/*
 * Copyright (C) 2017 Tobias Brunner
 * HSR Hochschule fuer Technik Rapperswil
 *
 * Copyright (C) 2013 Martin Willi
 * Copyright (C) 2013 revosec AG
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

#include "test_suite.h"

#include <plugins/plugin_feature.h>
#include <credentials/keys/signature_params.h>

/**
 * Signature schemes to test
 */
static signature_scheme_t schemes[] = {
	SIGN_RSA_EMSA_PKCS1_NULL,
	SIGN_RSA_EMSA_PKCS1_MD5,
	SIGN_RSA_EMSA_PKCS1_SHA1,
	SIGN_RSA_EMSA_PKCS1_SHA2_224,
	SIGN_RSA_EMSA_PKCS1_SHA2_256,
	SIGN_RSA_EMSA_PKCS1_SHA2_384,
	SIGN_RSA_EMSA_PKCS1_SHA2_512,
	SIGN_RSA_EMSA_PSS,
};
/**
 * Default parameters for RSA PSS signatures
 */
static rsa_pss_params_t default_pss_params = {
	.hash = HASH_SHA256,
	.mgf1_hash = HASH_SHA256,
	.salt_len = HASH_SIZE_SHA256,
};

/**
 * Perform a signature verification "good" test having a keypair
 */
static void test_good_sig(private_key_t *privkey, public_key_t *pubkey)
{
	chunk_t sig, data = chunk_from_chars(0x01,0x02,0x03,0xFD,0xFE,0xFF);
	int i;

	for (i = 0; i < countof(schemes); i++)
	{
		rsa_pss_params_t *params = NULL;

		if (!lib->plugins->has_feature(lib->plugins,
						PLUGIN_PROVIDE(PUBKEY_VERIFY, schemes[i])) ||
			!lib->plugins->has_feature(lib->plugins,
						PLUGIN_PROVIDE(PRIVKEY_SIGN, schemes[i])))
		{
			warn("%N not supported, skip scheme", signature_scheme_names,
				 schemes[i]);
			continue;
		}
		if (schemes[i] == SIGN_RSA_EMSA_PSS)
		{
			params = &default_pss_params;
		}
		fail_unless(privkey->sign(privkey, schemes[i], params, data, &sig),
					"sign %N", signature_scheme_names, schemes[i]);
		fail_unless(pubkey->verify(pubkey, schemes[i], params, data, sig),
					"verify %N", signature_scheme_names, schemes[i]);
		free(sig.ptr);
	}
}

/**
 * Some special signatures that should never validate successfully
 */
static chunk_t invalid_sigs[] = {
	chunk_from_chars(),
	chunk_from_chars(0x00),
	chunk_from_chars(0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
	chunk_from_chars(0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
					 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00),
};

/**
 * Check public key that it properly fails against some crafted sigs
 */
static void test_bad_sigs(public_key_t *pubkey)
{
	chunk_t data = chunk_from_chars(0x01,0x02,0x03,0xFD,0xFE,0xFF);
	int s, i;

	for (s = 0; s < countof(schemes); s++)
	{
		rsa_pss_params_t *params = NULL;

		if (!lib->plugins->has_feature(lib->plugins,
						PLUGIN_PROVIDE(PUBKEY_VERIFY, schemes[s])))
		{
			warn("%N not supported, skip scheme", signature_scheme_names,
				 schemes[s]);
			continue;
		}
		if (schemes[s] == SIGN_RSA_EMSA_PSS)
		{
			params = &default_pss_params;
		}
		for (i = 0; i < countof(invalid_sigs); i++)
		{
			fail_if(
				pubkey->verify(pubkey, schemes[s], params, data, invalid_sigs[i]),
				"bad %N sig accepted %B", signature_scheme_names, schemes[s],
				&invalid_sigs[i]);
		}
	}
}

/**
 * RSA key sizes to test
 */
static int key_sizes[] = {
	1024, 1536, 2048, 3072, 4096,
};

START_TEST(test_gen)
{
	private_key_t *privkey;
	public_key_t *pubkey;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_RSA,
								 BUILD_KEY_SIZE, key_sizes[_i], BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert_int_eq(key_sizes[_i], privkey->get_keysize(privkey));
	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);
	ck_assert_int_eq(key_sizes[_i], pubkey->get_keysize(pubkey));

	test_good_sig(privkey, pubkey);

	test_bad_sigs(pubkey);

	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}
END_TEST

/**
 * Private keys to load
 */
static struct {
	chunk_t key;
	chunk_t pkcs8;
	chunk_t pub;
	chunk_t fp_pk;
	chunk_t fp_spki;
} keys[] = {
	{ chunk_from_chars( /* RSA-768 */
		0x30,0x82,0x01,0xcb,0x02,0x01,0x00,0x02,0x61,0x00,0xd1,0x5d,0x98,0x97,0x95,0x98,
		0x19,0x87,0x20,0x3f,0x10,0xb0,0x05,0x36,0x1e,0x1b,0xcd,0xc8,0x93,0x66,0xd7,0x43,
		0xed,0x84,0xb0,0x3e,0x96,0xd3,0xe7,0x27,0x0e,0xc0,0xba,0xdf,0x7e,0x32,0x05,0xd3,
		0x08,0xd6,0x44,0xd5,0x01,0x2b,0x3e,0x5d,0xc0,0x37,0xae,0x4f,0xe0,0xea,0x8d,0x2c,
		0x42,0x4c,0xa9,0xa2,0x42,0xbe,0xdd,0xdb,0xf7,0xd3,0x28,0x07,0x10,0x88,0x53,0x15,
		0xb2,0x4f,0xb5,0x9d,0x47,0x9b,0xd6,0xc8,0xfe,0x5b,0xa2,0xd7,0xe1,0x13,0xca,0x0b,
		0xce,0x7a,0xed,0xa2,0x3e,0xd5,0x9b,0xb8,0x8b,0x4f,0x02,0x03,0x01,0x00,0x01,0x02,
		0x60,0x2d,0x83,0x82,0x53,0x99,0xb2,0xaa,0x02,0x05,0x11,0x90,0xa8,0x23,0x49,0xe3,
		0x7b,0xb9,0xdd,0x9b,0xa5,0xa4,0xb0,0x60,0xa7,0x12,0xc5,0x58,0x76,0x92,0x6e,0x9c,
		0x37,0x6b,0xa8,0x80,0x3f,0x91,0xa2,0x91,0xee,0x3a,0xa2,0x6f,0x91,0x9e,0x0a,0x35,
		0x69,0xc0,0xa7,0xdc,0xd8,0x46,0xe4,0x29,0x1c,0x3d,0x34,0x30,0xa2,0xb9,0x0d,0x34,
		0x94,0xa1,0x12,0xa7,0x85,0xd3,0x2c,0x47,0x1b,0xf0,0x78,0xd5,0x22,0xfc,0xa5,0xe0,
		0x75,0xac,0x71,0x21,0xe8,0xe8,0x19,0x9f,0xbb,0x98,0x5c,0xa6,0x9d,0x42,0xd7,0x9c,
		0x89,0x02,0x31,0x00,0xee,0xaa,0x9e,0x82,0xe1,0xb2,0xdd,0x05,0xbc,0x2e,0x53,0xe9,
		0x64,0x4b,0x48,0x06,0x3a,0xfd,0x9e,0x91,0xce,0x1b,0x7f,0x66,0xbc,0xd2,0xc4,0xab,
		0xbf,0xc5,0x5d,0x1a,0xbd,0xd6,0xb5,0x9c,0x5c,0x18,0x01,0xe6,0x79,0x19,0xf2,0xc3,
		0x1d,0x66,0x88,0x2d,0x02,0x31,0x00,0xe0,0x92,0x34,0x1e,0x09,0xf2,0x1b,0xf9,0xbf,
		0x11,0x65,0x3f,0xc8,0x85,0x5a,0xe6,0xc0,0xcf,0x93,0x44,0xb0,0x50,0xe4,0x8b,0x6f,
		0x30,0xde,0x42,0x0c,0x8a,0x77,0x0d,0x98,0x7f,0x52,0x59,0x9e,0x87,0xb8,0x6e,0xdc,
		0xed,0x15,0x80,0xbd,0xbb,0xf2,0xeb,0x02,0x31,0x00,0xb0,0x6b,0x36,0x98,0x90,0xb5,
		0x62,0x63,0xa6,0xe2,0xa7,0xec,0x51,0xd2,0xc3,0xfe,0xb7,0x04,0x5a,0x7e,0x74,0xd8,
		0x26,0xa8,0x8e,0xd3,0x4d,0xc5,0x97,0x10,0x10,0xee,0x7f,0x7d,0x82,0xe9,0x7d,0xb9,
		0xd1,0x4d,0xc8,0x1e,0xc2,0x30,0x30,0x3f,0x66,0x51,0x02,0x31,0x00,0xaa,0x75,0x2f,
		0x4c,0x11,0xbe,0x8d,0x0f,0x8f,0xc1,0x13,0x7a,0x4b,0xa9,0x35,0x6b,0x6b,0xb4,0xe3,
		0x92,0xc2,0xc6,0x54,0x03,0xa6,0x5d,0x90,0x86,0xcf,0xe0,0x16,0x27,0xe2,0xb5,0xd9,
		0xfb,0x1e,0x82,0xe4,0x32,0x7a,0x4d,0x17,0x02,0x46,0x82,0x30,0x0b,0x02,0x30,0x09,
		0xf3,0xce,0x9b,0x02,0xc5,0x53,0xe9,0xa2,0x89,0xe2,0x3b,0x8c,0x8b,0xe9,0xc2,0xba,
		0x94,0x76,0x60,0x27,0x2b,0xe9,0x92,0xc1,0x5e,0x3c,0xc3,0x77,0x9b,0xc7,0xce,0xc6,
		0x67,0xd5,0x20,0x2c,0x54,0xa1,0x5d,0x2a,0x17,0x16,0x66,0xdf,0x5a,0xe9,0x87),
	  chunk_from_chars(
		0x30,0x82,0x01,0xe5,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x01,0xcf,0x30,0x82,0x01,0xcb,0x02,0x01,
		0x00,0x02,0x61,0x00,0xd1,0x5d,0x98,0x97,0x95,0x98,0x19,0x87,0x20,0x3f,0x10,0xb0,
		0x05,0x36,0x1e,0x1b,0xcd,0xc8,0x93,0x66,0xd7,0x43,0xed,0x84,0xb0,0x3e,0x96,0xd3,
		0xe7,0x27,0x0e,0xc0,0xba,0xdf,0x7e,0x32,0x05,0xd3,0x08,0xd6,0x44,0xd5,0x01,0x2b,
		0x3e,0x5d,0xc0,0x37,0xae,0x4f,0xe0,0xea,0x8d,0x2c,0x42,0x4c,0xa9,0xa2,0x42,0xbe,
		0xdd,0xdb,0xf7,0xd3,0x28,0x07,0x10,0x88,0x53,0x15,0xb2,0x4f,0xb5,0x9d,0x47,0x9b,
		0xd6,0xc8,0xfe,0x5b,0xa2,0xd7,0xe1,0x13,0xca,0x0b,0xce,0x7a,0xed,0xa2,0x3e,0xd5,
		0x9b,0xb8,0x8b,0x4f,0x02,0x03,0x01,0x00,0x01,0x02,0x60,0x2d,0x83,0x82,0x53,0x99,
		0xb2,0xaa,0x02,0x05,0x11,0x90,0xa8,0x23,0x49,0xe3,0x7b,0xb9,0xdd,0x9b,0xa5,0xa4,
		0xb0,0x60,0xa7,0x12,0xc5,0x58,0x76,0x92,0x6e,0x9c,0x37,0x6b,0xa8,0x80,0x3f,0x91,
		0xa2,0x91,0xee,0x3a,0xa2,0x6f,0x91,0x9e,0x0a,0x35,0x69,0xc0,0xa7,0xdc,0xd8,0x46,
		0xe4,0x29,0x1c,0x3d,0x34,0x30,0xa2,0xb9,0x0d,0x34,0x94,0xa1,0x12,0xa7,0x85,0xd3,
		0x2c,0x47,0x1b,0xf0,0x78,0xd5,0x22,0xfc,0xa5,0xe0,0x75,0xac,0x71,0x21,0xe8,0xe8,
		0x19,0x9f,0xbb,0x98,0x5c,0xa6,0x9d,0x42,0xd7,0x9c,0x89,0x02,0x31,0x00,0xee,0xaa,
		0x9e,0x82,0xe1,0xb2,0xdd,0x05,0xbc,0x2e,0x53,0xe9,0x64,0x4b,0x48,0x06,0x3a,0xfd,
		0x9e,0x91,0xce,0x1b,0x7f,0x66,0xbc,0xd2,0xc4,0xab,0xbf,0xc5,0x5d,0x1a,0xbd,0xd6,
		0xb5,0x9c,0x5c,0x18,0x01,0xe6,0x79,0x19,0xf2,0xc3,0x1d,0x66,0x88,0x2d,0x02,0x31,
		0x00,0xe0,0x92,0x34,0x1e,0x09,0xf2,0x1b,0xf9,0xbf,0x11,0x65,0x3f,0xc8,0x85,0x5a,
		0xe6,0xc0,0xcf,0x93,0x44,0xb0,0x50,0xe4,0x8b,0x6f,0x30,0xde,0x42,0x0c,0x8a,0x77,
		0x0d,0x98,0x7f,0x52,0x59,0x9e,0x87,0xb8,0x6e,0xdc,0xed,0x15,0x80,0xbd,0xbb,0xf2,
		0xeb,0x02,0x31,0x00,0xb0,0x6b,0x36,0x98,0x90,0xb5,0x62,0x63,0xa6,0xe2,0xa7,0xec,
		0x51,0xd2,0xc3,0xfe,0xb7,0x04,0x5a,0x7e,0x74,0xd8,0x26,0xa8,0x8e,0xd3,0x4d,0xc5,
		0x97,0x10,0x10,0xee,0x7f,0x7d,0x82,0xe9,0x7d,0xb9,0xd1,0x4d,0xc8,0x1e,0xc2,0x30,
		0x30,0x3f,0x66,0x51,0x02,0x31,0x00,0xaa,0x75,0x2f,0x4c,0x11,0xbe,0x8d,0x0f,0x8f,
		0xc1,0x13,0x7a,0x4b,0xa9,0x35,0x6b,0x6b,0xb4,0xe3,0x92,0xc2,0xc6,0x54,0x03,0xa6,
		0x5d,0x90,0x86,0xcf,0xe0,0x16,0x27,0xe2,0xb5,0xd9,0xfb,0x1e,0x82,0xe4,0x32,0x7a,
		0x4d,0x17,0x02,0x46,0x82,0x30,0x0b,0x02,0x30,0x09,0xf3,0xce,0x9b,0x02,0xc5,0x53,
		0xe9,0xa2,0x89,0xe2,0x3b,0x8c,0x8b,0xe9,0xc2,0xba,0x94,0x76,0x60,0x27,0x2b,0xe9,
		0x92,0xc1,0x5e,0x3c,0xc3,0x77,0x9b,0xc7,0xce,0xc6,0x67,0xd5,0x20,0x2c,0x54,0xa1,
		0x5d,0x2a,0x17,0x16,0x66,0xdf,0x5a,0xe9,0x87),
	  chunk_from_chars(
		0x30,0x7c,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,
		0x00,0x03,0x6b,0x00,0x30,0x68,0x02,0x61,0x00,0xd1,0x5d,0x98,0x97,0x95,0x98,0x19,
		0x87,0x20,0x3f,0x10,0xb0,0x05,0x36,0x1e,0x1b,0xcd,0xc8,0x93,0x66,0xd7,0x43,0xed,
		0x84,0xb0,0x3e,0x96,0xd3,0xe7,0x27,0x0e,0xc0,0xba,0xdf,0x7e,0x32,0x05,0xd3,0x08,
		0xd6,0x44,0xd5,0x01,0x2b,0x3e,0x5d,0xc0,0x37,0xae,0x4f,0xe0,0xea,0x8d,0x2c,0x42,
		0x4c,0xa9,0xa2,0x42,0xbe,0xdd,0xdb,0xf7,0xd3,0x28,0x07,0x10,0x88,0x53,0x15,0xb2,
		0x4f,0xb5,0x9d,0x47,0x9b,0xd6,0xc8,0xfe,0x5b,0xa2,0xd7,0xe1,0x13,0xca,0x0b,0xce,
		0x7a,0xed,0xa2,0x3e,0xd5,0x9b,0xb8,0x8b,0x4f,0x02,0x03,0x01,0x00,0x01),
	  chunk_from_chars(
		0x06,0xad,0x82,0xc3,0x58,0x22,0xbb,0x79,0xb5,0xfc,0x48,0xdb,0xa0,0x3c,0x39,0x60,
		0x00,0x85,0x06,0xca),
	  chunk_from_chars(
		0x60,0xa2,0x6d,0x32,0x97,0xc5,0x91,0x44,0xa6,0xa6,0xfe,0xe2,0x15,0xb7,0xa5,0xdc,
		0x79,0x60,0xfd,0x3b),
	},
	{ chunk_from_chars( /* RSA-1024 */
		0x30,0x82,0x02,0x5c,0x02,0x01,0x00,0x02,0x81,0x81,0x00,0xc0,0xbd,0x48,0x83,0xbc,
		0xea,0x0b,0x32,0x06,0x4b,0xf5,0x10,0x54,0x1b,0xba,0x88,0xc4,0x10,0x7e,0x47,0xec,
		0x0e,0xf9,0xb4,0xcf,0x9a,0x02,0xc6,0xb3,0xaf,0x35,0xc8,0xaf,0x78,0x1a,0xbc,0x37,
		0x1a,0x25,0x7a,0x37,0x24,0x73,0x53,0x9a,0xf0,0x44,0x64,0x5b,0x6b,0x64,0x4c,0xfa,
		0x83,0x3a,0x0f,0x77,0x5d,0x7b,0x21,0xa2,0x25,0x00,0x11,0xae,0x72,0x36,0x35,0xd9,
		0x0d,0xef,0x5a,0xdd,0x98,0x35,0x49,0xaf,0x44,0xa0,0x33,0x29,0xc0,0xca,0xf5,0x6f,
		0xfe,0xc1,0x06,0x4c,0x80,0x9a,0x54,0xbe,0x46,0x1a,0x96,0xb1,0xf3,0x29,0xb8,0x9d,
		0x07,0x84,0x03,0x68,0x6b,0x9f,0xbf,0xe5,0xd8,0x14,0x2a,0xe0,0xef,0xbd,0x1a,0x61,
		0x0d,0x3a,0xc8,0x67,0xcd,0x99,0x90,0xe3,0xe6,0x52,0x83,0x02,0x03,0x01,0x00,0x01,
		0x02,0x81,0x80,0x13,0xd2,0xa3,0xe5,0xa0,0xb0,0x0a,0xe2,0x0f,0x3c,0x65,0x57,0xa8,
		0xe9,0x87,0xd5,0x79,0xcc,0xc9,0xca,0xc8,0x8a,0xd5,0xc0,0x74,0x90,0x3e,0x1e,0xda,
		0x40,0xcd,0x42,0xf7,0x01,0x09,0x9c,0x37,0xfd,0x41,0x6e,0x2b,0x6e,0x5d,0x4a,0x1e,
		0x52,0x53,0x1b,0xbb,0x3c,0x9f,0xfe,0x91,0x79,0x48,0xfc,0x69,0x90,0xbc,0xbc,0x3d,
		0xcf,0xee,0x62,0x0a,0xbd,0x57,0x6b,0xa9,0x51,0x3e,0xc2,0x7f,0x26,0xb1,0xaa,0x38,
		0xeb,0x40,0x91,0x3a,0x3c,0x80,0x1e,0x4e,0xe2,0xff,0xa2,0x8e,0x56,0xbb,0xb3,0xeb,
		0x24,0x81,0x4c,0x19,0x2c,0x8f,0x51,0x4c,0x04,0x81,0xaf,0x5e,0xc2,0xa6,0xf9,0xd3,
		0x48,0xee,0xe9,0x6d,0x9b,0xe1,0xe5,0x17,0x4f,0x07,0x18,0xea,0x96,0xd3,0x2c,0xce,
		0x44,0x71,0x51,0x02,0x41,0x00,0xe9,0xe9,0x46,0x7e,0xe1,0xc2,0x86,0x94,0x65,0x77,
		0x9c,0xc7,0x76,0x5d,0xa0,0xd3,0xcc,0x1f,0xa3,0xc7,0xfe,0xbb,0x4e,0x27,0xd6,0x43,
		0x6b,0xbd,0x0d,0x05,0x7a,0x10,0xe8,0x48,0x97,0x30,0xaa,0x53,0x61,0x57,0x1f,0x8a,
		0xf7,0x39,0x5e,0xa6,0xfe,0xe9,0x2c,0x19,0x5e,0x53,0xea,0xc2,0xb2,0xc2,0x11,0x3c,
		0x18,0xab,0xcf,0xc4,0x91,0x1b,0x02,0x41,0x00,0xd2,0xf0,0xb1,0x49,0xa1,0x6f,0xf1,
		0x83,0xa3,0xd2,0xa1,0x0e,0xb3,0xb3,0x33,0x01,0xed,0xd0,0x28,0xc1,0x2f,0x88,0x80,
		0x9f,0x43,0x7c,0x7e,0x5d,0x4c,0x15,0x05,0x86,0xff,0x75,0x9b,0xf1,0x64,0xde,0x06,
		0xbf,0xdd,0x98,0x50,0xd9,0x4a,0x3a,0xd6,0x25,0x1c,0xdd,0xc8,0x56,0x12,0x11,0xb9,
		0x02,0x42,0xc7,0x1d,0x86,0xeb,0xd9,0xc2,0xb9,0x02,0x41,0x00,0x80,0x25,0x8c,0xb9,
		0x76,0x75,0x5b,0xc5,0x70,0xd1,0x56,0xd2,0xef,0xc5,0xdb,0x96,0x2c,0xfe,0x28,0x7c,
		0x28,0xd1,0xf4,0xbf,0x5e,0x63,0x11,0x63,0x40,0xfe,0xff,0x20,0xc4,0x21,0x00,0xb3,
		0x68,0x9c,0xc5,0x77,0x35,0x90,0xac,0x60,0x81,0xba,0x7b,0x6c,0xc2,0xfc,0x22,0xf1,
		0x56,0x6b,0xd4,0x02,0xfd,0xee,0x2e,0x95,0xf1,0xfd,0x7e,0x81,0x02,0x40,0x47,0xaf,
		0x84,0x90,0x81,0x4c,0x89,0xc7,0x32,0xe5,0x61,0xd6,0x9d,0x3b,0x49,0x1a,0x5e,0xb7,
		0x5f,0x22,0x48,0x05,0x1b,0xb1,0x04,0x3e,0x4a,0xb3,0x6a,0x27,0xba,0xb9,0x26,0x17,
		0xd1,0xe7,0x37,0x60,0x3c,0xea,0xf7,0x63,0xcc,0x16,0x0c,0x23,0xf2,0xa2,0xaa,0x2c,
		0xb4,0xe8,0x8b,0x3b,0x7a,0xa4,0x4a,0x0d,0x60,0xfb,0x79,0x2b,0x88,0x01,0x02,0x40,
		0x42,0xee,0x12,0x91,0xf9,0x80,0x1e,0x60,0x0b,0xaa,0xbe,0xfd,0x09,0x84,0x93,0x0d,
		0x09,0xd3,0x1e,0x37,0x52,0xb0,0xe8,0x51,0x4f,0xd3,0x9e,0xda,0x32,0x38,0x22,0x35,
		0xdb,0x25,0x8b,0x9f,0x1a,0xb5,0xf1,0x75,0xfa,0x4d,0x09,0x42,0x01,0x64,0xe6,0xc4,
		0x6e,0xba,0x2d,0x88,0x92,0xbe,0xa9,0x1f,0x85,0x38,0x10,0xa3,0x0e,0x1a,0x92,0x54),
	  chunk_from_chars(
		0x30,0x82,0x02,0x76,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x02,0x60,0x30,0x82,0x02,0x5c,0x02,0x01,
		0x00,0x02,0x81,0x81,0x00,0xc0,0xbd,0x48,0x83,0xbc,0xea,0x0b,0x32,0x06,0x4b,0xf5,
		0x10,0x54,0x1b,0xba,0x88,0xc4,0x10,0x7e,0x47,0xec,0x0e,0xf9,0xb4,0xcf,0x9a,0x02,
		0xc6,0xb3,0xaf,0x35,0xc8,0xaf,0x78,0x1a,0xbc,0x37,0x1a,0x25,0x7a,0x37,0x24,0x73,
		0x53,0x9a,0xf0,0x44,0x64,0x5b,0x6b,0x64,0x4c,0xfa,0x83,0x3a,0x0f,0x77,0x5d,0x7b,
		0x21,0xa2,0x25,0x00,0x11,0xae,0x72,0x36,0x35,0xd9,0x0d,0xef,0x5a,0xdd,0x98,0x35,
		0x49,0xaf,0x44,0xa0,0x33,0x29,0xc0,0xca,0xf5,0x6f,0xfe,0xc1,0x06,0x4c,0x80,0x9a,
		0x54,0xbe,0x46,0x1a,0x96,0xb1,0xf3,0x29,0xb8,0x9d,0x07,0x84,0x03,0x68,0x6b,0x9f,
		0xbf,0xe5,0xd8,0x14,0x2a,0xe0,0xef,0xbd,0x1a,0x61,0x0d,0x3a,0xc8,0x67,0xcd,0x99,
		0x90,0xe3,0xe6,0x52,0x83,0x02,0x03,0x01,0x00,0x01,0x02,0x81,0x80,0x13,0xd2,0xa3,
		0xe5,0xa0,0xb0,0x0a,0xe2,0x0f,0x3c,0x65,0x57,0xa8,0xe9,0x87,0xd5,0x79,0xcc,0xc9,
		0xca,0xc8,0x8a,0xd5,0xc0,0x74,0x90,0x3e,0x1e,0xda,0x40,0xcd,0x42,0xf7,0x01,0x09,
		0x9c,0x37,0xfd,0x41,0x6e,0x2b,0x6e,0x5d,0x4a,0x1e,0x52,0x53,0x1b,0xbb,0x3c,0x9f,
		0xfe,0x91,0x79,0x48,0xfc,0x69,0x90,0xbc,0xbc,0x3d,0xcf,0xee,0x62,0x0a,0xbd,0x57,
		0x6b,0xa9,0x51,0x3e,0xc2,0x7f,0x26,0xb1,0xaa,0x38,0xeb,0x40,0x91,0x3a,0x3c,0x80,
		0x1e,0x4e,0xe2,0xff,0xa2,0x8e,0x56,0xbb,0xb3,0xeb,0x24,0x81,0x4c,0x19,0x2c,0x8f,
		0x51,0x4c,0x04,0x81,0xaf,0x5e,0xc2,0xa6,0xf9,0xd3,0x48,0xee,0xe9,0x6d,0x9b,0xe1,
		0xe5,0x17,0x4f,0x07,0x18,0xea,0x96,0xd3,0x2c,0xce,0x44,0x71,0x51,0x02,0x41,0x00,
		0xe9,0xe9,0x46,0x7e,0xe1,0xc2,0x86,0x94,0x65,0x77,0x9c,0xc7,0x76,0x5d,0xa0,0xd3,
		0xcc,0x1f,0xa3,0xc7,0xfe,0xbb,0x4e,0x27,0xd6,0x43,0x6b,0xbd,0x0d,0x05,0x7a,0x10,
		0xe8,0x48,0x97,0x30,0xaa,0x53,0x61,0x57,0x1f,0x8a,0xf7,0x39,0x5e,0xa6,0xfe,0xe9,
		0x2c,0x19,0x5e,0x53,0xea,0xc2,0xb2,0xc2,0x11,0x3c,0x18,0xab,0xcf,0xc4,0x91,0x1b,
		0x02,0x41,0x00,0xd2,0xf0,0xb1,0x49,0xa1,0x6f,0xf1,0x83,0xa3,0xd2,0xa1,0x0e,0xb3,
		0xb3,0x33,0x01,0xed,0xd0,0x28,0xc1,0x2f,0x88,0x80,0x9f,0x43,0x7c,0x7e,0x5d,0x4c,
		0x15,0x05,0x86,0xff,0x75,0x9b,0xf1,0x64,0xde,0x06,0xbf,0xdd,0x98,0x50,0xd9,0x4a,
		0x3a,0xd6,0x25,0x1c,0xdd,0xc8,0x56,0x12,0x11,0xb9,0x02,0x42,0xc7,0x1d,0x86,0xeb,
		0xd9,0xc2,0xb9,0x02,0x41,0x00,0x80,0x25,0x8c,0xb9,0x76,0x75,0x5b,0xc5,0x70,0xd1,
		0x56,0xd2,0xef,0xc5,0xdb,0x96,0x2c,0xfe,0x28,0x7c,0x28,0xd1,0xf4,0xbf,0x5e,0x63,
		0x11,0x63,0x40,0xfe,0xff,0x20,0xc4,0x21,0x00,0xb3,0x68,0x9c,0xc5,0x77,0x35,0x90,
		0xac,0x60,0x81,0xba,0x7b,0x6c,0xc2,0xfc,0x22,0xf1,0x56,0x6b,0xd4,0x02,0xfd,0xee,
		0x2e,0x95,0xf1,0xfd,0x7e,0x81,0x02,0x40,0x47,0xaf,0x84,0x90,0x81,0x4c,0x89,0xc7,
		0x32,0xe5,0x61,0xd6,0x9d,0x3b,0x49,0x1a,0x5e,0xb7,0x5f,0x22,0x48,0x05,0x1b,0xb1,
		0x04,0x3e,0x4a,0xb3,0x6a,0x27,0xba,0xb9,0x26,0x17,0xd1,0xe7,0x37,0x60,0x3c,0xea,
		0xf7,0x63,0xcc,0x16,0x0c,0x23,0xf2,0xa2,0xaa,0x2c,0xb4,0xe8,0x8b,0x3b,0x7a,0xa4,
		0x4a,0x0d,0x60,0xfb,0x79,0x2b,0x88,0x01,0x02,0x40,0x42,0xee,0x12,0x91,0xf9,0x80,
		0x1e,0x60,0x0b,0xaa,0xbe,0xfd,0x09,0x84,0x93,0x0d,0x09,0xd3,0x1e,0x37,0x52,0xb0,
		0xe8,0x51,0x4f,0xd3,0x9e,0xda,0x32,0x38,0x22,0x35,0xdb,0x25,0x8b,0x9f,0x1a,0xb5,
		0xf1,0x75,0xfa,0x4d,0x09,0x42,0x01,0x64,0xe6,0xc4,0x6e,0xba,0x2d,0x88,0x92,0xbe,
		0xa9,0x1f,0x85,0x38,0x10,0xa3,0x0e,0x1a,0x92,0x54),
	  chunk_from_chars(
		0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,
		0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xc0,0xbd,0x48,
		0x83,0xbc,0xea,0x0b,0x32,0x06,0x4b,0xf5,0x10,0x54,0x1b,0xba,0x88,0xc4,0x10,0x7e,
		0x47,0xec,0x0e,0xf9,0xb4,0xcf,0x9a,0x02,0xc6,0xb3,0xaf,0x35,0xc8,0xaf,0x78,0x1a,
		0xbc,0x37,0x1a,0x25,0x7a,0x37,0x24,0x73,0x53,0x9a,0xf0,0x44,0x64,0x5b,0x6b,0x64,
		0x4c,0xfa,0x83,0x3a,0x0f,0x77,0x5d,0x7b,0x21,0xa2,0x25,0x00,0x11,0xae,0x72,0x36,
		0x35,0xd9,0x0d,0xef,0x5a,0xdd,0x98,0x35,0x49,0xaf,0x44,0xa0,0x33,0x29,0xc0,0xca,
		0xf5,0x6f,0xfe,0xc1,0x06,0x4c,0x80,0x9a,0x54,0xbe,0x46,0x1a,0x96,0xb1,0xf3,0x29,
		0xb8,0x9d,0x07,0x84,0x03,0x68,0x6b,0x9f,0xbf,0xe5,0xd8,0x14,0x2a,0xe0,0xef,0xbd,
		0x1a,0x61,0x0d,0x3a,0xc8,0x67,0xcd,0x99,0x90,0xe3,0xe6,0x52,0x83,0x02,0x03,0x01,
		0x00,0x01),
	  chunk_from_chars(
		0xda,0xab,0x50,0x22,0x4b,0x4f,0x3b,0xd0,0x82,0xc4,0xa4,0x14,0x06,0x64,0x0b,0x6f,
		0xad,0xbc,0x69,0xc0),
	  chunk_from_chars(
		0x92,0xf4,0x02,0x8a,0x3c,0x8f,0x05,0xad,0x3a,0x55,0x1f,0xcf,0x13,0xd8,0xff,0xf2,
		0x65,0x35,0x75,0xa0),
	},
	{ chunk_from_chars( /* RSA-1536 */
		0x30,0x82,0x03,0x7d,0x02,0x01,0x00,0x02,0x81,0xc1,0x00,0xba,0xe3,0x37,0x93,0x7e,
		0x42,0x13,0x3c,0xba,0x41,0xc1,0x7b,0xf0,0xcc,0x7a,0x44,0xc6,0x54,0xc8,0x77,0x01,
		0x70,0x2f,0x6e,0x4a,0xcf,0x2d,0x07,0xab,0x01,0xc0,0x43,0xab,0x8d,0x33,0xb3,0xd4,
		0xeb,0xe3,0x90,0xf6,0x01,0x03,0x75,0x03,0x1d,0xe8,0x06,0x40,0x15,0xfa,0x96,0x0b,
		0xd5,0x26,0x64,0xea,0x55,0x82,0x16,0x7b,0xd5,0x1e,0xaa,0x08,0xc7,0x30,0x1a,0x59,
		0xf8,0xd9,0xe3,0x9e,0x89,0xd9,0x92,0x2c,0x32,0x79,0x0e,0xb3,0x25,0xbc,0x1d,0x7c,
		0x59,0xde,0x05,0x47,0x8f,0x61,0x77,0xf5,0x4f,0xed,0x82,0x2c,0xf8,0x2a,0x3e,0x02,
		0xf3,0xc0,0x15,0x51,0xde,0x05,0xc4,0xfc,0x80,0x91,0xae,0x06,0x1b,0xd7,0x39,0x8e,
		0x9a,0x6d,0xb3,0x2f,0xb0,0xd0,0xc8,0x96,0xa6,0x88,0xb3,0x17,0xca,0x58,0xbe,0x38,
		0x2c,0x64,0x35,0x5a,0x29,0xb7,0xf8,0x74,0x3d,0xbb,0xec,0x90,0x01,0x04,0x64,0x3d,
		0x38,0x0f,0x87,0xce,0xd7,0xfc,0xd2,0x96,0x93,0x31,0x85,0x0d,0x2d,0xa5,0x91,0xe2,
		0xfc,0x7b,0xea,0xb0,0x89,0x24,0xaa,0x00,0x29,0x8c,0x26,0x7c,0x94,0x54,0x74,0xe4,
		0x11,0xa8,0x04,0x6f,0x40,0xeb,0xaf,0xed,0xac,0x75,0x33,0x02,0x03,0x01,0x00,0x01,
		0x02,0x81,0xc0,0x0a,0x96,0xec,0x63,0xc1,0xa0,0x39,0xd9,0xd3,0x8d,0xfd,0x4a,0x2a,
		0x13,0x54,0x0c,0x48,0x96,0xae,0x43,0x3c,0x04,0x20,0xd3,0xe5,0x8e,0x46,0xb5,0x6c,
		0x05,0xad,0xe0,0xc7,0xbc,0x39,0x05,0x44,0x17,0xd7,0xad,0xb3,0x9a,0xcc,0x18,0xd9,
		0xc3,0xdc,0x8d,0x5a,0x1d,0x44,0xb5,0x32,0xd7,0x71,0x94,0xff,0x48,0x38,0x16,0x51,
		0x0e,0xfa,0xed,0x54,0x91,0x00,0xd3,0x45,0x6c,0xd9,0xdf,0xd1,0x70,0x6b,0x31,0x22,
		0xaa,0xfb,0x7c,0x0f,0x3f,0xa0,0xa0,0xa5,0x16,0xac,0x83,0x6d,0x12,0x1d,0x4a,0x40,
		0x4e,0xb6,0x9c,0xf4,0x67,0xaa,0xa9,0xb0,0xc8,0xb4,0x0a,0xd5,0x3b,0x5c,0x19,0xed,
		0x86,0x83,0x5a,0x75,0xbc,0xeb,0x17,0xc8,0x16,0xa0,0x60,0x2e,0xb6,0x25,0xc5,0x4d,
		0x59,0xba,0x62,0xcb,0x3d,0x91,0x7c,0x79,0x6a,0x4b,0x4a,0x54,0xbd,0xb7,0xa3,0x89,
		0x7f,0xbf,0x0e,0x77,0xe1,0x54,0x29,0x0d,0x45,0x6d,0xa8,0x15,0xa5,0x17,0x8c,0xcf,
		0x27,0x9e,0x47,0x4e,0x2a,0x91,0x7e,0x4e,0x14,0x59,0x8c,0x62,0x91,0xa3,0x40,0xa5,
		0x9e,0x67,0xbb,0x02,0x97,0xb4,0xe7,0x06,0x04,0xbc,0x16,0x24,0x3d,0x49,0xb1,0xf0,
		0xae,0xfc,0x1d,0x02,0x61,0x00,0xde,0x86,0x5d,0x49,0x88,0xeb,0x5c,0xd3,0xe5,0x11,
		0x48,0x0b,0x1e,0x52,0x95,0xa9,0x65,0x99,0x89,0xcf,0x51,0xb0,0x08,0xdd,0xb5,0x5b,
		0x64,0x1a,0x34,0xd2,0xee,0x4b,0x2d,0x8b,0xc1,0xd5,0xd6,0x1d,0x6c,0x0c,0x7e,0xa5,
		0x66,0x12,0xec,0xaf,0x5d,0xe9,0x33,0xd4,0xba,0x18,0x71,0x84,0x97,0xbe,0xc0,0x75,
		0x63,0x19,0xae,0xc6,0xc7,0x65,0xf3,0xf6,0xda,0x3f,0x91,0xfa,0x5e,0x87,0xf3,0xbc,
		0xd2,0x64,0x8d,0xcf,0xfb,0xdd,0x7f,0x9b,0x6c,0x81,0xba,0x9b,0x4e,0x94,0x5e,0x83,
		0xd1,0xcb,0xb9,0xf4,0x39,0x7f,0x02,0x61,0x00,0xd7,0x00,0x6d,0x8e,0x1b,0xa1,0x44,
		0xd9,0xff,0xe6,0x42,0x72,0x18,0x55,0x26,0x3e,0x87,0x40,0x71,0xb2,0x67,0x37,0x16,
		0xe9,0xbd,0x51,0x7f,0x0e,0x79,0x0e,0x75,0xa9,0x1f,0x0f,0x6b,0xa5,0x7c,0x5f,0xc8,
		0xdc,0x17,0xde,0x53,0x88,0x97,0x90,0x88,0xf2,0x4d,0x66,0x5e,0x0e,0x11,0x16,0x92,
		0x1e,0x61,0x56,0xe6,0xf0,0x74,0x81,0x58,0x95,0x05,0x29,0x71,0x9b,0xa0,0x69,0xed,
		0x14,0x23,0xf6,0x36,0x9b,0x8f,0x06,0x3a,0x76,0xab,0xeb,0xce,0xe8,0xdc,0x79,0xc1,
		0x29,0xb9,0xfc,0x49,0x7a,0x26,0x59,0xd6,0x4d,0x02,0x61,0x00,0xaf,0x3c,0xac,0xd6,
		0x2d,0xe6,0xfb,0x91,0x3a,0xc1,0x23,0x34,0xee,0x4a,0x26,0xe5,0xe1,0xc6,0xc9,0xc9,
		0xe4,0x10,0x76,0xca,0xf1,0xf8,0xe8,0x99,0xe2,0xa3,0x81,0x58,0xde,0xa3,0x42,0xa0,
		0x3d,0x1f,0xaa,0x69,0x24,0x8a,0xe8,0x19,0x5b,0x1e,0xb7,0x1b,0xe0,0xdf,0x53,0x35,
		0xd0,0x9f,0x94,0x48,0x79,0x93,0x77,0xd9,0x4f,0xd3,0xe6,0x4f,0x19,0x92,0x7a,0x48,
		0xb9,0x92,0xab,0x42,0xf0,0xe4,0xef,0xe2,0x93,0xf3,0x07,0xeb,0x64,0x84,0x67,0x2c,
		0xba,0x61,0x77,0xbe,0x4b,0xb8,0x0f,0x4d,0x1a,0x41,0x83,0xcd,0x02,0x60,0x56,0xec,
		0x55,0x5e,0x9e,0xcd,0x14,0x89,0x0e,0x6c,0x89,0x70,0x97,0x65,0xd5,0x90,0x72,0x1e,
		0x1b,0xd9,0x84,0xe1,0x40,0xe2,0x3f,0x28,0x33,0xb6,0x26,0x3b,0x32,0x56,0xad,0xb8,
		0x0e,0x4d,0x59,0x7b,0x60,0x39,0x9b,0x6c,0xc7,0x58,0xf1,0xed,0xfd,0x6f,0xf8,0xda,
		0xea,0x2b,0xc5,0xbc,0xda,0x56,0x6e,0x04,0x34,0x5a,0x02,0xc0,0x48,0x8f,0xf7,0x06,
		0x4a,0x68,0x20,0xf2,0xb2,0x66,0xf2,0x23,0x18,0xf0,0xcb,0x62,0x39,0x40,0xc1,0x41,
		0x14,0xe6,0x10,0x3d,0x29,0x5b,0x35,0x56,0x4a,0x5e,0x98,0x22,0xba,0x01,0x02,0x61,
		0x00,0xcc,0x80,0xb7,0xb9,0xb9,0x4a,0xaf,0x47,0x00,0x3e,0x21,0x0f,0xb8,0x4e,0x7c,
		0xb1,0xe4,0x25,0xd6,0x19,0x26,0x54,0xc6,0x8c,0x30,0x88,0x54,0x70,0xcf,0x1f,0x62,
		0x75,0xcb,0x18,0x58,0x6c,0x14,0xb0,0x9b,0x13,0x90,0xa2,0x1a,0x5a,0x79,0xa3,0x82,
		0xf0,0x9b,0xba,0xf0,0x90,0xaf,0xa1,0xe8,0xa8,0x70,0xef,0x60,0x6a,0x68,0xed,0x5a,
		0x21,0x77,0x69,0x7a,0xf2,0xee,0x3e,0xe5,0x90,0xd2,0x33,0x71,0x3b,0x82,0x88,0x75,
		0xdd,0x8e,0x6e,0xbc,0x17,0x83,0xef,0x37,0x82,0x4e,0x83,0x30,0xcb,0x8a,0xbc,0x6c,
		0x41),
	  chunk_from_chars(
		0x30,0x82,0x03,0x97,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x03,0x81,0x30,0x82,0x03,0x7d,0x02,0x01,
		0x00,0x02,0x81,0xc1,0x00,0xba,0xe3,0x37,0x93,0x7e,0x42,0x13,0x3c,0xba,0x41,0xc1,
		0x7b,0xf0,0xcc,0x7a,0x44,0xc6,0x54,0xc8,0x77,0x01,0x70,0x2f,0x6e,0x4a,0xcf,0x2d,
		0x07,0xab,0x01,0xc0,0x43,0xab,0x8d,0x33,0xb3,0xd4,0xeb,0xe3,0x90,0xf6,0x01,0x03,
		0x75,0x03,0x1d,0xe8,0x06,0x40,0x15,0xfa,0x96,0x0b,0xd5,0x26,0x64,0xea,0x55,0x82,
		0x16,0x7b,0xd5,0x1e,0xaa,0x08,0xc7,0x30,0x1a,0x59,0xf8,0xd9,0xe3,0x9e,0x89,0xd9,
		0x92,0x2c,0x32,0x79,0x0e,0xb3,0x25,0xbc,0x1d,0x7c,0x59,0xde,0x05,0x47,0x8f,0x61,
		0x77,0xf5,0x4f,0xed,0x82,0x2c,0xf8,0x2a,0x3e,0x02,0xf3,0xc0,0x15,0x51,0xde,0x05,
		0xc4,0xfc,0x80,0x91,0xae,0x06,0x1b,0xd7,0x39,0x8e,0x9a,0x6d,0xb3,0x2f,0xb0,0xd0,
		0xc8,0x96,0xa6,0x88,0xb3,0x17,0xca,0x58,0xbe,0x38,0x2c,0x64,0x35,0x5a,0x29,0xb7,
		0xf8,0x74,0x3d,0xbb,0xec,0x90,0x01,0x04,0x64,0x3d,0x38,0x0f,0x87,0xce,0xd7,0xfc,
		0xd2,0x96,0x93,0x31,0x85,0x0d,0x2d,0xa5,0x91,0xe2,0xfc,0x7b,0xea,0xb0,0x89,0x24,
		0xaa,0x00,0x29,0x8c,0x26,0x7c,0x94,0x54,0x74,0xe4,0x11,0xa8,0x04,0x6f,0x40,0xeb,
		0xaf,0xed,0xac,0x75,0x33,0x02,0x03,0x01,0x00,0x01,0x02,0x81,0xc0,0x0a,0x96,0xec,
		0x63,0xc1,0xa0,0x39,0xd9,0xd3,0x8d,0xfd,0x4a,0x2a,0x13,0x54,0x0c,0x48,0x96,0xae,
		0x43,0x3c,0x04,0x20,0xd3,0xe5,0x8e,0x46,0xb5,0x6c,0x05,0xad,0xe0,0xc7,0xbc,0x39,
		0x05,0x44,0x17,0xd7,0xad,0xb3,0x9a,0xcc,0x18,0xd9,0xc3,0xdc,0x8d,0x5a,0x1d,0x44,
		0xb5,0x32,0xd7,0x71,0x94,0xff,0x48,0x38,0x16,0x51,0x0e,0xfa,0xed,0x54,0x91,0x00,
		0xd3,0x45,0x6c,0xd9,0xdf,0xd1,0x70,0x6b,0x31,0x22,0xaa,0xfb,0x7c,0x0f,0x3f,0xa0,
		0xa0,0xa5,0x16,0xac,0x83,0x6d,0x12,0x1d,0x4a,0x40,0x4e,0xb6,0x9c,0xf4,0x67,0xaa,
		0xa9,0xb0,0xc8,0xb4,0x0a,0xd5,0x3b,0x5c,0x19,0xed,0x86,0x83,0x5a,0x75,0xbc,0xeb,
		0x17,0xc8,0x16,0xa0,0x60,0x2e,0xb6,0x25,0xc5,0x4d,0x59,0xba,0x62,0xcb,0x3d,0x91,
		0x7c,0x79,0x6a,0x4b,0x4a,0x54,0xbd,0xb7,0xa3,0x89,0x7f,0xbf,0x0e,0x77,0xe1,0x54,
		0x29,0x0d,0x45,0x6d,0xa8,0x15,0xa5,0x17,0x8c,0xcf,0x27,0x9e,0x47,0x4e,0x2a,0x91,
		0x7e,0x4e,0x14,0x59,0x8c,0x62,0x91,0xa3,0x40,0xa5,0x9e,0x67,0xbb,0x02,0x97,0xb4,
		0xe7,0x06,0x04,0xbc,0x16,0x24,0x3d,0x49,0xb1,0xf0,0xae,0xfc,0x1d,0x02,0x61,0x00,
		0xde,0x86,0x5d,0x49,0x88,0xeb,0x5c,0xd3,0xe5,0x11,0x48,0x0b,0x1e,0x52,0x95,0xa9,
		0x65,0x99,0x89,0xcf,0x51,0xb0,0x08,0xdd,0xb5,0x5b,0x64,0x1a,0x34,0xd2,0xee,0x4b,
		0x2d,0x8b,0xc1,0xd5,0xd6,0x1d,0x6c,0x0c,0x7e,0xa5,0x66,0x12,0xec,0xaf,0x5d,0xe9,
		0x33,0xd4,0xba,0x18,0x71,0x84,0x97,0xbe,0xc0,0x75,0x63,0x19,0xae,0xc6,0xc7,0x65,
		0xf3,0xf6,0xda,0x3f,0x91,0xfa,0x5e,0x87,0xf3,0xbc,0xd2,0x64,0x8d,0xcf,0xfb,0xdd,
		0x7f,0x9b,0x6c,0x81,0xba,0x9b,0x4e,0x94,0x5e,0x83,0xd1,0xcb,0xb9,0xf4,0x39,0x7f,
		0x02,0x61,0x00,0xd7,0x00,0x6d,0x8e,0x1b,0xa1,0x44,0xd9,0xff,0xe6,0x42,0x72,0x18,
		0x55,0x26,0x3e,0x87,0x40,0x71,0xb2,0x67,0x37,0x16,0xe9,0xbd,0x51,0x7f,0x0e,0x79,
		0x0e,0x75,0xa9,0x1f,0x0f,0x6b,0xa5,0x7c,0x5f,0xc8,0xdc,0x17,0xde,0x53,0x88,0x97,
		0x90,0x88,0xf2,0x4d,0x66,0x5e,0x0e,0x11,0x16,0x92,0x1e,0x61,0x56,0xe6,0xf0,0x74,
		0x81,0x58,0x95,0x05,0x29,0x71,0x9b,0xa0,0x69,0xed,0x14,0x23,0xf6,0x36,0x9b,0x8f,
		0x06,0x3a,0x76,0xab,0xeb,0xce,0xe8,0xdc,0x79,0xc1,0x29,0xb9,0xfc,0x49,0x7a,0x26,
		0x59,0xd6,0x4d,0x02,0x61,0x00,0xaf,0x3c,0xac,0xd6,0x2d,0xe6,0xfb,0x91,0x3a,0xc1,
		0x23,0x34,0xee,0x4a,0x26,0xe5,0xe1,0xc6,0xc9,0xc9,0xe4,0x10,0x76,0xca,0xf1,0xf8,
		0xe8,0x99,0xe2,0xa3,0x81,0x58,0xde,0xa3,0x42,0xa0,0x3d,0x1f,0xaa,0x69,0x24,0x8a,
		0xe8,0x19,0x5b,0x1e,0xb7,0x1b,0xe0,0xdf,0x53,0x35,0xd0,0x9f,0x94,0x48,0x79,0x93,
		0x77,0xd9,0x4f,0xd3,0xe6,0x4f,0x19,0x92,0x7a,0x48,0xb9,0x92,0xab,0x42,0xf0,0xe4,
		0xef,0xe2,0x93,0xf3,0x07,0xeb,0x64,0x84,0x67,0x2c,0xba,0x61,0x77,0xbe,0x4b,0xb8,
		0x0f,0x4d,0x1a,0x41,0x83,0xcd,0x02,0x60,0x56,0xec,0x55,0x5e,0x9e,0xcd,0x14,0x89,
		0x0e,0x6c,0x89,0x70,0x97,0x65,0xd5,0x90,0x72,0x1e,0x1b,0xd9,0x84,0xe1,0x40,0xe2,
		0x3f,0x28,0x33,0xb6,0x26,0x3b,0x32,0x56,0xad,0xb8,0x0e,0x4d,0x59,0x7b,0x60,0x39,
		0x9b,0x6c,0xc7,0x58,0xf1,0xed,0xfd,0x6f,0xf8,0xda,0xea,0x2b,0xc5,0xbc,0xda,0x56,
		0x6e,0x04,0x34,0x5a,0x02,0xc0,0x48,0x8f,0xf7,0x06,0x4a,0x68,0x20,0xf2,0xb2,0x66,
		0xf2,0x23,0x18,0xf0,0xcb,0x62,0x39,0x40,0xc1,0x41,0x14,0xe6,0x10,0x3d,0x29,0x5b,
		0x35,0x56,0x4a,0x5e,0x98,0x22,0xba,0x01,0x02,0x61,0x00,0xcc,0x80,0xb7,0xb9,0xb9,
		0x4a,0xaf,0x47,0x00,0x3e,0x21,0x0f,0xb8,0x4e,0x7c,0xb1,0xe4,0x25,0xd6,0x19,0x26,
		0x54,0xc6,0x8c,0x30,0x88,0x54,0x70,0xcf,0x1f,0x62,0x75,0xcb,0x18,0x58,0x6c,0x14,
		0xb0,0x9b,0x13,0x90,0xa2,0x1a,0x5a,0x79,0xa3,0x82,0xf0,0x9b,0xba,0xf0,0x90,0xaf,
		0xa1,0xe8,0xa8,0x70,0xef,0x60,0x6a,0x68,0xed,0x5a,0x21,0x77,0x69,0x7a,0xf2,0xee,
		0x3e,0xe5,0x90,0xd2,0x33,0x71,0x3b,0x82,0x88,0x75,0xdd,0x8e,0x6e,0xbc,0x17,0x83,
		0xef,0x37,0x82,0x4e,0x83,0x30,0xcb,0x8a,0xbc,0x6c,0x41),
	  chunk_from_chars(
		0x30,0x81,0xdf,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,
		0x05,0x00,0x03,0x81,0xcd,0x00,0x30,0x81,0xc9,0x02,0x81,0xc1,0x00,0xba,0xe3,0x37,
		0x93,0x7e,0x42,0x13,0x3c,0xba,0x41,0xc1,0x7b,0xf0,0xcc,0x7a,0x44,0xc6,0x54,0xc8,
		0x77,0x01,0x70,0x2f,0x6e,0x4a,0xcf,0x2d,0x07,0xab,0x01,0xc0,0x43,0xab,0x8d,0x33,
		0xb3,0xd4,0xeb,0xe3,0x90,0xf6,0x01,0x03,0x75,0x03,0x1d,0xe8,0x06,0x40,0x15,0xfa,
		0x96,0x0b,0xd5,0x26,0x64,0xea,0x55,0x82,0x16,0x7b,0xd5,0x1e,0xaa,0x08,0xc7,0x30,
		0x1a,0x59,0xf8,0xd9,0xe3,0x9e,0x89,0xd9,0x92,0x2c,0x32,0x79,0x0e,0xb3,0x25,0xbc,
		0x1d,0x7c,0x59,0xde,0x05,0x47,0x8f,0x61,0x77,0xf5,0x4f,0xed,0x82,0x2c,0xf8,0x2a,
		0x3e,0x02,0xf3,0xc0,0x15,0x51,0xde,0x05,0xc4,0xfc,0x80,0x91,0xae,0x06,0x1b,0xd7,
		0x39,0x8e,0x9a,0x6d,0xb3,0x2f,0xb0,0xd0,0xc8,0x96,0xa6,0x88,0xb3,0x17,0xca,0x58,
		0xbe,0x38,0x2c,0x64,0x35,0x5a,0x29,0xb7,0xf8,0x74,0x3d,0xbb,0xec,0x90,0x01,0x04,
		0x64,0x3d,0x38,0x0f,0x87,0xce,0xd7,0xfc,0xd2,0x96,0x93,0x31,0x85,0x0d,0x2d,0xa5,
		0x91,0xe2,0xfc,0x7b,0xea,0xb0,0x89,0x24,0xaa,0x00,0x29,0x8c,0x26,0x7c,0x94,0x54,
		0x74,0xe4,0x11,0xa8,0x04,0x6f,0x40,0xeb,0xaf,0xed,0xac,0x75,0x33,0x02,0x03,0x01,
		0x00,0x01),
	  chunk_from_chars(
		0x21,0x00,0x8c,0xe1,0x78,0x25,0x67,0x19,0xb7,0xd0,0xcb,0x13,0x01,0x7a,0xa3,0x71,
		0x67,0x46,0x96,0xf1),
	  chunk_from_chars(
		0x58,0x4d,0x4c,0x27,0x98,0x15,0x76,0xab,0x29,0xc9,0xf3,0xd1,0xa0,0xc4,0x50,0x0d,
		0x42,0x30,0x43,0x19),
	},
	{ chunk_from_chars( /* RSA-2048 */
		0x30,0x82,0x04,0xa2,0x02,0x01,0x00,0x02,0x82,0x01,0x01,0x00,0xba,0xbf,0x27,0x0b,
		0x22,0x59,0xd8,0x6f,0xff,0x26,0x5d,0x41,0x3d,0xb0,0x94,0x58,0x5d,0xc0,0x46,0xb6,
		0x77,0xa9,0x78,0x10,0x6d,0xe9,0xbf,0xca,0x6f,0x04,0xe1,0xda,0x85,0x12,0x1e,0xe0,
		0xa6,0xc7,0xa2,0x71,0x04,0x8b,0x6e,0x84,0xf9,0x86,0x2b,0xeb,0x72,0x01,0x72,0xc8,
		0x0a,0x83,0xa6,0xf7,0xc0,0xd6,0x76,0x1d,0x28,0x38,0xb5,0x7e,0x6c,0x8c,0x6a,0x13,
		0xf4,0xf1,0x7f,0xf2,0x79,0xae,0x73,0xba,0x1a,0x3f,0x30,0x65,0xb6,0x23,0xa7,0x94,
		0x34,0x29,0x87,0xce,0x06,0x99,0xee,0x85,0x10,0xce,0x08,0xe2,0x8d,0xd5,0x47,0xf3,
		0xc8,0xf0,0x18,0x41,0xc0,0x59,0x66,0x06,0xda,0xb6,0x18,0xd2,0xa3,0xa0,0xbd,0x3a,
		0x90,0x7f,0x37,0x39,0xdf,0x98,0x55,0xa2,0x19,0x5e,0x37,0xbc,0x86,0xf3,0x02,0xf8,
		0x68,0x49,0x53,0xf2,0x4b,0x3d,0x7a,0xe3,0x1d,0xa4,0x15,0x10,0xa6,0xce,0x8c,0xb8,
		0xfd,0x95,0x54,0xa2,0x50,0xa2,0xd9,0x35,0x12,0x56,0xae,0xbc,0x51,0x33,0x6d,0xb8,
		0x63,0x7c,0x26,0xab,0x19,0x01,0xa5,0xda,0xfa,0x4b,0xb6,0x57,0xd3,0x4b,0xdd,0xc0,
		0x62,0xc5,0x05,0xb7,0xc3,0x2e,0x1f,0x17,0xc8,0x09,0x87,0x12,0x37,0x21,0xd7,0x7a,
		0x53,0xb0,0x47,0x60,0xa2,0xb5,0x23,0x3b,0x99,0xdf,0xea,0x8b,0x94,0xea,0x9d,0x53,
		0x5d,0x02,0x52,0xf7,0x29,0xfb,0x63,0xb0,0xff,0x27,0x5e,0xde,0x54,0x7d,0x95,0xd6,
		0x4e,0x58,0x12,0x06,0x60,0x22,0x33,0xf2,0x19,0x67,0x65,0xdd,0xf3,0x42,0xb5,0x00,
		0x51,0x35,0xe5,0x62,0x4d,0x90,0x44,0xfb,0x7f,0x5b,0xb5,0xe5,0x02,0x03,0x01,0x00,
		0x01,0x02,0x82,0x01,0x00,0x1c,0xf5,0x66,0xf5,0xce,0x4c,0x1d,0xe8,0xd2,0x29,0x6e,
		0x15,0x1f,0x9e,0x9a,0x06,0x70,0xf5,0x4f,0xd1,0xdc,0x51,0x02,0x8e,0x13,0xa9,0x47,
		0x85,0x39,0xfd,0x89,0x13,0x74,0x86,0xb8,0x94,0x90,0x30,0x4d,0x73,0x96,0xa7,0x93,
		0x8a,0x19,0xd2,0x91,0x4d,0x77,0xb6,0x9b,0x48,0xc3,0x7e,0xa2,0x5d,0xf1,0x80,0xa0,
		0x3c,0xc9,0xbf,0xaf,0x7f,0x4d,0x10,0x62,0x23,0xb9,0x9c,0x58,0x81,0xae,0x96,0x5b,
		0x9a,0x4c,0x97,0x27,0x67,0x62,0x5c,0xf9,0x8f,0xdd,0x1d,0xe2,0x92,0x13,0x8a,0x7b,
		0xc7,0x15,0x31,0xca,0x05,0x6d,0xc6,0x98,0xdb,0x88,0x39,0x99,0x1d,0x5b,0x19,0x51,
		0xdd,0xb6,0xbd,0x3d,0xb0,0xae,0x50,0x8e,0xff,0x7d,0xa8,0x48,0x95,0x58,0x23,0xbc,
		0x85,0xc0,0x46,0xd0,0xc0,0x0e,0xda,0xdd,0xa4,0x8e,0x8d,0x31,0x8b,0x89,0x0f,0x8b,
		0x76,0x9a,0xb5,0x99,0x56,0x5e,0xd3,0x0c,0x88,0x0b,0x03,0xf1,0xc9,0xe3,0x05,0x05,
		0x08,0x75,0xce,0x35,0x52,0xa0,0xc0,0xf2,0xf4,0xb9,0x87,0x22,0x21,0x3f,0x61,0xd6,
		0x99,0xae,0x0e,0x76,0x5d,0x9c,0x16,0xa3,0xe9,0xde,0x2d,0x2a,0x46,0xf7,0x89,0xbf,
		0x0d,0xb1,0x60,0xad,0xbc,0x24,0xe2,0xe5,0xb1,0xc1,0x1c,0x00,0x40,0x1c,0xbd,0xfa,
		0x6e,0xc7,0x0d,0xc1,0xda,0x4d,0x54,0x45,0x96,0xac,0xf7,0xfe,0x1b,0xf2,0x47,0x1e,
		0xf7,0x8b,0xcf,0x27,0xcc,0xe7,0x08,0xd6,0x43,0x60,0xea,0xda,0x19,0xd7,0x98,0x17,
		0x7c,0xab,0x0c,0x90,0x60,0x75,0x9f,0x8b,0xaa,0x13,0x63,0x98,0x9e,0xc6,0x41,0x9f,
		0xd4,0x85,0xa3,0xb2,0xb9,0x02,0x81,0x81,0x00,0xe1,0x20,0xf6,0xac,0xa9,0x01,0xbd,
		0x31,0xe6,0xb2,0x4e,0xcf,0x66,0xc3,0x11,0x0e,0x5b,0xfe,0x58,0x6b,0xc6,0x2d,0x7a,
		0x05,0x30,0x9a,0x6f,0xcc,0xcc,0xdf,0xd2,0x2c,0xe1,0x47,0x39,0x9e,0xf3,0x0c,0x81,
		0xd9,0x76,0x00,0xe2,0xb1,0x08,0x91,0xfb,0x12,0x04,0xf6,0x1f,0xea,0xff,0x82,0xe5,
		0x64,0x64,0x6f,0x14,0xbe,0x33,0x5f,0x41,0x5f,0x73,0x1f,0xa2,0x32,0xec,0x75,0xb3,
		0x98,0x4b,0x88,0x4d,0x1e,0xec,0x78,0xda,0x4c,0x2d,0xf8,0xbb,0xcf,0x0e,0x8f,0x2f,
		0x23,0xae,0xcd,0xe0,0x4c,0x13,0x1c,0x1c,0x16,0x8e,0xb9,0x9f,0x02,0x12,0x12,0xa5,
		0xf4,0x21,0xfe,0x57,0x08,0x7a,0xe8,0xbe,0x15,0xe9,0xdd,0x2a,0xd1,0x7b,0x39,0xd6,
		0x4f,0x70,0x74,0x7d,0xfd,0x39,0x97,0x80,0x8d,0x02,0x81,0x81,0x00,0xd4,0x5a,0xce,
		0x05,0x93,0x51,0x15,0x44,0xdd,0x4d,0x79,0x92,0x04,0xe6,0x64,0x7e,0x6c,0xb5,0x61,
		0x6b,0xc3,0xb3,0xae,0x4f,0x0a,0x75,0xbf,0x6c,0xec,0x47,0xf2,0xbc,0xea,0x76,0xc4,
		0xc2,0xe7,0xd2,0x50,0xc4,0xe0,0xaf,0x56,0x05,0x72,0x3c,0x34,0x8c,0x5b,0xae,0xb8,
		0x0e,0xfb,0x83,0x27,0xcf,0x61,0x05,0x44,0x97,0x3f,0x66,0x6d,0x26,0x7d,0xed,0xcd,
		0x5a,0x87,0x04,0xbc,0xb3,0x70,0x75,0x15,0x51,0xe9,0x18,0x85,0xf7,0x2a,0x45,0xd5,
		0xc7,0x93,0x32,0x07,0x2e,0x26,0x34,0x2d,0x18,0x63,0x45,0x06,0x6f,0xa9,0x75,0x5d,
		0x20,0x6b,0x0b,0x13,0x45,0x81,0x7e,0x5c,0xc5,0x48,0x16,0x4b,0x82,0x7c,0xad,0xbe,
		0xfd,0xa5,0x0a,0xd6,0xc2,0x21,0xfc,0xa5,0x84,0xaf,0xf3,0x10,0xb9,0x02,0x81,0x80,
		0x29,0x20,0x20,0x6f,0xc2,0x1f,0xf3,0x33,0xde,0x74,0xcc,0x38,0xcf,0x08,0xeb,0x60,
		0xb8,0x25,0x6a,0x79,0xa5,0xa6,0x41,0x18,0x19,0x9c,0xdc,0xb7,0x88,0xe5,0x8a,0x3b,
		0x70,0x9b,0xd6,0x46,0xd7,0x17,0x7d,0xd0,0xff,0xe1,0x81,0x87,0xdd,0x8c,0xed,0x54,
		0x89,0x5b,0x7c,0xd1,0x2d,0x03,0xf8,0x6b,0xb2,0x7d,0x28,0x48,0xe6,0x91,0x8c,0x1b,
		0xa7,0xa8,0x2b,0xb5,0x29,0xc5,0x06,0x9d,0xd7,0x8e,0x7a,0xa8,0x1f,0x82,0xa4,0x3e,
		0x2e,0x57,0xb5,0xd7,0x49,0x4d,0x96,0xca,0xe9,0xef,0xe9,0xfd,0x7b,0xb0,0x32,0xe1,
		0x5c,0x09,0x44,0xa6,0xd8,0x2e,0x57,0xea,0x95,0x1b,0x25,0x43,0x03,0x50,0xe9,0x08,
		0x8f,0xc4,0x3b,0x42,0x31,0x44,0x8b,0x85,0xcf,0x81,0x38,0x52,0xbd,0xe6,0x93,0x31,
		0x02,0x81,0x80,0x18,0x3d,0x79,0x51,0x07,0x9c,0xf4,0xd9,0x94,0x8d,0x78,0x78,0x23,
		0x99,0x0d,0x15,0xa5,0x61,0x1b,0x0a,0xcb,0x1f,0x22,0xa1,0xa1,0x27,0x09,0xbf,0xec,
		0x44,0xd6,0x3f,0x9c,0x60,0x0c,0x5b,0xd7,0x4c,0x99,0xad,0xaf,0x9c,0x34,0x2c,0x90,
		0xfa,0xb0,0x60,0xe9,0x42,0x4b,0x7e,0x62,0x55,0x79,0x60,0xe1,0xc9,0x51,0x28,0x16,
		0xb3,0xa1,0x78,0x08,0x5d,0xf1,0xd8,0x08,0x9b,0x90,0xd2,0xc6,0xde,0x86,0x9d,0x80,
		0x07,0x2d,0x9b,0xa6,0x36,0xac,0x8d,0x88,0x8e,0xe8,0x64,0xeb,0x35,0x7f,0x84,0x4e,
		0x28,0x9d,0xf0,0x77,0x1e,0x8f,0x8f,0xd8,0xc8,0x3d,0xdd,0xec,0x47,0x39,0x5d,0xc7,
		0xb9,0xcb,0xca,0xcc,0x62,0xa4,0xef,0x9d,0x3c,0x5c,0x81,0x72,0x91,0xbd,0x6f,0x25,
		0x0a,0x90,0xf9,0x02,0x81,0x80,0x51,0x42,0x23,0x64,0x3d,0xbc,0xcb,0xcb,0x77,0xd4,
		0x5c,0x6b,0xf4,0x16,0x3a,0x6b,0x05,0x5f,0xd4,0xf8,0x59,0xe6,0x98,0x0c,0x43,0x7e,
		0x6b,0x17,0x0d,0x01,0x23,0x6e,0x4c,0xff,0x35,0xe4,0xc5,0xba,0xe8,0x9e,0x12,0x94,
		0x34,0x78,0xe4,0x3d,0x35,0xa1,0xd4,0xa9,0xa3,0x7e,0xe4,0x57,0xef,0xa4,0x9a,0x6a,
		0x32,0xb3,0x9f,0xf8,0x3a,0xcf,0xea,0xf4,0xc7,0x59,0x92,0xd4,0x2a,0x5b,0x26,0x83,
		0x78,0x30,0x5f,0xdf,0x46,0xa6,0xb0,0x28,0x37,0x2b,0x55,0x08,0x4c,0xb6,0x6b,0xb8,
		0xa9,0x11,0x7d,0x0b,0xab,0x97,0x4d,0x8c,0xc3,0xbf,0x3b,0xcd,0x3e,0xad,0x80,0xce,
		0xe8,0xc6,0x01,0x35,0xd2,0x3e,0x31,0xdc,0x96,0xd7,0xc3,0xab,0x65,0xd1,0xc4,0xa3,
		0x47,0x14,0xa9,0xba,0xd0,0x30),
	  chunk_from_chars(
		0x30,0x82,0x04,0xbc,0x02,0x01,0x00,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,
		0x0d,0x01,0x01,0x01,0x05,0x00,0x04,0x82,0x04,0xa6,0x30,0x82,0x04,0xa2,0x02,0x01,
		0x00,0x02,0x82,0x01,0x01,0x00,0xba,0xbf,0x27,0x0b,0x22,0x59,0xd8,0x6f,0xff,0x26,
		0x5d,0x41,0x3d,0xb0,0x94,0x58,0x5d,0xc0,0x46,0xb6,0x77,0xa9,0x78,0x10,0x6d,0xe9,
		0xbf,0xca,0x6f,0x04,0xe1,0xda,0x85,0x12,0x1e,0xe0,0xa6,0xc7,0xa2,0x71,0x04,0x8b,
		0x6e,0x84,0xf9,0x86,0x2b,0xeb,0x72,0x01,0x72,0xc8,0x0a,0x83,0xa6,0xf7,0xc0,0xd6,
		0x76,0x1d,0x28,0x38,0xb5,0x7e,0x6c,0x8c,0x6a,0x13,0xf4,0xf1,0x7f,0xf2,0x79,0xae,
		0x73,0xba,0x1a,0x3f,0x30,0x65,0xb6,0x23,0xa7,0x94,0x34,0x29,0x87,0xce,0x06,0x99,
		0xee,0x85,0x10,0xce,0x08,0xe2,0x8d,0xd5,0x47,0xf3,0xc8,0xf0,0x18,0x41,0xc0,0x59,
		0x66,0x06,0xda,0xb6,0x18,0xd2,0xa3,0xa0,0xbd,0x3a,0x90,0x7f,0x37,0x39,0xdf,0x98,
		0x55,0xa2,0x19,0x5e,0x37,0xbc,0x86,0xf3,0x02,0xf8,0x68,0x49,0x53,0xf2,0x4b,0x3d,
		0x7a,0xe3,0x1d,0xa4,0x15,0x10,0xa6,0xce,0x8c,0xb8,0xfd,0x95,0x54,0xa2,0x50,0xa2,
		0xd9,0x35,0x12,0x56,0xae,0xbc,0x51,0x33,0x6d,0xb8,0x63,0x7c,0x26,0xab,0x19,0x01,
		0xa5,0xda,0xfa,0x4b,0xb6,0x57,0xd3,0x4b,0xdd,0xc0,0x62,0xc5,0x05,0xb7,0xc3,0x2e,
		0x1f,0x17,0xc8,0x09,0x87,0x12,0x37,0x21,0xd7,0x7a,0x53,0xb0,0x47,0x60,0xa2,0xb5,
		0x23,0x3b,0x99,0xdf,0xea,0x8b,0x94,0xea,0x9d,0x53,0x5d,0x02,0x52,0xf7,0x29,0xfb,
		0x63,0xb0,0xff,0x27,0x5e,0xde,0x54,0x7d,0x95,0xd6,0x4e,0x58,0x12,0x06,0x60,0x22,
		0x33,0xf2,0x19,0x67,0x65,0xdd,0xf3,0x42,0xb5,0x00,0x51,0x35,0xe5,0x62,0x4d,0x90,
		0x44,0xfb,0x7f,0x5b,0xb5,0xe5,0x02,0x03,0x01,0x00,0x01,0x02,0x82,0x01,0x00,0x1c,
		0xf5,0x66,0xf5,0xce,0x4c,0x1d,0xe8,0xd2,0x29,0x6e,0x15,0x1f,0x9e,0x9a,0x06,0x70,
		0xf5,0x4f,0xd1,0xdc,0x51,0x02,0x8e,0x13,0xa9,0x47,0x85,0x39,0xfd,0x89,0x13,0x74,
		0x86,0xb8,0x94,0x90,0x30,0x4d,0x73,0x96,0xa7,0x93,0x8a,0x19,0xd2,0x91,0x4d,0x77,
		0xb6,0x9b,0x48,0xc3,0x7e,0xa2,0x5d,0xf1,0x80,0xa0,0x3c,0xc9,0xbf,0xaf,0x7f,0x4d,
		0x10,0x62,0x23,0xb9,0x9c,0x58,0x81,0xae,0x96,0x5b,0x9a,0x4c,0x97,0x27,0x67,0x62,
		0x5c,0xf9,0x8f,0xdd,0x1d,0xe2,0x92,0x13,0x8a,0x7b,0xc7,0x15,0x31,0xca,0x05,0x6d,
		0xc6,0x98,0xdb,0x88,0x39,0x99,0x1d,0x5b,0x19,0x51,0xdd,0xb6,0xbd,0x3d,0xb0,0xae,
		0x50,0x8e,0xff,0x7d,0xa8,0x48,0x95,0x58,0x23,0xbc,0x85,0xc0,0x46,0xd0,0xc0,0x0e,
		0xda,0xdd,0xa4,0x8e,0x8d,0x31,0x8b,0x89,0x0f,0x8b,0x76,0x9a,0xb5,0x99,0x56,0x5e,
		0xd3,0x0c,0x88,0x0b,0x03,0xf1,0xc9,0xe3,0x05,0x05,0x08,0x75,0xce,0x35,0x52,0xa0,
		0xc0,0xf2,0xf4,0xb9,0x87,0x22,0x21,0x3f,0x61,0xd6,0x99,0xae,0x0e,0x76,0x5d,0x9c,
		0x16,0xa3,0xe9,0xde,0x2d,0x2a,0x46,0xf7,0x89,0xbf,0x0d,0xb1,0x60,0xad,0xbc,0x24,
		0xe2,0xe5,0xb1,0xc1,0x1c,0x00,0x40,0x1c,0xbd,0xfa,0x6e,0xc7,0x0d,0xc1,0xda,0x4d,
		0x54,0x45,0x96,0xac,0xf7,0xfe,0x1b,0xf2,0x47,0x1e,0xf7,0x8b,0xcf,0x27,0xcc,0xe7,
		0x08,0xd6,0x43,0x60,0xea,0xda,0x19,0xd7,0x98,0x17,0x7c,0xab,0x0c,0x90,0x60,0x75,
		0x9f,0x8b,0xaa,0x13,0x63,0x98,0x9e,0xc6,0x41,0x9f,0xd4,0x85,0xa3,0xb2,0xb9,0x02,
		0x81,0x81,0x00,0xe1,0x20,0xf6,0xac,0xa9,0x01,0xbd,0x31,0xe6,0xb2,0x4e,0xcf,0x66,
		0xc3,0x11,0x0e,0x5b,0xfe,0x58,0x6b,0xc6,0x2d,0x7a,0x05,0x30,0x9a,0x6f,0xcc,0xcc,
		0xdf,0xd2,0x2c,0xe1,0x47,0x39,0x9e,0xf3,0x0c,0x81,0xd9,0x76,0x00,0xe2,0xb1,0x08,
		0x91,0xfb,0x12,0x04,0xf6,0x1f,0xea,0xff,0x82,0xe5,0x64,0x64,0x6f,0x14,0xbe,0x33,
		0x5f,0x41,0x5f,0x73,0x1f,0xa2,0x32,0xec,0x75,0xb3,0x98,0x4b,0x88,0x4d,0x1e,0xec,
		0x78,0xda,0x4c,0x2d,0xf8,0xbb,0xcf,0x0e,0x8f,0x2f,0x23,0xae,0xcd,0xe0,0x4c,0x13,
		0x1c,0x1c,0x16,0x8e,0xb9,0x9f,0x02,0x12,0x12,0xa5,0xf4,0x21,0xfe,0x57,0x08,0x7a,
		0xe8,0xbe,0x15,0xe9,0xdd,0x2a,0xd1,0x7b,0x39,0xd6,0x4f,0x70,0x74,0x7d,0xfd,0x39,
		0x97,0x80,0x8d,0x02,0x81,0x81,0x00,0xd4,0x5a,0xce,0x05,0x93,0x51,0x15,0x44,0xdd,
		0x4d,0x79,0x92,0x04,0xe6,0x64,0x7e,0x6c,0xb5,0x61,0x6b,0xc3,0xb3,0xae,0x4f,0x0a,
		0x75,0xbf,0x6c,0xec,0x47,0xf2,0xbc,0xea,0x76,0xc4,0xc2,0xe7,0xd2,0x50,0xc4,0xe0,
		0xaf,0x56,0x05,0x72,0x3c,0x34,0x8c,0x5b,0xae,0xb8,0x0e,0xfb,0x83,0x27,0xcf,0x61,
		0x05,0x44,0x97,0x3f,0x66,0x6d,0x26,0x7d,0xed,0xcd,0x5a,0x87,0x04,0xbc,0xb3,0x70,
		0x75,0x15,0x51,0xe9,0x18,0x85,0xf7,0x2a,0x45,0xd5,0xc7,0x93,0x32,0x07,0x2e,0x26,
		0x34,0x2d,0x18,0x63,0x45,0x06,0x6f,0xa9,0x75,0x5d,0x20,0x6b,0x0b,0x13,0x45,0x81,
		0x7e,0x5c,0xc5,0x48,0x16,0x4b,0x82,0x7c,0xad,0xbe,0xfd,0xa5,0x0a,0xd6,0xc2,0x21,
		0xfc,0xa5,0x84,0xaf,0xf3,0x10,0xb9,0x02,0x81,0x80,0x29,0x20,0x20,0x6f,0xc2,0x1f,
		0xf3,0x33,0xde,0x74,0xcc,0x38,0xcf,0x08,0xeb,0x60,0xb8,0x25,0x6a,0x79,0xa5,0xa6,
		0x41,0x18,0x19,0x9c,0xdc,0xb7,0x88,0xe5,0x8a,0x3b,0x70,0x9b,0xd6,0x46,0xd7,0x17,
		0x7d,0xd0,0xff,0xe1,0x81,0x87,0xdd,0x8c,0xed,0x54,0x89,0x5b,0x7c,0xd1,0x2d,0x03,
		0xf8,0x6b,0xb2,0x7d,0x28,0x48,0xe6,0x91,0x8c,0x1b,0xa7,0xa8,0x2b,0xb5,0x29,0xc5,
		0x06,0x9d,0xd7,0x8e,0x7a,0xa8,0x1f,0x82,0xa4,0x3e,0x2e,0x57,0xb5,0xd7,0x49,0x4d,
		0x96,0xca,0xe9,0xef,0xe9,0xfd,0x7b,0xb0,0x32,0xe1,0x5c,0x09,0x44,0xa6,0xd8,0x2e,
		0x57,0xea,0x95,0x1b,0x25,0x43,0x03,0x50,0xe9,0x08,0x8f,0xc4,0x3b,0x42,0x31,0x44,
		0x8b,0x85,0xcf,0x81,0x38,0x52,0xbd,0xe6,0x93,0x31,0x02,0x81,0x80,0x18,0x3d,0x79,
		0x51,0x07,0x9c,0xf4,0xd9,0x94,0x8d,0x78,0x78,0x23,0x99,0x0d,0x15,0xa5,0x61,0x1b,
		0x0a,0xcb,0x1f,0x22,0xa1,0xa1,0x27,0x09,0xbf,0xec,0x44,0xd6,0x3f,0x9c,0x60,0x0c,
		0x5b,0xd7,0x4c,0x99,0xad,0xaf,0x9c,0x34,0x2c,0x90,0xfa,0xb0,0x60,0xe9,0x42,0x4b,
		0x7e,0x62,0x55,0x79,0x60,0xe1,0xc9,0x51,0x28,0x16,0xb3,0xa1,0x78,0x08,0x5d,0xf1,
		0xd8,0x08,0x9b,0x90,0xd2,0xc6,0xde,0x86,0x9d,0x80,0x07,0x2d,0x9b,0xa6,0x36,0xac,
		0x8d,0x88,0x8e,0xe8,0x64,0xeb,0x35,0x7f,0x84,0x4e,0x28,0x9d,0xf0,0x77,0x1e,0x8f,
		0x8f,0xd8,0xc8,0x3d,0xdd,0xec,0x47,0x39,0x5d,0xc7,0xb9,0xcb,0xca,0xcc,0x62,0xa4,
		0xef,0x9d,0x3c,0x5c,0x81,0x72,0x91,0xbd,0x6f,0x25,0x0a,0x90,0xf9,0x02,0x81,0x80,
		0x51,0x42,0x23,0x64,0x3d,0xbc,0xcb,0xcb,0x77,0xd4,0x5c,0x6b,0xf4,0x16,0x3a,0x6b,
		0x05,0x5f,0xd4,0xf8,0x59,0xe6,0x98,0x0c,0x43,0x7e,0x6b,0x17,0x0d,0x01,0x23,0x6e,
		0x4c,0xff,0x35,0xe4,0xc5,0xba,0xe8,0x9e,0x12,0x94,0x34,0x78,0xe4,0x3d,0x35,0xa1,
		0xd4,0xa9,0xa3,0x7e,0xe4,0x57,0xef,0xa4,0x9a,0x6a,0x32,0xb3,0x9f,0xf8,0x3a,0xcf,
		0xea,0xf4,0xc7,0x59,0x92,0xd4,0x2a,0x5b,0x26,0x83,0x78,0x30,0x5f,0xdf,0x46,0xa6,
		0xb0,0x28,0x37,0x2b,0x55,0x08,0x4c,0xb6,0x6b,0xb8,0xa9,0x11,0x7d,0x0b,0xab,0x97,
		0x4d,0x8c,0xc3,0xbf,0x3b,0xcd,0x3e,0xad,0x80,0xce,0xe8,0xc6,0x01,0x35,0xd2,0x3e,
		0x31,0xdc,0x96,0xd7,0xc3,0xab,0x65,0xd1,0xc4,0xa3,0x47,0x14,0xa9,0xba,0xd0,0x30),
	  chunk_from_chars(
		0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
		0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,
		0x00,0xba,0xbf,0x27,0x0b,0x22,0x59,0xd8,0x6f,0xff,0x26,0x5d,0x41,0x3d,0xb0,0x94,
		0x58,0x5d,0xc0,0x46,0xb6,0x77,0xa9,0x78,0x10,0x6d,0xe9,0xbf,0xca,0x6f,0x04,0xe1,
		0xda,0x85,0x12,0x1e,0xe0,0xa6,0xc7,0xa2,0x71,0x04,0x8b,0x6e,0x84,0xf9,0x86,0x2b,
		0xeb,0x72,0x01,0x72,0xc8,0x0a,0x83,0xa6,0xf7,0xc0,0xd6,0x76,0x1d,0x28,0x38,0xb5,
		0x7e,0x6c,0x8c,0x6a,0x13,0xf4,0xf1,0x7f,0xf2,0x79,0xae,0x73,0xba,0x1a,0x3f,0x30,
		0x65,0xb6,0x23,0xa7,0x94,0x34,0x29,0x87,0xce,0x06,0x99,0xee,0x85,0x10,0xce,0x08,
		0xe2,0x8d,0xd5,0x47,0xf3,0xc8,0xf0,0x18,0x41,0xc0,0x59,0x66,0x06,0xda,0xb6,0x18,
		0xd2,0xa3,0xa0,0xbd,0x3a,0x90,0x7f,0x37,0x39,0xdf,0x98,0x55,0xa2,0x19,0x5e,0x37,
		0xbc,0x86,0xf3,0x02,0xf8,0x68,0x49,0x53,0xf2,0x4b,0x3d,0x7a,0xe3,0x1d,0xa4,0x15,
		0x10,0xa6,0xce,0x8c,0xb8,0xfd,0x95,0x54,0xa2,0x50,0xa2,0xd9,0x35,0x12,0x56,0xae,
		0xbc,0x51,0x33,0x6d,0xb8,0x63,0x7c,0x26,0xab,0x19,0x01,0xa5,0xda,0xfa,0x4b,0xb6,
		0x57,0xd3,0x4b,0xdd,0xc0,0x62,0xc5,0x05,0xb7,0xc3,0x2e,0x1f,0x17,0xc8,0x09,0x87,
		0x12,0x37,0x21,0xd7,0x7a,0x53,0xb0,0x47,0x60,0xa2,0xb5,0x23,0x3b,0x99,0xdf,0xea,
		0x8b,0x94,0xea,0x9d,0x53,0x5d,0x02,0x52,0xf7,0x29,0xfb,0x63,0xb0,0xff,0x27,0x5e,
		0xde,0x54,0x7d,0x95,0xd6,0x4e,0x58,0x12,0x06,0x60,0x22,0x33,0xf2,0x19,0x67,0x65,
		0xdd,0xf3,0x42,0xb5,0x00,0x51,0x35,0xe5,0x62,0x4d,0x90,0x44,0xfb,0x7f,0x5b,0xb5,
		0xe5,0x02,0x03,0x01,0x00,0x01),
	  chunk_from_chars(
		0x4f,0xe8,0x82,0xee,0xaa,0x2c,0x7b,0x3f,0x3a,0xf1,0xb4,0xe1,0xe3,0x85,0xd4,0xb1,
		0xb4,0x34,0x5c,0x2d),
	  chunk_from_chars(
		0x29,0xe3,0x4b,0x6c,0x41,0x08,0xf2,0x6d,0xe5,0x1a,0x99,0x02,0xdf,0x9c,0x02,0xe3,
		0x67,0xdf,0xfd,0xb3),
	},
};

START_TEST(test_load)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t encoding, fp;

	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_RSA,
								 BUILD_BLOB_ASN1_DER, keys[_i].key, BUILD_END);
	ck_assert(privkey != NULL);
	ck_assert(privkey->get_encoding(privkey, PRIVKEY_ASN1_DER, &encoding));
	if (encoding.len == keys[_i].pkcs8.len)
	{
		ck_assert_chunk_eq(keys[_i].pkcs8, encoding);
	}
	else
	{
		ck_assert_chunk_eq(keys[_i].key, encoding);
	}
	chunk_clear(&encoding);

	ck_assert(privkey->get_fingerprint(privkey, KEYID_PUBKEY_SHA1, &fp));
	ck_assert_chunk_eq(keys[_i].fp_pk, fp);
	ck_assert(privkey->get_fingerprint(privkey, KEYID_PUBKEY_INFO_SHA1, &fp));
	ck_assert_chunk_eq(keys[_i].fp_spki, fp);

	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);
	ck_assert(pubkey->get_encoding(pubkey, PUBKEY_SPKI_ASN1_DER, &encoding));
	ck_assert_chunk_eq(keys[_i].pub, encoding);
	chunk_free(&encoding);

	ck_assert(pubkey->get_fingerprint(pubkey, KEYID_PUBKEY_SHA1, &fp));
	ck_assert_chunk_eq(keys[_i].fp_pk, fp);
	ck_assert(pubkey->get_fingerprint(pubkey, KEYID_PUBKEY_INFO_SHA1, &fp));
	ck_assert_chunk_eq(keys[_i].fp_spki, fp);

	test_good_sig(privkey, pubkey);

	test_bad_sigs(pubkey);

	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}
END_TEST

/**
 * Keys for the FIPS 186-4 SigGen PKCS#1 Ver1.5 test vectors
 */
static private_key_t *privkey_2048, *privkey_3072;
static public_key_t *pubkey_2048, *pubkey_3072;

static void create_key(chunk_t n, chunk_t e, chunk_t d, private_key_t **privkey,
					   public_key_t **pubkey)
{
	*privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_RSA,
								  BUILD_RSA_MODULUS, n, BUILD_RSA_PUB_EXP, e,
								  BUILD_RSA_PRIV_EXP, d, BUILD_END);
	ck_assert(*privkey != NULL);
	*pubkey = (*privkey)->get_public_key(*privkey);
	ck_assert(*pubkey != NULL);
}

START_SETUP(setup_keys)
{
	create_key(chunk_from_chars(
			0xce,0xa8,0x04,0x75,0x32,0x4c,0x1d,0xc8,0x34,0x78,0x27,0x81,0x8d,0xa5,0x8b,0xac,
			0x06,0x9d,0x34,0x19,0xc6,0x14,0xa6,0xea,0x1a,0xc6,0xa3,0xb5,0x10,0xdc,0xd7,0x2c,
			0xc5,0x16,0x95,0x49,0x05,0xe9,0xfe,0xf9,0x08,0xd4,0x5e,0x13,0x00,0x6a,0xdf,0x27,
			0xd4,0x67,0xa7,0xd8,0x3c,0x11,0x1d,0x1a,0x5d,0xf1,0x5e,0xf2,0x93,0x77,0x1a,0xef,
			0xb9,0x20,0x03,0x2a,0x5b,0xb9,0x89,0xf8,0xe4,0xf5,0xe1,0xb0,0x50,0x93,0xd3,0xf1,
			0x30,0xf9,0x84,0xc0,0x7a,0x77,0x2a,0x36,0x83,0xf4,0xdc,0x6f,0xb2,0x8a,0x96,0x81,
			0x5b,0x32,0x12,0x3c,0xcd,0xd1,0x39,0x54,0xf1,0x9d,0x5b,0x8b,0x24,0xa1,0x03,0xe7,
			0x71,0xa3,0x4c,0x32,0x87,0x55,0xc6,0x5e,0xd6,0x4e,0x19,0x24,0xff,0xd0,0x4d,0x30,
			0xb2,0x14,0x2c,0xc2,0x62,0xf6,0xe0,0x04,0x8f,0xef,0x6d,0xbc,0x65,0x2f,0x21,0x47,
			0x9e,0xa1,0xc4,0xb1,0xd6,0x6d,0x28,0xf4,0xd4,0x6e,0xf7,0x18,0x5e,0x39,0x0c,0xbf,
			0xa2,0xe0,0x23,0x80,0x58,0x2f,0x31,0x88,0xbb,0x94,0xeb,0xbf,0x05,0xd3,0x14,0x87,
			0xa0,0x9a,0xff,0x01,0xfc,0xbb,0x4c,0xd4,0xbf,0xd1,0xf0,0xa8,0x33,0xb3,0x8c,0x11,
			0x81,0x3c,0x84,0x36,0x0b,0xb5,0x3c,0x7d,0x44,0x81,0x03,0x1c,0x40,0xba,0xd8,0x71,
			0x3b,0xb6,0xb8,0x35,0xcb,0x08,0x09,0x8e,0xd1,0x5b,0xa3,0x1e,0xe4,0xba,0x72,0x8a,
			0x8c,0x8e,0x10,0xf7,0x29,0x4e,0x1b,0x41,0x63,0xb7,0xae,0xe5,0x72,0x77,0xbf,0xd8,
			0x81,0xa6,0xf9,0xd4,0x3e,0x02,0xc6,0x92,0x5a,0xa3,0xa0,0x43,0xfb,0x7f,0xb7,0x8d),
		chunk_from_chars(
			0x26,0x04,0x45),
		chunk_from_chars(
			0x09,0x97,0x63,0x4c,0x47,0x7c,0x1a,0x03,0x9d,0x44,0xc8,0x10,0xb2,0xaa,0xa3,0xc7,
			0x86,0x2b,0x0b,0x88,0xd3,0x70,0x82,0x72,0xe1,0xe1,0x5f,0x66,0xfc,0x93,0x89,0x70,
			0x9f,0x8a,0x11,0xf3,0xea,0x6a,0x5a,0xf7,0xef,0xfa,0x2d,0x01,0xc1,0x89,0xc5,0x0f,
			0x0d,0x5b,0xcb,0xe3,0xfa,0x27,0x2e,0x56,0xcf,0xc4,0xa4,0xe1,0xd3,0x88,0xa9,0xdc,
			0xd6,0x5d,0xf8,0x62,0x89,0x02,0x55,0x6c,0x8b,0x6b,0xb6,0xa6,0x41,0x70,0x9b,0x5a,
			0x35,0xdd,0x26,0x22,0xc7,0x3d,0x46,0x40,0xbf,0xa1,0x35,0x9d,0x0e,0x76,0xe1,0xf2,
			0x19,0xf8,0xe3,0x3e,0xb9,0xbd,0x0b,0x59,0xec,0x19,0x8e,0xb2,0xfc,0xca,0xae,0x03,
			0x46,0xbd,0x8b,0x40,0x1e,0x12,0xe3,0xc6,0x7c,0xb6,0x29,0x56,0x9c,0x18,0x5a,0x2e,
			0x0f,0x35,0xa2,0xf7,0x41,0x64,0x4c,0x1c,0xca,0x5e,0xbb,0x13,0x9d,0x77,0xa8,0x9a,
			0x29,0x53,0xfc,0x5e,0x30,0x04,0x8c,0x0e,0x61,0x9f,0x07,0xc8,0xd2,0x1d,0x1e,0x56,
			0xb8,0xaf,0x07,0x19,0x3d,0x0f,0xdf,0x3f,0x49,0xcd,0x49,0xf2,0xef,0x31,0x38,0xb5,
			0x13,0x88,0x62,0xf1,0x47,0x0b,0xd2,0xd1,0x6e,0x34,0xa2,0xb9,0xe7,0x77,0x7a,0x6c,
			0x8c,0x8d,0x4c,0xb9,0x4b,0x4e,0x8b,0x5d,0x61,0x6c,0xd5,0x39,0x37,0x53,0xe7,0xb0,
			0xf3,0x1c,0xc7,0xda,0x55,0x9b,0xa8,0xe9,0x8d,0x88,0x89,0x14,0xe3,0x34,0x77,0x3b,
			0xaf,0x49,0x8a,0xd8,0x8d,0x96,0x31,0xeb,0x5f,0xe3,0x2e,0x53,0xa4,0x14,0x5b,0xf0,
			0xba,0x54,0x8b,0xf2,0xb0,0xa5,0x0c,0x63,0xf6,0x7b,0x14,0xe3,0x98,0xa3,0x4b,0x0d),
		&privkey_2048, &pubkey_2048);

	create_key(chunk_from_chars(
			0xdc,0xa9,0x83,0x04,0xb7,0x29,0xe8,0x19,0xb3,0x40,0xe2,0x6c,0xec,0xb7,0x30,0xae,
			0xcb,0xd8,0x93,0x0e,0x33,0x4c,0x73,0x14,0x93,0xb1,0x80,0xde,0x97,0x0e,0x6d,0x3b,
			0xc5,0x79,0xf8,0x6c,0x8d,0x5d,0x03,0x2f,0x8c,0xd3,0x3c,0x43,0x97,0xee,0x7f,0xfd,
			0x01,0x9d,0x51,0xb0,0xa7,0xdb,0xe4,0xf5,0x25,0x05,0xa1,0xa3,0x4a,0xe3,0x5d,0x23,
			0xcf,0xaa,0xf5,0x94,0x41,0x9d,0x50,0x9f,0x46,0x9b,0x13,0x69,0x58,0x9f,0x9c,0x86,
			0x16,0xa7,0xd6,0x98,0x51,0x3b,0xc1,0xd4,0x23,0xd7,0x00,0x70,0xd3,0xd7,0x2b,0x99,
			0x6c,0x23,0xab,0xe6,0x8b,0x22,0xcc,0xc3,0x9a,0xab,0xd1,0x65,0x07,0x12,0x40,0x42,
			0xc8,0x8d,0x4d,0xa6,0xa7,0x45,0x12,0x88,0xec,0x87,0xc9,0x24,0x4b,0xe2,0x26,0xaa,
			0xc0,0x2d,0x18,0x17,0x68,0x2f,0x80,0xcc,0x34,0xc6,0xea,0xf3,0x7e,0xc8,0x4d,0x24,
			0x7a,0xae,0xde,0xbb,0x56,0xc3,0xbb,0xca,0xff,0xb5,0xcf,0x42,0xf6,0x1f,0xe1,0xb7,
			0xf3,0xfc,0x89,0x74,0x8e,0x21,0x39,0x73,0xbf,0x5f,0x67,0x9d,0x8b,0x8b,0x42,0xa4,
			0x7a,0xc4,0xaf,0xd9,0xe5,0x1e,0x1d,0x12,0x14,0xdf,0xe1,0xa7,0xe1,0x16,0x90,0x80,
			0xbd,0x9a,0xd9,0x17,0x58,0xf6,0xc0,0xf9,0xb2,0x2a,0xe4,0x0a,0xf6,0xb4,0x14,0x03,
			0xd8,0xf2,0xd9,0x6d,0xb5,0xa0,0x88,0xda,0xa5,0xef,0x86,0x83,0xf8,0x6f,0x50,0x1f,
			0x7a,0xd3,0xf3,0x58,0xb6,0x33,0x7d,0xa5,0x5c,0x6c,0xfc,0x00,0x31,0x97,0x42,0x0c,
			0x1c,0x75,0xab,0xdb,0x7b,0xe1,0x40,0x3e,0xa4,0xf3,0xe6,0x42,0x59,0xf5,0xc6,0xda,
			0x33,0x25,0xbb,0x87,0xd6,0x05,0xb6,0xe1,0x4b,0x53,0x50,0xe6,0xe1,0x45,0x5c,0x9d,
			0x49,0x7d,0x81,0x04,0x66,0x08,0xe3,0x87,0x95,0xdc,0x85,0xab,0xa4,0x06,0xc9,0xde,
			0x1f,0x4f,0x99,0x90,0xd5,0x15,0x3b,0x98,0xbb,0xab,0xbd,0xcb,0xd6,0xbb,0x18,0x85,
			0x43,0x12,0xb2,0xda,0x48,0xb4,0x11,0xe8,0x38,0xf2,0x6a,0xe3,0x10,0x9f,0x10,0x4d,
			0xfd,0x16,0x19,0xf9,0x91,0x82,0x4e,0xc8,0x19,0x86,0x1e,0x51,0x99,0xf2,0x6b,0xb9,
			0xb3,0xb2,0x99,0xbf,0xa9,0xec,0x2f,0xd6,0x91,0x27,0x1b,0x58,0xa8,0xad,0xec,0xbf,
			0x0f,0xf6,0x27,0xb5,0x43,0x36,0xf3,0xdf,0x70,0x03,0xd7,0x0e,0x37,0xd1,0x1d,0xdb,
			0xd9,0x30,0xd9,0xab,0xa7,0xe8,0x8e,0xd4,0x01,0xac,0xb4,0x40,0x92,0xfd,0x53,0xd5),
		chunk_from_chars(
			0xea,0xf0,0x5d),
		chunk_from_chars(
			0x2d,0x6d,0xb9,0x1e,0xb3,0x2e,0x36,0xe5,0xd5,0x12,0x7d,0xeb,0x03,0x4d,0x14,0x07,
			0x2f,0xe6,0x0c,0x1c,0xd1,0x3c,0x8c,0x3d,0xd9,0xad,0xbc,0x87,0x14,0x0b,0x5e,0x71,
			0x36,0xf4,0xf8,0x9e,0x61,0xbb,0xee,0x78,0x26,0xf4,0x5a,0xc1,0xd9,0x91,0x94,0xfb,
			0xaa,0x8c,0x5a,0x0b,0xb9,0x4d,0xb3,0x1d,0x93,0x72,0x3b,0x51,0x41,0x9d,0x9c,0x6f,
			0x6e,0xeb,0x5f,0x36,0x10,0xb6,0x7f,0x4b,0x4e,0x2a,0xde,0x05,0xcc,0x6b,0x89,0x90,
			0xe8,0x83,0x2c,0xf4,0xcd,0x40,0xf2,0xdf,0x03,0x88,0xc9,0xa5,0x20,0x72,0xe2,0x7e,
			0xfe,0xba,0xe2,0x0b,0x4a,0xd5,0x95,0x1f,0x4d,0x20,0xdd,0x18,0x94,0x3e,0x58,0xb7,
			0x86,0xd8,0x79,0x76,0x52,0xb2,0xbb,0x75,0x9c,0x31,0x9d,0x2b,0x00,0x46,0xdb,0xf6,
			0x9c,0x53,0xc0,0x75,0xd0,0x0c,0x28,0x7b,0x87,0x60,0x42,0xfa,0xfa,0x23,0xfe,0x4d,
			0xd7,0x05,0xe4,0xe4,0x23,0x27,0x7c,0x90,0x00,0x31,0x1e,0x94,0xea,0x3f,0x74,0x56,
			0xe3,0x2f,0xd1,0x2a,0xfe,0x4a,0x2b,0xde,0x35,0x8a,0x65,0x82,0x4f,0x10,0x55,0x06,
			0x48,0x23,0xc8,0x93,0xfc,0x93,0xbe,0x3b,0x8c,0x65,0x8b,0xb4,0x41,0xd7,0xf0,0xb0,
			0x0a,0xc2,0x46,0xbf,0x04,0x3a,0x9c,0x00,0x53,0xd3,0x19,0xf0,0x03,0xef,0x5a,0x55,
			0x33,0xf7,0x4d,0x63,0x0d,0x8c,0xe9,0x3b,0xab,0x41,0x6a,0x82,0x95,0x1e,0x05,0xb8,
			0x2c,0x60,0x36,0x59,0x3e,0xca,0x89,0xf0,0xeb,0xac,0xd7,0xd5,0x1e,0xd9,0x61,0x0a,
			0xf4,0x35,0x37,0xfc,0xd2,0x66,0xe5,0xe4,0x7c,0x0d,0x25,0xfe,0xda,0xd6,0xd0,0x47,
			0xa1,0xa1,0xee,0x3e,0xb4,0x44,0x36,0x7e,0x3e,0xff,0x7c,0x75,0x20,0xca,0x4f,0x77,
			0x9f,0x20,0x27,0xfe,0x45,0x03,0x62,0x04,0x16,0x84,0x54,0xdf,0x49,0x18,0xb5,0x47,
			0xa4,0xd1,0x9e,0x93,0x8f,0x3c,0x6d,0xb6,0xca,0x27,0x02,0xad,0x9b,0xbd,0xa1,0x26,
			0x1c,0x64,0xd0,0x0b,0x57,0x82,0x85,0xbd,0xcf,0xc9,0x85,0x1f,0x96,0xa4,0xf2,0xcd,
			0x14,0xd6,0x6b,0x9c,0x1f,0x65,0x74,0x2a,0x13,0x44,0x94,0x8c,0x9f,0x1d,0xa8,0xd3,
			0x38,0xed,0x4e,0x3d,0xeb,0x1e,0xba,0xdf,0x11,0xf8,0xc2,0x81,0x94,0x4e,0x88,0x49,
			0x82,0x34,0x96,0xf8,0x61,0x11,0xf3,0x78,0xbd,0xd0,0x84,0xc9,0x9f,0x65,0xfb,0x9b,
			0x4e,0xe6,0x27,0x1b,0x1d,0x1b,0xe4,0x24,0xc2,0x94,0xd1,0x85,0xd9,0xfd,0x9c,0xdf),
		&privkey_3072, &pubkey_3072);
}
END_SETUP

START_TEARDOWN(teardown_keys)
{
	pubkey_2048->destroy(pubkey_2048);
	privkey_2048->destroy(privkey_2048);
	pubkey_3072->destroy(pubkey_3072);
	privkey_3072->destroy(privkey_3072);
}
END_TEARDOWN

/**
 * Test vectors from FIPS 186-4
 */
static struct {
	signature_scheme_t scheme;
	chunk_t m;
	chunk_t s;
} pkcs15_2048[] = {
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x5a,0xf2,0x83,0xb1,0xb7,0x6a,0xb2,0xa6,0x95,0xd7,0x94,0xc2,0x3b,0x35,0xca,0x73,
			0x71,0xfc,0x77,0x9e,0x92,0xeb,0xf5,0x89,0xe3,0x04,0xc7,0xf9,0x23,0xd8,0xcf,0x97,
			0x63,0x04,0xc1,0x98,0x18,0xfc,0xd8,0x9d,0x6f,0x07,0xc8,0xd8,0xe0,0x8b,0xf3,0x71,
			0x06,0x8b,0xdf,0x28,0xae,0x6e,0xe8,0x3b,0x2e,0x02,0x32,0x8a,0xf8,0xc0,0xe2,0xf9,
			0x6e,0x52,0x8e,0x16,0xf8,0x52,0xf1,0xfc,0x54,0x55,0xe4,0x77,0x2e,0x28,0x8a,0x68,
			0xf1,0x59,0xca,0x6b,0xdc,0xf9,0x02,0xb8,0x58,0xa1,0xf9,0x47,0x89,0xb3,0x16,0x38,
			0x23,0xe2,0xd0,0x71,0x7f,0xf5,0x66,0x89,0xee,0xc7,0xd0,0xe5,0x4d,0x93,0xf5,0x20,
			0xd9,0x6e,0x1e,0xb0,0x45,0x15,0xab,0xc7,0x0a,0xe9,0x05,0x78,0xff,0x38,0xd3,0x1b),
		.s = chunk_from_chars(
			0x6b,0x8b,0xe9,0x7d,0x9e,0x51,0x8a,0x2e,0xde,0x74,0x6f,0xf4,0xa7,0xd9,0x1a,0x84,
			0xa1,0xfc,0x66,0x5b,0x52,0xf1,0x54,0xa9,0x27,0x65,0x0d,0xb6,0xe7,0x34,0x8c,0x69,
			0xf8,0xc8,0x88,0x1f,0x7b,0xcf,0x9b,0x1a,0x6d,0x33,0x66,0xee,0xd3,0x0c,0x3a,0xed,
			0x4e,0x93,0xc2,0x03,0xc4,0x3f,0x55,0x28,0xa4,0x5d,0xe7,0x91,0x89,0x57,0x47,0xad,
			0xe9,0xc5,0xfa,0x5e,0xee,0x81,0x42,0x7e,0xde,0xe0,0x20,0x82,0x14,0x7a,0xa3,0x11,
			0x71,0x2a,0x6a,0xd5,0xfb,0x17,0x32,0xe9,0x3b,0x3d,0x6c,0xd2,0x3f,0xfd,0x46,0xa0,
			0xb3,0xca,0xf6,0x2a,0x8b,0x69,0x95,0x7c,0xc6,0x8a,0xe3,0x9f,0x99,0x93,0xc1,0xa7,
			0x79,0x59,0x9c,0xdd,0xa9,0x49,0xbd,0xaa,0xba,0xbb,0x77,0xf2,0x48,0xfc,0xfe,0xaa,
			0x44,0x05,0x9b,0xe5,0x45,0x9f,0xb9,0xb8,0x99,0x27,0x8e,0x92,0x95,0x28,0xee,0x13,
			0x0f,0xac,0xd5,0x33,0x72,0xec,0xbc,0x42,0xf3,0xe8,0xde,0x29,0x98,0x42,0x58,0x60,
			0x40,0x64,0x40,0xf2,0x48,0xd8,0x17,0x43,0x2d,0xe6,0x87,0x11,0x2e,0x50,0x4d,0x73,
			0x40,0x28,0xe6,0xc5,0x62,0x0f,0xa2,0x82,0xca,0x07,0x64,0x70,0x06,0xcf,0x0a,0x2f,
			0xf8,0x3e,0x19,0xa9,0x16,0x55,0x4c,0xc6,0x18,0x10,0xc2,0xe8,0x55,0x30,0x5d,0xb4,
			0xe5,0xcf,0x89,0x3a,0x6a,0x96,0x76,0x73,0x65,0x79,0x45,0x56,0xff,0x03,0x33,0x59,
			0x08,0x4d,0x7e,0x38,0xa8,0x45,0x6e,0x68,0xe2,0x11,0x55,0xb7,0x61,0x51,0x31,0x4a,
			0x29,0x87,0x5f,0xee,0xe0,0x95,0x57,0x16,0x1c,0xbc,0x65,0x45,0x41,0xe8,0x9e,0x42),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xc4,0x30,0x11,0xf3,0xee,0x88,0xc9,0xc9,0xad,0xca,0xc8,0xbf,0x37,0x22,0x1a,0xfa,
			0x31,0x76,0x9d,0x34,0x7d,0xec,0x70,0x5e,0x53,0xac,0xa9,0x89,0x93,0xe7,0x46,0x06,
			0x59,0x18,0x67,0xcc,0xd2,0x89,0xba,0x1b,0x4f,0x19,0x36,0x5f,0x98,0x3e,0x0c,0x57,
			0x83,0x46,0xda,0x76,0xc5,0xe2,0x22,0x8a,0x07,0xe4,0xfc,0x9b,0x3d,0x48,0x07,0x16,
			0x33,0x71,0xa5,0x2b,0x68,0xb6,0x68,0x73,0x20,0x1d,0xc7,0xd6,0xb5,0x66,0x16,0xac,
			0x2e,0x4c,0xb5,0x22,0x12,0x07,0x87,0xdf,0x7f,0x15,0xa5,0xe8,0x76,0x3a,0x54,0xc1,
			0x79,0xc6,0x35,0xd6,0x58,0x16,0xbc,0x19,0x48,0x5d,0xe3,0xeb,0x35,0xa5,0x20,0x40,
			0x59,0x10,0x94,0xfe,0x0e,0x64,0x85,0xa7,0xe0,0xc6,0x0e,0x38,0xe7,0xc6,0x15,0x51),
		.s = chunk_from_chars(
			0xaa,0x3a,0x4e,0x12,0xeb,0x87,0x59,0x6c,0x71,0x1c,0x9a,0x22,0xbc,0xab,0xcb,0x9d,
			0xad,0xff,0xca,0xbc,0xec,0xbd,0x16,0x22,0x88,0x89,0xe9,0xbb,0x45,0x7d,0x5d,0x22,
			0x57,0x1a,0x72,0xf0,0x34,0xbe,0x47,0x83,0x38,0x4f,0x43,0xce,0x6f,0xff,0xc6,0x05,
			0x34,0xb8,0x33,0x1c,0xdd,0x5d,0x7c,0x77,0xf4,0x91,0x80,0xbf,0xd1,0x94,0xb5,0xfd,
			0x43,0xa5,0x08,0xc6,0x6d,0x78,0x6c,0x55,0x88,0x76,0x73,0x58,0x94,0xe6,0xa9,0x30,
			0x09,0x52,0xde,0x79,0x2f,0x74,0x70,0x45,0xe7,0x4d,0x87,0xfd,0x50,0x98,0x02,0x30,
			0x70,0x7a,0x34,0xa4,0xdf,0x01,0x3c,0xe0,0x50,0xbb,0xff,0x0d,0x6f,0x57,0x08,0x85,
			0xc9,0xc7,0xbf,0x8d,0xc4,0x99,0x13,0x2c,0xae,0xe0,0x71,0xb4,0x1d,0x81,0xff,0x91,
			0xb8,0xce,0x21,0xaa,0x2f,0x28,0x2c,0xbf,0x52,0x38,0x9f,0x23,0x9a,0xfe,0x14,0x90,
			0x89,0x0b,0xe2,0x1f,0x9d,0x80,0x8b,0x3d,0x70,0xb9,0x7e,0xfd,0x59,0xc0,0xb6,0x0e,
			0x46,0x60,0x88,0xbb,0x42,0x71,0x4f,0x21,0x2b,0xc9,0x0d,0xb7,0xe9,0x42,0xeb,0xce,
			0xe6,0x0e,0x7b,0x10,0x7f,0xff,0x44,0xfb,0x35,0x64,0xff,0x07,0xd6,0xd0,0x28,0x50,
			0x21,0x5f,0xd3,0x57,0xd8,0x97,0xc4,0xd3,0x2b,0xef,0x86,0x61,0x68,0x9f,0x2d,0x84,
			0xff,0x89,0x76,0x37,0xfb,0x6d,0x55,0x68,0xa7,0x27,0x0e,0x78,0x34,0x26,0xb7,0x4b,
			0x70,0x37,0x49,0x3e,0x51,0x55,0xfd,0x7c,0xb3,0xdd,0xdd,0xfd,0x36,0xbd,0x8a,0x9c,
			0x87,0x7d,0x71,0xd2,0xa9,0x66,0x05,0x7c,0x08,0x26,0x3d,0x29,0x39,0xc8,0x49,0x87),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x61,0xd7,0xb3,0x15,0x01,0x31,0x35,0x1e,0x7b,0x4c,0x8e,0x56,0x45,0xd3,0x8b,0xe9,
			0x33,0x5b,0x40,0x28,0x9a,0xf3,0x4c,0xc6,0xb6,0xfc,0x5e,0x48,0x49,0x3b,0xf8,0xb7,
			0x85,0x2c,0x73,0x98,0x2c,0x99,0x44,0x1e,0xf6,0x6c,0x7d,0x9d,0x33,0xc2,0x97,0x42,
			0xb1,0x40,0x6e,0x02,0xe0,0xaa,0x8d,0xd0,0x34,0xb1,0xac,0x13,0xcb,0x0d,0x77,0x57,
			0x50,0xcc,0x91,0x42,0x1f,0xea,0xd9,0xca,0xa9,0x21,0xec,0xa6,0x1a,0x02,0xeb,0x02,
			0x3a,0x45,0x7e,0x77,0x91,0x5e,0x18,0x3a,0xcf,0x51,0x7d,0x94,0x6b,0xc6,0x82,0x92,
			0x89,0x60,0x14,0xfd,0x21,0x4b,0x7c,0x8c,0x5e,0x14,0xe1,0x59,0x44,0xbe,0x0f,0x92,
			0x96,0x12,0x77,0x71,0xf7,0x36,0x76,0x6e,0x4f,0x81,0xda,0xb3,0x70,0x8e,0xa2,0xd0),
		.s = chunk_from_chars(
			0x84,0xe9,0x2a,0x14,0x5a,0xe6,0xbe,0x1f,0xf9,0x24,0x2d,0x9e,0xd2,0xd6,0x8d,0xe6,
			0x68,0xe8,0x02,0x52,0x4e,0x8a,0xc0,0xa7,0x9d,0xe6,0x2f,0xe7,0x40,0x48,0xc3,0x54,
			0x91,0xfd,0x2f,0xfd,0xb1,0x85,0x05,0x7e,0x66,0x6d,0xbf,0xaa,0xc8,0x4c,0x34,0xfd,
			0xe7,0x89,0x12,0x63,0xf8,0xb2,0xbc,0x74,0x74,0x62,0x30,0x32,0x0f,0x67,0xa7,0xbd,
			0x73,0x19,0xc9,0xb9,0xde,0x41,0x90,0x54,0x70,0x14,0xe2,0xd7,0xa2,0xa5,0x06,0x0d,
			0x62,0x00,0xaa,0xdc,0x3a,0x44,0xba,0xc0,0x29,0xff,0x39,0x92,0xed,0xd3,0x0e,0xc5,
			0x3a,0xb0,0xd9,0x12,0x3e,0xaa,0x6b,0x14,0x73,0x52,0xa0,0x73,0xa9,0x81,0x61,0xe6,
			0x4f,0x39,0x4b,0xb9,0x94,0x92,0xc6,0x97,0x7e,0x24,0xf4,0x45,0xc7,0x12,0x5b,0xfb,
			0x90,0xf8,0x7f,0xaf,0x26,0x22,0x72,0x13,0x4a,0xcb,0x18,0x82,0x3a,0x99,0xa5,0x22,
			0x8d,0x14,0x95,0x46,0x32,0x97,0xfd,0x77,0x48,0x77,0xfb,0x63,0xd4,0x91,0x81,0x06,
			0x34,0x7e,0x6f,0x29,0x31,0x5e,0x48,0x36,0x3f,0x39,0xb3,0x32,0x99,0xea,0xa3,0x2d,
			0x8d,0xa7,0x1b,0x22,0x9d,0x8f,0xfe,0xe5,0xf6,0x6f,0x72,0x2a,0xd3,0xaa,0x41,0x75,
			0xd3,0xf8,0x4e,0xce,0x9c,0xc8,0xec,0xa8,0xd6,0xf2,0xf3,0x56,0xa8,0x5c,0x15,0x24,
			0x89,0x6c,0x18,0xf7,0xb5,0xc8,0xf9,0xbc,0xde,0xf4,0x5c,0x49,0x6d,0x53,0x91,0x79,
			0x89,0x1d,0xdc,0x76,0xe5,0x20,0x8a,0xd8,0x35,0x3d,0x48,0xc6,0x24,0x05,0x4f,0x34,
			0x40,0xee,0xba,0x44,0x32,0xa1,0x06,0x54,0xa1,0x1e,0xf5,0x37,0x83,0xbd,0x11,0x6f),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xb6,0x77,0x1a,0xb0,0xe1,0x28,0xb4,0x1b,0x32,0xb8,0xb0,0x5e,0x05,0xad,0xd2,0x3c,
			0xe0,0xfb,0x87,0x7b,0x40,0xbf,0xcc,0x3b,0x99,0x2f,0x4c,0x86,0x98,0xd1,0xc8,0x28,
			0xab,0xec,0xbc,0xc1,0xc3,0x3d,0x40,0x18,0x59,0xea,0x2c,0xb2,0xaf,0xbc,0x7f,0xa4,
			0x58,0x88,0x02,0xa5,0xfa,0xee,0x28,0x67,0x53,0x46,0x39,0x28,0x7a,0xd8,0xaf,0x84,
			0x67,0x4b,0xe1,0x8d,0xb6,0x61,0xde,0x1d,0xa8,0xe1,0x9c,0x6b,0x6b,0xd4,0x52,0xdd,
			0x9b,0xf3,0x22,0x1d,0x08,0x61,0xfb,0x6f,0xba,0x96,0xbe,0x42,0x32,0x9b,0x9f,0x04,
			0xf3,0x7d,0xcf,0x3b,0x41,0xfc,0x58,0xd2,0x29,0x83,0x48,0xb0,0xc1,0x5d,0x11,0x90,
			0xb1,0x25,0x30,0x0c,0xf2,0x7e,0x0d,0xfa,0xd6,0x05,0x22,0xfc,0x49,0x84,0x60,0x53),
		.s = chunk_from_chars(
			0x62,0x76,0x92,0x55,0x68,0x62,0x6f,0x0c,0xbe,0x6f,0x51,0x50,0xb0,0x50,0xe1,0x70,
			0x25,0x82,0xf8,0xda,0xf9,0x9a,0x6f,0x88,0x0e,0xf7,0x5c,0xd9,0x6c,0x2d,0x42,0x08,
			0xfb,0x6e,0x91,0xb0,0x1b,0xa6,0xab,0xa2,0xa8,0x16,0xb2,0xd3,0xcb,0x97,0x5d,0xf8,
			0x50,0xb1,0xd2,0x68,0xc4,0x66,0x2d,0xd1,0xea,0x3a,0x30,0x0c,0x1d,0x71,0x71,0xc6,
			0x33,0xdd,0x2e,0xfb,0xac,0x30,0x00,0xc5,0x6a,0xb8,0x0f,0x98,0x9d,0xbc,0x18,0x24,
			0x3e,0x63,0x6b,0xa5,0xd4,0xd2,0x6a,0x7d,0x3f,0x19,0x65,0xad,0x3c,0xb0,0xf1,0xa8,
			0x51,0x3f,0x99,0x80,0x03,0xf7,0xb6,0x7e,0x2a,0xc5,0xc7,0x18,0xcb,0x68,0x8b,0x32,
			0x01,0xd5,0x6e,0x68,0xf0,0xb9,0xf8,0x62,0x57,0xb8,0x47,0x94,0xcd,0xff,0xbc,0x1f,
			0xe3,0xea,0x24,0xb7,0xbb,0x6e,0x9e,0xf0,0x53,0x9b,0xd4,0xfb,0xc1,0xaf,0xb5,0x5b,
			0xc1,0xdc,0xa3,0x99,0x96,0xea,0x8a,0x63,0x76,0x9f,0x6e,0x22,0x57,0x07,0xf6,0x90,
			0x47,0x55,0x5e,0x1a,0x4e,0xf3,0xc6,0x39,0xc5,0xf2,0xa4,0x97,0xb8,0x89,0x42,0x4a,
			0x90,0x14,0x86,0x39,0xbb,0x64,0xdf,0x0a,0x06,0xe0,0xb7,0xf0,0xe8,0xed,0x46,0x6a,
			0x97,0x7b,0xac,0xa3,0x2f,0x48,0x23,0x37,0xb2,0xab,0xe3,0x98,0x3e,0xae,0xc3,0xfe,
			0x10,0x75,0x01,0x6e,0x58,0x67,0x52,0x17,0x60,0xfd,0x06,0x07,0xd7,0x99,0xf1,0x76,
			0x6b,0x3f,0xf6,0xe2,0xae,0x15,0x5d,0x69,0x25,0x0f,0x8b,0xf0,0x8c,0x8e,0xdc,0xa0,
			0xb4,0xf3,0x1d,0x0f,0x83,0x8c,0xfd,0x29,0x8c,0xb7,0x31,0x2d,0xf9,0x3f,0x09,0x97),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x6a,0x81,0xcb,0x6c,0x7b,0x26,0x8f,0x4b,0x9f,0xb9,0x17,0x2a,0xdb,0xbb,0x36,0xa2,
			0x37,0xa0,0xdc,0xf1,0xc3,0xc8,0x3a,0x95,0xdc,0xb0,0x27,0x1a,0xac,0x6a,0xc3,0x30,
			0xf0,0x4a,0x5a,0x00,0xfe,0xe3,0x8b,0xc0,0x06,0x31,0xa9,0x85,0x98,0x18,0x61,0x59,
			0x66,0x0d,0x9d,0x8e,0x4c,0x14,0xa9,0x52,0x8d,0xea,0x94,0x83,0x60,0x83,0xda,0xc4,
			0xab,0xb7,0x3f,0xd0,0x0e,0x38,0xfe,0x0e,0x23,0xc7,0x23,0x66,0x04,0xa7,0x36,0x54,
			0x0e,0x52,0x19,0x3a,0xe5,0x6c,0x33,0xfb,0xb8,0xf5,0xcf,0xc5,0xc7,0xc2,0xbe,0x2e,
			0x22,0x2e,0x44,0x83,0xb3,0x0d,0x32,0x5c,0x7e,0xe1,0x4f,0x74,0x28,0x51,0xfc,0xb8,
			0xb6,0xd6,0x18,0x9e,0x98,0xb8,0x22,0xb8,0xe6,0x39,0x9d,0x89,0xe9,0x0f,0xb9,0x97),
		.s = chunk_from_chars(
			0xb6,0x79,0x91,0x05,0x0c,0x08,0x3e,0x64,0x50,0x97,0xdb,0x03,0xff,0xf3,0x47,0x58,
			0x86,0x8b,0xeb,0x19,0xe9,0xc0,0xc4,0x84,0x75,0xf0,0xf9,0x13,0x36,0x1e,0x71,0xd3,
			0xd6,0xf2,0x7a,0x8c,0x4f,0x0b,0x26,0x9b,0x49,0xe8,0x53,0x40,0x39,0xe5,0x3a,0xd3,
			0xba,0xb9,0xa3,0xe6,0x2a,0xbe,0x07,0x8e,0xe7,0x5e,0x7f,0xb5,0x95,0x90,0x06,0xfb,
			0xfb,0x01,0x4c,0xa7,0xb8,0x1b,0x3d,0x5a,0xfe,0x0e,0xe5,0xf6,0xfc,0x2d,0xfb,0xc4,
			0x50,0xf2,0x83,0x95,0x43,0x00,0x2f,0x33,0xf4,0xf3,0x54,0xf8,0x27,0x27,0x8c,0x76,
			0xc0,0x41,0x68,0x6e,0xea,0x78,0x86,0xeb,0xb2,0xa7,0xaf,0xa5,0x99,0x5c,0x6c,0xdd,
			0xb1,0xc0,0xb5,0x80,0x66,0xdd,0xb8,0xdc,0x54,0xa6,0x92,0x7c,0x14,0x6c,0x3b,0x2a,
			0x0f,0xa7,0xce,0xf2,0x89,0x03,0xc6,0xc6,0x72,0xbc,0x20,0xef,0x68,0xff,0xbf,0xab,
			0x24,0x7e,0xb6,0x88,0xab,0x4b,0xde,0x71,0x06,0xd9,0xc5,0x9d,0x21,0x53,0x09,0x6d,
			0xc9,0xe5,0x20,0x72,0x67,0x03,0x8d,0x88,0xe2,0x17,0x4e,0x76,0xad,0xc1,0x50,0x8a,
			0xe2,0x4e,0xb6,0x02,0x33,0x2e,0x53,0xc0,0xc2,0xe3,0x31,0x54,0xa6,0x6a,0x97,0xa0,
			0xf1,0x2f,0x66,0xc6,0x12,0x58,0xc7,0xbf,0x6b,0xbf,0x3f,0x1d,0xcb,0xe9,0xca,0xf2,
			0xfd,0x30,0xec,0x68,0xc0,0xa9,0xd0,0x9f,0x4f,0xd7,0x76,0x30,0x4b,0x54,0x0e,0x62,
			0xfc,0x85,0x12,0xbe,0xaa,0xbc,0x4b,0xe2,0x10,0x7a,0x1e,0xc1,0x8e,0x87,0xf6,0x1f,
			0x9d,0xb2,0x5e,0x87,0x1d,0xc0,0x69,0x3c,0xef,0x17,0xc2,0xa6,0x87,0xfc,0x85,0x4f),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x05,0x6c,0x1e,0x46,0x44,0x59,0x9e,0x31,0x83,0xdd,0x8d,0x2f,0x64,0xe4,0xbb,0x23,
			0x52,0xff,0x00,0xd0,0x12,0xab,0x76,0x3f,0x9a,0xd6,0xe5,0x60,0x27,0x9f,0x7f,0xf3,
			0x8a,0x5e,0xce,0xa9,0xc2,0xe4,0xea,0x87,0xd0,0x04,0xef,0x8c,0xc7,0x52,0xae,0x93,
			0x23,0x2a,0xa3,0x7b,0x5b,0xf4,0x28,0x84,0xba,0xa7,0xe7,0xfc,0x6a,0x8c,0x95,0x1c,
			0xd2,0x45,0xde,0x2d,0x22,0x0d,0x9b,0xee,0x2b,0x41,0x4b,0x3a,0x75,0x20,0xc1,0xe6,
			0x8b,0xcf,0x1a,0xe9,0x9a,0x9f,0xf2,0xbf,0x3a,0x93,0xd8,0x0f,0x8c,0x1d,0xfe,0x8b,
			0x85,0x29,0x35,0x17,0x89,0x5c,0x19,0x2e,0x3c,0x9e,0x89,0x82,0x95,0xd6,0x5b,0xe3,
			0x34,0xf4,0x4d,0x62,0xf5,0x35,0x3e,0xb6,0xc5,0xa2,0x9e,0xdf,0xb4,0xdb,0x23,0x09),
		.s = chunk_from_chars(
			0xae,0x05,0x20,0x4e,0x40,0x9d,0x72,0x7e,0xb9,0xe4,0xdc,0x24,0xbe,0x8f,0x86,0x33,
			0x28,0xc2,0x81,0x3d,0xa4,0xfc,0xef,0x28,0x86,0x6e,0x21,0xa5,0xda,0xb2,0x1a,0x48,
			0x53,0x21,0xb7,0x35,0x27,0x4a,0xf0,0x6b,0xf1,0x7e,0x27,0x15,0x18,0xe1,0x11,0x64,
			0xd7,0x22,0xab,0x07,0x35,0x48,0xf0,0x2e,0x1b,0x44,0x19,0x23,0xdb,0x6f,0x1c,0xee,
			0x65,0xa0,0x17,0xed,0xfb,0xaf,0x33,0x61,0xc6,0x7f,0xbc,0x2b,0x39,0xfe,0x03,0x8c,
			0xb5,0xcb,0x65,0xa6,0x40,0xf9,0x58,0x87,0x38,0x9c,0xe8,0xa5,0xad,0x2e,0xc6,0xe6,
			0x9d,0x3d,0x60,0x35,0x05,0xb0,0x25,0xf6,0xd6,0x33,0x0c,0x8b,0x64,0x88,0x02,0xca,
			0xf7,0xe6,0xfa,0x3f,0xe7,0xb3,0x81,0x41,0x65,0x99,0x86,0xcb,0x89,0xe6,0x23,0x2f,
			0x10,0x62,0x22,0x56,0x4d,0x5e,0x51,0x95,0xed,0xa6,0xa2,0x5f,0x99,0x06,0x85,0x72,
			0xc2,0xfa,0xfe,0x97,0xf1,0x47,0xf7,0xf2,0xf4,0x11,0x9f,0x21,0x38,0x5a,0xf1,0xfc,
			0xed,0x97,0xf7,0x86,0x32,0xd8,0xbf,0x4f,0xd9,0xa9,0x05,0x4d,0x8b,0x9a,0xa2,0xa9,
			0xf4,0xde,0xd5,0x87,0x84,0x7a,0x91,0xd4,0x2c,0x63,0x91,0x12,0x5f,0x10,0x3a,0xe2,
			0x88,0x54,0x7e,0x84,0x89,0x69,0x3a,0xe8,0x68,0x6b,0x84,0x89,0x1b,0x77,0x2b,0x10,
			0xc4,0x79,0x68,0x83,0xf6,0x6c,0xd4,0x59,0xa8,0xc1,0xa6,0xa4,0x18,0x7b,0xd6,0xb3,
			0x87,0xd3,0x49,0xe9,0x2d,0x7b,0x60,0x49,0x53,0x72,0x7c,0x9e,0x9f,0xdc,0x44,0x9e,
			0x73,0x45,0xe7,0xca,0x6b,0x33,0x9e,0x26,0xb0,0x86,0xf5,0x54,0x88,0x98,0xcb,0xe9),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xce,0xc5,0xc9,0xb6,0xf8,0x44,0x97,0xac,0x32,0x7f,0x68,0xef,0x88,0x66,0x41,0xfe,
			0xc9,0x95,0x17,0x8b,0x30,0x71,0x92,0x30,0x43,0x74,0x11,0x5e,0xfc,0xc5,0xee,0x96,
			0x27,0x0c,0x03,0xdb,0x0b,0x84,0x6d,0x67,0x4c,0x52,0x8f,0x9d,0x10,0x15,0x5a,0x3f,
			0x61,0xbe,0xcc,0xe1,0xd3,0xa2,0xb7,0x9d,0x66,0xcd,0xc4,0x09,0xad,0x99,0xb7,0x66,
			0x30,0x80,0xf5,0x1a,0x10,0x2f,0x43,0x61,0xe9,0xdb,0xd0,0x3f,0xfc,0xd8,0x76,0xb9,
			0x8e,0x68,0x3d,0x44,0x8b,0xd1,0x21,0x7e,0x6f,0xb2,0x15,0x1c,0x66,0x96,0x47,0x23,
			0xb2,0xca,0xa6,0x5c,0x4e,0x6c,0xa2,0x01,0xd1,0xc5,0x32,0xbd,0x94,0xd9,0x1c,0xd4,
			0x17,0x3b,0x71,0x9d,0xa1,0x26,0x56,0x39,0x27,0xca,0x0a,0x7f,0x6f,0xe4,0x25,0x36),
		.s = chunk_from_chars(
			0xc4,0x8a,0x8e,0x01,0xd4,0xbb,0xfe,0x0f,0x2f,0x05,0x65,0x93,0x37,0xea,0x71,0xd2,
			0x1f,0x38,0xd7,0xf7,0xa1,0x0b,0x00,0xb0,0x6e,0x1f,0x89,0x9e,0xaf,0x40,0xa8,0xe9,
			0x7e,0xad,0x64,0xbc,0xa3,0x7f,0x13,0xa5,0x5e,0xf1,0xcf,0x3f,0xb5,0x2c,0xee,0x27,
			0x9c,0xdc,0xb0,0x96,0x08,0x5a,0x46,0x7a,0xfa,0x97,0xb0,0x3d,0x78,0xd6,0x07,0x6e,
			0x47,0x2b,0x12,0xd6,0xbe,0x96,0x47,0xce,0xc3,0x2d,0x8d,0x91,0xa2,0x62,0x47,0x69,
			0x37,0x71,0x68,0x74,0x60,0xba,0x52,0x69,0xde,0x18,0xe1,0xed,0xef,0x60,0x22,0x53,
			0x3a,0x95,0x79,0xf9,0x1d,0x58,0x4f,0x9e,0x0c,0xee,0x11,0x00,0xc4,0x47,0xb7,0x75,
			0x76,0xb1,0xb4,0xee,0x16,0x3e,0xd4,0x70,0x01,0x47,0xa9,0xaa,0x61,0xbd,0xc4,0xe2,
			0x31,0x6d,0x2d,0x81,0x8c,0x10,0x28,0xed,0x1c,0x3e,0x37,0x2c,0x9f,0x6a,0x17,0x45,
			0x57,0x24,0x44,0x63,0x72,0x48,0x09,0x1b,0x83,0xf7,0xb5,0x39,0xf9,0xbd,0x58,0xb7,
			0x67,0x56,0x76,0x03,0x4c,0x20,0xe4,0xca,0x11,0x9b,0x91,0xc4,0xca,0x5d,0xc7,0x6a,
			0xcb,0xff,0x3d,0x04,0x62,0x89,0x83,0x52,0xc5,0x91,0xc2,0xca,0x6f,0x2d,0x8b,0x09,
			0xe2,0xe6,0x33,0x8a,0x84,0x33,0x6e,0x06,0xf0,0xcc,0x02,0x0e,0x9e,0xb8,0xda,0x78,
			0x58,0x89,0xb4,0x97,0xf3,0xb9,0x8e,0x82,0x7e,0xe7,0xa7,0xd3,0xf1,0xb0,0xb7,0x3c,
			0x19,0x58,0xe1,0x6a,0xa9,0x78,0x61,0xe6,0x67,0x59,0x70,0xce,0x31,0xd9,0xd1,0x19,
			0xbb,0x34,0x0b,0xe8,0x0f,0xd0,0xf4,0x3c,0x3d,0xbe,0x64,0xf2,0xa5,0x9d,0x62,0x9d),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x91,0x93,0xf8,0xb9,0x14,0xdf,0xe0,0xe6,0x25,0x21,0xf3,0x5a,0xfa,0x4f,0xa5,0xd4,
			0x28,0x35,0xe1,0x98,0xaf,0x67,0x38,0x09,0x37,0x7a,0x3e,0x7a,0x99,0x73,0x31,0x42,
			0xa1,0x80,0xdc,0x0e,0x13,0xe6,0xbb,0x7c,0xeb,0x3b,0x60,0xe5,0xe9,0xd5,0x15,0x79,
			0x4d,0x82,0xc3,0x92,0xe0,0x79,0x13,0x42,0x33,0x91,0xd2,0x2e,0x2b,0xb1,0x9a,0xa0,
			0xbd,0x88,0xaf,0xd7,0xf7,0x7e,0x27,0xa2,0x40,0xea,0x4e,0x2d,0xe0,0x85,0x48,0x1a,
			0xc3,0x1f,0xf8,0xd3,0x79,0x90,0x21,0x1f,0x82,0xf2,0xcb,0xf4,0xc9,0x0d,0xe9,0x8d,
			0x6e,0x13,0x38,0xbb,0xc8,0x8e,0x6a,0x80,0xab,0x96,0x84,0xda,0xe6,0x47,0x85,0xdd,
			0x10,0x72,0x48,0x04,0x85,0x93,0xab,0xc9,0xab,0x03,0xf1,0x73,0x7a,0x6f,0x65,0x30),
		.s = chunk_from_chars(
			0x5c,0x2f,0xe4,0x53,0xa8,0xb0,0x8c,0x90,0xb0,0x2e,0xb2,0xc9,0x99,0x42,0x42,0xd5,
			0x18,0xf3,0xf2,0x1b,0x36,0x88,0x95,0xcf,0xfd,0x62,0x40,0x50,0xe4,0x8a,0xa7,0x14,
			0x00,0x5a,0xe6,0x75,0xfe,0x79,0xaa,0x3c,0xad,0xd4,0xdf,0x55,0xbd,0xf1,0x2b,0xec,
			0x5b,0xe8,0xa4,0x1d,0x87,0x53,0x8f,0x7e,0x03,0x1b,0x78,0x2e,0x34,0xd3,0x92,0x46,
			0x8e,0x5f,0x14,0xbc,0x61,0x3b,0x8f,0x4d,0x28,0xc8,0xfb,0x79,0xa2,0x53,0x7e,0x1e,
			0x60,0x10,0x31,0xda,0x72,0x0a,0xcd,0x7b,0x2c,0x8d,0xcb,0xe9,0x85,0x86,0x24,0xa7,
			0xa9,0xa9,0x2a,0x06,0xf9,0x18,0x45,0xf7,0x32,0x37,0x0d,0x67,0x36,0x5c,0x64,0x64,
			0xf7,0xb6,0x8f,0x22,0xeb,0x3e,0xdf,0xee,0xc9,0x7e,0x32,0x85,0x02,0x4d,0x7f,0x69,
			0x43,0xb6,0xd5,0x0a,0x16,0xcc,0x96,0xd6,0x0f,0x68,0x03,0x51,0xde,0xaa,0x25,0xf0,
			0xbc,0x86,0x89,0x48,0x60,0x7a,0x6b,0xa7,0xf1,0x94,0x9b,0x85,0x94,0x3c,0x6a,0x92,
			0xbd,0x61,0x72,0xe8,0x1b,0xcc,0x05,0x50,0x14,0xb7,0x8a,0x73,0x39,0x72,0xe3,0xf3,
			0x9d,0x14,0x09,0x9d,0x16,0x07,0xa2,0x0f,0xf8,0x68,0x1c,0x29,0xae,0x1e,0xf9,0x9e,
			0xf1,0x15,0xed,0x6a,0x10,0x84,0xb5,0x14,0xb8,0x1a,0x69,0xd4,0xa1,0x5c,0xe1,0xe2,
			0x57,0x6f,0xdc,0xf2,0xb2,0xaf,0x61,0x5b,0x52,0xfe,0xc7,0x01,0x32,0x11,0x2d,0xcc,
			0x5b,0xc1,0x9e,0xc1,0x7f,0x32,0x28,0x14,0x60,0x62,0x34,0x20,0x31,0x73,0x53,0xe8,
			0xa2,0x55,0xfd,0xa5,0x02,0xbd,0x1f,0xb1,0x1a,0x58,0x83,0x2a,0xe2,0xc0,0x4f,0x9a),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x0e,0x57,0xef,0x40,0xb0,0x21,0xbf,0x87,0xf6,0x42,0xc5,0x75,0x6b,0x65,0x15,0xa0,
			0xe0,0x6c,0x15,0xa0,0x18,0x56,0xd7,0x16,0xc5,0x66,0xa6,0xed,0xb3,0x81,0xdf,0xdf,
			0x44,0xd9,0x03,0x3b,0x1c,0xc8,0x09,0xe6,0x1d,0xfe,0xf9,0xa0,0x96,0xdf,0xb6,0x89,
			0xb7,0x27,0x1b,0xe4,0x49,0xd0,0x4a,0x1a,0x9c,0x35,0x41,0x02,0xc0,0x77,0xaf,0x5f,
			0xf7,0x20,0x05,0xab,0x6b,0x06,0xcf,0x13,0x1d,0x73,0x45,0xc2,0x1e,0x82,0x1d,0x62,
			0x01,0xcc,0xa4,0xe0,0x90,0x44,0x0d,0x70,0xbe,0x60,0x09,0xd2,0xdd,0x7a,0x98,0xd3,
			0x11,0x75,0x1e,0x16,0x05,0xa3,0xb9,0x14,0xdc,0xe6,0xd2,0x62,0x6b,0x16,0xf2,0x33,
			0xa5,0xa3,0xd7,0x1d,0x56,0x7c,0xc8,0x20,0x15,0x2f,0x25,0xe4,0x73,0x51,0x42,0x42),
		.s = chunk_from_chars(
			0x76,0x43,0xaa,0x3f,0xe6,0x3e,0x66,0xf7,0x9d,0x6b,0x40,0x9d,0x14,0x5e,0xa8,0x20,
			0xc9,0xf7,0x35,0x6f,0x71,0xb4,0xac,0xdc,0xbd,0x43,0xfe,0x1e,0x99,0xf8,0x80,0x2c,
			0xd1,0x66,0x2b,0x16,0x24,0x0f,0x5c,0xfd,0x94,0xa7,0x69,0xb0,0xb3,0xf2,0xcb,0x0b,
			0x11,0x88,0x7e,0x88,0x6e,0x5b,0xa4,0x37,0x33,0x36,0x74,0x90,0xb3,0xfc,0x18,0x8f,
			0x2f,0xb3,0xa0,0xc0,0xc8,0xa6,0x8b,0x5d,0x27,0x26,0xc8,0xf7,0xa3,0x19,0x02,0xb6,
			0xb8,0x6c,0xd4,0x02,0x28,0x7d,0x38,0x5c,0x3e,0x3c,0x06,0x50,0x3c,0xe1,0x7f,0xd6,
			0xe5,0x4e,0x58,0x2f,0x4a,0x90,0x7a,0x91,0xf9,0x52,0xd2,0xa3,0x60,0xe2,0xfb,0xa0,
			0x00,0x28,0xe4,0xd3,0xb0,0x2a,0xab,0xf7,0xd2,0x20,0xb3,0x1d,0x1f,0x8e,0xe7,0xfa,
			0xa0,0x70,0x14,0x76,0x82,0xcc,0xc8,0xbc,0xc7,0x56,0xca,0x6a,0x68,0xfc,0x20,0x95,
			0x45,0x50,0xc3,0x17,0xe8,0x79,0x18,0x78,0x1a,0x3d,0x1f,0x19,0x23,0x50,0x30,0x91,
			0x09,0x0c,0x3c,0x60,0xca,0x1c,0x0b,0x1c,0x69,0x99,0x06,0xfb,0xf8,0x5a,0xa7,0x0a,
			0xd9,0xae,0x48,0x70,0x9f,0xf7,0x43,0xb8,0x2d,0xcc,0x31,0x07,0x4c,0xfc,0xea,0x62,
			0x3e,0xa4,0x5e,0x48,0x64,0x4b,0x19,0xa2,0x17,0x72,0xca,0x10,0x7e,0xd6,0x42,0x39,
			0xc5,0x65,0x74,0xa0,0x87,0xf1,0xa6,0xaa,0xdf,0x0f,0x4b,0x00,0xff,0xe5,0x81,0xc1,
			0x41,0x02,0x74,0xc8,0x75,0xe4,0x59,0x90,0x63,0xe4,0x6e,0x51,0x68,0x80,0x3f,0x0d,
			0x28,0xd2,0x1f,0xcd,0x35,0x09,0xb4,0xc6,0x22,0x29,0x95,0xad,0xd7,0x75,0x3b,0xf3),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x0c,0x84,0x91,0xfc,0x34,0x8d,0x34,0x1f,0xe8,0x5c,0x46,0xa5,0x61,0x15,0xf2,0x60,
			0x35,0xc5,0x9e,0x6a,0x2b,0xe7,0x65,0xc4,0x4e,0x2e,0xc8,0x3d,0x40,0x7e,0xa0,0x96,
			0xd1,0x3b,0x57,0xe3,0xd0,0xc7,0x58,0x34,0x22,0x46,0xc4,0x75,0x10,0xa5,0x67,0x93,
			0xe5,0xda,0xea,0xe1,0xb9,0x6d,0x4a,0xb9,0x88,0x37,0x89,0x66,0x87,0x6a,0xa3,0x41,
			0xb7,0xd1,0xc3,0x1b,0xba,0x59,0xb7,0xdb,0xe6,0xd1,0xa1,0x68,0x98,0xee,0xf0,0xca,
			0xca,0x92,0x8f,0x8c,0xe8,0x4d,0x5c,0x64,0xe0,0x25,0xdc,0x16,0x79,0x92,0x2d,0x95,
			0xe5,0xcd,0x3c,0x6b,0x99,0x4a,0x38,0x5c,0x5c,0x83,0x46,0x46,0x9e,0xf8,0x76,0x4c,
			0x0c,0x74,0xf5,0x33,0x61,0x91,0x85,0x0c,0x7f,0x7e,0x2b,0x14,0xbe,0x00,0x27,0xd8),
		.s = chunk_from_chars(
			0xca,0xcc,0x8d,0x9f,0x5e,0xcd,0x34,0xc1,0x43,0x48,0x84,0x61,0x13,0x5c,0x49,0x51,
			0x67,0x61,0x45,0xc6,0xe4,0x72,0xb9,0x2f,0x12,0xf7,0x58,0x04,0x6f,0x17,0x21,0x42,
			0xfa,0x38,0x8f,0x28,0x5f,0x3f,0xff,0x06,0x82,0x42,0x02,0x88,0x29,0x04,0x7e,0x24,
			0x80,0x59,0xed,0x4f,0xd3,0x9d,0x2c,0x5a,0xde,0x46,0x9d,0xc7,0xc3,0x93,0x45,0xe5,
			0x11,0x49,0x50,0xd2,0x03,0x1c,0xc7,0x46,0x5f,0xe7,0x12,0xc4,0x04,0x1d,0x05,0xc7,
			0x56,0xd3,0xf2,0xd8,0x8a,0x46,0xce,0xb9,0x9f,0x2e,0x24,0xa5,0x2e,0x95,0x8a,0x03,
			0xcd,0x25,0x19,0xa9,0xb1,0x37,0xe6,0x2d,0x5c,0xa2,0xb3,0x53,0xf7,0xb0,0x47,0xb6,
			0x25,0xc3,0x60,0x23,0x13,0xfd,0xb5,0x3c,0x8d,0xb2,0x3d,0x83,0x95,0x1a,0x59,0x9d,
			0xb3,0x28,0xfe,0xdc,0x4a,0xe0,0x6d,0xa8,0x9c,0xe7,0xf5,0x62,0x59,0xb5,0xc8,0x22,
			0x2f,0x7b,0xd3,0xd9,0x74,0x04,0x78,0xfd,0x28,0xe5,0x81,0x0d,0xb7,0x8a,0xee,0x86,
			0x23,0xfd,0xd3,0x9f,0x60,0x3f,0x8d,0xdf,0x98,0x08,0x1d,0x78,0x73,0x98,0x0c,0x4e,
			0xb0,0xe2,0x2a,0x9c,0xd4,0x08,0xf7,0xc4,0x13,0x4c,0x12,0xd2,0x04,0x9a,0x2d,0x12,
			0x0f,0x4b,0x62,0xe6,0xb3,0x82,0xb9,0x97,0xfc,0x37,0x5e,0xf7,0xac,0x95,0x5f,0xcf,
			0x80,0xb0,0x45,0xc3,0xd6,0x38,0x5f,0xf4,0x22,0xda,0xd3,0x50,0xc6,0x88,0x70,0x53,
			0x90,0x68,0xa1,0x62,0xa2,0xed,0xbb,0x93,0xce,0xef,0xed,0x96,0x77,0x93,0x9b,0x90,
			0xbd,0x3d,0xfa,0x0d,0xc0,0x53,0x46,0x0b,0x4e,0x23,0x32,0xef,0xa6,0x92,0x17,0x9a),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x6c,0xd5,0x9f,0xdd,0x3e,0xfd,0x89,0x3d,0x09,0x1a,0xfd,0xc3,0x15,0x5d,0x35,0x4f,
			0x10,0xd6,0xd8,0x81,0x67,0x42,0x7a,0x2c,0xf7,0x24,0x62,0x07,0xe5,0x17,0x91,0xa6,
			0xca,0x62,0x00,0xa9,0x14,0xcd,0x28,0x34,0xa9,0xb3,0xc7,0x9f,0xcd,0x59,0xe2,0x6e,
			0x45,0x7e,0x06,0x83,0xbc,0x33,0xd4,0x92,0x67,0xed,0xbd,0xd6,0xe5,0xd9,0x09,0x02,
			0x69,0x6f,0x1e,0x7b,0x1a,0x4a,0xff,0xc4,0xba,0x37,0x13,0x39,0x86,0x8c,0x28,0x01,
			0x5e,0xbb,0xb7,0x3e,0x26,0x26,0x69,0x86,0x6c,0x35,0xdb,0x97,0x4b,0xa6,0x9e,0x46,
			0x8f,0x25,0x83,0xb9,0x19,0x1d,0x15,0xd6,0x86,0xcd,0x66,0xfb,0x0b,0x9e,0x0f,0xf0,
			0xa3,0xb4,0x72,0x1a,0x6d,0xc3,0x42,0xf1,0x4f,0x24,0x46,0xb4,0xe0,0x28,0x59,0x5b),
		.s = chunk_from_chars(
			0x39,0x74,0x90,0x0b,0xec,0x3f,0xcb,0x08,0x1f,0x0e,0x5a,0x29,0x9a,0xdf,0x30,0xd0,
			0x87,0xaa,0xba,0xa6,0x33,0x91,0x14,0x10,0xe8,0x7a,0x49,0x79,0xbb,0xe3,0xfa,0x80,
			0xc3,0xab,0xcf,0x22,0x16,0x86,0x39,0x9a,0x49,0xbc,0x2f,0x1e,0x5a,0xc4,0x0c,0x35,
			0xdf,0x17,0x00,0xe4,0xb9,0xcb,0x7c,0x80,0x5a,0x89,0x66,0x46,0x57,0x3f,0x4a,0x57,
			0x0a,0x97,0x04,0xd2,0xa2,0xe6,0xba,0xee,0x4b,0x43,0xd9,0x16,0x90,0x68,0x84,0xad,
			0x3c,0xf2,0x83,0x52,0x9e,0xa2,0x65,0xe8,0xfc,0xb5,0xcc,0x1b,0xdf,0x7b,0x7d,0xee,
			0x85,0x94,0x1e,0x4b,0x4f,0xb2,0x5c,0x1f,0xc7,0xb9,0x51,0xfb,0x12,0x9a,0xb3,0x93,
			0xcb,0x06,0x9b,0xe2,0x71,0xc1,0xd9,0x54,0xda,0x3c,0x43,0x67,0x43,0x09,0xf1,0xd2,
			0x12,0x82,0x6f,0xab,0xb8,0xe8,0x12,0xde,0x2d,0x53,0xd1,0x25,0x97,0xde,0x04,0x0d,
			0x32,0xcb,0x28,0xc9,0xf8,0x13,0x15,0x9c,0xb1,0x8c,0x1b,0x51,0xf7,0xa8,0x74,0xcb,
			0xf2,0x29,0xcc,0x22,0x2c,0xae,0xb9,0x8e,0x35,0xec,0x5e,0x4b,0xf5,0xc5,0xe2,0x2c,
			0xc8,0x52,0x86,0x31,0xf1,0x51,0x17,0xe8,0xc2,0xbe,0x6e,0xac,0x91,0xf4,0x07,0x0e,
			0xec,0xdd,0x07,0xec,0xc6,0xdb,0x6c,0x46,0xea,0xa6,0x5f,0x47,0x2f,0x20,0x06,0x98,
			0x8e,0xfe,0xf0,0xb5,0x1c,0x53,0x8c,0x6e,0x04,0xd7,0x51,0x9c,0x8e,0x3d,0xa4,0xb1,
			0x72,0xb1,0xe2,0x76,0x10,0x89,0xed,0x3a,0xd1,0x19,0x79,0x92,0xef,0x37,0xc1,0x68,
			0xdc,0x88,0x1c,0x8b,0x5f,0x8b,0xbf,0xee,0x91,0x9f,0x7c,0x7a,0xfd,0x25,0xb8,0xfc),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xac,0xb3,0x0b,0xe9,0x09,0x2b,0x2f,0x18,0xf2,0x59,0x34,0xa0,0xd6,0x78,0xb6,0xbc,
			0xd6,0xb6,0x7c,0x2b,0x88,0xe7,0x58,0x84,0xf4,0x7b,0x4f,0xca,0xe3,0xad,0xfa,0x40,
			0x5a,0xfe,0x2c,0x7e,0x61,0xe2,0xd6,0xc5,0x08,0xb9,0x27,0x90,0xac,0x00,0xf7,0x6b,
			0x77,0xc9,0x65,0x08,0x26,0x68,0xbf,0x90,0x0f,0x70,0xa3,0x37,0x62,0xde,0x64,0x13,
			0xaf,0x93,0xaf,0x2e,0xa8,0x08,0x6f,0xda,0x29,0x3d,0xed,0x44,0x75,0xf2,0x3c,0x4c,
			0xc3,0x1a,0xd4,0x94,0xf9,0x8d,0x7d,0xd7,0xb7,0xfd,0x6f,0x7d,0x97,0x2b,0xb7,0x6c,
			0xb3,0x5a,0xdc,0x20,0x68,0x04,0xc3,0xfe,0x5a,0xcd,0xd0,0xe5,0xb8,0xb5,0x4e,0x07,
			0xc2,0x91,0x11,0xf7,0x88,0xbc,0x59,0x02,0xf4,0x0a,0xfa,0xc3,0x0a,0xfd,0xba,0xf2),
		.s = chunk_from_chars(
			0xb5,0xc6,0x0d,0x8d,0xa9,0xb3,0x94,0x38,0x78,0xcb,0x23,0x59,0xcf,0x65,0xe4,0x81,
			0x7c,0x07,0x94,0xf9,0x50,0x45,0x3c,0xa7,0x7c,0x81,0xa5,0xa1,0xc1,0x58,0x55,0x91,
			0xaa,0x50,0xa6,0x74,0x68,0xe3,0xb3,0x99,0xe4,0xfa,0xf1,0xd6,0x06,0xbe,0xa0,0xd9,
			0xe6,0xcc,0x1d,0x2d,0x70,0xdb,0x80,0x63,0x73,0x9e,0x0c,0x27,0xd3,0xdc,0x9f,0x9a,
			0xfe,0x88,0xde,0xa5,0x2e,0x73,0x29,0x8a,0x07,0xd0,0x5c,0x7d,0x97,0x07,0x00,0x2e,
			0xfa,0x53,0x7c,0x38,0x9e,0x38,0xbd,0x37,0xbc,0xa7,0x4e,0xb0,0xaf,0x62,0x61,0xa5,
			0xda,0x06,0x13,0x62,0x02,0xc8,0xad,0x48,0x7e,0xeb,0xd5,0x0b,0xef,0x74,0x76,0x70,
			0x89,0xc7,0x08,0x70,0xbe,0x1d,0x8f,0xab,0x91,0x56,0xf9,0xfd,0xbc,0x2f,0x2e,0x9c,
			0xc3,0x30,0xa9,0x50,0x18,0xce,0x79,0x43,0x98,0x4b,0xec,0xc2,0x56,0x21,0xbf,0xa6,
			0x60,0x18,0xef,0x83,0x20,0xb6,0x00,0x59,0xf9,0x41,0x15,0x6e,0x9c,0xdd,0x87,0xff,
			0x0d,0x82,0xcf,0x7b,0xe7,0x74,0x65,0xe0,0x20,0x3e,0x71,0x20,0xaa,0xec,0xed,0x84,
			0xab,0xd8,0x18,0x69,0x47,0xd4,0xac,0x3d,0xaf,0x3f,0x99,0x39,0x02,0xae,0xc4,0x7c,
			0x30,0x90,0x47,0x5c,0x85,0x7b,0x5d,0x35,0x9f,0x0a,0x55,0x72,0xd4,0x68,0x8e,0x5a,
			0x76,0xa4,0x65,0x38,0x68,0xff,0x54,0xce,0x9f,0x99,0x9e,0x6b,0xb5,0x59,0xd1,0xc1,
			0x1c,0x67,0xc1,0x5b,0xe9,0xd7,0xfe,0x5f,0x8c,0x17,0x04,0x30,0x1d,0x05,0x5f,0x3d,
			0x29,0x07,0x72,0x27,0x79,0xd6,0x01,0x20,0x36,0x08,0x4e,0x95,0x0d,0xe3,0x6f,0x4f),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x60,0x1a,0x6a,0xad,0x3f,0xaa,0x79,0x88,0xd5,0xae,0x52,0x8a,0x69,0x69,0x03,0x1b,
			0x10,0xa6,0xf3,0x92,0x16,0x94,0x6a,0xa8,0x9f,0xd4,0x53,0x2c,0x8e,0xd1,0x41,0xf9,
			0xa6,0x50,0xb1,0x26,0xef,0x48,0x8f,0x7c,0x5c,0xf3,0xfb,0x2d,0xaa,0x25,0x4c,0xc2,
			0x8b,0xdd,0x55,0x56,0x04,0x19,0xe8,0x02,0x14,0xef,0x99,0x98,0x96,0xda,0xc4,0x94,
			0x68,0x52,0xd2,0x4f,0xcd,0x9f,0xb7,0x76,0x10,0xee,0xbf,0xbb,0x6b,0xa5,0x8b,0xca,
			0x26,0xf4,0x56,0x7f,0x03,0xac,0x7e,0x56,0xda,0x55,0x3f,0x23,0x81,0x7b,0xc1,0x03,
			0xee,0x48,0x55,0x92,0xa0,0x58,0xfb,0x5e,0x3b,0xc8,0x29,0x9c,0x72,0x90,0xc7,0x1a,
			0x29,0x13,0x7e,0x75,0xdb,0xf5,0x32,0x8c,0x3a,0x2d,0xcd,0x34,0x16,0x5b,0x3f,0x2e),
		.s = chunk_from_chars(
			0x30,0x1d,0x60,0xd5,0x65,0x76,0xf3,0x66,0x3a,0x7f,0xbe,0x80,0x36,0xbb,0xe4,0xfb,
			0xc0,0xfb,0xd8,0x2c,0xd6,0xa4,0x2e,0x36,0xd7,0xbb,0xc8,0xb2,0x06,0x54,0x3d,0xc2,
			0xd5,0x6d,0x31,0x98,0xe7,0x91,0x1a,0xd1,0x38,0xca,0xd2,0x22,0xdd,0x99,0x05,0x0d,
			0xd1,0xf8,0x5f,0xe1,0x9c,0x8a,0x88,0xbf,0x67,0x13,0x5e,0x7f,0x8f,0x11,0xb5,0xf5,
			0xe4,0x85,0xc9,0x1f,0xc7,0xd4,0x78,0x06,0x9b,0x72,0xf4,0x6e,0xbc,0xdc,0xf2,0xd2,
			0xae,0x7d,0xe6,0xac,0x8f,0xe5,0x3b,0xb6,0xc0,0x49,0x11,0xd1,0x22,0xcc,0x23,0x1d,
			0xc2,0x10,0xb2,0x14,0x7e,0xbe,0x8b,0x05,0x2e,0x8b,0x2c,0xcc,0x09,0xf3,0x38,0xb3,
			0x49,0xde,0x20,0x25,0xcc,0x87,0xb2,0x61,0x9a,0x7b,0x16,0x33,0x47,0xca,0x66,0xa3,
			0x47,0x91,0xa2,0xe4,0x6b,0x4e,0x2a,0xc5,0x7e,0xb9,0xf6,0x02,0x9c,0xdb,0xe0,0x24,
			0xe8,0x96,0xd5,0x7f,0x7d,0x04,0x91,0xf7,0x78,0x33,0x12,0xf8,0xf0,0x6c,0x79,0x07,
			0x70,0x15,0x0c,0xd1,0x39,0xf6,0x1f,0xd2,0xb3,0xe7,0x04,0x1b,0x37,0x26,0x1c,0x6e,
			0x7e,0xa8,0x6d,0x4e,0x06,0xd9,0x30,0x0b,0x1a,0x56,0x67,0xcb,0x02,0x88,0xc5,0x50,
			0xb2,0xaf,0xb3,0x55,0x94,0x48,0x34,0xb4,0x61,0xce,0xad,0x13,0x79,0x42,0x76,0xbb,
			0x46,0xe5,0xe2,0x0a,0xec,0x7b,0x63,0xaa,0xca,0x4d,0x49,0x1a,0x50,0x0f,0xac,0xd5,
			0x9a,0x37,0xc5,0x27,0x79,0xcf,0x46,0x7d,0x74,0xaf,0x1e,0x62,0xb1,0xeb,0xe0,0xfd,
			0x0b,0xe1,0xca,0xcb,0x7c,0xe6,0xd0,0x50,0xd8,0x6e,0x4e,0xb7,0x6c,0xde,0x06,0x93),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x44,0xd3,0xe0,0xfc,0x90,0x10,0x0a,0x1c,0x93,0x16,0x06,0x3f,0x26,0xb1,0x80,0x32,
			0x6c,0xc2,0xe3,0x83,0x4c,0xe5,0x6e,0x43,0x24,0x52,0x8a,0x0b,0xbb,0x01,0x5b,0x3d,
			0x78,0x12,0x95,0x8c,0xd2,0x6b,0x91,0xbf,0x08,0xa3,0xa0,0xb1,0x12,0x1f,0x9f,0x9d,
			0xd7,0x7a,0xcb,0x98,0xa0,0x2a,0xd7,0x5f,0xcd,0x61,0x3c,0x53,0xc7,0x32,0xd1,0xc2,
			0x35,0xf5,0x9b,0x68,0x73,0xec,0xe6,0x36,0x3f,0x27,0x94,0x52,0xb6,0xa4,0xb6,0x5e,
			0x80,0xbb,0x59,0xfd,0x47,0xb9,0xa2,0x93,0x6d,0xcc,0x1e,0x4d,0xfe,0x1f,0x53,0x62,
			0xe3,0x45,0x9b,0x98,0x59,0xdb,0x32,0x09,0xa2,0x69,0x8d,0x27,0xfa,0x8a,0xed,0xfe,
			0xcd,0x4d,0x35,0xb9,0x27,0xda,0xf8,0x68,0x6c,0x59,0xd7,0x00,0x49,0x0f,0x0a,0xa3),
		.s = chunk_from_chars(
			0xaf,0x22,0x29,0xe9,0x4a,0x85,0x7b,0x89,0xe0,0xe8,0x90,0xda,0xca,0x3a,0x8f,0xe1,
			0x2e,0xbd,0xba,0x04,0x94,0x8d,0x18,0x83,0xa7,0xd7,0x81,0x6a,0x3b,0x68,0x2f,0x7d,
			0xa3,0x03,0x25,0x40,0xa8,0x76,0x9f,0x9c,0xca,0xc9,0x58,0x6c,0xf2,0x4e,0x8c,0x20,
			0x4b,0x45,0xb8,0x5d,0x1b,0xdc,0xc5,0xa5,0x45,0x0a,0x21,0x5b,0x40,0x48,0xea,0x42,
			0x98,0x3b,0x34,0x56,0xfa,0x8c,0x76,0xc6,0x78,0x6e,0x02,0x4f,0x70,0x5e,0x08,0x8d,
			0x69,0x45,0x59,0xd6,0x68,0xca,0xa8,0x68,0x4c,0xad,0x0f,0xc5,0x78,0x50,0xfc,0xaf,
			0x34,0xe4,0x58,0xae,0xe8,0xfa,0xd4,0xe0,0x9e,0x6f,0x19,0x65,0x57,0xd4,0xe8,0x86,
			0x02,0x84,0xd9,0x82,0xc0,0x10,0x5d,0x98,0xce,0x49,0x12,0xe9,0x6c,0x35,0x50,0xe2,
			0xa0,0xc7,0xe8,0xba,0xd5,0xab,0xc2,0x9a,0x9a,0x54,0x2f,0x57,0xa8,0xc6,0x05,0x79,
			0x03,0x80,0x67,0xb3,0xd5,0x39,0x1a,0xbc,0x21,0xb4,0xf9,0xde,0xb0,0x24,0xca,0x58,
			0xf9,0xb0,0xc3,0x8c,0x0d,0x1f,0x82,0x37,0x3f,0x52,0x8e,0x93,0x9b,0xd7,0x3a,0x24,
			0xd5,0x01,0xc5,0x91,0x16,0x88,0x14,0xc8,0x72,0xc5,0x25,0xdb,0x0e,0x56,0xca,0xe4,
			0x7d,0xf0,0x0f,0xa3,0x72,0x8d,0xc3,0xa0,0x97,0x69,0x65,0x32,0x3c,0xe8,0xd2,0xde,
			0xe2,0xb1,0x38,0xb5,0x0a,0xb7,0xaf,0xd4,0x84,0x95,0x11,0x46,0x73,0xe9,0x1b,0xb3,
			0xed,0x22,0x05,0xe2,0x6a,0x84,0x55,0x47,0x4c,0x3d,0x4e,0xc8,0x73,0x9b,0xbf,0xf6,
			0xdf,0x39,0xb2,0xb7,0x2e,0xe0,0x50,0x41,0x09,0x30,0x42,0x3b,0x14,0x72,0xb6,0xed),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x5a,0xf0,0x90,0x77,0xa1,0xf5,0x34,0xb8,0x98,0x22,0xb2,0x6c,0x32,0x72,0xad,0xf8,
			0x50,0x0d,0x3c,0x6b,0xd9,0x0f,0x9b,0x5e,0x0d,0x8b,0x21,0x1f,0x16,0xd0,0x72,0x0e,
			0xe0,0xea,0xf6,0x46,0x2b,0x6c,0x8a,0x80,0xdf,0x6d,0x75,0x35,0x9f,0xd1,0x9d,0x03,
			0xa0,0xca,0xfb,0x52,0xbc,0x9d,0x4c,0x37,0xc2,0xaa,0x09,0x99,0x11,0xa7,0x9a,0x92,
			0x65,0x2c,0xc7,0x17,0xf0,0x74,0x6f,0xdc,0xad,0x62,0x7c,0x72,0xf1,0xc2,0x16,0xb2,
			0x43,0xd2,0x17,0x5f,0x6d,0x00,0xbf,0x07,0xd3,0xf6,0xaa,0x2a,0x04,0xd4,0xfe,0x9f,
			0x8f,0xbc,0xe9,0x32,0x18,0x94,0x4b,0x92,0xaa,0x07,0xaf,0x6b,0x4f,0xcd,0x80,0xcf,
			0xde,0x2d,0x7a,0xda,0x15,0xc0,0x5e,0x96,0xe7,0x77,0xea,0x1c,0x17,0xdf,0x08,0xfc),
		.s = chunk_from_chars(
			0xa5,0x68,0x23,0xfa,0x57,0x7e,0x89,0x46,0xf1,0xd2,0xf6,0xe3,0x51,0xb7,0x38,0xb5,
			0x35,0x92,0x54,0x43,0x58,0x52,0x8a,0xf8,0x88,0x07,0xea,0x4f,0x19,0x01,0x7d,0xfe,
			0x81,0xa3,0xd6,0x9f,0x62,0xfb,0xff,0x64,0x95,0x50,0xd9,0xb3,0x10,0xfa,0xf2,0x7a,
			0x04,0x1f,0xe6,0x24,0xf0,0xa0,0x2b,0xdc,0xdd,0xb7,0x9b,0xfb,0x0a,0x46,0x57,0x39,
			0xec,0x8b,0x64,0xb7,0x48,0xcc,0x29,0xe5,0xa0,0x2c,0x77,0x7e,0x18,0x26,0xd3,0xe2,
			0xf1,0xee,0xe6,0xfe,0x2e,0xde,0xe4,0xa8,0xbc,0xac,0x51,0x9c,0x7c,0x7c,0xa5,0xc0,
			0x39,0xe7,0x6d,0x63,0x06,0x68,0x94,0x5a,0x1e,0x5e,0x86,0x18,0xe2,0x35,0x86,0x45,
			0x61,0xa4,0x40,0xe7,0x3e,0x39,0xf6,0xd6,0x84,0x2a,0xd7,0xda,0x64,0xef,0x5b,0x0c,
			0xe1,0xc4,0xab,0x88,0xdb,0x15,0x7b,0x68,0x10,0x71,0x74,0xad,0x7d,0x5c,0x9a,0x60,
			0x65,0x06,0x87,0x68,0xc1,0x1c,0x4c,0x96,0xff,0x67,0x05,0x0b,0x5d,0x07,0xb8,0xcd,
			0x02,0x7f,0xcd,0x0d,0x34,0x7e,0xc7,0x9a,0x19,0x7c,0xf4,0x34,0x35,0x98,0x5b,0xc1,
			0xae,0xb4,0x79,0xdb,0x00,0x22,0x28,0x9e,0x8d,0xd3,0xb3,0x1b,0xb7,0xc6,0x2d,0x88,
			0x31,0xcf,0xe6,0x95,0x2f,0x41,0xd2,0x4f,0x89,0xd7,0x53,0x78,0x95,0x35,0xf9,0x18,
			0xff,0x68,0xb3,0x69,0x50,0xaf,0x6f,0xd3,0x1d,0xee,0x1a,0xc4,0x76,0xa0,0xcf,0x93,
			0xaf,0xe9,0xf4,0xa7,0x66,0xf3,0xc4,0xd2,0xc0,0xc3,0xf9,0x28,0x25,0xd5,0x57,0x2e,
			0xb2,0xeb,0x8a,0x2b,0x64,0x4e,0x32,0x9e,0xea,0x16,0x83,0xf9,0x08,0x10,0xed,0x77),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xf6,0x0a,0x3a,0x54,0x37,0x68,0xfa,0xbe,0x37,0xf0,0x03,0x00,0x9a,0x8c,0x26,0xf7,
			0xdc,0x91,0xf1,0x42,0x2d,0x44,0x29,0xed,0x7f,0x9d,0x74,0x4c,0xdd,0x4b,0x55,0x2a,
			0xfe,0xf7,0x5d,0x24,0x1a,0xcd,0xa0,0x4f,0xfc,0x39,0x67,0x21,0x59,0xee,0x24,0x8e,
			0x60,0x2d,0xab,0x71,0x92,0x44,0x9e,0x2e,0xd4,0x55,0x29,0x95,0xc2,0x58,0xf0,0x0a,
			0x47,0x63,0x46,0xe3,0x6a,0x29,0xa0,0x12,0x6b,0xc2,0x49,0x04,0x0f,0xaa,0x57,0xc9,
			0x38,0x0b,0xdd,0x74,0xb8,0x3f,0x62,0xc5,0x67,0x90,0x92,0x05,0x74,0x43,0x34,0x32,
			0xf8,0xd6,0x5c,0x5c,0xd1,0x85,0xe2,0x4f,0xad,0x13,0x12,0x72,0x65,0xc6,0xa5,0xef,
			0x8d,0xb4,0xf1,0x14,0x49,0x3d,0x5c,0xfa,0x61,0xd9,0x16,0x64,0x98,0x14,0x08,0xe9),
		.s = chunk_from_chars(
			0x08,0xd3,0x96,0x48,0x1d,0xee,0xf1,0x8c,0xb0,0xbe,0xf7,0xc3,0xe8,0x26,0xfe,0x6e,
			0x5c,0x9e,0xcc,0x85,0xe5,0x23,0x0d,0x35,0xd6,0x67,0x72,0xb8,0xd2,0xd0,0x15,0xd4,
			0xe5,0xf5,0x79,0x4f,0xbe,0x05,0x50,0xdf,0x2f,0x74,0x57,0x30,0xd6,0xf8,0xd1,0xd3,
			0xb8,0x50,0xd1,0x64,0xfc,0xe4,0x63,0x08,0x05,0xe7,0x11,0xb5,0x93,0x08,0xf8,0x60,
			0x85,0x06,0xb7,0xe0,0x1e,0x8e,0x92,0x94,0xed,0x8b,0x7e,0x75,0x82,0x16,0x56,0x77,
			0xf1,0x80,0xe9,0x65,0x16,0x9d,0xca,0x81,0xb3,0xda,0xf2,0x4d,0x7b,0x92,0xfe,0x32,
			0xd6,0xa9,0xac,0x63,0x82,0x1d,0x48,0xb1,0xa0,0xa1,0x44,0xfc,0x7a,0x04,0xb0,0xbf,
			0xc6,0x3a,0x3b,0xc1,0x6a,0x0f,0xd8,0x37,0xb0,0x20,0x37,0xed,0x76,0xe5,0x0d,0x46,
			0xcb,0xfa,0x38,0x57,0xe6,0x58,0xe3,0x70,0xc5,0x86,0xab,0x1e,0xed,0x82,0x50,0x76,
			0x32,0x1a,0xc8,0xe8,0x2b,0xe3,0x74,0xba,0xcb,0x29,0x5e,0x4d,0x34,0x08,0xf0,0xcc,
			0x1f,0xc4,0xc3,0x00,0xb8,0x42,0x75,0xa5,0x1c,0x35,0x73,0xe9,0xca,0xbf,0xdb,0xe3,
			0xdc,0x51,0xe4,0xa6,0xf5,0x81,0x1d,0x86,0x0d,0x72,0x5a,0xaf,0x8f,0xd0,0xaf,0x19,
			0xa2,0x43,0x7b,0x0f,0x1c,0x80,0xf5,0xac,0x22,0x2f,0x6b,0x25,0xf1,0xfa,0x09,0xe9,
			0x33,0x99,0xa6,0x97,0x6b,0x1b,0x3c,0xa7,0x6a,0xfe,0x60,0x86,0xe9,0xb2,0x32,0xaa,
			0xe6,0xc7,0xb8,0x18,0x25,0x5b,0xf9,0x63,0xf3,0x1c,0x04,0xae,0x3f,0xa2,0x13,0x6c,
			0x0a,0x44,0x29,0x97,0xd4,0xcf,0x12,0xf3,0x95,0xfb,0x80,0x4a,0x47,0x55,0xb5,0x6b),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x2c,0x07,0xa8,0x1d,0xe5,0x89,0x55,0xb6,0x76,0xfe,0xc0,0x57,0x2d,0x48,0xd1,0x95,
			0x5b,0x48,0x75,0xff,0x62,0xa4,0x4b,0x00,0x10,0xc7,0xa1,0x07,0x2b,0x29,0x9e,0xe4,
			0x4d,0xd0,0xc0,0x76,0xf2,0x17,0x8a,0x83,0xd0,0xae,0x76,0xe7,0x67,0xe2,0x31,0xf1,
			0xd8,0x1e,0x07,0x0a,0xfa,0xb2,0x9c,0x97,0xab,0xd4,0xde,0x21,0x64,0xe4,0x37,0xb3,
			0x11,0xf5,0x07,0x84,0x1f,0x88,0x51,0xd6,0xd6,0x9a,0xb5,0x1e,0xe9,0xe2,0x9e,0x65,
			0x4b,0x54,0xbc,0xee,0x45,0xe9,0xb5,0x19,0xc6,0xa2,0x17,0x87,0xfa,0xcb,0x92,0x7f,
			0x1d,0x7d,0x64,0x91,0x92,0x66,0x14,0x79,0x2f,0xcc,0x63,0x46,0xdc,0xd0,0x80,0xbb,
			0x5c,0xf0,0x7b,0xf5,0x6a,0xd0,0xfc,0x4e,0x08,0x3a,0x35,0x82,0x14,0x63,0x15,0x10),
		.s = chunk_from_chars(
			0x9a,0xa3,0x91,0xe7,0xc2,0xf0,0xe9,0x20,0xaa,0xc2,0x7e,0xd9,0xfc,0x20,0x81,0xd3,
			0xc9,0xca,0xa3,0x73,0x58,0x83,0xd0,0x1a,0xd7,0xa7,0xe3,0xb1,0x18,0x67,0xd0,0xad,
			0x62,0x41,0x56,0x47,0x7b,0xbb,0xdd,0xe6,0x59,0xf4,0x74,0x68,0x2d,0x0d,0x77,0x44,
			0x89,0xe2,0xb5,0xb0,0x39,0xd1,0xeb,0x35,0x45,0x4c,0x9e,0x3e,0xed,0x78,0xcf,0xf9,
			0xc4,0x26,0x2e,0x3a,0xec,0xfc,0xa1,0xd8,0x17,0x54,0x2b,0x48,0x60,0x96,0x59,0x8e,
			0x11,0x14,0xbf,0xc0,0x3f,0x20,0xa4,0x5d,0xe3,0x6f,0x6d,0xf7,0x0d,0x14,0x4d,0x01,
			0xdc,0x48,0x66,0xa0,0xf8,0x33,0x19,0xe7,0xc2,0xb8,0x53,0x0f,0x8c,0x27,0xa4,0x1b,
			0x7a,0xdd,0x9f,0x69,0x2d,0x8a,0x8e,0x64,0x64,0x55,0xb6,0x7c,0x9e,0xc4,0x7a,0x4d,
			0x2c,0xe3,0xdf,0xe3,0x5d,0x6a,0x2e,0x89,0xd9,0xbe,0x50,0xc5,0xb6,0xda,0x39,0xbb,
			0x02,0x54,0xbd,0x23,0xa8,0x09,0xab,0x97,0xb2,0xb4,0x8a,0x06,0x8a,0x87,0xab,0xde,
			0x6b,0x6a,0x6e,0x35,0x95,0x5f,0xc9,0x2a,0x96,0x26,0xf9,0x60,0x7d,0x5b,0x3f,0x40,
			0x15,0x17,0x27,0x15,0x94,0xbe,0xf7,0x38,0x59,0x81,0x2b,0x6a,0x62,0x1e,0xd6,0xbd,
			0xaf,0x3c,0x5f,0x2a,0x90,0xb1,0xe1,0x68,0x0f,0x68,0xdc,0xfc,0xca,0xcb,0x65,0xe0,
			0x08,0x1f,0x1c,0xcb,0x6a,0x20,0x73,0x70,0x9d,0x1b,0xa0,0x67,0x06,0x50,0x16,0xed,
			0x73,0xeb,0xd7,0xeb,0xe9,0xe7,0xa7,0xb6,0x0c,0x8c,0x9d,0xd0,0x4a,0x56,0xfa,0xb3,
			0x07,0x02,0xc8,0xa6,0xdf,0x6a,0x35,0x3a,0x30,0x10,0x47,0xdf,0x4c,0x7a,0xff,0x62),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x35,0xec,0x92,0xaf,0xdb,0xc2,0xfc,0xef,0xe4,0x8f,0x1e,0x2f,0x6e,0x48,0x29,0xae,
			0x53,0xb3,0xda,0x04,0x59,0xcc,0x4e,0xa8,0xa9,0x68,0x18,0xb5,0x83,0x18,0x91,0xee,
			0x2f,0x50,0x6f,0xff,0x37,0xc8,0x99,0x06,0xd3,0x23,0x3a,0x51,0xa5,0xcf,0x14,0x69,
			0xa6,0x2c,0x18,0x50,0x61,0xf0,0x33,0x08,0x5f,0xca,0x6a,0x54,0xe2,0x45,0x29,0xc3,
			0xd6,0xf0,0xd8,0xe9,0x04,0xbc,0xb0,0xf0,0x89,0xa5,0xcd,0x50,0x86,0x94,0x84,0xda,
			0x1a,0x84,0xf6,0xfb,0x8d,0xe4,0xe5,0x3f,0xce,0x3d,0xc7,0x14,0x20,0x15,0x19,0xd1,
			0x10,0x13,0xf6,0xf6,0xaa,0x64,0xe8,0xb5,0xec,0x5c,0xfe,0xb2,0x7b,0x61,0x1f,0x08,
			0x95,0x05,0x9d,0x8c,0x47,0x72,0x0d,0x55,0xe0,0x0b,0x57,0x7c,0xa5,0x50,0x09,0x20),
		.s = chunk_from_chars(
			0x6b,0x0f,0x5b,0x50,0xe6,0x78,0xda,0x08,0x3e,0xd0,0xf1,0xb6,0x4e,0x94,0x3e,0x8c,
			0x62,0x79,0xc7,0x24,0x6a,0xf5,0xad,0x07,0x9c,0xdb,0xf2,0x23,0xe4,0x2a,0x0d,0x47,
			0x1e,0x56,0x31,0x4b,0xc0,0xd5,0x8f,0x20,0x2a,0xa6,0xc5,0xe1,0xe5,0x25,0x59,0x85,
			0xb0,0x79,0x5d,0x48,0xeb,0x3d,0x4b,0x8e,0x3f,0xc9,0x22,0x40,0xae,0x02,0xb4,0x08,
			0x8c,0x6c,0xe8,0xab,0x0e,0x8c,0x79,0xc6,0x8d,0xfd,0xc4,0x86,0x57,0xd6,0xa2,0x82,
			0x95,0x39,0x1b,0x9a,0x5a,0x5f,0x35,0x25,0x51,0x26,0xbf,0x8c,0xa5,0x3c,0xbc,0xc0,
			0x08,0x2e,0xab,0x52,0xec,0x10,0x9d,0x22,0xa1,0x18,0x5f,0x6d,0xc7,0x92,0xfc,0x29,
			0x0a,0xa8,0xdb,0xae,0xbb,0x2f,0xbe,0x40,0x4f,0x1d,0x03,0x9a,0xa6,0x34,0x3c,0xd7,
			0xaf,0x9f,0xcb,0x2d,0x1e,0x05,0xde,0xf4,0x80,0x96,0xc2,0x37,0xe1,0x0d,0xaa,0x7c,
			0xfa,0xc5,0xae,0x9b,0x3b,0x30,0x22,0x00,0x5d,0x0d,0x2d,0x5c,0x9c,0x5c,0x50,0x2b,
			0x2f,0x23,0x59,0x4e,0x80,0xd1,0x60,0x4b,0xbb,0x8f,0x5d,0xec,0x07,0xcd,0x3a,0xfe,
			0x1f,0x77,0x77,0x43,0xb0,0xb5,0x8a,0x4e,0x0e,0x4e,0x5c,0xaa,0x14,0x88,0x30,0xee,
			0xe0,0x47,0x96,0x8e,0x7f,0x40,0x66,0x1f,0x9f,0x1a,0x02,0xe1,0xa7,0xfd,0x2b,0x6c,
			0xaf,0x19,0x32,0x6a,0x75,0xe9,0x56,0x5e,0xfd,0xc0,0x11,0x4b,0xce,0xcb,0x14,0xdd,
			0xa0,0x6c,0x32,0x9c,0xf3,0x22,0xa5,0xbd,0x3e,0x6a,0xb4,0x8d,0x95,0xf2,0xd2,0xa9,
			0xc1,0xc1,0x23,0x3a,0x0a,0xa0,0x15,0xa7,0x38,0xf9,0x01,0xf1,0x31,0x48,0xb4,0x54),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x80,0xc9,0xde,0xbd,0xf9,0x31,0x74,0xd7,0x57,0x50,0xa6,0xcf,0x09,0xaf,0x71,0xfc,
			0x18,0xfd,0x51,0x3b,0xff,0x9c,0xb4,0x91,0xbe,0x60,0xaf,0x11,0x2a,0x93,0xf0,0x00,
			0x87,0x3c,0xf4,0x38,0x58,0xa0,0x7a,0xca,0x76,0x0a,0x37,0xe7,0x60,0xc8,0xcb,0x01,
			0xd2,0x76,0xf4,0x2d,0x99,0x7f,0x01,0xcc,0xa5,0xe0,0x8a,0x6a,0x60,0x2f,0x5f,0xe6,
			0x3e,0xdc,0xbe,0xd3,0x95,0xb8,0xc9,0x1f,0xb0,0xb3,0x36,0xf2,0x1f,0xea,0x49,0xd9,
			0x50,0xe1,0xff,0x24,0x64,0x0c,0x8d,0x8d,0x3b,0x95,0x08,0x1a,0xd1,0x59,0x66,0x44,
			0xce,0x34,0xa5,0x58,0x58,0x7e,0x4a,0x1e,0x2c,0xd5,0x0d,0xb9,0xed,0x1d,0xd3,0xce,
			0xbb,0xc6,0xdc,0xe8,0x08,0x4d,0x3e,0x1b,0xa7,0x06,0x92,0xe8,0x26,0x18,0xed,0x61),
		.s = chunk_from_chars(
			0x4a,0x15,0xa7,0x83,0xad,0xbf,0x27,0x46,0x22,0xd5,0xa6,0x10,0xbb,0x6f,0xc7,0x33,
			0x37,0x99,0x9e,0x44,0x5d,0xc2,0x13,0x3a,0xcc,0xb7,0x88,0xd6,0x20,0x3d,0x70,0xf3,
			0xcd,0xc6,0x3e,0x67,0xda,0xa4,0x17,0x1a,0x79,0x52,0xa4,0x98,0x64,0x56,0xfa,0xb3,
			0xc0,0x77,0xa8,0x94,0x1f,0xb2,0x59,0xe3,0x7a,0x5c,0x0c,0xbb,0x20,0xc4,0x08,0xfa,
			0x24,0xad,0x0e,0xc8,0x50,0xe9,0xbf,0x02,0x8c,0x36,0x04,0x60,0x99,0x41,0xf5,0xae,
			0x2f,0x18,0xbf,0x1a,0xc3,0x7a,0x24,0xf7,0x55,0xab,0xb9,0xc8,0x5d,0xdc,0xd0,0xbf,
			0x4a,0x12,0xfa,0xbd,0x9d,0x25,0x30,0x29,0xe0,0x81,0xf6,0x28,0xe2,0xbb,0xe9,0xf9,
			0xaf,0xe9,0x22,0x49,0x54,0xd8,0x31,0x5d,0xb8,0x6c,0x21,0x25,0x51,0x2b,0xb9,0x8c,
			0xe9,0xb3,0x69,0x30,0x99,0x4b,0x09,0x1a,0x8a,0x1d,0x7d,0x4e,0x2f,0x4a,0x0e,0x58,
			0xd0,0xa3,0x58,0x76,0xad,0xad,0x14,0x30,0x05,0x30,0xb3,0x9c,0x8d,0xc1,0x1d,0xed,
			0x3e,0xf2,0xfa,0x95,0xd5,0xf2,0x2e,0x67,0xca,0xe3,0x4c,0xc2,0x1a,0xd5,0xe2,0x3f,
			0x91,0x22,0xb5,0x3d,0xfb,0x79,0xf1,0xa2,0xac,0x63,0xc1,0x84,0x4e,0x9e,0xf0,0x69,
			0xa2,0xe4,0x1f,0x17,0x8d,0x6d,0xce,0xdc,0x51,0x8a,0xaf,0xcf,0x81,0xe0,0xeb,0xd8,
			0x82,0x55,0x6e,0x73,0x1c,0xb0,0xab,0x41,0xd9,0x57,0x27,0x4a,0x3f,0xbb,0xb7,0xce,
			0xf2,0x60,0x87,0x91,0x00,0x0c,0x6b,0x86,0x08,0x68,0xcb,0x73,0x93,0xe7,0xd0,0x3d,
			0x94,0x56,0x89,0xff,0xb7,0x75,0x55,0xef,0xe0,0x8f,0x46,0x14,0x51,0xd3,0x3c,0x11),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x31,0x39,0x5c,0xef,0x34,0x95,0x51,0x34,0x3a,0x49,0x27,0x1a,0x8d,0x81,0x2b,0x4c,
			0x7b,0x65,0xb4,0x55,0xb7,0xed,0xa8,0x11,0xfc,0xf7,0x41,0x61,0xf3,0x97,0x11,0x23,
			0x57,0xae,0x44,0x62,0x57,0xbe,0x26,0xc9,0x3c,0xfc,0xe5,0x5e,0x4b,0xa7,0x97,0x6d,
			0xed,0x99,0x7e,0xc1,0x0d,0x1c,0x8b,0x1a,0xc2,0xfe,0x22,0xdc,0x2e,0xe8,0x1d,0x05,
			0xa6,0xeb,0x13,0x61,0x12,0x5c,0xda,0x01,0x97,0xe2,0x4a,0xe9,0x74,0xcd,0x44,0x09,
			0x2a,0xa9,0xf3,0x6f,0xe0,0x13,0x52,0xba,0x05,0xcc,0xef,0xd2,0x37,0x0c,0xee,0xd6,
			0x64,0x19,0x50,0x56,0x2f,0x17,0x76,0xc3,0x95,0x22,0xe0,0x23,0xd0,0x9a,0x3b,0x09,
			0x7b,0xbe,0x9b,0xc5,0xf8,0x7d,0x05,0xd8,0x0f,0x88,0x30,0xab,0xd7,0xac,0x8c,0x80),
		.s = chunk_from_chars(
			0x16,0x2f,0x38,0x76,0x95,0xcf,0x9d,0x82,0xdd,0xa8,0x9c,0x74,0x93,0x18,0xe4,0x6c,
			0x9b,0xe8,0x95,0xec,0x36,0x4e,0xa4,0xae,0xce,0x97,0xcc,0xfa,0x63,0x92,0x5a,0xf3,
			0x71,0x08,0x94,0xda,0x2b,0x7b,0x59,0x67,0xe4,0x6f,0x4e,0xfa,0x80,0xca,0x25,0xd2,
			0xa9,0x65,0xa7,0xe1,0x5f,0x75,0xe0,0xaa,0x1b,0xd4,0x25,0x0f,0x8f,0x41,0x09,0x9e,
			0x6e,0x97,0x14,0xc3,0xfc,0x43,0x11,0x07,0x7a,0xe9,0xbd,0xdf,0xe3,0x5b,0xa4,0x72,
			0x75,0x31,0x52,0x9c,0x23,0x9d,0x54,0x6a,0xb1,0xc2,0x98,0x18,0x7f,0x16,0x5f,0x70,
			0x8c,0xcc,0x0a,0xe3,0x97,0x9a,0x8d,0xa1,0x93,0xe3,0x48,0x59,0xa5,0x9c,0x2c,0x3b,
			0xc4,0x22,0x53,0xc8,0x34,0x66,0x88,0xe6,0xbb,0xa6,0xfb,0x1b,0x01,0xb1,0x0c,0x1e,
			0xc2,0xc6,0x49,0x3d,0xed,0xcc,0x26,0x96,0x26,0x9d,0x85,0x1b,0xde,0x63,0xe2,0x7e,
			0x37,0xbe,0xd3,0x57,0x45,0x5c,0x8f,0xee,0x56,0x29,0xf9,0x4a,0xfa,0x7a,0x98,0x66,
			0x95,0xcf,0xd5,0xb9,0x92,0x12,0x65,0x7a,0x6c,0x88,0x46,0x44,0x59,0x60,0x86,0xb8,
			0x9e,0x0c,0x7c,0x05,0xe8,0x19,0xfa,0xeb,0xeb,0xef,0x74,0x5f,0xd2,0x95,0xaf,0x88,
			0x66,0xe0,0x75,0x0f,0x54,0x79,0xba,0xed,0x50,0xcb,0xb3,0xd0,0x59,0xf8,0xa5,0xeb,
			0x7e,0x0e,0x61,0xe2,0x73,0x3a,0xe5,0x0f,0x0c,0x1e,0xc4,0x2b,0xe7,0x1f,0x5d,0xff,
			0x32,0x41,0x95,0xcb,0x4f,0x0e,0x94,0x1a,0x21,0x56,0x15,0x13,0xc3,0x03,0x7d,0xb9,
			0x2f,0xec,0x95,0x56,0xb7,0x72,0xcc,0xab,0x23,0x9e,0x34,0xb1,0x87,0x6c,0x56,0xb1),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xa7,0xc3,0x09,0xd4,0x4a,0x57,0x18,0x8b,0xbd,0x7b,0x72,0x6b,0x98,0xb9,0x8c,0xe1,
			0x25,0x82,0x22,0x8e,0x14,0x15,0x86,0x48,0x70,0xa2,0x39,0x61,0xd2,0xaf,0xb8,0x2c,
			0xd5,0xbc,0x98,0xbe,0xc9,0x22,0xd5,0xf2,0xac,0x41,0x68,0xb0,0x56,0xda,0x17,0x6e,
			0xf3,0xba,0x91,0xf6,0xb6,0x99,0xba,0x6a,0xcc,0x41,0x44,0x86,0x8f,0xf3,0x7f,0x26,
			0xfd,0x06,0x72,0x08,0x68,0xd1,0x2a,0xd2,0x6e,0xcb,0x52,0x57,0x2c,0xf1,0x04,0x16,
			0xaf,0x68,0xdf,0x03,0xab,0x64,0x5a,0x8b,0x70,0x48,0x57,0xd2,0x19,0x0f,0xfc,0x3f,
			0x07,0xea,0xbe,0x3a,0x8e,0x2a,0xbe,0x34,0xed,0x61,0x59,0xe8,0x84,0xc4,0xfa,0xe1,
			0x41,0xd4,0x33,0x3d,0x5c,0x3e,0x0d,0xb0,0x44,0xff,0x9c,0xcc,0xd9,0xcb,0xd6,0x7f),
		.s = chunk_from_chars(
			0x14,0x8a,0xf6,0x1e,0xd5,0xea,0x8a,0x87,0xa0,0x8b,0x3f,0x40,0x39,0x29,0xbf,0x80,
			0x31,0xdb,0x4f,0xd3,0x99,0x9b,0x64,0x40,0x9b,0xa4,0x89,0xf9,0x7a,0x3e,0xe5,0x20,
			0x8e,0xa4,0x20,0x2d,0x2e,0xc1,0x87,0x34,0xf6,0x15,0x00,0x3a,0x51,0xf7,0x74,0x41,
			0x08,0x5b,0xe6,0xac,0x0f,0x11,0x81,0x0f,0xfa,0x2d,0xad,0x58,0xf0,0xe1,0x86,0xd5,
			0x52,0x0a,0xc2,0xb8,0xa5,0xd3,0x96,0x6e,0x8d,0x2a,0xbb,0x80,0x74,0xe1,0x3b,0x50,
			0xa4,0xe7,0xde,0x83,0xbe,0x10,0xa6,0x6f,0xdc,0x7c,0xa1,0x81,0x18,0xc5,0x77,0x4f,
			0x78,0x12,0x12,0xde,0x9e,0xfe,0xbc,0x63,0x76,0xfc,0xdd,0xdc,0x65,0xa3,0xb1,0xb8,
			0xf1,0xab,0x31,0x49,0x2f,0xe4,0x78,0x25,0x9c,0xe7,0x19,0xb3,0xdb,0x58,0x74,0x98,
			0xd8,0x79,0xa0,0x1d,0xec,0x96,0xe8,0xea,0xbe,0xb0,0x7f,0xf7,0x07,0x3f,0x3f,0x3e,
			0xb4,0x46,0x08,0x49,0x55,0xca,0x26,0x32,0x9a,0x79,0x13,0x15,0xa2,0xc2,0x59,0xd2,
			0x25,0xe2,0x6b,0x21,0x54,0xb2,0x04,0x7b,0x21,0xfa,0xba,0x68,0x11,0x5b,0xfd,0x96,
			0x2e,0x5e,0x24,0xec,0x52,0xd7,0xc5,0xd2,0x31,0xe3,0x04,0x4c,0xbc,0xd8,0xc8,0x80,
			0x48,0x55,0x70,0x3c,0xba,0xa6,0x22,0xb1,0x5b,0x6e,0xf7,0x8c,0x74,0x21,0xa3,0x67,
			0x16,0x6f,0x1b,0x02,0x57,0x6c,0x87,0x36,0x05,0x93,0xda,0x75,0xb7,0x18,0x9e,0xfa,
			0xfd,0x10,0x82,0xbd,0x59,0xf6,0x85,0x7f,0x17,0x01,0xf6,0x46,0xc2,0x4d,0x70,0xc9,
			0x52,0x73,0xc4,0x9d,0x5b,0x11,0xe6,0xaf,0xe2,0x58,0x82,0x1b,0x55,0xc1,0x68,0x0c),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xca,0x50,0x5d,0x45,0x91,0x12,0x16,0x64,0x99,0x07,0x47,0xd9,0x5d,0x95,0x55,0xcc,
			0x75,0xbf,0xc3,0xfd,0xae,0xec,0xee,0xaa,0x60,0xea,0xfa,0xb3,0xfc,0x32,0x0c,0xfc,
			0xe5,0x6e,0xb9,0x13,0x81,0x38,0xbf,0x13,0x8f,0x25,0xf3,0xc8,0xbb,0x02,0x7b,0x13,
			0x6f,0x5d,0x3d,0x90,0xed,0x48,0x97,0x77,0x9b,0x59,0x51,0xc0,0x9d,0xf5,0xd0,0x8b,
			0xa9,0xce,0x8c,0xbe,0x17,0xab,0xc4,0xf0,0x38,0x68,0x70,0x86,0xe9,0x3d,0x77,0x1b,
			0x68,0x43,0x22,0x26,0x66,0x33,0xd0,0xd6,0x5d,0x71,0xec,0x41,0x23,0x4a,0x1d,0xbe,
			0xc0,0x7a,0xbc,0x8f,0x7d,0xf2,0x8b,0xc4,0x3d,0xd8,0xa4,0x5b,0x10,0xce,0xaf,0xac,
			0x06,0x77,0x58,0x05,0x41,0x37,0x01,0x91,0x4e,0x3b,0xb3,0x7e,0xb6,0xba,0x5b,0x5e),
		.s = chunk_from_chars(
			0x58,0x9c,0xcd,0x4e,0xbf,0x97,0x64,0xf8,0x7e,0x6a,0xfa,0x7f,0x13,0xc4,0x06,0x25,
			0x79,0xb0,0x22,0x28,0x11,0x7b,0x15,0xa8,0x73,0x8a,0xb3,0x9c,0xd6,0x44,0x77,0x06,
			0x9c,0xb4,0xf5,0x2c,0xd8,0xd5,0xf4,0x57,0x4c,0x65,0x7b,0x45,0x38,0x35,0xca,0x3c,
			0xed,0xb8,0x24,0xf0,0x3b,0x92,0xa5,0x73,0xd6,0xd3,0xd9,0x13,0x61,0x31,0x3f,0x11,
			0xbd,0xcb,0x34,0xd2,0x05,0x9f,0xe2,0xe6,0xce,0x2b,0x85,0x44,0x61,0xaf,0x58,0xa9,
			0x29,0x4c,0x88,0xcb,0xfb,0x2a,0x63,0x99,0x76,0xb5,0x6e,0x47,0x48,0x02,0x6f,0x30,
			0x40,0xe2,0xfd,0x71,0x12,0xd6,0xad,0x44,0x50,0x06,0x89,0xac,0x77,0x7c,0x07,0x1d,
			0x17,0x39,0x19,0x69,0x76,0x2e,0x18,0x64,0x17,0xc4,0x40,0x0a,0xbd,0xda,0x5c,0x16,
			0xdc,0xe0,0x07,0x76,0x42,0xf1,0xfc,0x13,0x54,0xe0,0xe8,0xc1,0x4e,0x55,0x8c,0x92,
			0x3c,0x1b,0xfb,0x85,0x48,0x8b,0x83,0x50,0xf4,0x15,0x86,0x6a,0x60,0x87,0x1e,0xd7,
			0x15,0x1f,0x5f,0xbc,0x5b,0x88,0x05,0x00,0x01,0x19,0x77,0xc7,0x78,0xe1,0x7f,0xe8,
			0x91,0x8c,0x5d,0x34,0x3f,0x70,0xb0,0x0d,0x58,0xf7,0x18,0x95,0x61,0x25,0xfe,0x28,
			0xb3,0xa5,0xe2,0xd0,0x76,0x04,0xa2,0xb8,0xa8,0x77,0x20,0x44,0x34,0xce,0x90,0x3b,
			0x35,0xa0,0x30,0x93,0x6b,0xc7,0x19,0x51,0xca,0x59,0x3d,0xf9,0x7d,0x24,0xe8,0xe8,
			0xad,0x8f,0x2d,0xc9,0xb7,0x8f,0x76,0xef,0x13,0xa1,0xd3,0x86,0xca,0x85,0x7c,0xed,
			0x48,0xf1,0x9f,0x3e,0xbe,0x39,0x10,0x8f,0x9b,0x33,0xff,0x59,0xeb,0x05,0x56,0xb1),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x23,0x7a,0x7e,0x44,0xb0,0xa6,0xc2,0x68,0xbb,0x63,0x36,0x4b,0x95,0x8a,0xe0,0x2b,
			0x95,0xe7,0xee,0xd3,0x6b,0x3e,0xa5,0xbf,0xb1,0x8b,0x9b,0x81,0xc3,0x8e,0x26,0x63,
			0xd1,0x87,0x14,0x4e,0x32,0x3f,0x9c,0xea,0xfb,0x47,0x95,0x07,0xd1,0x84,0xe6,0x3c,
			0xfb,0xec,0x3e,0xcd,0xbb,0x8a,0x05,0xd2,0xdf,0xc8,0x92,0x96,0x93,0xed,0x9e,0x3e,
			0x79,0xe5,0xf8,0xab,0xfc,0x41,0x7b,0xa1,0xe1,0x7e,0x3e,0x28,0x1e,0x8a,0x0a,0x32,
			0xf0,0x84,0x11,0x7f,0x28,0xc3,0xdc,0xbe,0xc5,0x1b,0x86,0xf5,0xc8,0x5b,0x28,0x22,
			0x44,0x1a,0x94,0x23,0xb5,0xb4,0x46,0xd3,0x92,0x8f,0x97,0x76,0x26,0xa3,0x34,0x57,
			0x9b,0x39,0xcf,0xaf,0x58,0xf2,0x14,0xc9,0x8d,0x0c,0xdf,0x64,0x0b,0xe1,0xac,0x59),
		.s = chunk_from_chars(
			0xaf,0x07,0x6b,0xc2,0x13,0xca,0xf7,0x56,0x19,0xf4,0xbd,0x1d,0x78,0x7c,0xc1,0x98,
			0xf7,0xdf,0x33,0x24,0xa0,0xdd,0x87,0xa8,0x84,0x16,0xe0,0xa4,0xb8,0x1c,0x2f,0xb9,
			0xa9,0xdb,0x5f,0x98,0xae,0xd4,0x3b,0xc1,0x5f,0xe2,0x35,0x71,0x43,0xa6,0xe4,0xff,
			0x70,0x1d,0x9c,0x48,0xf5,0x1d,0xe9,0xeb,0x80,0x36,0x70,0xbb,0xc4,0xb0,0xae,0xa7,
			0x22,0x0b,0xe2,0xf8,0x4b,0x83,0x00,0x31,0x8c,0x77,0xa9,0xf6,0x15,0x98,0x6c,0x49,
			0x80,0xab,0xda,0x85,0xe3,0xad,0x00,0x89,0x56,0x4d,0xba,0xf7,0xf4,0x4d,0x81,0xb6,
			0x66,0x4e,0xec,0x03,0x11,0xad,0xb1,0x94,0xd4,0x6d,0xe9,0x6b,0xb1,0x7d,0x5a,0x5d,
			0x47,0x42,0x68,0x45,0x80,0x2c,0xa0,0xf4,0x9a,0x16,0x9e,0xb8,0x2b,0x75,0xaf,0xa1,
			0x91,0x02,0x7a,0x0c,0xc8,0xfc,0xe9,0xdd,0x16,0x05,0x53,0x50,0xdf,0x97,0x45,0xfc,
			0x72,0x00,0xff,0x9f,0x4e,0xa3,0xcf,0xbf,0xc6,0x6c,0x42,0x84,0x81,0x13,0xe3,0xbe,
			0x32,0x93,0xd5,0x10,0x38,0x2d,0x09,0x99,0xf0,0x32,0x51,0x55,0x27,0xbd,0x99,0xf6,
			0x6e,0xfa,0x2a,0x75,0x5e,0x01,0x12,0x47,0xb2,0x23,0xa6,0x8e,0x51,0x25,0x8b,0x6b,
			0xc3,0x19,0xa7,0xcd,0xef,0x4a,0xec,0x53,0x3e,0x9d,0xcd,0x8a,0xe2,0x6e,0x34,0x9e,
			0x5b,0x33,0xc7,0x91,0x21,0x90,0x7d,0xe5,0x09,0xa1,0xcb,0x83,0xc2,0xe5,0x9a,0x47,
			0xc1,0xa8,0x84,0xbf,0x68,0xe7,0x22,0x93,0x16,0xa6,0x2e,0x3c,0x49,0xd1,0xf5,0x42,
			0xeb,0xe7,0x10,0x5c,0xfc,0x27,0x09,0x92,0x68,0x12,0x0a,0x77,0x43,0x90,0x84,0x71),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xab,0x18,0x93,0x92,0x30,0xb0,0x96,0x64,0x6a,0x37,0xa7,0x81,0x62,0x9f,0xbd,0x92,
			0x70,0xf3,0x89,0x1a,0x5c,0xea,0xb4,0xa8,0xc3,0xbc,0x68,0x51,0xbc,0x34,0x11,0x5d,
			0xbc,0x06,0x65,0x41,0xb7,0x64,0xa2,0xce,0x88,0xcc,0x16,0xa7,0x93,0x24,0xe5,0xf8,
			0xa9,0x08,0x07,0x65,0x2c,0x63,0x90,0x41,0x73,0x3c,0x34,0x01,0x6f,0xd3,0x0a,0xf0,
			0x8f,0xed,0x90,0x24,0xe2,0x6c,0xf0,0xb0,0x7c,0x22,0x81,0x1b,0x1a,0xe7,0x91,0x11,
			0x09,0xe9,0x62,0x59,0x43,0x44,0x72,0x07,0xdc,0xd3,0xff,0xf3,0x9c,0x45,0xcb,0x69,
			0xee,0x73,0x1d,0x22,0xf8,0xf0,0x08,0x73,0x0c,0xe2,0xef,0xc5,0x3f,0x11,0x49,0x45,
			0x57,0x3e,0xa2,0xdd,0xeb,0xb6,0xe2,0x62,0xc5,0x27,0xd2,0x0f,0x8b,0xb1,0xdc,0x32),
		.s = chunk_from_chars(
			0x95,0xbd,0x0b,0xf2,0x36,0x2f,0x34,0xb2,0xe0,0x40,0x75,0xb2,0x93,0x4f,0x40,0x47,
			0x98,0x70,0x3e,0xa4,0x72,0xb8,0x1a,0xc3,0xcc,0x22,0x3a,0xec,0x48,0x6e,0x4c,0x3d,
			0x9c,0x5d,0x1c,0x2f,0x9e,0xe2,0x24,0x17,0x13,0x29,0x64,0xed,0x58,0xe4,0x99,0x37,
			0xf5,0xb2,0x57,0xd3,0x16,0xca,0x7f,0xff,0xe2,0x90,0xb1,0x9f,0x5b,0x58,0x10,0x38,
			0x36,0x81,0x2b,0xef,0x30,0xca,0x03,0x27,0x03,0x9d,0x8b,0x9e,0xa9,0x12,0x95,0x39,
			0x2f,0xc3,0x94,0xb8,0x81,0xe2,0xd2,0xac,0x9e,0x30,0xc5,0xa4,0x42,0x56,0x70,0x0f,
			0xc9,0xde,0x0d,0xba,0x29,0x82,0x73,0xae,0xc3,0x0c,0x4f,0x77,0x8d,0x2e,0x71,0x27,
			0xe8,0xb8,0xa8,0x8b,0x02,0x74,0xfc,0xe0,0x40,0x81,0xcc,0x13,0xad,0xbe,0xfe,0x55,
			0x50,0x14,0xe1,0xb5,0xd5,0xdc,0xf6,0x22,0x4c,0x5a,0xe2,0x77,0x54,0x23,0xa6,0x6c,
			0x81,0x81,0x8e,0xec,0x01,0x4a,0x3f,0xaf,0x9e,0xe7,0x5a,0x3f,0x6c,0x3e,0x51,0xc5,
			0x56,0xb0,0xa2,0x88,0xe8,0xc2,0x62,0x94,0x66,0x84,0xeb,0x62,0x8b,0x88,0xe3,0xf8,
			0x75,0xe6,0x2e,0xf6,0xe8,0x01,0xca,0xe7,0x5f,0x61,0xce,0xe4,0x04,0x97,0x1c,0x39,
			0xd2,0x4a,0x97,0x12,0xeb,0x34,0x2d,0xdc,0x66,0x35,0x15,0xde,0xc1,0x03,0xb1,0x8d,
			0x97,0xd7,0x8e,0xd6,0x82,0x12,0xf2,0x79,0x00,0xe7,0x7c,0x04,0x9b,0x60,0xc8,0x53,
			0x00,0x2b,0x08,0x02,0x2d,0xf5,0x6f,0x70,0x7e,0xfa,0x71,0x02,0x75,0x89,0xe1,0xa3,
			0xca,0x6e,0x41,0x5b,0xa5,0xf4,0x43,0x7e,0x97,0x8b,0x07,0xaf,0x3b,0x73,0xba,0x0d),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xa2,0x80,0xe8,0x9c,0xeb,0x2c,0x8c,0xf2,0x62,0x97,0x19,0x1b,0xaf,0x9a,0x95,0x5d,
			0x0d,0x52,0x37,0x5d,0xa0,0x23,0x63,0x3e,0x0a,0xfc,0xdb,0x0d,0x39,0xdc,0x33,0x5d,
			0x82,0x95,0x85,0x2e,0xf4,0xd0,0x67,0x14,0xe6,0x51,0x1a,0x95,0xd3,0x7c,0x04,0xd2,
			0x68,0x18,0x60,0x6a,0xda,0x54,0x35,0x9b,0x7d,0x07,0x84,0xaa,0x93,0x3c,0xc6,0x85,
			0x61,0xee,0x96,0xa8,0x89,0x10,0xaa,0x3d,0x93,0xd1,0x07,0x87,0xcd,0x1d,0x75,0x80,
			0x55,0x67,0x31,0xc1,0x74,0xa6,0xe3,0xa3,0x2d,0x9d,0xcf,0xa4,0x16,0x60,0x4f,0x0c,
			0x67,0x14,0x81,0xd0,0x51,0xf6,0x3d,0xb6,0x91,0x9f,0x4a,0xba,0x44,0x86,0xd1,0xb0,
			0xfd,0xc6,0x11,0x2c,0x15,0x21,0x55,0x9f,0x42,0x45,0x23,0xc2,0x6b,0x4f,0xb7,0x38),
		.s = chunk_from_chars(
			0xcd,0x60,0xde,0x3b,0x4a,0x12,0x89,0xa8,0x4c,0xa7,0x61,0xf9,0x0f,0xa6,0x3f,0x4d,
			0x56,0x88,0xbd,0x88,0x5f,0x4b,0x53,0x1c,0x85,0x15,0xad,0xd2,0xde,0x12,0x51,0xf9,
			0x93,0xff,0x7f,0x98,0x6b,0xef,0x3f,0xba,0x69,0x2e,0xcd,0xeb,0xc8,0x19,0x42,0xd7,
			0x42,0x9c,0x7a,0x59,0xc5,0xd3,0xf1,0xfb,0x87,0x2f,0xc1,0xda,0x19,0x15,0xe9,0x45,
			0x86,0xa5,0xc3,0xd9,0x63,0x60,0x36,0x19,0x00,0x8f,0x7e,0xfe,0xde,0xd1,0xd7,0x0b,
			0x0a,0x11,0xce,0x2c,0xd8,0x1b,0x5b,0x0d,0x86,0xb3,0x76,0x0c,0x94,0x83,0x67,0x4f,
			0x55,0xe9,0xfa,0x47,0xf2,0xf3,0x10,0xd5,0x88,0xfb,0x21,0x60,0xe8,0xb5,0xc3,0x2b,
			0xe4,0xe7,0xa9,0x68,0xd5,0xa8,0xd4,0xac,0x65,0x76,0xb7,0x1a,0x2b,0x91,0xcd,0x6a,
			0xf0,0x01,0x6c,0xbc,0x81,0x6d,0x4a,0xae,0x8c,0x70,0x64,0x9e,0x08,0xdc,0xe9,0x0b,
			0x3c,0xe5,0x2a,0xb4,0x9c,0xe2,0xcb,0x5b,0x0e,0xd8,0xa4,0x5e,0x33,0xd9,0x4c,0xf2,
			0xd4,0xcf,0xde,0xe1,0x15,0x12,0x70,0xb2,0x07,0x3a,0xef,0xfe,0xaf,0x71,0x7d,0x39,
			0xe0,0x41,0x92,0xb8,0xb6,0x93,0xc5,0x3f,0x21,0xa6,0x12,0x38,0x13,0x28,0x08,0x06,
			0x92,0x0b,0x7d,0xc5,0x82,0x20,0x1c,0x9d,0x11,0x70,0x50,0x32,0x06,0x71,0xe8,0x61,
			0x39,0xa0,0x27,0x97,0x6b,0x7e,0xcf,0x41,0x33,0x69,0xa9,0xfc,0x28,0xe0,0xbd,0x71,
			0x9c,0xeb,0x5e,0x10,0x7d,0xe7,0x99,0xf1,0xbc,0x2e,0x25,0x5a,0x9f,0x29,0x47,0x6d,
			0x45,0x74,0xd1,0x33,0x2f,0x66,0x46,0x8a,0xfb,0x90,0x04,0xff,0x7b,0x53,0x53,0x02),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x85,0xed,0x1e,0x3d,0xfc,0xd5,0xbc,0xa2,0x4c,0xad,0x1d,0x01,0xeb,0xe1,0x92,0xb7,
			0xd0,0x59,0xec,0x9b,0x88,0x44,0x36,0xe1,0x87,0x14,0xa4,0x3f,0xbc,0xc9,0xc6,0x4f,
			0x68,0x73,0x01,0x35,0x2f,0xf2,0x40,0x81,0x70,0x01,0xe7,0x57,0xd2,0x73,0x09,0xcd,
			0x1f,0xbb,0xda,0x94,0x56,0xb2,0x67,0xdb,0xfb,0x95,0x84,0x70,0xb2,0x4d,0x06,0x28,
			0x0c,0xf4,0x33,0x82,0xa1,0x94,0x77,0x87,0x5f,0x32,0x59,0xf4,0x21,0x0b,0xac,0x9b,
			0x83,0x1d,0x0a,0x07,0xf5,0xe9,0x7e,0x5f,0x0f,0x78,0x81,0x8c,0x25,0x9c,0x28,0x9e,
			0x1a,0x78,0x9b,0x6c,0x79,0x42,0xc9,0x7b,0xc1,0x48,0x5a,0x22,0x01,0x31,0xe5,0xeb,
			0xa5,0x86,0x64,0x3b,0x90,0x71,0xe5,0x36,0x6b,0xc4,0x82,0xdd,0x3c,0x3c,0x92,0x79),
		.s = chunk_from_chars(
			0x13,0x81,0x34,0xbb,0xec,0xef,0xaf,0xc7,0xca,0x8b,0x10,0x2c,0xbe,0x87,0xb0,0x12,
			0xf8,0xaa,0xda,0x88,0x78,0x99,0x50,0x02,0xcf,0x18,0x87,0x69,0x4b,0x5b,0xe3,0xb8,
			0xf0,0xbb,0x61,0x6b,0xc6,0xe0,0x79,0x62,0xd5,0x48,0x2d,0x3a,0x52,0xc5,0x2a,0xb9,
			0x1b,0x3e,0xe0,0x06,0x4d,0x24,0x55,0x8e,0x13,0xc7,0x5c,0x80,0xf6,0xa9,0x5b,0x7d,
			0xc4,0x98,0x44,0x28,0x79,0xd5,0xba,0xf8,0xff,0xa7,0xe2,0xf6,0x38,0x80,0x8b,0x97,
			0xff,0x70,0x13,0x6b,0xb6,0x45,0xe3,0x09,0x44,0xdd,0x97,0xa9,0x97,0xa0,0x20,0x51,
			0x69,0x55,0x3a,0x5b,0x9e,0x87,0x4c,0x5a,0x94,0x41,0xe1,0x8c,0x15,0xeb,0xed,0x76,
			0x04,0x3b,0x63,0x9d,0xfd,0x64,0xdb,0x79,0xe1,0x74,0x84,0x7a,0x10,0x27,0x24,0xa2,
			0xa0,0x5c,0x64,0x94,0x73,0xcc,0x7d,0xac,0xd3,0x9e,0x2e,0x1d,0x56,0x66,0xbb,0xb5,
			0xf0,0x12,0x46,0x74,0x70,0x48,0xff,0xfc,0xdf,0xcd,0xdf,0x78,0x2d,0xa2,0x4a,0x6d,
			0xcc,0x02,0x2b,0x26,0x95,0xf7,0x07,0x81,0xbd,0x9f,0x8f,0xf7,0xd0,0x3b,0xe2,0x2e,
			0xb8,0xfc,0x79,0x3f,0x5c,0x07,0x1a,0x66,0xd9,0xa6,0xea,0x46,0xc6,0xa2,0xcf,0x05,
			0x56,0x52,0x6b,0xa8,0xb0,0x85,0x07,0x35,0x46,0x44,0x80,0x81,0x73,0x2a,0xc1,0x5f,
			0x12,0x83,0x3c,0x1d,0xb1,0x70,0x1f,0xf7,0xf6,0x83,0x44,0xca,0x65,0xdf,0xf8,0x62,
			0x11,0xa0,0x03,0xad,0xbf,0x51,0x89,0xcf,0xae,0x79,0xea,0xa8,0xc8,0xb7,0x14,0x1e,
			0xa3,0x78,0xe4,0x4c,0xc9,0xc5,0xbf,0x02,0x4d,0x2c,0x71,0x0f,0xf5,0xcd,0x68,0xaf),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x0b,0xdb,0xa3,0x4e,0x35,0xfc,0xa6,0x5a,0x17,0x81,0xd4,0xd7,0xc9,0x33,0xa5,0xf2,
			0x10,0xd3,0xa5,0x94,0x83,0xae,0xbc,0x95,0xec,0x71,0xb3,0x2d,0xf1,0x3f,0xf4,0xab,
			0xf4,0x01,0x91,0x69,0x37,0xfd,0x88,0xff,0x44,0xab,0x46,0xb7,0x8c,0xc3,0x69,0x41,
			0x4e,0x9b,0xca,0xa8,0xba,0xb0,0xbb,0x85,0x57,0x82,0x8d,0x73,0xa2,0xa6,0x56,0xc2,
			0xf8,0x16,0xf0,0x70,0xb5,0xcb,0x45,0x54,0x9e,0x8e,0xca,0x9d,0x7c,0x0b,0x4a,0x7b,
			0x0a,0x27,0xe5,0x1c,0x11,0x93,0x58,0xda,0xd2,0xa1,0x7f,0xb3,0xa4,0x57,0x18,0xf9,
			0xde,0xc3,0xc9,0x4a,0xf7,0x8d,0x65,0xc3,0xec,0xd3,0x6b,0x71,0xe2,0x30,0xcf,0x08,
			0x0d,0x1e,0xfd,0xd8,0xd0,0x7f,0x1c,0xfc,0x26,0x76,0x8f,0xd5,0x40,0x7b,0xc2,0xb7),
		.s = chunk_from_chars(
			0x9f,0x48,0xde,0xb9,0x6b,0xec,0x0b,0x72,0xfb,0xc4,0xf1,0x2f,0x08,0xaf,0xb4,0x6b,
			0xcc,0xf1,0x9d,0x9e,0x0c,0xd0,0x36,0x8e,0xbe,0xb3,0x12,0xd8,0x38,0x72,0x62,0x63,
			0x80,0xac,0x92,0x8b,0x61,0x2c,0x5c,0xd7,0x74,0x38,0xd4,0x7a,0xa9,0xce,0xea,0x90,
			0x5a,0x9d,0xe7,0x18,0x2c,0x8e,0xf7,0x6e,0x8a,0x7a,0x03,0xd6,0xef,0xec,0x84,0x00,
			0xb6,0x49,0x63,0x62,0xbf,0x6a,0x30,0xce,0xb1,0xce,0xd2,0x18,0x5f,0xc7,0xc2,0x11,
			0x7b,0x6a,0x6d,0x88,0x8a,0xc2,0x0c,0x16,0x87,0xb0,0xf2,0xaa,0x9b,0x76,0x70,0x5f,
			0xd3,0x15,0x48,0x89,0xb6,0xac,0xaf,0x4e,0x63,0xbe,0x25,0x88,0x0c,0x71,0xe6,0xc2,
			0x39,0xec,0xfb,0x96,0x50,0x04,0xcd,0x63,0x21,0x25,0x7f,0x84,0x6a,0xfd,0x2a,0x65,
			0x90,0xc7,0x2a,0xd8,0x31,0x46,0xee,0xfc,0x7b,0x0d,0xc4,0x79,0x63,0x39,0xa7,0xf6,
			0x4d,0xa0,0xfb,0xe3,0x59,0xf9,0x4a,0xce,0x1f,0xd1,0x51,0xc5,0xac,0x7b,0xb5,0x70,
			0x7b,0x32,0xea,0xcf,0x56,0x4f,0xe1,0x62,0x2e,0x66,0xe1,0x84,0x4e,0x63,0x96,0x02,
			0xca,0x36,0x27,0x4a,0xe0,0x1f,0x93,0xe6,0xb2,0xbd,0x1e,0xff,0xd3,0x4a,0xb6,0x3d,
			0x85,0x2c,0xc9,0xca,0xf3,0xce,0x84,0x46,0xc2,0x9c,0x8a,0xe3,0xc6,0x11,0x0f,0xb7,
			0x53,0x8c,0xc8,0x37,0x1c,0x2a,0x39,0x81,0x24,0x9c,0xdc,0x1b,0xe2,0xb2,0x4b,0x6a,
			0x0c,0x95,0x17,0x64,0xd0,0xb7,0xef,0xa9,0x2a,0x22,0xcd,0x8e,0xd1,0x65,0xe1,0x82,
			0x86,0x35,0x79,0x37,0x79,0x97,0xa9,0xee,0x50,0xc8,0xac,0x3a,0xa4,0xdf,0x1a,0xca),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x9a,0xee,0xd8,0x5b,0x40,0xba,0x7f,0x86,0xa2,0x28,0xb5,0xa1,0x51,0x5b,0xa1,0x90,
			0xb2,0xef,0xff,0x66,0x99,0x3a,0x5e,0xce,0x19,0xd1,0x8b,0xaa,0x9b,0x4e,0x4d,0xf9,
			0x2e,0x51,0x52,0xfe,0x1e,0xc5,0x6a,0x9f,0xc8,0x65,0xf3,0x0b,0xac,0x7e,0x94,0x9f,
			0xc4,0xf6,0x2f,0x0b,0x15,0x8d,0x10,0xb0,0x83,0x63,0x6b,0x4d,0xe9,0xbb,0x05,0xdb,
			0x69,0xfe,0x31,0xb5,0x01,0x03,0xfe,0xfc,0x5f,0x8d,0xaf,0x3a,0xf7,0x15,0x6b,0x45,
			0x52,0xca,0x36,0x67,0xa9,0xd7,0x20,0xbb,0xb2,0xe4,0xbc,0xda,0xba,0xdf,0xd4,0xb7,
			0xf4,0xfc,0x5b,0xc8,0x11,0xfa,0xa3,0x67,0x10,0xa9,0xd1,0x77,0x58,0xa9,0x8d,0x4a,
			0x04,0x74,0xfe,0xc2,0x7e,0x9e,0xf5,0xb7,0x4f,0x5c,0x68,0x99,0x35,0x44,0x23,0x57),
		.s = chunk_from_chars(
			0x9e,0xec,0xdb,0xd7,0xfb,0xf6,0x18,0xdd,0xdd,0xfb,0x6e,0x75,0xd6,0x44,0x40,0xf6,
			0x04,0x45,0xb8,0x53,0xc5,0x42,0xfe,0x0f,0xba,0xaa,0x6a,0x43,0x12,0x94,0xe6,0xcb,
			0x66,0x83,0xae,0x1a,0x71,0xea,0x05,0x5e,0xb4,0x9c,0xd2,0xa3,0xcb,0x51,0x54,0xdc,
			0x93,0xd9,0xaa,0x16,0x63,0x99,0xf4,0xe6,0x29,0x4f,0x0e,0xb0,0x65,0x28,0x00,0xd7,
			0x1e,0x04,0x1c,0x1c,0xe1,0xad,0x84,0x9c,0x03,0xc9,0x63,0xbc,0x09,0x29,0xdc,0xdd,
			0x11,0xbe,0x5d,0x67,0xa0,0x50,0xd0,0x2b,0x64,0xb2,0x9e,0xab,0xa6,0x55,0x64,0x2b,
			0x64,0x36,0xfb,0xfb,0x16,0x36,0x90,0xbf,0x43,0x2f,0xdc,0xee,0xdd,0x10,0x6c,0x2f,
			0x49,0x72,0xec,0xbf,0x30,0x77,0xed,0x8b,0x75,0x3b,0xb6,0x05,0xec,0x1e,0xa0,0x30,
			0x20,0x83,0x9a,0x31,0x8a,0x24,0xf8,0xd4,0xc1,0xd7,0xd8,0xdf,0x99,0xa7,0xf0,0x01,
			0x0a,0xe4,0x1a,0x8b,0x06,0x8e,0x28,0x88,0x53,0x10,0x56,0xa7,0xda,0xbb,0xe9,0x21,
			0x87,0x8d,0xcd,0x3c,0x7d,0x69,0x41,0x68,0x67,0xf4,0x01,0x2a,0x60,0x6a,0xe8,0x68,
			0x55,0xf1,0x5a,0xed,0x0d,0xa1,0x25,0x0e,0x59,0x68,0x77,0x06,0xe8,0x9c,0x94,0x94,
			0xba,0xf3,0x7f,0x61,0xfb,0x17,0x03,0xb7,0x99,0x28,0x79,0x5f,0x90,0xcc,0xbe,0x29,
			0x3a,0x1e,0x94,0x72,0xf6,0xe0,0xf4,0xb8,0x90,0xfd,0xda,0x3e,0xa2,0x52,0x2e,0x3d,
			0x11,0xd5,0xab,0xdf,0x00,0x69,0x51,0x94,0x24,0xd1,0x47,0xb5,0x64,0x6a,0x5a,0x60,
			0x1f,0x19,0xec,0x89,0x72,0x9a,0x8b,0x48,0x46,0x1e,0x71,0xc0,0x8b,0xbe,0x9c,0xda),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x65,0x4e,0x18,0x9f,0x06,0xc7,0xd4,0x2d,0x55,0x39,0xa5,0x87,0x21,0x84,0xf8,0x33,
			0x6c,0xf1,0x00,0x69,0x1f,0x19,0x08,0x18,0xfd,0x02,0x08,0x2a,0xd6,0x8a,0x76,0x09,
			0xfd,0x09,0x5e,0x62,0xfc,0x32,0xb5,0x29,0x85,0x3a,0xeb,0xdd,0xac,0x3d,0xbf,0x0d,
			0x54,0xdd,0x57,0x1b,0xe7,0x2c,0x90,0x40,0x4b,0xcc,0x93,0xd0,0x11,0x54,0xa9,0xbf,
			0xef,0xf6,0x50,0x65,0x70,0x5f,0x8e,0x7e,0xea,0xdf,0x85,0x75,0xb1,0xca,0x48,0xe2,
			0x8a,0x1e,0xed,0x51,0x62,0x65,0xe3,0x45,0x40,0xdd,0x86,0x7c,0x79,0xd7,0xf1,0x75,
			0x23,0x5d,0x13,0x30,0xcb,0x17,0x06,0x35,0x6b,0x70,0x9b,0xd7,0x96,0xf4,0x3a,0xba,
			0xf6,0xfc,0xe9,0x93,0xf8,0x8e,0xaa,0x2f,0xc6,0x7f,0x0a,0xb7,0x76,0xda,0xf7,0x32),
		.s = chunk_from_chars(
			0xaf,0x90,0x29,0x8b,0xce,0xf6,0x15,0x30,0x9f,0x23,0x5d,0x5c,0x33,0x60,0xf0,0xdf,
			0x11,0xf5,0xfb,0x98,0x87,0x89,0xf2,0x13,0xd4,0xc4,0x61,0x34,0xfe,0xe5,0xeb,0x10,
			0x4a,0xa1,0xfa,0xbb,0x13,0x07,0xc9,0xa9,0x04,0x70,0x9d,0xe8,0x86,0x73,0xed,0x99,
			0x51,0xcb,0xa9,0x31,0x67,0xc6,0x7c,0x09,0xd8,0x27,0x02,0x1b,0x08,0xa2,0x2c,0x05,
			0x05,0x82,0x8a,0xb4,0xbe,0xb4,0x2e,0x59,0xa3,0x88,0x32,0xcb,0x4d,0xa2,0x4e,0xcf,
			0x91,0xf4,0x70,0xa3,0xb4,0x12,0xc0,0x71,0x2a,0x8a,0x59,0xf6,0xf2,0x73,0x9d,0x4e,
			0x9e,0xb4,0xcc,0x58,0xd2,0xc5,0x25,0x92,0xf1,0x45,0x2d,0xc6,0x57,0x59,0xab,0xe4,
			0x3e,0x8d,0x2b,0xc8,0x04,0xe2,0xef,0xb3,0xef,0xc9,0xb2,0x3c,0xc1,0x73,0x4f,0xf7,
			0xca,0xef,0xa4,0x6b,0x03,0xba,0x4b,0x39,0x7d,0x07,0x14,0xcd,0xb8,0x50,0x1a,0x81,
			0x2c,0x1b,0x9f,0x47,0x41,0x1c,0x91,0xcb,0xa5,0x3a,0x3d,0x3b,0x13,0x9e,0xdb,0xd7,
			0xcb,0xb5,0x43,0xf5,0xbf,0x38,0x29,0xba,0x7f,0x5f,0xaf,0xd8,0xa7,0x12,0xc0,0xb1,
			0x11,0x94,0x3f,0x53,0x20,0x93,0x53,0xaf,0xab,0xa1,0x76,0xb3,0xf5,0xdc,0x06,0x03,
			0x39,0xd0,0x9b,0x1f,0xb3,0xc2,0x13,0xda,0xe5,0xd0,0xf0,0x04,0xd3,0x02,0x82,0x85,
			0x60,0xfb,0x5d,0xeb,0xf9,0xfe,0x49,0x1e,0xaa,0x66,0xf5,0x97,0xaa,0x4d,0xe2,0x3e,
			0xee,0xf9,0x17,0x63,0x58,0x75,0x5c,0x95,0x2e,0xf9,0x6e,0x36,0x72,0x58,0x3b,0x6e,
			0xcd,0x95,0xa0,0x2e,0x8c,0xa7,0xb2,0x1d,0x7c,0x20,0xcb,0xb7,0xa7,0x57,0xaf,0x71),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x12,0x1f,0x80,0xb4,0x3f,0x97,0x57,0xb3,0xfa,0x80,0x90,0x6a,0xea,0xb2,0x32,0x19,
			0x5f,0x0e,0x2c,0x41,0xe5,0xbf,0x8c,0x09,0x1a,0xc0,0xf1,0xe0,0xbc,0x9e,0x43,0x64,
			0x06,0x80,0xa1,0x82,0x3d,0x64,0x9b,0xdf,0x86,0xab,0xa2,0x77,0xfa,0xd8,0xbc,0x85,
			0xfc,0x95,0x7d,0xa2,0xca,0xf7,0x32,0x30,0x53,0x02,0x5f,0xf9,0x49,0x70,0x6c,0x14,
			0x76,0xae,0x9b,0x09,0x53,0x28,0x3d,0x34,0xd7,0xc6,0x26,0x6f,0x8d,0xb6,0x5e,0xeb,
			0xe9,0x6d,0x19,0x5f,0xdc,0xe8,0xe9,0x65,0xa6,0x38,0x33,0x20,0xec,0x3d,0xe0,0x23,
			0x0a,0xb2,0x54,0x8e,0xaa,0x69,0xa4,0x7a,0x96,0xd8,0x03,0x98,0xca,0xd5,0x7e,0x14,
			0xce,0x9e,0xea,0xc0,0x42,0x1c,0x1a,0x6e,0xba,0x69,0x55,0x9d,0xcd,0x8f,0x06,0x59),
		.s = chunk_from_chars(
			0x06,0xa2,0xd7,0x45,0x85,0xf1,0x2e,0xa7,0xa8,0x05,0x27,0xb8,0xc6,0x35,0xa2,0x1c,
			0xc1,0x1b,0x45,0xdb,0xb0,0x88,0x5a,0x12,0x72,0x21,0x26,0x81,0x1d,0xd2,0x5d,0x65,
			0x7b,0xfa,0x9f,0xda,0x77,0x43,0x01,0xca,0x34,0x98,0xd0,0x5d,0xfd,0xfb,0x78,0xa6,
			0xaa,0x16,0xa9,0xf8,0xa9,0x5f,0x40,0xf1,0xf0,0x4b,0xd3,0x54,0xa5,0x22,0xf6,0xa2,
			0xd6,0x2b,0x32,0x4e,0xfa,0x3c,0x00,0x6c,0x22,0xc2,0x31,0x4b,0x01,0xfa,0x0e,0x91,
			0xa3,0xdb,0xa4,0x9a,0xa3,0x5b,0x46,0xb1,0x98,0x04,0xb0,0x7a,0xd9,0x8f,0xe4,0xbc,
			0x99,0x03,0x93,0xa4,0xa2,0x73,0xce,0x8f,0x1c,0x85,0xfc,0x19,0xcd,0x5e,0xae,0x9a,
			0xf0,0xb7,0xd1,0x95,0x7b,0xb2,0x34,0x09,0x77,0x8a,0x01,0x0b,0x00,0xc6,0x95,0x9e,
			0x1b,0x67,0x06,0x6f,0xdb,0x9f,0x84,0x95,0xb4,0xde,0x4d,0xcb,0xb9,0x87,0x35,0x81,
			0x45,0xb1,0xff,0x6a,0x39,0xef,0x6f,0xc5,0x88,0xcd,0xa1,0x74,0x4e,0x0a,0xb9,0xe7,
			0xeb,0x00,0x2c,0x29,0xa7,0x85,0x31,0xd2,0x51,0x57,0xc5,0xc2,0xcd,0x64,0x70,0x55,
			0x15,0x60,0xa0,0x28,0x45,0xdb,0x6d,0xbe,0xe2,0x42,0xf9,0x65,0xa2,0x55,0x40,0x6f,
			0x6e,0xf4,0x7b,0x32,0x21,0xa5,0x11,0x0e,0xdb,0x44,0xd3,0x8b,0x94,0x19,0x1a,0xea,
			0xf4,0x33,0xc0,0xec,0xe3,0x48,0x0b,0x9d,0x1b,0x06,0xd8,0xb8,0xb6,0xc0,0xa2,0x32,
			0xa0,0x4c,0x56,0x78,0x88,0xe6,0x37,0x2f,0x2e,0x94,0xbc,0x2b,0xe6,0xb8,0x27,0xf8,
			0x71,0x2a,0xf4,0x8c,0x6f,0x1e,0x4f,0x22,0x3f,0x55,0x28,0xfc,0xf3,0x48,0x79,0x9d),
	},
};

START_TEST(test_sign_pkcs15_2048)
{
	signature_scheme_t scheme = pkcs15_2048[_i].scheme;
	chunk_t sig;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, scheme)) ||
		!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PRIVKEY_SIGN, scheme)))
	{
		warn("%N not supported, skip test", signature_scheme_names, scheme);
		return;
	}
	fail_unless(privkey_2048->sign(privkey_2048, scheme, NULL, pkcs15_2048[_i].m, &sig),
				"sign %N", signature_scheme_names, scheme);
	ck_assert_chunk_eq(pkcs15_2048[_i].s, sig);
	fail_unless(pubkey_2048->verify(pubkey_2048, scheme, NULL, pkcs15_2048[_i].m, sig),
				"verify %N", signature_scheme_names, scheme);
	free(sig.ptr);
}
END_TEST

/**
 * Test vectors from FIPS 186-4
 */
static struct {
	signature_scheme_t scheme;
	chunk_t m;
	chunk_t s;
} pkcs15_3072[] = {
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xbc,0xf6,0x07,0x43,0x33,0xa7,0xed,0xe5,0x92,0xff,0xc9,0xec,0xf1,0xc5,0x11,0x81,
			0x28,0x7e,0x0a,0x69,0x36,0x3f,0x46,0x7d,0xe4,0xbf,0x6b,0x5a,0xa5,0xb0,0x37,0x59,
			0xc1,0x50,0xc1,0xc2,0xb2,0x3b,0x02,0x3c,0xce,0x83,0x93,0x88,0x27,0x02,0xb8,0x6f,
			0xb0,0xef,0x9e,0xf9,0xa1,0xb0,0xe1,0xe0,0x1c,0xef,0x51,0x44,0x10,0xf0,0xf6,0xa0,
			0x5e,0x22,0x52,0xfd,0x3a,0xf4,0xe5,0x66,0xd4,0xe9,0xf7,0x9b,0x38,0xef,0x91,0x0a,
			0x73,0xed,0xcd,0xfa,0xf8,0x9b,0x4f,0x0a,0x42,0x96,0x14,0xda,0xba,0xb4,0x6b,0x08,
			0xda,0x94,0x40,0x5e,0x93,0x7a,0xa0,0x49,0xec,0x5a,0x7a,0x8d,0xed,0x33,0xa3,0x38,
			0xbb,0x9f,0x1d,0xd4,0x04,0xa7,0x99,0xe1,0x9d,0xdb,0x3a,0x83,0x6a,0xa3,0x9c,0x77),
		.s = chunk_from_chars(
			0xd1,0xd2,0x1b,0x8d,0xfa,0x55,0xf0,0x68,0x1e,0x8f,0xa8,0x61,0x35,0xcf,0x29,0x2d,
			0x71,0xb7,0x66,0x97,0x13,0xc2,0x91,0xd8,0xf8,0xdc,0x24,0x64,0x64,0xde,0x3b,0xbb,
			0x96,0x1b,0x59,0x6d,0xfc,0x8f,0xda,0x6c,0x82,0x3c,0x38,0x40,0x08,0xd0,0x5b,0xcb,
			0x3d,0xcc,0xc3,0x6a,0xcc,0xf1,0xb2,0xbe,0xde,0x1a,0x95,0xe5,0x22,0x58,0xd7,0xd1,
			0xbd,0xf1,0xfc,0x44,0xe1,0x80,0x72,0xab,0xd4,0x5c,0x13,0x92,0x01,0x5e,0xe7,0x16,
			0x92,0x69,0x0e,0xf8,0xcd,0xaa,0xed,0x33,0x7d,0xd8,0x54,0x67,0x83,0xf9,0x61,0xbb,
			0x96,0x20,0xeb,0x5c,0x7b,0x8b,0x67,0x16,0xe8,0xc6,0x00,0x35,0x1f,0xab,0x77,0x65,
			0xee,0x38,0xa1,0x5d,0x32,0xd8,0xa2,0xc0,0x94,0x98,0x25,0xc4,0x9a,0x7f,0x25,0xee,
			0xdd,0x9b,0xe7,0xb8,0x07,0xbb,0xfd,0x51,0x79,0x13,0x78,0x66,0x20,0xd2,0x49,0x82,
			0x3d,0xae,0x6f,0xe2,0xfd,0x39,0xac,0x63,0x9d,0xd7,0x48,0x21,0xb0,0xc1,0x20,0xb4,
			0x2f,0x31,0xc2,0xc6,0x39,0xd2,0xc6,0x1b,0x39,0x5f,0x09,0xf8,0x68,0x51,0xbc,0x80,
			0x9b,0x34,0xc4,0x98,0x1a,0xc6,0x5c,0xf2,0x5b,0x2e,0x8a,0xdc,0xbc,0xe1,0x90,0xef,
			0x2e,0xf6,0x7a,0x01,0x89,0x03,0x9c,0x91,0x10,0xf2,0x67,0x01,0xc3,0xee,0xd7,0x31,
			0xc8,0xd9,0xea,0xd1,0x78,0x22,0x0f,0xfc,0xac,0x7f,0x0f,0x67,0x8a,0xa2,0x22,0x68,
			0xe1,0xd0,0x19,0x42,0xec,0x51,0xe8,0x0e,0xef,0x06,0xe2,0x11,0x28,0x30,0x85,0x5e,
			0x87,0xba,0xfe,0x8c,0xc9,0xc2,0x2f,0xd7,0x37,0xc7,0xab,0xbc,0xa5,0xeb,0x7a,0x22,
			0x1d,0x38,0x35,0xa8,0x66,0x10,0xd2,0x4b,0x50,0x7b,0x5d,0xcb,0x46,0x18,0xaa,0x42,
			0x1f,0x63,0xa5,0x60,0x9e,0xf5,0xd6,0x8f,0x57,0x60,0xfd,0xdf,0x97,0x01,0x35,0x60,
			0x2e,0xfa,0xd0,0x85,0x1b,0xbf,0xf9,0x8f,0xe8,0x7f,0xa5,0x8b,0xc3,0x65,0xf3,0x8e,
			0xe7,0xec,0x8e,0xf5,0xaa,0xb1,0x7f,0xd1,0x1d,0x89,0xd9,0x1e,0xf4,0xc6,0x04,0xe0,
			0xd1,0xf0,0x01,0xd0,0xe0,0x88,0x69,0xdf,0x92,0x25,0xe3,0xb4,0xce,0xf5,0x2f,0xf8,
			0x68,0x15,0xe1,0x3b,0x3e,0xfd,0xf4,0x57,0x76,0xf9,0x35,0x37,0x69,0xa8,0xa5,0x1f,
			0xe7,0xd8,0x91,0xa7,0xef,0x70,0x35,0xee,0xcf,0xa2,0x59,0x84,0x87,0x38,0x37,0x68,
			0x86,0xed,0xc9,0x1c,0xc7,0x8f,0x6d,0xa3,0x1c,0x2f,0x07,0xee,0x36,0x2c,0x3d,0x82),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x2b,0xca,0xd6,0xe7,0x44,0xf2,0x49,0x0b,0xa6,0xa6,0xe0,0x72,0x28,0x32,0x41,0x7e,
			0xbd,0x91,0x0f,0x91,0x46,0xeb,0x62,0xba,0xaa,0x5c,0x74,0x95,0x29,0xf7,0x9d,0x6c,
			0xed,0x0b,0x81,0xa2,0xe2,0xa4,0x88,0x52,0xc8,0x55,0x8e,0x33,0x87,0x35,0xdc,0xbf,
			0xc2,0x28,0x57,0x94,0xae,0x60,0xf8,0x1a,0x25,0x23,0x7c,0x66,0xf6,0xce,0x5d,0x5e,
			0x80,0x1a,0x00,0x1e,0x7f,0x9e,0x30,0x9b,0x25,0x95,0xcb,0x86,0x6d,0xe2,0xbb,0x74,
			0xac,0x51,0x28,0x3b,0x68,0x20,0xec,0x9f,0x6e,0xbe,0x48,0x2e,0x1f,0xd2,0xd5,0x68,
			0x0b,0x7f,0xbd,0x23,0xc1,0xe6,0x2a,0x2e,0xe4,0xed,0xff,0x35,0x82,0x3f,0xc7,0xe4,
			0xa2,0x95,0xea,0x4f,0x1c,0x33,0x27,0x92,0xae,0xb5,0x3e,0xb4,0x4b,0x0b,0xed,0xd2),
		.s = chunk_from_chars(
			0x37,0xd9,0x60,0xfe,0x39,0x12,0x98,0xbb,0xdc,0x22,0x3f,0xa1,0xeb,0x1d,0x3c,0xd9,
			0xa4,0x6b,0xa8,0xc6,0x2e,0x1d,0xa8,0xc5,0x63,0xc8,0x9a,0x8f,0x0e,0x67,0xb8,0x64,
			0xfc,0x89,0x83,0x7f,0xfc,0x08,0xaa,0xb7,0x12,0x2b,0x84,0xc4,0x35,0xc7,0xf9,0x40,
			0x6e,0x16,0x5a,0x10,0x29,0x85,0x7c,0x1e,0x4d,0xea,0x65,0x35,0x69,0x27,0x72,0x73,
			0xb1,0xd9,0xb0,0xa9,0xf5,0xb0,0xdc,0x24,0xaf,0xdd,0x21,0x44,0x76,0xd4,0x72,0x08,
			0xad,0x52,0x21,0xa7,0xd7,0x93,0xca,0xb8,0x06,0x71,0xfb,0x49,0x87,0xc8,0x6b,0xd6,
			0x14,0x48,0x80,0xc5,0x9d,0x24,0x87,0x14,0x00,0xf6,0x4b,0xdc,0x6d,0x49,0x6d,0xbd,
			0x49,0x7f,0x3d,0xbf,0x64,0x28,0x64,0xfe,0x49,0xaf,0x3e,0x21,0x51,0x5e,0x62,0xd6,
			0x0f,0x00,0x71,0xdb,0x48,0x84,0xf4,0x96,0x70,0xea,0xa9,0xe4,0xe4,0x98,0x2f,0x26,
			0x9a,0xbe,0x72,0x42,0x44,0x28,0x88,0x59,0xc2,0xad,0xf6,0x0a,0x09,0xfa,0xaa,0xbb,
			0x07,0x99,0x0e,0x09,0xe5,0x6d,0xe2,0x54,0xba,0xbb,0xee,0x14,0xbe,0x7e,0xb6,0xed,
			0xa0,0xcd,0xb2,0x2f,0x3d,0x0d,0xe8,0x72,0x48,0x04,0x67,0x3f,0xb9,0x9f,0x86,0xef,
			0xb4,0x26,0x3d,0xcc,0x50,0x17,0xab,0xc9,0x1b,0xd9,0xcd,0x83,0x36,0x79,0x47,0x5b,
			0xfa,0xc5,0x0a,0x2b,0xe8,0xdb,0x86,0x29,0x6b,0xbf,0x80,0x17,0x88,0x93,0x57,0x37,
			0x13,0x14,0x60,0x4e,0x83,0xd6,0x8b,0x6e,0xfe,0xcd,0x4b,0x79,0xf0,0xa8,0xaf,0xa0,
			0xdf,0xfa,0x44,0x8f,0xb7,0xfc,0xe6,0xd3,0x44,0x70,0x9a,0x67,0x0e,0x0c,0xff,0x43,
			0x2c,0x3e,0x18,0x7b,0xcf,0xf7,0xfd,0xc4,0xf4,0xe9,0xab,0xe1,0x09,0x5c,0x46,0xb0,
			0x1d,0x88,0xb6,0x04,0x4b,0xb9,0x50,0xe9,0x28,0x59,0x01,0x0d,0x9a,0x0e,0x3b,0x2d,
			0x1f,0x27,0xa0,0x96,0xea,0xca,0xa2,0x42,0x63,0xa2,0xa0,0x52,0x3d,0x6e,0x0d,0xa1,
			0xfb,0xa8,0xaf,0x76,0x81,0x96,0xf7,0xa5,0x1f,0x92,0xfd,0xf1,0x52,0xbe,0xf0,0x62,
			0xdd,0x1f,0x83,0x27,0xce,0xe1,0xd3,0x44,0xc2,0x00,0xc2,0x11,0x5a,0xc6,0xec,0x1d,
			0xd8,0x51,0x4c,0xef,0x9e,0x36,0xd0,0xce,0x8c,0x32,0xe5,0x87,0x83,0xc4,0xfc,0xba,
			0x90,0x1a,0xa7,0x0c,0x2b,0x42,0x96,0x64,0x88,0x00,0x2f,0xf1,0x71,0xd3,0x64,0x14,
			0xa1,0x44,0xbf,0x46,0x77,0x51,0x83,0xa8,0x81,0x5d,0xe9,0xee,0x3e,0x81,0xf3,0x1b),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xc3,0x97,0x8b,0xd0,0x50,0xd4,0x6d,0xa4,0xa7,0x92,0x27,0xd8,0x27,0x0a,0x22,0x02,
			0x95,0x34,0x82,0x87,0x59,0x30,0xfb,0x1a,0xea,0xe4,0xe6,0x7f,0x87,0xe7,0x94,0x95,
			0x28,0x9d,0xe2,0x93,0xb4,0xa4,0x0d,0x92,0x74,0x6f,0xc8,0x4c,0xc8,0x31,0x8c,0x23,
			0x18,0xfd,0x30,0x65,0x0e,0x2b,0xb9,0xce,0x02,0xfd,0x73,0x4e,0xb6,0x83,0x41,0x0d,
			0x44,0xbb,0x31,0xad,0x54,0xfd,0x53,0xcf,0x92,0x96,0xcc,0xd8,0x60,0xb4,0x26,0xf5,
			0xc7,0x82,0xea,0x5c,0xb4,0x93,0x71,0xd5,0x61,0x84,0xf7,0x79,0x11,0xdd,0xf1,0xba,
			0x00,0x39,0xa0,0xa4,0x9a,0xa7,0xe7,0x63,0xeb,0x4f,0x5a,0x04,0x57,0x59,0x97,0x80,
			0x8b,0x0a,0xd9,0xf6,0xb3,0x30,0xca,0x38,0xed,0xc1,0x99,0x89,0xfe,0xbf,0x4d,0xa5),
		.s = chunk_from_chars(
			0x9a,0xed,0x20,0xa8,0xbd,0xaf,0x26,0xf1,0xf1,0x19,0x02,0x0d,0x8f,0x3e,0xa6,0xce,
			0x91,0x51,0x38,0xd4,0xc8,0x7d,0xce,0x02,0x5e,0x7f,0x4e,0x49,0x53,0x6c,0x8e,0xc0,
			0x79,0xed,0xc6,0xca,0xf0,0xd6,0x03,0xbf,0x42,0xbd,0x6a,0x45,0x4a,0x6d,0x52,0xd0,
			0xd9,0x9f,0xd0,0xf5,0x9f,0xfb,0x3b,0x22,0xe9,0xe6,0x7b,0x3d,0x0b,0xb2,0xd2,0x75,
			0xd9,0xae,0xdc,0x6d,0xa9,0x6a,0x72,0xcb,0xff,0x35,0xc4,0x3e,0x7f,0x39,0xa9,0x96,
			0xfa,0x8a,0x6d,0x33,0x8a,0x07,0x25,0xf7,0x85,0x25,0x4f,0xe9,0x1a,0x20,0x83,0x4b,
			0xa5,0x57,0xfe,0xdf,0xe7,0x15,0x2b,0x99,0x56,0xfe,0xdd,0xfd,0x94,0x17,0x41,0xef,
			0xf9,0x17,0x7c,0x2f,0xbb,0x55,0xe2,0x00,0xbb,0xe4,0x21,0x62,0xb3,0x2a,0x94,0x0c,
			0xc3,0x00,0xab,0x37,0x55,0x57,0xdf,0xfd,0x48,0xdf,0xa5,0x39,0xf5,0x0e,0xdd,0x52,
			0xdf,0x15,0x8d,0x90,0x72,0xd1,0x49,0x82,0xe9,0x63,0x03,0xbc,0x61,0x2c,0x2c,0x25,
			0x06,0xdb,0xca,0x3a,0x93,0x9d,0x62,0x6d,0x2e,0x7f,0xb4,0x44,0xc6,0xad,0x7d,0x8d,
			0x9f,0x3b,0xba,0x82,0x10,0xb2,0xac,0x2f,0x69,0x67,0x83,0xc3,0x49,0xfc,0x52,0x80,
			0xc1,0x05,0x40,0x2a,0x4b,0x3d,0x86,0xbe,0xf5,0x02,0x6c,0x3d,0xd9,0x99,0xe3,0xb2,
			0x23,0x80,0xf9,0xdc,0xce,0x40,0xe3,0xa9,0xcc,0x9f,0x1d,0x7b,0xc3,0x8e,0xf3,0xdd,
			0x7e,0x94,0x13,0xbb,0x57,0x98,0x00,0xc0,0xe6,0xc3,0xe9,0xab,0x91,0x2d,0xa8,0xfe,
			0xc1,0xa4,0xab,0x21,0x39,0x8e,0x96,0x80,0xba,0x0d,0x04,0xf3,0xb4,0xc8,0xd5,0x3c,
			0x02,0xf0,0x5c,0x7a,0xe4,0x9b,0x70,0xa5,0x61,0x1c,0xf8,0x2e,0x38,0xde,0x84,0xaa,
			0x8c,0x24,0x26,0xf0,0xb6,0x3e,0xa0,0x1b,0x28,0x9f,0x20,0x1d,0x3a,0xf4,0x0d,0xad,
			0x5d,0x6e,0x5b,0xcc,0xc7,0x5b,0x99,0x59,0xe5,0xc9,0x75,0x8e,0x79,0x10,0x5a,0xf7,
			0xa9,0xaf,0xb1,0x2a,0xee,0x57,0x7c,0xb3,0x99,0x18,0x79,0xdb,0x0f,0xd8,0x66,0x2c,
			0x5b,0xc4,0x90,0x22,0x75,0x24,0x98,0xa3,0x01,0xd9,0x5f,0x4b,0x1d,0x08,0xc0,0x1e,
			0xbc,0x31,0x3f,0x89,0xc0,0x0b,0x1e,0xc2,0x73,0x5a,0x07,0x98,0x3f,0xd5,0x28,0xe6,
			0x38,0x82,0x45,0x03,0x6f,0x0e,0xd4,0xa2,0xdb,0xb6,0x5d,0xd3,0x3a,0xb7,0xf1,0x24,
			0xc0,0x14,0xec,0x16,0x79,0xf1,0xc2,0xf1,0x1e,0xdf,0xfb,0x93,0xfa,0x2d,0x1d,0x73),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x0c,0x11,0x95,0x02,0xc2,0xa0,0x19,0x20,0xa0,0x90,0xe4,0x33,0x57,0xe7,0xb2,0x8e,
			0x33,0xc7,0xee,0x85,0x8b,0x43,0x30,0xe0,0x5c,0x71,0x04,0x89,0x31,0xc0,0xed,0x88,
			0x46,0x8c,0xa9,0x31,0xec,0xf0,0xb7,0x9c,0x2f,0xdc,0x17,0x56,0xb7,0x67,0x51,0x56,
			0xec,0x66,0xb8,0x33,0x5e,0x3d,0xf0,0x94,0x63,0xf5,0xae,0xe7,0x02,0x8f,0xbf,0x56,
			0x0f,0x98,0x4c,0xf6,0x98,0xfe,0x5c,0x42,0x80,0x22,0x9a,0xc9,0x6a,0x2e,0x59,0x23,
			0xd8,0xa9,0xd5,0x29,0x94,0x49,0xbb,0x66,0x50,0x08,0xec,0xc8,0x89,0x79,0x7e,0x9b,
			0xb1,0x5d,0x04,0xb8,0x8c,0x72,0x10,0xfa,0xdb,0x8b,0xf6,0xf2,0x38,0xe5,0xd2,0xdc,
			0x41,0xb9,0xcc,0xd1,0xf8,0x0e,0x9a,0x3e,0x6a,0xd1,0x47,0x94,0x8f,0x27,0x33,0x41),
		.s = chunk_from_chars(
			0x8a,0xbf,0x2a,0x30,0x77,0x4e,0x6e,0x73,0x38,0xec,0xa0,0x9c,0xcc,0xac,0xa3,0x68,
			0x43,0x99,0x94,0x04,0x92,0xfb,0x94,0xb2,0x3b,0x5a,0xd6,0x2c,0xe3,0xe1,0x1d,0x2d,
			0xbe,0xf8,0x96,0x6b,0xa5,0x26,0x99,0x79,0xeb,0x96,0x53,0xba,0xad,0x71,0x95,0x16,
			0xd3,0xe8,0x39,0x90,0x79,0xa2,0xf6,0x70,0x27,0x5a,0x2e,0xd4,0x2c,0x82,0x0a,0x9a,
			0x31,0xfc,0xd7,0x03,0xa7,0x66,0x37,0xe0,0xd7,0x13,0xf3,0x2d,0x79,0x2b,0x9a,0xe3,
			0x6d,0x72,0x88,0xf6,0x0c,0x2d,0x1a,0xe5,0x26,0x83,0xbb,0x15,0x94,0x1b,0x1c,0xd8,
			0x90,0xd2,0xcd,0x64,0x99,0x8b,0x77,0x25,0x85,0xe7,0x60,0x32,0xa1,0x70,0x2e,0x06,
			0x52,0xcb,0xf2,0x59,0xa1,0xce,0xae,0x69,0x5d,0x40,0xcf,0x2f,0x4f,0x6d,0x81,0x34,
			0x1c,0x8b,0xc9,0x08,0x2c,0xb9,0x6c,0x75,0x2c,0x35,0x5d,0xfb,0xe2,0x96,0xdd,0x21,
			0xd6,0x98,0x46,0xfa,0x37,0x61,0x3e,0x73,0x81,0x7b,0x2a,0x07,0x04,0x66,0x58,0xc9,
			0xe3,0xfc,0x6d,0x09,0x1e,0x17,0x59,0x1b,0xb1,0xa4,0xfb,0x6e,0x2a,0xc0,0x0a,0x31,
			0x94,0xc1,0x48,0x8e,0x16,0xa9,0xd2,0x90,0x37,0x86,0xdb,0x86,0xae,0x90,0xe9,0x6a,
			0xcb,0x4d,0xe9,0x90,0x1a,0xaf,0x1b,0x06,0x51,0xfb,0x76,0xa5,0x8d,0xcb,0x3d,0xb4,
			0x73,0xef,0xbf,0xb8,0x31,0xef,0x8e,0x30,0xf8,0x99,0x67,0xdd,0xd3,0xa6,0xc2,0xf1,
			0x89,0x79,0xa0,0x45,0x06,0x57,0xcd,0xae,0xef,0x6e,0x59,0x37,0x7c,0x6d,0xb1,0xec,
			0x46,0x06,0x5f,0x61,0x40,0x24,0xa6,0x9c,0x51,0x8a,0x55,0x99,0x42,0x59,0x4a,0x46,
			0x26,0x6e,0x0d,0x3c,0xa1,0x33,0x42,0x96,0xb9,0x68,0xa2,0x3a,0x4b,0x11,0xc6,0x3a,
			0x97,0xe2,0x9e,0xb1,0x6b,0x24,0xc0,0x2d,0x54,0x5d,0x5b,0x42,0x7e,0x6a,0xa5,0x85,
			0x33,0x33,0x18,0xe6,0x3a,0x20,0x45,0x24,0xe0,0xe4,0x2a,0xc1,0xed,0xb7,0x0d,0x34,
			0x56,0x78,0x0d,0xbe,0xad,0x31,0xf7,0x85,0xf0,0xb2,0xa7,0x7f,0xfe,0xb0,0xd3,0x73,
			0x84,0xcb,0x5f,0x65,0xb4,0xe3,0x6c,0xa2,0x41,0xf3,0xb2,0xb0,0x59,0x10,0x5f,0xaa,
			0xa3,0x22,0x2d,0x6c,0x13,0x5e,0xa5,0xa3,0x66,0x51,0xae,0xa3,0x96,0xd2,0x2f,0xc4,
			0xea,0x1b,0x40,0x4d,0x7e,0x83,0x4b,0x6d,0xf1,0xfb,0x83,0x8b,0xb5,0xba,0x0d,0x78,
			0x4a,0x96,0xe2,0xae,0x28,0x43,0xdb,0x3e,0xee,0xa4,0x96,0xc7,0xad,0x2b,0x42,0x41),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xdd,0xbd,0x84,0x68,0xbd,0xb0,0x36,0xf4,0x79,0x9f,0x42,0x8b,0xc8,0xb4,0x37,0x4e,
			0xd9,0xb7,0xcd,0xe5,0x41,0x33,0x7a,0xc4,0x39,0xd4,0x41,0xac,0x06,0x14,0xcb,0x75,
			0xb8,0x16,0xb8,0x0c,0x17,0xd2,0x37,0xb8,0xdb,0x73,0xd4,0xa1,0x1b,0xfd,0x92,0x92,
			0x08,0x33,0x3a,0xfe,0xdb,0xb8,0xf2,0x41,0x0c,0x74,0x11,0x29,0xc5,0x39,0x32,0xb5,
			0x96,0xa7,0x88,0x1c,0x6a,0x4d,0x71,0x11,0xba,0x10,0x4d,0x46,0x00,0xd1,0x90,0x2f,
			0x6f,0x4a,0x16,0x08,0xe1,0x39,0xb7,0x19,0x11,0xc1,0x1c,0x39,0x0a,0x0d,0xd0,0x91,
			0xdf,0x36,0x9a,0xa2,0x9d,0x67,0x0b,0x8a,0x7e,0x3f,0x53,0x82,0x5f,0x76,0x59,0xac,
			0x74,0xc4,0x0a,0x0c,0x3b,0xfe,0xf0,0xd3,0xae,0x83,0x07,0xe4,0xbd,0xd6,0xcd,0x91),
		.s = chunk_from_chars(
			0x4e,0x37,0x7e,0x24,0x59,0x81,0x5d,0x5b,0x33,0x91,0x5f,0xa6,0x3c,0xd4,0x77,0xb5,
			0xbe,0x7c,0x6b,0x7f,0x78,0x14,0xd1,0x35,0x00,0x34,0xce,0x71,0x0b,0xe6,0x7e,0xd6,
			0x91,0x39,0xdb,0x62,0x2e,0xf6,0x0e,0xc6,0xb7,0x63,0x8e,0x94,0xb2,0x02,0x36,0x8b,
			0xac,0x63,0x1e,0x05,0x77,0x02,0xb0,0xe6,0x48,0x7b,0x32,0x4a,0x6b,0x98,0xed,0x7e,
			0x03,0xd1,0xf3,0xf2,0x0a,0x98,0x14,0xb0,0x0e,0x21,0x7a,0x46,0x48,0xe4,0xbb,0xc4,
			0x49,0xa2,0xaf,0x40,0x5c,0xa4,0xb5,0x9f,0x84,0x38,0xdd,0xfd,0x75,0xd3,0x4d,0x10,
			0x64,0xe5,0x8b,0xfb,0x32,0x5c,0x55,0xbd,0x54,0xea,0x6c,0xdf,0x77,0x12,0xba,0x80,
			0x7c,0x3e,0x4c,0x66,0x5d,0x62,0x0c,0xd5,0x95,0x13,0xd7,0xbc,0x08,0x55,0x24,0x7e,
			0xb6,0x70,0xec,0xc2,0x92,0x50,0x96,0x61,0x81,0x27,0x02,0x70,0x32,0x75,0xd9,0xb2,
			0xf8,0x7e,0xf2,0x79,0xd7,0x70,0x0e,0x69,0xd9,0x95,0xdb,0x98,0x14,0x4a,0x14,0xc8,
			0x17,0x74,0xa4,0xcd,0x89,0x0e,0xc0,0x3d,0x13,0xf8,0x58,0xf3,0x76,0x9e,0x50,0x48,
			0xed,0x55,0xca,0xa8,0x12,0x01,0xe8,0x78,0x5d,0x37,0x71,0xce,0x6d,0xa5,0x11,0x75,
			0xd0,0x17,0xd2,0x11,0xfa,0x70,0x37,0x94,0x41,0x6f,0x46,0x9b,0x11,0x29,0xd7,0x31,
			0xab,0xde,0x74,0x4d,0xa5,0xb2,0xfa,0xcd,0x7a,0x9b,0x09,0x3d,0x6c,0x97,0x43,0x50,
			0x9b,0x01,0x03,0xba,0xb9,0xc8,0x1c,0x6e,0x5f,0x38,0xbc,0x97,0x18,0xe3,0xe4,0xfa,
			0xa8,0x64,0x75,0xd1,0x37,0x25,0xa8,0x29,0xac,0x61,0xdf,0x8d,0x15,0xf0,0xb2,0x7c,
			0xb4,0x0d,0x0e,0xba,0x0b,0x24,0x6b,0x9c,0x36,0x0b,0x56,0x9b,0x81,0xb3,0xab,0xf3,
			0x80,0xee,0xc2,0x74,0x92,0x31,0x6b,0xc2,0x92,0xe5,0x15,0x0e,0xe0,0x60,0x72,0x19,
			0xa2,0xbd,0x80,0xba,0x98,0x4c,0x7e,0x3f,0x19,0x89,0xbc,0x51,0xe4,0xc5,0xda,0x3a,
			0xe5,0x07,0x06,0x76,0xe0,0xc1,0x50,0xd0,0x37,0xa8,0x6a,0x0f,0x91,0xbf,0xc0,0x7c,
			0xde,0x64,0xc1,0x9f,0x9c,0x7a,0x7a,0xf4,0x4d,0x69,0x29,0x97,0x00,0x41,0x44,0x8d,
			0x3b,0x17,0xc2,0x49,0xd5,0xe0,0xb5,0x86,0x2e,0x9a,0x25,0x20,0x9e,0x8f,0x97,0xd7,
			0xa0,0xf0,0x30,0x18,0x15,0x04,0xfe,0xad,0x22,0x66,0xc8,0x73,0xfd,0x23,0x59,0x83,
			0xdf,0x3d,0x06,0x57,0xb9,0x20,0x96,0xe2,0xb4,0x90,0xdf,0x33,0xca,0x11,0x57,0x33),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xf9,0x96,0xf3,0xad,0xc2,0xab,0xa5,0x05,0xad,0x4a,0xe5,0x2b,0xc5,0xa4,0x33,0x71,
			0xa3,0x3d,0x0f,0x28,0xe1,0x95,0x0b,0x66,0xd2,0x08,0x24,0x06,0x70,0xf3,0x52,0xef,
			0x96,0x18,0x5e,0x9a,0x70,0x44,0xf4,0xce,0x2f,0x2f,0xf9,0xae,0x01,0xa3,0x1e,0xf6,
			0x40,0xe0,0xb6,0x82,0xe9,0x40,0xc5,0x10,0x51,0x17,0x59,0x46,0x13,0xdd,0x1d,0xf7,
			0x4d,0x8f,0x2b,0xa2,0x0c,0x52,0x22,0x3b,0x04,0x5a,0x78,0x2e,0x85,0x0a,0x12,0xa2,
			0xaa,0x5c,0x12,0xfa,0xd4,0x84,0xf1,0xa2,0x56,0xd0,0xcd,0x08,0x72,0xd3,0x04,0xe8,
			0x85,0xc2,0x01,0xcd,0x7e,0x1e,0x56,0xd5,0x94,0x93,0x0b,0xb4,0x39,0x21,0x36,0xfb,
			0x49,0x79,0xcc,0x9b,0x88,0xaa,0xb7,0xa4,0x4b,0xfc,0x29,0x53,0x75,0x1c,0x2f,0x4c),
		.s = chunk_from_chars(
			0x30,0xb3,0x48,0x62,0x4f,0xaa,0x99,0x85,0xfc,0xd9,0x5f,0x9c,0x7e,0xad,0x3a,0xfe,
			0x64,0x56,0xba,0xdf,0x8c,0x0f,0xed,0xbd,0xad,0xb3,0xa9,0x00,0x3a,0x67,0x02,0x97,
			0x3a,0xcd,0xb4,0xe8,0x66,0x52,0x36,0x7d,0xb2,0x3e,0x0a,0x81,0x41,0x88,0x0d,0x66,
			0x31,0x83,0x4f,0x9f,0x17,0x1c,0x94,0xa8,0xfe,0x9c,0x31,0x5b,0xcb,0x86,0x80,0xec,
			0xfb,0x5a,0x4f,0x59,0xb4,0x5d,0x4e,0x4c,0x3c,0x05,0x82,0x8b,0x7f,0xaa,0xa8,0xe4,
			0x23,0x4a,0xad,0xa4,0xe7,0x66,0x64,0x6c,0xc5,0x10,0xd0,0x7b,0x42,0xbd,0x38,0x83,
			0xa8,0x3b,0x5b,0xcb,0x92,0xd9,0xe7,0xcc,0x1d,0xdf,0x59,0x0a,0x69,0x01,0x11,0xbf,
			0xc6,0x2a,0x51,0xaf,0x7e,0x55,0x54,0x3e,0xa5,0x18,0x8c,0x92,0x45,0x3d,0x41,0xd3,
			0xe8,0xfd,0xab,0xee,0x3e,0x1d,0xef,0xa9,0xd0,0xaf,0xdb,0x85,0xc8,0x15,0x3a,0x50,
			0x19,0xae,0x45,0x56,0x3e,0xa3,0x08,0x0a,0x30,0x22,0x66,0x81,0x68,0xf0,0xc2,0x73,
			0xa6,0xdb,0x1a,0xfa,0xdc,0xd5,0xed,0xbc,0xa5,0x02,0x1c,0x2e,0x53,0xf4,0xd9,0x51,
			0xc6,0x04,0x20,0x6a,0xe1,0x0f,0x28,0x7f,0x45,0x18,0x67,0x27,0x1d,0x37,0x04,0x82,
			0x79,0x1c,0xdf,0xdc,0xb6,0xa4,0x01,0x0f,0x6b,0x3d,0x9b,0x92,0x85,0x63,0xd1,0x68,
			0xda,0x19,0xf1,0xc1,0xe5,0x70,0xf8,0xc1,0x58,0xf3,0xd4,0x90,0xb2,0x9a,0xa2,0x3a,
			0xbd,0x1f,0xfd,0xf2,0x08,0x66,0xc3,0x4c,0x6e,0x63,0xb9,0xe8,0xa9,0xa0,0x2d,0x7a,
			0x1b,0x19,0x6d,0x05,0x5f,0x4c,0x53,0xce,0x82,0xb4,0x00,0xe4,0xab,0x9e,0x1b,0x9d,
			0x70,0xd0,0x04,0x9d,0x6d,0x57,0xcf,0x0a,0x49,0x49,0xcf,0xc6,0x8d,0x63,0x38,0x82,
			0x88,0x2d,0xcf,0xdf,0xc5,0x0c,0xf4,0x49,0xdf,0x10,0xac,0xf2,0x03,0x05,0xc2,0xaa,
			0x43,0xbd,0xa1,0x0f,0xd8,0xa1,0x0b,0x4e,0xca,0xa2,0x31,0x00,0xaa,0x47,0xe9,0x29,
			0x36,0xdc,0xe1,0xbf,0xb8,0xd6,0x59,0x52,0x35,0xbb,0xfe,0x2c,0x85,0x85,0xcb,0x16,
			0x47,0xb2,0xbe,0xac,0xb1,0xe1,0xd4,0xb6,0xce,0xf7,0x58,0x81,0x1a,0x68,0x33,0x0f,
			0xa9,0xc3,0xa8,0x25,0x73,0xc0,0x8f,0xa2,0xcd,0xa5,0xa0,0x3f,0x34,0x25,0x55,0x4e,
			0x45,0xd9,0x8c,0x16,0x45,0xc5,0xbd,0x27,0xd1,0x2e,0x6c,0x20,0xb2,0xc4,0x62,0xa7,
			0x46,0xe8,0x82,0xa3,0x42,0x1a,0x7b,0x1b,0x1e,0x25,0xb4,0xc3,0x6c,0x8b,0x16,0xa1),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x6a,0xce,0x05,0x2d,0x7e,0x99,0xcd,0x97,0x3b,0xb5,0xc9,0xf6,0x67,0x9b,0x1c,0x30,
			0x5e,0x07,0x20,0x89,0x65,0xfe,0x58,0xc6,0x3b,0x10,0xa6,0x92,0xf1,0xdb,0xbe,0x22,
			0xfc,0xd0,0xdb,0x15,0x89,0x3a,0xb1,0x9e,0x10,0x7b,0xa2,0xe4,0x2c,0x99,0x34,0xa9,
			0xaa,0xfa,0xc3,0x2a,0xdf,0x6c,0x73,0x47,0x3f,0x69,0x69,0xe4,0x2c,0x98,0x3b,0x8f,
			0x0c,0x96,0xa4,0x63,0x9e,0xf7,0x7d,0x2c,0x8e,0x88,0xe8,0xcc,0x47,0xd7,0xcf,0xdd,
			0x08,0xf6,0x8d,0x97,0x3a,0x7b,0xea,0xf4,0x01,0xcb,0x4d,0x13,0x11,0x99,0x2d,0xda,
			0xc3,0xa9,0xc9,0xe0,0x67,0xda,0x19,0x8a,0xdc,0x63,0x04,0x74,0x5f,0x5d,0xd3,0x12,
			0xa1,0x82,0xe6,0x97,0x1c,0x34,0xa5,0x15,0xa6,0xc1,0xba,0xe6,0x47,0xe5,0x7e,0x4c),
		.s = chunk_from_chars(
			0x5f,0x0e,0x74,0xf4,0x54,0x75,0x4a,0x30,0x74,0xfa,0xaf,0xc6,0x05,0xf3,0xc9,0xaf,
			0x47,0x60,0x4a,0x89,0x83,0x65,0x0a,0x9b,0x62,0x11,0xfb,0x19,0x1d,0x9a,0xfa,0x53,
			0x15,0xdf,0x4d,0xb4,0x50,0x1f,0xd4,0xf0,0x4c,0x74,0x1d,0x76,0x46,0x56,0xd4,0xa5,
			0xd0,0x06,0x38,0x8a,0xd8,0xfd,0xb2,0x19,0xec,0x6b,0x75,0x69,0x08,0xe2,0x3b,0x30,
			0xcb,0x63,0x9f,0xfa,0x7b,0xbf,0x28,0x74,0x71,0x3b,0xfd,0x5a,0x10,0x62,0xc1,0x9d,
			0x04,0xe0,0xe4,0xa7,0x4b,0x14,0x44,0x6a,0x7f,0xdf,0x5c,0xb8,0x12,0xe9,0xac,0x7b,
			0x60,0x12,0xd9,0xae,0x99,0x1c,0x47,0x65,0x6d,0x2a,0xde,0xd2,0x40,0x74,0xbb,0x8a,
			0x38,0xb1,0xa8,0x8b,0x1c,0x2b,0x13,0x1e,0x5b,0x09,0xc9,0x37,0x57,0xfd,0xb2,0xd6,
			0xb6,0x9a,0xa8,0x26,0x5a,0x43,0x5f,0xba,0x00,0xae,0xb3,0x6a,0x1f,0x62,0x9b,0xc3,
			0x4b,0x87,0x60,0x89,0xd2,0x8a,0x94,0x8d,0xd6,0xab,0x4c,0x89,0x94,0x30,0xda,0x60,
			0xa2,0x6f,0x6c,0x13,0x60,0x3f,0xc8,0x89,0xc7,0xb2,0x93,0x6c,0xa3,0xc5,0x15,0x6b,
			0xd7,0xfa,0x6e,0x34,0xea,0xc9,0xe0,0x48,0x00,0x83,0x3e,0xf0,0xcb,0x9b,0x6e,0xef,
			0x78,0x8c,0x0e,0xf0,0x02,0x1a,0x45,0x36,0xfb,0x83,0x71,0xfa,0x3e,0x2c,0x8b,0xb8,
			0xbe,0xfa,0xc1,0x6e,0x80,0x92,0xd6,0x9c,0x57,0x1c,0x1e,0x15,0xfd,0x25,0x5e,0xc0,
			0xa0,0x7a,0xcf,0x9a,0xe9,0x95,0x38,0x31,0xef,0xd3,0xdc,0xbe,0xf4,0x4e,0x0f,0xcc,
			0xeb,0xb1,0xaf,0x95,0x9d,0x71,0xf5,0x01,0x30,0xe8,0xac,0xb4,0xfa,0x23,0x19,0x26,
			0x1f,0xba,0x12,0xf2,0x71,0x5d,0xef,0x82,0xbf,0xaf,0xbf,0x40,0xe3,0x45,0xec,0x5d,
			0xcd,0xab,0x5c,0x1b,0xf5,0xf6,0x6b,0x1d,0x0e,0x9f,0x7a,0x9c,0x62,0xc9,0x37,0x57,
			0x46,0xe1,0xae,0x0c,0x8f,0x14,0xa4,0x89,0x18,0x43,0x83,0xe8,0x1d,0xce,0x20,0x70,
			0xad,0x4b,0x52,0x5d,0xf7,0x6b,0x44,0x6b,0x1f,0x22,0x92,0x1d,0x42,0x4d,0x9b,0xa3,
			0xce,0x21,0x57,0x75,0x01,0xdf,0x62,0x80,0xfd,0xc6,0x9f,0x02,0x39,0xae,0x11,0x27,
			0xb6,0x99,0x50,0x75,0x9d,0x5f,0x0b,0x69,0x3f,0x54,0xe8,0x7e,0x07,0x63,0x62,0x3b,
			0xf5,0xd3,0xff,0x69,0x43,0x00,0x81,0xb9,0xc9,0xe2,0x44,0x5a,0x05,0xe1,0x15,0x67,
			0x5e,0x09,0x0b,0xca,0xb2,0xaa,0x1d,0x75,0xce,0xee,0x2a,0xd6,0x19,0xec,0x8b,0x80),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x0e,0x49,0x74,0x0f,0xdc,0xca,0x6b,0xfc,0xe2,0x94,0xc1,0x1f,0x45,0x40,0x78,0x05,
			0xb3,0xda,0x41,0x2b,0x01,0xef,0x3f,0xb5,0x13,0xe7,0x0e,0x62,0xfd,0x95,0x04,0xc0,
			0x67,0x0d,0xb6,0x9c,0x36,0xb6,0xbe,0xbd,0x69,0xa0,0xbc,0xd2,0x40,0x17,0x9b,0xa8,
			0xa4,0x78,0x16,0xa0,0xc3,0x43,0x7a,0x61,0xfb,0x72,0xad,0xca,0xf9,0x09,0x6f,0x2a,
			0x22,0xef,0xe0,0xb4,0x31,0xfc,0x42,0x2d,0x22,0x53,0x01,0xe8,0x50,0xf2,0xf0,0xf4,
			0xda,0x87,0xd6,0x94,0x4a,0x85,0x29,0xef,0x79,0x78,0x19,0x09,0xad,0x96,0xd1,0xf2,
			0x05,0x96,0xf9,0x3e,0x17,0xc5,0x7f,0xb4,0xd7,0x56,0x97,0x4b,0xbb,0xf9,0x00,0x52,
			0x1c,0xb0,0x89,0xee,0xe0,0xde,0xd5,0xc9,0x56,0xa1,0x5b,0x09,0x61,0x62,0xb0,0x7f),
		.s = chunk_from_chars(
			0x7b,0xbb,0x3d,0xdd,0x17,0xa4,0x2b,0xe7,0xcc,0x4e,0x7e,0xaf,0x45,0x65,0x09,0xa4,
			0xba,0x58,0xd4,0x0c,0x49,0xa3,0xd9,0x95,0x73,0xb7,0x33,0xe1,0x94,0x2f,0x9f,0xca,
			0x20,0xba,0x8b,0x91,0x07,0x08,0xd6,0xe7,0x50,0x36,0x7e,0x84,0x73,0x02,0xfc,0x60,
			0x3b,0x80,0x63,0xc1,0x9a,0xf8,0x83,0xe7,0x50,0x7f,0xb0,0xd9,0xcc,0x2b,0xe3,0x74,
			0x79,0xa3,0x7c,0xca,0x25,0xb8,0xc7,0xc4,0x6f,0x6b,0xf6,0x61,0xdc,0x6a,0x32,0x32,
			0xf8,0x8b,0x48,0x3f,0x1b,0x8f,0x41,0xb4,0x6d,0x49,0xba,0x3f,0x17,0x95,0xd6,0x8e,
			0xaa,0xd4,0xa2,0x55,0x6f,0xb5,0xd7,0x87,0x3b,0xbb,0x65,0x01,0xec,0xf0,0x6a,0xc5,
			0x58,0x23,0x5e,0xd1,0x39,0x90,0xb0,0xe1,0x6f,0x67,0x96,0x5b,0x09,0x36,0x6b,0xcb,
			0x36,0x2c,0xfc,0x6f,0xb9,0x78,0xf4,0xf6,0x8d,0x81,0x46,0xdc,0x8b,0x81,0x98,0x04,
			0xdf,0x42,0x4e,0x8c,0xa5,0xb6,0x3c,0xf1,0xfc,0xf9,0x7b,0xbf,0x30,0x0d,0x0b,0x99,
			0x88,0x60,0x79,0x8a,0x63,0x42,0x43,0x83,0xfc,0xd8,0x1d,0x37,0x77,0x3d,0x59,0xbb,
			0x13,0xb4,0xfa,0x5d,0x46,0x8c,0xd1,0x28,0xbb,0xab,0x18,0xa8,0xce,0x51,0x73,0xbe,
			0x5d,0x9d,0x54,0xd3,0x17,0x7f,0x02,0x45,0x78,0x84,0x09,0x97,0x3d,0xf4,0xa9,0x01,
			0x6b,0x94,0x4b,0xae,0xfb,0xf3,0xbf,0x11,0x46,0xa9,0x39,0x3d,0x22,0xe3,0x5e,0xc2,
			0xbe,0x0a,0xe6,0xf4,0xc3,0x1d,0xc4,0x98,0x1f,0x40,0xfc,0x1b,0xaf,0x38,0x26,0x00,
			0x69,0x9e,0xaf,0xce,0xa9,0x2c,0xbe,0x24,0xe2,0x6e,0xe8,0x46,0xfa,0x23,0xbc,0x19,
			0x3b,0x6e,0x72,0x14,0x01,0xb7,0xac,0x3f,0x5f,0x4e,0xbe,0xb6,0x33,0x97,0x9f,0x8e,
			0xf3,0x5f,0x4a,0xb1,0x11,0x7a,0x86,0x9d,0x5b,0x9d,0xbb,0x74,0x82,0xf0,0xd5,0xa5,
			0x9e,0x41,0x63,0x54,0x8d,0x25,0x12,0xae,0x06,0x72,0x05,0xb5,0x7d,0x03,0x0c,0x48,
			0x3f,0x72,0x0d,0x2c,0x44,0x35,0x04,0x28,0xf5,0x26,0x89,0x43,0xfc,0x5f,0x6e,0xa1,
			0xc8,0x8e,0x2e,0xc1,0x3a,0xb3,0xdc,0x14,0x56,0xe9,0x6a,0x3b,0x8e,0x7c,0x12,0x1a,
			0xf4,0xd6,0xa5,0xfe,0x4e,0xe5,0x5e,0x99,0xfb,0xc3,0x59,0x2a,0x48,0x7c,0x19,0x4b,
			0xc2,0xf2,0xbf,0x6e,0x79,0xfb,0x79,0xc2,0x87,0x6c,0xf3,0x36,0x5e,0x07,0x5b,0xee,
			0xac,0xc7,0xdb,0x4d,0xb7,0xee,0x69,0xe7,0xf1,0xfe,0x12,0xa3,0x27,0xe6,0xcb,0x0f),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0x0e,0x67,0x5d,0xac,0x9a,0xec,0x91,0x01,0x06,0xa6,0xab,0x21,0x9b,0x4c,0xce,0xb5,
			0x2d,0xed,0x25,0x49,0xe8,0x99,0xc9,0xa2,0x4d,0x5e,0xe5,0x51,0x77,0x76,0x18,0x88,
			0xa3,0xbe,0x1a,0x2d,0xef,0x6a,0xa3,0x2d,0x62,0xf7,0x88,0x13,0x2d,0x62,0x27,0xd9,
			0x30,0x98,0x06,0xfd,0xc0,0x2d,0xb7,0xd8,0xa8,0x50,0xff,0x2c,0x6d,0xff,0x37,0xfc,
			0xd7,0x77,0xf1,0xa0,0xac,0xef,0xdf,0x18,0xbf,0x85,0xf1,0xa1,0x29,0x79,0xbe,0x86,
			0xd7,0x99,0x25,0x39,0x45,0xfc,0x34,0xa2,0x88,0xf3,0x48,0xb7,0x92,0x3d,0x76,0x4d,
			0xb2,0x7a,0x2a,0x2d,0x5a,0xe2,0x0e,0x6b,0x25,0x37,0x2e,0xf3,0x18,0xf8,0x59,0x65,
			0x29,0xd8,0xca,0x23,0xfd,0x6f,0x08,0xa8,0xf6,0x2e,0x0a,0x1b,0x6d,0x98,0x9f,0x23),
		.s = chunk_from_chars(
			0x80,0x52,0xd9,0x5f,0x12,0xce,0x0e,0x6e,0x53,0xa5,0xa3,0x56,0xa0,0xeb,0x35,0x3b,
			0xdc,0xc1,0xa6,0x65,0x14,0xd6,0xcf,0xb3,0xa3,0xd9,0x61,0x55,0x31,0x0b,0xdd,0xa0,
			0xa0,0xd1,0x79,0x5f,0x97,0x64,0x3f,0x3a,0x44,0x96,0x63,0x4f,0x2d,0xd9,0xb9,0x5a,
			0x21,0x38,0xee,0x39,0x0e,0x1e,0x74,0xbe,0x31,0x34,0xf3,0xf4,0x7a,0x91,0x9e,0xe7,
			0xb5,0x9f,0x8e,0xcd,0x27,0x2a,0xb8,0x8c,0x82,0xcb,0xce,0x7c,0x21,0x7e,0x5f,0x92,
			0xd0,0x57,0xa5,0xb0,0x0f,0xbf,0x05,0x75,0xcd,0xae,0xcd,0x7d,0xc2,0x85,0xa4,0x21,
			0x8c,0x8a,0x95,0x52,0x16,0x59,0x8f,0x07,0x42,0x67,0x1e,0x01,0x8e,0x8e,0x4e,0x76,
			0x83,0x9a,0x57,0x5f,0x50,0xb2,0x10,0x2a,0x8b,0x77,0xd1,0xb8,0x4f,0x6d,0xce,0x98,
			0xd7,0x8e,0x57,0x58,0xe0,0xa6,0xf9,0x2b,0xf3,0x5d,0x6a,0x2f,0x18,0xad,0x40,0x09,
			0x25,0xd7,0x88,0x0f,0x9e,0xfc,0x77,0x4a,0x8c,0x7e,0xbf,0x64,0x88,0x5c,0xd2,0xf6,
			0xf6,0x29,0xb5,0x4a,0x7c,0x12,0xec,0x91,0xd3,0x9b,0x3c,0x25,0x18,0x24,0x1f,0xdc,
			0x32,0x2d,0x9b,0x23,0x5a,0x8e,0xa4,0x4f,0x77,0xe8,0x2f,0x3d,0xc4,0xf7,0x28,0xf6,
			0x20,0xc0,0x7d,0x1e,0x7f,0xf4,0x09,0x4f,0x29,0xc6,0x74,0xab,0x0f,0x08,0x02,0xef,
			0xa1,0xc9,0xe6,0x48,0x1e,0xbb,0x84,0xe0,0xbf,0x13,0xef,0x46,0x8d,0x8c,0xca,0x11,
			0x45,0x70,0xb9,0xed,0xcd,0xdf,0x98,0xac,0x4a,0x83,0x4f,0xe7,0xa0,0xd5,0xc6,0xfa,
			0xe8,0xa6,0x0a,0x48,0x39,0x9f,0x3c,0x8a,0xf4,0x2f,0xf4,0x02,0x6e,0x42,0xa8,0x1a,
			0xac,0x36,0x11,0x4f,0xfc,0x05,0x3f,0x3f,0x72,0x9b,0x7c,0xf9,0xa9,0x7a,0x56,0x84,
			0x8e,0xbe,0xa0,0x11,0x5a,0xa8,0x29,0x83,0x41,0xaa,0x22,0x69,0x63,0xeb,0xdf,0x57,
			0xab,0x2d,0x8e,0x4b,0x90,0x00,0xdd,0x05,0x1a,0x6c,0x5d,0x69,0xf6,0x0e,0x1d,0xc1,
			0xb3,0x3f,0x20,0x94,0xfd,0xbf,0x8e,0x5b,0x62,0x7b,0xc0,0x76,0x4d,0xb9,0x52,0x2c,
			0xbb,0xc0,0x81,0xdb,0xf3,0x8c,0x21,0xb1,0x3f,0x98,0x08,0x13,0xbd,0x2b,0x00,0xc7,
			0x57,0xeb,0xb8,0xc0,0xb2,0x12,0x13,0x15,0x2e,0x69,0x40,0x39,0xf3,0x06,0xf7,0x34,
			0x28,0x57,0x65,0x1f,0x72,0x2b,0xdd,0xa0,0x12,0x12,0xa8,0x55,0x27,0x99,0xbd,0xa6,
			0xef,0x07,0xc5,0x20,0x7d,0xc7,0x44,0xef,0x79,0x69,0xaf,0xd5,0xaf,0x2e,0x6f,0x12),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
		.m = chunk_from_chars(
			0xf6,0xa7,0xa6,0xe5,0x26,0x59,0x12,0x5f,0xbb,0xc8,0x72,0x74,0x17,0x28,0x3b,0x9a,
			0x64,0x44,0x1f,0x87,0x12,0x1e,0x27,0xf3,0x86,0xd5,0x01,0x9f,0x10,0xcc,0x9b,0x96,
			0x1e,0x09,0xf1,0xb3,0xb0,0xdb,0x23,0x63,0x0c,0xc0,0xca,0xac,0xb3,0x85,0x8c,0x6f,
			0x93,0xaf,0xee,0xea,0x7e,0x1a,0x6a,0x80,0xdb,0xe0,0xc2,0xbd,0x9c,0x7c,0x93,0x95,
			0x70,0x30,0x2d,0xec,0x39,0xa4,0xa2,0x5c,0xc0,0xcf,0x1d,0x32,0xa7,0x1a,0x75,0xb9,
			0xa0,0xc3,0x02,0xbc,0xdd,0x80,0xb0,0x46,0xc8,0x66,0x51,0xac,0xf3,0x08,0x38,0xcd,
			0x52,0xe3,0x03,0x99,0xa8,0xfa,0xb8,0xd0,0x3f,0xbd,0x14,0x0c,0xdc,0x2f,0x1f,0x02,
			0xf2,0x48,0x04,0x05,0x16,0x98,0x20,0xcc,0xb3,0x2e,0x59,0x74,0xff,0xb8,0xb1,0xc8),
		.s = chunk_from_chars(
			0x84,0x60,0x3a,0xcb,0xfe,0x1f,0x2f,0x76,0x9f,0x1a,0x62,0xb0,0xf2,0x87,0xf3,0x06,
			0x94,0x0b,0x22,0x54,0x76,0x71,0x4a,0x4b,0x68,0x27,0xc0,0x2d,0x7b,0xd0,0x52,0xf3,
			0x03,0xf3,0x0a,0x5f,0xa6,0xda,0x83,0xe6,0x06,0x15,0x30,0x56,0x69,0xca,0x9e,0xc1,
			0x77,0xc5,0xb3,0x2b,0x14,0x15,0xee,0xbe,0xf7,0x86,0x20,0x29,0x6e,0xba,0xd6,0xdb,
			0xbd,0x52,0x08,0x39,0xd3,0xaa,0xcc,0x97,0x81,0xac,0x86,0x02,0xdd,0xce,0x07,0x36,
			0xdc,0xfa,0x72,0x90,0xb4,0x5f,0x15,0x5b,0x8e,0x92,0x4d,0x0a,0xfd,0xf7,0xdf,0xc8,
			0xd1,0x99,0xbf,0x09,0x50,0x9d,0x01,0x76,0xa6,0x8b,0x14,0x57,0x56,0xee,0xf5,0x3d,
			0xe4,0x56,0xe1,0x70,0x78,0x85,0x98,0x49,0xa3,0x52,0xa5,0xbb,0x65,0x42,0x39,0xd8,
			0xeb,0xaf,0x88,0x00,0xca,0x82,0x63,0xd3,0x4a,0x86,0x8d,0x52,0xbf,0x8f,0x22,0x64,
			0x4d,0xd9,0xf3,0xc0,0x5b,0xd8,0x91,0xcd,0x92,0xf2,0x63,0x53,0x0c,0x58,0x96,0x02,
			0x3c,0x6b,0x21,0x3d,0xdb,0x64,0xed,0xe1,0x77,0x0f,0xf1,0x68,0x6c,0x34,0x03,0x6e,
			0x28,0x1e,0x91,0x1d,0x9d,0xc9,0x60,0x35,0x4f,0xd8,0x44,0xcb,0x7b,0x22,0xdc,0x0c,
			0xd8,0x1a,0x96,0x20,0x3b,0xa8,0x18,0x40,0x1c,0xcc,0x22,0x5f,0x85,0x7e,0x59,0xa5,
			0xcb,0x7b,0xa6,0xdf,0xc7,0xf5,0x13,0x5e,0xa3,0x27,0x81,0xe6,0x3d,0xaa,0x14,0xfb,
			0xda,0x1b,0xac,0xc1,0x8e,0xbc,0x50,0x82,0x4d,0x40,0x28,0xb8,0xfd,0xec,0xda,0x49,
			0xe8,0x10,0xba,0xe5,0xac,0xc8,0xad,0xc0,0xdc,0xa2,0xe2,0x36,0xfc,0x83,0x2a,0x97,
			0x33,0x0a,0x12,0x14,0xfa,0x0a,0xed,0x15,0xcd,0x10,0xc0,0x49,0xef,0xb6,0x5c,0xe8,
			0x55,0xc0,0x60,0xf0,0x5b,0xef,0xb3,0x17,0xb8,0x06,0x58,0x43,0xc4,0xeb,0x5a,0x03,
			0x71,0xfc,0x6f,0x20,0x9f,0x6f,0xfb,0x94,0x8c,0x88,0x1f,0x2f,0x20,0x91,0xca,0xf0,
			0xf5,0x9f,0x60,0xb7,0x2c,0x5f,0x67,0x27,0x1b,0xae,0x96,0xb9,0x13,0xfd,0x21,0xfa,
			0x1d,0xfa,0x97,0x5d,0x5e,0xcd,0x62,0xb0,0xd5,0x08,0x73,0xb6,0x86,0xd2,0x9c,0x88,
			0x0d,0x36,0xed,0xca,0xd3,0x3e,0xc3,0xe2,0x21,0x6c,0x9c,0xfc,0xfb,0x4f,0x98,0x4c,
			0x23,0xfd,0xe8,0x15,0xe2,0x80,0xa8,0x02,0x42,0x86,0x08,0xbe,0xd3,0x73,0x9a,0xf9,
			0x20,0x0d,0xe1,0xf8,0x5e,0xde,0xe2,0x83,0x4c,0x04,0x94,0x2c,0x06,0x8a,0xac,0xd2),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xbb,0x29,0x4b,0x95,0xd9,0x13,0x00,0x5b,0x11,0x09,0x87,0xcd,0xe4,0x58,0x87,0x48,
			0x4a,0xe6,0xdf,0x79,0x48,0x73,0xdf,0xc5,0xc4,0x1f,0xb7,0xe8,0x99,0x2c,0x2f,0xdc,
			0xe7,0x06,0x99,0xfc,0xac,0x80,0x04,0x69,0x99,0x61,0xb3,0xad,0x1e,0x1f,0xce,0x9e,
			0xc8,0xea,0x56,0x85,0xcc,0xec,0x5e,0x80,0xe4,0xd0,0x79,0x25,0x59,0x81,0x6f,0x68,
			0x61,0x34,0x34,0xbf,0xac,0xa8,0x1a,0x84,0x3a,0xac,0x45,0x9a,0x6f,0xe3,0x5f,0x53,
			0x69,0xc4,0x8e,0x91,0x91,0xe4,0xa3,0x2c,0x70,0x78,0x95,0x94,0xc5,0x15,0x2d,0xb8,
			0xd4,0xbb,0x02,0x26,0x00,0x12,0xa8,0x73,0x9c,0xf3,0x25,0xdd,0xff,0x2a,0xa4,0x2f,
			0xd6,0x7b,0x6e,0xe5,0xbf,0xe3,0x15,0x91,0x13,0x1f,0xf2,0x7d,0x02,0x73,0xd2,0x92),
		.s = chunk_from_chars(
			0x32,0x63,0x7c,0x60,0x79,0x8b,0x45,0x0b,0xff,0x10,0x0b,0xff,0x12,0x83,0x83,0x57,
			0xde,0xff,0x28,0x1d,0x5b,0x31,0xe4,0xf4,0xc2,0xcf,0xc9,0x6e,0xb7,0x79,0xce,0x6d,
			0x31,0xb1,0xce,0x8b,0xd7,0xaa,0x7f,0xa8,0x8d,0xdc,0x42,0x79,0xc8,0xc3,0x28,0x06,
			0x04,0xb0,0x18,0xcc,0xf4,0x52,0x00,0x4a,0x14,0x88,0xed,0x47,0x50,0x18,0x1c,0x50,
			0x25,0x63,0x65,0x11,0xac,0x67,0x24,0xfe,0x51,0x76,0x1c,0x27,0xd7,0xcf,0x9a,0x0c,
			0x87,0x82,0xea,0x22,0x31,0x26,0x88,0x53,0xc4,0xb1,0xf7,0xac,0xb0,0x00,0x5e,0x56,
			0x87,0xc8,0xf3,0xdf,0x16,0xc9,0x62,0xf0,0x2c,0xe5,0x6b,0x23,0xd3,0x87,0xa2,0xba,
			0xad,0xc8,0xbe,0xc9,0x42,0x29,0xc3,0x55,0x75,0x26,0xe6,0x17,0x07,0xa8,0xb5,0x92,
			0x93,0xa9,0x76,0xe3,0x2c,0x7f,0xa1,0x33,0x28,0x50,0x88,0xf3,0xce,0x3e,0x67,0x77,
			0x88,0xaa,0xa9,0x47,0xe7,0x62,0x2c,0x75,0x7e,0x84,0x4b,0x11,0x75,0x92,0xbe,0x99,
			0xfe,0x45,0x37,0x6f,0x8b,0x30,0x13,0xe8,0x77,0x2e,0xc9,0x2c,0x5b,0xb0,0xb9,0xfa,
			0x30,0x1b,0x95,0x54,0x45,0x99,0x69,0x0a,0xd9,0x36,0x68,0xd8,0x3b,0x2d,0xaa,0x7d,
			0xf0,0x5c,0x66,0x21,0x4e,0x27,0x50,0x14,0x78,0x0a,0x91,0x2d,0x8b,0x19,0x32,0xd7,
			0xa6,0x55,0x05,0x8e,0x74,0x3f,0x50,0xb0,0x74,0xb1,0xd9,0x69,0x1c,0xa2,0x3a,0x2f,
			0x95,0xf6,0xaf,0xfb,0xd5,0x16,0xd6,0x4c,0xcb,0x2a,0xa4,0x3c,0x23,0x6e,0xb9,0x5d,
			0x36,0xd2,0x72,0x54,0x5e,0x3b,0xeb,0x8f,0xf5,0xaa,0xcd,0x95,0xb3,0x0f,0x7f,0x1d,
			0x64,0x18,0xaf,0x04,0x2c,0xd9,0xa0,0xcf,0x01,0x89,0x84,0x62,0x62,0x32,0x2a,0x18,
			0x87,0x5a,0xe4,0xc3,0xe6,0x8e,0x4e,0x8f,0xfa,0xa0,0x27,0x6c,0xdd,0x99,0xa0,0x04,
			0x7c,0x86,0xc0,0xf7,0x1d,0x2d,0xee,0xfd,0x50,0x64,0x2d,0x29,0xc1,0x95,0xe6,0xd1,
			0x4f,0xb4,0x6f,0xba,0xc3,0x3a,0x50,0x8c,0x1f,0x03,0xa2,0x32,0xde,0x08,0xaa,0xe0,
			0x9f,0xaf,0x1d,0xa8,0xed,0x2b,0xa2,0xae,0x84,0xbc,0xca,0x88,0xb7,0x8d,0xcc,0xbd,
			0xe9,0xaf,0xde,0x08,0xa3,0xbe,0xb3,0x22,0xdc,0x79,0x35,0x6b,0x29,0xc8,0x48,0x41,
			0x69,0x89,0x14,0xb0,0x50,0xbe,0xb7,0x5a,0x7b,0x2f,0x67,0x01,0xaa,0x81,0x01,0xa5,
			0xa4,0x95,0x5e,0xe2,0x7b,0xaf,0xe8,0x1b,0x21,0xd0,0x3b,0x43,0xe3,0xc7,0x73,0x98),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xf9,0x46,0xc6,0xbd,0x5e,0x1d,0x6b,0x89,0x09,0x2f,0x3c,0x48,0x7c,0x05,0x68,0xfa,
			0x07,0xc3,0x56,0xfa,0xe9,0xb8,0xe8,0x31,0xb8,0x32,0x02,0x89,0x03,0x97,0x46,0xa4,
			0x35,0xb1,0x22,0xcf,0xbc,0x4a,0x0d,0x31,0x6b,0xf9,0x0d,0x48,0x1d,0x3b,0x7d,0x97,
			0x9c,0xc5,0x0d,0x98,0xc1,0x19,0x0a,0xf8,0xdc,0x58,0xe0,0x03,0x55,0x57,0xdd,0x5e,
			0x94,0xf4,0x37,0xf4,0x1f,0xab,0x51,0x32,0x02,0x64,0x3a,0x77,0x74,0x8f,0x76,0xc6,
			0xb7,0x73,0x02,0xbf,0x40,0xc3,0x92,0xcd,0x18,0x73,0x1d,0xa0,0x82,0xc9,0x9b,0xde,
			0xde,0xb7,0x0e,0x15,0xcd,0x68,0xbf,0xf5,0x96,0x19,0xca,0xbc,0xc9,0x2a,0xdc,0xf1,
			0x22,0x75,0x3c,0x55,0xaf,0xde,0x08,0x17,0x35,0x2b,0xc2,0x47,0xd1,0x17,0x0b,0x8d),
		.s = chunk_from_chars(
			0x50,0x70,0x6b,0xa4,0x9d,0x9a,0x31,0x66,0x88,0xa3,0xee,0x80,0xa0,0xbd,0x98,0x67,
			0x57,0xd4,0x3e,0xc8,0x32,0x85,0xaf,0x9e,0x78,0x19,0x6b,0xd5,0x2c,0x90,0x0d,0x40,
			0xb2,0x80,0xfa,0x0d,0xe5,0x4e,0x35,0xac,0xe7,0xd6,0x66,0x00,0x12,0xf1,0xa6,0x62,
			0x04,0x09,0x2f,0x0e,0x63,0x4b,0x97,0xe0,0xe5,0x16,0x65,0xb4,0x07,0x5e,0x36,0xf1,
			0x42,0x22,0x66,0xc7,0xca,0xd7,0xb2,0xd9,0x98,0x1b,0x91,0x3d,0xf3,0xfa,0x3e,0x6a,
			0x5a,0x1c,0xad,0xfc,0x63,0x78,0xa8,0x54,0x0e,0x0f,0xaa,0x26,0xf1,0xcc,0x6f,0xb2,
			0xfb,0x49,0x2a,0x80,0xd0,0xa6,0x94,0x5b,0xce,0x5b,0xbc,0x23,0xdd,0xb3,0xb1,0x07,
			0x01,0xf0,0x24,0x9b,0x27,0x40,0x7a,0x67,0x00,0x80,0x2e,0x88,0x42,0xef,0x3c,0xc7,
			0x61,0xc4,0x82,0x3a,0xcb,0x5d,0x14,0x53,0x50,0x8d,0xcd,0xbb,0x97,0x9e,0x7b,0xd8,
			0xd0,0x01,0x28,0xe6,0x0a,0x9b,0x37,0x89,0x16,0x7c,0x91,0x41,0x7d,0x93,0xf0,0xe9,
			0xfb,0xb0,0x0c,0x9a,0xf1,0x49,0x8e,0x09,0xeb,0x64,0x85,0xeb,0x94,0xce,0xa4,0x88,
			0x3f,0x6a,0x25,0x6e,0xab,0x2c,0xaa,0x82,0x6d,0xe4,0xfd,0xac,0x01,0xba,0xca,0x3a,
			0x21,0x6e,0x3d,0x20,0x4a,0x3d,0x83,0x7f,0xfd,0x4d,0x0b,0xe2,0xb2,0xce,0xf7,0x11,
			0x90,0x90,0x54,0xc4,0xda,0x1d,0x5b,0x93,0xa8,0xf9,0x84,0x51,0xc7,0x00,0x2a,0xe8,
			0x4a,0x5e,0x70,0x80,0xd9,0x86,0x71,0xc5,0x0e,0x3c,0x91,0xc4,0x08,0x7d,0x04,0x77,
			0xb1,0x04,0xf9,0x16,0x01,0x0e,0x74,0x2f,0x2d,0x20,0x7f,0xb4,0x0d,0x12,0x2d,0x8f,
			0x21,0x1a,0xf6,0xd7,0xc5,0xec,0xa4,0x95,0x42,0xd9,0xac,0xb0,0xf1,0x66,0xe3,0x6a,
			0xbc,0x37,0x15,0x50,0x70,0xc1,0x2e,0x9f,0x28,0xb9,0x07,0xd6,0x7a,0x2c,0xa7,0x0b,
			0xfc,0xe5,0x54,0xe1,0xc4,0x4c,0x91,0x52,0x0e,0x98,0xfc,0x9a,0xd0,0xc0,0xee,0x47,
			0x7f,0x75,0x05,0x16,0x47,0x6a,0x94,0x16,0x80,0x66,0xce,0x47,0x00,0x00,0x30,0xa9,
			0x9c,0x23,0xe2,0xc3,0x87,0x55,0xde,0x94,0x6d,0x5e,0xdf,0x0d,0x6a,0xa9,0x42,0x12,
			0xf9,0x92,0x31,0x5b,0x24,0x8c,0x1f,0x82,0x72,0x3b,0x29,0xc4,0x22,0x16,0xc7,0x8c,
			0xdc,0xb6,0x68,0xf1,0x12,0x78,0x26,0x1c,0xee,0x92,0x52,0xc8,0xfd,0x0e,0xd3,0x7d,
			0x0a,0x85,0x80,0xca,0x9b,0x9f,0xde,0x75,0x05,0x61,0x59,0x43,0x71,0x2d,0xa1,0x9a),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x9a,0x33,0x7d,0x4c,0x0b,0xb9,0xa0,0x05,0xb4,0x7f,0x47,0x65,0xd6,0x96,0xd1,0x9d,
			0xec,0x58,0xbc,0x84,0x82,0xf2,0x17,0x3a,0x4a,0x20,0x3a,0x0b,0x6d,0x38,0xb4,0x96,
			0x1f,0x6a,0x85,0x2e,0x76,0x46,0x8e,0x80,0x7c,0x7e,0x45,0x76,0x83,0xee,0xad,0x5c,
			0xb8,0xd9,0x86,0x42,0xfb,0x76,0xc0,0xa1,0xee,0xab,0x36,0x41,0x4c,0x18,0x99,0x59,
			0x7d,0x57,0xaa,0xf9,0x67,0x82,0xad,0xa5,0x86,0xf6,0x1a,0x42,0x3f,0x57,0x95,0x37,
			0x71,0xd5,0x20,0xcc,0x4e,0xad,0x90,0xd5,0x69,0xf2,0x3d,0x95,0x0f,0x8d,0xfe,0xdd,
			0xdb,0x83,0x55,0x74,0x85,0x76,0xe6,0xbb,0xfb,0x6f,0x2e,0x91,0xb3,0xda,0x71,0x75,
			0x3f,0xd2,0xf4,0xea,0x22,0x9f,0x6d,0x20,0xe2,0x7d,0xb8,0xd0,0x5e,0x9f,0xcb,0x68),
		.s = chunk_from_chars(
			0xcf,0xf7,0xaa,0x7f,0x87,0x56,0x42,0xfb,0x93,0x43,0xe0,0x7e,0xf5,0xe7,0x30,0x3b,
			0xbf,0x5f,0x06,0x9b,0x44,0xc1,0x9f,0xbf,0x83,0xe5,0x9d,0x42,0x2e,0x25,0x26,0x7e,
			0xf9,0x30,0x74,0x14,0xb6,0xb1,0xef,0x61,0x71,0x1e,0xd0,0x01,0x32,0x76,0xd1,0xa2,
			0xad,0x98,0x39,0x04,0x74,0x02,0x7a,0x0a,0x70,0x3b,0xfe,0x8a,0x6e,0x87,0x70,0x60,
			0x59,0xd8,0x9c,0x06,0x09,0x80,0xc9,0xc9,0xe6,0x0d,0xc7,0xe1,0xfb,0x9f,0x77,0x7a,
			0x41,0x78,0x5a,0xb4,0xd2,0xb6,0x63,0xba,0x0e,0x3c,0x19,0x21,0x54,0x5c,0x47,0x9c,
			0x2a,0x38,0x3a,0x50,0xda,0x8e,0x48,0x9c,0xb2,0x2b,0x71,0x10,0x1d,0x0e,0xc1,0x48,
			0xac,0x70,0x92,0x87,0x32,0xa7,0x72,0x19,0x5a,0x14,0x0d,0x08,0x01,0x52,0x76,0x2a,
			0x9c,0x40,0x80,0x3a,0x39,0xfa,0x2a,0x69,0x78,0xc2,0xa7,0x5a,0xc4,0xd8,0xbd,0x1b,
			0xcc,0xaa,0x1f,0x42,0x04,0xba,0x65,0xed,0xdd,0xf3,0x2f,0xed,0xf2,0xd9,0xd0,0xa3,
			0xae,0xd9,0xb0,0x6c,0x47,0xe7,0x17,0x73,0x3c,0x57,0x78,0x12,0xd7,0x23,0xdb,0xa7,
			0x4a,0x85,0x2b,0x29,0x05,0x23,0x5c,0x81,0x2d,0xc5,0xf1,0xd0,0xdf,0x0f,0x0d,0xe7,
			0x3d,0xfb,0x86,0x22,0x1c,0x6f,0xfd,0xd1,0xed,0xa1,0x19,0xbb,0xe9,0x8d,0x14,0x8a,
			0xdd,0x36,0xa4,0xfe,0x50,0x48,0x9b,0x06,0xaa,0xee,0xfc,0xb5,0xc2,0x06,0x6d,0x90,
			0xfa,0x79,0x73,0x87,0x06,0xcd,0x18,0xe4,0x74,0xd6,0x96,0x09,0xff,0x12,0x10,0xc7,
			0x7d,0xe7,0xcd,0x23,0xba,0x2a,0x77,0x5a,0x43,0x29,0xcb,0x27,0x1a,0x82,0x6d,0x60,
			0x2c,0x40,0x1a,0x71,0x43,0x90,0x19,0xce,0xc1,0x0c,0xd9,0xf1,0x84,0xc4,0xd0,0x45,
			0x84,0x21,0x18,0x27,0xb1,0x9e,0xad,0xac,0x32,0x58,0xd8,0xa0,0xf2,0x63,0x16,0x13,
			0xf0,0x51,0xaa,0xe0,0xc6,0x13,0x05,0x0c,0xb2,0x44,0x42,0xf1,0x5e,0xd4,0xfe,0x0d,
			0xbd,0x29,0x0e,0x42,0x62,0x91,0x41,0xbd,0x2c,0xd5,0x6d,0x20,0x58,0x4a,0x1d,0x10,
			0xe1,0xf2,0xc2,0xa9,0xec,0x73,0x14,0x33,0xd5,0xbc,0xd1,0xd3,0x18,0xbe,0xd5,0x24,
			0x3b,0x4b,0x7d,0x0f,0x9a,0x79,0x82,0x06,0x1c,0x55,0xdf,0xaa,0x86,0xb2,0xc0,0x18,
			0x45,0xc0,0x21,0xfd,0xd2,0xa9,0x78,0xd4,0x20,0x34,0x21,0x2f,0x43,0xb3,0x35,0x1b,
			0x6a,0xde,0xb0,0x3b,0xdd,0x6c,0xaf,0x7d,0xe0,0x59,0x50,0x2f,0x16,0xd7,0x73,0x48),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x32,0xfd,0x45,0xe7,0x3f,0x6f,0x69,0x49,0xf2,0x0c,0xab,0x78,0xc0,0xcc,0x31,0xd8,
			0x14,0xba,0xea,0x63,0x89,0x54,0x6a,0x36,0x5d,0x35,0xf5,0x4f,0x23,0xf1,0xd9,0x95,
			0xb7,0x41,0x01,0x18,0x77,0x60,0xc8,0x9b,0xb0,0xb4,0x0b,0x50,0x57,0xb1,0x82,0xe2,
			0xfa,0xfb,0x50,0xb8,0xf5,0xca,0xd8,0x79,0xe9,0x93,0xd3,0xcb,0x6a,0xe5,0x9f,0x61,
			0xf8,0x91,0xda,0x34,0x31,0x0d,0x30,0x10,0x44,0x1a,0x71,0x53,0xa9,0xa5,0xe7,0xf2,
			0x10,0xeb,0xe6,0xbc,0x97,0xe1,0xa4,0xe3,0x3f,0xd3,0x4b,0xb8,0xa1,0x4b,0x4d,0xb6,
			0xdd,0x34,0xf8,0xc2,0xd4,0x3f,0x4a,0xb1,0x97,0x86,0x06,0x0b,0x1e,0x70,0x07,0x0e,
			0x3e,0xd4,0xd5,0xf6,0xd5,0x61,0x76,0x7c,0x48,0x3d,0x87,0x9d,0x2f,0xec,0x8b,0x9c),
		.s = chunk_from_chars(
			0xc3,0x89,0x61,0x37,0x17,0xec,0x74,0x76,0xec,0xda,0x21,0x44,0xd0,0xe8,0xc8,0xf9,
			0xd6,0x6f,0xb4,0x69,0xc1,0x67,0xc4,0x20,0x9e,0xc0,0xbd,0xee,0xbf,0xb4,0x71,0x66,
			0x5d,0x33,0xda,0xd4,0x7b,0x8f,0x3c,0x31,0x9a,0x76,0xfe,0x8a,0x8a,0x9f,0x66,0x2b,
			0x6c,0x69,0x0b,0x74,0x90,0x3d,0x17,0xf6,0x1e,0x23,0x14,0xe5,0xea,0x8d,0x26,0x67,
			0x0e,0xe4,0xdb,0x4d,0xad,0x29,0x5b,0x27,0x7c,0xa0,0x8a,0xde,0x88,0x0d,0xe2,0xe4,
			0x2d,0x12,0xb9,0x29,0x52,0x76,0x4c,0x1d,0xc8,0x08,0xc2,0x66,0xdb,0xbe,0xdb,0x67,
			0x01,0x58,0xee,0xf3,0x6e,0x89,0x6f,0x55,0xa2,0x03,0xfb,0x99,0x55,0x6d,0xed,0x05,
			0x97,0x41,0x0b,0xa3,0x74,0x86,0xb1,0xd8,0x41,0xf3,0xd6,0xd5,0xc0,0xb3,0x9f,0x2f,
			0x49,0xf0,0xc5,0x79,0x48,0x24,0xfb,0xa9,0x4a,0x8e,0xc7,0xc2,0xb2,0xc9,0x1e,0xad,
			0xd5,0xc8,0xcb,0xe4,0x48,0x95,0xfe,0x3b,0xe3,0xbc,0x17,0x27,0xd6,0xfc,0x0e,0x53,
			0x64,0xf5,0x35,0x78,0x63,0x9d,0x3b,0x3a,0xf6,0x96,0xb7,0x50,0xa0,0x78,0x53,0x69,
			0x4f,0xfe,0x14,0x5a,0x28,0xc0,0x36,0x20,0xc7,0x8d,0xd7,0x37,0x7d,0x09,0x4d,0x92,
			0xc3,0xe0,0x95,0x46,0x88,0x3d,0x47,0x03,0xe6,0x2a,0x98,0xdd,0xf8,0x1f,0xd0,0x1f,
			0xcd,0xf3,0xc4,0xb2,0x15,0x22,0x4f,0xe2,0xb1,0xb4,0x99,0x2a,0xbf,0x31,0xf2,0x0d,
			0x12,0xaf,0xa8,0x68,0x20,0x23,0x90,0xde,0x33,0x4a,0x84,0x6b,0x2d,0x58,0xb2,0x53,
			0xea,0x8a,0xb3,0xc5,0x26,0x5d,0x84,0x77,0x3a,0x65,0x9e,0x8b,0xac,0x7a,0xf4,0x41,
			0x23,0xd9,0xea,0x15,0x06,0x2e,0x65,0xd4,0xd4,0x19,0xcf,0x2d,0x97,0x07,0x7d,0x06,
			0x24,0xf8,0xe5,0xc3,0x6f,0x2c,0x7b,0x35,0xcc,0xf9,0x54,0x35,0xd5,0xc3,0x68,0x86,
			0xff,0x91,0x05,0xa6,0xc1,0xea,0x22,0x5e,0x15,0xea,0x8c,0xbc,0x7b,0x6b,0xf6,0x85,
			0x61,0x51,0xcd,0x76,0xfb,0xb7,0x5b,0x5b,0x98,0xf0,0xe3,0xdb,0x51,0x6a,0x8e,0x21,
			0x81,0x89,0xfc,0xb1,0xcd,0x5d,0xe3,0xca,0xfe,0xaa,0x33,0xef,0x13,0x5c,0x5d,0x8b,
			0x8a,0xa5,0xf8,0x81,0xaf,0xaa,0xca,0xf4,0xc0,0x8b,0xd7,0x28,0x12,0x55,0xbc,0x2a,
			0x33,0xb7,0x6d,0x4a,0x36,0xe0,0xb1,0x70,0xc4,0x55,0x88,0x23,0x9e,0x5b,0x38,0xc6,
			0x79,0xb0,0x8c,0xf8,0x02,0xaf,0x73,0xb6,0xd7,0x9b,0x39,0x35,0x94,0x94,0x61,0xe7),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xab,0x66,0xcc,0x48,0x7e,0xc9,0x51,0xf2,0x11,0x9d,0x6e,0x0f,0xa1,0x7a,0x6d,0x8f,
			0xeb,0x7d,0x07,0x14,0x9b,0xec,0x7d,0xb2,0x07,0x18,0xe4,0xf3,0x1d,0x88,0xc0,0x1f,
			0x9a,0x53,0xd5,0xba,0x7e,0xce,0x3a,0x4d,0xbc,0x67,0xaf,0x6a,0x35,0xd1,0x30,0xea,
			0xe7,0x62,0xcb,0x79,0x62,0xb9,0xae,0x55,0x7c,0xa3,0x84,0x52,0x46,0x40,0x02,0x22,
			0x3f,0x61,0xbc,0xd3,0xc7,0x35,0x3e,0x99,0xd6,0x25,0x58,0xce,0xed,0xfc,0xb9,0x37,
			0x4d,0x4b,0xbf,0x89,0x68,0x0c,0x8e,0x2b,0x95,0x85,0x60,0x3e,0x07,0x6f,0x1c,0xdb,
			0x00,0x58,0x29,0x9b,0x42,0x46,0x84,0x5d,0xc7,0x9d,0x10,0x43,0xb1,0x42,0x2e,0xfe,
			0x84,0x01,0x8e,0x4c,0x93,0x2c,0x45,0xbe,0xb8,0x85,0x1f,0xbf,0x48,0x5e,0x36,0xd2),
		.s = chunk_from_chars(
			0xb5,0x13,0x31,0x55,0x2b,0x08,0xbe,0x35,0xa1,0x69,0x8a,0xa6,0x20,0x3d,0x84,0xdb,
			0xff,0xf9,0x00,0x1e,0xd5,0xdd,0x77,0x6f,0x2b,0xe4,0xdd,0xfc,0x07,0xdd,0x46,0x20,
			0xe9,0x65,0x4e,0x82,0xa3,0x34,0x65,0xbd,0x20,0xf1,0x18,0x63,0xc0,0xed,0x02,0xa0,
			0xae,0xa2,0x7a,0x44,0xd4,0x14,0xc3,0x28,0xa9,0x38,0xbf,0x87,0x7e,0x15,0x83,0x8a,
			0xb9,0x9d,0x67,0x0d,0x01,0x41,0x42,0x62,0xe8,0x86,0x5d,0xc1,0xd9,0xfc,0x30,0xfd,
			0x08,0x12,0x69,0x9f,0xa6,0x90,0xc3,0x4f,0x30,0x2f,0x63,0x7e,0xc8,0x02,0xcd,0x40,
			0xac,0x85,0x91,0xe9,0x76,0xc0,0xb8,0xbc,0xcb,0x1b,0x01,0x37,0xaf,0x64,0xa2,0x87,
			0x02,0x10,0xe8,0xfa,0x3d,0xc4,0x31,0xfe,0x09,0x56,0xb8,0xad,0xdf,0xf1,0xe4,0xb1,
			0x8c,0xf0,0x7e,0x07,0x8a,0xa9,0x3a,0xf8,0x1b,0xb3,0x02,0x3c,0x9e,0x59,0x4e,0x66,
			0x59,0x5f,0xd9,0x2b,0x10,0x22,0x6e,0xa1,0x26,0x00,0x5f,0x47,0x24,0x42,0x73,0x52,
			0xc3,0x8e,0x9e,0x85,0xfc,0x2e,0x07,0x23,0xf8,0x0a,0xf1,0xf6,0x15,0x99,0x55,0x0b,
			0x5e,0xf5,0x4c,0x5b,0x38,0xca,0x40,0x57,0x38,0x01,0x7b,0x89,0xcb,0x94,0x68,0xd9,
			0x74,0x1c,0xd6,0xbd,0xf7,0x11,0x21,0x62,0x25,0x1b,0xa1,0xd0,0x83,0xcc,0x37,0x0a,
			0x4a,0x82,0x61,0xc3,0x9b,0x6b,0x94,0xbf,0x21,0xa5,0x3b,0x75,0x64,0x53,0x1a,0xe9,
			0xeb,0xc4,0xcc,0xea,0x7e,0xbb,0x8b,0xd3,0x14,0xb2,0xe1,0x3b,0x58,0xed,0x10,0x18,
			0xae,0x5b,0x41,0x5e,0x0f,0x9e,0x3e,0x19,0xa5,0xea,0xd3,0xa4,0x46,0x03,0xf9,0x06,
			0x74,0xa1,0x90,0xfe,0xbd,0xe2,0x5f,0x8a,0xd8,0x77,0x8a,0xee,0xad,0x4d,0x0f,0x64,
			0xfb,0xae,0x37,0x16,0x6a,0x54,0xe3,0xa7,0x63,0xe3,0x55,0x59,0xbf,0x8c,0x3f,0x17,
			0x3f,0x19,0xff,0x7b,0xab,0x98,0xf3,0xef,0x80,0x3d,0xd5,0x6c,0x07,0x62,0x83,0x99,
			0xaf,0xf8,0x74,0x85,0xee,0x73,0xdb,0xc3,0xdb,0x34,0xec,0xc7,0xbf,0xf3,0xa5,0x32,
			0x26,0xcf,0x87,0xbc,0x81,0xd2,0x56,0xe8,0x0c,0x09,0x52,0x0c,0x8f,0x38,0xe9,0xbc,
			0xda,0x09,0x5e,0x36,0x35,0x12,0x8e,0x1b,0xed,0xd9,0x97,0x06,0x00,0x54,0x6a,0x75,
			0x1e,0xb1,0x1d,0xab,0x42,0xe2,0x89,0xd6,0xfd,0xfe,0xa0,0x4b,0xd5,0x8d,0x45,0x71,
			0xa7,0x9d,0x24,0xbc,0xe4,0x50,0x8c,0x54,0xe1,0xec,0x4c,0xf7,0x5b,0x98,0x5f,0xd3),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xfe,0xf7,0xfe,0x89,0xb9,0xa5,0x99,0x02,0xa7,0x0a,0x1d,0x9c,0xaa,0xd0,0x9c,0xed,
			0x8b,0xee,0x41,0x45,0xed,0xcb,0xe3,0xef,0x7f,0xa6,0xda,0xb3,0x76,0x35,0x12,0x9f,
			0x3b,0x8c,0x5e,0x08,0x60,0x41,0x0e,0xcb,0xd9,0xce,0xc3,0xd8,0x69,0x36,0x82,0xf2,
			0x5a,0xec,0x08,0xb0,0x71,0xf0,0x5d,0xc8,0x21,0x3b,0xac,0x8c,0xff,0x5d,0x52,0xb5,
			0x76,0x65,0x35,0x60,0xbc,0x01,0x57,0x56,0x04,0xe6,0xab,0x90,0xf6,0x72,0x27,0xfb,
			0x5c,0x90,0x1a,0x78,0x1e,0xdd,0xc0,0x27,0x70,0x09,0x13,0xe5,0x4a,0x7f,0xe5,0x13,
			0x18,0x48,0x2c,0x9a,0xb4,0x2c,0x9d,0x2b,0x91,0x1b,0x7c,0xcc,0x39,0xcc,0xb2,0x90,
			0xf9,0xa4,0x20,0xa5,0xda,0xd9,0x33,0x94,0xd4,0xd7,0xb8,0xc5,0x3f,0xe3,0xf2,0x42),
		.s = chunk_from_chars(
			0x45,0x06,0x8c,0xa6,0xd8,0x2f,0x2c,0x12,0x39,0x25,0xcd,0xe1,0x19,0x71,0x21,0x5d,
			0x8f,0xa4,0xa4,0xdf,0x68,0x48,0xbb,0x76,0x54,0x86,0x87,0x00,0x97,0x87,0x64,0x85,
			0x46,0x38,0x92,0x1b,0xea,0x58,0x69,0x28,0x0d,0xc6,0xad,0x95,0x81,0xab,0x43,0xff,
			0x70,0x12,0x96,0x99,0x48,0xa5,0x67,0x7f,0xa0,0xa6,0x61,0x36,0xa3,0x16,0xa4,0xbf,
			0xec,0xb8,0x9a,0xdf,0x41,0x31,0xb5,0xbe,0xdf,0x3d,0x46,0x93,0xb7,0x80,0xd1,0x33,
			0xaf,0x9b,0xf9,0xc1,0x33,0x30,0x5b,0xe7,0x83,0x74,0xaf,0xda,0x3b,0xa3,0x85,0x42,
			0x03,0x32,0x44,0x81,0xa9,0xd1,0x0b,0x9c,0xa9,0xb9,0x2d,0xc7,0xd7,0x4d,0xf5,0x31,
			0x87,0x2d,0xdf,0xc7,0x6c,0xaa,0x82,0xde,0x02,0x0e,0x2c,0x41,0x56,0x43,0xcb,0xcc,
			0x42,0x80,0xe6,0xd2,0xf4,0x37,0x1f,0xda,0x7d,0x92,0x49,0x31,0x4a,0x8f,0x43,0x76,
			0x48,0x99,0x1a,0x9b,0x03,0xd7,0x1b,0x58,0x39,0xad,0x38,0xa1,0x55,0x5a,0xd3,0x45,
			0x26,0x99,0x4b,0xa5,0x68,0x70,0xb6,0xea,0x18,0x01,0x12,0x95,0xf2,0xca,0x2b,0x07,
			0x13,0xb2,0xe9,0x2a,0xd7,0x76,0x80,0xc0,0xdc,0x5b,0xed,0x8d,0x3b,0x9b,0x31,0xac,
			0x14,0xdf,0x76,0x99,0x49,0xc4,0xa4,0x3e,0xa6,0x7f,0x6d,0xee,0xb3,0xdc,0x9e,0xd5,
			0x89,0xea,0x4e,0x8a,0x2c,0xf6,0x69,0x5d,0xf4,0x6f,0x94,0x6f,0x14,0x67,0xb2,0x8e,
			0x87,0x54,0x77,0xae,0x4e,0x64,0x50,0x80,0xfa,0xfd,0xa6,0xdd,0x55,0x1d,0x2c,0x02,
			0xfd,0x6b,0x2b,0x19,0x4f,0xc0,0xbd,0xb0,0x50,0xe0,0x6d,0x4c,0x78,0x41,0x05,0xf5,
			0xa3,0x3b,0x53,0xe7,0x30,0x98,0x05,0x59,0x63,0x07,0x1e,0xfc,0x1b,0xf3,0x97,0xfd,
			0x32,0x5f,0x3a,0x6f,0x4e,0x10,0xd7,0x6f,0x04,0x11,0xa0,0x01,0xe6,0x2e,0xc7,0x37,
			0x29,0x01,0x83,0x16,0xf5,0x63,0x10,0xf8,0x93,0xa5,0x93,0x63,0xd1,0xf6,0xfe,0x5c,
			0x17,0x44,0x4b,0x6c,0x72,0x8a,0x49,0x33,0xb7,0x52,0x12,0xfd,0xfa,0x25,0x8e,0x40,
			0x18,0xb7,0x76,0x39,0x51,0xab,0x4e,0x50,0x96,0x41,0x1d,0xf9,0xe5,0xbc,0x16,0xdf,
			0x38,0x96,0xe4,0x6c,0x97,0x3d,0x32,0xac,0x92,0x76,0xa4,0xe2,0xb5,0xb8,0x0e,0x3d,
			0x8d,0x79,0x8d,0xc0,0x47,0x0b,0x45,0x09,0x6b,0x4d,0x73,0x86,0x69,0xce,0x05,0x2e,
			0xd8,0x18,0xe5,0x60,0xaf,0x1e,0x92,0xc9,0x15,0x18,0x7d,0x66,0xcc,0x30,0x8b,0x70),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x82,0xb3,0x84,0x0e,0xeb,0x95,0xc9,0xc5,0x77,0x24,0xc7,0x0f,0x11,0x2b,0x6c,0x2d,
			0xc6,0x17,0xc3,0x17,0x85,0xac,0xd0,0xc8,0x23,0xf8,0xbc,0xdd,0xa2,0x85,0x32,0x5e,
			0xb3,0xd3,0x08,0xdc,0x79,0x05,0x22,0xbc,0x90,0xdb,0x93,0xd2,0x4e,0xe0,0x06,0x32,
			0x49,0xe5,0x5d,0x42,0x19,0xad,0x97,0x14,0x5f,0xea,0xf7,0xf3,0x06,0x68,0x62,0x3c,
			0xc8,0x89,0x0a,0x70,0xf4,0xf1,0x49,0x86,0x6f,0x82,0xcf,0x86,0xf9,0x8b,0x00,0x53,
			0xb2,0x3c,0x98,0xc8,0xdd,0x5e,0x91,0x07,0xe3,0x41,0x46,0x0e,0x9b,0xf5,0xd8,0x8c,
			0xc8,0xbc,0xd1,0xf2,0xe4,0xc0,0x07,0xcc,0x1c,0x02,0xc4,0x52,0x9b,0x93,0x23,0x3a,
			0x0b,0x06,0xbd,0xd1,0x59,0x25,0x85,0x4a,0xb9,0xe3,0xf1,0x56,0xeb,0x92,0x5b,0xf5),
		.s = chunk_from_chars(
			0x05,0x93,0xb9,0xfd,0x44,0x21,0x45,0x23,0x76,0xd2,0x7b,0xc7,0xa2,0x80,0x10,0x1c,
			0xfd,0x6e,0x88,0xa6,0x72,0x7d,0x7d,0x77,0xcf,0x65,0xce,0xb7,0x23,0xec,0xd2,0x57,
			0xf3,0x2f,0xe1,0x02,0x77,0xe8,0x57,0x98,0xe0,0xda,0x75,0x91,0x77,0x36,0xda,0x1a,
			0x3b,0xfc,0x22,0xad,0xc7,0x65,0x8f,0xbb,0x84,0xda,0x6e,0xbe,0xa0,0xb0,0x7d,0x1c,
			0xc4,0x05,0x73,0x2f,0xb0,0x40,0xb5,0x85,0xc1,0xb6,0x3c,0x80,0x34,0x06,0x9b,0xff,
			0xb8,0x22,0x06,0x56,0xf1,0xac,0x54,0xce,0x69,0x37,0x20,0xd6,0xfb,0x1b,0x5a,0xec,
			0x67,0xb0,0x3c,0x88,0x7c,0x80,0x77,0xda,0x14,0x8d,0x10,0xf4,0x8a,0xf7,0xc0,0x28,
			0xf9,0x92,0xb1,0x8f,0x13,0xc0,0xe5,0x75,0x30,0xc0,0x86,0xd7,0x75,0x48,0x3d,0xa5,
			0xf6,0x6f,0x3a,0x6a,0x19,0x18,0x78,0x68,0x34,0x0a,0xc6,0x3c,0x62,0x12,0xbc,0xbd,
			0x6c,0xbb,0x7b,0xed,0xa8,0x62,0x0a,0xfd,0x9b,0x66,0xde,0x47,0x47,0x3e,0xf2,0x4d,
			0x1b,0x6a,0x36,0xf4,0xec,0xe9,0xad,0xd4,0x95,0x14,0xfd,0xf1,0xd8,0x4c,0x7a,0x78,
			0x5b,0x7f,0x0e,0x00,0xf3,0x82,0x23,0x58,0x99,0x79,0x0f,0x47,0x2d,0x13,0xf4,0x85,
			0x58,0xa4,0x31,0x47,0x42,0xf3,0x76,0x80,0x8d,0xec,0x96,0xed,0xd2,0xe2,0x29,0xe9,
			0x43,0xf7,0xb9,0x83,0xbe,0xa5,0xec,0x6e,0xdf,0xa5,0xe9,0xbb,0x37,0xf5,0x88,0xe5,
			0x5e,0xf6,0x2e,0xbc,0x92,0x14,0xbe,0xaf,0x9d,0xa5,0x02,0x43,0x4e,0x10,0x88,0xdf,
			0x27,0x2c,0x6c,0x77,0xc1,0xe1,0xd8,0x97,0xc4,0x7b,0xea,0xb7,0x7e,0x3b,0xbe,0x31,
			0x7f,0x8d,0x43,0xd2,0x1f,0xd7,0xe9,0x43,0x37,0xc7,0xe2,0x63,0xe2,0x86,0x7b,0xf5,
			0x80,0xa2,0xa8,0xec,0xb9,0xe3,0x6a,0xb7,0xd3,0xe1,0xd5,0xcf,0x9a,0x23,0x23,0x09,
			0x53,0xd5,0x9d,0xf0,0xd7,0xe2,0x35,0x58,0xfb,0x61,0x2b,0x79,0x18,0xab,0xba,0x31,
			0xb1,0x64,0xce,0x17,0x88,0x18,0xa1,0xa9,0xe6,0xb6,0x68,0x7f,0x4d,0xe6,0x85,0xd7,
			0x0e,0x16,0xbe,0xf6,0xe1,0x92,0xfa,0xed,0xfe,0x0b,0x2b,0x95,0x47,0x7d,0x37,0xb0,
			0xa3,0xa2,0xd0,0x02,0xf3,0x3e,0xf4,0x32,0x1c,0xb9,0x05,0x04,0x0c,0xe0,0x6f,0xda,
			0x1c,0x98,0xa0,0x08,0x76,0x7f,0xbc,0x78,0x1a,0x1e,0xaf,0x33,0x75,0xda,0xb8,0x66,
			0x4b,0x59,0x03,0x36,0xb9,0x9e,0x15,0x7b,0x86,0x87,0xa6,0x60,0x2f,0xef,0x6a,0x3b),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0xe1,0x53,0xcc,0xa4,0x43,0x1e,0xd9,0x71,0x3f,0x47,0x44,0xba,0x05,0x4f,0x5f,0x19,
			0x1c,0xb3,0x7b,0x28,0x01,0x08,0xae,0x3a,0x11,0x4a,0xd3,0x49,0xa8,0x72,0xd1,0x30,
			0x8b,0x46,0x21,0x1a,0x83,0x75,0x8a,0x3b,0x4b,0xe3,0x2f,0xbe,0xac,0x42,0xcc,0xfe,
			0xe7,0xe2,0x3d,0xf8,0x53,0xca,0x40,0x01,0x47,0x07,0x7b,0xb4,0x3a,0x44,0xc1,0x2f,
			0x29,0x9b,0x91,0x7f,0x3a,0xab,0xdf,0x58,0x9e,0xeb,0x17,0x09,0xbb,0x3d,0x60,0xb0,
			0x8b,0xc7,0x1e,0xaa,0x3f,0xfe,0xba,0x4e,0x29,0x03,0xa5,0xdb,0xd8,0x33,0x9a,0xae,
			0x85,0xfa,0x24,0xb9,0xae,0xe7,0x61,0x30,0x00,0x06,0x05,0x85,0x7a,0x6a,0xa1,0x97,
			0xd0,0x09,0x26,0x27,0x0d,0xcd,0xa5,0x8b,0x7d,0xe7,0x58,0xa6,0xca,0x67,0xe6,0x17),
		.s = chunk_from_chars(
			0xa8,0x35,0xcd,0x41,0x46,0xbe,0xf4,0x65,0x64,0x2d,0x49,0x49,0x36,0x26,0x8a,0x31,
			0x1a,0x54,0x90,0xd2,0xc9,0xf9,0x16,0x6c,0x6c,0xe9,0x82,0x16,0xa9,0xa2,0x3a,0x64,
			0x35,0x97,0x30,0x0a,0x00,0x50,0xe6,0x44,0x5a,0xbd,0x5a,0x9b,0xfc,0x7a,0x2d,0x9b,
			0x70,0x72,0x6c,0x82,0x4c,0x38,0x3b,0xf5,0xac,0xad,0xdd,0xdc,0x34,0xd4,0x34,0xa3,
			0x1e,0x53,0x14,0xd2,0x5f,0xb5,0x8e,0x25,0x8f,0x51,0x88,0x66,0xc1,0x36,0xe5,0x28,
			0x55,0xc1,0x6f,0xe6,0x4f,0xf8,0xf1,0xc4,0xd6,0x6c,0x4e,0x9e,0x39,0xb8,0xcb,0x11,
			0x96,0xd8,0x09,0x44,0xd0,0x74,0x6c,0x0a,0x3e,0x17,0x69,0xcd,0x41,0x67,0xdf,0x72,
			0xab,0x5e,0x4c,0x9d,0xba,0xe9,0xcb,0x35,0xf4,0x82,0x8e,0x12,0x09,0x9f,0x9b,0x36,
			0xa5,0xa7,0x0c,0x48,0xd4,0xae,0xc9,0x87,0x2d,0x7b,0x19,0xe1,0x29,0x1b,0x33,0xcb,
			0xdf,0x08,0xa2,0x26,0x3d,0x50,0x0c,0x0a,0x83,0xb5,0x23,0x7e,0xf6,0xce,0x92,0xde,
			0x34,0x4b,0x3b,0x41,0xd0,0xd0,0x74,0x04,0xfc,0xd5,0x46,0x7b,0x04,0x6b,0x52,0xb8,
			0xf8,0x5f,0xc6,0xb5,0xd7,0xaf,0xc4,0x37,0xf1,0xee,0x9e,0x78,0x39,0x0c,0xa9,0xbb,
			0x6c,0xec,0x61,0x88,0x85,0xec,0xe2,0x97,0x58,0xf2,0xfd,0x6f,0x4e,0x5f,0x4f,0x89,
			0x69,0x35,0xde,0x5f,0x67,0xcc,0x04,0x05,0x5a,0x4c,0x4c,0x0f,0xba,0x5d,0xef,0x8d,
			0x2c,0xaa,0x17,0x93,0x31,0xa8,0x55,0x01,0xed,0x25,0x82,0x2a,0xe7,0x9d,0xa9,0xbc,
			0x81,0x5c,0xc3,0x9c,0x6a,0x97,0x92,0x11,0x08,0x3e,0x86,0x83,0x13,0x6c,0x94,0x2e,
			0x1e,0x17,0xe9,0xeb,0x8f,0x84,0xaa,0xcf,0x09,0x1a,0xa1,0xe5,0x16,0x65,0xfa,0xe4,
			0x46,0xbc,0x48,0xc3,0x04,0xaf,0x65,0x39,0x1f,0x27,0x9a,0xfb,0x98,0xb9,0x2e,0x04,
			0xc2,0xb7,0x3d,0x9d,0x94,0xe9,0x91,0x19,0x8f,0xe7,0x78,0x1f,0x0f,0x96,0x96,0xfc,
			0xba,0x2c,0x03,0x48,0x5f,0x76,0xe6,0xde,0x30,0xb9,0x53,0x5c,0xf3,0x90,0x3d,0xb2,
			0xf3,0xaf,0xa8,0x51,0xa4,0x7b,0xcd,0xe7,0x2d,0x4e,0xd2,0xe8,0xfa,0xbf,0x9b,0xb7,
			0xd4,0x69,0x6c,0xb4,0xab,0x8c,0x28,0x9b,0x0c,0x21,0xe1,0xf9,0x79,0xeb,0xc5,0x32,
			0xe2,0x80,0xcd,0x90,0x10,0xdf,0x4e,0xe7,0x2f,0x84,0xbb,0x9e,0x82,0x75,0x28,0x28,
			0xf1,0x67,0x03,0x0c,0x0f,0xe3,0x48,0xeb,0xc3,0x1e,0xc1,0x7b,0x8f,0x07,0xd9,0x4b),
	},{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x9c,0x63,0x89,0x9d,0xfc,0x7b,0xdc,0x0d,0xb3,0x84,0x72,0x72,0x44,0xca,0xf7,0x1e,
			0xcf,0xb9,0xb8,0x79,0x2b,0x9f,0x57,0xe9,0x36,0xb3,0xc2,0xf5,0x69,0x55,0x65,0xa9,
			0xb0,0x97,0x9f,0x3c,0x78,0xfd,0x73,0xf0,0x09,0x81,0x81,0x3a,0x16,0xda,0x34,0x23,
			0x92,0xfe,0x3c,0xee,0xc6,0xe6,0x3f,0xfb,0xa1,0x91,0xcb,0xeb,0x4f,0x4b,0x90,0x05,
			0x0d,0x2f,0xcc,0xd8,0x3b,0xeb,0x06,0x22,0xb2,0xc3,0xff,0xf1,0x59,0xd9,0xe6,0x08,
			0xf3,0xab,0xcb,0x84,0x3b,0xdd,0x56,0xc0,0x33,0x39,0xb9,0x75,0xb9,0xf4,0xe3,0x26,
			0x5b,0x32,0xf6,0xbb,0x6c,0xcd,0xfc,0x6c,0x57,0x52,0xd6,0xe0,0x34,0x4d,0x74,0x96,
			0x99,0xc7,0x4c,0x85,0xb3,0x0c,0x04,0xff,0x95,0xb2,0x72,0xdb,0xcf,0xd6,0xc7,0xd3),
		.s = chunk_from_chars(
			0x4d,0x38,0xa2,0x97,0x30,0x2a,0xd0,0x77,0x0d,0x97,0x29,0xce,0x5b,0x72,0x12,0xee,
			0xf2,0x87,0xce,0x02,0x50,0xf4,0x03,0xe3,0x2b,0x4a,0xcc,0x36,0x17,0xdc,0x0d,0x2e,
			0xdc,0xcc,0xc2,0xd5,0x80,0xdd,0xbd,0xbc,0xa5,0x72,0x2b,0x70,0x70,0x40,0x58,0xa3,
			0xb8,0x07,0xf5,0x92,0xe4,0x00,0xbd,0x56,0x3f,0xca,0xa8,0xb0,0x66,0xa6,0x14,0xb4,
			0x90,0x6f,0x14,0x33,0x96,0x8e,0xd2,0xf5,0x20,0xa2,0xf6,0xb0,0x34,0xd4,0xb2,0xd6,
			0x89,0x0a,0x24,0x1a,0xfd,0x1a,0xdb,0x86,0x39,0xa6,0xca,0xd9,0xdb,0xfd,0x2e,0x27,
			0x8d,0xfe,0xbf,0x79,0x74,0x0d,0x75,0xf2,0x95,0x75,0x9d,0x29,0x13,0x0b,0x19,0xab,
			0x19,0x98,0x3d,0xd6,0x8f,0x77,0x9d,0xe4,0x1f,0xfe,0xfd,0x4e,0x82,0xb5,0xe6,0x2f,
			0x72,0xf9,0x0e,0xfb,0x73,0x43,0x7f,0x08,0xa2,0x50,0x3d,0xd9,0x81,0x9d,0xae,0x20,
			0xba,0x97,0x06,0xc1,0x99,0xde,0x9c,0xf8,0x84,0x43,0x3e,0xeb,0x75,0x62,0x86,0xa8,
			0x5e,0xae,0x14,0xbf,0x9f,0x6d,0xbe,0xb7,0x05,0x46,0x1d,0x91,0x82,0x22,0x82,0xf1,
			0x8e,0xfb,0xb1,0x05,0x89,0xa5,0x78,0xf2,0xc9,0xc3,0x45,0xb0,0x79,0xa7,0xe9,0xdd,
			0x07,0xfd,0x4b,0x34,0x05,0x1b,0x27,0x11,0x97,0x29,0x90,0x6c,0x77,0xdf,0xb7,0xd2,
			0xf8,0xfa,0x6b,0xdd,0x5f,0xaa,0x1e,0x13,0x2b,0xfb,0xa9,0xd3,0x91,0xe6,0x63,0x95,
			0xe6,0x7f,0x01,0x35,0x3f,0xa2,0x75,0xea,0xce,0x8b,0x53,0xaa,0x91,0xcb,0x6f,0xb6,
			0x93,0xe1,0x91,0x91,0xd4,0x2a,0x4c,0x1a,0x85,0xa0,0xc5,0x04,0xb1,0xc8,0x5f,0x49,
			0xa4,0xd6,0x09,0x36,0xde,0xe4,0x64,0x6a,0xca,0x62,0xa9,0x4a,0xa4,0xbc,0x78,0x28,
			0xc1,0xff,0xaf,0xde,0x8b,0xe6,0x56,0x31,0x7d,0x50,0x6a,0xbe,0xc1,0x79,0xcc,0x90,
			0x19,0x1d,0x12,0x35,0x6f,0xf5,0x06,0x44,0xd3,0xe0,0x1a,0xa5,0xbc,0xfd,0xd7,0x1d,
			0x3c,0x82,0x8d,0xc3,0x53,0x9d,0xc0,0xcf,0x3f,0xe8,0xb9,0xb9,0x1e,0x0c,0x25,0x24,
			0xf6,0xa3,0x71,0x03,0x79,0xc9,0x0a,0xff,0xd0,0xd0,0xa5,0x0d,0x74,0x38,0x7f,0x9c,
			0xa8,0x8b,0x46,0x46,0x3e,0xf1,0xbd,0xba,0x58,0xcc,0x9a,0x36,0xe5,0xc2,0xc4,0x35,
			0xa2,0x0d,0x96,0x83,0x50,0xd1,0x5d,0x94,0x1c,0x32,0x12,0xcd,0xce,0x81,0x55,0x92,
			0xb3,0x10,0xd2,0x59,0x86,0x0d,0xe1,0xdc,0x1a,0x3d,0x70,0xac,0x22,0x30,0x2a,0x51),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_384,
		.m = chunk_from_chars(
			0x04,0x84,0x6c,0x2e,0x67,0x6a,0xc7,0x31,0x60,0xbf,0x4e,0x45,0x65,0x2b,0xdc,0x6c,
			0xc4,0xd4,0xc9,0x28,0x45,0x77,0xb4,0x32,0x0a,0xb7,0x7f,0x6e,0xbb,0xb5,0x9a,0x1f,
			0xe0,0xe0,0x85,0x58,0x8e,0x0f,0x90,0xb3,0x46,0xcd,0xe6,0x44,0x1a,0xf3,0xc9,0xd0,
			0x11,0x7d,0x1f,0x3b,0xcd,0x96,0x2e,0x40,0x6b,0xf5,0xa4,0x65,0xab,0x6c,0xda,0x2d,
			0x51,0xbe,0x59,0x8f,0xcb,0xb2,0x9e,0xa7,0x13,0x65,0x1a,0xac,0xd7,0xe4,0x7d,0x22,
			0xd8,0xfa,0x34,0x50,0x90,0x47,0x30,0xf5,0x17,0x92,0xea,0x37,0x47,0x61,0xa4,0xdc,
			0x1f,0xc6,0xf1,0xbc,0x65,0x7b,0x77,0x76,0x8f,0x31,0xf4,0x63,0xe4,0x26,0x7f,0xc8,
			0xdf,0xf6,0x11,0x50,0xd4,0xb3,0x43,0xb9,0xd5,0x37,0x59,0xcd,0xd7,0xb9,0x80,0x94),
		.s = chunk_from_chars(
			0x10,0x3b,0xee,0x57,0xe2,0x5b,0xe8,0xc3,0xa2,0xf7,0x74,0xe7,0x39,0xb4,0x7f,0x93,
			0x43,0x5e,0x41,0x49,0x32,0xc0,0x49,0x4b,0x6b,0x6a,0xa2,0x47,0x5b,0xf7,0xc9,0x30,
			0x5c,0x73,0x74,0x7e,0x0a,0xdf,0x82,0xc2,0x03,0x20,0x07,0xb3,0xf7,0x5a,0x69,0xc9,
			0x31,0x12,0x61,0x7a,0x62,0x56,0x6c,0x5a,0x2d,0xea,0xa2,0x5f,0xb9,0x52,0x09,0xda,
			0x49,0xfe,0x9c,0x16,0x1c,0xb2,0xff,0xa4,0x0f,0xd9,0xd7,0x7f,0x1f,0xf6,0x60,0xc8,
			0xb6,0xcd,0x3b,0x54,0xe3,0xe7,0x9a,0x75,0x9c,0x57,0xc5,0x71,0x98,0x02,0xc9,0x31,
			0x1d,0xb7,0x04,0xba,0x3c,0x67,0xb4,0xa3,0x11,0x37,0x54,0xa4,0x1b,0x8d,0xa5,0x9c,
			0x64,0x5b,0xe3,0x90,0x9e,0x7d,0xb7,0xe7,0xcf,0x72,0x94,0xda,0xb4,0x4f,0x74,0x24,
			0x0f,0x81,0xa2,0x81,0xee,0xcd,0x6e,0xf3,0x1c,0x7c,0xf1,0x8b,0x1a,0x19,0xc7,0xd0,
			0x2a,0x31,0x2b,0x91,0xd6,0xed,0xfa,0xa9,0x54,0x46,0x2d,0x34,0x74,0x0a,0xf5,0xab,
			0x70,0x8d,0xb5,0xa1,0x0b,0x00,0xc5,0x42,0xbe,0x82,0xfa,0x2b,0x20,0x26,0xb0,0x9e,
			0xf3,0x8a,0x40,0x01,0x45,0x7e,0x27,0xa6,0x02,0x37,0x70,0xe4,0xb4,0xd5,0x00,0x32,
			0x67,0xc8,0x5c,0x9e,0xea,0x1d,0x5f,0x8d,0x77,0x0b,0xd4,0x0b,0x55,0x4d,0x5b,0x4d,
			0xaf,0x14,0x6d,0xcc,0xab,0xac,0x3e,0xa8,0xa1,0x3a,0x05,0xc3,0xbd,0xdf,0xc9,0x71,
			0xc5,0x15,0x8f,0xac,0x02,0x7c,0xa1,0x9b,0x72,0x32,0x62,0x1e,0x9d,0x2e,0x37,0xb6,
			0xa6,0x55,0xaf,0x54,0x5e,0x44,0xa2,0x98,0xbe,0x78,0xcd,0x47,0x5c,0x22,0xa4,0x8b,
			0xff,0x7c,0x34,0x94,0xa5,0xf8,0xa6,0xab,0xdf,0x1a,0x46,0xf9,0xde,0x08,0x2e,0x37,
			0x4f,0xd5,0x98,0x86,0x7d,0x61,0xe4,0xd5,0x1d,0xae,0xd8,0x41,0x52,0xe4,0x3c,0xc6,
			0xa2,0xaf,0xfa,0xe2,0x05,0xed,0xc5,0x26,0x13,0x48,0x0d,0x41,0x1a,0xba,0x84,0xfc,
			0xc9,0xb6,0x9d,0x1c,0x28,0xf1,0x6f,0x76,0x83,0x69,0x01,0xa7,0xc5,0xb3,0xeb,0x2f,
			0x2c,0x94,0x0d,0x0a,0x3f,0xad,0x38,0xa8,0xef,0xab,0x96,0x8a,0x0c,0x85,0xeb,0x22,
			0xe1,0x1d,0x3d,0x08,0x61,0x13,0x6c,0xed,0x5f,0x06,0x73,0x4f,0xdf,0x8d,0x4f,0x15,
			0x1d,0x23,0x86,0x1b,0x1c,0xba,0x9b,0x9c,0x58,0x0d,0x33,0x50,0xc7,0x6d,0x4d,0xc8,
			0x08,0x46,0x1d,0x5f,0x87,0x2e,0xc5,0x48,0xb2,0xb4,0x27,0xdf,0xf7,0x4b,0x1d,0x1a),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xdb,0x6c,0x9d,0x4b,0xad,0xb1,0xd9,0xb7,0x4d,0x68,0x34,0x64,0x48,0xb4,0xd5,0x34,
			0x06,0x31,0x78,0x3b,0x5a,0x35,0xac,0x24,0x58,0x56,0x3e,0xd0,0x67,0x2c,0xf5,0x41,
			0x97,0x58,0x7f,0xb7,0x34,0xc4,0xac,0x18,0x9b,0x2d,0xda,0x95,0x4c,0xdf,0xb1,0x8b,
			0x41,0xc0,0x10,0xa7,0x7e,0x90,0x46,0x4e,0xea,0x6f,0x86,0x3c,0x5d,0xa0,0x95,0x6b,
			0xfa,0x8c,0xc6,0x36,0xbf,0x0a,0x28,0xbe,0x5a,0xdd,0xfe,0x8d,0x3e,0x7e,0x6f,0x79,
			0xf7,0x1d,0x7f,0xcb,0xba,0xe2,0x3e,0xa1,0x41,0x78,0x3f,0x91,0xd6,0xcc,0x4c,0x8f,
			0xad,0x12,0x58,0x11,0x76,0x0a,0xb5,0x71,0x33,0x81,0x88,0x92,0x47,0x1a,0x79,0xc6,
			0xd0,0x4e,0xaf,0xef,0x37,0xb2,0xfb,0xe5,0x06,0x78,0x53,0x18,0xf9,0x39,0x83,0x77),
		.s = chunk_from_chars(
			0xd4,0x80,0xd5,0xa9,0x79,0xad,0x1a,0x0c,0x4c,0xa3,0x29,0xeb,0xd8,0x8a,0x4a,0xa6,
			0x94,0x8a,0x8c,0xf6,0x6a,0x3c,0x0b,0xfe,0xe2,0x25,0x44,0x09,0xc5,0x30,0x54,0xd6,
			0xff,0xf5,0x9f,0x72,0xa4,0x6f,0x02,0xc6,0x68,0x14,0x6a,0x14,0x4f,0x8f,0x2b,0xa7,
			0xc4,0xe6,0xb4,0xde,0x31,0x40,0x0e,0xba,0x00,0xae,0x3e,0xe8,0x75,0x89,0xdc,0xb6,
			0xea,0x13,0x9e,0x70,0xf7,0x70,0x4f,0x69,0x1b,0xc3,0x7d,0x72,0x2f,0x62,0xbb,0x3b,
			0x2c,0xd3,0x03,0xa3,0x4d,0x92,0xfd,0xe4,0xde,0xb5,0x4a,0x64,0xdd,0x39,0x18,0x43,
			0x82,0xd5,0x9c,0xca,0xf0,0xc0,0x7a,0x7e,0xa4,0x10,0x7d,0x08,0x08,0x26,0x0e,0xd8,
			0xd4,0x21,0xcb,0x8b,0x14,0x07,0xcd,0xf9,0xe9,0x15,0x15,0x92,0x82,0xb9,0xf7,0xbf,
			0xfd,0xbf,0x40,0xd8,0x77,0x88,0x5d,0xa7,0x39,0x9e,0xde,0xbd,0x30,0x0a,0x7e,0x77,
			0xa9,0x08,0xf7,0x56,0x65,0x9a,0x18,0x24,0xf9,0x5c,0x8a,0x81,0x2a,0xa5,0x40,0xeb,
			0xaa,0x64,0xab,0x54,0xa2,0x33,0x72,0x3d,0xb5,0x5c,0xaa,0x8b,0x44,0x66,0xea,0x9a,
			0xe6,0x61,0x4a,0xd1,0xbb,0x86,0x9e,0x9d,0x8e,0x0d,0x03,0x2f,0x39,0x01,0x67,0x1e,
			0x94,0xc0,0xb6,0x73,0xbe,0x65,0x37,0xcd,0x54,0x27,0x8e,0xd3,0xda,0x2e,0x1e,0xdb,
			0xc0,0x4e,0xe3,0xa9,0xe8,0x07,0x0d,0x73,0xba,0x0f,0xfb,0x93,0xe6,0x0f,0x30,0xb8,
			0x7f,0xf3,0x86,0x2e,0x9c,0x53,0x90,0x8f,0x2c,0x8e,0x99,0x91,0x56,0x68,0xc1,0xf4,
			0x66,0x35,0xe0,0x5b,0xf7,0x16,0x30,0x51,0xff,0x9d,0x92,0xbc,0x71,0xa6,0x26,0x55,
			0x3c,0x69,0xdf,0xdd,0x06,0xa4,0x9f,0x7f,0xf1,0xed,0x51,0xe9,0x18,0xf3,0xed,0x80,
			0x1d,0xae,0x62,0xca,0x27,0x6d,0x70,0x63,0xd7,0x2a,0x6e,0xbc,0x13,0x6b,0xa0,0x6c,
			0xfe,0xdf,0x5a,0xa2,0x32,0x77,0xe8,0x10,0x08,0xc6,0x3b,0x2e,0x00,0x83,0xd0,0xfd,
			0x68,0x14,0xf6,0xd4,0xb4,0xb4,0x0a,0x42,0xe8,0xc0,0x20,0x6f,0x3c,0x35,0x6a,0x5e,
			0xc7,0x09,0xb7,0xc8,0xa4,0xb7,0x4b,0x7b,0x48,0xd5,0x3c,0x9d,0x86,0x94,0xd2,0x73,
			0x59,0xc2,0xc7,0x70,0x19,0x38,0xd2,0xf0,0x16,0x17,0x21,0xa5,0x73,0x13,0xbb,0x1a,
			0x2e,0x11,0xda,0x21,0x58,0x72,0x49,0x81,0x82,0x49,0x3d,0x85,0x17,0x04,0x3b,0x4c,
			0x03,0xf9,0x34,0x46,0xaa,0xc9,0x38,0x30,0x27,0x65,0x42,0x02,0x6c,0xe8,0x30,0x55),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xd5,0xdd,0x3b,0x6c,0xe9,0x77,0x2d,0x9a,0x97,0xfe,0x21,0x64,0x84,0x97,0x78,0x3b,
			0xac,0x5b,0xb5,0x25,0x4a,0xad,0x82,0xb6,0xf7,0xcb,0xf4,0x3b,0x15,0xa4,0x0f,0x38,
			0x6e,0xea,0x8d,0x15,0x19,0x67,0xdb,0x14,0x9e,0x94,0x65,0x86,0x59,0x68,0x13,0x3f,
			0x24,0x6e,0x13,0x47,0x30,0x1a,0xda,0xd2,0x34,0x5d,0x65,0x72,0xca,0x77,0xc5,0x8c,
			0x15,0x0d,0xda,0x09,0xa8,0x7b,0x5f,0x4d,0xa3,0x6b,0x26,0x6d,0x1f,0xa7,0xa5,0x9c,
			0xcd,0x2b,0xb2,0xe7,0xd9,0x7f,0x8b,0x23,0x15,0x43,0x19,0x23,0x53,0x0b,0x76,0x2e,
			0x12,0x6e,0xac,0xaf,0x5e,0x5a,0xc0,0x2f,0xf1,0xaa,0xef,0x81,0x9e,0xfb,0x37,0x3c,
			0xf0,0xbb,0x19,0x6f,0x0e,0x82,0x9e,0x8f,0xe1,0xa6,0x98,0xb4,0x79,0x0a,0x2a,0x05),
		.s = chunk_from_chars(
			0xbf,0x9e,0x8b,0x4f,0x2a,0xe5,0x13,0xf7,0x3d,0x78,0x89,0x58,0x00,0x37,0x33,0xdb,
			0xe2,0x09,0x57,0xb1,0x47,0xb1,0x7c,0x3f,0x4f,0xd6,0xd0,0x24,0xe8,0xe8,0x3f,0x07,
			0xb6,0x5d,0x9f,0x3d,0xbc,0x3b,0x1f,0xe8,0x4d,0xa0,0x21,0xce,0xab,0xfc,0xcd,0x8c,
			0x57,0xa0,0x14,0xfb,0xe5,0xa2,0xbc,0xe3,0xe4,0x05,0x1b,0x7d,0x03,0xe0,0x9f,0xc0,
			0x35,0x0b,0x6a,0x21,0xfa,0xd2,0x14,0xae,0x7a,0x07,0x32,0x77,0xc7,0x7a,0x40,0xdc,
			0x44,0xa5,0xae,0xea,0x51,0x94,0xa7,0x56,0xb6,0x9c,0x93,0x97,0x7b,0x69,0xee,0x92,
			0x94,0x36,0x0e,0xaa,0x73,0xa5,0x74,0x54,0x8f,0xa6,0xa9,0x74,0xa7,0xcd,0x5a,0x6a,
			0xdc,0xf0,0x9e,0x80,0x63,0x11,0x56,0xaf,0x85,0xa8,0xe5,0xc5,0x31,0x7e,0x18,0x9e,
			0xea,0xd4,0x7e,0x2e,0xad,0x65,0xc3,0x81,0x39,0x6b,0x5c,0xac,0xde,0x26,0x0e,0x93,
			0x72,0x84,0xa8,0xe9,0x0e,0xff,0x2c,0xbc,0xb9,0xde,0xe2,0x29,0x25,0xf2,0xf7,0x25,
			0x6f,0x74,0xc6,0x7c,0xf3,0xff,0xc7,0xb8,0xce,0x65,0x7e,0x8d,0x13,0x5f,0x0f,0x37,
			0x6d,0x9d,0x93,0x6a,0x79,0x79,0x2c,0x98,0x16,0x14,0xd9,0x8e,0x3f,0x7d,0x66,0x2a,
			0x4f,0xd4,0x6d,0xcd,0xa9,0x69,0x16,0xb3,0x2f,0x36,0x6e,0xd2,0x7d,0xab,0x18,0x8f,
			0x18,0x4b,0x98,0x4d,0xf0,0xb5,0x59,0x71,0x0d,0x8f,0xf2,0x04,0x0b,0xe4,0x62,0xf9,
			0x19,0x43,0x50,0x1b,0xda,0x48,0x40,0xfd,0xd5,0xc8,0xec,0x15,0xd1,0x89,0x06,0x4d,
			0xef,0x75,0x6e,0x54,0x5d,0xb3,0x19,0xe0,0x07,0xc4,0x33,0xf0,0x46,0x8a,0x67,0x23,
			0x35,0x7b,0xa4,0x7d,0x15,0x6a,0xb7,0x65,0x2b,0x06,0xae,0x2b,0x18,0x87,0x4f,0x07,
			0x71,0xc6,0x26,0x46,0x6d,0xbd,0x64,0x23,0xe6,0xcb,0xc5,0x18,0xb5,0xe4,0xae,0x7b,
			0x8f,0x15,0xe0,0xf2,0xd0,0x47,0x1a,0x95,0x16,0xdf,0xa9,0x59,0x16,0x97,0xf7,0x42,
			0x86,0x23,0x24,0xd8,0xd1,0x03,0xfb,0x63,0x1d,0x6c,0x20,0x73,0xd4,0x06,0xb6,0x5c,
			0xde,0xe7,0xbd,0xa5,0x43,0xe2,0xe9,0xeb,0xff,0x99,0x06,0x98,0x5d,0x1c,0xb3,0x65,
			0x17,0x2e,0xa6,0x23,0xed,0x7a,0xa4,0xc7,0xa3,0x22,0xf0,0x98,0x46,0x80,0xe3,0x4e,
			0x99,0xbc,0x62,0x31,0xb0,0x2e,0x3d,0x14,0x58,0x16,0x08,0xbc,0x55,0xbc,0xa7,0xfb,
			0xe2,0x2d,0x7f,0x03,0xe9,0x04,0xda,0x45,0x52,0xe0,0x09,0xe5,0x60,0x7f,0x04,0x18),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x59,0x16,0x52,0xb6,0xeb,0x1b,0x52,0xc9,0xbe,0xbd,0x58,0x32,0x56,0xc2,0x22,0x86,
			0x80,0x11,0x0b,0x87,0x89,0x17,0xde,0xa5,0xad,0x69,0xe8,0xc5,0xd2,0xab,0x51,0x42,
			0x77,0xb0,0xac,0x31,0xe7,0xe2,0xcc,0xea,0xb2,0xe5,0xd9,0xc4,0x5d,0x77,0xa4,0x1f,
			0x59,0x9b,0x38,0xa8,0x32,0xf6,0xb2,0xd8,0x09,0x79,0x52,0xbe,0x44,0x40,0xd1,0xff,
			0x84,0xba,0xf5,0x1b,0xd7,0x0b,0x64,0xf1,0x30,0xae,0xb6,0x86,0x14,0x5f,0xcd,0x02,
			0x95,0x38,0x69,0xfb,0x84,0x1a,0xf7,0xf6,0xe3,0x4e,0xaa,0x2b,0x99,0x6c,0xcd,0x89,
			0x69,0x7c,0x58,0xfa,0x25,0x5c,0xc1,0xe8,0x1f,0x62,0x14,0x00,0xe1,0x41,0x46,0x36,
			0x1e,0x31,0xc7,0x09,0xe8,0x4a,0x56,0x08,0x22,0x31,0x19,0x95,0x39,0xf7,0xed,0xe9),
		.s = chunk_from_chars(
			0x1d,0xe7,0x9d,0x72,0x16,0xdd,0xe1,0x25,0xde,0xb7,0x7c,0x34,0xd9,0x0a,0xb3,0x21,
			0xa4,0xde,0x5f,0xb1,0x1c,0x29,0x66,0x56,0xad,0x9b,0xf9,0xa2,0x46,0x53,0x59,0x11,
			0x17,0xac,0xe4,0x15,0xe1,0x8e,0xad,0xce,0x92,0x82,0x3f,0x31,0xaf,0xe5,0x6f,0xc8,
			0xe2,0x94,0x94,0xe3,0x7c,0xf2,0xba,0x85,0xab,0xc3,0xba,0xc6,0x6e,0x01,0x95,0x84,
			0x79,0x9a,0xee,0x23,0x4a,0xd5,0x55,0x9e,0x21,0xc7,0xfd,0x4f,0xfd,0x24,0xd8,0x26,
			0x49,0xf6,0x79,0xb4,0xc0,0x5d,0x8c,0x15,0xd3,0xd4,0x57,0x4a,0x2e,0x76,0xb1,0xf3,
			0xee,0x9f,0x8d,0xec,0x0a,0xf6,0x0b,0x0c,0xed,0x1b,0xe8,0xa1,0x9c,0x2f,0xa7,0x1b,
			0xcb,0xc1,0xfb,0x19,0x08,0x99,0xec,0x85,0x56,0x95,0x8e,0x07,0x82,0xac,0xe7,0x19,
			0x6b,0x36,0x65,0x86,0x56,0xcf,0x36,0x4d,0x37,0x73,0xde,0x86,0x26,0x0f,0xd8,0x98,
			0x76,0x04,0xef,0x35,0xea,0xe8,0xf3,0x8e,0xc2,0xcb,0x0d,0xa8,0x64,0xcc,0xa7,0x19,
			0x21,0x9c,0x2a,0xd7,0x1c,0x08,0x50,0x6c,0x41,0x2e,0xc7,0x79,0x95,0xf3,0x74,0x39,
			0xc8,0x56,0x97,0x7b,0x71,0xdf,0xb9,0x64,0x79,0x90,0xef,0x70,0xfa,0xf4,0x32,0x73,
			0xae,0x60,0x83,0x9c,0xd0,0x67,0x9e,0xc9,0xaa,0x42,0xbf,0x91,0x4e,0x42,0x1b,0x79,
			0x7c,0xba,0x21,0x8a,0x40,0x0f,0xf9,0xdb,0xaa,0x20,0x6c,0xb9,0xc2,0xb0,0x59,0x6c,
			0x70,0x9a,0x32,0x2b,0x73,0xcb,0x82,0x72,0x1d,0x79,0xf9,0xdb,0x24,0x21,0x1b,0xf0,
			0x75,0xa1,0xce,0xf7,0x4e,0x8f,0x6d,0x2b,0xa0,0x7f,0xe0,0xdc,0x8a,0x60,0xf4,0x8a,
			0xf5,0x11,0xad,0x46,0x9d,0xcd,0x06,0xe0,0x7a,0x4c,0xe6,0x80,0x72,0x13,0x9c,0x46,
			0xd8,0xbe,0x5e,0x72,0x12,0x53,0xc3,0xb1,0x8b,0x3c,0x94,0x48,0x5c,0xe5,0x5c,0x0e,
			0x7c,0x1c,0xbc,0x39,0xb7,0x7b,0xc6,0xbb,0x7e,0x5e,0x9f,0x42,0xb1,0x53,0x9e,0x44,
			0x2d,0xa8,0x57,0x65,0x8c,0x9e,0x77,0x1c,0xcb,0x86,0xbe,0x73,0x97,0x64,0x7e,0xfb,
			0xc0,0xcc,0xb2,0xc3,0xad,0x31,0xac,0x4e,0x32,0xbf,0x24,0x8c,0xc0,0xce,0xd3,0xa4,
			0xf0,0x94,0x52,0x6b,0x25,0x63,0x1c,0xb5,0x02,0x47,0x09,0x61,0x29,0xb0,0x8a,0x9c,
			0x2c,0xdf,0xb7,0x75,0x97,0x8b,0x0f,0xee,0xe2,0x65,0xa6,0xc4,0x19,0x91,0xc1,0xdc,
			0x44,0x52,0x61,0x5b,0x78,0xc9,0x06,0xc7,0xed,0x1b,0xd2,0x07,0x96,0x9d,0x98,0xd0),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x8d,0xff,0xaa,0x91,0x51,0x27,0x1a,0xd2,0x26,0x22,0xf2,0x28,0xc8,0x92,0xe1,0xd9,
			0x74,0x8b,0x3c,0x39,0x43,0x97,0xf2,0xcb,0xb6,0xfe,0xbe,0xaa,0x92,0x44,0xa0,0x27,
			0xee,0xf2,0x8d,0xb4,0x8a,0x9a,0x66,0x01,0x62,0x15,0x27,0x64,0x83,0x0f,0x61,0x7e,
			0x1e,0xc6,0xea,0x1c,0xdb,0x0e,0xd2,0x5b,0x6f,0x99,0x9a,0x10,0x71,0x75,0xa1,0x66,
			0x69,0xd6,0xdf,0xc9,0x2b,0x16,0xd5,0x03,0x63,0xfa,0xc4,0xa5,0x70,0x37,0x1e,0xa9,
			0x76,0x34,0x3a,0x55,0xae,0x12,0x4b,0x63,0x01,0xea,0x93,0x5e,0xd6,0x55,0xd4,0x4f,
			0x28,0x32,0x08,0x99,0xdb,0xa3,0x51,0x22,0x50,0x59,0x33,0xb3,0x37,0x12,0x01,0xa2,
			0xa4,0x5f,0x95,0xae,0x65,0xab,0x44,0x2a,0x94,0x79,0x12,0x5e,0x68,0xed,0x21,0x2a),
		.s = chunk_from_chars(
			0xb3,0x29,0xae,0xf8,0x3a,0x56,0xdd,0xc5,0x7c,0xd9,0xa0,0xe1,0x5e,0xb0,0xb0,0xb7,
			0xae,0xa7,0xd7,0x8d,0x5e,0x8c,0xa3,0x98,0x2b,0xd3,0x1c,0xc8,0x25,0xa0,0xcd,0x1c,
			0x44,0x4d,0x9f,0x7b,0xea,0x9e,0x7a,0x27,0xf3,0xbb,0xb3,0x76,0x10,0x60,0xff,0x95,
			0xfe,0xe1,0xa3,0xe8,0x64,0xd2,0x10,0x8f,0xc4,0x0b,0x64,0x78,0x6a,0x96,0xa6,0xd6,
			0x2d,0x20,0x12,0x17,0xe0,0x3a,0x8b,0xa2,0xc0,0x7e,0xe9,0x4c,0x26,0x71,0x49,0xd1,
			0xe7,0x2c,0xc5,0x77,0x9b,0x73,0x7e,0x85,0x47,0xac,0xd6,0xaa,0x4b,0xba,0x3f,0xf3,
			0x8b,0xf9,0x68,0x7e,0x9e,0x82,0xf5,0x11,0xb5,0x97,0xad,0x7e,0xc1,0xd7,0x95,0xc3,
			0x6a,0x98,0xbf,0x83,0xa9,0x0f,0xc8,0x6b,0x0c,0xad,0x41,0x95,0x33,0x60,0x73,0x89,
			0x21,0x93,0x6a,0x45,0x86,0x74,0xb2,0xe9,0xa7,0x01,0x2a,0xc3,0x02,0x9f,0xdb,0x0a,
			0x9d,0x12,0x31,0x82,0x02,0xd2,0x54,0x4a,0x0d,0x97,0x6e,0xe5,0x36,0xe0,0x3b,0x7e,
			0x8d,0x89,0x4b,0x3b,0x9c,0x76,0x2d,0xab,0x01,0x10,0x84,0x9c,0xc1,0xea,0xad,0x74,
			0x7e,0x3d,0x88,0xd7,0xdc,0xf4,0x9f,0x82,0x4d,0xf0,0x27,0xe6,0x45,0xc0,0xb9,0x29,
			0x4e,0x65,0x5d,0x9f,0xc9,0xe1,0xef,0x95,0xeb,0x53,0xaa,0xff,0x57,0x75,0xc3,0x49,
			0x48,0x6d,0x4b,0x5d,0x67,0xdb,0xa2,0x9b,0x62,0x17,0xf8,0xb9,0x97,0x66,0x12,0xb5,
			0x7e,0x16,0xfc,0x1f,0x99,0x98,0x3f,0x2a,0xf0,0x45,0x79,0x93,0x86,0x06,0x87,0x9b,
			0x7c,0x72,0x53,0xe8,0x70,0x71,0x4b,0x4f,0x0f,0x24,0xe2,0x6d,0xc8,0xc7,0xa6,0xfc,
			0xef,0xfb,0x5f,0x98,0xe3,0xb2,0xfb,0x5d,0xb9,0x49,0xd2,0xf9,0x8c,0xd1,0xae,0x1a,
			0xa5,0x52,0x69,0x6b,0x48,0xc3,0x9f,0x67,0x8e,0x15,0x43,0x51,0xcc,0x75,0x6d,0x3e,
			0x9a,0x97,0xf7,0x92,0x79,0x85,0x3e,0xbd,0x0d,0xb9,0xae,0x68,0x59,0xfb,0x2d,0x57,
			0x21,0x38,0x5d,0x06,0xf5,0x56,0x5a,0x3a,0x8f,0xf0,0x99,0x2d,0x51,0x7a,0xcd,0xa1,
			0xaf,0x69,0xa9,0x28,0x54,0xa1,0xb3,0x2a,0x79,0xcb,0x9e,0x44,0x2a,0x90,0xb0,0x55,
			0xbb,0x2e,0xc3,0xaf,0x8d,0x99,0x26,0xa0,0xd8,0x57,0xe3,0xcb,0x1e,0x7e,0x4a,0x73,
			0x00,0xd1,0xac,0xcb,0x94,0x92,0xec,0x78,0x32,0xaf,0x45,0x35,0x29,0xff,0x0f,0x4a,
			0x6a,0xd3,0x25,0x97,0x57,0xf7,0x07,0xf7,0x13,0xaa,0xa5,0xdf,0x23,0x1f,0x74,0x87),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x71,0xd4,0x16,0x3e,0x70,0x8c,0x12,0x1e,0x93,0x1b,0xb9,0x69,0x2b,0x21,0x7d,0xdd,
			0xd3,0x5c,0x73,0x46,0xf6,0x1c,0xfc,0x95,0x91,0xf7,0xa4,0x31,0x3a,0xbd,0x4a,0x92,
			0x62,0xaf,0x82,0x0b,0xd7,0xeb,0x37,0xe7,0x8c,0x2b,0x95,0xb8,0x9d,0xaf,0x25,0xec,
			0x8e,0x78,0x3a,0xa1,0xd4,0xb7,0x8d,0xbb,0x96,0x85,0x24,0x33,0xb4,0xd4,0x78,0xb1,
			0x09,0xa6,0xd6,0x5e,0xed,0x7d,0x06,0xf3,0xfe,0x12,0x2b,0x17,0x21,0x49,0xea,0xe7,
			0xc3,0x65,0xce,0xd6,0x65,0x78,0xeb,0xb7,0x57,0x1e,0xc2,0x18,0xc3,0x6b,0x65,0xd2,
			0xee,0x22,0xdc,0xde,0xbb,0x28,0xc6,0x6a,0x71,0x38,0x43,0x2c,0xbd,0xd7,0x12,0xf7,
			0xfb,0x8b,0xf7,0x8c,0xb1,0x48,0x60,0xb2,0x5c,0x2b,0x47,0x89,0x70,0x6b,0x5a,0x1b),
		.s = chunk_from_chars(
			0x25,0x22,0xee,0x3b,0xda,0x30,0xc0,0x43,0x4e,0x54,0xb1,0x99,0xda,0x8c,0x97,0x33,
			0x96,0x4f,0xd4,0x02,0xb7,0x07,0xf5,0xb3,0x30,0xf4,0xf7,0x54,0xa0,0x50,0x2c,0x7a,
			0x71,0x3c,0x78,0x14,0xf0,0xe8,0x51,0xa4,0xa4,0xdb,0x72,0x69,0x0d,0xb9,0x6e,0xa8,
			0xb8,0x81,0x3b,0xd8,0x62,0x9a,0x94,0x8b,0xb3,0x0c,0x1b,0x82,0x72,0xa8,0x16,0xb3,
			0x0a,0x75,0x5f,0xc6,0xfb,0x17,0x54,0x16,0x7c,0x3e,0xb1,0xf1,0x94,0x39,0x59,0x07,
			0xa5,0x6c,0xf5,0xa7,0x3b,0x41,0x54,0x38,0x3a,0x05,0xb7,0x8b,0x73,0x1f,0xed,0xd9,
			0x07,0x7f,0x3c,0x22,0x67,0xa5,0xcf,0x92,0x66,0x97,0x87,0x1f,0xe0,0xa4,0xbe,0xd9,
			0xc2,0x19,0x55,0x2d,0xd1,0xc8,0x7a,0xff,0x50,0x61,0x30,0x94,0xbc,0xaa,0x2d,0xec,
			0x42,0xa3,0x53,0x80,0xa6,0xba,0xc6,0x73,0xda,0x25,0x94,0xf8,0x24,0xa8,0xf3,0x2f,
			0x21,0xd7,0x59,0x3a,0x3e,0x49,0xc7,0x8e,0xe2,0x80,0x19,0x3a,0x47,0x86,0x21,0xd3,
			0xb0,0x95,0xc1,0x6d,0xce,0x72,0x93,0x53,0x14,0xd4,0xa2,0x32,0x3e,0xeb,0xe7,0x85,
			0x5c,0xa4,0x73,0x8a,0x19,0xb5,0xa3,0x1a,0x5f,0x95,0xab,0x91,0xfb,0xe1,0x28,0x9c,
			0x02,0xfe,0xa7,0xa6,0x5b,0x91,0x32,0x7b,0x7b,0x97,0x90,0x55,0x62,0x89,0xe1,0xb9,
			0x88,0xe4,0x5d,0x50,0xeb,0x8c,0xea,0x15,0x81,0xde,0x5d,0x5d,0xfd,0x21,0x00,0x1c,
			0x73,0xb4,0x39,0x21,0xd8,0xb2,0x1b,0x96,0x44,0xb0,0xf2,0xb9,0x6e,0xe6,0xb0,0x9d,
			0x73,0x70,0x9c,0x33,0x33,0x81,0x43,0xd6,0xa2,0xfe,0xc5,0x59,0xa4,0x36,0xc5,0xec,
			0x86,0x5d,0x3a,0xcc,0xa5,0xfe,0xe6,0x54,0xf1,0x32,0x5a,0xe5,0x72,0x55,0xdf,0xd4,
			0x21,0x88,0xc8,0x4d,0xcb,0x1f,0x7c,0x1e,0x86,0x02,0x8a,0x74,0xe3,0x1d,0x73,0x60,
			0x78,0x74,0x1e,0xe9,0x7c,0x39,0xa5,0x6e,0x4d,0xe0,0x0f,0xc1,0x2b,0x80,0x51,0x83,
			0x5b,0xbd,0x0d,0x8f,0xca,0xe7,0x37,0x32,0x20,0x99,0xad,0xc1,0x01,0x71,0x07,0x02,
			0x2d,0xd1,0x5c,0x11,0x4d,0xa5,0x7e,0x78,0xb9,0x56,0x81,0xba,0x99,0x45,0x61,0x5b,
			0x59,0xda,0x90,0xf5,0xa2,0xa9,0x9a,0x25,0x2e,0xb4,0x2b,0x20,0x06,0xee,0xdd,0x6e,
			0x78,0x47,0x6c,0x29,0x05,0x47,0x3e,0xe6,0xb4,0xf2,0x3c,0x1c,0x5c,0xf0,0xb8,0x04,
			0x51,0xc5,0x42,0x6e,0xa0,0x09,0x14,0x1c,0xb3,0xfc,0xb0,0xdf,0x2d,0xed,0x92,0xbe),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xd0,0x0e,0x15,0x29,0x22,0x8c,0x79,0xa2,0x0a,0x1c,0x36,0x68,0xff,0xa4,0xa5,0x41,
			0x40,0xbb,0x17,0x0b,0xc5,0xc6,0x69,0xfd,0x75,0x60,0xd9,0x30,0x99,0x00,0x17,0x5e,
			0x91,0xd5,0xa0,0xe9,0xc5,0xf5,0x47,0x1f,0xdf,0xb7,0x14,0xbc,0x38,0x5d,0x52,0xb0,
			0x8f,0xf7,0xe4,0x23,0x01,0x84,0xd8,0xb7,0x35,0x59,0x3f,0x0d,0xd8,0xc7,0x3b,0x8a,
			0x49,0xf8,0x59,0x5b,0x95,0x1a,0x21,0xb6,0xa5,0xbf,0xec,0x63,0xb6,0x84,0xf6,0x7c,
			0x0a,0xf1,0xb4,0x71,0xdd,0xa1,0x68,0x4e,0x9b,0xa3,0xf2,0x41,0x50,0x1f,0xe9,0x57,
			0x60,0x3d,0xea,0x86,0x78,0x42,0x30,0xf0,0xc4,0xfd,0x65,0x66,0x63,0x61,0xb8,0x2b,
			0x18,0x73,0x30,0xfb,0x42,0x67,0x40,0x4c,0x0e,0x05,0x9b,0xd4,0xeb,0x52,0x49,0x4b),
		.s = chunk_from_chars(
			0x18,0x35,0xdd,0x97,0xe5,0x09,0x3a,0x33,0xce,0x1e,0x62,0xd6,0x83,0x86,0x3f,0x6b,
			0x35,0x07,0xf3,0x58,0xa6,0x2f,0xc8,0x79,0xb5,0x24,0x35,0x0f,0xbc,0x73,0x30,0x68,
			0x1c,0xb0,0xc6,0x82,0xee,0xf4,0x33,0x04,0x19,0xca,0xf8,0x54,0x3b,0xd9,0x26,0x9b,
			0x6d,0x91,0xd8,0xe1,0x07,0xec,0x38,0xb6,0xe9,0xc6,0xea,0xab,0xf9,0x06,0x45,0x72,
			0x05,0xd5,0x2a,0x90,0x0e,0x05,0x57,0x9a,0xa1,0x1f,0xc5,0x81,0x37,0x52,0x64,0xe6,
			0x9a,0x92,0x57,0x98,0xe5,0xa3,0x48,0xe5,0xa1,0x6f,0x15,0x67,0xd5,0xd0,0xe4,0x08,
			0x53,0x38,0x0b,0x34,0xde,0xac,0x93,0xad,0x73,0x77,0xaa,0xe8,0xa2,0x7b,0x09,0x0d,
			0x0d,0x3a,0x92,0xbf,0x7a,0x82,0x4d,0x92,0x6e,0x2e,0x35,0xa0,0xc3,0xbd,0x0e,0x99,
			0x0b,0x59,0x11,0x20,0xd7,0x4d,0xd9,0xb0,0x52,0xa7,0x35,0x68,0xe3,0xc3,0xf2,0x9c,
			0x5a,0x77,0xfb,0x1c,0x92,0x1b,0xce,0x9c,0x1e,0x7f,0x76,0x4a,0xa6,0x7b,0xac,0x11,
			0x9f,0x58,0x39,0xa5,0x30,0x38,0x60,0xed,0xeb,0x63,0x48,0x14,0xc2,0x38,0x6c,0x83,
			0x1f,0xee,0x62,0x00,0xcf,0x55,0xb6,0xbf,0xea,0x05,0x8b,0x79,0x5a,0x0f,0xcf,0x26,
			0xeb,0x72,0x16,0xae,0x1b,0x75,0x87,0xc8,0x2e,0x56,0x85,0xe5,0x84,0x17,0x0c,0xbd,
			0xdc,0x89,0xa7,0x7e,0x09,0x89,0xd4,0xce,0x5c,0x3c,0x7f,0xdb,0x66,0x4a,0xae,0xaa,
			0xdb,0xce,0x1f,0x23,0x1e,0x64,0x79,0x8f,0x6f,0x9a,0x85,0x45,0x6b,0x5a,0x93,0xa5,
			0x02,0x12,0x6a,0x80,0xe2,0xd2,0x1f,0x46,0x92,0x1c,0xc3,0x60,0x1f,0x5e,0xcd,0xbd,
			0x56,0x99,0x8a,0x63,0xb8,0x65,0xfc,0xe7,0xeb,0x29,0x9f,0x76,0xaf,0x40,0xe9,0x12,
			0x81,0xbf,0xc0,0x19,0xf4,0x0e,0x0d,0x46,0x81,0x1e,0x38,0x36,0x91,0xe4,0x02,0x4c,
			0x94,0x56,0x6f,0x18,0x02,0x4f,0xf2,0xb2,0x2a,0xa7,0xe1,0x27,0x02,0x33,0xff,0x16,
			0xe9,0x2f,0x89,0xc6,0x85,0x09,0xea,0x0b,0xe2,0xd3,0x45,0x11,0x58,0x1d,0x47,0x22,
			0x07,0xd1,0xb6,0x5f,0x7e,0xde,0x45,0x13,0x3d,0xe8,0x7a,0x5f,0xfb,0x92,0x62,0xc1,
			0xff,0x84,0x08,0x8f,0xf0,0x4c,0x01,0x83,0xf4,0x84,0x67,0x99,0x6a,0x94,0xd8,0x2b,
			0xa7,0x51,0x0c,0xb0,0xb3,0x6c,0xf2,0x54,0x82,0x09,0xa5,0x06,0x03,0x37,0x5c,0xb8,
			0x2e,0x67,0x8f,0x51,0x49,0x33,0x45,0xca,0x33,0xf9,0x34,0x5f,0xfd,0xf5,0x4b,0xe9),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xa3,0x59,0x26,0x68,0x55,0x61,0xf0,0x9f,0x30,0x92,0x5e,0x94,0xd7,0x4e,0x56,0x61,
			0x89,0x2a,0x2d,0xdd,0x52,0x4f,0x75,0x1f,0x83,0x21,0x16,0x3d,0x61,0x1e,0xa1,0x59,
			0x1a,0x08,0xe0,0xdf,0xfd,0x46,0xb2,0x08,0xe9,0x88,0x15,0xa3,0x06,0xaa,0x85,0x14,
			0xb4,0xdb,0x85,0x9d,0xc1,0xfe,0x7b,0xdc,0xdf,0x50,0xc0,0x95,0x55,0x4b,0xf8,0xb2,
			0xf4,0xcb,0x9f,0x88,0x4d,0x70,0xe5,0x5c,0x21,0x43,0xbc,0x26,0x19,0x9c,0x2f,0x94,
			0xb7,0x43,0xf5,0x52,0x8d,0xd5,0x46,0x89,0xad,0x69,0xed,0xa6,0x60,0x74,0x9f,0x5c,
			0x1b,0xea,0x8b,0xec,0xae,0xa6,0x32,0xa4,0xbf,0x0c,0x79,0xa5,0x77,0xed,0xfc,0xea,
			0x7b,0xaa,0xa6,0x86,0x1e,0x9d,0x7f,0x2d,0xd5,0xb4,0xc4,0xf6,0xeb,0x5f,0x3d,0x5f),
		.s = chunk_from_chars(
			0xb1,0xa9,0xc4,0x5a,0x26,0x4d,0x2c,0x9a,0xf4,0x41,0xa7,0xb2,0xd3,0x30,0xdd,0x78,
			0x80,0x89,0xcc,0xef,0x20,0x5d,0x5d,0x66,0x6b,0xfe,0x86,0x43,0x67,0xbe,0x97,0x38,
			0x12,0x4e,0x9d,0x74,0x64,0x8a,0xd9,0x91,0x60,0xbd,0x3a,0xf8,0x1a,0x81,0x85,0x8b,
			0xab,0xe6,0x67,0xa5,0xd9,0x5c,0x98,0x0f,0xe2,0xf6,0xac,0x34,0x86,0x1e,0xb2,0xec,
			0x9b,0x4b,0x4e,0x8b,0x64,0x2e,0xf3,0x82,0x0f,0x56,0xca,0x38,0x8a,0x55,0x65,0x30,
			0xd4,0x27,0x54,0xc4,0x72,0x12,0xe9,0xb2,0xf2,0x52,0x38,0xa1,0xef,0x5a,0xfe,0x29,
			0xbe,0x63,0x40,0x8c,0xf3,0x8c,0xaa,0x2d,0x23,0xa7,0x88,0x24,0xae,0x0b,0x92,0x59,
			0x75,0xd3,0xe9,0x83,0x55,0x8d,0xf6,0xd2,0xe9,0xb1,0xd3,0x4a,0x18,0xb1,0xd9,0x73,
			0xff,0xac,0xcc,0x74,0x5e,0x52,0x7c,0xe7,0x6c,0x66,0x3e,0x90,0x37,0x19,0x35,0x5e,
			0x45,0xcd,0x6d,0x11,0x8e,0xd0,0xb8,0x5b,0x70,0xcb,0xb8,0xe4,0x96,0x41,0x13,0x53,
			0xf8,0x4f,0x88,0x66,0xa0,0x1f,0xad,0xc8,0x19,0xca,0x0f,0xf9,0x5b,0xbe,0x2c,0xc6,
			0x8c,0x8c,0xf7,0x8d,0xa5,0x58,0x1b,0xec,0xc9,0x62,0x47,0xb9,0x11,0xd1,0x85,0xed,
			0x1f,0xae,0x36,0xc4,0xca,0xd2,0x62,0x08,0xeb,0x80,0x88,0x3f,0x42,0xa0,0x81,0x23,
			0xda,0xc6,0x8d,0x88,0xf2,0xf9,0x89,0x3c,0xde,0x02,0xef,0x5a,0x57,0x66,0x1d,0xb2,
			0xb3,0xe1,0xe9,0x26,0x9c,0xbb,0x0e,0x15,0xc4,0x07,0xbc,0xf5,0x5d,0x92,0xe6,0x79,
			0x38,0x3c,0x90,0x80,0x2c,0xd0,0xbf,0xfd,0x46,0x96,0x46,0xdc,0xb6,0x0c,0xa0,0x1a,
			0x1d,0xea,0xd4,0x32,0x28,0x93,0x40,0x18,0x39,0x1d,0xd8,0x1f,0x8b,0x7e,0x79,0x7e,
			0x52,0x7f,0xbe,0x18,0x15,0xb9,0x1b,0xf3,0xcd,0x6a,0x1f,0x2f,0xfb,0xf5,0xdd,0x16,
			0x6a,0xcd,0x55,0x26,0x76,0x1c,0xa8,0xba,0xb5,0xd4,0x63,0xfb,0x9f,0xb8,0x20,0x65,
			0x9f,0x5c,0xd5,0x0f,0x81,0x50,0xf1,0x2f,0x7e,0x8d,0x52,0xe7,0x77,0x73,0xc1,0xe6,
			0x48,0x0c,0x2c,0xc1,0x84,0xd4,0x11,0xd6,0x41,0xf7,0x1a,0x9d,0xed,0xc2,0xc5,0xfc,
			0x2e,0xc3,0x7a,0x27,0x70,0xa9,0x38,0x3b,0xfb,0xf6,0xa4,0x89,0xcf,0x32,0xb5,0x6a,
			0x12,0xcf,0x99,0x37,0x8e,0x39,0xb5,0x0b,0xda,0xdb,0x9f,0x05,0x91,0xb2,0x06,0x5f,
			0x9d,0x44,0xe5,0x11,0xc9,0xdf,0xb6,0x15,0x8f,0xdd,0xdd,0xd1,0xbc,0x2c,0xec,0xe6),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x12,0x71,0xa0,0xdd,0xb9,0x9a,0x0e,0x1e,0x9a,0x50,0x1c,0xa3,0x3c,0x13,0x1b,0x0a,
			0x1c,0x78,0x20,0xa3,0x97,0x79,0x08,0x69,0x09,0x0f,0xba,0x37,0x37,0x03,0xac,0x38,
			0xea,0x00,0xa9,0xa0,0xdd,0xee,0xd1,0x99,0xd9,0x7b,0xe1,0x80,0x1f,0xfa,0xb4,0x52,
			0x06,0x71,0x0a,0x61,0xe5,0xed,0x89,0x4c,0x33,0x19,0x01,0x2d,0xed,0x0f,0xf4,0x14,
			0x38,0x6e,0x56,0xb5,0x48,0xad,0x91,0x5d,0x80,0xaf,0xcc,0x2b,0xdb,0x97,0x6d,0x7c,
			0x8a,0xdd,0xdc,0xa7,0xdf,0xa2,0x8a,0xeb,0x69,0x40,0x33,0xa5,0x61,0x26,0x60,0xc6,
			0x44,0xe3,0x2f,0x85,0xc2,0x80,0x56,0x51,0xd7,0x13,0x66,0x0a,0x38,0x91,0x4d,0x70,
			0xf0,0xe4,0x1f,0xdc,0x4b,0x3d,0x16,0x2e,0xf3,0xac,0xd7,0x06,0x59,0xee,0xf6,0x37),
		.s = chunk_from_chars(
			0xbf,0xfd,0x01,0x0b,0x2e,0xc4,0xe4,0xa3,0x27,0x77,0xb7,0x76,0x19,0xb8,0x76,0x22,
			0xf8,0x92,0x1d,0xab,0x56,0xe1,0x02,0xc8,0xd8,0x24,0xfe,0x52,0xb5,0xdf,0x7a,0x20,
			0x3f,0xe7,0x17,0x99,0xee,0xaf,0xdc,0xc0,0xc8,0x87,0x2d,0xba,0x6a,0x37,0x44,0x07,
			0xb5,0x63,0x9a,0xeb,0x5a,0x30,0xa9,0x04,0x71,0x2f,0x15,0x09,0x7d,0xba,0x0f,0x2d,
			0x62,0xe8,0x45,0x41,0x23,0x95,0xcf,0x09,0x54,0x0a,0xbd,0x6e,0x10,0xc1,0xa2,0xe2,
			0x3d,0xbf,0x2f,0xe1,0xdf,0xd2,0xb0,0x2a,0xf4,0xee,0xa4,0x75,0x15,0x95,0x7f,0xa3,
			0x73,0x8b,0x06,0x41,0x1a,0x55,0x1f,0x8f,0x8d,0xc4,0xb8,0x5e,0xa7,0xf5,0xa3,0xa1,
			0xe2,0x6c,0xcc,0x44,0x98,0xbd,0x64,0xaf,0x80,0x38,0xc1,0xda,0x5c,0xbd,0x8e,0x80,
			0xb3,0xcb,0xac,0xde,0xf1,0xa4,0x1e,0xc5,0xaf,0x20,0x55,0x66,0xc8,0xdd,0x80,0xb2,
			0xea,0xda,0xf9,0x7d,0xd0,0xaa,0x98,0x33,0xba,0x3f,0xd0,0xe4,0xb6,0x73,0xe2,0xf8,
			0x96,0x0b,0x04,0xed,0xa7,0x61,0x61,0x64,0x39,0x14,0x24,0x2b,0x96,0x1e,0x74,0xde,
			0xae,0x49,0x7c,0xaf,0x00,0x5b,0x00,0x51,0x5d,0x78,0x49,0x2e,0xc2,0xc2,0xde,0xb6,
			0x0a,0x57,0xb9,0xdc,0xe3,0x6e,0x68,0xdd,0x82,0x00,0x7d,0x94,0x2a,0xe7,0xc0,0x23,
			0xe1,0x21,0x0f,0x0b,0xe8,0xa3,0xeb,0x3f,0x00,0x48,0x24,0x07,0x4b,0x8f,0x72,0x5e,
			0xaf,0x8a,0xc7,0x73,0xe6,0x0f,0xbb,0xb7,0xcb,0xa9,0x63,0x0e,0x88,0xb6,0x9c,0x8b,
			0xcb,0x2d,0x74,0xdb,0xdb,0x29,0xbf,0xff,0x8b,0x22,0x54,0x5b,0x80,0xbb,0x63,0x4e,
			0x4c,0x05,0xf7,0x3e,0x00,0x2a,0x92,0x8e,0xfd,0x5a,0x6a,0xa4,0x56,0x21,0xce,0x1b,
			0x03,0x2a,0x22,0x44,0xde,0x48,0xf4,0xdf,0x43,0x58,0x15,0x66,0x78,0xcb,0xe0,0x39,
			0xc9,0xeb,0xe4,0xce,0xe9,0x45,0xa2,0x5b,0x90,0x38,0x46,0x9f,0xe0,0x0c,0x30,0x92,
			0x93,0x6a,0x8c,0xff,0x93,0x69,0x04,0x5f,0x90,0x67,0x33,0xa9,0xd2,0xab,0x36,0x60,
			0x18,0x20,0x69,0xb1,0x57,0xca,0x8f,0x9b,0x99,0xa7,0x1f,0xc1,0x53,0xc6,0x83,0x01,
			0xe9,0x7a,0x38,0xfc,0x3a,0x87,0xae,0x2b,0x6f,0x03,0x75,0x4e,0x6d,0xa8,0x2d,0x0b,
			0x07,0x26,0xe0,0x70,0x39,0x79,0xc9,0x32,0x02,0x89,0xfe,0xef,0xbc,0xdd,0xcd,0x9d,
			0x70,0x6b,0x71,0xb5,0x1e,0x9a,0x1b,0x9d,0xc1,0x41,0x2e,0x6e,0xd4,0xb5,0x66,0x76),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0xf3,0x0c,0x78,0x3b,0x4e,0xae,0xb4,0x65,0x76,0x7f,0xa1,0xb9,0x6d,0x0a,0xf5,0x24,
			0x35,0xd8,0x5f,0xab,0x91,0x2b,0x6a,0xba,0x10,0xef,0xa5,0xb9,0x46,0xed,0x01,0xe1,
			0x5d,0x42,0x7a,0x4e,0xcd,0x0f,0xf9,0x55,0x67,0x73,0x79,0x17,0x98,0xb6,0x69,0x56,
			0xec,0xc7,0x52,0x88,0xd1,0xe9,0xba,0x2a,0x9e,0xa9,0x48,0x57,0xd3,0x13,0x29,0x99,
			0xa2,0x25,0xb1,0xff,0xaf,0x84,0x46,0x70,0x15,0x6e,0x7a,0x3e,0xa9,0xf0,0x77,0xfe,
			0x82,0x59,0xa0,0x98,0xb9,0xee,0x75,0x9a,0x6d,0xdf,0xb7,0xd2,0x0a,0x7a,0xcd,0x1b,
			0xcb,0x9f,0x67,0x77,0x7e,0x74,0x61,0x5e,0x88,0x59,0xea,0x56,0x28,0x1f,0xe5,0xc4,
			0x00,0x74,0x8f,0x02,0xd1,0xa2,0x63,0xb1,0x86,0x7a,0x3b,0x51,0x74,0x8a,0xb7,0x0f),
		.s = chunk_from_chars(
			0x34,0x5e,0x2f,0x60,0xf7,0xc8,0x2c,0x89,0xef,0x7d,0xfd,0x7d,0xff,0x2b,0xc2,0x34,
			0x8b,0xab,0x02,0x04,0x79,0x33,0x08,0x99,0xd4,0x41,0x02,0x13,0xb3,0x5e,0x98,0xd9,
			0xba,0xc9,0x2f,0xd8,0xae,0x80,0x6b,0x5b,0xce,0x8a,0x6c,0x4b,0xd8,0x27,0x5b,0x0f,
			0xac,0xb4,0xdd,0x13,0xf9,0xd6,0x8b,0xa6,0x71,0x41,0xfa,0x50,0x85,0x26,0x4d,0xa6,
			0xdd,0x68,0x5a,0x6d,0x21,0x21,0x70,0xa2,0xc9,0xcb,0xf2,0xcf,0x59,0x30,0x18,0x0e,
			0xff,0xc2,0x50,0x86,0x8c,0x98,0x4b,0xf5,0x0f,0xf6,0x9d,0x60,0x69,0xea,0x28,0xf5,
			0xbc,0x1b,0x63,0x70,0x5d,0x07,0x32,0x41,0x6f,0xd8,0x29,0xa5,0xf5,0xd6,0x21,0x74,
			0x62,0xc2,0x2a,0x33,0xfd,0x46,0x52,0xf7,0xc1,0xd1,0x98,0x79,0x46,0x46,0xc0,0x84,
			0x06,0x02,0x4e,0x81,0x63,0xa7,0xeb,0xe3,0x9c,0xfb,0x51,0x4c,0x54,0x43,0x89,0x7b,
			0x58,0x94,0xdd,0x19,0xa2,0x13,0xe0,0x37,0xf2,0x7e,0x0f,0xfb,0xd6,0xc5,0x44,0x7a,
			0x80,0x5a,0x54,0xdf,0xdf,0x4f,0x65,0x81,0x9d,0x4e,0x0f,0xbe,0xe2,0x5e,0x3d,0xac,
			0x47,0xfb,0x6b,0x63,0x6e,0x8d,0xe6,0x19,0x0a,0xdc,0xcb,0xce,0xe9,0x37,0xd0,0x97,
			0x7b,0x35,0xb9,0x73,0x60,0x6b,0x0c,0xa3,0x48,0x75,0x8b,0x50,0xcd,0xbb,0xa0,0x28,
			0xb7,0x3d,0x0e,0xf0,0x1c,0x56,0x01,0x4c,0x03,0x1c,0x59,0x8f,0xe8,0xdb,0x87,0xd2,
			0xca,0x46,0x44,0x77,0x0a,0xaa,0x04,0x51,0xc3,0x76,0xde,0xd8,0x2f,0xf5,0xc6,0xb8,
			0xe7,0xd2,0xed,0x9d,0x1c,0x8a,0x17,0xc3,0x12,0x2c,0x12,0x82,0x73,0xc6,0x0f,0xd1,
			0xb0,0x08,0x8d,0xfb,0xc9,0xc9,0x27,0xf1,0x62,0xe4,0x38,0x79,0x40,0x59,0x64,0xcb,
			0x11,0xef,0x78,0x99,0x12,0x3f,0xeb,0x8f,0x88,0xdd,0x27,0x34,0xdf,0x98,0xaa,0x69,
			0x6d,0x93,0x6a,0x8d,0xf0,0x70,0x00,0xe8,0x4a,0xf9,0x01,0x01,0xf7,0x00,0x6a,0x9b,
			0xd2,0x54,0x9f,0xdd,0x0a,0xd3,0xf9,0xde,0x09,0x30,0x12,0xd3,0x2d,0x2a,0xfa,0xa8,
			0x28,0x01,0x7e,0xe9,0xc6,0x07,0xcb,0xf5,0xb5,0x4f,0x22,0x36,0x66,0xd4,0xb5,0xf3,
			0xe2,0x6e,0x0d,0xfe,0xc0,0x03,0x96,0x1b,0x83,0xd8,0x3d,0xe3,0x9f,0xf6,0xa0,0xe8,
			0x1e,0x18,0x83,0xc1,0xdb,0x4a,0xaa,0xf0,0x82,0xfe,0xc5,0xaa,0x30,0xa7,0xe5,0x78,
			0x55,0x3d,0x89,0x77,0x4c,0x67,0x90,0x77,0x90,0xc9,0x6d,0xc4,0xf5,0xbe,0x4c,0x8c),
	},
	{
		.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_512,
		.m = chunk_from_chars(
			0x13,0x2c,0xf5,0x0c,0x66,0xac,0x4c,0xc5,0x43,0x39,0x75,0x1a,0x0e,0xbb,0x86,0x5e,
			0x1d,0x3d,0x32,0x05,0x62,0xfc,0x90,0x5c,0x4a,0xbd,0x1e,0x78,0xe4,0x64,0x06,0x6c,
			0x46,0xc3,0xa0,0xc0,0x2d,0xb0,0x37,0x1e,0xe3,0x5a,0x10,0x4d,0x66,0xdd,0xa8,0x64,
			0xc6,0x13,0x3e,0x37,0xcf,0xad,0x91,0x16,0xe8,0x83,0xeb,0xb7,0x3b,0x29,0x5e,0x70,
			0x16,0xc3,0x4e,0xa9,0x91,0x1a,0x30,0x92,0x72,0xef,0x90,0x11,0x4d,0x8f,0x59,0xff,
			0xf0,0xa7,0x51,0x93,0xfe,0x5a,0xe3,0x1e,0xd9,0x91,0x21,0xf9,0xc5,0x92,0x09,0xbc,
			0x4b,0xd5,0x07,0xb1,0xdc,0x12,0xbc,0x89,0xb7,0x9f,0xfe,0x4d,0x0d,0xf9,0x20,0x97,
			0x62,0xa1,0x73,0x01,0x36,0x29,0x0c,0xde,0xe5,0x8e,0xc8,0x28,0xcc,0xc8,0x8e,0xba),
		.s = chunk_from_chars(
			0xb1,0x25,0x03,0xb7,0xb2,0xf7,0x83,0x61,0x88,0x84,0x17,0x4b,0xcb,0x9b,0xe1,0x08,
			0x77,0x96,0x04,0x31,0xed,0x63,0x63,0xc8,0x07,0xe1,0x2d,0xb7,0x1b,0x8b,0x6b,0xd9,
			0xd6,0x40,0x1d,0x06,0x4e,0x25,0x37,0x40,0x15,0x8e,0x8b,0x90,0x01,0x52,0xd3,0x7f,
			0xaf,0x20,0x33,0x3a,0x7d,0x80,0xb3,0xd4,0x7c,0x7c,0x7a,0x3f,0xa1,0x20,0x91,0xce,
			0x31,0xcd,0x8a,0xae,0x27,0x2a,0x4d,0xa1,0x5f,0xe2,0xcb,0x5c,0xfd,0xea,0x54,0x11,
			0x95,0xa4,0x69,0xc9,0x6b,0xcf,0x69,0x5e,0x0b,0x52,0x6d,0xfa,0x48,0xa5,0x90,0x03,
			0xc6,0x76,0x3a,0xf8,0x13,0x63,0x92,0xc4,0xb8,0xd2,0x4d,0xb3,0x14,0x74,0x6f,0x42,
			0xac,0xa5,0x50,0xac,0xc6,0x5e,0x07,0x49,0x13,0xab,0x82,0x23,0x2e,0xb8,0x59,0x35,
			0x09,0x15,0x8a,0x8b,0xa3,0x4b,0xc0,0xf0,0xe3,0x12,0x5a,0x83,0x4a,0x3e,0xd2,0xd6,
			0xa8,0xcb,0x1d,0x08,0x5f,0x23,0x4a,0xe8,0x68,0xb8,0x6a,0xea,0x8d,0x6f,0x82,0xe1,
			0x3a,0x08,0x84,0x24,0x85,0x06,0x6e,0x48,0xaa,0xe4,0x83,0x78,0x73,0x15,0x0f,0x44,
			0x47,0x5e,0x12,0x60,0x2b,0x55,0x2d,0xcb,0x34,0xd1,0xf9,0xfd,0xaa,0xdb,0xc6,0xbf,
			0xf5,0x13,0x4c,0x6f,0xc7,0x62,0x63,0x88,0x8b,0xe6,0x7e,0xfe,0x63,0xee,0x18,0x40,
			0xfa,0x08,0xc4,0x99,0x38,0x85,0x8a,0x9d,0x48,0xb1,0x05,0x8d,0x18,0x97,0x6b,0xf2,
			0xe3,0xbf,0xc6,0x25,0x55,0x2f,0x75,0xb3,0xea,0x44,0xeb,0x91,0xdd,0x36,0x68,0x65,
			0xf2,0x40,0xa0,0xc3,0x36,0xa0,0x11,0x0e,0x0f,0xa0,0x9d,0x09,0xcd,0x94,0xc7,0x0c,
			0xbc,0x88,0x95,0xae,0x3d,0x44,0xae,0x3d,0xff,0x54,0x5f,0x0e,0x8c,0x8c,0xc6,0x62,
			0xec,0xd4,0x0f,0x90,0x99,0xa9,0x52,0x49,0x43,0x96,0xc6,0xb4,0x23,0xeb,0xb4,0x63,
			0x40,0x99,0x69,0x28,0x1c,0xdd,0x54,0xad,0x87,0xa3,0x08,0xe4,0x87,0xce,0x19,0x74,
			0x5b,0x30,0xd5,0xda,0x76,0xb9,0x8d,0x2a,0xa9,0xa0,0x07,0xa5,0x57,0x83,0xb3,0x03,
			0x7e,0x5b,0x86,0x62,0x32,0x28,0x10,0xbd,0xd1,0x1d,0x86,0xdc,0x3f,0x61,0x45,0x11,
			0x49,0x39,0x1f,0xb2,0xf1,0x4e,0xd9,0xc1,0x7c,0x75,0x16,0x23,0xa4,0x04,0x2c,0xe7,
			0xed,0xb8,0x75,0xee,0x27,0xbc,0xd1,0xf1,0x9d,0x6d,0xc9,0x28,0x3a,0xd0,0x6d,0x15,
			0xe0,0x97,0xe2,0xb0,0xb1,0x5a,0x7e,0xb7,0x12,0x8a,0xdb,0xca,0x0a,0xa6,0xad,0xcc),
	},
};

START_TEST(test_sign_pkcs15_3072)
{
	signature_scheme_t scheme = pkcs15_3072[_i].scheme;
	chunk_t sig;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, scheme)) ||
		!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PRIVKEY_SIGN, scheme)))
	{
		warn("%N not supported, skip test", signature_scheme_names, scheme);
		return;
	}
	fail_unless(privkey_3072->sign(privkey_3072, scheme, NULL, pkcs15_3072[_i].m, &sig),
				"sign %N", signature_scheme_names, scheme);
	ck_assert_chunk_eq(pkcs15_3072[_i].s, sig);
	fail_unless(pubkey_3072->verify(pubkey_3072, scheme, NULL, pkcs15_3072[_i].m, sig),
				"verify %N", signature_scheme_names, scheme);
	free(sig.ptr);
}
END_TEST

/**
 * Keys for the FIPS 186-4 SigGen RSASSA-PSS test vectors
 */
static private_key_t *privpss_2048, *privpss_3072;
static public_key_t *pubpss_2048, *pubpss_3072;
static bool pss_salt_supported;

static void pss_salt_test()
{
	chunk_t sig_salt = chunk_from_chars(
			0x3c,0x34,0xa4,0x56,0xdb,0x84,0x7b,0x7b,0xbb,0xbb,0xb0,0x15,0x5c,0xe2,0xe2,0x26,
			0xde,0xa9,0x27,0xb8,0x71,0x1b,0x28,0x0d,0xe3,0xe4,0xc2,0x6b,0x44,0x39,0xae,0x78,
			0x3a,0x93,0x68,0x6c,0xa0,0x91,0x19,0xd3,0x15,0x3b,0x93,0x15,0x92,0x13,0xaa,0x22,
			0x76,0x18,0xfd,0xbf,0x53,0x7f,0x2c,0xa7,0x24,0x5d,0xfa,0x93,0x78,0xff,0xdf,0x25,
			0x25,0x6c,0x65,0x2d,0xc1,0x6c,0xfd,0x43,0x87,0x3c,0x83,0x5b,0xe0,0x76,0xaf,0x99,
			0xa7,0xe4,0x11,0x3d,0xa6,0xa7,0xa4,0x50,0x0c,0xb8,0x4b,0x9b,0xd7,0x29,0xa6,0x6c,
			0x39,0x15,0xef,0xd9,0x46,0x5d,0x64,0x7e,0xf9,0x90,0x4a,0x7e,0x53,0xeb,0xfd,0x2f,
			0x32,0xdc,0x38,0xdf,0x3d,0xef,0xf7,0xa5,0x87,0xf0,0x5c,0x33,0x78,0x61,0x40,0x36,
			0x8d,0x16,0x14,0xf3,0xeb,0xfb,0x58,0x99,0x06,0xe9,0x48,0x67,0x93,0xe7,0x84,0x3c,
			0x6c,0x4a,0x3c,0x7e,0x0b,0xcc,0x62,0xcf,0xdb,0x7b,0x85,0xff,0x6c,0x3d,0x9f,0x7c,
			0x22,0x9e,0x0a,0x83,0x0b,0x3a,0x6a,0xe9,0xc2,0xbe,0x2f,0xd2,0xab,0x66,0xcb,0x1a,
			0x52,0x27,0x71,0x72,0x46,0xbf,0xed,0x8d,0xaa,0x47,0xaf,0x88,0x3a,0xf2,0xc2,0x6d,
			0xf4,0x24,0x01,0x4c,0x21,0xa1,0x40,0x63,0x93,0xeb,0xd9,0x56,0xf5,0xd2,0x74,0xe6,
			0x16,0x94,0x89,0x74,0x31,0x71,0xba,0xbd,0x6a,0x89,0x4a,0x52,0x9a,0x79,0x07,0x68,
			0x34,0x24,0x54,0xfd,0x21,0x54,0x7d,0xbe,0x3a,0x46,0xa6,0x28,0xa5,0xbc,0xea,0x30,
			0xb6,0x85,0xda,0x7e,0x45,0x24,0xc9,0xba,0x5f,0x0d,0xee,0xea,0x1b,0x54,0x67,0xc6);
	chunk_t sig_zero = chunk_from_chars(
			0xb0,0x7e,0x08,0xf4,0xb7,0x02,0xf7,0x91,0x7a,0x81,0x65,0xc3,0x87,0x39,0xe7,0x67,
			0xfa,0x28,0x4d,0xe5,0x65,0xe3,0x95,0x94,0xfd,0x30,0x1d,0x94,0x98,0x81,0xce,0x0c,
			0x0e,0x8d,0x40,0xc1,0x37,0x43,0x9b,0xbd,0x81,0x0e,0x19,0x8c,0xee,0xd8,0xd9,0x7a,
			0x95,0x96,0x76,0x57,0x0b,0x86,0xc5,0x69,0xe9,0x4d,0x9a,0x82,0x80,0xd5,0xf7,0x95,
			0x15,0x7c,0xfc,0x82,0xe6,0x0a,0x1a,0x13,0x84,0xb3,0xbc,0xe6,0xe3,0xc0,0xd4,0x84,
			0xcf,0xd4,0xdd,0xbd,0x5a,0x53,0x49,0x1f,0x72,0x47,0xbf,0x93,0x59,0xf4,0xf2,0xfb,
			0xcf,0xf8,0x42,0xb6,0xdb,0xb4,0xfb,0xea,0xa6,0x2e,0xdc,0xb5,0x3c,0x41,0x73,0xaa,
			0xba,0x0a,0xb3,0x7b,0x23,0x03,0x0a,0x71,0x7f,0x49,0xba,0x17,0xaa,0x35,0x15,0xeb,
			0xd0,0xea,0xae,0xb2,0x7a,0xf2,0xed,0x58,0x74,0xb0,0xe5,0xa8,0x1e,0x59,0x10,0x88,
			0xe6,0xe6,0xf7,0xc6,0x41,0x78,0xf5,0x88,0x35,0x03,0xf9,0x82,0x42,0xb5,0x42,0x43,
			0xc6,0x4f,0xeb,0xbf,0xa0,0x3b,0xef,0x61,0x35,0x80,0x2c,0xcb,0x3f,0x3a,0x3e,0x3d,
			0x7a,0x75,0x5e,0x0b,0xdb,0xc8,0xdf,0xc1,0x2f,0x73,0x55,0xec,0x2b,0x20,0xd0,0xf3,
			0xa3,0x8a,0xaf,0x3a,0xb3,0xe5,0xad,0xa2,0x9c,0x4f,0x6b,0x64,0x6b,0xb0,0x54,0x38,
			0x96,0xaa,0xb6,0x9c,0x8b,0xe9,0x7d,0x41,0xa5,0x0a,0xde,0x24,0x4d,0x5a,0x37,0xe6,
			0xe3,0xbd,0xbb,0x1c,0x00,0x8a,0x22,0xd5,0x51,0x70,0x18,0xaf,0x13,0x05,0x6e,0x32,
			0xf7,0xef,0xc3,0xe5,0x4b,0x6f,0x94,0xa0,0x59,0xa8,0xf8,0x7f,0x42,0x08,0x3b,0x51);
	chunk_t sig;
	rsa_pss_params_t params = {
		.hash = HASH_SHA256,
		.mgf1_hash = HASH_SHA256,
		.salt = chunk_from_chars(0x01,0x02,0x03,0x04),
	};

	pss_salt_supported = FALSE;
	if (!lib->plugins->has_feature(lib->plugins,
						PLUGIN_PROVIDE(PRIVKEY_SIGN, SIGN_RSA_EMSA_PSS)))
	{
		return;
	}
	/* we should get one of two signatures, either one without a salt or
	 * one with the configured salt */
	fail_unless(privpss_2048->sign(privpss_2048, SIGN_RSA_EMSA_PSS,
				&params, chunk_from_chars(0x04,0x03,0x02,0x01), &sig),
				"sign %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);

	if (chunk_equals(sig, sig_salt))
	{	/* matches signature with passed salt */
		pss_salt_supported = TRUE;
	}
	else if (!chunk_equals(sig, sig_zero))
	{	/* does not match signature without salt */
		fail("unexpected signature received: %B", &sig);
	}
	free(sig.ptr);
}

START_SETUP(setup_keys_pss)
{
	create_key(chunk_from_chars(
			0xc5,0x06,0x2b,0x58,0xd8,0x53,0x9c,0x76,0x5e,0x1e,0x5d,0xba,0xf1,0x4c,0xf7,0x5d,
			0xd5,0x6c,0x2e,0x13,0x10,0x5f,0xec,0xfd,0x1a,0x93,0x0b,0xbb,0x59,0x48,0xff,0x32,
			0x8f,0x12,0x6a,0xbe,0x77,0x93,0x59,0xca,0x59,0xbc,0xa7,0x52,0xc3,0x08,0xd2,0x81,
			0x57,0x3b,0xc6,0x17,0x8b,0x6c,0x0f,0xef,0x7d,0xc4,0x45,0xe4,0xf8,0x26,0x43,0x04,
			0x37,0xb9,0xf9,0xd7,0x90,0x58,0x1d,0xe5,0x74,0x9c,0x2c,0xb9,0xcb,0x26,0xd4,0x2b,
			0x2f,0xee,0x15,0xb6,0xb2,0x6f,0x09,0xc9,0x96,0x70,0x33,0x64,0x23,0xb8,0x6b,0xc5,
			0xbe,0xc7,0x11,0x13,0x15,0x7b,0xe2,0xd9,0x44,0xd7,0xff,0x3e,0xeb,0xff,0xb2,0x84,
			0x13,0x14,0x3e,0xa3,0x67,0x55,0xdb,0x0a,0xe6,0x2f,0xf5,0xb7,0x24,0xee,0xcb,0x3d,
			0x31,0x6b,0x6b,0xac,0x67,0xe8,0x9c,0xac,0xd8,0x17,0x19,0x37,0xe2,0xab,0x19,0xbd,
			0x35,0x3a,0x89,0xac,0xea,0x8c,0x36,0xf8,0x1c,0x89,0xa6,0x20,0xd5,0xfd,0x2e,0xff,
			0xea,0x89,0x66,0x01,0xc7,0xf9,0xda,0xca,0x7f,0x03,0x3f,0x63,0x5a,0x3a,0x94,0x33,
			0x31,0xd1,0xb1,0xb4,0xf5,0x28,0x87,0x90,0xb5,0x3a,0xf3,0x52,0xf1,0x12,0x1c,0xa1,
			0xbe,0xf2,0x05,0xf4,0x0d,0xc0,0x12,0xc4,0x12,0xb4,0x0b,0xdd,0x27,0x58,0x5b,0x94,
			0x64,0x66,0xd7,0x5f,0x7e,0xe0,0xa7,0xf9,0xd5,0x49,0xb4,0xbe,0xce,0x6f,0x43,0xac,
			0x3e,0xe6,0x5f,0xe7,0xfd,0x37,0x12,0x33,0x59,0xd9,0xf1,0xa8,0x50,0xad,0x45,0x0a,
			0xaf,0x5c,0x94,0xeb,0x11,0xde,0xa3,0xfc,0x0f,0xc6,0xe9,0x85,0x6b,0x18,0x05,0xef),
		chunk_from_chars(
			0x86,0xc9,0x4f),
		chunk_from_chars(
			0x49,0xe5,0x78,0x6b,0xb4,0xd3,0x32,0xf9,0x45,0x86,0x32,0x7b,0xde,0x08,0x88,0x75,
			0x37,0x9b,0x75,0xd1,0x28,0x48,0x8f,0x08,0xe5,0x74,0xab,0x47,0x15,0x30,0x2a,0x87,
			0xee,0xa5,0x2d,0x4c,0x4a,0x23,0xd8,0xb9,0x7a,0xf7,0x94,0x48,0x04,0x33,0x7c,0x5f,
			0x55,0xe1,0x6b,0xa9,0xff,0xaf,0xc0,0xc9,0xfd,0x9b,0x88,0xec,0xa4,0x43,0xf3,0x9b,
			0x79,0x67,0x17,0x0d,0xdb,0x8c,0xe7,0xdd,0xb9,0x3c,0x60,0x87,0xc8,0x06,0x6c,0x4a,
			0x95,0x53,0x8a,0x44,0x1b,0x9d,0xc8,0x0d,0xc9,0xf7,0x81,0x00,0x54,0xfd,0x1e,0x5c,
			0x9d,0x02,0x50,0xc9,0x78,0xbb,0x2d,0x74,0x8a,0xbe,0x1e,0x94,0x65,0xd7,0x1a,0x81,
			0x65,0xd3,0x12,0x6d,0xce,0x5d,0xb2,0xad,0xac,0xc0,0x03,0xe9,0x06,0x2b,0xa3,0x7a,
			0x54,0xb6,0x3e,0x5f,0x49,0xa4,0xea,0xfe,0xbd,0x7e,0x4b,0xf5,0xb0,0xa7,0x96,0xc2,
			0xb3,0xa9,0x50,0xfa,0x09,0xc7,0x98,0xd3,0xfa,0x3e,0x86,0xc4,0xb6,0x2c,0x33,0xba,
			0x93,0x65,0xed,0xa0,0x54,0xe5,0xfe,0x74,0xa4,0x1f,0x21,0xb5,0x95,0x02,0x6a,0xcf,
			0x10,0x93,0xc9,0x0a,0x8c,0x71,0x72,0x2f,0x91,0xaf,0x1e,0xd2,0x9a,0x41,0xa2,0x44,
			0x9a,0x32,0x0f,0xc7,0xba,0x31,0x20,0xe3,0xe8,0xc3,0xe4,0x24,0x0c,0x04,0x92,0x5c,
			0xc6,0x98,0xec,0xd6,0x6c,0x7c,0x90,0x6b,0xdf,0x24,0x0a,0xda,0xd9,0x72,0xb4,0xdf,
			0xf4,0x86,0x9d,0x40,0x0b,0x5d,0x13,0xe3,0x3e,0xeb,0xa3,0x8e,0x07,0x5e,0x87,0x2b,
			0x0e,0xd3,0xe9,0x1c,0xc9,0xc2,0x83,0x86,0x7a,0x4f,0xfc,0x39,0x01,0xd2,0x06,0x9f),
		&privpss_2048, &pubpss_2048);

	create_key(chunk_from_chars(
			0xa7,0xa1,0x88,0x2a,0x7f,0xb8,0x96,0x78,0x60,0x34,0xd0,0x7f,0xb1,0xb9,0xf6,0x32,
			0x7c,0x27,0xbd,0xd7,0xce,0x6f,0xe3,0x9c,0x28,0x5a,0xe3,0xb6,0xc3,0x42,0x59,0xad,
			0xc0,0xdc,0x4f,0x7b,0x9c,0x7d,0xec,0x3c,0xa4,0xa2,0x0d,0x34,0x07,0x33,0x9e,0xed,
			0xd7,0xa1,0x2a,0x42,0x1d,0xa1,0x8f,0x59,0x54,0x67,0x3c,0xac,0x2f,0xf0,0x59,0x15,
			0x6e,0xcc,0x73,0xc6,0x86,0x1e,0xc7,0x61,0xe6,0xa0,0xf2,0xa5,0xa0,0x33,0xa6,0x76,
			0x8c,0x6a,0x42,0xd8,0xb4,0x59,0xe1,0xb4,0x93,0x23,0x49,0xe8,0x4e,0xfd,0x92,0xdf,
			0x59,0xb4,0x59,0x35,0xf3,0xd0,0xe3,0x08,0x17,0xc6,0x62,0x01,0xaa,0x99,0xd0,0x7a,
			0xe3,0x6c,0x5d,0x74,0xf4,0x08,0xd6,0x9c,0xc0,0x8f,0x04,0x41,0x51,0xff,0x49,0x60,
			0xe5,0x31,0x36,0x0c,0xb1,0x90,0x77,0x83,0x3a,0xdf,0x7b,0xce,0x77,0xec,0xfa,0xa1,
			0x33,0xc0,0xcc,0xc6,0x3c,0x93,0xb8,0x56,0x81,0x45,0x69,0xe0,0xb9,0x88,0x4e,0xe5,
			0x54,0x06,0x1b,0x9a,0x20,0xab,0x46,0xc3,0x82,0x63,0xc0,0x94,0xda,0xe7,0x91,0xaa,
			0x61,0xa1,0x7f,0x8d,0x16,0xf0,0xe8,0x5b,0x7e,0x5c,0xe3,0xb0,0x67,0xec,0xe8,0x9e,
			0x20,0xbc,0x4e,0x8f,0x1a,0xe8,0x14,0xb2,0x76,0xd2,0x34,0xe0,0x4f,0x4e,0x76,0x6f,
			0x50,0x1d,0xa7,0x4e,0xa7,0xe3,0x81,0x7c,0x24,0xea,0x35,0xd0,0x16,0x67,0x6c,0xec,
			0xe6,0x52,0xb8,0x23,0xb0,0x51,0x62,0x55,0x73,0xca,0x92,0x75,0x7f,0xc7,0x20,0xd2,
			0x54,0xec,0xf1,0xdc,0xbb,0xfd,0x21,0xd9,0x83,0x07,0x56,0x1e,0xca,0xab,0x54,0x54,
			0x80,0xc7,0xc5,0x2a,0xd7,0xe9,0xfa,0x6b,0x59,0x7f,0x5f,0xe5,0x50,0x55,0x9c,0x2f,
			0xe9,0x23,0x20,0x5a,0xc1,0x76,0x1a,0x99,0x73,0x7c,0xa0,0x2d,0x7b,0x19,0x82,0x2e,
			0x00,0x8a,0x89,0x69,0x34,0x9c,0x87,0xfb,0x87,0x4c,0x81,0x62,0x0e,0x38,0xf6,0x13,
			0xc8,0x52,0x1f,0x03,0x81,0xfe,0x5b,0xa5,0x5b,0x74,0x82,0x7d,0xad,0x3e,0x1c,0xf2,
			0xaa,0x29,0xc6,0x93,0x36,0x29,0xf2,0xb2,0x86,0xad,0x11,0xbe,0x88,0xfa,0x64,0x36,
			0xe7,0xe3,0xf6,0x4a,0x75,0xe3,0x59,0x52,0x90,0xdc,0x0d,0x1c,0xd5,0xee,0xe7,0xaa,
			0xac,0x54,0x95,0x9c,0xc5,0x3b,0xd5,0xa9,0x34,0xa3,0x65,0xe7,0x2d,0xd8,0x1a,0x2b,
			0xd4,0xfb,0x9a,0x67,0x82,0x1b,0xff,0xed,0xf2,0xef,0x2b,0xd9,0x49,0x13,0xde,0x8b),
		chunk_from_chars(
			0x14,0x15,0xa7),
		chunk_from_chars(
			0x07,0x3a,0x5f,0xc4,0xcd,0x64,0x2f,0x61,0x13,0xdf,0xfc,0x4f,0x84,0x03,0x5c,0xee,
			0x3a,0x2b,0x8a,0xcc,0x54,0x97,0x03,0x75,0x1a,0x1d,0x6a,0x5e,0xaa,0x13,0x48,0x72,
			0x29,0xa5,0x8e,0xf7,0xd7,0xa5,0x22,0xbb,0x9f,0x4f,0x25,0x51,0x0f,0x1a,0xa0,0xf7,
			0x4c,0x6a,0x8f,0xc8,0xa5,0xc5,0xbe,0x8b,0x91,0xa6,0x74,0xed,0xe5,0x0e,0x92,0xf7,
			0xe3,0x4a,0x90,0xa3,0xc9,0xda,0x99,0x9f,0xff,0xb1,0xd6,0x95,0xe4,0x58,0x8f,0x45,
			0x12,0x56,0xc1,0x63,0x48,0x4c,0x15,0x13,0x50,0xcb,0x9c,0x78,0x25,0xa7,0xd9,0x10,
			0x84,0x5e,0xe5,0xcf,0x82,0x6f,0xec,0xf9,0xa7,0xc0,0xfb,0xbb,0xba,0x22,0xbb,0x4a,
			0x53,0x1c,0x13,0x1d,0x2e,0x77,0x61,0xba,0x89,0x8f,0x00,0x2e,0xbe,0xf8,0xab,0x87,
			0x21,0x85,0x11,0xf8,0x1d,0x32,0x66,0xe1,0xec,0x07,0xa7,0xca,0x86,0x22,0x51,0x4c,
			0x6d,0xfd,0xc8,0x6c,0x67,0x67,0x9a,0x2c,0x8f,0x5f,0x03,0x1d,0xe9,0xa0,0xc2,0x2b,
			0x5a,0x88,0x06,0x0b,0x46,0xee,0x0c,0x64,0xd3,0xb9,0xaf,0x3c,0x0a,0x37,0x9b,0xcd,
			0x9c,0x6a,0x1b,0x51,0xcf,0x64,0x80,0x45,0x6d,0x3f,0xd6,0xde,0xf9,0x4c,0xd2,0xa6,
			0xc1,0x71,0xdd,0x3f,0x01,0x0e,0x3c,0x9d,0x66,0x2b,0xc8,0x57,0x20,0x82,0x48,0xc9,
			0x4e,0xbc,0xb9,0xfd,0x99,0x7b,0x9f,0xf4,0xa7,0xe5,0xfd,0x95,0x55,0x85,0x69,0x90,
			0x65,0x25,0xe7,0x41,0xd7,0x83,0x44,0xf6,0xf6,0xcf,0xdb,0xd5,0x9d,0x4f,0xaa,0x52,
			0xee,0x3f,0xa9,0x64,0xfb,0x7c,0xcc,0xb2,0xd6,0xbe,0x19,0x35,0xd2,0x11,0xfe,0x14,
			0x98,0x21,0x77,0x16,0x27,0x39,0x39,0xa9,0x46,0x08,0x1f,0xd8,0x50,0x99,0x13,0xfd,
			0x47,0x74,0x7c,0x5c,0x2f,0x03,0xef,0xd4,0xd6,0xfc,0x9c,0x6f,0xcf,0xd8,0x40,0x2e,
			0x9f,0x40,0xa0,0xa5,0xb3,0xde,0x3c,0xa2,0xb3,0xc0,0xfa,0xc9,0x45,0x69,0x38,0xfa,
			0xa6,0xcf,0x2c,0x20,0xe3,0x91,0x2e,0x59,0x81,0xc9,0x87,0x6d,0x8c,0xa1,0xff,0x29,
			0xb8,0x7a,0x15,0xee,0xae,0x0c,0xcc,0xe3,0xf8,0xa8,0xf1,0xe4,0x05,0x09,0x1c,0x08,
			0x3b,0x98,0xbc,0xc5,0xfe,0x0d,0x0d,0xea,0xae,0x33,0xc6,0x7c,0x03,0x94,0x43,0x7f,
			0x0e,0xcc,0xb3,0x85,0xb7,0xef,0xb1,0x7a,0xee,0xbb,0xa8,0xaf,0xae,0xcc,0xa3,0x0a,
			0x2f,0x63,0xea,0xc8,0xf0,0xac,0x8f,0x1e,0xac,0xad,0x85,0xbb,0xca,0xf3,0x96,0x0b),
		&privpss_3072, &pubpss_3072);

	pss_salt_test();
}
END_SETUP

START_TEARDOWN(teardown_keys_pss)
{
	pubpss_2048->destroy(pubpss_2048);
	privpss_2048->destroy(privpss_2048);
	pubpss_3072->destroy(pubpss_3072);
	privpss_3072->destroy(privpss_3072);
}
END_TEARDOWN

/**
 * SigGen test vectors from FIPS 186-4 (first two for each hash algo)
 */
static struct {
	rsa_pss_params_t params;
	chunk_t m;
	chunk_t s;
} pss_sign_2048[] = {
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256,
			.salt_len = 20,
			.salt = chunk_from_chars(
				0xe1,0x25,0x6f,0xc1,0xee,0xef,0x81,0x77,0x3f,0xdd,0x54,0x65,0x7e,0x40,0x07,0xfd,
				0xe6,0xbc,0xb9,0xb1),
		},
		.m = chunk_from_chars(
			0xdf,0xc2,0x26,0x04,0xb9,0x5d,0x15,0x32,0x80,0x59,0x74,0x5c,0x6c,0x98,0xeb,0x9d,
			0xfb,0x34,0x7c,0xf9,0xf1,0x70,0xaf,0xf1,0x9d,0xee,0xec,0x55,0x5f,0x22,0x28,0x5a,
			0x67,0x06,0xc4,0xec,0xbf,0x0f,0xb1,0x45,0x8c,0x60,0xd9,0xbf,0x91,0x3f,0xba,0xe6,
			0xf4,0xc5,0x54,0xd2,0x45,0xd9,0x46,0xb4,0xbc,0x5f,0x34,0xae,0xc2,0xac,0x6b,0xe8,
			0xb3,0x3d,0xc8,0xe0,0xe3,0xa9,0xd6,0x01,0xdf,0xd5,0x36,0x78,0xf5,0x67,0x44,0x43,
			0xf6,0x7d,0xf7,0x8a,0x3a,0x9e,0x09,0x33,0xe5,0xf1,0x58,0xb1,0x69,0xac,0x8d,0x1c,
			0x4c,0xd0,0xfb,0x87,0x2c,0x14,0xca,0x8e,0x00,0x1e,0x54,0x2e,0xa0,0xf9,0xcf,0xda,
			0x88,0xc4,0x2d,0xca,0xd8,0xa7,0x40,0x97,0xa0,0x0c,0x22,0x05,0x5b,0x0b,0xd4,0x1f),
		.s = chunk_from_chars(
			0x8b,0x46,0xf2,0xc8,0x89,0xd8,0x19,0xf8,0x60,0xaf,0x0a,0x6c,0x4c,0x88,0x9e,0x4d,
			0x14,0x36,0xc6,0xca,0x17,0x44,0x64,0xd2,0x2a,0xe1,0x1b,0x9c,0xcc,0x26,0x5d,0x74,
			0x3c,0x67,0xe5,0x69,0xac,0xcb,0xc5,0xa8,0x0d,0x4d,0xd5,0xf1,0xbf,0x40,0x39,0xe2,
			0x3d,0xe5,0x2a,0xec,0xe4,0x02,0x91,0xc7,0x5f,0x89,0x36,0xc5,0x8c,0x9a,0x2f,0x77,
			0xa7,0x80,0xbb,0xe7,0xad,0x31,0xeb,0x76,0x74,0x2f,0x7b,0x2b,0x8b,0x14,0xca,0x1a,
			0x71,0x96,0xaf,0x7e,0x67,0x3a,0x3c,0xfc,0x23,0x7d,0x50,0xf6,0x15,0xb7,0x5c,0xf4,
			0xa7,0xea,0x78,0xa9,0x48,0xbe,0xda,0xf9,0x24,0x24,0x94,0xb4,0x1e,0x1d,0xb5,0x1f,
			0x43,0x7f,0x15,0xfd,0x25,0x51,0xbb,0x5d,0x24,0xee,0xfb,0x1c,0x3e,0x60,0xf0,0x36,
			0x94,0xd0,0x03,0x3a,0x1e,0x0a,0x9b,0x9f,0x5e,0x4a,0xb9,0x7d,0x45,0x7d,0xff,0x9b,
			0x9d,0xa5,0x16,0xdc,0x22,0x6d,0x6d,0x65,0x29,0x50,0x03,0x08,0xed,0x74,0xa2,0xe6,
			0xd9,0xf3,0xc1,0x05,0x95,0x78,0x8a,0x52,0xa1,0xbc,0x06,0x64,0xae,0xdf,0x33,0xef,
			0xc8,0xba,0xdd,0x03,0x7e,0xb7,0xb8,0x80,0x77,0x2b,0xdb,0x04,0xa6,0x04,0x6e,0x9e,
			0xde,0xee,0x41,0x97,0xc2,0x55,0x07,0xfb,0x0f,0x11,0xab,0x1c,0x9f,0x63,0xf5,0x3c,
			0x88,0x20,0xea,0x84,0x05,0xcf,0xd7,0x72,0x16,0x92,0x47,0x5b,0x4d,0x72,0x35,0x5f,
			0xa9,0xa3,0x80,0x4f,0x29,0xe6,0xb6,0xa7,0xb0,0x59,0xc4,0x44,0x1d,0x54,0xb2,0x8e,
			0x4e,0xed,0x25,0x29,0xc6,0x10,0x3b,0x54,0x32,0xc7,0x13,0x32,0xce,0x74,0x2b,0xcc),
	},
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256,
			.salt_len = 20,
			.salt = chunk_from_chars(
				0xe1,0x25,0x6f,0xc1,0xee,0xef,0x81,0x77,0x3f,0xdd,0x54,0x65,0x7e,0x40,0x07,0xfd,
				0xe6,0xbc,0xb9,0xb1),
		},
		.m = chunk_from_chars(
			0xfd,0x6a,0x06,0x3e,0x61,0xc2,0xb3,0x54,0xfe,0x8c,0xb3,0x7a,0x5f,0x37,0x88,0xb5,
			0xc0,0x1f,0xf1,0x5a,0x72,0x5f,0x6b,0x81,0x81,0xe6,0xf6,0xb7,0x95,0xce,0x1c,0xf3,
			0x16,0xe9,0x30,0xcc,0x93,0x9c,0xd4,0xe8,0x65,0xf0,0xbd,0xb8,0x8f,0xe6,0xbb,0x62,
			0xe9,0x0b,0xf3,0xff,0x7e,0x4d,0x6f,0x07,0x32,0x0d,0xda,0x09,0xa8,0x75,0x84,0xa0,
			0x62,0x0c,0xad,0xa2,0x2a,0x87,0xff,0x9a,0xb1,0xe3,0x5c,0x79,0x77,0xb0,0xda,0x88,
			0xea,0xb0,0x0c,0xa1,0xd2,0xa0,0x84,0x9f,0xec,0x56,0x95,0x13,0xd5,0x0c,0x5e,0x39,
			0x2a,0xfc,0x03,0x2a,0xee,0x2d,0x3e,0x52,0x2c,0x8c,0x17,0x25,0xdd,0x3e,0xef,0x0e,
			0x0b,0x35,0xc3,0xa8,0x37,0x01,0xaf,0x31,0xf9,0xe9,0xb1,0x3c,0xe6,0x3b,0xb0,0xa5),
		.s = chunk_from_chars(
			0x49,0x2b,0x6f,0x68,0x84,0xdf,0x46,0x1f,0xe1,0x05,0x16,0xb6,0xb8,0xcc,0x20,0x53,
			0x85,0xc2,0x01,0x08,0xec,0x47,0xd5,0xdb,0x69,0x28,0x3f,0x4a,0x76,0x88,0xe3,0x18,
			0xcf,0xdc,0x3c,0x49,0x1f,0xb2,0x92,0x25,0x32,0x5a,0xeb,0x46,0xef,0xc7,0x5e,0x85,
			0x58,0x40,0x91,0x0b,0xba,0xf0,0xd1,0xc8,0xd4,0x78,0x45,0x42,0xb9,0x70,0x75,0x4a,
			0xaa,0x84,0xbf,0xe4,0x7c,0x77,0xb3,0xa1,0xb5,0x03,0x7d,0x4d,0x79,0x75,0x94,0x71,
			0xe9,0x6c,0xc7,0xa5,0x27,0xa0,0xed,0x06,0x7e,0x21,0x70,0x9e,0xf7,0xf4,0xc4,0x11,
			0x1b,0x60,0xb8,0xc0,0x80,0x82,0xc8,0x18,0x0c,0x7c,0x96,0xb6,0x1c,0x0f,0x71,0x02,
			0xed,0x9b,0x90,0xe2,0x4d,0xe1,0x1e,0x62,0x98,0xbb,0x24,0x45,0x18,0xf9,0xb4,0x46,
			0xce,0x64,0x1f,0xe9,0x95,0xe9,0xcc,0x29,0x9e,0xd4,0x11,0xb6,0x5e,0xb2,0x5e,0xaa,
			0xe9,0xe5,0x53,0x48,0x4a,0x0a,0x7e,0x95,0x6e,0xad,0xf0,0x84,0x08,0x88,0xc7,0x0e,
			0x5c,0xa6,0xeb,0xc3,0xe4,0x79,0xf8,0xc6,0x9c,0x53,0xcf,0x31,0x37,0x0a,0xb3,0x85,
			0xe8,0xb6,0x73,0xdc,0x45,0xa0,0xc1,0x96,0x4e,0xc4,0x94,0x68,0xd1,0x82,0x46,0x21,
			0x3a,0x8f,0x93,0xa2,0xa9,0x6a,0xad,0x5a,0x27,0x01,0xc1,0x91,0xa1,0x4a,0x31,0x51,
			0x9e,0x4f,0x36,0x54,0x4d,0x66,0x87,0x08,0xff,0x37,0xbe,0x54,0x81,0xcb,0x0f,0xfa,
			0x2b,0x0e,0x1f,0x14,0x5e,0x29,0xf8,0x57,0x5d,0xfa,0x9e,0xc3,0x0c,0x6c,0xb4,0x1c,
			0x39,0x34,0x39,0x29,0x22,0x10,0xea,0x80,0x6a,0x50,0x55,0x98,0xeb,0xdf,0x08,0x33),
	},
	{
		.params = { .hash = HASH_SHA384, .mgf1_hash = HASH_SHA384,
			.salt_len = 25,
			.salt = chunk_from_chars(
				0xb7,0x50,0x58,0x76,0x71,0xaf,0xd7,0x68,0x86,0xe8,0xff,0xb7,0x86,0x5e,0x78,0xf7,
				0x06,0x64,0x1b,0x2e,0x42,0x51,0xb4,0x87,0x06),
		},
		.m = chunk_from_chars(
			0x83,0x3a,0xa2,0xb1,0xdc,0xc7,0x76,0x07,0xa4,0x4e,0x80,0x4e,0xe7,0x7d,0x45,0x40,
			0x85,0x86,0xc5,0x36,0x86,0x1f,0x66,0x48,0xad,0xcd,0x2f,0xb6,0x50,0x63,0x36,0x87,
			0x67,0xc5,0x5c,0x6f,0xe2,0xf2,0x37,0xf6,0x40,0x42,0x50,0xd7,0x5d,0xec,0x8f,0xa6,
			0x8b,0xca,0xf3,0xb6,0xe5,0x61,0x86,0x3a,0xe0,0x1c,0x91,0xaa,0x23,0xd8,0x0c,0x69,
			0x99,0xa5,0x58,0xa4,0xc4,0xcb,0x31,0x7d,0x54,0x0c,0xde,0x69,0xf8,0x29,0xaa,0xd6,
			0x74,0xa8,0x98,0x12,0xf4,0xd3,0x53,0x68,0x9f,0x04,0x64,0x8c,0x70,0x20,0xa7,0x39,
			0x41,0x62,0x00,0x18,0x29,0x5a,0x4a,0xe4,0x08,0x35,0x90,0xcc,0x60,0x3e,0x80,0x18,
			0x67,0xa5,0x1c,0x10,0x5a,0x7f,0xb3,0x19,0x13,0x0f,0x10,0x22,0xde,0x44,0xf1,0x3e),
		.s = chunk_from_chars(
			0x2c,0xa3,0x7a,0x3d,0x6a,0xbd,0x28,0xc1,0xea,0xf9,0xbd,0xe5,0xe7,0xac,0x17,0xf1,
			0xfa,0x79,0x9c,0xe1,0xb4,0xb8,0x99,0xd1,0x99,0x85,0xc2,0xff,0x7c,0x8b,0xa9,0x59,
			0xfe,0x54,0xe5,0xaf,0xb8,0xbc,0x40,0x21,0xa1,0xf1,0xc6,0x87,0xee,0xbb,0x8c,0xba,
			0x80,0x0d,0x1c,0x51,0x63,0x6b,0x1f,0x68,0xdc,0x3e,0x48,0xf6,0x3e,0x2d,0xa6,0xbc,
			0x6d,0x09,0xc6,0x66,0x8f,0x68,0xe5,0x08,0xc5,0xd8,0xc1,0x9b,0xef,0x15,0x47,0x59,
			0xe2,0xf8,0x9a,0xde,0x15,0x27,0x17,0x37,0x0a,0x89,0x44,0xf5,0x37,0x57,0x82,0x96,
			0x38,0x0d,0x1f,0xe6,0xbe,0x80,0x9e,0x8b,0x11,0x3d,0x2b,0x9d,0x89,0xe6,0xa4,0x6f,
			0x5c,0x33,0x3d,0x4f,0xd4,0x87,0x70,0xfc,0x1e,0xa1,0xc5,0x48,0x10,0x45,0x75,0xb8,
			0x4c,0xf0,0x71,0x04,0x2b,0xfe,0x5a,0xcf,0x49,0x63,0x92,0xbe,0x83,0x51,0xa4,0x1c,
			0x46,0xa2,0xca,0xb0,0x86,0x4c,0x4c,0x1c,0x5b,0x5e,0x0c,0x7b,0x27,0xe7,0xb8,0x8c,
			0x69,0xf3,0x7f,0xfa,0x7e,0x1a,0x8c,0xd9,0x8f,0x34,0x3a,0xc8,0x4a,0x4a,0xd6,0x70,
			0x25,0xa4,0x0e,0xd8,0xf6,0x64,0xe9,0xd6,0x30,0x33,0x7d,0xe6,0xe4,0x8b,0xb2,0x12,
			0x5e,0x25,0x52,0x12,0x36,0x09,0x49,0x1f,0x18,0x3a,0xfd,0x92,0x63,0x44,0x87,0xf0,
			0xb2,0xcf,0x97,0x1f,0x26,0x26,0xe8,0x88,0x58,0x87,0x9d,0x45,0xa2,0x9b,0x0f,0xef,
			0xb6,0x6c,0xd4,0x1b,0x2e,0x4e,0x96,0x83,0x85,0xbd,0x9f,0xc8,0xc7,0x21,0x19,0x76,
			0xbc,0x6b,0xd3,0xe1,0xad,0x6d,0xf6,0x08,0x56,0x98,0x5a,0x82,0x5f,0x47,0x26,0xd2),
	},
	{
		.params = { .hash = HASH_SHA384, .mgf1_hash = HASH_SHA384,
			.salt_len = 25,
			.salt = chunk_from_chars(
				0xb7,0x50,0x58,0x76,0x71,0xaf,0xd7,0x68,0x86,0xe8,0xff,0xb7,0x86,0x5e,0x78,0xf7,
				0x06,0x64,0x1b,0x2e,0x42,0x51,0xb4,0x87,0x06),
		},
		.m = chunk_from_chars(
			0x89,0x25,0xb8,0x7e,0x9d,0x1d,0x73,0x9d,0x8f,0x97,0x54,0x50,0xb7,0x9d,0x09,0x19,
			0xdd,0xe6,0x3e,0x8a,0x9e,0xaa,0x1c,0xb5,0x11,0xb4,0x0f,0xe3,0xab,0xb9,0xcd,0x89,
			0x60,0xe8,0x94,0x77,0x0b,0xc2,0xb2,0x53,0x10,0x2c,0x4b,0x46,0x40,0xc3,0x57,0xf5,
			0xfd,0x6f,0xea,0xb3,0x9e,0x3b,0xb8,0xf4,0x15,0x64,0xd8,0x05,0xce,0xaf,0xc8,0xfb,
			0xdb,0x00,0xb2,0xea,0x4f,0x29,0xed,0x57,0xe7,0x00,0xc7,0xef,0xf0,0xb4,0x82,0x79,
			0x64,0x61,0x9c,0x09,0x57,0xe1,0x54,0x76,0x91,0xe6,0x69,0x0f,0x7d,0x45,0x25,0x8a,
			0x42,0x95,0x9a,0x3d,0x2f,0xf9,0x2c,0x91,0x5c,0x3a,0x4f,0xb3,0x8e,0x19,0x92,0x8c,
			0x5c,0xe3,0xdd,0xf4,0x90,0x45,0xf6,0x22,0xd0,0x62,0x4a,0x67,0x7e,0x23,0xeb,0x1d),
		.s = chunk_from_chars(
			0x43,0xef,0x93,0xd1,0x4e,0x89,0xb0,0x5d,0x5e,0x0d,0xb2,0xdb,0xd5,0x7a,0x12,0x40,
			0x39,0x10,0x64,0x6b,0x4b,0x0a,0x24,0xd9,0xb8,0x0d,0x94,0x79,0x54,0x59,0x1a,0xfa,
			0x6e,0x98,0x09,0xe9,0x6d,0x7d,0x3e,0x71,0x10,0x03,0xee,0x0a,0x91,0x86,0xab,0x3d,
			0x8e,0x0b,0x4d,0x34,0x25,0xc6,0xda,0x4b,0x5f,0x78,0x99,0x53,0x7e,0x73,0x7b,0x71,
			0xdf,0x9e,0xd6,0x35,0x55,0x29,0xaa,0xce,0x77,0xa7,0xcb,0xa9,0x6b,0x5b,0x0a,0x86,
			0x39,0x92,0x52,0xf1,0x28,0x6a,0x6f,0xca,0xb1,0x80,0xb5,0x98,0x45,0x5d,0xfe,0x1d,
			0xe4,0xb8,0x04,0x70,0xd0,0x63,0x18,0xd5,0xf7,0xa5,0x2e,0x45,0xb6,0xd0,0xbc,0xc0,
			0x0b,0xd3,0x65,0x81,0x9a,0x4a,0x14,0x2b,0x83,0x07,0x27,0x75,0xf4,0x85,0xf6,0x3c,
			0x80,0x04,0xf5,0x33,0x78,0xa9,0xa0,0xd2,0x34,0x5d,0x07,0xb1,0xb3,0x26,0x23,0x8e,
			0xd0,0x70,0xd1,0xe6,0x9f,0xc0,0xb5,0xcf,0x85,0x3a,0x80,0x7c,0xfb,0x72,0x35,0x62,
			0xd1,0xf5,0x68,0x24,0x82,0xe8,0xa4,0x84,0x05,0x88,0xbc,0xc7,0x15,0x4c,0xe0,0x74,
			0x0c,0x76,0x86,0x16,0xcf,0x04,0xd7,0xaa,0x10,0x36,0x42,0x91,0x7e,0xc5,0xb4,0xb5,
			0x14,0xa3,0x73,0x4d,0x9e,0x0c,0x58,0x42,0x7c,0xff,0x42,0xf2,0x7f,0x43,0xfd,0xfc,
			0x85,0x99,0x1e,0x04,0x5a,0xcd,0x17,0xaf,0x6f,0xba,0x7b,0xda,0xb8,0x18,0xe9,0x0e,
			0xb4,0x11,0x76,0x84,0xe8,0x9f,0x91,0x63,0xdf,0xf7,0xb9,0x8b,0x82,0xa0,0x8b,0xaa,
			0x2b,0x49,0xac,0xde,0x48,0x0c,0x57,0x02,0xc3,0x35,0x23,0x7d,0x1b,0xe7,0x71,0xb7),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512,
			.salt_len = 30,
			.salt = chunk_from_chars(
				0xaa,0x10,0xfe,0xc3,0xf8,0x3b,0x7a,0x97,0xe0,0x92,0x87,0x7a,0x5b,0xf9,0x08,0x12,
				0x83,0xf5,0x02,0xa0,0xa4,0x6b,0x50,0xe3,0x95,0xab,0x98,0x3a,0x49,0xac),
		},
		.m = chunk_from_chars(
			0x5f,0x0f,0xe2,0xaf,0xa6,0x1b,0x62,0x8c,0x43,0xea,0x3b,0x6b,0xa6,0x05,0x67,0xb1,
			0xae,0x95,0xf6,0x82,0x07,0x6f,0x01,0xdf,0xb6,0x4d,0xe0,0x11,0xf2,0x5e,0x9c,0x4b,
			0x36,0x02,0xa7,0x8b,0x94,0xce,0xcb,0xc1,0x4c,0xd7,0x61,0x33,0x9d,0x2d,0xc3,0x20,
			0xdb,0xa5,0x04,0xa3,0xc2,0xdc,0xde,0xdb,0x0a,0x78,0xeb,0x49,0x3b,0xb1,0x18,0x79,
			0xc3,0x11,0x58,0xe5,0x46,0x77,0x95,0x16,0x35,0x62,0xec,0x0c,0xa2,0x6c,0x19,0xe0,
			0x53,0x15,0x30,0xa8,0x15,0xc2,0x8f,0x9b,0x52,0x06,0x10,0x76,0xe6,0x1f,0x83,0x1e,
			0x2f,0xc4,0x5b,0x86,0x63,0x1e,0xa7,0xd3,0x27,0x14,0x44,0xbe,0x5d,0xcb,0x51,0x3a,
			0x3d,0x6d,0xe4,0x57,0xa7,0x2a,0xfb,0x67,0xb7,0x7d,0xb6,0x5f,0x9b,0xb1,0xc3,0x80),
		.s = chunk_from_chars(
			0x5e,0x07,0x12,0xbb,0x36,0x3e,0x50,0x34,0xef,0x6b,0x23,0xc1,0x19,0xe3,0xb4,0x98,
			0x64,0x44,0x45,0xfa,0xab,0x5a,0x4c,0x0b,0x4e,0x21,0x7e,0x4c,0x83,0x2a,0xb3,0x4c,
			0x14,0x2d,0x7f,0x81,0xdb,0xf8,0xaf,0xfd,0xb2,0xda,0xce,0xfa,0xbb,0x2f,0x83,0x52,
			0x4c,0x5a,0xa8,0x83,0xfc,0x5f,0x06,0xe5,0x28,0xb2,0x32,0xd9,0x0f,0xbe,0xa9,0xca,
			0x08,0xae,0x5a,0xc1,0x80,0xd4,0x77,0xea,0xed,0x27,0xd1,0x37,0xe2,0xb5,0x1b,0xd6,
			0x13,0xb6,0x9c,0x54,0x3d,0x55,0x5b,0xfc,0x7c,0xd8,0x1a,0x4f,0x79,0x57,0x53,0xc8,
			0xc6,0x4c,0x6b,0x5d,0x2a,0xcd,0x9e,0x26,0xd6,0x22,0x5f,0x5b,0x26,0xe4,0xe6,0x6a,
			0x94,0x5f,0xd6,0x47,0x7a,0x27,0x7b,0x58,0x0d,0xbe,0xaa,0x46,0xd0,0xbe,0x49,0x8d,
			0xf9,0xa0,0x93,0x39,0x29,0x26,0xc9,0x05,0x64,0x19,0x45,0xec,0x5b,0x95,0x97,0x52,
			0x5e,0x44,0x9a,0xf3,0x74,0x3f,0x80,0x55,0x47,0x88,0xfc,0x35,0x8b,0xc0,0x40,0x1a,
			0x96,0x8f,0xf9,0x8a,0xaf,0x34,0xe5,0x0b,0x35,0x27,0x51,0xf3,0x22,0x74,0x75,0x0f,
			0xf5,0xc1,0xfb,0xa5,0x03,0x05,0x02,0x04,0xce,0xc9,0xc7,0x7d,0xee,0xde,0x7f,0x8f,
			0xa2,0x08,0x45,0xd9,0x5f,0x51,0x77,0x03,0x0b,0xc9,0x1d,0x51,0xf2,0x6f,0x29,0xd2,
			0xa6,0x5b,0x87,0x0d,0xc7,0x2b,0x81,0xe5,0xef,0x9e,0xee,0xf9,0x90,0xd7,0xc7,0x14,
			0x5b,0xbf,0x1a,0x3b,0xc7,0xae,0xdd,0x19,0xfa,0x7c,0xbb,0x02,0x07,0x56,0x52,0x5f,
			0x18,0x02,0x21,0x6c,0x13,0x29,0x6f,0xd6,0xaa,0xc1,0x1b,0xf2,0xd2,0xd9,0x04,0x94),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512,
			.salt_len = 30,
			.salt = chunk_from_chars(
				0xaa,0x10,0xfe,0xc3,0xf8,0x3b,0x7a,0x97,0xe0,0x92,0x87,0x7a,0x5b,0xf9,0x08,0x12,
				0x83,0xf5,0x02,0xa0,0xa4,0x6b,0x50,0xe3,0x95,0xab,0x98,0x3a,0x49,0xac),
		},
		.m = chunk_from_chars(
			0x9e,0x88,0x0c,0xe5,0x9f,0x54,0x7d,0x59,0x2c,0x30,0x9c,0x22,0xa2,0x97,0x4b,0xa5,
			0xa5,0x2c,0xf1,0xc1,0x64,0xf2,0xd8,0xa8,0x1e,0xbb,0xd4,0xed,0xe6,0xe3,0x26,0xde,
			0xa3,0x3d,0x9f,0x13,0x5a,0x4e,0x09,0x47,0xb0,0xb9,0xc2,0x67,0xaa,0xfb,0xaa,0xe9,
			0xb8,0x58,0x3f,0x5f,0xf2,0x15,0x07,0x4c,0xa1,0xe8,0x2f,0x36,0x01,0xad,0x71,0xfc,
			0x45,0x5a,0x3b,0x6a,0xdc,0x35,0x0d,0x0b,0xf3,0x45,0x22,0x3e,0x3b,0x06,0x54,0x8c,
			0xec,0x61,0x3a,0x39,0x0a,0xda,0x93,0x19,0xe7,0x0c,0xe7,0xa5,0xe9,0x52,0x6b,0x4e,
			0x8d,0xc8,0x26,0x12,0xac,0x72,0x52,0x4c,0xfd,0xba,0x05,0xd0,0xdc,0x20,0x10,0x37,
			0x49,0x2d,0x27,0x78,0x34,0xa8,0x43,0xb9,0xf8,0x0d,0x45,0x64,0x25,0x3b,0xdc,0x7c),
		.s = chunk_from_chars(
			0x8c,0x4f,0x81,0x9e,0x68,0x20,0x81,0xbb,0x16,0xdd,0xd4,0x59,0x66,0x2a,0x80,0x78,
			0xbc,0xa4,0x79,0x3e,0x18,0x11,0x00,0x33,0x53,0x94,0x60,0xb4,0x08,0xc0,0xaf,0x74,
			0x7e,0xa5,0xd9,0x41,0xf7,0x12,0x69,0x1f,0x5d,0x9d,0xdb,0x64,0x31,0x66,0xfd,0x96,
			0x5f,0x5b,0x51,0xb8,0x19,0xd5,0x51,0x41,0xd6,0x7c,0x15,0x53,0xb2,0x7a,0x46,0x82,
			0xe6,0x7d,0x55,0x55,0xb6,0x4d,0x7c,0xd3,0xdb,0x7f,0xc5,0xc2,0xe7,0x01,0xdd,0x26,
			0xe4,0x22,0xaf,0x8a,0x1f,0xb5,0x2c,0xd5,0xf5,0xa0,0x9e,0x0d,0x6d,0xb9,0x00,0xa9,
			0x92,0xf3,0x18,0xde,0xeb,0x6f,0x6e,0x39,0xdf,0xd6,0xaf,0x44,0xcb,0x21,0x7c,0x68,
			0x54,0x08,0x9c,0xea,0xa1,0x6e,0x3f,0x9b,0x10,0x0e,0xf8,0xe7,0x8f,0x6b,0x45,0x34,
			0x58,0xb8,0xef,0x6d,0x71,0x49,0x3e,0x7c,0x6e,0x45,0x28,0x2c,0x61,0x7f,0xa8,0x7c,
			0xcd,0xd4,0xa0,0xf2,0xf9,0xf7,0x16,0x62,0x81,0x80,0x6f,0xb4,0x1d,0x0f,0xe1,0x88,
			0xe0,0x0c,0x40,0xaf,0xea,0xa0,0x7d,0x2d,0xa0,0x9a,0x2c,0xd7,0x80,0x52,0xf8,0xd5,
			0x6b,0x7a,0xf4,0x0d,0x4c,0x73,0x14,0xcc,0xf0,0x2e,0x49,0x0d,0x5e,0x21,0x23,0xbf,
			0x67,0x6f,0x2b,0xcb,0xda,0xbe,0xff,0xcf,0x58,0x79,0x29,0x98,0xdd,0x0f,0x67,0xed,
			0x24,0xe4,0x83,0xd8,0x97,0x6b,0x00,0xd6,0x15,0x1a,0x6e,0x0b,0xa7,0x40,0xbd,0xb5,
			0x7c,0x9b,0xc2,0x7f,0xe5,0xdf,0x91,0x26,0xa4,0x70,0x20,0x07,0x5e,0xb2,0x22,0xd5,
			0xca,0x24,0x70,0x72,0x44,0x60,0xc5,0xad,0xf0,0x67,0xb5,0x75,0x02,0x87,0xcd,0x00),
	},
};

START_TEST(test_sign_pss_2048)
{
	chunk_t sig;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, SIGN_RSA_EMSA_PSS)) ||
		!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PRIVKEY_SIGN, SIGN_RSA_EMSA_PSS)))
	{
		warn("%N not supported, skip test", signature_scheme_names,
			 SIGN_RSA_EMSA_PSS);
		return;
	}
	if (!pss_salt_supported)
	{
		warn("static salt for %N not supported, skip test",
			 signature_scheme_names, SIGN_RSA_EMSA_PSS);
		return;
	}
	fail_unless(privpss_2048->sign(privpss_2048, SIGN_RSA_EMSA_PSS,
							  &pss_sign_2048[_i].params, pss_sign_2048[_i].m, &sig),
				"sign %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	ck_assert_chunk_eq(pss_sign_2048[_i].s, sig);
	fail_unless(pubpss_2048->verify(pubpss_2048, SIGN_RSA_EMSA_PSS,
							   &pss_sign_2048[_i].params, pss_sign_2048[_i].m, sig),
				"verify %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	free(sig.ptr);
}
END_TEST

/**
 * SigGen test vectors from FIPS 186-4 (first two for each hash algo)
 */
static struct {
	rsa_pss_params_t params;
	chunk_t m;
	chunk_t s;
} pss_sign_3072[] = {
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256,
			.salt_len = 32,
			.salt = chunk_from_chars(
				0x3e,0x07,0xad,0xe7,0x2a,0x3f,0x52,0x53,0x0f,0x53,0x13,0x5a,0x5d,0x7d,0x93,0x21,
				0x74,0x35,0xba,0x00,0x1e,0xa5,0x5a,0x8f,0x5d,0x5d,0x13,0x04,0x68,0x48,0x74,0xbc),
		},
		.m = chunk_from_chars(
			0xc1,0x64,0x99,0x11,0x0e,0xd5,0x77,0x20,0x2a,0xed,0x2d,0x3e,0x4d,0x51,0xde,0xd6,
			0xc6,0x63,0x73,0xfa,0xef,0x65,0x33,0xa8,0x60,0xe1,0x93,0x4c,0x63,0x48,0x4f,0x87,
			0xa8,0xd9,0xb9,0x2f,0x3a,0xc4,0x51,0x97,0xb2,0x90,0x97,0x10,0xab,0xba,0x1d,0xaf,
			0x75,0x9f,0xe0,0x51,0x0e,0x9b,0xd8,0xdd,0x4d,0x73,0xce,0xc9,0x61,0xf0,0x6e,0xe0,
			0x7a,0xcd,0x9d,0x42,0xc6,0xd4,0x0d,0xac,0x9f,0x43,0x0e,0xf9,0x03,0x74,0xa7,0xe9,
			0x44,0xbd,0xe5,0x22,0x00,0x96,0x73,0x74,0x54,0xf9,0x6b,0x61,0x4d,0x0f,0x6c,0xdd,
			0x9f,0x08,0xed,0x52,0x9a,0x4a,0xd0,0xe7,0x59,0xcf,0x3a,0x02,0x3d,0xc8,0xa3,0x0b,
			0x9a,0x87,0x29,0x74,0xaf,0x9b,0x2a,0xf6,0xdc,0x3d,0x11,0x1d,0x0f,0xeb,0x70,0x06),
		.s = chunk_from_chars(
			0x43,0x35,0x70,0x7d,0xa7,0x35,0xcf,0xd1,0x04,0x11,0xc9,0xc0,0x48,0xca,0x9b,0x60,
			0xbb,0x46,0xe2,0xfe,0x36,0x1e,0x51,0xfb,0xe3,0x36,0xf9,0x50,0x8d,0xc9,0x45,0xaf,
			0xe0,0x75,0x50,0x3d,0x24,0xf8,0x36,0x61,0x0f,0x21,0x78,0x99,0x6b,0x52,0xc4,0x11,
			0x69,0x30,0x52,0xd5,0xd7,0xae,0xd9,0x76,0x54,0xa4,0x00,0x74,0xed,0x20,0xed,0x66,
			0x89,0xc0,0x50,0x1b,0x7f,0xba,0xc2,0x1d,0xc4,0x6b,0x66,0x5a,0xc0,0x79,0x76,0x00,
			0x86,0x41,0x44,0x06,0xcd,0x66,0xf8,0x53,0x7d,0x1e,0xbf,0x0d,0xce,0x4c,0xf0,0xc9,
			0x8d,0x4c,0x30,0xc7,0x1d,0xa3,0x59,0xe9,0xcd,0x40,0x1f,0xf4,0x97,0x18,0xfd,0xd4,
			0xd0,0xf9,0x9e,0xfe,0x70,0xad,0x8d,0xd8,0xba,0x13,0x04,0xce,0xfb,0x88,0xf2,0x4b,
			0x0e,0xed,0xf7,0x01,0x16,0xda,0x15,0x93,0x2c,0x76,0xf0,0x06,0x95,0x51,0xa2,0x45,
			0xb5,0xfc,0x3b,0x91,0xec,0x10,0x1f,0x1d,0x63,0xb9,0x85,0x3b,0x59,0x8c,0x6f,0xa1,
			0xc1,0xac,0xdb,0xac,0xf9,0x62,0x63,0x56,0xc7,0x60,0x11,0x9b,0xe0,0x95,0x56,0x44,
			0x30,0x18,0x96,0xd9,0xd0,0xd3,0xea,0x5e,0x64,0x43,0xcb,0x72,0xca,0x29,0xf4,0xd4,
			0x52,0x46,0xd1,0x6d,0x74,0xd0,0x05,0x68,0xc2,0x19,0x18,0x2f,0xeb,0x19,0x11,0x79,
			0xe4,0x59,0x3d,0xc1,0x52,0xc6,0x08,0xfd,0x80,0x53,0x63,0x29,0xa5,0x33,0xb3,0xa6,
			0x31,0x56,0x68,0x14,0xcd,0x65,0x4f,0x58,0x7c,0x2d,0x8c,0xe6,0x96,0x08,0x5e,0x6e,
			0xd1,0xb0,0xb0,0x27,0x8e,0x60,0xa0,0x49,0xec,0x7a,0x39,0x9f,0x94,0xfc,0xca,0xe6,
			0x46,0x23,0x71,0xa6,0x96,0x95,0xef,0x52,0x5e,0x00,0x93,0x6f,0xa7,0xd9,0x78,0x1f,
			0x9e,0xe2,0x89,0xd4,0x10,0x5e,0xe8,0x27,0xa2,0x79,0x96,0x58,0x30,0x33,0xce,0xdb,
			0x2f,0x29,0x7e,0x7b,0x49,0x26,0xd9,0x06,0xce,0x0d,0x09,0xd8,0x41,0x28,0x40,0x6a,
			0xb3,0x3d,0x7d,0xa0,0xf8,0xa1,0xd4,0xd2,0xf6,0x66,0x56,0x86,0x86,0xc3,0x94,0xd1,
			0x39,0xb0,0xe5,0xe9,0x93,0x37,0x75,0x8d,0xe8,0x59,0x10,0xa5,0xfa,0x25,0xca,0x2a,
			0xa6,0xd8,0xfb,0x1c,0x77,0x72,0x44,0xe7,0xd9,0x8d,0xe4,0xc7,0x9b,0xbd,0x42,0x6a,
			0x5e,0x6f,0x65,0x7e,0x37,0x47,0x7e,0x01,0x24,0x74,0x32,0xf8,0x37,0x97,0xfb,0xf3,
			0x1b,0x50,0xd0,0x2b,0x83,0xf6,0x9d,0xed,0x26,0xd4,0x94,0x5b,0x2b,0xc3,0xf8,0x6e),
	},
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256,
			.salt_len = 32,
			.salt = chunk_from_chars(
				0x3e,0x07,0xad,0xe7,0x2a,0x3f,0x52,0x53,0x0f,0x53,0x13,0x5a,0x5d,0x7d,0x93,0x21,
				0x74,0x35,0xba,0x00,0x1e,0xa5,0x5a,0x8f,0x5d,0x5d,0x13,0x04,0x68,0x48,0x74,0xbc),
		},
		.m = chunk_from_chars(
			0x60,0x40,0x2d,0xed,0x89,0xd0,0x97,0x9a,0xfb,0x49,0xf8,0x50,0x8e,0xb9,0x78,0xa8,
			0x41,0xab,0xc2,0xae,0xc5,0x9c,0xac,0xef,0x40,0xb3,0x1a,0xd3,0x4b,0xac,0x1f,0x2d,
			0x3c,0x16,0x66,0x11,0xab,0xbe,0xd1,0xe6,0x2f,0x6b,0x5f,0xbb,0x69,0xcb,0x53,0xdf,
			0x44,0xae,0x93,0xab,0x7a,0x72,0x4e,0xa3,0x5b,0xbe,0xe1,0xbe,0xca,0x74,0xfc,0x01,
			0x88,0xe0,0x00,0x52,0xb5,0x36,0xac,0x8c,0x93,0x3b,0xf9,0xcf,0x8e,0x42,0x42,0x1a,
			0x79,0x5a,0xa8,0x1b,0x1b,0xc6,0xb5,0x45,0xea,0xad,0x40,0x24,0x16,0x13,0x90,0xed,
			0xc9,0x08,0xc4,0x5a,0xae,0x1f,0x71,0xb4,0xb0,0x22,0x8e,0x31,0x04,0x04,0x8d,0x81,
			0x69,0x17,0xcb,0xa4,0xae,0x7f,0x2a,0xfe,0x75,0xe7,0xfc,0xad,0x38,0x73,0x24,0x1a),
		.s = chunk_from_chars(
			0x5f,0x18,0x30,0x09,0x70,0x8b,0x37,0x96,0x37,0xda,0xc2,0xb1,0x42,0x93,0x70,0x9a,
			0xa6,0xd7,0xe8,0x6c,0x26,0x7a,0x0b,0x69,0x0a,0x3c,0x27,0x50,0x31,0x13,0x98,0x91,
			0x26,0x7c,0x64,0xe5,0xed,0xec,0xdf,0xf1,0x4c,0x2c,0xc2,0xf2,0xd9,0x85,0xb6,0x2f,
			0x90,0x0a,0xee,0x6e,0x04,0xca,0x51,0xa7,0x0a,0x5f,0x94,0x64,0x63,0x69,0x1c,0xf1,
			0x6c,0x2d,0x45,0x54,0x7c,0x53,0x74,0xf1,0x5b,0xdb,0x88,0x81,0x64,0x1d,0x30,0x40,
			0xef,0x57,0x80,0x75,0x32,0xcf,0x5b,0x2c,0xed,0x07,0x62,0x3d,0x0f,0x63,0x8b,0x39,
			0xeb,0xc2,0xf2,0xce,0x28,0x3e,0xea,0x22,0x47,0xe1,0xdf,0x3a,0xf5,0x43,0x05,0x54,
			0xd1,0xd4,0xb8,0x8b,0x7b,0x21,0x62,0x29,0x93,0x41,0x99,0x71,0xb7,0xd0,0xd5,0x44,
			0x91,0x22,0xa1,0x0f,0xc3,0x1b,0x2d,0xdc,0xc5,0x3f,0xf7,0x51,0xff,0x4b,0xf4,0xd3,
			0x36,0xfa,0xc6,0x67,0xb6,0x46,0x78,0x02,0x72,0xdb,0x89,0xa3,0xea,0x42,0x26,0xaf,
			0xa2,0x08,0x77,0xbf,0xb8,0x6b,0xa3,0xff,0x42,0x04,0xe5,0xcd,0x56,0xe1,0x3a,0x1d,
			0xc9,0xd5,0x3f,0x5c,0x94,0x65,0xb9,0x7a,0x18,0x2b,0x2b,0xf6,0x71,0x51,0x2e,0xf8,
			0x9e,0x6c,0x39,0x69,0xf9,0x73,0x07,0xa3,0xe4,0xbe,0xba,0x39,0xa7,0x8e,0x0a,0xd1,
			0xbb,0x97,0x99,0xcd,0xa9,0x29,0x76,0xca,0x39,0xd9,0x9d,0xb4,0xac,0x14,0x9c,0x84,
			0xbb,0x9b,0xc8,0x99,0x7e,0x8d,0x5e,0x05,0x6d,0x67,0xca,0x23,0xfe,0x4b,0xe2,0x8e,
			0x66,0xc4,0xbc,0x00,0xa2,0x5d,0x65,0xbb,0x9d,0x7d,0x62,0x3f,0xea,0x2d,0x3b,0x9c,
			0xf8,0x59,0xdf,0xd9,0xef,0xa9,0xe5,0x22,0x68,0xbf,0xa2,0x97,0xaf,0xb1,0xcc,0x28,
			0x83,0xdb,0x0c,0x9c,0x42,0xfc,0x04,0x18,0x0e,0x2e,0xc6,0xf4,0x96,0x57,0xc7,0x00,
			0x8e,0x40,0x25,0x06,0x1f,0x89,0x68,0x86,0x61,0x38,0x95,0xa3,0x5b,0xc2,0xd3,0x65,
			0x5a,0x8f,0x50,0xa9,0xfc,0xa2,0xac,0x64,0x8f,0x35,0x2e,0xb0,0x6b,0xfb,0xa2,0xfc,
			0x34,0x0a,0xae,0xea,0xd4,0xa8,0x45,0x7c,0x65,0xe2,0xe8,0xfd,0xba,0x56,0x8c,0x60,
			0xa6,0xd8,0xd3,0x81,0xf5,0xd9,0xca,0xa3,0x01,0x27,0x77,0x1f,0x4a,0x94,0xfd,0xb8,
			0xcd,0xe7,0xbe,0x4f,0xa7,0xb4,0xf8,0x9f,0xe3,0x79,0xdd,0x3e,0x1c,0xa6,0x6a,0xe1,
			0xfd,0xd6,0x3b,0xeb,0xdc,0x00,0x15,0x44,0x8e,0x61,0xef,0x16,0x66,0x59,0x4b,0x8f),
	},
	{
		.params = { .hash = HASH_SHA384, .mgf1_hash = HASH_SHA384,
			.salt_len = 48,
			.salt = chunk_from_chars(
				0x61,0xa7,0x62,0xf8,0x96,0x8d,0x5f,0x36,0x7e,0x2d,0xbc,0xac,0xb4,0x02,0x16,0x53,
				0xdc,0x75,0x43,0x7d,0x90,0x00,0xe3,0x16,0x9d,0x94,0x37,0x29,0x70,0x38,0x37,0xa5,
				0xcb,0xf4,0xde,0x62,0xbd,0xed,0xc9,0x5f,0xd0,0xd1,0x00,0x4e,0x84,0x75,0x14,0x52),
		},
		.m = chunk_from_chars(
			0x92,0x21,0xf0,0xfe,0x91,0x15,0x84,0x35,0x54,0xd5,0x68,0x5d,0x9f,0xe6,0x9d,0xc4,
			0x9e,0x95,0xce,0xb5,0x79,0x39,0x86,0xe4,0x28,0xb8,0xa1,0x0b,0x89,0x4c,0x01,0xd6,
			0xaf,0x87,0x82,0xfd,0x7d,0x95,0x2f,0xaf,0x74,0xc2,0xb6,0x37,0xca,0x3b,0x19,0xda,
			0xbc,0x19,0xa7,0xfe,0x25,0x9b,0x2b,0x92,0x4e,0xb3,0x63,0xa9,0x08,0xc5,0xb3,0x68,
			0xf8,0xab,0x1b,0x23,0x33,0xfc,0x67,0xc3,0x0b,0x8e,0xa5,0x6b,0x28,0x39,0xdc,0x5b,
			0xda,0xde,0xfb,0x14,0xad,0xa8,0x10,0xbc,0x3e,0x92,0xba,0xc5,0x4e,0x2a,0xe1,0xca,
			0x15,0x94,0xa4,0xb9,0xd8,0xd1,0x93,0x37,0xbe,0x42,0x1f,0x40,0xe0,0x67,0x4e,0x0e,
			0x9f,0xed,0xb4,0x3d,0x3a,0xe8,0x9e,0x2c,0xa0,0x5d,0x90,0xa6,0x82,0x03,0xf2,0xc2),
		.s = chunk_from_chars(
			0x96,0x87,0x11,0x5b,0xe4,0x78,0xe4,0xb6,0x42,0xcd,0x36,0x93,0x92,0xb9,0xdd,0x0f,
			0x35,0x76,0xe7,0x04,0xaf,0x72,0x18,0xb1,0xf9,0x4d,0x7f,0x8f,0xe7,0xf0,0x70,0x73,
			0xe3,0xe8,0xe1,0x18,0x6f,0xa7,0x68,0x97,0x7d,0x6b,0x51,0x4e,0x51,0x34,0x59,0xf2,
			0x37,0x3d,0xf6,0xec,0x52,0xe3,0xde,0x9b,0xd8,0x3f,0xcc,0x5c,0xc3,0xe6,0xb9,0x7f,
			0x8b,0x3f,0xb5,0x34,0x16,0x3c,0x64,0xf5,0x26,0x76,0x20,0x70,0x0e,0x9d,0x8c,0x52,
			0xb3,0xdf,0x61,0xa7,0xc3,0x74,0x8e,0xf1,0x59,0xd6,0xb3,0x90,0x89,0x5a,0xfa,0x3a,
			0xf5,0x91,0x09,0xa5,0x47,0x8d,0x01,0x6d,0x96,0xc4,0x9f,0x68,0xdf,0xc7,0x35,0xba,
			0x2a,0xaf,0xd5,0x01,0x2c,0x13,0x51,0x5e,0xd6,0x64,0x4f,0x0d,0x41,0x09,0xc4,0x55,
			0x56,0xe1,0x4a,0x38,0x21,0xe1,0xaa,0x24,0xbe,0xb8,0xa8,0x1a,0x48,0xda,0x27,0xf1,
			0x31,0xde,0x84,0xf7,0xba,0x51,0x58,0x1d,0x81,0xb8,0xff,0x31,0xba,0x92,0xb8,0xa1,
			0xfd,0xe8,0x67,0xf0,0x7e,0x32,0xe6,0xc2,0x70,0x92,0x53,0x44,0x81,0x74,0xdd,0x31,
			0x32,0x4d,0xbc,0x32,0xb0,0x5f,0x07,0x58,0x7f,0x76,0xa9,0x99,0x7d,0xec,0xb8,0x0f,
			0x38,0xd8,0xc1,0x3d,0x0f,0x6e,0xb3,0xc1,0x0e,0x3d,0x96,0xa2,0x29,0x3f,0x74,0x64,
			0xf1,0xe0,0x46,0x02,0xef,0x6e,0x84,0xc2,0xd0,0x24,0x5d,0x7d,0xb2,0x56,0xa6,0x7d,
			0x13,0x2a,0x47,0xca,0xe9,0xab,0xe0,0x6b,0x61,0xa8,0x96,0x8f,0x50,0xa1,0x74,0x99,
			0x95,0xdc,0x15,0xef,0x0d,0xcb,0x1d,0x5f,0x59,0x59,0xe4,0xd4,0x54,0xc8,0x54,0x7b,
			0xbb,0x4d,0x19,0x56,0x98,0xf4,0x84,0x61,0x7b,0xfd,0x12,0x2a,0xca,0xae,0x2d,0x0e,
			0x8c,0x76,0xd2,0x8b,0x24,0x00,0x5a,0xb0,0x3c,0xaa,0x78,0x1e,0xa9,0x7b,0x1c,0x4d,
			0x93,0x96,0xa1,0x6f,0x79,0x98,0xee,0xe7,0xdd,0xd9,0xde,0x4c,0xab,0xe5,0x70,0x32,
			0xd9,0x43,0x8a,0x5d,0x99,0xc6,0xb3,0x4a,0x95,0x61,0x22,0x35,0x02,0x63,0xc7,0xe9,
			0x98,0xbc,0x61,0xde,0xc9,0x13,0x81,0x01,0x2e,0x68,0x6d,0x07,0x9e,0x39,0xe9,0x6b,
			0x1e,0xa4,0xbf,0xdb,0x7c,0xdf,0x63,0x0d,0xdb,0x42,0x2c,0x6b,0x58,0x0e,0x55,0x06,
			0xc9,0xcc,0x3d,0x6c,0x10,0x0f,0x20,0x41,0xd1,0x7c,0xea,0xaa,0xa5,0x45,0x89,0x24,
			0x9f,0x04,0xa1,0x37,0x0f,0xfa,0x3b,0xf3,0xff,0x1a,0xde,0xb8,0x90,0x68,0x86,0x98),
	},
	{
		.params = { .hash = HASH_SHA384, .mgf1_hash = HASH_SHA384,
			.salt_len = 48,
			.salt = chunk_from_chars(
				0x61,0xa7,0x62,0xf8,0x96,0x8d,0x5f,0x36,0x7e,0x2d,0xbc,0xac,0xb4,0x02,0x16,0x53,
				0xdc,0x75,0x43,0x7d,0x90,0x00,0xe3,0x16,0x9d,0x94,0x37,0x29,0x70,0x38,0x37,0xa5,
				0xcb,0xf4,0xde,0x62,0xbd,0xed,0xc9,0x5f,0xd0,0xd1,0x00,0x4e,0x84,0x75,0x14,0x52),
		},
		.m = chunk_from_chars(
			0x75,0x2a,0x99,0x16,0xf4,0x49,0xae,0xbf,0x81,0x4c,0xe5,0x9c,0xa6,0xe8,0x2f,0xa8,
			0x03,0x8e,0x46,0x85,0x41,0x92,0x41,0xc1,0x48,0x8c,0x66,0x59,0xb2,0xff,0x3f,0x7b,
			0x7f,0x38,0xf0,0x90,0x0a,0x79,0xc7,0x7a,0x3b,0x57,0x15,0x1a,0xff,0x61,0x3c,0x16,
			0xf5,0x02,0x0a,0xd9,0x6b,0xa9,0x45,0xdb,0x88,0x26,0x87,0x22,0xca,0x58,0x4c,0x09,
			0xb4,0x05,0x4a,0x40,0xc0,0x09,0x01,0x14,0x9b,0xb3,0x92,0xf0,0x91,0x6c,0xd4,0x24,
			0x46,0x99,0xa5,0xe6,0xa8,0xc3,0x7e,0x96,0x21,0xf5,0x4b,0x47,0x11,0x66,0x79,0x7a,
			0x7b,0x58,0x50,0x2c,0xff,0x40,0x83,0x14,0x08,0x27,0x05,0x26,0x46,0x50,0x1f,0x5b,
			0x5f,0x1b,0xc0,0xb4,0xe1,0x29,0x14,0x7d,0x7c,0xc1,0x57,0xcf,0x6e,0x73,0xec,0x58),
		.s = chunk_from_chars(
			0x66,0x46,0xa8,0x8e,0xe4,0xb8,0x45,0xda,0x49,0x31,0x27,0x4c,0x23,0x84,0x0d,0xad,
			0xa6,0x14,0x5f,0xe0,0xaf,0x95,0x48,0x29,0xd1,0xd5,0x66,0x61,0x54,0x6a,0x25,0xe4,
			0x63,0x16,0xe2,0x16,0xbb,0x6b,0x94,0x46,0xb3,0x68,0x88,0x4b,0xa1,0x49,0x69,0xa6,
			0xf6,0x8c,0xcb,0xc1,0xcf,0x5b,0x4e,0x7a,0x6d,0x3a,0xab,0xec,0x67,0xf6,0x49,0x63,
			0xf6,0x3b,0x08,0x8f,0xa8,0x17,0xc8,0x55,0xd7,0x76,0xdd,0xca,0xda,0x57,0xe5,0xda,
			0xa5,0x0f,0xc1,0xc8,0x77,0x38,0x9c,0x3c,0xb9,0xd9,0x90,0x95,0xa8,0x69,0xa9,0x63,
			0xbc,0x91,0xec,0x24,0xb2,0x42,0x2e,0xf6,0xb8,0xdd,0x18,0xfd,0x20,0xd2,0xb2,0x15,
			0xfe,0xe6,0xe9,0x8c,0xda,0x41,0x5a,0xe4,0x4d,0x2d,0x26,0x16,0xfe,0x17,0x08,0x29,
			0x2a,0x3e,0xf5,0x0a,0x07,0x51,0x70,0xb3,0xa7,0xeb,0xab,0x02,0x91,0x8a,0xb0,0x30,
			0x17,0x94,0xc1,0x7f,0xb3,0x5e,0x20,0x38,0xf3,0x69,0xd9,0x4d,0xd4,0x95,0x69,0xc0,
			0x66,0xf7,0xc3,0x92,0x88,0x9d,0xc4,0xb8,0x78,0xc5,0x0c,0x7e,0x52,0x58,0x6b,0x50,
			0x81,0x11,0x4d,0x20,0x23,0x38,0xd2,0x33,0x04,0xf1,0x6f,0x91,0x2d,0x51,0x9a,0x9a,
			0xd2,0x1b,0xaf,0xf0,0xe3,0xd2,0x17,0x61,0xf3,0x73,0xd0,0x84,0x21,0xe1,0x01,0x08,
			0xa9,0x83,0x04,0x8f,0xcb,0x90,0xeb,0x2a,0xdc,0x7c,0x7f,0x12,0xff,0xa1,0x57,0x1b,
			0x09,0x1c,0x78,0x1b,0x25,0x5a,0x77,0xa8,0x80,0xe9,0x79,0x75,0xf1,0x4f,0x42,0xba,
			0xf5,0xaa,0x28,0x5e,0xcc,0x14,0x21,0x57,0xc3,0xe1,0xad,0xdd,0x6a,0xa0,0xc0,0x92,
			0x53,0xa1,0x1c,0x59,0x14,0x4a,0xbd,0x3b,0x1e,0x21,0x2d,0x89,0xe2,0x7e,0xd9,0x6f,
			0xb7,0x57,0x56,0xaf,0xc2,0x0e,0xc6,0x74,0x23,0xb1,0x51,0x19,0x4c,0xb0,0xb0,0x64,
			0x8c,0x65,0x99,0x87,0xa5,0x58,0x3c,0xb7,0x75,0x77,0x79,0xd8,0xa3,0x9e,0x20,0x5e,
			0x71,0x01,0xa5,0x35,0x1c,0xe1,0xaf,0x2c,0x9c,0x6b,0x08,0x47,0xcc,0xa5,0x7a,0xf5,
			0x25,0x93,0x32,0x39,0x05,0xe3,0xd2,0x29,0x7c,0x0d,0x54,0x54,0x1a,0x01,0x25,0x62,
			0x16,0x40,0xfe,0x1d,0xee,0xf1,0x3e,0x75,0x9f,0x8f,0x6c,0x56,0xa2,0xec,0x2a,0x94,
			0x83,0x1a,0xc2,0xc6,0x14,0xb9,0x11,0xe7,0x9e,0xdd,0x54,0x2f,0xef,0x65,0x1f,0x5a,
			0x82,0x7f,0x48,0x05,0x75,0xae,0x22,0x0c,0x49,0x5f,0x2a,0x28,0x42,0xf9,0x9e,0xc4),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512,
			.salt_len = 62,
			.salt = chunk_from_chars(
				0x2d,0x0c,0x49,0xb2,0x07,0x89,0xf3,0x95,0x02,0xee,0xfd,0x09,0x2a,0x2b,0x6a,0x9b,
				0x27,0x57,0xc1,0x45,0x61,0x47,0x56,0x9a,0x68,0x5f,0xca,0x44,0x92,0xa8,0xd5,0xb0,
				0xe6,0x23,0x43,0x08,0x38,0x5d,0x3d,0x62,0x96,0x44,0xca,0x37,0xe3,0x39,0x96,0x16,
				0xc2,0x66,0xf1,0x99,0xb6,0x52,0x1a,0x99,0x87,0xb2,0xbe,0x9e,0xe7,0x83),
		},
		.m = chunk_from_chars(
			0x44,0x24,0x0c,0xe5,0x19,0xf0,0x02,0x39,0xbd,0x66,0xba,0x03,0xc8,0x4d,0x31,0x60,
			0xb1,0xce,0x39,0xe3,0x93,0x28,0x66,0xe5,0x31,0xa6,0x2b,0x1c,0x37,0xcf,0x41,0x70,
			0xc3,0xdc,0x48,0x09,0x23,0x6f,0xb1,0xad,0xe1,0x81,0xdb,0x49,0xfc,0x9c,0x7c,0xcd,
			0x79,0x4b,0x43,0x3d,0x1a,0xd0,0xbc,0x05,0x6e,0x14,0x73,0x8e,0x0a,0xe4,0x5c,0x0e,
			0x15,0x59,0x72,0xa4,0x0a,0x98,0x9f,0xa4,0xb9,0xbc,0xdc,0x30,0x8f,0x11,0x99,0x08,
			0x18,0x83,0x5f,0xa2,0xc2,0x56,0xb4,0x7e,0xe4,0x17,0x3f,0xb4,0xfe,0xd2,0x2c,0xcf,
			0x43,0x85,0xd2,0xdd,0x54,0xd5,0x93,0xc7,0x4f,0x00,0x04,0xdf,0x08,0x13,0x4e,0xb8,
			0x96,0x5d,0xd5,0x3a,0x12,0x23,0x17,0xf5,0x9b,0x95,0xd6,0xb6,0x9d,0x01,0x79,0x58),
		.s = chunk_from_chars(
			0x8f,0x47,0xab,0xc2,0x32,0x6e,0x22,0xcf,0x62,0x40,0x45,0x08,0xb4,0x42,0xe8,0x1a,
			0xd4,0x5a,0xff,0xf7,0x27,0x40,0x96,0xb9,0xa1,0x3e,0x47,0x8c,0xdd,0x0a,0x72,0xf9,
			0x9a,0x76,0xbf,0x51,0x7f,0x1b,0xb0,0xf8,0x72,0xa5,0x23,0xd8,0xc5,0x88,0xd4,0x40,
			0x25,0x69,0xe9,0x48,0xfd,0x6a,0x10,0x8a,0xe1,0xa4,0x5c,0x65,0x83,0x08,0x28,0xa1,
			0x0e,0x94,0xd4,0x32,0x76,0x53,0x14,0xba,0x82,0xea,0xd3,0x10,0xfc,0x87,0xac,0x99,
			0xa5,0xb3,0x9f,0x30,0xab,0x88,0x20,0xbf,0x69,0xe6,0x93,0x4a,0x9c,0x1c,0x91,0x5c,
			0x19,0xf3,0x6e,0xa7,0x71,0x7e,0xaf,0xf7,0xaf,0x67,0xb4,0x99,0x13,0x15,0xb1,0x87,
			0x3b,0xa9,0x29,0xbe,0xdf,0x18,0xa9,0x75,0xbe,0x80,0x8e,0x7a,0xa1,0x4a,0x67,0x26,
			0x12,0x6c,0x79,0xcc,0x93,0xf6,0x95,0x41,0xc5,0xce,0xfd,0xeb,0x5b,0x67,0xec,0x27,
			0x9d,0x8f,0x5a,0x44,0x65,0x83,0xe4,0xb4,0xfa,0xed,0x16,0x85,0x14,0x0e,0xe4,0xb3,
			0xb7,0x57,0xc8,0xff,0x4a,0x1e,0xf9,0xcd,0x76,0xa8,0x8e,0x05,0x31,0x9e,0xe6,0x20,
			0x03,0xd2,0xd7,0x72,0x90,0xc9,0x4c,0x57,0x9b,0x0c,0xa2,0xab,0x0d,0xeb,0x31,0x76,
			0xef,0x10,0xa3,0xfd,0xb8,0x5c,0x80,0xff,0xbc,0x9e,0x2a,0x66,0x5a,0x23,0x74,0x4f,
			0xc8,0x36,0xf9,0xa9,0xa1,0x03,0xcd,0x9f,0xb7,0x56,0x95,0x23,0x56,0xa2,0xf1,0xac,
			0xdd,0x68,0xa6,0x45,0xe2,0x01,0x79,0x00,0x65,0x58,0xb5,0xd4,0xd0,0xb9,0xb0,0xbd,
			0x3a,0xdf,0x5e,0x29,0x0f,0x49,0xda,0xe6,0x0b,0x9d,0x19,0x92,0x09,0x53,0xea,0x8b,
			0xb2,0x37,0xd5,0xb3,0xdc,0xfe,0x14,0x9a,0x60,0xf1,0x2a,0x4e,0xe3,0xa8,0x89,0xb3,
			0x3b,0xcd,0x3a,0x3b,0x75,0x3d,0x61,0x07,0x57,0xcb,0xcd,0x09,0x3d,0xd5,0xa7,0x34,
			0x25,0x53,0x33,0x68,0x96,0x95,0xab,0x63,0x69,0x63,0xe3,0xd2,0x15,0xa8,0xe7,0x7f,
			0xf3,0x19,0x73,0x71,0x8a,0x49,0x44,0xa1,0xe9,0xe4,0x4f,0x45,0x75,0x4d,0x39,0xf6,
			0xfa,0x43,0x1c,0x53,0xf9,0xa2,0xef,0x36,0xe1,0x6a,0x5f,0x70,0x63,0x6e,0xb5,0xfb,
			0xa5,0x4e,0x15,0xc2,0x0a,0x71,0x4f,0x28,0x09,0xa7,0xcf,0xf4,0xb8,0xdc,0x11,0x65,
			0xf8,0x36,0x60,0x7e,0xb5,0xa5,0xa3,0xbb,0x0c,0x45,0x67,0xee,0xe2,0x69,0x41,0xfe,
			0xf4,0x6f,0xb4,0x1e,0x73,0xb5,0x65,0xc0,0xcf,0x8c,0x72,0xe4,0x04,0x22,0x12,0x64),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512,
			.salt_len = 62,
			.salt = chunk_from_chars(
				0x2d,0x0c,0x49,0xb2,0x07,0x89,0xf3,0x95,0x02,0xee,0xfd,0x09,0x2a,0x2b,0x6a,0x9b,
				0x27,0x57,0xc1,0x45,0x61,0x47,0x56,0x9a,0x68,0x5f,0xca,0x44,0x92,0xa8,0xd5,0xb0,
				0xe6,0x23,0x43,0x08,0x38,0x5d,0x3d,0x62,0x96,0x44,0xca,0x37,0xe3,0x39,0x96,0x16,
				0xc2,0x66,0xf1,0x99,0xb6,0x52,0x1a,0x99,0x87,0xb2,0xbe,0x9e,0xe7,0x83),
		},
		.m = chunk_from_chars(
			0x06,0xd5,0x53,0x4b,0x77,0x69,0x25,0x6e,0x8c,0xf6,0x5c,0x6c,0xe5,0x2a,0x3e,0x86,
			0x96,0x5a,0x1f,0xd1,0x2c,0x75,0x82,0xd2,0xeb,0x36,0x82,0x4a,0x5a,0x9d,0x70,0x53,
			0x02,0x9f,0xbe,0xac,0x72,0x1d,0x1b,0x52,0x86,0x13,0xe0,0x50,0xe9,0x12,0xab,0xd7,
			0xd9,0xf0,0x49,0x91,0x2a,0xbe,0xda,0x33,0x8e,0xfa,0x2f,0x52,0x13,0x06,0x77,0x77,
			0xed,0xd9,0x1b,0x75,0x76,0xf5,0xe6,0xfa,0x73,0x98,0x69,0x65,0x99,0x37,0x9e,0xd7,
			0x50,0x28,0xcb,0x8d,0xb6,0x9f,0xa9,0x6d,0xe7,0xdb,0xc6,0xde,0x7c,0xa1,0x28,0xdd,
			0x51,0xea,0x33,0x4e,0x8c,0xd9,0xcd,0x8f,0xda,0xef,0xbf,0x53,0xfc,0x82,0x5e,0xae,
			0x83,0x6b,0x6c,0x6c,0xd7,0x00,0x39,0xa7,0x7e,0x42,0x0d,0x99,0x9b,0x57,0xca,0xae),
		.s = chunk_from_chars(
			0x91,0x3f,0xc1,0x18,0xd5,0xac,0x1e,0xdf,0xfb,0x4b,0x8f,0xcf,0xa4,0xe8,0x59,0x86,
			0xb4,0x62,0x31,0xce,0xf3,0xda,0xd9,0x11,0xd5,0xe9,0x53,0x4c,0xc8,0x82,0x61,0xf6,
			0xb6,0x96,0x9b,0x75,0xa3,0xf2,0x5d,0x83,0xec,0xe7,0xec,0x20,0x34,0xb0,0x1d,0x3b,
			0x2b,0xe6,0xc5,0xbd,0x95,0x8c,0xc4,0xaf,0xcd,0x44,0x83,0x9e,0x39,0x53,0xf0,0x1e,
			0x4a,0x15,0xea,0x5e,0xf6,0xe1,0xb4,0xb0,0xe8,0xae,0x90,0xbd,0xfd,0x40,0x41,0x99,
			0xe8,0xf8,0x65,0x47,0xf6,0x7f,0xf6,0xb8,0x4f,0x21,0x62,0xc4,0x31,0x1c,0xc9,0xee,
			0xe0,0x6b,0xfb,0x2f,0xe4,0x61,0x98,0xaf,0xb9,0x74,0x5d,0x9c,0x44,0x38,0x33,0xbf,
			0x23,0x87,0xeb,0x92,0x40,0x6a,0x63,0x39,0x52,0x13,0x96,0xf2,0xcb,0xda,0x55,0xd9,
			0x8f,0xe6,0x40,0x74,0xd2,0xf2,0xe2,0x7b,0x8b,0xc6,0xa7,0x9b,0xe3,0xd1,0xcc,0x56,
			0x88,0x69,0xb0,0xb5,0x0f,0xcb,0xf7,0x02,0xb0,0x83,0x16,0x68,0xfb,0xfd,0xed,0xc2,
			0xd1,0xb5,0x49,0x1e,0x8e,0xc6,0x23,0xed,0xeb,0x60,0xac,0x87,0x0e,0x6e,0x8d,0x05,
			0x85,0x93,0xfb,0xbc,0x93,0x8f,0xbf,0x74,0x17,0x00,0xef,0xc2,0xb2,0x46,0x7e,0x7e,
			0xb2,0x54,0xae,0x00,0x85,0x09,0xe9,0x16,0x07,0xf8,0xe5,0x0a,0xa1,0x6a,0x4e,0x85,
			0x1a,0xbc,0xa7,0xc8,0xd2,0x0c,0x6f,0xf6,0x1c,0xfe,0xe6,0xc1,0xfb,0x67,0x60,0x98,
			0xe5,0xcd,0xf1,0x27,0xc9,0xb7,0x95,0x38,0xfd,0x1e,0x6c,0x01,0x41,0x61,0x05,0x4c,
			0xaf,0x43,0xb7,0x34,0xfa,0x69,0xfe,0x06,0xa0,0x0d,0x76,0xf7,0x10,0xac,0xc1,0x98,
			0xf3,0xda,0x90,0x6a,0x7d,0x2e,0x73,0xa2,0xca,0x88,0x25,0x26,0xcc,0x35,0x4d,0xd7,
			0x63,0x0a,0x30,0x3d,0x8f,0x32,0xc6,0x55,0xb5,0xb3,0x3c,0xf7,0x88,0x59,0xbe,0xea,
			0xba,0x3f,0x9a,0xe0,0x52,0xc8,0xd7,0x47,0x1c,0xd2,0xbd,0x9e,0xdf,0x42,0xfd,0x8f,
			0x70,0xc3,0xb0,0xaa,0x79,0xc0,0x76,0x92,0x80,0x68,0xca,0x97,0x70,0x95,0x9a,0xfa,
			0x63,0x2c,0xa6,0xaa,0xba,0x66,0x79,0xe4,0x5d,0x68,0x88,0xc5,0x01,0x25,0xa7,0x3b,
			0x9d,0xeb,0x00,0xd4,0x2a,0x12,0x5f,0x25,0xdf,0x54,0x34,0xbe,0xff,0x0d,0x5b,0x0e,
			0xe1,0x3a,0x16,0xb1,0x70,0x45,0xce,0xce,0x0f,0x2d,0xa7,0x57,0x7d,0x79,0xd7,0xcd,
			0x75,0xa4,0xb6,0xc5,0xbc,0x34,0x5f,0x46,0x0a,0x17,0x34,0x87,0xb5,0x1b,0xc6,0xa6),
	},
};

START_TEST(test_sign_pss_3072)
{
	chunk_t sig;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, SIGN_RSA_EMSA_PSS)) ||
		!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PRIVKEY_SIGN, SIGN_RSA_EMSA_PSS)))
	{
		warn("%N not supported, skip test", signature_scheme_names,
			 SIGN_RSA_EMSA_PSS);
		return;
	}
	if (!pss_salt_supported)
	{
		warn("static salt for %N not supported, skip test",
			 signature_scheme_names, SIGN_RSA_EMSA_PSS);
		return;
	}
	fail_unless(privpss_3072->sign(privpss_3072, SIGN_RSA_EMSA_PSS,
							  &pss_sign_3072[_i].params, pss_sign_3072[_i].m, &sig),
				"sign %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	ck_assert_chunk_eq(pss_sign_3072[_i].s, sig);
	fail_unless(pubpss_3072->verify(pubpss_3072, SIGN_RSA_EMSA_PSS,
							   &pss_sign_3072[_i].params, pss_sign_3072[_i].m, sig),
				"verify %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	free(sig.ptr);
}
END_TEST

/**
 * Test vectors from FIPS 186-4 (those with 0 salt are 3072-bit verify vectors)
 */
static struct {
	rsa_pss_params_t params;
	chunk_t n;
	chunk_t d;
	chunk_t e;
	chunk_t m;
	chunk_t s;
} pss_sign_ns[] = {
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256, },
		.n = chunk_from_chars(
			0xce,0x49,0x24,0xff,0x47,0x0f,0xb9,0x9d,0x17,0xf6,0x65,0x95,0x56,0x1a,0x74,0xde,
			0xd2,0x20,0x92,0xd1,0xdc,0x27,0x12,0x2a,0xe1,0x5c,0xa8,0xca,0xc4,0xbf,0xae,0x11,
			0xda,0xa9,0xe3,0x7a,0x94,0x14,0x30,0xdd,0x1b,0x81,0xaa,0xf4,0x72,0xf3,0x20,0x83,
			0x5e,0xe2,0xfe,0x74,0x4c,0x83,0xf1,0x32,0x08,0x82,0xa8,0xa0,0x23,0x16,0xce,0xb3,
			0x75,0xf5,0xc4,0x90,0x92,0x32,0xbb,0x2c,0x65,0x20,0xb2,0x49,0xc8,0x8b,0xe4,0xf4,
			0x7b,0x8b,0x86,0xfd,0xd9,0x36,0x78,0xc6,0x9e,0x64,0xf5,0x00,0x89,0xe9,0x07,0xa5,
			0x50,0x4f,0xdd,0x43,0xf0,0xca,0xd2,0x4a,0xaa,0x9e,0x31,0x7e,0xf2,0xec,0xad,0xe3,
			0xb5,0xc1,0xfd,0x31,0xf3,0xc3,0x27,0xd7,0x0a,0x0e,0x2d,0x48,0x67,0xe6,0xfe,0x3f,
			0x26,0x27,0x2e,0x8b,0x6a,0x3c,0xce,0x17,0x84,0x3e,0x35,0x9b,0x82,0xeb,0x7a,0x4c,
			0xad,0x8c,0x42,0x46,0x01,0x79,0xcb,0x6c,0x07,0xfa,0x25,0x2e,0xfa,0xec,0x42,0x8f,
			0xd5,0xca,0xe5,0x20,0x8b,0x29,0x8b,0x25,0x51,0x09,0x02,0x6e,0x21,0x27,0x24,0x24,
			0xec,0x0c,0x52,0xe1,0xe5,0xf7,0x2c,0x5a,0xb0,0x6f,0x5d,0x2a,0x05,0xe7,0x7c,0x19,
			0x3b,0x64,0x7e,0xc9,0x48,0xbb,0x84,0x4e,0x0c,0x2e,0xf1,0x30,0x7f,0x53,0xcb,0x80,
			0x0d,0x4f,0x55,0x52,0x3d,0x86,0x03,0x8b,0xb9,0xe2,0x10,0x99,0xa8,0x61,0xb6,0xb9,
			0xbc,0xc9,0x69,0xe5,0xdd,0xdb,0xdf,0x71,0x71,0xb3,0x7d,0x61,0x63,0x81,0xb7,0x8c,
			0x3b,0x22,0xef,0x66,0x51,0x0b,0x27,0x65,0xd9,0x61,0x75,0x56,0xb1,0x75,0x59,0x98,
			0x79,0xd8,0x55,0x81,0x00,0xad,0x90,0xb8,0x30,0xe8,0x7a,0xd4,0x60,0xa2,0x21,0x08,
			0xba,0xa5,0xed,0x0f,0x2b,0xa9,0xdf,0xc0,0x51,0x67,0xf8,0xab,0x61,0xfc,0x9f,0x8a,
			0xe0,0x16,0x03,0xf9,0xdd,0x5e,0x66,0xce,0x1e,0x64,0x2b,0x60,0x4b,0xca,0x92,0x94,
			0xb5,0x7f,0xb7,0xc0,0xd8,0x3f,0x05,0x4b,0xac,0xf4,0x45,0x4c,0x29,0x8a,0x27,0x2c,
			0x44,0xbc,0x71,0x8f,0x54,0x60,0x5b,0x91,0xe0,0xbf,0xaf,0xd7,0x72,0xae,0xba,0xf3,
			0x82,0x88,0x46,0xc9,0x30,0x18,0xf9,0x8e,0x31,0x57,0x08,0xd5,0x0b,0xe8,0x40,0x1e,
			0xb9,0xa8,0x77,0x8d,0xcb,0xd0,0xd6,0xdb,0x93,0x70,0x86,0x04,0x11,0xb0,0x04,0xcd,
			0x37,0xfb,0xb8,0xb5,0xdf,0x87,0xed,0xee,0x7a,0xae,0x94,0x9f,0xff,0x34,0x60,0x7b),
		.d = chunk_from_chars(
			0x25,0x8f,0x08,0x40,0x36,0xb7,0xff,0xda,0x1d,0x0a,0xa0,0x37,0x3a,0x50,0x01,0x1d,
			0xd9,0x76,0xb7,0xfd,0x0e,0xe4,0xb8,0x89,0x65,0x4b,0x04,0x4a,0xb2,0x41,0xfb,0x75,
			0x46,0x75,0x46,0x69,0x09,0x42,0x9b,0x1a,0xcb,0xa9,0xd9,0xc1,0xab,0xf2,0xe9,0xbb,
			0x49,0x4c,0xea,0x81,0xc4,0xba,0x10,0xdc,0xd1,0x03,0x6f,0x36,0xea,0x81,0xdc,0x24,
			0xce,0x98,0x3e,0x3a,0xe7,0xda,0x7c,0xf8,0x10,0xdd,0xc0,0x5c,0x96,0xf9,0xcc,0x3a,
			0x90,0x46,0xfd,0xf5,0x8c,0x99,0x02,0x17,0x2c,0x7e,0x53,0xa1,0xbc,0xed,0x1b,0x78,
			0x84,0xf7,0x28,0x13,0x3b,0xe9,0xb4,0xa9,0x11,0x02,0x3e,0x31,0x59,0xd5,0xf2,0x52,
			0xf4,0x07,0xa8,0x08,0x0c,0x88,0xf1,0x22,0xcf,0x4a,0x9e,0x53,0xf1,0x03,0xae,0xcb,
			0x41,0x2c,0xd4,0x4d,0x9d,0x53,0xc1,0x45,0x75,0x7b,0x14,0xeb,0x85,0xa5,0xb0,0xd7,
			0xf8,0xbe,0x88,0xc5,0x6b,0xb0,0x0e,0x73,0x57,0xd4,0x3d,0x6a,0x82,0x89,0x53,0xf9,
			0x31,0x24,0xd1,0xb3,0x9c,0x0c,0xc1,0x37,0xdf,0xf2,0x97,0x2a,0x40,0x2e,0xbf,0xe2,
			0x9e,0xb6,0x14,0xc6,0x57,0x8e,0x10,0x2c,0x61,0xa6,0x00,0x18,0x33,0x32,0x3d,0x4b,
			0x79,0xbe,0xe1,0x01,0xe7,0x6a,0x9c,0x59,0xa3,0x58,0x47,0x1b,0x62,0x25,0x68,0x85,
			0x84,0xfb,0xdd,0x79,0x0a,0x1e,0x38,0xa6,0x0a,0x5f,0x8b,0xf6,0x47,0xf7,0x37,0x46,
			0x80,0xaa,0x1d,0x6c,0xc0,0x37,0x2f,0xd1,0x2e,0xf2,0x33,0xbf,0x6b,0xf7,0x26,0xfa,
			0x4a,0xf4,0x5e,0x1e,0xad,0x9b,0x58,0xdf,0x08,0xf6,0x2a,0xa7,0x6f,0xe9,0xfd,0x9b,
			0xb1,0xa9,0x75,0xbb,0x1c,0x4d,0xdb,0x9b,0x00,0x54,0x53,0xf9,0x57,0xdf,0xe4,0x14,
			0x8d,0x26,0x44,0xc1,0xc4,0x90,0x87,0x74,0x31,0xb6,0x7e,0x97,0x5c,0x5e,0x02,0xb2,
			0xdc,0x40,0x8d,0xe0,0x9e,0x53,0x1c,0x05,0xc0,0x51,0x73,0x11,0xa5,0xcf,0xeb,0x41,
			0x65,0xb5,0xf4,0x40,0x60,0xbb,0x34,0x33,0xff,0xf6,0xee,0x8f,0x0a,0xd3,0xf5,0x59,
			0xb8,0x45,0x8f,0x20,0xcb,0xdc,0xa8,0x46,0x49,0xf0,0xc8,0xa3,0xb6,0x98,0x9f,0x67,
			0x6b,0xc0,0xfe,0x46,0x91,0x03,0x2d,0x2a,0x08,0x97,0x8f,0x90,0x53,0xab,0xf2,0x1c,
			0x1d,0x08,0x1f,0x8e,0xc3,0x27,0x35,0xdd,0x1f,0xf0,0x40,0x7c,0x33,0x02,0xbf,0x55,
			0xd1,0x67,0x19,0x7d,0xbe,0x92,0xc6,0x78,0x29,0x4d,0x5f,0x1f,0x83,0x2d,0xa5,0xbb),
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0x08,0x97,0xd4,0x0e,0x7c,0x0f,0x2d,0xfc,0x07,0xb0,0xc7,0xfd,0xda,0xf5,0xfd,0x8f,
			0xcc,0x6a,0xf9,0xc1,0xfd,0xc1,0x7b,0xeb,0xb9,0x23,0xd5,0x9c,0x9f,0xc4,0x3b,0xd4,
			0x02,0xba,0x39,0x73,0x8f,0x0f,0x85,0xf2,0x30,0x15,0xf7,0x51,0x31,0xf9,0xd6,0x50,
			0xa2,0x9b,0x55,0xe2,0xfc,0x9d,0x5d,0xdf,0x07,0xbb,0x8d,0xf9,0xfa,0x5a,0x80,0xf1,
			0xe4,0x63,0x4e,0x0b,0x4c,0x51,0x55,0xbf,0x14,0x89,0x39,0xb1,0xa4,0xea,0x29,0xe3,
			0x44,0xa6,0x64,0x29,0xc8,0x50,0xfc,0xde,0x73,0x36,0xda,0xd6,0x16,0xf0,0x03,0x93,
			0x78,0x39,0x1a,0xbc,0xfa,0xfe,0x25,0xca,0x7b,0xb5,0x94,0x05,0x7a,0xf0,0x7f,0xaf,
			0x7a,0x32,0x2f,0x7f,0xab,0x01,0xe0,0x51,0xc6,0x3c,0xc5,0x1b,0x39,0xaf,0x4d,0x23),
		.s = chunk_from_chars(
			0x8e,0xbe,0xd0,0x02,0xd4,0xf5,0x4d,0xe5,0x89,0x8a,0x5f,0x2e,0x69,0xd7,0x70,0xed,
			0x5a,0x5c,0xe1,0xd4,0x5a,0xd6,0xdd,0x9c,0xe5,0xf1,0x17,0x9d,0x1c,0x46,0xda,0xa4,
			0xd0,0x39,0x4e,0x21,0xa9,0x9d,0x80,0x33,0x58,0xd9,0xab,0xfd,0x23,0xbb,0x53,0x16,
			0x63,0x94,0xf9,0x97,0xb9,0x09,0xe6,0x75,0x66,0x20,0x66,0x32,0x4c,0xa1,0xf2,0xb7,
			0x31,0xde,0xba,0x17,0x05,0x25,0xc4,0xee,0x8f,0xa7,0x52,0xd2,0xd7,0xf2,0x01,0xb1,
			0x02,0x19,0x48,0x9f,0x57,0x84,0xe3,0x99,0xd9,0x16,0x30,0x2f,0xd4,0xb7,0xad,0xf8,
			0x84,0x90,0xdf,0x87,0x65,0x01,0xc4,0x67,0x42,0xa9,0x3c,0xfb,0x3a,0xaa,0xb9,0x60,
			0x2e,0x65,0xd7,0xe6,0x0d,0x7c,0x4c,0xea,0xdb,0x7e,0xb6,0x7e,0x42,0x1d,0x18,0x03,
			0x23,0xa6,0xd3,0x8f,0x38,0xb9,0xf9,0x99,0x21,0x3e,0xbf,0xcc,0xc7,0xe0,0x4f,0x06,
			0x0f,0xbd,0xb7,0xc2,0x10,0x20,0x65,0x22,0xb4,0x94,0xe1,0x99,0xe9,0x8c,0x6c,0x24,
			0xe4,0x57,0xf8,0x69,0x66,0x44,0xfd,0xca,0xeb,0xc1,0xb9,0x03,0x1c,0x81,0x83,0x22,
			0xc2,0x9d,0x13,0x5e,0x11,0x72,0xfa,0x0f,0xdf,0x7b,0xe1,0x00,0x7d,0xab,0xca,0xab,
			0x49,0x66,0x33,0x2e,0x7e,0xa1,0x45,0x6b,0x6c,0xe8,0x79,0xcd,0x91,0x0c,0x91,0x10,
			0x10,0x4f,0xc7,0xd3,0xdc,0xab,0x07,0x6f,0x2b,0xd1,0x82,0xbb,0x83,0x27,0xa8,0x63,
			0x25,0x45,0x70,0xcd,0xf2,0xab,0x38,0xe0,0xcd,0xa3,0x17,0x79,0xde,0xaa,0xd6,0x16,
			0xe3,0x43,0x7e,0xd6,0x59,0xd7,0x4e,0x5a,0x4e,0x04,0x5a,0x70,0x13,0x38,0x90,0xb8,
			0x1b,0xc4,0xf2,0x4a,0xb6,0xda,0x67,0xa2,0xee,0x0c,0xe1,0x5b,0xab,0xa3,0x37,0xd0,
			0x91,0xcb,0x5a,0x1c,0x44,0xda,0x69,0x0f,0x81,0x14,0x5b,0x02,0x52,0xa6,0x54,0x9b,
			0xbb,0x20,0xcd,0x5c,0xc4,0x7a,0xfe,0xc7,0x55,0xeb,0x37,0xfe,0xd5,0x5a,0x9a,0x33,
			0xd3,0x65,0x57,0x42,0x45,0x03,0xd8,0x05,0xa0,0xa1,0x20,0xb7,0x69,0x41,0xf4,0x15,
			0x0d,0x89,0x34,0x2d,0x7a,0x7f,0xa3,0xa2,0xb0,0x8c,0x51,0x5e,0x6f,0x68,0x42,0x9c,
			0xf7,0xaf,0xd1,0xa3,0xfc,0xe0,0xf4,0x28,0x35,0x1a,0x6f,0x9e,0xda,0x3a,0xb2,0x4a,
			0x7e,0xf5,0x91,0x99,0x4c,0x21,0xfb,0xf1,0x00,0x1f,0x99,0x23,0x9e,0x88,0x34,0x0f,
			0x9b,0x35,0x9e,0xc7,0x2e,0x8a,0x21,0x2a,0x19,0x20,0xe6,0xcf,0x99,0x3f,0xf8,0x48),
	},
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256, },
		.n = chunk_from_chars(
			0x9c,0x21,0x3e,0xa3,0x73,0x63,0x1f,0x57,0x2e,0x5e,0x46,0x22,0x5b,0x95,0xa0,0xf5,
			0xea,0x8a,0xb0,0xa5,0xec,0x70,0x90,0xa3,0xb0,0x18,0x1c,0x59,0x06,0xdc,0x22,0xfd,
			0x1b,0xd7,0x3d,0x11,0x47,0x12,0x42,0xa2,0xed,0x18,0x24,0xe6,0x01,0x01,0x7f,0x5b,
			0x53,0x54,0xb9,0x2f,0xdb,0x43,0xd4,0xda,0x00,0xa8,0x24,0x27,0xd0,0x53,0x66,0xa4,
			0xd5,0x52,0xc4,0x0d,0x69,0xd2,0x00,0x48,0x5d,0x5d,0x5d,0xb8,0x3c,0xf5,0x23,0xe6,
			0x1a,0x83,0x4b,0x40,0xcc,0xd4,0x01,0x08,0x7f,0xbb,0x93,0xd8,0x1d,0xbe,0xd7,0xb2,
			0xee,0x23,0x81,0xa1,0x24,0x1a,0xc6,0x8f,0x2a,0xfc,0x02,0x15,0x7e,0xe0,0xc7,0x3c,
			0xc6,0x6c,0x02,0xa6,0xc6,0xeb,0x2d,0xc3,0x51,0x45,0xae,0x55,0xd7,0x70,0x84,0x12,
			0xa3,0x41,0x0f,0x20,0x4c,0x49,0x23,0x76,0xd6,0x31,0x5c,0xed,0xf2,0x53,0xaf,0x91,
			0xf3,0x1d,0xca,0xb3,0x4f,0x72,0xc2,0x06,0xad,0x81,0xe8,0x00,0x50,0x98,0x64,0xfe,
			0x92,0x55,0xd0,0x46,0xac,0x25,0xdb,0xdf,0x49,0x54,0xd2,0x35,0x03,0x24,0x72,0x2e,
			0x73,0xc1,0xf2,0x5d,0x08,0x9f,0x87,0x54,0x2a,0xd9,0x07,0xfb,0x37,0xec,0xea,0xef,
			0xb3,0x30,0xf4,0x32,0x5e,0x97,0xf5,0xea,0xbc,0x04,0x09,0x6a,0x8f,0xab,0xba,0x97,
			0x85,0x89,0xe3,0x55,0x44,0x5d,0x95,0x43,0x27,0x4c,0x1c,0x38,0xff,0x84,0x9b,0xa8,
			0xc2,0x91,0x1f,0x07,0x03,0x06,0x34,0xc1,0x32,0xcc,0xaf,0x4e,0x4f,0x57,0xa5,0xad,
			0x92,0x44,0xf5,0xdf,0x0e,0xe2,0x5a,0xf8,0xef,0x2f,0xc7,0xb2,0x9f,0xdf,0x7c,0xc1,
			0x84,0x04,0xe2,0x0e,0xfa,0xaf,0xfa,0x45,0x1e,0xc4,0x1a,0xb8,0x38,0xd5,0x94,0x90,
			0x6f,0xf2,0xca,0xd5,0x2d,0xcb,0xd0,0xe9,0xa6,0x8e,0xf7,0xb0,0x6c,0x25,0x37,0x10,
			0xe1,0x31,0x8d,0x09,0xad,0x07,0x01,0x27,0x67,0xa8,0x91,0x24,0x17,0x7d,0xf5,0x0d,
			0x16,0x84,0x67,0x9e,0x14,0x30,0x68,0x89,0xd9,0xa7,0xac,0x51,0x43,0xd4,0x86,0x1b,
			0x7d,0x6a,0xe7,0x79,0x92,0xef,0xa7,0x3e,0x0a,0xba,0x9d,0xa0,0xad,0x9a,0x68,0x88,
			0xbc,0x80,0x4d,0xbd,0x07,0xbc,0x26,0xa8,0xa5,0xdf,0xbd,0x29,0x2a,0x0e,0xfa,0x96,
			0x86,0x7f,0xdb,0x92,0xe8,0x45,0xc3,0x6e,0x34,0x33,0xcf,0x29,0x2e,0x0e,0x31,0x66,
			0x24,0x80,0x25,0x7f,0xcf,0x46,0x6f,0x7f,0x65,0xd8,0x14,0xbb,0x3e,0x33,0x99,0x2f),
		.d = chunk_from_chars(
			0x3e,0xac,0x18,0x66,0xe2,0x35,0xd6,0x4b,0x9a,0x04,0xd7,0x3c,0x3f,0x49,0x70,0x7f,
			0x53,0x06,0xef,0x23,0x6f,0xcd,0xdb,0x7d,0xec,0x19,0x14,0x8a,0x4b,0xf3,0x48,0x28,
			0xc2,0x9b,0xc5,0x0e,0x75,0xa0,0xdc,0x38,0xd2,0x67,0x1b,0x50,0x28,0x59,0x61,0x8b,
			0xf3,0xf7,0xe5,0xf8,0x71,0x30,0x5a,0xd1,0xdf,0x49,0x72,0x75,0xb6,0x2d,0x36,0x03,
			0x1d,0xf6,0xc9,0x5c,0x9b,0x9d,0x68,0x73,0xa4,0x20,0x66,0x40,0x48,0x26,0x40,0x78,
			0x80,0x19,0x4c,0x61,0x13,0x2c,0x03,0x3c,0x76,0xac,0x16,0x9a,0xd4,0x01,0xcb,0x6e,
			0xd7,0x16,0xe7,0x02,0xbb,0xe3,0x16,0x11,0x72,0x9f,0x3a,0xca,0x1f,0x54,0x38,0xc6,
			0xb7,0x25,0xd9,0x01,0x17,0xca,0x53,0xe2,0x59,0xce,0x65,0xad,0x2f,0xbc,0x8a,0x1a,
			0x30,0xfa,0xa6,0x06,0xc3,0xb9,0x7f,0x1a,0xba,0xc5,0xa9,0x01,0x0f,0xa0,0x78,0x8e,
			0x9d,0x74,0x04,0xfd,0xcf,0xdc,0x55,0xf9,0xec,0x8e,0x91,0xaa,0x5f,0xdd,0x54,0xec,
			0x93,0x25,0xf0,0x21,0x48,0xc7,0xc5,0xdb,0x18,0xfe,0x52,0xed,0x62,0x75,0x8f,0xcd,
			0xb9,0x63,0x72,0x69,0x3f,0x99,0x8b,0x53,0x46,0x5a,0x97,0xca,0xcb,0x3b,0xbd,0x4d,
			0x0b,0xc3,0xd8,0x73,0x16,0x18,0xa2,0x78,0xe8,0x42,0x26,0x0e,0xaa,0xe8,0xdf,0x65,
			0xbe,0x8b,0xb9,0xf6,0xf0,0x85,0x22,0x0a,0x06,0x2a,0x11,0x77,0xf1,0x09,0x2a,0x9e,
			0xbd,0x46,0x71,0x35,0x0f,0xb0,0x5d,0x0c,0x89,0x3c,0x07,0xd8,0x36,0xe3,0x7e,0x73,
			0xc0,0x92,0xac,0x42,0x6e,0x63,0x0e,0x9b,0x6c,0xde,0xf3,0x6d,0x32,0x6c,0xd3,0x0e,
			0x5e,0xd8,0xff,0x37,0xce,0xd3,0x5c,0x4a,0xc4,0x7c,0xf0,0x12,0x61,0x61,0x6e,0xdd,
			0xb1,0x91,0x95,0x7e,0xaa,0x38,0xb2,0x60,0x0e,0x08,0xe7,0x84,0xb2,0x18,0x92,0x48,
			0xcf,0x40,0xf9,0xde,0xe6,0x5f,0x06,0x6a,0x8f,0x1e,0xe1,0x02,0x8d,0xe6,0x01,0xb6,
			0xbc,0x29,0x91,0x5b,0x9c,0xba,0xee,0xc2,0xdc,0x4f,0xe7,0xb1,0x3b,0x84,0x0c,0xf7,
			0x9a,0x4d,0x93,0xb0,0xc9,0xf4,0x7b,0x64,0x47,0xc3,0xf1,0xf8,0x32,0xdc,0x8c,0xf7,
			0xe5,0xf9,0x8f,0x68,0xf9,0x7c,0xc5,0xd9,0x93,0x46,0xb1,0x24,0x23,0x52,0xf7,0x4d,
			0x94,0xa7,0x27,0xa5,0x30,0x3b,0x99,0x67,0x0b,0xb8,0x34,0x19,0x25,0x5a,0xc5,0x12,
			0x70,0xbb,0xe5,0xd8,0x13,0x28,0x40,0x26,0x14,0xba,0xe1,0xc8,0x23,0x64,0x08,0xd9),
		.e = chunk_from_chars(0xef,0xab,0xb9),
		.m = chunk_from_chars(
			0x3a,0xbd,0x43,0xf1,0xd7,0x41,0xdd,0xf0,0xb7,0x52,0xdd,0x94,0x19,0x7a,0xb6,0x56,
			0x28,0x8f,0xf4,0x65,0xe5,0xdb,0xcc,0xec,0x61,0x79,0xf8,0x93,0x2f,0x02,0xc2,0x48,
			0x19,0x6a,0x5e,0x3f,0x12,0x24,0x7c,0x37,0x6c,0x8d,0x1e,0x00,0x4b,0x87,0xa7,0x3a,
			0x5a,0x64,0x35,0x5f,0xbc,0x07,0x31,0x04,0x46,0x45,0x25,0x5b,0xeb,0x91,0xcd,0xfd,
			0x77,0xd9,0x70,0xe6,0x81,0xab,0x70,0xc1,0x9a,0x9c,0x21,0x67,0xcb,0xea,0xd7,0xa2,
			0x3f,0x60,0x43,0x36,0x3c,0x7a,0xff,0x22,0x58,0xca,0x67,0x23,0xe9,0x9a,0xab,0x7a,
			0xbd,0xd3,0x22,0xef,0x0b,0x1e,0xe1,0x16,0xaa,0x48,0x8d,0xd1,0x81,0xeb,0x6d,0x16,
			0x3f,0x4b,0xc3,0xf2,0x4c,0x7e,0x5d,0xce,0xff,0xd0,0x67,0xf2,0x11,0x65,0x83,0x47),
		.s = chunk_from_chars(
			0x47,0x40,0x11,0x5f,0x25,0x1b,0x35,0xcf,0xe1,0x9d,0x38,0x16,0xcc,0x1f,0xbf,0x02,
			0xb9,0xa4,0xd9,0x47,0x0d,0xc6,0x2b,0x41,0xcd,0xb7,0x22,0x84,0xc3,0x0f,0xff,0xd3,
			0x4e,0x70,0x57,0x55,0x9a,0x73,0xb4,0x4a,0xc4,0x9f,0xa7,0x28,0x5f,0x3e,0x9a,0x68,
			0x0b,0xf1,0x94,0x95,0x62,0xa4,0x7f,0x01,0xbf,0xc5,0x5e,0xb7,0xbd,0xa9,0xe2,0x91,
			0xc9,0xa5,0xce,0x1a,0x5d,0x7b,0x61,0x95,0x90,0xc0,0xe1,0xbd,0x36,0x94,0x3f,0xa4,
			0x1b,0x9c,0xc9,0xb2,0xa3,0xa7,0x42,0xc2,0x94,0x2c,0x53,0xec,0x3e,0x45,0xc7,0x71,
			0x76,0xe4,0xbf,0x32,0xb7,0x10,0x1a,0xb4,0xb0,0x5a,0x99,0x6a,0x21,0xfe,0x12,0x92,
			0x18,0x41,0xf8,0xf0,0xfa,0xd1,0xe7,0x1b,0xde,0xff,0x5a,0x7f,0x53,0xe8,0x76,0x63,
			0x67,0x31,0x1a,0x98,0xb9,0xd8,0x61,0x8a,0x61,0x98,0xaa,0x16,0x62,0xe5,0x87,0x33,
			0x2c,0x24,0x28,0x3f,0x9e,0xcf,0x2b,0xb2,0xd8,0x25,0xf5,0x97,0xd1,0x82,0x56,0xce,
			0x9c,0x58,0x64,0xf1,0xff,0xca,0x37,0xc4,0xec,0x8d,0xf4,0xf9,0x45,0xcf,0x22,0xc9,
			0x74,0xf5,0xef,0x3d,0xbb,0x9f,0x17,0x0b,0x39,0x78,0xbd,0xdc,0xc9,0x1b,0x13,0xb9,
			0xe8,0x55,0x0b,0xfc,0xdb,0x74,0xac,0x73,0x49,0xd5,0x31,0xe1,0xe3,0x28,0x0f,0xc0,
			0x05,0xac,0x34,0xc0,0x36,0x08,0x99,0x3b,0x78,0x76,0xca,0xf2,0x7c,0x8b,0xd5,0xa9,
			0x73,0x06,0x29,0x20,0x82,0xf7,0x7d,0x91,0x44,0x00,0x5b,0xcd,0x10,0x46,0xf2,0x73,
			0x03,0x18,0x5b,0x6a,0x7c,0xac,0x53,0x9f,0x1e,0x05,0x80,0x5e,0xe8,0xd0,0x77,0x2f,
			0x37,0xac,0xc3,0xe7,0xfa,0x09,0x20,0x17,0x1c,0x19,0xec,0x92,0x66,0xc4,0xdd,0x6d,
			0x9e,0x51,0x76,0x6c,0x44,0x33,0xa2,0x83,0x1c,0x3e,0x5f,0xb7,0xd8,0x43,0x41,0x62,
			0x4e,0xa9,0xa2,0x5e,0x1a,0x2e,0xb2,0x7e,0xf2,0xc7,0xe8,0xde,0x49,0x1d,0x94,0xd8,
			0xa8,0xa3,0x19,0x19,0xbd,0xd7,0xcd,0xf6,0x07,0x3a,0x1d,0x88,0xf4,0x52,0xff,0x8d,
			0x15,0xe5,0x7c,0xf6,0x50,0x4a,0x81,0x84,0x3c,0x24,0x35,0x1c,0x61,0xa3,0x73,0xb7,
			0x3a,0xcd,0x84,0xc7,0x6d,0xe9,0x16,0xf2,0x90,0xe0,0xe9,0xbd,0xf5,0x1d,0x8f,0xfa,
			0x63,0xf8,0x67,0xaf,0x58,0x7b,0x5c,0x75,0x91,0x11,0x67,0x10,0x79,0xea,0x28,0x84,
			0x64,0x66,0xcc,0x33,0xdf,0x76,0x83,0x63,0x9d,0x01,0x83,0x37,0xeb,0xe1,0x3c,0xb8),
	},
	{
		.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256, },
		.n = chunk_from_chars(
			0xcd,0x61,0x41,0xe4,0xc8,0xdc,0x69,0x97,0xfc,0x65,0x30,0x0a,0x6e,0x2c,0x74,0x6f,
			0x39,0x10,0x22,0xf1,0x56,0x61,0x23,0x1a,0x83,0x2b,0xe9,0x66,0xaa,0x49,0x8e,0x67,
			0x87,0x77,0x76,0x69,0x44,0xdb,0x54,0xa5,0x99,0xcd,0x6d,0xdb,0xda,0xf7,0x53,0x3f,
			0x42,0x9e,0x6e,0x11,0x97,0xb7,0xeb,0x06,0x19,0x13,0xf5,0x0b,0x09,0xbe,0x04,0xab,
			0x70,0xb1,0xb7,0x02,0xfc,0xe9,0xad,0xa2,0x79,0xea,0x80,0x89,0x67,0x7a,0x37,0x70,
			0x1e,0x64,0x19,0x0f,0x24,0x3d,0xfa,0xbe,0x7c,0xc2,0x54,0xf0,0x8a,0x61,0x43,0xc9,
			0xc5,0x89,0xac,0x4a,0x90,0x88,0x1c,0x0d,0x2b,0x62,0xe9,0x80,0x13,0xe9,0x2d,0x04,
			0x9e,0xe9,0xca,0x11,0xa4,0x25,0xad,0x45,0x0e,0xc5,0xa6,0x99,0xae,0x17,0x67,0x2d,
			0x86,0xef,0xde,0x3f,0xbc,0x81,0x20,0x3f,0x2b,0x50,0x0e,0xf4,0x17,0x46,0xb9,0xe9,
			0xaf,0x26,0x42,0xb3,0x0c,0xbf,0x75,0xe7,0x88,0x9e,0x50,0x08,0x36,0xd6,0xdd,0x32,
			0xbc,0xc5,0xd8,0xb6,0x90,0x21,0xb7,0x64,0xa5,0x91,0xd6,0x85,0x07,0x76,0xcd,0xdf,
			0x0b,0x72,0x40,0xc7,0x5d,0xda,0x9e,0x2d,0x19,0x7f,0x2c,0xd9,0xc6,0x78,0x7b,0x16,
			0x44,0x5b,0x55,0xea,0xec,0xcd,0xfd,0xd1,0x7f,0xb7,0x2d,0x7f,0xe5,0x18,0x9a,0xa1,
			0xa2,0xa6,0xad,0x06,0xb4,0xf2,0xbf,0xc8,0xf6,0xc9,0x1c,0x3a,0x2a,0x80,0xa8,0x3a,
			0x31,0x13,0xe0,0xb4,0x20,0xb7,0x06,0x54,0xd6,0xa5,0x07,0x5d,0x38,0xd9,0xa1,0x2d,
			0x1e,0x0f,0xdd,0xd2,0xcb,0x23,0xb7,0xf3,0xef,0x94,0x99,0x91,0xbf,0x49,0x04,0x8d,
			0xc4,0x00,0x20,0xdc,0xf0,0x42,0xca,0xf0,0x08,0x83,0x36,0x35,0x91,0xdd,0x6a,0x90,
			0x78,0x9a,0xc2,0x12,0xa0,0xce,0xd9,0x5f,0xb4,0x0b,0xca,0xda,0xac,0x2c,0x97,0xee,
			0x7f,0xf3,0x02,0xc3,0x7c,0xf2,0xaa,0x72,0x98,0x31,0x1d,0xb8,0x5a,0x2d,0x4d,0x1b,
			0x9f,0x03,0x70,0x23,0xc2,0x1b,0xf2,0xe9,0xb9,0xab,0xec,0xc7,0x80,0xa7,0x0c,0xf2,
			0xa5,0x40,0x02,0x63,0x42,0x72,0xd7,0xdc,0xf2,0x5e,0xe4,0xc7,0xdb,0xb0,0x44,0x37,
			0xf3,0x0f,0x50,0xbd,0xcf,0x49,0x62,0xd4,0x32,0x48,0x4a,0x5d,0xab,0x7b,0x60,0xb3,
			0x1d,0x80,0x25,0xb9,0x84,0xc8,0x21,0xfa,0xdb,0xf2,0xb6,0x23,0xf2,0xf9,0x2a,0x65,
			0x31,0xdc,0xe1,0x1a,0x1f,0x25,0x2e,0x60,0x30,0x16,0x07,0x8f,0x39,0x86,0xe0,0x81),
		.d = chunk_from_chars(
			0x1c,0xc4,0x40,0x8e,0x92,0xd1,0xae,0xd2,0x2f,0x6c,0x04,0x3d,0x82,0x14,0x03,0x04,
			0xe7,0x82,0x5e,0x74,0xeb,0xaa,0x10,0x40,0x8d,0x33,0x95,0x46,0x30,0xb6,0x3b,0x02,
			0x3a,0xc2,0x13,0x6e,0xaf,0x45,0x16,0xd7,0x3a,0xe5,0x4f,0x7f,0xf6,0xf8,0xeb,0x34,
			0x1f,0x1d,0x39,0xae,0xd5,0xc9,0x8d,0x92,0xf1,0x85,0x8a,0xe4,0x11,0xb9,0xe0,0xe2,
			0x06,0xe0,0xe2,0xb2,0x25,0x19,0xbc,0x20,0xd3,0x04,0xbc,0xb4,0x7d,0x81,0xc4,0xf4,
			0x9e,0x59,0x9b,0x86,0x95,0x26,0xb7,0x58,0x88,0xb0,0x2f,0x29,0xdc,0x59,0x58,0xe8,
			0x21,0x6f,0x6a,0xe4,0x75,0xbf,0xf7,0x15,0xa7,0x41,0x63,0x32,0x04,0xbf,0xed,0x27,
			0xbd,0xc6,0x25,0x22,0x60,0x99,0x9f,0xf1,0x77,0xf1,0xa4,0xf7,0xa0,0x60,0x57,0x18,
			0xe9,0xad,0x45,0xb8,0x99,0x2b,0x59,0x46,0x6c,0x11,0xb4,0xda,0x7e,0x0e,0x1a,0x97,
			0x4c,0x28,0xc0,0xe1,0x54,0xe0,0x78,0x68,0xa0,0x1e,0x3f,0xa7,0x4e,0x6c,0xde,0x47,
			0x95,0x26,0xc4,0x6b,0x73,0x18,0xed,0x5f,0x12,0xe7,0xa2,0x27,0x01,0xf7,0x39,0xa3,
			0x10,0xec,0xcd,0x59,0x94,0xd4,0xbc,0x7f,0x8d,0xf6,0xe1,0x5e,0x3b,0x84,0xe1,0x83,
			0x93,0x64,0x62,0x32,0xfe,0x6e,0xc4,0x8b,0xe9,0xed,0x26,0xca,0xb8,0x74,0x96,0x00,
			0x9d,0x25,0xa5,0xbb,0x3c,0xdd,0xbd,0x32,0xb1,0xda,0x3f,0x07,0x03,0x8f,0xfc,0x78,
			0x14,0xdf,0x10,0x33,0xff,0xda,0x30,0xab,0x05,0xf0,0x75,0x5e,0x33,0x00,0x22,0xc9,
			0xcb,0x6b,0x78,0xdf,0xb4,0x75,0x02,0xb6,0x29,0xeb,0x4b,0xad,0xaf,0xf3,0x44,0x25,
			0x96,0x28,0x9f,0xa3,0x29,0xd5,0x08,0x7b,0xf7,0x0f,0x53,0x4f,0x50,0xff,0xa6,0x5e,
			0xeb,0x64,0x14,0x1d,0x0d,0xb4,0x31,0xa4,0x15,0x9d,0x39,0x0f,0x82,0x33,0x98,0x12,
			0x71,0xf4,0x8c,0x66,0x6c,0x50,0xab,0x7f,0xa4,0xf1,0xe7,0xe3,0x96,0x34,0x71,0x84,
			0xe1,0xd9,0x77,0xa0,0x8d,0xae,0xb2,0xfa,0xbe,0x8d,0xca,0x0c,0x2f,0xb9,0x41,0xd3,
			0xe5,0x39,0x8a,0x0d,0x1d,0xb9,0x18,0xee,0x59,0x3c,0x4b,0x4d,0xc9,0x67,0x13,0x1a,
			0x4b,0xc1,0x80,0x80,0xad,0x57,0xa9,0x42,0x8e,0x72,0x69,0x1b,0x0d,0xd6,0xb1,0xd5,
			0x66,0xe0,0x2a,0xd4,0x5f,0x30,0x8b,0xe8,0xbc,0x37,0xf3,0xfe,0xd6,0xc8,0x8d,0x28,
			0x38,0x56,0x67,0x33,0x57,0x04,0xa4,0xa5,0x50,0xf8,0x70,0xea,0x59,0xca,0x18,0x4f),
		.e = chunk_from_chars(0x2c,0xc9,0x2f),
		.m = chunk_from_chars(
			0xd1,0xac,0xd4,0xa6,0x03,0x5f,0x23,0xbf,0xe6,0x70,0x06,0x13,0x2a,0x1c,0xd4,0x74,
			0xef,0x58,0xc0,0x11,0x36,0x70,0xf5,0x3c,0x95,0xee,0xe5,0x70,0x30,0xbd,0xdd,0x92,
			0xe4,0x8e,0xdd,0x77,0xc5,0x7b,0xa8,0xa7,0xcb,0xea,0x17,0x60,0xc8,0xb5,0xb2,0xde,
			0x9d,0x88,0x81,0xda,0xae,0x8a,0x20,0x51,0x93,0x3f,0x12,0x86,0x11,0xad,0x57,0x4a,
			0x48,0xbd,0x41,0x7b,0x2d,0xe5,0x83,0xcb,0x3b,0x04,0x8b,0xc2,0x66,0x8d,0x12,0x07,
			0x51,0xe8,0xf1,0xdb,0xc0,0x15,0x36,0xe6,0x50,0x93,0x77,0x36,0x66,0x80,0x66,0x85,
			0x65,0x01,0xa6,0xba,0x24,0xe4,0xdd,0xb3,0x9f,0x84,0x0b,0x42,0xec,0xed,0x1a,0x75,
			0x71,0x41,0xb6,0x1b,0x55,0x5b,0x5e,0x83,0x34,0xb1,0xbb,0x87,0x17,0x7f,0x31,0xc9),
		.s = chunk_from_chars(
			0x55,0x61,0x3f,0x64,0x33,0x1a,0xbd,0x97,0x1e,0x82,0x2c,0xc3,0xd9,0x72,0xf6,0x7e,
			0x78,0x2b,0x6a,0xf8,0x1a,0xac,0x0b,0x80,0xe2,0xf6,0x32,0x4d,0xa6,0xd9,0xf3,0x3c,
			0x46,0xf8,0xb7,0xe2,0x4c,0x94,0xde,0xee,0x7e,0x98,0x52,0x5a,0xaf,0x4c,0xa6,0x84,
			0x37,0x53,0xdf,0x3d,0x44,0x4e,0x1e,0x05,0x40,0xf9,0xc1,0x96,0xff,0xbb,0xcb,0x4f,
			0xca,0x7a,0xad,0x4b,0xa9,0x42,0x88,0xf9,0x11,0x65,0x0e,0x3c,0xaf,0x6d,0xd2,0x2f,
			0x87,0x92,0x3f,0xa4,0xb1,0x2e,0x2d,0x6b,0x11,0x5f,0x2f,0xe7,0x73,0xaf,0x25,0x63,
			0xcc,0x76,0x0d,0xe8,0x8c,0xfa,0x6e,0x16,0x87,0x8d,0xd9,0xc0,0x7f,0x0d,0x45,0xb1,
			0x0a,0x22,0x4d,0x1a,0x7e,0xe1,0xd2,0x6f,0xfa,0xdf,0xd3,0x8b,0x78,0x00,0x4a,0x20,
			0x45,0xb3,0x5d,0x9c,0xb9,0xfc,0xcd,0xc7,0x15,0x66,0x8a,0x74,0x01,0xdb,0x7a,0xde,
			0xf1,0x09,0xeb,0xca,0xaf,0xc5,0x26,0x9d,0x37,0x7c,0x1e,0x8f,0x7d,0x42,0x92,0x45,
			0x9c,0x80,0x1a,0x93,0xc4,0xd0,0x7e,0xf4,0x77,0xf8,0x71,0x75,0xaa,0x35,0xe0,0x02,
			0xa0,0x3c,0x20,0x51,0x98,0x5f,0xcb,0x69,0x63,0xe9,0xbf,0x96,0xe2,0x8e,0x55,0xeb,
			0x52,0xa3,0x55,0x44,0x75,0xf9,0x09,0x5f,0x24,0xa9,0xe4,0x22,0xfa,0x84,0xb6,0x05,
			0x90,0xe8,0xb8,0xd6,0xec,0x91,0xdc,0xf6,0xa5,0x84,0xcc,0x8a,0x93,0x4e,0x55,0x1b,
			0xef,0xee,0xdc,0x0f,0x0f,0x48,0xda,0xdf,0x29,0x3f,0x0f,0x92,0x60,0xee,0x22,0xd7,
			0x2e,0xa4,0x6e,0x27,0x10,0x45,0xb6,0x65,0x16,0xb2,0xbe,0xbe,0xed,0x52,0x72,0x2a,
			0xec,0x5f,0x08,0x40,0x85,0xf8,0xfa,0xa6,0x89,0x30,0x8b,0xc1,0x3f,0xa5,0xb9,0xe5,
			0xb0,0xaa,0xdf,0x27,0x66,0x25,0x8e,0x01,0xd4,0xb8,0xdd,0xbe,0x0d,0x32,0x8a,0xa1,
			0x12,0x23,0x7e,0xc1,0x0b,0x11,0x5b,0xe8,0x3f,0xae,0x8a,0x01,0x13,0xf8,0x89,0x0e,
			0xda,0xb7,0x9d,0x04,0x63,0xc9,0xec,0x6e,0x2d,0xb4,0x50,0x46,0x93,0x53,0x5d,0xcf,
			0x76,0x95,0xf8,0x44,0x25,0xe9,0xf7,0xb7,0x3e,0x4f,0x18,0xbc,0x07,0x2f,0x51,0x31,
			0x98,0x9b,0x0c,0xdb,0x07,0xf4,0x96,0xf1,0x5e,0x56,0x2d,0x07,0xf0,0x71,0x1c,0xf9,
			0x79,0xf1,0xaa,0x4d,0x14,0x0d,0x97,0x83,0xef,0xd6,0xf9,0x6c,0x41,0xfe,0x5f,0xb7,
			0x9e,0x5d,0x5d,0x68,0x9d,0xd5,0xa7,0x05,0xf1,0xff,0x83,0x77,0x54,0x03,0x1a,0x39),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, },
		.n = chunk_from_chars(
			0x99,0xa5,0xc8,0xd0,0x94,0xa5,0xf9,0x17,0x03,0x46,0x67,0xa0,0x40,0x8b,0x7e,0xcf,
			0xca,0xac,0xc3,0xf9,0x78,0x44,0x44,0xe2,0x17,0x73,0xc3,0x46,0x1e,0xc3,0x55,0xf0,
			0xd0,0xf5,0x2a,0x5d,0xb0,0x56,0x8a,0x71,0xd3,0x88,0x69,0x67,0x88,0xef,0x66,0xae,
			0x73,0x40,0xc6,0xb2,0x8d,0xbf,0x92,0x5f,0xe8,0x35,0x57,0x98,0x65,0x75,0xf7,0x9c,
			0xca,0x69,0x21,0x72,0x21,0x39,0x7e,0xd5,0x80,0x8a,0x26,0xf7,0xe7,0xe7,0x14,0xc9,
			0x32,0x35,0xf9,0x14,0xd4,0x5c,0x4a,0x9a,0xf4,0x61,0x9b,0x20,0xf5,0x11,0xad,0x64,
			0x4b,0xd3,0x41,0x2d,0xfd,0xf0,0xff,0x71,0x7f,0x7a,0xac,0x74,0x6f,0x31,0x0b,0xfa,
			0x9a,0x14,0x1a,0xc3,0xdb,0xf0,0x1c,0x1f,0xc7,0x4f,0xeb,0xd1,0x97,0x93,0x84,0x19,
			0xc2,0x62,0x29,0x35,0x05,0xc3,0x5f,0x40,0x2f,0x90,0x53,0xad,0x13,0xc5,0x1a,0x59,
			0x60,0xec,0xde,0x55,0xec,0x82,0x9e,0x95,0x3f,0x94,0x1a,0xf7,0x33,0xe5,0x87,0x05,
			0x91,0x37,0x67,0xe7,0xa7,0x20,0x0d,0x1d,0x09,0xe7,0xe7,0xe2,0xd2,0x69,0xfa,0x29,
			0xa5,0x58,0xbb,0x16,0x30,0x4b,0x05,0x9f,0x13,0xf4,0xca,0x56,0x0a,0x81,0x01,0xfe,
			0x37,0x20,0xb4,0xa7,0x79,0xec,0x12,0x64,0x27,0x32,0x6c,0xaa,0x13,0x2a,0x3d,0x36,
			0x11,0xd7,0xdb,0xc5,0x03,0x36,0xfa,0xc7,0x89,0xec,0x40,0x6b,0x39,0x7e,0x1e,0x36,
			0xd7,0xda,0xf9,0xb6,0x24,0xbf,0x63,0x9c,0x82,0xb8,0x59,0x28,0x87,0x47,0x69,0x0c,
			0x73,0x0c,0x98,0x0b,0x2f,0x5a,0x23,0x9d,0xd9,0x5a,0xd5,0x38,0x9a,0x2e,0xc9,0x0c,
			0x57,0x78,0x60,0x47,0x13,0x71,0x03,0x83,0xae,0x55,0xd4,0xd2,0x8c,0x06,0xd4,0xac,
			0x26,0xf0,0xd1,0x23,0x1f,0x1d,0x67,0x62,0xc8,0xe0,0xd9,0x18,0x11,0x81,0x56,0xbc,
			0x63,0x77,0x60,0xda,0xea,0x18,0x47,0x46,0xb8,0xdc,0xf6,0xf6,0x1d,0xb2,0x74,0xa7,
			0xdd,0xce,0xaa,0x07,0x49,0x37,0xab,0xab,0xad,0x45,0x49,0xb9,0x7a,0xb9,0x92,0x49,
			0x4a,0x80,0x72,0x08,0xab,0xd7,0x89,0x82,0x3f,0x5d,0x75,0xc4,0xb9,0x94,0x08,0x9c,
			0x80,0x72,0xcf,0xc2,0x54,0xe0,0xd8,0x20,0x2f,0xd8,0x96,0x47,0x6e,0x96,0xad,0x9d,
			0x30,0x9a,0x0e,0x8e,0x73,0x01,0x28,0x2f,0x07,0xeb,0x2a,0xe8,0xed,0xef,0xb7,0xdb,
			0xbe,0x13,0xb9,0x6e,0x8b,0x40,0x24,0xc6,0xb8,0x4d,0xe0,0xa0,0x5e,0x15,0x02,0x85),
		.d = chunk_from_chars(
			0x03,0xb1,0x90,0x5b,0x88,0x3e,0xa9,0xd3,0x2a,0x37,0x96,0x4c,0xf2,0xb3,0x19,0x9c,
			0xb1,0xee,0x5f,0x61,0x1f,0xf2,0xaa,0x16,0x32,0x69,0x94,0x2b,0x35,0x43,0x6a,0x4a,
			0x70,0x8e,0x73,0x71,0x85,0x7b,0x60,0x3f,0xe8,0x4e,0x43,0x7f,0xea,0x0f,0xb9,0xc1,
			0x9c,0xb0,0x9b,0x3c,0x39,0xac,0x82,0xfa,0xf7,0x3f,0x12,0x0a,0xb2,0xdd,0x49,0x57,
			0x0e,0xd9,0xc4,0x3c,0x70,0xf4,0x30,0x20,0x4f,0x77,0x74,0x3b,0x29,0x51,0x23,0x21,
			0x8a,0x5b,0x12,0x83,0xda,0xdf,0x93,0xcf,0xe3,0x73,0x75,0x00,0x0f,0xbe,0x64,0xa5,
			0x3f,0xa4,0xe4,0x0f,0x9a,0x8e,0x40,0x37,0x3f,0x75,0x92,0x35,0x66,0x5a,0xb1,0xf8,
			0x4d,0xd2,0xc8,0xa8,0x0a,0x8c,0x18,0x30,0xe7,0xd6,0x82,0xbd,0xba,0xbc,0xed,0xf3,
			0x22,0xe6,0xc7,0x87,0xcd,0x20,0x84,0xab,0x75,0x27,0x04,0x0e,0x78,0x5a,0x8e,0xf8,
			0x62,0xe0,0x25,0x66,0x99,0x5e,0x3a,0x3c,0x1d,0xb9,0xff,0x04,0x60,0xee,0xdf,0x80,
			0x78,0x39,0xb8,0x01,0x70,0xfd,0x4f,0xcb,0xd0,0x85,0x04,0x24,0x90,0x7d,0x8f,0x28,
			0x65,0x2c,0xb7,0xe9,0xd3,0x7e,0x1b,0xda,0x2e,0xad,0xb8,0x89,0xe5,0x55,0x0f,0x3c,
			0x49,0x01,0xdb,0x4d,0xb5,0x77,0x5d,0xd9,0xd1,0xe8,0x63,0xda,0xc1,0xe5,0xd9,0x47,
			0xef,0x0b,0x7b,0xcf,0x74,0x6c,0xbf,0xd3,0xe6,0x00,0x0f,0x40,0x92,0xb0,0xb1,0xee,
			0xa7,0x3d,0x15,0x43,0x11,0xb5,0x1b,0xde,0x7e,0x4d,0xff,0x4a,0x5f,0xcd,0x89,0xb8,
			0x28,0x53,0x45,0x65,0x4c,0x4f,0xc6,0x97,0x53,0x4c,0x79,0x41,0x35,0xfd,0x85,0xe8,
			0x73,0x65,0x6e,0xfe,0x2d,0x90,0x28,0xc7,0x36,0xb4,0x6c,0x59,0x1f,0x4d,0x5b,0xa1,
			0x3b,0x44,0xba,0xce,0x0b,0x39,0x7c,0xc1,0x7f,0x5d,0x77,0x93,0x95,0x05,0x9c,0x59,
			0xd3,0x64,0x52,0x93,0xab,0x0d,0x63,0xca,0xe4,0x39,0x20,0x23,0x61,0x9e,0xc5,0x7d,
			0xc4,0x33,0x93,0x23,0x94,0xa9,0xf5,0x9a,0x69,0x0c,0x0d,0xd1,0xe1,0x1c,0xf0,0x92,
			0xba,0x54,0x1c,0x55,0xf3,0xfa,0x89,0x24,0xd2,0x55,0xa1,0x68,0xc3,0xad,0x26,0xe6,
			0xb4,0xe5,0xe0,0x50,0x02,0x6f,0x65,0x0a,0x2a,0xcb,0xb1,0xee,0xc5,0x74,0xcc,0x8b,
			0x14,0x47,0x0b,0x7d,0x3b,0x12,0x2b,0x00,0x95,0x72,0x0f,0x97,0xc6,0x17,0x5c,0x53,
			0xe7,0x60,0x36,0xd5,0xe9,0xb6,0xa2,0xaa,0xba,0x76,0x31,0xab,0x3b,0x7b,0x8a,0xe9),
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0xcc,0x21,0x59,0x3a,0x6a,0x0f,0x73,0x7e,0x29,0x70,0xb7,0xc0,0x79,0x84,0xb0,0x70,
			0xd7,0x61,0x72,0x62,0x96,0xa0,0x7e,0x24,0xe0,0x56,0xe6,0x8f,0xf8,0x46,0xb2,0x9c,
			0xc1,0x54,0x81,0x79,0x84,0x3d,0x74,0xdc,0xee,0x86,0x47,0x98,0x58,0xb2,0xc1,0x6e,
			0x4c,0xb8,0x4f,0x25,0x44,0xb4,0xec,0xdc,0xb4,0xdd,0x43,0xa0,0x4b,0xb7,0x18,0x3a,
			0x76,0x8a,0xe4,0x4a,0x27,0x12,0xbf,0x9a,0xd4,0x78,0x83,0xac,0xc2,0x81,0x2f,0x95,
			0x83,0x06,0x89,0x0e,0xbe,0xa4,0x08,0xc9,0x2e,0xb4,0xf0,0x01,0xed,0x7d,0xbf,0x55,
			0xf3,0xa9,0xc8,0xd6,0xd9,0xf6,0x1e,0x5f,0xe3,0x2e,0xb3,0x25,0x3e,0x59,0xc1,0x8e,
			0x86,0x31,0x69,0x47,0x8c,0xd6,0x9b,0x91,0x55,0xc3,0x35,0xdb,0x66,0x01,0x6f,0x96),
		.s = chunk_from_chars(
			0x0a,0xa5,0x72,0xa6,0x84,0x5b,0x87,0x0b,0x89,0x09,0xa6,0x83,0xbb,0x7e,0x6e,0x76,
			0x16,0xf7,0x7b,0xef,0xf2,0x87,0x46,0x11,0x6d,0x8b,0xc4,0xb7,0x33,0x55,0x46,0xb5,
			0x1e,0x80,0x06,0xed,0x0f,0xc9,0xa0,0xd6,0x6f,0x63,0xce,0x0b,0x9e,0xbf,0x79,0x2d,
			0x7e,0xfd,0x43,0x05,0xd7,0x62,0x4d,0x54,0x54,0x00,0xa5,0xfd,0x6a,0x06,0xb7,0x8f,
			0x17,0x4b,0x86,0x80,0x3f,0x7c,0xd1,0xcc,0x93,0xe3,0xa9,0x72,0x86,0xf0,0xea,0x59,
			0x0e,0x40,0xff,0x26,0x19,0x5a,0xa2,0x19,0xfe,0x15,0x10,0xa0,0x16,0x78,0x52,0x23,
			0x60,0x6d,0x93,0x11,0xa1,0x6c,0x59,0xa8,0xfe,0x4a,0x6d,0xa6,0xec,0xd0,0xc1,0xd7,
			0x77,0x50,0x39,0x29,0x0c,0x2a,0xaa,0x17,0xed,0x1e,0xb1,0xb5,0x43,0x74,0xf7,0xe5,
			0x72,0xdb,0x13,0xcc,0xa3,0xa6,0x38,0x57,0x5f,0x80,0x04,0xaa,0x54,0xa2,0xfa,0x98,
			0x42,0x2f,0xc0,0x7e,0x43,0xad,0x3a,0x20,0xdd,0x93,0x00,0x14,0x93,0x44,0x26,0x77,
			0xd8,0x83,0x91,0x4d,0xc7,0x4e,0xc1,0xcb,0xeb,0xbb,0xd3,0xd2,0xb6,0xba,0xd4,0x66,
			0x6d,0x91,0x45,0x7b,0x69,0xb4,0x6a,0x1a,0x61,0xf2,0x12,0x98,0xf1,0xa6,0x79,0x42,
			0xec,0x86,0xc8,0x76,0x32,0x2d,0xd3,0x66,0xed,0x16,0x78,0x14,0xe9,0xc8,0xfc,0x90,
			0x40,0xc5,0xb4,0xb7,0xa8,0x59,0xbb,0xd8,0x80,0xcb,0x6b,0xc2,0x41,0xb9,0xe3,0x27,
			0xce,0x77,0x9e,0x07,0x83,0xb1,0xcf,0x44,0x5e,0x0b,0x2f,0x57,0x71,0xb3,0xf5,0x82,
			0x2a,0x13,0x64,0x39,0x1c,0x15,0x4d,0xc5,0x06,0xff,0xf1,0xfb,0x9d,0x9a,0x35,0xf8,
			0x01,0x99,0xa6,0xb3,0x0b,0x4b,0x92,0xb9,0x26,0x19,0xa4,0x0e,0x21,0xae,0xa1,0x92,
			0x84,0x01,0x58,0x63,0xc4,0x48,0x66,0xc6,0x1e,0xd9,0x04,0xa7,0xad,0x19,0xee,0x04,
			0xd9,0x66,0xc0,0xaa,0xe3,0x90,0x63,0x62,0x43,0x56,0x55,0x81,0xff,0x20,0xbd,0x6e,
			0x3c,0xfb,0x6e,0x31,0xf5,0xaf,0xba,0x96,0x4b,0x31,0x1d,0xc2,0xd0,0x23,0xa2,0x19,
			0x98,0xc8,0xdd,0x50,0xca,0x45,0x36,0x99,0x19,0x0b,0xd4,0x67,0x42,0x9e,0x2f,0x88,
			0xac,0xe2,0x9c,0x4d,0x1d,0xa4,0xda,0x61,0xaa,0xc1,0xed,0xa2,0x38,0x02,0x30,0xaa,
			0x8d,0xbb,0x63,0xc7,0x5a,0x3c,0x1e,0xc0,0x4d,0xa3,0xa1,0xf8,0x80,0xc9,0xc7,0x47,
			0xac,0xdb,0x74,0xa8,0x39,0x5a,0xf5,0x8f,0x5f,0x04,0x40,0x15,0xcc,0xaf,0x6e,0x94),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, },
		.n = chunk_from_chars(
			0xb4,0x3d,0x4a,0x44,0x6d,0xe4,0x5a,0xa8,0xf3,0x36,0xb9,0x3a,0x4c,0x59,0x23,0xf3,
			0xa5,0x38,0x6e,0xf8,0xdd,0x1e,0x94,0xc4,0x23,0x00,0xde,0x08,0x80,0xe9,0xbd,0x08,
			0x28,0xfb,0x32,0xe3,0x6e,0x4c,0x50,0xcb,0xf6,0x66,0x03,0x7a,0x8f,0x2e,0x05,0xf4,
			0x57,0x73,0x89,0x6c,0x10,0xae,0xa9,0x75,0xb3,0xdb,0xf4,0xc4,0xcd,0xfc,0x24,0xa0,
			0x38,0xc5,0xc0,0x6d,0x36,0x1b,0xaf,0x84,0xc3,0x8f,0xc2,0x2c,0x03,0xa3,0x6b,0x9d,
			0xce,0x38,0xe0,0x90,0x11,0x1d,0x9c,0x13,0x23,0xd7,0xa7,0x7d,0x3e,0x04,0xb7,0x13,
			0xfa,0xf7,0x40,0x96,0x5a,0x9e,0x18,0x83,0xee,0x37,0x75,0x48,0x9a,0xb5,0x14,0xce,
			0x48,0x07,0x86,0xf9,0xeb,0x74,0x1c,0x60,0xab,0x89,0x6a,0x9d,0x6e,0xae,0x3a,0x53,
			0xed,0x92,0x68,0x76,0x8c,0x21,0xf7,0x9e,0x30,0x75,0x9e,0x0b,0x01,0xab,0x7f,0xa2,
			0x24,0xea,0x82,0x29,0xc2,0x93,0x78,0x00,0x58,0xf2,0x58,0xe9,0x22,0x6d,0x73,0x74,
			0xac,0x42,0x5e,0xf1,0xd2,0xb6,0xe0,0x6e,0x5b,0x26,0x3d,0xf0,0xc6,0xd6,0x6c,0x00,
			0xed,0x26,0xcb,0xf2,0x46,0xa5,0xaf,0x0a,0x01,0x63,0x33,0x68,0x86,0xef,0x8d,0x92,
			0x9b,0x37,0x74,0x9a,0x08,0xf0,0xec,0x1d,0xb0,0x59,0x73,0xa8,0xaf,0xc8,0x17,0x78,
			0xb6,0xcc,0x91,0x06,0xf9,0x2b,0x14,0x53,0xf1,0x52,0x86,0x97,0xb1,0xdc,0x8d,0xd0,
			0xb2,0x55,0xe8,0x01,0x06,0x0f,0xe1,0x79,0xb2,0xd1,0x0a,0x9c,0x4c,0x3e,0x13,0xf3,
			0xc5,0x6f,0xa6,0xd5,0x51,0x66,0xf6,0x46,0x1a,0xf4,0xaa,0xf4,0xf4,0x16,0x8f,0xd5,
			0xea,0x6d,0xad,0xed,0xaa,0x3f,0x9f,0x1d,0xe4,0xde,0x99,0x3d,0x88,0x44,0x35,0x7c,
			0x0a,0xf7,0x9a,0x09,0x0e,0xcc,0x80,0x57,0x0c,0x64,0x15,0x45,0xbe,0xb4,0x2a,0x12,
			0x48,0xa5,0x2d,0xe6,0x12,0xf2,0xe0,0xf8,0x83,0x44,0x96,0x33,0x1a,0x73,0x54,0xf7,
			0xeb,0x91,0xa1,0x94,0x3b,0x5c,0xb3,0xb6,0xcf,0x19,0x84,0x51,0x73,0x5f,0xb5,0x54,
			0x92,0x2f,0x04,0xf0,0x09,0xa5,0x2a,0x15,0xd9,0x93,0x69,0xad,0xc2,0xe4,0x6b,0x09,
			0xbb,0x87,0x1f,0x4f,0x3e,0xc1,0x90,0x5a,0xcd,0x79,0x2b,0x8c,0x81,0xa3,0xd7,0x4b,
			0x31,0x6f,0xf9,0xd2,0x0b,0x93,0xf4,0x88,0x17,0xae,0x61,0x86,0x77,0xdc,0x94,0x51,
			0xf5,0x82,0xec,0x99,0x95,0xf4,0x45,0x52,0xf4,0x24,0x49,0x53,0xcd,0xe8,0x3f,0x11),
		.d = chunk_from_chars(
			0x05,0x50,0x8e,0x4a,0xec,0x1e,0x7e,0xad,0xdd,0x40,0x20,0x4c,0x8a,0x3a,0x38,0x90,
			0x48,0x79,0x2b,0x4d,0x1f,0x15,0x0d,0xc5,0x3a,0x10,0x9a,0xc8,0x05,0x5f,0xa9,0x49,
			0x22,0x78,0x2c,0x43,0xc3,0x07,0x12,0x36,0x65,0x8b,0x1c,0x7b,0x5d,0x07,0xb2,0xef,
			0xae,0xef,0x39,0x44,0x9e,0x02,0x80,0x97,0x98,0x0b,0x0a,0x3a,0x20,0x40,0x68,0x47,
			0xed,0x3f,0x79,0x5d,0x03,0xd5,0xfc,0xed,0x2a,0x39,0x08,0x24,0x58,0x9f,0x57,0x1f,
			0x5d,0xbb,0x79,0x5d,0x97,0x7e,0x8b,0x76,0x9a,0xdd,0x7f,0x4e,0xe7,0x65,0x82,0x9d,
			0x59,0xb4,0xb5,0xb5,0x8e,0xf0,0xda,0xa6,0xc8,0x68,0x2f,0x13,0x84,0x49,0xd8,0x7a,
			0xd4,0x30,0x7f,0xc0,0x46,0xcc,0xf6,0x36,0x03,0xef,0xf9,0xee,0xb1,0x48,0x2d,0xd7,
			0x4d,0xa2,0x55,0x98,0x4f,0x2b,0xd3,0x00,0xab,0xa6,0xb9,0x54,0x9d,0x5d,0x6e,0xab,
			0x02,0x7f,0x73,0xa8,0xf9,0x69,0x5a,0x81,0x40,0xdd,0x6c,0x83,0x9e,0xd6,0xf0,0x48,
			0x19,0xc3,0x8e,0xac,0xd8,0x75,0x3b,0xbb,0xbd,0x66,0xf3,0x4d,0xd1,0x6d,0xa4,0x04,
			0x4f,0x93,0xa5,0x00,0xc4,0x80,0x3f,0x33,0xf6,0xbc,0x20,0xbe,0x75,0xd1,0x30,0x25,
			0x46,0x1f,0xc5,0xb7,0x83,0xd9,0xc7,0x64,0x64,0xe8,0x95,0xaa,0x82,0x3f,0x18,0xd7,
			0xfe,0x7b,0xac,0xd8,0x91,0xe7,0xf9,0x60,0xfe,0xd6,0xf7,0xe8,0x77,0x41,0x3d,0x46,
			0x0a,0x1d,0xdc,0x4e,0x47,0xe7,0xde,0xcb,0x3f,0xae,0xf4,0x1c,0x34,0x74,0xfc,0x9b,
			0xb4,0x33,0x03,0xb8,0x02,0x71,0x30,0xb4,0x86,0x21,0x58,0xd0,0x17,0xf0,0x57,0x0b,
			0x51,0x9b,0x3f,0x87,0x55,0xe6,0x8b,0xba,0x10,0x38,0xca,0x25,0x0a,0x1c,0x7c,0xda,
			0x22,0xc3,0xbd,0xa9,0xaa,0xca,0x24,0xa5,0x0a,0x56,0x8c,0xca,0x52,0xab,0x72,0x21,
			0x41,0xbf,0x8b,0xdc,0x07,0x77,0x0e,0x69,0x32,0x96,0x64,0x52,0x72,0x58,0x25,0x90,
			0x32,0x8e,0x4f,0x11,0x69,0xac,0x57,0xab,0xa7,0x74,0xa9,0x81,0x89,0x89,0xc4,0xa0,
			0x69,0x10,0x4a,0x9d,0xcc,0xe8,0x72,0xea,0x45,0xb2,0x39,0x2a,0xf7,0x56,0xd9,0x72,
			0x06,0xac,0x90,0x7c,0x6f,0xde,0x9c,0xe5,0x33,0x08,0x3f,0x54,0x68,0xc3,0x04,0xff,
			0x2e,0x4d,0xef,0xca,0xfb,0xd5,0xe2,0x35,0xa2,0x41,0x56,0xee,0xf6,0x05,0x94,0xe9,
			0xf2,0xf4,0x97,0x40,0xa6,0x0f,0xf3,0x62,0x37,0xc1,0x7e,0x93,0x7c,0x27,0x51,0xf5),
		.e = chunk_from_chars(0x09,0x8f,0xfb),
		.m = chunk_from_chars(
			0xdd,0xdb,0x05,0xcb,0x9b,0xf1,0x0c,0x14,0xe7,0xdd,0x1e,0x9f,0x3b,0x3d,0x2b,0x32,
			0x9a,0x17,0xf3,0x16,0x76,0x28,0x10,0x11,0xd2,0x78,0x37,0x94,0xa4,0x32,0xbb,0x34,
			0x7d,0xb3,0x52,0x5e,0x6b,0xe1,0x1c,0x47,0x1f,0xbe,0xe1,0x23,0x4b,0x3d,0x9b,0x97,
			0x4e,0x54,0x34,0x70,0x13,0x52,0x90,0x95,0x36,0x73,0xce,0x3a,0x69,0xb1,0xcb,0x57,
			0x17,0xde,0xe8,0x59,0x47,0xf0,0x0e,0x17,0xc2,0x9c,0xef,0x07,0x78,0x26,0x8e,0xb2,
			0x20,0x77,0x01,0x65,0x1f,0x70,0x75,0x2a,0xab,0x7e,0x74,0xf6,0x30,0x6e,0x63,0x24,
			0xf2,0x83,0x4f,0x22,0xf5,0xc6,0xe9,0x6b,0x1a,0x9e,0xce,0xb5,0x8a,0xa0,0x0c,0x6b,
			0x57,0xa8,0xe2,0x5d,0x61,0x29,0xc8,0xb7,0x77,0xc1,0xaf,0x2f,0xbf,0x11,0x8a,0x83),
		.s = chunk_from_chars(
			0x11,0x10,0x7c,0x6d,0xa7,0xa7,0x6d,0xba,0x8c,0xe8,0xde,0x3d,0x5c,0x90,0xc6,0xbc,
			0xca,0xf3,0x3f,0xb9,0xa7,0xec,0x4d,0x40,0xa9,0x7f,0xbc,0xde,0xba,0xf6,0xe6,0x5c,
			0x09,0x5b,0x67,0xdf,0xa1,0x71,0xc5,0x4a,0x33,0x64,0xa6,0xde,0xd7,0x18,0xe1,0xbe,
			0xbb,0xb4,0x97,0xd1,0x5f,0x1d,0xe1,0x33,0xb5,0x82,0x97,0xe0,0x8c,0x1b,0x20,0x0f,
			0x84,0x66,0x57,0x9c,0xb8,0xe9,0xc3,0xdb,0xe5,0xf7,0x24,0x28,0x2e,0x5b,0xb2,0x85,
			0x70,0xa4,0x1d,0x8d,0x35,0xad,0x6e,0x13,0x1e,0xb2,0xa1,0xd3,0x29,0xd8,0xb0,0x8d,
			0x10,0xfa,0xf7,0x6f,0xdb,0xe7,0x42,0x17,0xa2,0x76,0x93,0x6b,0x15,0x93,0xb1,0xd3,
			0xa1,0x18,0x91,0xd1,0x86,0x41,0xc0,0xbe,0xe0,0xc3,0x7e,0x68,0xbf,0x75,0xad,0xd9,
			0xe7,0xdc,0xf3,0xb9,0xb4,0x41,0x3f,0xa3,0xec,0x90,0x67,0x60,0x38,0x16,0x1f,0x68,
			0xfe,0x3a,0xf6,0x4d,0x69,0xa1,0x6f,0xc2,0x2c,0x8e,0xa7,0xc0,0x6f,0x86,0xbc,0xbb,
			0xa6,0x67,0x9b,0x6f,0xa4,0x58,0x58,0xc8,0x7e,0xc2,0xda,0x88,0x96,0x92,0x0d,0xfd,
			0x77,0xfd,0x16,0xd9,0x53,0xe9,0x84,0x62,0x42,0x03,0xcc,0x21,0x78,0xd2,0x20,0xfa,
			0x47,0xba,0x9a,0x6d,0xdb,0xe7,0x59,0xdb,0x3d,0xc4,0x22,0x4c,0x7c,0xcb,0x09,0xb2,
			0x9e,0x20,0x07,0x22,0xa7,0x45,0xd3,0x32,0x92,0x09,0x32,0x3a,0x08,0x2a,0x3f,0x7a,
			0x2c,0x62,0x4b,0x40,0x53,0xfa,0x42,0x18,0x3d,0xd2,0x2b,0x7f,0x49,0xc4,0x56,0x9f,
			0x8c,0x3d,0x49,0x51,0x57,0x53,0xb9,0x93,0xef,0x81,0x22,0xc3,0x69,0x85,0xe9,0x6a,
			0x32,0x89,0xaf,0x8d,0x6e,0x22,0x93,0x4b,0xbc,0xf7,0x55,0x1b,0x15,0x70,0x92,0x31,
			0x58,0xd1,0xb5,0x54,0xc1,0x3f,0x4f,0xe0,0x8f,0xaf,0xf4,0x89,0x8b,0xe9,0x2f,0x6e,
			0x66,0xff,0x74,0xa8,0x0b,0x13,0x70,0x3e,0x0d,0x26,0x65,0xdb,0x71,0x5c,0x55,0x9e,
			0x68,0xbe,0xad,0x6f,0x6b,0x75,0x93,0xf0,0x2e,0xfb,0xd6,0x8c,0x67,0x62,0x56,0xae,
			0x86,0x95,0xa5,0x1d,0xab,0x90,0xe3,0x29,0x86,0x69,0x7f,0x4e,0xde,0x07,0x66,0x28,
			0x17,0x6c,0xa6,0xb6,0x51,0x91,0x8d,0x74,0x9f,0xd0,0x1a,0x16,0xa8,0x7a,0xda,0xad,
			0x4a,0xae,0x93,0xb8,0xc0,0x5d,0x92,0x56,0x5e,0xc6,0x82,0x17,0xaa,0xab,0x96,0x76,
			0x23,0xda,0x3e,0x5a,0x50,0x95,0x64,0x2c,0xc1,0x7d,0x73,0x0a,0x6d,0x95,0x9c,0xfa),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, },
		.n = chunk_from_chars(
			0xb4,0x3d,0x4a,0x44,0x6d,0xe4,0x5a,0xa8,0xf3,0x36,0xb9,0x3a,0x4c,0x59,0x23,0xf3,
			0xa5,0x38,0x6e,0xf8,0xdd,0x1e,0x94,0xc4,0x23,0x00,0xde,0x08,0x80,0xe9,0xbd,0x08,
			0x28,0xfb,0x32,0xe3,0x6e,0x4c,0x50,0xcb,0xf6,0x66,0x03,0x7a,0x8f,0x2e,0x05,0xf4,
			0x57,0x73,0x89,0x6c,0x10,0xae,0xa9,0x75,0xb3,0xdb,0xf4,0xc4,0xcd,0xfc,0x24,0xa0,
			0x38,0xc5,0xc0,0x6d,0x36,0x1b,0xaf,0x84,0xc3,0x8f,0xc2,0x2c,0x03,0xa3,0x6b,0x9d,
			0xce,0x38,0xe0,0x90,0x11,0x1d,0x9c,0x13,0x23,0xd7,0xa7,0x7d,0x3e,0x04,0xb7,0x13,
			0xfa,0xf7,0x40,0x96,0x5a,0x9e,0x18,0x83,0xee,0x37,0x75,0x48,0x9a,0xb5,0x14,0xce,
			0x48,0x07,0x86,0xf9,0xeb,0x74,0x1c,0x60,0xab,0x89,0x6a,0x9d,0x6e,0xae,0x3a,0x53,
			0xed,0x92,0x68,0x76,0x8c,0x21,0xf7,0x9e,0x30,0x75,0x9e,0x0b,0x01,0xab,0x7f,0xa2,
			0x24,0xea,0x82,0x29,0xc2,0x93,0x78,0x00,0x58,0xf2,0x58,0xe9,0x22,0x6d,0x73,0x74,
			0xac,0x42,0x5e,0xf1,0xd2,0xb6,0xe0,0x6e,0x5b,0x26,0x3d,0xf0,0xc6,0xd6,0x6c,0x00,
			0xed,0x26,0xcb,0xf2,0x46,0xa5,0xaf,0x0a,0x01,0x63,0x33,0x68,0x86,0xef,0x8d,0x92,
			0x9b,0x37,0x74,0x9a,0x08,0xf0,0xec,0x1d,0xb0,0x59,0x73,0xa8,0xaf,0xc8,0x17,0x78,
			0xb6,0xcc,0x91,0x06,0xf9,0x2b,0x14,0x53,0xf1,0x52,0x86,0x97,0xb1,0xdc,0x8d,0xd0,
			0xb2,0x55,0xe8,0x01,0x06,0x0f,0xe1,0x79,0xb2,0xd1,0x0a,0x9c,0x4c,0x3e,0x13,0xf3,
			0xc5,0x6f,0xa6,0xd5,0x51,0x66,0xf6,0x46,0x1a,0xf4,0xaa,0xf4,0xf4,0x16,0x8f,0xd5,
			0xea,0x6d,0xad,0xed,0xaa,0x3f,0x9f,0x1d,0xe4,0xde,0x99,0x3d,0x88,0x44,0x35,0x7c,
			0x0a,0xf7,0x9a,0x09,0x0e,0xcc,0x80,0x57,0x0c,0x64,0x15,0x45,0xbe,0xb4,0x2a,0x12,
			0x48,0xa5,0x2d,0xe6,0x12,0xf2,0xe0,0xf8,0x83,0x44,0x96,0x33,0x1a,0x73,0x54,0xf7,
			0xeb,0x91,0xa1,0x94,0x3b,0x5c,0xb3,0xb6,0xcf,0x19,0x84,0x51,0x73,0x5f,0xb5,0x54,
			0x92,0x2f,0x04,0xf0,0x09,0xa5,0x2a,0x15,0xd9,0x93,0x69,0xad,0xc2,0xe4,0x6b,0x09,
			0xbb,0x87,0x1f,0x4f,0x3e,0xc1,0x90,0x5a,0xcd,0x79,0x2b,0x8c,0x81,0xa3,0xd7,0x4b,
			0x31,0x6f,0xf9,0xd2,0x0b,0x93,0xf4,0x88,0x17,0xae,0x61,0x86,0x77,0xdc,0x94,0x51,
			0xf5,0x82,0xec,0x99,0x95,0xf4,0x45,0x52,0xf4,0x24,0x49,0x53,0xcd,0xe8,0x3f,0x11),
		.d = chunk_from_chars(
			0x05,0x50,0x8e,0x4a,0xec,0x1e,0x7e,0xad,0xdd,0x40,0x20,0x4c,0x8a,0x3a,0x38,0x90,
			0x48,0x79,0x2b,0x4d,0x1f,0x15,0x0d,0xc5,0x3a,0x10,0x9a,0xc8,0x05,0x5f,0xa9,0x49,
			0x22,0x78,0x2c,0x43,0xc3,0x07,0x12,0x36,0x65,0x8b,0x1c,0x7b,0x5d,0x07,0xb2,0xef,
			0xae,0xef,0x39,0x44,0x9e,0x02,0x80,0x97,0x98,0x0b,0x0a,0x3a,0x20,0x40,0x68,0x47,
			0xed,0x3f,0x79,0x5d,0x03,0xd5,0xfc,0xed,0x2a,0x39,0x08,0x24,0x58,0x9f,0x57,0x1f,
			0x5d,0xbb,0x79,0x5d,0x97,0x7e,0x8b,0x76,0x9a,0xdd,0x7f,0x4e,0xe7,0x65,0x82,0x9d,
			0x59,0xb4,0xb5,0xb5,0x8e,0xf0,0xda,0xa6,0xc8,0x68,0x2f,0x13,0x84,0x49,0xd8,0x7a,
			0xd4,0x30,0x7f,0xc0,0x46,0xcc,0xf6,0x36,0x03,0xef,0xf9,0xee,0xb1,0x48,0x2d,0xd7,
			0x4d,0xa2,0x55,0x98,0x4f,0x2b,0xd3,0x00,0xab,0xa6,0xb9,0x54,0x9d,0x5d,0x6e,0xab,
			0x02,0x7f,0x73,0xa8,0xf9,0x69,0x5a,0x81,0x40,0xdd,0x6c,0x83,0x9e,0xd6,0xf0,0x48,
			0x19,0xc3,0x8e,0xac,0xd8,0x75,0x3b,0xbb,0xbd,0x66,0xf3,0x4d,0xd1,0x6d,0xa4,0x04,
			0x4f,0x93,0xa5,0x00,0xc4,0x80,0x3f,0x33,0xf6,0xbc,0x20,0xbe,0x75,0xd1,0x30,0x25,
			0x46,0x1f,0xc5,0xb7,0x83,0xd9,0xc7,0x64,0x64,0xe8,0x95,0xaa,0x82,0x3f,0x18,0xd7,
			0xfe,0x7b,0xac,0xd8,0x91,0xe7,0xf9,0x60,0xfe,0xd6,0xf7,0xe8,0x77,0x41,0x3d,0x46,
			0x0a,0x1d,0xdc,0x4e,0x47,0xe7,0xde,0xcb,0x3f,0xae,0xf4,0x1c,0x34,0x74,0xfc,0x9b,
			0xb4,0x33,0x03,0xb8,0x02,0x71,0x30,0xb4,0x86,0x21,0x58,0xd0,0x17,0xf0,0x57,0x0b,
			0x51,0x9b,0x3f,0x87,0x55,0xe6,0x8b,0xba,0x10,0x38,0xca,0x25,0x0a,0x1c,0x7c,0xda,
			0x22,0xc3,0xbd,0xa9,0xaa,0xca,0x24,0xa5,0x0a,0x56,0x8c,0xca,0x52,0xab,0x72,0x21,
			0x41,0xbf,0x8b,0xdc,0x07,0x77,0x0e,0x69,0x32,0x96,0x64,0x52,0x72,0x58,0x25,0x90,
			0x32,0x8e,0x4f,0x11,0x69,0xac,0x57,0xab,0xa7,0x74,0xa9,0x81,0x89,0x89,0xc4,0xa0,
			0x69,0x10,0x4a,0x9d,0xcc,0xe8,0x72,0xea,0x45,0xb2,0x39,0x2a,0xf7,0x56,0xd9,0x72,
			0x06,0xac,0x90,0x7c,0x6f,0xde,0x9c,0xe5,0x33,0x08,0x3f,0x54,0x68,0xc3,0x04,0xff,
			0x2e,0x4d,0xef,0xca,0xfb,0xd5,0xe2,0x35,0xa2,0x41,0x56,0xee,0xf6,0x05,0x94,0xe9,
			0xf2,0xf4,0x97,0x40,0xa6,0x0f,0xf3,0x62,0x37,0xc1,0x7e,0x93,0x7c,0x27,0x51,0xf5),
		.e = chunk_from_chars(0x09,0x8f,0xfb),
		.m = chunk_from_chars(
			0xdd,0xdb,0x05,0xcb,0x9b,0xf1,0x0c,0x14,0xe7,0xdd,0x1e,0x9f,0x3b,0x3d,0x2b,0x32,
			0x9a,0x17,0xf3,0x16,0x76,0x28,0x10,0x11,0xd2,0x78,0x37,0x94,0xa4,0x32,0xbb,0x34,
			0x7d,0xb3,0x52,0x5e,0x6b,0xe1,0x1c,0x47,0x1f,0xbe,0xe1,0x23,0x4b,0x3d,0x9b,0x97,
			0x4e,0x54,0x34,0x70,0x13,0x52,0x90,0x95,0x36,0x73,0xce,0x3a,0x69,0xb1,0xcb,0x57,
			0x17,0xde,0xe8,0x59,0x47,0xf0,0x0e,0x17,0xc2,0x9c,0xef,0x07,0x78,0x26,0x8e,0xb2,
			0x20,0x77,0x01,0x65,0x1f,0x70,0x75,0x2a,0xab,0x7e,0x74,0xf6,0x30,0x6e,0x63,0x24,
			0xf2,0x83,0x4f,0x22,0xf5,0xc6,0xe9,0x6b,0x1a,0x9e,0xce,0xb5,0x8a,0xa0,0x0c,0x6b,
			0x57,0xa8,0xe2,0x5d,0x61,0x29,0xc8,0xb7,0x77,0xc1,0xaf,0x2f,0xbf,0x11,0x8a,0x83),
		.s = chunk_from_chars(
			0x11,0x10,0x7c,0x6d,0xa7,0xa7,0x6d,0xba,0x8c,0xe8,0xde,0x3d,0x5c,0x90,0xc6,0xbc,
			0xca,0xf3,0x3f,0xb9,0xa7,0xec,0x4d,0x40,0xa9,0x7f,0xbc,0xde,0xba,0xf6,0xe6,0x5c,
			0x09,0x5b,0x67,0xdf,0xa1,0x71,0xc5,0x4a,0x33,0x64,0xa6,0xde,0xd7,0x18,0xe1,0xbe,
			0xbb,0xb4,0x97,0xd1,0x5f,0x1d,0xe1,0x33,0xb5,0x82,0x97,0xe0,0x8c,0x1b,0x20,0x0f,
			0x84,0x66,0x57,0x9c,0xb8,0xe9,0xc3,0xdb,0xe5,0xf7,0x24,0x28,0x2e,0x5b,0xb2,0x85,
			0x70,0xa4,0x1d,0x8d,0x35,0xad,0x6e,0x13,0x1e,0xb2,0xa1,0xd3,0x29,0xd8,0xb0,0x8d,
			0x10,0xfa,0xf7,0x6f,0xdb,0xe7,0x42,0x17,0xa2,0x76,0x93,0x6b,0x15,0x93,0xb1,0xd3,
			0xa1,0x18,0x91,0xd1,0x86,0x41,0xc0,0xbe,0xe0,0xc3,0x7e,0x68,0xbf,0x75,0xad,0xd9,
			0xe7,0xdc,0xf3,0xb9,0xb4,0x41,0x3f,0xa3,0xec,0x90,0x67,0x60,0x38,0x16,0x1f,0x68,
			0xfe,0x3a,0xf6,0x4d,0x69,0xa1,0x6f,0xc2,0x2c,0x8e,0xa7,0xc0,0x6f,0x86,0xbc,0xbb,
			0xa6,0x67,0x9b,0x6f,0xa4,0x58,0x58,0xc8,0x7e,0xc2,0xda,0x88,0x96,0x92,0x0d,0xfd,
			0x77,0xfd,0x16,0xd9,0x53,0xe9,0x84,0x62,0x42,0x03,0xcc,0x21,0x78,0xd2,0x20,0xfa,
			0x47,0xba,0x9a,0x6d,0xdb,0xe7,0x59,0xdb,0x3d,0xc4,0x22,0x4c,0x7c,0xcb,0x09,0xb2,
			0x9e,0x20,0x07,0x22,0xa7,0x45,0xd3,0x32,0x92,0x09,0x32,0x3a,0x08,0x2a,0x3f,0x7a,
			0x2c,0x62,0x4b,0x40,0x53,0xfa,0x42,0x18,0x3d,0xd2,0x2b,0x7f,0x49,0xc4,0x56,0x9f,
			0x8c,0x3d,0x49,0x51,0x57,0x53,0xb9,0x93,0xef,0x81,0x22,0xc3,0x69,0x85,0xe9,0x6a,
			0x32,0x89,0xaf,0x8d,0x6e,0x22,0x93,0x4b,0xbc,0xf7,0x55,0x1b,0x15,0x70,0x92,0x31,
			0x58,0xd1,0xb5,0x54,0xc1,0x3f,0x4f,0xe0,0x8f,0xaf,0xf4,0x89,0x8b,0xe9,0x2f,0x6e,
			0x66,0xff,0x74,0xa8,0x0b,0x13,0x70,0x3e,0x0d,0x26,0x65,0xdb,0x71,0x5c,0x55,0x9e,
			0x68,0xbe,0xad,0x6f,0x6b,0x75,0x93,0xf0,0x2e,0xfb,0xd6,0x8c,0x67,0x62,0x56,0xae,
			0x86,0x95,0xa5,0x1d,0xab,0x90,0xe3,0x29,0x86,0x69,0x7f,0x4e,0xde,0x07,0x66,0x28,
			0x17,0x6c,0xa6,0xb6,0x51,0x91,0x8d,0x74,0x9f,0xd0,0x1a,0x16,0xa8,0x7a,0xda,0xad,
			0x4a,0xae,0x93,0xb8,0xc0,0x5d,0x92,0x56,0x5e,0xc6,0x82,0x17,0xaa,0xab,0x96,0x76,
			0x23,0xda,0x3e,0x5a,0x50,0x95,0x64,0x2c,0xc1,0x7d,0x73,0x0a,0x6d,0x95,0x9c,0xfa),
	},
	{
		.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, },
		.n = chunk_from_chars(
			0xa3,0xf2,0x23,0x5a,0xd2,0x05,0x3b,0x4c,0x83,0xfa,0x38,0xf8,0x28,0x4e,0xd8,0x05,
			0x42,0x16,0x21,0xfe,0x98,0x84,0x5f,0xb0,0x1b,0x68,0x9f,0x5b,0x82,0xb3,0x25,0x11,
			0xb6,0xd1,0x61,0x73,0xe7,0xb4,0x0a,0x66,0xa3,0xa9,0x99,0xc1,0x89,0xbe,0xb9,0xe0,
			0x68,0x22,0x15,0x0a,0xc8,0xbe,0x67,0x71,0x86,0x37,0x0c,0x82,0x3b,0x52,0x77,0xd9,
			0x09,0xde,0x07,0x56,0x4e,0x28,0x1c,0xca,0x2f,0x13,0x87,0x3d,0x9d,0x07,0xb7,0xbd,
			0x85,0xa2,0xb9,0xac,0x66,0xf4,0xce,0x4f,0x5e,0x38,0xb8,0xe9,0xee,0xbe,0xc0,0x4c,
			0x8c,0xaf,0x31,0x1e,0x37,0x5d,0x69,0xe8,0x08,0x51,0xd5,0x59,0xb8,0xe9,0x0e,0x85,
			0xba,0x6b,0x96,0x47,0x67,0x90,0xf7,0x27,0xc2,0x5a,0xa8,0x16,0x30,0x62,0xec,0x85,
			0x43,0xfc,0xc7,0x75,0x9b,0xe6,0x2c,0x77,0x68,0xec,0xc3,0x7f,0x34,0x0b,0xb0,0x61,
			0x02,0x76,0x2b,0xf0,0x44,0x1c,0xa1,0xaa,0x2c,0x7a,0x81,0xbf,0x37,0xdc,0x8b,0x27,
			0x43,0x9d,0x3a,0xbb,0xa9,0x38,0x12,0xc9,0xbb,0x44,0xfe,0x4d,0x6a,0x94,0xba,0xae,
			0x70,0x93,0x79,0xf5,0xce,0x5d,0x0c,0x8f,0x81,0xd0,0x00,0x86,0xb9,0xca,0xa3,0x02,
			0x68,0x19,0x58,0x8f,0x49,0x1b,0x52,0x58,0x07,0x89,0x9c,0xda,0xb3,0x3d,0x8e,0x99,
			0x21,0x50,0xd2,0xb1,0x05,0xd3,0xaa,0xb6,0x15,0x21,0x7c,0x6a,0x3d,0x74,0x08,0x31,
			0xc7,0xdc,0x76,0xfa,0xab,0xd9,0xc9,0xb9,0x81,0x7e,0xad,0x0b,0x49,0x45,0x66,0xde,
			0x14,0x33,0xff,0xf5,0xba,0x46,0x04,0xc6,0xb8,0x44,0x6f,0x6f,0xc3,0x5e,0x74,0x6a,
			0xff,0x84,0xff,0x8b,0xd7,0x50,0x04,0x10,0xd1,0x0e,0x82,0xbf,0x4c,0x90,0x36,0x48,
			0x9d,0xe4,0x7d,0xee,0x9a,0x32,0x7a,0x5c,0x45,0x10,0xd8,0x56,0x13,0x21,0xb9,0x1d,
			0x55,0x55,0x9a,0x4c,0xba,0x85,0xe0,0xc3,0x61,0x76,0x70,0x84,0xb2,0x52,0x17,0xe8,
			0xa6,0x3c,0x4e,0x15,0x1a,0x1e,0x88,0x68,0x9f,0xee,0xcf,0xfd,0x16,0xfa,0x0a,0x65,
			0xae,0x41,0xd2,0xba,0xbc,0xa9,0x9c,0xf1,0xb9,0x59,0xc3,0xc0,0x76,0xc0,0xf7,0x59,
			0x74,0x14,0x6f,0x2c,0xc4,0x94,0x12,0x6f,0xbe,0xca,0xd4,0x21,0x7b,0x9a,0xaa,0x00,
			0xf1,0x69,0xfa,0x51,0x25,0x27,0xff,0x5a,0x0b,0x50,0xda,0x46,0xd6,0xbe,0x87,0x0e,
			0xce,0xf2,0xaf,0x7a,0x1e,0x6c,0x45,0x56,0xf6,0xf7,0xa0,0xa0,0x0b,0x9f,0x47,0xcb),
		.d = chunk_from_chars(
			0x3c,0x27,0x68,0x73,0x67,0xeb,0xd2,0x14,0x0b,0x4f,0x4f,0x20,0x24,0xae,0x38,0xe3,
			0xac,0xde,0xe1,0xfa,0x8a,0xb7,0xce,0xa9,0x91,0x4e,0xec,0xb5,0x80,0x11,0x30,0x15,
			0x1b,0x87,0x9f,0x7d,0xd9,0xc8,0xc5,0x23,0xfb,0xeb,0x16,0xea,0x4c,0x35,0x8b,0xfb,
			0x2b,0x6e,0x49,0x2b,0x46,0x66,0xb4,0xff,0x3f,0x2d,0xce,0xae,0x50,0xc3,0x3e,0xd5,
			0xb2,0x84,0x45,0x40,0x52,0x15,0xb6,0x4d,0xd8,0xd9,0x20,0x06,0xe6,0x81,0xa4,0x5f,
			0x78,0x98,0x34,0x31,0xc0,0xdb,0x66,0x4a,0x78,0x75,0x59,0x47,0x9a,0x3a,0x27,0x9b,
			0xd6,0xab,0x71,0xf9,0xd1,0x5f,0x01,0x09,0x37,0x05,0x47,0xcb,0x12,0xb7,0xaa,0x90,
			0xa8,0x4c,0x1d,0x16,0x51,0xb9,0xbe,0xe5,0xd2,0xe6,0x22,0x60,0x11,0x49,0x18,0x97,
			0xeb,0xf8,0x91,0xfa,0x67,0xcd,0x19,0xfa,0x4f,0xed,0x77,0xcc,0xd5,0xf9,0x63,0x43,
			0x74,0x3d,0x8b,0xd2,0x2e,0x46,0xda,0xbd,0xab,0x46,0x9a,0xdc,0x46,0x4a,0xcf,0x29,
			0xc3,0xa0,0x8f,0x57,0x48,0xf4,0xb0,0x0b,0xb1,0x55,0x87,0xee,0x44,0x4b,0x76,0x5c,
			0x06,0x9d,0xe3,0xe7,0x1d,0x09,0xe2,0x4a,0x69,0xb6,0x27,0xac,0x6f,0xc3,0x49,0x7d,
			0x9d,0x32,0xd7,0x97,0x1f,0xc4,0x47,0x1d,0x5d,0xd4,0x39,0x65,0x2d,0x73,0x93,0xfa,
			0x3b,0x43,0x4b,0x52,0x54,0xa4,0xcb,0x10,0x64,0xde,0xa1,0xd1,0xba,0x41,0x18,0x1a,
			0x6c,0x36,0x72,0xe3,0xe0,0xf6,0xa2,0x84,0xf2,0x31,0x21,0xdc,0x7a,0x14,0xa4,0xc7,
			0xa7,0x6e,0xbc,0x2b,0xfd,0x09,0xcb,0x69,0x84,0x1c,0xa9,0x59,0x07,0xcc,0x30,0xc9,
			0x4e,0x69,0x27,0x51,0x1c,0x72,0xe9,0x7d,0xd3,0x33,0x0b,0xb4,0x56,0x48,0x8d,0x91,
			0x9f,0xaf,0xd6,0xd8,0xf1,0x97,0x59,0xab,0x4c,0x37,0x46,0x63,0x14,0x75,0x1e,0x53,
			0xab,0x13,0x92,0x72,0xe8,0xd6,0x7c,0x0d,0xc0,0x42,0x49,0xd3,0x00,0xee,0x63,0x5a,
			0x20,0xa4,0xf6,0x5e,0xd7,0xcf,0xb3,0x0d,0x18,0x13,0xd0,0xb7,0x58,0x58,0xf2,0xf8,
			0xf9,0x10,0xc0,0x5b,0xcc,0x20,0x13,0xba,0x5c,0x93,0x21,0xc4,0x95,0xe1,0x41,0xa6,
			0x17,0xfa,0x73,0x18,0x19,0x18,0x90,0x9d,0xab,0xc3,0xf9,0xde,0x59,0x4a,0x19,0x33,
			0x12,0xe1,0x47,0x72,0xb4,0xa5,0x9b,0x4f,0xb7,0x2e,0x73,0x15,0xc0,0x55,0x23,0x6a,
			0x78,0x9e,0xb4,0xd2,0x24,0x60,0x97,0x45,0x72,0x1e,0x73,0x03,0x1c,0x91,0x13,0xe7),
		.e = chunk_from_chars(0xb3,0xf5,0x7f),
		.m = chunk_from_chars(
			0xbe,0x2f,0x3e,0x1d,0xc8,0xa3,0x71,0x15,0x70,0x40,0x1b,0xd5,0x35,0x18,0x54,0x26,
			0x94,0x4d,0x09,0x4e,0x84,0x81,0xa1,0x2a,0x43,0x8d,0xe0,0x7d,0x54,0x76,0x0c,0x88,
			0xc9,0x9d,0x4f,0xdb,0xbe,0x35,0x5d,0x6a,0x26,0xfa,0x56,0xe3,0xca,0x20,0xee,0x3f,
			0x8e,0x8a,0xcb,0x98,0xf6,0x3d,0x2f,0x3a,0xea,0x14,0xd6,0xfc,0xb6,0xb5,0x22,0xd1,
			0x55,0xc3,0x75,0x9a,0xef,0x56,0xde,0x3e,0xa0,0xa8,0xf9,0xfd,0x7b,0x11,0x10,0x01,
			0xcf,0x35,0x86,0x36,0xa8,0x7c,0x76,0x5c,0x99,0xc2,0x97,0x5b,0xb9,0x50,0x63,0xd6,
			0xec,0x0b,0x78,0x02,0x64,0xec,0x3e,0xb9,0x67,0xb0,0xca,0xca,0x52,0xd1,0x02,0x94,
			0xde,0xb4,0x02,0xd3,0xa2,0x24,0xbf,0xb9,0xd9,0xff,0xea,0x41,0x66,0x2f,0x18,0xc0),
		.s = chunk_from_chars(
			0x78,0x7c,0xdd,0x6e,0x1d,0x4f,0xdf,0x9a,0x0d,0x9f,0x96,0x5e,0xb8,0x57,0x25,0x23,
			0x2a,0x9e,0xfc,0xc1,0x2a,0xbf,0xa1,0xef,0x25,0xa8,0x1e,0x09,0x83,0x11,0x1d,0x90,
			0x00,0xd4,0x94,0xfc,0x7d,0x32,0x01,0xeb,0x3b,0xba,0x32,0x73,0x02,0x72,0x7f,0x70,
			0x86,0x14,0x7a,0x75,0x5b,0x48,0x27,0x03,0x0c,0x72,0x76,0x53,0x6f,0x42,0x55,0x93,
			0xab,0x2e,0x91,0x27,0xa1,0x49,0xe7,0x54,0xde,0x7a,0xd7,0x7f,0x8c,0x20,0x43,0x26,
			0x7d,0xb4,0x9f,0x8a,0x35,0x03,0x1d,0x83,0xf1,0x3d,0x14,0x0d,0x5d,0xf4,0xd4,0x24,
			0xb4,0x74,0x54,0x04,0x1a,0x23,0xb9,0x2f,0xf6,0x81,0x8e,0x74,0x9d,0x65,0xd0,0x1f,
			0xc5,0x0b,0xeb,0xf6,0x91,0x52,0xf3,0xf5,0xfc,0xb4,0x87,0x3b,0x10,0x36,0x21,0x9e,
			0x22,0xb1,0xe7,0x4f,0x83,0x68,0xc8,0xc5,0x01,0xce,0x65,0xf2,0xc9,0x29,0xd9,0x0a,
			0x8e,0xc8,0x99,0x63,0x0e,0x80,0x25,0x47,0xa7,0xca,0x6e,0xf1,0x8a,0xb3,0xcb,0x3e,
			0xb4,0xa6,0x91,0xee,0x68,0xae,0xbe,0xaf,0x1b,0x9c,0x05,0x5a,0xd1,0x22,0x18,0x03,
			0x9c,0xf4,0x80,0xcd,0x8d,0x29,0x43,0x32,0xc5,0xe1,0x6e,0xbb,0xe6,0xaf,0x11,0xf8,
			0xf4,0xbf,0x49,0xf9,0xb4,0xed,0x2f,0x51,0x11,0x26,0xae,0x78,0x0a,0x3b,0x78,0x4b,
			0xe8,0xf4,0x42,0x6a,0xbd,0x17,0xf8,0x60,0x00,0x74,0x48,0x3f,0x2a,0xf3,0xb7,0x1a,
			0x89,0x64,0xc6,0xe0,0xfa,0x00,0x04,0x9a,0x1d,0x94,0x0d,0x34,0xcc,0x08,0x83,0x9e,
			0x0c,0x59,0x25,0x3d,0x99,0xe9,0x0d,0x17,0x87,0x1d,0x48,0x96,0x74,0x69,0x56,0x63,
			0x62,0x61,0x66,0xd3,0x6f,0xf9,0x1d,0x8c,0x22,0x99,0xa2,0xf0,0x51,0xea,0xe2,0xd6,
			0x0e,0x8e,0xd0,0xbc,0x3f,0xac,0x1e,0x49,0x0b,0x47,0x0c,0x12,0xf3,0xd6,0x97,0xf6,
			0xfb,0xfd,0x88,0x0d,0xe2,0xe9,0x0e,0x9f,0xcb,0xd4,0x85,0xfa,0x33,0x93,0x19,0x83,
			0x72,0xfb,0x01,0xe4,0xce,0xc5,0xc1,0x59,0x17,0xec,0xdd,0x42,0xe5,0x7c,0x43,0xec,
			0xf5,0x5a,0x8c,0x0e,0xcb,0xdc,0xef,0x1b,0xce,0x4e,0x36,0xd9,0x6d,0x46,0xb1,0x12,
			0x57,0x0b,0x53,0xf8,0x2f,0x3d,0x20,0x64,0xb0,0x8a,0xc7,0x86,0x13,0x67,0x0a,0x28,
			0xea,0x69,0xd7,0x9c,0x71,0x7e,0xb1,0xc2,0x94,0x09,0x0d,0xbd,0x56,0x1f,0xa6,0xe5,
			0x04,0xd0,0x9d,0x26,0x57,0x24,0xe3,0x7a,0x2d,0xc6,0xf4,0x45,0xf6,0xf5,0x28,0xc9),
	},
};

START_TEST(test_sign_pss_no_salt)
{
	private_key_t *privkey;
	public_key_t *pubkey;
	chunk_t sig;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, SIGN_RSA_EMSA_PSS)) ||
		!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PRIVKEY_SIGN, SIGN_RSA_EMSA_PSS)))
	{
		warn("%N not supported, skip test", signature_scheme_names,
			 SIGN_RSA_EMSA_PSS);
		return;
	}
	privkey = lib->creds->create(lib->creds, CRED_PRIVATE_KEY, KEY_RSA,
								 BUILD_RSA_MODULUS, pss_sign_ns[_i].n,
								 BUILD_RSA_PUB_EXP, pss_sign_ns[_i].e,
								 BUILD_RSA_PRIV_EXP, pss_sign_ns[_i].d, BUILD_END);
	ck_assert(privkey != NULL);
	pubkey = privkey->get_public_key(privkey);
	ck_assert(pubkey != NULL);
	fail_unless(privkey->sign(privkey, SIGN_RSA_EMSA_PSS,
							  &pss_sign_ns[_i].params, pss_sign_ns[_i].m, &sig),
				"sign %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	ck_assert_chunk_eq(pss_sign_ns[_i].s, sig);
	fail_unless(pubkey->verify(pubkey, SIGN_RSA_EMSA_PSS,
							   &pss_sign_ns[_i].params, pss_sign_ns[_i].m, sig),
				"verify %N", signature_scheme_names, SIGN_RSA_EMSA_PSS);
	free(sig.ptr);
	pubkey->destroy(pubkey);
	privkey->destroy(privkey);
}
END_TEST

/**
 * Test vectors from FIPS 186-4 (only SHA-256)
 */
static struct {
	signature_scheme_t scheme;
	chunk_t n;
	struct {
		bool exp;
		chunk_t e;
		chunk_t m;
		chunk_t s;
	} tests[6];
} pkcs15_verify[] = {
{
	.scheme = SIGN_RSA_EMSA_PKCS1_SHA2_256,
	.n = chunk_from_chars(
			0xc4,0x7a,0xba,0xcc,0x2a,0x84,0xd5,0x6f,0x36,0x14,0xd9,0x2f,0xd6,0x2e,0xd3,0x6d,
			0xdd,0xe4,0x59,0x66,0x4b,0x93,0x01,0xdc,0xd1,0xd6,0x17,0x81,0xcf,0xcc,0x02,0x6b,
			0xcb,0x23,0x99,0xbe,0xe7,0xe7,0x56,0x81,0xa8,0x0b,0x7b,0xf5,0x00,0xe2,0xd0,0x8c,
			0xea,0xe1,0xc4,0x2e,0xc0,0xb7,0x07,0x92,0x7f,0x2b,0x2f,0xe9,0x2a,0xe8,0x52,0x08,
			0x7d,0x25,0xf1,0xd2,0x60,0xcc,0x74,0x90,0x5e,0xe5,0xf9,0xb2,0x54,0xed,0x05,0x49,
			0x4a,0x9f,0xe0,0x67,0x32,0xc3,0x68,0x09,0x92,0xdd,0x6f,0x0d,0xc6,0x34,0x56,0x8d,
			0x11,0x54,0x2a,0x70,0x5f,0x83,0xae,0x96,0xd2,0xa4,0x97,0x63,0xd5,0xfb,0xb2,0x43,
			0x98,0xed,0xf3,0x70,0x2b,0xc9,0x4b,0xc1,0x68,0x19,0x01,0x66,0x49,0x2b,0x86,0x71,
			0xde,0x87,0x4b,0xb9,0xce,0xcb,0x05,0x8c,0x6c,0x83,0x44,0xaa,0x8c,0x93,0x75,0x4d,
			0x6e,0xff,0xcd,0x44,0xa4,0x1e,0xd7,0xde,0x0a,0x9d,0xcd,0x91,0x44,0x43,0x7f,0x21,
			0x2b,0x18,0x88,0x1d,0x04,0x2d,0x33,0x1a,0x46,0x18,0xa9,0xe6,0x30,0xef,0x9b,0xb6,
			0x63,0x05,0xe4,0xfd,0xf8,0xf0,0x39,0x1b,0x3b,0x23,0x13,0xfe,0x54,0x9f,0x01,0x89,
			0xff,0x96,0x8b,0x92,0xf3,0x3c,0x26,0x6a,0x4b,0xc2,0xcf,0xfc,0x89,0x7d,0x19,0x37,
			0xee,0xb9,0xe4,0x06,0xf5,0xd0,0xea,0xa7,0xa1,0x47,0x82,0xe7,0x6a,0xf3,0xfc,0xe9,
			0x8f,0x54,0xed,0x23,0x7b,0x4a,0x04,0xa4,0x15,0x9a,0x5f,0x62,0x50,0xa2,0x96,0xa9,
			0x02,0x88,0x02,0x04,0xe6,0x1d,0x89,0x1c,0x4d,0xa2,0x9f,0x2d,0x65,0xf3,0x4c,0xbb),
	.tests = {
	{
		.exp = TRUE,
		.e = chunk_from_chars(0x49,0xd2,0xa1),
		.m = chunk_from_chars(
			0x95,0x12,0x3c,0x8d,0x1b,0x23,0x65,0x40,0xb8,0x69,0x76,0xa1,0x1c,0xea,0x31,0xf8,
			0xbd,0x4e,0x6c,0x54,0xc2,0x35,0x14,0x7d,0x20,0xce,0x72,0x2b,0x03,0xa6,0xad,0x75,
			0x6f,0xbd,0x91,0x8c,0x27,0xdf,0x8e,0xa9,0xce,0x31,0x04,0x44,0x4c,0x0b,0xbe,0x87,
			0x73,0x05,0xbc,0x02,0xe3,0x55,0x35,0xa0,0x2a,0x58,0xdc,0xda,0x30,0x6e,0x63,0x2a,
			0xd3,0x0b,0x3d,0xc3,0xce,0x0b,0xa9,0x7f,0xdf,0x46,0xec,0x19,0x29,0x65,0xdd,0x9c,
			0xd7,0xf4,0xa7,0x1b,0x02,0xb8,0xcb,0xa3,0xd4,0x42,0x64,0x6e,0xee,0xc4,0xaf,0x59,
			0x08,0x24,0xca,0x98,0xd7,0x4f,0xbc,0xa9,0x34,0xd0,0xb6,0x86,0x7a,0xa1,0x99,0x1f,
			0x30,0x40,0xb7,0x07,0xe8,0x06,0xde,0x6e,0x66,0xb5,0x93,0x4f,0x05,0x50,0x9b,0xea),
		.s = chunk_from_chars(
			0x51,0x26,0x5d,0x96,0xf1,0x1a,0xb3,0x38,0x76,0x28,0x91,0xcb,0x29,0xbf,0x3f,0x1d,
			0x2b,0x33,0x05,0x10,0x70,0x63,0xf5,0xf3,0x24,0x5a,0xf3,0x76,0xdf,0xcc,0x70,0x27,
			0xd3,0x93,0x65,0xde,0x70,0xa3,0x1d,0xb0,0x5e,0x9e,0x10,0xeb,0x61,0x48,0xcb,0x7f,
			0x64,0x25,0xf0,0xc9,0x3c,0x4f,0xb0,0xe2,0x29,0x1a,0xdb,0xd2,0x2c,0x77,0x65,0x6a,
			0xfc,0x19,0x68,0x58,0xa1,0x1e,0x1c,0x67,0x0d,0x9e,0xeb,0x59,0x26,0x13,0xe6,0x9e,
			0xb4,0xf3,0xaa,0x50,0x17,0x30,0x74,0x3a,0xc4,0x46,0x44,0x86,0xc7,0xae,0x68,0xfd,
			0x50,0x9e,0x89,0x6f,0x63,0x88,0x4e,0x94,0x24,0xf6,0x9c,0x1c,0x53,0x97,0x95,0x9f,
			0x1e,0x52,0xa3,0x68,0x66,0x7a,0x59,0x8a,0x1f,0xc9,0x01,0x25,0x27,0x3d,0x93,0x41,
			0x29,0x5d,0x2f,0x8e,0x1c,0xc4,0x96,0x9b,0xf2,0x28,0xc8,0x60,0xe0,0x7a,0x35,0x46,
			0xbe,0x2e,0xed,0xa1,0xcd,0xe4,0x8e,0xe9,0x4d,0x06,0x28,0x01,0xfe,0x66,0x6e,0x4a,
			0x7a,0xe8,0xcb,0x9c,0xd7,0x92,0x62,0xc0,0x17,0xb0,0x81,0xaf,0x87,0x4f,0xf0,0x04,
			0x53,0xca,0x43,0xe3,0x4e,0xfd,0xb4,0x3f,0xff,0xb0,0xbb,0x42,0xa4,0xe2,0xd3,0x2a,
			0x5e,0x5c,0xc9,0xe8,0x54,0x6a,0x22,0x1f,0xe9,0x30,0x25,0x0e,0x5f,0x53,0x33,0xe0,
			0xef,0xe5,0x8f,0xfe,0xbf,0x19,0x36,0x9a,0x3b,0x8a,0xe5,0xa6,0x7f,0x6a,0x04,0x8b,
			0xc9,0xef,0x91,0x5b,0xda,0x25,0x16,0x07,0x29,0xb5,0x08,0x66,0x7a,0xda,0x84,0xa0,
			0xc2,0x7e,0x7e,0x26,0xcf,0x2a,0xbc,0xa4,0x13,0xe5,0xe4,0x69,0x3f,0x4a,0x94,0x05),
	},
	{
		.exp = FALSE,
		.e = chunk_from_chars(0x49,0xd2,0xa1),
		.m = chunk_from_chars(
			0xf8,0x9f,0xd2,0xf6,0xc4,0x5a,0x8b,0x50,0x66,0xa6,0x51,0x41,0x0b,0x8e,0x53,0x4b,
			0xfe,0xc0,0xd9,0xa3,0x6f,0x3e,0x2b,0x88,0x74,0x57,0xaf,0xd4,0x4d,0xd6,0x51,0xd1,
			0xec,0x79,0x27,0x4d,0xb5,0xa4,0x55,0xf1,0x82,0x57,0x2f,0xce,0xea,0x5e,0x9e,0x39,
			0xc3,0xc7,0xc5,0xd9,0xe5,0x99,0xe4,0xfe,0x31,0xc3,0x7c,0x34,0xd2,0x53,0xb4,0x19,
			0xc3,0xe8,0xfb,0x6b,0x91,0x6a,0xef,0x65,0x63,0xf8,0x7d,0x4c,0x37,0x22,0x4a,0x45,
			0x6e,0x59,0x52,0x69,0x8b,0xa3,0xd0,0x1b,0x38,0x94,0x5d,0x99,0x8a,0x79,0x5b,0xd2,
			0x85,0xd6,0x94,0x78,0xe3,0x13,0x1f,0x55,0x11,0x72,0x84,0xe2,0x7b,0x44,0x1f,0x16,
			0x09,0x5d,0xca,0x7c,0xe9,0xc5,0xb6,0x88,0x90,0xb0,0x9a,0x2b,0xfb,0xb0,0x10,0xa5),
		.s = chunk_from_chars(
			0xba,0x48,0x53,0x87,0x08,0x51,0x2d,0x45,0xc0,0xed,0xca,0xc5,0x7a,0x9b,0x4f,0xb6,
			0x37,0xe9,0x72,0x1f,0x72,0x00,0x3c,0x60,0xf1,0x3f,0x5c,0x9a,0x36,0xc9,0x68,0xce,
			0xf9,0xbe,0x8f,0x54,0x66,0x54,0x18,0x14,0x1c,0x3d,0x9e,0xcc,0x02,0xa5,0xbf,0x95,
			0x2c,0xfc,0x05,0x5f,0xb5,0x1e,0x18,0x70,0x5e,0x9d,0x88,0x50,0xf4,0xe1,0xf5,0xa3,
			0x44,0xaf,0x55,0x0d,0xe8,0x4f,0xfd,0x08,0x05,0xe2,0x7e,0x55,0x7f,0x6a,0xa5,0x0d,
			0x26,0x45,0x31,0x4c,0x64,0xc1,0xc7,0x1a,0xa6,0xbb,0x44,0xfa,0xf8,0xf2,0x9c,0xa6,
			0x57,0x8e,0x24,0x41,0xd4,0x51,0x0e,0x36,0x05,0x2f,0x46,0x55,0x1d,0xf3,0x41,0xb2,
			0xdc,0xf4,0x3f,0x76,0x1f,0x08,0xb9,0x46,0xca,0x0b,0x70,0x81,0xda,0xdb,0xb8,0x8e,
			0x95,0x5e,0x82,0x0f,0xd7,0xf6,0x57,0xc4,0xdd,0x9f,0x45,0x54,0xd1,0x67,0xdd,0x7c,
			0x9a,0x48,0x7e,0xd4,0x1c,0xed,0x2b,0x40,0x06,0x80,0x98,0xde,0xed,0xc9,0x51,0x06,
			0x0f,0xaf,0x7e,0x15,0xb1,0xf0,0xf8,0x0a,0xe6,0x7f,0xf2,0xee,0x28,0xa2,0x38,0xd8,
			0x0b,0xf7,0x2d,0xd7,0x1c,0x8d,0x95,0xc7,0x9b,0xc1,0x56,0x11,0x4e,0xce,0x8e,0xc8,
			0x37,0x57,0x3a,0x4b,0x66,0x89,0x8d,0x45,0xb4,0x5a,0x5e,0xac,0xd0,0xb0,0xe4,0x14,
			0x47,0xd8,0xfa,0x08,0xa3,0x67,0xf4,0x37,0x64,0x5e,0x50,0xc9,0x92,0x0b,0x88,0xa1,
			0x6b,0xc0,0x88,0x01,0x47,0xac,0xfb,0x9a,0x79,0xde,0x9e,0x35,0x1b,0x3f,0xa0,0x0b,
			0x3f,0x4e,0x9f,0x18,0x2f,0x45,0x55,0x3d,0xff,0xca,0x55,0xe3,0x93,0xc5,0xea,0xb6),
	},
	{
		.exp = FALSE,
		.e = chunk_from_chars(0x49,0xd2,0xa1),
		.m = chunk_from_chars(
			0x91,0x5c,0x5e,0x4c,0x16,0xac,0xfa,0x0f,0x49,0xde,0x43,0xd6,0x49,0x1f,0x00,0x60,
			0xa9,0x44,0x03,0x44,0x75,0xba,0x51,0x85,0x72,0xc0,0x83,0x66,0xa8,0xd3,0x6c,0x7f,
			0x1e,0x6a,0xfc,0x11,0xe5,0xe4,0x64,0x97,0x57,0xbf,0x7b,0x9d,0xa1,0x0a,0x61,0xd5,
			0x7f,0x1d,0x62,0x68,0x47,0x87,0x1d,0x8a,0x29,0x48,0xe5,0x51,0xb5,0x41,0x67,0xc7,
			0x9d,0xe8,0x8d,0x3e,0xbd,0x40,0xa3,0xe3,0x58,0x09,0xb9,0x96,0xa5,0x33,0x48,0xf9,
			0x8a,0x99,0x18,0xc7,0xa7,0xec,0x60,0x68,0x96,0xed,0x30,0xc2,0x71,0xe0,0x0c,0x51,
			0x95,0x3d,0xd9,0x7a,0xa6,0xa8,0xfe,0x1c,0xd4,0x23,0xc3,0x69,0x5c,0x83,0xfc,0xf4,
			0x51,0x20,0xec,0x0a,0x9c,0xd1,0x64,0x46,0x42,0x18,0x2b,0x60,0xe5,0x99,0xa2,0x46),
		.s = chunk_from_chars(
			0x3d,0x57,0xea,0x59,0x61,0xdb,0x8f,0xc1,0x44,0x30,0x1c,0xa4,0x27,0x8f,0x79,0x99,
			0x11,0x22,0x9d,0x86,0x5e,0xa3,0xe9,0x92,0xc7,0xfb,0xc4,0xd0,0x3c,0x65,0x51,0x72,
			0x9e,0x26,0x03,0x4e,0x95,0xdd,0x71,0xda,0x31,0x23,0x40,0xe4,0x05,0x1c,0x9d,0xd9,
			0xb1,0x2f,0x77,0x00,0xa8,0x21,0xfe,0x3b,0x7c,0x37,0x78,0x5d,0x51,0x06,0x35,0x0b,
			0x66,0x7a,0xc2,0x55,0xa5,0x7c,0x13,0xda,0x58,0x42,0xd9,0x0b,0xca,0xde,0xa9,0xe6,
			0xb1,0xf7,0x20,0xc6,0x07,0xd6,0x89,0x3a,0x2c,0xaa,0x3c,0x5f,0x3c,0x40,0x74,0xe9,
			0x14,0x45,0x1a,0x45,0x38,0x0a,0x76,0x7c,0x29,0x1a,0x67,0xca,0xc3,0xf1,0xca,0xb1,
			0xfb,0xd0,0x5a,0xdc,0x37,0x03,0x68,0x56,0xa8,0x40,0x4e,0x7c,0xea,0x36,0x54,0x01,
			0x94,0x66,0xde,0x44,0x9a,0xd6,0xe9,0x2b,0x27,0x25,0x4f,0x3d,0x25,0x94,0x9b,0x1b,
			0x86,0x00,0x65,0x40,0x64,0x55,0xa1,0x3d,0xb7,0xc5,0xfe,0x25,0xd1,0xaf,0x7a,0x84,
			0xcd,0xdf,0x77,0x92,0xc6,0x4e,0x16,0x26,0x0c,0x95,0x0d,0x60,0xbd,0x86,0xd0,0x05,
			0x92,0x41,0x48,0xad,0x09,0x7c,0x12,0x6b,0x84,0x94,0x7a,0xb6,0xe8,0x9d,0x48,0xf6,
			0x1e,0x71,0x1d,0x62,0x52,0x2b,0x6e,0x48,0xf1,0x61,0x86,0xd1,0x33,0x9e,0x6a,0xb3,
			0xf5,0x8c,0x35,0x9e,0xb2,0x4c,0xb6,0x80,0x43,0x73,0x75,0x91,0xcd,0x7d,0x93,0x90,
			0xa4,0x68,0xc0,0x02,0x2b,0x3b,0x25,0x3b,0xe5,0x2f,0x1a,0x7f,0xc4,0x08,0xf8,0x4e,
			0x9f,0xfb,0x4c,0x34,0xfa,0x9e,0x01,0x60,0x58,0x51,0xd6,0x58,0x3a,0xa1,0x30,0x32),
	},
	{
		.exp = FALSE,
		.e = chunk_from_chars(0x07,0x48,0x5b),
		.m = chunk_from_chars(
			0x03,0xd2,0xf0,0x69,0x35,0x17,0xcf,0xfb,0x2b,0x72,0x4c,0x1f,0x30,0x50,0x2c,0x53,
			0x59,0xc0,0x51,0xc1,0xbc,0xd8,0x8d,0xc1,0xdd,0x54,0xb8,0x9e,0x69,0x81,0x00,0x9d,
			0x27,0x5a,0x81,0x3b,0x2b,0xf0,0x16,0xb7,0x4d,0x0f,0x6e,0xd0,0xd9,0x1e,0x62,0xd0,
			0x88,0x47,0x85,0xc9,0xaf,0xd8,0xfd,0x1f,0xb7,0xe9,0x92,0x46,0xcd,0x40,0x05,0xcd,
			0xda,0x71,0xa3,0x9c,0xb6,0x49,0x19,0x7a,0x99,0x6d,0x8a,0xd2,0xd2,0x3f,0xdf,0xb6,
			0xbb,0x01,0x5f,0x24,0xec,0x3d,0x7f,0x88,0xaf,0x64,0xfb,0x83,0xb4,0xb5,0x25,0xeb,
			0x06,0x60,0x7d,0x13,0x3e,0xec,0x83,0x4c,0xf7,0xd6,0xc9,0xab,0x81,0x7b,0x4c,0x0d,
			0xda,0x37,0x04,0x59,0xd9,0xcf,0xba,0x05,0xad,0x0c,0x1a,0xdc,0x86,0xa9,0x09,0xfe),
		.s = chunk_from_chars(
			0x51,0x1a,0xbd,0x82,0x21,0x8c,0xab,0x34,0x49,0x79,0xb2,0x88,0x7b,0x02,0x60,0x0d,
			0x24,0x27,0xf1,0xeb,0x12,0xac,0x01,0xd9,0x76,0x84,0xc2,0xa4,0x43,0xa9,0x27,0x28,
			0x34,0xc3,0xf7,0x9c,0xde,0xd0,0x7a,0x39,0xdb,0xee,0x37,0x70,0xdd,0xe8,0x27,0xa7,
			0x4d,0xc9,0x94,0xb1,0x7b,0xfd,0x8a,0x26,0xd0,0x7b,0x23,0x9d,0x26,0xd5,0x8c,0x42,
			0xf7,0x9d,0x56,0x02,0x64,0xc3,0x1b,0x7e,0x1c,0x3d,0xdd,0xef,0x6d,0x75,0x56,0xf2,
			0x28,0xc3,0x94,0x41,0x4f,0x4c,0xec,0x56,0x1c,0x3d,0xa2,0x68,0x6a,0x8e,0xeb,0xec,
			0x77,0x02,0xf3,0x28,0x50,0x80,0x9a,0x93,0xde,0xeb,0x84,0xb2,0xa0,0x2f,0xcd,0xba,
			0x22,0x4d,0x2f,0xd9,0xef,0xb8,0xe0,0x56,0xe7,0x96,0xf4,0x9b,0x57,0xd5,0x6e,0x9f,
			0x3e,0x90,0xd0,0xb4,0x9b,0x08,0xbd,0xee,0x93,0xa2,0xe1,0x2e,0x67,0x6f,0xb4,0xd4,
			0xfa,0x83,0x8c,0x5b,0xd8,0x8e,0xda,0x00,0x8f,0x1b,0x59,0x2a,0x72,0x46,0x55,0x87,
			0xbe,0x0a,0xe1,0x7d,0x9b,0x15,0x6b,0x90,0x4f,0x44,0xa7,0xe0,0x4d,0x3b,0x58,0xd2,
			0x4a,0xd6,0x7b,0x71,0xb0,0xf4,0xc6,0x99,0xfa,0x51,0x63,0x95,0x46,0xb6,0x2b,0x9f,
			0x83,0x59,0x7f,0xf0,0x3d,0x46,0x5f,0x1b,0xb3,0x96,0xae,0x15,0xe9,0x2d,0x0e,0x92,
			0xe8,0x56,0x47,0xd5,0xdf,0x11,0x3e,0x2c,0x75,0x18,0xd0,0xe3,0xad,0x2e,0x7a,0xa7,
			0xda,0xc7,0x20,0xc9,0x83,0x47,0xaa,0x15,0x1e,0x4f,0x37,0xfe,0xa0,0x81,0xdb,0xed,
			0x35,0x0c,0xc9,0xc9,0x3f,0x60,0x6b,0x38,0xf2,0x1a,0x3e,0x5d,0xe6,0xd1,0x40,0xd2),
	},
	{
		.exp = FALSE,
		.e = chunk_from_chars(0x49,0xd2,0xa1),
		.m = chunk_from_chars(
			0xdf,0xfe,0x42,0xbf,0xda,0x88,0x6e,0x1a,0x73,0xfe,0x8a,0x8d,0xfc,0xf7,0x1c,0x9f,
			0xb4,0x4d,0xeb,0x05,0x45,0x88,0xa9,0xbb,0x91,0x99,0xd5,0x54,0xae,0xcc,0xe0,0x8f,
			0x2f,0xf8,0x8f,0x2a,0xa6,0xf8,0xa0,0xfb,0x67,0x5f,0xb0,0x3c,0x8e,0x68,0x5c,0x27,
			0x43,0x2c,0xa7,0xc3,0x3c,0x18,0x9b,0xfd,0x84,0x9d,0x34,0xfa,0x7b,0x29,0x79,0xac,
			0x1f,0x57,0xec,0xa3,0x89,0x63,0x24,0x26,0xba,0xe0,0xb9,0x83,0x98,0xad,0x60,0xa3,
			0x34,0x25,0x57,0xe1,0x4e,0x96,0x04,0x1c,0x1b,0xf4,0xd9,0x0b,0x46,0xcf,0x7a,0xd1,
			0x34,0x83,0x22,0xd2,0x8c,0xaf,0x43,0xc4,0xf7,0xe8,0x6c,0x09,0x24,0xae,0x70,0x3c,
			0x10,0x9e,0xc5,0x0a,0x84,0xea,0x2a,0x43,0xdf,0x07,0x8c,0x30,0x15,0xa5,0x2b,0x28),
		.s = chunk_from_chars(
			0x8f,0x4d,0xd4,0x79,0x23,0x9f,0x2d,0x08,0xdc,0x05,0xd7,0xd4,0x05,0x39,0x28,0x8b,
			0x67,0xc4,0xd7,0x72,0x10,0xec,0xb1,0x6b,0xe7,0x6f,0x0b,0x19,0x25,0xe8,0xb0,0x88,
			0x57,0x08,0x31,0xe3,0x61,0xa1,0xca,0x57,0x89,0x31,0x35,0xf8,0xaf,0x64,0xb8,0xe2,
			0x99,0x6b,0x8d,0x63,0x58,0x99,0xda,0x4e,0x04,0xc6,0x8a,0xcb,0x9b,0x1b,0x38,0x13,
			0x69,0x7d,0x57,0xda,0x90,0xc5,0x7f,0x18,0x50,0x9e,0x0a,0xb6,0x70,0x5c,0x70,0x4f,
			0xeb,0x44,0x8c,0xca,0x5c,0x07,0xd2,0x58,0xec,0xd8,0x84,0xab,0x93,0xf5,0x08,0xce,
			0xfd,0xb2,0x5f,0x2b,0xc3,0x06,0x1c,0x40,0x06,0x09,0x9e,0x2e,0x33,0xb2,0x79,0x72,
			0xc3,0xed,0xb0,0xa0,0xa3,0x31,0x14,0xd3,0x81,0xc8,0x2a,0xb5,0x06,0xd0,0x41,0xff,
			0x68,0x0a,0xf5,0x95,0xef,0x34,0x00,0xa8,0xbb,0x67,0x74,0x03,0x0d,0x2e,0x38,0xdd,
			0x30,0x42,0x72,0x09,0x2b,0xd3,0x2a,0x55,0x30,0x17,0xf7,0xbd,0xa4,0xb9,0x98,0xb2,
			0x7a,0xa8,0xac,0xa1,0x2d,0xef,0x32,0x7b,0x1f,0x11,0x06,0x3a,0x53,0x42,0xb0,0xd5,
			0x57,0x38,0x18,0x34,0x17,0xd3,0x21,0xc5,0x68,0x2f,0xc4,0xab,0x64,0xe7,0x91,0x74,
			0x21,0x6f,0xee,0xbb,0x98,0x95,0x21,0xe1,0xe3,0xd8,0x27,0x64,0x70,0x68,0x00,0x3b,
			0xe3,0x4f,0xe1,0xd0,0x93,0x96,0x4d,0x28,0xf4,0x87,0x7c,0x49,0xb4,0x06,0x56,0x72,
			0x44,0x85,0x97,0xa8,0x9b,0x91,0x91,0x9c,0xfb,0x55,0xca,0x13,0x83,0x6e,0x7e,0x6f,
			0x3b,0x3f,0xd0,0x4f,0x41,0x7c,0xf1,0xc1,0x6d,0x98,0x72,0x53,0x8b,0xf4,0xe8,0x7a),
	},
	{
		.exp = FALSE,
		.e = chunk_from_chars(0x49,0xd2,0xa1),
		.m = chunk_from_chars(
			0xcf,0xe9,0x97,0x88,0xf5,0x5e,0xc6,0x94,0x49,0x42,0xbd,0x0a,0x18,0x7d,0x51,0xb8,
			0x0f,0xd8,0xbd,0x40,0x51,0xbd,0x4f,0x07,0xc7,0x3e,0x61,0x4e,0xb7,0x5a,0x8b,0x9f,
			0x99,0x7b,0x17,0x6b,0x26,0x42,0xb5,0xf1,0xb1,0x87,0x70,0x61,0xba,0x9c,0xe1,0x42,
			0xc1,0xd2,0xa3,0x11,0x58,0x3f,0x07,0x2b,0x7c,0xbe,0x08,0xed,0x25,0x36,0x81,0x19,
			0x1c,0x20,0x9d,0x7b,0x0d,0x43,0x8f,0xcd,0xdd,0xc2,0x84,0xd9,0x3d,0x59,0xd6,0xdd,
			0x80,0xe4,0x83,0x33,0xa9,0x21,0xdd,0x31,0xc9,0xb6,0x83,0x4f,0x88,0x76,0x8f,0x87,
			0x01,0xe0,0x11,0x02,0xd3,0xe8,0xbd,0xf0,0x74,0xfb,0xe0,0xb8,0xc9,0x3d,0x99,0x51,
			0xf4,0x15,0x45,0xef,0x6e,0xeb,0x3b,0xe3,0x55,0x30,0xba,0xbc,0x07,0x9f,0x1f,0xb3),
		.s = chunk_from_chars(
			0x9f,0xd6,0xf6,0x10,0x7e,0x83,0x81,0x07,0xf9,0x06,0xc2,0x6c,0xb2,0x91,0x07,0x04,
			0x59,0x9f,0x17,0x5b,0x6a,0x84,0xdb,0x48,0x5f,0xbc,0x30,0x77,0x6e,0xb7,0xfd,0x53,
			0xbf,0xe2,0x0c,0x38,0xc5,0x37,0xb1,0x54,0xa3,0xe5,0x19,0xb6,0x62,0xbd,0x9f,0xdc,
			0x8e,0x30,0x45,0xe2,0x1f,0x6e,0x5a,0xe9,0x7d,0x0f,0xf6,0xa9,0xd8,0x63,0x28,0x25,
			0x54,0x45,0x25,0xd8,0x4f,0x99,0xf8,0x0e,0x3e,0xd4,0xe6,0x9d,0xc5,0xe2,0x19,0xd5,
			0x9c,0xcf,0xbb,0x37,0xc2,0x3c,0x84,0xfe,0x3b,0x3e,0x6f,0xb2,0x2f,0x40,0x2f,0x94,
			0xe5,0x22,0x5c,0x63,0x87,0xfd,0xf8,0xbc,0xdb,0x35,0x08,0xf8,0x83,0x29,0x08,0xfe,
			0x05,0x77,0x15,0x21,0xe9,0x22,0x34,0x34,0x80,0x04,0xe8,0xfe,0x19,0xa8,0xf2,0x4b,
			0xeb,0xca,0xb9,0xf0,0x74,0x32,0x7c,0x88,0xd0,0x66,0xbc,0x12,0x08,0x17,0x48,0xd6,
			0x96,0xbe,0x61,0x35,0xc6,0xae,0xa3,0x22,0x20,0xea,0x78,0x6e,0xbd,0x78,0x00,0xe6,
			0x93,0x63,0x65,0xff,0x25,0x83,0x1c,0x28,0xcb,0x6c,0x8a,0x59,0x23,0x7f,0xf8,0x4f,
			0x5c,0xf8,0x90,0x36,0xcf,0xf1,0x88,0xee,0x0f,0x9a,0x61,0x95,0xf2,0xb1,0xac,0xa2,
			0xe4,0x44,0x2a,0xf8,0x36,0x9f,0x1b,0x49,0x32,0x2f,0xa2,0xf8,0x91,0xb8,0x3a,0x14,
			0xa9,0x7b,0x60,0xc6,0xae,0xaf,0xd6,0xc2,0x92,0x80,0x47,0xaf,0xfd,0xa9,0xc8,0xd8,
			0x69,0xff,0x52,0x94,0xbb,0x59,0x43,0xad,0x14,0xa6,0xd6,0x4e,0x78,0x4d,0x12,0x6c,
			0x46,0x9d,0x51,0xe2,0x92,0xb9,0xce,0x33,0xe1,0xd8,0x37,0x1b,0xa5,0xf4,0x67,0xb3),
	},
	},
},
};

START_TEST(test_verify_pkcs15)
{
	signature_scheme_t scheme = pkcs15_verify[_i].scheme;
	public_key_t *pubkey;
	bool result;
	int i;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, scheme)))
	{
		warn("%N not supported, skip test", signature_scheme_names, scheme);
		return;
	}
	for (i = 0; i < countof(pkcs15_verify[_i].tests); i++)
	{
		pubkey = lib->creds->create(lib->creds, CRED_PUBLIC_KEY, KEY_RSA,
							BUILD_RSA_MODULUS, pkcs15_verify[_i].n,
							BUILD_RSA_PUB_EXP, pkcs15_verify[_i].tests[i].e,
							BUILD_END);
		ck_assert(pubkey != NULL);
		result = pubkey->verify(pubkey, scheme, NULL,
								pkcs15_verify[_i].tests[i].m,
								pkcs15_verify[_i].tests[i].s);
		fail_unless(result == pkcs15_verify[_i].tests[i].exp,
					"verify %N [%d]", signature_scheme_names, scheme, i);
		pubkey->destroy(pubkey);
	}
}
END_TEST

/**
 * Test vectors from FIPS 186-4
 */
static struct {
	rsa_pss_params_t params;
	chunk_t n;
	struct {
		bool exp;
		chunk_t e;
		chunk_t m;
		chunk_t s;
	} tests[6];
} pss_verify[] = {
{
	.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256, .salt_len = 32, },
	.n = chunk_from_chars(
			0xa4,0x7d,0x04,0xe7,0xca,0xcd,0xba,0x4e,0xa2,0x6e,0xca,0x8a,0x4c,0x6e,0x14,0x56,
			0x3c,0x2c,0xe0,0x3b,0x62,0x3b,0x76,0x8c,0x0d,0x49,0x86,0x8a,0x57,0x12,0x13,0x01,
			0xdb,0xf7,0x83,0xd8,0x2f,0x4c,0x05,0x5e,0x73,0x96,0x0e,0x70,0x55,0x01,0x87,0xd0,
			0xaf,0x62,0xac,0x34,0x96,0xf0,0xa3,0xd9,0x10,0x3c,0x2e,0xb7,0x91,0x9a,0x72,0x75,
			0x2f,0xa7,0xce,0x8c,0x68,0x8d,0x81,0xe3,0xae,0xe9,0x94,0x68,0x88,0x7a,0x15,0x28,
			0x8a,0xfb,0xb7,0xac,0xb8,0x45,0xb7,0xc5,0x22,0xb5,0xc6,0x4e,0x67,0x8f,0xcd,0x3d,
			0x22,0xfe,0xb8,0x4b,0x44,0x27,0x27,0x00,0xbe,0x52,0x7d,0x2b,0x20,0x25,0xa3,0xf8,
			0x3c,0x23,0x83,0xbf,0x6a,0x39,0xcf,0x5b,0x4e,0x48,0xb3,0xcf,0x2f,0x56,0xee,0xf0,
			0xdf,0xff,0x18,0x55,0x5e,0x31,0x03,0x7b,0x91,0x52,0x48,0x69,0x48,0x76,0xf3,0x04,
			0x78,0x14,0x41,0x51,0x64,0xf2,0xc6,0x60,0x88,0x1e,0x69,0x4b,0x58,0xc2,0x80,0x38,
			0xa0,0x32,0xad,0x25,0x63,0x4a,0xad,0x7b,0x39,0x17,0x1d,0xee,0x36,0x8e,0x3d,0x59,
			0xbf,0xb7,0x29,0x9e,0x46,0x01,0xd4,0x58,0x7e,0x68,0xca,0xaf,0x8d,0xb4,0x57,0xb7,
			0x5a,0xf4,0x2f,0xc0,0xcf,0x1a,0xe7,0xca,0xce,0xd2,0x86,0xd7,0x7f,0xac,0x6c,0xed,
			0xb0,0x3a,0xd9,0x4f,0x14,0x33,0xd2,0xc9,0x4d,0x08,0xe6,0x0b,0xc1,0xfd,0xef,0x05,
			0x43,0xcd,0x29,0x51,0xe7,0x65,0xb3,0x82,0x30,0xfd,0xd1,0x8d,0xe5,0xd2,0xca,0x62,
			0x7d,0xdc,0x03,0x2f,0xe0,0x5b,0xbd,0x2f,0xf2,0x1e,0x2d,0xb1,0xc2,0xf9,0x4d,0x8b),
	.tests = {
	{	/* Format of the EM is incorrect - hash moved to left */
		.exp = FALSE,
		.e = chunk_from_chars(0x10,0xe4,0x3f),
		.m = chunk_from_chars(
			0x75,0x18,0xc8,0x5b,0x67,0xe7,0xae,0xf7,0xf2,0x6b,0xf0,0x06,0x89,0x9f,0xae,0xf7,
			0x6e,0x07,0x6f,0x0c,0x6c,0x94,0x6e,0x5d,0xc9,0xc8,0x35,0x21,0x77,0x1a,0x6d,0x29,
			0x8a,0x9c,0xf5,0xad,0xef,0xdb,0x31,0x4b,0x5a,0x07,0xa5,0x4d,0x80,0x54,0xc2,0x2b,
			0x87,0x9f,0xff,0x50,0xba,0x55,0x2c,0x21,0x82,0x91,0x03,0x3c,0x91,0x84,0x01,0xfd,
			0x61,0x1a,0x74,0x47,0xdd,0xda,0xd4,0x81,0x5e,0x0f,0x56,0xde,0xd8,0x25,0xbf,0xe2,
			0x56,0x55,0x76,0x22,0xa3,0x85,0xde,0x4b,0x4a,0x69,0xe2,0x65,0xc1,0xef,0xd2,0x59,
			0xe2,0xda,0x6d,0xb1,0x9a,0xac,0x3f,0xa0,0xe5,0xca,0x2d,0x42,0xfa,0xdb,0x4e,0x24,
			0xc2,0x71,0xfc,0x07,0x8f,0xeb,0x2b,0xe1,0x0b,0x9a,0xfa,0x25,0x6f,0x22,0x88,0x44),
		.s = chunk_from_chars(
			0x99,0x2d,0x48,0xb2,0x1b,0xb3,0xd2,0x21,0x9b,0x44,0xe8,0xfc,0xc8,0x63,0x3c,0xf3,
			0xae,0xb5,0x91,0xde,0x90,0xf4,0x38,0x64,0x96,0xac,0x7e,0xcd,0x28,0x4c,0xb6,0x3d,
			0x7d,0xff,0x81,0xa5,0x0b,0x8c,0x4f,0xed,0x9f,0x2e,0xf7,0x37,0x69,0x2e,0xa6,0xbe,
			0x05,0x24,0x8c,0xa1,0x38,0x94,0x7b,0x49,0xb4,0xe7,0xf3,0xcc,0xe6,0x64,0x0e,0x04,
			0x9a,0xc2,0x15,0x4c,0x40,0xf5,0x7e,0x22,0xfa,0x14,0xf9,0x7e,0x7a,0x95,0x07,0xe1,
			0xdc,0x98,0xb2,0x06,0xce,0x6e,0xa0,0xe1,0x80,0x03,0x91,0x99,0xd1,0xbe,0x0a,0x15,
			0xd1,0xf5,0x09,0x3a,0x45,0x9e,0x51,0x01,0xaa,0xca,0x2a,0x23,0xcb,0x1f,0x59,0xca,
			0xd2,0xf1,0xfb,0x99,0xdc,0x95,0x6b,0x9d,0x43,0x44,0xba,0xd2,0xc1,0x12,0x1d,0x63,
			0xb9,0x15,0x00,0x4a,0xcb,0xfc,0x7a,0xc6,0x0a,0xc9,0xa7,0xb0,0xb1,0xc6,0x81,0x2b,
			0x30,0xbf,0xe0,0x87,0xf7,0xf0,0xc7,0xd1,0x62,0x5f,0x9c,0x4f,0x45,0x85,0x15,0xe1,
			0x14,0x78,0xe3,0x60,0x4a,0xa3,0x9d,0x14,0xd0,0x8b,0xea,0x30,0xb0,0x1f,0xcd,0x61,
			0x89,0xe6,0xf9,0xb7,0x01,0xd3,0x60,0xe4,0x71,0x4d,0x45,0x55,0x6b,0x29,0x81,0x5c,
			0x8d,0x8f,0xa8,0xe4,0x6e,0x10,0x74,0x9b,0xa5,0xe8,0xd4,0x45,0xa4,0xc0,0xf4,0x87,
			0xe7,0x0a,0xb5,0x89,0x0b,0x7c,0xcc,0x16,0x51,0x28,0x2a,0x54,0xe8,0x7e,0x7d,0xb4,
			0xbb,0x2f,0x7d,0x4a,0x67,0x1e,0x71,0xc4,0x3c,0x55,0xcf,0x64,0x86,0x41,0x6f,0x17,
			0x1d,0x19,0x55,0x03,0x74,0x74,0xd0,0x6a,0x71,0xdd,0x07,0x87,0x67,0x84,0x8e,0x5d),
	},
	{
		.exp = TRUE,
		.e = chunk_from_chars(0x10,0xe4,0x3f),
		.m = chunk_from_chars(
			0xe0,0x02,0x37,0x7a,0xff,0xb0,0x4f,0x0f,0xe4,0x59,0x8d,0xe9,0xd9,0x2d,0x31,0xd6,
			0xc7,0x86,0x04,0x0d,0x57,0x76,0x97,0x65,0x56,0xa2,0xcf,0xc5,0x5e,0x54,0xa1,0xdc,
			0xb3,0xcb,0x1b,0x12,0x6b,0xd6,0xa4,0xbe,0xd2,0xa1,0x84,0x99,0x0c,0xce,0xa7,0x73,
			0xfc,0xc7,0x9d,0x24,0x65,0x53,0xe6,0xc6,0x4f,0x68,0x6d,0x21,0xad,0x41,0x52,0x67,
			0x3c,0xaf,0xec,0x22,0xae,0xb4,0x0f,0x6a,0x08,0x4e,0x8a,0x5b,0x49,0x91,0xf4,0xc6,
			0x4c,0xf8,0xa9,0x27,0xef,0xfd,0x0f,0xd7,0x75,0xe7,0x1e,0x83,0x29,0xe4,0x1f,0xdd,
			0x44,0x57,0xb3,0x91,0x11,0x73,0x18,0x7b,0x4f,0x09,0xa8,0x17,0xd7,0x9e,0xa2,0x39,
			0x7f,0xc1,0x2d,0xfe,0x3d,0x9c,0x9a,0x02,0x90,0xc8,0xea,0xd3,0x1b,0x66,0x90,0xa6),
		.s = chunk_from_chars(
			0x4f,0x9b,0x42,0x5c,0x20,0x58,0x46,0x0e,0x4a,0xb2,0xf5,0xc9,0x63,0x84,0xda,0x23,
			0x27,0xfd,0x29,0x15,0x0f,0x01,0x95,0x5a,0x76,0xb4,0xef,0xe9,0x56,0xaf,0x06,0xdc,
			0x08,0x77,0x9a,0x37,0x4e,0xe4,0x60,0x7e,0xab,0x61,0xa9,0x3a,0xdc,0x56,0x08,0xf4,
			0xec,0x36,0xe4,0x7f,0x2a,0x0f,0x75,0x4e,0x8f,0xf8,0x39,0xa8,0xa1,0x9b,0x1d,0xb1,
			0xe8,0x84,0xea,0x4c,0xf3,0x48,0xcd,0x45,0x50,0x69,0xeb,0x87,0xaf,0xd5,0x36,0x45,
			0xb4,0x4e,0x28,0xa0,0xa5,0x68,0x08,0xf5,0x03,0x1d,0xa5,0xba,0x91,0x12,0x76,0x8d,
			0xfb,0xfc,0xa4,0x4e,0xbe,0x63,0xa0,0xc0,0x57,0x2b,0x73,0x1d,0x66,0x12,0x2f,0xb7,
			0x16,0x09,0xbe,0x14,0x80,0xfa,0xa4,0xe4,0xf7,0x5e,0x43,0x95,0x51,0x59,0xd7,0x0f,
			0x08,0x1e,0x2a,0x32,0xfb,0xb1,0x9a,0x48,0xb9,0xf1,0x62,0xcf,0x6b,0x2f,0xb4,0x45,
			0xd2,0xd6,0x99,0x4b,0xc5,0x89,0x10,0xa2,0x6b,0x59,0x43,0x47,0x78,0x03,0xcd,0xaa,
			0xa1,0xbd,0x74,0xb0,0xda,0x0a,0x5d,0x05,0x3d,0x8b,0x1d,0xc5,0x93,0x09,0x1d,0xb5,
			0x38,0x83,0x83,0xc2,0x60,0x79,0xf3,0x44,0xe2,0xae,0xa6,0x00,0xd0,0xe3,0x24,0x16,
			0x4b,0x45,0x0f,0x7b,0x9b,0x46,0x51,0x11,0xb7,0x26,0x5f,0x3b,0x1b,0x06,0x30,0x89,
			0xae,0x7e,0x26,0x23,0xfc,0x0f,0xda,0x80,0x52,0xcf,0x4b,0xf3,0x37,0x91,0x02,0xfb,
			0xf7,0x1d,0x7c,0x98,0xe8,0x25,0x86,0x64,0xce,0xed,0x63,0x7d,0x20,0xf9,0x5f,0xf0,
			0x11,0x18,0x81,0xe6,0x50,0xce,0x61,0xf2,0x51,0xd9,0xc3,0xa6,0x29,0xef,0x22,0x2d),
	},
	{	/* Message changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x10,0xe4,0x3f),
		.m = chunk_from_chars(
			0xa3,0xbf,0x44,0xca,0xe8,0xaa,0x83,0x47,0xfd,0x07,0xd8,0x4a,0x33,0xee,0xc5,0xdb,
			0xbd,0xd7,0xb6,0x43,0x13,0x68,0x88,0x7c,0x98,0x8c,0x4b,0xe7,0x79,0xc5,0x47,0x3d,
			0xd8,0xc3,0x3e,0xc8,0x2a,0x35,0xf1,0xd3,0xdd,0xdf,0xe5,0x5f,0x3e,0xed,0x67,0x17,
			0x9b,0x87,0xce,0x86,0xa4,0xa5,0x00,0x88,0x17,0x25,0x38,0xfe,0x9d,0x1b,0x06,0xc6,
			0xef,0x68,0x97,0xeb,0x3c,0x8e,0x36,0x18,0xcf,0xc2,0x13,0x53,0xed,0x43,0x43,0xe7,
			0xfc,0xeb,0x09,0xa2,0xeb,0x03,0x54,0x41,0xcd,0x5c,0x88,0x29,0xc7,0x9b,0x81,0x58,
			0x2d,0xd5,0xd6,0x9a,0xe8,0x5c,0x5a,0x00,0x1b,0xd8,0xe9,0x8e,0x06,0x99,0x61,0x34,
			0x2a,0x2b,0xee,0x00,0xad,0x2b,0x8b,0x91,0x01,0x5a,0xc5,0xcf,0xc1,0xf0,0xc2,0xd9),
		.s = chunk_from_chars(
			0x87,0x7f,0x20,0xee,0xd6,0x0f,0x8c,0xe2,0x86,0x10,0x8a,0x5d,0xde,0x9b,0x68,0x28,
			0xb3,0x7e,0x3f,0xbd,0xb0,0x8f,0xe1,0x53,0xe5,0x91,0x51,0x38,0x97,0x44,0x0f,0x21,
			0xf8,0x12,0x14,0x59,0x8f,0xba,0x08,0xea,0x07,0x73,0x94,0xba,0x8c,0x2a,0x44,0xaa,
			0x4f,0x0d,0x8f,0x3a,0x5f,0xba,0xec,0x3d,0xc6,0x9b,0x3b,0xdd,0xfb,0xe2,0x83,0x97,
			0xc9,0x0a,0xdf,0x35,0xd0,0x8e,0xa7,0x71,0xc7,0xaa,0xa3,0x1e,0xb0,0x64,0x13,0xc1,
			0xc6,0x2b,0x77,0x61,0x8a,0xf9,0x40,0xf4,0xc7,0x18,0x59,0xfa,0x43,0x84,0xd2,0x9b,
			0x48,0xe5,0xcf,0xc9,0x41,0xd6,0x9b,0xf0,0xa3,0x80,0x4d,0x20,0x08,0xe7,0x58,0x74,
			0x2b,0x8e,0xd6,0x87,0x54,0xbc,0x71,0xd2,0x31,0x62,0x3d,0x18,0x13,0x47,0xc3,0x68,
			0x33,0xa7,0xd7,0x16,0x0f,0x74,0x2a,0x37,0xce,0x7d,0x43,0x2d,0x74,0x8e,0x51,0x4a,
			0xa7,0xd8,0x15,0x6b,0x50,0xc5,0x32,0x15,0x13,0x90,0xd0,0x86,0xcd,0xcf,0x9d,0x59,
			0xf1,0x22,0xc6,0xd9,0x7f,0x4c,0xcb,0x73,0x72,0x89,0xf7,0xb0,0x0a,0x23,0x7c,0xb6,
			0xb4,0xaa,0xe6,0xba,0x79,0xd4,0x1f,0xf7,0x3d,0x01,0x9a,0x26,0xb5,0x9a,0xde,0x04,
			0xc9,0x67,0x35,0x6e,0x2a,0xad,0x52,0xf1,0x15,0x35,0x7f,0xfb,0x76,0x76,0xf1,0x90,
			0xdb,0x38,0xdc,0xfc,0x98,0x66,0x6e,0x5b,0x25,0x85,0x59,0xc8,0xc8,0x5f,0xa3,0x29,
			0x42,0xcb,0xcb,0x99,0xd7,0x57,0xe8,0x84,0x7e,0x56,0xa1,0x68,0x7b,0x33,0x02,0x41,
			0x56,0x98,0x70,0x81,0x91,0xb1,0x36,0xd9,0x23,0x34,0x9b,0x02,0xfe,0x38,0xb6,0xbe),
	},
	{	/* Signature changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x10,0xe4,0x3f),
		.m = chunk_from_chars(
			0xe1,0xc4,0x6c,0x30,0x9b,0x63,0x66,0xfb,0x4d,0x56,0xac,0x08,0xc9,0x39,0x3c,0xee,
			0x9a,0x7c,0x95,0xbb,0xe7,0xb7,0xc0,0xe7,0x9a,0x3d,0x91,0x87,0xc0,0xf4,0x2b,0xc3,
			0x33,0x64,0xc2,0x8a,0x77,0x0d,0xa5,0x85,0xe3,0xfe,0x7b,0x49,0x01,0xa3,0xcc,0xd0,
			0x37,0xdf,0xc4,0x2a,0xa6,0x5a,0x34,0x70,0x52,0x1d,0xda,0xfa,0x83,0x5c,0xe2,0xd1,
			0x6c,0x92,0xac,0x67,0x0b,0xd4,0xd0,0x86,0x50,0x5e,0x60,0x87,0x81,0x73,0x6d,0xc4,
			0xdd,0x64,0xcc,0x50,0x80,0xee,0x19,0xe5,0x86,0xc8,0xfd,0x1d,0x73,0x7d,0xad,0xe5,
			0xd3,0x78,0xb3,0x2f,0x1d,0x5d,0xf1,0xe8,0xdd,0xa0,0xe3,0x2a,0x12,0x50,0x24,0xb2,
			0xd5,0x33,0x34,0x94,0x3c,0x18,0x78,0x2d,0x7e,0x69,0x82,0x5a,0x58,0x00,0x93,0xe7),
		.s = chunk_from_chars(
			0x8e,0xd1,0xf2,0x8f,0xd1,0x6d,0x45,0xd4,0x16,0xa2,0x15,0x54,0xe1,0x04,0xc0,0x06,
			0xfd,0x78,0x68,0xe5,0x89,0x5e,0x8b,0x99,0x83,0x1a,0xe0,0x93,0x81,0x35,0xb5,0x43,
			0x61,0x0d,0xf6,0x4a,0x8c,0x35,0x74,0xd0,0x81,0x18,0xbf,0xe3,0x96,0xf9,0xa5,0x60,
			0x9a,0x8d,0xbd,0xa2,0x1b,0x9a,0x85,0x30,0xff,0x0b,0xa9,0x0e,0x62,0x9d,0x6a,0xbe,
			0x30,0xd2,0xc1,0xb5,0x90,0x60,0x0d,0xb9,0x71,0xfc,0xda,0x80,0xe6,0xea,0xa8,0x40,
			0x17,0xe2,0x09,0xb9,0xbd,0x3b,0x64,0x1f,0x3c,0x81,0xd5,0xd2,0x7f,0x84,0x2b,0xec,
			0x80,0x19,0x79,0x0e,0xd9,0x9a,0x0e,0x5d,0xb4,0xae,0xdc,0x1c,0x07,0x0b,0x04,0x7c,
			0x19,0x41,0x0c,0xbc,0x56,0xe9,0xa0,0xff,0x12,0xd8,0xf6,0xe5,0xd7,0x37,0x1b,0x10,
			0x11,0xec,0xfe,0xcf,0x7b,0xe7,0xa7,0x4f,0x94,0x40,0x35,0x90,0xa5,0x2f,0x95,0x23,
			0x8d,0xd6,0x9e,0x0b,0x5f,0x4c,0x1f,0xcd,0xe9,0x7e,0xcf,0xdb,0x1a,0xcc,0x38,0x03,
			0xe5,0x9a,0xd8,0xb3,0x08,0x8b,0x2b,0xc5,0x09,0xe3,0xdd,0x12,0xd4,0x0d,0x87,0x56,
			0x25,0xdc,0x83,0x62,0xc5,0x79,0x17,0x67,0x99,0xc7,0x5e,0x4f,0xad,0xcd,0xb3,0x92,
			0xc6,0x8f,0x40,0x1f,0x68,0xd8,0x54,0xe4,0x63,0x77,0xf0,0x84,0xc0,0x81,0xf9,0xd8,
			0x37,0x43,0x03,0x9f,0x69,0x34,0x72,0x2e,0x30,0xef,0x3f,0x02,0x26,0xbc,0x84,0x1d,
			0x79,0xa4,0xeb,0x68,0xc5,0xcc,0xcb,0xb6,0xae,0x0e,0x92,0x00,0x44,0x4e,0x50,0xff,
			0x0d,0x09,0x53,0x04,0x7e,0xf9,0x55,0xd2,0xd3,0x9a,0x70,0xc3,0xb8,0x37,0xc5,0xf4),
	},
	{	/* Format of the EM is incorrect - 00 on end of pad removed */
		.exp = FALSE,
		.e = chunk_from_chars(0x10,0xe4,0x3f),
		.m = chunk_from_chars(
			0x92,0x5d,0x59,0xf9,0x53,0xcb,0x3f,0xfb,0x6d,0x5a,0x3a,0x55,0xc0,0x79,0xcb,0x10,
			0x83,0x99,0x75,0x36,0xe3,0x3d,0x7c,0x8a,0xed,0x50,0xed,0x76,0xae,0xbc,0xde,0x45,
			0x99,0x38,0xf7,0x92,0x29,0x61,0x32,0x00,0xc7,0x0d,0xde,0x2c,0xed,0xdc,0xea,0xe0,
			0x8c,0x10,0x60,0x8a,0xab,0x9e,0x30,0xec,0x51,0x84,0x2f,0x14,0xa6,0x5e,0x5f,0x8f,
			0x55,0x34,0x71,0xda,0x34,0x97,0x88,0x19,0x27,0xec,0x40,0x0b,0x42,0x07,0xef,0x3e,
			0x2d,0xfc,0x2b,0x7f,0xcd,0x31,0x8c,0x95,0x20,0xb8,0xb2,0x2f,0x69,0xdc,0x8a,0x1a,
			0x8e,0xfa,0xce,0xb7,0xbe,0x93,0xcb,0xad,0x56,0x9e,0x67,0xdb,0x06,0x23,0x62,0x91,
			0x30,0x05,0xdc,0xff,0x90,0x20,0x18,0xed,0x22,0x93,0x7f,0xad,0x40,0x5f,0xe8,0x4e),
		.s = chunk_from_chars(
			0x62,0xb0,0x7f,0x6d,0x1b,0x8f,0x13,0x65,0x1d,0x7f,0x22,0xce,0x2c,0xe0,0x10,0x61,
			0x09,0x00,0x29,0xdb,0x5a,0xf7,0xdc,0xec,0xfe,0xaf,0xef,0x20,0x62,0x1d,0xd9,0xe2,
			0x54,0xa0,0xfb,0x91,0x4f,0x76,0xa3,0xd7,0x96,0x62,0x25,0x74,0x89,0xcb,0x81,0x22,
			0x70,0x85,0x83,0xd3,0x07,0x78,0x79,0x1a,0x77,0xda,0x83,0xc7,0xbe,0xa8,0x11,0x40,
			0xc6,0x1e,0x4d,0x04,0x84,0x80,0x6e,0x20,0xfb,0x85,0xf2,0x4d,0x1b,0xbf,0x77,0x4e,
			0xf2,0xed,0x38,0x80,0x9c,0x9b,0x14,0xf2,0xa5,0x8c,0x6e,0x86,0x49,0xb7,0x60,0xba,
			0xa9,0x01,0x54,0x45,0x22,0xed,0x94,0xbd,0x40,0x5c,0x77,0x20,0x1d,0x07,0xc8,0xd1,
			0x28,0x64,0xa8,0xd1,0xe9,0x7a,0x4d,0x32,0x2c,0x29,0x99,0x4b,0x21,0x4f,0xc8,0x3c,
			0x2e,0xcd,0x5c,0x95,0x5b,0x9b,0xde,0xc4,0x24,0xe7,0xca,0x5a,0x13,0x25,0xec,0x0a,
			0xae,0x4a,0xb0,0xc2,0x02,0xb9,0x80,0xa2,0x18,0x7f,0x09,0x6a,0xaa,0xaa,0x5e,0x85,
			0x55,0x0e,0xbd,0x32,0x57,0x99,0xf4,0xf3,0x0e,0xf2,0xea,0xd0,0x7e,0x79,0xc7,0xa4,
			0x75,0x66,0x7f,0x59,0x65,0xe6,0xb5,0x02,0x69,0x51,0x36,0x59,0xab,0x59,0x62,0xa3,
			0x91,0xc4,0x3c,0xbc,0x3a,0x3d,0xa3,0x4c,0x0f,0xdd,0x15,0x46,0xc4,0x0e,0xa7,0xe2,
			0xeb,0x53,0x52,0xce,0x6a,0x06,0xce,0x6a,0x63,0x85,0xca,0x0d,0xdd,0x5d,0x16,0x2c,
			0x13,0x78,0x36,0xdf,0x9e,0xa1,0xf8,0x9c,0xfa,0x00,0xc3,0xeb,0x16,0x71,0xa4,0x3b,
			0xd6,0x25,0x52,0x6f,0x3b,0x6b,0xa8,0xe4,0x8a,0x7a,0x2d,0x56,0xfb,0x4f,0x01,0xc6),
	},
	{	/* Public Key e changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x3c,0x6c,0xd1),
		.m = chunk_from_chars(
			0x08,0xd3,0xac,0x24,0xa5,0x95,0xda,0x81,0x1c,0xc9,0xbb,0xa7,0x88,0x28,0xf1,0x45,
			0x2f,0xf3,0x90,0xae,0x65,0x3f,0x22,0xd1,0xad,0x91,0xef,0x6b,0x22,0xaa,0x7b,0x7b,
			0x15,0xa4,0x4f,0xf7,0xf8,0x3e,0xfc,0xbd,0x7a,0x75,0x5e,0xcc,0xf4,0x54,0x1e,0xb4,
			0x04,0x0c,0x7a,0x4b,0x77,0x47,0x49,0xa2,0x6d,0xba,0x39,0x37,0xb7,0xf9,0x5c,0x6c,
			0x84,0x90,0xe3,0x38,0x3d,0x4f,0x29,0x1d,0xad,0xe5,0xf3,0x5a,0x65,0xb1,0xf6,0x61,
			0x5f,0xd4,0x99,0x8b,0xe1,0x8b,0xfd,0x0b,0xa4,0xbc,0x3a,0x21,0x36,0xce,0xec,0x90,
			0x9d,0xbe,0xef,0x51,0x3d,0x6f,0x66,0x89,0xfc,0x42,0x02,0xb9,0xa3,0xe7,0x81,0x34,
			0x87,0x73,0x74,0xd7,0x6a,0xb2,0x46,0xf4,0x9c,0xbe,0x7a,0x8f,0x65,0xd0,0x34,0xcd),
		.s = chunk_from_chars(
			0x72,0xea,0x7d,0x66,0x9a,0xc6,0x99,0xa1,0x49,0xc1,0x3a,0xa1,0x68,0xeb,0x3e,0x14,
			0x8f,0x59,0x0b,0xbc,0x04,0x24,0x95,0x1f,0x1c,0x31,0x44,0xc5,0x41,0x91,0x55,0x73,
			0xec,0xf0,0x5c,0xab,0xb5,0x27,0x54,0x88,0xc9,0x8d,0x0e,0xbe,0x55,0xb1,0xf1,0xaf,
			0x49,0xef,0x8c,0xc7,0x2c,0xf0,0x0e,0x84,0x98,0x97,0xc6,0xaf,0xa5,0x38,0x33,0xca,
			0xf9,0xff,0xc0,0x0e,0xb8,0x4c,0xfe,0x55,0xb8,0x27,0x7e,0x93,0xa6,0xac,0x3e,0x04,
			0xb1,0x11,0xc9,0xd0,0x7a,0xd6,0x20,0x82,0xca,0x32,0x85,0x4c,0x40,0xde,0x1e,0x09,
			0x56,0xa1,0x8f,0x84,0x60,0x43,0x84,0x84,0x40,0x3a,0xbb,0x91,0xe6,0xfa,0xd1,0x2b,
			0x28,0xf1,0x14,0x15,0x7f,0x1b,0x6e,0xf5,0x63,0xd2,0xfa,0x43,0x8c,0xcb,0x8a,0x16,
			0xaa,0x3a,0xfe,0x6e,0xc5,0xe9,0x83,0x65,0xc3,0x0d,0x9a,0x35,0x5e,0x7e,0x14,0xb0,
			0x39,0xbb,0xde,0xcf,0x59,0xf9,0x12,0x48,0xd0,0xf1,0xe3,0x17,0xd2,0xe3,0xc8,0x19,
			0x09,0x6e,0xd6,0x8c,0xf9,0x83,0x61,0xd7,0xf9,0x31,0x0b,0x35,0x4f,0x4f,0x45,0xe2,
			0xf7,0xc4,0xdf,0xb8,0x2c,0x21,0x8c,0xd4,0x38,0xf3,0x17,0x4a,0x6f,0x01,0x7f,0x0f,
			0x14,0x8a,0x71,0xc4,0x0a,0xd5,0xb7,0xe2,0xc8,0x08,0x1d,0xdb,0xae,0xaa,0x76,0x4d,
			0x60,0x1b,0xb2,0xe0,0x75,0xeb,0x36,0x18,0xaa,0xba,0xa4,0xb5,0x06,0x00,0x22,0xc6,
			0xd2,0x3e,0xf0,0x72,0x64,0x36,0x86,0xa7,0x59,0x22,0x7d,0xae,0xf0,0x33,0x1b,0x28,
			0xee,0x76,0xda,0x4a,0xb3,0xb5,0xad,0x5d,0xda,0x89,0x91,0xb9,0x93,0x7d,0x1a,0x59),
	},
	},
},{
	.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, .salt_len = 64, },
	.n = chunk_from_chars(
			0xa6,0x77,0x52,0x5e,0x1a,0x69,0x54,0x6a,0x96,0xdc,0x7b,0x11,0x23,0x50,0xd5,0xe4,
			0x86,0x4f,0x0f,0x82,0xe9,0x99,0xa7,0x14,0xfa,0x9f,0x43,0xac,0x68,0x15,0x17,0xd3,
			0x97,0x59,0x10,0xc2,0xd8,0x06,0xbb,0x3e,0xe6,0xdb,0xf5,0xdb,0xa1,0xd9,0x69,0xb3,
			0x88,0x89,0xe1,0x13,0xc2,0xda,0x76,0xee,0xf4,0x41,0x2a,0x60,0xcb,0xd8,0x9f,0xaf,
			0x35,0xb2,0xbd,0xcb,0x0d,0xe3,0x6a,0x2c,0xb7,0x62,0xcd,0x8f,0x2f,0x29,0xae,0xd9,
			0x98,0x2a,0x9a,0xb6,0x08,0x86,0xcc,0x8f,0xbf,0xee,0x9b,0x2a,0xe0,0x9c,0x88,0x16,
			0x1e,0x91,0x59,0xd4,0xfc,0x83,0x3a,0xdc,0x4f,0x80,0xe4,0xbf,0x62,0x9d,0x5a,0x95,
			0x51,0xac,0xce,0x7a,0x39,0x38,0x63,0x0c,0x2b,0xf9,0x95,0x60,0x97,0x64,0x2e,0x3b,
			0xc6,0x0a,0xc6,0x52,0x20,0x17,0x84,0x1b,0x65,0xc7,0xa2,0x51,0x97,0x86,0x5e,0x69,
			0x77,0x53,0xb0,0x81,0x69,0x85,0x36,0x81,0x91,0x14,0x43,0xa2,0xb2,0x5f,0x1b,0x7c,
			0x46,0x96,0xf9,0x46,0x15,0x5b,0x26,0x64,0xb6,0x7b,0x40,0x87,0x8d,0x3b,0x45,0xc3,
			0xe0,0xd7,0x03,0x4d,0x5b,0x5e,0xe6,0xf5,0xba,0x8f,0xb3,0xca,0xe7,0x79,0x7e,0x85,
			0x78,0x99,0x02,0xcf,0x8f,0x9f,0x86,0xed,0x3e,0xf2,0x5a,0xe0,0x73,0x61,0x78,0xaa,
			0xe2,0x60,0xfe,0x87,0x5b,0xfe,0xf5,0xbc,0xde,0x9e,0xc0,0x5f,0x11,0xe1,0x8f,0xc7,
			0x37,0x5e,0xdc,0xd4,0xa5,0x53,0x36,0x18,0xe6,0xf9,0x91,0xdd,0x48,0xaa,0x30,0x62,
			0xe6,0x03,0x1e,0x29,0x1d,0xfc,0xdc,0x6e,0x7f,0xc1,0x4e,0xc6,0x0e,0x53,0x9f,0xcb),
	.tests = {
	{	/* Public Key e changed */
		.exp = FALSE,
		.e = chunk_from_chars(0xea,0xc8,0x39),
		.m = chunk_from_chars(
			0x9c,0x3d,0x5d,0x23,0xd2,0x74,0x6d,0x15,0xd6,0x16,0xbe,0xbf,0x3c,0xf7,0x20,0xc6,
			0xe6,0x01,0x2a,0x71,0xca,0xe2,0x20,0x02,0xf5,0x02,0x1a,0x47,0xd0,0xb8,0x63,0x6c,
			0xa3,0xbd,0x20,0x13,0x57,0xe1,0x32,0xa6,0x80,0xfc,0x5d,0xec,0x9b,0x28,0xa9,0xdb,
			0x93,0x2d,0x08,0xae,0x8b,0x3d,0x3a,0x37,0xd7,0xe2,0xee,0x75,0x4b,0x34,0x2a,0x69,
			0xb9,0x4f,0xec,0x26,0xb5,0x04,0x12,0x28,0x9b,0xcf,0x77,0xe6,0xd4,0x09,0x5f,0xaa,
			0x54,0x5f,0x15,0xa1,0x67,0x83,0xd2,0x2e,0xae,0x21,0xe1,0x84,0x64,0x15,0x01,0x74,
			0xe6,0xdb,0x0b,0x83,0x73,0x47,0xd4,0x40,0x30,0x76,0x55,0xd5,0x6f,0x04,0x09,0xdb,
			0x30,0x7f,0x97,0x73,0xe8,0x1c,0xb1,0x92,0x82,0xa9,0x3c,0x9c,0xa4,0xc3,0xb1,0x35),
		.s = chunk_from_chars(
			0x98,0x65,0x7f,0xd8,0x16,0x39,0x67,0xfa,0x7d,0x26,0x3b,0xd4,0x5b,0xb8,0x90,0x03,
			0x5a,0xdb,0xcd,0xd1,0x64,0x5f,0xd4,0x8b,0x28,0xfe,0xbf,0xb9,0xb4,0xe1,0x51,0x72,
			0x54,0x0e,0x38,0xb7,0xc2,0xf6,0x73,0xc4,0x0a,0x20,0x5f,0xd4,0x0b,0x08,0xb6,0x0b,
			0x4b,0x81,0xed,0x6e,0x23,0x6c,0xdf,0x08,0xf0,0xd6,0xb1,0x1f,0x50,0xdc,0x74,0xc6,
			0x0d,0xc4,0x66,0xac,0x37,0x2e,0x0f,0x46,0x78,0x83,0xaa,0x9a,0x39,0x8f,0x4a,0xee,
			0xf8,0x7b,0x04,0x0e,0x14,0xa5,0x15,0x02,0xdd,0x46,0x7e,0x8e,0x8d,0xd8,0x98,0x12,
			0xdf,0xaf,0x6b,0x1d,0xc1,0xc2,0xf6,0xc2,0x84,0x48,0xaf,0x08,0x45,0x90,0xc0,0x5a,
			0xec,0x49,0x9d,0xd3,0xb1,0x48,0xe6,0x6f,0x3d,0x71,0xcf,0x75,0xe2,0x39,0xdb,0x6d,
			0x21,0xf4,0x07,0x4b,0x8b,0xd9,0xa6,0xbd,0xe5,0xca,0x66,0x86,0x34,0xbd,0x47,0x95,
			0x32,0x76,0xff,0x2d,0x0e,0xbb,0xe0,0x1a,0xfc,0xfe,0x0e,0x38,0x19,0x03,0x73,0x6d,
			0x6a,0x6c,0x67,0x2a,0x45,0xfb,0xa4,0xee,0x32,0x6e,0x34,0x2d,0xc5,0x92,0x51,0x69,
			0x51,0x7c,0x5f,0x57,0xe9,0x29,0x07,0x24,0x57,0x6a,0x22,0x5b,0xa8,0x9c,0xb4,0xdd,
			0x09,0x1f,0x4e,0x65,0x13,0xbe,0x10,0xdd,0x41,0x81,0x85,0x5b,0xb4,0x04,0x5d,0x6e,
			0xf6,0x43,0x7c,0x16,0xd3,0xb5,0x58,0x9e,0xf9,0xd6,0x83,0x66,0x82,0x71,0x1c,0x7d,
			0x66,0x02,0x5a,0xe3,0x7b,0x52,0x55,0x80,0xf0,0xdf,0xcf,0x3d,0xb7,0xfe,0x57,0xd7,
			0xc6,0xb1,0x57,0x77,0xcc,0x41,0x60,0x03,0x07,0xe5,0x8a,0x17,0x21,0xb6,0xf7,0xbc),
	},
	{	/* Format of the EM is incorrect - hash moved to left */
		.exp = FALSE,
		.e = chunk_from_chars(0x8d,0x88,0x53),
		.m = chunk_from_chars(
			0x77,0x59,0x42,0x0c,0x8d,0x1a,0x39,0xba,0x0e,0x3e,0x16,0x81,0xd9,0xb7,0x57,0xfc,
			0xf3,0x0c,0xb4,0x0d,0x84,0x4f,0x4a,0x71,0x22,0x4f,0xf9,0x98,0x52,0x0e,0x0b,0xb4,
			0x4c,0x76,0x16,0x11,0xff,0x47,0x23,0xa4,0x55,0x54,0x0f,0xe6,0xb8,0xef,0x32,0xe3,
			0xd5,0xa2,0x3f,0x8d,0xa0,0x20,0x6b,0x57,0x98,0x7a,0xd2,0xff,0x4c,0x96,0x16,0xab,
			0x1e,0xde,0x49,0x38,0x47,0x35,0x0d,0xcf,0x1b,0x2f,0xf9,0xb9,0x8e,0x81,0x3f,0x74,
			0xf8,0xb6,0x8c,0x06,0x15,0x24,0x30,0x91,0x50,0x1f,0x7f,0x28,0x41,0x6c,0x77,0xe1,
			0x74,0xf8,0xa3,0x2b,0x6c,0xca,0x3e,0x62,0xe7,0x37,0x9c,0xa1,0x64,0x55,0xc6,0xd8,
			0xe3,0xb2,0x65,0x1e,0xec,0x45,0xe1,0x48,0xd2,0xa4,0xc9,0xba,0x39,0x78,0x76,0x7d),
		.s = chunk_from_chars(
			0x91,0xce,0x8e,0x92,0x1f,0x85,0xde,0x84,0x5b,0x9b,0xf7,0x50,0x3d,0xc6,0x5a,0x32,
			0x82,0x24,0xd1,0x5b,0x23,0xe8,0xea,0x65,0xdc,0x3d,0xa6,0xd8,0x04,0xfb,0xef,0x19,
			0x6c,0x30,0x5e,0x0f,0xba,0x05,0x9a,0x7d,0x84,0x2c,0x8c,0x0f,0x58,0x08,0x60,0xcc,
			0xe7,0x76,0x52,0xd4,0xdc,0x77,0x90,0x73,0xd6,0xa3,0xb0,0xa9,0x2b,0xf2,0xdd,0x3e,
			0xc6,0xc0,0x61,0x8d,0xa1,0x48,0x50,0x36,0xfd,0x59,0x4b,0xd9,0x62,0xf7,0xb5,0xa1,
			0xc6,0x1d,0x53,0xf1,0x62,0x69,0x12,0xc7,0xe3,0x12,0xaa,0x83,0xeb,0x43,0x26,0x28,
			0x8a,0x52,0x43,0x7c,0x73,0x87,0x91,0x6b,0xcb,0x11,0x08,0x4a,0x40,0xc2,0x19,0xc8,
			0x4c,0xab,0x2f,0xf6,0x6e,0x83,0x14,0x3f,0x85,0x86,0x47,0xf2,0xdc,0x91,0x4d,0xa0,
			0xc7,0x85,0x8b,0xa3,0x9a,0xa8,0xad,0x81,0x47,0xea,0x5c,0x9c,0x52,0x06,0x9e,0xb2,
			0xb2,0xca,0x94,0x1d,0x8a,0x50,0x23,0x74,0x8d,0xc4,0x01,0xe7,0x66,0x45,0x00,0x84,
			0x6c,0x85,0xca,0xdf,0x02,0x87,0xe3,0x9b,0x8b,0x8b,0x46,0x79,0x4a,0x82,0xbf,0xff,
			0xa7,0x52,0x96,0x0a,0xb3,0xb9,0xea,0xa0,0x25,0x30,0xb5,0x90,0x3c,0x24,0xae,0xfa,
			0x10,0x8f,0x10,0xae,0xe3,0x5b,0x6a,0x8a,0x3c,0xdf,0x87,0x95,0xa1,0x82,0xe5,0xc0,
			0x4e,0xd3,0x6b,0xb6,0x8c,0x18,0x78,0x56,0x23,0xb4,0x2f,0x09,0x79,0xff,0xf3,0x19,
			0x11,0x2e,0x15,0x14,0xce,0xe5,0x1d,0xb4,0x37,0xee,0x32,0x28,0x9f,0x61,0xde,0xd6,
			0x11,0x86,0xc6,0xed,0x44,0xad,0x72,0x2d,0x18,0xd5,0x56,0x82,0x1f,0x88,0x82,0x21),
	},
	{	/* Message changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x8d,0x88,0x53),
		.m = chunk_from_chars(
			0x86,0x4f,0x4f,0x58,0x90,0xa0,0x36,0x53,0xd0,0x8d,0xaa,0x83,0xd9,0xe2,0x99,0x2f,
			0xab,0x33,0x93,0xb7,0xee,0x15,0x2d,0x6f,0x7b,0x8e,0xa3,0xad,0x1e,0xc9,0xc1,0x21,
			0x9a,0x0e,0x03,0x65,0xee,0xf2,0xfb,0xd5,0xd7,0xcb,0xf3,0xa1,0x96,0x67,0xd4,0x21,
			0xf3,0xfe,0x46,0x68,0x82,0x64,0x19,0x1b,0x25,0x83,0xe4,0x84,0xd2,0x09,0xeb,0xe2,
			0x97,0x5e,0x46,0x03,0xdb,0xd9,0x40,0x15,0xe6,0x33,0xfe,0xba,0xa4,0x3c,0x61,0x5a,
			0xa5,0xcc,0x2c,0xbd,0x69,0xfd,0x6a,0xd9,0xdb,0x97,0x0a,0xe8,0x1b,0xcc,0xcd,0xbe,
			0x80,0x26,0x62,0x5a,0xde,0xbd,0xcf,0xbb,0x04,0xad,0x12,0xf5,0x89,0xc7,0x08,0x83,
			0xc9,0x87,0x8f,0xdf,0x2a,0x46,0xe7,0x50,0xa4,0xb1,0x5f,0x03,0x80,0x7a,0x2d,0x5e),
		.s = chunk_from_chars(
			0x24,0x5f,0xda,0x2e,0x66,0xd2,0xba,0xb1,0x02,0xd0,0xda,0x40,0x98,0xd7,0x51,0x65,
			0x0f,0x5e,0xf4,0x73,0x8d,0x14,0xda,0xbe,0x86,0x6c,0x78,0x20,0x09,0xea,0xd9,0xd6,
			0xd0,0x5d,0x43,0xac,0x0a,0x30,0x14,0xa3,0x82,0xd6,0x2a,0xc9,0x71,0xee,0x19,0x79,
			0x8f,0x3b,0x8e,0xf4,0x9f,0x9b,0xc2,0x47,0xfa,0x07,0xb3,0x68,0xac,0x2d,0x0e,0x8a,
			0x4f,0xb2,0x10,0xaf,0x5a,0xcd,0x87,0xb6,0xb5,0x12,0xe1,0x35,0x86,0x1b,0x30,0x10,
			0x34,0x03,0x30,0x47,0x32,0xa1,0x0b,0x75,0xc9,0xa1,0x49,0x55,0x2d,0x1c,0xcf,0x0f,
			0x99,0x12,0x96,0x8c,0x64,0x31,0x39,0x8b,0x7d,0x48,0xdf,0xf4,0x8c,0xc7,0xcb,0x71,
			0x84,0xe7,0x6a,0x0a,0xb9,0x4a,0xc9,0xa6,0xb0,0x03,0x5b,0x91,0xb3,0xee,0xb2,0xd1,
			0x32,0x6e,0xbd,0xdf,0xc1,0x32,0x50,0x22,0x45,0x38,0xc1,0x43,0xfc,0x0d,0xfa,0x39,
			0x95,0x92,0x64,0x15,0xa6,0x58,0xdb,0x8e,0x66,0x91,0x50,0x40,0x50,0xa1,0xa3,0x0e,
			0x34,0xb2,0x1a,0xf8,0xd1,0x27,0x07,0xbc,0x44,0x54,0xf1,0x74,0x38,0x5c,0xd8,0x4c,
			0x72,0x9e,0x2d,0xdf,0xc5,0xcc,0x91,0x78,0x81,0x20,0x59,0xb5,0xda,0x4a,0x1a,0x60,
			0x5a,0x2a,0x81,0x64,0x2c,0xcc,0x1b,0xc5,0xcf,0x95,0x39,0x65,0xf7,0x05,0x67,0x3b,
			0x12,0xc5,0xb4,0x6a,0xc8,0xc1,0xa0,0x3c,0x10,0x01,0xae,0x4d,0xf1,0xc3,0x9c,0xc8,
			0x3c,0xb5,0x4b,0x10,0xcb,0x7d,0x33,0xcd,0x51,0x08,0x03,0x3c,0x26,0x8c,0xee,0xeb,
			0xba,0x6c,0x6a,0xda,0x92,0x88,0x2f,0x46,0x56,0x2d,0xc6,0x58,0xd3,0x52,0x55,0x74),
	},
	{	/* Signature changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x8d,0x88,0x53),
		.m = chunk_from_chars(
			0x71,0xab,0xf8,0x95,0xe5,0x6b,0xa8,0xf6,0xae,0x8d,0x0e,0x9e,0x66,0x90,0xc0,0x9c,
			0x75,0x92,0x70,0xa7,0x3d,0xb8,0xc1,0xaa,0x95,0xd0,0x59,0x80,0x79,0x35,0x37,0xfb,
			0xff,0xf3,0x47,0x2c,0x8d,0x2c,0x34,0xde,0x4a,0xbb,0x7e,0x64,0xd2,0x16,0xcc,0x95,
			0x2e,0x79,0x83,0x14,0x03,0x41,0x97,0xd5,0x09,0x96,0xa2,0xdc,0xbf,0x4c,0x33,0x48,
			0x5e,0x0b,0x68,0x91,0x0b,0xae,0xbf,0x0e,0x50,0xea,0x29,0xba,0xcd,0x30,0x60,0x37,
			0x2b,0xd4,0x7b,0x13,0x52,0x6e,0xc0,0x4b,0xdc,0x81,0xb9,0x0d,0xc9,0x5a,0x8a,0xc2,
			0x74,0x3b,0x81,0x4c,0xc5,0xb9,0xef,0x8c,0xa9,0x63,0x36,0x28,0xbf,0xa4,0x24,0x8b,
			0x55,0xeb,0x7f,0x2d,0x92,0x08,0xe1,0x14,0xf4,0xda,0xc6,0x9b,0xfe,0x27,0xe4,0x72),
		.s = chunk_from_chars(
			0x24,0x3c,0x57,0x0c,0xd1,0xbd,0x85,0x52,0x2e,0x64,0x81,0x73,0x61,0xcb,0xb9,0x49,
			0xc8,0x38,0x76,0xd7,0xa3,0x60,0x95,0x13,0xae,0xbf,0x50,0x6f,0xc0,0x67,0x50,0xba,
			0x38,0xe8,0x20,0x23,0x2c,0x98,0xac,0xbe,0x00,0xff,0xa5,0x0b,0x36,0xc5,0x38,0x56,
			0x00,0x69,0xfd,0x13,0x99,0x2c,0xa0,0xdf,0x7d,0x9f,0x6d,0x95,0x6b,0x88,0x33,0x34,
			0xf1,0xc3,0xaf,0x8a,0x74,0xad,0xee,0xd5,0xbe,0x15,0xb0,0x33,0xd8,0x5b,0x20,0x7b,
			0xe2,0xfe,0x60,0x0a,0x99,0x02,0xd6,0x99,0xcc,0xef,0x1f,0x4a,0x8d,0x58,0x86,0x72,
			0xc4,0x46,0x40,0x1e,0xb3,0x90,0xd1,0x74,0xd5,0x72,0xb9,0x18,0xf9,0xb3,0x49,0xe2,
			0xe7,0x04,0x77,0x53,0x00,0x02,0x38,0x50,0xa5,0x34,0xcd,0x14,0xe0,0x04,0x6e,0x38,
			0x51,0x81,0xd3,0x45,0xa0,0x1d,0xbd,0x23,0x0c,0x79,0x65,0xff,0xfd,0x71,0xd4,0xe9,
			0xf9,0xbd,0x27,0x4d,0x2e,0xf4,0x6c,0x55,0x6c,0x73,0x32,0x12,0x13,0x62,0xe7,0x17,
			0x06,0xc3,0x9f,0x08,0x91,0x9e,0xfa,0x3a,0x4f,0x2f,0x14,0x21,0x50,0xc8,0xce,0xf4,
			0xb3,0x0e,0xf7,0x88,0xf6,0xff,0xfc,0x2e,0xd2,0xac,0xa0,0x26,0x2e,0xba,0x87,0x3a,
			0x25,0x17,0x95,0x84,0x5d,0xa5,0x26,0xb7,0x8e,0x2b,0x24,0xd6,0x6f,0x21,0xd4,0x0d,
			0x5f,0xcb,0xe5,0x54,0x2a,0x75,0x53,0xa9,0x17,0x8b,0xe5,0xa9,0x80,0x5b,0xb5,0x35,
			0x79,0xbd,0x10,0x8c,0xff,0x73,0x3b,0x37,0x64,0x07,0x89,0x31,0x8d,0xb6,0x20,0xa6,
			0xa9,0x44,0x44,0xe9,0x20,0x39,0x23,0xae,0x32,0x55,0x00,0xb6,0x76,0xb1,0x47,0x98),
	},
	{	/* Format of the EM is incorrect - 00 on end of pad removed */
		.exp = FALSE,
		.e = chunk_from_chars(0x8d,0x88,0x53),
		.m = chunk_from_chars(
			0x4b,0x64,0x8d,0xe2,0xe3,0x90,0x67,0x7c,0xe1,0x72,0x71,0x91,0x49,0x8b,0x76,0xd2,
			0xd2,0x43,0x8d,0x0c,0xed,0xdf,0x3d,0xb5,0x02,0xad,0xe5,0xc5,0xfd,0x81,0x5b,0xc7,
			0x09,0x65,0xd4,0x2f,0x9b,0x8d,0x04,0xee,0xb3,0x89,0x86,0x44,0x5f,0xbe,0x7d,0xd5,
			0x80,0x17,0x7c,0x00,0xd0,0xf8,0xa3,0x6c,0x24,0x74,0x0b,0xe3,0xef,0xbf,0x65,0x64,
			0x08,0xe6,0xdd,0x73,0xad,0xbd,0x45,0xb4,0xc7,0x19,0x6f,0xb8,0x6f,0x40,0xda,0x17,
			0xb2,0x9e,0x91,0xbf,0x91,0xba,0x51,0x8b,0x4c,0x68,0xea,0x18,0xe1,0x3f,0xe7,0xd4,
			0xcf,0x4d,0x71,0x57,0x1f,0x0a,0x71,0xf7,0xa5,0x8b,0xf5,0x96,0x84,0xf8,0xd5,0xbf,
			0xf0,0x40,0x96,0xf2,0x38,0x0b,0x8e,0x3b,0x03,0xf1,0x49,0x09,0xd9,0xe9,0x2c,0x3e),
		.s = chunk_from_chars(
			0x27,0x7f,0xe8,0x1a,0x15,0x40,0x34,0x1c,0x9c,0x04,0xfb,0x5d,0x3c,0x8f,0x70,0xc1,
			0xe7,0x2f,0x87,0xfa,0x7d,0x04,0xb9,0x3e,0xe5,0x4d,0x56,0xd1,0x18,0x4c,0x14,0x1e,
			0xc4,0xe1,0x25,0xe2,0xe5,0xcf,0xd0,0xac,0xff,0x10,0x55,0xc7,0x66,0x97,0xa1,0xc3,
			0x07,0x9c,0x37,0x1c,0xce,0x0b,0xd8,0x4e,0xe4,0x43,0x7b,0xfa,0x18,0x06,0xaa,0x76,
			0xdf,0x29,0x55,0x93,0x53,0x31,0x91,0x5f,0xff,0x87,0x9d,0x87,0xc7,0x37,0x2e,0x75,
			0xe5,0xe8,0xfe,0xcc,0x90,0x6f,0x9d,0x34,0xa5,0xad,0xc0,0xb7,0x54,0x09,0xa8,0x7f,
			0xb6,0x41,0x1e,0x29,0x90,0xcd,0x23,0x7a,0xc8,0x22,0x3f,0x72,0x23,0x44,0x4f,0x5d,
			0x52,0x7a,0x11,0x5c,0x85,0xb9,0xba,0x10,0x28,0x7f,0x3e,0x82,0x5e,0x1d,0xc5,0x8e,
			0xb3,0x7c,0x8a,0x9c,0xa0,0x55,0x15,0x5f,0xf6,0xa5,0xaa,0xa6,0x56,0x56,0x90,0xdc,
			0xee,0xf6,0xf6,0x47,0x72,0x9d,0x07,0x1c,0xdd,0x9f,0x06,0x88,0xc7,0xfe,0xe8,0xf9,
			0x29,0x3a,0xbe,0x65,0x8d,0x0e,0x1e,0xf8,0x55,0xc8,0x6a,0x28,0xd8,0xd0,0xf8,0xf8,
			0x2f,0x72,0x17,0x31,0x86,0x35,0x30,0x80,0xb8,0x4f,0x14,0x2f,0x96,0x59,0x07,0x37,
			0x71,0x78,0xfc,0x1e,0xa0,0x08,0x19,0xd8,0x1a,0xfc,0x24,0x74,0xbe,0xcf,0x79,0xe1,
			0xe3,0xb7,0x0c,0x7c,0x2a,0x84,0x07,0xd5,0x07,0x95,0xc3,0x13,0x1b,0xb6,0xdf,0x78,
			0x72,0x0c,0x31,0xb2,0xf8,0xc4,0x3f,0x70,0x54,0x64,0xd8,0x98,0x20,0x43,0xc4,0x5f,
			0x04,0xe3,0x39,0xaa,0x1c,0x6f,0x4e,0x26,0xaa,0x70,0x92,0xb3,0xe6,0x26,0x12,0x99),
	},
	{
		.exp = TRUE,
		.e = chunk_from_chars(0x8d,0x88,0x53),
		.m = chunk_from_chars(
			0x78,0x11,0xa4,0x07,0xfe,0x65,0x3a,0xd2,0x34,0x3d,0x83,0xc0,0x49,0x9f,0xc1,0x1e,
			0x29,0x51,0xef,0x0a,0x47,0x91,0xa3,0xcd,0x9a,0x06,0x39,0x6b,0xe5,0xf7,0x2e,0x78,
			0x3c,0xbb,0xca,0x2c,0xc4,0x70,0x02,0xba,0xba,0xf0,0x9c,0xde,0xb7,0x01,0x94,0xb5,
			0x32,0xcc,0xbf,0xd2,0x46,0x91,0xae,0x8e,0xb5,0x98,0xd5,0x9f,0x2e,0x6b,0xec,0xed,
			0xcb,0x42,0x96,0xa1,0xde,0xbf,0x41,0x7e,0xed,0xdb,0x4d,0x74,0xfb,0x21,0x70,0x72,
			0x09,0x1a,0x59,0x7d,0xdd,0x08,0x93,0xff,0x02,0xd6,0xad,0x61,0x10,0x50,0x98,0xdb,
			0x2e,0x90,0x50,0x8b,0xd4,0xb8,0xbe,0xc5,0xd6,0xe7,0xde,0xab,0x9e,0x65,0x1d,0xfd,
			0xb8,0x21,0x05,0x32,0x95,0x5e,0x1b,0xc7,0x88,0xa9,0x08,0xd7,0x15,0x0e,0xf8,0xec),
		.s = chunk_from_chars(
			0x3a,0xd1,0xf0,0xfa,0xa9,0xa3,0x65,0x86,0x15,0x4a,0x38,0x2a,0x1f,0x2c,0x10,0xe8,
			0x19,0xdc,0x31,0x8a,0x68,0xbe,0x35,0xf5,0x4f,0x95,0x40,0x1b,0x0e,0xf2,0xba,0x5c,
			0xc8,0x95,0xf0,0xd6,0xbd,0xe2,0x8c,0x8d,0x3b,0x36,0x4f,0x60,0xde,0x03,0xce,0x75,
			0xa7,0xaf,0x29,0xe5,0x80,0x7c,0x1e,0xec,0x4c,0xd7,0x06,0x24,0x75,0x6e,0x26,0x31,
			0xeb,0x46,0xaf,0x01,0x0c,0xa8,0xcf,0x93,0xee,0x7d,0x86,0xa4,0xaf,0x3c,0xed,0x5d,
			0xd2,0xa8,0xa4,0x1c,0x1f,0xee,0xe6,0xb4,0x57,0x20,0x70,0x87,0x39,0x39,0xae,0x7a,
			0x2c,0xe7,0x51,0x93,0x94,0x59,0x37,0x46,0x1d,0x00,0x64,0xee,0xba,0x07,0x76,0x0c,
			0x49,0x5c,0x5c,0x70,0xbd,0xb0,0x4d,0x89,0x95,0x1d,0xe3,0xf9,0x65,0x25,0xf8,0xe4,
			0x46,0x12,0x49,0x3d,0x49,0x07,0x31,0xe7,0xef,0x76,0x79,0xa2,0x4b,0x0b,0x1e,0x2b,
			0x24,0xc8,0x00,0x3c,0x2f,0x94,0x11,0x4e,0x41,0x93,0xd6,0xe5,0x4c,0x0a,0xf9,0xe8,
			0x70,0x53,0x0b,0x00,0x85,0x30,0x21,0x0a,0xc0,0xb0,0xe4,0xc7,0xc7,0x9f,0x37,0x93,
			0x84,0xe1,0xbb,0x31,0x9b,0x5f,0x2a,0x0b,0x62,0x25,0x17,0xae,0x5d,0x27,0xf0,0xea,
			0xaf,0x79,0x58,0xcb,0x0b,0x41,0x84,0x91,0x26,0x07,0x50,0x92,0xe8,0x6e,0x7e,0x7b,
			0xe5,0xee,0xac,0xef,0x9a,0x8e,0x3e,0xc5,0x95,0x43,0x2e,0xf6,0x19,0x85,0x8f,0xe0,
			0xd0,0x51,0x78,0x71,0xb8,0xb4,0x95,0xc0,0x1a,0xf6,0xf1,0x8d,0x4e,0x6d,0x25,0x0d,
			0xbd,0x19,0x28,0x0e,0xf4,0xf8,0xa3,0x7d,0x1b,0x59,0xa4,0x8f,0xa4,0x1b,0x83,0x1d),
	},
	},
},
{
	.params = { .hash = HASH_SHA256, .mgf1_hash = HASH_SHA256, },
	.n = chunk_from_chars(
			0xce,0x49,0x24,0xff,0x47,0x0f,0xb9,0x9d,0x17,0xf6,0x65,0x95,0x56,0x1a,0x74,0xde,
			0xd2,0x20,0x92,0xd1,0xdc,0x27,0x12,0x2a,0xe1,0x5c,0xa8,0xca,0xc4,0xbf,0xae,0x11,
			0xda,0xa9,0xe3,0x7a,0x94,0x14,0x30,0xdd,0x1b,0x81,0xaa,0xf4,0x72,0xf3,0x20,0x83,
			0x5e,0xe2,0xfe,0x74,0x4c,0x83,0xf1,0x32,0x08,0x82,0xa8,0xa0,0x23,0x16,0xce,0xb3,
			0x75,0xf5,0xc4,0x90,0x92,0x32,0xbb,0x2c,0x65,0x20,0xb2,0x49,0xc8,0x8b,0xe4,0xf4,
			0x7b,0x8b,0x86,0xfd,0xd9,0x36,0x78,0xc6,0x9e,0x64,0xf5,0x00,0x89,0xe9,0x07,0xa5,
			0x50,0x4f,0xdd,0x43,0xf0,0xca,0xd2,0x4a,0xaa,0x9e,0x31,0x7e,0xf2,0xec,0xad,0xe3,
			0xb5,0xc1,0xfd,0x31,0xf3,0xc3,0x27,0xd7,0x0a,0x0e,0x2d,0x48,0x67,0xe6,0xfe,0x3f,
			0x26,0x27,0x2e,0x8b,0x6a,0x3c,0xce,0x17,0x84,0x3e,0x35,0x9b,0x82,0xeb,0x7a,0x4c,
			0xad,0x8c,0x42,0x46,0x01,0x79,0xcb,0x6c,0x07,0xfa,0x25,0x2e,0xfa,0xec,0x42,0x8f,
			0xd5,0xca,0xe5,0x20,0x8b,0x29,0x8b,0x25,0x51,0x09,0x02,0x6e,0x21,0x27,0x24,0x24,
			0xec,0x0c,0x52,0xe1,0xe5,0xf7,0x2c,0x5a,0xb0,0x6f,0x5d,0x2a,0x05,0xe7,0x7c,0x19,
			0x3b,0x64,0x7e,0xc9,0x48,0xbb,0x84,0x4e,0x0c,0x2e,0xf1,0x30,0x7f,0x53,0xcb,0x80,
			0x0d,0x4f,0x55,0x52,0x3d,0x86,0x03,0x8b,0xb9,0xe2,0x10,0x99,0xa8,0x61,0xb6,0xb9,
			0xbc,0xc9,0x69,0xe5,0xdd,0xdb,0xdf,0x71,0x71,0xb3,0x7d,0x61,0x63,0x81,0xb7,0x8c,
			0x3b,0x22,0xef,0x66,0x51,0x0b,0x27,0x65,0xd9,0x61,0x75,0x56,0xb1,0x75,0x59,0x98,
			0x79,0xd8,0x55,0x81,0x00,0xad,0x90,0xb8,0x30,0xe8,0x7a,0xd4,0x60,0xa2,0x21,0x08,
			0xba,0xa5,0xed,0x0f,0x2b,0xa9,0xdf,0xc0,0x51,0x67,0xf8,0xab,0x61,0xfc,0x9f,0x8a,
			0xe0,0x16,0x03,0xf9,0xdd,0x5e,0x66,0xce,0x1e,0x64,0x2b,0x60,0x4b,0xca,0x92,0x94,
			0xb5,0x7f,0xb7,0xc0,0xd8,0x3f,0x05,0x4b,0xac,0xf4,0x45,0x4c,0x29,0x8a,0x27,0x2c,
			0x44,0xbc,0x71,0x8f,0x54,0x60,0x5b,0x91,0xe0,0xbf,0xaf,0xd7,0x72,0xae,0xba,0xf3,
			0x82,0x88,0x46,0xc9,0x30,0x18,0xf9,0x8e,0x31,0x57,0x08,0xd5,0x0b,0xe8,0x40,0x1e,
			0xb9,0xa8,0x77,0x8d,0xcb,0xd0,0xd6,0xdb,0x93,0x70,0x86,0x04,0x11,0xb0,0x04,0xcd,
			0x37,0xfb,0xb8,0xb5,0xdf,0x87,0xed,0xee,0x7a,0xae,0x94,0x9f,0xff,0x34,0x60,0x7b),
	.tests = {
	{	/* Public Key e changed */
		.exp = FALSE,
		.e = chunk_from_chars(0xfe,0x30,0x79),
		.m = chunk_from_chars(
			0xe4,0x9f,0x58,0x5e,0xec,0xcf,0x2b,0xf7,0x26,0x56,0x41,0xfb,0x8c,0x0f,0x94,0xc7,
			0x17,0xe2,0xff,0x1d,0x90,0x45,0xae,0xca,0xa3,0x02,0xd2,0x85,0x35,0x3b,0x99,0x1b,
			0xf7,0xac,0x5d,0xc9,0x3b,0x31,0x1c,0xe9,0x07,0x88,0x28,0xd2,0x68,0x57,0x1f,0xf9,
			0x09,0x71,0x1e,0x5c,0x04,0x55,0x32,0x20,0xf8,0xf8,0x0f,0x78,0x5c,0xc4,0x05,0xca,
			0x13,0xe0,0x2f,0x0d,0x40,0xb2,0xee,0x76,0x5b,0xa2,0x95,0x53,0x85,0x21,0x66,0x37,
			0x18,0xea,0xbe,0x57,0x83,0x88,0x8c,0x34,0x55,0x19,0x07,0x7a,0x97,0x51,0xa1,0x28,
			0x5f,0xc2,0x36,0xf2,0xa2,0x5a,0x8a,0xe4,0x4a,0x2d,0xf2,0x47,0x88,0x74,0x51,0xc8,
			0x6c,0xd6,0x46,0xd7,0xb3,0xe7,0xa4,0x4e,0xe0,0xef,0x23,0x53,0x8e,0xec,0x55,0x7f),
		.s = chunk_from_chars(
			0x4e,0x85,0xf6,0x8a,0x5b,0x06,0xb0,0x6a,0x17,0xd0,0xf3,0xf2,0x7b,0x3a,0x5a,0x11,
			0x9e,0x7d,0xb0,0x2a,0xbc,0x2d,0x9b,0x4a,0xfc,0x69,0x82,0x20,0xda,0x11,0x52,0x4a,
			0x88,0x5f,0x33,0xcd,0x7a,0x10,0xae,0x89,0xc9,0x8b,0x02,0x7b,0x69,0x22,0x4a,0xce,
			0xf4,0x71,0x3a,0x14,0x63,0xf1,0x68,0xc8,0xbe,0xf5,0x51,0xef,0x8f,0xed,0xb2,0x19,
			0xb6,0xad,0x0b,0x3e,0x99,0xd6,0x21,0x66,0x43,0xe5,0x8a,0x51,0xbb,0x2a,0xe9,0x3b,
			0xbe,0xf7,0x69,0x61,0x49,0x14,0xea,0xb1,0x37,0xc1,0x99,0x3b,0x14,0x91,0x71,0xb8,
			0x63,0x3f,0x4a,0x31,0x8f,0x69,0x77,0x29,0x96,0xef,0x7d,0xc3,0xf7,0x74,0x8f,0x37,
			0x56,0xd5,0x8e,0xcd,0xc3,0x93,0x76,0x32,0x71,0x7f,0xb4,0x0c,0xb7,0xed,0x6e,0x5c,
			0x72,0xe1,0x72,0xac,0x58,0xec,0x01,0xf4,0xe3,0x2f,0xff,0xc4,0x45,0xb6,0x0f,0x98,
			0xa6,0x28,0xfc,0x1b,0x0f,0xa4,0xcf,0xb6,0x68,0x6d,0xeb,0x12,0x59,0x50,0xb8,0x62,
			0xf3,0x47,0xe9,0xeb,0x81,0x20,0xfb,0x2b,0x5a,0xa2,0x3d,0x6d,0x86,0xea,0xf1,0xed,
			0xeb,0xeb,0x13,0x37,0x93,0x54,0x1c,0x4d,0xbe,0xa0,0xf1,0x4a,0x9f,0x74,0x73,0x3d,
			0xa4,0xed,0x11,0xd1,0x27,0x4d,0x46,0x4e,0x09,0xa5,0x78,0x08,0x43,0xd6,0x75,0x0b,
			0xac,0xe0,0xe9,0x70,0x29,0x30,0x82,0x87,0xdd,0x39,0x6e,0xfa,0x0f,0x32,0x62,0x81,
			0x71,0xfc,0x5e,0xc2,0x0d,0x3c,0x82,0x61,0x9b,0x78,0x4e,0x4c,0xdb,0x66,0xcb,0xdb,
			0x28,0xcd,0xd2,0x63,0xa4,0x6a,0x3e,0xc6,0x3e,0x1c,0xad,0x76,0x59,0xdc,0x3b,0x33,
			0x80,0x14,0x32,0xd2,0xb5,0xb5,0xe1,0x0a,0x77,0x00,0x83,0xb9,0x33,0xa8,0x05,0xa9,
			0xc7,0x6c,0xc2,0x6c,0x91,0x2f,0x95,0x2c,0xec,0x5f,0xd8,0x41,0x3a,0x8c,0x1a,0xda,
			0xee,0x80,0x14,0x9f,0xa1,0x98,0x55,0x31,0x50,0x75,0x82,0x52,0x92,0xdb,0x24,0xde,
			0x32,0x5f,0xa6,0xbf,0x3b,0x4c,0x06,0x65,0x2f,0xc8,0x32,0x0d,0xef,0x42,0x36,0xc0,
			0x88,0xdd,0x5a,0xe4,0x33,0x15,0xe0,0x36,0x72,0xfb,0x99,0x9c,0x35,0x4e,0xf6,0x1a,
			0xc3,0x80,0xb1,0xb1,0xc9,0x6d,0x71,0x1f,0xc7,0x77,0xe3,0x45,0xcc,0xb9,0x45,0x36,
			0x35,0x5a,0x32,0x14,0x66,0xee,0xdc,0xf2,0x35,0x5d,0xd5,0x1f,0x68,0x80,0x23,0xd6,
			0xb5,0x99,0x39,0x0f,0x3a,0xff,0x62,0x01,0x36,0x9d,0x81,0x03,0xaf,0x92,0x6c,0x83),
	},
	{	/* Message changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0x46,0x21,0xb1,0x7c,0xd9,0xf5,0xb6,0x23,0xfe,0x73,0xb5,0xfe,0x28,0x0c,0xe9,0xac,
			0x84,0x08,0x05,0x60,0x8a,0xcd,0x6e,0x41,0xd5,0x5e,0xa7,0x11,0x32,0x22,0x0c,0x0d,
			0xf7,0xe7,0xc4,0x15,0x96,0x26,0xf1,0x0d,0x71,0x88,0x29,0x83,0xf0,0xaa,0x2a,0x92,
			0xd1,0x1d,0xc9,0x06,0xc0,0xb2,0x2c,0xc0,0x28,0xf4,0x39,0x5d,0x48,0xf5,0x4e,0x12,
			0x89,0x4e,0x33,0xda,0x0f,0x61,0x4d,0xd4,0x8e,0xe1,0x14,0xe6,0x5f,0x95,0xc7,0xa7,
			0xd3,0x58,0x5e,0x7c,0xc7,0x65,0xc0,0x01,0x78,0xd1,0x36,0xaa,0x99,0x59,0x1f,0xaa,
			0xa3,0x5e,0xe6,0x13,0x6d,0x2e,0x32,0x3f,0xfc,0x85,0x5c,0x70,0x9c,0x54,0x26,0xb3,
			0x2f,0xc0,0xaa,0x0a,0xc6,0x6e,0x90,0xc9,0x6e,0xfe,0x84,0x41,0x4d,0xd5,0xe7,0x9c),
		.s = chunk_from_chars(
			0xb6,0x0a,0x4d,0xd6,0x29,0xd6,0x03,0x0f,0xe6,0x52,0x2f,0x6b,0x75,0x4f,0x0e,0x75,
			0x1d,0xe4,0xb2,0x55,0x2c,0x60,0x7e,0xfc,0xcb,0x2f,0x90,0xda,0x91,0x78,0x75,0x83,
			0xb6,0xfc,0x51,0xbc,0xb6,0x0a,0xb2,0x19,0x38,0xa4,0x8c,0xa6,0xab,0x3e,0xf8,0xab,
			0x75,0xb5,0x6a,0xbb,0x9d,0xf1,0xfa,0xa4,0xdb,0xd8,0x4b,0x41,0x20,0x66,0xf3,0xf9,
			0x2b,0xff,0x77,0x8a,0x89,0xf7,0xdf,0x4f,0x55,0x31,0x7c,0xbc,0x40,0xa7,0x80,0xfb,
			0x87,0xf0,0xc8,0x44,0xc2,0xd6,0x4e,0x23,0x24,0x74,0xa3,0xe9,0x31,0xc1,0x68,0xb3,
			0x30,0x86,0x65,0x79,0x68,0x5c,0x51,0xa5,0x93,0x7a,0x2e,0x80,0xea,0x2c,0x6e,0xd0,
			0x0f,0xe1,0x23,0xf1,0x4b,0xbe,0xf5,0x5c,0x97,0x74,0xbd,0x62,0x0e,0x1e,0x82,0x1e,
			0x01,0x28,0xcf,0x49,0xda,0xb6,0xf8,0x85,0x3c,0x08,0x80,0x1c,0x8a,0x00,0x91,0x9a,
			0x6c,0x01,0x3c,0x3a,0x83,0xf9,0x99,0xc6,0x6a,0x5c,0xb4,0x9c,0x91,0x86,0x5d,0xf6,
			0x0d,0xb8,0xbe,0x81,0x3b,0xc3,0xd8,0xb3,0x5d,0x85,0xd7,0x9c,0xf0,0x1a,0xbe,0xd2,
			0xf6,0x0f,0x60,0xed,0xc9,0x71,0x53,0x78,0x0c,0x0c,0x12,0xfe,0x45,0xe5,0xe4,0x87,
			0xfd,0x95,0x93,0x93,0x07,0x9d,0xbe,0xe5,0xaf,0x46,0xc0,0xa7,0xc4,0x21,0x4f,0xab,
			0x75,0xc5,0xb5,0xdd,0xd6,0xf0,0x28,0x86,0x69,0xe0,0xa9,0xbe,0x33,0xf4,0xb5,0x78,
			0x25,0x24,0xe8,0x38,0x31,0x5c,0xc0,0x31,0xd9,0x7b,0xeb,0x15,0x96,0x02,0x6f,0x12,
			0x9a,0x21,0xe9,0x61,0xdd,0xe6,0xbc,0x34,0xc4,0x92,0xf3,0x02,0x6a,0xf7,0xf5,0x03,
			0xb8,0xff,0x87,0xb7,0x77,0x56,0x19,0xf8,0xd1,0xe1,0x7f,0x97,0x2c,0x85,0x94,0x0a,
			0xff,0xbb,0x64,0xd3,0x31,0x0f,0xb9,0xf7,0x4d,0x9d,0x16,0xae,0xb0,0x77,0xb8,0xb9,
			0x97,0xb1,0x80,0x20,0xeb,0x99,0x2a,0xb6,0x13,0x22,0x84,0x7f,0xe6,0xcb,0x62,0xf7,
			0x3d,0x0a,0xbd,0x81,0x63,0x3a,0xc5,0xc5,0xbe,0x65,0x19,0xdd,0xbc,0x93,0x34,0xbb,
			0x56,0x44,0x9b,0xdf,0x96,0x93,0x0d,0x65,0xd8,0x06,0x1d,0xb8,0x91,0x1d,0x4e,0xd6,
			0xa5,0x9a,0xe8,0xd2,0x27,0x6d,0x04,0x59,0x6a,0x38,0x8e,0x75,0x2c,0xf9,0x9c,0xbd,
			0x39,0x5b,0x83,0x7e,0x7c,0x5a,0xee,0x27,0x8a,0x7c,0x4b,0x43,0xc7,0x8d,0x3d,0x74,
			0xc8,0x8f,0x49,0xcb,0xc8,0xd8,0x16,0xf5,0x3c,0xca,0x15,0x69,0x27,0xfb,0x92,0xd4),
	},
	{
		.exp = TRUE,
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0x08,0x97,0xd4,0x0e,0x7c,0x0f,0x2d,0xfc,0x07,0xb0,0xc7,0xfd,0xda,0xf5,0xfd,0x8f,
			0xcc,0x6a,0xf9,0xc1,0xfd,0xc1,0x7b,0xeb,0xb9,0x23,0xd5,0x9c,0x9f,0xc4,0x3b,0xd4,
			0x02,0xba,0x39,0x73,0x8f,0x0f,0x85,0xf2,0x30,0x15,0xf7,0x51,0x31,0xf9,0xd6,0x50,
			0xa2,0x9b,0x55,0xe2,0xfc,0x9d,0x5d,0xdf,0x07,0xbb,0x8d,0xf9,0xfa,0x5a,0x80,0xf1,
			0xe4,0x63,0x4e,0x0b,0x4c,0x51,0x55,0xbf,0x14,0x89,0x39,0xb1,0xa4,0xea,0x29,0xe3,
			0x44,0xa6,0x64,0x29,0xc8,0x50,0xfc,0xde,0x73,0x36,0xda,0xd6,0x16,0xf0,0x03,0x93,
			0x78,0x39,0x1a,0xbc,0xfa,0xfe,0x25,0xca,0x7b,0xb5,0x94,0x05,0x7a,0xf0,0x7f,0xaf,
			0x7a,0x32,0x2f,0x7f,0xab,0x01,0xe0,0x51,0xc6,0x3c,0xc5,0x1b,0x39,0xaf,0x4d,0x23),
		.s = chunk_from_chars(
			0x8e,0xbe,0xd0,0x02,0xd4,0xf5,0x4d,0xe5,0x89,0x8a,0x5f,0x2e,0x69,0xd7,0x70,0xed,
			0x5a,0x5c,0xe1,0xd4,0x5a,0xd6,0xdd,0x9c,0xe5,0xf1,0x17,0x9d,0x1c,0x46,0xda,0xa4,
			0xd0,0x39,0x4e,0x21,0xa9,0x9d,0x80,0x33,0x58,0xd9,0xab,0xfd,0x23,0xbb,0x53,0x16,
			0x63,0x94,0xf9,0x97,0xb9,0x09,0xe6,0x75,0x66,0x20,0x66,0x32,0x4c,0xa1,0xf2,0xb7,
			0x31,0xde,0xba,0x17,0x05,0x25,0xc4,0xee,0x8f,0xa7,0x52,0xd2,0xd7,0xf2,0x01,0xb1,
			0x02,0x19,0x48,0x9f,0x57,0x84,0xe3,0x99,0xd9,0x16,0x30,0x2f,0xd4,0xb7,0xad,0xf8,
			0x84,0x90,0xdf,0x87,0x65,0x01,0xc4,0x67,0x42,0xa9,0x3c,0xfb,0x3a,0xaa,0xb9,0x60,
			0x2e,0x65,0xd7,0xe6,0x0d,0x7c,0x4c,0xea,0xdb,0x7e,0xb6,0x7e,0x42,0x1d,0x18,0x03,
			0x23,0xa6,0xd3,0x8f,0x38,0xb9,0xf9,0x99,0x21,0x3e,0xbf,0xcc,0xc7,0xe0,0x4f,0x06,
			0x0f,0xbd,0xb7,0xc2,0x10,0x20,0x65,0x22,0xb4,0x94,0xe1,0x99,0xe9,0x8c,0x6c,0x24,
			0xe4,0x57,0xf8,0x69,0x66,0x44,0xfd,0xca,0xeb,0xc1,0xb9,0x03,0x1c,0x81,0x83,0x22,
			0xc2,0x9d,0x13,0x5e,0x11,0x72,0xfa,0x0f,0xdf,0x7b,0xe1,0x00,0x7d,0xab,0xca,0xab,
			0x49,0x66,0x33,0x2e,0x7e,0xa1,0x45,0x6b,0x6c,0xe8,0x79,0xcd,0x91,0x0c,0x91,0x10,
			0x10,0x4f,0xc7,0xd3,0xdc,0xab,0x07,0x6f,0x2b,0xd1,0x82,0xbb,0x83,0x27,0xa8,0x63,
			0x25,0x45,0x70,0xcd,0xf2,0xab,0x38,0xe0,0xcd,0xa3,0x17,0x79,0xde,0xaa,0xd6,0x16,
			0xe3,0x43,0x7e,0xd6,0x59,0xd7,0x4e,0x5a,0x4e,0x04,0x5a,0x70,0x13,0x38,0x90,0xb8,
			0x1b,0xc4,0xf2,0x4a,0xb6,0xda,0x67,0xa2,0xee,0x0c,0xe1,0x5b,0xab,0xa3,0x37,0xd0,
			0x91,0xcb,0x5a,0x1c,0x44,0xda,0x69,0x0f,0x81,0x14,0x5b,0x02,0x52,0xa6,0x54,0x9b,
			0xbb,0x20,0xcd,0x5c,0xc4,0x7a,0xfe,0xc7,0x55,0xeb,0x37,0xfe,0xd5,0x5a,0x9a,0x33,
			0xd3,0x65,0x57,0x42,0x45,0x03,0xd8,0x05,0xa0,0xa1,0x20,0xb7,0x69,0x41,0xf4,0x15,
			0x0d,0x89,0x34,0x2d,0x7a,0x7f,0xa3,0xa2,0xb0,0x8c,0x51,0x5e,0x6f,0x68,0x42,0x9c,
			0xf7,0xaf,0xd1,0xa3,0xfc,0xe0,0xf4,0x28,0x35,0x1a,0x6f,0x9e,0xda,0x3a,0xb2,0x4a,
			0x7e,0xf5,0x91,0x99,0x4c,0x21,0xfb,0xf1,0x00,0x1f,0x99,0x23,0x9e,0x88,0x34,0x0f,
			0x9b,0x35,0x9e,0xc7,0x2e,0x8a,0x21,0x2a,0x19,0x20,0xe6,0xcf,0x99,0x3f,0xf8,0x48),
	},
	{	/* Format of the EM is incorrect - 00 on end of pad removed */
		.exp = FALSE,
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0x78,0x67,0xb6,0x5e,0xd9,0x82,0xed,0x6c,0xdd,0x2d,0x06,0x11,0x57,0xbe,0x90,0xf8,
			0x5b,0xcd,0x58,0x03,0x50,0xf1,0x25,0x31,0x45,0xcb,0xa5,0xc5,0x89,0x46,0xa8,0xa5,
			0x75,0x1c,0x8c,0x00,0x8d,0x9d,0xf8,0x33,0xc8,0xac,0xce,0xee,0x2b,0x0a,0x5a,0x92,
			0x9c,0xd7,0xd0,0xde,0xf6,0x55,0xf5,0xcb,0x59,0xf0,0x1c,0xb4,0xc4,0x7b,0x54,0xba,
			0xe5,0xbd,0xe0,0x67,0x2f,0x2c,0xe7,0x92,0x2e,0xf8,0x6d,0x82,0x17,0x4b,0x82,0x56,
			0xa4,0xd0,0xb9,0xa3,0x1e,0x72,0xdc,0x60,0xbb,0x66,0xde,0xff,0x2b,0x6b,0x11,0xdd,
			0x6e,0x50,0x99,0xdc,0xe8,0xb7,0x21,0x4e,0xb7,0x1a,0xcd,0x16,0x44,0x0b,0x6f,0x09,
			0x18,0xc0,0xfd,0x9b,0xf2,0xce,0x43,0xb7,0x16,0x35,0xd5,0xee,0xe7,0x9d,0x48,0xd4),
		.s = chunk_from_chars(
			0x0a,0x61,0x70,0x81,0xdd,0x97,0x94,0xc9,0x37,0xe4,0xc7,0x16,0x3e,0x28,0x65,0x43,
			0x95,0x58,0xd4,0x1e,0x68,0x8b,0x23,0x0c,0x20,0xa1,0xea,0x71,0x84,0x66,0x43,0xb8,
			0x32,0x1d,0x77,0xca,0xd5,0x82,0xec,0x9a,0x70,0xf4,0x0d,0x7a,0x8e,0xd9,0xf7,0x22,
			0xc5,0xb9,0x01,0x2f,0x43,0x6c,0x7a,0x61,0x7d,0xed,0x2a,0xd1,0x86,0x43,0x45,0x7b,
			0x6f,0xd3,0x30,0x35,0xfc,0x2a,0x2c,0xbd,0x52,0xf3,0xda,0x41,0x58,0x01,0x5e,0x01,
			0xfa,0xb5,0x5b,0x4e,0x26,0xe2,0xde,0x96,0x02,0x10,0x1d,0xdb,0x86,0xe8,0xd2,0xae,
			0xb8,0x9b,0xf4,0x00,0x69,0x33,0xa1,0x1b,0x5f,0xbc,0xe2,0xbd,0x09,0xe3,0x1f,0xda,
			0x18,0xa8,0x2e,0xc4,0x05,0x91,0x0d,0xd3,0xd1,0xcf,0x1e,0x46,0x5c,0x17,0x68,0x72,
			0xda,0x3d,0xb1,0xc2,0xb3,0x08,0x9b,0x48,0xe7,0x68,0xef,0x15,0x5a,0xd5,0xf5,0x64,
			0x17,0x49,0x7a,0x64,0x8e,0xd0,0x42,0x7d,0x45,0xec,0xdd,0x03,0x8d,0x3c,0xc8,0xc8,
			0x7c,0xe9,0x9a,0x92,0xf0,0x8c,0x76,0x41,0xf7,0xfa,0x39,0xce,0xb4,0xec,0x83,0xec,
			0xe0,0x99,0x4a,0xc6,0x84,0x8b,0xb6,0xef,0xb4,0xe7,0x93,0x27,0x14,0x5e,0x53,0x96,
			0xb1,0x0f,0xaa,0x1a,0xc3,0xa6,0x3e,0xcd,0x4a,0x4a,0x2c,0x9c,0xba,0x94,0x6b,0x0e,
			0x5f,0x18,0x5b,0x2c,0xd1,0xa0,0x42,0x93,0xee,0x46,0x39,0x3d,0x37,0x4d,0xb8,0xbe,
			0x0a,0x84,0x93,0x77,0xeb,0x11,0x06,0x7e,0xad,0x3b,0x76,0xe5,0x14,0x55,0x1d,0x48,
			0x4d,0x5c,0x51,0x54,0x4a,0x1d,0x0d,0xa9,0x6c,0x17,0xb5,0xb2,0xef,0x95,0x3d,0xcd,
			0xc1,0xc6,0xd5,0x92,0x03,0x0e,0x46,0x8e,0xb8,0x18,0x6a,0x5b,0xf6,0x60,0xb7,0x1d,
			0xa5,0xd9,0x82,0xbc,0x54,0x24,0xcf,0x1a,0x4f,0x8c,0xe9,0x52,0xbc,0x76,0x3b,0x4e,
			0xaa,0xdb,0x2a,0x3f,0x78,0x7c,0x8a,0xf0,0x1d,0xf2,0xf4,0x3d,0xb5,0xe0,0xad,0x71,
			0x18,0x24,0xac,0xec,0xe2,0xd9,0x9d,0x5b,0xf9,0xfd,0xee,0x0f,0xdf,0x1f,0xdd,0x37,
			0x0e,0xc7,0x91,0xa9,0x46,0xa4,0xb6,0xaf,0x20,0xb2,0x77,0x47,0x9a,0x83,0x87,0x76,
			0xe9,0xc0,0x92,0x30,0xc3,0x63,0x25,0xb8,0xc8,0xef,0x84,0xdb,0x28,0x2f,0xe8,0x9c,
			0xed,0x24,0xb0,0xd7,0x86,0x9b,0x5d,0x24,0x6f,0xb3,0xc6,0x0f,0xcb,0x26,0x13,0x09,
			0x89,0x77,0x14,0xe8,0x70,0x19,0x5b,0x9b,0xee,0xd0,0xb2,0x65,0x80,0x0f,0xc6,0xe6),
	},
	{	/* Format of the EM is incorrect - hash moved to left */
		.exp = FALSE,
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0x4d,0x36,0xeb,0x2c,0x3a,0xd2,0x33,0x43,0x69,0x23,0xe5,0x80,0xfa,0xdd,0xb4,0x5a,
			0xd3,0x59,0x67,0x10,0x8b,0xe8,0xd9,0x9a,0x87,0x67,0x45,0xdf,0x62,0x13,0xc0,0x28,
			0x92,0x9f,0x07,0xd5,0x49,0x84,0x7b,0x4f,0x9a,0x99,0x6a,0x3d,0xdd,0xe3,0x90,0xfb,
			0x54,0xae,0xde,0x47,0x0f,0xdc,0x7a,0x3e,0x0c,0x7e,0x06,0x88,0xa3,0xa1,0x25,0xcf,
			0x21,0x6a,0x3b,0x75,0xb4,0x66,0x75,0x86,0x87,0x1b,0x0a,0xeb,0x2d,0xe3,0xc0,0xe1,
			0x43,0xfa,0xb1,0xaa,0x51,0xd5,0x4f,0x82,0xf2,0xb5,0xa6,0xd5,0x35,0x7d,0xe1,0xaf,
			0x42,0xc0,0x10,0x74,0x41,0x1f,0x28,0xd1,0x77,0xd2,0x4b,0xf2,0xb2,0x84,0x4a,0xf6,
			0xe8,0x64,0x69,0xa0,0x1b,0x79,0x62,0x4f,0x7f,0x35,0xac,0x30,0xdf,0x4e,0xfc,0xb2),
		.s = chunk_from_chars(
			0x81,0x38,0x92,0x2f,0xd1,0xa8,0x73,0x33,0xf1,0x31,0x6e,0x20,0x7e,0x58,0x51,0xaf,
			0x65,0xf2,0x7d,0x9f,0x5c,0xda,0x0f,0x7b,0x7c,0xab,0x00,0x54,0xa2,0xde,0xab,0x0a,
			0x5c,0x24,0x6f,0x18,0x34,0xb8,0xfa,0x0a,0x9a,0xe7,0x55,0xb6,0xad,0xd3,0xb6,0xdd,
			0x93,0xc6,0x94,0xfa,0x49,0xbe,0xa2,0x8a,0x56,0x35,0xe5,0x6d,0x98,0x41,0xd2,0x83,
			0x32,0x0e,0x7a,0x6f,0x98,0x12,0xc1,0x02,0xa6,0x0f,0xc2,0x50,0x5a,0x08,0x1e,0xe3,
			0x84,0x9f,0x81,0x51,0x91,0xb9,0xd7,0xa6,0xc4,0x1d,0xb6,0x7a,0xa1,0xa0,0x53,0xf2,
			0x2d,0x15,0x14,0x22,0x6a,0xad,0x39,0x22,0xcb,0x0b,0x5b,0xe7,0xdc,0x86,0xd6,0xcd,
			0x5d,0xaf,0x28,0xba,0xda,0x17,0x9f,0xbe,0xda,0x50,0x37,0x2b,0xce,0xe5,0xfd,0x8b,
			0x56,0x7c,0xa1,0x82,0x6b,0x08,0x1c,0xc0,0xe6,0xcf,0x58,0xcd,0x7e,0xd9,0x35,0xc6,
			0x92,0x2b,0x92,0x4b,0xed,0xd7,0xe4,0xf3,0xd4,0x8f,0x19,0x95,0xa5,0x6b,0xb1,0xd8,
			0xaf,0xf5,0x50,0x5e,0xa9,0x7c,0x56,0xcd,0xcb,0xaa,0x8e,0x93,0x25,0x4e,0x8c,0x6a,
			0x39,0xdb,0x12,0x76,0xc6,0xf5,0x3a,0x65,0x51,0xf1,0x62,0xf4,0x03,0x18,0x2b,0x4b,
			0xca,0x89,0x2a,0x6c,0xf5,0x27,0xc4,0xf9,0x62,0x96,0xb2,0xce,0x17,0xff,0xdf,0xa0,
			0x34,0x7d,0x52,0x90,0x94,0x0d,0x81,0xed,0x99,0xa3,0xe8,0xb0,0x0e,0x85,0x9d,0xc0,
			0xbc,0xb8,0x3e,0x90,0xe9,0xcd,0x9a,0x5b,0x64,0xcb,0x9d,0xb1,0xf3,0xf0,0x1d,0x26,
			0xd0,0x88,0x35,0xff,0xdf,0xe4,0x05,0x3d,0x9c,0x79,0x20,0x96,0x3b,0x07,0xfb,0x19,
			0x9e,0x5c,0x01,0xd4,0x4d,0x41,0x82,0x58,0x61,0x04,0x85,0x02,0x13,0xf8,0xca,0x60,
			0xc0,0x08,0x1e,0x62,0xb5,0xfe,0xcb,0xad,0x7e,0xce,0xd9,0x3c,0x35,0xa1,0x37,0x4d,
			0x27,0xb6,0x71,0x50,0xdd,0xfb,0x44,0xac,0x68,0xf6,0xaa,0x00,0xb5,0x5e,0x38,0xe6,
			0x81,0xb4,0x56,0xe6,0x06,0x55,0x26,0x1d,0x8d,0xae,0x04,0x7d,0x67,0xa4,0xee,0x2c,
			0x55,0xec,0xf2,0x13,0x57,0x1d,0xa1,0xdf,0x54,0x9a,0x48,0x89,0xea,0x3a,0x28,0x5b,
			0x5b,0x2f,0xe0,0x70,0x7d,0x74,0xd4,0x70,0xc7,0xec,0xe3,0xa2,0x15,0x26,0xf5,0xd4,
			0xaf,0x70,0x85,0xdd,0x11,0xe6,0xaf,0x40,0x51,0x52,0xa0,0x51,0x10,0xfc,0xd6,0xaa,
			0x06,0xce,0x93,0xf1,0xd2,0xba,0xba,0x3e,0xd5,0xdb,0x73,0x29,0x3f,0x97,0x7e,0xf1),
	},
	{	/* Signature changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x73,0xb1,0x93),
		.m = chunk_from_chars(
			0xe2,0xa9,0x2b,0x14,0x3c,0x8a,0x00,0x6c,0xee,0x8a,0xfe,0xbb,0x66,0x31,0x19,0x74,
			0x5d,0x26,0xf4,0xff,0xac,0xaa,0x53,0x5c,0x61,0x65,0xd3,0x0a,0x12,0x65,0xd9,0x27,
			0x71,0x64,0xc8,0xd8,0x21,0x49,0x77,0xeb,0xc8,0xf2,0xe2,0xbb,0x66,0x31,0x1f,0x54,
			0xe5,0x14,0x94,0xd8,0xf1,0x6a,0x68,0x22,0x09,0x82,0x37,0xa8,0xd6,0x36,0x0a,0xa6,
			0xbc,0x28,0x86,0x79,0xb0,0x4c,0x63,0x48,0x49,0xa7,0x20,0xea,0x19,0x56,0xf5,0x04,
			0x3d,0xd1,0x0a,0x72,0x33,0x71,0xe6,0xe9,0xec,0x30,0x2a,0xda,0x22,0xb1,0x7f,0x99,
			0xec,0x80,0xcd,0xbe,0x90,0x9a,0xae,0xcc,0x98,0x30,0x22,0x10,0x28,0xa8,0x84,0xcc,
			0x8e,0xe5,0xbc,0x95,0x1b,0xc8,0xb7,0xfa,0x31,0xf6,0x14,0x66,0x9e,0xdc,0x20,0x48),
		.s = chunk_from_chars(
			0xc8,0x1a,0xc3,0x94,0xf8,0x2c,0xb2,0x7d,0x75,0x95,0x07,0x59,0x22,0x4b,0x17,0x87,
			0xf8,0x72,0x25,0xae,0x55,0x49,0xbe,0x13,0xdc,0x1f,0x74,0x32,0x6e,0xb6,0x68,0xe5,
			0xe0,0x9d,0x03,0xe0,0x3e,0xcf,0x03,0x9f,0xd0,0x8d,0xc3,0x01,0xe7,0x0b,0xd0,0x79,
			0x92,0xb2,0x0a,0x27,0x57,0xb5,0x76,0x6b,0xf6,0x22,0x05,0x2d,0x69,0xfb,0xfb,0xfa,
			0xca,0x96,0x4d,0xa3,0x3a,0xf7,0x1c,0x87,0x87,0xeb,0xa9,0xed,0x5c,0xaf,0x1b,0xf8,
			0x6e,0x48,0x58,0x7d,0x26,0xb2,0x1f,0x9f,0x1b,0xe6,0x01,0x24,0x6e,0x1d,0x9c,0xdc,
			0xea,0x9a,0xf7,0x79,0x6a,0xe3,0x0d,0xd6,0xbc,0xbf,0x1b,0x25,0x73,0x4f,0x89,0xe8,
			0x6a,0xe9,0x05,0x1c,0x7a,0x2c,0xf3,0x73,0x09,0x14,0x40,0x6f,0x15,0x41,0xbe,0xed,
			0x81,0x2a,0xc6,0xc9,0xe1,0x8c,0xfd,0x5b,0xf2,0x65,0x52,0x33,0x60,0x01,0x88,0x60,
			0x14,0x1b,0xcf,0xc5,0x1c,0x89,0xef,0x87,0x96,0x36,0xae,0x2d,0x7f,0x7e,0x41,0xe3,
			0x13,0x37,0xf7,0xaf,0xc3,0xe2,0x01,0xc6,0x80,0x5a,0x76,0xaf,0x89,0xdc,0x9e,0xa7,
			0x74,0x13,0xbe,0x1f,0x13,0x38,0xe8,0x45,0xe9,0xd1,0x6c,0xf8,0x6e,0x5e,0x04,0xa5,
			0x2d,0x8b,0x57,0x94,0x21,0x29,0x6f,0xa7,0x02,0xad,0x07,0xec,0xbf,0xa0,0x23,0x7a,
			0xc9,0x96,0xee,0x2e,0x91,0xee,0xac,0x8e,0xf4,0x89,0x02,0xa7,0x6c,0x94,0x7b,0x50,
			0x1e,0x26,0x5c,0x4b,0x3f,0xbd,0x0c,0x51,0x6a,0x04,0x16,0xea,0x13,0xc7,0x9b,0x7e,
			0x5d,0x5b,0x94,0x35,0xae,0xf3,0xf3,0x71,0xd3,0x8f,0xf9,0xdf,0x8f,0xdd,0xed,0x2f,
			0x26,0x5b,0x27,0xda,0x5a,0x84,0xfe,0x54,0x43,0x63,0x5e,0x26,0x08,0x07,0xdb,0x9a,
			0x15,0x51,0x38,0x7e,0x6c,0x9c,0x59,0x64,0x98,0xd2,0x80,0xd9,0x17,0x0d,0xaa,0x62,
			0x95,0x4e,0xd3,0x2b,0x14,0xd0,0x44,0x98,0x0e,0xca,0x76,0xa8,0xdb,0x63,0x48,0x3d,
			0x8c,0x4c,0xbf,0xa6,0x69,0xec,0x14,0x0e,0x7c,0x4a,0x30,0x4c,0xc1,0x5e,0x46,0x8d,
			0x96,0xff,0xf3,0x4d,0x77,0x69,0x25,0x81,0xb1,0x0e,0x7d,0x25,0xef,0x07,0x5d,0x65,
			0x27,0x58,0xf5,0x0e,0x9a,0xc4,0xcf,0x98,0x48,0x46,0x63,0x88,0x05,0x15,0x18,0xcf,
			0x93,0xe1,0x83,0xf9,0x10,0xa4,0xf5,0x03,0xfb,0xeb,0x65,0x4f,0x4b,0x94,0x24,0xdb,
			0xad,0xf6,0x1d,0x2d,0x50,0xf6,0x0f,0x8f,0x02,0xb4,0xea,0x0e,0x7b,0xc3,0xb3,0x98),
	},
	},
},
{
	.params = { .hash = HASH_SHA512, .mgf1_hash = HASH_SHA512, },
	.n = chunk_from_chars(
			0x99,0xa5,0xc8,0xd0,0x94,0xa5,0xf9,0x17,0x03,0x46,0x67,0xa0,0x40,0x8b,0x7e,0xcf,
			0xca,0xac,0xc3,0xf9,0x78,0x44,0x44,0xe2,0x17,0x73,0xc3,0x46,0x1e,0xc3,0x55,0xf0,
			0xd0,0xf5,0x2a,0x5d,0xb0,0x56,0x8a,0x71,0xd3,0x88,0x69,0x67,0x88,0xef,0x66,0xae,
			0x73,0x40,0xc6,0xb2,0x8d,0xbf,0x92,0x5f,0xe8,0x35,0x57,0x98,0x65,0x75,0xf7,0x9c,
			0xca,0x69,0x21,0x72,0x21,0x39,0x7e,0xd5,0x80,0x8a,0x26,0xf7,0xe7,0xe7,0x14,0xc9,
			0x32,0x35,0xf9,0x14,0xd4,0x5c,0x4a,0x9a,0xf4,0x61,0x9b,0x20,0xf5,0x11,0xad,0x64,
			0x4b,0xd3,0x41,0x2d,0xfd,0xf0,0xff,0x71,0x7f,0x7a,0xac,0x74,0x6f,0x31,0x0b,0xfa,
			0x9a,0x14,0x1a,0xc3,0xdb,0xf0,0x1c,0x1f,0xc7,0x4f,0xeb,0xd1,0x97,0x93,0x84,0x19,
			0xc2,0x62,0x29,0x35,0x05,0xc3,0x5f,0x40,0x2f,0x90,0x53,0xad,0x13,0xc5,0x1a,0x59,
			0x60,0xec,0xde,0x55,0xec,0x82,0x9e,0x95,0x3f,0x94,0x1a,0xf7,0x33,0xe5,0x87,0x05,
			0x91,0x37,0x67,0xe7,0xa7,0x20,0x0d,0x1d,0x09,0xe7,0xe7,0xe2,0xd2,0x69,0xfa,0x29,
			0xa5,0x58,0xbb,0x16,0x30,0x4b,0x05,0x9f,0x13,0xf4,0xca,0x56,0x0a,0x81,0x01,0xfe,
			0x37,0x20,0xb4,0xa7,0x79,0xec,0x12,0x64,0x27,0x32,0x6c,0xaa,0x13,0x2a,0x3d,0x36,
			0x11,0xd7,0xdb,0xc5,0x03,0x36,0xfa,0xc7,0x89,0xec,0x40,0x6b,0x39,0x7e,0x1e,0x36,
			0xd7,0xda,0xf9,0xb6,0x24,0xbf,0x63,0x9c,0x82,0xb8,0x59,0x28,0x87,0x47,0x69,0x0c,
			0x73,0x0c,0x98,0x0b,0x2f,0x5a,0x23,0x9d,0xd9,0x5a,0xd5,0x38,0x9a,0x2e,0xc9,0x0c,
			0x57,0x78,0x60,0x47,0x13,0x71,0x03,0x83,0xae,0x55,0xd4,0xd2,0x8c,0x06,0xd4,0xac,
			0x26,0xf0,0xd1,0x23,0x1f,0x1d,0x67,0x62,0xc8,0xe0,0xd9,0x18,0x11,0x81,0x56,0xbc,
			0x63,0x77,0x60,0xda,0xea,0x18,0x47,0x46,0xb8,0xdc,0xf6,0xf6,0x1d,0xb2,0x74,0xa7,
			0xdd,0xce,0xaa,0x07,0x49,0x37,0xab,0xab,0xad,0x45,0x49,0xb9,0x7a,0xb9,0x92,0x49,
			0x4a,0x80,0x72,0x08,0xab,0xd7,0x89,0x82,0x3f,0x5d,0x75,0xc4,0xb9,0x94,0x08,0x9c,
			0x80,0x72,0xcf,0xc2,0x54,0xe0,0xd8,0x20,0x2f,0xd8,0x96,0x47,0x6e,0x96,0xad,0x9d,
			0x30,0x9a,0x0e,0x8e,0x73,0x01,0x28,0x2f,0x07,0xeb,0x2a,0xe8,0xed,0xef,0xb7,0xdb,
			0xbe,0x13,0xb9,0x6e,0x8b,0x40,0x24,0xc6,0xb8,0x4d,0xe0,0xa0,0x5e,0x15,0x02,0x85),
	.tests = {
	{	/* Format of the EM is incorrect - hash moved to left */
		.exp = FALSE,
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0x4d,0x44,0x66,0x5e,0x0e,0x29,0x87,0xcf,0x7d,0xd1,0xe9,0xd6,0x20,0x01,0x51,0xf5,
			0x00,0xea,0x3b,0x85,0xa2,0x4f,0x72,0xfb,0x38,0xe5,0xdd,0x69,0x19,0xff,0x9c,0x48,
			0x6f,0x6a,0x54,0x74,0x2f,0x96,0xc6,0x78,0x4f,0xf7,0xa7,0x0c,0x33,0xab,0xab,0xdc,
			0x68,0x5d,0xe4,0x8b,0x94,0x69,0x4a,0xf8,0xad,0x41,0x9b,0xc0,0x6a,0x0f,0x65,0x04,
			0xf7,0x9c,0x60,0x6c,0xed,0x70,0x02,0x6b,0x81,0x3b,0x4e,0xc5,0xfc,0x73,0xed,0x4f,
			0x6b,0x3f,0x7f,0xf6,0x97,0x94,0x33,0x03,0x37,0x9d,0x95,0x9e,0x65,0x61,0xa8,0x90,
			0x4f,0xa0,0x35,0x4d,0x2f,0xfd,0x62,0x31,0x7d,0x63,0x86,0x0a,0x62,0xa1,0x3c,0xd7,
			0x71,0x57,0xb7,0xcf,0x92,0x16,0x55,0xa5,0x3e,0xc3,0x76,0x0b,0xf2,0x3c,0x8e,0x75),
		.s = chunk_from_chars(
			0x4f,0x48,0xe5,0x1a,0x04,0x1e,0x40,0xa0,0x29,0xd5,0x86,0xcb,0x7a,0xfd,0x7e,0xc1,
			0x10,0x31,0xf8,0x6d,0x7b,0xee,0x58,0xfc,0x13,0xa9,0x22,0x7d,0xd2,0x09,0xf1,0xd7,
			0xbe,0xcf,0x32,0x09,0xcd,0x43,0x05,0x5e,0x93,0x16,0x5f,0x04,0xf2,0x32,0x76,0xd4,
			0x6b,0x0b,0x64,0x66,0x9b,0x88,0xc2,0xba,0x54,0xad,0x6b,0xcb,0x5e,0x9d,0xa8,0x18,
			0x10,0x01,0x4e,0x14,0x0e,0xd0,0xa1,0x42,0x34,0xb1,0xfc,0x0f,0xa4,0xcf,0xa4,0xf2,
			0x58,0xa7,0xb5,0xcf,0x73,0xa3,0x36,0x53,0x4f,0x37,0x58,0x0a,0x38,0x30,0x87,0x5f,
			0x31,0x19,0x09,0x2e,0x48,0x54,0xfa,0x80,0xd8,0x78,0x33,0xf7,0xe9,0x62,0x7a,0xb5,
			0x1e,0x05,0x6a,0x8d,0xef,0xf3,0xa4,0x51,0xe8,0xaa,0xd7,0x83,0x35,0x81,0x6b,0x53,
			0xae,0x0d,0x89,0x9b,0x06,0x07,0x40,0x3b,0xf0,0x8e,0x8c,0x32,0xe1,0xcf,0x0b,0xaa,
			0x8f,0xe0,0xd7,0x8d,0x1b,0x8d,0x72,0xf8,0x5f,0x17,0xc1,0xc5,0xb8,0x87,0x0c,0xec,
			0x49,0x94,0x97,0xb4,0x2d,0xe9,0x82,0x40,0x8f,0xba,0xdb,0xe3,0xb4,0x5a,0x13,0xb4,
			0xb5,0x37,0x1c,0xe2,0xa4,0xd9,0x60,0x0b,0x7d,0x14,0x01,0x16,0x25,0xcb,0xe7,0x3a,
			0x79,0x31,0x39,0x54,0xf4,0xf3,0xec,0x0b,0x3b,0xf6,0xdc,0x94,0x52,0x97,0x32,0x5e,
			0x3a,0xaf,0x5b,0xc8,0x9a,0x27,0x67,0x0f,0x5b,0x25,0x36,0xa5,0x4c,0xae,0xa3,0xbc,
			0x66,0x41,0xae,0x62,0x8a,0x9a,0x4d,0xa6,0x1d,0xe7,0x49,0x66,0x1a,0x93,0xb6,0xae,
			0x68,0x72,0x2d,0x10,0xd7,0x90,0x2b,0x39,0x1f,0x9e,0x9e,0xbf,0x3a,0xb9,0xc1,0x86,
			0x01,0x7b,0x52,0xea,0xc6,0xfd,0x44,0x2a,0x25,0x6f,0xe7,0xc9,0x50,0x01,0x65,0xe3,
			0xc7,0xa4,0x02,0xaa,0x01,0x74,0xe6,0xb6,0xf3,0x5e,0x4e,0x12,0xe3,0x5a,0x9d,0x42,
			0x9b,0x21,0xf6,0xcc,0x98,0xeb,0xa7,0x3c,0x80,0xed,0x8a,0xc3,0x3a,0xcd,0x79,0x48,
			0x8e,0x62,0x0b,0x27,0x49,0x23,0xe7,0x8c,0xe5,0xd9,0x1a,0x61,0x96,0xca,0x59,0x05,
			0x4f,0xa7,0x7c,0xd6,0x7e,0x0d,0x3f,0xc0,0x4d,0x71,0xb9,0x4b,0x0c,0xda,0x54,0x78,
			0xf1,0xb3,0xd6,0x91,0x50,0xf4,0x09,0x27,0xc1,0xab,0x39,0x20,0x36,0x58,0x36,0x09,
			0xd8,0xef,0xda,0x3f,0xe4,0x86,0x43,0xef,0x2f,0x75,0xf1,0x16,0x4e,0xa7,0xf2,0x55,
			0x89,0xb5,0xbb,0xb5,0x60,0xc9,0xa0,0x42,0x40,0x8f,0x85,0x94,0x4c,0xa0,0x8f,0x92),
	},
	{	/* Format of the EM is incorrect - 00 on end of pad removed */
		.exp = FALSE,
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0x1a,0xbc,0xa8,0xd8,0x1c,0xc9,0x8a,0x5d,0x50,0x20,0xe9,0x50,0x80,0xb5,0xfc,0xe5,
			0x9f,0xfe,0x70,0xdc,0xff,0x54,0x48,0x02,0xc4,0x9d,0xe7,0xfa,0x99,0x28,0x6b,0x3d,
			0xa1,0xd8,0x3d,0x60,0xb0,0x85,0xe6,0x96,0xbc,0x9f,0x25,0x75,0x8e,0x14,0x3e,0xfd,
			0xbd,0x74,0x94,0xd6,0xdd,0x54,0x7a,0xd9,0xf4,0xf0,0x47,0xd2,0x2d,0x14,0xa8,0x84,
			0xd1,0x3c,0x3b,0xee,0x72,0xfa,0x59,0xcd,0x6a,0x98,0x33,0x62,0x84,0xfe,0x20,0x75,
			0xf8,0xd7,0x35,0x9a,0x9d,0xf0,0x17,0xb3,0x5a,0xe0,0x0d,0xb6,0xdd,0xee,0x59,0xb1,
			0x90,0x2f,0xc9,0x0b,0xc9,0xef,0x89,0x0c,0x42,0x52,0x3b,0x9f,0xc8,0x3b,0x49,0x57,
			0x37,0x6d,0x7a,0xb1,0xa5,0xf1,0xc4,0x99,0xde,0xb7,0x62,0x2f,0x11,0x80,0x26,0xca),
		.s = chunk_from_chars(
			0x85,0xe1,0x57,0x65,0xa4,0x01,0xc1,0x0d,0x86,0xac,0xf3,0xa0,0x5f,0xab,0x14,0xe0,
			0x54,0x87,0xb7,0x93,0x56,0x9a,0xe6,0x80,0xf7,0x7d,0x20,0xc2,0x5b,0x17,0x63,0xd1,
			0x84,0x63,0x4f,0xa0,0xb0,0x1d,0x97,0x9d,0xce,0x9d,0x80,0x36,0x70,0xd7,0xf9,0xf6,
			0xad,0xcd,0x59,0x47,0x6f,0xb1,0xc8,0xc3,0xaa,0xe5,0x12,0xf9,0x4a,0x9c,0xcf,0x6e,
			0x57,0x3a,0xee,0x6a,0x28,0xc0,0xdc,0xfd,0x2d,0xae,0x85,0x22,0x68,0x5f,0x03,0xec,
			0xec,0x8c,0x04,0x50,0x36,0xd1,0xcb,0x05,0x4a,0x08,0x64,0x54,0xa2,0x83,0x26,0xd5,
			0x31,0x9b,0xc7,0x00,0x4b,0xc8,0x7c,0x75,0x66,0xa5,0x68,0x52,0x55,0x8f,0x58,0x3a,
			0x38,0xa4,0xe7,0xe7,0xbe,0x79,0x81,0xdc,0xf3,0xec,0x41,0xc1,0x23,0x45,0xf8,0x45,
			0x40,0x97,0x92,0xed,0x87,0xb3,0xf1,0xb8,0xfc,0x89,0xc2,0x88,0xa2,0x96,0x8e,0x73,
			0x27,0x3a,0x36,0xf2,0xb6,0x72,0x58,0xfc,0x67,0xe8,0x88,0x60,0x85,0x66,0xa2,0x3a,
			0x6a,0x56,0x45,0x32,0x8f,0x1e,0xdd,0x8c,0x10,0x59,0x93,0xb4,0x86,0xeb,0x33,0x35,
			0x23,0x22,0x12,0xa1,0xdc,0xfd,0x5d,0xa4,0x0f,0xb3,0x90,0x2f,0xd2,0xcf,0xa8,0x6e,
			0x85,0x88,0xeb,0x3c,0x40,0xdf,0x8d,0x9e,0xdf,0x3c,0x41,0x93,0x4c,0xe9,0x1b,0x50,
			0xc0,0xd5,0xca,0x1a,0x38,0x40,0xca,0x40,0xe3,0xc8,0x41,0xb3,0x95,0x20,0xcb,0x5e,
			0x56,0x54,0xaf,0x39,0x61,0xec,0x94,0x08,0xd2,0x2e,0xdb,0x34,0x50,0x27,0xee,0xa7,
			0x11,0x9d,0xd5,0x52,0x65,0x63,0x42,0xe3,0x16,0xc6,0x0c,0x5f,0x5d,0xa0,0x28,0x30,
			0x8b,0x03,0x1b,0xac,0xc7,0x4d,0x1f,0xdd,0x69,0x07,0xbc,0x69,0x44,0xfb,0x85,0x60,
			0x82,0xcb,0x14,0xf1,0x2c,0xb5,0x09,0xca,0x4e,0xfb,0x51,0x6e,0xd0,0xc6,0x08,0x96,
			0x77,0x1b,0x47,0x84,0x42,0x02,0x69,0x13,0xe6,0x1a,0x32,0x61,0x1d,0xde,0x1a,0x88,
			0x07,0xf2,0xe1,0xcb,0x76,0xd0,0xc0,0xc6,0x9c,0x70,0x5b,0xeb,0x8f,0xe7,0x13,0xe8,
			0x48,0xcc,0x03,0xb1,0xef,0x89,0x65,0x26,0x7c,0xfc,0x3e,0xbd,0x43,0xf0,0x00,0x1b,
			0x33,0xb9,0xe9,0xad,0x41,0x8e,0xe0,0xd5,0x92,0xf1,0x84,0xd2,0x02,0x33,0x1d,0x4b,
			0x51,0xa9,0x6a,0x17,0xb0,0x14,0xd4,0x71,0x38,0xf2,0x8c,0xaa,0x39,0xd9,0xdf,0xbd,
			0xa6,0x08,0x94,0x95,0x6d,0x0c,0x04,0x3e,0x42,0x92,0x87,0x7f,0x04,0x4b,0x55,0xa2),
	},
	{	/* Signature changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0x20,0x85,0xe8,0xb9,0x46,0xd2,0x05,0x97,0x52,0xf8,0xf0,0xfe,0xe5,0x28,0xe7,0xc4,
			0xcc,0x0a,0xc1,0xfa,0x24,0x53,0x2c,0x1e,0xc2,0x9a,0x3a,0x92,0xd2,0x68,0x1c,0x4d,
			0xbd,0x8e,0x2d,0xd1,0x51,0xbb,0x32,0x5d,0x1c,0x4d,0xdf,0x63,0x18,0x68,0x9e,0x3c,
			0x8e,0x45,0x8d,0x1b,0x34,0xcd,0xc5,0x40,0x98,0x67,0xe2,0x6e,0xc2,0x0e,0xc0,0x9e,
			0x15,0xa6,0x82,0xf5,0x77,0x90,0xd9,0xf4,0xb2,0x7c,0x83,0xef,0x19,0x7e,0x11,0xc0,
			0x6e,0x65,0xee,0xbc,0x03,0x48,0x9d,0xe6,0x20,0x61,0x42,0x63,0xa5,0x1e,0x64,0x2e,
			0xf0,0x76,0xf1,0x0d,0x01,0x7a,0x82,0xc4,0xdc,0x2c,0xe1,0x88,0x2f,0x93,0xc9,0xb6,
			0x35,0x0f,0xec,0xc5,0xda,0xeb,0x3a,0x7a,0x20,0x31,0x61,0x5a,0xc1,0xc2,0x90,0x65),
		.s = chunk_from_chars(
			0x65,0xbd,0x57,0xd4,0xa3,0x74,0x95,0xd5,0x4e,0xec,0x7f,0x63,0x21,0xf7,0x10,0x1a,
			0x46,0x43,0xd9,0x43,0x2a,0x2a,0xfc,0x8e,0xaa,0x14,0x7f,0xb2,0x9e,0x7a,0xb1,0x10,
			0x6e,0xb3,0x1a,0x6e,0xa7,0xae,0x3b,0x48,0xe4,0x0d,0xe6,0x3c,0xc3,0x68,0xe7,0xa6,
			0xdf,0x4c,0x99,0x82,0x59,0xc5,0x9c,0xe4,0x0b,0x8a,0xc2,0xf8,0x76,0x3b,0xee,0x91,
			0x92,0x41,0x7d,0x20,0x20,0xcb,0xe1,0x79,0xbb,0xa8,0xae,0xe9,0xbe,0x89,0x71,0x46,
			0x6f,0x5e,0x53,0xb3,0xf8,0x20,0x04,0xab,0x6a,0xa5,0x29,0xd0,0x85,0x45,0xbd,0x89,
			0x44,0x35,0xd8,0xdc,0xee,0xb3,0xa8,0xe9,0x05,0x91,0x6b,0xbd,0xb0,0x15,0xbf,0xf6,
			0x5e,0x9e,0x79,0x2a,0x88,0xe1,0xc8,0x44,0x9f,0x41,0x1e,0xda,0xcb,0xa5,0x04,0xcd,
			0x07,0x30,0x60,0x5d,0xc8,0x3f,0xee,0x64,0xf8,0xdd,0x17,0xf5,0x77,0x07,0x56,0x94,
			0x40,0x2f,0x18,0x62,0x79,0x0d,0x96,0x29,0xf9,0xfa,0x8a,0xc3,0xab,0x4e,0xe3,0x41,
			0xdd,0x39,0x79,0xeb,0x17,0x15,0xc4,0xf7,0x42,0xfb,0xd8,0x9b,0xa3,0xb9,0x05,0x65,
			0xea,0x45,0x87,0x43,0x85,0x62,0x49,0x3f,0x61,0x1d,0x5a,0xc9,0x98,0x9f,0xd4,0x05,
			0xa0,0x1d,0x23,0xb1,0x34,0x69,0xa2,0xa7,0xa8,0x5b,0x00,0x47,0x48,0x3a,0x66,0x5c,
			0xe3,0x89,0x9a,0x41,0x21,0x07,0xca,0xcc,0x11,0x08,0x7c,0xb2,0x60,0x7b,0xa9,0xd0,
			0x40,0x91,0xc4,0x18,0x24,0x72,0x42,0xcd,0x8d,0x81,0x9a,0xe1,0x7e,0x49,0xa2,0x1b,
			0xc4,0xf9,0x66,0xac,0x3d,0x37,0xd2,0xa6,0x1a,0x8b,0xfe,0xed,0xe0,0xf2,0xef,0x2d,
			0xa3,0xaa,0x32,0xbc,0x38,0xab,0xb6,0xa1,0x6a,0x44,0x75,0xe9,0x67,0x2c,0x4a,0xaa,
			0x31,0xd2,0xe8,0x27,0x7f,0x3d,0xd9,0x66,0x3f,0x39,0x60,0xf1,0x6c,0x01,0x99,0x8d,
			0x50,0xa9,0x9b,0xcc,0x68,0x1e,0xa0,0x2b,0x13,0x75,0x55,0xaf,0x2c,0xdd,0x28,0xea,
			0x43,0x28,0x15,0x88,0x06,0x6b,0xc5,0x5c,0x8f,0x7a,0xab,0x45,0xde,0xdf,0xa2,0xf6,
			0xbe,0xfb,0x5b,0x13,0x91,0x1c,0xb9,0xce,0xc2,0xb3,0xfb,0x07,0xbc,0x00,0xb2,0xda,
			0xca,0xff,0x82,0x37,0x20,0x1d,0x3e,0xbe,0xac,0x57,0x50,0x83,0x01,0x16,0x24,0xb4,
			0xbe,0x01,0x83,0x79,0x51,0x4d,0xec,0x97,0x20,0x9f,0xaa,0x29,0xb1,0xf9,0x3f,0xda,
			0x75,0xf0,0x77,0x0f,0x3c,0x73,0xe7,0x00,0xa2,0xe4,0x84,0x53,0x06,0x7c,0x90,0x9e),
	},
	{
		.exp = TRUE,
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0xcc,0x21,0x59,0x3a,0x6a,0x0f,0x73,0x7e,0x29,0x70,0xb7,0xc0,0x79,0x84,0xb0,0x70,
			0xd7,0x61,0x72,0x62,0x96,0xa0,0x7e,0x24,0xe0,0x56,0xe6,0x8f,0xf8,0x46,0xb2,0x9c,
			0xc1,0x54,0x81,0x79,0x84,0x3d,0x74,0xdc,0xee,0x86,0x47,0x98,0x58,0xb2,0xc1,0x6e,
			0x4c,0xb8,0x4f,0x25,0x44,0xb4,0xec,0xdc,0xb4,0xdd,0x43,0xa0,0x4b,0xb7,0x18,0x3a,
			0x76,0x8a,0xe4,0x4a,0x27,0x12,0xbf,0x9a,0xd4,0x78,0x83,0xac,0xc2,0x81,0x2f,0x95,
			0x83,0x06,0x89,0x0e,0xbe,0xa4,0x08,0xc9,0x2e,0xb4,0xf0,0x01,0xed,0x7d,0xbf,0x55,
			0xf3,0xa9,0xc8,0xd6,0xd9,0xf6,0x1e,0x5f,0xe3,0x2e,0xb3,0x25,0x3e,0x59,0xc1,0x8e,
			0x86,0x31,0x69,0x47,0x8c,0xd6,0x9b,0x91,0x55,0xc3,0x35,0xdb,0x66,0x01,0x6f,0x96),
		.s = chunk_from_chars(
			0x0a,0xa5,0x72,0xa6,0x84,0x5b,0x87,0x0b,0x89,0x09,0xa6,0x83,0xbb,0x7e,0x6e,0x76,
			0x16,0xf7,0x7b,0xef,0xf2,0x87,0x46,0x11,0x6d,0x8b,0xc4,0xb7,0x33,0x55,0x46,0xb5,
			0x1e,0x80,0x06,0xed,0x0f,0xc9,0xa0,0xd6,0x6f,0x63,0xce,0x0b,0x9e,0xbf,0x79,0x2d,
			0x7e,0xfd,0x43,0x05,0xd7,0x62,0x4d,0x54,0x54,0x00,0xa5,0xfd,0x6a,0x06,0xb7,0x8f,
			0x17,0x4b,0x86,0x80,0x3f,0x7c,0xd1,0xcc,0x93,0xe3,0xa9,0x72,0x86,0xf0,0xea,0x59,
			0x0e,0x40,0xff,0x26,0x19,0x5a,0xa2,0x19,0xfe,0x15,0x10,0xa0,0x16,0x78,0x52,0x23,
			0x60,0x6d,0x93,0x11,0xa1,0x6c,0x59,0xa8,0xfe,0x4a,0x6d,0xa6,0xec,0xd0,0xc1,0xd7,
			0x77,0x50,0x39,0x29,0x0c,0x2a,0xaa,0x17,0xed,0x1e,0xb1,0xb5,0x43,0x74,0xf7,0xe5,
			0x72,0xdb,0x13,0xcc,0xa3,0xa6,0x38,0x57,0x5f,0x80,0x04,0xaa,0x54,0xa2,0xfa,0x98,
			0x42,0x2f,0xc0,0x7e,0x43,0xad,0x3a,0x20,0xdd,0x93,0x00,0x14,0x93,0x44,0x26,0x77,
			0xd8,0x83,0x91,0x4d,0xc7,0x4e,0xc1,0xcb,0xeb,0xbb,0xd3,0xd2,0xb6,0xba,0xd4,0x66,
			0x6d,0x91,0x45,0x7b,0x69,0xb4,0x6a,0x1a,0x61,0xf2,0x12,0x98,0xf1,0xa6,0x79,0x42,
			0xec,0x86,0xc8,0x76,0x32,0x2d,0xd3,0x66,0xed,0x16,0x78,0x14,0xe9,0xc8,0xfc,0x90,
			0x40,0xc5,0xb4,0xb7,0xa8,0x59,0xbb,0xd8,0x80,0xcb,0x6b,0xc2,0x41,0xb9,0xe3,0x27,
			0xce,0x77,0x9e,0x07,0x83,0xb1,0xcf,0x44,0x5e,0x0b,0x2f,0x57,0x71,0xb3,0xf5,0x82,
			0x2a,0x13,0x64,0x39,0x1c,0x15,0x4d,0xc5,0x06,0xff,0xf1,0xfb,0x9d,0x9a,0x35,0xf8,
			0x01,0x99,0xa6,0xb3,0x0b,0x4b,0x92,0xb9,0x26,0x19,0xa4,0x0e,0x21,0xae,0xa1,0x92,
			0x84,0x01,0x58,0x63,0xc4,0x48,0x66,0xc6,0x1e,0xd9,0x04,0xa7,0xad,0x19,0xee,0x04,
			0xd9,0x66,0xc0,0xaa,0xe3,0x90,0x63,0x62,0x43,0x56,0x55,0x81,0xff,0x20,0xbd,0x6e,
			0x3c,0xfb,0x6e,0x31,0xf5,0xaf,0xba,0x96,0x4b,0x31,0x1d,0xc2,0xd0,0x23,0xa2,0x19,
			0x98,0xc8,0xdd,0x50,0xca,0x45,0x36,0x99,0x19,0x0b,0xd4,0x67,0x42,0x9e,0x2f,0x88,
			0xac,0xe2,0x9c,0x4d,0x1d,0xa4,0xda,0x61,0xaa,0xc1,0xed,0xa2,0x38,0x02,0x30,0xaa,
			0x8d,0xbb,0x63,0xc7,0x5a,0x3c,0x1e,0xc0,0x4d,0xa3,0xa1,0xf8,0x80,0xc9,0xc7,0x47,
			0xac,0xdb,0x74,0xa8,0x39,0x5a,0xf5,0x8f,0x5f,0x04,0x40,0x15,0xcc,0xaf,0x6e,0x94),
	},
	{	/* Public Key e changed */
		.exp = FALSE,
		.e = chunk_from_chars(0xa4,0x5b,0x6b),
		.m = chunk_from_chars(
			0xa5,0x89,0xc8,0x78,0x8c,0x95,0x99,0x61,0xfe,0xf9,0x86,0x94,0xbb,0xfe,0xff,0xce,
			0x5d,0x69,0x07,0x18,0x99,0xad,0x96,0x9a,0xc2,0x5f,0x3c,0xb4,0x8e,0xea,0x08,0x4b,
			0x1d,0x84,0xa8,0x61,0x37,0x61,0xd1,0xe9,0xd6,0x26,0xe2,0xd9,0xe4,0xa0,0xc4,0x80,
			0x45,0xb6,0x14,0x1a,0x18,0x9c,0x84,0xa2,0x3a,0x4a,0x7e,0xe7,0x0c,0x2d,0x0b,0xe2,
			0x77,0x1c,0xf4,0x47,0x2d,0x8d,0x27,0x5a,0x31,0x09,0x5b,0x04,0x99,0xdd,0xf7,0x26,
			0x93,0x13,0x30,0x2d,0xa6,0xa0,0x72,0xe7,0x3a,0xdf,0x02,0x95,0x5b,0x3e,0xe1,0x41,
			0xba,0x38,0xf3,0x51,0xf4,0x83,0x60,0x5d,0x17,0x8a,0x3b,0x3e,0x63,0x1d,0x62,0x67,
			0x4d,0x67,0xa5,0x79,0xca,0x21,0x2a,0x11,0xc9,0x06,0x0f,0xe4,0x01,0x87,0x52,0x0f),
		.s = chunk_from_chars(
			0x06,0x24,0x18,0x75,0x08,0x45,0xd3,0x29,0x1e,0x63,0x3e,0x47,0x40,0x64,0x80,0xe3,
			0x2f,0xa2,0x98,0x02,0x75,0xf2,0xe4,0xac,0x79,0xd2,0x84,0xad,0x83,0xed,0x11,0x19,
			0x0e,0x7b,0x85,0xbf,0xd6,0xed,0x9b,0xcf,0xf6,0x33,0x45,0x58,0x2f,0xcc,0xd8,0x8a,
			0x57,0x89,0x41,0xab,0xb9,0x87,0xd2,0xdc,0xb5,0x49,0xa0,0xac,0x2e,0xd5,0xfe,0x7c,
			0xe8,0x25,0x7b,0xb1,0xe6,0x28,0x81,0x55,0xff,0xf3,0xdc,0x9b,0xc1,0x14,0x0f,0x27,
			0x77,0xa8,0x45,0x75,0x98,0x63,0x60,0xd0,0x76,0x60,0x5e,0x65,0x5c,0x9d,0x73,0xf9,
			0xbf,0x01,0x0a,0x43,0x7e,0xb7,0xb2,0x80,0xd6,0x10,0x83,0x74,0xce,0x8d,0xa4,0xc7,
			0x02,0xf6,0x6a,0x9d,0x1d,0x01,0xb1,0x41,0xc6,0xa6,0x98,0x72,0x9d,0x0f,0xcd,0xb0,
			0x63,0x1b,0x21,0x57,0x8d,0xde,0xd7,0x21,0xef,0x5c,0x49,0x08,0x12,0x26,0x6c,0xf6,
			0xf0,0x12,0x70,0xa8,0xc6,0xc2,0xf0,0x84,0x66,0xc9,0xb3,0xc6,0x26,0xc0,0xf1,0xed,
			0xeb,0x5b,0x4f,0xb3,0x2e,0x95,0x8a,0x4a,0x38,0x8c,0x6c,0x6e,0x73,0xb5,0x3a,0x50,
			0x61,0x37,0x35,0x43,0x58,0xbd,0x00,0xbb,0xd2,0x05,0x8c,0x9c,0x12,0x79,0x04,0x10,
			0xa4,0x5f,0x43,0x48,0x75,0x48,0x44,0xb3,0x57,0xc4,0xe1,0x7f,0xb3,0x8c,0x8c,0xac,
			0x87,0xa6,0xbe,0x3c,0xd0,0x6a,0x08,0x94,0x3d,0xb0,0xe8,0xb8,0x67,0xd4,0x73,0xe7,
			0xd3,0x58,0x1e,0xd2,0xa3,0x2d,0x25,0xd7,0xb4,0x5e,0xec,0xf8,0xfe,0xa1,0x0d,0x0a,
			0xde,0x1c,0x88,0x85,0x91,0xc7,0x1f,0xc8,0x6a,0xbd,0x14,0x46,0xfa,0xdc,0xa2,0x88,
			0x1d,0xe8,0x7a,0x6d,0x05,0x56,0x6c,0x50,0x00,0xc1,0xe0,0x75,0xb0,0xc1,0xc5,0x48,
			0x1f,0xfb,0x6e,0x98,0x2a,0x3f,0x8f,0x44,0xfa,0x8a,0xd0,0xb6,0xa9,0x2f,0x1c,0xde,
			0xdd,0x9c,0x38,0xd3,0x26,0xa6,0xef,0xad,0x9d,0x03,0x59,0xa6,0x9a,0xf4,0x20,0x34,
			0xb7,0xa4,0xac,0x38,0x8f,0x9c,0x7d,0x35,0x6e,0xdf,0x4d,0x6e,0x45,0x50,0xad,0xde,
			0xcc,0x71,0x92,0x24,0x2b,0x3b,0x1f,0xc4,0x16,0x46,0x02,0x91,0x9d,0x34,0x11,0xed,
			0x43,0xe2,0xb3,0x94,0x86,0x7b,0xf8,0x40,0xea,0x61,0x75,0x0b,0xac,0x4f,0x1b,0x72,
			0xd2,0x8a,0xf3,0x5e,0xb0,0xb3,0xaa,0xf6,0xee,0xd6,0x54,0xd1,0xde,0x88,0x32,0x40,
			0xc0,0xe2,0x23,0x4e,0xa2,0xdd,0xf6,0xf8,0x93,0x4c,0xdc,0xac,0xea,0x80,0x2e,0x26),
	},
	{	/* Message changed */
		.exp = FALSE,
		.e = chunk_from_chars(0x08,0xa6,0x49),
		.m = chunk_from_chars(
			0x70,0xa4,0xa8,0x98,0x02,0x85,0x68,0x69,0x9c,0xcd,0x8a,0x53,0x28,0x8a,0x74,0x78,
			0x51,0x88,0x4d,0xb2,0x97,0x1a,0xb1,0x7a,0xa1,0x16,0xa9,0x09,0xd4,0x22,0xda,0xbc,
			0x24,0x2e,0xfc,0xbd,0x2a,0x42,0x8e,0xe0,0x77,0x7a,0xc8,0xcb,0x29,0x4f,0xd2,0x5a,
			0xbc,0x17,0x33,0x42,0x22,0xcf,0x8c,0xf4,0x15,0x19,0x86,0xc4,0xac,0x81,0x09,0x4a,
			0x2c,0x04,0xaf,0x08,0x21,0xd3,0x78,0x4e,0xae,0x54,0xbf,0x5b,0x22,0x6a,0x6c,0xc5,
			0xcb,0xf6,0x6c,0x1b,0x0f,0x9e,0x5e,0xc8,0xb8,0xcd,0x8f,0x3a,0x90,0xfa,0x29,0x44,
			0x2e,0x2e,0xa7,0xc1,0x0f,0xc0,0x8c,0x7b,0xe9,0x63,0x55,0x4b,0x3b,0x64,0xac,0x81,
			0x29,0x2c,0xde,0x85,0x87,0x0c,0x8f,0xbf,0x34,0x3a,0xad,0xb5,0xa9,0x16,0xa2,0x73),
		.s = chunk_from_chars(
			0x07,0x88,0x43,0xb0,0x4c,0x98,0x55,0xe6,0x0b,0x09,0x65,0x97,0x0e,0x34,0x85,0xa8,
			0x51,0x70,0x33,0xef,0xb5,0x7b,0x77,0x31,0x50,0x48,0x3e,0x7d,0x79,0xa5,0x49,0xd2,
			0x17,0x15,0x9f,0xc1,0x94,0xc7,0x8d,0x1d,0x7e,0xb8,0x72,0xad,0xe2,0x31,0xd1,0xc9,
			0x5d,0xaf,0xcb,0x7d,0x52,0x92,0xd7,0xb1,0x13,0xa9,0x95,0x67,0xf4,0x92,0xcd,0x84,
			0x6f,0x88,0xcd,0xc1,0xd0,0x35,0xa0,0x4a,0xe6,0xa5,0xef,0x28,0xa8,0x67,0x33,0x87,
			0xa2,0x47,0x9c,0x3e,0x0c,0x5a,0x12,0x04,0xbf,0x13,0x60,0x2b,0x49,0x09,0x7f,0xd4,
			0xbe,0xc1,0x62,0x84,0x56,0x6c,0x36,0xe4,0x5e,0xa3,0x96,0x35,0x67,0xf3,0x6d,0xae,
			0xe1,0xee,0x86,0x69,0x71,0x37,0xcb,0x25,0x82,0x58,0xb5,0xc3,0x18,0x92,0x6f,0xed,
			0x50,0x5c,0xd6,0x78,0x4c,0xcc,0x95,0xde,0x4e,0xdc,0x20,0x34,0xe6,0x45,0x21,0x27,
			0x6f,0x90,0x55,0x82,0x44,0x79,0xb4,0xab,0xd5,0x1a,0x73,0x15,0xe8,0x1e,0x20,0x4d,
			0xc0,0x0f,0x4b,0x07,0x7f,0x26,0xea,0x0a,0x72,0x56,0xbc,0xfb,0x0e,0xdf,0x88,0x62,
			0x4c,0x4e,0x3a,0x05,0x99,0x14,0xb7,0x5d,0xdf,0xcf,0x94,0x5c,0x56,0x20,0xeb,0x6b,
			0x06,0xb3,0x01,0x1e,0x4b,0xab,0xc2,0x68,0x57,0xbd,0xd9,0xbb,0x06,0x5d,0xfd,0xe5,
			0x15,0xc3,0x7e,0xaa,0x6a,0xd8,0x35,0x90,0x53,0x75,0x66,0x68,0x0a,0xd1,0xd7,0x76,
			0x06,0x78,0x27,0x28,0x8c,0x80,0x49,0x7b,0x76,0xc4,0x89,0xcb,0x39,0x76,0x2d,0xc1,
			0x99,0x7f,0xaa,0x41,0x08,0x6b,0x1e,0x37,0x7d,0x3c,0x9c,0xb2,0x9e,0x45,0x99,0xde,
			0xa3,0xb1,0x23,0xd8,0x50,0x46,0x3e,0xd2,0x00,0x87,0xfa,0x74,0x4a,0x54,0x81,0xa4,
			0x53,0x7a,0x62,0x52,0xae,0xa0,0xbc,0x18,0xb2,0xbc,0x96,0x2d,0xeb,0x58,0x9a,0xe5,
			0x3a,0x88,0x23,0x9e,0xac,0x7e,0x79,0x06,0xf4,0xb1,0x6e,0x68,0xc0,0xe7,0xde,0x24,
			0x78,0xc4,0xe4,0xda,0x86,0x90,0x08,0xa6,0x54,0x55,0xc6,0x91,0x54,0x7b,0x16,0xa9,
			0xd9,0x64,0xfe,0x0f,0xaa,0xac,0x4a,0xbb,0x22,0x30,0x9e,0x07,0x53,0x40,0x8e,0x22,
			0xa8,0x66,0x8c,0x0e,0x96,0x68,0x1f,0x0f,0xed,0xf3,0xad,0xeb,0xd7,0xc3,0xc6,0xea,
			0x0b,0x45,0x2e,0x07,0xa6,0xd9,0x4d,0x94,0x37,0xe8,0xb0,0x3e,0x7b,0x26,0x20,0x3a,
			0xff,0x7c,0x17,0x06,0xb5,0x8a,0x20,0x56,0x86,0x42,0xa8,0x71,0x9b,0x98,0x38,0xf0),
	},
	},
},
};

START_TEST(test_verify_pss)
{
	public_key_t *pubkey;
	bool result;
	int i;

	if (!lib->plugins->has_feature(lib->plugins,
					PLUGIN_PROVIDE(PUBKEY_VERIFY, SIGN_RSA_EMSA_PSS)))
	{
		warn("%N not supported, skip test", signature_scheme_names,
			 SIGN_RSA_EMSA_PSS);
		return;
	}
	for (i = 0; i < countof(pss_verify[_i].tests); i++)
	{
		pubkey = lib->creds->create(lib->creds, CRED_PUBLIC_KEY, KEY_RSA,
							BUILD_RSA_MODULUS, pss_verify[_i].n,
							BUILD_RSA_PUB_EXP, pss_verify[_i].tests[i].e,
							BUILD_END);
		ck_assert(pubkey != NULL);
		result = pubkey->verify(pubkey, SIGN_RSA_EMSA_PSS,
								&pss_verify[_i].params,
								pss_verify[_i].tests[i].m,
								pss_verify[_i].tests[i].s);
		fail_unless(result == pss_verify[_i].tests[i].exp,
					"verify %N [%d]", signature_scheme_names,
					SIGN_RSA_EMSA_PSS, i);
		pubkey->destroy(pubkey);
	}
}
END_TEST

Suite *rsa_suite_create()
{
	Suite *s;
	TCase *tc;
	int gen_count = countof(key_sizes);

	s = suite_create("rsa");

	if (getenv("TESTS_REDUCED_KEYLENGTHS") != NULL)
	{
		gen_count = min(1, gen_count);
	}

	tc = tcase_create("generate");
	tcase_add_loop_test(tc, test_gen, 0, gen_count);
	tcase_set_timeout(tc, 8);
	suite_add_tcase(s, tc);

	tc = tcase_create("load");
	tcase_add_loop_test(tc, test_load, 0, countof(keys));
	suite_add_tcase(s, tc);

	tc = tcase_create("sign");
	tcase_add_checked_fixture(tc, setup_keys, teardown_keys);
	tcase_add_loop_test(tc, test_sign_pkcs15_2048, 0, countof(pkcs15_2048));
	tcase_add_loop_test(tc, test_sign_pkcs15_3072, 0, countof(pkcs15_3072));
	suite_add_tcase(s, tc);

	tc = tcase_create("sign pss");
	tcase_add_checked_fixture(tc, setup_keys_pss, teardown_keys_pss);
	tcase_add_loop_test(tc, test_sign_pss_2048, 0, countof(pss_sign_2048));
	tcase_add_loop_test(tc, test_sign_pss_3072, 0, countof(pss_sign_3072));
	tcase_add_loop_test(tc, test_sign_pss_no_salt, 0, countof(pss_sign_ns));
	suite_add_tcase(s, tc);

	tc = tcase_create("verify");
	tcase_add_loop_test(tc, test_verify_pkcs15, 0, countof(pkcs15_verify));
	tcase_add_loop_test(tc, test_verify_pss, 0, countof(pss_verify));
	suite_add_tcase(s, tc);

	return s;
}
