/*
 * Copyright (c) 2013-2016, The Linux Foundation. All rights reserved.
 * Copyright (c) 2017-2021, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*****************************************************************************
 * Includes
 *****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>
#include <linux/spinlock.h>
#include <linux/fb.h>
#include <drm/drmP.h>
#include <drm/drm_fb_helper.h>

#ifndef CONFIG_DRM_NEC_FB_RESERVED
#define CONFIG_DRM_NEC_FB_RESERVED 0
#endif // CONFIG_DRM_NEC_FB_RESERVED

#ifdef NEC_PANEL_ENABLE_PERF_CHECK
#include <time.h>
#include <sys/time.h>
#endif // NEC_PANEL_ENABLE_PERF_CHECK

#include "nec_fb.h"
#include "nec_panel.h"


/*****************************************************************************
 * Types
 *****************************************************************************/
/* for debug */
#define NEC_PANEL_DEBUG		(0)
#define NEC_PANEL_VERBOSE	(0)

#if NEC_PANEL_DEBUG
#define NEC_PANEL_DPRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define NEC_PANEL_DPRINT(fmt, ...)
#endif /* NEC_PANEL_DEBUG */

#if NEC_PANEL_VERBOSE
#define NEC_PANEL_VPRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define NEC_PANEL_VPRINT(fmt, ...)
#endif /* NEC_PANEL_VERBOSE */

#define NEC_PANEL_PRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#define NEC_PANEL_PRINT_ERR(fmt, ...) \
		pr_err("%s: " fmt, __func__, ## __VA_ARGS__)


/*****************************************************************************
 * Statics
 *****************************************************************************/
static bool   nec_panel_init = false;

static struct nec_panel_resource	*nec_panel_res = NULL;
static struct nec_panel_lcd_operation	*nec_lcd_ops   = NULL;
static struct nec_fb_panel_operation	*nec_panel_ops = NULL;

static struct mutex send_cmd_mutex = __MUTEX_INITIALIZER(send_cmd_mutex);
static struct mutex panel_on_mutex = __MUTEX_INITIALIZER(panel_on_mutex);

/*****************************************************************************
 * Private funciton declairs
 *****************************************************************************/
static void nec_panel_set_panel_is_on(bool on, int user);
static bool nec_panel_is_panel_on_with_state_change(bool change_condition);
static void nec_panel_set_panel_change_state(int state);
static int  nec_panel_get_panel_change_state(void);
static void nec_panel_wait_panel_state_change(void);
static bool nec_panel_is_panel_using_by_other(int user);
static void nec_panel_set_panel_use(int user, bool use);

static void nec_panel_send_command_locked(u8 cmd, u8 *data, u32 data_cnt);
static void nec_panel_read_command_locked(u8 cmd, u32 data_cnt);

static void nec_panel_perf_check_start(int part_type);
static void nec_panel_perf_check_end(int part_type);
static void nec_panel_perf_check_count_inc(void);
static void nec_panel_print_perf_result(void);
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
static const char * nec_panel_get_perf_type_str(int type);
#endif // NEC_PANEL_ENABLE_PERF_CHECK
static void nec_panel_update_status(void);

/*****************************************************************************
 * Panel operations (for FB driver)
 *****************************************************************************/
static int  nec_panel_alloc_fb_mem(struct nec_fb_data *nfd);
static void nec_panel_reset_panel(bool hw_reset, bool sw_reset);
static int  nec_panel_pan_display(struct nec_fb_data *nfd,
					struct fb_var_screeninfo *var);
static int  nec_panel_write_lcd_data(u8 cmd, u8 *data, u32 data_cnt);
static int  nec_panel_read_lcd_data(u8 cmd, u32 data_cnt);


/*****************************************************************************
 * Driver register/unregister operations
 *****************************************************************************/
static int  nec_panel_probe(struct platform_device *pdev);
static int  nec_panel_init_lcd_operation(void);
static int  nec_panel_init_panel_operation(
				struct nec_fb_panel_operation *panel_ops);
static int  nec_panel_init_panel_io(struct platform_device *pdev,
				struct nec_panel_io *panel_io);
static int  nec_panel_init_panel_info(struct nec_panel_info *panel_info);

static int  nec_panel_remove(struct platform_device *pdev);

static struct drm_panel *_panel = NULL;

void nec_lcd_attach(struct drm_panel *panel, struct nec_panel_lcd_operation *ops)
{
	if (panel) {
		_panel = panel;
	}

	if (!ops)
		return;

	if (nec_lcd_ops) {	// When attach after probed this.
		memcpy(nec_lcd_ops, ops, sizeof(struct nec_panel_lcd_operation));
	}
	else {				// When attach first before probing this.
		nec_lcd_ops = ops;
	}
	NEC_PANEL_VPRINT("lcd ops attached\n");
}

/*****************************************************************************
 * Private funcitons
 *****************************************************************************/
static void nec_panel_set_panel_is_on(bool on, int user)
{
	mutex_lock(&panel_on_mutex);
	nec_panel_res->panel_change_state = NEC_PANEL_STATE_CHANGED;
	nec_panel_res->panel_is_on = on;
	if (on)
		nec_panel_res->use_panel[user] = true;
	else
		nec_panel_res->use_panel[user] = false;
	mutex_unlock(&panel_on_mutex);
}

static bool nec_panel_is_panel_on_with_state_change(bool change_condition)
{
	bool panel_is_on = false;

	nec_panel_wait_panel_state_change();

	mutex_lock(&panel_on_mutex);
	panel_is_on = nec_panel_res->panel_is_on;
	if (panel_is_on == change_condition)
		nec_panel_res->panel_change_state = NEC_PANEL_STATE_CHANGING;
	mutex_unlock(&panel_on_mutex);

	return panel_is_on;
}

static void nec_panel_set_panel_change_state(int state)
{
	mutex_lock(&panel_on_mutex);
	nec_panel_res->panel_change_state = state;
	mutex_unlock(&panel_on_mutex);
}

static int nec_panel_get_panel_change_state(void)
{
	int state;
	mutex_lock(&panel_on_mutex);
	state = nec_panel_res->panel_change_state;
	mutex_unlock(&panel_on_mutex);
	return state;
}

static void nec_panel_wait_panel_state_change(void)
{
	int state;
	u32 timeout;
	u32 sleep_us = NEC_PANEL_PANEL_STATE_CHECK_CYCLE_US;

	/* Get timeout time */
	timeout = (u32)ktime_to_ms(ktime_get()) +
			NEC_PANEL_MAX_PANEL_CHANGE_WAIT_MS;

	/* Wait panel change state */
	do {
		state = nec_panel_get_panel_change_state();
		if (state == NEC_PANEL_STATE_CHANGED)
			break;

		if (timeout < (u32)ktime_to_ms(ktime_get())) {
			NEC_PANEL_PRINT_ERR("Timed out\n");
			break;
		}

		usleep_range(sleep_us, sleep_us);
	} while (1);
}

static bool nec_panel_is_panel_using_by_other(int user)
{
	int i = 0;
	bool use = false;

	mutex_lock(&panel_on_mutex);
	for (i = 0; i < NEC_PANEL_NUM_USER; i ++) {
		if (user == i)
			continue;

		if (nec_panel_res->use_panel[i]) {
			use = true;
			break;
		}
	}
	mutex_unlock(&panel_on_mutex);

	return use;
}

static void nec_panel_set_panel_use(int user, bool use)
{
	mutex_lock(&panel_on_mutex);
	nec_panel_res->use_panel[user] = use;
	mutex_unlock(&panel_on_mutex);
}

static void nec_panel_send_command_locked(u8 cmd, u8 *data, u32 data_cnt)
{
	int sz = NEC_PANEL_CLAMP(data_cnt, 0, NEC_PANEL_MAX_LCD_CMD_DATA);

	/* Write command */
	if (nec_lcd_ops->lcd_send_cmd) {
		nec_lcd_ops->lcd_send_cmd(_panel, cmd, data, sz);
	}
}

static void nec_panel_read_command_locked(u8 cmd, u32 data_cnt)
{
	int sz = NEC_PANEL_CLAMP(data_cnt, 0, NEC_PANEL_MAX_LCD_CMD_DATA);

	if (nec_lcd_ops->lcd_recv_cmd) {
		(void)nec_lcd_ops->lcd_recv_cmd(_panel, cmd, NULL, sz);
	}
}

static void nec_panel_perf_check_start(int part_type)
{
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	struct timeval tv;
	u32 idx = nec_panel_res->perf_index;

	do_gettimeofday(&tv);
	nec_panel_res->part[part_type].start_sec[idx]  = tv.tv_sec;
	nec_panel_res->part[part_type].start_usec[idx] = tv.tv_usec;
#endif // NEC_PANEL_ENABLE_PERF_CHECK
}

static void nec_panel_perf_check_end(int part_type)
{
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	struct timeval tv;
	u32 idx = nec_panel_res->perf_index;

	do_gettimeofday(&tv);
	nec_panel_res->part[part_type].end_sec[idx]  = tv.tv_sec;
	nec_panel_res->part[part_type].end_usec[idx] = tv.tv_usec;
#endif // NEC_PANEL_ENABLE_PERF_CHECK
}

static void nec_panel_perf_check_count_inc(void)
{
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	u32 idx = nec_panel_res->perf_index;

	nec_panel_res->perf_check_count++;
	nec_panel_res->perf_index = NEC_PANEL_PERF_NEXT_IDX(idx);
#endif // NEC_PANEL_ENABLE_PERF_CHECK
}

static void nec_panel_print_perf_result(void)
{
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	struct nec_panel_resource *res = nec_panel_res;
	u32 check_count = res->perf_check_count;
	u32 type, idx;
	u64 start, end, total;

	check_count = NEC_PANEL_CLAMP(check_count, 0, NEC_PANEL_PERF_NUM_DATA);
	if (!check_count)
		return;

	for (type = 0; type < NEC_PANEL_PERF_NUM_PART; type++, total = 0) {
		for (idx = 0; idx < check_count; idx++) {
			start = ((res->part[type].start_sec[idx] * USEC_PER_SEC) +
				res->part[type].start_usec[idx]);
			end   = ((res->part[type].end_sec[idx] * USEC_PER_SEC) +
				res->part[type].end_usec[idx]);
			total += (end - start);
		}
		NEC_PANEL_PRINT("type=%-5s: total:%10llu usec / %d times)\n",
				nec_panel_get_perf_type_str(type),
				total, check_count);
	}
#endif // NEC_PANEL_ENABLE_PERF_CHECK
}

#ifdef NEC_PANEL_ENABLE_PERF_CHECK
static const char * nec_panel_get_perf_type_str(int type)
{
	switch (type) {
	case NEC_PANEL_PERF_PART_DMA:
		return NEC_PANEL_PERF_PART_NAME_DMA;
	case NEC_PANEL_PERF_PART_SYNC_WAIT:
		return NEC_PANEL_PERF_PART_NAME_SYNC;
	case NEC_PANEL_PERF_PART_PAN:
	default:
		return NEC_PANEL_PERF_PART_NAME_PAN;
	}
}
#endif // NEC_PANEL_ENABLE_PERF_CHECK

static void nec_panel_update_status(void)
{
	int display_on = 0, lcd_on = 0;

	if (nec_lcd_ops->lcd_get_status) {
		nec_lcd_ops->lcd_get_status(_panel, &display_on, &lcd_on);
	}

	mutex_lock(&panel_on_mutex);
	nec_panel_res->display_is_on = display_on;
	nec_panel_res->panel_is_on = lcd_on;
	mutex_unlock(&panel_on_mutex);
}

/*****************************************************************************
 *  Panel Operations
 *****************************************************************************/
static int nec_panel_alloc_fb_mem(struct nec_fb_data *nfd)
{
	size_t size = 0;
	u32    base_size = 0;

	struct fb_info *fbi = nfd->fbi;
	struct fb_fix_screeninfo *fix = &(fbi->fix);
	struct fb_var_screeninfo *var = &(fbi->var);
	struct nec_panel_buf_addr *fb_buf = &(nec_panel_res->fb_buf);

	if (!nec_panel_res) {
		NEC_PANEL_PRINT_ERR("nec panel driver is not initialized\n");
		return -EINVAL;
	}

	/*
	 * Allocate Frame Buffer memory
	 */
	base_size = fix->line_length * var->yres_virtual;
	size = PAGE_ALIGN(base_size);
	if (!(fb_buf->virt)) {
		fb_buf->virt = (void *)dmam_alloc_coherent(
					&(nec_panel_res->pdev->dev),
					size,
					&(fb_buf->phys),
					GFP_KERNEL);
	}
	if (!(fb_buf->virt)) {
#ifndef CONFIG_64BIT
		NEC_PANEL_PRINT_ERR("Failed to alloc fb mem"
					" (sz=0x%08x).\n", size);
#else
		NEC_PANEL_PRINT_ERR("Failed to alloc fb mem"
					" (sz=0x%08lx).\n", size);
#endif // CONFIG_64BIT
		return -ENOMEM;
	}
#ifndef CONFIG_64BIT
	pr_info("framebuffer: size=%d(0x%08x) vaddr=0x%08x paddr=0x%08x\n",
		size, size, (int)fb_buf->virt, (int)fb_buf->phys);
#else
	pr_info("framebuffer: size=%ld(0x%08lx) vaddr=0x%p paddr=0x%016llx\n",
		size, size, fb_buf->virt, fb_buf->phys);
#endif // CONFIG_64BIT

	/* Update fix/var fb info */
	fbi->screen_base = fb_buf->virt;
	fix->smem_start  = fb_buf->phys;
	fix->smem_len    = size;

	return 0;
}

static void nec_panel_reset_panel(bool hw_reset, bool sw_reset)
{
	/* Panel reset */
	if (nec_lcd_ops->lcd_reset)
		nec_lcd_ops->lcd_reset(_panel, hw_reset, sw_reset);
}

static int  nec_panel_pan_display(struct nec_fb_data *nfd,
					struct fb_var_screeninfo *var)
{
	struct drm_fb_helper *helper = get_drm_fb_helper();
	struct fb_info *fbi;
	int err = 0;

	NEC_PANEL_DPRINT("enter \n");
	if (!helper) {
		NEC_PANEL_DPRINT("exit, not yet registered DRM helper \n");
		return -EINVAL;	// Not assigned
	}
	fbi = helper->fbdev;
	if (!fbi) {
		NEC_PANEL_DPRINT("exit, not yet registered DRM framebuffer \n");
		return -EINVAL;	// Not assigned
	}

	nec_panel_perf_check_start(NEC_PANEL_PERF_PART_PAN);

	// Display DRM FB
	err = drm_fb_helper_pan_display(var, fbi);

	nec_panel_perf_check_end(NEC_PANEL_PERF_PART_PAN);
	nec_panel_perf_check_count_inc();

	NEC_PANEL_DPRINT("exit ret = %d\n", err);

	return err;
}

static int nec_panel_write_lcd_data(u8 cmd, u8 *data, u32 data_cnt)
{
	nec_panel_send_command(cmd, data, data_cnt);
	return 0;
}

static int nec_panel_read_lcd_data(u8 cmd, u32 data_cnt)
{
	nec_panel_read_command(cmd, data_cnt);
	return 0;
}


/*****************************************************************************
 *  Public funcitons
 *****************************************************************************/
bool nec_panel_is_initialized(void)
{
	return nec_panel_init && (get_drm_fb_helper());
}

bool nec_panel_is_panel_on(void)
{
	bool panel_is_on = false;

	mutex_lock(&panel_on_mutex);
	panel_is_on = nec_panel_res->panel_is_on;
	mutex_unlock(&panel_on_mutex);

	return panel_is_on;
}


/*****************************************************************************/
int nec_panel_set_lcd_operation(struct nec_panel_lcd_operation *lcd_ops)
{
	nec_lcd_ops = lcd_ops;
	return 0;
}


int nec_panel_get_panel_resource(struct nec_panel_resource **res)
{
	if (!nec_panel_res) {
		NEC_PANEL_PRINT_ERR("nec panel is not initialized\n");
		return -1;
	}

	*res = nec_panel_res;
	return 0;
}

int nec_panel_get_panel_info(struct nec_panel_info *panel_info)
{
	struct nec_panel_info *pinfo = NULL;

	if (!nec_panel_res) {
		NEC_PANEL_PRINT_ERR("nec panel is not initialized\n");
		return -1;
	}

	pinfo = &(nec_panel_res->panel_info);
	panel_info->xres	= pinfo->xres;
	panel_info->yres	= pinfo->yres;
	panel_info->bpp		= pinfo->bpp;
	panel_info->Bpp		= pinfo->Bpp;
	panel_info->max_fps	= pinfo->max_fps;

	return 0;
}


/*****************************************************************************/
void nec_panel_set_no_reset_once(void)
{
	nec_panel_res->with_no_reset = true;
}

int nec_panel_on(int user)
{
	bool with_reset = true;
	int err = 0;

	/* If any user uses panel, skip lcd on */
	if (nec_panel_is_panel_on_with_state_change(false)) {
		nec_panel_set_panel_use(user, true);
		return 0;
	}

	/* Splash image needs 'no-reset' */
	if (user == NEC_PANEL_USER_SPLASH) {
		if (nec_panel_res->with_no_reset) {
			with_reset = false;
			nec_panel_res->with_no_reset = false;
		}
	}

	/* Panel on */
	if (nec_lcd_ops->lcd_on) {
		if (with_reset) {
			nec_panel_reset_panel(true, false);
		}
		err = nec_lcd_ops->lcd_on(_panel);
	}

	nec_panel_set_panel_is_on(true, user);
	return err;
}

void nec_panel_off(int user)
{
	/* Already panel is off */
	if (!nec_panel_is_panel_on_with_state_change(true))
		return;

	/* If any user uses panel, skip lcd off */
	if (nec_panel_is_panel_using_by_other(user)) {
		nec_panel_set_panel_use(user, false);
		nec_panel_set_panel_change_state(NEC_PANEL_STATE_CHANGED);
		return;
	}

	/* Backlight off, if need */
	nec_panel_set_backlight(0);

	/* Display off, if need */
	nec_panel_display_off();

	/* Panel off */
	if (nec_lcd_ops->lcd_off)
		nec_lcd_ops->lcd_off(_panel);

	nec_panel_set_panel_is_on(false, user);
	return;
}

void nec_panel_force_off(void)
{
	int user = 0;

	/* Already panel is off */
	if (!nec_panel_is_panel_on_with_state_change(true))
		return;

	/* Backlight off, if need */
	nec_panel_set_backlight(0);

	/* Display off, if need */
	nec_panel_display_off();

	/* Panel off */
	if (nec_lcd_ops->lcd_off)
		nec_lcd_ops->lcd_off(_panel);

	for (user = 0; user < NEC_PANEL_NUM_USER; user++)
		nec_panel_set_panel_is_on(false, user);

	return;
}

void nec_panel_display_on(void)
{
	if (!nec_panel_is_panel_on())
		return;

	if (nec_panel_res->display_is_on)
		return;

	if (nec_lcd_ops->lcd_display_on)
		nec_lcd_ops->lcd_display_on(_panel);

	nec_panel_res->display_is_on = true;
	NEC_PANEL_VPRINT("func = %s\n", nec_lcd_ops->lcd_display_on ? "ok" : "none");
	return;
}

void nec_panel_display_off(void)
{
	if (!nec_panel_res->display_is_on)
		return;

	/* Backlight off, if need */
	nec_panel_set_backlight(0);

	if (nec_lcd_ops->lcd_display_off)
		nec_lcd_ops->lcd_display_off(_panel);

	nec_panel_res->display_is_on = false;
	NEC_PANEL_VPRINT("func = %s\n", nec_lcd_ops->lcd_display_off ? "ok" : "none");
	return;
}

int nec_panel_set_backlight(int on)
{
	int err = 0;

	if (!nec_panel_is_panel_on())
		return -1;

	if (nec_lcd_ops->lcd_set_backlight)
		err = nec_lcd_ops->lcd_set_backlight(_panel, on);

	NEC_PANEL_VPRINT("backlight = %d func = %s\n", on, nec_lcd_ops->lcd_set_backlight ? "ok" : "none");

	return err;
}

int nec_panel_get_backlight(void)
{
	int backlight = 0;

	if (nec_lcd_ops->lcd_get_backlight)
		backlight = nec_lcd_ops->lcd_get_backlight(_panel);

	NEC_PANEL_VPRINT("backlight = %d func = %s\n", backlight, nec_lcd_ops->lcd_get_backlight ? "ok" : "none");

	return backlight;
}

int nec_panel_set_brightness(int brightness)
{
	int err = 0;

	if (!nec_panel_is_panel_on())
		return -1;

	if (nec_lcd_ops->lcd_set_brightness)
		err = nec_lcd_ops->lcd_set_brightness(_panel, brightness);

	NEC_PANEL_VPRINT("brightness = %d func = %s\n", brightness, nec_lcd_ops->lcd_set_brightness ? "ok" : "none");

	return err;
}

int nec_panel_get_brightness(void)
{
	int brightness = -1;

	if (nec_lcd_ops->lcd_get_brightness)
		brightness = nec_lcd_ops->lcd_get_brightness(_panel);
	NEC_PANEL_VPRINT("brightness = %d func = %s\n", brightness, nec_lcd_ops->lcd_get_brightness ? "ok" : "none");

	return brightness;
}

int nec_panel_set_refresh_rate(int refresh_rate)
{
	if (!nec_panel_is_panel_on())
		return -1;
	if (nec_lcd_ops->lcd_set_refresh_rate)
		nec_lcd_ops->lcd_set_refresh_rate(_panel, refresh_rate);

	NEC_PANEL_VPRINT("refresh rate = %d func = %s\n", refresh_rate, nec_lcd_ops->lcd_set_refresh_rate ? "ok" : "none");

	nec_panel_init_panel_info(&(nec_panel_res->panel_info));

	return 0;
}

int nec_panel_get_refresh_rate(void)
{
	int refresh_rate = -1;

	if (nec_lcd_ops->lcd_get_refresh_rate)
		refresh_rate = nec_lcd_ops->lcd_get_refresh_rate(_panel);

	NEC_PANEL_VPRINT("refresh rate = %d func = %s\n", refresh_rate, nec_lcd_ops->lcd_get_refresh_rate ? "ok" : "none");

	return refresh_rate;
}


/*****************************************************************************/
void nec_panel_send_command(u8 cmd, u8 *data, u32 data_cnt)
{
	mutex_lock(&send_cmd_mutex);
	nec_panel_send_command_locked(cmd, data, data_cnt);
	mutex_unlock(&send_cmd_mutex);
}

void nec_panel_read_command(u8 cmd, u32 data_cnt)
{
	mutex_lock(&send_cmd_mutex);
	nec_panel_read_command_locked(cmd, data_cnt);
	mutex_unlock(&send_cmd_mutex);
}

int nec_panel_send_frame(u32 x_start, u32 y_start, u32 x_end, u32 y_end,
		u32 *src_p, u32 *src_v, u32 bytes, u32 page)
{
	int err = 0;
	u32 dx, y, n;
	u8 *fbp, *inp;
	int pitch;
	int bytes_per_pixel = 2;	// Support only 16bit
	struct drm_fb_helper *helper = get_drm_fb_helper();
	struct fb_info *fbi;
	struct fb_var_screeninfo *var, ovar;
	struct fb_fix_screeninfo *fix;
	struct drm_framebuffer *fb;

	NEC_PANEL_DPRINT("start \n");
	NEC_PANEL_PRINT("(%d,%d)-(%d,%d) bytes:%d page: %d\n", x_start, y_start, x_end, y_end, bytes, page);

	if (!helper) {
		NEC_PANEL_PRINT_ERR("exit, not yet registered DRM helper \n");
		return -EINVAL;
	}
	fbi = helper->fbdev;	// Use DRM's FB info
	if (!fbi) {
		NEC_PANEL_PRINT_ERR("exit, not yet registered DRM framebuffer \n");
		return -EINVAL;
	}

	// Use lock_fb_info() for protecting section which changes FB info to original temporary
	if (!lock_fb_info(fbi)) {
		NEC_PANEL_PRINT_ERR("exit, cannot lock fb info \n");
		return -EBUSY;
	}

	var = &(fbi->var);
	fix = &(fbi->fix);

	fb = helper->fb;
	if (!fb) {
		unlock_fb_info(fbi);
		NEC_PANEL_PRINT_ERR("exit, FB is invalid\n");
		return -EINVAL;
	}
	if (fb->format->depth != 16) {
		unlock_fb_info(fbi);
		NEC_PANEL_PRINT_ERR("exit, FB depth %d not suported\n", fb->format->depth);
		return -EINVAL;
	}

	nec_fb_set_original_fb_info(helper, &ovar);
	// fbi->var was modified to original

	if (x_start > var->xres || x_end > var->xres ||
		y_start > var->yres || y_end > var->yres ||
		(CONFIG_DRM_NEC_FB_RESERVED >= 100 && page >= (CONFIG_DRM_NEC_FB_RESERVED / 100))) {
			unlock_fb_info(fbi);
			NEC_PANEL_DPRINT("exit, target region is out of %dx%d, or page invalid\n", var->xres, var->yres);
			return -EINVAL;
	}
	pitch = fix->line_length;

	// Set bottom area of original DRM FB for NEC proprietary
	var->yoffset += var->yres * page;

	NEC_PANEL_DPRINT("change display area temporary (%d,%d)-%dx%d->(%d,%d)-%dx%d\n", 
		ovar.xoffset, ovar.yoffset, ovar.xres, ovar.yres,
		var->xoffset, var->yoffset, var->xres, var->yres);
	NEC_PANEL_DPRINT("pitch: %d\n", pitch);

	n = 0;
	fbp = fbi->screen_base + (var->yoffset + y_start) * pitch;
	inp = (u8*)src_v;
	for (y = y_start; y <= y_end; y++) {
		if (n >= bytes) {
			break;
		}

		dx = (x_end - x_start + 1) * bytes_per_pixel;
		if ((n + dx) > bytes) {
			dx = (bytes - n) / bytes_per_pixel * bytes_per_pixel;
		}
		memcpy(fbp + x_start * bytes_per_pixel, inp, dx);
		fbp += pitch;
		inp += dx;
		n += dx;
	}
	// Display whole screen by PAN
	err = drm_fb_helper_pan_display(var, fbi);

	nec_fb_set_tricked_fb_info(helper, &ovar);

	unlock_fb_info(fbi);

	/* Display on, if need */
	if (!err)
		nec_panel_display_on();

	NEC_PANEL_DPRINT("exit ret = %d \n", err);

	return err;
}


/*****************************************************************************
 * Driver register/unregister operations
 *****************************************************************************/
static int nec_panel_probe(struct platform_device *pdev)
{
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	int j = 0;
#endif // NEC_PANEL_ENABLE_PERF_CHECK
	int i = 0;
	int err = 0;

	NEC_PANEL_DPRINT("start probe\n");

	/* Get memory for nec panel resource */
	if (!nec_panel_res)
		nec_panel_res = devm_kzalloc(&pdev->dev,
			sizeof(struct nec_panel_resource), GFP_KERNEL);
	if (!nec_panel_res) {
		NEC_PANEL_PRINT_ERR("Cannot allocate "
					"nec resource memory!\n");
		return -ENOMEM;
	}
	if (nec_panel_res->res_init) {
		pr_err("nec panel has already initialized\n");
		return -EINVAL;
	}

	/* Get memory for lcd operations */
	if (!nec_lcd_ops)
		nec_lcd_ops = devm_kzalloc(&pdev->dev,
			sizeof(struct nec_panel_lcd_operation), GFP_KERNEL);
	if (!nec_lcd_ops) {
		NEC_PANEL_PRINT_ERR("Cannot allocate "
					"lcd operations memory!\n");
		return -ENOMEM;
	}

	/* Get memory for panel operations */
	if (!nec_panel_ops)
		nec_panel_ops = devm_kzalloc(&pdev->dev,
			sizeof(struct nec_fb_panel_operation), GFP_KERNEL);
	if (!nec_panel_ops) {
		NEC_PANEL_PRINT_ERR("Cannot allocate "
					"panel operations memory!\n");
		return -ENOMEM;
	}

	/*
	 * Init lcd operation (for LCD device)
	 */
	err = nec_panel_init_lcd_operation();
	if (err)
		return err;
	/* Init lcd io */
	err = nec_panel_init_panel_io(pdev, &(nec_panel_res->panel_io));
	if (err)
		return err;
	/* Init panel info */
	nec_panel_init_panel_info(&(nec_panel_res->panel_info));

	/* Set driver's data */
	pdev->id = 0;
	nec_panel_res->pdev = pdev;
	for (i = 0; i < NEC_PANEL_NUM_USER; i++)
		nec_panel_res->use_panel[i] = false;
	nec_panel_res->panel_is_on = false;
	nec_panel_res->display_is_on = false;
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	nec_panel_res->perf_index = 0;
	nec_panel_res->perf_check_count = 0;
	for (i = 0; i < NEC_PANEL_PERF_NUM_PART; i++) {
		for (j = 0; j < NEC_PANEL_PERF_NUM_DATA; j++) {
			nec_panel_res->part[i].start_sec[j]  = 0;
			nec_panel_res->part[i].start_usec[j] = 0;
			nec_panel_res->part[i].end_sec[j]    = 0;
			nec_panel_res->part[i].end_usec[j]   = 0;
		}
	}
#endif // NEC_PANEL_ENABLE_PERF_CHECK
	nec_panel_res->res_init = true;
	platform_set_drvdata(pdev, nec_panel_res);

	/*
	 * Init panel operation (for FB)
	 */
	err = nec_panel_init_panel_operation(nec_panel_ops);
	if (err)
		return err;

	nec_panel_init = true;
	NEC_PANEL_DPRINT("done\n");

	return 0;
}

static int nec_panel_init_lcd_operation(void)
{
	NEC_PANEL_DPRINT("Set lcd device operations\n");
	return 0;
}

static int nec_panel_init_panel_operation(
		struct nec_fb_panel_operation *panel_ops)
{
	if (!panel_ops) {
		/* NOT REACH HERE. */
		NEC_PANEL_PRINT_ERR("nec panel driver is not initialized\n");
		return -EINVAL;
	}

	/* Set nec panel operation for frame buffer driver */
	panel_ops->alloc_fb_mem		= nec_panel_alloc_fb_mem;
	panel_ops->panel_on		= nec_panel_on;
	panel_ops->panel_off		= nec_panel_off;
	panel_ops->panel_force_off	= nec_panel_force_off;
	panel_ops->panel_is_on		= nec_panel_is_panel_on;
	panel_ops->panel_disp_on	= nec_panel_display_on;
	panel_ops->panel_disp_off	= nec_panel_display_off;
	panel_ops->panel_reset		= nec_panel_reset_panel;
	panel_ops->start_dma		= nec_panel_pan_display;
	panel_ops->set_brightness	= nec_panel_set_brightness;
	panel_ops->get_brightness	= nec_panel_get_brightness;
	panel_ops->set_backlight	= nec_panel_set_backlight;
	panel_ops->get_backlight	= nec_panel_get_backlight;
	panel_ops->set_refresh_rate	= nec_panel_set_refresh_rate;
	panel_ops->get_refresh_rate	= nec_panel_get_refresh_rate;
	panel_ops->write_panel_cmd	= nec_panel_write_lcd_data;
	panel_ops->read_panel_cmd	= nec_panel_read_lcd_data;
	panel_ops->print_perf		= nec_panel_print_perf_result;
	panel_ops->update_panel_status	= nec_panel_update_status;

	/* Set operation to FB */
	nec_fb_set_panel_operation(panel_ops);
	NEC_PANEL_DPRINT("Set nec panel operations to FB\n");

	return 0;
}

static int nec_panel_init_panel_io(struct platform_device *pdev,
					struct nec_panel_io *panel_io)
{
	int err = 0;

	/* Specifiv settings, if LCD need */
	if (nec_lcd_ops->lcd_init) {
		err = nec_lcd_ops->lcd_init(_panel);
		if (!err)
			return 0;
		if (err != -ENOTSUPP) {
			return -1;
		}
	}

	/*
	 * NEC Panel default setting
	 */
	// NO setting required in this driver

	/* IO init done */
	panel_io->init = true;
	return 0;
}

static int nec_panel_init_panel_info(struct nec_panel_info *panel_info)
{
	panel_info->xres	= NEC_PANEL_DEFAULT_XRES;
	panel_info->yres	= NEC_PANEL_DEFAULT_YRES;
	panel_info->bpp		= NEC_PANEL_DEFAULT_BITS_PER_PIXEL;
	panel_info->Bpp		= NEC_PANEL_DEFAULT_BYTES_PER_PIXEL;

	if (nec_lcd_ops->lcd_get_refresh_rate)
		panel_info->max_fps = nec_lcd_ops->lcd_get_refresh_rate(_panel);
	else
		panel_info->max_fps = NEC_PANEL_DEFAULT_REFRESHRATE;

	NEC_PANEL_DPRINT("done\n");
	return 0;
}

/*****************************************************************************/
static int nec_panel_remove(struct platform_device *pdev)
{
	NEC_PANEL_DPRINT("done\n");
	return 0;
}


/*****************************************************************************/
static const struct of_device_id nec_panel_of_match[] = {
	{ .compatible = NEC_PANEL_NAME, },
	{ },
};
MODULE_DEVICE_TABLE(of, nec_panel_of_match);

static struct platform_driver nec_panel_driver = {
	.probe		= nec_panel_probe,
	.remove		= nec_panel_remove,
	.shutdown	= NULL,
	.driver		= {
		.name		= NEC_PANEL_NAME,
		.owner		= THIS_MODULE,
		.of_match_table	= nec_panel_of_match,
	},
};
module_platform_driver(nec_panel_driver);
MODULE_LICENSE("GPL v2");
