/*
 * Copyright (c) 2013 - 2015, The Linux Foundation. All rights reserved.
 * Copyright (c) 2017 - 2021, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#ifndef NEC_PANEL_H
#define NEC_PANEL_H

/*****************************************************************************
 * Includes
 *****************************************************************************/
#include <drm/drm_panel.h>


/*****************************************************************************
 * Types
 *****************************************************************************/
/* NOTE: Set the same name as nec_panel_device.
 * 	See arch/arm/plat-nec/devs.c. */
#define NEC_PANEL_NAME		"nec-panel"

/* Default resolution */
/*	NOTE: NEC Panel supports only 16 bits per pixel */
#define NEC_PANEL_BITS_PER_BYTE			(8)
#define NEC_PANEL_DEFAULT_XRES			(240)
#define NEC_PANEL_DEFAULT_YRES			(320)
#define NEC_PANEL_DEFAULT_BITS_PER_PIXEL	(16)
#define NEC_PANEL_DEFAULT_BYTES_PER_PIXEL \
	(NEC_PANEL_DEFAULT_BITS_PER_PIXEL / NEC_PANEL_BITS_PER_BYTE)
#define NEC_PANEL_DEFAULT_REFRESHRATE		(60)

/* Brightness */
#define NEC_PANEL_MIN_LCD_BRIGHTNESS		(0x00)
#define NEC_PANEL_MAX_LCD_BRIGHTNESS		(0xFF)

/* Panel commands */
#define NEC_PANEL_MIN_LCD_CMD			(0x00)
#define NEC_PANEL_MAX_LCD_CMD			(0xFF)
#define NEC_PANEL_MIN_LCD_CMD_DATA		(0)
#define NEC_PANEL_MAX_LCD_CMD_DATA		(16)

/* Panel user */
#define NEC_PANEL_USER_FB	(0)
#define NEC_PANEL_USER_SPLASH	(1)
#define NEC_PANEL_NUM_USER	(2)

/* Panel state */
#define NEC_PANEL_STATE_CHANGED		(0)
#define NEC_PANEL_STATE_CHANGING	(1)

#define NEC_PANEL_MAX_PANEL_CHANGE_WAIT_MS	(3000) /* msec */
#define NEC_PANEL_PANEL_STATE_CHECK_CYCLE_US	(100 * 1000) /* micro sec */

/* Performance check */
/* 	If check performance, enable the below define. */
//#define NEC_PANEL_ENABLE_PERF_CHECK
enum {
	NEC_PANEL_PERF_PART_PAN = 0,
	NEC_PANEL_PERF_PART_SYNC_WAIT,
	NEC_PANEL_PERF_PART_DMA,
	NEC_PANEL_PERF_NUM_PART,
};
#define NEC_PANEL_PERF_PART_NAME_PAN	"pan"
#define NEC_PANEL_PERF_PART_NAME_SYNC	"vsync"
#define NEC_PANEL_PERF_PART_NAME_DMA	"dma"

#define NEC_PANEL_PERF_NUM_DATA		(0x80)
#define NEC_PANEL_PERF_DATA_MASK	(0x7F)
#define NEC_PANEL_PERF_NEXT_IDX(curr) \
	(((curr) + 0x01) & NEC_PANEL_PERF_DATA_MASK)



/*****************************************************************************
 *  Macros
 *****************************************************************************/
#define NEC_PANEL_CLAMP( _val , _min , _max ) \
	( ( ( _val ) < ( _min ) ) ? ( _min ) : \
	( ( ( _max ) < ( _val ) ) ? ( _max ) : ( _val ) ) )


/*****************************************************************************
 *  Structures
 *****************************************************************************/
struct nec_panel_io {
	bool init;
};

struct nec_panel_info {
	u32 xres;
	u32 yres;
	u32 bpp;	// bits per pixel
	u32 Bpp;	// Bytes per pixel
	u32 max_fps;
};

struct nec_panel_buf_addr {
	void *virt;
#ifndef CONFIG_64BIT
	u32   phys;
#else
	u64   phys;
#endif
};

#ifdef NEC_PANEL_ENABLE_PERF_CHECK
struct nec_panel_perf_info {
	u64 start_sec[NEC_PANEL_PERF_NUM_DATA];
	u64 start_usec[NEC_PANEL_PERF_NUM_DATA];
	u64 end_sec[NEC_PANEL_PERF_NUM_DATA];
	u64 end_usec[NEC_PANEL_PERF_NUM_DATA];
};
#endif // NEC_PANEL_ENABLE_PERF_CHECK

struct nec_panel_resource {
	bool res_init;

	bool use_panel[NEC_PANEL_NUM_USER];
	bool panel_is_on;
	int  panel_change_state;
	bool display_is_on;

	bool with_no_reset;

	struct platform_device *pdev;

	struct nec_panel_buf_addr fb_buf;

	struct nec_panel_io   panel_io;
	struct nec_panel_info panel_info;

#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	u32 perf_index;
	u32 perf_check_count;
	struct nec_panel_perf_info part[NEC_PANEL_PERF_NUM_PART];
#endif // NEC_PANEL_ENABLE_PERF_CHECK
};

struct nec_panel_lcd_operation {
    int (*lcd_init)(struct drm_panel *panel);
    int (*lcd_reset)(struct drm_panel *panel, int hwreset, int swreset);
    int (*lcd_on)(struct drm_panel *panel);
    int (*lcd_off)(struct drm_panel *panel);
    void (*lcd_display_on)(struct drm_panel *panel);
    void (*lcd_display_off)(struct drm_panel *panel);
    void (*lcd_get_status)(struct drm_panel *panel, int *display_on, int *lcd_on);
    int (*lcd_set_backlight)(struct drm_panel *panel, int onoff);
    int (*lcd_get_backlight)(struct drm_panel *panel);
    int (*lcd_set_brightness)(struct drm_panel *panel, unsigned int level);
    int (*lcd_get_brightness)(struct drm_panel *panel);
    void (*lcd_set_refresh_rate)(struct drm_panel *panel, int refresh_rate);
    int (*lcd_get_refresh_rate)(struct drm_panel *panel);
    void (*lcd_send_cmd)(struct drm_panel *panel, u8 cmd, u8 *data, int data_len);
    int (*lcd_recv_cmd)(struct drm_panel *panel, u8 cmd, u8 *data, int max_data_len);
    int (*lcd_wait_vsync)(struct drm_panel *panel, int timeout_ms);
};

/*****************************************************************************
 * Public funciton declairs
 *****************************************************************************/
bool nec_panel_is_initialized(void);
bool nec_panel_is_panel_on(void);

int  nec_panel_set_lcd_operation(struct nec_panel_lcd_operation *lcd_ops);
int  nec_panel_get_panel_resource(struct nec_panel_resource **res);
int  nec_panel_get_panel_info(struct nec_panel_info *panel_info);

void nec_panel_set_no_reset_once(void);

int  nec_panel_on(int user);
void nec_panel_off(int user);
void nec_panel_force_off(void);
void nec_panel_display_on(void);
void nec_panel_display_off(void);
int  nec_panel_set_backlight(int on);
int  nec_panel_get_backlight(void);
int  nec_panel_set_brightness(int brightness);
int  nec_panel_get_brightness(void);
int  nec_panel_set_refresh_rate(int refresh_rate);
int  nec_panel_get_refresh_rate(void);

void nec_panel_send_command(u8 cmd, u8 *data, u32 data_cnt);
void nec_panel_read_command(u8 cmd, u32 data_cnt);

int  nec_panel_send_frame(u32 x_start, u32 y_start, u32 x_end, u32 y_end,
			u32 *src_p, u32 *src_v, u32 bytes, u32 page);

#endif /* NEC_PANEL_H */
