/*
 * Copyright (c) 2013 - 2015, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013 NEC AccessTechinica, Ltd., All rights reserved.
 * Copyright (c) 2014 - 2021, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*****************************************************************************
 *  Includes
 *****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>
//TODO: Build error because of platform dependence // #include <linux/necpf_panel.h>

#include <linux/kthread.h>
#include <linux/mtd/mtd.h>

// Enable below when any module will use display_square_animation() which is currently not used.
// #define NEC_PANEL_USE_ANIMATION

#include "nec_panel.h"
#include "images/nec_image.h"


/*****************************************************************************
 *  Types
 *****************************************************************************/
/* NOTE: Set the same name as nec_splash_device.
 *       See arch/arm/plat-nec/devs.c. */
#define NEC_SPLASH_NAME			"nec-splash"


#define PANEL_DISP_ON_BIT               (0)
#define PANEL_DISP_ON_MASK              (0x1)
  #define PANEL_STATE_DISP_ON           (0x1)

#define PANEL_DISP_BOOT_BIT             (1)
#define PANEL_DISP_BOOT_MASK            (0x1)
  #define PANEL_STATE_BOOTING           (0x1)

#define PANEL_DISP_VERTICAL_BIT         (2)
#define PANEL_DISP_VERTICAL_MASK        (0x1)
  #define PANEL_STATE_HORIZON           (0x0)
  #define PANEL_STATE_VERTICAL          (0x1)

#define PANEL_DISP_SLEEP_BIT            (4)
#define PANEL_DISP_SLEEP_MASK           (0x1)
  #define PANEL_STATE_RESUME            (0x0)
  #define PANEL_STATE_SUSPEND           (0x1)
#define PANEL_DISP_WAKEONBT_BIT         (5)
#define PANEL_DISP_WAKEONBT_MASK        (0x1)
  #define PANEL_STATE_WAKEONBT_OFF      (0x0)
  #define PANEL_STATE_WAKEONBT_ON       (0x1)
#define PANEL_DISP_PWOFF_BIT            (6)
#define PANEL_DISP_PWOFF_MASK           (0x1)
  #define PANEL_STATE_POWEROFF          (0x1)
#define PANEL_DISP_BATTERY_BIT          (8)
#define PANEL_DISP_BATTERY_MASK         (0x7f)
#define PANEL_DISP_CHARGE_BIT           (15)
#define PANEL_DISP_CHARGE_MASK          (0x1)
  #define PANEL_STATE_CHARGE_DISABLE    (0x0)
  #define PANEL_STATE_CHARGE_ENABLE     (0x1)

#define PANEL_DISP_CARRIER_SW_BIT       (16)
#define PANEL_DISP_CARRIER_SW_MASK      (0x1)
  #define PANEL_STATE_CARRIER_SW_EXEC   (0x1)

#define PANEL_DISP_PROGRESS_BAR_BIT     (19)
#define PANEL_DISP_PROGRESS_BAR_MASK    (0x1)
  #define PANEL_STATE_PROGRESS_BAR      (0x1)

#define PANEL_DISP_PROGRESS_RATE_BIT    (20)
#define PANEL_DISP_PROGRESS_RATE_MASK   (0xf)

#if defined(CONFIG_NEC_SUPPORT_LEGAL_PAGE)
#define PANEL_DISP_LEGAL_PAGE_BIT       (28)
#define PANEL_DISP_LEGAL_PAGE_MASK      (0xf)
#endif /* CONFIG_NEC_SUPPORT_LEGAL_PAGE */

#define PANEL_DISP_PROGRESS_BAR_INTERVAL (5000) /* msec */

enum writemode {
	WRITEMODE_DIRECT,
	WRITEMODE_BUFFER
};

#define BOOT_FLAG_LCD_BIT		(0)
#define BOOT_FLAG_LCD_MASK		(0x1)
  #define BOOT_FLAG_LCD_DISABLE		(0x0)
  #define BOOT_FLAG_LCD_ENABLE		(0x1)
#define BOOT_FLAG_DISP_WAY_BIT		(1)
#define BOOT_FLAG_DISP_WAY_MASK		(0x1)
  #define BOOT_FLAG_DISP_VERICAL	(0x0)
  #define BOOT_FLAG_DISP_HORIZONTAL	(0x1)
#define BOOT_FLAG_ATERM_DISP_BIT	(2)
#define BOOT_FLAG_ATERM_DISP_MASK	(0x1)
  #define BOOT_FLAG_ATERM_NONDISP	(0x0)
  #define BOOT_FLAG_ATERM_DISP		(0x1)
#define BOOT_FLAG_SWAP_SIM_BIT		(3)
#define BOOT_FLAG_SWAP_SIM_MASK		(0x1)
  #define BOOT_FLAG_SWAP_SIM_NONDISP	(0x0)
  #define BOOT_FLAG_SWAP_SIM_DISP	(0x1)
#define BOOT_FLAG_SWAP_SIM_SELECT_BIT	(4)
#define BOOT_FLAG_SWAP_SIM_SELECT_MASK	(0x1)
  #define BOOT_FLAG_FM_SIM1_TO_SIM2	(0x0)
  #define BOOT_FLAG_FM_SIM2_TO_SIM1	(0x1)
#define BOOT_FLAG_ATERM_INHERIT_BIT	(5)
#define BOOT_FLAG_ATERM_INHERIT_MASK	(0x1)
  #define BOOT_FLAG_ATERM_NONINHERIT	(0x0)
  #define BOOT_FLAG_ATERM_INHERIT	(0x1)
#define BOOT_FLAG_BATT_DETECT_BIT	(6)
#define BOOT_FLAG_BATT_DETECT_MASK	(0x1)
  #define BOOT_FLAG_BATT_DETECT		(0x0)
  #define BOOT_FLAG_BATT_UNDETECT	(0x1)
#define BOOT_FLAG_NON_BATT_BOOT_BIT	(7)
#define BOOT_FLAG_NON_BATT_BOOT_MASK	(0x1)
  #define BOOT_FLAG_NON_BATT_DISABLE	(0x0)
  #define BOOT_FLAG_NON_BATT_ENABLE	(0x1)
#define BOOT_FLAG_TEST_TYPE_BIT		(28)
#define BOOT_FLAG_TEST_TYPE_MASK	(0x7)
  #define BOOT_FLAG_TEST_TYPE_BT	(0)
#define BOOT_FLAG_TEST_MODE_BIT		(31)
#define BOOT_FLAG_TEST_MODE_MASK	(0x1)
  #define BOOT_FLAG_TEST_MODE_ENABLE	(1)

#define GET_BOOT_FLAG_INFO(x, name)					\
	((x >> BOOT_FLAG_##name##_BIT) & BOOT_FLAG_##name##_MASK)
#define SET_BOOT_FLAG_INFO(x, name, val)					\
do {								\
	x &= ~(BOOT_FLAG_##name##_MASK << BOOT_FLAG_##name##_BIT);	\
	x |= ((val & BOOT_FLAG_##name##_MASK) << BOOT_FLAG_##name##_BIT); \
} while(0)

/*****************************************************************************
 *  Macros
 *****************************************************************************/
#define GET_PANEL_DISP_INFO(name)	\
	((panel_disp_flag >> PANEL_DISP_##name##_BIT) & PANEL_DISP_##name##_MASK)
#define SET_PANEL_DISP_INFO(name, val)	\
	panel_disp_flag |= ((val & PANEL_DISP_##name##_MASK) << PANEL_DISP_##name##_BIT)


/*****************************************************************************
 *  Statics
 *****************************************************************************/
static enum writemode writemode = WRITEMODE_DIRECT;
static u16 display_buffer[NEC_IMG_SIZE];

static volatile u32 panel_disp_flag = 0;

#define NAND_INIT_COMP_OK (2)
#ifdef NEC_PANEL_CHECK_NAND_STATUS
static volatile int nand_init_comp = 0;
#else // !NEC_PANEL_CHECK_NAND_STATUS
static volatile int nand_init_comp = NAND_INIT_COMP_OK;		// TODO: How to determine NAND is ok?
#endif // !NEC_PANEL_CHECK_NAND_STATUS

struct panel_softc {
	wait_queue_head_t wait_q;
	wait_queue_head_t sleep_wait_q;
	int wait_condition;
	struct mutex display_mutex;
	atomic_t sleeped;
} panel_sc;

static struct platform_device *bb_pdev;

static struct task_struct *nec_display_polling = NULL;
static void *bb_virt;
#ifndef CONFIG_64BIT
static u32 bb_phys;
#else
static u64 bb_phys;
#endif

/*****************************************************************************
 * SysFs operations
 *****************************************************************************/
static ssize_t nec_splash_sysfs_show_flag(struct device *dev,
	struct device_attribute *attr, char *buf);
static ssize_t nec_splash_sysfs_set_flag(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count);

static DEVICE_ATTR(ctl_flag, 0664, nec_splash_sysfs_show_flag,
	nec_splash_sysfs_set_flag);

static struct attribute *fs_attrs[] = {
	&dev_attr_ctl_flag.attr,
	NULL,
};

static struct attribute_group fs_attr_group = {
	.attrs = fs_attrs,
};

/*****************************************************************************
 * Private funciton declairs
 *****************************************************************************/
static void set_display_writemode(enum writemode mode);
static enum writemode get_display_writemode(void);

static void display_on(void);
static void flush_displaybuffer(void);
static void display_reset(struct display_info_t *info);
static void display_all_reset(void);

static void display_picto(struct image_info_t *info,
			struct coordinate_info_t *coord_info, const u16 *data);
#ifdef NEC_PANEL_USE_ANIMATION
static void display_animation_picto(int index, int vertical);
#endif // NEC_PANEL_USE_ANIMATION
static void display_sim_switch(int vertical);
#ifdef NEC_PANEL_USE_ANIMATION
static void display_square_animation(int vertical, int msec);
#endif // NEC_PANEL_USE_ANIMATION
static void display_boot_animation(int vertical, int msec);
static void display_resume(void);
static void display_battery_picto(int remain, int charging, int vertical);
static void display_battery_remain(int remain, int vertical);
static void display_battery_picto_pwoff(int remain, int charging, int vertical);
static void display_battery_remain_pwoff(int remain, int vertical);
static void display_bt_wave(int enable, int vertical);
static void display_sleep_picto(int index, int vertical);
static void display_suspend(void);
static void display_poweroff(void);
static void display_progress_picto(int rate, int vertical);
static void display_progress_bar(int vertical);
static void display_progress_animation(int vertical, int msec, int start, int end);
#if defined(CONFIG_NEC_SUPPORT_LEGAL_PAGE)
static void display_legal_page(int kind, int vertical);
#endif /* CONFIG_NEC_SUPPORT_LEGAL_PAGE */

static int nec_display_polling_thread(void *arg);
static void nec_display_init(struct platform_device *pdev);

static inline uint32_t get_msm_boot_flag(void) {
#if !defined(USE_NEC_BOOTPARAM)
	struct mtd_info *mtd = NULL;
	u8 *buf;
	int ret;
	size_t len;
	struct boot_info_t {
#define CONFIG_HEADER_MAGIC (0x594e4147) //GANY
		uint32_t magic;
		u8 reserved1[28];
		u32 longlife;
		u32 modechg;
		u32 lang;
		u32 reserved2;
	} *boot_info;
	static u32 boot_flag = 0;
	static u32 exist_boot_flag = 0;

	printk(KERN_DEBUG "get_msm_boot_flag enter\n");

	if (exist_boot_flag)
		return boot_flag;

	SET_BOOT_FLAG_INFO(boot_flag, LCD, BOOT_FLAG_LCD_ENABLE);
	SET_BOOT_FLAG_INFO(boot_flag, DISP_WAY, BOOT_FLAG_DISP_HORIZONTAL);

	if (nand_init_comp < NAND_INIT_COMP_OK) {
		printk("%s: uninitialized NAND driver\n", __func__);
		goto err1;
	}

	mtd = get_mtd_device_nm("bootnv");
	if (IS_ERR(mtd)) { // Must check with IS_ERR macro
		printk("%s: fail to get device env mtd block\n", __func__);
		goto err1;
	}

	buf = kmalloc(mtd->writesize, GFP_KERNEL);
	if (!buf) {
		printk("%s: kmalloc error\n", __func__);
		goto err1;
	}

	ret = mtd_read(mtd, 0, mtd->writesize, &len, buf);
	if (ret < 0) { // Must check error as negative value
		printk("%s: Error read flash\n", __func__);
		goto err2;
	}

	boot_info = (struct boot_info_t *) buf;

	if (boot_info->magic != CONFIG_HEADER_MAGIC) {
		printk("%s: format error 0x%x", __func__, boot_info->magic);
		goto err2;
	}

	if (boot_info->modechg == 0) {
		SET_BOOT_FLAG_INFO(boot_flag, ATERM_DISP, BOOT_FLAG_ATERM_DISP);
	}

	exist_boot_flag = 1;
	kfree(buf);

	printk(KERN_DEBUG "get_msm_boot_flag exit %08x\n", boot_flag);
	return boot_flag;

 err2:
	kfree(buf);
 err1:
	SET_BOOT_FLAG_INFO(boot_flag, ATERM_DISP, BOOT_FLAG_ATERM_DISP);
	return boot_flag;
#else /* USE_NEC_BOOTPARAM */
#error Not support
#endif /* !USE_NEC_BOOTPARAM */
}

/*****************************************************************************
 * Driver register/unregister operations
 *****************************************************************************/
static int  nec_splash_probe(struct platform_device *pdev);
static int  nec_splash_sysfs_add(struct device *dev);


/*****************************************************************************
 *  SysFs Operations
 *****************************************************************************/
static ssize_t nec_splash_sysfs_show_flag(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0x%0x\n", panel_disp_flag);
}

static ssize_t nec_splash_sysfs_set_flag(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	panel_disp_flag = val;
	sc->wait_condition = 1;
	mutex_unlock(&panel_sc.display_mutex);
	wake_up(&sc->wait_q);
	if ((val >> PANEL_DISP_ON_BIT) & PANEL_DISP_ON_MASK)
		return count;   /* 点灯するケース */

	/* 消灯するケース */
	pr_info("@@@@@@@@@@@@@@ %s pre wait_event\n", __func__);
	wait_event(sc->sleep_wait_q, atomic_read(&sc->sleeped) == 1);
	pr_info("@@@@@@@@@@@@@@@ %s after wait_event\n", __func__);

	return count;
}


/*****************************************************************************
 *  Private funcitons
 *****************************************************************************/
void necpf_panel_disp_allow(void)
{
	if (nand_init_comp >= NAND_INIT_COMP_OK)
		return;
	mutex_lock(&panel_sc.display_mutex);
	nand_init_comp++;
	mutex_unlock(&panel_sc.display_mutex);
	pr_info("@@@@@@@@@@@@@@ %s nand_init_comp=%d\n", __func__, nand_init_comp);
}

static void set_display_writemode(enum writemode mode)
{
	writemode = mode;
}

static enum writemode get_display_writemode(void)
{
	return writemode;
}

static void display_on(void)
{
	nec_panel_on(NEC_PANEL_USER_SPLASH);
	nec_panel_set_brightness(177); // TODO: Change default brightness if need
}

static void flush_displaybuffer(void)
{
	memcpy(bb_virt, display_buffer, NEC_IMG_SIZE * 2);

	/* Panel power on, if need. */
	nec_panel_on(NEC_PANEL_USER_SPLASH);

	nec_panel_send_frame(0, 0, NEC_IMG_WIDTH - 1, NEC_IMG_HEIGHT - 1,
			(u32 *)bb_phys, (u32 *)bb_virt, NEC_IMG_SIZE*2, 0);

	nec_panel_set_backlight(1);
}

static void display_reset(struct display_info_t *info)
{
	int bit16_len;

	if (info == NULL)
		return;
	if ((info->image == NULL) || (info->coord == NULL))
		return;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->image->height; y++)
			for (x = 0; x < info->image->width; x++)
				display_buffer[NEC_IMG_WIDTH*(info->coord->y_start + y) + info->coord->x_start + x] = 0;
	}
	else {
		bit16_len = info->image->width * info->image->height * 2;
		memset(bb_virt, 0, bit16_len);

		/* Panel power on, if need. */
		nec_panel_on(NEC_PANEL_USER_SPLASH);

		nec_panel_send_frame(info->coord->x_start, info->coord->y_start,
				info->coord->x_end, info->coord->y_end,
				(u32 *)bb_phys, (u32 *)bb_virt, bit16_len, 0);
	}
}

static void display_all_reset(void)
{
	struct display_info_t disp_info;
	struct image_info_t image;
	struct coordinate_info_t coord_info;

	image.width = NEC_IMG_WIDTH;
	image.height = NEC_IMG_HEIGHT;
	coord_info.x_start = 0;
	coord_info.y_start = 0;
	coord_info.x_end = NEC_IMG_WIDTH - 1;
	coord_info.y_end = NEC_IMG_HEIGHT - 1;

	disp_info.image = &image;
	disp_info.coord = &coord_info;

	display_reset(&disp_info);
}

static void display_picto(struct image_info_t *info,
			struct coordinate_info_t *coord_info, const u16 *data)
{
	int bit16_len = 0;

	if (!info || !coord_info || !data)	// picto information is not set
		return;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->height; y++) {
			for (x = 0; x < info->width; x++) {
				display_buffer[NEC_IMG_WIDTH*(coord_info->y_start + y) + coord_info->x_start + x]
					= data[info->width * y + x];
			}
		}
	}
	else {
		bit16_len = info->width * info->height * 2;

		memcpy(bb_virt, data, bit16_len);

		/* Panel power on, if need. */
		nec_panel_on(NEC_PANEL_USER_SPLASH);

		nec_panel_send_frame(coord_info->x_start, coord_info->y_start,
			coord_info->x_end, coord_info->y_end,
			(u32 *)bb_phys, (u32 *)bb_virt, bit16_len, 0);

		/* Set backlight on */
		nec_panel_set_backlight(1);
	}
}

#ifdef NEC_PANEL_USE_ANIMATION
static void display_animation_picto(int index, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;

	if ((index > 1) || (index < 0))
		index = 0;

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		/* landscape */
		coord_info = &coord_info_landscape[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		/* portrait */
		coord_info = &coord_info_portrait[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	data = (u16 *)anime_data[index].data;
	display_picto(info, coord_info, data);
}
#endif // NEC_PANEL_USE_ANIMATION

static void display_sim_switch(int vertical)
{
#ifdef CONFIG_NEC_SUPPORT_DUAL_SIM
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;
	u16 *data;
	u32 switch_type = GET_BOOT_FLAG_INFO(get_msm_boot_flag(),
						SWAP_SIM_SELECT);

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		if (switch_type) {
			info = (struct image_info_t *)&SIM_Fm2To1_ls;
			data = (u16 *)SIM_Fm2To1_ls.data;
		} else {
			info = (struct image_info_t *)&SIM_Fm1To2_ls;
			data = (u16 *)SIM_Fm1To2_ls.data;
		}
		coord_info = &coord_info_landscape[SIM_SWITCH];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		if (switch_type) {
			info = (struct image_info_t *)&SIM_Fm2To1_pr;
			data = (u16 *)SIM_Fm2To1_pr.data;
		} else {
			info = (struct image_info_t *)&SIM_Fm1To2_pr;
			data = (u16 *)SIM_Fm1To2_pr.data;
		}
		coord_info = &coord_info_portrait[SIM_SWITCH];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}
	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();
	display_picto(info, coord_info, data);
	flush_displaybuffer();
	set_display_writemode(WRITEMODE_DIRECT);
#else
	/* not support */
	return;
#endif /* CONFIG_NEC_SUPPORT_DUAL_SIM */
}

#ifdef NEC_PANEL_USE_ANIMATION
static void display_square_animation(int vertical, int msec)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	while (GET_PANEL_DISP_INFO(ON) &&
		GET_PANEL_DISP_INFO(BOOT)) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		display_animation_picto(i%2, vertical);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		wait_event_timeout(sc->wait_q, sc->wait_condition,
					msecs_to_jiffies(msec));
		i++;
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}
#endif // NEC_PANEL_USE_ANIMATION

static void display_boot_animation(int vertical, int msec)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	u16 *data;
	u32 boot_flag = get_msm_boot_flag();

	if (GET_BOOT_FLAG_INFO(boot_flag,SWAP_SIM)
		== BOOT_FLAG_SWAP_SIM_DISP) {

		display_sim_switch(vertical);

		mutex_lock(&panel_sc.display_mutex);
		/* 500ms * 10 */
		while ((GET_PANEL_DISP_INFO(ON) &&
			GET_PANEL_DISP_INFO(BOOT)) && i < 10) {
			mutex_unlock(&panel_sc.display_mutex);
			wait_event_timeout(sc->wait_q, sc->wait_condition,
						msecs_to_jiffies(500));
			i++;
			mutex_lock(&panel_sc.display_mutex);
		}
		mutex_unlock(&panel_sc.display_mutex);
	}

	if (GET_BOOT_FLAG_INFO(boot_flag, ATERM_DISP)
		== BOOT_FLAG_ATERM_DISP) {
		/* Aterm Logo */
		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			info = (struct image_info_t *)&main_logo_ls;
			coord_info = &coord_info_landscape[ATERM_LOGO];
			data = (u16 *)main_logo_ls.data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			info = (struct image_info_t *)&main_logo_pr;
			coord_info = &coord_info_portrait[ATERM_LOGO];
			data = (u16 *)main_logo_pr.data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}

		set_display_writemode(WRITEMODE_BUFFER);
		if (GET_BOOT_FLAG_INFO(boot_flag, ATERM_INHERIT)
			!= BOOT_FLAG_ATERM_INHERIT) {
			display_all_reset();
		}
		display_picto(info, coord_info, data);
	}

	display_progress_animation(vertical, msec, 2, 10);

	display_all_reset();
}

static void display_resume(void)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;
	int vertical;

	mutex_lock(&panel_sc.display_mutex);
#if !defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
	vertical = 0;
#elif defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && !defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
	vertical = 1;
#else
	vertical = GET_PANEL_DISP_INFO(VERTICAL);
#endif /* CONFIG_NEC_SUPPORT_PORTRAIT_DISP && CONFIG_NEC_SUPPORT_LANDSCAPE_DISP */
	mutex_unlock(&panel_sc.display_mutex);

	/* Aterm logo */
	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		info = (struct image_info_t *)&main_logo_ls;
		coord_info = &coord_info_landscape[ATERM_LOGO];
		data = (u16 *)main_logo_ls.data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		info = (struct image_info_t *)&main_logo_pr;
		coord_info = &coord_info_portrait[ATERM_LOGO];
		data = (u16 *)main_logo_pr.data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();
	display_picto(info, coord_info, data);

	mutex_lock(&panel_sc.display_mutex);
	SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	mutex_unlock(&panel_sc.display_mutex);

	display_progress_animation(vertical, 100, 0, 10);
	display_all_reset();

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		flush_displaybuffer();
		set_display_writemode(WRITEMODE_DIRECT);
	}
}

static void display_battery_picto(int remain, int charging, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;
	int batt_lv = BATT_CHG_LV1;

	if (remain <= 20)
		batt_lv = BATT_CHG_LV1;
	else if (remain <= 40)
		batt_lv = BATT_CHG_LV2;
	else if (remain <= 60)
		batt_lv = BATT_CHG_LV3;
	else if (remain <= 80)
		batt_lv = BATT_CHG_LV4;
	else
		batt_lv = BATT_CHG_LV5;

	if (!charging) {
		batt_lv += BATT_LV1;
		if (remain <= 5)
			batt_lv = BATT_LV0;
	}

	if (remain > 100 || remain < 0) {
		if (!charging)
			batt_lv = BATT_LV0;
		else
			batt_lv = BATT_CHG_LV1;
	}

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		info = (struct image_info_t *)&batt_data_ls[batt_lv];
		coord_info = &coord_info_landscape[BATTERY_PICTO];
		data = (u16 *)batt_data_ls[batt_lv].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		info = (struct image_info_t *)&batt_data_pr[batt_lv];
		coord_info = &coord_info_portrait[BATTERY_PICTO];
		data = (u16 *)batt_data_pr[batt_lv].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	display_picto(info, coord_info, data);
}

static void display_battery_remain(int remain, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *image = NULL;
	struct coordinate_info_t *coord_info = NULL;
	struct display_info_t disp_info;
	int i;
	unsigned int num, tmp;
	int divider = 100;
	int num_disp = 0;

	/* % */
	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		image = (struct image_info_t *)&num_data_ls[PERCENT];
		coord_info = &coord_info_landscape[REMAIN_PER_PICTO];
		data = (u16 *)num_data_ls[PERCENT].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		image = (struct image_info_t *)&num_data_pr[PERCENT];
		coord_info = &coord_info_portrait[REMAIN_PER_PICTO];
		data = (u16 *)num_data_pr[PERCENT].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}
	display_picto(image, coord_info, data);

	if (remain > 100 || remain < 0) {
		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			image = (struct image_info_t *)&num_data_ls[MINUS];
			coord_info = &coord_info_landscape[REMAIN_NUM2_PICTO];
			data = (u16 *)num_data_ls[MINUS].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			image = (struct image_info_t *)&num_data_pr[MINUS];
			coord_info = &coord_info_portrait[REMAIN_NUM2_PICTO];
			data = (u16 *)num_data_pr[MINUS].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			coord_info = &coord_info_landscape[REMAIN_NUM1_PICTO];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			coord_info = &coord_info_portrait[REMAIN_NUM1_PICTO];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			coord_info = &coord_info_landscape[REMAIN_NUM0_PICTO];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			coord_info = &coord_info_portrait[REMAIN_NUM0_PICTO];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);

		return;
	}

	num_disp = 0;
	tmp = remain;
	for (i = REMAIN_NUM0_PICTO; i <= REMAIN_NUM2_PICTO; i++) {
		if (divider <= 0)
			break;
		num = tmp / divider;
		tmp = tmp % divider;
		divider = divider / 10;
		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			image = (struct image_info_t *)&num_data_ls[num];
			coord_info = &coord_info_landscape[i];
			data = (u16 *)num_data_ls[num].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			image = (struct image_info_t *)&num_data_pr[num];
			coord_info = &coord_info_portrait[i];
			data = (u16 *)num_data_pr[num].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}

		if ((num == 0) && (i != REMAIN_NUM2_PICTO) && !num_disp) {
			disp_info.image = image;
			disp_info.coord = coord_info;
			display_reset(&disp_info);
			continue;
		}

		display_picto(image, coord_info, data);
		num_disp = 1;
	}
}

static void display_battery_picto_pwoff(int remain, int charging, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;
	int batt_lv = BATT_CHG_LV1;

	if (remain <= 20)
		batt_lv = BATT_CHG_LV1;
	else if (remain <= 40)
		batt_lv = BATT_CHG_LV2;
	else if (remain <= 60)
		batt_lv = BATT_CHG_LV3;
	else if (remain <= 80)
		batt_lv = BATT_CHG_LV4;
	else
		batt_lv = BATT_CHG_LV5;

	if (!charging) {
		batt_lv += BATT_LV1;
		if (remain <= 5)
			batt_lv = BATT_LV0;
	}

	if (remain > 100 || remain < 0) {
		if (!charging)
			batt_lv = BATT_LV0;
		else
			batt_lv = BATT_CHG_LV1;
	}

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		info = (struct image_info_t *)&batt_data_ls[batt_lv];
		coord_info = &coord_info_landscape[PWOFF_BATTERY_PICTO];
		data = (u16 *)batt_data_ls[batt_lv].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		info = (struct image_info_t *)&batt_data_pr[batt_lv];
		coord_info = &coord_info_portrait[PWOFF_BATTERY_PICTO];
		data = (u16 *)batt_data_pr[batt_lv].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	display_picto(info, coord_info, data);
}

static void display_battery_remain_pwoff(int remain, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *image = NULL;
	struct coordinate_info_t *coord_info = NULL;
	struct display_info_t disp_info;
	int i;
	unsigned int num, tmp;
	int divider = 100;
	int num_disp = 0;

	/* % */
	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		image = (struct image_info_t *)&num_data_ls[PERCENT];
		coord_info = &coord_info_landscape[PWOFF_REMAIN_PER_PICTO];
		data = (u16 *)num_data_ls[PERCENT].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		image = (struct image_info_t *)&num_data_pr[PERCENT];
		coord_info = &coord_info_portrait[PWOFF_REMAIN_PER_PICTO];
		data = (u16 *)num_data_pr[PERCENT].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}
	display_picto(image, coord_info, data);

	if (remain > 100 || remain < 0) {
		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			image = (struct image_info_t *)&num_data_ls[MINUS];
			coord_info = &coord_info_landscape[PWOFF_REMAIN_NUM2_PICTO];
			data = (u16 *)num_data_ls[MINUS].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			image = (struct image_info_t *)&num_data_pr[MINUS];
			coord_info = &coord_info_portrait[PWOFF_REMAIN_NUM2_PICTO];
			data = (u16 *)num_data_pr[MINUS].data;
#endif
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			coord_info = &coord_info_landscape[PWOFF_REMAIN_NUM1_PICTO];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			coord_info = &coord_info_portrait[PWOFF_REMAIN_NUM1_PICTO];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			coord_info = &coord_info_landscape[PWOFF_REMAIN_NUM0_PICTO];
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			coord_info = &coord_info_portrait[PWOFF_REMAIN_NUM0_PICTO];
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);

		return;
	}

	num_disp = 0;
	tmp = remain;
	for (i = PWOFF_REMAIN_NUM0_PICTO; i <= PWOFF_REMAIN_NUM2_PICTO; i++) {
		if (divider <= 0)
			break;
		num = tmp / divider;
		tmp = tmp % divider;
		divider = divider / 10;
		if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
			image = (struct image_info_t *)&num_data_ls[num];
			coord_info = &coord_info_landscape[i];
			data = (u16 *)num_data_ls[num].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
			image = (struct image_info_t *)&num_data_pr[num];
			coord_info = &coord_info_portrait[i];
			data = (u16 *)num_data_pr[num].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		}

		if ((num == 0) && (i != PWOFF_REMAIN_NUM2_PICTO) && !num_disp) {
			disp_info.image = image;
			disp_info.coord = coord_info;
			display_reset(&disp_info);
			continue;
		}

		display_picto(image, coord_info, data);
		num_disp = 1;
	}
}

static void display_bt_wave(int enable, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *image = NULL;
	struct coordinate_info_t *coord_info = NULL;

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		image = (struct image_info_t *)&bt_wave_data_ls;
		coord_info = &coord_info_landscape[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_data_ls.data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		image = (struct image_info_t *)&bt_wave_data_pr;
		coord_info = &coord_info_portrait[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_data_pr.data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	if (!enable) {
		struct display_info_t disp_info;
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);
	} else {
		display_picto(image, coord_info, data);
	}
}

static void display_sleep_picto(int index, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;

	if (index > 1 || index < 0)
		index = 0;

	if (!vertical) {
#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
		/* landscape */
		coord_info = &coord_info_landscape[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_data_ls[index];
		data = (u16 *)sleep_data_ls[index].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	} else {
#ifdef CONFIG_NEC_SUPPORT_PORTRAIT_DISP
		/* portrait */
		coord_info = &coord_info_portrait[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_data_pr[index];
		data = (u16 *)sleep_data_pr[index].data;
#endif // CONFIG_NEC_SUPPORT_PORTRAIT_DISP
	}

	display_picto(info, coord_info, data);
}

static void display_suspend(void)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;
	int batt_remain;
	int bt_enable;
	int charging;
	int vertical;
	u32 prev_panel_disp_flag = panel_disp_flag;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {
		batt_remain = GET_PANEL_DISP_INFO(BATTERY);
		bt_enable = GET_PANEL_DISP_INFO(WAKEONBT);
		charging = GET_PANEL_DISP_INFO(CHARGE);
#if !defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 0;
#elif defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && !defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 1;
#else
		vertical = GET_PANEL_DISP_INFO(VERTICAL);
#endif /* CONFIG_NEC_SUPPORT_PORTRAIT_DISP && CONFIG_NEC_SUPPORT_LANDSCAPE_DISP */

		mutex_unlock(&panel_sc.display_mutex);
		set_display_writemode(WRITEMODE_BUFFER);
		display_all_reset();

		if (batt_remain > 100 || batt_remain < 0)
			batt_remain = -2;

		display_battery_picto(batt_remain, charging, vertical);

		display_battery_remain(batt_remain, vertical);

		display_bt_wave(bt_enable, vertical);

		mutex_lock(&panel_sc.display_mutex);
		while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {
			if (prev_panel_disp_flag != panel_disp_flag) {
				prev_panel_disp_flag = panel_disp_flag;
				break;
			}
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			display_sleep_picto(i%2, vertical);
			if (get_display_writemode() == WRITEMODE_BUFFER) {
				flush_displaybuffer();
				set_display_writemode(WRITEMODE_DIRECT);
			}
			wait_event_timeout(sc->wait_q, sc->wait_condition,
				msecs_to_jiffies(1000));
			i++;
			mutex_lock(&panel_sc.display_mutex);
		}

		mutex_unlock(&panel_sc.display_mutex);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static void display_poweroff(void)
{
	int batt_remain;
	int bt_enable;
	int charging;
	int vertical;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(PWOFF)) {

		batt_remain = GET_PANEL_DISP_INFO(BATTERY);
		bt_enable = GET_PANEL_DISP_INFO(WAKEONBT);
		charging = GET_PANEL_DISP_INFO(CHARGE);
#if !defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 0;
#elif defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && !defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 1;
#else
		vertical = GET_PANEL_DISP_INFO(VERTICAL);
#endif /* CONFIG_NEC_SUPPORT_PORTRAIT_DISP && CONFIG_NEC_SUPPORT_LANDSCAPE_DISP */

		mutex_unlock(&panel_sc.display_mutex);
		set_display_writemode(WRITEMODE_BUFFER);
		display_all_reset();


		if (batt_remain > 100 || batt_remain < 0)
			batt_remain = -2;

		display_battery_picto_pwoff(batt_remain, charging, vertical);

		display_battery_remain_pwoff(batt_remain, vertical);

		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static void display_progress_picto(int rate, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;

	if ((rate > PROG_STAT_10) || (rate < 0))
		return;

#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	/* landscape only */
	coord_info = &coord_info_landscape[PROGRESS_BAR_PICTO];
	info = (struct image_info_t *)&prog_stat_ls[rate];

	data = (u16 *)prog_stat_ls[rate].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	display_picto(info, coord_info, data);
}

static void display_progress_bar(int vertical)
{
	int rate;
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(PROGRESS_BAR)) {
		rate = GET_PANEL_DISP_INFO(PROGRESS_RATE);

		mutex_unlock(&panel_sc.display_mutex);
		display_progress_picto(rate, vertical);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		wait_event_timeout(sc->wait_q, sc->wait_condition,
				   msecs_to_jiffies(1000));
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static void display_progress_animation(int vertical, int msec, int start, int end)
{
	int rate = 0;
	struct panel_softc *sc = &panel_sc;

	if (start > 0)
		rate = start;
	if (end < 0)
		end = 10;

	mutex_lock(&panel_sc.display_mutex);
	while (GET_PANEL_DISP_INFO(ON) &&
		GET_PANEL_DISP_INFO(BOOT)) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		if (rate <= end) {
			display_progress_picto(rate, vertical);
			if (get_display_writemode() == WRITEMODE_BUFFER) {
				flush_displaybuffer();
				set_display_writemode(WRITEMODE_DIRECT);
			}
		}
		wait_event_timeout(sc->wait_q, sc->wait_condition,
					msecs_to_jiffies(msec));
		rate++;
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

#if defined(CONFIG_NEC_SUPPORT_LEGAL_PAGE)
static void display_legal_page(int kind, int vertical)
{
	u16 *data = NULL;
	struct image_info_t *info = NULL;
	struct coordinate_info_t *coord_info = NULL;

	if ((kind > LEGAL_PAGE_05) || (kind < LEGAL_PAGE_01))
		kind = LEGAL_PAGE_01;

#ifdef CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	/* landscape only */
	coord_info = &coord_info_landscape[LEGAL_PAGE];
	info = (struct image_info_t *)&legal_ls[kind];

	data = (u16 *)legal_ls[kind].data;
#endif // CONFIG_NEC_SUPPORT_LANDSCAPE_DISP
	display_picto(info, coord_info, data);
}

static void display_legal(int vertical)
{
	int kind;
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(LEGAL_PAGE)) {
		kind = GET_PANEL_DISP_INFO(LEGAL_PAGE) - 1;
		sc->wait_condition = 0;

		mutex_unlock(&panel_sc.display_mutex);
		display_legal_page(kind, vertical);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		wait_event(sc->wait_q, sc->wait_condition);
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}
#endif /* CONFIG_NEC_SUPPORT_LEGAL_PAGE */

static int nec_display_polling_thread(void *arg)
{
	int disp_len = NEC_IMG_SIZE * 2;
	struct panel_softc *sc = &panel_sc;
	int vertical = 1;
	u32 boot_flag = 0;
	int goto_off = 1;

	printk(KERN_DEBUG "display polling thread start!\n");
	// TODO: Allocate DRM framebuffer for overlay
	//       set MediaTek overlay registers directly, or allocate and assign a plane.
	bb_virt = dma_alloc_coherent(&bb_pdev->dev, disp_len,
					&bb_phys, GFP_KERNEL | GFP_DMA);

	if (!bb_virt) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ Could not allocate external memory\n");
		return 0;
	}

#if !defined(USE_NEC_BOOTPARAM)
	mutex_lock(&panel_sc.display_mutex);
	while (nand_init_comp < NAND_INIT_COMP_OK) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		wait_event_timeout(sc->wait_q, sc->wait_condition,
					msecs_to_jiffies(500));
		mutex_lock(&panel_sc.display_mutex);
	}
	pr_info("@@@@@@@@@@@@@@ %s panel disp start\n", __func__);
	boot_flag = get_msm_boot_flag();
	if ((GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) ||
		((GET_BOOT_FLAG_INFO(boot_flag, TEST_MODE)
		== BOOT_FLAG_TEST_MODE_ENABLE))
		) {
		panel_disp_flag = 0;
	} else {
		SET_PANEL_DISP_INFO(ON, PANEL_STATE_DISP_ON);
		SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	}
	mutex_unlock(&panel_sc.display_mutex);
#endif /* !USE_NEC_BOOTPARAM */

	if (GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ LCD disable\n");
	} else {
#if !defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 0;
#elif defined(CONFIG_NEC_SUPPORT_PORTRAIT_DISP) && !defined(CONFIG_NEC_SUPPORT_LANDSCAPE_DISP)
		vertical = 1;
#else
		if (GET_BOOT_FLAG_INFO(boot_flag, DISP_WAY)
			== BOOT_FLAG_DISP_HORIZONTAL) {
			/* landscape */
			vertical = 0;
		} else {
			/* portrait */
			vertical = 1;
		}
#endif /* CONFIG_NEC_SUPPORT_PORTRAIT_DISP && CONFIG_NEC_SUPPORT_LANDSCAPE_DISP */
		nec_panel_set_no_reset_once();

		display_on();
		display_boot_animation(vertical, PANEL_DISP_PROGRESS_BAR_INTERVAL);
	}

	while (!kthread_should_stop()) {
		mutex_lock(&panel_sc.display_mutex);
		if (!GET_PANEL_DISP_INFO(ON)) {
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			dev_err(&bb_pdev->dev,
				"*************** sleep!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);
			if (goto_off) {
				display_all_reset();
				nec_panel_off(NEC_PANEL_USER_SPLASH);
				goto_off = 0;
			}
			atomic_set(&sc->sleeped, 1);
			wake_up(&sc->sleep_wait_q);
			wait_event(sc->wait_q, sc->wait_condition);
			atomic_set(&sc->sleeped, 0);
			display_on();
			dev_err(&bb_pdev->dev,
				"*************** wake up!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);

			display_all_reset();
			continue;
		}

		mutex_unlock(&panel_sc.display_mutex);
		if (GET_PANEL_DISP_INFO(BOOT)) {
			display_boot_animation(vertical, PANEL_DISP_PROGRESS_BAR_INTERVAL);
		} else if (GET_PANEL_DISP_INFO(PROGRESS_BAR)) {
			display_progress_bar(vertical);
		} else if (GET_PANEL_DISP_INFO(CARRIER_SW)) {
			/* not support*/
			/* display_carrier_switch(vertical); */
#if defined(CONFIG_NEC_SUPPORT_LEGAL_PAGE)
		} else if (GET_PANEL_DISP_INFO(LEGAL_PAGE)) {
			display_legal(vertical);
			goto_off = 1;
#endif /* CONFIG_NEC_SUPPORT_LEGAL_PAGE */
		} else {
			if (GET_PANEL_DISP_INFO(PWOFF)) {
				display_poweroff();
				goto_off = 1;
			} else if (GET_PANEL_DISP_INFO(SLEEP)) {
				display_suspend();
				goto_off = 1;
			} else {
				display_resume();
				goto_off = 1;
			}
		}
	}
	return 0;
}

static void nec_display_init(struct platform_device *pdev)
{
	nec_display_polling = kthread_run(nec_display_polling_thread, NULL, "nec-display");
	if (IS_ERR(nec_display_polling)) {
		pr_err("failed to create polling thread: %ld\n", PTR_ERR(nec_display_polling));
		return;
	}
}


/*****************************************************************************
 * Driver register/unregister operations
 *****************************************************************************/
static int nec_splash_probe(struct platform_device *pdev)
{
	struct panel_softc *sc = &panel_sc;
	int rc;

	printk(KERN_DEBUG "%s enter", __func__);
	if (!nec_panel_is_initialized()) {
		pr_info("Defer splash probe until nec_panel initialized.\n");
		return -EPROBE_DEFER;
	}

	mutex_init(&panel_sc.display_mutex);

	rc = nec_splash_sysfs_add(&pdev->dev);
	if (rc)
		return rc;

	init_waitqueue_head(&sc->wait_q);
	init_waitqueue_head(&sc->sleep_wait_q);

	bb_pdev = pdev;
	nec_display_init(pdev);

	pr_info("%s: done.\n", __func__);
	printk(KERN_DEBUG "%s exit", __func__);

	return 0;
}

static int nec_splash_sysfs_add(struct device *dev)
{
	int retval;
#if defined(USE_NEC_BOOTPARAM)
	u32 boot_flag = get_msm_boot_flag();

	retval = sysfs_create_group(&dev->kobj, &fs_attr_group);

	mutex_lock(&panel_sc.display_mutex);
	if ((GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) ||
		((GET_BOOT_FLAG_INFO(boot_flag, TEST_MODE)
		== BOOT_FLAG_TEST_MODE_ENABLE))
		) {
		panel_disp_flag = 0;
	} else {
		SET_PANEL_DISP_INFO(ON, PANEL_STATE_DISP_ON);
		SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	}
	mutex_unlock(&panel_sc.display_mutex);
#else /* !USE_NEC_BOOTPARAM */
	retval = sysfs_create_group(&dev->kobj, &fs_attr_group);
#endif /* USE_NEC_BOOTPARAM */

	return retval;
}


/*****************************************************************************/
static const struct of_device_id nec_splash_of_match[] = {
	{ .compatible = NEC_SPLASH_NAME, },
	{ },
};
MODULE_DEVICE_TABLE(of, nec_splash_of_match);

static struct platform_driver nec_splash_driver = {
	.probe = nec_splash_probe,
	.driver = {
		.name = NEC_SPLASH_NAME,
		.of_match_table = nec_splash_of_match,
	},
};
module_platform_driver(nec_splash_driver);
MODULE_LICENSE("GPL v2");
