// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020 MediaTek Inc.
 * Author: Andy Teng <andy.teng@mediatek.com>
 *
 */

#include <linux/module.h>
#include "pinctrl-mtk-mt6880.h"
#include "pinctrl-paris.h"

/* MT6880 have multiple bases to program pin configuration listed as the below:
 * gpio:0x10005000,     iocfg_rm:0x11C10000, iocfg_rb:0x11C20000,
 * iocfg_bl:0x11D00000, iocfg_bm:0x11D10000, iocfg_br:0x11D20000,
 * iocfg_lt:0x11E00000, iocfg_tl:0x11F00000
 * _i_based could be used to indicate what base the pin should be mapped into.
 */

#define PIN_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits, \
		       32, 0)

#define PINS_FIELD_BASE(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits) \
	PIN_FIELD_CALC(s_pin, e_pin, i_base, s_addr, x_addrs, s_bit, x_bits,  \
		       32, 1)

static const struct mtk_pin_field_calc mt6880_pin_mode_range[] = {
	PIN_FIELD_BASE(0, 7, 0, 0x0300, 0x10, 0, 4),
	PIN_FIELD_BASE(8, 15, 0, 0x0310, 0x10, 0, 4),
	PIN_FIELD_BASE(16, 23, 0, 0x0320, 0x10, 0, 4),
	PIN_FIELD_BASE(24, 31, 0, 0x0330, 0x10, 0, 4),
	PIN_FIELD_BASE(32, 39, 0, 0x0340, 0x10, 0, 4),
	PIN_FIELD_BASE(40, 47, 0, 0x0350, 0x10, 0, 4),
	PIN_FIELD_BASE(48, 55, 0, 0x0360, 0x10, 0, 4),
	PIN_FIELD_BASE(56, 63, 0, 0x0370, 0x10, 0, 4),
	PIN_FIELD_BASE(64, 71, 0, 0x0380, 0x10, 0, 4),
	PIN_FIELD_BASE(72, 79, 0, 0x0390, 0x10, 0, 4),
	PIN_FIELD_BASE(80, 87, 0, 0x03a0, 0x10, 0, 4),
	PIN_FIELD_BASE(88, 95, 0, 0x03b0, 0x10, 0, 4),
	PIN_FIELD_BASE(96, 103, 0, 0x03c0, 0x10, 0, 4),
	PIN_FIELD_BASE(104, 111, 0, 0x03d0, 0x10, 0, 4),
	PIN_FIELD_BASE(112, 119, 0, 0x03e0, 0x10, 0, 4),
	PIN_FIELD_BASE(120, 127, 0, 0x03f0, 0x10, 0, 4),
	PIN_FIELD_BASE(128, 135, 0, 0x0400, 0x10, 0, 4),
	PIN_FIELD_BASE(136, 143, 0, 0x0410, 0x10, 0, 4),
	PIN_FIELD_BASE(144, 151, 0, 0x0420, 0x10, 0, 4),
	PIN_FIELD_BASE(152, 159, 0, 0x0430, 0x10, 0, 4),
	PIN_FIELD_BASE(160, 167, 0, 0x0440, 0x10, 0, 4),
	PIN_FIELD_BASE(168, 175, 0, 0x0450, 0x10, 0, 4),
	PIN_FIELD_BASE(176, 183, 0, 0x0460, 0x10, 0, 4),
	PIN_FIELD_BASE(184, 191, 0, 0x0470, 0x10, 0, 4),
	PIN_FIELD_BASE(192, 199, 0, 0x0480, 0x10, 0, 4),
	PIN_FIELD_BASE(200, 207, 0, 0x0490, 0x10, 0, 4),
	PIN_FIELD_BASE(208, 215, 0, 0x04a0, 0x10, 0, 4),
	PIN_FIELD_BASE(216, 223, 0, 0x04b0, 0x10, 0, 4),
	PIN_FIELD_BASE(224, 231, 0, 0x04c0, 0x10, 0, 4),
	PIN_FIELD_BASE(232, 234, 0, 0x04d0, 0x10, 0, 4),
};

static const struct mtk_pin_field_calc mt6880_pin_dir_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x0000, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x0010, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 159, 0, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(160, 191, 0, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(192, 223, 0, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(224, 234, 0, 0x0070, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_di_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x0200, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x0210, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x0220, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x0230, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 159, 0, 0x0240, 0x10, 0, 1),
	PIN_FIELD_BASE(160, 191, 0, 0x0250, 0x10, 0, 1),
	PIN_FIELD_BASE(192, 223, 0, 0x0260, 0x10, 0, 1),
	PIN_FIELD_BASE(224, 234, 0, 0x0270, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_do_range[] = {
	PIN_FIELD_BASE(0, 31, 0, 0x0100, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 63, 0, 0x0110, 0x10, 0, 1),
	PIN_FIELD_BASE(64, 95, 0, 0x0120, 0x10, 0, 1),
	PIN_FIELD_BASE(96, 127, 0, 0x0130, 0x10, 0, 1),
	PIN_FIELD_BASE(128, 159, 0, 0x0140, 0x10, 0, 1),
	PIN_FIELD_BASE(160, 191, 0, 0x0150, 0x10, 0, 1),
	PIN_FIELD_BASE(192, 223, 0, 0x0160, 0x10, 0, 1),
	PIN_FIELD_BASE(224, 234, 0, 0x0170, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_ies_range[] = {
	PIN_FIELD_BASE(0, 3, 3, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(4, 7, 1, 0x0070, 0x10, 24, 1),
	PIN_FIELD_BASE(8, 9, 4, 0x00a0, 0x10, 29, 1),
	PIN_FIELD_BASE(10, 25, 4, 0x00a0, 0x10, 5, 1),
	PIN_FIELD_BASE(26, 31, 1, 0x0070, 0x10, 18, 1),
	PIN_FIELD_BASE(32, 39, 4, 0x00a0, 0x10, 21, 1),
	PIN_FIELD_BASE(40, 40, 6, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(41, 42, 6, 0x0030, 0x10, 19, 1),
	PIN_FIELD_BASE(43, 43, 6, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(44, 44, 6, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(45, 45, 6, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(46, 46, 6, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(47, 47, 6, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(48, 48, 6, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(49, 49, 6, 0x0030, 0x10, 1, 1),
	PIN_FIELD_BASE(50, 50, 6, 0x0030, 0x10, 6, 1),
	PIN_FIELD_BASE(51, 51, 6, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(52, 52, 6, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(53, 53, 6, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(54, 54, 6, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(55, 56, 6, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(57, 57, 6, 0x0030, 0x10, 18, 1),
	PIN_FIELD_BASE(58, 58, 6, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(59, 59, 6, 0x0030, 0x10, 17, 1),
	PIN_FIELD_BASE(60, 60, 6, 0x0030, 0x10, 16, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(63, 66, 3, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(67, 68, 3, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(70, 71, 3, 0x0040, 0x10, 16, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(73, 74, 3, 0x0040, 0x10, 19, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0040, 0x10, 18, 1),
	PIN_FIELD_BASE(76, 77, 3, 0x0040, 0x10, 22, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0040, 0x10, 21, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0040, 0x10, 26, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0040, 0x10, 27, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(83, 84, 3, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(85, 86, 3, 0x0040, 0x10, 29, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x0040, 0x10, 28, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x0040, 0x10, 31, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(91, 91, 4, 0x00b0, 0x10, 6, 1),
	PIN_FIELD_BASE(92, 92, 4, 0x00b0, 0x10, 3, 1),
	PIN_FIELD_BASE(93, 93, 4, 0x00b0, 0x10, 5, 1),
	PIN_FIELD_BASE(94, 94, 4, 0x00b0, 0x10, 4, 1),
	PIN_FIELD_BASE(95, 95, 4, 0x00b0, 0x10, 2, 1),
	PIN_FIELD_BASE(96, 96, 4, 0x00b0, 0x10, 7, 1),
	PIN_FIELD_BASE(97, 97, 4, 0x00b0, 0x10, 14, 1),
	PIN_FIELD_BASE(98, 98, 4, 0x00b0, 0x10, 17, 1),
	PIN_FIELD_BASE(99, 100, 4, 0x00b0, 0x10, 15, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x00b0, 0x10, 29, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x00b0, 0x10, 28, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x00b0, 0x10, 30, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x00b0, 0x10, 27, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x00b0, 0x10, 24, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x00b0, 0x10, 21, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x00b0, 0x10, 25, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x00b0, 0x10, 22, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x00b0, 0x10, 26, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x00b0, 0x10, 23, 1),
	PIN_FIELD_BASE(111, 116, 4, 0x00b0, 0x10, 8, 1),
	PIN_FIELD_BASE(117, 117, 4, 0x00b0, 0x10, 20, 1),
	PIN_FIELD_BASE(118, 118, 4, 0x00c0, 0x10, 5, 1),
	PIN_FIELD_BASE(119, 120, 4, 0x00c0, 0x10, 3, 1),
	PIN_FIELD_BASE(121, 121, 4, 0x00b0, 0x10, 31, 1),
	PIN_FIELD_BASE(122, 124, 4, 0x00c0, 0x10, 0, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x00a0, 0x10, 31, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(127, 128, 4, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x00a0, 0x10, 2, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x00b0, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 132, 4, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(133, 133, 4, 0x00c0, 0x10, 6, 1),
	PIN_FIELD_BASE(134, 134, 4, 0x00b0, 0x10, 19, 1),
	PIN_FIELD_BASE(135, 135, 4, 0x00b0, 0x10, 18, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0010, 0x10, 12, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0010, 0x10, 11, 1),
	PIN_FIELD_BASE(138, 139, 5, 0x0010, 0x10, 15, 1),
	PIN_FIELD_BASE(140, 141, 5, 0x0010, 0x10, 13, 1),
	PIN_FIELD_BASE(142, 143, 5, 0x0010, 0x10, 0, 1),
	PIN_FIELD_BASE(144, 151, 5, 0x0010, 0x10, 3, 1),
	PIN_FIELD_BASE(152, 152, 5, 0x0010, 0x10, 2, 1),
	PIN_FIELD_BASE(153, 163, 2, 0x0010, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 165, 1, 0x0070, 0x10, 30, 1),
	PIN_FIELD_BASE(166, 167, 1, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x0070, 0x10, 29, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x0070, 0x10, 12, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(174, 174, 1, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(175, 175, 1, 0x0070, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 1, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(177, 177, 1, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(178, 178, 1, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 1, 0x0070, 0x10, 14, 1),
	PIN_FIELD_BASE(180, 180, 1, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 1, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(182, 182, 1, 0x0070, 0x10, 7, 1),
	PIN_FIELD_BASE(183, 183, 1, 0x0070, 0x10, 15, 1),
	PIN_FIELD_BASE(184, 184, 1, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(185, 185, 1, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(186, 186, 1, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(187, 187, 1, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(188, 188, 1, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(189, 189, 1, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(190, 190, 1, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(191, 191, 1, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(192, 192, 1, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(193, 193, 1, 0x0080, 0x10, 14, 1),
	PIN_FIELD_BASE(194, 194, 1, 0x0080, 0x10, 13, 1),
	PIN_FIELD_BASE(195, 195, 1, 0x0080, 0x10, 15, 1),
	PIN_FIELD_BASE(196, 196, 1, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(197, 197, 1, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(198, 198, 1, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(199, 199, 1, 0x0080, 0x10, 21, 1),
	PIN_FIELD_BASE(200, 200, 1, 0x0080, 0x10, 16, 1),
	PIN_FIELD_BASE(201, 201, 1, 0x0080, 0x10, 22, 1),
	PIN_FIELD_BASE(202, 205, 1, 0x0080, 0x10, 17, 1),
	PIN_FIELD_BASE(206, 207, 1, 0x0070, 0x10, 16, 1),
	PIN_FIELD_BASE(208, 208, 1, 0x0070, 0x10, 28, 1),
	PIN_FIELD_BASE(209, 209, 7, 0x0030, 0x10, 9, 1),
	PIN_FIELD_BASE(210, 210, 7, 0x0030, 0x10, 7, 1),
	PIN_FIELD_BASE(211, 212, 7, 0x0030, 0x10, 5, 1),
	PIN_FIELD_BASE(213, 213, 7, 0x0030, 0x10, 10, 1),
	PIN_FIELD_BASE(214, 214, 7, 0x0030, 0x10, 8, 1),
	PIN_FIELD_BASE(215, 222, 7, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(223, 223, 7, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(224, 224, 7, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(225, 225, 7, 0x0030, 0x10, 24, 1),
	PIN_FIELD_BASE(226, 226, 7, 0x0030, 0x10, 23, 1),
	PIN_FIELD_BASE(227, 227, 7, 0x0030, 0x10, 25, 1),
	PIN_FIELD_BASE(228, 228, 7, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(229, 229, 7, 0x0030, 0x10, 14, 1),
	PIN_FIELD_BASE(230, 230, 7, 0x0030, 0x10, 4, 1),
	PIN_FIELD_BASE(231, 231, 7, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(232, 232, 7, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(233, 234, 7, 0x0030, 0x10, 1, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_smt_range[] = {
	PIN_FIELD_BASE(0, 0, 3, 0x00e0, 0x10, 6, 1),
	PINS_FIELD_BASE(1, 2, 3, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(3, 3, 3, 0x00e0, 0x10, 6, 1),
	PINS_FIELD_BASE(4, 5, 1, 0x0140, 0x10, 14, 1),
	PINS_FIELD_BASE(6, 7, 1, 0x0140, 0x10, 13, 1),
	PINS_FIELD_BASE(8, 9, 4, 0x0190, 0x10, 10, 1),
	PINS_FIELD_BASE(10, 12, 4, 0x0190, 0x10, 11, 1),
	PINS_FIELD_BASE(13, 16, 4, 0x0190, 0x10, 12, 1),
	PINS_FIELD_BASE(17, 19, 4, 0x0190, 0x10, 13, 1),
	PIN_FIELD_BASE(20, 20, 4, 0x0190, 0x10, 1, 1),
	PIN_FIELD_BASE(21, 21, 4, 0x0190, 0x10, 13, 1),
	PINS_FIELD_BASE(22, 24, 4, 0x0190, 0x10, 1, 1),
	PIN_FIELD_BASE(25, 25, 4, 0x0190, 0x10, 2, 1),
	PIN_FIELD_BASE(26, 31, 1, 0x0140, 0x10, 0, 1),
	PIN_FIELD_BASE(32, 32, 4, 0x0190, 0x10, 8, 1),
	PINS_FIELD_BASE(33, 36, 4, 0x0190, 0x10, 9, 1),
	PINS_FIELD_BASE(37, 38, 4, 0x0190, 0x10, 10, 1),
	PIN_FIELD_BASE(39, 39, 4, 0x0190, 0x10, 11, 1),
	PIN_FIELD_BASE(40, 41, 6, 0x00d0, 0x10, 18, 1),
	PIN_FIELD_BASE(42, 42, 6, 0x00d0, 0x10, 18, 1),
	PIN_FIELD_BASE(43, 43, 6, 0x00d0, 0x10, 9, 1),
	PIN_FIELD_BASE(44, 44, 6, 0x00d0, 0x10, 8, 1),
	PIN_FIELD_BASE(45, 45, 6, 0x00d0, 0x10, 7, 1),
	PIN_FIELD_BASE(46, 46, 6, 0x00d0, 0x10, 6, 1),
	PIN_FIELD_BASE(47, 47, 6, 0x00d0, 0x10, 11, 1),
	PIN_FIELD_BASE(48, 48, 6, 0x00d0, 0x10, 1, 1),
	PIN_FIELD_BASE(49, 49, 6, 0x00d0, 0x10, 0, 1),
	PIN_FIELD_BASE(50, 50, 6, 0x00d0, 0x10, 5, 1),
	PIN_FIELD_BASE(51, 51, 6, 0x00d0, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 6, 0x00d0, 0x10, 3, 1),
	PIN_FIELD_BASE(53, 53, 6, 0x00d0, 0x10, 2, 1),
	PIN_FIELD_BASE(54, 54, 6, 0x00d0, 0x10, 10, 1),
	PIN_FIELD_BASE(55, 56, 6, 0x00d0, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 6, 0x00d0, 0x10, 17, 1),
	PIN_FIELD_BASE(58, 58, 6, 0x00d0, 0x10, 14, 1),
	PIN_FIELD_BASE(59, 59, 6, 0x00d0, 0x10, 16, 1),
	PIN_FIELD_BASE(60, 60, 6, 0x00d0, 0x10, 15, 1),
	PINS_FIELD_BASE(61, 64, 3, 0x00e0, 0x10, 1, 1),
	PINS_FIELD_BASE(65, 68, 3, 0x00e0, 0x10, 2, 1),
	PINS_FIELD_BASE(69, 72, 3, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(73, 74, 3, 0x00e0, 0x10, 4, 1),
	PINS_FIELD_BASE(75, 76, 3, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(77, 77, 3, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x00e0, 0x10, 10, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x00e0, 0x10, 8, 1),
	PINS_FIELD_BASE(83, 84, 3, 0x00e0, 0x10, 6, 1),
	PINS_FIELD_BASE(85, 90, 3, 0x00e0, 0x10, 0, 1),
	PINS_FIELD_BASE(91, 94, 4, 0x0190, 0x10, 0, 1),
	PINS_FIELD_BASE(95, 98, 4, 0x0190, 0x10, 5, 1),
	PINS_FIELD_BASE(99, 100, 4, 0x0190, 0x10, 6, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0190, 0x10, 22, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0190, 0x10, 21, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0190, 0x10, 23, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0190, 0x10, 20, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0190, 0x10, 17, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0190, 0x10, 14, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0190, 0x10, 18, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0190, 0x10, 15, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0190, 0x10, 19, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0190, 0x10, 16, 1),
	PINS_FIELD_BASE(111, 112, 4, 0x0190, 0x10, 6, 1),
	PINS_FIELD_BASE(113, 116, 4, 0x0190, 0x10, 7, 1),
	PIN_FIELD_BASE(117, 117, 4, 0x0190, 0x10, 8, 1),
	PIN_FIELD_BASE(118, 118, 4, 0x0190, 0x10, 28, 1),
	PINS_FIELD_BASE(119, 120, 4, 0x0190, 0x10, 8, 1),
	PIN_FIELD_BASE(121, 124, 4, 0x0190, 0x10, 24, 1),
	PINS_FIELD_BASE(125, 127, 4, 0x0190, 0x10, 2, 1),
	PINS_FIELD_BASE(128, 131, 4, 0x0190, 0x10, 3, 1),
	PINS_FIELD_BASE(132, 135, 4, 0x0190, 0x10, 4, 1),
	PINS_FIELD_BASE(136, 137, 5, 0x0050, 0x10, 2, 1),
	PINS_FIELD_BASE(138, 139, 5, 0x0050, 0x10, 1, 1),
	PINS_FIELD_BASE(140, 141, 5, 0x0050, 0x10, 0, 1),
	PINS_FIELD_BASE(142, 143, 5, 0x0050, 0x10, 2, 1),
	PINS_FIELD_BASE(144, 147, 5, 0x0050, 0x10, 3, 1),
	PINS_FIELD_BASE(148, 151, 5, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(152, 152, 5, 0x0050, 0x10, 5, 1),
	PINS_FIELD_BASE(153, 154, 2, 0x0050, 0x10, 0, 1),
	PINS_FIELD_BASE(155, 156, 2, 0x0050, 0x10, 1, 1),
	PINS_FIELD_BASE(157, 160, 2, 0x0050, 0x10, 2, 1),
	PINS_FIELD_BASE(161, 163, 2, 0x0050, 0x10, 3, 1),
	PINS_FIELD_BASE(164, 165, 1, 0x0140, 0x10, 7, 1),
	PINS_FIELD_BASE(166, 167, 1, 0x0140, 0x10, 8, 1),
	PINS_FIELD_BASE(168, 171, 1, 0x0140, 0x10, 9, 1),
	PINS_FIELD_BASE(172, 175, 1, 0x0140, 0x10, 10, 1),
	PINS_FIELD_BASE(176, 179, 1, 0x0140, 0x10, 11, 1),
	PINS_FIELD_BASE(180, 183, 1, 0x0140, 0x10, 12, 1),
	PIN_FIELD_BASE(184, 184, 1, 0x0140, 0x10, 13, 1),
	PIN_FIELD_BASE(185, 185, 1, 0x0140, 0x10, 23, 1),
	PIN_FIELD_BASE(186, 186, 1, 0x0140, 0x10, 22, 1),
	PIN_FIELD_BASE(187, 187, 1, 0x0140, 0x10, 24, 1),
	PIN_FIELD_BASE(188, 188, 1, 0x0140, 0x10, 21, 1),
	PIN_FIELD_BASE(189, 191, 1, 0x0140, 0x10, 14, 1),
	PINS_FIELD_BASE(192, 193, 1, 0x0140, 0x10, 15, 1),
	PIN_FIELD_BASE(194, 195, 1, 0x0140, 0x10, 14, 1),
	PIN_FIELD_BASE(196, 196, 1, 0x0140, 0x10, 13, 1),
	PIN_FIELD_BASE(197, 197, 1, 0x0140, 0x10, 20, 1),
	PIN_FIELD_BASE(198, 198, 1, 0x0140, 0x10, 19, 1),
	PINS_FIELD_BASE(199, 201, 1, 0x0140, 0x10, 16, 1),
	PINS_FIELD_BASE(202, 205, 1, 0x0140, 0x10, 17, 1),
	PINS_FIELD_BASE(206, 207, 1, 0x0140, 0x10, 18, 1),
	PIN_FIELD_BASE(208, 208, 1, 0x0140, 0x10, 6, 1),
	PINS_FIELD_BASE(209, 210, 7, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(211, 211, 7, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(212, 212, 7, 0x0070, 0x10, 7, 1),
	PINS_FIELD_BASE(213, 214, 7, 0x0070, 0x10, 8, 1),
	PINS_FIELD_BASE(215, 216, 7, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(217, 217, 7, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(218, 219, 7, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(220, 220, 7, 0x0070, 0x10, 4, 1),
	PINS_FIELD_BASE(221, 222, 7, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(223, 223, 7, 0x0070, 0x10, 6, 1),
	PINS_FIELD_BASE(224, 226, 7, 0x0070, 0x10, 7, 1),
	PINS_FIELD_BASE(227, 229, 7, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(230, 230, 7, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(231, 231, 7, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(232, 232, 7, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(233, 234, 7, 0x0070, 0x10, 1, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_pu_range[] = {
	PIN_FIELD_BASE(0, 3, 3, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(4, 7, 1, 0x00c0, 0x10, 18, 1),
	PIN_FIELD_BASE(8, 9, 4, 0x0110, 0x10, 29, 1),
	PIN_FIELD_BASE(10, 25, 4, 0x0110, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 39, 4, 0x0110, 0x10, 21, 1),
	PIN_FIELD_BASE(40, 42, 6, 0x0060, 0x10, 0, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0090, 0x10, 11, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0090, 0x10, 10, 1),
	PIN_FIELD_BASE(63, 66, 3, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(67, 68, 3, 0x0090, 0x10, 13, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0090, 0x10, 12, 1),
	PIN_FIELD_BASE(70, 71, 3, 0x0090, 0x10, 16, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0090, 0x10, 15, 1),
	PIN_FIELD_BASE(73, 74, 3, 0x0090, 0x10, 19, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0090, 0x10, 18, 1),
	PIN_FIELD_BASE(76, 77, 3, 0x0090, 0x10, 22, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0090, 0x10, 21, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0090, 0x10, 26, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0090, 0x10, 24, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0090, 0x10, 27, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0090, 0x10, 25, 1),
	PIN_FIELD_BASE(83, 84, 3, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(91, 91, 4, 0x0120, 0x10, 6, 1),
	PIN_FIELD_BASE(92, 92, 4, 0x0120, 0x10, 3, 1),
	PIN_FIELD_BASE(93, 93, 4, 0x0120, 0x10, 5, 1),
	PIN_FIELD_BASE(94, 94, 4, 0x0120, 0x10, 4, 1),
	PIN_FIELD_BASE(95, 95, 4, 0x0120, 0x10, 2, 1),
	PIN_FIELD_BASE(96, 97, 4, 0x0120, 0x10, 7, 1),
	PIN_FIELD_BASE(98, 98, 4, 0x0120, 0x10, 11, 1),
	PIN_FIELD_BASE(99, 100, 4, 0x0120, 0x10, 9, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x0120, 0x10, 23, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x0120, 0x10, 22, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x0120, 0x10, 24, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x0120, 0x10, 21, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x0120, 0x10, 18, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x0120, 0x10, 15, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x0120, 0x10, 19, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x0120, 0x10, 16, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x0120, 0x10, 20, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x0120, 0x10, 17, 1),
	PIN_FIELD_BASE(117, 117, 4, 0x0120, 0x10, 14, 1),
	PIN_FIELD_BASE(118, 118, 4, 0x0130, 0x10, 1, 1),
	PIN_FIELD_BASE(119, 120, 4, 0x0120, 0x10, 29, 1),
	PIN_FIELD_BASE(121, 124, 4, 0x0120, 0x10, 25, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x0110, 0x10, 31, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x0120, 0x10, 0, 1),
	PIN_FIELD_BASE(127, 128, 4, 0x0110, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x0110, 0x10, 2, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x0120, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 132, 4, 0x0110, 0x10, 0, 1),
	PIN_FIELD_BASE(133, 133, 4, 0x0130, 0x10, 2, 1),
	PIN_FIELD_BASE(134, 134, 4, 0x0120, 0x10, 13, 1),
	PIN_FIELD_BASE(135, 135, 4, 0x0120, 0x10, 12, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0030, 0x10, 12, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0030, 0x10, 11, 1),
	PIN_FIELD_BASE(138, 139, 5, 0x0030, 0x10, 15, 1),
	PIN_FIELD_BASE(140, 141, 5, 0x0030, 0x10, 13, 1),
	PIN_FIELD_BASE(142, 143, 5, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(144, 151, 5, 0x0030, 0x10, 3, 1),
	PIN_FIELD_BASE(152, 152, 5, 0x0030, 0x10, 2, 1),
	PIN_FIELD_BASE(153, 163, 2, 0x0030, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 167, 1, 0x00c0, 0x10, 24, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x00c0, 0x10, 23, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x00c0, 0x10, 0, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x00c0, 0x10, 4, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x00c0, 0x10, 12, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x00c0, 0x10, 8, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x00c0, 0x10, 1, 1),
	PIN_FIELD_BASE(174, 174, 1, 0x00c0, 0x10, 5, 1),
	PIN_FIELD_BASE(175, 175, 1, 0x00c0, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 1, 0x00c0, 0x10, 9, 1),
	PIN_FIELD_BASE(177, 177, 1, 0x00c0, 0x10, 2, 1),
	PIN_FIELD_BASE(178, 178, 1, 0x00c0, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 1, 0x00c0, 0x10, 14, 1),
	PIN_FIELD_BASE(180, 180, 1, 0x00c0, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 1, 0x00c0, 0x10, 3, 1),
	PIN_FIELD_BASE(182, 182, 1, 0x00c0, 0x10, 7, 1),
	PIN_FIELD_BASE(183, 183, 1, 0x00c0, 0x10, 15, 1),
	PIN_FIELD_BASE(184, 184, 1, 0x00c0, 0x10, 11, 1),
	PIN_FIELD_BASE(185, 185, 1, 0x00d0, 0x10, 0, 1),
	PIN_FIELD_BASE(186, 186, 1, 0x00c0, 0x10, 31, 1),
	PIN_FIELD_BASE(187, 187, 1, 0x00d0, 0x10, 1, 1),
	PIN_FIELD_BASE(188, 188, 1, 0x00c0, 0x10, 30, 1),
	PIN_FIELD_BASE(189, 189, 1, 0x00d0, 0x10, 4, 1),
	PIN_FIELD_BASE(190, 190, 1, 0x00d0, 0x10, 3, 1),
	PIN_FIELD_BASE(191, 191, 1, 0x00d0, 0x10, 5, 1),
	PIN_FIELD_BASE(192, 192, 1, 0x00d0, 0x10, 2, 1),
	PIN_FIELD_BASE(193, 193, 1, 0x00d0, 0x10, 8, 1),
	PIN_FIELD_BASE(194, 194, 1, 0x00d0, 0x10, 7, 1),
	PIN_FIELD_BASE(195, 195, 1, 0x00d0, 0x10, 9, 1),
	PIN_FIELD_BASE(196, 196, 1, 0x00d0, 0x10, 6, 1),
	PIN_FIELD_BASE(197, 197, 1, 0x00c0, 0x10, 29, 1),
	PIN_FIELD_BASE(198, 198, 1, 0x00c0, 0x10, 28, 1),
	PIN_FIELD_BASE(199, 199, 1, 0x00d0, 0x10, 15, 1),
	PIN_FIELD_BASE(200, 200, 1, 0x00d0, 0x10, 10, 1),
	PIN_FIELD_BASE(201, 201, 1, 0x00d0, 0x10, 16, 1),
	PIN_FIELD_BASE(202, 205, 1, 0x00d0, 0x10, 11, 1),
	PIN_FIELD_BASE(206, 207, 1, 0x00c0, 0x10, 16, 1),
	PIN_FIELD_BASE(208, 208, 1, 0x00c0, 0x10, 22, 1),
	PIN_FIELD_BASE(209, 209, 7, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(210, 210, 7, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(211, 212, 7, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(213, 213, 7, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(214, 214, 7, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(215, 222, 7, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(223, 223, 7, 0x0050, 0x10, 13, 1),
	PIN_FIELD_BASE(224, 224, 7, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(225, 225, 7, 0x0050, 0x10, 24, 1),
	PIN_FIELD_BASE(226, 226, 7, 0x0050, 0x10, 23, 1),
	PIN_FIELD_BASE(227, 227, 7, 0x0050, 0x10, 25, 1),
	PIN_FIELD_BASE(228, 228, 7, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(229, 229, 7, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(230, 230, 7, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(231, 231, 7, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(232, 232, 7, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(233, 234, 7, 0x0050, 0x10, 1, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_pd_range[] = {
	PIN_FIELD_BASE(0, 3, 3, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(4, 7, 1, 0x0090, 0x10, 18, 1),
	PIN_FIELD_BASE(8, 9, 4, 0x00d0, 0x10, 29, 1),
	PIN_FIELD_BASE(10, 25, 4, 0x00d0, 0x10, 5, 1),
	PIN_FIELD_BASE(32, 39, 4, 0x00d0, 0x10, 21, 1),
	PIN_FIELD_BASE(40, 42, 6, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(61, 61, 3, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(62, 62, 3, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(63, 66, 3, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(67, 68, 3, 0x0070, 0x10, 13, 1),
	PIN_FIELD_BASE(69, 69, 3, 0x0070, 0x10, 12, 1),
	PIN_FIELD_BASE(70, 71, 3, 0x0070, 0x10, 16, 1),
	PIN_FIELD_BASE(72, 72, 3, 0x0070, 0x10, 15, 1),
	PIN_FIELD_BASE(73, 74, 3, 0x0070, 0x10, 19, 1),
	PIN_FIELD_BASE(75, 75, 3, 0x0070, 0x10, 18, 1),
	PIN_FIELD_BASE(76, 77, 3, 0x0070, 0x10, 22, 1),
	PIN_FIELD_BASE(78, 78, 3, 0x0070, 0x10, 21, 1),
	PIN_FIELD_BASE(79, 79, 3, 0x0070, 0x10, 26, 1),
	PIN_FIELD_BASE(80, 80, 3, 0x0070, 0x10, 24, 1),
	PIN_FIELD_BASE(81, 81, 3, 0x0070, 0x10, 27, 1),
	PIN_FIELD_BASE(82, 82, 3, 0x0070, 0x10, 25, 1),
	PIN_FIELD_BASE(83, 84, 3, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(91, 91, 4, 0x00e0, 0x10, 6, 1),
	PIN_FIELD_BASE(92, 92, 4, 0x00e0, 0x10, 3, 1),
	PIN_FIELD_BASE(93, 93, 4, 0x00e0, 0x10, 5, 1),
	PIN_FIELD_BASE(94, 94, 4, 0x00e0, 0x10, 4, 1),
	PIN_FIELD_BASE(95, 95, 4, 0x00e0, 0x10, 2, 1),
	PIN_FIELD_BASE(96, 97, 4, 0x00e0, 0x10, 7, 1),
	PIN_FIELD_BASE(98, 98, 4, 0x00e0, 0x10, 11, 1),
	PIN_FIELD_BASE(99, 100, 4, 0x00e0, 0x10, 9, 1),
	PIN_FIELD_BASE(101, 101, 4, 0x00e0, 0x10, 23, 1),
	PIN_FIELD_BASE(102, 102, 4, 0x00e0, 0x10, 22, 1),
	PIN_FIELD_BASE(103, 103, 4, 0x00e0, 0x10, 24, 1),
	PIN_FIELD_BASE(104, 104, 4, 0x00e0, 0x10, 21, 1),
	PIN_FIELD_BASE(105, 105, 4, 0x00e0, 0x10, 18, 1),
	PIN_FIELD_BASE(106, 106, 4, 0x00e0, 0x10, 15, 1),
	PIN_FIELD_BASE(107, 107, 4, 0x00e0, 0x10, 19, 1),
	PIN_FIELD_BASE(108, 108, 4, 0x00e0, 0x10, 16, 1),
	PIN_FIELD_BASE(109, 109, 4, 0x00e0, 0x10, 20, 1),
	PIN_FIELD_BASE(110, 110, 4, 0x00e0, 0x10, 17, 1),
	PIN_FIELD_BASE(117, 117, 4, 0x00e0, 0x10, 14, 1),
	PIN_FIELD_BASE(118, 118, 4, 0x00f0, 0x10, 1, 1),
	PIN_FIELD_BASE(119, 120, 4, 0x00e0, 0x10, 29, 1),
	PIN_FIELD_BASE(121, 124, 4, 0x00e0, 0x10, 25, 1),
	PIN_FIELD_BASE(125, 125, 4, 0x00d0, 0x10, 31, 1),
	PIN_FIELD_BASE(126, 126, 4, 0x00e0, 0x10, 0, 1),
	PIN_FIELD_BASE(127, 128, 4, 0x00d0, 0x10, 3, 1),
	PIN_FIELD_BASE(129, 129, 4, 0x00d0, 0x10, 2, 1),
	PIN_FIELD_BASE(130, 130, 4, 0x00e0, 0x10, 1, 1),
	PIN_FIELD_BASE(131, 132, 4, 0x00d0, 0x10, 0, 1),
	PIN_FIELD_BASE(133, 133, 4, 0x00f0, 0x10, 2, 1),
	PIN_FIELD_BASE(134, 134, 4, 0x00e0, 0x10, 13, 1),
	PIN_FIELD_BASE(135, 135, 4, 0x00e0, 0x10, 12, 1),
	PIN_FIELD_BASE(136, 136, 5, 0x0020, 0x10, 12, 1),
	PIN_FIELD_BASE(137, 137, 5, 0x0020, 0x10, 11, 1),
	PIN_FIELD_BASE(138, 139, 5, 0x0020, 0x10, 15, 1),
	PIN_FIELD_BASE(140, 141, 5, 0x0020, 0x10, 13, 1),
	PIN_FIELD_BASE(142, 143, 5, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(144, 151, 5, 0x0020, 0x10, 3, 1),
	PIN_FIELD_BASE(152, 152, 5, 0x0020, 0x10, 2, 1),
	PIN_FIELD_BASE(153, 163, 2, 0x0020, 0x10, 0, 1),
	PIN_FIELD_BASE(164, 167, 1, 0x0090, 0x10, 24, 1),
	PIN_FIELD_BASE(168, 168, 1, 0x0090, 0x10, 23, 1),
	PIN_FIELD_BASE(169, 169, 1, 0x0090, 0x10, 0, 1),
	PIN_FIELD_BASE(170, 170, 1, 0x0090, 0x10, 4, 1),
	PIN_FIELD_BASE(171, 171, 1, 0x0090, 0x10, 12, 1),
	PIN_FIELD_BASE(172, 172, 1, 0x0090, 0x10, 8, 1),
	PIN_FIELD_BASE(173, 173, 1, 0x0090, 0x10, 1, 1),
	PIN_FIELD_BASE(174, 174, 1, 0x0090, 0x10, 5, 1),
	PIN_FIELD_BASE(175, 175, 1, 0x0090, 0x10, 13, 1),
	PIN_FIELD_BASE(176, 176, 1, 0x0090, 0x10, 9, 1),
	PIN_FIELD_BASE(177, 177, 1, 0x0090, 0x10, 2, 1),
	PIN_FIELD_BASE(178, 178, 1, 0x0090, 0x10, 6, 1),
	PIN_FIELD_BASE(179, 179, 1, 0x0090, 0x10, 14, 1),
	PIN_FIELD_BASE(180, 180, 1, 0x0090, 0x10, 10, 1),
	PIN_FIELD_BASE(181, 181, 1, 0x0090, 0x10, 3, 1),
	PIN_FIELD_BASE(182, 182, 1, 0x0090, 0x10, 7, 1),
	PIN_FIELD_BASE(183, 183, 1, 0x0090, 0x10, 15, 1),
	PIN_FIELD_BASE(184, 184, 1, 0x0090, 0x10, 11, 1),
	PIN_FIELD_BASE(185, 185, 1, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(186, 186, 1, 0x0090, 0x10, 31, 1),
	PIN_FIELD_BASE(187, 187, 1, 0x00a0, 0x10, 1, 1),
	PIN_FIELD_BASE(188, 188, 1, 0x0090, 0x10, 30, 1),
	PIN_FIELD_BASE(189, 189, 1, 0x00a0, 0x10, 4, 1),
	PIN_FIELD_BASE(190, 190, 1, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(191, 191, 1, 0x00a0, 0x10, 5, 1),
	PIN_FIELD_BASE(192, 192, 1, 0x00a0, 0x10, 2, 1),
	PIN_FIELD_BASE(193, 193, 1, 0x00a0, 0x10, 8, 1),
	PIN_FIELD_BASE(194, 194, 1, 0x00a0, 0x10, 7, 1),
	PIN_FIELD_BASE(195, 195, 1, 0x00a0, 0x10, 9, 1),
	PIN_FIELD_BASE(196, 196, 1, 0x00a0, 0x10, 6, 1),
	PIN_FIELD_BASE(197, 197, 1, 0x0090, 0x10, 29, 1),
	PIN_FIELD_BASE(198, 198, 1, 0x0090, 0x10, 28, 1),
	PIN_FIELD_BASE(199, 199, 1, 0x00a0, 0x10, 15, 1),
	PIN_FIELD_BASE(200, 200, 1, 0x00a0, 0x10, 10, 1),
	PIN_FIELD_BASE(201, 201, 1, 0x00a0, 0x10, 16, 1),
	PIN_FIELD_BASE(202, 205, 1, 0x00a0, 0x10, 11, 1),
	PIN_FIELD_BASE(206, 207, 1, 0x0090, 0x10, 16, 1),
	PIN_FIELD_BASE(208, 208, 1, 0x0090, 0x10, 22, 1),
	PIN_FIELD_BASE(209, 209, 7, 0x0040, 0x10, 9, 1),
	PIN_FIELD_BASE(210, 210, 7, 0x0040, 0x10, 7, 1),
	PIN_FIELD_BASE(211, 212, 7, 0x0040, 0x10, 5, 1),
	PIN_FIELD_BASE(213, 213, 7, 0x0040, 0x10, 10, 1),
	PIN_FIELD_BASE(214, 214, 7, 0x0040, 0x10, 8, 1),
	PIN_FIELD_BASE(215, 222, 7, 0x0040, 0x10, 15, 1),
	PIN_FIELD_BASE(223, 223, 7, 0x0040, 0x10, 13, 1),
	PIN_FIELD_BASE(224, 224, 7, 0x0040, 0x10, 12, 1),
	PIN_FIELD_BASE(225, 225, 7, 0x0040, 0x10, 24, 1),
	PIN_FIELD_BASE(226, 226, 7, 0x0040, 0x10, 23, 1),
	PIN_FIELD_BASE(227, 227, 7, 0x0040, 0x10, 25, 1),
	PIN_FIELD_BASE(228, 228, 7, 0x0040, 0x10, 11, 1),
	PIN_FIELD_BASE(229, 229, 7, 0x0040, 0x10, 14, 1),
	PIN_FIELD_BASE(230, 230, 7, 0x0040, 0x10, 4, 1),
	PIN_FIELD_BASE(231, 231, 7, 0x0040, 0x10, 0, 1),
	PIN_FIELD_BASE(232, 232, 7, 0x0040, 0x10, 3, 1),
	PIN_FIELD_BASE(233, 234, 7, 0x0040, 0x10, 1, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_drv_range[] = {
	PIN_FIELD_BASE(0, 3, 3, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(4, 7, 1, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(8, 8, 4, 0x0030, 0x10, 24, 3),
	PIN_FIELD_BASE(9, 9, 4, 0x0020, 0x10, 15, 3),
	PIN_FIELD_BASE(10, 16, 4, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(17, 25, 4, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(26, 30, 1, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(31, 31, 1, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(32, 32, 4, 0x0010, 0x10, 27, 3),
	PIN_FIELD_BASE(33, 37, 4, 0x0020, 0x10, 0, 3),
	PINS_FIELD_BASE(38, 39, 4, 0x0030, 0x10, 24, 3),
	PIN_FIELD_BASE(40, 41, 6, 0x0010, 0x10, 24, 3),
	PIN_FIELD_BASE(42, 42, 6, 0x0010, 0x10, 24, 3),
	PIN_FIELD_BASE(43, 43, 6, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(44, 44, 6, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(45, 45, 6, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(46, 46, 6, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(47, 47, 6, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(48, 48, 6, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(49, 49, 6, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(50, 50, 6, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(51, 51, 6, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(52, 52, 6, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(53, 53, 6, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(54, 54, 6, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(55, 56, 6, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(57, 57, 6, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(58, 58, 6, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(59, 59, 6, 0x0010, 0x10, 18, 3),
	PIN_FIELD_BASE(60, 60, 6, 0x0010, 0x10, 15, 3),
	PINS_FIELD_BASE(61, 64, 3, 0x0000, 0x10, 14, 3),
	PINS_FIELD_BASE(65, 68, 3, 0x0000, 0x10, 17, 3),
	PINS_FIELD_BASE(69, 72, 3, 0x0000, 0x10, 20, 3),
	PIN_FIELD_BASE(73, 74, 3, 0x0000, 0x10, 23, 3),
	PINS_FIELD_BASE(75, 76, 3, 0x0000, 0x10, 23, 3),
	PIN_FIELD_BASE(77, 77, 3, 0x0000, 0x10, 26, 3),
	PIN_FIELD_BASE(78, 78, 3, 0x0000, 0x10, 23, 3),
	PIN_FIELD_BASE(79, 79, 3, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(80, 80, 3, 0x0000, 0x10, 29, 3),
	PIN_FIELD_BASE(81, 81, 3, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(82, 82, 3, 0x0010, 0x10, 0, 3),
	PINS_FIELD_BASE(83, 84, 3, 0x0000, 0x10, 26, 3),
	PINS_FIELD_BASE(85, 90, 3, 0x0000, 0x10, 12, 2),
	PIN_FIELD_BASE(91, 91, 4, 0x0030, 0x10, 0, 3),
	PIN_FIELD_BASE(92, 92, 4, 0x0020, 0x10, 21, 3),
	PIN_FIELD_BASE(93, 93, 4, 0x0020, 0x10, 27, 3),
	PIN_FIELD_BASE(94, 94, 4, 0x0020, 0x10, 24, 3),
	PIN_FIELD_BASE(95, 95, 4, 0x0020, 0x10, 18, 3),
	PIN_FIELD_BASE(96, 97, 4, 0x0030, 0x10, 3, 3),
	PIN_FIELD_BASE(98, 98, 4, 0x0030, 0x10, 15, 3),
	PIN_FIELD_BASE(99, 100, 4, 0x0030, 0x10, 9, 3),
	PIN_FIELD_BASE(101, 101, 4, 0x0040, 0x10, 27, 3),
	PIN_FIELD_BASE(102, 102, 4, 0x0040, 0x10, 24, 3),
	PIN_FIELD_BASE(103, 103, 4, 0x0050, 0x10, 0, 3),
	PIN_FIELD_BASE(104, 104, 4, 0x0040, 0x10, 21, 3),
	PIN_FIELD_BASE(105, 105, 4, 0x0040, 0x10, 12, 3),
	PIN_FIELD_BASE(106, 106, 4, 0x0040, 0x10, 3, 3),
	PIN_FIELD_BASE(107, 107, 4, 0x0040, 0x10, 15, 3),
	PIN_FIELD_BASE(108, 108, 4, 0x0040, 0x10, 6, 3),
	PIN_FIELD_BASE(109, 109, 4, 0x0040, 0x10, 18, 3),
	PIN_FIELD_BASE(110, 110, 4, 0x0040, 0x10, 9, 3),
	PINS_FIELD_BASE(111, 114, 4, 0x0030, 0x10, 18, 3),
	PINS_FIELD_BASE(115, 116, 4, 0x0030, 0x10, 21, 3),
	PIN_FIELD_BASE(118, 118, 4, 0x0050, 0x10, 15, 3),
	PINS_FIELD_BASE(119, 120, 4, 0x0030, 0x10, 21, 3),
	PIN_FIELD_BASE(121, 124, 4, 0x0050, 0x10, 3, 3),
	PIN_FIELD_BASE(125, 126, 4, 0x0030, 0x10, 24, 3),
	PIN_FIELD_BASE(127, 128, 4, 0x0000, 0x10, 3, 3),
	PIN_FIELD_BASE(129, 129, 4, 0x0000, 0x10, 0, 3),
	PINS_FIELD_BASE(130, 132, 4, 0x0030, 0x10, 27, 3),
	PINS_FIELD_BASE(133, 135, 4, 0x0040, 0x10, 0, 3),
	PINS_FIELD_BASE(136, 137, 5, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(138, 139, 5, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(140, 141, 5, 0x0000, 0x10, 0, 3),
	PINS_FIELD_BASE(142, 143, 5, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(144, 147, 5, 0x0000, 0x10, 9, 3),
	PINS_FIELD_BASE(148, 151, 5, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(152, 152, 5, 0x0000, 0x10, 15, 3),
	PINS_FIELD_BASE(153, 154, 2, 0x0000, 0x10, 0, 3),
	PINS_FIELD_BASE(155, 156, 2, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(157, 160, 2, 0x0000, 0x10, 6, 3),
	PINS_FIELD_BASE(161, 163, 2, 0x0000, 0x10, 9, 3),
	PINS_FIELD_BASE(164, 165, 1, 0x0010, 0x10, 18, 3),
	PINS_FIELD_BASE(166, 167, 1, 0x0010, 0x10, 21, 3),
	PIN_FIELD_BASE(168, 168, 1, 0x0010, 0x10, 24, 3),
	PIN_FIELD_BASE(169, 169, 1, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(170, 170, 1, 0x0000, 0x10, 6, 3),
	PIN_FIELD_BASE(171, 171, 1, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(172, 172, 1, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(173, 173, 1, 0x0000, 0x10, 3, 3),
	PINS_FIELD_BASE(174, 176, 1, 0x0010, 0x10, 24, 3),
	PINS_FIELD_BASE(177, 180, 1, 0x0010, 0x10, 27, 3),
	PINS_FIELD_BASE(181, 184, 1, 0x0020, 0x10, 0, 3),
	PIN_FIELD_BASE(185, 185, 1, 0x0020, 0x10, 24, 3),
	PIN_FIELD_BASE(186, 186, 1, 0x0020, 0x10, 21, 3),
	PIN_FIELD_BASE(187, 187, 1, 0x0020, 0x10, 27, 3),
	PIN_FIELD_BASE(188, 188, 1, 0x0020, 0x10, 18, 3),
	PINS_FIELD_BASE(189, 190, 1, 0x0020, 0x10, 3, 3),
	PINS_FIELD_BASE(191, 193, 1, 0x0020, 0x10, 6, 3),
	PIN_FIELD_BASE(194, 195, 1, 0x0020, 0x10, 3, 3),
	PIN_FIELD_BASE(196, 196, 1, 0x0020, 0x10, 3, 3),
	PIN_FIELD_BASE(197, 197, 1, 0x0020, 0x10, 15, 3),
	PIN_FIELD_BASE(198, 198, 1, 0x0020, 0x10, 12, 3),
	PIN_FIELD_BASE(199, 199, 1, 0x0030, 0x10, 15, 3),
	PIN_FIELD_BASE(200, 200, 1, 0x0030, 0x10, 0, 3),
	PIN_FIELD_BASE(201, 201, 1, 0x0030, 0x10, 18, 3),
	PIN_FIELD_BASE(202, 205, 1, 0x0030, 0x10, 3, 3),
	PINS_FIELD_BASE(206, 207, 1, 0x0020, 0x10, 9, 3),
	PIN_FIELD_BASE(208, 208, 1, 0x0010, 0x10, 15, 3),
	PINS_FIELD_BASE(209, 210, 7, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(211, 211, 7, 0x0010, 0x10, 15, 3),
	PIN_FIELD_BASE(212, 213, 7, 0x0010, 0x10, 12, 3),
	PIN_FIELD_BASE(214, 214, 7, 0x0010, 0x10, 12, 3),
	PINS_FIELD_BASE(215, 216, 7, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(217, 217, 7, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(218, 219, 7, 0x0010, 0x10, 6, 3),
	PIN_FIELD_BASE(220, 220, 7, 0x0010, 0x10, 6, 3),
	PINS_FIELD_BASE(221, 222, 7, 0x0010, 0x10, 9, 3),
	PIN_FIELD_BASE(223, 223, 7, 0x0000, 0x10, 21, 3),
	PIN_FIELD_BASE(224, 224, 7, 0x0000, 0x10, 18, 3),
	PIN_FIELD_BASE(225, 225, 7, 0x0010, 0x10, 0, 3),
	PIN_FIELD_BASE(226, 226, 7, 0x0000, 0x10, 27, 3),
	PIN_FIELD_BASE(227, 227, 7, 0x0010, 0x10, 3, 3),
	PIN_FIELD_BASE(228, 228, 7, 0x0000, 0x10, 15, 3),
	PIN_FIELD_BASE(229, 229, 7, 0x0000, 0x10, 24, 3),
	PIN_FIELD_BASE(230, 230, 7, 0x0000, 0x10, 12, 3),
	PIN_FIELD_BASE(231, 231, 7, 0x0000, 0x10, 0, 3),
	PIN_FIELD_BASE(232, 232, 7, 0x0000, 0x10, 9, 3),
	PIN_FIELD_BASE(233, 234, 7, 0x0000, 0x10, 3, 3),
};

static const struct mtk_pin_field_calc mt6880_pin_pupd_range[] = {
	PIN_FIELD_BASE(26, 31, 1, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(43, 43, 6, 0x0050, 0x10, 9, 1),
	PIN_FIELD_BASE(44, 44, 6, 0x0050, 0x10, 8, 1),
	PIN_FIELD_BASE(45, 45, 6, 0x0050, 0x10, 7, 1),
	PIN_FIELD_BASE(46, 46, 6, 0x0050, 0x10, 6, 1),
	PIN_FIELD_BASE(47, 47, 6, 0x0050, 0x10, 11, 1),
	PIN_FIELD_BASE(48, 48, 6, 0x0050, 0x10, 1, 1),
	PIN_FIELD_BASE(49, 49, 6, 0x0050, 0x10, 0, 1),
	PIN_FIELD_BASE(50, 50, 6, 0x0050, 0x10, 5, 1),
	PIN_FIELD_BASE(51, 51, 6, 0x0050, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 6, 0x0050, 0x10, 3, 1),
	PIN_FIELD_BASE(53, 53, 6, 0x0050, 0x10, 2, 1),
	PIN_FIELD_BASE(54, 54, 6, 0x0050, 0x10, 10, 1),
	PIN_FIELD_BASE(55, 56, 6, 0x0050, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 6, 0x0050, 0x10, 17, 1),
	PIN_FIELD_BASE(58, 58, 6, 0x0050, 0x10, 14, 1),
	PIN_FIELD_BASE(59, 59, 6, 0x0050, 0x10, 16, 1),
	PIN_FIELD_BASE(60, 60, 6, 0x0050, 0x10, 15, 1),
	PIN_FIELD_BASE(85, 86, 3, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(111, 116, 4, 0x0100, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_r0_range[] = {
	PIN_FIELD_BASE(26, 31, 1, 0x00e0, 0x10, 0, 1),
	PIN_FIELD_BASE(43, 43, 6, 0x0070, 0x10, 9, 1),
	PIN_FIELD_BASE(44, 44, 6, 0x0070, 0x10, 8, 1),
	PIN_FIELD_BASE(45, 45, 6, 0x0070, 0x10, 7, 1),
	PIN_FIELD_BASE(46, 46, 6, 0x0070, 0x10, 6, 1),
	PIN_FIELD_BASE(47, 47, 6, 0x0070, 0x10, 11, 1),
	PIN_FIELD_BASE(48, 48, 6, 0x0070, 0x10, 1, 1),
	PIN_FIELD_BASE(49, 49, 6, 0x0070, 0x10, 0, 1),
	PIN_FIELD_BASE(50, 50, 6, 0x0070, 0x10, 5, 1),
	PIN_FIELD_BASE(51, 51, 6, 0x0070, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 6, 0x0070, 0x10, 3, 1),
	PIN_FIELD_BASE(53, 53, 6, 0x0070, 0x10, 2, 1),
	PIN_FIELD_BASE(54, 54, 6, 0x0070, 0x10, 10, 1),
	PIN_FIELD_BASE(55, 56, 6, 0x0070, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 6, 0x0070, 0x10, 17, 1),
	PIN_FIELD_BASE(58, 58, 6, 0x0070, 0x10, 14, 1),
	PIN_FIELD_BASE(59, 59, 6, 0x0070, 0x10, 16, 1),
	PIN_FIELD_BASE(60, 60, 6, 0x0070, 0x10, 15, 1),
	PIN_FIELD_BASE(85, 86, 3, 0x00a0, 0x10, 1, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x00a0, 0x10, 0, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x00a0, 0x10, 3, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x00a0, 0x10, 5, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x00a0, 0x10, 4, 1),
	PIN_FIELD_BASE(111, 116, 4, 0x0140, 0x10, 0, 1),
};

static const struct mtk_pin_field_calc mt6880_pin_r1_range[] = {
	PIN_FIELD_BASE(26, 31, 1, 0x00f0, 0x10, 0, 1),
	PIN_FIELD_BASE(43, 43, 6, 0x0080, 0x10, 9, 1),
	PIN_FIELD_BASE(44, 44, 6, 0x0080, 0x10, 8, 1),
	PIN_FIELD_BASE(45, 45, 6, 0x0080, 0x10, 7, 1),
	PIN_FIELD_BASE(46, 46, 6, 0x0080, 0x10, 6, 1),
	PIN_FIELD_BASE(47, 47, 6, 0x0080, 0x10, 11, 1),
	PIN_FIELD_BASE(48, 48, 6, 0x0080, 0x10, 1, 1),
	PIN_FIELD_BASE(49, 49, 6, 0x0080, 0x10, 0, 1),
	PIN_FIELD_BASE(50, 50, 6, 0x0080, 0x10, 5, 1),
	PIN_FIELD_BASE(51, 51, 6, 0x0080, 0x10, 4, 1),
	PIN_FIELD_BASE(52, 52, 6, 0x0080, 0x10, 3, 1),
	PIN_FIELD_BASE(53, 53, 6, 0x0080, 0x10, 2, 1),
	PIN_FIELD_BASE(54, 54, 6, 0x0080, 0x10, 10, 1),
	PIN_FIELD_BASE(55, 56, 6, 0x0080, 0x10, 12, 1),
	PIN_FIELD_BASE(57, 57, 6, 0x0080, 0x10, 17, 1),
	PIN_FIELD_BASE(58, 58, 6, 0x0080, 0x10, 14, 1),
	PIN_FIELD_BASE(59, 59, 6, 0x0080, 0x10, 16, 1),
	PIN_FIELD_BASE(60, 60, 6, 0x0080, 0x10, 15, 1),
	PIN_FIELD_BASE(85, 86, 3, 0x00b0, 0x10, 1, 1),
	PIN_FIELD_BASE(87, 87, 3, 0x00b0, 0x10, 0, 1),
	PIN_FIELD_BASE(88, 88, 3, 0x00b0, 0x10, 3, 1),
	PIN_FIELD_BASE(89, 89, 3, 0x00b0, 0x10, 5, 1),
	PIN_FIELD_BASE(90, 90, 3, 0x00b0, 0x10, 4, 1),
	PIN_FIELD_BASE(111, 116, 4, 0x0150, 0x10, 0, 1),
};

static const struct mtk_eint_hw mt6880_eint_hw = {
	.port_mask = 7,
	.ports     = 7,
	.ap_num    = 224,
	.db_cnt    = 32,
};

static const struct mtk_pin_reg_calc mt6880_reg_cals[PINCTRL_PIN_REG_MAX] = {
	[PINCTRL_PIN_REG_MODE] = MTK_RANGE(mt6880_pin_mode_range),
	[PINCTRL_PIN_REG_DIR] = MTK_RANGE(mt6880_pin_dir_range),
	[PINCTRL_PIN_REG_DI] = MTK_RANGE(mt6880_pin_di_range),
	[PINCTRL_PIN_REG_DO] = MTK_RANGE(mt6880_pin_do_range),
	[PINCTRL_PIN_REG_SMT] = MTK_RANGE(mt6880_pin_smt_range),
	[PINCTRL_PIN_REG_IES] = MTK_RANGE(mt6880_pin_ies_range),
	[PINCTRL_PIN_REG_PU] = MTK_RANGE(mt6880_pin_pu_range),
	[PINCTRL_PIN_REG_PD] = MTK_RANGE(mt6880_pin_pd_range),
	[PINCTRL_PIN_REG_DRV] = MTK_RANGE(mt6880_pin_drv_range),
	[PINCTRL_PIN_REG_PUPD] = MTK_RANGE(mt6880_pin_pupd_range),
	[PINCTRL_PIN_REG_R0] = MTK_RANGE(mt6880_pin_r0_range),
	[PINCTRL_PIN_REG_R1] = MTK_RANGE(mt6880_pin_r1_range),
};

static const char * const mt6880_pinctrl_register_base_names[] = {
	"gpio", "iocfg_rm", "iocfg_rb", "iocfg_bl", "iocfg_bm",
	"iocfg_br", "iocfg_lt", "iocfg_tl",
};

static const struct mtk_pin_soc mt6880_data = {
	.reg_cal = mt6880_reg_cals,
	.pins = mtk_pins_mt6880,
	.npins = ARRAY_SIZE(mtk_pins_mt6880),
	.ngrps = ARRAY_SIZE(mtk_pins_mt6880),
	.eint_hw = &mt6880_eint_hw,
	.gpio_m = 0,
	.ies_present = true,
	.base_names = mt6880_pinctrl_register_base_names,
	.nbase_names = ARRAY_SIZE(mt6880_pinctrl_register_base_names),
	.bias_set_combo = mtk_pinconf_bias_set_combo,
	.bias_get_combo = mtk_pinconf_bias_get_combo,
	.drive_set = mtk_pinconf_drive_set_raw,
	.drive_get = mtk_pinconf_drive_get_raw,
	.adv_pull_get = mtk_pinconf_adv_pull_get,
	.adv_pull_set = mtk_pinconf_adv_pull_set,
};

static const struct of_device_id mt6880_pinctrl_of_match[] = {
	{ .compatible = "mediatek,mt6880-pinctrl", },
	{ }
};

static int mt6880_pinctrl_probe(struct platform_device *pdev)
{
	return mtk_paris_pinctrl_probe(pdev, &mt6880_data);
}

static struct platform_driver mt6880_pinctrl_driver = {
	.driver = {
		.name = "mt6880-pinctrl",
		.of_match_table = mt6880_pinctrl_of_match,
		.pm = &mtk_paris_pinctrl_pm_ops,
	},
	.probe = mt6880_pinctrl_probe,
};

static int __init mt6880_pinctrl_init(void)
{
	return platform_driver_register(&mt6880_pinctrl_driver);
}
arch_initcall(mt6880_pinctrl_init);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("MediaTek MT6880 Pinctrl Driver");
