/*
 * BQ2589x battery charging driver
 *
 * Copyright (C) 2013 Texas Instruments
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.

 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <linux/gpio.h>
#include <linux/i2c.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/power_supply.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/kthread.h>
#include <linux/delay.h>
#include <linux/of_gpio.h>

#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#ifdef CONFIG_REGMAP
#include <linux/regmap.h>
#endif

/* DEBUG */
#if 0
#undef pr_debug
#define pr_debug pr_info
#endif

/* #define WD_DEBUG */
#ifdef WD_DEBUG
static int wd_timeout_dbg = 0;
module_param(wd_timeout_dbg, int, 0644);
#endif /* WD_DEBUG */


#include "bq25898s_reg.h"

#define BQ2589X_MANUFACTURER		"Texas Instruments"

enum bq2589x_part_no {
	BQ25898  = 0x00,
	BQ25898S = 0x01,
	BQ25898D = 0x02,
};

struct bq2589x_config {
	bool	enable_auto_dpdm;

	u32		charge_voltage;
	u32		charge_current;
	u32		precharge_current;

	u32		iindpm_threshold;
	u32		vindpm_threshold;

	bool	enable_term;
	u32		term_current;

	bool	use_absolute_vindpm;

	bool	enable_init;
};

#define BITORDER_REVERSE
#if !defined(BITORDER_REVERSE)
struct bq2589x_state{
	u8 vbus    : 3;
	u8 charge  : 2;
	bool pgood : 1;
	int        : 1;
	bool vsys  : 1;
};

struct bq2589x_fault{
	bool wdt_timeout : 1;
	bool boost : 1;
	u8 charge : 2;
	bool bat : 1;
	u8 ntc : 3;
};
#else
struct bq2589x_state{
	bool vsys  : 1;
	int        : 1;
	bool pgood : 1;
	u8 charge  : 2;
	u8 vbus    : 3;
};

struct bq2589x_fault{
	u8 ntc : 3;
	bool bat : 1;
	u8 charge : 2;
	bool boost : 1;
	bool wdt_timeout : 1;
};
#endif

struct bq2589x {
	struct device *dev;
	struct i2c_client *client;
	struct power_supply *charger_psy;
	int    psy_usb_type;
	enum   bq2589x_part_no part_no;
	int    revision;

	struct	bq2589x_config cfg;
	union{
		u8 byte;
		struct bq2589x_state data;
	} state;
	union{
		u8 byte;
		struct bq2589x_fault data;
	} fault;
	bool vbus_good;
	bool online;

	int		enable_gpio;
	enum of_gpio_flags gpio_flags;
	struct 	work_struct irq_work;
	struct 	delayed_work polling_work;
	int		polling_interval_sec;

	int		rsoc;
	int		rechg_thres;
	struct 	power_supply *batt_psy;
	struct regulator_dev *otg_rdev;
	
	int		ilim_det_result;
	int		ilim_from_tcpc;
#if defined(CONFIG_PLATFORM_NEC)
	struct mutex 	float_v_lock;
	int				real_usb_psy_ma;
	int				prev_usb_psy_dis_ma;
	int				prev_usb_psy_sav_ma;
	bool			gadget_status;
	struct  power_supply *usb_psy;
	bool			chage_stop;
	bool			chg_enabled;
	bool			notify_src;
	int				cradle_status;
	int				detect_interval_sec;
	int				monitor_interval_sec;
	int				current_charging_voltage_step;
#endif /* defined(CONFIG_PLATFORM_NEC) */
};

#if defined(CONFIG_PLATFORM_NEC)
#define BQ2589X_NEC_HW_INIT_REG_CHECK_VALUE	0xFC
struct init_reg_t {
	u8 reg;
	u8 val;
};
static struct init_reg_t hw_init_regs[] = {
	{0x00, 0x48},
	{0x01, 0x01},
	{0x02, 0x5d},
	{0x03, 0x1c},
	{0x04, 0x1e},
	{0x05, 0x10},
	{0x06, 0x5f},
	{0x07, 0x4c},
	{0x08, 0x03},
	{0x09, 0x0c},
	{0x0a, 0x74},
	{0x0d, 0x12},
};

static char charge_voltage_table[] = {
	0x5F, /* 4.208V */
	0x53, /* 4.16V */
	0x4B, /* 4.128V */
	0x3F, /* 4.08V */
	0x37, /* 4.048V */
	0x2B, /* 4.00V */
	0x23, /* 3.968V */
};

static char float_voltage = 0x5F;

#define CURRENT_500_MA		500
#define CURRENT_1000_MA		1000
#define CURRENT_1100_MA		1100
#define CURRENT_1400_MA		1400
#define CURRENT_1800_MA		1800

static int bq2589x_set_charging_voltage(struct bq2589x *bq, int step);
static int bq2589x_notify_usb_src_change(struct bq2589x *bq,int src_status);
static int bq2589x_get_prop_cradle_status(struct bq2589x *bq);
static int bq2589x_nec_hw_init_regs(struct bq2589x *bq, int recovery);
static int bq2589x_get_prop_charge_status(struct bq2589x *bq);
static int bq2589x_get_prop_usb_type_check(struct bq2589x *bq);
static int bq2589x_get_prop_machine_status(struct bq2589x *bq);
static int bq2589x_adc_read_temperature(struct bq2589x *bq);
static int bq2589x_set_watchdog_timer_value(struct bq2589x *bq);
static int bq2589x_enter_ship_mode(struct bq2589x *bq);
#endif /* defined(CONFIG_PLATFORM_NEC) */


#ifdef CONFIG_REGMAP
static const struct regmap_config bq2589x_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = BQ25898S_REG_14,
	.cache_type = REGCACHE_NONE,
};
#endif /* CONFIG_REGMAP */

static DEFINE_MUTEX(bq2589x_i2c_lock);
static DEFINE_MUTEX(bq2589x_state_lock);

static int bq2589x_read_byte(struct bq2589x *bq, u8 reg, u8 *data)
{
	int ret;

	mutex_lock(&bq2589x_i2c_lock);
	ret = i2c_smbus_read_byte_data(bq->client, reg);
	if (ret < 0) {
		dev_err(bq->dev, "failed to read 0x%.2x\n", reg);
		mutex_unlock(&bq2589x_i2c_lock);
		return ret;
	}

	*data = (u8)ret;
	mutex_unlock(&bq2589x_i2c_lock);

	return 0;
}

static int bq2589x_read_byte_mask(struct bq2589x *bq, u8 *data, u8 reg, u8 mask, u8 shift)
{
	int ret;
	u8 val;

	if (shift > 8)
		return -EINVAL;

	ret = bq2589x_read_byte(bq, reg, &val);
	if (ret < 0)
		return ret;

	*data = (val & mask) >> shift;

	return 0;
}

static int bq2589x_write_byte(struct bq2589x *bq, u8 reg, u8 data)
{
	int ret;
	mutex_lock(&bq2589x_i2c_lock);
	ret = i2c_smbus_write_byte_data(bq->client, reg, data);
	mutex_unlock(&bq2589x_i2c_lock);
	return ret;
}

static int bq2589x_write_byte_mask(struct bq2589x *bq, u8 reg, u8 val, u8 mask, u8 shift)
{
	int ret;
	u8 data;

	if (shift > 8)
		return -EINVAL;

	ret = bq2589x_read_byte(bq, reg, &data);
	if (ret < 0)
		return ret;

	data &= ~mask;
	data |= val << shift;

	return bq2589x_write_byte(bq, reg, data);
}

static int bq2589x_detect_device(struct bq2589x *bq)
{
	int ret;
	u8 data;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_14, &data);
	if (ret == 0) {
		bq->part_no = (data & BQ25898S_PN_MASK) >> BQ25898S_PN_SHIFT;
		bq->revision = (data & BQ25898S_DEV_REV_MASK) >> BQ25898S_DEV_REV_SHIFT;
	}

	return ret;
}

static int bq2589x_enable_charger(struct bq2589x *bq, bool enable)
{
#if defined(CONFIG_PLATFORM_NEC)
	pr_debug("%s enable=%d chage_stop=%d\n",__func__,enable,bq->chage_stop);
	if( bq->chage_stop == 1 ) {
		/* chg disable */
	    return bq2589x_write_byte_mask(bq, BQ25898S_REG_03,
			BQ25898S_CHG_DISABLE,
			BQ25898S_CHG_CONFIG_MASK, BQ25898S_CHG_CONFIG_SHIFT);
	} else {
		return bq2589x_write_byte_mask(bq, BQ25898S_REG_03,
			enable ? BQ25898S_CHG_ENABLE: BQ25898S_CHG_DISABLE,
			BQ25898S_CHG_CONFIG_MASK, BQ25898S_CHG_CONFIG_SHIFT);
	}
#else /* defined(CONFIG_PLATFORM_NEC) */
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_03,
		enable ? BQ25898S_CHG_ENABLE: BQ25898S_CHG_DISABLE,
		BQ25898S_CHG_CONFIG_MASK, BQ25898S_CHG_CONFIG_SHIFT);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int bq2589x_enable_term(struct bq2589x* bq, bool enable)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_07,
		enable ? BQ25898S_TERM_ENABLE : BQ25898S_TERM_DISABLE,
		BQ25898S_EN_TERM_MASK, BQ25898S_EN_TERM_SHIFT);
}

int bq2589x_adc_start(struct bq2589x *bq, bool oneshot)
{
	u8 val;
	int ret;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_02, &val);
	if (ret < 0) {
		dev_err(bq->dev, "%s failed to read register 0x02:%d\n", __func__, ret);
		return ret;
	}

	if (((val & BQ25898S_CONV_RATE_MASK) >> BQ25898S_CONV_RATE_SHIFT) == BQ25898S_ADC_CONTINUE_ENABLE)
		return 0; /*is doing continuous scan*/
	if (oneshot)
		ret = bq2589x_write_byte_mask(bq, BQ25898S_REG_02, BQ25898S_CONV_START, BQ25898S_CONV_START_MASK, BQ25898S_CONV_START_SHIFT);
	else
		ret = bq2589x_write_byte_mask(bq, BQ25898S_REG_02, BQ25898S_ADC_CONTINUE_ENABLE, BQ25898S_CONV_RATE_MASK, BQ25898S_CONV_RATE_SHIFT);
	return ret;
}

int bq2589x_adc_stop(struct bq2589x *bq)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_02,
		BQ25898S_ADC_CONTINUE_DISABLE,
		BQ25898S_CONV_RATE_MASK, BQ25898S_CONV_RATE_SHIFT);
}


int bq2589x_adc_read_battery_volt(struct bq2589x *bq)
{
	uint8_t val;
	int volt;
	int ret;
	ret = bq2589x_read_byte(bq, BQ25898S_REG_0E, &val);
	if (ret < 0) {
		dev_err(bq->dev, "read battery voltage failed :%d\n", ret);
		return ret;
	} else{
		volt = BQ25898S_BATV_BASE + ((val & BQ25898S_BATV_MASK) >> BQ25898S_BATV_SHIFT) * BQ25898S_BATV_LSB ;
		return volt;
	}
}


int bq2589x_adc_read_sys_volt(struct bq2589x *bq)
{
	uint8_t val;
	int volt;
	int ret;
	ret = bq2589x_read_byte(bq, BQ25898S_REG_0F, &val);
	if (ret < 0) {
		dev_err(bq->dev, "read system voltage failed :%d\n", ret);
		return ret;
	} else{
		volt = BQ25898S_SYSV_BASE + ((val & BQ25898S_SYSV_MASK) >> BQ25898S_SYSV_SHIFT) * BQ25898S_SYSV_LSB ;
		return volt;
	}
}

int bq2589x_adc_read_vbus_volt(struct bq2589x *bq)
{
	uint8_t val;
	int volt;
	int ret;
	ret = bq2589x_read_byte(bq, BQ25898S_REG_11, &val);
	if (ret < 0) {
		dev_err(bq->dev, "read vbus voltage failed :%d\n", ret);
		return ret;
	} else{
		volt = BQ25898S_VBUSV_BASE + ((val & BQ25898S_VBUSV_MASK) >> BQ25898S_VBUSV_SHIFT) * BQ25898S_VBUSV_LSB ;
		return volt;
	}
}

int bq2589x_adc_read_charge_current(struct bq2589x *bq)
{
	uint8_t val;
	int volt;
	int ret;
	ret = bq2589x_read_byte(bq, BQ25898S_REG_12, &val);
	if (ret < 0) {
		dev_err(bq->dev, "read charge current failed :%d\n", ret);
		return ret;
	} else{
		volt = (int)(BQ25898S_ICHGR_BASE + ((val & BQ25898S_ICHGR_MASK) >> BQ25898S_ICHGR_SHIFT) * BQ25898S_ICHGR_LSB) ;
		return volt;
	}
}

int bq2589x_set_charge_current(struct bq2589x *bq, int curr)
{
	u8 ichg;

	ichg = (curr - BQ25898S_ICHG_BASE)/BQ25898S_ICHG_LSB;
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_04, ichg, BQ25898S_ICHG_MASK, BQ25898S_ICHG_SHIFT);

}

int bq2589x_set_term_current(struct bq2589x *bq, int curr)
{
	u8 iterm;

	iterm = (curr - BQ25898S_ITERM_BASE) / BQ25898S_ITERM_LSB;

	return bq2589x_write_byte_mask(bq, BQ25898S_REG_05, iterm, BQ25898S_ITERM_MASK, BQ25898S_ITERM_SHIFT);
}


int bq2589x_set_prechg_current(struct bq2589x *bq, int curr)
{
	u8 iprechg;

	iprechg = (curr - BQ25898S_IPRECHG_BASE) / BQ25898S_IPRECHG_LSB;

	return bq2589x_write_byte_mask(bq, BQ25898S_REG_05, iprechg, BQ25898S_IPRECHG_MASK, BQ25898S_IPRECHG_SHIFT);
}

int bq2589x_set_charge_voltage(struct bq2589x *bq, int volt)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_06,
		(volt - BQ25898S_VREG_BASE)/BQ25898S_VREG_LSB,
		BQ25898S_VREG_MASK, BQ25898S_VREG_SHIFT);
}

int bq2589x_set_input_volt_limit(struct bq2589x *bq, int volt)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_0D,
		(volt - BQ25898S_VINDPM_BASE) / BQ25898S_VINDPM_LSB,
		BQ25898S_VINDPM_MASK, BQ25898S_VINDPM_SHIFT);
}

int bq2589x_set_input_current_limit(struct bq2589x *bq, int curr)
{
	u8 val;

	val = (curr - BQ25898S_IINLIM_BASE) / BQ25898S_IINLIM_LSB;
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_00,
		val,
		BQ25898S_IINLIM_MASK, BQ25898S_IINLIM_SHIFT);
}

int bq2589x_set_vindpm_offset(struct bq2589x *bq, int offset)
{
	u8 val;

	if (offset == 400)
		val = BQ25898S_VINDPMOS_400MV;
	else
		val = BQ25898S_VINDPMOS_600MV;

	return bq2589x_write_byte_mask(bq, BQ25898S_REG_01,
		val,
		BQ25898S_VINDPMOS_MASK, BQ25898S_VINDPMOS_SHIFT);

}

int bq2589x_set_watchdog_timer(struct bq2589x *bq, u8 timeout)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_07,
		(u8)((timeout - BQ25898S_WDT_BASE) / BQ25898S_WDT_LSB),
		BQ25898S_WDT_MASK, BQ25898S_WDT_SHIFT);
}

int bq2589x_disable_watchdog_timer(struct bq2589x *bq)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_07,
		BQ25898S_WDT_DISABLE,
		BQ25898S_WDT_MASK, BQ25898S_WDT_SHIFT);
}

int bq2589x_reset_watchdog_timer(struct bq2589x *bq)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_03,
		BQ25898S_WDT_RESET,
		BQ25898S_WDT_RESET_MASK, BQ25898S_WDT_RESET_SHIFT);
}

int bq2589x_reset_chip(struct bq2589x *bq)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_14,
		BQ25898S_RESET,
		BQ25898S_RESET_MASK, BQ25898S_RESET_SHIFT);
}

int bq2589x_set_hiz_mode(struct bq2589x *bq, int enable)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_00,
		enable ? BQ25898S_HIZ_ENABLE : BQ25898S_HIZ_DISABLE,
		BQ25898S_ENHIZ_MASK, BQ25898S_ENHIZ_SHIFT);
}

int bq2589x_get_hiz_mode(struct bq2589x *bq, u8 *state)
{
	int ret;

	ret = bq2589x_read_byte_mask(bq, state, BQ25898S_REG_00, BQ25898S_ENHIZ_MASK, BQ25898S_ENHIZ_SHIFT);
	if (ret)
		return ret;

	return 0;
}

static int bq2589x_enable_auto_dpdm(struct bq2589x* bq, bool enable)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_02,
		enable ? BQ25898S_AUTO_DPDM_ENABLE : BQ25898S_AUTO_DPDM_DISABLE,
		BQ25898S_AUTO_DPDM_EN_MASK, BQ25898S_AUTO_DPDM_EN_SHIFT);
}

static int bq2589x_set_absolute_vindpm(struct bq2589x* bq, bool enable)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_0D,
		enable ? BQ25898S_FORCE_VINDPM_ENABLE : BQ25898S_FORCE_VINDPM_DISABLE,
		BQ25898S_FORCE_VINDPM_MASK, BQ25898S_FORCE_VINDPM_SHIFT);
}

static int bq2589x_set_en12v(struct bq2589x* bq, bool enable)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_01,
		enable ? BQ25898S_EN12V_ENABLE : BQ25898S_EN12V_DISABLE,
		BQ25898S_EN12V_MASK, BQ25898S_EN12V_SHIFT);
}

#define bq2589x_get_value_func(_name, _reg, _mask, _shift, _lsb, _base) \
	static int bq2589x_get_##_name(struct bq2589x *bq)                  \
	{                                                                      \
		int ret;                                                           \
		u8 data;                                                           \
		ret = bq2589x_read_byte_mask(bq, &data, _reg, _mask, _shift);      \
		return (ret < 0)? 0 : (data * _lsb) + _base;                       \
	}

bq2589x_get_value_func(input_current_limit, BQ25898S_REG_00, BQ25898S_IINLIM_MASK,  BQ25898S_IINLIM_SHIFT,  BQ25898S_IINLIM_LSB,  BQ25898S_IINLIM_BASE);
bq2589x_get_value_func(mimimum_sysV_limit,  BQ25898S_REG_03, BQ25898S_SYS_MIN_MASK, BQ25898S_SYS_MIN_SHIFT, BQ25898S_SYS_MIN_LSB, BQ25898S_SYS_MIN_BASE);
bq2589x_get_value_func(charge_current,      BQ25898S_REG_04, BQ25898S_ICHG_MASK,    BQ25898S_ICHG_SHIFT,    BQ25898S_ICHG_LSB,    BQ25898S_ICHG_BASE);
bq2589x_get_value_func(precharge_current,   BQ25898S_REG_05, BQ25898S_IPRECHG_MASK, BQ25898S_IPRECHG_SHIFT, BQ25898S_IPRECHG_LSB, BQ25898S_IPRECHG_BASE);
bq2589x_get_value_func(termination_current, BQ25898S_REG_05, BQ25898S_ITERM_MASK,   BQ25898S_ITERM_SHIFT,   BQ25898S_ITERM_LSB,   BQ25898S_ITERM_BASE);
bq2589x_get_value_func(charge_voltage,      BQ25898S_REG_06, BQ25898S_VREG_MASK,    BQ25898S_VREG_SHIFT,    BQ25898S_VREG_LSB,    BQ25898S_VREG_BASE);
bq2589x_get_value_func(watchdog_timer,      BQ25898S_REG_07, BQ25898S_WDT_MASK,     BQ25898S_WDT_SHIFT,     BQ25898S_WDT_LSB,     BQ25898S_WDT_BASE);
bq2589x_get_value_func(VINDPM_threshold,    BQ25898S_REG_0D, BQ25898S_VINDPM_MASK,  BQ25898S_VINDPM_SHIFT,  BQ25898S_VINDPM_LSB,  BQ25898S_VINDPM_BASE);
bq2589x_get_value_func(adc_batt_voltage,    BQ25898S_REG_0E, BQ25898S_BATV_MASK,    BQ25898S_BATV_SHIFT,    BQ25898S_BATV_LSB,    BQ25898S_BATV_BASE);
bq2589x_get_value_func(adc_sys_voltage,     BQ25898S_REG_0F, BQ25898S_SYSV_MASK,    BQ25898S_SYSV_SHIFT,    BQ25898S_SYSV_LSB,    BQ25898S_SYSV_BASE);
bq2589x_get_value_func(adc_vbus_voltage,    BQ25898S_REG_11, BQ25898S_VBUSV_MASK,   BQ25898S_VBUSV_SHIFT,   BQ25898S_VBUSV_LSB,   BQ25898S_VBUSV_BASE);
bq2589x_get_value_func(adc_charge_current,  BQ25898S_REG_12, BQ25898S_ICHGR_MASK,   BQ25898S_ICHGR_SHIFT,   BQ25898S_ICHGR_LSB,   BQ25898S_ICHGR_BASE);
bq2589x_get_value_func(IDPM_limit,          BQ25898S_REG_13, BQ25898S_IDPM_LIM_MASK,BQ25898S_IDPM_LIM_SHIFT,BQ25898S_IDPM_LIM_LSB,BQ25898S_IDPM_LIM_BASE);


static ssize_t bq2589x_fs_show_registers(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct bq2589x *bq = dev_get_drvdata(dev);
	u8 addr;
	u8 val;
	u8 tmpbuf[300];
	int len;
	int idx = 0;
	int ret ;

	idx = snprintf(buf, PAGE_SIZE, "%s:\n", "Charger");
	for (addr = 0x0; addr <= 0x14; addr++) {
		ret = bq2589x_read_byte(bq, addr, &val);
		if (ret == 0) {
			len = snprintf(tmpbuf, PAGE_SIZE - idx,"Reg[0x%.2x] = 0x%.2x\n", addr, val);
			memcpy(&buf[idx], tmpbuf, len);
			idx += len;
		}
	}

	return idx;
}

static ssize_t bq2589x_fs_get_enable(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;
	struct bq2589x *bq = dev_get_drvdata(dev);

	ret = gpio_get_value(bq->enable_gpio);

	return sprintf(buf, "%d\n", !ret);	//CE = active low
}

static ssize_t bq2589x_fs_set_enable(struct device *dev,
				struct device_attribute *attr, const char *buf,
				size_t count)
{
	struct bq2589x *bq = dev_get_drvdata(dev);
	long val;

	if (kstrtol(buf, 10, &val) < 0)
		return -EINVAL;

	gpio_set_value(bq->enable_gpio, !val);	//CE = active low
	return count;
}

static ssize_t bq2589x_fs_set_reset(struct device *dev,
				struct device_attribute *attr, const char *buf,
				size_t count)
{
	struct bq2589x *bq = dev_get_drvdata(dev);
	long val;

	if (kstrtol(buf, 10, &val) < 0)
		return -EINVAL;

	if(val)
		bq2589x_reset_chip(bq);
	return count;
}

/* set current and voltage limit entries (in mA or mV) */
static ssize_t bq2589x_fs_set_limit(struct device *dev,
				       struct device_attribute *attr,
				       const char *buf,
				       size_t count)
{
	struct bq2589x *bq = dev_get_drvdata(dev);
	long val;
	int ret;

	if (kstrtol(buf, 10, &val) < 0)
		return -EINVAL;

	if (strcmp(attr->attr.name, "current_limit") == 0)
		ret = bq2589x_set_input_current_limit(bq, val);
	else if (strcmp(attr->attr.name, "charge_voltage") == 0)
		ret = bq2589x_set_charge_voltage(bq, val);
	else if (strcmp(attr->attr.name, "charge_current") == 0)
		ret = bq2589x_set_charge_current(bq, val);
	else if (strcmp(attr->attr.name, "termination_current") == 0)
		ret = bq2589x_set_term_current(bq, val);
	else
		return -EINVAL;

	if (ret < 0)
		return ret;
	return count;
}

/* show current and voltage limit entries (in mA or mV) */
static ssize_t bq2589x_fs_show_limit(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct bq2589x *bq = dev_get_drvdata(dev);
	int ret;

	if (strcmp(attr->attr.name, "current_limit") == 0)
		ret = bq2589x_get_input_current_limit(bq);
	else if (strcmp(attr->attr.name, "min_systemV") == 0)
		ret = bq2589x_get_mimimum_sysV_limit(bq);
	else if (strcmp(attr->attr.name, "charge_voltage") == 0)
		ret = bq2589x_get_charge_voltage(bq);
	else if (strcmp(attr->attr.name, "charge_current") == 0)
		ret = bq2589x_get_charge_current(bq);
	else if (strcmp(attr->attr.name, "precharge_current") == 0)
		ret = bq2589x_get_precharge_current(bq);
	else if (strcmp(attr->attr.name, "termination_current") == 0)
		ret = bq2589x_get_termination_current(bq);
	else if (strcmp(attr->attr.name, "watchdog_timer") == 0)
		ret = bq2589x_get_watchdog_timer(bq);
	else if (strcmp(attr->attr.name, "vindpm_threshold") == 0)
		ret = bq2589x_get_VINDPM_threshold(bq);
	else if (strcmp(attr->attr.name, "adc_batt_voltage") == 0)
		ret = bq2589x_get_adc_batt_voltage(bq);
	else if (strcmp(attr->attr.name, "adc_sys_voltage") == 0)
		ret = bq2589x_get_adc_sys_voltage(bq);
	else if (strcmp(attr->attr.name, "adc_vbus_voltage") == 0)
		ret = bq2589x_get_adc_vbus_voltage(bq);
	else if (strcmp(attr->attr.name, "adc_charge_current") == 0)
		ret = bq2589x_get_adc_charge_current(bq);
	else if (strcmp(attr->attr.name, "idpm_limit") == 0)
		ret = bq2589x_get_IDPM_limit(bq);
	else
		return -EINVAL;

	if (ret < 0)
		return ret;
	return sprintf(buf, "%d\n", ret);
}

static DEVICE_ATTR(registers, S_IRUGO, bq2589x_fs_show_registers, NULL);
static DEVICE_ATTR(enable, S_IRUGO | S_IWUSR, bq2589x_fs_get_enable, bq2589x_fs_set_enable);
static DEVICE_ATTR(reset, S_IWUSR, NULL, bq2589x_fs_set_reset);

static DEVICE_ATTR(current_limit, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(charge_voltage, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(charge_current, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(precharge_current, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(termination_current, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(watchdog_timer, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(vindpm_threshold, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(adc_batt_voltage, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(adc_sys_voltage, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(adc_vbus_voltage, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(adc_charge_current, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);
static DEVICE_ATTR(idpm_limit, S_IRUGO | S_IWUSR, bq2589x_fs_show_limit, bq2589x_fs_set_limit);

static struct attribute *bq2589x_attributes[] = {
	&dev_attr_registers.attr,
	&dev_attr_enable.attr,
	&dev_attr_reset.attr,

	&dev_attr_current_limit.attr,
	&dev_attr_charge_voltage.attr,
	&dev_attr_charge_current.attr,
	&dev_attr_precharge_current.attr,
	&dev_attr_termination_current.attr,
	&dev_attr_watchdog_timer.attr,
	&dev_attr_vindpm_threshold.attr,
	&dev_attr_adc_batt_voltage.attr,
	&dev_attr_adc_sys_voltage.attr,
	&dev_attr_adc_vbus_voltage.attr,
	&dev_attr_adc_charge_current.attr,
	&dev_attr_idpm_limit.attr,

	NULL,
};

static const struct attribute_group bq2589x_attr_group = {
	.attrs = bq2589x_attributes,
};


static int bq2589x_parse_dt(struct device *dev, struct bq2589x *bq)
{
	int ret;
	struct device_node *np = dev->of_node;

	bq->enable_gpio = of_get_named_gpio_flags(np, "ti,bq2589x,enable-gpio", 0, &bq->gpio_flags);
	if (!gpio_is_valid(bq->enable_gpio)) {
		dev_dbg(bq->dev, "%s:enable-gpio define is wrong:%d\n", __func__, bq->enable_gpio);
	}
	bq->cfg.enable_init = of_property_read_bool(np, "ti,bq2589x,enable-gpio-init");

	ret = of_property_read_u32(np, "ti,bq2589x,monitor-interval",&bq->polling_interval_sec);
	if (ret)
		bq->polling_interval_sec = 10;

	bq->cfg.enable_auto_dpdm = of_property_read_bool(np, "ti,bq2589x,enable-auto-dpdm");
	bq->cfg.enable_term = of_property_read_bool(np, "ti,bq2589x,enable-termination");
	bq->cfg.use_absolute_vindpm = of_property_read_bool(np, "ti,bq2589x,use-absolute-vindpm");

	ret = of_property_read_u32(np, "ti,bq2589x,charge-voltage",&bq->cfg.charge_voltage);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,charge-current",&bq->cfg.charge_current);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,precharge-current",&bq->cfg.precharge_current);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,term-current",&bq->cfg.term_current);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,input-current-limit",&bq->cfg.iindpm_threshold);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,input-voltage-limit",&bq->cfg.vindpm_threshold);
	if (ret)
		goto err;

	ret = of_property_read_u32(np, "ti,bq2589x,recharge-soc-threshold",&bq->rechg_thres);
	if (ret)
		bq->rechg_thres = 95;

	return 0;

err:
	dev_info(bq->dev, "%s: Device tree parse error:%d\n", __func__, ret);
	return ret;
}

#define POWER_SUPPLY_BATTERY_NAME	"bq27421-battery"

static int bq2589x_read_batt_rsoc(struct bq2589x *bq)
{
	union power_supply_propval ret = {0,};

#if defined(CONFIG_PLATFORM_NEC)
	if (!bq->batt_psy)
		bq->batt_psy = power_supply_get_by_name(POWER_SUPPLY_BATTERY_NAME);
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (!bq->batt_psy)
		bq->batt_psy = power_supply_get_by_name("battery");
#endif /* !defined(CONFIG_PLATFORM_NEC) */

	if (bq->batt_psy) {
		if (bq->batt_psy->desc) {
			bq->batt_psy->desc->get_property(bq->batt_psy,POWER_SUPPLY_PROP_CAPACITY,&ret);
			return ret.intval;
		}
	}

	return 50;
}

#if !defined(CONFIG_PLATFORM_NEC)
static int bq2589x_adjust_absolute_vindpm(struct bq2589x *bq)
{
	u16 vbus_volt;
	u16 vindpm_volt;
	int ret;

	vbus_volt = bq2589x_adc_read_vbus_volt(bq);
	if (vbus_volt < 0){
		dev_err(bq->dev, "%s:Failed to read vbus voltage:%d\n", __func__, ret);
		return ret;
	}
	if (vbus_volt < 6000)
		vindpm_volt = vbus_volt - 600;
	else
		vindpm_volt = vbus_volt - 1200;

	/* BQ25898S_REG_0D Write */
	ret = bq2589x_set_input_volt_limit(bq, vindpm_volt);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Set absolute vindpm threshold %d Failed:%d\n", __func__, vindpm_volt, ret);
		return ret;
	}
	else
		dev_info(bq->dev, "%s:Set absolute vindpm threshold %d successfully\n", __func__, vindpm_volt);
	return 0;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

int bq2589x_set_charge_profile(struct bq2589x *bq)
{
	int ret;

#if defined(CONFIG_PLATFORM_NEC)
	/* BQ25898S_REG_00 Write */
	ret = bq2589x_set_input_current_limit(bq, bq->real_usb_psy_ma);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set input current limit:%d\n", __func__, ret);
		return ret;
	}

	return ret;
#else /* defined(CONFIG_PLATFORM_NEC) */
	/* BQ25898S_REG_06 Write */
	ret = bq2589x_set_charge_voltage(bq, bq->cfg.charge_voltage);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set charge voltage:%d\n", __func__, ret);
		return ret;
	}

	/* BQ25898S_REG_04 Write */
	ret = bq2589x_set_charge_current(bq, bq->cfg.charge_current);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set charge current:%d\n", __func__, ret);
		return ret;
	}

	/* BQ25898S_REG_05 Write */
	ret = bq2589x_set_prechg_current(bq, bq->cfg.precharge_current);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set precharge current:%d\n", __func__, ret);
		return ret;
	}

	/* BQ25898S_REG_05 Write */
	ret = bq2589x_set_term_current(bq, bq->cfg.term_current);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set termination current:%d\n", __func__, ret);
		return ret;
	}

	/* BQ25898S_REG_00 Write */
	ret = bq2589x_set_input_current_limit(bq, bq->cfg.iindpm_threshold);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set input current limit:%d\n", __func__, ret);
		return ret;
	}
	/* BQ25898S_REG_0D Write */
	return bq2589x_adjust_absolute_vindpm(bq);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}


void bq2589x_adapter_in_handler(struct bq2589x *bq)
{
	int ret;

	ret = bq2589x_set_charge_profile(bq);
	if (ret)
		return;

#if defined(CONFIG_PLATFORM_NEC)
	ret = bq2589x_enable_charger(bq, true);
	dev_info(bq->dev, "%s: start charging\n", __func__);

	ret = bq2589x_set_watchdog_timer_value(bq);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to enable watchdog timer:%d\n", __func__, ret);
	}
#else /* defined(CONFIG_PLATFORM_NEC) */
	/* recharge limitation with Gas-gauge-SoC */
	bq->rsoc = bq2589x_read_batt_rsoc(bq);
	if (bq->state.data.charge == BQ25898S_CHRG_STAT_IDLE) {
		if (bq->rsoc > bq->rechg_thres ) {
			dev_dbg(bq->dev, "%s: RSOC=%d(> %d), no need start re-charge\n", __func__, bq->rsoc, bq->rechg_thres);
		} else {
			ret = bq2589x_enable_charger(bq, true);
			dev_info(bq->dev, "%s: start charging (soc: %d < %d)\n", __func__,
					bq->rsoc, bq->rechg_thres);
		}
	}
	ret = bq2589x_set_watchdog_timer(bq, 40);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to enable watchdog timer:%d\n", __func__, ret);
	}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

	schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
}

void bq2589x_adapter_out_handler(struct bq2589x *bq)
{
	int ret;

	ret = bq2589x_enable_charger(bq, false);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to disable charger:%d\n", __func__, ret);
	}
	else {
		dev_info(bq->dev, "%s:slave charge stopped\n", __func__);
	}

	ret = bq2589x_disable_watchdog_timer(bq);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to disable watchdog timer:%d\n", __func__, ret);
	}

#if defined(CONFIG_PLATFORM_NEC)
	bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
	bq->real_usb_psy_ma = CURRENT_500_MA;
	/* BQ25898S_REG_00 Write */
	ret = bq2589x_set_input_current_limit(bq, bq->real_usb_psy_ma);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set input current limit:%d\n", __func__, ret);
	}
#else  /* defined(CONFIG_PLATFORM_NEC) */
	cancel_delayed_work_sync(&bq->polling_work);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int bq2589x_init_device(struct bq2589x *bq)
{
	int ret;

	ret = bq2589x_reset_chip(bq);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to reset all registers:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_disable_watchdog_timer(bq);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to disable watchdog timer:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_enable_auto_dpdm(bq, bq->cfg.enable_auto_dpdm);//always disable autodpdm when acting as slave
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set auto dpdm:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_enable_term(bq, bq->cfg.enable_term);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set termination:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_set_absolute_vindpm(bq, bq->cfg.use_absolute_vindpm); /* use absolute mode vindpm setting*/
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set absolute vindpm:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_set_en12v(bq, true);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to set en12v:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_enable_charger(bq, false);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to disable charger:%d\n", __func__, ret);
		return ret;
	}

	ret = bq2589x_adc_start(bq,false);
	if (ret < 0) {
		dev_err(bq->dev, "%s:Failed to start ADC:%d\n", __func__, ret);
	}

	bq2589x_set_charge_profile(bq);

	if (gpio_is_valid(bq->enable_gpio)) {
		gpio_set_value(bq->enable_gpio, !bq->cfg.enable_init);	//CE = active low
	}

	return ret;
}

static void bq2589x_polling_workfunc(struct work_struct *work)
{
	struct bq2589x *bq = container_of(work, struct bq2589x, polling_work.work);
	u8 status = 0;
	int ret;
	int chg_current,vbus_volt,vsys_volt,vbat_volt;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_0B, &bq->state.byte);
	if(!bq->state.data.pgood)
		return;

	/* recharge limitation with Gas-gauge-SoC */
	bq->rsoc = bq2589x_read_batt_rsoc(bq);
#if !defined(CONFIG_PLATFORM_NEC)
	if (bq->state.data.charge == BQ25898S_CHRG_STAT_IDLE) {
		if (bq->rsoc > bq->rechg_thres ) {
			dev_dbg(bq->dev, "%s: RSOC=%d(> %d), no need start re-charge\n", __func__, bq->rsoc, bq->rechg_thres);
		} else {
			ret = bq2589x_enable_charger(bq, true);
			dev_info(bq->dev, "%s: start charging (soc: %d < %d)\n", __func__,
					bq->rsoc, bq->rechg_thres);
		}
	}
#endif /* !defined(CONFIG_PLATFORM_NEC) */
#ifdef WD_DEBUG
	if (wd_timeout_dbg == 1) {
		printk(KERN_ERR "@@@ wd_timuout_dbg enable skip reset wd\n");
	} else {
		bq2589x_reset_watchdog_timer(bq);
	}
#else
	bq2589x_reset_watchdog_timer(bq);
#endif

	vbus_volt = bq2589x_adc_read_vbus_volt(bq);
	vsys_volt = bq2589x_adc_read_sys_volt(bq);
	vbat_volt = bq2589x_adc_read_battery_volt(bq);
	chg_current = bq2589x_adc_read_charge_current(bq);

#if defined(CONFIG_PLATFORM_NEC)
	dev_info(bq->dev, "%s:soc:%d, state:0x%02x, vbus:%d, vsys:%d, vbat:%d, chg curr:%d ce:%d Ilim=%d\n", __func__,
				bq->rsoc, bq->state.byte, vbus_volt, vsys_volt, vbat_volt, chg_current,
				bq2589x_get_prop_charge_status(bq),
				bq2589x_get_input_current_limit(bq)
	);
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_info(bq->dev, "%s:soc:%d, state:0x%02x, vbus:%d, vsys:%d, vbat:%d, chg curr:%d\n", __func__,
					bq->rsoc, bq->state.byte, vbus_volt, vsys_volt, vbat_volt, chg_current);
#endif /* defined(CONFIG_PLATFORM_NEC) */

	ret = bq2589x_read_byte(bq, BQ25898S_REG_13, &status);
	if (ret == 0 && (status & BQ25898S_VDPM_STAT_MASK))
		dev_info(bq->dev, "%s:VINDPM occurred\n", __func__);
	if (ret == 0 && (status & BQ25898S_IDPM_STAT_MASK))
		dev_info(bq->dev, "%s:IINDPM occurred\n", __func__);

	if (bq->notify_src) {
		bq->notify_src = 0;
		bq->online = 0;
		bq->polling_interval_sec = bq->monitor_interval_sec;
		dev_info(bq->dev,"@@@ %s notify_src in polling interval_sec=%d\n",__FUNCTION__,bq->polling_interval_sec);
		schedule_work(&bq->irq_work);
	}

	schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
}

static void bq2589x_charger_irq_workfunc(struct work_struct *work)
{
	struct bq2589x *bq = container_of(work, struct bq2589x, irq_work);
	u8 val, state;
	int ret;
#if defined(CONFIG_PLATFORM_NEC)
#define WD_TIMEOUT_CHECK	3
	int i;
	int charge_work = 1;
#endif /* defined(CONFIG_PLATFORM_NEC) */

	msleep(5);

	state = bq->state.byte;
	/* Read STATUS and FAULT registers */
	ret = bq2589x_read_byte(bq, BQ25898S_REG_0B, &bq->state.byte);
	ret = bq2589x_read_byte(bq, BQ25898S_REG_0C, &bq->fault.byte);
	ret = bq2589x_read_byte_mask(bq, &val, BQ25898S_REG_11, BQ25898S_VBUS_GD_MASK, BQ25898S_VBUS_GD_SHIFT);
	bq->vbus_good = val;

	dev_info(bq->dev, "%s:vbus_gd:%d, stat:%02x det-%01x/chg-%01x/pg-%01x/vsys-%01x\n", __func__,
		bq->vbus_good,
		bq->state.byte, bq->state.data.vbus, bq->state.data.charge, bq->state.data.pgood, bq->state.data.vsys);
	if(state != bq->state.byte) {
		if (bq->state.data.charge == BQ25898S_CHRG_STAT_IDLE)
			dev_info(bq->dev, "%s:not charging\n", __func__);
		else if (bq->state.data.charge == BQ25898S_CHRG_STAT_PRECHG)
			dev_info(bq->dev, "%s:precharging\n", __func__);
		else if (bq->state.data.charge == BQ25898S_CHRG_STAT_FASTCHG)
			dev_info(bq->dev, "%s:fast charging\n", __func__);
		else if (bq->state.data.charge == BQ25898S_CHRG_STAT_CHGDONE){
			dev_info(bq->dev, "%s:charge done!\n", __func__);
			bq2589x_enable_charger(bq, false);	//if use recharge limitation with Gas-gauge-SoC
			schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
		}
	}

	if (bq->fault.byte) {
		dev_info(bq->dev, "%s:charge fault:%02x\n", __func__, bq->fault.byte);
#if defined(CONFIG_PLATFORM_NEC)
		for (i = 0; i < WD_TIMEOUT_CHECK; i++) {
			ret = bq2589x_read_byte(bq, BQ25898S_REG_0C, &bq->fault.byte);
			if (bq->fault.data.wdt_timeout == 0)
				break;
		}
		if (bq->fault.data.wdt_timeout == 1) {
#ifdef WD_DEBUG
			if (wd_timeout_dbg == 1) {
				printk(KERN_ERR "@@@ wd_timuout_dbg enable\n");
				wd_timeout_dbg = 0;
			}
#endif /* WD_DEBUG */
			dev_err(bq->dev, "@@@ WD timeout!!\n");
			ret = bq2589x_nec_hw_init_regs(bq,1);
			if (ret < 0) {
				dev_err(bq->dev, "Failed to nec_hw_init\n");
			}
			return;
		}
		if (bq->vbus_good == 0) {
			return;
		}
		if (bq->fault.data.ntc == 0) {
			return;
		}
		charge_work = 0;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	}

	if (bq->online != bq->state.data.pgood) {
		bq->online = bq->state.data.pgood;
		switch (bq->state.data.vbus) {
			case BQ25898S_VBUS_STAT_NOPOWER:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
				bq->real_usb_psy_ma = CURRENT_500_MA;
				break;
			case BQ25898S_VBUS_STAT_USB_SDP:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_SDP;
				bq->real_usb_psy_ma = CURRENT_500_MA;
				break;
			case BQ25898S_VBUS_STAT_USB_CDP:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_CDP;
				bq->real_usb_psy_ma = CURRENT_500_MA;
				break;
			case BQ25898S_VBUS_STAT_USB_DCP:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
				bq->real_usb_psy_ma = CURRENT_1800_MA;
				break;
			case BQ25898S_VBUS_STAT_HVDCP:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
				bq->real_usb_psy_ma = CURRENT_1000_MA;
				break;
			case BQ25898S_VBUS_STAT_UNKNOWN:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
				bq->real_usb_psy_ma = CURRENT_500_MA;
				break;
			case BQ25898S_VBUS_STAT_NO_STDADP:
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
				bq->real_usb_psy_ma = CURRENT_1800_MA;
				break;
			default:
				dev_err(bq->dev, "%s: unkown type =0x%x\n", __func__, bq->state.data.vbus);
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
				bq->real_usb_psy_ma = CURRENT_500_MA;
				break;
		}
		power_supply_changed(bq->charger_psy);
		msleep(1000);
		bq->cradle_status = bq2589x_get_prop_cradle_status(bq);
		dev_info(bq->dev, "%s:cradle_status:%d\n", __func__, bq->cradle_status);
		if (bq->cradle_status == 1) {
			if (bq->online == 1) {
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
				bq->real_usb_psy_ma = CURRENT_1800_MA;
			} else {
				bq->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
				bq->real_usb_psy_ma = CURRENT_500_MA;
			}
		}
		bq2589x_set_charge_profile(bq);
		bq2589x_notify_usb_src_change(bq,(int)bq->online);
		bq->ilim_det_result = bq2589x_get_input_current_limit(bq) * 1000;
		dev_info(bq->dev, "%s:Ilim Det-reslt:%d uA\n", __func__, bq->ilim_det_result);
#if defined(CONFIG_PLATFORM_NEC)
		charge_work = 1;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	}

#if defined(CONFIG_PLATFORM_NEC)
	if (charge_work == 0) {
		dev_info(bq->dev, "%s:charge_work == 0 in\n", __func__);
		return;
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

#if defined(CONFIG_PLATFORM_NEC)
	if (bq->state.data.pgood) {
		if (bq->state.data.charge == BQ25898S_CHRG_STAT_IDLE) {
			bq2589x_adapter_in_handler(bq);
		} else {
			ret = bq2589x_set_watchdog_timer_value(bq);
			if (ret < 0) {
				dev_err(bq->dev, "%s:Failed to enable watchdog timer:%d\n", __func__, ret);
			}
			cancel_delayed_work_sync(&bq->polling_work);
			schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
		}
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (bq->state.data.pgood
		&& bq->state.data.charge == BQ25898S_CHRG_STAT_IDLE) {
		bq2589x_adapter_in_handler(bq);
#endif /* defined(CONFIG_PLATFORM_NEC) */
	} else if (!bq->state.data.pgood) {
		bq2589x_adapter_out_handler(bq);
	}
}

static int bq2589x_power_supply_get_property(struct power_supply *psy,
						enum power_supply_property psp,
						union power_supply_propval *val)
{
	struct bq2589x *bq = power_supply_get_drvdata(psy);
	struct bq2589x_state state;
	struct bq2589x_fault fault;

	mutex_lock(&bq2589x_state_lock);
	state = bq->state.data;
	fault = bq->fault.data;
	mutex_unlock(&bq2589x_state_lock);

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		if (!state.pgood)
			val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
		else if (state.charge == BQ25898S_CHRG_STAT_IDLE)
			val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
		else if (state.charge == BQ25898S_CHRG_STAT_PRECHG ||
				 state.charge == BQ25898S_CHRG_STAT_FASTCHG)
			val->intval = POWER_SUPPLY_STATUS_CHARGING;
		else if (state.charge == BQ25898S_CHRG_STAT_CHGDONE)
			val->intval = POWER_SUPPLY_STATUS_FULL;
		else
			val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
		break;

	case POWER_SUPPLY_PROP_MANUFACTURER:
		val->strval = BQ2589X_MANUFACTURER;
		break;

	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = bq->online;
		break;

	case POWER_SUPPLY_PROP_HEALTH:
		if (!fault.charge && !fault.bat && !fault.boost)
			val->intval = POWER_SUPPLY_HEALTH_GOOD;
		else if (fault.bat)
			val->intval = POWER_SUPPLY_HEALTH_OVERVOLTAGE;
		else if (fault.charge == BQ25898S_FAULT_CHRG_TIMER)
			val->intval = POWER_SUPPLY_HEALTH_SAFETY_TIMER_EXPIRE;
		else if (fault.charge == BQ25898S_FAULT_CHRG_THERMAL)
			val->intval = POWER_SUPPLY_HEALTH_OVERHEAT;
		else
			val->intval = POWER_SUPPLY_HEALTH_UNSPEC_FAILURE;
		break;

	case POWER_SUPPLY_PROP_CONSTANT_CHARGE_CURRENT:
		val->intval = bq2589x_get_adc_charge_current(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_CONSTANT_CHARGE_CURRENT_MAX:
		val->intval =  bq2589x_get_charge_current(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_CONSTANT_CHARGE_VOLTAGE:
		val->intval = bq2589x_get_adc_batt_voltage(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_CONSTANT_CHARGE_VOLTAGE_MAX:
		val->intval = bq2589x_get_charge_voltage(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_INPUT_CURRENT_LIMIT:
		val->intval =  bq2589x_get_input_current_limit(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_PRECHARGE_CURRENT:
		val->intval = bq2589x_get_precharge_current(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_CHARGE_TERM_CURRENT:
		val->intval = bq2589x_get_termination_current(bq) * 1000;
		break;
	case POWER_SUPPLY_PROP_USB_TYPE:
		val->intval = bq->psy_usb_type;
		break;
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_VBUS_CONNECTION:
		val->intval = bq->vbus_good; /* BQ25898S_REG_11 BQ25898S_VBUS_GD_SHIFT(7) */
		break;
	case POWER_SUPPLY_PROP_CRADLE_STATUS:
		val->intval = bq2589x_get_prop_cradle_status(bq);
		break;
	case POWER_SUPPLY_PROP_MACHINE_STATUS:
		val->intval = bq2589x_get_prop_machine_status(bq);
		break;
	case POWER_SUPPLY_PROP_CHARGE_STOP:
		val->intval = bq->chage_stop;
		pr_debug("%s POWER_SUPPLY_PROP_CHARGE_STOP val=%d\n",__func__,val->intval);
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		val->intval = bq->real_usb_psy_ma;
		break;
	case POWER_SUPPLY_PROP_SAVE_CURRENT:
		val->intval = bq->real_usb_psy_ma;
		break;
	case POWER_SUPPLY_PROP_GADGET_STATUS:
		val->intval = bq->gadget_status;
		pr_debug("%s POWER_SUPPLY_PROP_GADGET_STATUS val=%d\n",__func__,val->intval);
		break;
	case POWER_SUPPLY_PROP_CHARGE_VOLTAGE:
		val->intval = float_voltage;
		pr_debug("%s POWER_SUPPLY_PROP_CHARGE_VOLTAGE val=0x%x com=%s\n",__func__,val->intval,current->comm);
		break;
	case POWER_SUPPLY_PROP_DCP_TYPE:
		val->intval = bq2589x_get_prop_usb_type_check(bq);
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		val->intval = bq->notify_src;
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = bq2589x_adc_read_temperature(bq);
		break;
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
		val->intval = bq2589x_get_prop_charge_status(bq);
		break;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	default:
		return -EINVAL;
	}

	return 0;
}

static int bq2589x_power_supply_set_property(struct power_supply *psy,
						enum power_supply_property psp,
						const union power_supply_propval *val)
{
	struct bq2589x *bq = power_supply_get_drvdata(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_INPUT_CURRENT_LIMIT:
		bq->ilim_from_tcpc = val->intval;
		dev_info(bq->dev, "%s:Ilim from tcpc:%d uA\n", __func__, bq->ilim_from_tcpc);
		return 0;
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
		dev_dbg(bq->dev, "%s POWER_SUPPLY_PROP_CHARGING_ENABLED val=%d\n",__func__,val->intval);
		bq->chg_enabled = val->intval;
		bq2589x_enable_charger(bq,bq->chg_enabled);
		break;
	case POWER_SUPPLY_PROP_CHARGE_STOP:
		dev_dbg(bq->dev, "%s POWER_SUPPLY_PROP_CHARGE_STOP val=%d\n",__func__,val->intval);
		bq->chage_stop = val->intval;
		bq2589x_enable_charger(bq,bq->chg_enabled);
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		dev_dbg(bq->dev, "%s POWER_SUPPLY_PROP_FORCE_DISCHARGE val=%d\n",__func__,val->intval);
		if (val->intval) {
			if (val->intval != bq->real_usb_psy_ma) {
				printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Force input current %dmA\n", val->intval);
				bq->prev_usb_psy_dis_ma = bq->real_usb_psy_ma;
				bq2589x_set_input_current_limit(bq,val->intval);
				bq->real_usb_psy_ma = val->intval;
			}
		} else {
			if ((bq->prev_usb_psy_dis_ma != 0) && (bq->prev_usb_psy_dis_ma != bq->real_usb_psy_ma)) {
				if (bq->online) {
					printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Restore input current %dmA to %dmA\n", bq->real_usb_psy_ma, bq->prev_usb_psy_dis_ma);
					bq2589x_set_input_current_limit(bq,bq->prev_usb_psy_dis_ma);
					bq->real_usb_psy_ma = bq->prev_usb_psy_dis_ma;
				}
				/* printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Restore stay\n"); */
				bq->prev_usb_psy_dis_ma = 0;
			}
		}
		break;
	case POWER_SUPPLY_PROP_SAVE_CURRENT:
		dev_dbg(bq->dev, "%s POWER_SUPPLY_PROP_SAVE_CURRENT val=%d\n",__func__,val->intval);
		if (val->intval) {
			if ((val->intval != bq->real_usb_psy_ma) && ((bq->real_usb_psy_ma == CURRENT_1000_MA)||(bq->real_usb_psy_ma == CURRENT_1800_MA))) {
				printk("POWER_SUPPLY_PROP_SAVE_CURRENT Force input current %dmA\n", val->intval);
				bq->prev_usb_psy_sav_ma = bq->real_usb_psy_ma;
				bq2589x_set_input_current_limit(bq,val->intval);
				bq->real_usb_psy_ma = val->intval;
			}
		} else {
			if ((bq->prev_usb_psy_sav_ma != 0) && (bq->prev_usb_psy_sav_ma != bq->real_usb_psy_ma)) {
				if (bq->online) {
					printk("POWER_SUPPLY_PROP_SAVE_CURRENT Restore input current %dmA to %dmA\n", bq->real_usb_psy_ma, bq->prev_usb_psy_sav_ma);
					bq2589x_set_input_current_limit(bq,bq->prev_usb_psy_sav_ma);
					bq->real_usb_psy_ma = bq->prev_usb_psy_sav_ma;
				}
				/* printk("POWER_SUPPLY_PROP_SAVE_CURRENT Restore stay\n"); */
				bq->prev_usb_psy_sav_ma = 0;
			}
		}
		break;
	case POWER_SUPPLY_PROP_GADGET_STATUS:
		dev_dbg(bq->dev, "%s POWER_SUPPLY_PROP_GADGET_STATUS val=%d\n",__func__,val->intval);
		bq->gadget_status = val->intval;
		break;
	case POWER_SUPPLY_PROP_CHARGE_VOLTAGE:
		printk("####bq2589x_power_supply_set_property bq2589x_set_charging_voltage\n");
		bq2589x_set_charging_voltage(bq, val->intval);
		bq->current_charging_voltage_step = val->intval;
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		dev_info(bq->dev, "%s POWER_SUPPLY_PROP_NOTIFY_SRC val=%d\n",__func__,val->intval);
		cancel_delayed_work_sync(&bq->polling_work);
		bq->polling_interval_sec = bq->detect_interval_sec;
		bq->notify_src = 1;
		dev_info(bq->dev,"@@@ %s notify_src in polling interval_sec=%d\n",__FUNCTION__,bq->polling_interval_sec);
		schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
		break;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	default:
		return -EINVAL;
	}
	return 0;
}

static int bq2589x_is_writeable(struct power_supply *psy,
				       enum power_supply_property prop)
{
	int rc = 0;

	switch (prop) {
	case POWER_SUPPLY_PROP_INPUT_CURRENT_LIMIT:
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_CHARGE_STOP:
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
#endif /* defined(CONFIG_PLATFORM_NEC) */
		rc = 1;
		break;
	default:
		rc = 0;
		break;
	}
	return rc;
}

static enum power_supply_usb_type bq2589x_power_supply_usb_types[] = {
	POWER_SUPPLY_USB_TYPE_UNKNOWN,
	POWER_SUPPLY_USB_TYPE_SDP,
	POWER_SUPPLY_USB_TYPE_DCP,
	POWER_SUPPLY_USB_TYPE_CDP,
};

static enum power_supply_property bq2589x_power_supply_props[] = {
	POWER_SUPPLY_PROP_MANUFACTURER,
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_ONLINE,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_CONSTANT_CHARGE_CURRENT,
	POWER_SUPPLY_PROP_CONSTANT_CHARGE_CURRENT_MAX,
	POWER_SUPPLY_PROP_CONSTANT_CHARGE_VOLTAGE,
	POWER_SUPPLY_PROP_CONSTANT_CHARGE_VOLTAGE_MAX,
	POWER_SUPPLY_PROP_INPUT_CURRENT_LIMIT,
	POWER_SUPPLY_PROP_PRECHARGE_CURRENT,
	POWER_SUPPLY_PROP_CHARGE_TERM_CURRENT,
	POWER_SUPPLY_PROP_USB_TYPE
#if defined(CONFIG_PLATFORM_NEC)
	,
	POWER_SUPPLY_PROP_VBUS_CONNECTION,
	POWER_SUPPLY_PROP_CRADLE_STATUS,
	POWER_SUPPLY_PROP_MACHINE_STATUS,
	POWER_SUPPLY_PROP_CHARGE_STOP,
	POWER_SUPPLY_PROP_FORCE_DISCHARGE,
	POWER_SUPPLY_PROP_SAVE_CURRENT,
	POWER_SUPPLY_PROP_GADGET_STATUS,
	POWER_SUPPLY_PROP_CHARGE_VOLTAGE,
	POWER_SUPPLY_PROP_DCP_TYPE,
	POWER_SUPPLY_PROP_NOTIFY_SRC
	,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_CHARGING_ENABLED,
#endif /* defined(CONFIG_PLATFORM_NEC) */
};

static char *bq2589x_charger_supplied_to[] = {
	"battery",
};

static const struct power_supply_desc bq2589x_power_supply_desc = {
	.name = "bq2589x-charger",
	.type = POWER_SUPPLY_TYPE_USB,
	.usb_types = bq2589x_power_supply_usb_types,
	.num_usb_types = ARRAY_SIZE(bq2589x_power_supply_usb_types),
	.properties = bq2589x_power_supply_props,
	.num_properties = ARRAY_SIZE(bq2589x_power_supply_props),
	.get_property = bq2589x_power_supply_get_property,
	.set_property = bq2589x_power_supply_set_property,
	.property_is_writeable = bq2589x_is_writeable,
};

static int bq2589x_power_supply_init(struct bq2589x *bq)
{
	struct power_supply_config psy_cfg = { .drv_data = bq, };

	psy_cfg.supplied_to = bq2589x_charger_supplied_to;
	psy_cfg.num_supplicants = ARRAY_SIZE(bq2589x_charger_supplied_to);

	bq->charger_psy = power_supply_register(bq->dev, &bq2589x_power_supply_desc, &psy_cfg);

	return PTR_ERR_OR_ZERO(bq->charger_psy);
}


int bq2589x_reg_enable_regulator_otg(struct regulator_dev *rdev)
{
	/* TODO */
	return 0;
}

int bq2589x_reg_disable_regulator_otg(struct regulator_dev *rdev)
{
	/* TODO */
	return 0;
}

static int bq2589x_reg_set_current_limit(struct regulator_dev *rdev,
						int min_uA, int max_uA)
{
	/* TODO */
	return 0;
}

static int bq2589x_reg_get_current_limit(struct regulator_dev *rdev)
{
	/* TODO */
	return 0;
}


#if defined(CONFIG_PLATFORM_NEC)
static int bq2589x_set_charging_voltage(struct bq2589x *bq, int step)
{
	int rc;

	mutex_lock(&bq->float_v_lock);
	rc = bq2589x_write_byte(bq, BQ25898S_REG_06, charge_voltage_table[step]);
	if (rc < 0) {
		dev_err(bq->dev, "Unable to set BQ25898S_REG_06 rc = %d\n", rc);
		mutex_unlock(&bq->float_v_lock);
		return rc;
	}
	printk("####bq2589x_set_charging_voltage OK step%d set%02x\n", step, charge_voltage_table[step]);
	float_voltage = charge_voltage_table[step];
	mutex_unlock(&bq->float_v_lock);
	return 0;
}

#define POWER_SUPPLY_TYPEC_NAME		"type-c-usb"

static int bq2589x_notify_usb_src_change(struct bq2589x *bq,int src_status)
{
	union power_supply_propval pval;
	int ret = -ENODEV;

	pval.intval = src_status;

	if (!bq->batt_psy)
		bq->batt_psy = power_supply_get_by_name(POWER_SUPPLY_BATTERY_NAME);

	if (bq->batt_psy) {
		ret = power_supply_set_property(bq->batt_psy, POWER_SUPPLY_PROP_NOTIFY_SRC, &pval);
	}

	return ret;
}

/* 1:connected 0:unconnected -1:-ENODEV */
static int bq2589x_get_prop_cradle_status(struct bq2589x *bq)
{
	union power_supply_propval pval;
	int ret = -ENODEV;

	if (!bq->usb_psy)
		bq->usb_psy = power_supply_get_by_name(POWER_SUPPLY_TYPEC_NAME);

	if (bq->usb_psy) {
		ret = power_supply_get_property(bq->usb_psy, POWER_SUPPLY_PROP_CRADLE_STATUS, &pval);
		if (ret == 0) {
			ret = pval.intval;
		}
	}
	return ret;
}

static int bq2589x_get_prop_charge_status(struct bq2589x *bq)
{
	int rc;
	u8 reg;

	rc = bq2589x_read_byte(bq, BQ25898S_REG_03, &reg);
	if(rc < 0) {
		printk(KERN_ERR "%s BQ25898S_REG_03 read err=%d\n",__func__,rc);
		return rc;
	}

	pr_debug("%s BQ25898S_REG_03 val=0x%x\n",__func__,reg);
	return ((reg & BQ25898S_CHG_CONFIG_MASK) >> BQ25898S_CHG_CONFIG_SHIFT);
}

static int bq2589x_get_prop_usb_type_check(struct bq2589x *bq)
{
	int ret;
	u8 byte;
	struct bq2589x_state *state;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_0B, &byte);
	if(ret < 0) {
		printk(KERN_ERR "%s BQ25898S_REG_0B read err=%d\n",__func__,ret);
		return 0;
	}

	state = (struct bq2589x_state *)&byte;
	ret = 0;
	pr_debug("%s BQ25898S_REG_0B val=0x%x\n",__func__,byte);
	switch (state->vbus) {
	case BQ25898S_VBUS_STAT_NOPOWER:
		pr_debug("%s BQ25898S_VBUS_STAT_NOPOWER\n",__func__);
		break;
	case BQ25898S_VBUS_STAT_USB_SDP:
		pr_debug("%s BQ25898S_VBUS_STAT_USB_SDP\n",__func__);
		break;
	case BQ25898S_VBUS_STAT_USB_CDP:
		pr_debug("%s BQ25898S_VBUS_STAT_USB_CDP\n",__func__);
		break;
	case BQ25898S_VBUS_STAT_USB_DCP:
		pr_debug("%s BQ25898S_VBUS_STAT_USB_DCP\n",__func__);
		break;
	case BQ25898S_VBUS_STAT_HVDCP:
		pr_debug("%s BQ25898S_VBUS_STAT_HVDCP\n",__func__);
		break;
	case BQ25898S_VBUS_STAT_UNKNOWN:
		pr_debug("%s BQ25898S_VBUS_STAT_UNKNOWN\n",__func__);
		if ((bq->cradle_status == 0) && (bq->online))
			ret = 1;
		break;
	case BQ25898S_VBUS_STAT_NO_STDADP:
		pr_debug("%s BQ25898S_VBUS_STAT_NO_STDADP\n",__func__);
		break;
	}
	return ret;
}

#define MACHINE_STATUS_GOOD				0
#define MACHINE_STATUS_HOT_HARD_LIMIT	1

static int bq2589x_get_prop_machine_status(struct bq2589x *bq)
{
	int ret;
	u8 byte;
	struct bq2589x_fault *fault;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_0C, &byte);
	if(ret < 0) {
		printk(KERN_ERR "%s BQ25898S_REG_0C read err=%d\n",__func__,ret);
		return 0;
	}

	pr_debug("%s BQ25898S_REG_0C val=0x%x\n",__func__,byte);
	fault = (struct bq2589x_fault *)&byte;

	if (fault->ntc == 0x6) {
		ret = MACHINE_STATUS_HOT_HARD_LIMIT;
	} else {
		ret = MACHINE_STATUS_GOOD;
	}
	return ret;
}

static int bq2589x_adc_read_temperature(struct bq2589x *bq)
{
	uint8_t val;
	int temp;
	int ts_value;
	int ret;
	ret = bq2589x_read_byte(bq, BQ25898S_REG_10, &val);

	if (ret < 0) {
		printk(KERN_ERR "%s BQ25898S_REG_10 read err=%d\n",__func__,ret);
		return ret;
	} else {
		ts_value = (BQ2589X_TSPCT_BASE * 1000) + ((val & BQ2589X_TSPCT_MASK) >> BQ2589X_TSPCT_SHIFT) * BQ2589X_TSPCT_LSB;
		pr_debug("%s ts_val=%d\n",__func__,ts_value);

		/* 112.85                - (1.5248                  *  (ts_value / 1000)       ) */
		/* (112.85 * 1000*10000) - ((1.5248 *10000)         *  (ts_value))             ) */
		temp = (1128500000 - (15248 * ts_value)) /100000;
		return temp;
	}
}

static int bq2589x_nec_hw_init_regs(struct bq2589x *bq, int recovery)
{
	struct init_reg_t *init_regs = hw_init_regs;
	uint8_t val;
	int ret;
	int i;

	ret = bq2589x_read_byte(bq, BQ25898S_REG_07, &val);
	if (ret < 0)
		return ret;

	for (i = 0; i < ARRAY_SIZE(hw_init_regs); i++, init_regs++) {
		ret = bq2589x_write_byte(bq, init_regs->reg, init_regs->val);
		if (ret) {
			dev_err(bq->dev, "bq2589x_write_byte error %02x %02x ret=%d\n",
			    init_regs->reg, init_regs->val, ret);
			break;
		}
	}

	if (recovery == 1) {
		ret = bq2589x_set_input_current_limit(bq, bq->real_usb_psy_ma);
		if (ret < 0) {
			dev_err(bq->dev, "%s:Failed to set input current limit:%d\n", __func__, ret);
		}

		ret = bq2589x_enable_charger(bq, bq->chg_enabled);
		dev_info(bq->dev, "%s: start charging ret=%d\n", __func__,ret);

		ret = bq2589x_set_watchdog_timer_value(bq);
		if (ret < 0) {
			dev_err(bq->dev, "%s:Failed to enable watchdog timer:%d\n", __func__, ret);
		}

		ret = bq2589x_set_charging_voltage(bq,bq->current_charging_voltage_step);
		if (ret < 0) {
			dev_err(bq->dev, "%s:Failed to charging_voltage:%d\n", __func__, ret);
		}
	}
	return ret;
}

static int bq2589x_set_watchdog_timer_value(struct bq2589x *bq)
{
	int ret;

	/* BQ25898S_REG_07 Write */
	/* 160 sec */
	ret = bq2589x_set_watchdog_timer(bq, (160-1));

	return ret;
}

static int bq2589x_enter_ship_mode(struct bq2589x *bq)
{
	return bq2589x_write_byte_mask(bq, BQ25898S_REG_09,
		BQ2589X_BATFET_OFF,
		BQ2589X_BATFET_DIS_MASK,BQ2589X_BATFET_DIS_SHIFT);
}
#endif /* defined(CONFIG_PLATFORM_NEC) */


static const struct regulator_ops bq2589x_chg_otg_ops = {
	.enable = bq2589x_reg_enable_regulator_otg,
	.disable = bq2589x_reg_disable_regulator_otg,
	.is_enabled = regulator_is_enabled_regmap,
	.list_voltage = regulator_list_voltage_linear,
	.set_voltage_sel = regulator_set_voltage_sel_regmap,
	.get_voltage_sel = regulator_get_voltage_sel_regmap,
	.set_current_limit = bq2589x_reg_set_current_limit,
	.get_current_limit = bq2589x_reg_get_current_limit,
};

static const struct regulator_desc bq2589x_otg_rdesc = {
	.of_match = "usb-otg-vbus",
	.name = "usb-otg-vbus",
	.ops = &bq2589x_chg_otg_ops,
	.owner = THIS_MODULE,
	.type = REGULATOR_VOLTAGE,
	.min_uV = 4550000,
	.uV_step = 64000, /* step  64mV */
	.n_voltages = 16, /* 4550mV to 5510mV */
	.vsel_reg = BQ25898S_REG_0A,
	.vsel_mask = BQ25898S_BOOSTV_MASK,
	.csel_reg = BQ25898S_REG_0A,
	.csel_mask = BQ25898S_BST_ILIM_MASK,
	.enable_reg = BQ25898S_REG_03,
	.enable_mask = BQ25898S_OTG_CONFIG_MASK,
};

static irqreturn_t bq2589x_charger_interrupt(int irq, void *data)
{
	struct bq2589x *bq = data;

	schedule_work(&bq->irq_work);
	return IRQ_HANDLED;
}

static int bq2589x_charger_probe(struct i2c_client *client,
			   const struct i2c_device_id *id)
{
	struct bq2589x *bq;
	struct regulator_config config = { };
	int ret;

	bq = devm_kzalloc(&client->dev, sizeof(struct bq2589x), GFP_KERNEL);
	if (!bq) {
		dev_err(&client->dev, "%s: out of memory\n", __func__);
		return -ENOMEM;
	}

	bq->dev = &client->dev;
	bq->client = client;
	i2c_set_clientdata(client, bq);

#if defined(CONFIG_PLATFORM_NEC)
	mutex_init(&bq->float_v_lock);
	bq->prev_usb_psy_dis_ma = 0;
	bq->prev_usb_psy_sav_ma = 0;
	bq->real_usb_psy_ma = CURRENT_500_MA;
	bq->chage_stop = 0;
	bq->cradle_status = 0;
	bq->current_charging_voltage_step = 0;
#endif /* defined(CONFIG_PLATFORM_NEC) */

	ret = bq2589x_detect_device(bq);
	if (!ret && bq->part_no == BQ25898S ) {
		dev_info(bq->dev, "%s: charger device bq25898S detected, revision:%d\n", __func__, bq->revision);
	} else if (!ret && bq->part_no == BQ25898D ) {
		dev_info(bq->dev, "%s: charger device bq25898D detected, revision:%d\n", __func__, bq->revision);
	} else {
		dev_info(bq->dev, "%s: no bq25898S charger device found:%d\n", __func__, ret);
		return -ENODEV;
	}

	bq->batt_psy = devm_power_supply_get_by_phandle(bq->dev, "battery");
	if(bq->batt_psy) {
		dev_info(bq->dev, "%s: battery device %s detected\n", __func__, bq->batt_psy->desc->name);
	} else {
		dev_info(bq->dev, "%s: batt_psy is not found.\n", __func__);
	}

	if (client->dev.of_node)
		bq2589x_parse_dt(&client->dev, bq);

	ret = bq2589x_init_device(bq);
	if (ret) {
		dev_err(bq->dev, "device init failure: %d\n", ret);
		goto err_0;
	}

	if (client->irq < 0) {
		dev_err(bq->dev, "%s: irq not defined (%d)\n", __func__, client->irq);
		return -EINVAL;
	}

	INIT_WORK(&bq->irq_work, bq2589x_charger_irq_workfunc);
	INIT_DELAYED_WORK(&bq->polling_work, bq2589x_polling_workfunc);

#if !defined(CONFIG_PLATFORM_NEC)
	schedule_work(&bq->irq_work);	//initial-read state & fault register
	schedule_delayed_work(&bq->polling_work, 0); //polling start
#endif /* !defined(CONFIG_PLATFORM_NEC) */

	ret = sysfs_create_group(&bq->dev->kobj, &bq2589x_attr_group);
	if (ret) {
		dev_err(bq->dev, "failed to register sysfs. err: %d\n", ret);
		goto err_irq;
	}

	ret = devm_request_threaded_irq(&client->dev, client->irq, NULL, bq2589x_charger_interrupt,
					IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
					"bq2589x_charger_irq", bq);
	if (ret) {
		dev_err(bq->dev, "%s:Request IRQ %d failed: %d\n", __func__, client->irq, ret);
		goto err_irq;
	} else {
		dev_info(bq->dev, "%s:irq = %d\n", __func__, client->irq);
#if defined(CONFIG_PLATFORM_NEC)
		enable_irq_wake(client->irq);
#endif /* defined(CONFIG_PLATFORM_NEC) */
	}

	/* otg regulator */
	config.dev = bq->dev;
	config.driver_data = bq;
	#ifdef CONFIG_REGMAP
	config.regmap = devm_regmap_init_i2c(bq->client, &bq2589x_regmap_config);
	#endif

	bq->otg_rdev = devm_regulator_register(bq->dev, &bq2589x_otg_rdesc, &config);
	if (IS_ERR(bq->otg_rdev)) {
		dev_notice(bq->dev, "%s : regulator register fail\n", __func__);
		goto err_regulator_dev;
	}

#if defined(CONFIG_PLATFORM_NEC)
	ret = bq2589x_nec_hw_init_regs(bq,0);
	if (ret < 0) {
		dev_err(bq->dev, "Failed to nec_hw_init\n");
		goto err_regulator_dev;
	}

	ret = bq2589x_power_supply_init(bq);
	if (ret < 0) {
		dev_err(bq->dev, "Failed to register power supply\n");
		goto err_regulator_dev;
	}
	bq->monitor_interval_sec = bq->polling_interval_sec;
	bq->detect_interval_sec = 1;
	bq->polling_interval_sec = bq->detect_interval_sec;
	bq->notify_src = 1;
	schedule_delayed_work(&bq->polling_work, bq->polling_interval_sec * HZ);
#endif /* defined(CONFIG_PLATFORM_NEC) */

	return 0;

err_regulator_dev:
err_irq:
	cancel_work_sync(&bq->irq_work);
	cancel_delayed_work_sync(&bq->polling_work);
err_0:
	return ret;
}

static void bq2589x_charger_shutdown(struct i2c_client *client)
{
	struct bq2589x *bq = i2c_get_clientdata(client);

	dev_info(bq->dev, "%s: shutdown\n", __func__);

#if defined(CONFIG_PLATFORM_NEC)
	bq2589x_enter_ship_mode(bq);
#endif /* defined(CONFIG_PLATFORM_NEC) */
	sysfs_remove_group(&bq->dev->kobj, &bq2589x_attr_group);
	cancel_work_sync(&bq->irq_work);
	cancel_delayed_work_sync(&bq->polling_work);

	free_irq(bq->client->irq, bq);
}

#ifdef CONFIG_PM_SLEEP
static int bq2589x_suspend(struct device *dev)
{
	struct bq2589x *bq = dev_get_drvdata(dev);

	dev_info(bq->dev, "%s: adc stop\n", __func__);
	bq2589x_adc_stop(bq);
	return 0;
}

static int bq2589x_resume(struct device *dev)
{
	struct bq2589x *bq = dev_get_drvdata(dev);

	dev_info(bq->dev, "%s: adc\n", __func__);
	bq2589x_adc_start(bq,false);

	return 0;
}

static const struct dev_pm_ops bq2589x_pm = {
	SET_SYSTEM_SLEEP_PM_OPS(bq2589x_suspend, bq2589x_resume)
};
#endif /* CONFIG_PM_SLEEP */

static struct of_device_id bq2589x_charger_match_table[] = {
	{.compatible = "ti,bq25898s",},
	{.compatible = "ti,bq25898d",},
	{},
};


static const struct i2c_device_id bq2589x_charger_id[] = {
	{ "bq25898s", BQ25898S },
	{ "bq25898d", BQ25898D },
	{},
};

MODULE_DEVICE_TABLE(i2c, bq2589x_charger_id);

static struct i2c_driver bq2589x_charger_driver = {
	.driver		= {
		.name	= "bq25898x",
		.of_match_table = bq2589x_charger_match_table,
#ifdef CONFIG_PM_SLEEP
		.pm = &bq2589x_pm,
#endif /* CONFIG_PM_SLEEP */
	},
	.id_table	= bq2589x_charger_id,

	.probe		= bq2589x_charger_probe,
	.shutdown   = bq2589x_charger_shutdown,
};

module_i2c_driver(bq2589x_charger_driver);

MODULE_DESCRIPTION("TI BQ2589x Charger Driver");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Texas Instruments");
