/*
 * Copyright (c) 2013 - 2015, The Linux Foundation. All rights reserved.
 * Copyright (c) 2015 - 2022, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*****************************************************************************
 *  Includes
 *****************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>

#include <linux/kthread.h>

#include "nec_charger_manager.h"

#ifdef CHARGING_CONTROL_DEBUG
#define CC_DBG_STATE_MAX	6  /*(CHARGING_CONTROL_STATE_5 + 1) */
#define CC_DBG_COUNT_MAX	12

struct charging_control_debug_param {
	int debug_remain;
	int debug_ChargeStatus;
	int debug_FullCharge;
};

struct charging_control_debug_info {
	int debug_state;
	int debug_count;
	struct charging_control_debug_param  debug_table[CC_DBG_STATE_MAX][CC_DBG_COUNT_MAX];
};

struct charging_control_debug_info g_debug_info;

struct charging_control_debug_param charge_save_param[CC_DBG_STATE_MAX][CC_DBG_COUNT_MAX] = 
{
	/* charge_save_param */
	/* CHARGING_CONTROL_STATE_0 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_1 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_2 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_3 */
	{
		{70,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{69,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{68,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{67,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{66,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{65,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{64,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{63,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{62,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{61,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{60,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_4 */
	{
		{60,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{59,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{57,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{56,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{54,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{53,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{52,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{51,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{50,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_5 */
	{
		{50,POWER_SUPPLY_STATUS_CHARGING,-1},
		{51,POWER_SUPPLY_STATUS_CHARGING,-1},
		{52,POWER_SUPPLY_STATUS_CHARGING,-1},
		{53,POWER_SUPPLY_STATUS_CHARGING,-1},
		{54,POWER_SUPPLY_STATUS_CHARGING,-1},
		{55,POWER_SUPPLY_STATUS_CHARGING,-1},
		{56,POWER_SUPPLY_STATUS_CHARGING,-1},
		{57,POWER_SUPPLY_STATUS_CHARGING,-1},
		{58,POWER_SUPPLY_STATUS_CHARGING,-1},
		{59,POWER_SUPPLY_STATUS_CHARGING,-1},
		{60,POWER_SUPPLY_STATUS_CHARGING,-1},
		{-1,-1,-1}
	}
};

struct charging_control_debug_param longlife_param[CC_DBG_STATE_MAX][CC_DBG_COUNT_MAX] = 
{
	/* longlife_param */
	/* CHARGING_CONTROL_STATE_0 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_1 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_2 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_3 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_4 */
	{
		{70,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{69,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{68,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{67,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{66,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{65,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{64,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{63,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{62,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{61,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{60,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_5 */
	{
		{60,POWER_SUPPLY_STATUS_CHARGING,-1},
		{61,POWER_SUPPLY_STATUS_CHARGING,-1},
		{62,POWER_SUPPLY_STATUS_CHARGING,-1},
		{63,POWER_SUPPLY_STATUS_CHARGING,-1},
		{64,POWER_SUPPLY_STATUS_CHARGING,-1},
		{65,POWER_SUPPLY_STATUS_CHARGING,-1},
		{66,POWER_SUPPLY_STATUS_CHARGING,-1},
		{67,POWER_SUPPLY_STATUS_CHARGING,-1},
		{68,POWER_SUPPLY_STATUS_CHARGING,-1},
		{69,POWER_SUPPLY_STATUS_CHARGING,-1},
		{70,POWER_SUPPLY_STATUS_CHARGING,-1},
		{-1,-1,-1}
	}
};

struct charging_control_debug_param charging_param[CC_DBG_STATE_MAX][CC_DBG_COUNT_MAX] = 
{
	/* charging_param */
	/* CHARGING_CONTROL_STATE_0 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_1 */
	{
		{90,POWER_SUPPLY_STATUS_CHARGING,-1},
		{91,POWER_SUPPLY_STATUS_CHARGING,-1},
		{92,POWER_SUPPLY_STATUS_CHARGING,-1},
		{93,POWER_SUPPLY_STATUS_CHARGING,-1},
		{94,POWER_SUPPLY_STATUS_CHARGING,-1},
		{95,POWER_SUPPLY_STATUS_CHARGING,-1},
		{96,POWER_SUPPLY_STATUS_CHARGING,-1},
		{97,POWER_SUPPLY_STATUS_CHARGING,-1},
		{98,POWER_SUPPLY_STATUS_CHARGING,-1},
		{99,POWER_SUPPLY_STATUS_CHARGING,-1},
		{100,POWER_SUPPLY_STATUS_CHARGING,POWER_SUPPLY_STATUS_FULL},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_2 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_3 */
	{
		{90,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{89,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{85,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{82,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{81,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{80,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{79,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{75,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{72,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{71,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{70,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_4 */
	{
		{70,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{69,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{68,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{67,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{66,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{65,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{64,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{63,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{62,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{61,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{60,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_5 */
	{
		{60,POWER_SUPPLY_STATUS_CHARGING,-1},
		{61,POWER_SUPPLY_STATUS_CHARGING,-1},
		{62,POWER_SUPPLY_STATUS_CHARGING,-1},
		{63,POWER_SUPPLY_STATUS_CHARGING,-1},
		{64,POWER_SUPPLY_STATUS_CHARGING,-1},
		{65,POWER_SUPPLY_STATUS_CHARGING,-1},
		{66,POWER_SUPPLY_STATUS_CHARGING,-1},
		{67,POWER_SUPPLY_STATUS_CHARGING,-1},
		{68,POWER_SUPPLY_STATUS_CHARGING,-1},
		{69,POWER_SUPPLY_STATUS_CHARGING,-1},
		{70,POWER_SUPPLY_STATUS_CHARGING,-1},
		{-1,-1,-1}
	}
};
struct charging_control_debug_param poweroff_charging_param[CC_DBG_STATE_MAX][CC_DBG_COUNT_MAX] = 
{
	/* poweroff_charging_param */
	/* CHARGING_CONTROL_STATE_0 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_1 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_2 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_3 */
	{	{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},{-1,-1,-1},
		{-1,-1,-1},{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_4 */
	{
		{100,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{95,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{90,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{89,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{87,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{85,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{84,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{83,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{82,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{81,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{80,POWER_SUPPLY_STATUS_NOT_CHARGING,-1},
		{-1,-1,-1}
	},
	/* CHARGING_CONTROL_STATE_5 */
	{
		{80,POWER_SUPPLY_STATUS_CHARGING,-1},
		{81,POWER_SUPPLY_STATUS_CHARGING,-1},
		{85,POWER_SUPPLY_STATUS_CHARGING,-1},
		{90,POWER_SUPPLY_STATUS_CHARGING,-1},
		{94,POWER_SUPPLY_STATUS_CHARGING,-1},
		{95,POWER_SUPPLY_STATUS_CHARGING,-1},
		{96,POWER_SUPPLY_STATUS_CHARGING,-1},
		{97,POWER_SUPPLY_STATUS_CHARGING,-1},
		{98,POWER_SUPPLY_STATUS_CHARGING,-1},
		{99,POWER_SUPPLY_STATUS_CHARGING,-1},
		{100,POWER_SUPPLY_STATUS_CHARGING,POWER_SUPPLY_STATUS_FULL},
		{-1,-1,-1}
	}
};

#endif /* CHARGING_CONTROL_DEBUG */

int charger_manager_setForceDischarge(struct charger_manager *cm,int enable);
int charger_manager_getVBUSConnect(struct charger_manager *cm);
int charger_manager_getChargeStatus(struct charger_manager *cm);
int charger_manager_setChargeEnable(struct charger_manager *cm,int enable);
int charger_manager_getChargeEnable(struct charger_manager *cm);
int charger_manager_check_battery_full_status(struct charger_manager *cm);

static int charger_manager_control_charging(struct charger_manager *cm);
static int charger_manager_control_poweroff_charging(struct charger_manager *cm);
static int charger_manager_control_longlife(struct charger_manager *cm);
static int charger_manager_control_charge_save(struct charger_manager *cm);

int charger_manager_setForceDischarge(struct charger_manager *cm,int enable)
{
	union power_supply_propval pval;
	int ret;

	if (cm->charger_psy == NULL)
		return -1;

	cm->discharge_status = (enable != 0 ? true : false);
#define CURRENT_100_MA		100
	pval.intval =  enable != 0 ? CURRENT_100_MA : 0;

	ret = power_supply_set_property(cm->charger_psy, POWER_SUPPLY_PROP_FORCE_DISCHARGE, &pval);

	return ret;
}

int charger_manager_getVBUSConnect(struct charger_manager *cm)
{
	union power_supply_propval pval;
	int ret;
	int vbus_connection = 0;

	if (cm->charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(cm->charger_psy, POWER_SUPPLY_PROP_VBUS_CONNECTION, &pval);
	if (ret ) {
		dev_err(cm->dev, "%s POWER_SUPPLY_PROP_VBUS_CONNECTION ret=%d val=%d\n",__FUNCTION__,ret,pval.intval);
		return -2;
	}

	vbus_connection = pval.intval;

	if (!vbus_connection) {
		charger_manager_reset_flag(cm);
	}

	return vbus_connection;
}

int charger_manager_getChargeStatus(struct charger_manager *cm)
{
	union power_supply_propval pval;
	int ret;
	int charge_status = POWER_SUPPLY_STATUS_UNKNOWN;

	if (cm->charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(cm->charger_psy, POWER_SUPPLY_PROP_STATUS, &pval);
	if (ret ) {
		dev_info(cm->dev, "%s POWER_SUPPLY_PROP_STATUS ret=%d val=%d\n",__FUNCTION__,ret,pval.intval);
		return -2;
	}

	charge_status = pval.intval;
#ifdef CHARGING_CONTROL_DEBUG
	if (cm->charging_control_debug == 1) {
		int debug_status;
		debug_status = g_debug_info.debug_table[g_debug_info.debug_state][g_debug_info.debug_count].debug_ChargeStatus;
		if ((debug_status != -1) && (debug_status != charge_status)) {
			dev_err(cm->dev, "%s debug in state=%d count=%d status=%d\n",__FUNCTION__,g_debug_info.debug_state,g_debug_info.debug_count,debug_status);
			dev_err(cm->dev,  "charge_status =%d write =%d\n",charge_status,debug_status);
			charge_status = debug_status;
		}
	}
#endif /* CHARGING_CONTROL_DEBUG */
	if (charge_status == POWER_SUPPLY_STATUS_CHARGING)
		return 1;

	return 0;
}

int charger_manager_setChargeEnable(struct charger_manager *cm,int enable)
{
	union power_supply_propval pval;
	int ret;

	if (cm->charger_psy == NULL)
		return -1;

	pval.intval = enable != 0 ? true : false;
	ret = power_supply_set_property(cm->charger_psy, POWER_SUPPLY_PROP_CHARGING_ENABLED, &pval);

	return ret;
}

int charger_manager_getChargeStop(struct charger_manager *cm)
{
	int ret;
	int enable;
	union power_supply_propval pval;

	if (cm->charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(cm->charger_psy, POWER_SUPPLY_PROP_CHARGE_STOP, &pval);
	if (ret ) {
		dev_err(cm->dev, "%s POWER_SUPPLY_PROP_CHARGE_STOP ret=%d val=%d\n",__FUNCTION__,ret,pval.intval);
		return -2;
	}

	enable = pval.intval;
	return enable;
}

int charger_manager_getChargeEnable(struct charger_manager *cm)
{
	int ret;
	int enable;
	union power_supply_propval pval;

	if (cm->charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(cm->charger_psy, POWER_SUPPLY_PROP_CHARGING_ENABLED, &pval);
	if (ret ) {
		dev_err(cm->dev, "%s POWER_SUPPLY_PROP_CHARGING_ENABLED ret=%d val=%d\n",__FUNCTION__,ret,pval.intval);
		return -2;
	}

	enable = pval.intval;
	return enable;
}

int charger_manager_check_battery_full_status(struct charger_manager *cm)
{
	union power_supply_propval pval;
	int ret;
	int status;

	if (cm->battery_psy == NULL)
		return -1;

	ret = power_supply_get_property(cm->battery_psy, POWER_SUPPLY_PROP_STATUS, &pval);
	if (ret ) {
		dev_err(cm->dev, "%s POWER_SUPPLY_PROP_STATUS ret=%d val=%d\n",__FUNCTION__,ret,pval.intval);
		return -2;
	}

	status = pval.intval;
#ifdef CHARGING_CONTROL_DEBUG
	if (cm->charging_control_debug == 1) {
		int debug_FullCharge;
		debug_FullCharge = g_debug_info.debug_table[g_debug_info.debug_state][g_debug_info.debug_count].debug_FullCharge;
		if ((debug_FullCharge != -1) && (debug_FullCharge != status)) {
			dev_err(cm->dev, "%s debug in state=%d count=%d debug_FullCharge=%d\n",__FUNCTION__,g_debug_info.debug_state,g_debug_info.debug_count,debug_FullCharge);
			dev_err(cm->dev,  "status=0x%x write debug_FullCharge=0x%x\n",status,debug_FullCharge);
			status = debug_FullCharge;
		}
	}
#endif /* CHARGING_CONTROL_DEBUG */
	if (status == POWER_SUPPLY_STATUS_FULL)
		return 1;

	return 0;
}

int charger_manager_init(struct charger_manager *cm)
{
	dev_info(cm->dev, "%s\n",__FUNCTION__);
	cm->discharge_status = false;
	cm->power_on_state = true;

	cm->charging_control_state = CHARGING_CONTROL_STATE_0;
	cm->charging_control_start_t = 0;
	cm->polling_interval = POLLING_INTERVAL;
	cm->dischg_counter = DEFAULT_COUNT_TO_FORCED_DISCHG;
	cm->dischg_start_period = DEFAULT_COUNT_TO_FORCED_DISCHG;

#ifdef CHARGING_CONTROL_DEBUG
	cm->charging_control_debug = 0;
#endif /* CHARGING_CONTROL_DEBUG */

	cm->charge_save = CHARGE_SAVE_DISABLE;
	cm->longlife = LONGLIFE_DISABLE;

	return 0;
}

void
charger_manager_reset_flag(struct charger_manager *cm)
{
	if (cm->charging_control_state != CHARGING_CONTROL_STATE_0)
		dev_info(cm->dev, "Clear control flag state=%d\n",cm->charging_control_state);

	charger_manager_setForceDischarge(cm,0);
	charger_manager_setChargeEnable(cm,1);
	cm->charging_control_state = CHARGING_CONTROL_STATE_0;
	cm->charging_control_start_t = 0;
}

#ifdef CHARGING_CONTROL_DEBUG
int charger_manager_set_debug(struct charger_manager *cm,int enable)
{
	dev_info(cm->dev, "%s enable=%d\n",__FUNCTION__,enable);
	if (enable == 1) {
		cm->charging_control_debug = 1;
		g_debug_info.debug_state = CHARGING_CONTROL_STATE_0;
		g_debug_info.debug_count = 0;
		if(cm->charge_save) {
			memcpy(&g_debug_info.debug_table,&charge_save_param,sizeof(g_debug_info.debug_table));
		} else if (cm->longlife) {
			memcpy(&g_debug_info.debug_table,&longlife_param,sizeof(g_debug_info.debug_table));
		} else {
			if (cm->power_on_state == false) {
				memcpy(&g_debug_info.debug_table,&poweroff_charging_param,sizeof(g_debug_info.debug_table));
			} else {
				memcpy(&g_debug_info.debug_table,&charging_param,sizeof(g_debug_info.debug_table));
			}
		}
	} else {
		cm->charging_control_debug = 0;
	}

	return 0;
}
#endif /* CHARGING_CONTROL_DEBUG */

int charger_manager_get_force_discharge_info(struct charger_manager *cm)
{
	int ret = 0;
	dev_info(cm->dev, "%s\n",__FUNCTION__);

	if ((cm->charging_control_state >= CHARGING_CONTROL_STATE_3) ||
		(cm->charging_control_state == CHARGING_CONTROL_STATE_0) ||
		(cm->power_on_state == false) ||
		(cm->charge_save != CHARGE_SAVE_DISABLE) ||
		(cm->longlife != LONGLIFE_DISABLE)) {
		ret = 1;
	}

	return ret;
}

int charger_manager_set_force_discharge_req(struct charger_manager *cm)
{
	int ret;
	
	ret = charger_manager_get_force_discharge_info(cm);
	if (ret == 1)
		return -1;

	dev_info(cm->dev, "POWER_SUPPLY_PROP_FORCE_DISCHARGE  Force Discharge start\n");

	charger_manager_setChargeEnable(cm,0);
	charger_manager_setForceDischarge(cm,1);
	cm->dischg_counter = cm->dischg_start_period;
	cm->charging_control_state = CHARGING_CONTROL_STATE_3;
	dev_info(cm->dev, "->CHARGING_CONTROL_STATE_3 line=%d\n",__LINE__);

	return 0;
}


static int charger_manager_control_charging(struct charger_manager *cm)
{
	int vbus_stat,charge_status,battery_full_status;
	int charge_enable;
	int charge_stop;
	
	vbus_stat = charger_manager_getVBUSConnect(cm);
	if (vbus_stat <= 0) {
		if (vbus_stat) {
			dev_err(cm->dev,"%s %u: vbus connect get error.\n",	__func__, __LINE__);
			return -1;
		}
		return 0;
	}

	if (cm->charging_control_start_t == 0) {
		cm->charging_control_start_t = jiffies + (cm->polling_interval * HZ);
		dev_info(cm->dev, "update charging_control_start_t=0x%lx\n",cm->charging_control_start_t);
		return 0;
	}

	charge_status = charger_manager_getChargeStatus(cm);
	if (charge_status < 0) {
		dev_err(cm->dev, "%s charge status get error. charge_status=%d\n",__func__,charge_status);
		charger_manager_setChargeEnable(cm,1);
		return -2;
	}

	charge_enable = charger_manager_getChargeEnable(cm);
	if (charge_enable < 0) {
		dev_err(cm->dev, "%s charge enable get error. charge_enable=%d\n",__func__,charge_enable);
		charger_manager_setChargeEnable(cm,1);
		return -3;
	}

	switch(cm->charging_control_state) {
		case CHARGING_CONTROL_STATE_0:
			dev_info(cm->dev, "CHARGING_CONTROL_STATE_0->1 check\n");
			if (time_after(cm->charging_control_start_t, jiffies) ) {
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_1 line=%d\n",__LINE__);
				cm->charging_control_state = CHARGING_CONTROL_STATE_1;
				charger_manager_setForceDischarge(cm,0);
				charger_manager_setChargeEnable(cm,1);
			}
			break;
		case CHARGING_CONTROL_STATE_1:
			if (charge_status == 1) {
				battery_full_status = charger_manager_check_battery_full_status(cm);
				if (battery_full_status == 1) {
					dev_info(cm->dev,"%s %u: batter Full status\n",__func__, __LINE__);
					charger_manager_setChargeEnable(cm,0);
					cm->charging_control_state = CHARGING_CONTROL_STATE_2;
					dev_info(cm->dev, "->CHARGING_CONTROL_STATE_2 line=%d\n",__LINE__);
					cm->dischg_counter = cm->dischg_start_period;
				}
			} else {
				if (charge_enable == 0) {
					charge_stop = charger_manager_getChargeStop(cm);
					if (charge_stop == 0) {
						/* discharge status */
						dev_info(cm->dev, "CHARGING_CONTROL_STATE_1 detect no charge status!! start discharge timer\n");
						charger_manager_setChargeEnable(cm,0);
						cm->dischg_counter = cm->dischg_start_period;
						cm->charging_control_state = CHARGING_CONTROL_STATE_2;
						dev_info(cm->dev, "->CHARGING_CONTROL_STATE_2 line=%d\n",__LINE__);
					}
				}
			}
			break;
		case CHARGING_CONTROL_STATE_2:
			if (cm->dischg_counter-- <= 0) {
				charger_manager_setForceDischarge(cm,1);
				dev_info(cm->dev, "Force Discharge start\n");
				cm->dischg_counter = cm->dischg_start_period;
				cm->charging_control_state = CHARGING_CONTROL_STATE_3;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_3 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charging status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_2 detect charging status!!\n");
					charger_manager_reset_flag(cm);
				}
			}
			break;
		case CHARGING_CONTROL_STATE_3:
			if (cm->battery_remain <= 70) {
				charger_manager_setForceDischarge(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charging status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_3 detect charging status!!\n");
					charger_manager_reset_flag(cm);
				}
			}
			break;
		case CHARGING_CONTROL_STATE_4:
			if (cm->battery_remain <= 60) {
				dev_info(cm->dev,
					 "restart charging(batt=%d%%)\n",
					 cm->battery_remain);
				printk("CC (re)start charge\n");
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charge status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_4 detect charging status!!\n");
					charger_manager_reset_flag(cm);
				}
			}
			break;
		case CHARGING_CONTROL_STATE_5:
			if (cm->battery_remain >= 70) {
				dev_info(cm->dev,
					 "stop charging(batt=%d%%)\n",
					 cm->battery_remain);
				dev_info(cm->dev,"CC stop charge\n");
				charger_manager_setChargeEnable(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				if (charge_status == 0) {
					/* discharge status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_5 detect discharge status!!\n");
				}
			}
			break;
		default:
			dev_err(cm->dev, "%s invalid state=%d\n",__FUNCTION__,cm->charging_control_state);
			break;
	}

	return 0;
}

static int charger_manager_control_poweroff_charging(struct charger_manager *cm)
{
	int vbus_stat,charge_status,battery_full_status;

	vbus_stat = charger_manager_getVBUSConnect(cm);
	if (vbus_stat <= 0) {
		if (vbus_stat) {
			dev_err(cm->dev,"%s %u: vbus connect get error.\n",	__func__, __LINE__);
			return -1;
		}
		return 0;
	}

	charge_status = charger_manager_getChargeStatus(cm);
	if (charge_status < 0) {
		dev_err(cm->dev, "%s charge status get error. charge_status=%d\n",__func__,charge_status);
		charger_manager_setChargeEnable(cm,1);
		return -2;
	}

	switch(cm->charging_control_state) {
		case CHARGING_CONTROL_STATE_0:
			if (cm->battery_remain == 100) {
				charger_manager_setChargeEnable(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			}
			break;
		case CHARGING_CONTROL_STATE_4:
			if (cm->battery_remain <= 80) {
				dev_info(cm->dev,
					 "restart charging(batt=%d%%)\n",
					 cm->battery_remain);
				dev_info(cm->dev, "power off (re)start charge\n");
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charging status*/
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_4 detect charging status!!\n");
				}
			}
			break;
		case CHARGING_CONTROL_STATE_5:
			if (charge_status == 1) {
				battery_full_status = charger_manager_check_battery_full_status(cm);
				if (battery_full_status == 1) {
					dev_info(cm->dev,
						 "power off stop charging(batt=%d%%)\n",
						 cm->battery_remain);
					charger_manager_setChargeEnable(cm,0);
					cm->charging_control_state = CHARGING_CONTROL_STATE_4;
					dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
				}
			} else {
				if (charge_status == 0) {
					/* discharge status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_5 detect discharge status!!\n");
				}
			}
			break;
		default:
			dev_err(cm->dev, "%s invalid state=%d\n",__FUNCTION__,cm->charging_control_state);
			break;
	}

	return 0;
}

static int charger_manager_control_longlife(struct charger_manager *cm)
{
	int vbus_stat,charge_status;

	vbus_stat = charger_manager_getVBUSConnect(cm);
	if (vbus_stat <= 0) {
		if (vbus_stat) {
			dev_err(cm->dev,"%s %u: vbus connect get error.\n",	__func__, __LINE__);
			return -1;
		}
		return 0;
	}

	if (cm->charging_control_start_t == 0) {
		cm->charging_control_start_t = jiffies + (cm->polling_interval * HZ);
		dev_info(cm->dev, "update charging_control_start_t=0x%lx\n",cm->charging_control_start_t);
		return 0;
	}

	charge_status = charger_manager_getChargeStatus(cm);
	if (charge_status < 0) {
		dev_err(cm->dev, "%s charge status get error. charge_status=%d\n",__func__,charge_status);
		charger_manager_setChargeEnable(cm,1);
		return -2;
	}

	switch(cm->charging_control_state) {
		case CHARGING_CONTROL_STATE_0:
			if (cm->battery_remain >= 70) {
				charger_manager_setChargeEnable(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			}
			break;
		case CHARGING_CONTROL_STATE_4:
			if (cm->battery_remain <= 60) {
				dev_info(cm->dev,
					 "restart charging(batt=%d%%)\n",
					 cm->battery_remain);
				dev_info(cm->dev, "LL (re)start charge\n");
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charging status*/
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_4 detect charging status!!\n");
				}
			}
			break;
		case CHARGING_CONTROL_STATE_5:
			if (cm->battery_remain >= 70) {
				dev_info(cm->dev,
					 "stop charging(batt=%d%%)\n",
					 cm->battery_remain);
				dev_info(cm->dev, "LL stop charge\n");
				charger_manager_setChargeEnable(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				if (charge_status == 0) {
					/* discharge status */
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_5 detect discharge status!!\n");
				}
			}
			break;
		default:
			dev_err(cm->dev, "%s invalid state=%d\n",__FUNCTION__,cm->charging_control_state);
			break;
	}

	return 0;
}

static int charger_manager_control_charge_save(struct charger_manager *cm)
{
	int vbus_stat,charge_status;

	vbus_stat = charger_manager_getVBUSConnect(cm);
	if (vbus_stat <= 0) {
		if (vbus_stat) {
			dev_err(cm->dev,
				"%s %u: vbus connect get error.\n",
				__func__, __LINE__);
			return -1;
		}
		return 0;
	}

	if (cm->charging_control_start_t == 0) {
		cm->charging_control_start_t = jiffies + (cm->polling_interval * HZ);
		dev_info(cm->dev, "update charging_control_start_t=0x%lx\n",cm->charging_control_start_t);
		return 0;
	}

	charge_status = charger_manager_getChargeStatus(cm);
	if (charge_status < 0) {
		dev_err(cm->dev, "%s charge status get error. charge_status=%d\n",__func__,charge_status);
		charger_manager_setChargeEnable(cm,1);
		return -2;
	}

	switch(cm->charging_control_state) {
		case CHARGING_CONTROL_STATE_0:
			dev_info(cm->dev, "CHARGING_CONTROL_STATE_0->1 check\n");
			if (time_after(cm->charging_control_start_t, jiffies) ) {
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_1 line=%d\n",__LINE__);
				cm->charging_control_state = CHARGING_CONTROL_STATE_1;
				charger_manager_setForceDischarge(cm,0);
				charger_manager_setChargeEnable(cm,1);
			}
			break;
		case CHARGING_CONTROL_STATE_1:
			if (cm->battery_remain > 60) {
				cm->charging_control_state = CHARGING_CONTROL_STATE_3;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_3 line=%d\n",__LINE__);
				charger_manager_setChargeEnable(cm,0);
				charger_manager_setForceDischarge(cm,1);
			} else {
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			}
			break;
		case CHARGING_CONTROL_STATE_3:
			if ((cm->battery_remain <= 60) || (cm->power_on_state == false)) {
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
				charger_manager_setForceDischarge(cm,0);
				charger_manager_setChargeEnable(cm,0);
			}
			break;
		case CHARGING_CONTROL_STATE_4:
			if (cm->battery_remain <= 50) {
				dev_info(cm->dev,
					 "restart charging(batt=%d%%)\n",
					 cm->battery_remain);
				dev_info(cm->dev, "CS (re)start charge\n");
				charger_manager_setChargeEnable(cm,1);
				cm->charging_control_state = CHARGING_CONTROL_STATE_5;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_5 line=%d\n",__LINE__);
			} else {
				if (charge_status == 1) {
					/* charge status*/
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_4 detect charging status!!\n");
				}
			}
			break;
		case CHARGING_CONTROL_STATE_5:
			if (cm->battery_remain >= 60) {
				dev_info(cm->dev,"stop charging(batt=%d%%)\n",cm->battery_remain);
				dev_info(cm->dev, "CS stop charge\n");
				charger_manager_setChargeEnable(cm,0);
				cm->charging_control_state = CHARGING_CONTROL_STATE_4;
				dev_info(cm->dev, "->CHARGING_CONTROL_STATE_4 line=%d\n",__LINE__);
			} else {
				if (charge_status == 0) {
					/* discharge status*/
					dev_err(cm->dev, "CHARGING_CONTROL_STATE_5 detect no charging status!!\n");
				}
			}
			break;
		default:
			dev_err(cm->dev, "%s invalid state=%d\n",__FUNCTION__,cm->charging_control_state);
			break;
	}

	return 0;
}

int charger_manager_control(struct charger_manager *cm)
{
#ifdef CHARGING_CONTROL_DEBUG
	int debug_remain = -1;
#endif /* CHARGING_CONTROL_DEBUG */

	dev_info(cm->dev, "charging control state=%d battery_remain=%d\n",cm->charging_control_state,cm->battery_remain);

#ifdef CHARGING_CONTROL_DEBUG
	if ((cm->charging_control_debug == 1) && (cm->charging_control_state != CHARGING_CONTROL_STATE_0)) {
		dev_err(cm->dev, "%s debug in state=%d count=%d\n",__FUNCTION__,g_debug_info.debug_state,g_debug_info.debug_count);
		if (cm->charging_control_state != g_debug_info.debug_state) {
			dev_info(cm->dev, "debug state change =%d -> %d \n",g_debug_info.debug_state,cm->charging_control_state);
			g_debug_info.debug_state = cm->charging_control_state;
			/* reset */
			g_debug_info.debug_count = 0;
		}

		debug_remain = g_debug_info.debug_table[g_debug_info.debug_state][g_debug_info.debug_count].debug_remain;
		if (debug_remain != -1) {
			dev_err(cm->dev, "battery_remain=%d -> write =%d\n",cm->battery_remain,debug_remain);
			cm->battery_remain = debug_remain;
		}
	}
#endif /* CHARGING_CONTROL_DEBUG */

	if(cm->charge_save) {
		charger_manager_control_charge_save(cm);
	} else if (cm->longlife) {
		charger_manager_control_longlife(cm);
	} else {
		if (cm->power_on_state == false) {
			charger_manager_control_poweroff_charging(cm);
		} else {
			charger_manager_control_charging(cm);
		}
	}
#ifdef CHARGING_CONTROL_DEBUG
	if (cm->charging_control_debug == 1) {
		if (debug_remain != -1) {
			g_debug_info.debug_count++;
		}
	}
#endif

	return 0;
}

