// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2020 MediaTek Inc.
 */

#include <linux/err.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <md_cooling.h>
#include <mtk_thermal_trace.h>

static unsigned long find_max_mutt_cc_level(unsigned int id,
				unsigned long lv)
{
	struct md_cooling_device *md_cdev;
	unsigned long final_lv = lv;
	int i, pa_num;

	pa_num = get_pa_num();

	for (i = 0; i < pa_num; i++) {
		if (i == id)
			continue;

		md_cdev = get_md_cdev(MD_COOLING_TYPE_MUTT_CC, i);
		if (md_cdev && md_cdev->target_level)
			return 1;

	}

	return final_lv;
}

static int md_cooling_mutt_cc_get_max_state(
	struct thermal_cooling_device *cdev, unsigned long *state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;

	*state = md_cdev->max_level;

	return 0;
}

static int md_cooling_mutt_cc_get_cur_state(
	struct thermal_cooling_device *cdev, unsigned long *state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;

	*state = md_cdev->target_level;

	return 0;
}

static int md_cooling_mutt_cc_set_cur_state(
		struct thermal_cooling_device *cdev, unsigned long state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;
	struct device *dev = (struct device *)md_cdev->dev_data;
	enum md_status status;
	enum md_cc_control_status new_cc_status;
	unsigned int msg;
	unsigned long cc_state;
	int ret = 0;

	/* Request state should be less than max_level */
	if (WARN_ON(state > md_cdev->max_level))
		return -EINVAL;

	if (md_cdev->target_level == state)
		return 0;

	status = get_md_status();
	if (is_md_off(status)) {
		dev_info(dev, "skip mutt_cc due to MD is off\n");
		md_cdev->target_level = MD_COOLING_UNLIMITED_LV;
		return 0;
	}
	if (is_mutt_enabled(status)) {
		dev_info(dev, "skip cc control due to MUTT PA is enabled\n");
		trace_md_mutt_cc(md_cdev, status);
		return -EACCES;
	}

	msg = mutt_cc_lv_to_tmc_msg(md_cdev->pa_id, state);
	ret = send_throttle_msg(msg);
	if (!ret)
		md_cdev->target_level = state;

	cc_state = find_max_mutt_cc_level(md_cdev->pa_id,
				state);
	new_cc_status = (cc_state) ? MD_CC_CONTROL_ENABLED
				: MD_CC_CONTROL_DISABLED;
	set_md_cc_control_status(new_cc_status);
	dev_info(dev, "%s: set lv = %ld done\n", md_cdev->name, state);
	trace_md_mutt_cc(md_cdev, status);

	return ret;
}

static const struct of_device_id md_cooling_mutt_cc_of_match[] = {
	{ .compatible = "mediatek,md-cooler-mutt_cc", },
	{},
};
MODULE_DEVICE_TABLE(of, md_cooling_mutt_cc_of_match);

static struct thermal_cooling_device_ops md_cooling_mutt_cc_ops = {
	.get_max_state		= md_cooling_mutt_cc_get_max_state,
	.get_cur_state		= md_cooling_mutt_cc_get_cur_state,
	.set_cur_state		= md_cooling_mutt_cc_set_cur_state,
};

static int md_cooling_mutt_cc_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device *dev = &pdev->dev;
	int ret = -1;

	if (!np) {
		dev_err(dev, "MD cooler DT node not found\n");
		return -ENODEV;
	}

	ret = md_cooling_register(np,
				MD_COOLING_TYPE_MUTT_CC,
				MAX_MUTT_CC_LV,
				NULL,
				&md_cooling_mutt_cc_ops,
				dev);
	if (ret) {
		dev_err(dev, "register mutt-cc cdev failed!\n");
		return ret;
	}

	return ret;
}

static int md_cooling_mutt_cc_remove(struct platform_device *pdev)
{
	md_cooling_unregister(MD_COOLING_TYPE_MUTT);

	return 0;
}

static struct platform_driver md_cooling_mutt_cc_driver = {
	.probe = md_cooling_mutt_cc_probe,
	.remove = md_cooling_mutt_cc_remove,
	.driver = {
		.name = "mtk-md-cooling-mutt-cc",
		.of_match_table = md_cooling_mutt_cc_of_match,
	},
};
module_platform_driver(md_cooling_mutt_cc_driver);

MODULE_AUTHOR("Meng-Hsun Chou <meng-hsun.chou@mediatek.com>");
MODULE_DESCRIPTION("Mediatek modem cooling UL_control driver");
MODULE_LICENSE("GPL v2");
