// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2019 MediaTek Inc.
 * Author: Kai Chieh Chuang <kaichieh.chuang@mediatek.com>
 */

#include <linux/init.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/miscdevice.h>
#include <linux/mm.h>
#include <linux/dma-mapping.h>
#include <linux/notifier.h>

#include <mtk_ccci_common.h>

#define USIP_EMP_IOC_MAGIC 'D'
#define GET_USIP_EMI_SIZE _IOWR(USIP_EMP_IOC_MAGIC, 0xF0, unsigned long long)

#define NUM_MPU_REGION 3

int EMI_TABLE[3][3]
	= {{0, 0, 0x30000}, {1, 0x30000, 0x8000}, {2, 0x38000, 0x28000} };

enum {
	SP_EMI_AP_USIP_PARAMETER,
	SP_EMI_ADSP_USIP_PHONECALL,
	SP_EMI_ADSP_USIP_SMARTPA
};

enum {
	SP_EMI_TYPE,
	SP_EMI_OFFSET,
	SP_EMI_SIZE
};

struct usip_info {
	bool memory_ready;
	size_t memory_size;

	void *memory_area;
	dma_addr_t memory_addr;
	phys_addr_t addr_phy;
};

static struct usip_info usip;

static long usip_ioctl(struct file *fp, unsigned int cmd, unsigned long arg)
{
	int ret = 0;
	long size_for_spe = 0;

	pr_info("%s(), cmd 0x%x, arg %lu, memory_size = %ld addr_phy = 0x%llx\n",
		 __func__, cmd, arg, usip.memory_size, usip.addr_phy);

	size_for_spe = EMI_TABLE[SP_EMI_AP_USIP_PARAMETER][SP_EMI_SIZE];
	switch (cmd) {

	case GET_USIP_EMI_SIZE:
		if (!usip.memory_ready) {
			pr_info("no phy addr from ccci");
			ret = -ENODEV;
		} else if (copy_to_user((void __user *)arg, &size_for_spe,
			   sizeof(size_for_spe))) {
			pr_warn("Fail copy to user Ptr:%p, r_sz:%zu",
				(char *)&size_for_spe, sizeof(size_for_spe));
			ret = -1;
		}
		break;

	default:
		pr_debug("%s(), default\n", __func__);
		break;
	}
	return ret;
}

#ifdef CONFIG_COMPAT
static long usip_compat_ioctl(struct file *fp, unsigned int cmd,
			      unsigned long arg)
{
	long ret;

	if (!fp->f_op || !fp->f_op->unlocked_ioctl)
		return -ENOTTY;
	ret = fp->f_op->unlocked_ioctl(fp, cmd, arg);
	if (ret < 0)
		pr_err("%s(), fail, ret %ld, cmd 0x%x, arg %lu\n",
		       __func__, ret, cmd, arg);
	return ret;
}
#endif

int usip_mmap_data(struct usip_info *usip, struct vm_area_struct *area)
{
	long size;
	unsigned long offset;
	size_t align_bytes = PAGE_ALIGN(usip->memory_size);
	unsigned long pfn;
	int ret;

	pr_info("%s(), memory ready %d, size %zu, align_bytes %zu\n",
		__func__, usip->memory_ready, usip->memory_size, align_bytes);

	if (!usip->memory_ready)
		return -EBADFD;

	size = area->vm_end - area->vm_start;
	offset = area->vm_pgoff << PAGE_SHIFT;

	if ((size_t)size > align_bytes)
		return -EINVAL;
	if (offset > align_bytes - size)
		return -EINVAL;

	/*area->vm_ops = &usip_vm_ops_data;*/
	/*area->vm_private_data = usip;*/

	pfn = usip->addr_phy >> PAGE_SHIFT;
	/* ensure that memory does not get swapped to disk */
	area->vm_flags |= VM_IO;
	/* ensure non-cacheable */
	area->vm_page_prot = pgprot_noncached(area->vm_page_prot);
	ret = remap_pfn_range(area, area->vm_start,
			pfn, size, area->vm_page_prot);
	if (ret)
		pr_err("%s(), ret %d, remap failed 0x%lx, phys_addr %pa -> vm_start 0x%lx\n",
		       __func__, ret, pfn,
		       &usip->addr_phy,
		       area->vm_start);
	/*Comment*/
	smp_mb();

	return ret;
}

static int usip_mmap(struct file *file, struct vm_area_struct *area)
{
	unsigned long offset;
	struct usip_info *usip = file->private_data;

	pr_debug("%s(), vm_flags 0x%lx, vm_pgoff 0x%lx\n",
		 __func__, area->vm_flags, area->vm_pgoff);

	offset = area->vm_pgoff << PAGE_SHIFT;
	switch (offset) {
	default:
		return usip_mmap_data(usip, area);
	}
	return 0;
}
static void usip_get_addr(void)
{
#if IS_ENABLED(CONFIG_MTK_ECCCI_DRIVER)
	int size_o = 0;
	phys_addr_t phys_addr;

	phys_addr_t srw_base;
	unsigned int srw_size;
	phys_addr_t r_rw_base;
	unsigned int r_rw_size;

	phys_addr = get_smem_phy_start_addr(MD_SYS1,
					    SMEM_USER_RAW_USIP, &size_o);
	if (phys_addr == 0) {
		pr_info("%s(), cannot get emi addr from ccci", __func__);
		usip.memory_ready = false;
	} else {
		usip.memory_ready = true;

		get_md_resv_mem_info(MD_SYS1, &r_rw_base, &r_rw_size,
				     &srw_base, &srw_size);
		pr_info("%s(), 0x%llx %d 0x%llx %d 0x%llx", __func__,
			r_rw_base, r_rw_size, srw_base, srw_size, phys_addr);

		usip.memory_size = size_o;
		usip.addr_phy = phys_addr;
	}
#endif
}

static int usip_open(struct inode *inode, struct file *file)
{
	file->private_data = &usip;

	if (!usip.memory_ready) {
		usip_get_addr();
	}

	return 0;
}

static const struct file_operations usip_fops = {
	.owner = THIS_MODULE,
	.open = usip_open,
	.unlocked_ioctl = usip_ioctl,
#ifdef CONFIG_COMPAT
	.compat_ioctl = usip_compat_ioctl,
#endif
	.mmap = usip_mmap,
};

#if IS_ENABLED(CONFIG_MTK_ECCCI_DRIVER)
static struct miscdevice usip_miscdevice = {
	.minor      = MISC_DYNAMIC_MINOR,
	.name       = "usip",
	.fops       = &usip_fops,
};
#endif

static int __init usip_init(void)
{
	int ret;

	usip.memory_ready = false;
	usip.memory_size = 0;
	usip.addr_phy = 0;

#if IS_ENABLED(CONFIG_MTK_ECCCI_DRIVER)
	ret = misc_register(&usip_miscdevice);
	if (ret) {
		pr_err("%s(), cannot register miscdev on minor %d, ret %d\n",
		       __func__, usip_miscdevice.minor, ret);
		ret = -ENODEV;
	}
#else
	ret = -ENODEV;
#endif

	/* init usip info */
	usip.memory_addr = 0x11220000L;


#ifdef CONFIG_MTK_AURISYS_PHONE_CALL_SUPPORT
	adsp_A_register_notify(&audio_call_notifier);
#endif

	return ret;
}

static void __exit usip_exit(void)
{
#if IS_ENABLED(CONFIG_MTK_ECCCI_DRIVER)
	misc_deregister(&usip_miscdevice);
#endif
}

module_init(usip_init);
module_exit(usip_exit);

MODULE_DESCRIPTION("Mediatek uSip memory control");
MODULE_LICENSE("GPL v2");

