#include <typedefs.h>
#include <platform.h>
#include <spmi.h>
#include <spmi_sw.h>
#include <regulator/mtk_regulator.h>
#include <regulator/mtk_regulator_errno.h>
#include <pmic.h>
#include <sec_devinfo.h>
#include <rtc.h>
#include <mt6330.h>

#ifdef EXT_BUCK_MT6315
#include <mt6315-spmi.h>
#endif



static struct spmi_device *mt6330_sdev;

#define MT6330_SWCID 0x30

/* show vcore for MD before MD boot up */
static const struct mtk_regulator empty_regulator;
void wk_vcore_check(int check_vol)
{
	struct mtk_regulator reg_vcore = empty_regulator;
	int ret = 0;

	ret = mtk_regulator_get("vcore", &reg_vcore);
	if (ret)
		pal_log_err("mtk_regulator_get reg_vcore failed\n");
	pal_log_warn("vcore = %d uV\n", mtk_regulator_get_voltage(&reg_vcore));
}
//==============================================================================
// extern function
//==============================================================================

U32 get_dram_type(void)
{
	U32 val = 0;
	pmic_read_interface(PMIC_VM_MODE_ADDR, &val,
		PMIC_VM_MODE_MASK, PMIC_VM_MODE_SHIFT);

	return val;
}

//==============================================================================
// PMIC APIs
//==============================================================================

U32 pmic_read_interface_burst(U32 RegNum, u16 *val)
{
	int ret = 0;
	u16 rdata = 0;

	ret = spmi_ext_register_readl(mt6330_sdev, RegNum, &rdata, 2);
	if (ret < 0) {
		pal_log_info("%s: fail, addr = 0x%x, ret = %d\n",
			   __func__, RegNum, ret);
		return ret;
	}
	*val = rdata;

	return 0;
}


U32 pmic_read_interface(U32 RegNum, u8 *val, u8 MASK, u8 SHIFT)
{
	int ret = 0;
	u8 rdata = 0;

	ret = spmi_ext_register_readl(mt6330_sdev, RegNum, &rdata, 1);
	if (ret < 0) {
		pal_log_info("%s: fail, addr = 0x%x, ret = %d\n",
			   __func__, RegNum, ret);
		return ret;
	}
	rdata &= (MASK << SHIFT);
	*val = (rdata >> SHIFT);

	return 0;
}

U32 pmic_config_interface_burst(U32 RegNum, u16 val, u16 MASK, u16 SHIFT)
{
	int ret = 0;
	u8 org = 0;

	ret = spmi_ext_register_readl(mt6330_sdev, RegNum, &org, 2);
	if (ret < 0) {
		dbg_print("%s: fail, addr = 0x%x, ret = %d\n",
			  __func__, RegNum, ret);
		return ret;
	}
	org &= ~(MASK << SHIFT);
	org |= (val << SHIFT);

	ret = spmi_ext_register_writel(mt6330_sdev, RegNum, &org, 2);
	if (ret < 0) {
		dbg_print("%s: fail, addr = 0x%x, ret = %d\n",
			  __func__, RegNum, ret);
		return ret;
	}

	return 0;
}

U32 pmic_config_interface(U32 RegNum, u8 val, u8 MASK, u8 SHIFT)
{
	int ret = 0;
	u8 org = 0;

	ret = spmi_ext_register_readl(mt6330_sdev, RegNum, &org, 1);
	if (ret < 0) {
		dbg_print("%s: fail, addr = 0x%x, ret = %d\n",
			  __func__, RegNum, ret);
		return ret;
	}
	org &= ~(MASK << SHIFT);
	org |= (val << SHIFT);

	ret = spmi_ext_register_writel(mt6330_sdev, RegNum, &org, 1);
	if (ret < 0) {
		dbg_print("%s: fail, addr = 0x%x, ret = %d\n",
			  __func__, RegNum, ret);
		return ret;
	}

	return 0;
}

U8 upmu_get_reg_value(U32 reg)
{
	U32 ret = 0;
	U8 temp_val = 0;

	ret = pmic_read_interface(reg, &temp_val, 0xFF, 0x0);

	return temp_val;
}

U32 upmu_set_reg_value(U32 reg, U8 reg_val)
{
	U32 ret = 0;

	ret = pmic_config_interface(reg, reg_val, 0xFF, 0x0);

	return ret;
}

static U8 pmic_get_register_value(struct pmuflag_t pmureg)
{
	U32 ret;
	U8 val = 0;

	ret = pmic_read_interface(pmureg.addr, &val, pmureg.mask, pmureg.shift);

	return val;
}

static U32 pmic_set_register_value(struct pmuflag_t pmureg, U32 val)
{
	U32 ret;

	ret = pmic_config_interface(pmureg.addr, val, pmureg.mask, pmureg.shift);

	return ret;
}

U8 pmic_get_chip_version(void)
{
	U8 val = 0;

	pmic_read_interface(PMIC_SWCID0_ADDR, &val,
			    PMIC_SWCID0_MASK,
			    PMIC_SWCID0_SHIFT);

	return val;
}

int pmic_detect_powerkey(void)
{
	U32 ret = 0;
	U32 val = 0;

	ret = pmic_read_interface(PMIC_PWRKEY_DEB_ADDR, &val,
		PMIC_PWRKEY_DEB_MASK,
		PMIC_PWRKEY_DEB_SHIFT);

	return (1 - val);
}

U32 upmu_is_chr_det(void)
{
	U32 ret = 0;
	U32 val = 0;

	ret = pmic_read_interface(PMIC_RGS_CHRDET_ADDR, &val,
		PMIC_RGS_CHRDET_MASK,
		PMIC_RGS_CHRDET_SHIFT);
	pal_log_info("[PMIC]IsUsbCableIn %d\n", val);

	return val;
}

void PMIC_enable_long_press_reboot(void)
{
#if CFG_LPRST_SUPPORT
	/* PWRKEY */
	pmic_config_interface(PMIC_RG_PWRKEY_KEY_MODE_ADDR, 0x00,
			      PMIC_RG_PWRKEY_KEY_MODE_MASK,
			      PMIC_RG_PWRKEY_KEY_MODE_SHIFT);
	pmic_config_interface(PMIC_RG_PWRKEY_RST_EN_ADDR, 0x01,
			      PMIC_RG_PWRKEY_RST_EN_MASK,
			      PMIC_RG_PWRKEY_RST_EN_SHIFT);
	pmic_config_interface(PMIC_RG_PWRKEY_RST_TD_ADDR,
			      KPD_PMIC_LPRST_TD,
			      PMIC_RG_PWRKEY_RST_TD_MASK,
			      PMIC_RG_PWRKEY_RST_TD_SHIFT);
#else
	pmic_config_interface(PMIC_RG_PWRKEY_RST_EN_ADDR, 0x00,
			      PMIC_RG_PWRKEY_RST_EN_MASK,
			      PMIC_RG_PWRKEY_RST_EN_SHIFT);
#endif /* CFG_LPRST_SUPPORT */
}

U32 PMIC_VUSB_EN(void)
{
	int ret = 0;

	ret = pmic_config_interface(PMIC_RG_LDO_VUSB_EN_ADDR, 1,
		PMIC_RG_LDO_VUSB_EN_MASK,
		PMIC_RG_LDO_VUSB_EN_SHIFT);

	return ret;
}

/*
 * PMIC Export API
 */
static U32 g_sts_rtca;
static U32 g_sts_spar;
static U32 g_sts_crst;
static U32 g_just_rst;
static U32 g_pwrkey_release;
static U32 g_pmic_boot_status;

/* check if RTC Alarm happened at power on */
U32 is_pmic_rtc_alarm(void)
{
	return g_sts_rtca;
}
/* check if RTC SPAR happened at power on */
U32 is_pmic_spar(void)
{
	return g_sts_spar;
}
/* check if PMIC cold reset at previous power off */
U32 is_pmic_cold_reset(void)
{
	return g_sts_crst;
}

U32 is_pmic_long_press_reset(void)
{
	return g_just_rst;
}

U32 get_pmic_boot_status(void)
{
	return g_pmic_boot_status;
}

/*
 * used for measure long press counter time which unit is 32ms
 * unit of T is ms
 */
#define LONG_PRESS_COUNT_TIME(T) (T / 32)
/*
 * if detect PWRKEY pressed, check is PWRKEY short pressed
 * return 1 means PWRKEY is short pressed, it is not a valid boot up
 */
/*TODO check PWRKEY_PRESS_STS_H*/
U32 is_pwrkey_short_press(void)
{
	U32 val = 0;

	/* if detect PWRKEY pressed, check is PWRKEY short pressed */
	pmic_read_interface(PMIC_PWRKEY_LONG_PRESS_COUNT_ADDR, &val,
		PMIC_PWRKEY_LONG_PRESS_COUNT_MASK, PMIC_PWRKEY_LONG_PRESS_COUNT_SHIFT);
	pal_log_info("[%s] pwrkey_release=%d, LONG_PRESS_COUNT=%d\n",
		__func__, g_pwrkey_release, val << 5);
	if (g_pwrkey_release) {
		/* PWRKEY released during power up and press time not enough */
		if (val < LONG_PRESS_COUNT_TIME(800))
			return 1;
	}
	return 0;
}

int PMIC_POWER_HOLD(unsigned int hold)
{
	unsigned int val = 0;

	if (hold > 1) {
		pal_log_err("[PMIC] POWER_HOLD error\n");
		return -1;
	}

#if EXT_BUCK_MT6315
	if (hold == 0)
		mt6315_all_seq_off(1);
#endif
	pmic_config_interface(PMIC_RG_PWRHOLD_ADDR, hold,
		PMIC_RG_PWRHOLD_MASK, PMIC_RG_PWRHOLD_SHIFT);

	/* hold:  1(ON)  0(OFF) */
	pmic_read_interface(PMIC_RG_PWRHOLD_ADDR, &val,
		PMIC_RG_PWRHOLD_MASK, PMIC_RG_PWRHOLD_SHIFT);
	if (hold == 1)
		pal_log_info("[PMIC]POWER_HOLD :0x%x\n", val);
	return 0;
}

static void pmic_check_rst(unsigned int poff_sts, unsigned int top_sts)
{
	unsigned int val;

	/*
	 * TOP_RST_STATUS is used to indicate which reset happened
	 * If a reset happened, the corresponding bit will be clear
	 */
	val = upmu_get_reg_value(MT6330_TOP_RST_STATUS);
	if (val != 0x7F) {
		if (val == 0)
			pal_log_warn("[%s] PORSTB\n", __func__);
		else if (((val >> 1) & 1) == 0 && (poff_sts >> 9) & 1)
			pal_log_warn("[%s] DDLO_RSTB\n", __func__);
		else if (((val >> 2) & 1) == 0 && (poff_sts >> 0) & 1)
			pal_log_warn("[%s] UVLO_RSTB\n", __func__);
		val = upmu_set_reg_value(MT6330_TOP_RST_STATUS_SET, 0x78 | val);
	}
	if ((poff_sts >> 8) & 1)
		pal_log_warn("[%s] BWDT\n", __func__);
	if ((poff_sts >> 6) & 1)
		pal_log_warn("[%s] Long press shutdown\n", __func__);
	if ((poff_sts >> 5) & 1)
		pal_log_warn("[%s] Cold Reset\n", __func__);
	if ((poff_sts >> 13) & 1)
		pal_log_warn("[%s] PWRKEY short press\n", __func__);
	if ((poff_sts >> 10) & 1)
		pal_log_warn("[%s] AP Watchdog\n", __func__);

	if ((top_sts >> 4) & 1)
		pal_log_warn("[%s] PMIC_RESETB SHDN\n", __func__);
	if ((top_sts >> 5) & 1)
		pal_log_warn("[%s] PMIC_RESETB WRST\n", __func__);

	/* Long press shutdown status */
	pmic_read_interface(PMIC_JUST_PWRKEY_RST_ADDR, &g_just_rst,
		PMIC_JUST_PWRKEY_RST_MASK, PMIC_JUST_PWRKEY_RST_SHIFT);
	pal_log_warn("[PMIC]just_rst = %d\n", g_just_rst);
	/* LK mtk_key will clear long press status */
}

static void record_pmic_boot_status(unsigned int pon_sts, unsigned int poff_sts)
{
	/* Need to check MT6315 OT mask bit */
	unsigned int MT6315_OT_MASK = 0x1;
	unsigned int val = 0;

	val = upmu_get_reg_value(PMIC_RG_RSV_SWREG_ADDR);
	pal_log_warn("[PMIC]RG_RSV_SWREG[0x%x]=0x%x\n",
		     PMIC_RG_RSV_SWREG_ADDR, val);
	if ((poff_sts & 0x80) && (val & MT6315_OT_MASK)) {
		g_pmic_boot_status = 1; /* OC/PG */
		/* Clear OT status */
		pmic_config_interface(PMIC_RG_RSV_SWREG_ADDR, 0,
				      MT6315_OT_MASK, 0);
	return;
}
	if (poff_sts & 0x6)
		g_pmic_boot_status = 1; /* OC/PG */
	else if (poff_sts & 0x40)
		g_pmic_boot_status = 2; /* Long press */
	else if (poff_sts & 0x1 || poff_sts & 0x200)
		g_pmic_boot_status = 3; /* UVLO/DDLO */
}

void pmic_dbg_status(unsigned char option)
{
	/*--option = 1--*/
	/* UVLO off */
	/* Write 1 to clear PONSTS and POFFSTS */
	/* power not good */
	/* buck oc */
	/* thermal shutdown 150 */
	/* long press shutdown */
	/* WDTRST */
	/* CLK TRIM */
	/*--option = 0--*/
	/* After writing 1, need to write 0 back */

	int ret_val = 0;
	unsigned int pon_sts = 0;
	unsigned int poff_sts = 0;
	unsigned int top_sts = 0;

	if (option) {
	/* pwhold must set to 1 to update debug status */
		PMIC_POWER_HOLD(1);
	/*--Check if PWRKEY released during boot up--*/
		pmic_read_interface(PMIC_PUP_PKEY_RELEASE_ADDR, &g_pwrkey_release,
			PMIC_PUP_PKEY_RELEASE_MASK, PMIC_PUP_PKEY_RELEASE_SHIFT);
	/*--UVLO off--*/
		pal_log_info("[PMIC]TOP_RST_STATUS[0x%x]=0x%x\n",
			MT6330_TOP_RST_STATUS, upmu_get_reg_value(MT6330_TOP_RST_STATUS));
	/*special for RTC Alarm and SPAR*/
		pon_sts = upmu_get_reg_value(MT6330_PONSTS);
		g_sts_rtca = (pon_sts >> PMIC_STS_RTCA_SHIFT) & PMIC_STS_RTCA_MASK;
		g_sts_spar = (pon_sts >> PMIC_STS_SPAR_SHIFT) & PMIC_STS_SPAR_MASK;
		pal_log_warn("[PMIC]PONSTS[0x%x]=0x%x\n",
			MT6330_PONSTS, pon_sts);
	/*special for cold reset*/
		poff_sts = (upmu_get_reg_value(MT6330_POFFSTS0) << 8 |
			   (upmu_get_reg_value(MT6330_POFFSTS1)));
		pal_log_info("[PMIC]POFFSTS[0x%x]=0x%x\n",
			MT6330_POFFSTS1, poff_sts);
		g_sts_crst = (poff_sts >> PMIC_STS_CRST_SHIFT) & PMIC_STS_CRST_MASK;

	/*pmic_resetb*/
		top_sts = upmu_get_reg_value(MT6330_TOPSTATUS);
		pal_log_warn("[PMIC]TOPSTS[0x%x]=0x%x\n", MT6330_TOPSTATUS, top_sts);

#ifdef MTK_PLAT_DEBUG_HW
	/*--power not good--*/
		pal_log_info("[PMIC]PG_SDN_STS0[0x%x]=0x%x\n",
			MT6330_PG_SDN_STS0, upmu_get_reg_value(MT6330_PG_SDN_STS0));
		pal_log_info("[PMIC]PG_SDN_STS1[0x%x]=0x%x\n",
			MT6330_PG_SDN_STS1, upmu_get_reg_value(MT6330_PG_SDN_STS1));
		pal_log_info("[PMIC]PG_SDN_STS2[0x%x]=0x%x\n",
			MT6330_PG_SDN_STS2, upmu_get_reg_value(MT6330_PG_SDN_STS2));
		pal_log_info("[PMIC]PG_SDN_STS3[0x%x]=0x%x\n",
			MT6330_PG_SDN_STS3, upmu_get_reg_value(MT6330_PG_SDN_STS3));
	/*--buck oc--*/
		pal_log_info("[PMIC]OC_SDN_STS0[0x%x]=0x%x\n",
			MT6330_OC_SDN_STS0, upmu_get_reg_value(MT6330_OC_SDN_STS0));
		pal_log_info("[PMIC]OC_SDN_STS1[0x%x]=0x%x\n",
			MT6330_OC_SDN_STS1, upmu_get_reg_value(MT6330_OC_SDN_STS1));
		pal_log_info("[PMIC]OC_SDN_STS2[0x%x]=0x%x\n",
			MT6330_OC_SDN_STS2, upmu_get_reg_value(MT6330_OC_SDN_STS2));
		pal_log_info("[PMIC]BUCK_OC_SDN_STATUS[0x%x]=0x%x\n",
			MT6330_BUCK_TOP_OC_CON0, upmu_get_reg_value(MT6330_BUCK_TOP_OC_CON0));
		pal_log_info("[PMIC]BUCK_OC_SDN_EN[0x%x]=0x%x\n",
			MT6330_BUCK_TOP_ELR0, upmu_get_reg_value(MT6330_BUCK_TOP_ELR0));
	/*--thermal shutdown 150--*/
		pal_log_info("[PMIC]THERMALSTATUS[0x%x]=0x%x\n",
			MT6330_THERMALSTATUS, upmu_get_reg_value(MT6330_THERMALSTATUS));
	/*--long press shutdown--*/
		pal_log_info("[PMIC]STRUP_CON4[0x%x]=0x%x\n",
			MT6330_STRUP_CON4, upmu_get_reg_value(MT6330_STRUP_CON4));
		pal_log_info("[PMIC]STRUP_CON12[0x%x]=0x%x\n",
			MT6330_STRUP_CON12, upmu_get_reg_value(MT6330_STRUP_CON12));
	/*--WDTRST--*/
		pal_log_info("[PMIC]TOP_RST_MISC0[0x%x]=0x%x\n",
			MT6330_TOP_RST_MISC0, upmu_get_reg_value(MT6330_TOP_RST_MISC0));
		pal_log_info("[PMIC]TOP_RST_MISC1[0x%x]=0x%x\n",
			MT6330_TOP_RST_MISC1, upmu_get_reg_value(MT6330_TOP_RST_MISC1));
	/*--CLK TRIM--*/
		pal_log_info("[PMIC]TOP_CLK_TRIM[0x%x]=0x%x\n",
			MT6330_TOP_CLK_TRIM, upmu_get_reg_value(MT6330_TOP_CLK_TRIM));
	/*--WDTRSTB_STATUS will be clear in kernel--*/
		mt6330_wdtdbg_vosel();
#endif

	/*--Check PMIC reset reason--*/
		pmic_check_rst(poff_sts, top_sts);

		record_pmic_boot_status(pon_sts, poff_sts);
	} else {
		/*--Clear PONSTS and POFFSTS(include PG status and BUCK OC status)--*/
		ret_val = pmic_config_interface(PMIC_RG_POFFSTS_CLR_ADDR, 0x1,
			PMIC_RG_POFFSTS_CLR_MASK, PMIC_RG_POFFSTS_CLR_SHIFT);
		ret_val = pmic_config_interface(PMIC_RG_PONSTS_CLR_ADDR, 0x1,
			PMIC_RG_PONSTS_CLR_MASK, PMIC_RG_PONSTS_CLR_SHIFT);
		ret_val = pmic_config_interface(PMIC_RG_POFFSTS_CLR_ADDR, 0x0,
			PMIC_RG_POFFSTS_CLR_MASK, PMIC_RG_POFFSTS_CLR_SHIFT);
		ret_val = pmic_config_interface(PMIC_RG_PONSTS_CLR_ADDR, 0x0,
			PMIC_RG_PONSTS_CLR_MASK, PMIC_RG_PONSTS_CLR_SHIFT);
		/*--Clear TOPSTS--*/
		ret_val = pmic_config_interface(PMIC_TOP_STATUS_CLR_ADDR, 0x1,
			PMIC_TOP_STATUS_CLR_MASK, PMIC_TOP_STATUS_CLR_SHIFT);
		ret_val = pmic_config_interface(PMIC_TOP_STATUS_CLR_ADDR, 0x0,
			PMIC_TOP_STATUS_CLR_MASK, PMIC_TOP_STATUS_CLR_SHIFT);
		/*--clear OC_SDN_STATUS--*/
		ret_val = pmic_config_interface(MT6330_BUCK_TOP_OC_CON0, 0xFF, 0xFF, 0);
		/*--Clear thermal shutdown--*/
		ret_val = pmic_config_interface(PMIC_RG_STRUP_THR_CLR_ADDR, 0x1,
			PMIC_RG_STRUP_THR_CLR_MASK, PMIC_RG_STRUP_THR_CLR_SHIFT);
		ret_val = pmic_config_interface(PMIC_RG_STRUP_THR_CLR_ADDR, 0x0,
			PMIC_RG_STRUP_THR_CLR_MASK, PMIC_RG_STRUP_THR_CLR_SHIFT);
		/*--Long press shutdown will be clear by pmic_check_rst()--*/
	}
}

void pmic_wdt_set(void)
{
	unsigned int ret_val = 0;
	/*--Reset digital only--*/
	/*--Enable WDT--*/
	ret_val = pmic_config_interface(PMIC_TOP_RST_MISC1_SET_ADDR, 0x20, 0xFF, 0); /*--[5]=1, RG_WDTRSTB_DEB--*/
	ret_val = pmic_config_interface(PMIC_TOP_RST_MISC1_CLR_ADDR, 0x02, 0xFF, 0); /*--[1]=0, RG_WDTRSTB_MODE--*/
	ret_val = pmic_config_interface(PMIC_TOP_RST_MISC1_SET_ADDR, 0x01, 0xFF, 0); /*--[0]=1, RG_WDTRSTB_EN--*/
	pal_log_info("[%s] TOP_RST_MISC1=0x%x\n", __func__,
			upmu_get_reg_value(MT6330_TOP_RST_MISC1));
}

/******************************************************************************************************
* AUXADC
******************************************************************************************************/
#define VOLT_FULL		1800
#define MAX_TIMEOUT_COUNT	320
#if 0 /* TODO */
/* extend r_val to 10 times from PMIC MT6330 */
struct pmic_auxadc_channel_new pmic_auxadc_channel[] = {
	/* BATADC */
	PMIC_AUXADC_GEN(15, 35, 0, PMIC_AUXADC_RQST_CH0,
		PMIC_AUXADC_ADC_RDY_CH0_BY_AP, PMIC_AUXADC_ADC_OUT_CH0_BY_AP),
	/* BAT TEMP */
	PMIC_AUXADC_GEN(12, 25, 3, PMIC_AUXADC_RQST_CH3,
		PMIC_AUXADC_ADC_RDY_CH3, PMIC_AUXADC_ADC_OUT_CH3),
	/* CHIP TEMP */
	PMIC_AUXADC_GEN(12, 10, 4, PMIC_AUXADC_RQST_CH4,
		PMIC_AUXADC_ADC_RDY_CH4, PMIC_AUXADC_ADC_OUT_CH4),
	/* VCORE_TEMP */
	PMIC_AUXADC_GEN(12, 10, 4, PMIC_AUXADC_RQST_CH4_BY_THR1,
		PMIC_AUXADC_ADC_RDY_CH4_BY_THR1, PMIC_AUXADC_ADC_OUT_CH4_BY_THR1),
	/* VPROC_TEMP */
	PMIC_AUXADC_GEN(12, 10, 4, PMIC_AUXADC_RQST_CH4_BY_THR2,
		PMIC_AUXADC_ADC_RDY_CH4_BY_THR2, PMIC_AUXADC_ADC_OUT_CH4_BY_THR2),
	/* VGPU_TEMP */
	PMIC_AUXADC_GEN(12, 10, 4, PMIC_AUXADC_RQST_CH4_BY_THR3,
		PMIC_AUXADC_ADC_RDY_CH4_BY_THR3, PMIC_AUXADC_ADC_OUT_CH4_BY_THR3),
	/* ACCDET Multi-Key */
	PMIC_AUXADC_GEN(12, 10, 5, PMIC_AUXADC_RQST_CH5,
		PMIC_AUXADC_ADC_RDY_CH5, PMIC_AUXADC_ADC_OUT_CH5),
	/* DCXO voltage */
	PMIC_AUXADC_GEN(12, 15, 6, PMIC_AUXADC_RQST_CH6,
		PMIC_AUXADC_ADC_RDY_CH6, PMIC_AUXADC_ADC_OUT_CH6),
	/* TSX TEMP(GPS) */
	PMIC_AUXADC_GEN(15, 10, 7, PMIC_AUXADC_RQST_CH7_BY_GPS,
		PMIC_AUXADC_ADC_RDY_CH7_BY_GPS, PMIC_AUXADC_ADC_OUT_CH7_BY_GPS),
	/* HP OFFSET CAL */
	PMIC_AUXADC_GEN(15, 10, 9, PMIC_AUXADC_RQST_CH9,
		PMIC_AUXADC_ADC_RDY_CH9, PMIC_AUXADC_ADC_OUT_CH9),
	/* DCXO TEMP */
	PMIC_AUXADC_GEN(15, 10, 10, PMIC_AUXADC_RQST_DCXO_BY_GPS,
		PMIC_AUXADC_ADC_RDY_DCXO_BY_GPS, PMIC_AUXADC_ADC_OUT_DCXO_BY_GPS),
	/* VBIF */
	PMIC_AUXADC_GEN(12, 25, 11, PMIC_AUXADC_RQST_CH11,
		PMIC_AUXADC_ADC_RDY_CH11, PMIC_AUXADC_ADC_OUT_CH11),
};

int pmic_get_auxadc_value(unsigned short channel)
{
	int count = 0;
	signed int adc_result = 0, reg_val = 0;
	struct pmic_auxadc_channel_new *auxadc_channel;

	if (channel >= AUXADC_LIST_MAX) {
		pal_log_err("[%s] Invalid channel(%d)\n", __func__, channel);
		return -1;
	}
	auxadc_channel = &pmic_auxadc_channel[channel];

	pmic_set_register_value(auxadc_channel->channel_rqst, 1);
	udelay(10);

	while (pmic_get_register_value(auxadc_channel->channel_rdy) != 1) {
		udelay(100);
		count++;
		if (count > MAX_TIMEOUT_COUNT) {
			pal_log_err("[%s] (%d) Time out!\n",
				__func__, auxadc_channel->ch_num);
			break;
		}
	}
	reg_val = pmic_get_register_value(auxadc_channel->channel_out);

	/* Audio request HPOFS to return raw data */
	/* V = (reg_val * r_val * 1800 / 10) >> (resolution bits) */
	if (channel == AUXADC_LIST_HPOFS_CAL)
		adc_result = reg_val;
	else
		adc_result = ((reg_val * auxadc_channel->r_val * VOLT_FULL) / 10)
				>> auxadc_channel->resolution;

	pal_log_info("[%s] channel = %d, reg_val = 0x%x, adc_result = %d\n",
		__func__, auxadc_channel->ch_num, reg_val, adc_result);
	return adc_result;
}
#endif

static void pwrkey_dbg_status(void)
{
	U32 val = 0;

	pmic_read_interface(PMIC_PWRKEY_LONG_PRESS_COUNT_ADDR, &val,
			    PMIC_PWRKEY_LONG_PRESS_COUNT_MASK,
			    PMIC_PWRKEY_LONG_PRESS_COUNT_SHIFT);

	pal_log_info("[%s] powerkey:%d count=%d(ms) g_pwrkey_release=%d\n",
		     __func__, pmic_detect_powerkey(),
		     val << 5, g_pwrkey_release);
}

static void pmic_enable_smart_reset(unsigned char smart_en,
				    unsigned char smart_sdn_en)
{
	pmic_config_interface(PMIC_RG_SMART_RST_MODE_ADDR, smart_en,
			      PMIC_RG_SMART_RST_MODE_MASK,
			      PMIC_RG_SMART_RST_MODE_SHIFT);
	pmic_config_interface(PMIC_RG_SMART_RST_SDN_EN_ADDR, smart_sdn_en,
			      PMIC_RG_SMART_RST_SDN_EN_MASK,
			      PMIC_RG_SMART_RST_SDN_EN_SHIFT);
	pal_log_info("[%s] smart_en:%d, smart_sdn_en:%d\n",
		     __func__, smart_en, smart_sdn_en);
}

static U16 get_spmi_dbg_data(struct spmi_device *dev, U8 mux_sel)
{
	U16 rdata = 0;

	/* set DBGMUX_SEL */
	spmi_ext_register_writel(dev, PMIC_RG_SPMI_DBGMUX_SEL_ADDR, &mux_sel, 1);
	/* read spmi_debug[15:0] data*/
	spmi_ext_register_readl(dev, PMIC_RG_SPMI_DBGMUX_OUT_L_ADDR, &rdata, 2);

	return rdata;
}

void mt6330_dump_record_reg(struct spmi_device *dev)
{
	U8 wdata;
	U16 dbg_data = 0, idx;
	U16 addr, data, cmd;
	int i;

	/* pause debug log feature by setting RG_DEBUG_DIS_TRIG 0->1->0 */
	wdata = 0x1 << PMIC_RG_DEBUG_DIS_TRIG_SHIFT;
	spmi_ext_register_writel(dev, PMIC_RG_DEBUG_DIS_TRIG_ADDR, &wdata, 1);
	wdata = 0;
	spmi_ext_register_writel(dev, PMIC_RG_DEBUG_DIS_TRIG_ADDR, &wdata, 1);
	/* DBGMUX_SEL = 0 */
	dbg_data = get_spmi_dbg_data(dev, 0);
	idx = dbg_data & 0xF;
	for (i = 0; i < 16; i++) {
		/* debug_addr start from index 1 */
		addr = get_spmi_dbg_data(dev, ((idx + i) % 16) + 1);
		dbg_data = get_spmi_dbg_data(dev, ((idx + i) % 16) + 17);
		data = dbg_data & 0xFF;
		cmd = (dbg_data >> 8) & 0x7;
		pal_log_warn("record %s addr=0x%x, data=0x%x, cmd=%d%s\n",
			     cmd <= 3 ? "write" : "read",
			     addr, data, cmd,
			     i == 15 ? "(the last)" : "");
	}
	/* enable debug log feature by setting RG_DEBUG_EN_TRIG 0->1->0 */
	wdata = 0x1 << PMIC_RG_DEBUG_EN_TRIG_SHIFT;
	spmi_ext_register_writel(dev, PMIC_RG_DEBUG_EN_TRIG_ADDR, &wdata, 1);
	wdata = 0;
	spmi_ext_register_writel(dev, PMIC_RG_DEBUG_EN_TRIG_ADDR, &wdata, 1);
}
/*
 * PMIC Init Code
 */
U32 pmic_init(void)
{
	U32 ret_code = PMIC_TEST_PASS;
	unsigned int val = 0;

	pal_log_info("[PMIC]Preloader Start\n");

	mt6330_sdev = get_spmi_device(SPMI_MASTER_1, SPMI_SLAVE_4);
	if (!mt6330_sdev) {
		pal_log_info("%s: get spmi device fail\n", __func__);
		return -ENODEV;
	}

	pal_log_info("[PMIC] CHIP Code = 0x30%x\n",
		     pmic_get_chip_version());

	/* Boot debug status */
	pmic_dbg_status(1);

	pwrkey_dbg_status();

	/* remove pmic_disable_usbdl_wo_battery() */

	/* Enable PMIC WDTRST function (depends on main chip RST function) TBD */
	pmic_wdt_set();

	mt6330_probe();
#ifdef EXT_BUCK_MT6315
	mt6315_spmi_probe();
#endif

	pal_log_info("[PMIC]Init done\n");

	/* default disable smart reset, enable it by RGU driver */
	pmic_enable_smart_reset(0, 0);

#if 0 /* Test PMIC AUXADC */
	pmic_get_auxadc_value(AUXADC_LIST_BATADC);
	pmic_get_auxadc_value(AUXADC_LIST_BATTEMP);
	pmic_get_auxadc_value(AUXADC_LIST_VBIF);
	pmic_get_auxadc_value(AUXADC_LIST_TSX_TEMP);
#endif
	pmic_config_interface(PMIC_RG_CPS_W_KEY_ADDR, 0x29,
			      PMIC_RG_CPS_W_KEY_MASK,
			      PMIC_RG_CPS_W_KEY_SHIFT);
	pmic_config_interface(PMIC_RG_CPS_W_KEY_H_ADDR, 0x47,
			      PMIC_RG_CPS_W_KEY_H_MASK,
			      PMIC_RG_CPS_W_KEY_H_SHIFT);
	pmic_config_interface(PMIC_RG_SMART_RST_MODE_ADDR, 0x0,
			      PMIC_RG_SMART_RST_MODE_MASK,
			      PMIC_RG_SMART_RST_MODE_SHIFT);
	pmic_config_interface(PMIC_RG_SMART_RST_SDN_EN_ADDR, 0x0,
			      PMIC_RG_SMART_RST_SDN_EN_MASK,
			      PMIC_RG_SMART_RST_SDN_EN_SHIFT);
	pmic_config_interface(PMIC_RG_CPS_W_KEY_ADDR, 0,
			      PMIC_RG_CPS_W_KEY_MASK,
			      PMIC_RG_CPS_W_KEY_SHIFT);
	pmic_config_interface(PMIC_RG_CPS_W_KEY_H_ADDR, 0,
			      PMIC_RG_CPS_W_KEY_H_MASK,
			      PMIC_RG_CPS_W_KEY_H_SHIFT);

	return ret_code;
}

U32 pmic_reinit(void)
{
	U32 ret_code = PMIC_TEST_PASS;
	pal_log_info("[PMIC]re-init Start\n");

	mt6330_sdev = get_spmi_device(SPMI_MASTER_1, SPMI_SLAVE_4);
	if (!mt6330_sdev) {
		pal_log_info("%s: get spmi device fail\n", __func__);
		return -ENODEV;
	}
	pal_log_info("[PMIC] CHIP Code = 0x30%x\n",
		     pmic_get_chip_version());

	mt6330_probe();
#ifdef EXT_BUCK_MT6315
	mt6315_spmi_probe();
#endif

	pal_log_info("[PMIC]re-init done\n");

	return ret_code;
}
