#include <inttypes.h>
#include <ctype.h>
#include <fcntl.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <getopt.h>
#include <asm/types.h>
#include <pthread.h>

#include "../inc/nand_api.h"
#include "../inc/common.h"
#include "../inc/xalloc.h"
#include "../inc/libmtd.h"

struct nand_ops {
	struct mtd_dev_info mtd;
	unsigned int *eb_map;
	unsigned int valid_eb_cnt;
	int fd;
	off_t offset;
};

/* As disscussed with upper user,
 * the max number of files opened simultaneously is less than 10.
 * consider to keep code simple,
 * just define a array for saving files' objects.
 */
#define OPEN_FILES_MAX 10

struct nand_handler {
	libmtd_t mtd_desc;
	struct nand_ops nops[OPEN_FILES_MAX];
	int nops_count;
	unsigned char *buf;
	int eb_size;
};

static struct nand_handler nhandler;
static pthread_mutex_t nmutex = PTHREAD_MUTEX_INITIALIZER;

struct split_info {
	off_t head;
	size_t head_len;
	off_t body;
	size_t body_len;
	off_t tail;
	size_t tail_len;
};

static void data_split(struct split_info *split, off_t offset,
			size_t len, unsigned int align)
{
	size_t val;

	split->head = offset;
	val = offset / align * align;
	val = align - (offset - val);
	if (val == align)
		split->head_len = 0;
	else if (val > len)
		split->head_len = len;
	else
		split->head_len = val;

	split->body = offset + split->head_len;
	split->body_len = (len - split->head_len) / align * align;

	split->tail = split->body + split->body_len;
	split->tail_len = len - split->head_len - split->body_len;
}

static inline int offset_to_eb(off_t offset)
{
	return offset / nhandler.eb_size;
}

static inline int offset_to_eb_offset(off_t offset)
{
	return offset % nhandler.eb_size;
}

static inline unsigned int map_eb(unsigned int eb, int nops_idx)
{
	/* this case would be rarely appeared */
	if (eb >= nhandler.nops[nops_idx].valid_eb_cnt)
		printf("warning: the block leb:%d maybe not valid, fd:%d\n", eb, nhandler.nops[nops_idx].fd);

	return nhandler.nops[nops_idx].eb_map[eb];
}

static int create_eb_map(struct nand_ops *ops)
{
	int eb_cnt = 0;
	int ret = 0, i;

	if (!ops->eb_map)
		ops->eb_map = xzalloc((size_t)ops->mtd.eb_cnt * 4);

	for (i = 0; i < ops->mtd.eb_cnt; i++) {
		ret = mtd_is_bad(&ops->mtd, ops->fd, i);
		if (ret == 0) {
			ops->eb_map[eb_cnt++] = i;
		} else if (ret > 0) {
			printf("eb: %d is bad\n", i);
		} else {
			free(ops->eb_map);
			ops->eb_map = NULL;
			return ret;
		}
	}

	ops->valid_eb_cnt = eb_cnt;
	printf("valid eb count:%d\n", ops->valid_eb_cnt);

	return ret;
}

static int get_current_ops(int fd)
{
	int i;

	for (i = 0; i < OPEN_FILES_MAX; i++) {
		if (nhandler.nops[i].fd == fd)
			return i;
	}

	printf("Fail to get the file we want, there must be a error.\n");
	/*it wouldn't reach here, no need to check if value returned is valid.*/
	return -1;
}

static int get_empty_ops()
{
	int i;

	for (i = 0; i < OPEN_FILES_MAX; i++) {
		if (nhandler.nops[i].fd == 0)
			return i;
	}

	printf("Fail to get an empty ops object, open too many files.\n");
	return -1;
}

int nand_open(const char *pathname, int flags)
{
	struct mtd_dev_info *mtd;
	int current, fd;
	int ret = -1;

	pthread_mutex_lock(&nmutex);

	if (nhandler.nops_count == 0) {
		nhandler.mtd_desc = libmtd_open();
		if (!nhandler.mtd_desc) {
			printf("libmtd initialize fail\n");
			return ret;
		}
	}

	fd = open(pathname, flags);
	if (fd == -1) {
		printf("open file: %s fail\n", pathname);
		return ret;
	}

	current = get_empty_ops();
	nhandler.nops[current].fd = fd;

	mtd = &(nhandler.nops[current].mtd);
	ret = mtd_get_dev_info(nhandler.mtd_desc, pathname, mtd);
	if (ret < 0) {
		printf("get mtd info fail\n");
		return ret;
	}

	ret = create_eb_map(&(nhandler.nops[current]));
	if (ret < 0) {
		printf("check bad block fail\n");
		return ret;
	}

	if (nhandler.nops_count == 0) {
		nhandler.buf = xmalloc(mtd->eb_size);
		nhandler.eb_size = mtd->eb_size;
	}

	nhandler.nops_count++;

	pthread_mutex_unlock(&nmutex);

	return fd;
}

ssize_t nand_read(int fd, void *buf, size_t count)
{
	struct nand_ops *nops;
	struct split_info split;
	int eb, eb_cnt, eb_off;
	char *lbuf = buf;
	int current, ret, i;

	current = get_current_ops(fd);
	nops = &(nhandler.nops[current]);

	data_split(&split, nops->offset, count, nops->mtd.eb_size);

	if (split.head_len) {
		eb = offset_to_eb(split.head);
		eb_off = offset_to_eb_offset(split.head);

		ret = mtd_read(&nops->mtd, fd, map_eb(eb, current), eb_off,
			       lbuf, split.head_len);
		if (ret < 0)
			return ret;

		lbuf += split.head_len;
	}

	if (split.body_len) {
		eb = offset_to_eb(split.body);
		eb_cnt = split.body_len / nops->mtd.eb_size;

		for (i = 0; i < eb_cnt; i++) {
			ret = mtd_read(&nops->mtd, fd, map_eb(eb + i, current),
				       0, lbuf, nops->mtd.eb_size);
			if (ret < 0)
				return ret;

			lbuf += nops->mtd.eb_size;
		}

	}

	if (split.tail_len) {
		eb = offset_to_eb(split.tail);

		ret = mtd_read(&nops->mtd, fd, map_eb(eb, current), 0,
			       lbuf, split.tail_len);
		if (ret < 0)
			return ret;
	}

	nops->offset += count;
	return count;
}

static int bad_block_handle(libmtd_t desc, const struct mtd_dev_info *mtd,
	void *data, int len, struct nand_ops *ops)
{
	int eb, eb_off, ret;

	eb = offset_to_eb(ops->offset);
	if (eb + 1 > ops->valid_eb_cnt) {
		printf("there is no good block for bbh.\n");
		return -1;
	}

	ret = mtd_erase(desc, &ops->mtd, ops->fd, ops->eb_map[eb + 1]);
	if (ret < 0)
		return ret;

	eb_off = offset_to_eb_offset(ops->offset);
	if (eb_off) {
		ret = mtd_read(&ops->mtd, ops->fd, ops->eb_map[eb], 0,
			       nhandler.buf, eb_off);
		if (ret < 0)
			return ret;

		ret = mtd_write(desc, &ops->mtd, ops->fd, ops->eb_map[eb + 1],
				0, nhandler.buf, eb_off, NULL, 0, 0);
		if (ret < 0)
			return ret;
	}

	if (data && len) {
		ret = mtd_write(desc, &ops->mtd, ops->fd, ops->eb_map[eb + 1],
				eb_off, data, len, NULL, 0, 0);
		if (ret < 0)
			return ret;
	}

	ret = mtd_mark_bad(&ops->mtd, ops->fd, ops->eb_map[eb]);
	if (ret < 0)
		return ret;

	return create_eb_map(ops);
}

/* $count must be page_size alligned */
ssize_t nand_write(int fd, void *buf, size_t count)
{
	struct nand_ops *nops;
	struct split_info split;
	int eb, eb_cnt, eb_off;
	char *lbuf = buf;
	int current, ret, i;

	current = get_current_ops(fd);
	nops = &(nhandler.nops[current]);

	data_split(&split, nops->offset, count, nops->mtd.eb_size);

	if (split.head_len) {
		eb = offset_to_eb(split.head);
		eb_off = offset_to_eb_offset(split.head);

		/* this eb should be already erased, so write directly */
		ret = mtd_write(nhandler.mtd_desc, &nops->mtd, fd,
				map_eb(eb, current), eb_off, lbuf,
				split.head_len, NULL, 0, 0);
		if (ret < 0) {
			if (errno != EIO)
				return ret;
			ret = bad_block_handle(nhandler.mtd_desc,
				&nops->mtd, lbuf, split.head_len, nops);
			if (ret < 0)
				return ret;
		}

		lbuf += split.head_len;
		nops->offset += split.head_len;
	}

	if (split.body_len) {
		eb = offset_to_eb(split.body);
		eb_cnt = split.body_len / nops->mtd.eb_size;

		for (i = 0; i < eb_cnt; i++) {
			ret = mtd_erase(nhandler.mtd_desc, &nops->mtd, fd,
					map_eb(eb + i, current));
			printf("line: %d\n", __LINE__);
			if (ret < 0) {
				if (errno != EIO)
					return ret;
				ret = bad_block_handle(nhandler.mtd_desc,
						&nops->mtd, NULL, 0, nops);
				if (ret < 0)
					return ret;
			}

			ret = mtd_write(nhandler.mtd_desc, &nops->mtd, fd,
					map_eb(eb + i, current), 0, lbuf,
					nops->mtd.eb_size, NULL, 0, 0);
			if (ret < 0) {
				if (errno != EIO)
					return ret;
				ret = bad_block_handle(nhandler.mtd_desc,
						&nops->mtd, lbuf,
						nops->mtd.eb_size, nops);
				if (ret < 0)
					return ret;
			}

			lbuf += nops->mtd.eb_size;
			nops->offset += nops->mtd.eb_size;
		}

	}

	if (split.tail_len) {
		eb = offset_to_eb(split.tail);

		ret = mtd_erase(nhandler.mtd_desc, &nops->mtd, fd,
				map_eb(eb, current));
		if (ret < 0) {
			if (errno != EIO)
				return ret;
			ret = bad_block_handle(nhandler.mtd_desc,
					&nops->mtd, NULL, 0, nops);
			if (ret < 0)
				return ret;
		}

		ret = mtd_write(nhandler.mtd_desc, &nops->mtd, fd,
				map_eb(eb, current), 0, lbuf,
				split.tail_len, NULL, 0, 0);
		if (ret < 0) {
			if (errno != EIO)
				return ret;
			ret = bad_block_handle(nhandler.mtd_desc,
					&nops->mtd, lbuf,
					split.tail_len, nops);
			if (ret < 0)
				return ret;
		}

		nops->offset += split.tail_len;
	}

	return count;
}

off_t nand_seek(int fd, off_t offset, int whence)
{
	int current = get_current_ops(fd);

	switch (whence) {
	case SEEK_SET:
		nhandler.nops[current].offset = offset;
		break;

	case SEEK_CUR:
		nhandler.nops[current].offset += offset;
		break;

	case SEEK_END:
	default:
		printf("invalid perameter\n");
		return -1;
	}

	return lseek(fd, nhandler.nops[current].offset, SEEK_SET);
}

off64_t nand_seek64(int fd, off64_t offset, int whence)
{
	return (off64_t)nand_seek(fd, (off_t)offset, whence);
}

int nand_close(int fd)
{
	int ret;
	int current = get_current_ops(fd);

	pthread_mutex_lock(&nmutex);

	nhandler.nops_count--;

	if (nhandler.nops_count == 0) {
		libmtd_close(nhandler.mtd_desc);
		free(nhandler.buf);
	}

	free(nhandler.nops[current].eb_map);
	memset(&nhandler.nops[current], 0, sizeof(struct nand_ops));
	ret = close(fd);

	pthread_mutex_unlock(&nmutex);

	return ret;
}

off_t nand_query_offset(int fd)
{
	int current = get_current_ops(fd);
	struct nand_ops *nops;

	if (current == -1) {
		printf("nand_query_offset fail\n");
		return 0;
	}

	nops = &(nhandler.nops[current]);
	return nops->offset;
}

size_t nand_query_blk_size(int fd)
{
	int current = get_current_ops(fd);
	struct nand_ops *nops;

	if (current == -1) {
		printf("nand_query_blk_size fail\n");
		return 0;
	}

	nops = &(nhandler.nops[current]);
	return nops->mtd.eb_size;
}
