/*
 * This file is part of sp-oops-extract
 * 
 * MTD Oops/Panic log extraction program
 *
 * Copyright (C) 2007, 2008 Nokia Corporation. All rights reserved.
 *
 * Author: Richard Purdie <rpurdie@openedhand.com>
 * Contact: Eero Tamminen <eero.tamminen@nokia.com>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#define _XOPEN_SOURCE 500

#include <errno.h>
#include <fcntl.h>
#include <malloc.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <err.h>
#include <stdint.h>
#include <stdlib.h>

#include <stdarg.h>

#include <syslog.h>

#define OOPS_PAGE_SIZE 131072

#define MEMGETINFO                _IOR('M', 1, struct mtd_info_user)

#define u_int32_t	uint32_t

struct mtd_info_user {
	uint8_t type;
	uint32_t flags;
	uint32_t size;	 // Total size of the MTD
	uint32_t erasesize;
	uint32_t writesize;
	uint32_t oobsize;   // Amount of OOB data per block (e.g. 16)
	/* The below two fields are obsolete and broken, do not use them
	 * (TODO: remove at some point) */
	uint32_t ecctype;
	uint32_t eccsize;
};

enum oops_log_level {
	LOG_LEVEL_1  = 0x01,
	LOG_LEVEL_2,
	LOG_LEVEL_3,
};

enum oops_option {
	OOPS_OPTION_SELECT_DISP  = 0x01,
	OOPS_OPTION_CHECK,
};

#define oopslog_print_level_lo(type,fmt,args...) \
{ \
	if (type >= LOG_LEVEL_1) \
		printf(fmt,##args); \
} \

#define oopslog_print_level_hi(type,fmt,args...) \
{ \
	if (type >= LOG_LEVEL_2) \
		printf(fmt,##args); \
} \

#define oopslog_print_level_dbg(type,fmt,args...) \
{ \
	syslog(LOG_ERR ,"[crash_log]"fmt,##args); \
} \




static int try_to_check_for_bad_blocks(void) {
	/* FIXME
	 * Checking for bad blocks should be implemented here.
	 */
	return 0;
}

//int main(int argc, const char *argv[])
int main(int argc, char *argv[])
{
	const char *progname;
	u_int32_t *count, maxcount = 0xffffffff;
	u_int32_t *magic_ptr, magic_value = 0x5d005d00;
	
	unsigned char *charbuf;
	unsigned long size;
	const char *device;
	struct stat sbuf;
	struct mtd_info_user meminfo;
	int i, j, maxpos;
	void *buf;
	int fd;
	int c;
	u_int32_t oops_page_size = OOPS_PAGE_SIZE;
	int log_level=LOG_LEVEL_2;
	char *checkfile_path = NULL;
	char entry_log[512];
	char *p_entry_comp_log;
	char checkfile_dat[512];
	int option;
	int  check_find;

	progname = argv[0];
	memset(checkfile_dat,0,sizeof(checkfile_dat));
	p_entry_comp_log = NULL;
	option = 0;

	while ((c = getopt(argc, argv, "r:hl:c")) > 0) {
		switch (c) {
		case 'h':
			errx(-1, "Usage: %s [-r RECORD_SIZE] [-h] [-l checkfile] [-c checkfile] devicefile\n", argv[0]);
			break;
		case 'r':
			oops_page_size = atoi(optarg);
			break;
		case 'l':
			log_level=LOG_LEVEL_2;
			checkfile_path = optarg;
			option = OOPS_OPTION_SELECT_DISP;
			break;
		case 'c':
			log_level=LOG_LEVEL_1;
			option = OOPS_OPTION_CHECK;
			break;
		}
	}
	argc -= optind;
	argv += optind;

	oopslog_print_level_dbg(log_level, "Oops Log extractor (build %s %s).\n", __DATE__, __TIME__);
	if (checkfile_path) {
		FILE *fp;
		oopslog_print_level_dbg(log_level, "checkfile =%s\n", checkfile_path);
		fp = fopen(checkfile_path, "r");
		if (fp) {
			if (fgets(checkfile_dat, sizeof(checkfile_dat), fp)) {
				p_entry_comp_log = checkfile_dat;
				oopslog_print_level_hi(log_level, "entry_comp_log=%s\n", p_entry_comp_log);
				oopslog_print_level_dbg(log_level, "entry_comp_log=%s\n", p_entry_comp_log);
			}
			fclose(fp);
		} else {
			oopslog_print_level_dbg(log_level, "fopen err\n");
		}
	}

	if (argc < 1) {
		errx(-1, "Usage: %s devicefile\n", progname);
	}

	device = argv[0];

	buf = malloc(oops_page_size);
	if (!buf) {
		err(-1, "Unable to allocate memory");
	}

	if (stat(device, &sbuf)) {
		err(-1, "Unable to stat file '%s'", device);
	}

	fd = open(device, O_RDONLY);
	if (fd < 0) {
		err(-1, "Unable to open file '%s'", device);
	}


	if (S_ISBLK(sbuf.st_mode)) {
		if (ioctl(fd, BLKGETSIZE, &size) < 0) {
			err(-1, "BLKGETSIZE ioctl failed for device '%s'", device);			
		}
		size = size * 512;
		if (try_to_check_for_bad_blocks())
			warnx("%s is probably an mtdblock device. Thus it's not possible to check for bad blocks", device);
	}
	else if (S_ISCHR(sbuf.st_mode)) {
		if (ioctl(fd, MEMGETINFO, &meminfo) != 0) {
			warn("MEMGETINFO ioctl failed for device '%s'", device);
			errx(-1, "%s is maybe not an mtd device?", device);
		}
		size = meminfo.size;
		if (try_to_check_for_bad_blocks()) {
			warnx("Checking for bad blocks failed");		
			warnx("Are you sure that %s is an mtd device?", device);
		}		

	} 
	else if (S_ISREG(sbuf.st_mode)) {
		size = sbuf.st_size;
	}
	else if (S_ISLNK(sbuf.st_mode)) {
		errx(-1, "%s is a symbolik link", device); 
	}
	else if (S_ISDIR(sbuf.st_mode)) {
		errx(-1, "%s is a directory", device);
	}
	else if (S_ISFIFO(sbuf.st_mode)) {
		errx(-1, "%s is a FIFO", device); 
	}
	else if (S_ISSOCK(sbuf.st_mode)) {
		errx(-1, "%s is a socket", device);
	}
	else 
		errx(-1, "%s is something weird", device);

	charbuf = buf;	
	count = (u_int32_t *) buf;
	magic_ptr = (u_int32_t *) (buf + sizeof(u_int32_t));

	for (i = 0; i < (size / oops_page_size); i++) {
		pread(fd, buf, oops_page_size, i * oops_page_size);
		if (*count == 0xffffffff || *magic_ptr != magic_value)
			continue;
		if (maxcount == 0xffffffff) {
			maxcount = *count;
			maxpos = i;
		} else if ((*count < 0x40000000) && (maxcount > 0xc0000000)) {
			maxcount = *count;
			maxpos = i;
		} else if ((*count > maxcount) && (*count < 0xc0000000)) {
			maxcount = *count;
			maxpos = i;
		} else if ((*count > maxcount) && (*count > 0xc0000000) 
					&& (maxcount > 0x80000000)) {
			maxcount = *count;
			maxpos = i;
		}
	}

	if (maxcount == 0xffffffff) {
		oopslog_print_level_hi(log_level, "No logs present\n");
		oopslog_print_level_dbg(log_level, "No logs present\n");
		return 0;
	}
	oopslog_print_level_dbg(log_level, "Last log is at position %d with count %d\n", maxpos, maxcount);
	check_find = 0;

	for (i = 0; i < (size / oops_page_size); i++) {
		int bufend;

		maxpos++;
		if ((maxpos * oops_page_size) >= size)
			maxpos = 0;

		pread(fd, buf, oops_page_size, maxpos * oops_page_size);
		if (*count == 0xffffffff || *magic_ptr != magic_value)
			continue;

		for (j = oops_page_size - 1; j > 3; j--) {
			if (charbuf[j] != 0xff)
				break;
		}
		snprintf(entry_log, sizeof(entry_log), "Log Entry %d (at position %d)\n", *count, maxpos);
		oopslog_print_level_dbg(log_level, "entry_log = %s\n",entry_log);
		if (option != 0) {
			if (option == OOPS_OPTION_CHECK) {
				/* c option */
				check_find = 1;
				oopslog_print_level_dbg(log_level, "c option continue\n");
				oopslog_print_level_lo(log_level, "%s",entry_log);
				oopslog_print_level_dbg(log_level, "%s",entry_log);
				continue;
			}
			if (p_entry_comp_log) {
				if (strncmp(entry_log, p_entry_comp_log, strlen(p_entry_comp_log)) == 0) {
					oopslog_print_level_dbg(log_level, "hit!! p_entry_comp_log=%s \n", p_entry_comp_log);
				} else {
					if (option == OOPS_OPTION_SELECT_DISP) {
						/* l option */
						oopslog_print_level_dbg(log_level, "l option continue\n");
						continue;
					}
				}
			} else {
				if (option == OOPS_OPTION_SELECT_DISP) {
					/* l option */
					oopslog_print_level_dbg(log_level, "l option break\n");
					break;
				}
			}
		}
		bufend = j;
		for (j = 8; j <= bufend; j++) {
			oopslog_print_level_hi(log_level, "%c", charbuf[j]);
		}
		oopslog_print_level_hi(log_level, "\n");

		if (option != 0) {
			/* l option */
			oopslog_print_level_dbg(log_level, "l option break");
			break;
		}
	}
	oopslog_print_level_dbg(log_level, "End");
	return 0;
}
