#ifndef __RTL_CRYPTO_TESTCASE_H__
#define __RTL_CRYPTO_TESTCASE_H__

#define MAX_TAP			8
struct cipher_testvec {
	char *key;
	char *iv;
	char *input;
	char *result;
	unsigned short tap[MAX_TAP];
	int np;
	unsigned char also_non_np;
	unsigned char fail;
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short rlen;
};

/*******************AES ECB TestCase**********************/

static struct cipher_testvec aes_enc_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.ilen	= 16,
		.result	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.rlen	= 16,
	}, { /* Generated with Crypto++ */
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec aes_dec_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.klen	= 16,
		.input	= "\x69\xc4\xe0\xd8\x6a\x7b\x04\x30"
			  "\xd8\xcd\xb7\x80\x70\xb4\xc5\x5a",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17",
		.klen	= 24,
		.input	= "\xdd\xa9\x7c\xa4\x86\x4c\xdf\xe0"
			  "\x6e\xaf\x70\xa0\xec\x0d\x71\x91",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, {
		.key	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.klen	= 32,
		.input	= "\x8e\xa2\xb7\xca\x51\x67\x45\xbf"
			  "\xea\xfc\x49\x90\x4b\x49\x60\x89",
		.ilen	= 16,
		.result	= "\x00\x11\x22\x33\x44\x55\x66\x77"
			  "\x88\x99\xaa\xbb\xcc\xdd\xee\xff",
		.rlen	= 16,
	}, { /* Generated with Crypto++ */
		.key	= "\xA6\xC9\x83\xA6\xC9\xEC\x0F\x32"
			  "\x55\x0F\x32\x55\x78\x9B\xBE\x78"
			  "\x9B\xBE\xE1\x04\x27\xE1\x04\x27"
			  "\x4A\x6D\x90\x4A\x6D\x90\xB3\xD6",
		.klen	= 32,
		.input	= "\x71\x73\xF7\xDB\x24\x93\x21\x6D"
			  "\x61\x1E\xBB\x63\x42\x79\xDB\x64"
			  "\x6F\x82\xC0\xCA\xA3\x9B\xFA\x0B"
			  "\xD9\x08\xC7\x4A\x90\xAE\x8F\x5F"
			  "\x5E\x06\xF0\x5F\x31\x51\x18\x37"
			  "\x45\xD7\xCA\x3A\xFD\x6C\x3F\xE1"
			  "\xDD\x8D\x22\x65\x2B\x00\x50\xCE"
			  "\xBA\x28\x67\xD7\xCE\x0E\x0D\xEA"
			  "\x78\x69\x7F\xAE\x8F\x8B\x69\x37"
			  "\x75\xE0\xDC\x96\xE0\xB7\xF4\x09"
			  "\xCB\x6D\xA2\xFB\xDA\xAF\x09\xF8"
			  "\x81\x82\x27\xFA\x45\x9C\x29\xA4"
			  "\x22\x8B\x78\x69\x5B\x46\xF9\x39"
			  "\x1B\xCC\xF9\x1D\x09\xEB\xBC\x5C"
			  "\x41\x72\x51\x97\x1D\x07\x49\xA0"
			  "\x1B\x8E\x65\x4B\xB2\x6A\x12\x03"
			  "\x6A\x60\x95\xAC\xBD\xAC\x1A\x64"
			  "\xDE\x5A\xA5\xF0\x83\x2F\xCB\xCA"
			  "\x22\x74\xA6\x6C\x9B\x73\xCE\x3F"
			  "\xE1\x8B\x22\x17\x59\x0C\x47\x89"
			  "\x33\xA1\xD6\x47\x03\x19\x4F\xA8"
			  "\x67\x69\xF0\x5B\xF0\x20\xAD\x06"
			  "\x27\x81\x92\xD8\xC5\xBA\x98\x12"
			  "\xBE\x24\xB5\x2F\x75\x02\xC2\xAD"
			  "\x12\x2F\x07\x32\xEE\x39\xAF\x64"
			  "\x05\x8F\xB3\xD4\xEB\x1B\x46\x6E"
			  "\xD9\x21\xF9\xC4\xB7\xC9\x45\x68"
			  "\xB4\xA1\x74\x9F\x82\x47\xEB\xCC"
			  "\xBD\x0A\x14\x95\x0F\x8B\xA8\x2F"
			  "\x4B\x1B\xA7\xBF\x82\xA6\x43\x0C"
			  "\xB9\x39\x4A\xA8\x10\x6F\x50\x7B"
			  "\x25\xFB\x26\x81\xE0\x2F\xF0\x96"
			  "\x8D\x8B\xAC\x92\x0F\xF6\xED\x64"
			  "\x63\x29\x4C\x8E\x18\x13\xC5\xBF"
			  "\xFC\xA0\xD9\xBF\x7C\x3A\x0E\x29"
			  "\x6F\xD1\x6C\x6F\xA5\xDA\xBF\xB1"
			  "\x30\xEA\x44\x2D\xC3\x8F\x16\xE1"
			  "\x66\xFA\xA3\x21\x3E\xFC\x13\xCA"
			  "\xF0\xF6\xF0\x59\xBD\x8F\x38\x50"
			  "\x31\xCB\x69\x3F\x96\x15\xD6\xF5"
			  "\xAE\xFF\xF6\xAA\x41\x85\x4C\x10"
			  "\x58\xE3\xF9\x44\xE6\x28\xDA\x9A"
			  "\xDC\x6A\x80\x34\x73\x97\x1B\xC5"
			  "\xCA\x26\x16\x77\x0E\x60\xAB\x89"
			  "\x0F\x04\x27\xBD\xCE\x3E\x71\xB4"
			  "\xA0\xD7\x22\x7E\xDB\xEB\x24\x70"
			  "\x42\x71\x51\x78\x70\xB3\xE0\x3D"
			  "\x84\x8E\x8D\x7B\xD0\x6D\xEA\x92"
			  "\x11\x08\x42\x4F\xE5\xAD\x26\x92"
			  "\xD2\x00\xAE\xA8\xE3\x4B\x37\x47"
			  "\x22\xC1\x95\xC1\x63\x7F\xCB\x03"
			  "\xF3\xE3\xD7\x9D\x60\xC7\xBC\xEA"
			  "\x35\xA2\xFD\x45\x52\x39\x13\x6F"
			  "\xC1\x53\xF3\x53\xDF\x33\x84\xD7"
			  "\xD2\xC8\x37\xB0\x75\xE3\x41\x46"
			  "\xB3\xC7\x83\x2E\x8A\xBB\xA4\xE5"
			  "\x7F\x3C\xFD\x8B\xEB\xEA\x63\xBD"
			  "\xB7\x46\xE7\xBF\x09\x9C\x0D\x0F"
			  "\x40\x86\x7F\x51\xE1\x11\x9C\xCB"
			  "\x88\xE6\x68\x47\xE3\x2B\xC5\xFF"
			  "\x09\x79\xA0\x43\x5C\x0D\x08\x58"
			  "\x17\xBB\xC0\x6B\x62\x3F\x56\xE9",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};


/*******************AES CBC TestCase**********************/
static struct cipher_testvec aes_cbc_enc_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv	= "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input	= "Single block msg",
		.ilen   = 16,
		.result = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen   = 32,
		.result = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

static struct cipher_testvec aes_cbc_dec_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
			  "\x51\x2e\x03\xd5\x34\x12\x00\x06",
		.klen   = 16,
		.iv     = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
			  "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
		.input  = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
			  "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
		.ilen   = 16,
		.result = "Single block msg",
		.rlen   = 16,
	}, {
		.key    = "\xc2\x86\x69\x6d\x88\x7c\x9a\xa0"
			  "\x61\x1b\xbb\x3e\x20\x25\xa4\x5a",
		.klen   = 16,
		.iv     = "\x56\x2e\x17\x99\x6d\x09\x3d\x28"
			  "\xdd\xb3\xba\x69\x5a\x2e\x6f\x58",
		.input  = "\xd2\x96\xcd\x94\xc2\xcc\xcf\x8a"
			  "\x3a\x86\x30\x28\xb5\xe1\xdc\x0a"
			  "\x75\x86\x60\x2d\x25\x3c\xff\xf9"
			  "\x1b\x82\x66\xbe\xa6\xd6\x1a\xb1",
		.ilen   = 32,
		.result = "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\x4f\x02\x1d\xb2\x43\xbc\x63\x3d"
			  "\x71\x78\x18\x3a\x9f\xa0\x71\xe8"
			  "\xb4\xd9\xad\xa9\xad\x7d\xed\xf4"
			  "\xe5\xe7\x38\x76\x3f\x69\x14\x5a"
			  "\x57\x1b\x24\x20\x12\xfb\x7a\xe0"
			  "\x7f\xa9\xba\xac\x3d\xf1\x02\xe0"
			  "\x08\xb0\xe2\x79\x88\x59\x88\x81"
			  "\xd9\x20\xa9\xe6\x4f\x56\x15\xcd",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f",
		.input	= "\xf5\x8c\x4c\x04\xd6\xe5\xf1\xba"
			  "\x77\x9e\xab\xfb\x5f\x7b\xfb\xd6"
			  "\x9c\xfc\x4e\x96\x7e\xdb\x80\x8d"
			  "\x67\x9f\x77\x7b\xc6\x70\x2c\x7d"
			  "\x39\xf2\x33\x69\xa9\xd9\xba\xcf"
			  "\xa5\x30\xe2\x63\x04\x23\x14\x61"
			  "\xb2\xeb\x05\xe2\xc3\x9b\xe9\xfc"
			  "\xda\x6c\x19\x07\x8c\x6a\x9d\x1b",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xEA\x65\x8A\x19\xB0\x66\xC1\x3F"
			  "\xCE\xF1\x97\x75\xC1\xFD\xB5\xAF"
			  "\x52\x65\xF7\xFF\xBC\xD8\x2D\x9F"
			  "\x2F\xB9\x26\x9B\x6F\x10\xB7\xB8"
			  "\x26\xA1\x02\x46\xA2\xAD\xC6\xC0"
			  "\x11\x15\xFF\x6D\x1E\x82\x04\xA6"
			  "\xB1\x74\xD1\x08\x13\xFD\x90\x7C"
			  "\xF5\xED\xD3\xDB\x5A\x0A\x0C\x2F"
			  "\x0A\x70\xF1\x88\x07\xCF\x21\x26"
			  "\x40\x40\x8A\xF5\x53\xF7\x24\x4F"
			  "\x83\x38\x43\x5F\x08\x99\xEB\xE3"
			  "\xDC\x02\x64\x67\x50\x6E\x15\xC3"
			  "\x01\x1A\xA0\x81\x13\x65\xA6\x73"
			  "\x71\xA6\x3B\x91\x83\x77\xBE\xFA"
			  "\xDB\x71\x73\xA6\xC1\xAE\x43\xC3"
			  "\x36\xCE\xD6\xEB\xF9\x30\x1C\x4F"
			  "\x80\x38\x5E\x9C\x6E\xAB\x98\x2F"
			  "\x53\xAF\xCF\xC8\x9A\xB8\x86\x43"
			  "\x3E\x86\xE7\xA1\xF4\x2F\x30\x40"
			  "\x03\xA8\x6C\x50\x42\x9F\x77\x59"
			  "\x89\xA0\xC5\xEC\x9A\xB8\xDD\x99"
			  "\x16\x24\x02\x07\x48\xAE\xF2\x31"
			  "\x34\x0E\xC3\x85\xFE\x1C\x95\x99"
			  "\x87\x58\x98\x8B\xE7\xC6\xC5\x70"
			  "\x73\x81\x07\x7C\x56\x2F\xD8\x1B"
			  "\xB7\xB9\x2B\xAB\xE3\x01\x87\x0F"
			  "\xD8\xBB\xC0\x0D\xAC\x2C\x2F\x98"
			  "\x3C\x0B\xA2\x99\x4A\x8C\xF7\x04"
			  "\xE0\xE0\xCF\xD1\x81\x5B\xFE\xF5"
			  "\x24\x04\xFD\xB8\xDF\x13\xD8\xCD"
			  "\xF1\xE3\x3D\x98\x50\x02\x77\x9E"
			  "\xBC\x22\xAB\xFA\xC2\x43\x1F\x66"
			  "\x20\x02\x23\xDA\xDF\xA0\x89\xF6"
			  "\xD8\xF3\x45\x24\x53\x6F\x16\x77"
			  "\x02\x3E\x7B\x36\x5F\xA0\x3B\x78"
			  "\x63\xA2\xBD\xB5\xA4\xCA\x1E\xD3"
			  "\x57\xBC\x0B\x9F\x43\x51\x28\x4F"
			  "\x07\x50\x6C\x68\x12\x07\xCF\xFA"
			  "\x6B\x72\x0B\xEB\xF8\x88\x90\x2C"
			  "\x7E\xF5\x91\xD1\x03\xD8\xD5\xBD"
			  "\x22\x39\x7B\x16\x03\x01\x69\xAF"
			  "\x3D\x38\x66\x28\x0C\xBE\x5B\xC5"
			  "\x03\xB4\x2F\x51\x8A\x56\x17\x2B"
			  "\x88\x42\x6D\x40\x68\x8F\xD0\x11"
			  "\x19\xF9\x1F\x43\x79\x95\x31\xFA"
			  "\x28\x7A\x3D\xF7\x66\xEB\xEF\xAC"
			  "\x06\xB2\x01\xAD\xDB\x68\xDB\xEC"
			  "\x8D\x53\x6E\x72\x68\xA3\xC7\x63"
			  "\x43\x2B\x78\xE0\x04\x29\x8F\x72"
			  "\xB2\x2C\xE6\x84\x03\x30\x6D\xCD"
			  "\x26\x92\x37\xE1\x2F\xBB\x8B\x9D"
			  "\xE4\x4C\xF6\x93\xBC\xD9\xAD\x44"
			  "\x52\x65\xC7\xB0\x0E\x3F\x0E\x61"
			  "\x56\x5D\x1C\x6D\xA7\x05\x2E\xBC"
			  "\x58\x08\x15\xAB\x12\xAB\x17\x4A"
			  "\x5E\x1C\xF2\xCD\xB8\xA2\xAE\xFB"
			  "\x9B\x2E\x0E\x85\x34\x80\x0E\x3F"
			  "\x4C\xB8\xDB\xCE\x1C\x90\xA1\x61"
			  "\x6C\x69\x09\x35\x9E\xD4\xF4\xAD"
			  "\xBC\x06\x41\xE3\x01\xB4\x4E\x0A"
			  "\xE0\x1F\x91\xF8\x82\x96\x2D\x65"
			  "\xA3\xAA\x13\xCC\x50\xFF\x7B\x02",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	},
};

/*************************AES CTR TestCase******************/
static struct cipher_testvec aes_ctr_dec_tv_template[] = {
	{ /* From NIST Special Publication 800-38A, Appendix F.5 */
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.ilen	= 64,
		.result	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.ilen	= 496,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.ilen	= 499,
		.result	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

static struct cipher_testvec aes_ctr_enc_tv_template[] = {
	{ /* From NIST Special Publication 800-38A, Appendix F.5 */
		.key	= "\x2b\x7e\x15\x16\x28\xae\xd2\xa6"
			  "\xab\xf7\x15\x88\x09\xcf\x4f\x3c",
		.klen	= 16,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x87\x4d\x61\x91\xb6\x20\xe3\x26"
			  "\x1b\xef\x68\x64\x99\x0d\xb6\xce"
			  "\x98\x06\xf6\x6b\x79\x70\xfd\xff"
			  "\x86\x17\x18\x7b\xb9\xff\xfd\xff"
			  "\x5a\xe4\xdf\x3e\xdb\xd5\xd3\x5e"
			  "\x5b\x4f\x09\x02\x0d\xb0\x3e\xab"
			  "\x1e\x03\x1d\xda\x2f\xbe\x03\xd1"
			  "\x79\x21\x70\xa0\xf3\x00\x9c\xee",
		.rlen	= 64,
	}, {
		.key	= "\x8e\x73\xb0\xf7\xda\x0e\x64\x52"
			  "\xc8\x10\xf3\x2b\x80\x90\x79\xe5"
			  "\x62\xf8\xea\xd2\x52\x2c\x6b\x7b",
		.klen	= 24,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x1a\xbc\x93\x24\x17\x52\x1c\xa2"
			  "\x4f\x2b\x04\x59\xfe\x7e\x6e\x0b"
			  "\x09\x03\x39\xec\x0a\xa6\xfa\xef"
			  "\xd5\xcc\xc2\xc6\xf4\xce\x8e\x94"
			  "\x1e\x36\xb2\x6b\xd1\xeb\xc6\x70"
			  "\xd1\xbd\x1d\x66\x56\x20\xab\xf7"
			  "\x4f\x78\xa7\xf6\xd2\x98\x09\x58"
			  "\x5a\x97\xda\xec\x58\xc6\xb0\x50",
		.rlen	= 64,
	}, {
		.key	= "\x60\x3d\xeb\x10\x15\xca\x71\xbe"
			  "\x2b\x73\xae\xf0\x85\x7d\x77\x81"
			  "\x1f\x35\x2c\x07\x3b\x61\x08\xd7"
			  "\x2d\x98\x10\xa3\x09\x14\xdf\xf4",
		.klen	= 32,
		.iv	= "\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			  "\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff",
		.input	= "\x6b\xc1\xbe\xe2\x2e\x40\x9f\x96"
			  "\xe9\x3d\x7e\x11\x73\x93\x17\x2a"
			  "\xae\x2d\x8a\x57\x1e\x03\xac\x9c"
			  "\x9e\xb7\x6f\xac\x45\xaf\x8e\x51"
			  "\x30\xc8\x1c\x46\xa3\x5c\xe4\x11"
			  "\xe5\xfb\xc1\x19\x1a\x0a\x52\xef"
			  "\xf6\x9f\x24\x45\xdf\x4f\x9b\x17"
			  "\xad\x2b\x41\x7b\xe6\x6c\x37\x10",
		.ilen	= 64,
		.result	= "\x60\x1e\xc3\x13\x77\x57\x89\xa5"
			  "\xb7\xa7\xf5\x04\xbb\xf3\xd2\x28"
			  "\xf4\x43\xe3\xca\x4d\x62\xb5\x9a"
			  "\xca\x84\xe9\x90\xca\xca\xf5\xc5"
			  "\x2b\x09\x30\xda\xa2\x3d\xe9\x4c"
			  "\xe8\x70\x17\xba\x2d\x84\x98\x8d"
			  "\xdf\xc9\xc5\x8d\xb6\x7a\xad\xa6"
			  "\x13\xc2\xdd\x08\x45\x79\x41\xa6",
		.rlen	= 64,
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFF"
			  "\xFF\xFF\xFF\xFF\xFF\xFF\xFF\xFD",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12",
		.ilen	= 496,
		.result	= "\x04\xF3\xD3\x88\x17\xEF\xDC\xEF"
			  "\x8B\x04\xF8\x3A\x66\x8D\x1A\x53"
			  "\x57\x1F\x4B\x23\xE4\xA0\xAF\xF9"
			  "\x69\x95\x35\x98\x8D\x4D\x8C\xC1"
			  "\xF0\xB2\x7F\x80\xBB\x54\x28\xA2"
			  "\x7A\x1B\x9F\x77\xEC\x0E\x6E\xDE"
			  "\xF0\xEC\xB8\xE4\x20\x62\xEE\xDB"
			  "\x5D\xF5\xDD\xE3\x54\xFC\xDD\xEB"
			  "\x6A\xEE\x65\xA1\x21\xD6\xD7\x81"
			  "\x47\x61\x12\x4D\xC2\x8C\xFA\x78"
			  "\x1F\x28\x02\x01\xC3\xFC\x1F\xEC"
			  "\x0F\x10\x4F\xB3\x12\x45\xC6\x3B"
			  "\x7E\x08\xF9\x5A\xD0\x5D\x73\x2D"
			  "\x58\xA4\xE5\xCB\x1C\xB4\xCE\x74"
			  "\x32\x41\x1F\x31\x9C\x08\xA2\x5D"
			  "\x67\xEB\x72\x1D\xF8\xE7\x70\x54"
			  "\x34\x4B\x31\x69\x84\x66\x96\x44"
			  "\x56\xCC\x1E\xD9\xE6\x13\x6A\xB9"
			  "\x2D\x0A\x05\x45\x2D\x90\xCC\xDF"
			  "\x16\x5C\x5F\x79\x34\x52\x54\xFE"
			  "\xFE\xCD\xAD\x04\x2E\xAD\x86\x06"
			  "\x1F\x37\xE8\x28\xBC\xD3\x8F\x5B"
			  "\x92\x66\x87\x3B\x8A\x0A\x1A\xCC"
			  "\x6E\xAB\x9F\x0B\xFA\x5C\xE6\xFD"
			  "\x3C\x98\x08\x12\xEC\xAA\x9E\x11"
			  "\xCA\xB2\x1F\xCE\x5E\x5B\xB2\x72"
			  "\x9C\xCC\x5D\xC5\xE0\x32\xC0\x56"
			  "\xD5\x45\x16\xD2\xAF\x13\x66\xF7"
			  "\x8C\x67\xAC\x79\xB2\xAF\x56\x27"
			  "\x3F\xCC\xFE\xCB\x1E\xC0\x75\xF1"
			  "\xA7\xC9\xC3\x1D\x8E\xDD\xF9\xD4"
			  "\x42\xC8\x21\x08\x16\xF7\x01\xD7"
			  "\xAC\x8E\x3F\x1D\x56\xC1\x06\xE4"
			  "\x9C\x62\xD6\xA5\x6A\x50\x44\xB3"
			  "\x35\x1C\x82\xB9\x10\xF9\x42\xA1"
			  "\xFC\x74\x9B\x44\x4F\x25\x02\xE3"
			  "\x08\xF5\xD4\x32\x39\x08\x11\xE8"
			  "\xD2\x6B\x50\x53\xD4\x08\xD1\x6B"
			  "\x3A\x4A\x68\x7B\x7C\xCD\x46\x5E"
			  "\x0D\x07\x19\xDB\x67\xD7\x98\x91"
			  "\xD7\x17\x10\x9B\x7B\x8A\x9B\x33"
			  "\xAE\xF3\x00\xA6\xD4\x15\xD9\xEA"
			  "\x85\x99\x22\xE8\x91\x38\x70\x83"
			  "\x93\x01\x24\x6C\xFA\x9A\xB9\x07"
			  "\xEA\x8D\x3B\xD9\x2A\x43\x59\x16"
			  "\x2F\x69\xEE\x84\x36\x44\x76\x98"
			  "\xF3\x04\x2A\x7C\x74\x3D\x29\x2B"
			  "\x0D\xAD\x8F\x44\x82\x9E\x57\x8D"
			  "\xAC\xED\x18\x1F\x50\xA4\xF5\x98"
			  "\x1F\xBD\x92\x91\x1B\x2D\xA6\xD6"
			  "\xD2\xE3\x02\xAA\x92\x3B\xC6\xB3"
			  "\x1B\x39\x72\xD5\x26\xCA\x04\xE0"
			  "\xFC\x58\x78\xBB\xB1\x3F\xA1\x9C"
			  "\x42\x24\x3E\x2E\x22\xBB\x4B\xBA"
			  "\xF4\x52\x0A\xE6\xAE\x47\xB4\x7D"
			  "\x1D\xA8\xBE\x81\x1A\x75\xDA\xAC"
			  "\xA6\x25\x1E\xEF\x3A\xC0\x6C\x63"
			  "\xEF\xDC\xC9\x79\x10\x26\xE8\x61"
			  "\x29\xFC\xA4\x05\xDF\x7D\x5C\x63"
			  "\x10\x09\x9B\x46\x9B\xF2\x2C\x2B"
			  "\xFA\x3A\x05\x4C\xFA\xD1\xFF\xFE"
			  "\xF1\x4C\xE5\xB2\x91\x64\x0C\x51",
		.rlen	= 496,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 496 - 16, 16 },
	}, { /* Generated with Crypto++ */
		.key	= "\xC9\x83\xA6\xC9\xEC\x0F\x32\x55"
			  "\x0F\x32\x55\x78\x9B\xBE\x78\x9B"
			  "\xBE\xE1\x04\x27\xE1\x04\x27\x4A"
			  "\x6D\x90\x4A\x6D\x90\xB3\xD6\xF9",
		.klen	= 32,
		.iv	= "\xE7\x82\x1D\xB8\x53\x11\xAC\x47"
			  "\xE2\x7D\x18\xD6\x71\x0C\xA7\x42",
		.input	= "\x50\xB9\x22\xAE\x17\x80\x0C\x75"
			  "\xDE\x47\xD3\x3C\xA5\x0E\x9A\x03"
			  "\x6C\xF8\x61\xCA\x33\xBF\x28\x91"
			  "\x1D\x86\xEF\x58\xE4\x4D\xB6\x1F"
			  "\xAB\x14\x7D\x09\x72\xDB\x44\xD0"
			  "\x39\xA2\x0B\x97\x00\x69\xF5\x5E"
			  "\xC7\x30\xBC\x25\x8E\x1A\x83\xEC"
			  "\x55\xE1\x4A\xB3\x1C\xA8\x11\x7A"
			  "\x06\x6F\xD8\x41\xCD\x36\x9F\x08"
			  "\x94\xFD\x66\xF2\x5B\xC4\x2D\xB9"
			  "\x22\x8B\x17\x80\xE9\x52\xDE\x47"
			  "\xB0\x19\xA5\x0E\x77\x03\x6C\xD5"
			  "\x3E\xCA\x33\x9C\x05\x91\xFA\x63"
			  "\xEF\x58\xC1\x2A\xB6\x1F\x88\x14"
			  "\x7D\xE6\x4F\xDB\x44\xAD\x16\xA2"
			  "\x0B\x74\x00\x69\xD2\x3B\xC7\x30"
			  "\x99\x02\x8E\xF7\x60\xEC\x55\xBE"
			  "\x27\xB3\x1C\x85\x11\x7A\xE3\x4C"
			  "\xD8\x41\xAA\x13\x9F\x08\x71\xFD"
			  "\x66\xCF\x38\xC4\x2D\x96\x22\x8B"
			  "\xF4\x5D\xE9\x52\xBB\x24\xB0\x19"
			  "\x82\x0E\x77\xE0\x49\xD5\x3E\xA7"
			  "\x10\x9C\x05\x6E\xFA\x63\xCC\x35"
			  "\xC1\x2A\x93\x1F\x88\xF1\x5A\xE6"
			  "\x4F\xB8\x21\xAD\x16\x7F\x0B\x74"
			  "\xDD\x46\xD2\x3B\xA4\x0D\x99\x02"
			  "\x6B\xF7\x60\xC9\x32\xBE\x27\x90"
			  "\x1C\x85\xEE\x57\xE3\x4C\xB5\x1E"
			  "\xAA\x13\x7C\x08\x71\xDA\x43\xCF"
			  "\x38\xA1\x0A\x96\xFF\x68\xF4\x5D"
			  "\xC6\x2F\xBB\x24\x8D\x19\x82\xEB"
			  "\x54\xE0\x49\xB2\x1B\xA7\x10\x79"
			  "\x05\x6E\xD7\x40\xCC\x35\x9E\x07"
			  "\x93\xFC\x65\xF1\x5A\xC3\x2C\xB8"
			  "\x21\x8A\x16\x7F\xE8\x51\xDD\x46"
			  "\xAF\x18\xA4\x0D\x76\x02\x6B\xD4"
			  "\x3D\xC9\x32\x9B\x04\x90\xF9\x62"
			  "\xEE\x57\xC0\x29\xB5\x1E\x87\x13"
			  "\x7C\xE5\x4E\xDA\x43\xAC\x15\xA1"
			  "\x0A\x73\xFF\x68\xD1\x3A\xC6\x2F"
			  "\x98\x01\x8D\xF6\x5F\xEB\x54\xBD"
			  "\x26\xB2\x1B\x84\x10\x79\xE2\x4B"
			  "\xD7\x40\xA9\x12\x9E\x07\x70\xFC"
			  "\x65\xCE\x37\xC3\x2C\x95\x21\x8A"
			  "\xF3\x5C\xE8\x51\xBA\x23\xAF\x18"
			  "\x81\x0D\x76\xDF\x48\xD4\x3D\xA6"
			  "\x0F\x9B\x04\x6D\xF9\x62\xCB\x34"
			  "\xC0\x29\x92\x1E\x87\xF0\x59\xE5"
			  "\x4E\xB7\x20\xAC\x15\x7E\x0A\x73"
			  "\xDC\x45\xD1\x3A\xA3\x0C\x98\x01"
			  "\x6A\xF6\x5F\xC8\x31\xBD\x26\x8F"
			  "\x1B\x84\xED\x56\xE2\x4B\xB4\x1D"
			  "\xA9\x12\x7B\x07\x70\xD9\x42\xCE"
			  "\x37\xA0\x09\x95\xFE\x67\xF3\x5C"
			  "\xC5\x2E\xBA\x23\x8C\x18\x81\xEA"
			  "\x53\xDF\x48\xB1\x1A\xA6\x0F\x78"
			  "\x04\x6D\xD6\x3F\xCB\x34\x9D\x06"
			  "\x92\xFB\x64\xF0\x59\xC2\x2B\xB7"
			  "\x20\x89\x15\x7E\xE7\x50\xDC\x45"
			  "\xAE\x17\xA3\x0C\x75\x01\x6A\xD3"
			  "\x3C\xC8\x31\x9A\x03\x8F\xF8\x61"
			  "\xED\x56\xBF\x28\xB4\x1D\x86\x12"
			  "\x7B\xE4\x4D",
		.ilen	= 499,
		.result	= "\xDA\x4E\x3F\xBC\xE8\xB6\x3A\xA2"
			  "\xD5\x4D\x84\x4A\xA9\x0C\xE1\xA5"
			  "\xB8\x73\xBC\xF9\xBB\x59\x2F\x44"
			  "\x8B\xAB\x82\x6C\xB4\x32\x9A\xDE"
			  "\x5A\x0B\xDB\x7A\x6B\xF2\x38\x9F"
			  "\x06\xF7\xF7\xFF\xFF\xC0\x8A\x2E"
			  "\x76\xEA\x06\x32\x23\xF3\x59\x2E"
			  "\x75\xDE\x71\x86\x3C\x98\x23\x44"
			  "\x5B\xF2\xFA\x6A\x00\xBB\xC1\xAD"
			  "\x58\xBD\x3E\x6F\x2E\xB4\x19\x04"
			  "\x70\x8B\x92\x55\x23\xE9\x6A\x3A"
			  "\x78\x7A\x1B\x10\x85\x52\x9C\x12"
			  "\xE4\x55\x81\x21\xCE\x53\xD0\x3B"
			  "\x63\x77\x2C\x74\xD1\xF5\x60\xF3"
			  "\xA1\xDE\x44\x3C\x8F\x4D\x2F\xDD"
			  "\x8A\xFE\x3C\x42\x8E\xD3\xF2\x8E"
			  "\xA8\x28\x69\x65\x31\xE1\x45\x83"
			  "\xE4\x49\xC4\x9C\xA7\x28\xAA\x21"
			  "\xCD\x5D\x0F\x15\xB7\x93\x07\x26"
			  "\xB0\x65\x6D\x91\x90\x23\x7A\xC6"
			  "\xDB\x68\xB0\xA1\x8E\xA4\x76\x4E"
			  "\xC6\x91\x83\x20\x92\x4D\x63\x7A"
			  "\x45\x18\x18\x74\x19\xAD\x71\x01"
			  "\x6B\x23\xAD\x9D\x4E\xE4\x6E\x46"
			  "\xC9\x73\x7A\xF9\x02\x95\xF4\x07"
			  "\x0E\x7A\xA6\xC5\xAE\xFA\x15\x2C"
			  "\x51\x71\xF1\xDC\x22\xB6\xAC\xD8"
			  "\x19\x24\x44\xBC\x0C\xFB\x3C\x2D"
			  "\xB1\x50\x47\x15\x0E\xDB\xB6\xD7"
			  "\xE8\x61\xE5\x95\x52\x1E\x3E\x49"
			  "\x70\xE9\x66\x04\x4C\xE1\xAF\xBD"
			  "\xDD\x15\x3B\x20\x59\x24\xFF\xB0"
			  "\x39\xAA\xE7\xBF\x23\xA3\x6E\xD5"
			  "\x15\xF0\x61\x4F\xAE\x89\x10\x58"
			  "\x5A\x33\x95\x52\x2A\xB5\x77\x9C"
			  "\xA5\x43\x80\x40\x27\x2D\xAE\xD9"
			  "\x3F\xE0\x80\x94\x78\x79\xCB\x7E"
			  "\xAD\x12\x44\x4C\xEC\x27\xB0\xEE"
			  "\x0B\x05\x2A\x82\x99\x58\xBB\x7A"
			  "\x8D\x6D\x9D\x8E\xE2\x8E\xE7\x93"
			  "\x2F\xB3\x09\x8D\x06\xD5\xEE\x70"
			  "\x16\xAE\x35\xC5\x52\x0F\x46\x1F"
			  "\x71\xF9\x5E\xF2\x67\xDC\x98\x2F"
			  "\xA3\x23\xAA\xD5\xD0\x49\xF4\xA6"
			  "\xF6\xB8\x32\xCD\xD6\x85\x73\x60"
			  "\x59\x20\xE7\x55\x0E\x91\xE2\x0C"
			  "\x3F\x1C\xEB\x3D\xDF\x52\x64\xF2"
			  "\x7D\x8B\x5D\x63\x16\xB9\xB2\x5D"
			  "\x5E\xAB\xB2\x97\xAB\x78\x44\xE7"
			  "\xC6\x72\x20\xC5\x90\x9B\xDC\x5D"
			  "\xB0\xEF\x44\xEF\x87\x31\x8D\xF4"
			  "\xFB\x81\x5D\xF7\x96\x96\xD4\x50"
			  "\x89\xA7\xF6\xB9\x67\x76\x40\x9E"
			  "\x9D\x40\xD5\x2C\x30\xB8\x01\x8F"
			  "\xE4\x7B\x71\x48\xA9\xA0\xA0\x1D"
			  "\x87\x52\xA4\x91\xA9\xD7\xA9\x51"
			  "\xD9\x59\xF7\xCC\x63\x22\xC1\x8D"
			  "\x84\x7B\xD8\x22\x32\x5C\x6F\x1D"
			  "\x6E\x9F\xFA\xDD\x49\x40\xDC\x37"
			  "\x14\x8C\xE1\x80\x1B\xDD\x36\x2A"
			  "\xD0\xE9\x54\x99\x5D\xBA\x3B\x11"
			  "\xD8\xFE\xC9\x5B\x5C\x25\xE5\x76"
			  "\xFB\xF2\x3F",
		.rlen	= 499,
		.also_non_np = 1,
		.np	= 2,
		.tap	= { 499 - 16, 16 },
	},
};

#if 0
static struct cipher_testvec aes_ctr_rfc3686_dec_tv_template[] = {
	{ /* From RFC 3686 */
		.key	= "\xae\x68\x52\xf8\x12\x10\x67\xcc"
			  "\x4b\xf7\xa5\x76\x55\x77\xf3\x9e"
			  "\x00\x00\x00\x30",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "\xe4\x09\x5d\x4f\xb7\xa7\xb3\x79"
			  "\x2d\x61\x75\xa3\x26\x13\x11\xb8",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\x7e\x24\x06\x78\x17\xfa\xe0\xd7"
			  "\x43\xd6\xce\x1f\x32\x53\x91\x63"
			  "\x00\x6c\xb6\xdb",
		.klen	= 20,
		.iv	= "\xc0\x54\x3b\x59\xda\x48\xd9\x0b",
		.input	= "\x51\x04\xa1\x06\x16\x8a\x72\xd9"
			  "\x79\x0d\x41\xee\x8e\xda\xd3\x88"
			  "\xeb\x2e\x1e\xfc\x46\xda\x57\xc8"
			  "\xfc\xe6\x30\xdf\x91\x41\xbe\x28",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	}, {
		.key	= "\x16\xaf\x5b\x14\x5f\xc9\xf5\x79"
			  "\xc1\x75\xf9\x3e\x3b\xfb\x0e\xed"
			  "\x86\x3d\x06\xcc\xfd\xb7\x85\x15"
			  "\x00\x00\x00\x48",
		.klen	= 28,
		.iv	= "\x36\x73\x3c\x14\x7d\x6d\x93\xcb",
		.input	= "\x4b\x55\x38\x4f\xe2\x59\xc9\xc8"
			  "\x4e\x79\x35\xa0\x03\xcb\xe9\x28",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\x7c\x5c\xb2\x40\x1b\x3d\xc3\x3c"
			  "\x19\xe7\x34\x08\x19\xe0\xf6\x9c"
			  "\x67\x8c\x3d\xb8\xe6\xf6\xa9\x1a"
			  "\x00\x96\xb0\x3b",
		.klen	= 28,
		.iv	= "\x02\x0c\x6e\xad\xc2\xcb\x50\x0d",
		.input	= "\x45\x32\x43\xfc\x60\x9b\x23\x32"
			  "\x7e\xdf\xaa\xfa\x71\x31\xcd\x9f"
			  "\x84\x90\x70\x1c\x5a\xd4\xa7\x9c"
			  "\xfc\x1f\xe0\xff\x42\xf4\xfb\x00",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	}, {
		.key	= "\x77\x6b\xef\xf2\x85\x1d\xb0\x6f"
			  "\x4c\x8a\x05\x42\xc8\x69\x6f\x6c"
			  "\x6a\x81\xaf\x1e\xec\x96\xb4\xd3"
			  "\x7f\xc1\xd6\x89\xe6\xc1\xc1\x04"
			  "\x00\x00\x00\x60",
		.klen	= 36,
		.iv	= "\xdb\x56\x72\xc9\x7a\xa8\xf0\xb2",
		.input	= "\x14\x5a\xd0\x1d\xbf\x82\x4e\xc7"
			  "\x56\x08\x63\xdc\x71\xe3\xe0\xc0",
		.ilen	= 16,
		.result	= "Single block msg",
		.rlen	= 16,
	}, {
		.key	= "\xf6\xd6\x6d\x6b\xd5\x2d\x59\xbb"
			  "\x07\x96\x36\x58\x79\xef\xf8\x86"
			  "\xc6\x6d\xd5\x1a\x5b\x6a\x99\x74"
			  "\x4b\x50\x59\x0c\x87\xa2\x38\x84"
			  "\x00\xfa\xac\x24",
		.klen	= 36,
		.iv	= "\xc1\x58\x5e\xf1\x5a\x43\xd8\x75",
		.input	= "\xf0\x5e\x23\x1b\x38\x94\x61\x2c"
			  "\x49\xee\x00\x0b\x80\x4e\xb2\xa9"
			  "\xb8\x30\x6b\x50\x8f\x83\x9d\x6a"
			  "\x55\x30\x83\x1d\x93\x44\xaf\x1c",
		.ilen	= 32,
		.result	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.rlen	= 32,
	},
};

static struct cipher_testvec aes_ctr_rfc3686_enc_tv_template[] = {
	{ /* From RFC 3686 */
		.key	= "\xae\x68\x52\xf8\x12\x10\x67\xcc"
			  "\x4b\xf7\xa5\x76\x55\x77\xf3\x9e"
			  "\x00\x00\x00\x30",
		.klen	= 20,
		.iv	= "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input	= "Single block msg",
		.ilen	= 16,
		.result = "\xe4\x09\x5d\x4f\xb7\xa7\xb3\x79"
			  "\x2d\x61\x75\xa3\x26\x13\x11\xb8",
		.rlen	= 16,
	}, {
		.key	= "\x7e\x24\x06\x78\x17\xfa\xe0\xd7"
			  "\x43\xd6\xce\x1f\x32\x53\x91\x63"
			  "\x00\x6c\xb6\xdb",
		.klen	= 20,
		.iv	= "\xc0\x54\x3b\x59\xda\x48\xd9\x0b",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result = "\x51\x04\xa1\x06\x16\x8a\x72\xd9"
			  "\x79\x0d\x41\xee\x8e\xda\xd3\x88"
			  "\xeb\x2e\x1e\xfc\x46\xda\x57\xc8"
			  "\xfc\xe6\x30\xdf\x91\x41\xbe\x28",
		.rlen	= 32,
	}, {
		.key	= "\x16\xaf\x5b\x14\x5f\xc9\xf5\x79"
			  "\xc1\x75\xf9\x3e\x3b\xfb\x0e\xed"
			  "\x86\x3d\x06\xcc\xfd\xb7\x85\x15"
			  "\x00\x00\x00\x48",
		.klen	= 28,
		.iv	= "\x36\x73\x3c\x14\x7d\x6d\x93\xcb",
		.input	= "Single block msg",
		.ilen	= 16,
		.result	= "\x4b\x55\x38\x4f\xe2\x59\xc9\xc8"
			  "\x4e\x79\x35\xa0\x03\xcb\xe9\x28",
		.rlen	= 16,
	}, {
		.key	= "\x7c\x5c\xb2\x40\x1b\x3d\xc3\x3c"
			  "\x19\xe7\x34\x08\x19\xe0\xf6\x9c"
			  "\x67\x8c\x3d\xb8\xe6\xf6\xa9\x1a"
			  "\x00\x96\xb0\x3b",
		.klen	= 28,
		.iv	= "\x02\x0c\x6e\xad\xc2\xcb\x50\x0d",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result	= "\x45\x32\x43\xfc\x60\x9b\x23\x32"
			  "\x7e\xdf\xaa\xfa\x71\x31\xcd\x9f"
			  "\x84\x90\x70\x1c\x5a\xd4\xa7\x9c"
			  "\xfc\x1f\xe0\xff\x42\xf4\xfb\x00",
		.rlen	= 32,
	}, {
		.key	= "\x77\x6b\xef\xf2\x85\x1d\xb0\x6f"
			  "\x4c\x8a\x05\x42\xc8\x69\x6f\x6c"
			  "\x6a\x81\xaf\x1e\xec\x96\xb4\xd3"
			  "\x7f\xc1\xd6\x89\xe6\xc1\xc1\x04"
			  "\x00\x00\x00\x60",
		.klen	= 36,
		.iv	= "\xdb\x56\x72\xc9\x7a\xa8\xf0\xb2",
		.input	= "Single block msg",
		.ilen	= 16,
		.result = "\x14\x5a\xd0\x1d\xbf\x82\x4e\xc7"
			  "\x56\x08\x63\xdc\x71\xe3\xe0\xc0",
		.rlen	= 16,
	}, {
		.key	= "\xf6\xd6\x6d\x6b\xd5\x2d\x59\xbb"
			  "\x07\x96\x36\x58\x79\xef\xf8\x86"
			  "\xc6\x6d\xd5\x1a\x5b\x6a\x99\x74"
			  "\x4b\x50\x59\x0c\x87\xa2\x38\x84"
			  "\x00\xfa\xac\x24",
		.klen	= 36,
		.iv	= "\xc1\x58\x5e\xf1\x5a\x43\xd8\x75",
		.input	= "\x00\x01\x02\x03\x04\x05\x06\x07"
			  "\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			  "\x10\x11\x12\x13\x14\x15\x16\x17"
			  "\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f",
		.ilen	= 32,
		.result = "\xf0\x5e\x23\x1b\x38\x94\x61\x2c"
			  "\x49\xee\x00\x0b\x80\x4e\xb2\xa9"
			  "\xb8\x30\x6b\x50\x8f\x83\x9d\x6a"
			  "\x55\x30\x83\x1d\x93\x44\xaf\x1c",
		.rlen	= 32,
	}, {
	// generated using Crypto++
		.key = "\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x00\x00\x00\x00",
		.klen = 32 + 4,
		.iv = "\x00\x00\x00\x00\x00\x00\x00\x00",
		.input =
			"\x00\x01\x02\x03\x04\x05\x06\x07"
			"\x08\x09\x0a\x0b\x0c\x0d\x0e\x0f"
			"\x10\x11\x12\x13\x14\x15\x16\x17"
			"\x18\x19\x1a\x1b\x1c\x1d\x1e\x1f"
			"\x20\x21\x22\x23\x24\x25\x26\x27"
			"\x28\x29\x2a\x2b\x2c\x2d\x2e\x2f"
			"\x30\x31\x32\x33\x34\x35\x36\x37"
			"\x38\x39\x3a\x3b\x3c\x3d\x3e\x3f"
			"\x40\x41\x42\x43\x44\x45\x46\x47"
			"\x48\x49\x4a\x4b\x4c\x4d\x4e\x4f"
			"\x50\x51\x52\x53\x54\x55\x56\x57"
			"\x58\x59\x5a\x5b\x5c\x5d\x5e\x5f"
			"\x60\x61\x62\x63\x64\x65\x66\x67"
			"\x68\x69\x6a\x6b\x6c\x6d\x6e\x6f"
			"\x70\x71\x72\x73\x74\x75\x76\x77"
			"\x78\x79\x7a\x7b\x7c\x7d\x7e\x7f"
			"\x80\x81\x82\x83\x84\x85\x86\x87"
			"\x88\x89\x8a\x8b\x8c\x8d\x8e\x8f"
			"\x90\x91\x92\x93\x94\x95\x96\x97"
			"\x98\x99\x9a\x9b\x9c\x9d\x9e\x9f"
			"\xa0\xa1\xa2\xa3\xa4\xa5\xa6\xa7"
			"\xa8\xa9\xaa\xab\xac\xad\xae\xaf"
			"\xb0\xb1\xb2\xb3\xb4\xb5\xb6\xb7"
			"\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf"
			"\xc0\xc1\xc2\xc3\xc4\xc5\xc6\xc7"
			"\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf"
			"\xd0\xd1\xd2\xd3\xd4\xd5\xd6\xd7"
			"\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf"
			"\xe0\xe1\xe2\xe3\xe4\xe5\xe6\xe7"
			"\xe8\xe9\xea\xeb\xec\xed\xee\xef"
			"\xf0\xf1\xf2\xf3\xf4\xf5\xf6\xf7"
			"\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"
			"\x00\x03\x06\x09\x0c\x0f\x12\x15"
			"\x18\x1b\x1e\x21\x24\x27\x2a\x2d"
			"\x30\x33\x36\x39\x3c\x3f\x42\x45"
			"\x48\x4b\x4e\x51\x54\x57\x5a\x5d"
			"\x60\x63\x66\x69\x6c\x6f\x72\x75"
			"\x78\x7b\x7e\x81\x84\x87\x8a\x8d"
			"\x90\x93\x96\x99\x9c\x9f\xa2\xa5"
			"\xa8\xab\xae\xb1\xb4\xb7\xba\xbd"
			"\xc0\xc3\xc6\xc9\xcc\xcf\xd2\xd5"
			"\xd8\xdb\xde\xe1\xe4\xe7\xea\xed"
			"\xf0\xf3\xf6\xf9\xfc\xff\x02\x05"
			"\x08\x0b\x0e\x11\x14\x17\x1a\x1d"
			"\x20\x23\x26\x29\x2c\x2f\x32\x35"
			"\x38\x3b\x3e\x41\x44\x47\x4a\x4d"
			"\x50\x53\x56\x59\x5c\x5f\x62\x65"
			"\x68\x6b\x6e\x71\x74\x77\x7a\x7d"
			"\x80\x83\x86\x89\x8c\x8f\x92\x95"
			"\x98\x9b\x9e\xa1\xa4\xa7\xaa\xad"
			"\xb0\xb3\xb6\xb9\xbc\xbf\xc2\xc5"
			"\xc8\xcb\xce\xd1\xd4\xd7\xda\xdd"
			"\xe0\xe3\xe6\xe9\xec\xef\xf2\xf5"
			"\xf8\xfb\xfe\x01\x04\x07\x0a\x0d"
			"\x10\x13\x16\x19\x1c\x1f\x22\x25"
			"\x28\x2b\x2e\x31\x34\x37\x3a\x3d"
			"\x40\x43\x46\x49\x4c\x4f\x52\x55"
			"\x58\x5b\x5e\x61\x64\x67\x6a\x6d"
			"\x70\x73\x76\x79\x7c\x7f\x82\x85"
			"\x88\x8b\x8e\x91\x94\x97\x9a\x9d"
			"\xa0\xa3\xa6\xa9\xac\xaf\xb2\xb5"
			"\xb8\xbb\xbe\xc1\xc4\xc7\xca\xcd"
			"\xd0\xd3\xd6\xd9\xdc\xdf\xe2\xe5"
			"\xe8\xeb\xee\xf1\xf4\xf7\xfa\xfd"
			"\x00\x05\x0a\x0f\x14\x19\x1e\x23"
			"\x28\x2d\x32\x37\x3c\x41\x46\x4b"
			"\x50\x55\x5a\x5f\x64\x69\x6e\x73"
			"\x78\x7d\x82\x87\x8c\x91\x96\x9b"
			"\xa0\xa5\xaa\xaf\xb4\xb9\xbe\xc3"
			"\xc8\xcd\xd2\xd7\xdc\xe1\xe6\xeb"
			"\xf0\xf5\xfa\xff\x04\x09\x0e\x13"
			"\x18\x1d\x22\x27\x2c\x31\x36\x3b"
			"\x40\x45\x4a\x4f\x54\x59\x5e\x63"
			"\x68\x6d\x72\x77\x7c\x81\x86\x8b"
			"\x90\x95\x9a\x9f\xa4\xa9\xae\xb3"
			"\xb8\xbd\xc2\xc7\xcc\xd1\xd6\xdb"
			"\xe0\xe5\xea\xef\xf4\xf9\xfe\x03"
			"\x08\x0d\x12\x17\x1c\x21\x26\x2b"
			"\x30\x35\x3a\x3f\x44\x49\x4e\x53"
			"\x58\x5d\x62\x67\x6c\x71\x76\x7b"
			"\x80\x85\x8a\x8f\x94\x99\x9e\xa3"
			"\xa8\xad\xb2\xb7\xbc\xc1\xc6\xcb"
			"\xd0\xd5\xda\xdf\xe4\xe9\xee\xf3"
			"\xf8\xfd\x02\x07\x0c\x11\x16\x1b"
			"\x20\x25\x2a\x2f\x34\x39\x3e\x43"
			"\x48\x4d\x52\x57\x5c\x61\x66\x6b"
			"\x70\x75\x7a\x7f\x84\x89\x8e\x93"
			"\x98\x9d\xa2\xa7\xac\xb1\xb6\xbb"
			"\xc0\xc5\xca\xcf\xd4\xd9\xde\xe3"
			"\xe8\xed\xf2\xf7\xfc\x01\x06\x0b"
			"\x10\x15\x1a\x1f\x24\x29\x2e\x33"
			"\x38\x3d\x42\x47\x4c\x51\x56\x5b"
			"\x60\x65\x6a\x6f\x74\x79\x7e\x83"
			"\x88\x8d\x92\x97\x9c\xa1\xa6\xab"
			"\xb0\xb5\xba\xbf\xc4\xc9\xce\xd3"
			"\xd8\xdd\xe2\xe7\xec\xf1\xf6\xfb"
			"\x00\x07\x0e\x15\x1c\x23\x2a\x31"
			"\x38\x3f\x46\x4d\x54\x5b\x62\x69"
			"\x70\x77\x7e\x85\x8c\x93\x9a\xa1"
			"\xa8\xaf\xb6\xbd\xc4\xcb\xd2\xd9"
			"\xe0\xe7\xee\xf5\xfc\x03\x0a\x11"
			"\x18\x1f\x26\x2d\x34\x3b\x42\x49"
			"\x50\x57\x5e\x65\x6c\x73\x7a\x81"
			"\x88\x8f\x96\x9d\xa4\xab\xb2\xb9"
			"\xc0\xc7\xce\xd5\xdc\xe3\xea\xf1"
			"\xf8\xff\x06\x0d\x14\x1b\x22\x29"
			"\x30\x37\x3e\x45\x4c\x53\x5a\x61"
			"\x68\x6f\x76\x7d\x84\x8b\x92\x99"
			"\xa0\xa7\xae\xb5\xbc\xc3\xca\xd1"
			"\xd8\xdf\xe6\xed\xf4\xfb\x02\x09"
			"\x10\x17\x1e\x25\x2c\x33\x3a\x41"
			"\x48\x4f\x56\x5d\x64\x6b\x72\x79"
			"\x80\x87\x8e\x95\x9c\xa3\xaa\xb1"
			"\xb8\xbf\xc6\xcd\xd4\xdb\xe2\xe9"
			"\xf0\xf7\xfe\x05\x0c\x13\x1a\x21"
			"\x28\x2f\x36\x3d\x44\x4b\x52\x59"
			"\x60\x67\x6e\x75\x7c\x83\x8a\x91"
			"\x98\x9f\xa6\xad\xb4\xbb\xc2\xc9"
			"\xd0\xd7\xde\xe5\xec\xf3\xfa\x01"
			"\x08\x0f\x16\x1d\x24\x2b\x32\x39"
			"\x40\x47\x4e\x55\x5c\x63\x6a\x71"
			"\x78\x7f\x86\x8d\x94\x9b\xa2\xa9"
			"\xb0\xb7\xbe\xc5\xcc\xd3\xda\xe1"
			"\xe8\xef\xf6\xfd\x04\x0b\x12\x19"
			"\x20\x27\x2e\x35\x3c\x43\x4a\x51"
			"\x58\x5f\x66\x6d\x74\x7b\x82\x89"
			"\x90\x97\x9e\xa5\xac\xb3\xba\xc1"
			"\xc8\xcf\xd6\xdd\xe4\xeb\xf2\xf9"
			"\x00\x09\x12\x1b\x24\x2d\x36\x3f"
			"\x48\x51\x5a\x63\x6c\x75\x7e\x87"
			"\x90\x99\xa2\xab\xb4\xbd\xc6\xcf"
			"\xd8\xe1\xea\xf3\xfc\x05\x0e\x17"
			"\x20\x29\x32\x3b\x44\x4d\x56\x5f"
			"\x68\x71\x7a\x83\x8c\x95\x9e\xa7"
			"\xb0\xb9\xc2\xcb\xd4\xdd\xe6\xef"
			"\xf8\x01\x0a\x13\x1c\x25\x2e\x37"
			"\x40\x49\x52\x5b\x64\x6d\x76\x7f"
			"\x88\x91\x9a\xa3\xac\xb5\xbe\xc7"
			"\xd0\xd9\xe2\xeb\xf4\xfd\x06\x0f"
			"\x18\x21\x2a\x33\x3c\x45\x4e\x57"
			"\x60\x69\x72\x7b\x84\x8d\x96\x9f"
			"\xa8\xb1\xba\xc3\xcc\xd5\xde\xe7"
			"\xf0\xf9\x02\x0b\x14\x1d\x26\x2f"
			"\x38\x41\x4a\x53\x5c\x65\x6e\x77"
			"\x80\x89\x92\x9b\xa4\xad\xb6\xbf"
			"\xc8\xd1\xda\xe3\xec\xf5\xfe\x07"
			"\x10\x19\x22\x2b\x34\x3d\x46\x4f"
			"\x58\x61\x6a\x73\x7c\x85\x8e\x97"
			"\xa0\xa9\xb2\xbb\xc4\xcd\xd6\xdf"
			"\xe8\xf1\xfa\x03\x0c\x15\x1e\x27"
			"\x30\x39\x42\x4b\x54\x5d\x66\x6f"
			"\x78\x81\x8a\x93\x9c\xa5\xae\xb7"
			"\xc0\xc9\xd2\xdb\xe4\xed\xf6\xff"
			"\x08\x11\x1a\x23\x2c\x35\x3e\x47"
			"\x50\x59\x62\x6b\x74\x7d\x86\x8f"
			"\x98\xa1\xaa\xb3\xbc\xc5\xce\xd7"
			"\xe0\xe9\xf2\xfb\x04\x0d\x16\x1f"
			"\x28\x31\x3a\x43\x4c\x55\x5e\x67"
			"\x70\x79\x82\x8b\x94\x9d\xa6\xaf"
			"\xb8\xc1\xca\xd3\xdc\xe5\xee\xf7"
			"\x00\x0b\x16\x21\x2c\x37\x42\x4d"
			"\x58\x63\x6e\x79\x84\x8f\x9a\xa5"
			"\xb0\xbb\xc6\xd1\xdc\xe7\xf2\xfd"
			"\x08\x13\x1e\x29\x34\x3f\x4a\x55"
			"\x60\x6b\x76\x81\x8c\x97\xa2\xad"
			"\xb8\xc3\xce\xd9\xe4\xef\xfa\x05"
			"\x10\x1b\x26\x31\x3c\x47\x52\x5d"
			"\x68\x73\x7e\x89\x94\x9f\xaa\xb5"
			"\xc0\xcb\xd6\xe1\xec\xf7\x02\x0d"
			"\x18\x23\x2e\x39\x44\x4f\x5a\x65"
			"\x70\x7b\x86\x91\x9c\xa7\xb2\xbd"
			"\xc8\xd3\xde\xe9\xf4\xff\x0a\x15"
			"\x20\x2b\x36\x41\x4c\x57\x62\x6d"
			"\x78\x83\x8e\x99\xa4\xaf\xba\xc5"
			"\xd0\xdb\xe6\xf1\xfc\x07\x12\x1d"
			"\x28\x33\x3e\x49\x54\x5f\x6a\x75"
			"\x80\x8b\x96\xa1\xac\xb7\xc2\xcd"
			"\xd8\xe3\xee\xf9\x04\x0f\x1a\x25"
			"\x30\x3b\x46\x51\x5c\x67\x72\x7d"
			"\x88\x93\x9e\xa9\xb4\xbf\xca\xd5"
			"\xe0\xeb\xf6\x01\x0c\x17\x22\x2d"
			"\x38\x43\x4e\x59\x64\x6f\x7a\x85"
			"\x90\x9b\xa6\xb1\xbc\xc7\xd2\xdd"
			"\xe8\xf3\xfe\x09\x14\x1f\x2a\x35"
			"\x40\x4b\x56\x61\x6c\x77\x82\x8d"
			"\x98\xa3\xae\xb9\xc4\xcf\xda\xe5"
			"\xf0\xfb\x06\x11\x1c\x27\x32\x3d"
			"\x48\x53\x5e\x69\x74\x7f\x8a\x95"
			"\xa0\xab\xb6\xc1\xcc\xd7\xe2\xed"
			"\xf8\x03\x0e\x19\x24\x2f\x3a\x45"
			"\x50\x5b\x66\x71\x7c\x87\x92\x9d"
			"\xa8\xb3\xbe\xc9\xd4\xdf\xea\xf5"
			"\x00\x0d\x1a\x27\x34\x41\x4e\x5b"
			"\x68\x75\x82\x8f\x9c\xa9\xb6\xc3"
			"\xd0\xdd\xea\xf7\x04\x11\x1e\x2b"
			"\x38\x45\x52\x5f\x6c\x79\x86\x93"
			"\xa0\xad\xba\xc7\xd4\xe1\xee\xfb"
			"\x08\x15\x22\x2f\x3c\x49\x56\x63"
			"\x70\x7d\x8a\x97\xa4\xb1\xbe\xcb"
			"\xd8\xe5\xf2\xff\x0c\x19\x26\x33"
			"\x40\x4d\x5a\x67\x74\x81\x8e\x9b"
			"\xa8\xb5\xc2\xcf\xdc\xe9\xf6\x03"
			"\x10\x1d\x2a\x37\x44\x51\x5e\x6b"
			"\x78\x85\x92\x9f\xac\xb9\xc6\xd3"
			"\xe0\xed\xfa\x07\x14\x21\x2e\x3b"
			"\x48\x55\x62\x6f\x7c\x89\x96\xa3"
			"\xb0\xbd\xca\xd7\xe4\xf1\xfe\x0b"
			"\x18\x25\x32\x3f\x4c\x59\x66\x73"
			"\x80\x8d\x9a\xa7\xb4\xc1\xce\xdb"
			"\xe8\xf5\x02\x0f\x1c\x29\x36\x43"
			"\x50\x5d\x6a\x77\x84\x91\x9e\xab"
			"\xb8\xc5\xd2\xdf\xec\xf9\x06\x13"
			"\x20\x2d\x3a\x47\x54\x61\x6e\x7b"
			"\x88\x95\xa2\xaf\xbc\xc9\xd6\xe3"
			"\xf0\xfd\x0a\x17\x24\x31\x3e\x4b"
			"\x58\x65\x72\x7f\x8c\x99\xa6\xb3"
			"\xc0\xcd\xda\xe7\xf4\x01\x0e\x1b"
			"\x28\x35\x42\x4f\x5c\x69\x76\x83"
			"\x90\x9d\xaa\xb7\xc4\xd1\xde\xeb"
			"\xf8\x05\x12\x1f\x2c\x39\x46\x53"
			"\x60\x6d\x7a\x87\x94\xa1\xae\xbb"
			"\xc8\xd5\xe2\xef\xfc\x09\x16\x23"
			"\x30\x3d\x4a\x57\x64\x71\x7e\x8b"
			"\x98\xa5\xb2\xbf\xcc\xd9\xe6\xf3"
			"\x00\x0f\x1e\x2d\x3c\x4b\x5a\x69"
			"\x78\x87\x96\xa5\xb4\xc3\xd2\xe1"
			"\xf0\xff\x0e\x1d\x2c\x3b\x4a\x59"
			"\x68\x77\x86\x95\xa4\xb3\xc2\xd1"
			"\xe0\xef\xfe\x0d\x1c\x2b\x3a\x49"
			"\x58\x67\x76\x85\x94\xa3\xb2\xc1"
			"\xd0\xdf\xee\xfd\x0c\x1b\x2a\x39"
			"\x48\x57\x66\x75\x84\x93\xa2\xb1"
			"\xc0\xcf\xde\xed\xfc\x0b\x1a\x29"
			"\x38\x47\x56\x65\x74\x83\x92\xa1"
			"\xb0\xbf\xce\xdd\xec\xfb\x0a\x19"
			"\x28\x37\x46\x55\x64\x73\x82\x91"
			"\xa0\xaf\xbe\xcd\xdc\xeb\xfa\x09"
			"\x18\x27\x36\x45\x54\x63\x72\x81"
			"\x90\x9f\xae\xbd\xcc\xdb\xea\xf9"
			"\x08\x17\x26\x35\x44\x53\x62\x71"
			"\x80\x8f\x9e\xad\xbc\xcb\xda\xe9"
			"\xf8\x07\x16\x25\x34\x43\x52\x61"
			"\x70\x7f\x8e\x9d\xac\xbb\xca\xd9"
			"\xe8\xf7\x06\x15\x24\x33\x42\x51"
			"\x60\x6f\x7e\x8d\x9c\xab\xba\xc9"
			"\xd8\xe7\xf6\x05\x14\x23\x32\x41"
			"\x50\x5f\x6e\x7d\x8c\x9b\xaa\xb9"
			"\xc8\xd7\xe6\xf5\x04\x13\x22\x31"
			"\x40\x4f\x5e\x6d\x7c\x8b\x9a\xa9"
			"\xb8\xc7\xd6\xe5\xf4\x03\x12\x21"
			"\x30\x3f\x4e\x5d\x6c\x7b\x8a\x99"
			"\xa8\xb7\xc6\xd5\xe4\xf3\x02\x11"
			"\x20\x2f\x3e\x4d\x5c\x6b\x7a\x89"
			"\x98\xa7\xb6\xc5\xd4\xe3\xf2\x01"
			"\x10\x1f\x2e\x3d\x4c\x5b\x6a\x79"
			"\x88\x97\xa6\xb5\xc4\xd3\xe2\xf1"
			"\x00\x11\x22\x33\x44\x55\x66\x77"
			"\x88\x99\xaa\xbb\xcc\xdd\xee\xff"
			"\x10\x21\x32\x43\x54\x65\x76\x87"
			"\x98\xa9\xba\xcb\xdc\xed\xfe\x0f"
			"\x20\x31\x42\x53\x64\x75\x86\x97"
			"\xa8\xb9\xca\xdb\xec\xfd\x0e\x1f"
			"\x30\x41\x52\x63\x74\x85\x96\xa7"
			"\xb8\xc9\xda\xeb\xfc\x0d\x1e\x2f"
			"\x40\x51\x62\x73\x84\x95\xa6\xb7"
			"\xc8\xd9\xea\xfb\x0c\x1d\x2e\x3f"
			"\x50\x61\x72\x83\x94\xa5\xb6\xc7"
			"\xd8\xe9\xfa\x0b\x1c\x2d\x3e\x4f"
			"\x60\x71\x82\x93\xa4\xb5\xc6\xd7"
			"\xe8\xf9\x0a\x1b\x2c\x3d\x4e\x5f"
			"\x70\x81\x92\xa3\xb4\xc5\xd6\xe7"
			"\xf8\x09\x1a\x2b\x3c\x4d\x5e\x6f"
			"\x80\x91\xa2\xb3\xc4\xd5\xe6\xf7"
			"\x08\x19\x2a\x3b\x4c\x5d\x6e\x7f"
			"\x90\xa1\xb2\xc3\xd4\xe5\xf6\x07"
			"\x18\x29\x3a\x4b\x5c\x6d\x7e\x8f"
			"\xa0\xb1\xc2\xd3\xe4\xf5\x06\x17"
			"\x28\x39\x4a\x5b\x6c\x7d\x8e\x9f"
			"\xb0\xc1\xd2\xe3\xf4\x05\x16\x27"
			"\x38\x49\x5a\x6b\x7c\x8d\x9e\xaf"
			"\xc0\xd1\xe2\xf3\x04\x15\x26\x37"
			"\x48\x59\x6a\x7b\x8c\x9d\xae\xbf"
			"\xd0\xe1\xf2\x03\x14\x25\x36\x47"
			"\x58\x69\x7a\x8b\x9c\xad\xbe\xcf"
			"\xe0\xf1\x02\x13\x24\x35\x46\x57"
			"\x68\x79\x8a\x9b\xac\xbd\xce\xdf"
			"\xf0\x01\x12\x23\x34\x45\x56\x67"
			"\x78\x89\x9a\xab\xbc\xcd\xde\xef"
			"\x00\x13\x26\x39\x4c\x5f\x72\x85"
			"\x98\xab\xbe\xd1\xe4\xf7\x0a\x1d"
			"\x30\x43\x56\x69\x7c\x8f\xa2\xb5"
			"\xc8\xdb\xee\x01\x14\x27\x3a\x4d"
			"\x60\x73\x86\x99\xac\xbf\xd2\xe5"
			"\xf8\x0b\x1e\x31\x44\x57\x6a\x7d"
			"\x90\xa3\xb6\xc9\xdc\xef\x02\x15"
			"\x28\x3b\x4e\x61\x74\x87\x9a\xad"
			"\xc0\xd3\xe6\xf9\x0c\x1f\x32\x45"
			"\x58\x6b\x7e\x91\xa4\xb7\xca\xdd"
			"\xf0\x03\x16\x29\x3c\x4f\x62\x75"
			"\x88\x9b\xae\xc1\xd4\xe7\xfa\x0d"
			"\x20\x33\x46\x59\x6c\x7f\x92\xa5"
			"\xb8\xcb\xde\xf1\x04\x17\x2a\x3d"
			"\x50\x63\x76\x89\x9c\xaf\xc2\xd5"
			"\xe8\xfb\x0e\x21\x34\x47\x5a\x6d"
			"\x80\x93\xa6\xb9\xcc\xdf\xf2\x05"
			"\x18\x2b\x3e\x51\x64\x77\x8a\x9d"
			"\xb0\xc3\xd6\xe9\xfc\x0f\x22\x35"
			"\x48\x5b\x6e\x81\x94\xa7\xba\xcd"
			"\xe0\xf3\x06\x19\x2c\x3f\x52\x65"
			"\x78\x8b\x9e\xb1\xc4\xd7\xea\xfd"
			"\x10\x23\x36\x49\x5c\x6f\x82\x95"
			"\xa8\xbb\xce\xe1\xf4\x07\x1a\x2d"
			"\x40\x53\x66\x79\x8c\x9f\xb2\xc5"
			"\xd8\xeb\xfe\x11\x24\x37\x4a\x5d"
			"\x70\x83\x96\xa9\xbc\xcf\xe2\xf5"
			"\x08\x1b\x2e\x41\x54\x67\x7a\x8d"
			"\xa0\xb3\xc6\xd9\xec\xff\x12\x25"
			"\x38\x4b\x5e\x71\x84\x97\xaa\xbd"
			"\xd0\xe3\xf6\x09\x1c\x2f\x42\x55"
			"\x68\x7b\x8e\xa1\xb4\xc7\xda\xed"
			"\x00\x15\x2a\x3f\x54\x69\x7e\x93"
			"\xa8\xbd\xd2\xe7\xfc\x11\x26\x3b"
			"\x50\x65\x7a\x8f\xa4\xb9\xce\xe3"
			"\xf8\x0d\x22\x37\x4c\x61\x76\x8b"
			"\xa0\xb5\xca\xdf\xf4\x09\x1e\x33"
			"\x48\x5d\x72\x87\x9c\xb1\xc6\xdb"
			"\xf0\x05\x1a\x2f\x44\x59\x6e\x83"
			"\x98\xad\xc2\xd7\xec\x01\x16\x2b"
			"\x40\x55\x6a\x7f\x94\xa9\xbe\xd3"
			"\xe8\xfd\x12\x27\x3c\x51\x66\x7b"
			"\x90\xa5\xba\xcf\xe4\xf9\x0e\x23"
			"\x38\x4d\x62\x77\x8c\xa1\xb6\xcb"
			"\xe0\xf5\x0a\x1f\x34\x49\x5e\x73"
			"\x88\x9d\xb2\xc7\xdc\xf1\x06\x1b"
			"\x30\x45\x5a\x6f\x84\x99\xae\xc3"
			"\xd8\xed\x02\x17\x2c\x41\x56\x6b"
			"\x80\x95\xaa\xbf\xd4\xe9\xfe\x13"
			"\x28\x3d\x52\x67\x7c\x91\xa6\xbb"
			"\xd0\xe5\xfa\x0f\x24\x39\x4e\x63"
			"\x78\x8d\xa2\xb7\xcc\xe1\xf6\x0b"
			"\x20\x35\x4a\x5f\x74\x89\x9e\xb3"
			"\xc8\xdd\xf2\x07\x1c\x31\x46\x5b"
			"\x70\x85\x9a\xaf\xc4\xd9\xee\x03"
			"\x18\x2d\x42\x57\x6c\x81\x96\xab"
			"\xc0\xd5\xea\xff\x14\x29\x3e\x53"
			"\x68\x7d\x92\xa7\xbc\xd1\xe6\xfb"
			"\x10\x25\x3a\x4f\x64\x79\x8e\xa3"
			"\xb8\xcd\xe2\xf7\x0c\x21\x36\x4b"
			"\x60\x75\x8a\x9f\xb4\xc9\xde\xf3"
			"\x08\x1d\x32\x47\x5c\x71\x86\x9b"
			"\xb0\xc5\xda\xef\x04\x19\x2e\x43"
			"\x58\x6d\x82\x97\xac\xc1\xd6\xeb"
			"\x00\x17\x2e\x45\x5c\x73\x8a\xa1"
			"\xb8\xcf\xe6\xfd\x14\x2b\x42\x59"
			"\x70\x87\x9e\xb5\xcc\xe3\xfa\x11"
			"\x28\x3f\x56\x6d\x84\x9b\xb2\xc9"
			"\xe0\xf7\x0e\x25\x3c\x53\x6a\x81"
			"\x98\xaf\xc6\xdd\xf4\x0b\x22\x39"
			"\x50\x67\x7e\x95\xac\xc3\xda\xf1"
			"\x08\x1f\x36\x4d\x64\x7b\x92\xa9"
			"\xc0\xd7\xee\x05\x1c\x33\x4a\x61"
			"\x78\x8f\xa6\xbd\xd4\xeb\x02\x19"
			"\x30\x47\x5e\x75\x8c\xa3\xba\xd1"
			"\xe8\xff\x16\x2d\x44\x5b\x72\x89"
			"\xa0\xb7\xce\xe5\xfc\x13\x2a\x41"
			"\x58\x6f\x86\x9d\xb4\xcb\xe2\xf9"
			"\x10\x27\x3e\x55\x6c\x83\x9a\xb1"
			"\xc8\xdf\xf6\x0d\x24\x3b\x52\x69"
			"\x80\x97\xae\xc5\xdc\xf3\x0a\x21"
			"\x38\x4f\x66\x7d\x94\xab\xc2\xd9"
			"\xf0\x07\x1e\x35\x4c\x63\x7a\x91"
			"\xa8\xbf\xd6\xed\x04\x1b\x32\x49"
			"\x60\x77\x8e\xa5\xbc\xd3\xea\x01"
			"\x18\x2f\x46\x5d\x74\x8b\xa2\xb9"
			"\xd0\xe7\xfe\x15\x2c\x43\x5a\x71"
			"\x88\x9f\xb6\xcd\xe4\xfb\x12\x29"
			"\x40\x57\x6e\x85\x9c\xb3\xca\xe1"
			"\xf8\x0f\x26\x3d\x54\x6b\x82\x99"
			"\xb0\xc7\xde\xf5\x0c\x23\x3a\x51"
			"\x68\x7f\x96\xad\xc4\xdb\xf2\x09"
			"\x20\x37\x4e\x65\x7c\x93\xaa\xc1"
			"\xd8\xef\x06\x1d\x34\x4b\x62\x79"
			"\x90\xa7\xbe\xd5\xec\x03\x1a\x31"
			"\x48\x5f\x76\x8d\xa4\xbb\xd2\xe9"
			"\x00\x19\x32\x4b\x64\x7d\x96\xaf"
			"\xc8\xe1\xfa\x13\x2c\x45\x5e\x77"
			"\x90\xa9\xc2\xdb\xf4\x0d\x26\x3f"
			"\x58\x71\x8a\xa3\xbc\xd5\xee\x07"
			"\x20\x39\x52\x6b\x84\x9d\xb6\xcf"
			"\xe8\x01\x1a\x33\x4c\x65\x7e\x97"
			"\xb0\xc9\xe2\xfb\x14\x2d\x46\x5f"
			"\x78\x91\xaa\xc3\xdc\xf5\x0e\x27"
			"\x40\x59\x72\x8b\xa4\xbd\xd6\xef"
			"\x08\x21\x3a\x53\x6c\x85\x9e\xb7"
			"\xd0\xe9\x02\x1b\x34\x4d\x66\x7f"
			"\x98\xb1\xca\xe3\xfc\x15\x2e\x47"
			"\x60\x79\x92\xab\xc4\xdd\xf6\x0f"
			"\x28\x41\x5a\x73\x8c\xa5\xbe\xd7"
			"\xf0\x09\x22\x3b\x54\x6d\x86\x9f"
			"\xb8\xd1\xea\x03\x1c\x35\x4e\x67"
			"\x80\x99\xb2\xcb\xe4\xfd\x16\x2f"
			"\x48\x61\x7a\x93\xac\xc5\xde\xf7"
			"\x10\x29\x42\x5b\x74\x8d\xa6\xbf"
			"\xd8\xf1\x0a\x23\x3c\x55\x6e\x87"
			"\xa0\xb9\xd2\xeb\x04\x1d\x36\x4f"
			"\x68\x81\x9a\xb3\xcc\xe5\xfe\x17"
			"\x30\x49\x62\x7b\x94\xad\xc6\xdf"
			"\xf8\x11\x2a\x43\x5c\x75\x8e\xa7"
			"\xc0\xd9\xf2\x0b\x24\x3d\x56\x6f"
			"\x88\xa1\xba\xd3\xec\x05\x1e\x37"
			"\x50\x69\x82\x9b\xb4\xcd\xe6\xff"
			"\x18\x31\x4a\x63\x7c\x95\xae\xc7"
			"\xe0\xf9\x12\x2b\x44\x5d\x76\x8f"
			"\xa8\xc1\xda\xf3\x0c\x25\x3e\x57"
			"\x70\x89\xa2\xbb\xd4\xed\x06\x1f"
			"\x38\x51\x6a\x83\x9c\xb5\xce\xe7"
			"\x00\x1b\x36\x51\x6c\x87\xa2\xbd"
			"\xd8\xf3\x0e\x29\x44\x5f\x7a\x95"
			"\xb0\xcb\xe6\x01\x1c\x37\x52\x6d"
			"\x88\xa3\xbe\xd9\xf4\x0f\x2a\x45"
			"\x60\x7b\x96\xb1\xcc\xe7\x02\x1d"
			"\x38\x53\x6e\x89\xa4\xbf\xda\xf5"
			"\x10\x2b\x46\x61\x7c\x97\xb2\xcd"
			"\xe8\x03\x1e\x39\x54\x6f\x8a\xa5"
			"\xc0\xdb\xf6\x11\x2c\x47\x62\x7d"
			"\x98\xb3\xce\xe9\x04\x1f\x3a\x55"
			"\x70\x8b\xa6\xc1\xdc\xf7\x12\x2d"
			"\x48\x63\x7e\x99\xb4\xcf\xea\x05"
			"\x20\x3b\x56\x71\x8c\xa7\xc2\xdd"
			"\xf8\x13\x2e\x49\x64\x7f\x9a\xb5"
			"\xd0\xeb\x06\x21\x3c\x57\x72\x8d"
			"\xa8\xc3\xde\xf9\x14\x2f\x4a\x65"
			"\x80\x9b\xb6\xd1\xec\x07\x22\x3d"
			"\x58\x73\x8e\xa9\xc4\xdf\xfa\x15"
			"\x30\x4b\x66\x81\x9c\xb7\xd2\xed"
			"\x08\x23\x3e\x59\x74\x8f\xaa\xc5"
			"\xe0\xfb\x16\x31\x4c\x67\x82\x9d"
			"\xb8\xd3\xee\x09\x24\x3f\x5a\x75"
			"\x90\xab\xc6\xe1\xfc\x17\x32\x4d"
			"\x68\x83\x9e\xb9\xd4\xef\x0a\x25"
			"\x40\x5b\x76\x91\xac\xc7\xe2\xfd"
			"\x18\x33\x4e\x69\x84\x9f\xba\xd5"
			"\xf0\x0b\x26\x41\x5c\x77\x92\xad"
			"\xc8\xe3\xfe\x19\x34\x4f\x6a\x85"
			"\xa0\xbb\xd6\xf1\x0c\x27\x42\x5d"
			"\x78\x93\xae\xc9\xe4\xff\x1a\x35"
			"\x50\x6b\x86\xa1\xbc\xd7\xf2\x0d"
			"\x28\x43\x5e\x79\x94\xaf\xca\xe5"
			"\x00\x1d\x3a\x57\x74\x91\xae\xcb"
			"\xe8\x05\x22\x3f\x5c\x79\x96\xb3"
			"\xd0\xed\x0a\x27\x44\x61\x7e\x9b"
			"\xb8\xd5\xf2\x0f\x2c\x49\x66\x83"
			"\xa0\xbd\xda\xf7\x14\x31\x4e\x6b"
			"\x88\xa5\xc2\xdf\xfc\x19\x36\x53"
			"\x70\x8d\xaa\xc7\xe4\x01\x1e\x3b"
			"\x58\x75\x92\xaf\xcc\xe9\x06\x23"
			"\x40\x5d\x7a\x97\xb4\xd1\xee\x0b"
			"\x28\x45\x62\x7f\x9c\xb9\xd6\xf3"
			"\x10\x2d\x4a\x67\x84\xa1\xbe\xdb"
			"\xf8\x15\x32\x4f\x6c\x89\xa6\xc3"
			"\xe0\xfd\x1a\x37\x54\x71\x8e\xab"
			"\xc8\xe5\x02\x1f\x3c\x59\x76\x93"
			"\xb0\xcd\xea\x07\x24\x41\x5e\x7b"
			"\x98\xb5\xd2\xef\x0c\x29\x46\x63"
			"\x80\x9d\xba\xd7\xf4\x11\x2e\x4b"
			"\x68\x85\xa2\xbf\xdc\xf9\x16\x33"
			"\x50\x6d\x8a\xa7\xc4\xe1\xfe\x1b"
			"\x38\x55\x72\x8f\xac\xc9\xe6\x03"
			"\x20\x3d\x5a\x77\x94\xb1\xce\xeb"
			"\x08\x25\x42\x5f\x7c\x99\xb6\xd3"
			"\xf0\x0d\x2a\x47\x64\x81\x9e\xbb"
			"\xd8\xf5\x12\x2f\x4c\x69\x86\xa3"
			"\xc0\xdd\xfa\x17\x34\x51\x6e\x8b"
			"\xa8\xc5\xe2\xff\x1c\x39\x56\x73"
			"\x90\xad\xca\xe7\x04\x21\x3e\x5b"
			"\x78\x95\xb2\xcf\xec\x09\x26\x43"
			"\x60\x7d\x9a\xb7\xd4\xf1\x0e\x2b"
			"\x48\x65\x82\x9f\xbc\xd9\xf6\x13"
			"\x30\x4d\x6a\x87\xa4\xc1\xde\xfb"
			"\x18\x35\x52\x6f\x8c\xa9\xc6\xe3"
			"\x00\x1f\x3e\x5d\x7c\x9b\xba\xd9"
			"\xf8\x17\x36\x55\x74\x93\xb2\xd1"
			"\xf0\x0f\x2e\x4d\x6c\x8b\xaa\xc9"
			"\xe8\x07\x26\x45\x64\x83\xa2\xc1"
			"\xe0\xff\x1e\x3d\x5c\x7b\x9a\xb9"
			"\xd8\xf7\x16\x35\x54\x73\x92\xb1"
			"\xd0\xef\x0e\x2d\x4c\x6b\x8a\xa9"
			"\xc8\xe7\x06\x25\x44\x63\x82\xa1"
			"\xc0\xdf\xfe\x1d\x3c\x5b\x7a\x99"
			"\xb8\xd7\xf6\x15\x34\x53\x72\x91"
			"\xb0\xcf\xee\x0d\x2c\x4b\x6a\x89"
			"\xa8\xc7\xe6\x05\x24\x43\x62\x81"
			"\xa0\xbf\xde\xfd\x1c\x3b\x5a\x79"
			"\x98\xb7\xd6\xf5\x14\x33\x52\x71"
			"\x90\xaf\xce\xed\x0c\x2b\x4a\x69"
			"\x88\xa7\xc6\xe5\x04\x23\x42\x61"
			"\x80\x9f\xbe\xdd\xfc\x1b\x3a\x59"
			"\x78\x97\xb6\xd5\xf4\x13\x32\x51"
			"\x70\x8f\xae\xcd\xec\x0b\x2a\x49"
			"\x68\x87\xa6\xc5\xe4\x03\x22\x41"
			"\x60\x7f\x9e\xbd\xdc\xfb\x1a\x39"
			"\x58\x77\x96\xb5\xd4\xf3\x12\x31"
			"\x50\x6f\x8e\xad\xcc\xeb\x0a\x29"
			"\x48\x67\x86\xa5\xc4\xe3\x02\x21"
			"\x40\x5f\x7e\x9d\xbc\xdb\xfa\x19"
			"\x38\x57\x76\x95\xb4\xd3\xf2\x11"
			"\x30\x4f\x6e\x8d\xac\xcb\xea\x09"
			"\x28\x47\x66\x85\xa4\xc3\xe2\x01"
			"\x20\x3f\x5e\x7d\x9c\xbb\xda\xf9"
			"\x18\x37\x56\x75\x94\xb3\xd2\xf1"
			"\x10\x2f\x4e\x6d\x8c\xab\xca\xe9"
			"\x08\x27\x46\x65\x84\xa3\xc2\xe1"
			"\x00\x21\x42\x63",
		.ilen = 4100,
		.result =
			"\xf0\x5c\x74\xad\x4e\xbc\x99\xe2"
			"\xae\xff\x91\x3a\x44\xcf\x38\x32"
			"\x1e\xad\xa7\xcd\xa1\x39\x95\xaa"
			"\x10\xb1\xb3\x2e\x04\x31\x8f\x86"
			"\xf2\x62\x74\x70\x0c\xa4\x46\x08"
			"\xa8\xb7\x99\xa8\xe9\xd2\x73\x79"
			"\x7e\x6e\xd4\x8f\x1e\xc7\x8e\x31"
			"\x0b\xfa\x4b\xce\xfd\xf3\x57\x71"
			"\xe9\x46\x03\xa5\x3d\x34\x00\xe2"
			"\x18\xff\x75\x6d\x06\x2d\x00\xab"
			"\xb9\x3e\x6c\x59\xc5\x84\x06\xb5"
			"\x8b\xd0\x89\x9c\x4a\x79\x16\xc6"
			"\x3d\x74\x54\xfa\x44\xcd\x23\x26"
			"\x5c\xcf\x7e\x28\x92\x32\xbf\xdf"
			"\xa7\x20\x3c\x74\x58\x2a\x9a\xde"
			"\x61\x00\x1c\x4f\xff\x59\xc4\x22"
			"\xac\x3c\xd0\xe8\x6c\xf9\x97\x1b"
			"\x58\x9b\xad\x71\xe8\xa9\xb5\x0d"
			"\xee\x2f\x04\x1f\x7f\xbc\x99\xee"
			"\x84\xff\x42\x60\xdc\x3a\x18\xa5"
			"\x81\xf9\xef\xdc\x7a\x0f\x65\x41"
			"\x2f\xa3\xd3\xf9\xc2\xcb\xc0\x4d"
			"\x8f\xd3\x76\x96\xad\x49\x6d\x38"
			"\x3d\x39\x0b\x6c\x80\xb7\x54\x69"
			"\xf0\x2c\x90\x02\x29\x0d\x1c\x12"
			"\xad\x55\xc3\x8b\x68\xd9\xcc\xb3"
			"\xb2\x64\x33\x90\x5e\xca\x4b\xe2"
			"\xfb\x75\xdc\x63\xf7\x9f\x82\x74"
			"\xf0\xc9\xaa\x7f\xe9\x2a\x9b\x33"
			"\xbc\x88\x00\x7f\xca\xb2\x1f\x14"
			"\xdb\xc5\x8e\x7b\x11\x3c\x3e\x08"
			"\xf3\x83\xe8\xe0\x94\x86\x2e\x92"
			"\x78\x6b\x01\xc9\xc7\x83\xba\x21"
			"\x6a\x25\x15\x33\x4e\x45\x08\xec"
			"\x35\xdb\xe0\x6e\x31\x51\x79\xa9"
			"\x42\x44\x65\xc1\xa0\xf1\xf9\x2a"
			"\x70\xd5\xb6\xc6\xc1\x8c\x39\xfc"
			"\x25\xa6\x55\xd9\xdd\x2d\x4c\xec"
			"\x49\xc6\xeb\x0e\xa8\x25\x2a\x16"
			"\x1b\x66\x84\xda\xe2\x92\xe5\xc0"
			"\xc8\x53\x07\xaf\x80\x84\xec\xfd"
			"\xcd\xd1\x6e\xcd\x6f\x6a\xf5\x36"
			"\xc5\x15\xe5\x25\x7d\x77\xd1\x1a"
			"\x93\x36\xa9\xcf\x7c\xa4\x54\x4a"
			"\x06\x51\x48\x4e\xf6\x59\x87\xd2"
			"\x04\x02\xef\xd3\x44\xde\x76\x31"
			"\xb3\x34\x17\x1b\x9d\x66\x11\x9f"
			"\x1e\xcc\x17\xe9\xc7\x3c\x1b\xe7"
			"\xcb\x50\x08\xfc\xdc\x2b\x24\xdb"
			"\x65\x83\xd0\x3b\xe3\x30\xea\x94"
			"\x6c\xe7\xe8\x35\x32\xc7\xdb\x64"
			"\xb4\x01\xab\x36\x2c\x77\x13\xaf"
			"\xf8\x2b\x88\x3f\x54\x39\xc4\x44"
			"\xfe\xef\x6f\x68\x34\xbe\x0f\x05"
			"\x16\x6d\xf6\x0a\x30\xe7\xe3\xed"
			"\xc4\xde\x3c\x1b\x13\xd8\xdb\xfe"
			"\x41\x62\xe5\x28\xd4\x8d\xa3\xc7"
			"\x93\x97\xc6\x48\x45\x1d\x9f\x83"
			"\xdf\x4b\x40\x3e\x42\x25\x87\x80"
			"\x4c\x7d\xa8\xd4\x98\x23\x95\x75"
			"\x41\x8c\xda\x41\x9b\xd4\xa7\x06"
			"\xb5\xf1\x71\x09\x53\xbe\xca\xbf"
			"\x32\x03\xed\xf0\x50\x1c\x56\x39"
			"\x5b\xa4\x75\x18\xf7\x9b\x58\xef"
			"\x53\xfc\x2a\x38\x23\x15\x75\xcd"
			"\x45\xe5\x5a\x82\x55\xba\x21\xfa"
			"\xd4\xbd\xc6\x94\x7c\xc5\x80\x12"
			"\xf7\x4b\x32\xc4\x9a\x82\xd8\x28"
			"\x8f\xd9\xc2\x0f\x60\x03\xbe\x5e"
			"\x21\xd6\x5f\x58\xbf\x5c\xb1\x32"
			"\x82\x8d\xa9\xe5\xf2\x66\x1a\xc0"
			"\xa0\xbc\x58\x2f\x71\xf5\x2f\xed"
			"\xd1\x26\xb9\xd8\x49\x5a\x07\x19"
			"\x01\x7c\x59\xb0\xf8\xa4\xb7\xd3"
			"\x7b\x1a\x8c\x38\xf4\x50\xa4\x59"
			"\xb0\xcc\x41\x0b\x88\x7f\xe5\x31"
			"\xb3\x42\xba\xa2\x7e\xd4\x32\x71"
			"\x45\x87\x48\xa9\xc2\xf2\x89\xb3"
			"\xe4\xa7\x7e\x52\x15\x61\xfa\xfe"
			"\xc9\xdd\x81\xeb\x13\xab\xab\xc3"
			"\x98\x59\xd8\x16\x3d\x14\x7a\x1c"
			"\x3c\x41\x9a\x16\x16\x9b\xd2\xd2"
			"\x69\x3a\x29\x23\xac\x86\x32\xa5"
			"\x48\x9c\x9e\xf3\x47\x77\x81\x70"
			"\x24\xe8\x85\xd2\xf5\xb5\xfa\xff"
			"\x59\x6a\xd3\x50\x59\x43\x59\xde"
			"\xd9\xf1\x55\xa5\x0c\xc3\x1a\x1a"
			"\x18\x34\x0d\x1a\x63\x33\xed\x10"
			"\xe0\x1d\x2a\x18\xd2\xc0\x54\xa8"
			"\xca\xb5\x9a\xd3\xdd\xca\x45\x84"
			"\x50\xe7\x0f\xfe\xa4\x99\x5a\xbe"
			"\x43\x2d\x9a\xcb\x92\x3f\x5a\x1d"
			"\x85\xd8\xc9\xdf\x68\xc9\x12\x80"
			"\x56\x0c\xdc\x00\xdc\x3a\x7d\x9d"
			"\xa3\xa2\xe8\x4d\xbf\xf9\x70\xa0"
			"\xa4\x13\x4f\x6b\xaf\x0a\x89\x7f"
			"\xda\xf0\xbf\x9b\xc8\x1d\xe5\xf8"
			"\x2e\x8b\x07\xb5\x73\x1b\xcc\xa2"
			"\xa6\xad\x30\xbc\x78\x3c\x5b\x10"
			"\xfa\x5e\x62\x2d\x9e\x64\xb3\x33"
			"\xce\xf9\x1f\x86\xe7\x8b\xa2\xb8"
			"\xe8\x99\x57\x8c\x11\xed\x66\xd9"
			"\x3c\x72\xb9\xc3\xe6\x4e\x17\x3a"
			"\x6a\xcb\x42\x24\x06\xed\x3e\x4e"
			"\xa3\xe8\x6a\x94\xda\x0d\x4e\xd5"
			"\x14\x19\xcf\xb6\x26\xd8\x2e\xcc"
			"\x64\x76\x38\x49\x4d\xfe\x30\x6d"
			"\xe4\xc8\x8c\x7b\xc4\xe0\x35\xba"
			"\x22\x6e\x76\xe1\x1a\xf2\x53\xc3"
			"\x28\xa2\x82\x1f\x61\x69\xad\xc1"
			"\x7b\x28\x4b\x1e\x6c\x85\x95\x9b"
			"\x51\xb5\x17\x7f\x12\x69\x8c\x24"
			"\xd5\xc7\x5a\x5a\x11\x54\xff\x5a"
			"\xf7\x16\xc3\x91\xa6\xf0\xdc\x0a"
			"\xb6\xa7\x4a\x0d\x7a\x58\xfe\xa5"
			"\xf5\xcb\x8f\x7b\x0e\xea\x57\xe7"
			"\xbd\x79\xd6\x1c\x88\x23\x6c\xf2"
			"\x4d\x29\x77\x53\x35\x6a\x00\x8d"
			"\xcd\xa3\x58\xbe\x77\x99\x18\xf8"
			"\xe6\xe1\x8f\xe9\x37\x8f\xe3\xe2"
			"\x5a\x8a\x93\x25\xaf\xf3\x78\x80"
			"\xbe\xa6\x1b\xc6\xac\x8b\x1c\x91"
			"\x58\xe1\x9f\x89\x35\x9d\x1d\x21"
			"\x29\x9f\xf4\x99\x02\x27\x0f\xa8"
			"\x4f\x79\x94\x2b\x33\x2c\xda\xa2"
			"\x26\x39\x83\x94\xef\x27\xd8\x53"
			"\x8f\x66\x0d\xe4\x41\x7d\x34\xcd"
			"\x43\x7c\x95\x0a\x53\xef\x66\xda"
			"\x7e\x9b\xf3\x93\xaf\xd0\x73\x71"
			"\xba\x40\x9b\x74\xf8\xd7\xd7\x41"
			"\x6d\xaf\x72\x9c\x8d\x21\x87\x3c"
			"\xfd\x0a\x90\xa9\x47\x96\x9e\xd3"
			"\x88\xee\x73\xcf\x66\x2f\x52\x56"
			"\x6d\xa9\x80\x4c\xe2\x6f\x62\x88"
			"\x3f\x0e\x54\x17\x48\x80\x5d\xd3"
			"\xc3\xda\x25\x3d\xa1\xc8\xcb\x9f"
			"\x9b\x70\xb3\xa1\xeb\x04\x52\xa1"
			"\xf2\x22\x0f\xfc\xc8\x18\xfa\xf9"
			"\x85\x9c\xf1\xac\xeb\x0c\x02\x46"
			"\x75\xd2\xf5\x2c\xe3\xd2\x59\x94"
			"\x12\xf3\x3c\xfc\xd7\x92\xfa\x36"
			"\xba\x61\x34\x38\x7c\xda\x48\x3e"
			"\x08\xc9\x39\x23\x5e\x02\x2c\x1a"
			"\x18\x7e\xb4\xd9\xfd\x9e\x40\x02"
			"\xb1\x33\x37\x32\xe7\xde\xd6\xd0"
			"\x7c\x58\x65\x4b\xf8\x34\x27\x9c"
			"\x44\xb4\xbd\xe9\xe9\x4c\x78\x7d"
			"\x4b\x9f\xce\xb1\xcd\x47\xa5\x37"
			"\xe5\x6d\xbd\xb9\x43\x94\x0a\xd4"
			"\xd6\xf9\x04\x5f\xb5\x66\x6c\x1a"
			"\x35\x12\xe3\x36\x28\x27\x36\x58"
			"\x01\x2b\x79\xe4\xba\x6d\x10\x7d"
			"\x65\xdf\x84\x95\xf4\xd5\xb6\x8f"
			"\x2b\x9f\x96\x00\x86\x60\xf0\x21"
			"\x76\xa8\x6a\x8c\x28\x1c\xb3\x6b"
			"\x97\xd7\xb6\x53\x2a\xcc\xab\x40"
			"\x9d\x62\x79\x58\x52\xe6\x65\xb7"
			"\xab\x55\x67\x9c\x89\x7c\x03\xb0"
			"\x73\x59\xc5\x81\xf5\x18\x17\x5c"
			"\x89\xf3\x78\x35\x44\x62\x78\x72"
			"\xd0\x96\xeb\x31\xe7\x87\x77\x14"
			"\x99\x51\xf2\x59\x26\x9e\xb5\xa6"
			"\x45\xfe\x6e\xbd\x07\x4c\x94\x5a"
			"\xa5\x7d\xfc\xf1\x2b\x77\xe2\xfe"
			"\x17\xd4\x84\xa0\xac\xb5\xc7\xda"
			"\xa9\x1a\xb6\xf3\x74\x11\xb4\x9d"
			"\xfb\x79\x2e\x04\x2d\x50\x28\x83"
			"\xbf\xc6\x52\xd3\x34\xd6\xe8\x7a"
			"\xb6\xea\xe7\xa8\x6c\x15\x1e\x2c"
			"\x57\xbc\x48\x4e\x5f\x5c\xb6\x92"
			"\xd2\x49\x77\x81\x6d\x90\x70\xae"
			"\x98\xa1\x03\x0d\x6b\xb9\x77\x14"
			"\xf1\x4e\x23\xd3\xf8\x68\xbd\xc2"
			"\xfe\x04\xb7\x5c\xc5\x17\x60\x8f"
			"\x65\x54\xa4\x7a\x42\xdc\x18\x0d"
			"\xb5\xcf\x0f\xd3\xc7\x91\x66\x1b"
			"\x45\x42\x27\x75\x50\xe5\xee\xb8"
			"\x7f\x33\x2c\xba\x4a\x92\x4d\x2c"
			"\x3c\xe3\x0d\x80\x01\xba\x0d\x29"
			"\xd8\x3c\xe9\x13\x16\x57\xe6\xea"
			"\x94\x52\xe7\x00\x4d\x30\xb0\x0f"
			"\x35\xb8\xb8\xa7\xb1\xb5\x3b\x44"
			"\xe1\x2f\xfd\x88\xed\x43\xe7\x52"
			"\x10\x93\xb3\x8a\x30\x6b\x0a\xf7"
			"\x23\xc6\x50\x9d\x4a\xb0\xde\xc3"
			"\xdc\x9b\x2f\x01\x56\x36\x09\xc5"
			"\x2f\x6b\xfe\xf1\xd8\x27\x45\x03"
			"\x30\x5e\x5c\x5b\xb4\x62\x0e\x1a"
			"\xa9\x21\x2b\x92\x94\x87\x62\x57"
			"\x4c\x10\x74\x1a\xf1\x0a\xc5\x84"
			"\x3b\x9e\x72\x02\xd7\xcc\x09\x56"
			"\xbd\x54\xc1\xf0\xc3\xe3\xb3\xf8"
			"\xd2\x0d\x61\xcb\xef\xce\x0d\x05"
			"\xb0\x98\xd9\x8e\x4f\xf9\xbc\x93"
			"\xa6\xea\xc8\xcf\x10\x53\x4b\xf1"
			"\xec\xfc\x89\xf9\x64\xb0\x22\xbf"
			"\x9e\x55\x46\x9f\x7c\x50\x8e\x84"
			"\x54\x20\x98\xd7\x6c\x40\x1e\xdb"
			"\x69\x34\x78\x61\x24\x21\x9c\x8a"
			"\xb3\x62\x31\x8b\x6e\xf5\x2a\x35"
			"\x86\x13\xb1\x6c\x64\x2e\x41\xa5"
			"\x05\xf2\x42\xba\xd2\x3a\x0d\x8e"
			"\x8a\x59\x94\x3c\xcf\x36\x27\x82"
			"\xc2\x45\xee\x58\xcd\x88\xb4\xec"
			"\xde\xb2\x96\x0a\xaf\x38\x6f\x88"
			"\xd7\xd8\xe1\xdf\xb9\x96\xa9\x0a"
			"\xb1\x95\x28\x86\x20\xe9\x17\x49"
			"\xa2\x29\x38\xaa\xa5\xe9\x6e\xf1"
			"\x19\x27\xc0\xd5\x2a\x22\xc3\x0b"
			"\xdb\x7c\x73\x10\xb9\xba\x89\x76"
			"\x54\xae\x7d\x71\xb3\x93\xf6\x32"
			"\xe6\x47\x43\x55\xac\xa0\x0d\xc2"
			"\x93\x27\x4a\x8e\x0e\x74\x15\xc7"
			"\x0b\x85\xd9\x0c\xa9\x30\x7a\x3e"
			"\xea\x8f\x85\x6d\x3a\x12\x4f\x72"
			"\x69\x58\x7a\x80\xbb\xb5\x97\xf3"
			"\xcf\x70\xd2\x5d\xdd\x4d\x21\x79"
			"\x54\x4d\xe4\x05\xe8\xbd\xc2\x62"
			"\xb1\x3b\x77\x1c\xd6\x5c\xf3\xa0"
			"\x79\x00\xa8\x6c\x29\xd9\x18\x24"
			"\x36\xa2\x46\xc0\x96\x65\x7f\xbd"
			"\x2a\xed\x36\x16\x0c\xaa\x9f\xf4"
			"\xc5\xb4\xe2\x12\xed\x69\xed\x4f"
			"\x26\x2c\x39\x52\x89\x98\xe7\x2c"
			"\x99\xa4\x9e\xa3\x9b\x99\x46\x7a"
			"\x3a\xdc\xa8\x59\xa3\xdb\xc3\x3b"
			"\x95\x0d\x3b\x09\x6e\xee\x83\x5d"
			"\x32\x4d\xed\xab\xfa\x98\x14\x4e"
			"\xc3\x15\x45\x53\x61\xc4\x93\xbd"
			"\x90\xf4\x99\x95\x4c\xe6\x76\x92"
			"\x29\x90\x46\x30\x92\x69\x7d\x13"
			"\xf2\xa5\xcd\x69\x49\x44\xb2\x0f"
			"\x63\x40\x36\x5f\x09\xe2\x78\xf8"
			"\x91\xe3\xe2\xfa\x10\xf7\xc8\x24"
			"\xa8\x89\x32\x5c\x37\x25\x1d\xb2"
			"\xea\x17\x8a\x0a\xa9\x64\xc3\x7c"
			"\x3c\x7c\xbd\xc6\x79\x34\xe7\xe2"
			"\x85\x8e\xbf\xf8\xde\x92\xa0\xae"
			"\x20\xc4\xf6\xbb\x1f\x38\x19\x0e"
			"\xe8\x79\x9c\xa1\x23\xe9\x54\x7e"
			"\x37\x2f\xe2\x94\x32\xaf\xa0\x23"
			"\x49\xe4\xc0\xb3\xac\x00\x8f\x36"
			"\x05\xc4\xa6\x96\xec\x05\x98\x4f"
			"\x96\x67\x57\x1f\x20\x86\x1b\x2d"
			"\x69\xe4\x29\x93\x66\x5f\xaf\x6b"
			"\x88\x26\x2c\x67\x02\x4b\x52\xd0"
			"\x83\x7a\x43\x1f\xc0\x71\x15\x25"
			"\x77\x65\x08\x60\x11\x76\x4c\x8d"
			"\xed\xa9\x27\xc6\xb1\x2a\x2c\x6a"
			"\x4a\x97\xf5\xc6\xb7\x70\x42\xd3"
			"\x03\xd1\x24\x95\xec\x6d\xab\x38"
			"\x72\xce\xe2\x8b\x33\xd7\x51\x09"
			"\xdc\x45\xe0\x09\x96\x32\xf3\xc4"
			"\x84\xdc\x73\x73\x2d\x1b\x11\x98"
			"\xc5\x0e\x69\x28\x94\xc7\xb5\x4d"
			"\xc8\x8a\xd0\xaa\x13\x2e\x18\x74"
			"\xdd\xd1\x1e\xf3\x90\xe8\xfc\x9a"
			"\x72\x4a\x0e\xd1\xe4\xfb\x0d\x96"
			"\xd1\x0c\x79\x85\x1b\x1c\xfe\xe1"
			"\x62\x8f\x7a\x73\x32\xab\xc8\x18"
			"\x69\xe3\x34\x30\xdf\x13\xa6\xe5"
			"\xe8\x0e\x67\x7f\x81\x11\xb4\x60"
			"\xc7\xbd\x79\x65\x50\xdc\xc4\x5b"
			"\xde\x39\xa4\x01\x72\x63\xf3\xd1"
			"\x64\x4e\xdf\xfc\x27\x92\x37\x0d"
			"\x57\xcd\x11\x4f\x11\x04\x8e\x1d"
			"\x16\xf7\xcd\x92\x9a\x99\x30\x14"
			"\xf1\x7c\x67\x1b\x1f\x41\x0b\xe8"
			"\x32\xe8\xb8\xc1\x4f\x54\x86\x4f"
			"\xe5\x79\x81\x73\xcd\x43\x59\x68"
			"\x73\x02\x3b\x78\x21\x72\x43\x00"
			"\x49\x17\xf7\x00\xaf\x68\x24\x53"
			"\x05\x0a\xc3\x33\xe0\x33\x3f\x69"
			"\xd2\x84\x2f\x0b\xed\xde\x04\xf4"
			"\x11\x94\x13\x69\x51\x09\x28\xde"
			"\x57\x5c\xef\xdc\x9a\x49\x1c\x17"
			"\x97\xf3\x96\xc1\x7f\x5d\x2e\x7d"
			"\x55\xb8\xb3\x02\x09\xb3\x1f\xe7"
			"\xc9\x8d\xa3\x36\x34\x8a\x77\x13"
			"\x30\x63\x4c\xa5\xcd\xc3\xe0\x7e"
			"\x05\xa1\x7b\x0c\xcb\x74\x47\x31"
			"\x62\x03\x43\xf1\x87\xb4\xb0\x85"
			"\x87\x8e\x4b\x25\xc7\xcf\xae\x4b"
			"\x36\x46\x3e\x62\xbc\x6f\xeb\x5f"
			"\x73\xac\xe6\x07\xee\xc1\xa1\xd6"
			"\xc4\xab\xc9\xd6\x89\x45\xe1\xf1"
			"\x04\x4e\x1a\x6f\xbb\x4f\x3a\xa3"
			"\xa0\xcb\xa3\x0a\xd8\x71\x35\x55"
			"\xe4\xbc\x2e\x04\x06\xe6\xff\x5b"
			"\x1c\xc0\x11\x7c\xc5\x17\xf3\x38"
			"\xcf\xe9\xba\x0f\x0e\xef\x02\xc2"
			"\x8d\xc6\xbc\x4b\x67\x20\x95\xd7"
			"\x2c\x45\x5b\x86\x44\x8c\x6f\x2e"
			"\x7e\x9f\x1c\x77\xba\x6b\x0e\xa3"
			"\x69\xdc\xab\x24\x57\x60\x47\xc1"
			"\xd1\xa5\x9d\x23\xe6\xb1\x37\xfe"
			"\x93\xd2\x4c\x46\xf9\x0c\xc6\xfb"
			"\xd6\x9d\x99\x69\xab\x7a\x07\x0c"
			"\x65\xe7\xc4\x08\x96\xe2\xa5\x01"
			"\x3f\x46\x07\x05\x7e\xe8\x9a\x90"
			"\x50\xdc\xe9\x7a\xea\xa1\x39\x6e"
			"\x66\xe4\x6f\xa5\x5f\xb2\xd9\x5b"
			"\xf5\xdb\x2a\x32\xf0\x11\x6f\x7c"
			"\x26\x10\x8f\x3d\x80\xe9\x58\xf7"
			"\xe0\xa8\x57\xf8\xdb\x0e\xce\x99"
			"\x63\x19\x3d\xd5\xec\x1b\x77\x69"
			"\x98\xf6\xe4\x5f\x67\x17\x4b\x09"
			"\x85\x62\x82\x70\x18\xe2\x9a\x78"
			"\xe2\x62\xbd\xb4\xf1\x42\xc6\xfb"
			"\x08\xd0\xbd\xeb\x4e\x09\xf2\xc8"
			"\x1e\xdc\x3d\x32\x21\x56\x9c\x4f"
			"\x35\xf3\x61\x06\x72\x84\xc4\x32"
			"\xf2\xf1\xfa\x0b\x2f\xc3\xdb\x02"
			"\x04\xc2\xde\x57\x64\x60\x8d\xcf"
			"\xcb\x86\x5d\x97\x3e\xb1\x9c\x01"
			"\xd6\x28\x8f\x99\xbc\x46\xeb\x05"
			"\xaf\x7e\xb8\x21\x2a\x56\x85\x1c"
			"\xb3\x71\xa0\xde\xca\x96\xf1\x78"
			"\x49\xa2\x99\x81\x80\x5c\x01\xf5"
			"\xa0\xa2\x56\x63\xe2\x70\x07\xa5"
			"\x95\xd6\x85\xeb\x36\x9e\xa9\x51"
			"\x66\x56\x5f\x1d\x02\x19\xe2\xf6"
			"\x4f\x73\x38\x09\x75\x64\x48\xe0"
			"\xf1\x7e\x0e\xe8\x9d\xf9\xed\x94"
			"\xfe\x16\x26\x62\x49\x74\xf4\xb0"
			"\xd4\xa9\x6c\xb0\xfd\x53\xe9\x81"
			"\xe0\x7a\xbf\xcf\xb5\xc4\x01\x81"
			"\x79\x99\x77\x01\x3b\xe9\xa2\xb6"
			"\xe6\x6a\x8a\x9e\x56\x1c\x8d\x1e"
			"\x8f\x06\x55\x2c\x6c\xdc\x92\x87"
			"\x64\x3b\x4b\x19\xa1\x13\x64\x1d"
			"\x4a\xe9\xc0\x00\xb8\x95\xef\x6b"
			"\x1a\x86\x6d\x37\x52\x02\xc2\xe0"
			"\xc8\xbb\x42\x0c\x02\x21\x4a\xc9"
			"\xef\xa0\x54\xe4\x5e\x16\x53\x81"
			"\x70\x62\x10\xaf\xde\xb8\xb5\xd3"
			"\xe8\x5e\x6c\xc3\x8a\x3e\x18\x07"
			"\xf2\x2f\x7d\xa7\xe1\x3d\x4e\xb4"
			"\x26\xa7\xa3\x93\x86\xb2\x04\x1e"
			"\x53\x5d\x86\xd6\xde\x65\xca\xe3"
			"\x4e\xc1\xcf\xef\xc8\x70\x1b\x83"
			"\x13\xdd\x18\x8b\x0d\x76\xd2\xf6"
			"\x37\x7a\x93\x7a\x50\x11\x9f\x96"
			"\x86\x25\xfd\xac\xdc\xbe\x18\x93"
			"\x19\x6b\xec\x58\x4f\xb9\x75\xa7"
			"\xdd\x3f\x2f\xec\xc8\x5a\x84\xab"
			"\xd5\xe4\x8a\x07\xf6\x4d\x23\xd6"
			"\x03\xfb\x03\x6a\xea\x66\xbf\xd4"
			"\xb1\x34\xfb\x78\xe9\x55\xdc\x7c"
			"\x3d\x9c\xe5\x9a\xac\xc3\x7a\x80"
			"\x24\x6d\xa0\xef\x25\x7c\xb7\xea"
			"\xce\x4d\x5f\x18\x60\xce\x87\x22"
			"\x66\x2f\xd5\xdd\xdd\x02\x21\x75"
			"\x82\xa0\x1f\x58\xc6\xd3\x62\xf7"
			"\x32\xd8\xaf\x1e\x07\x77\x51\x96"
			"\xd5\x6b\x1e\x7e\x80\x02\xe8\x67"
			"\xea\x17\x0b\x10\xd2\x3f\x28\x25"
			"\x4f\x05\x77\x02\x14\x69\xf0\x2c"
			"\xbe\x0c\xf1\x74\x30\xd1\xb9\x9b"
			"\xfc\x8c\xbb\x04\x16\xd9\xba\xc3"
			"\xbc\x91\x8a\xc4\x30\xa4\xb0\x12"
			"\x4c\x21\x87\xcb\xc9\x1d\x16\x96"
			"\x07\x6f\x23\x54\xb9\x6f\x79\xe5"
			"\x64\xc0\x64\xda\xb1\xae\xdd\x60"
			"\x6c\x1a\x9d\xd3\x04\x8e\x45\xb0"
			"\x92\x61\xd0\x48\x81\xed\x5e\x1d"
			"\xa0\xc9\xa4\x33\xc7\x13\x51\x5d"
			"\x7f\x83\x73\xb6\x70\x18\x65\x3e"
			"\x2f\x0e\x7a\x12\x39\x98\xab\xd8"
			"\x7e\x6f\xa3\xd1\xba\x56\xad\xbd"
			"\xf0\x03\x01\x1c\x85\x35\x9f\xeb"
			"\x19\x63\xa1\xaf\xfe\x2d\x35\x50"
			"\x39\xa0\x65\x7c\x95\x7e\x6b\xfe"
			"\xc1\xac\x07\x7c\x98\x4f\xbe\x57"
			"\xa7\x22\xec\xe2\x7e\x29\x09\x53"
			"\xe8\xbf\xb4\x7e\x3f\x8f\xfc\x14"
			"\xce\x54\xf9\x18\x58\xb5\xff\x44"
			"\x05\x9d\xce\x1b\xb6\x82\x23\xc8"
			"\x2e\xbc\x69\xbb\x4a\x29\x0f\x65"
			"\x94\xf0\x63\x06\x0e\xef\x8c\xbd"
			"\xff\xfd\xb0\x21\x6e\x57\x05\x75"
			"\xda\xd5\xc4\xeb\x8d\x32\xf7\x50"
			"\xd3\x6f\x22\xed\x5f\x8e\xa2\x5b"
			"\x80\x8c\xc8\x78\x40\x24\x4b\x89"
			"\x30\xce\x7a\x97\x0e\xc4\xaf\xef"
			"\x9b\xb4\xcd\x66\x74\x14\x04\x2b"
			"\xf7\xce\x0b\x1c\x6e\xc2\x78\x8c"
			"\xca\xc5\xd0\x1c\x95\x4a\x91\x2d"
			"\xa7\x20\xeb\x86\x52\xb7\x67\xd8"
			"\x0c\xd6\x04\x14\xde\x51\x74\x75"
			"\xe7\x11\xb4\x87\xa3\x3d\x2d\xad"
			"\x4f\xef\xa0\x0f\x70\x00\x6d\x13"
			"\x19\x1d\x41\x50\xe9\xd8\xf0\x32"
			"\x71\xbc\xd3\x11\xf2\xac\xbe\xaf"
			"\x75\x46\x65\x4e\x07\x34\x37\xa3"
			"\x89\xfe\x75\xd4\x70\x4c\xc6\x3f"
			"\x69\x24\x0e\x38\x67\x43\x8c\xde"
			"\x06\xb5\xb8\xe7\xc4\xf0\x41\x8f"
			"\xf0\xbd\x2f\x0b\xb9\x18\xf8\xde"
			"\x64\xb1\xdb\xee\x00\x50\x77\xe1"
			"\xc7\xff\xa6\xfa\xdd\x70\xf4\xe3"
			"\x93\xe9\x77\x35\x3d\x4b\x2f\x2b"
			"\x6d\x55\xf0\xfc\x88\x54\x4e\x89"
			"\xc1\x8a\x23\x31\x2d\x14\x2a\xb8"
			"\x1b\x15\xdd\x9e\x6e\x7b\xda\x05"
			"\x91\x7d\x62\x64\x96\x72\xde\xfc"
			"\xc1\xec\xf0\x23\x51\x6f\xdb\x5b"
			"\x1d\x08\x57\xce\x09\xb8\xf6\xcd"
			"\x8d\x95\xf2\x20\xbf\x0f\x20\x57"
			"\x98\x81\x84\x4f\x15\x5c\x76\xe7"
			"\x3e\x0a\x3a\x6c\xc4\x8a\xbe\x78"
			"\x74\x77\xc3\x09\x4b\x5d\x48\xe4"
			"\xc8\xcb\x0b\xea\x17\x28\xcf\xcf"
			"\x31\x32\x44\xa4\xe5\x0e\x1a\x98"
			"\x94\xc4\xf0\xff\xae\x3e\x44\xe8"
			"\xa5\xb3\xb5\x37\x2f\xe8\xaf\x6f"
			"\x28\xc1\x37\x5f\x31\xd2\xb9\x33"
			"\xb1\xb2\x52\x94\x75\x2c\x29\x59"
			"\x06\xc2\x25\xe8\x71\x65\x4e\xed"
			"\xc0\x9c\xb1\xbb\x25\xdc\x6c\xe7"
			"\x4b\xa5\x7a\x54\x7a\x60\xff\x7a"
			"\xe0\x50\x40\x96\x35\x63\xe4\x0b"
			"\x76\xbd\xa4\x65\x00\x1b\x57\x88"
			"\xae\xed\x39\x88\x42\x11\x3c\xed"
			"\x85\x67\x7d\xb9\x68\x82\xe9\x43"
			"\x3c\x47\x53\xfa\xe8\xf8\x9f\x1f"
			"\x9f\xef\x0f\xf7\x30\xd9\x30\x0e"
			"\xb9\x9f\x69\x18\x2f\x7e\xf8\xf8"
			"\xf8\x8c\x0f\xd4\x02\x4d\xea\xcd"
			"\x0a\x9c\x6f\x71\x6d\x5a\x4c\x60"
			"\xce\x20\x56\x32\xc6\xc5\x99\x1f"
			"\x09\xe6\x4e\x18\x1a\x15\x13\xa8"
			"\x7d\xb1\x6b\xc0\xb2\x6d\xf8\x26"
			"\x66\xf8\x3d\x18\x74\x70\x66\x7a"
			"\x34\x17\xde\xba\x47\xf1\x06\x18"
			"\xcb\xaf\xeb\x4a\x1e\x8f\xa7\x77"
			"\xe0\x3b\x78\x62\x66\xc9\x10\xea"
			"\x1f\xb7\x29\x0a\x45\xa1\x1d\x1e"
			"\x1d\xe2\x65\x61\x50\x9c\xd7\x05"
			"\xf2\x0b\x5b\x12\x61\x02\xc8\xe5"
			"\x63\x4f\x20\x0c\x07\x17\x33\x5e"
			"\x03\x9a\x53\x0f\x2e\x55\xfe\x50"
			"\x43\x7d\xd0\xb6\x7e\x5a\xda\xae"
			"\x58\xef\x15\xa9\x83\xd9\x46\xb1"
			"\x42\xaa\xf5\x02\x6c\xce\x92\x06"
			"\x1b\xdb\x66\x45\x91\x79\xc2\x2d"
			"\xe6\x53\xd3\x14\xfd\xbb\x44\x63"
			"\xc6\xd7\x3d\x7a\x0c\x75\x78\x9d"
			"\x5c\xa6\x39\xb3\xe5\x63\xca\x8b"
			"\xfe\xd3\xef\x60\x83\xf6\x8e\x70"
			"\xb6\x67\xc7\x77\xed\x23\xef\x4c"
			"\xf0\xed\x2d\x07\x59\x6f\xc1\x01"
			"\x34\x37\x08\xab\xd9\x1f\x09\xb1"
			"\xce\x5b\x17\xff\x74\xf8\x9c\xd5"
			"\x2c\x56\x39\x79\x0f\x69\x44\x75"
			"\x58\x27\x01\xc4\xbf\xa7\xa1\x1d"
			"\x90\x17\x77\x86\x5a\x3f\xd9\xd1"
			"\x0e\xa0\x10\xf8\xec\x1e\xa5\x7f"
			"\x5e\x36\xd1\xe3\x04\x2c\x70\xf7"
			"\x8e\xc0\x98\x2f\x6c\x94\x2b\x41"
			"\xb7\x60\x00\xb7\x2e\xb8\x02\x8d"
			"\xb8\xb0\xd3\x86\xba\x1d\xd7\x90"
			"\xd6\xb6\xe1\xfc\xd7\xd8\x28\x06"
			"\x63\x9b\xce\x61\x24\x79\xc0\x70"
			"\x52\xd0\xb6\xd4\x28\x95\x24\x87"
			"\x03\x1f\xb7\x9a\xda\xa3\xfb\x52"
			"\x5b\x68\xe7\x4c\x8c\x24\xe1\x42"
			"\xf7\xd5\xfd\xad\x06\x32\x9f\xba"
			"\xc1\xfc\xdd\xc6\xfc\xfc\xb3\x38"
			"\x74\x56\x58\x40\x02\x37\x52\x2c"
			"\x55\xcc\xb3\x9e\x7a\xe9\xd4\x38"
			"\x41\x5e\x0c\x35\xe2\x11\xd1\x13"
			"\xf8\xb7\x8d\x72\x6b\x22\x2a\xb0"
			"\xdb\x08\xba\x35\xb9\x3f\xc8\xd3"
			"\x24\x90\xec\x58\xd2\x09\xc7\x2d"
			"\xed\x38\x80\x36\x72\x43\x27\x49"
			"\x4a\x80\x8a\xa2\xe8\xd3\xda\x30"
			"\x7d\xb6\x82\x37\x86\x92\x86\x3e"
			"\x08\xb2\x28\x5a\x55\x44\x24\x7d"
			"\x40\x48\x8a\xb6\x89\x58\x08\xa0"
			"\xd6\x6d\x3a\x17\xbf\xf6\x54\xa2"
			"\xf5\xd3\x8c\x0f\x78\x12\x57\x8b"
			"\xd5\xc2\xfd\x58\x5b\x7f\x38\xe3"
			"\xcc\xb7\x7c\x48\xb3\x20\xe8\x81"
			"\x14\x32\x45\x05\xe0\xdb\x9f\x75"
			"\x85\xb4\x6a\xfc\x95\xe3\x54\x22"
			"\x12\xee\x30\xfe\xd8\x30\xef\x34"
			"\x50\xab\x46\x30\x98\x2f\xb7\xc0"
			"\x15\xa2\x83\xb6\xf2\x06\x21\xa2"
			"\xc3\x26\x37\x14\xd1\x4d\xb5\x10"
			"\x52\x76\x4d\x6a\xee\xb5\x2b\x15"
			"\xb7\xf9\x51\xe8\x2a\xaf\xc7\xfa"
			"\x77\xaf\xb0\x05\x4d\xd1\x68\x8e"
			"\x74\x05\x9f\x9d\x93\xa5\x3e\x7f"
			"\x4e\x5f\x9d\xcb\x09\xc7\x83\xe3"
			"\x02\x9d\x27\x1f\xef\x85\x05\x8d"
			"\xec\x55\x88\x0f\x0d\x7c\x4c\xe8"
			"\xa1\x75\xa0\xd8\x06\x47\x14\xef"
			"\xaa\x61\xcf\x26\x15\xad\xd8\xa3"
			"\xaa\x75\xf2\x78\x4a\x5a\x61\xdf"
			"\x8b\xc7\x04\xbc\xb2\x32\xd2\x7e"
			"\x42\xee\xb4\x2f\x51\xff\x7b\x2e"
			"\xd3\x02\xe8\xdc\x5d\x0d\x50\xdc"
			"\xae\xb7\x46\xf9\xa8\xe6\xd0\x16"
			"\xcc\xe6\x2c\x81\xc7\xad\xe9\xf0"
			"\x05\x72\x6d\x3d\x0a\x7a\xa9\x02"
			"\xac\x82\x93\x6e\xb6\x1c\x28\xfc"
			"\x44\x12\xfb\x73\x77\xd4\x13\x39"
			"\x29\x88\x8a\xf3\x5c\xa6\x36\xa0"
			"\x2a\xed\x7e\xb1\x1d\xd6\x4c\x6b"
			"\x41\x01\x18\x5d\x5d\x07\x97\xa6"
			"\x4b\xef\x31\x18\xea\xac\xb1\x84"
			"\x21\xed\xda\x86",
		.rlen = 4100,
		.np	= 2,
		.tap	= { 4064, 36 },
	},
};
#endif
#endif
