#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/kobject.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/sysfs.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/suspend.h>
#include <linux/interrupt.h>

#include <asm/uaccess.h>
#include <linux/debugfs.h>
#include <linux/string.h>

#include "btnuvar.h"

#if NPSOC > 0
#include "psocvar.h"
#endif

#define BTNU_DEV_NAME	"btnu_dev"

static int btnu_io_read(void *arg);
#if NPSOC > 0
static int psoc_io_read(void *arg);
#endif

#define POWER_SW_GPIO	84
#define RESET_SW_GPIO	20
#define WPS_SW_GPIO	74

struct btnu_softc {
	int devnum;
	int port;
	int is_psoc;
	int is_fake;
	const char *name;
	struct btnucom data;
} btnu_list[] = {
	{0, 10, 1, 1, "SET"},
	{1, RESET_SW_GPIO, 0, 0, "RESET"},
	{2, POWER_SW_GPIO, 0, 1, "Power"},
	{3,  1, 1, 1, "CRADLE"},
	{4, 86, 0, 1, "RT/AP"},
	{5, WPS_SW_GPIO, 0, 0, "WPS"},
};

#ifdef CONFIG_PM_SLEEP
static int power_sw_detect;
module_param(power_sw_detect, int, 0644);

static irqreturn_t
power_sw_wakeup_interrupt(int irq, void *ignored)
{
	pr_info("%s %d\n", __func__, power_sw_detect);
	power_sw_detect = 1;
	return IRQ_HANDLED;
}
#endif

static struct dentry *btnu_debugfs;

static int
btnu_debugfs_open(struct inode *inode, struct file *file)
{
	if (inode->i_private)
		file->private_data = inode->i_private;

	return 0;
}

static ssize_t
btnu_debugfs_read(struct file *file, char __user *buf, size_t len, loff_t *ppos)
{
	struct btnu_softc *btnu = file->private_data;

	printk(KERN_INFO "io_dummy : %d \n", btnu->data.io_dummy);

	return 0;
}

static ssize_t
btnu_debugfs_write(struct file *file, const char __user *buf, size_t len, loff_t *ppos)
{
	struct btnu_softc *btnu = file->private_data;
	size_t ret = 0;
	char *tmp = kmalloc(len, GFP_KERNEL);
	char *p;

	if (!tmp)
		return -ENOMEM;

	ret = copy_from_user(tmp, buf, len);
	if (ret)
		goto free_buf;

	p = strchr(tmp, '\n');
	*p = '\0';

	if (strcmp(tmp, "1") == 0)
		btnu->data.io_dummy = 1;
	else if (strcmp(tmp, "0") == 0)
		btnu->data.io_dummy = 0;

	printk(KERN_INFO "Set io_dummy value to %d \n", btnu->data.io_dummy);

free_buf:
	kfree(tmp);

	return len;
}

static struct file_operations btnu_fops = {
	.owner = THIS_MODULE,
	.read = btnu_debugfs_read,
	.write = btnu_debugfs_write,
	.open = btnu_debugfs_open,
};

static int
btnu_debugfs_init(void)
{
	btnu_debugfs = debugfs_create_dir("btnu", NULL);
	if (!btnu_debugfs)
		return -1;

	return 0;
}

static int
btnu_debugfs_create_file(struct btnu_softc *btnu)
{
	struct dentry *entry = NULL;
	char btn_name[8];

	if (!btnu_debugfs)
		goto err;

	snprintf(btn_name, sizeof(btn_name), "btn%d", btnu->devnum);
	entry = debugfs_create_file(btn_name, 0777, btnu_debugfs, btnu, &btnu_fops);
	if (!entry)
		goto err ;

	return 0;

err:
	debugfs_remove_recursive(btnu_debugfs);

	return -1;
}

static void
btnu_debugfs_cleanup(void)
{
	if (btnu_debugfs) {
		debugfs_remove_recursive(btnu_debugfs);
		btnu_debugfs = NULL;
	}
}

static int __init
btnu_init(void)
{
	int i, free_target, status;
#ifdef CONFIG_PM_SLEEP
	int irq, err;
#endif

	btnu_debugfs_init();

	for (i = 0; i < ARRAY_SIZE(btnu_list); i++) {
		if (!btnu_list[i].is_psoc) {
			if (!btnu_list[i].is_fake) {
				status = gpio_request(btnu_list[i].port,
				    btnu_list[i].name);
				if (status)
					goto err;
			}
			else {
				btnu_debugfs_create_file(&btnu_list[i]);
			}

			btnu_list[i].data.io_read = btnu_io_read;
			status = btnucom_attach(&btnu_list[i].data,
			    btnu_list[i].devnum, &btnu_list[i]);
			if (status) {
				if (!btnu_list[i].is_fake) {
					gpio_free(btnu_list[i].port);
				}
				goto err;
			}
			if (!btnu_list[i].is_fake) {
				gpio_export(btnu_list[i].port, 0);
			}
		}
#if NPSOC > 0
		else {
			btnu_list[i].data.io_read = psoc_io_read;
			status = btnucom_attach(&btnu_list[i].data,
			    btnu_list[i].devnum, &btnu_list[i]);
			if (status)
				goto err;
		}
#endif
	}

#ifdef CONFIG_PM_SLEEP
	if (0) {
	irq = gpio_to_irq(POWER_SW_GPIO);
	err = request_irq(irq, &power_sw_wakeup_interrupt,
	    IRQF_TRIGGER_FALLING, "power_sw_wakeup", NULL);
	enable_irq_wake(irq);
	}
#endif

	return 0;

err:
	free_target = i;
	for (i = 0; i < free_target; i++) {
		btnucom_detach(&btnu_list[i].data);
		if (!btnu_list[i].is_psoc) {
			if (!btnu_list[i].is_fake) {
				gpio_unexport(btnu_list[i].port);
				gpio_free(btnu_list[i].port);
			}
		}
	}

	return status;
}

static void __exit
btnu_exit(void)
{
	int i;

	btnu_debugfs_cleanup();

	for (i = 0; i < ARRAY_SIZE(btnu_list); i++) {
		btnucom_detach(&btnu_list[i].data);
		if (!btnu_list[i].is_psoc) {
			if (!btnu_list[i].is_fake) {
				gpio_unexport(btnu_list[i].port);
				gpio_free(btnu_list[i].port);
			}
		}
	}
}

static int
btnu_io_read(void *arg)
{
	struct btnu_softc *sc = arg;

	if (sc->is_fake) {
		return sc->data.io_dummy;
	}

	return gpio_get_value(sc->port) ? 0 : 1;
}

#if NPSOC > 0
static int
psoc_io_read(void *arg)
{
	struct btnu_softc *sc = arg;

	return psoc_stat_read(sc->port) ? 1 : 0;
}
#endif

module_init(btnu_init);
module_exit(btnu_exit);

MODULE_DESCRIPTION("NECPF button device interface");
MODULE_LICENSE("GPL v2");
