#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/gpio.h>

#include "ledvar.h"
#include "led_stone.h"

#if NPSOC > 0
#include "psocvar.h"
#endif

static const ledtype_t ledtype_alias[] = {
	[LEDT_ALL] = LEDT_ALL,
	[LEDT_STAT] = LEDT_STAT,
	[LEDT_RSSI] = LEDT_RSSI,
	[LEDT_WIFI] = LEDT_WIFI,
	[LEDT_POWER] = LEDT_POWER,
	[LEDT_REAL_MAX] = LEDT_REAL_MAX,
	[LEDT_REBOOT] = LEDT_STAT,
	[LEDT_INIT] = LEDT_STAT,
 	[LEDT_FW_UPDATE] = LEDT_STAT,
 	[LEDT_LTE] = LEDT_WAN,
	[LEDT_WPS] = LEDT_WIFI,
	[LEDT_LTE_STATUS] = LEDT_STAT,
	[LEDT_INIT_STATUS] = LEDT_STAT,
	[LEDT_SIM_STATUS] = LEDT_STAT,

	[LEDT_MAX] = LEDT_MAX,
};

static const ledact_type_t ledact_alias[] = {
	[LEDACT_NORMAL] = LEDACT_NORMAL,
	[LEDACT_NORMAL_RED] = LEDACT_NORMAL_RED,
	[LEDACT_NORMAL_ORANGE] = LEDACT_NORMAL_ORANGE,
	[LEDACT_NORMAL_BLUE] = LEDACT_NORMAL_BLUE,
	[LEDACT_BLINK1] = LEDACT_BLINK1,
	[LEDACT_BLINK1_RED] = LEDACT_BLINK1_RED,
	[LEDACT_BLINK1_ORANGE] = LEDACT_BLINK1_ORANGE,
	[LEDACT_INACTIVATE] = LEDACT_INACTIVATE,
	[LEDACT_INACTIVATE2] = LEDACT_INACTIVATE2,
	[LEDACT_CLEAR] = LEDACT_CLEAR,
	[LEDACT_ALLCLEAR] = LEDACT_ALLCLEAR,
	[LEDACT_NORMAL2] = LEDACT_NORMAL2,
	[LEDACT_NORMAL_RED2] = LEDACT_NORMAL_RED2,
	[LEDACT_NORMAL_ORANGE2] = LEDACT_NORMAL_ORANGE2,
	[LEDACT_NORMAL_ORANGE3] = LEDACT_NORMAL_ORANGE3,
	[LEDACT_BLINK1_2] = LEDACT_BLINK1_2,
	[LEDACT_BLINK1_3] = LEDACT_BLINK1_3,
	[LEDACT_BLINK1_4] = LEDACT_BLINK1_4,
	[LEDACT_BLINK1_RED2] = LEDACT_BLINK1_RED2,
	[LEDACT_BLINK10_RED] = LEDACT_BLINK10_RED,
	[LEDACT_BLINK11_ORANGE] = LEDACT_BLINK11_ORANGE,
	[LEDACT_BLINK12] = LEDACT_BLINK12,
	[LEDACT_BLINK12_RED] = LEDACT_BLINK12_RED,
	[LEDACT_BLINK12_ORANGE] = LEDACT_BLINK12_ORANGE,
	[LEDACT_BLINK13] = LEDACT_BLINK13,
	[LEDACT_BLINK13_RED] = LEDACT_BLINK13_RED,
	[LEDACT_BLINK13_ORANGE] = LEDACT_BLINK13_ORANGE,
	[LEDACT_BLINK13_BLUE] = LEDACT_BLINK13_BLUE,
	[LEDACT_REAL_MAX] = LEDACT_REAL_MAX,
	[LEDACT_REBOOT] = LEDACT_BLINK12_ORANGE,
	[LEDACT_VUP_FIRMWARE_UPDATE] = LEDACT_NORMAL_ORANGE3,
	[LEDACT_VUP_FW_DOWNLOAD] = LEDACT_NORMAL_ORANGE3,
	[LEDACT_EMERG] = LEDACT_NORMAL_RED2,
	[LEDACT_WAN_CONNECTING] = LEDACT_BLINK13_BLUE,
	[LEDACT_WAN_CONNECTED] = LEDACT_NORMAL_BLUE,
	[LEDACT_WAN_WAITING] = LEDACT_NORMAL,
	[LEDACT_LTE_POWER_HIGH] = LEDACT_NORMAL_BLUE,
	[LEDACT_SYS_WAIT] = LEDACT_BLINK13_ORANGE,

	[LEDACT_LTE_POWER_MID] = LEDACT_NORMAL,
	[LEDACT_LTE_POWER_LOW] = LEDACT_NORMAL,

	[LEDACT_LTE_POWER_OUT] = LEDACT_NORMAL,

	[LEDACT_CONFIG] = LEDACT_BLINK12_ORANGE,
	[LEDACT_INIT_STATUS] = LEDACT_NORMAL_ORANGE,
	[LEDACT_SIM_UNDETECTED] = LEDACT_NORMAL_RED,
	[LEDACT_WPS_FAIL] = LEDACT_NORMAL_RED,
	[LEDACT_WPS_COMPLETE] = LEDACT_NORMAL_ORANGE2,
	[LEDACT_WPS_SETTING] = LEDACT_BLINK1_2,
	[LEDACT_VUP_HAVE_NEWFW] = LEDACT_BLINK1_2,
	[LEDACT_WIFI_5G_AVAILABLE] = LEDACT_NORMAL_ORANGE,
	[LEDACT_WIFI_2G_AVAILABLE] = LEDACT_NORMAL,
	[LEDACT_WIFI_STOPPED] = LEDACT_NORMAL_RED2,
	[LEDACT_DFS_RUNNING] = LEDACT_BLINK1_RED2,
	[LEDACT_WPS_SESSION_OVERLAP] = LEDACT_BLINK10_RED,
	[LEDACT_ZWAVE_STOPPED] = LEDACT_INACTIVATE,
	[LEDACT_ZWAVE_FAILED] = LEDACT_NORMAL_RED2,
	[LEDACT_ZWAVE_PAIR_FAILED] = LEDACT_NORMAL_RED,
	[LEDACT_ZWAVE_PAIR_OK] = LEDACT_NORMAL_ORANGE2,
	[LEDACT_ZWAVE_INCLUSION] = LEDACT_BLINK1_ORANGE,
	[LEDACT_ZWAVE_EXCLUSION] = LEDACT_BLINK11_ORANGE,
	[LEDACT_ZWAVE_AVAIL] = LEDACT_NORMAL,

	[LEDACT_BLE_STOPPED] = LEDACT_INACTIVATE,
	[LEDACT_BLE_PAIR_FAILED] = LEDACT_NORMAL_RED,
	[LEDACT_BLE_PAIR_OK] = LEDACT_NORMAL_ORANGE2,
	[LEDACT_BLE_PAIRING] = LEDACT_BLINK1_ORANGE,
	[LEDACT_BLE_AVAIL] = LEDACT_NORMAL,

	[LEDACT_INIT_PREPARATION] = LEDACT_BLINK13_RED,
	[LEDACT_PINLOCK] = LEDACT_NORMAL_RED2,
	[LEDACT_PUKLOCK] = LEDACT_NORMAL_RED2,
	[LEDACT_PIN_UNAPPROVAL] = LEDACT_NORMAL_RED2,
	[LEDACT_WAN_NOT_CONNECTED] = LEDACT_NORMAL_RED2,


	[LEDACT_MAX] = LEDACT_MAX,
};

#define LEDACT_BITMAP(x) ((ledact_bitmap_t) 1 << ((x) - 1))

static const action_list_t action_list[] = {
	{LEDST_ORANGE, LEDACT_BITMAP(LEDACT_NORMAL_ORANGE3), act_on, LEDCOLOR_ORANGE, 0, 0, 0},	
	{LEDST_OFF, LEDACT_BITMAP(LEDACT_INACTIVATE), act_off, 0, 0, 0, 0},
	{LEDST_OFF, LEDACT_BITMAP(LEDACT_INACTIVATE2), act_off, 0, 0, 0, 0},
	{LEDST_RED, LEDACT_BITMAP(LEDACT_NORMAL_RED2), act_on, LEDCOLOR_RED, 0, 0, 0},
	{LEDST_BLINK_RED, LEDACT_BITMAP(LEDACT_BLINK1_RED2), act_blink1, LEDCOLOR_RED, 0, 0, 1},
	{LEDST_BLINK_RED, LEDACT_BITMAP(LEDACT_BLINK10_RED), act_blink14, LEDCOLOR_RED, 0, 0, 1},
	{LEDST_RED, LEDACT_BITMAP(LEDACT_NORMAL_RED), act_on, LEDCOLOR_RED, 0, 0, 0},
	{LEDST_ORANGE, LEDACT_BITMAP(LEDACT_NORMAL_ORANGE2), act_on, LEDCOLOR_ORANGE, 0, 0, 0},
	{LEDST_BLINK_GREEN, LEDACT_BITMAP(LEDACT_BLINK1_2), act_blink1, LEDCOLOR_GREEN, 0, 0, 1},
	{LEDST_ORANGE, LEDACT_BITMAP(LEDACT_NORMAL_ORANGE), act_on, LEDCOLOR_ORANGE, 0, 0, 0},
	{LEDST_BLINK_ORANGE, LEDACT_BITMAP(LEDACT_BLINK12_ORANGE), act_blink16, LEDCOLOR_ORANGE, 0, 0, 1},
	{LEDST_BLINK_ORANGE, LEDACT_BITMAP(LEDACT_BLINK13_ORANGE), act_blink17, LEDCOLOR_ORANGE, 0, 0, 1},
	{LEDST_BLINK_GREEN, LEDACT_BITMAP(LEDACT_BLINK12), act_blink16, LEDCOLOR_GREEN, 0, 0, 1},
	{LEDST_BLINK_GREEN, LEDACT_BITMAP(LEDACT_BLINK13), act_blink17, LEDCOLOR_GREEN, 0, 0, 1},
	{LEDST_BLINK_RED, LEDACT_BITMAP(LEDACT_BLINK12_RED), act_blink16, LEDCOLOR_RED, 0, 0, 1},
	{LEDST_BLINK_RED, LEDACT_BITMAP(LEDACT_BLINK13_RED), act_blink17, LEDCOLOR_RED, 0, 0, 1},
	{LEDST_BLINK_BLUE, LEDACT_BITMAP(LEDACT_BLINK13_BLUE), act_blink17, LEDCOLOR_BLUE, 0, 0, 1},
	{LEDST_BLUE, LEDACT_BITMAP(LEDACT_NORMAL_BLUE), act_on, LEDCOLOR_BLUE, 0, 0, 0},
	{LEDST_GREEN, LEDACT_BITMAP(LEDACT_NORMAL), act_on, LEDCOLOR_GREEN, 0, 0, 0},
	{LEDST_BLUE, LEDACT_BITMAP(LEDACT_NORMAL_BLUE), act_blink1, LEDCOLOR_BLUE, 0, 0, 0},
	{LEDST_BLUE, LEDACT_BITMAP(LEDACT_NORMAL_BLUE), act_off, LEDCOLOR_BLUE, 0, 0, 0},
	{LEDST_OFF, LEDACT_NONE, act_off, 0, 0, 0, 0}
};

const struct led_bitmap led_stone_bitmap[LEDT_REAL_MAX] = {
	[LEDT_ALL] =
	{{0, LED_ALL_GRN, LED_ALL_RED, LED_ALL_ORANGE, LED_RSSI_BLU, 0, 0, 0}, LED_ALL},
	[LEDT_RSSI] =
	{{0, LED_RSSI_GRN, LED_RSSI_RED, LED_RSSI_ORANGE, LED_RSSI_BLU, 0, 0}, LED_RSSI_ALL},
	[LEDT_WIFI] =
	{{0, LED_WIFI_GRN, LED_WIFI_RED, LED_WIFI_ORANGE, 0, 0, 0, 0}, LED_WIFI_ORANGE},
	[LEDT_POWER] =
	{{0, LED_POWER_GRN, LED_POWER_RED, LED_POWER_ORANGE,  0, 0, 0, 0}, LED_POWER_ORANGE},

};

static const type_name_list_t type_name_list[] = {
	{"all", LEDT_ALL},
	{"rssi", LEDT_RSSI},
	{"power", LEDT_POWER},
	{"wifi", LEDT_WIFI},
	{NULL, LEDT_MAX}
};

static const act_name_list_t act_name_list[] = {
	{"oneshot", LEDACT_ONESHOT},
	{"normal", LEDACT_NORMAL},
	{"normal_red", LEDACT_NORMAL_RED},
	{"normal_orange", LEDACT_NORMAL_ORANGE},
	{"normal_blue", LEDACT_NORMAL_BLUE},
	{"no_config", LEDACT_ALIAS_NO_CONFIG},
	{"blink1", LEDACT_BLINK1},
	{"blink2", LEDACT_BLINK2},
	{"blink3", LEDACT_BLINK3},
	{"blink1_orange", LEDACT_BLINK1_ORANGE},
	{"dms_connect_wait", LEDACT_ALIAS_DMS_CONNECT_WAIT},
	{"auto_subnet_err_act", LEDACT_AUTO_SUBNET_ERR},
	{"blink1_green_orange", LEDACT_BLINK1_GREEN_ORANGE},
	{"blink1_red", LEDACT_BLINK1_RED},
	{"blink3_red", LEDACT_BLINK3_RED},
	{"blink1_orange3", LEDACT_BLINK1_ORANGE3},
	{"flush", LEDACT_FLUSH},
	{"flush_green", LEDACT_FLUSH_GREEN},
	{"oneshot_orange", LEDACT_ONESHOT_ORANGE},
	{"normal_red2", LEDACT_NORMAL_RED2},
	{"blink1_orange2", LEDACT_BLINK1_ORANGE2},
	{"blink1_green_orange2", LEDACT_BLINK1_GREEN_ORANGE2},
	{"blink2_red", LEDACT_BLINK2_RED},
	{"blink5_orange", LEDACT_BLINK5_ORANGE},
	{"blink5_green", LEDACT_BLINK5},
	{"normal_orange2", LEDACT_NORMAL_ORANGE2},
	{"blink7_orange_green", LEDACT_BLINK7_ORANGE_GREEN},
	{"blink8_red", LEDACT_BLINK8_RED},
	{"blink1_red2", LEDACT_BLINK1_RED2},
	{"blink9_red_green_orange", LEDACT_BLINK9_RED_GREEN_ORANGE},
	{"blink3_red_green", LEDACT_BLINK3_RED_GREEN},
	{"1sec_orange", LEDACT_1SEC_ORANGE},
	{"inactivate", LEDACT_INACTIVATE},

	{"clear", LEDACT_CLEAR},
	{"allclear", LEDACT_ALLCLEAR},

	{"normal2", LEDACT_NORMAL2},
	{"blink1_green2", LEDACT_BLINK1_2},
	{"blink1_green3", LEDACT_BLINK1_3},
	{"blink1_green4", LEDACT_BLINK1_4},
	{"blink10_red", LEDACT_BLINK10_RED},
	{"blink11_orange", LEDACT_BLINK11_ORANGE},

	{"blink12_green", LEDACT_BLINK12},
	{"blink12_red", LEDACT_BLINK12_RED},
	{"blink12_orange", LEDACT_BLINK12_ORANGE},
	{"blink13_green", LEDACT_BLINK13},
	{"blink13_red", LEDACT_BLINK13_RED},
	{"blink13_orange", LEDACT_BLINK13_ORANGE},
	{"blink13_blue", LEDACT_BLINK13_BLUE},

	{"wps_fail", LEDACT_WPS_FAIL},
	{"wps_session_overlap", LEDACT_WPS_SESSION_OVERLAP},
	{"vup_firmware_update", LEDACT_VUP_FIRMWARE_UPDATE},
	{"vup_fw_download", LEDACT_VUP_FW_DOWNLOAD},
	{"wps_complete", LEDACT_WPS_COMPLETE},
	{"wps_start", LEDACT_WPS_START},
	{"wps_setting", LEDACT_WPS_SETTING},
	{"vup_have_newfw", LEDACT_VUP_HAVE_NEWFW},
	{"wan_connected", LEDACT_WAN_CONNECTED},
	{"wan_connecting", LEDACT_WAN_CONNECTING},
	{"wan_waiting", LEDACT_WAN_WAITING},
	{"emerg", LEDACT_EMERG},
	{"init_preparation", LEDACT_INIT_PREPARATION},
	{"wifi_2g", LEDACT_WIFI_2G_AVAILABLE},
	{"wifi_5g", LEDACT_WIFI_5G_AVAILABLE},
	{"wifi_stopped", LEDACT_WIFI_STOPPED},
	{"dfs_run", LEDACT_DFS_RUNNING},
	{"lte_high", LEDACT_LTE_POWER_HIGH},
	{"lte_low", LEDACT_LTE_POWER_LOW},
	{"sys_wait", LEDACT_SYS_WAIT},

	{NULL, LEDACT_MAX}
};

#define GPIO_PORT_LEDMAX -1
#define GPIO_PORT_NOTUSE -2

struct led_gpio_arg {
	union {
		int pin;
		int *pin_tbl;
	} u;
};

static void gpio_write_one(const struct led_gpio_arg *, int);
static void gpio_write_multi(const struct led_gpio_arg *, int);
#if NPSOC > 0
static void psoc_write_one(const struct led_gpio_arg *, int);
#endif

#define GPIO_PORT_LED_RSSI_BLU		6
#define GPIO_PORT_LED_POWER_GRN		10
#define GPIO_PORT_LED_POWER_RED		11
#define GPIO_PORT_LED_RSSI_GRN		19
#define GPIO_PORT_LED_WIFI_RED		76
#define GPIO_PORT_LED_WIFI_GRN		75
#define GPIO_PORT_LED_RSSI_RED		77

static const struct gpio led_gpios[] = {
	{GPIO_PORT_LED_RSSI_BLU,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "RSSI_LED_BLU"},
	{GPIO_PORT_LED_POWER_RED,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "POWER_LED_RED"},
	{GPIO_PORT_LED_POWER_GRN,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "POWER_LED_GRN"},
	{GPIO_PORT_LED_RSSI_RED,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "RSSI_LED_RED"},
	{GPIO_PORT_LED_RSSI_GRN,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "RSSI_LED_GRN"},
	{GPIO_PORT_LED_WIFI_RED,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "WIFI_LED_RED"},
	{GPIO_PORT_LED_WIFI_GRN,	GPIOF_OUT_INIT_LOW  | GPIOF_EXPORT, "WIFI_LED_GRN"},
};

struct led_gpio {
	void (*io)(const struct led_gpio_arg *, int);
	const struct led_gpio_arg arg;
} led_gpio_tbl[] = {
	[LED_ALL_GRN] = {.io = gpio_write_multi,
			 .arg ={.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_POWER_GRN,
					GPIO_PORT_LED_RSSI_GRN,
					GPIO_PORT_LED_WIFI_GRN,
					GPIO_PORT_LEDMAX}}}},
	[LED_ALL_RED] = {.io = gpio_write_multi,
			 .arg ={.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_POWER_RED,
					GPIO_PORT_LED_RSSI_RED,
					GPIO_PORT_LED_WIFI_RED,
					GPIO_PORT_LEDMAX}}}},
	[LED_ALL_ORANGE] = {.io = gpio_write_multi,
			    .arg ={.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_POWER_RED,
					GPIO_PORT_LED_POWER_GRN,
					GPIO_PORT_LED_RSSI_RED,
					GPIO_PORT_LED_RSSI_GRN,
					GPIO_PORT_LED_WIFI_RED,
					GPIO_PORT_LED_WIFI_GRN,
					GPIO_PORT_LEDMAX}}}},
	[LED_ALL] = {.io = gpio_write_multi,
			 .arg ={.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_POWER_RED,
					GPIO_PORT_LED_POWER_GRN,
					GPIO_PORT_LED_RSSI_RED,
					GPIO_PORT_LED_RSSI_GRN,
					GPIO_PORT_LED_WIFI_RED,
					GPIO_PORT_LED_WIFI_GRN,
				 	GPIO_PORT_LED_RSSI_BLU,
					GPIO_PORT_LEDMAX}}}},
	[LED_POWER_GRN] = {.io = gpio_write_one,
			  .arg = {.u = {.pin = GPIO_PORT_LED_POWER_GRN}}},
	[LED_POWER_RED] = {.io = gpio_write_one,
			  .arg = {.u = {.pin = GPIO_PORT_LED_POWER_RED}}},
	[LED_POWER_ORANGE] = {.io = gpio_write_multi,
			  .arg = {.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_POWER_GRN,
					GPIO_PORT_LED_POWER_RED,
					GPIO_PORT_LEDMAX}}}},
	[LED_RSSI_GRN] = {.io = gpio_write_one,
			 .arg = {.u = {.pin = GPIO_PORT_LED_RSSI_GRN}}},
	[LED_RSSI_RED] = {.io = gpio_write_one,
			 .arg = {.u = {.pin = GPIO_PORT_LED_RSSI_RED}}},
	[LED_RSSI_BLU] = {.io = gpio_write_one,
			  .arg = {.u = {.pin = GPIO_PORT_LED_RSSI_BLU}}},
	[LED_RSSI_ORANGE] = {.io = gpio_write_multi,
			    .arg = {.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_RSSI_GRN,
					GPIO_PORT_LED_RSSI_RED,
					GPIO_PORT_LEDMAX}}}},
	[LED_RSSI_ALL] = {.io = gpio_write_multi,
			    .arg = {.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_RSSI_GRN,
					GPIO_PORT_LED_RSSI_RED,
			    	GPIO_PORT_LED_RSSI_BLU,
					GPIO_PORT_LEDMAX}}}},
	[LED_WIFI_GRN] = {.io = gpio_write_one,
			 .arg = {.u = {.pin = GPIO_PORT_LED_WIFI_GRN}}},
	[LED_WIFI_RED] = {.io = gpio_write_one,
			 .arg = {.u = {.pin = GPIO_PORT_LED_WIFI_RED}}},
	[LED_WIFI_ORANGE] = {.io = gpio_write_multi,
			    .arg = {.u = {.pin_tbl = (int []){
					GPIO_PORT_LED_WIFI_GRN,
					GPIO_PORT_LED_WIFI_RED,
					GPIO_PORT_LEDMAX}}}},
};

static void
led_write_gpio(struct led_softc *sc, led_t type, int on)
{
	const struct led_gpio *tbl = led_gpio_tbl;

	if (type != 0)
#if defined(LED_GPIO_ACTIVE_LOW)
		tbl[type].io(&tbl[type].arg, on ? 0 : 1);
#else
		tbl[type].io(&tbl[type].arg, on ? 1 : 0);
#endif
}

static void
gpio_write_one(const struct led_gpio_arg *arg, int on)
{
	if (arg->u.pin >= 0)
		gpio_set_value(arg->u.pin, on);
}

static void
gpio_write_multi(const struct led_gpio_arg *arg, int on)
{
	int *pin_tbl;

	for (pin_tbl = arg->u.pin_tbl;
	     *pin_tbl != GPIO_PORT_LEDMAX; pin_tbl++) {
		if (*pin_tbl >= 0)
			gpio_set_value(*pin_tbl, on);
	}
}

#if NPSOC > 0
static void
psoc_write_one(const struct led_gpio_arg *arg, int on)
{
	switch (arg->u.pin) {
	case PSOC_LED_POWER_FLICKER_GRN:
		psoc_ledflicker_set(on ? 0 : 1);
		break;
	default:
		break;
	}
}
#endif

static struct led_softc *lsc;

static int __init
led_board_init(void)
{
	struct led_softc *sc;
	int rc;

	rc = gpio_request_array(led_gpios, ARRAY_SIZE(led_gpios));
	if (rc)
		return rc;

	lsc = sc = kmalloc(sizeof(*sc), GFP_KERNEL);
	if (sc == NULL)
		return -ENOMEM;

	memset(sc, 0, sizeof(*sc));

	sc->type_alias = ledtype_alias;
	sc->act_alias = ledact_alias;
	sc->action_list = action_list;
	sc->blinkspeed_list = NULL;
	sc->bitmap = led_stone_bitmap;
	sc->type_name_list = type_name_list;
	sc->act_name_list = act_name_list;
	sc->ledarg = sc;
	sc->ledwrite = (void (*)(void *, uint32_t, int)) led_write_gpio;

	led_attach_common((void *)sc);

	return 0;
}

static void __exit
led_board_exit(void)
{
	led_detach_common(lsc);
	kfree(lsc);

	gpio_free_array(led_gpios, ARRAY_SIZE(led_gpios));
}

module_init(led_board_init);
module_exit(led_board_exit);

MODULE_DESCRIPTION("NECPF led device interface");
MODULE_LICENSE("GPL v2");
