#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/slab.h>

#include "ledvar.h"

static void led_tick(unsigned long userp);

struct led_softc *led_sc = NULL;

#define LEDINTERVAL 50

#if defined(LED_DEBUG)
#define led_debug(format, args...)  \
	do {					       \
		printk("%s:%d ", __func__, __LINE__);  \
		printk(format , ## args);	       \
	} while (0)
#else
#define led_debug(format, args...)
#endif

#if defined(LED_NIGHTMODE)
static void nightmode_init(struct led_softc *sc);
static DEFINE_SPINLOCK(lednm_lock);
#endif

static const status_name_list_t status_name_list[] = {
	{"none", LEDST_NONE},
	{"off", LEDST_OFF},
	{"green", LEDST_GREEN},
	{"red", LEDST_RED},
	{"orange", LEDST_ORANGE},
	{"blink_green", LEDST_BLINK_GREEN},
	{"blink_red", LEDST_BLINK_RED},
	{"blink_orange", LEDST_BLINK_ORANGE},
	{"blue", LEDST_BLUE},
	{"lightblue", LEDST_LIGHTBLUE},
	{"pink", LEDST_PINK},
	{"white", LEDST_WHITE},
	{"blink_blue", LEDST_BLINK_BLUE},
	{"blink_lightblue", LEDST_BLINK_LIGHTBLUE},
	{"blink_pink", LEDST_BLINK_PINK},
	{"blink_white", LEDST_BLINK_WHITE},
	{NULL, LEDST_MAX}
};

static const blinkmode_name_list_t blinkmode_name_list[] = {
	{"normal", LEDBM_NORMAL},
	{"ignore_eco", LEDBM_IGNORE_ECO},
	{NULL, LEDBM_MAX}
};
#if defined(LED_NIGHTMODE)
static const nm_act_name_list_t nm_act_name_list[] = {
	{"all", LEDNM_ALL},
	{"except_power", LEDNM_EXCEPT_POWER},
	{"off", LEDNM_OFF},
	{"recover", LEDNM_RECOVER},
	{"oneshot_off", LEDNM_ONESHOT_OFF},
	{"temp_off", LEDNM_TEMP_OFF},
	{"oneshot_off_end", LEDNM_ONESHOT_OFF_END},
	{"temp_off_end", LEDNM_TEMP_OFF_END},
	{NULL, LEDNM_MAX}
};
#endif

static int (* const ledfunc_list[])(ledact_t *) = {
	[LEDFUNC_ACT_ONESHOT] = act_oneshot,
	[LEDFUNC_ACT_ONESHOT_RED] = act_oneshot_red,
	[LEDFUNC_ACT_ONESHOT_ORANGE] = act_oneshot_orange,
	[LEDFUNC_ACT_1SEC_ORANGE] = act_1sec_orange,
	[LEDFUNC_ACT_2SEC_GREEN] = act_2sec_green,
	[LEDFUNC_ACT_BLINK_NORMAL1] = act_blink_normal1,
	[LEDFUNC_ACT_FLUSH] = act_flush,
	[LEDFUNC_ACT_FLUSH_GREEN] = act_flush_green,
	[LEDFUNC_ACT_FLUSH2_RED] = act_flush2_red,
	[LEDFUNC_ACT_FLUSH3_LIGHTBLUE] = act_flush3_lightblue,
	[LEDFUNC_ACT_ON] = act_on,
	[LEDFUNC_ACT_OFF] = act_off,
	[LEDFUNC_ACT_BLINK1] = act_blink1,
	[LEDFUNC_ACT_BLINK2] = act_blink2,
	[LEDFUNC_ACT_BLINK3] = act_blink3,
	[LEDFUNC_ACT_BLINK4] = act_blink4,
	[LEDFUNC_ACT_BLINK5] = act_blink5,
	[LEDFUNC_ACT_BLINK6] = act_blink6,
	[LEDFUNC_ACT_BLINK7] = act_blink7,
	[LEDFUNC_ACT_BLINK8] = act_blink8,
	[LEDFUNC_ACT_BLINK9] = act_blink9,
	[LEDFUNC_ACT_BLINK10] = act_blink10,
	[LEDFUNC_ACT_MAX] = NULL
};

void
led_attach_common(void *self)
{
	struct led_softc *sc = (struct led_softc *) self;

	memset(&sc->led, 0, sizeof(sc->led));

	if (sc->type_alias == NULL)
		panic("led: type_alias is null");
	if (sc->act_alias == NULL)
		panic("led: act_alias is null");
	if (sc->action_list == NULL)
		panic("led: action_list is null");
	if (sc->type_name_list == NULL)
		panic("led: type_name_list is null");
	if (sc->act_name_list == NULL)
		panic("led: act_name_list is null");
	if (sc->bitmap == NULL)
		panic("led: bitmap is null");
	if (sc->ledwrite == NULL)
		panic("led: ledwrite function is null");

	sc->led.chnd.expires = jiffies + HZ;
	sc->led.chnd.function = led_tick;
	sc->led.chnd.data = (unsigned long)sc;
	init_timer(&sc->led.chnd);

	led_sc = sc;


#if defined(LED_NIGHTMODE)
	nightmode_init(sc);
#endif

#if CTL_POWER_LED
	ledwrite(LEDT_ALL, LEDACT_INACTIVATE2, 0);
#endif

	ledwrite(LEDT_POWER, LEDACT_NORMAL, 1);

	add_timer(&sc->led.chnd);


}
EXPORT_SYMBOL(led_attach_common);

void
led_detach_common(void *self)
{
	struct led_softc *sc = (struct led_softc *) self;

	if (led_sc == sc) {
		del_timer(&sc->led.chnd);
		led_sc = NULL;
	}
}
EXPORT_SYMBOL(led_detach_common);

int
ledactive(ledtype_t type)
{
	struct led_softc *sc = led_sc;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return 0;
	}
	return sc->bitmap[type].mask == 0 ? 0 : 1;
}
EXPORT_SYMBOL(ledactive);

ledact_status_t
ledstatus(ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return LEDST_NONE;
	}
	action = sc->led.action[type].action;
	for (idx = 0; sc->action_list[idx].action != 0; idx++) {
		if (sc->action_list[idx].status == LEDST_NONE)
			continue;
		if (action & sc->action_list[idx].action)
			break;
	}
	return sc->action_list[idx].status;
}
EXPORT_SYMBOL(ledstatus);

ledact_status_t
ledstatus_inactive(ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX) {
		return LEDST_NONE;
	}
	action = sc->led.action[type].action;
	action &= ~((ledact_bitmap_t) 1 <<
		    (sc->act_alias[LEDACT_INACTIVATE] - 1));
	for (idx = 0; sc->action_list[idx].action != 0; idx++) {
		if (sc->action_list[idx].status == LEDST_NONE)
			continue;
		if (action & sc->action_list[idx].action)
			break;
	}
	return sc->action_list[idx].status;
}
EXPORT_SYMBOL(ledstatus_inactive);

ledact_blinkspeed_t
ledstatus_blink(ledact_blinkmode_t mode, ledtype_t type)
{
	struct led_softc *sc = led_sc;
	ledact_bitmap_t action;
	int idx;

	if (sc->blinkspeed_list == NULL)
		return LEDBS_NONE;

	type = sc->type_alias[type];
	if (type >= LEDT_REAL_MAX)
		return LEDBS_NONE;

	action = sc->led.action[type].action;

	if (mode == LEDBM_IGNORE_ECO)
		action &= ~((ledact_bitmap_t) 1 <<
			    (sc->act_alias[LEDACT_INACTIVATE] - 1));
	for (idx = 0; sc->blinkspeed_list[idx].action != 0; idx++) {
		if (action & sc->blinkspeed_list[idx].action)
			break;
	}
	return sc->blinkspeed_list[idx].blinkspeed;
}
EXPORT_SYMBOL(ledstatus_blink);

#if defined(LED_NIGHTMODE)
static void
nightmode_init(struct led_softc *sc)
{
	unsigned long flags;
	spin_lock_irqsave(&lednm_lock, flags);
	sc->nightmode.stat = 1 << LEDNM_ONESHOT_OFF;
	sc->nightmode.temp_stat = sc->nightmode.stat;
	sc->nightmode.count = 0;
	spin_unlock_irqrestore(&lednm_lock, flags);
}

static int
ledwrite_multi_type(const struct multi_type_list *list, int enable)
{
	const struct multi_type_list *li;
	int ret = 0;

	if (enable) {
		for (li = list; li->type != LEDT_MAX; li++) {
			ret = ledwrite(li->type, li->action, enable);
			if (ret < 0)
				return ret;
		}
		return ret;
	}
	for (li = list; li->type != LEDT_MAX; li++) {
		if (li->acts != NULL) {
			const ledact_type_t *acts;

			for (acts = li->acts; *acts != LEDACT_MAX; acts++)
				ledwrite(li->type, *acts, enable);
		}
		ret = ledwrite(li->type, li->action, enable);
		if (ret < 0)
			return ret;
	}
	return ret;
}

static int
nightmode_refresh(struct led_softc *sc)
{
	if (sc->nightmode.temp_stat == sc->nightmode.stat)
		return 0;

	if (!(sc->nightmode.stat & (1 << LEDNM_ALL)) &&
	    !(sc->nightmode.stat & (1 << LEDNM_EXCEPT_POWER))) {
		ledwrite_multi_type(nm_all_list, 0);
		return 0;
	}

	if ((sc->nightmode.stat & (1 << LEDNM_TEMP_OFF)) ||
	    (sc->nightmode.stat & (1 << LEDNM_ONESHOT_OFF))) {
		ledwrite_multi_type(nm_all_list, 0);
		return 0;
	}

	switch (sc->nightmode.stat &
		((1 << LEDNM_ALL) | (1 << LEDNM_EXCEPT_POWER))) {
	case (1 << LEDNM_ALL):
		ledwrite_multi_type(nm_all_list, 1);
		return 0;
	case (1 << LEDNM_EXCEPT_POWER):
		ledwrite_multi_type(nm_all_list, 1);
		ledwrite(LEDT_POWER, LEDACT_INACTIVATE2, 0);
		return 0;
	default:
		return -1;
	}
}

int
ledwrite_nightmode(lednm_act_t action)
{
	struct led_softc *sc = led_sc;
	unsigned long flags;

	if (action >= LEDNM_MAX) {
		return 0;
	}

	spin_lock_irqsave(&lednm_lock, flags);
	sc->nightmode.temp_stat = sc->nightmode.stat;
	switch (action) {
	case LEDNM_TEMP_OFF:
		sc->nightmode.stat |= 1 << LEDNM_TEMP_OFF;

		led_debug("action=%d\n", action);
		break;
	case LEDNM_RECOVER:
		sc->nightmode.stat &= ~((1 << LEDNM_TEMP_OFF) |
					(1 << LEDNM_ONESHOT_OFF));

		led_debug("action=%d\n", action);
		break;
	case LEDNM_ONESHOT_OFF:
		sc->nightmode.stat |= 1 << LEDNM_ONESHOT_OFF;
		sc->nightmode.count = 0;

		led_debug("action=%d\n", action);
		break;
	case LEDNM_OFF:
		sc->nightmode.stat &= ~((1 << LEDNM_TEMP_OFF) |
					(1 << LEDNM_ONESHOT_OFF) |
					(1 << LEDNM_ALL) |
					(1 << LEDNM_EXCEPT_POWER));

		led_debug("action=%d\n", action);
		break;
	case LEDNM_ALL:
		sc->nightmode.stat &= ~(1 << LEDNM_EXCEPT_POWER);
		sc->nightmode.stat |= 1 << LEDNM_ALL;

		led_debug("action=%d\n", action);
		break;
	case LEDNM_EXCEPT_POWER:
		sc->nightmode.stat &= ~(1 << LEDNM_ALL);
		sc->nightmode.stat |= 1 << LEDNM_EXCEPT_POWER;

		led_debug("action=%d\n", action);
		break;
	case LEDNM_TEMP_OFF_END:
		sc->nightmode.stat &= ~(1 << LEDNM_TEMP_OFF);

		led_debug("action=%d\n", action);
		break;
	case LEDNM_ONESHOT_OFF_END:
		sc->nightmode.stat &= ~(1 << LEDNM_ONESHOT_OFF);
		sc->nightmode.count = 0;

		led_debug("action=%d\n", action);
		break;
	default:
		led_debug("action=%d\n", action);
		break;
	}

	nightmode_refresh(sc);
	spin_unlock_irqrestore(&lednm_lock, flags);
	return 0;
}
EXPORT_SYMBOL(ledwrite_nightmode);

#define LEDNM_ONESHOT_MAX_COUNT (((LEDNM_ONESHOT_OFF_SEC) * 1000) /	\
				 (LEDINTERVAL))
static int
nightmode_oneshot_off_end(struct led_softc *sc)
{
	unsigned long flags;
	spin_lock_irqsave(&lednm_lock, flags);

	if (!(sc->nightmode.stat & (1 << LEDNM_ONESHOT_OFF))) {
		spin_unlock_irqrestore(&lednm_lock, flags);
		return 0;
	}
	sc->nightmode.count++;
	if (sc->nightmode.count < LEDNM_ONESHOT_MAX_COUNT) {
		spin_unlock_irqrestore(&lednm_lock, flags);
		return 0;
	}
	spin_unlock_irqrestore(&lednm_lock, flags);

	led_debug("expired\n");
	ledwrite_nightmode(LEDNM_ONESHOT_OFF_END);
	return 0;
}

void
led_nm_status(u_int32_t *stat)
{
	struct led_softc *sc = led_sc;
	unsigned long flags;
	spin_lock_irqsave(&lednm_lock, flags);
	*stat = sc->nightmode.stat;
	spin_unlock_irqrestore(&lednm_lock, flags);
}
EXPORT_SYMBOL(led_nm_status);
#endif

#define LED_MAX_INTERVAL (60000 / LEDINTERVAL)
#define LEDACT_BITMAP(x) ((ledact_bitmap_t) 1 << ((x) - 1))
static void
led_tick(unsigned long userp)
{
	struct led_softc *sc = (struct led_softc *)userp;
	ledact_t *act;
	int type;
	int refresh = 0;
	int alloff = 0;

	sc->led.count = (sc->led.count + 1) % LED_MAX_INTERVAL;
	for (type = 0, act = &sc->led.action[type];
	     type < LEDT_REAL_MAX; type++, act++) {
		ledact_bitmap_t action = act->action;
		int on;
		int idx;
		
		if (type == LEDT_ALL && !action) {
			if (act->exec_action) {
				refresh = 1;
				act->exec_action = action;
			}
			continue;
		}

		if (type == LEDT_ALL &&
		    (action & LEDACT_BITMAP(LEDACT_ALLCLEAR))) {
			refresh = 1;
			alloff = 1;
			act->action = 0;
			act->exec_action = 0;
			continue;
		}
		if ((alloff && refresh) ||
		    (action & LEDACT_BITMAP(LEDACT_CLEAR))) {
			action = 0;
			act->action = 0;
		}

		for (idx = 0; sc->action_list[idx].action != 0; idx++) {
			if (action & sc->action_list[idx].action) {
				act->current_action = action & sc->action_list[idx].action;
				break;
			}
		}
		if (sc->action_list[idx].sync) {
			act->count = sc->led.count;
		}
		on = sc->action_list[idx].func(act);
		if (refresh || (act->exec_action != action) ||
		    (on != act->on)) {
			int color = 0;
			switch (on) {
			case 0:
				color = sc->action_list[idx].off_color;
				break;
			case 1:
				color = sc->action_list[idx].on_color;
				break;
			case 2:
				color = sc->action_list[idx].color_3rd;
				break;
			default:
				break;
			}

			sc->ledwrite(sc->ledarg, sc->bitmap[type].mask, 0);
			if (color != LEDCOLOR_NONE) {
				sc->ledwrite(
					sc->ledarg,
					sc->bitmap[type].color[color], 1);
			}
			act->on = on;
			act->exec_action = action;
		}
		if (type == LEDT_ALL && action)
			break;
	}
#if defined(LED_NIGHTMODE)
	nightmode_oneshot_off_end(sc);
#endif

	mod_timer(&sc->led.chnd, jiffies + HZ / (1000 / LEDINTERVAL));
}

#ifndef LEDACT_FLUSH_1
#define LEDACT_FLUSH_1 LEDACT_DUMMY
#endif
int
act_oneshot(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_1] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_oneshot);

#ifndef LEDACT_FLUSH_10
#define LEDACT_FLUSH_10 LEDACT_DUMMY
#endif
int
act_oneshot_red(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &=  ~((ledact_bitmap_t) 1 <<
				  (sc->act_alias[LEDACT_FLUSH_10] - 1));
		act->count = 0;
	}
	return on;
}

#ifndef LEDACT_FLUSH_2
#define LEDACT_FLUSH_2 LEDACT_DUMMY
#endif
int
act_oneshot_orange(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = act->on;

	on = (on + 1) % 2;
	if (++(act->count) >= 2) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_2] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_oneshot_orange);

#ifndef LEDACT_FLUSH_3
#define LEDACT_FLUSH_3 LEDACT_DUMMY
#endif
#define LED1SEC_INTERVAL (1000 / LEDINTERVAL)
int
act_1sec_orange(ledact_t *act)
{
	struct led_softc *sc = led_sc;

	if (++act->count <= LED1SEC_INTERVAL)
		return 1;
	act->action &= ~((ledact_bitmap_t) 1 <<
			 (sc->act_alias[LEDACT_FLUSH_3] - 1));
	act->count = 0;
	return 0;
}
EXPORT_SYMBOL(act_1sec_orange);

#ifndef LEDACT_FLUSH_4
#define LEDACT_FLUSH_4 LEDACT_DUMMY
#endif
#define LED2SEC_INTERVAL (2000 / LEDINTERVAL)
int
act_2sec_green(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = 1;

	act->count++;
	if (act->count >= LED2SEC_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_4] - 1));
		act->count = 0;
		on = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_2sec_green);

#ifndef LEDACT_FLUSH_5
#define LEDACT_FLUSH_5 LEDACT_DUMMY
#endif
#define LEDBLINK_NORMAL1_BLINK_INTERVAL (1000 / LEDINTERVAL)
#define LEDBLINK_NORMAL1_BLINK_COUNT 3
#define LEDBLINK_NORMAL1_ON_INTERVAL (2000 / LEDINTERVAL)
#define LEDBLINK_NORMAL1_INTERVAL (LEDBLINK_NORMAL1_BLINK_INTERVAL * \
				   LEDBLINK_NORMAL1_BLINK_COUNT + \
				   LEDBLINK_NORMAL1_ON_INTERVAL)
int
act_blink_normal1(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	int on = 0;
	unsigned count = act->count % LEDBLINK_NORMAL1_BLINK_INTERVAL;

	act->count++;
	if (act->count < LEDBLINK_NORMAL1_BLINK_INTERVAL *
	    LEDBLINK_NORMAL1_BLINK_COUNT)
		on = (count < (LEDBLINK_NORMAL1_BLINK_INTERVAL / 2)) ? 1 : 0;
	else if (act->count >= LEDBLINK_NORMAL1_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_5] - 1));
		act->count = 0;
		on = 0;
	}
	else
		on = 1;

	return on;
}
EXPORT_SYMBOL(act_blink_normal1);

#ifndef LEDACT_FLUSH_6
#define LEDACT_FLUSH_6 LEDACT_DUMMY
#endif
#define LEDFLUSH_INTERVAL (1000 / LEDINTERVAL)
#define LEDFLUSH_ONESHOT_COUNT 3
#define LEDFLUSH_TOTAL_INTERVAL (LEDFLUSH_INTERVAL * LEDFLUSH_ONESHOT_COUNT)
int
act_flush(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH_TOTAL_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_6] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_flush);

#ifndef LEDACT_FLUSH_7
#define LEDACT_FLUSH_7 LEDACT_DUMMY
#endif
int
act_flush_green(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH_TOTAL_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_7] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_flush_green);

#ifndef LEDACT_FLUSH_8
#define LEDACT_FLUSH_8 LEDACT_DUMMY
#endif
#define LEDFLUSH2_ONESHOT_COUNT 10
#define LEDFLUSH2_TOTAL_INTERVAL (LEDFLUSH_INTERVAL * LEDFLUSH2_ONESHOT_COUNT)
int
act_flush2_red(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH_INTERVAL;
	int on = (count < (LEDFLUSH_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH2_TOTAL_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_8] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_flush2_red);

#ifndef LEDACT_FLUSH_9
#define LEDACT_FLUSH_9 LEDACT_DUMMY
#endif
#define LEDFLUSH3_INTERVAL (500 / LEDINTERVAL)
#define LEDFLUSH3_ONESHOT_COUNT 4
#define LEDFLUSH3_TOTAL_INTERVAL (LEDFLUSH3_INTERVAL * LEDFLUSH3_ONESHOT_COUNT)
int
act_flush3_lightblue(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH3_INTERVAL;
	int on = (count < (LEDFLUSH3_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH3_TOTAL_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_9] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_flush3_lightblue);

#ifndef LEDACT_FLUSH_10
#define LEDACT_FLUSH_10 LEDACT_DUMMY
#endif
#define LEDFLUSH4_INTERVAL (400 / LEDINTERVAL)
#define LEDFLUSH4_TOTAL_INTERVAL (3000 / LEDINTERVAL)
int
act_flush4(ledact_t *act)
{
	struct led_softc *sc = led_sc;
	unsigned count = act->count % LEDFLUSH4_INTERVAL;
	int on = (count < (LEDFLUSH4_INTERVAL / 2)) ? 1 : 0;

	act->count++;
	if (act->count >= LEDFLUSH4_TOTAL_INTERVAL) {
		act->action &= ~((ledact_bitmap_t) 1 <<
				 (sc->act_alias[LEDACT_FLUSH_10] - 1));
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_flush4);

int act_on(ledact_t *act)
{
	return 1;
}
EXPORT_SYMBOL(act_on);

int act_off(ledact_t *act)
{
	return 0;
}
EXPORT_SYMBOL(act_off);

#define LEDBLINK1_ON_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK1_OFF_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK1_INTERVAL (LEDBLINK1_ON_INTERVAL + LEDBLINK1_OFF_INTERVAL)
int act_blink1(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK1_INTERVAL;
	int on = (count < LEDBLINK1_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK1_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink1);

#define LEDBLINK2_ON_INTERVAL (1000 / LEDINTERVAL)
#define LEDBLINK2_OFF_INTERVAL (1000 / LEDINTERVAL)
#define LEDBLINK2_INTERVAL (LEDBLINK2_ON_INTERVAL + LEDBLINK2_OFF_INTERVAL)
int act_blink2(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK2_INTERVAL;
	int on = (count < LEDBLINK2_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK2_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink2);

#define LEDBLINK3_ON_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK3_OFF_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK3_INTERVAL (LEDBLINK3_ON_INTERVAL + LEDBLINK3_OFF_INTERVAL)
int act_blink3(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK3_INTERVAL;
	int on = (count < LEDBLINK3_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK3_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink3);

#define LEDBLINK4_ON_INTERVAL (400 / LEDINTERVAL)
#define LEDBLINK4_OFF_INTERVAL (400 / LEDINTERVAL)
#define LEDBLINK4_INTERVAL (LEDBLINK4_ON_INTERVAL + LEDBLINK4_OFF_INTERVAL)
int act_blink4(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK4_INTERVAL;
	int on = (count < LEDBLINK4_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK4_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink4);

#define LEDBLINK5_ON_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK5_OFF_INTERVAL (900 / LEDINTERVAL)
#define LEDBLINK5_INTERVAL (LEDBLINK5_ON_INTERVAL + LEDBLINK5_OFF_INTERVAL)
int act_blink5(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK5_INTERVAL;
	int on = (count < LEDBLINK5_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK5_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink5);

#define LEDBLINK6_ON_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK6_OFF_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK6_INTERVAL (LEDBLINK6_ON_INTERVAL + LEDBLINK6_OFF_INTERVAL)
int act_blink6(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK6_INTERVAL;
	int on = (count < LEDBLINK6_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK6_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink6);

#define LEDBLINK7_ON_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK7_OFF_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK7_INTERVAL (LEDBLINK7_ON_INTERVAL + LEDBLINK7_OFF_INTERVAL)
int act_blink7(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK7_INTERVAL;
	int on = (count < LEDBLINK7_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK7_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink7);

#define LEDBLINK8_ON_INTERVAL (1000 / LEDINTERVAL)
#define LEDBLINK8_OFF_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK8_INTERVAL (LEDBLINK8_ON_INTERVAL + LEDBLINK8_OFF_INTERVAL)
int act_blink8(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK8_INTERVAL;
	unsigned count2 = act->count % LEDBLINK6_INTERVAL;
	int on;

	if (count < LEDBLINK8_ON_INTERVAL && count2 < LEDBLINK6_ON_INTERVAL) {
		on = 1;
	} else {
		on = 0;
	}

	act->count++;
	if (act->count >= LEDBLINK8_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink8);

#define LEDBLINK9_1ST_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK9_2ND_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK9_3RD_INTERVAL (200 / LEDINTERVAL)
#define LEDBLINK9_INTERVAL (LEDBLINK9_1ST_INTERVAL + \
				LEDBLINK9_2ND_INTERVAL + \
				LEDBLINK9_3RD_INTERVAL)
int act_blink9(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK9_INTERVAL;
	int on;

	if (count < LEDBLINK9_1ST_INTERVAL) {
		on = 1;
	} else if (count >= LEDBLINK9_1ST_INTERVAL &&
		   count < (LEDBLINK9_1ST_INTERVAL +
			    LEDBLINK9_2ND_INTERVAL)) {
		on = 0;
	} else {
		on = 2;
	}

	act->count++;
	if (act->count >= LEDBLINK9_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink9);

#define LEDBLINK10_ON_INTERVAL (250 / LEDINTERVAL)
#define LEDBLINK10_OFF_INTERVAL (250 / LEDINTERVAL)
#define LEDBLINK10_INTERVAL (LEDBLINK10_ON_INTERVAL + LEDBLINK10_OFF_INTERVAL)
int act_blink10(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK10_INTERVAL;
	int on = (count < LEDBLINK10_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK10_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink10);

#define LEDBLINK11_ON_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK11_OFF_INTERVAL (4900 / LEDINTERVAL)
#define LEDBLINK11_INTERVAL (LEDBLINK11_ON_INTERVAL + LEDBLINK11_OFF_INTERVAL)
int act_blink11(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK11_INTERVAL;
	int on = (count < LEDBLINK11_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK11_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink11);

#define LEDBLINK12_ON_INTERVAL (3500 / LEDINTERVAL)
#define LEDBLINK12_OFF_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK12_INTERVAL (LEDBLINK12_ON_INTERVAL + LEDBLINK12_OFF_INTERVAL)
int act_blink12(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK12_INTERVAL;
	int on = (count < LEDBLINK12_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK12_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink12);

#define LEDBLINK13_ON_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK13_OFF_INTERVAL (4500 / LEDINTERVAL)
#define LEDBLINK13_INTERVAL (LEDBLINK13_ON_INTERVAL + LEDBLINK13_OFF_INTERVAL)
int act_blink13(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK13_INTERVAL;
	int on = (count < LEDBLINK13_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK13_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink13);

#define LEDBLINK14_ON_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK14_OFF_INTERVAL (100 / LEDINTERVAL)
#define LEDBLINK14_INTERVAL (LEDBLINK14_ON_INTERVAL + LEDBLINK14_OFF_INTERVAL)
int act_blink14(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK14_INTERVAL;
	int on = (count < LEDBLINK14_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK14_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink14);

#define LEDBLINK15_ON_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK15_OFF_INTERVAL (300 / LEDINTERVAL)
#define LEDBLINK15_INTERVAL (LEDBLINK15_ON_INTERVAL + LEDBLINK15_OFF_INTERVAL)
int act_blink15(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK15_INTERVAL;
	int on = (count < LEDBLINK15_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK15_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink15);

#define LEDBLINK16_ON_INTERVAL (150 / LEDINTERVAL)
#define LEDBLINK16_OFF_INTERVAL (350 / LEDINTERVAL)
#define LEDBLINK16_INTERVAL (LEDBLINK16_ON_INTERVAL + LEDBLINK16_OFF_INTERVAL)
int act_blink16(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK16_INTERVAL;
	int on = (count < LEDBLINK16_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK16_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink16);

#define LEDBLINK17_ON_INTERVAL (1500 / LEDINTERVAL)
#define LEDBLINK17_OFF_INTERVAL (500 / LEDINTERVAL)
#define LEDBLINK17_INTERVAL (LEDBLINK17_ON_INTERVAL + LEDBLINK17_OFF_INTERVAL)
int act_blink17(ledact_t *act)
{
	unsigned count = act->count % LEDBLINK17_INTERVAL;
	int on = (count < LEDBLINK17_ON_INTERVAL) ? 1 : 0;

	act->count++;
	if (act->count >= LEDBLINK17_INTERVAL) {
		act->count = 0;
	}
	return on;
}
EXPORT_SYMBOL(act_blink17);

static int
type_name_lookup(struct led_softc *sc, struct led_type_name_info *info)
{
	int i;
	const type_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = sc->type_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->type = list->type;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	for (i = 0, list = sc->type_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->type = list->type;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static int
act_name_lookup(struct led_softc *sc, struct led_act_name_info *info)
{
	int i;
	const act_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = sc->act_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->act = list->act;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	for (i = 0, list = sc->act_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->act = list->act;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

#if defined(LED_NIGHTMODE)
static int
nm_act_name_lookup(struct led_softc *sc, struct led_nm_act_name_info *info)
{
	int i;
        const nm_act_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = nm_act_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->nm_act = list->nm_act;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	for (i = 0, list = nm_act_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->nm_act = list->nm_act;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}
#endif

static int
status_name_lookup(struct led_softc *sc, struct led_status_name_info *info)
{
	int i;
	const status_name_list_t *list;

	if (info->idx == LED_IDX_MATCHSTATUS) {
		for (i = 0, list = status_name_list;
		     list->name != NULL; list++, i++) {
			if (info->status == list->status)
				break;
		}
		if (list->name != NULL) {
			strlcpy(info->name, list->name, sizeof(info->name));
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	for (i = 0, list = status_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->status = list->status;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static int
blinkmode_name_lookup(struct led_softc *sc, struct led_blinkmode_name_info *info)
{
	int i;
	const blinkmode_name_list_t *list;

	if (info->idx == LED_IDX_MATCHNAME) {
		for (i = 0, list = blinkmode_name_list;
		     list->name != NULL; list++, i++) {
			if (strcmp(list->name, info->name) == 0)
				break;
		}
		if (list->name != NULL) {
			info->mode = list->mode;
			info->idx = i;
			return 0;
		}
		info->idx = -1;
		return -EINVAL;
	}
	for (i = 0, list = blinkmode_name_list;
	     list->name != NULL; list++, i++) {
		if (i == info->idx)
			break;
	}
	if (list->name != NULL) {
		info->mode = list->mode;
		strlcpy(info->name, list->name, sizeof(info->name));
		return 0;
	}
	return -EINVAL;
}

static long
ledioctl(struct file *filp, unsigned int cmd, unsigned long data)
{
	struct led_softc *sc = led_sc;

	if (sc == NULL)
		return -ENXIO;

	switch (cmd) {
	case LED_IOC_WRITE: {
		struct led_write_info *info = (struct led_write_info *) data;

		if ((info->type >= LEDT_MAX) || (info->action >= LEDACT_MAX)) {
			return 0;
		}
		ledwrite(info->type, info->action, info->enable);
		return 0;
	}
	case LED_IOC_SCAN: {
		struct led_scan_info *info = (struct led_scan_info *) data;

		info->active = ledactive(info->type) ? 1 : 0;
		return 0;
	}
	case LED_IOC_STATUS: {
		struct led_status_info *info = (struct led_status_info *) data;
		
		info->status = ledstatus(info->type);
		return 0;
	}
	case LED_IOC_STATUS_INACTIVE: {
		struct led_status_info *info = (struct led_status_info *) data;
		
		info->status = ledstatus_inactive(info->type);
		return 0;
	}
	case LED_IOC_TYPENAME : {
		struct led_type_name_info *info = (void *)data;

		return type_name_lookup(sc, info);
	}
	case LED_IOC_ACTNAME: {
		struct led_act_name_info *info = (void *)data;

		return act_name_lookup(sc, info);
	}
	case LED_IOC_STATUSNAME: {
		struct led_status_name_info *info = (void *)data;

		return status_name_lookup(sc, info);
	}
	case LED_IOC_STATUS_BLINKSPEED: {
		struct led_blinkspeed_info *info =
			(struct led_blinkspeed_info *) data;
		
		info->blinkspeed = ledstatus_blink(info->mode, info->type);
		return 0;
	}
	case LED_IOC_BLINKMODENAME: {
		struct led_blinkmode_name_info *info = (void *)data;

		return blinkmode_name_lookup(sc, info);
	}
#if defined(LED_NIGHTMODE)
	case LED_IOC_NM_ACTION: {
		struct led_nm_act_info *info =
			(struct led_nm_act_info *)data;

		if (info->action >= LEDNM_MAX) {
			return 0;
		}
		ledwrite_nightmode(info->action);
		return 0;
	}
	case LED_IOC_NM_STATUS: {
		struct led_nm_status_info *info =
			(struct led_nm_status_info *) data;

		led_nm_status(&info->stat);
		return 0;
	}
	case LED_IOC_NM_ACTNAME: {
		struct led_nm_act_name_info *info = (void *)data;

		return nm_act_name_lookup(sc, info);
        }
#endif
	default:
		break;
	}
	return -EFAULT;
}

static const struct file_operations ledcom_fops = {
	.owner   = THIS_MODULE,
	.unlocked_ioctl   = ledioctl,
};

static struct cdev ledcom_cdev;
static dev_t ledcom_dev;
static struct class *ledcom_class;

static int __init
ledcom_init(void)
{
	int rc;
	struct device *dev;

	ledcom_class = class_create(THIS_MODULE, "ledcom");
	if (IS_ERR(ledcom_class))
		return PTR_ERR(ledcom_class);

	rc = alloc_chrdev_region(&ledcom_dev, 0, 1, "led");
	if (rc)
		goto err_class_destroy;

	cdev_init(&ledcom_cdev, &ledcom_fops);
	ledcom_cdev.owner = THIS_MODULE;
	rc = cdev_add(&ledcom_cdev, ledcom_dev, 1);
	if (rc)
		goto err_unregister_chrdev;

	dev = device_create(ledcom_class, NULL, ledcom_dev, NULL, "led%d", 0);
	if (IS_ERR(dev)) {
		rc = PTR_ERR(dev);
		goto err_cdev_del;
	}

	return 0;

err_class_destroy:
	class_destroy(ledcom_class);
err_cdev_del:
	cdev_del(&ledcom_cdev);
err_unregister_chrdev:
	unregister_chrdev_region(ledcom_dev, 1);

	return rc;
}

static void __exit
ledcom_exit(void)
{
	device_destroy(ledcom_class, ledcom_dev);
	cdev_del(&ledcom_cdev);
	unregister_chrdev_region(ledcom_dev, 1);
	class_destroy(ledcom_class);
}

module_init(ledcom_init);
module_exit(ledcom_exit);

MODULE_DESCRIPTION("NECPF led common interface");
MODULE_LICENSE("GPL v2");
