#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>
#include <fcntl.h>
#include <unistd.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <time.h>
#include "../utils/includes.h"
#include "../utils/common.h"
#include "../ap/hostapd.h"
#include "wifi_unimgr_iface.h"
#include <pfmg/pfmgvar.h>
#include <wdf_sock_if.h>
#ifdef USE_CONFIG
#include "config_control.h"
#endif /* USE_CONFIG */

static const struct wifi_unimgr_iface *hostapd_get_wifi_unimgr_iface_tbl(const char *);
static int hostapd_send_wdf_msg(unsigned short, void *, size_t);

struct wifi_unimgr_iface {
	const char *pfmg_name;
	int wdf_iface;
} wifi_unimgr_iface_tbl[] = {
	{ PFMG_IFN_ALIAS_WLAN		, IF_24_PRI	},
	{ PFMG_IFN_ALIAS_WLAN_2ND	, IF_24_SEC	},
//	{ PFMG_IFN_ALIAS_WLAN_3RD	, IF_24_TER	},
	{ PFMG_IFN_ALIAS_WLAN2		, IF_5_PRI	},
	{ PFMG_IFN_ALIAS_WLAN2_2ND	, IF_5_SEC	},
	{ NULL				, -1		},
};

int
hostapd_wifi_unimgr_set_entry(struct hostapd_data *hapd, const uint8_t *macaddr, int fail_reason)
{
	int ret = 0;
#ifdef USE_CONFIG
	ret = cfg_sci_set_entry_connect_fail_reason(hapd, macaddr, fail_reason);
#endif /* USE_CONFIG */

	return ret;
}

int
hostapd_wifi_unimgr_del_entry(const uint8_t *macaddr)
{
	int ret = 0;
#ifdef USE_CONFIG
	ret = cfg_sci_del_entry_connect_fail_reason(macaddr);
#endif /* USE_CONFIG */

	return ret;
}

int
hostapd_wdf_ind_sta_connect(const uint8_t *macaddr, const char *ifname)
{
	const struct wifi_unimgr_iface *iface_tbl;
	struct wdf_sockmsg_ind_sta_connect msg_data;
	size_t msg_data_len = 0;

	if (macaddr == NULL || ifname == NULL)
		return -1;

	iface_tbl = hostapd_get_wifi_unimgr_iface_tbl(ifname);
	if (iface_tbl == NULL)
		return -1;

	/* create send data */
	os_memset(&msg_data, 0, sizeof(msg_data));
	os_memcpy(msg_data.mac, macaddr, sizeof(msg_data.mac));
	msg_data.interface = iface_tbl->wdf_iface;

	msg_data_len = sizeof(msg_data);

	return hostapd_send_wdf_msg(WDF_SOCKMSG_IND_STA_CONNECT, &msg_data, msg_data_len);
}

static const struct wifi_unimgr_iface *
hostapd_get_wifi_unimgr_iface_tbl(const char *ifname)
{
	struct wifi_unimgr_iface *tbl;
	const char *pfmg_ifname;

	for (tbl = wifi_unimgr_iface_tbl; tbl->pfmg_name != NULL; tbl++) {
		if ((pfmg_ifname = pfmg_ifrealname(tbl->pfmg_name)) == NULL) {
			wpa_printf(MSG_ERROR, "%s: pfmg ifname get error %s",
			    __func__, tbl->pfmg_name);
			continue;
		}
		if (os_strcmp(ifname, pfmg_ifname) == 0)
			return tbl;
	}

	return NULL;
}

static int
hostapd_send_wdf_msg(unsigned short req, void *msg_data, size_t msg_data_len)
{
	int sock = -1;
	struct sockaddr_un addr;
	struct wdf_sockmsg msg_head;
	size_t msg_head_len = 0;
	size_t sendlen = 0;
	char *sendbuf = NULL;
	int ret = -1;

	/* create send header */
	os_memset(&msg_head, 0, sizeof(msg_head));
	msg_head_len = sizeof(msg_head);
	msg_head.len = msg_data_len;
	msg_head.req = req;

	/* create send msg (header + data) */
	sendlen = msg_head_len + msg_data_len;
	sendbuf = os_calloc(1, sendlen);
	if (sendbuf == NULL)
		goto end;
	os_memcpy(sendbuf, (char *)&msg_head, msg_head_len);
	os_memcpy(sendbuf + msg_head_len, (char *)msg_data, msg_data_len);

	/* socket */
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
		wpa_printf(MSG_ERROR, "%s: socket error %s", __func__, strerror(errno));
		goto end;
	}
	os_memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_UNIX;
	os_strlcpy(addr.sun_path, WDF_SOCK_PATH, sizeof(addr.sun_path));
	if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		wpa_printf(MSG_ERROR, "%s: connect error %s", __func__, strerror(errno));
		goto end;
	}

	/* send data */
	if (sendto(sock, sendbuf, sendlen, 0, NULL, 0) < 0) {
		wpa_printf(MSG_ERROR, "%s: sendto error %s", __func__, strerror(errno));
		goto end;
	}

	ret = 0;
end:
	if (sock >= 0)
		close(sock);

	if (sendbuf != NULL)
		os_free(sendbuf);

	return ret;
}
