#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/cdev.h>
#include <linux/device.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/slab.h>
#include <linux/poll.h>
#include <linux/reboot.h>

#include <backupvar.h>
#include "btnuvar.h"

#if !defined(BTNU_LONGTIME)
#define BTNU_LONGTIME 0
#endif

static ssize_t btnucomread(struct file *, char __user *, size_t, loff_t *);
static ssize_t btnucomopen(struct inode *, struct file *);
static unsigned int btnucompoll(struct file *, struct poll_table_struct *);
static ssize_t btnucomrelease(struct inode *, struct file *);
static long btnucomioctl(struct file *, unsigned int, unsigned long);
static void btnu_tick(unsigned long);

static const struct file_operations btnucom_fops = {
	.owner   = THIS_MODULE,
	.read    = btnucomread,
	.open    = btnucomopen,
	.poll    = btnucompoll,
	.release = btnucomrelease,
	.unlocked_ioctl   = btnucomioctl,
};

static DEFINE_SPINLOCK(btnucom_irq_lock);
static struct class *btnucom_class;

int
btnucom_attach(struct btnucom *sc, int unit, void *parent)
{
	static const struct {
		int unit;
		unsigned timeout;
	} timeout_list[] = {
#if defined(BTNU_TIMEOUT_0)
		{0, BTNU_TIMEOUT_0},
#endif
#if defined(BTNU_TIMEOUT_1)
		{1, BTNU_TIMEOUT_1},
#endif
#if defined(BTNU_TIMEOUT_2)
		{2, BTNU_TIMEOUT_2},
#endif
		{-1, BTNU_TIMEOUT}
	};
	int i;
	int rc;
	struct device *dev;

	rc = alloc_chrdev_region(&sc->dev, unit, 1, "btnu");
	if (rc)
		return rc;

	cdev_init(&sc->cdev, &btnucom_fops);
	sc->cdev.owner = THIS_MODULE;
	rc = cdev_add(&sc->cdev, sc->dev, 1);
	if (rc)
		goto err_unregister_chrdev;

	dev = device_create(btnucom_class, NULL, sc->dev, NULL, "btnu%d", unit);
	if (IS_ERR(dev)) {
		rc = PTR_ERR(dev);
		goto err_cdev_del;
	}

	sc->parent = parent;
	sc->tick_callout.expires = jiffies + HZ / (1000 / BTNU_POLLTIME);
	sc->tick_callout.function = &btnu_tick;
	sc->tick_callout.data = (unsigned long)sc;
	init_timer(&sc->tick_callout);
	sc->open_count = 0;

	sc->sc_current = sc->sc_detect = 0;
	sc->sc_count = sc->sc_detect_count = 0;
	sc->sc_threshold = BTNU_THRESHOLD / BTNU_POLLTIME;
	sc->sc_longtime = BTNU_LONGTIME / BTNU_POLLTIME;
	for (i = 0; timeout_list[i].unit >= 0; i++) {
		if (timeout_list[i].unit == unit) {
			break;
		}
	}
	sc->sc_timeout = timeout_list[i].timeout / BTNU_POLLTIME;

	LIST_INIT(&sc->event_list);

	add_timer(&sc->tick_callout);

	return 0;

err_cdev_del:
	cdev_del(&sc->cdev);

err_unregister_chrdev:
	unregister_chrdev_region(sc->dev, 1);

	return rc;
}
EXPORT_SYMBOL(btnucom_attach);

void
btnucom_detach(struct btnucom *sc)
{
	device_destroy(btnucom_class, sc->dev);
	del_timer(&sc->tick_callout);
	cdev_del(&sc->cdev);
	unregister_chrdev_region(sc->dev, 1);
}
EXPORT_SYMBOL(btnucom_detach);

static void
btnu_enqueue_event(struct btnucom *sc, int event, unsigned tick)
{
	unsigned next;
	struct btnu_event_info *info;
	
	LIST_FOREACH(info, &sc->event_list, event_links) {
		next = (info->sc_event_in + 1) % BTNU_MAX_EVENTS;
		if (next != info->sc_event_out) {
			info->sc_event[info->sc_event_in].type = event;
			info->sc_event[info->sc_event_in].press_msec =
				tick * BTNU_POLLTIME;
			info->sc_event_in = next;
		}
		wake_up_interruptible(&info->sc_rsel);
	}
}

static int
btnu_exist_event(struct btnucom *sc, struct task_struct *p)
{
	struct btnu_event_info *info;
	
	LIST_FOREACH(info, &sc->event_list, event_links) {
		if (info->p == p) {
			return info->sc_event_in != info->sc_event_out;
		}
	}
	return 0;
}

static int
btnu_dequeue_event(
	struct btnucom *sc, struct btnu_read_event_info *rinfo,
	struct task_struct *p)
{
	struct btnu_event_info *info;

	rinfo->event = BTNU_EVENT_NONE;
	rinfo->press_msec = 0;
	LIST_FOREACH(info, &sc->event_list, event_links) {
		if (info->p == p) {
			if (info->sc_event_in == info->sc_event_out) {
				return BTNU_EVENT_NONE;
			}
			rinfo->event = info->sc_event[info->sc_event_out].type;
			rinfo->press_msec =
				info->sc_event[info->sc_event_out].press_msec;
			info->sc_event_out =
				(info->sc_event_out + 1) % BTNU_MAX_EVENTS;
			return 0;
		}
	}
	return -1;
}

static void deferred_restart(struct work_struct *dummy) {
	if (backup_traffic_data() == 0) {
		printk(KERN_ERR "Backup Success.\n");
	} else {
		printk(KERN_ERR "Some backup Failed.\n");
	}
	kernel_restart(NULL);
}
static DECLARE_WORK(restart_work, deferred_restart);

static void
btnu_tick(unsigned long arg)
{
	struct btnucom *sc = (struct btnucom *)arg;
	int is_current, detect;

	is_current = sc->io_read(sc->parent) ? 1 : 0;
	detect = sc->sc_detect;
	sc->sc_count++;
	if (is_current != sc->sc_current) {
		sc->sc_count = 0;
		sc->sc_current = is_current;
	} else if (sc->sc_count >= sc->sc_threshold) {
		detect = is_current;
		sc->sc_count = sc->sc_threshold;
	}

	if (detect != sc->sc_detect) {
		if (!detect) {
			if (sc->sc_detect_count < sc->sc_timeout) {
				btnu_enqueue_event(sc, BTNU_EVENT_PUSH,
						   sc->sc_detect_count);
			} else {
				btnu_enqueue_event(sc, BTNU_EVENT_OFF, 0);
			}
		} else {
			btnu_enqueue_event(sc, BTNU_EVENT_ON, 0);
		}
		sc->sc_detect_count = 0;
	} else {
		sc->sc_detect_count++;
	}
	if ((sc->sc_longtime > 0) && detect &&
	    (sc->sc_detect_count == sc->sc_longtime)) {
		btnu_enqueue_event(sc, BTNU_EVENT_LONGPRESS,
				   sc->sc_detect_count);
	}
	if (detect && sc->sc_detect_count == sc->sc_timeout) {
#ifdef BTNU_TIMEOUT_REBOOT
		schedule_work(&restart_work);
#endif
		btnu_enqueue_event(sc, BTNU_EVENT_TIMEOUT,
				   sc->sc_detect_count);
		sc->sc_detect_count = sc->sc_timeout + 1;
	}
	sc->sc_detect = detect;

	mod_timer(&sc->tick_callout, jiffies + HZ / (1000 / BTNU_POLLTIME));
}

static void
btnu_remove_list(struct btnucom *sc, struct task_struct *p)
{
	struct btnu_event_info *info;
	int match;

	do {
		match = 0;
		LIST_FOREACH(info, &sc->event_list, event_links) {
			if ((p == NULL) || (info->p == p)) {
				LIST_REMOVE(info, event_links);
				kfree(info);
				match = 1;
				if (p != NULL) {
					return;
				} else {
					break;
				}
			}
		}
	} while (match);
}

static ssize_t
btnucomopen(struct inode *inode, struct file *filp)
{
	struct btnucom *sc;
	struct btnu_event_info *info;

	sc = container_of(inode->i_cdev, struct btnucom, cdev);

	if (sc == NULL) {
		return -ENXIO;
	}

	if (sc->open_count > 0) {
		return -EBUSY;
	}
	LIST_FOREACH(info, &sc->event_list, event_links) {
		if (info->p == current) {
			return -EBUSY;
		}
	}
	info = kmalloc(sizeof(*info), GFP_KERNEL);
	if (info == NULL) {
		return -ENOMEM;
	}
	memset(info, 0, sizeof(*info));
	info->p = current;
	init_waitqueue_head(&info->sc_rsel);
	spin_lock_bh(&btnucom_irq_lock);
	LIST_INSERT_HEAD(&sc->event_list,
			 (struct btnu_event_info *) info, event_links);
	sc->open_count++;
	filp->private_data = sc;
	spin_unlock_bh(&btnucom_irq_lock);

	return 0;
}

static ssize_t
btnucomread(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	char strbuf[256];
	int len;
	struct btnucom *sc = filp->private_data;

	if (*f_pos != 0)
		return 0;

	snprintf(strbuf, sizeof(strbuf),
		 "current: %s detect: %s\n",
		 sc->sc_current ? "on" : "off",
		 sc->sc_detect ? "on" : "off");

	len = strlen(strbuf);

	if (copy_to_user(buf, strbuf, len))
		return -EFAULT;

	*f_pos += len;

	return len;
}

static int
btnucomrelease(struct inode *inode, struct file *filp)
{
	struct btnucom *sc = filp->private_data;

	if (sc == NULL) {
		return -ENXIO;
	}

	if (sc->open_count > 0) {
		sc->open_count--;
	}

	btnu_remove_list(sc, current);
	return 0;
}

static long
btnucomioctl(struct file *filp, unsigned int cmd, unsigned long data)
{
	struct btnucom *sc = filp->private_data;

	switch (cmd) {
	case BTNU_IOC_READ: {
		struct btnu_read_info *info =
			(struct btnu_read_info *) data;
		info->on = sc->sc_detect;
		break;
	}
	case BTNU_IOC_EVENT: {
		struct btnu_read_event_info *info =
			(struct btnu_read_event_info *) data;
		spin_lock_bh(&btnucom_irq_lock);
		btnu_dequeue_event(sc, info, current);
		spin_unlock_bh(&btnucom_irq_lock);
		break;
	}
	case BTNU_IOC_READ_EXT: {
		struct btnu_read_info_ext *info_ext =
			(struct btnu_read_info_ext *) data;
		info_ext->on = sc->sc_detect;
		info_ext->detect_time_ms = sc->sc_detect_count * BTNU_POLLTIME;
		break;
	}
	default:
		printk(KERN_ERR "cmd: 0x%x\n", cmd);
		break;
	}
	return 0;
}

static unsigned int
btnucompoll(struct file *filp, poll_table *wait)
{
	struct btnucom *sc = filp->private_data;
	struct btnu_event_info *info;
	int match = 0;

	LIST_FOREACH(info, &sc->event_list, event_links) {
		if (info->p == current) {
			match = 1;
			break;
		}
	}
	if (!match)
		return 0;

	poll_wait(filp, &info->sc_rsel, wait);

	if (btnu_exist_event(sc, current))
		return POLLIN | POLLRDNORM;

	return 0;
}

static int __init
btnucom_init(void)
{
	btnucom_class = class_create(THIS_MODULE, "btnucom");
	if (IS_ERR(btnucom_class))
		return PTR_ERR(btnucom_class);

	return 0;
}

static void __exit
btnucom_exit(void)
{
	class_destroy(btnucom_class);
}

module_init(btnucom_init);
module_exit(btnucom_exit);

MODULE_DESCRIPTION("NECPF button common interface");
MODULE_LICENSE("GPL v2");
