/* Copyright (c) 2011, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013-2014, The Linux Foundation. All rights reserved.
 * Copyright (c) 2014-2015, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#define pr_fmt(fmt) "%s: " fmt, __func__

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/platform_device.h>
#include <linux/leds.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#include <linux/module.h>
#include <linux/necpf_panel.h>

#ifdef CONFIG_HAS_EARLYSUSPEND
#include <linux/earlysuspend.h>

/* Early-suspend level */
#define LED_SUSPEND_LEVEL 1
#endif

#define PDM_DEBUG (0)
#if PDM_DEBUG
#define PDM_DPRINTK(fmt, ...) printk(fmt, ## __VA_ARGS__)
#else
#define PDM_DPRINTK(fmt, ...)
#endif /* PDM_DEBUG */
#include <linux/gpio.h>

#define SUPPORT_PDM_ACCESS (1)

/* Enable control */
#define LED_EN_GPIO (68)
static int led_en_gpio = LED_EN_GPIO;
static bool led_enabled = false;

#if 0 /* BUG#38264 */
/* PDM0 (Clock) Control
 *     PDM_AHB_CBCR: 0x01844004 (GCC BASE: 0x01800000, PDM_AHB_CBCR: 0x44004)
 *     PDM_XO4_CBCR: 0x01844008 (GCC BASE: 0x01800000, PDM_XO8_CBCR: 0x44008)
 */
#define PDM_CLK_BASE        (0x01844000)
#define PDM_CLK_SIZE        (0xF)
#define PDM_AHB_CBCR_OFFSET (0x4)
#define PDM_XO4_CBCR_OFFSET (0x8)
static void __iomem *pdm_gcc_base = NULL;
#endif /* BUG#38264 */

/* PDM0 (Brightness) Control */
#define PDM_DUTY_MAXVAL BIT(16)
#define PDM_DUTY_REFVAL BIT(15)

#define PDM_MIN_BRIGHTNESS (40)
#define PDM_MAX_BRIGHTNESS (172)
#define PDM_CLAMP(val, min, max) \
do { \
	if (val < min) val = min; \
	if (val > max) val = max; \
} while (0)

struct pdm_led_data {
	struct led_classdev cdev;
	void __iomem *perph_base;
	int pdm_offset;
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend early_suspend;
#endif
};

static struct mutex pdm_mutex = __MUTEX_INITIALIZER(pdm_mutex);

static int msm_led_gpio_varidation(void)
{
	mutex_lock(&pdm_mutex);
	if (!gpio_is_valid(led_en_gpio)) {
		mutex_unlock(&pdm_mutex);
		pr_err("%s: enable gpio not specified\n", __func__);
		return -EINVAL;
	}
	mutex_unlock(&pdm_mutex);
	return 0;
}

#if SUPPORT_PDM_ACCESS
static void msm_pdm_led_set_pdm_id(struct platform_device *pdev,
				struct pdm_led_data *led)
{
	u32 tcxo_pdm_ctl;

	mutex_lock(&pdm_mutex);
	msm_pdm_led_clk_prepare_enable();

	/* program tcxo_pdm_ctl register to enable specific pdm */
	tcxo_pdm_ctl = readl_relaxed(led->perph_base);
	tcxo_pdm_ctl |= (1 << pdev->id);
	writel_relaxed(tcxo_pdm_ctl, led->perph_base);

	if (!led_enabled)
		msm_pdm_led_clk_disable_unprepare();

	mutex_unlock(&pdm_mutex);
}
#endif /* SUPPORT_PDM_ACCESS */

#if 0 /* BUG#38264 */
static void msm_pdm_led_clk_enable(int enable)
{
	u32 ahb, xo4;

	ahb = readl_relaxed(pdm_gcc_base + PDM_AHB_CBCR_OFFSET);
	xo4 = readl_relaxed(pdm_gcc_base + PDM_XO4_CBCR_OFFSET);

	if (enable) {
		ahb |= BIT(0);
		xo4 |= BIT(0);
	} else {
		ahb &= ~(BIT(0));
		xo4 &= ~(BIT(0));
	}

	writel_relaxed(ahb, (pdm_gcc_base + PDM_AHB_CBCR_OFFSET));
	writel_relaxed(xo4, (pdm_gcc_base + PDM_XO4_CBCR_OFFSET));
}

static void msm_pdm_led_clk_prepare_enable(void)
{
	msm_pdm_led_clk_enable(1);
	PDM_DPRINTK("%s: done\n", __func__);
}

static void msm_pdm_led_clk_disable_unprepare(void)
{
	msm_pdm_led_clk_enable(0);
	PDM_DPRINTK("%s: done\n", __func__);
}
#endif /* BUG#38264 */

int msm_pdm_led_on(void)
{
	mutex_lock(&pdm_mutex);
	if (led_enabled) {
		mutex_unlock(&pdm_mutex);
		return 0;
	}

	msm_pdm_led_clk_prepare_enable();

	gpio_set_value(led_en_gpio, 1);
	msleep(10);

	PDM_DPRINTK("%s: LED %s\n", __func__,
		(1 == gpio_get_value(led_en_gpio)) ? "turn on": "off");

	led_enabled = true;
	mutex_unlock(&pdm_mutex);
	return 0;
}
EXPORT_SYMBOL(msm_pdm_led_on);

void msm_pdm_led_off(void)
{
	mutex_lock(&pdm_mutex);
	if (!led_enabled) {
		mutex_unlock(&pdm_mutex);
		return;
	}

	gpio_set_value(led_en_gpio, 0);
	msleep(10);

	PDM_DPRINTK("%s: LED %s\n", __func__,
		(0 == gpio_get_value(led_en_gpio)) ? "turn off": "on");

	msm_pdm_led_clk_disable_unprepare();

	led_enabled = false;
	mutex_unlock(&pdm_mutex);
	return;
}
EXPORT_SYMBOL(msm_pdm_led_off);

bool msm_pdm_get_led_status(void)
{
	bool enabled;

	mutex_lock(&pdm_mutex);
	enabled = led_enabled;
	mutex_unlock(&pdm_mutex);

	PDM_DPRINTK("%s: LED status: %s\n", __func__, enabled ? "ON": "off");
	return enabled;
}
EXPORT_SYMBOL(msm_pdm_get_led_status);

static void msm_led_brightness_set_percent(struct pdm_led_data *led,
						int duty_per)
{
	u16 duty_val;

#if PDM_DEBUG
#if SUPPORT_PDM_ACCESS
	u32 reg_val;
#endif /* SUPPORT_PDM_ACCESS */
#endif /* PDM_DEBUG */

	duty_val = PDM_DUTY_REFVAL - ((PDM_DUTY_MAXVAL * duty_per) / 100);
	duty_val &= 0xFFFF;

	if (!duty_per)
		duty_val--;

	/* Update lcd brightness */
	PDM_DPRINTK("%s: duty_val=0x%02x ==> 0x%08x\n", __func__, duty_val,
			(u32)(led->perph_base + led->pdm_offset));

	mutex_lock(&pdm_mutex);
	msm_pdm_led_clk_prepare_enable();

#if SUPPORT_PDM_ACCESS
	writel_relaxed(duty_val, led->perph_base + led->pdm_offset);
#else
	PDM_DPRINTK("%s: But diasbling brightness controls.\n", __func__);
#endif /* SUPPORT_PDM_ACCESS */


#if PDM_DEBUG
	/* For debug */
#if SUPPORT_PDM_ACCESS
	reg_val = readl_relaxed(led->perph_base + led->pdm_offset);
	pr_err("%s: readl_relaxed(0x%08X) => 0x%08X\n", __func__,
		(unsigned int)(led->perph_base + led->pdm_offset), (unsigned int)reg_val);
	usleep(10 * 1000);
#endif /* SUPPORT_PDM_ACCESS */
#endif /* PDM_DEBUG */

	if (!led_enabled)
		msm_pdm_led_clk_disable_unprepare();

	mutex_unlock(&pdm_mutex);
}

static void msm_led_brightness_set(struct led_classdev *led_cdev,
				enum led_brightness value)
{
	int percent;
	u32 brightness = value;

	struct pdm_led_data *led =
		container_of(led_cdev, struct pdm_led_data, cdev);

	PDM_CLAMP(brightness, PDM_MIN_BRIGHTNESS, PDM_MAX_BRIGHTNESS);

	percent = (brightness * 100) / LED_FULL;
	PDM_DPRINTK("%s; brightness=%d/percent=%d\n",
			__func__, brightness, percent);

	msm_led_brightness_set_percent(led, percent);
}

#ifdef CONFIG_PM_SLEEP
static int msm_led_pdm_suspend(struct device *dev)
{
	msm_pdm_led_off();

	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void msm_led_pdm_early_suspend(struct early_suspend *h)
{
	struct pdm_led_data *led = container_of(h,
			struct pdm_led_data, early_suspend);

	msm_led_pdm_suspend(led->cdev.dev->parent);
}

#endif

static const struct dev_pm_ops msm_led_pdm_pm_ops = {
#ifndef CONFIG_HAS_EARLYSUSPEND
	.suspend	= msm_led_pdm_suspend,
#endif
};
#endif

static int msm_pdm_led_probe(struct platform_device *pdev)
{
	const struct led_info *pdata = pdev->dev.platform_data;
	struct pdm_led_data *led;
	struct resource *res, *ioregion;
	int rc;
	int initial_percent;

	if (!pdata) {
		pr_err("platform data is invalid\n");
		return -EINVAL;
	}

	PDM_DPRINTK("%s: id = %d\n", __func__, pdev->id);
	if (pdev->id > 2) {
		pr_err("pdm id is invalid\n");
		return -EINVAL;
	}

	led = kzalloc(sizeof(struct pdm_led_data), GFP_KERNEL);
	if (!led)
		return -ENOMEM;

	/* Enable runtime PM ops, start in ACTIVE mode */
	rc = pm_runtime_set_active(&pdev->dev);
	if (rc < 0)
		dev_dbg(&pdev->dev, "unable to set runtime pm state\n");
	pm_runtime_enable(&pdev->dev);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		pr_err("get resource failed\n");
		rc = -EINVAL;
		goto err_get_res;
	}

	ioregion = request_mem_region(res->start, resource_size(res),
						pdev->name);
	if (!ioregion) {
		pr_err("request for mem region failed\n");
		rc = -ENOMEM;
		goto err_get_res;
	}

	led->perph_base = ioremap(res->start, resource_size(res));
	if (!led->perph_base) {
		pr_err("ioremap failed\n");
		rc = -ENOMEM;
		goto err_ioremap;
	}

#if 0 /* BUG#38264 */
	pdm_gcc_base = ioremap(PDM_CLK_BASE, PDM_CLK_SIZE);
	if (!pdm_gcc_base) {
		pr_err("ioremap failed\n");
		rc = -ENOMEM;
		goto err_gcc_ioremap;
	}
	PDM_DPRINTK("%s: pdm gcc p=0x%08x(v=0x%08x)\n",
		__func__, PDM_CLK_BASE, (u32)pdm_gcc_base);
#endif /* BUG#38264 */

	rc = msm_led_gpio_varidation();
	if (rc < 0)
		goto err_led_reg;

	/* Pulse Density Modulation(PDM) ids start with 0 and
	 * every PDM0 register takes 8 bytes
	 */
	led->pdm_offset = ((pdev->id) + 1) * 4;
	PDM_DPRINTK("%s: pdm base p=0x%08x(v=0x%08x)/pdm0 offset=0x%08x\n",
		__func__, (u32)res->start, (u32)led->perph_base,
		(u32)led->pdm_offset);

#if SUPPORT_PDM_ACCESS
	/* program tcxo_pdm_ctl register to enable specific pdm */
	msm_pdm_led_set_pdm_id(pdev, led);

	/* Start with LED in off state */
	initial_percent = (PDM_MIN_BRIGHTNESS * 100) / LED_FULL;
	msm_led_brightness_set_percent(led, initial_percent);
#else
	/* Dummy set */
	initial_percent = 0;
	PDM_DPRINTK("%s: Brightness control is not supported!\n", __func__);
#endif /* SUPPORT_PDM_ACCESS */

	/* set cdev parameter */
	led->cdev.brightness_set = msm_led_brightness_set;
	led->cdev.name = pdata->name ? : "leds-msm-pdm";

	rc = led_classdev_register(&pdev->dev, &led->cdev);
	if (rc) {
		pr_err("led class registration failed\n");
		goto err_led_reg;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
	led->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN +
						LED_SUSPEND_LEVEL;
	led->early_suspend.suspend = msm_led_pdm_early_suspend;
	register_early_suspend(&led->early_suspend);
#endif

	platform_set_drvdata(pdev, led);
	PDM_DPRINTK("%s: done\n", __func__);
	return 0;

err_led_reg:
#if 0  /* BUG#38264 */
	iounmap(pdm_gcc_base);
err_gcc_ioremap:
#endif /* BUG#38264 */
	iounmap(led->perph_base);
err_ioremap:
	release_mem_region(res->start, resource_size(res));
err_get_res:
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	kfree(led);
	return rc;
}

static int msm_pdm_led_remove(struct platform_device *pdev)
{
	struct pdm_led_data *led = platform_get_drvdata(pdev);
	struct resource *res = platform_get_resource(pdev, IORESOURCE_MEM, 0);

#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&led->early_suspend);
#endif
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);

	led_classdev_unregister(&led->cdev);
	msm_led_brightness_set_percent(led, 0);
	msm_pdm_led_clk_disable_unprepare();
#if 0 /* BUG#38264 */
	iounmap(pdm_gcc_base);
#endif /* BUG#38264 */
	iounmap(led->perph_base);
	release_mem_region(res->start, resource_size(res));
	kfree(led);

	return 0;
}

static struct platform_driver msm_pdm_led_driver = {
	.probe		= msm_pdm_led_probe,
	.remove		= msm_pdm_led_remove,
	.driver		= {
		.name	= "leds-msm-pdm",
		.owner	= THIS_MODULE,
#ifdef CONFIG_PM_SLEEP
		.pm	= &msm_led_pdm_pm_ops,
#endif
	},
};

static int __init msm_pdm_led_init(void)
{
	return platform_driver_register(&msm_pdm_led_driver);
}
module_init(msm_pdm_led_init);

static void __exit msm_pdm_led_exit(void)
{
	platform_driver_unregister(&msm_pdm_led_driver);
}
module_exit(msm_pdm_led_exit);

MODULE_DESCRIPTION("MSM PDM LEDs driver");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:leds-msm-pdm");
