/*
 * USB Network driver infrastructure
 * Copyright (C) 2000-2005 by David Brownell
 * Copyright (C) 2003-2005 David Hollis <dhollis@davehollis.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This is a generic "USB networking" framework that works with several
 * kinds of full and high speed networking devices:  host-to-host cables,
 * smart usb peripherals, and actual Ethernet adapters.
 *
 * These devices usually differ in terms of control protocols (if they
 * even have one!) and sometimes they define new framing to wrap or batch
 * Ethernet packets.  Otherwise, they talk to USB pretty much the same,
 * so interface (un)binding, endpoint I/O queues, fault handling, and other
 * issues can usefully be addressed by this framework.
 */

// #define	DEBUG			// error path messages, extra info
// #define	VERBOSE			// more; success messages

#include <linux/module.h>
#include <linux/init.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ctype.h>
#include <linux/ethtool.h>
#include <linux/workqueue.h>
#include <linux/mii.h>
#include <linux/usb.h>
#include <linux/usb/usbnet.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/pm_runtime.h>
#include <linux/debugfs.h>

#ifdef CONFIG_USBNET_IPA
#include <linux/ip.h>
#include <linux/ipv6.h>
#include <linux/ipa.h>
#endif /* CONFIG_USBNET_IPA */

#include <linux/types.h>
#define DRIVER_VERSION		"22-Aug-2005"


/*-------------------------------------------------------------------------*/

/*
 * Nineteen USB 1.1 max size bulk transactions per frame (ms), max.
 * Several dozen bytes of IPv4 data can fit in two such transactions.
 * One maximum size Ethernet packet takes twenty four of them.
 * For high speed, each frame comfortably fits almost 36 max size
 * Ethernet packets (so queues should be bigger).
 *
 * The goal is to let the USB host controller be busy for 5msec or
 * more before an irq is required, under load.  Jumbograms change
 * the equation.
 */
#define	MAX_QUEUE_MEMORY	(600 * 1518)
#define	RX_QLEN(dev)		((dev)->rx_qlen)
#define	TX_QLEN(dev)		((dev)->tx_qlen)

// reawaken network queue this soon after stopping; else watchdog barks
#define TX_TIMEOUT_JIFFIES	(5*HZ)

// throttle rx/tx briefly after some faults, so khubd might disconnect()
// us (it polls at HZ/4 usually) before we report too many false errors.
#define THROTTLE_JIFFIES	(HZ/8)

// between wakeups
#define UNLINK_TIMEOUT_MS	3

#ifdef CONFIG_USBNET_IPA
#define IPA_USBNET_MAX_PKTS_DESC 160
#define IPA_USBNET_DNE_PKTS_DESC IPA_USBNET_MAX_PKTS_DESC * 2
#define IPA_USBNET_MIN_PKTS_DESC 5
#endif /* CONFIG_USBNET_IPA */
/* timeout value for odu bridge resources */
#define IPA_ODU_RM_TIMEOUT_MSEC 10000

/*-------------------------------------------------------------------------*/

// randomly generated ethernet address
static u8	node_id [ETH_ALEN];

static const char driver_name [] = "usbnet";

struct workqueue_struct	*usbnet_wq;

static DECLARE_WAIT_QUEUE_HEAD(unlink_wakeup);

/* use ethtool to change the level for any given device */
static int msg_level = -1;
module_param (msg_level, int, 0);
MODULE_PARM_DESC (msg_level, "Override default message level");

#ifdef CONFIG_USBNET_IPA
static struct usbnet_ipa_ctx *usbnet_ipa;
unsigned int enable_ipa_bridge = 0;

static int usbnet_debugfs_init(struct usbnet *);
void usbnet_debugfs_exit(struct usbnet *);

static int usbnet_ipa_init(struct usbnet *);
static int usbnet_ipa_cleanup(struct usbnet_ipa_ctx *);

#else /* CONFIG_USBNET_IPA */
#ifdef CONFIG_USBNET_IPA_BRIDGE
	unsigned int enable_ipa_bridge = 1;
#else
	unsigned int enable_ipa_bridge = 0;
#endif

#endif /* CONFIG_USBNET_IPA */

#if defined(CONFIG_PLATFORM_NEC)
#ifdef IPA_DEBUG
#define usbnet_ipa_debug(fmt, ...) \
	printk(KERN_DEBUG fmt, ## __VA_ARGS__)
#else
#define usbnet_ipa_debug(fmt, ...)
#endif /* IPA_DEBUG */

#define usbnet_ipa_info(fmt, ...) \
	printk(KERN_INFO fmt, ## __VA_ARGS__)

#else /* defined(CONFIG_PLATFORM_NEC) */
#define usbnet_ipa_debug(fmt, ...)
#define usbnet_ipa_info(fmt, ...)

#endif /* defined(CONFIG_PLATFORM_NEC) */

/*-------------------------------------------------------------------------*/

/* handles CDC Ethernet and many other network "bulk data" interfaces */
int usbnet_get_endpoints(struct usbnet *dev, struct usb_interface *intf)
{
	int				tmp;
	struct usb_host_interface	*alt = NULL;
	struct usb_host_endpoint	*in = NULL, *out = NULL;
	struct usb_host_endpoint	*status = NULL;

	for (tmp = 0; tmp < intf->num_altsetting; tmp++) {
		unsigned	ep;

		in = out = status = NULL;
		alt = intf->altsetting + tmp;

		/* take the first altsetting with in-bulk + out-bulk;
		 * remember any status endpoint, just in case;
		 * ignore other endpoints and altsettings.
		 */
		for (ep = 0; ep < alt->desc.bNumEndpoints; ep++) {
			struct usb_host_endpoint	*e;
			int				intr = 0;

			e = alt->endpoint + ep;
			switch (e->desc.bmAttributes) {
			case USB_ENDPOINT_XFER_INT:
				if (!usb_endpoint_dir_in(&e->desc))
					continue;
				intr = 1;
				/* FALLTHROUGH */
			case USB_ENDPOINT_XFER_BULK:
				break;
			default:
				continue;
			}
			if (usb_endpoint_dir_in(&e->desc)) {
				if (!intr && !in)
					in = e;
				else if (intr && !status)
					status = e;
			} else {
				if (!out)
					out = e;
			}
		}
		if (in && out)
			break;
	}
	if (!alt || !in || !out)
		return -EINVAL;

	if (alt->desc.bAlternateSetting != 0 ||
	    !(dev->driver_info->flags & FLAG_NO_SETINT)) {
		tmp = usb_set_interface (dev->udev, alt->desc.bInterfaceNumber,
				alt->desc.bAlternateSetting);
		if (tmp < 0)
			return tmp;
	}

	dev->in = usb_rcvbulkpipe (dev->udev,
			in->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
	dev->out = usb_sndbulkpipe (dev->udev,
			out->desc.bEndpointAddress & USB_ENDPOINT_NUMBER_MASK);
	dev->status = status;
	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_get_endpoints);

int usbnet_get_ethernet_addr(struct usbnet *dev, int iMACAddress)
{
	int 		tmp, i;
	unsigned char	buf [13];

	tmp = usb_string(dev->udev, iMACAddress, buf, sizeof buf);
	if (tmp != 12) {
		dev_dbg(&dev->udev->dev,
			"bad MAC string %d fetch, %d\n", iMACAddress, tmp);
		if (tmp >= 0)
			tmp = -EINVAL;
		return tmp;
	}
	for (i = tmp = 0; i < 6; i++, tmp += 2)
		dev->net->dev_addr [i] =
			(hex_to_bin(buf[tmp]) << 4) + hex_to_bin(buf[tmp + 1]);
	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_get_ethernet_addr);

static void intr_complete (struct urb *urb)
{
	struct usbnet	*dev = urb->context;
	int		status = urb->status;

	switch (status) {
	/* success */
	case 0:
		dev->driver_info->status(dev, urb);
		break;

	/* software-driven interface shutdown */
	case -ENOENT:		/* urb killed */
	case -ESHUTDOWN:	/* hardware gone */
		netif_dbg(dev, ifdown, dev->net,
			  "intr shutdown, code %d\n", status);
		return;

	/* NOTE:  not throttling like RX/TX, since this endpoint
	 * already polls infrequently
	 */
	default:
		netdev_dbg(dev->net, "intr status %d\n", status);
		break;
	}

	status = usb_submit_urb (urb, GFP_ATOMIC);
	if (status != 0)
		netif_err(dev, timer, dev->net,
			  "intr resubmit --> %d\n", status);
}

static int init_status (struct usbnet *dev, struct usb_interface *intf)
{
	char		*buf = NULL;
	unsigned	pipe = 0;
	unsigned	maxp;
	unsigned	period;

	if (!dev->driver_info->status)
		return 0;

	pipe = usb_rcvintpipe (dev->udev,
			dev->status->desc.bEndpointAddress
				& USB_ENDPOINT_NUMBER_MASK);
	maxp = usb_maxpacket (dev->udev, pipe, 0);

	/* avoid 1 msec chatter:  min 8 msec poll rate */
	period = max ((int) dev->status->desc.bInterval,
		(dev->udev->speed == USB_SPEED_HIGH) ? 7 : 3);

	buf = kmalloc (maxp, GFP_KERNEL);
	if (buf) {
		dev->interrupt = usb_alloc_urb (0, GFP_KERNEL);
		if (!dev->interrupt) {
			kfree (buf);
			return -ENOMEM;
		} else {
			usb_fill_int_urb(dev->interrupt, dev->udev, pipe,
				buf, maxp, intr_complete, dev, period);
			dev->interrupt->transfer_flags |= URB_FREE_BUFFER;
			dev_dbg(&intf->dev,
				"status ep%din, %d bytes period %d\n",
				usb_pipeendpoint(pipe), maxp, period);
		}
	}
	return 0;
}

/* Submit the interrupt URB if not previously submitted, increasing refcount */
int usbnet_status_start(struct usbnet *dev, gfp_t mem_flags)
{
	int ret = 0;

	WARN_ON_ONCE(dev->interrupt == NULL);
	if (dev->interrupt) {
		mutex_lock(&dev->interrupt_mutex);

		if (++dev->interrupt_count == 1)
			ret = usb_submit_urb(dev->interrupt, mem_flags);

		dev_dbg(&dev->udev->dev, "incremented interrupt URB count to %d\n",
			dev->interrupt_count);
		mutex_unlock(&dev->interrupt_mutex);
	}
	return ret;
}
EXPORT_SYMBOL_GPL(usbnet_status_start);

/* For resume; submit interrupt URB if previously submitted */
static int __usbnet_status_start_force(struct usbnet *dev, gfp_t mem_flags)
{
	int ret = 0;

	mutex_lock(&dev->interrupt_mutex);
	if (dev->interrupt_count) {
		ret = usb_submit_urb(dev->interrupt, mem_flags);
		dev_dbg(&dev->udev->dev,
			"submitted interrupt URB for resume\n");
	}
	mutex_unlock(&dev->interrupt_mutex);
	return ret;
}

/* Kill the interrupt URB if all submitters want it killed */
void usbnet_status_stop(struct usbnet *dev)
{
	if (dev->interrupt) {
		mutex_lock(&dev->interrupt_mutex);
		WARN_ON(dev->interrupt_count == 0);

		if (dev->interrupt_count && --dev->interrupt_count == 0)
			usb_kill_urb(dev->interrupt);

		dev_dbg(&dev->udev->dev,
			"decremented interrupt URB count to %d\n",
			dev->interrupt_count);
		mutex_unlock(&dev->interrupt_mutex);
	}
}
EXPORT_SYMBOL_GPL(usbnet_status_stop);

/* For suspend; always kill interrupt URB */
static void __usbnet_status_stop_force(struct usbnet *dev)
{
	if (dev->interrupt) {
		mutex_lock(&dev->interrupt_mutex);
		usb_kill_urb(dev->interrupt);
		dev_dbg(&dev->udev->dev, "killed interrupt URB for suspend\n");
		mutex_unlock(&dev->interrupt_mutex);
	}
}

#ifdef CONFIG_USBNET_IPA
#ifdef IPA_DEBUG
static void usbnet_dump_skb_data(struct sk_buff *skb)
{
	char tmp[80];
	char *p = skb->data;
	char *t = tmp;
	int i;

	printk(KERN_INFO "===== start =====\n");
	printk(KERN_INFO "skb->len %d (0x%x)\n", skb->len, skb->len);
	for (i = 0; i < skb->len; i++) {
		t += sprintf(t, "%02x ", *p++ & 0xff);
		if (((i + 1) & 0x0f) == 0) {
			printk(KERN_INFO " %s\n", tmp);
			t = tmp;
		}
	}
	if (i & 0x0f)
		printk(KERN_INFO " %s\n", tmp);
	printk(KERN_INFO "===== end =====\n");
}
#endif /* IPA_DEBUG */

static void usbnet_remove_padding(struct sk_buff *skb)
{
	uint16_t eth_type;
	eth_type = be16_to_cpu(skb->protocol);

	if (eth_type == ETH_P_IP) {
		struct iphdr *ip_hdr = NULL;
		ip_hdr = (struct iphdr *)(skb_mac_header(skb) + ETH_HLEN);
		skb_trim(skb, ntohs(ip_hdr->tot_len) + ETH_HLEN);
	} else if (eth_type == ETH_P_IPV6) {
		struct ipv6hdr *ip6_hdr = NULL;
		ip6_hdr = (struct ipv6hdr *)(skb_mac_header(skb) + ETH_HLEN);
		skb_trim(skb, ntohs(ip6_hdr->payload_len) + sizeof(struct ipv6hdr) + ETH_HLEN);
	}
}

static int usbnet_send_pkt_to_ipa(struct sk_buff *skb)
{
	struct ipa_tx_meta metadata;
	struct ethhdr *eth = NULL;
	struct ethhdr *pusheth = NULL;
	struct usbnet_ipa_desc_node *node = NULL;
	bool schedule_ipa_work = false;
	int ret;

	memset(&metadata, 0, sizeof(metadata));

	eth = eth_hdr(skb);

	pusheth = (struct ethhdr *) skb_push(skb, ETH_HLEN);
	memcpy(pusheth->h_dest, eth->h_dest, ETH_ALEN);
	memcpy(pusheth->h_source, eth->h_source, ETH_ALEN);
	memcpy(&pusheth->h_proto, &eth->h_proto, sizeof(__be16));

	usbnet_ipa_debug("%s:%d ipa_tx_dp called \n", __func__, __LINE__);

	spin_lock(&usbnet_ipa->flow_ctrl_lock);

	if (usbnet_ipa->freeq_cnt == 0) {
		dev_kfree_skb(skb);
		/* @@ TODO : increase drop count */
		if (usbnet_ipa->ipa_free_desc_cnt > 0)
		    schedule_ipa_work = true;
		goto end_to_tx;
	}

	/* BUG#36799 : Remove extra padding */
	usbnet_remove_padding(skb);

#ifdef IPA_DEBUG
	usbnet_dump_skb_data(skb);
#endif /* IPA_DEBUG */

	if ((usbnet_ipa->pendq_cnt == 0) && (usbnet_ipa->ipa_free_desc_cnt > 0)) {
		metadata.dma_address_valid = false;
		usbnet_ipa->stats.rx_ipa_send++;
		ret = ipa_tx_dp(IPA_CLIENT_USBNET_PROD, skb, &metadata);

		if (ret) {
			usbnet_ipa_debug("%s:%d failed to transmit packet to ipa. ", __func__, __LINE__);
			kfree(skb);
			/* @@ TODO : increase fail count */
			usbnet_ipa->stats.rx_ipa_send_fail++;
		} else {
			usbnet_ipa->ipa_free_desc_cnt--;
		}
	} else if (usbnet_ipa->pendq_cnt <= IPA_USBNET_DNE_PKTS_DESC) {
		node = list_first_entry(&usbnet_ipa->free_queue_head, struct usbnet_ipa_desc_node, link);
		list_del(&node->link);
		node->skb_ptr = skb;
		list_add_tail(&node->link, &usbnet_ipa->pend_queue_head);
		usbnet_ipa->freeq_cnt--;
		usbnet_ipa->pendq_cnt++;
		schedule_ipa_work = true;
	} else {
		usbnet_ipa->stats.flow_control_pkt_drop--;
	}

end_to_tx:
	spin_unlock(&usbnet_ipa->flow_ctrl_lock);
	if (schedule_ipa_work) {
		schedule_work(&usbnet_ipa->ipa_send_task);
	}

	return ret;
}
#endif /* CONFIG_USBNET_IPA */

/* Passes this packet up the stack, updating its accounting.
 * Some link protocols batch packets, so their rx_fixup paths
 * can return clones as well as just modify the original skb.
 */
void usbnet_skb_return (struct usbnet *dev, struct sk_buff *skb)
{
	int	status;
	struct sk_buff *pend_skb;
	struct usbnet_ipa_ctx	*punet_ipa = dev->pusbnet_ipa;
	struct	ipa_tx_meta ipa_meta = {0x0};
	u32	pkts_to_send, ret, qlen = 0;
	u8	protocol_type = (skb->data[ETH_HLEN] & 0xf0);
#ifdef CONFIG_USBNET_IPA
	uint16_t eth_type;
#endif /* CONFIG_USBNET_IPA */

	usbnet_ipa_debug("%s:%d skb=%p ->dev=%p ->len=%d enable_ipa_bridge=%d\n", __func__, __LINE__,skb, skb->dev,skb->len,enable_ipa_bridge);

	if (enable_ipa_bridge  &&
		(protocol_type == 0x40 || protocol_type == 0x60)) {
		spin_lock(&dev->ipa_pendq.lock);
		qlen = skb_queue_len(&dev->ipa_pendq);
		/* drop pkts */
		if (!dev->ipa_free_desc_cnt &&
		    qlen > (2 * USBNET_IPA_SYS_PIPE_DNE_PKTS)) {
			dev->pusbnet_ipa->stats.flow_control_pkt_drop++;
			pr_debug_ratelimited("drop pkt ipa pending qlen = %d\n",
					     qlen);
			dev_kfree_skb(skb);
			goto unlock_and_schedule;
		}

		__skb_queue_tail(&dev->ipa_pendq, skb);
		if (dev->ipa_free_desc_cnt) {
			/* send fixed number of packets to ODU bridge driver
			 * now. Number should be smaller than total desc count
			 * to keep both usbnet and odu driver busy and prevent
			 * pending packet counts to quickly reach to drop count
			 * limit when descs are available without waiting for
			 * work to get a chance to run.
			 */
			pkts_to_send = dev->ipa_free_desc_cnt;
			pkts_to_send = (pkts_to_send > 50) ? 50 : pkts_to_send;
			while (pkts_to_send &&
			       (pend_skb = __skb_dequeue(&dev->ipa_pendq))) {
				ipa_meta.dma_address_valid = false;
				/* Send Packet to ODU bridge Driver */
				spin_unlock(&dev->ipa_pendq.lock);
				ret = odu_bridge_tx_dp(pend_skb, &ipa_meta);
				spin_lock(&dev->ipa_pendq.lock);
				if (ret) {
					pr_err_ratelimited("%s: ret %d\n",
							   __func__, ret);
					dev_kfree_skb(pend_skb);
					punet_ipa->stats.rx_ipa_send_fail++;
					goto unlock_and_schedule;
				} else {
					dev->pusbnet_ipa->stats.rx_ipa_send++;
					dev->ipa_free_desc_cnt--;
				}
				pkts_to_send--;
			}
		}
unlock_and_schedule:
		spin_unlock(&dev->ipa_pendq.lock);
		if (dev->ipa_pendq.qlen)
			queue_work(usbnet_wq, &dev->ipa_send_task);
		return;
	}

	if (test_bit(EVENT_RX_PAUSED, &dev->flags)) {
		skb_queue_tail(&dev->rxq_pause, skb);
		return;
	}

	if (!skb->protocol)
		skb->protocol = eth_type_trans(skb, dev->net);

	usbnet_ipa_debug("%s:%d eth_type 0x%x (ETH_P_IP : 0x%x, ETH_P_IPV6 : 0x%x)\n", __func__, __LINE__, be16_to_cpu(skb->protocol), ETH_P_IP, ETH_P_IPV6);

	dev->net->stats.rx_packets++;
	dev->net->stats.rx_bytes += skb->len;

	netif_dbg(dev, rx_status, dev->net, "< rx, len %zu, type 0x%x\n",
		  skb->len + sizeof (struct ethhdr), skb->protocol);
	memset (skb->cb, 0, sizeof (struct skb_data));

	if (skb_defer_rx_timestamp(skb))
		return;

#ifdef CONFIG_USBNET_IPA
	/* neither ip or ipv6 not transmit to ipa */
	eth_type = be16_to_cpu(skb->protocol);
	/* @@ TODO : fragment packetłȂƂɉǂ */
	if (eth_type == ETH_P_IP || eth_type == ETH_P_IPV6) {
		status = usbnet_send_pkt_to_ipa(skb);
	} else {
		status = netif_rx_ni(skb);
		if (status != NET_RX_SUCCESS)
			netif_dbg(dev, rx_err, dev->net,
				  "netif_rx status %d\n", status);
	}
#else /* CONFIG_USBNET_IPA */
	status = netif_rx_ni(skb);
	if (status != NET_RX_SUCCESS)
		netif_dbg(dev, rx_err, dev->net,
			  "netif_rx status %d\n", status);
#endif /* CONFIG_USBNET_IPA */
}
EXPORT_SYMBOL_GPL(usbnet_skb_return);

/* must be called if hard_mtu or rx_urb_size changed */
void usbnet_update_max_qlen(struct usbnet *dev)
{
	enum usb_device_speed speed = dev->udev->speed;

	switch (speed) {
	case USB_SPEED_HIGH:
		dev->rx_qlen = MAX_QUEUE_MEMORY / dev->rx_urb_size;
		dev->tx_qlen = MAX_QUEUE_MEMORY / dev->hard_mtu;
		break;
	case USB_SPEED_SUPER:
		/* Not take default 5ms qlen for super speed HC to
		 * save memory, and iperf tests show 2.5ms qlen can
		 * work well
		 */
		dev->rx_qlen = 5 * MAX_QUEUE_MEMORY / dev->rx_urb_size;
		dev->tx_qlen = 5 * MAX_QUEUE_MEMORY / dev->hard_mtu;
		break;
	default:
		dev->rx_qlen = dev->tx_qlen = 4;
	}
}
EXPORT_SYMBOL_GPL(usbnet_update_max_qlen);

/*-------------------------------------------------------------------------
 *
 * Network Device Driver (peer link to "Host Device", from USB host)
 *
 *-------------------------------------------------------------------------*/

int usbnet_change_mtu (struct net_device *net, int new_mtu)
{
	struct usbnet	*dev = netdev_priv(net);
	int		ll_mtu = new_mtu + net->hard_header_len;
	int		old_hard_mtu = dev->hard_mtu;
	int		old_rx_urb_size = dev->rx_urb_size;

	if (new_mtu <= 0)
		return -EINVAL;
	// no second zero-length packet read wanted after mtu-sized packets
	if ((ll_mtu % dev->maxpacket) == 0)
		return -EDOM;
	net->mtu = new_mtu;

	dev->hard_mtu = net->mtu + net->hard_header_len;
	if (dev->rx_urb_size == old_hard_mtu) {
		dev->rx_urb_size = dev->hard_mtu;
		if (dev->rx_urb_size > old_rx_urb_size)
			usbnet_unlink_rx_urbs(dev);
	}

	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_change_mtu);

#if defined(CONFIG_PLATFORM_NEC)
int usbnet_eth_mac_addr(struct net_device *dev, void *p)
{
	int ret;
	struct usbnet *usbnet_dev;
	ret = eth_mac_addr(dev,p);
	if (ret < 0)
		return ret;

#ifdef CONFIG_USBNET_IPA
	usbnet_dev = usbnet_ipa->dev;
	usbnet_ipa_cleanup(usbnet_ipa);

	usbnet_ipa_init(usbnet_dev);
#endif /* CONFIG_USBNET_IPA */
	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_eth_mac_addr);
#endif /* defined(CONFIG_PLATFORM_NEC) */

/* The caller must hold list->lock */
static void __usbnet_queue_skb(struct sk_buff_head *list,
			struct sk_buff *newsk, enum skb_state state)
{
	struct skb_data *entry = (struct skb_data *) newsk->cb;

	__skb_queue_tail(list, newsk);
	entry->state = state;
}

/*-------------------------------------------------------------------------*/

/* some LK 2.4 HCDs oopsed if we freed or resubmitted urbs from
 * completion callbacks.  2.5 should have fixed those bugs...
 */

static enum skb_state defer_bh(struct usbnet *dev, struct sk_buff *skb,
		struct sk_buff_head *list, enum skb_state state)
{
	unsigned long		flags;
	enum skb_state 		old_state;
	struct skb_data *entry = (struct skb_data *) skb->cb;

	spin_lock_irqsave(&list->lock, flags);
	old_state = entry->state;
	entry->state = state;
	__skb_unlink(skb, list);
	spin_unlock(&list->lock);
	spin_lock(&dev->done.lock);
	__skb_queue_tail(&dev->done, skb);
	if (dev->done.qlen == 1)
		queue_work(usbnet_wq, &dev->bh_w);
	spin_unlock_irqrestore(&dev->done.lock, flags);
	return old_state;
}

/* some work can't be done in tasklets, so we use keventd
 *
 * NOTE:  annoying asymmetry:  if it's active, schedule_work() fails,
 * but tasklet_schedule() doesn't.  hope the failure is rare.
 */
void usbnet_defer_kevent (struct usbnet *dev, int work)
{
	set_bit (work, &dev->flags);
	if (!schedule_work (&dev->kevent)) {
		if (net_ratelimit())
			netdev_err(dev->net, "kevent %d may have been dropped\n", work);
	} else {
		netdev_dbg(dev->net, "kevent %d scheduled\n", work);
	}
}
EXPORT_SYMBOL_GPL(usbnet_defer_kevent);

/*-------------------------------------------------------------------------*/

static void rx_complete(struct urb *urb);

#if defined(CONFIG_PLATFORM_NEC)
extern void msm_watchdog_clear(void);	/* WAR BUG#29394 */
#endif /* defined(CONFIG_PLATFORM_NEC) */

static int rx_submit (struct usbnet *dev, struct urb *urb, gfp_t flags)
{
	struct sk_buff		*skb;
	struct skb_data		*entry;
	int			retval = 0;
	unsigned long		lockflags;
	size_t			size = dev->rx_urb_size;

	/* prevent rx skb allocation when error ratio is high */
	if (test_bit(EVENT_RX_KILL, &dev->flags)) {
		usb_free_urb(urb);
		return -ENOLINK;
	}

#if defined(CONFIG_PLATFORM_NEC)
	msm_watchdog_clear();		/* WAR BUG#29394 */
#endif /* defined(CONFIG_PLATFORM_NEC) */

	skb = __netdev_alloc_skb_ip_align(dev->net, size, flags);
	if (!skb) {
		netif_dbg(dev, rx_err, dev->net, "no rx skb\n");
		usbnet_defer_kevent (dev, EVENT_RX_MEMORY);
		usb_free_urb (urb);
		return -ENOMEM;
	}

	entry = (struct skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->length = 0;

	usb_fill_bulk_urb (urb, dev->udev, dev->in,
		skb->data, size, rx_complete, skb);

	spin_lock_irqsave (&dev->rxq.lock, lockflags);

	if (netif_running (dev->net) &&
	    netif_device_present (dev->net) &&
	    !test_bit (EVENT_RX_HALT, &dev->flags) &&
	    !test_bit (EVENT_DEV_ASLEEP, &dev->flags)) {
		switch (retval = usb_submit_urb (urb, GFP_ATOMIC)) {
		case -EPIPE:
			usbnet_defer_kevent (dev, EVENT_RX_HALT);
			break;
		case -ENOMEM:
			usbnet_defer_kevent (dev, EVENT_RX_MEMORY);
			break;
		case -ENODEV:
			netif_dbg(dev, ifdown, dev->net, "device gone\n");
			netif_device_detach (dev->net);
			break;
		case -EHOSTUNREACH:
			retval = -ENOLINK;
			break;
		default:
			netif_dbg(dev, rx_err, dev->net,
				  "rx submit, %d\n", retval);
			queue_work(usbnet_wq, &dev->bh_w);
			break;
		case 0:
			usb_mark_last_busy(dev->udev);
			__usbnet_queue_skb(&dev->rxq, skb, rx_start);
		}
	} else {
		netif_dbg(dev, ifdown, dev->net, "rx: stopped\n");
		retval = -ENOLINK;
	}
	spin_unlock_irqrestore (&dev->rxq.lock, lockflags);
	if (retval) {
		dev_kfree_skb_any (skb);
		usb_free_urb (urb);
	}
	return retval;
}


/*-------------------------------------------------------------------------*/

static inline void rx_process (struct usbnet *dev, struct sk_buff *skb)
{
	if (dev->driver_info->rx_fixup &&
	    !dev->driver_info->rx_fixup (dev, skb)) {
		/* With RX_ASSEMBLE, rx_fixup() must update counters */
		if (!(dev->driver_info->flags & FLAG_RX_ASSEMBLE))
			dev->net->stats.rx_errors++;
		goto done;
	}
	// else network stack removes extra byte if we forced a short packet

	/* all data was already cloned from skb inside the driver */
	if (dev->driver_info->flags & FLAG_MULTI_PACKET)
		goto done;

	if (skb->len < ETH_HLEN) {
		dev->net->stats.rx_errors++;
		dev->net->stats.rx_length_errors++;
		netif_dbg(dev, rx_err, dev->net, "rx length %d\n", skb->len);
	} else {
		usbnet_skb_return(dev, skb);
		return;
	}

done:
	skb_queue_tail(&dev->done, skb);
}

/*-------------------------------------------------------------------------*/

static void rx_complete(struct urb *urb)
{
	struct sk_buff		*skb = (struct sk_buff *) urb->context;
	struct skb_data		*entry = (struct skb_data *) skb->cb;
	struct usbnet		*dev = entry->dev;
	int			urb_status = urb->status;
	enum skb_state		state;

	skb_put (skb, urb->actual_length);
	state = rx_done;
	entry->urb = NULL;

	usbnet_ipa_debug("%s:%d skb=%p\n", __func__, __LINE__, skb);
	switch (urb_status) {
	/* success */
	case 0:
		break;

	/* stalls need manual reset. this is rare ... except that
	 * when going through USB 2.0 TTs, unplug appears this way.
	 * we avoid the highspeed version of the ETIMEDOUT/EILSEQ
	 * storm, recovering as needed.
	 */
	case -EPIPE:
		dev->net->stats.rx_errors++;
		usbnet_defer_kevent (dev, EVENT_RX_HALT);
		// FALLTHROUGH

	/* software-driven interface shutdown */
	case -ECONNRESET:		/* async unlink */
	case -ESHUTDOWN:		/* hardware gone */
		netif_dbg(dev, ifdown, dev->net,
			  "rx shutdown, code %d\n", urb_status);
		goto block;

	/* we get controller i/o faults during khubd disconnect() delays.
	 * throttle down resubmits, to avoid log floods; just temporarily,
	 * so we still recover when the fault isn't a khubd delay.
	 */
	case -EPROTO:
	case -ETIME:
	case -EILSEQ:
		dev->net->stats.rx_errors++;
		if (!timer_pending (&dev->delay)) {
			mod_timer (&dev->delay, jiffies + THROTTLE_JIFFIES);
			netif_dbg(dev, link, dev->net,
				  "rx throttle %d\n", urb_status);
		}
block:
		state = rx_cleanup;
		entry->urb = urb;
		urb = NULL;
		break;

	/* data overrun ... flush fifo? */
	case -EOVERFLOW:
		dev->net->stats.rx_over_errors++;
		// FALLTHROUGH

	default:
		state = rx_cleanup;
		dev->net->stats.rx_errors++;
		netif_dbg(dev, rx_err, dev->net, "rx status %d\n", urb_status);
		break;
	}

	/* stop rx if packet error rate is high */
	if (++dev->pkt_cnt > 30) {
		dev->pkt_cnt = 0;
		dev->pkt_err = 0;
	} else {
		if (state == rx_cleanup)
			dev->pkt_err++;
		if (dev->pkt_err > 20)
			set_bit(EVENT_RX_KILL, &dev->flags);
	}

	state = defer_bh(dev, skb, &dev->rxq, state);

	if (urb) {
#if !defined(CONFIG_PLATFORM_NEC)
		/* observing memory allocation failure in atomic context for
		 * high throughput use cases of ipa bridge. Avoid recycling
		 * of rx urb for ipa bridge, let usbnet_bh submit the rx urb.
		 */
		if (!enable_ipa_bridge && netif_running(dev->net) &&
		    !test_bit (EVENT_RX_HALT, &dev->flags) &&
		    state != unlink_start) {
			rx_submit (dev, urb, GFP_ATOMIC);
			usb_mark_last_busy(dev->udev);
			return;
		}
#endif /* !defined(CONFIG_PLATFORM_NEC) */
		usb_free_urb (urb);
	}
	netif_dbg(dev, rx_err, dev->net, "no read resubmitted\n");
}
EXPORT_SYMBOL_GPL(rx_complete);

/*-------------------------------------------------------------------------*/
void usbnet_pause_rx(struct usbnet *dev)
{
	set_bit(EVENT_RX_PAUSED, &dev->flags);

	netif_dbg(dev, rx_status, dev->net, "paused rx queue enabled\n");
}
EXPORT_SYMBOL_GPL(usbnet_pause_rx);

void usbnet_resume_rx(struct usbnet *dev)
{
	struct sk_buff *skb;
	int num = 0;

	clear_bit(EVENT_RX_PAUSED, &dev->flags);

	while ((skb = skb_dequeue(&dev->rxq_pause)) != NULL) {
		usbnet_skb_return(dev, skb);
		num++;
	}

	queue_work(usbnet_wq, &dev->bh_w);

	netif_dbg(dev, rx_status, dev->net,
		  "paused rx queue disabled, %d skbs requeued\n", num);
}
EXPORT_SYMBOL_GPL(usbnet_resume_rx);

void usbnet_purge_paused_rxq(struct usbnet *dev)
{
	skb_queue_purge(&dev->rxq_pause);
}
EXPORT_SYMBOL_GPL(usbnet_purge_paused_rxq);

/*-------------------------------------------------------------------------*/

// unlink pending rx/tx; completion handlers do all other cleanup

static int unlink_urbs (struct usbnet *dev, struct sk_buff_head *q)
{
	unsigned long		flags;
	struct sk_buff		*skb;
	int			count = 0;

	spin_lock_irqsave (&q->lock, flags);
	while (!skb_queue_empty(q)) {
		struct skb_data		*entry;
		struct urb		*urb;
		int			retval;

		skb_queue_walk(q, skb) {
			entry = (struct skb_data *) skb->cb;
			if (entry->state != unlink_start)
				goto found;
		}
		break;
found:
		entry->state = unlink_start;
		urb = entry->urb;

		/*
		 * Get reference count of the URB to avoid it to be
		 * freed during usb_unlink_urb, which may trigger
		 * use-after-free problem inside usb_unlink_urb since
		 * usb_unlink_urb is always racing with .complete
		 * handler(include defer_bh).
		 */
		usb_get_urb(urb);
		spin_unlock_irqrestore(&q->lock, flags);
		// during some PM-driven resume scenarios,
		// these (async) unlinks complete immediately
		retval = usb_unlink_urb (urb);
		if (retval != -EINPROGRESS && retval != 0)
			netdev_dbg(dev->net, "unlink urb err, %d\n", retval);
		else
			count++;
		usb_put_urb(urb);
		spin_lock_irqsave(&q->lock, flags);
	}
	spin_unlock_irqrestore (&q->lock, flags);
	return count;
}

// Flush all pending rx urbs
// minidrivers may need to do this when the MTU changes

void usbnet_unlink_rx_urbs(struct usbnet *dev)
{
	if (netif_running(dev->net)) {
		(void) unlink_urbs (dev, &dev->rxq);
		queue_work(usbnet_wq, &dev->bh_w);
	}
}
EXPORT_SYMBOL_GPL(usbnet_unlink_rx_urbs);

/*-------------------------------------------------------------------------*/

// precondition: never called in_interrupt
void usbnet_terminate_urbs(struct usbnet *dev)
{
	DECLARE_WAITQUEUE(wait, current);
	int temp;

	/* ensure there are no more active urbs */
	add_wait_queue(&dev->wait, &wait);
	set_current_state(TASK_UNINTERRUPTIBLE);
	temp = unlink_urbs(dev, &dev->txq) +
		unlink_urbs(dev, &dev->rxq);

	/* maybe wait for deletions to finish. */
	while (!skb_queue_empty(&dev->rxq)
		&& !skb_queue_empty(&dev->txq)
		&& !skb_queue_empty(&dev->done)) {
			schedule_timeout(msecs_to_jiffies(UNLINK_TIMEOUT_MS));
			set_current_state(TASK_UNINTERRUPTIBLE);
			netif_dbg(dev, ifdown, dev->net,
				  "waited for %d urb completions\n", temp);
	}
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&dev->wait, &wait);
}
EXPORT_SYMBOL_GPL(usbnet_terminate_urbs);

int usbnet_stop (struct net_device *net)
{
	struct usbnet		*dev = netdev_priv(net);
	struct driver_info	*info = dev->driver_info;
	int			retval, pm;

	clear_bit(EVENT_DEV_OPEN, &dev->flags);
	netif_stop_queue (net);

	netif_info(dev, ifdown, dev->net,
		   "stop stats: rx/tx %lu/%lu, errs %lu/%lu\n",
		   net->stats.rx_packets, net->stats.tx_packets,
		   net->stats.rx_errors, net->stats.tx_errors);

	/* to not race resume */
	pm = usb_autopm_get_interface(dev->intf);
	/* allow minidriver to stop correctly (wireless devices to turn off
	 * radio etc) */
	if (info->stop) {
		retval = info->stop(dev);
		if (retval < 0)
			netif_info(dev, ifdown, dev->net,
				   "stop fail (%d) usbnet usb-%s-%s, %s\n",
				   retval,
				   dev->udev->bus->bus_name, dev->udev->devpath,
				   info->description);
	}

	if (!(info->flags & FLAG_AVOID_UNLINK_URBS))
		usbnet_terminate_urbs(dev);

	usbnet_status_stop(dev);

	usbnet_purge_paused_rxq(dev);

	/* deferred work (task, timer, softirq) must also stop.
	 * can't flush_scheduled_work() until we drop rtnl (later),
	 * else workers could deadlock; so make workers a NOP.
	 */
	dev->flags = 0;
	del_timer_sync (&dev->delay);
	cancel_work_sync(&dev->bh_w);
	if (!pm)
		usb_autopm_put_interface(dev->intf);

	if (info->manage_power &&
	    !test_and_clear_bit(EVENT_NO_RUNTIME_PM, &dev->flags))
		info->manage_power(dev, 0);
	else
		usb_autopm_put_interface(dev->intf);

	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_stop);

/*-------------------------------------------------------------------------*/

// posts reads, and enables write queuing

// precondition: never called in_interrupt

int usbnet_open (struct net_device *net)
{
	struct usbnet		*dev = netdev_priv(net);
	int			retval;
	struct driver_info	*info = dev->driver_info;

	if ((retval = usb_autopm_get_interface(dev->intf)) < 0) {
		netif_info(dev, ifup, dev->net,
			   "resumption fail (%d) usbnet usb-%s-%s, %s\n",
			   retval,
			   dev->udev->bus->bus_name,
			   dev->udev->devpath,
			   info->description);
		goto done_nopm;
	}

	// put into "known safe" state
	if (info->reset && (retval = info->reset (dev)) < 0) {
		netif_info(dev, ifup, dev->net,
			   "open reset fail (%d) usbnet usb-%s-%s, %s\n",
			   retval,
			   dev->udev->bus->bus_name,
			   dev->udev->devpath,
			   info->description);
		goto done;
	}

	/* hard_mtu or rx_urb_size may change in reset() */
	usbnet_update_max_qlen(dev);

	// insist peer be connected
	if (info->check_connect && (retval = info->check_connect (dev)) < 0) {
		netif_dbg(dev, ifup, dev->net, "can't open; %d\n", retval);
		goto done;
	}

	/* start any status interrupt transfer */
	if (dev->interrupt) {
		retval = usbnet_status_start(dev, GFP_KERNEL);
		if (retval < 0) {
			netif_err(dev, ifup, dev->net,
				  "intr submit %d\n", retval);
			goto done;
		}
	}

	set_bit(EVENT_DEV_OPEN, &dev->flags);
	netif_start_queue (net);
	netif_info(dev, ifup, dev->net,
		   "open: enable queueing (rx %d, tx %d) mtu %d %s framing\n",
		   (int)RX_QLEN(dev), (int)TX_QLEN(dev),
		   dev->net->mtu,
		   (dev->driver_info->flags & FLAG_FRAMING_NC) ? "NetChip" :
		   (dev->driver_info->flags & FLAG_FRAMING_GL) ? "GeneSys" :
		   (dev->driver_info->flags & FLAG_FRAMING_Z) ? "Zaurus" :
		   (dev->driver_info->flags & FLAG_FRAMING_RN) ? "RNDIS" :
		   (dev->driver_info->flags & FLAG_FRAMING_AX) ? "ASIX" :
		   "simple");

	/* reset rx error state */
	dev->pkt_cnt = 0;
	dev->pkt_err = 0;
	clear_bit(EVENT_RX_KILL, &dev->flags);

	// delay posting reads until we're fully open
	queue_work(usbnet_wq, &dev->bh_w);
	if (info->manage_power) {
		retval = info->manage_power(dev, 1);
		if (retval < 0) {
			retval = 0;
			set_bit(EVENT_NO_RUNTIME_PM, &dev->flags);
		} else {
			usb_autopm_put_interface(dev->intf);
		}
	}
	return retval;
done:
	usb_autopm_put_interface(dev->intf);
done_nopm:
	return retval;
}
EXPORT_SYMBOL_GPL(usbnet_open);

/*-------------------------------------------------------------------------*/

/* ethtool methods; minidrivers may need to add some more, but
 * they'll probably want to use this base set.
 */

int usbnet_get_settings (struct net_device *net, struct ethtool_cmd *cmd)
{
	struct usbnet *dev = netdev_priv(net);

	if (!dev->mii.mdio_read)
		return -EOPNOTSUPP;

	return mii_ethtool_gset(&dev->mii, cmd);
}
EXPORT_SYMBOL_GPL(usbnet_get_settings);

int usbnet_set_settings (struct net_device *net, struct ethtool_cmd *cmd)
{
	struct usbnet *dev = netdev_priv(net);
	int retval;

	if (!dev->mii.mdio_write)
		return -EOPNOTSUPP;

	retval = mii_ethtool_sset(&dev->mii, cmd);

	/* link speed/duplex might have changed */
	if (dev->driver_info->link_reset)
		dev->driver_info->link_reset(dev);

	/* hard_mtu or rx_urb_size may change in link_reset() */
	usbnet_update_max_qlen(dev);

	return retval;

}
EXPORT_SYMBOL_GPL(usbnet_set_settings);

u32 usbnet_get_link (struct net_device *net)
{
	struct usbnet *dev = netdev_priv(net);

	/* If a check_connect is defined, return its result */
	if (dev->driver_info->check_connect)
		return dev->driver_info->check_connect (dev) == 0;

	/* if the device has mii operations, use those */
	if (dev->mii.mdio_read)
		return mii_link_ok(&dev->mii);

	/* Otherwise, dtrt for drivers calling netif_carrier_{on,off} */
	return ethtool_op_get_link(net);
}
EXPORT_SYMBOL_GPL(usbnet_get_link);

int usbnet_nway_reset(struct net_device *net)
{
	struct usbnet *dev = netdev_priv(net);

	if (!dev->mii.mdio_write)
		return -EOPNOTSUPP;

	return mii_nway_restart(&dev->mii);
}
EXPORT_SYMBOL_GPL(usbnet_nway_reset);

void usbnet_get_drvinfo (struct net_device *net, struct ethtool_drvinfo *info)
{
	struct usbnet *dev = netdev_priv(net);

	strlcpy (info->driver, dev->driver_name, sizeof info->driver);
	strlcpy (info->version, DRIVER_VERSION, sizeof info->version);
	strlcpy (info->fw_version, dev->driver_info->description,
		sizeof info->fw_version);
	usb_make_path (dev->udev, info->bus_info, sizeof info->bus_info);
}
EXPORT_SYMBOL_GPL(usbnet_get_drvinfo);

u32 usbnet_get_msglevel (struct net_device *net)
{
	struct usbnet *dev = netdev_priv(net);

	return dev->msg_enable;
}
EXPORT_SYMBOL_GPL(usbnet_get_msglevel);

void usbnet_set_msglevel (struct net_device *net, u32 level)
{
	struct usbnet *dev = netdev_priv(net);

	dev->msg_enable = level;
}
EXPORT_SYMBOL_GPL(usbnet_set_msglevel);

/* drivers may override default ethtool_ops in their bind() routine */
static const struct ethtool_ops usbnet_ethtool_ops = {
	.get_settings		= usbnet_get_settings,
	.set_settings		= usbnet_set_settings,
	.get_link		= usbnet_get_link,
	.nway_reset		= usbnet_nway_reset,
	.get_drvinfo		= usbnet_get_drvinfo,
	.get_msglevel		= usbnet_get_msglevel,
	.set_msglevel		= usbnet_set_msglevel,
	.get_ts_info		= ethtool_op_get_ts_info,
};

/*-------------------------------------------------------------------------*/

static void __handle_link_change(struct usbnet *dev)
{
	if (!test_bit(EVENT_DEV_OPEN, &dev->flags))
		return;

	if (!netif_carrier_ok(dev->net)) {
		/* kill URBs for reading packets to save bus bandwidth */
		unlink_urbs(dev, &dev->rxq);

		/*
		 * tx_timeout will unlink URBs for sending packets and
		 * tx queue is stopped by netcore after link becomes off
		 */
	} else {
		/* submitting URBs for reading packets */
		queue_work(usbnet_wq, &dev->bh_w);
	}

	/* hard_mtu or rx_urb_size may change during link change */
	usbnet_update_max_qlen(dev);

	clear_bit(EVENT_LINK_CHANGE, &dev->flags);
}

/* work that cannot be done in interrupt context uses keventd.
 *
 * NOTE:  with 2.5 we could do more of this using completion callbacks,
 * especially now that control transfers can be queued.
 */
static void
kevent (struct work_struct *work)
{
	struct usbnet		*dev =
		container_of(work, struct usbnet, kevent);
	int			status;

	/* usb_clear_halt() needs a thread context */
	if (test_bit (EVENT_TX_HALT, &dev->flags)) {
		unlink_urbs (dev, &dev->txq);
		status = usb_autopm_get_interface(dev->intf);
		if (status < 0)
			goto fail_pipe;
		status = usb_clear_halt (dev->udev, dev->out);
		usb_autopm_put_interface(dev->intf);
		if (status < 0 &&
		    status != -EPIPE &&
		    status != -ESHUTDOWN) {
			if (netif_msg_tx_err (dev))
fail_pipe:
				netdev_err(dev->net, "can't clear tx halt, status %d\n",
					   status);
		} else {
			clear_bit (EVENT_TX_HALT, &dev->flags);
			if (status != -ESHUTDOWN)
				netif_wake_queue (dev->net);
		}
	}
	if (test_bit (EVENT_RX_HALT, &dev->flags)) {
		unlink_urbs (dev, &dev->rxq);
		status = usb_autopm_get_interface(dev->intf);
		if (status < 0)
			goto fail_halt;
		status = usb_clear_halt (dev->udev, dev->in);
		usb_autopm_put_interface(dev->intf);
		if (status < 0 &&
		    status != -EPIPE &&
		    status != -ESHUTDOWN) {
			if (netif_msg_rx_err (dev))
fail_halt:
				netdev_err(dev->net, "can't clear rx halt, status %d\n",
					   status);
		} else {
			clear_bit (EVENT_RX_HALT, &dev->flags);
			queue_work(usbnet_wq, &dev->bh_w);
		}
	}

	/* tasklet could resubmit itself forever if memory is tight */
	if (test_bit (EVENT_RX_MEMORY, &dev->flags)) {
		struct urb	*urb = NULL;
		int resched = 1;

		if (netif_running (dev->net))
			urb = usb_alloc_urb (0, GFP_KERNEL);
		else
			clear_bit (EVENT_RX_MEMORY, &dev->flags);
		if (urb != NULL) {
			clear_bit (EVENT_RX_MEMORY, &dev->flags);
			status = usb_autopm_get_interface(dev->intf);
			if (status < 0) {
				usb_free_urb(urb);
				goto fail_lowmem;
			}
			if (rx_submit (dev, urb, GFP_KERNEL) == -ENOLINK)
				resched = 0;
			usb_autopm_put_interface(dev->intf);
fail_lowmem:
			if (resched)
				queue_work(usbnet_wq, &dev->bh_w);
		}
	}

	if (test_bit (EVENT_LINK_RESET, &dev->flags)) {
		struct driver_info	*info = dev->driver_info;
		int			retval = 0;

		clear_bit (EVENT_LINK_RESET, &dev->flags);
		status = usb_autopm_get_interface(dev->intf);
		if (status < 0)
			goto skip_reset;
		if(info->link_reset && (retval = info->link_reset(dev)) < 0) {
			usb_autopm_put_interface(dev->intf);
skip_reset:
			netdev_info(dev->net, "link reset failed (%d) usbnet usb-%s-%s, %s\n",
				    retval,
				    dev->udev->bus->bus_name,
				    dev->udev->devpath,
				    info->description);
		} else {
			usb_autopm_put_interface(dev->intf);
		}

		/* handle link change from link resetting */
		__handle_link_change(dev);
	}

	if (test_bit (EVENT_LINK_CHANGE, &dev->flags))
		__handle_link_change(dev);

	if (dev->flags)
		netdev_dbg(dev->net, "kevent done, flags = 0x%lx\n", dev->flags);
}

/*-------------------------------------------------------------------------*/

static void tx_complete (struct urb *urb)
{
	struct sk_buff		*skb = (struct sk_buff *) urb->context;
	struct skb_data		*entry = (struct skb_data *) skb->cb;
	struct usbnet		*dev = entry->dev;

	if (urb->status == 0) {
		if (!(dev->driver_info->flags & FLAG_MULTI_PACKET))
			dev->net->stats.tx_packets++;
		dev->net->stats.tx_bytes += entry->length;
	} else {
		dev->net->stats.tx_errors++;

		switch (urb->status) {
		case -EPIPE:
			usbnet_defer_kevent (dev, EVENT_TX_HALT);
			break;

		/* software-driven interface shutdown */
		case -ECONNRESET:		// async unlink
		case -ESHUTDOWN:		// hardware gone
			break;

		// like rx, tx gets controller i/o faults during khubd delays
		// and so it uses the same throttling mechanism.
		case -EPROTO:
		case -ETIME:
		case -EILSEQ:
			usb_mark_last_busy(dev->udev);
			if (!timer_pending (&dev->delay)) {
				mod_timer (&dev->delay,
					jiffies + THROTTLE_JIFFIES);
				netif_dbg(dev, link, dev->net,
					  "tx throttle %d\n", urb->status);
			}
			netif_stop_queue (dev->net);
			break;
		default:
			netif_dbg(dev, tx_err, dev->net,
				  "tx err %d\n", entry->urb->status);
			break;
		}
	}

	usb_autopm_put_interface_async(dev->intf);
	(void) defer_bh(dev, skb, &dev->txq, tx_done);
}

/*-------------------------------------------------------------------------*/

void usbnet_tx_timeout (struct net_device *net)
{
	struct usbnet		*dev = netdev_priv(net);

	unlink_urbs (dev, &dev->txq);
	queue_work(usbnet_wq, &dev->bh_w);

	// FIXME: device recovery -- reset?
}
EXPORT_SYMBOL_GPL(usbnet_tx_timeout);

/*-------------------------------------------------------------------------*/

netdev_tx_t usbnet_start_xmit (struct sk_buff *skb,
				     struct net_device *net)
{
	struct usbnet		*dev = netdev_priv(net);
	int			length;
	struct urb		*urb = NULL;
	struct skb_data		*entry;
	struct driver_info	*info = dev->driver_info;
	unsigned long		flags;
	int retval;

	usbnet_ipa_debug("%s:%d skb=%p ->dev=%p ->len=%d enable_ipa_bridge=%d\n", __func__, __LINE__,skb, skb->dev,skb->len,enable_ipa_bridge);

	if (enable_ipa_bridge && dev->txq.qlen >= TX_QLEN(dev))
		goto drop;

	if (skb)
		skb_tx_timestamp(skb);

	// some devices want funky USB-level framing, for
	// win32 driver (usually) and/or hardware quirks
	if (info->tx_fixup) {
		skb = info->tx_fixup (dev, skb, GFP_ATOMIC);
		if (!skb) {
			/* packet collected; minidriver waiting for more */
			if (info->flags & FLAG_MULTI_PACKET)
				goto not_drop;
			netif_dbg(dev, tx_err, dev->net, "can't tx_fixup skb\n");
			goto drop;
		}
	}
	length = skb->len;

	if (!(urb = usb_alloc_urb (0, GFP_ATOMIC))) {
		netif_dbg(dev, tx_err, dev->net, "no urb\n");
		goto drop;
	}

	entry = (struct skb_data *) skb->cb;
	entry->urb = urb;
	entry->dev = dev;
	entry->length = length;

	usb_fill_bulk_urb (urb, dev->udev, dev->out,
			skb->data, skb->len, tx_complete, skb);

	/* don't assume the hardware handles USB_ZERO_PACKET
	 * NOTE:  strictly conforming cdc-ether devices should expect
	 * the ZLP here, but ignore the one-byte packet.
	 * NOTE2: CDC NCM specification is different from CDC ECM when
	 * handling ZLP/short packets, so cdc_ncm driver will make short
	 * packet itself if needed.
	 */
	if (length % dev->maxpacket == 0) {
		if (!(info->flags & FLAG_SEND_ZLP)) {
			if (!(info->flags & FLAG_MULTI_PACKET)) {
				urb->transfer_buffer_length++;
				if (skb_tailroom(skb)) {
					skb->data[skb->len] = 0;
					__skb_put(skb, 1);
				}
			}
		} else
			urb->transfer_flags |= URB_ZERO_PACKET;
	}

	spin_lock_irqsave(&dev->txq.lock, flags);
	retval = usb_autopm_get_interface_async(dev->intf);
	if (retval < 0) {
		spin_unlock_irqrestore(&dev->txq.lock, flags);
		goto drop;
	}

#ifdef CONFIG_PM
	/* if this triggers the device is still a sleep */
	if (test_bit(EVENT_DEV_ASLEEP, &dev->flags)) {
		/* transmission will be done in resume */
		usb_anchor_urb(urb, &dev->deferred);
		/* no use to process more packets */
		netif_stop_queue(net);
		usb_put_urb(urb);
		spin_unlock_irqrestore(&dev->txq.lock, flags);
		netdev_dbg(dev->net, "Delaying transmission for resumption\n");
		goto deferred;
	}
#endif

	switch ((retval = usb_submit_urb (urb, GFP_ATOMIC))) {
	case -EPIPE:
		netif_stop_queue (net);
		usbnet_defer_kevent (dev, EVENT_TX_HALT);
		usb_autopm_put_interface_async(dev->intf);
		break;
	default:
		usb_autopm_put_interface_async(dev->intf);
		netif_dbg(dev, tx_err, dev->net,
			  "tx: submit urb err %d\n", retval);
		break;
	case 0:
		net->trans_start = jiffies;
		__usbnet_queue_skb(&dev->txq, skb, tx_start);
		if (!enable_ipa_bridge && dev->txq.qlen >= TX_QLEN(dev))
			netif_stop_queue (net);
	}
	spin_unlock_irqrestore (&dev->txq.lock, flags);

	if (retval) {
		netif_dbg(dev, tx_err, dev->net, "drop, code %d\n", retval);
drop:
		dev->net->stats.tx_dropped++;
not_drop:
		if (skb)
			dev_kfree_skb_any (skb);
		usb_free_urb (urb);
	} else
		netif_dbg(dev, tx_queued, dev->net,
			  "> tx, len %d, type 0x%x\n", length, skb->protocol);
#ifdef CONFIG_PM
deferred:
#endif
	return NETDEV_TX_OK;
}
EXPORT_SYMBOL_GPL(usbnet_start_xmit);

static int rx_alloc_submit(struct usbnet *dev, gfp_t flags)
{
	struct urb	*urb;
	int		i;
	int		ret = 0;

	/* don't refill the queue all at once */
	for (i = 0; i < 10 && dev->rxq.qlen < RX_QLEN(dev); i++) {
		urb = usb_alloc_urb(0, flags);
		if (urb != NULL) {
			ret = rx_submit(dev, urb, flags);
			if (ret)
				goto err;
		} else {
			ret = -ENOMEM;
			goto err;
		}
	}
err:
	return ret;
}

/*-------------------------------------------------------------------------*/

// tasklet (work deferred from completions, in_irq) or timer

static void usbnet_bh (unsigned long param)
{
	struct usbnet		*dev = (struct usbnet *) param;
	struct sk_buff		*skb;
	struct skb_data		*entry;

	set_wake_up_idle(true);
	while ((skb = skb_dequeue (&dev->done))) {
		entry = (struct skb_data *) skb->cb;
		switch (entry->state) {
		case rx_done:
			entry->state = rx_cleanup;
			rx_process (dev, skb);
			continue;
		case tx_done:
		case rx_cleanup:
			usb_free_urb (entry->urb);
			dev_kfree_skb (skb);
			continue;
		default:
			netdev_dbg(dev->net, "bogus skb state %d\n", entry->state);
		}
	}
	set_wake_up_idle(false);

	/* restart RX again after disabling due to high error rate */
	clear_bit(EVENT_RX_KILL, &dev->flags);

	/* waiting for all pending urbs to complete?
	 * only then can we forgo submitting anew
	 */
	if (waitqueue_active(&dev->wait)) {
		if (dev->txq.qlen + dev->rxq.qlen + dev->done.qlen == 0)
			wake_up_all(&unlink_wakeup);

	// or are we maybe short a few urbs?
	} else if (netif_running (dev->net) &&
		   netif_device_present (dev->net) &&
		   netif_carrier_ok(dev->net) &&
		   !timer_pending (&dev->delay) &&
		   !test_bit (EVENT_RX_HALT, &dev->flags)) {
		int	temp = dev->rxq.qlen;

		if (temp < RX_QLEN(dev)) {
			if (rx_alloc_submit(dev, GFP_KERNEL) == -ENOLINK)
				return;
			if (temp != dev->rxq.qlen)
				netif_dbg(dev, link, dev->net,
					  "rxqlen %d --> %d\n",
					  temp, dev->rxq.qlen);
			if (dev->rxq.qlen < RX_QLEN(dev))
				queue_work(usbnet_wq, &dev->bh_w);
		}
		if (!enable_ipa_bridge && dev->txq.qlen < TX_QLEN(dev))
			netif_wake_queue (dev->net);
	}
}

static void usbnet_bh_w(struct work_struct *work)
{
	struct usbnet		*dev =
		container_of(work, struct usbnet, bh_w);
	unsigned long param = (unsigned long)dev;

	usbnet_bh(param);
}

#ifdef CONFIG_USBNET_IPA
void
usbnet_ipa_tx_notify(void *priv, enum ipa_dp_evt_type evt, unsigned long data)
{
	struct usbnet *dev = priv;
	struct usbnet_ipa_ctx *usbnet_ipa = dev->pusbnet_ipa;
	struct sk_buff *skb = (struct sk_buff *) data;
	int status;
	bool schedule_ipa_work = false;

	usbnet_ipa_debug("%s:%d evt 0x%x \n", __func__, __LINE__, evt);

	switch(evt) {
	case IPA_RECEIVE:
		usbnet_ipa_debug("%s:%d receive IPA_RECEIVE event : 0x%x skb=%p\n", __func__, __LINE__, evt,skb);
		usbnet_ipa->stats.rx_ipa_excep++;
		skb->dev = dev->net;
		skb->protocol = eth_type_trans(skb, dev->net);

		usbnet_ipa_debug("%s:%d skb->dev %p\n", __func__, __LINE__, skb->dev);
		status = netif_rx_ni(skb);
		if (status != NET_RX_SUCCESS)
			netif_dbg(dev, rx_err, dev->net,
				  "netif_rx status %d\n", status);
		break;
	case IPA_WRITE_DONE:
		usbnet_ipa_debug("%s:%d receive WRITE DONE event : 0x%x skb=%p\n", __func__, __LINE__, evt,skb);
		usbnet_ipa->stats.rx_ipa_write_done++;
		kfree_skb(skb);
		spin_lock_bh(&usbnet_ipa->flow_ctrl_lock);
		usbnet_ipa->ipa_free_desc_cnt++;
		if ((usbnet_ipa->pendq_cnt > 0) &&
		    (usbnet_ipa->ipa_free_desc_cnt >= usbnet_ipa->ipa_low_watermark)) {
			schedule_ipa_work = true;
			usbnet_ipa->stats.ipa_low_watermark_cnt++;
		}

		spin_unlock_bh(&usbnet_ipa->flow_ctrl_lock);
		if (schedule_ipa_work)
			schedule_work(&usbnet_ipa->ipa_send_task);
		break;
	default:
		usbnet_ipa_info("%s:%d wrong event : 0x%x \n", __func__, __LINE__, evt);
		return;
	}
}

void
usbnet_ipa_rx_notify(void *priv, enum ipa_dp_evt_type evt, unsigned long data)
{
	struct usbnet *dev = priv;
	struct usbnet_ipa_ctx *usbnet_ipa = dev->pusbnet_ipa;
	struct sk_buff *skb = (struct sk_buff *)data;
	int ret;

	usbnet_ipa_debug("%s:%d evt 0x%x \n", __func__, __LINE__, evt);

	if (evt != IPA_RECEIVE) {
		usbnet_ipa_info("%s:%d error unexpected event (evt 0x%x) \n", __func__, __LINE__, evt);
		WARN_ON(1);
		return;
	}

	skb->dev = dev->net;
	ret = dev_queue_xmit(skb);
	if (ret) {
		usbnet_ipa->stats.tx_ipa_send_err++;
		usbnet_ipa_debug("%s:%d failed dev_queue_xmit %d \n", __func__, __LINE__, ret);
	}else{
		usbnet_ipa->stats.tx_ipa_send++;
	}
}

int
usbnet_ipa_sys_pipe_setup(struct usbnet_ipa_ctx *ctx)
{
	struct ipa_sys_connect_params usbnet_prod_params;
	struct ipa_sys_connect_params usbnet_cons_params;
	struct ipa_sys_connect_params *p;
	int res;

	memset(&usbnet_prod_params, 0, sizeof(usbnet_prod_params));
	memset(&usbnet_cons_params, 0, sizeof(usbnet_cons_params));

	p = &usbnet_prod_params;
	p->client = IPA_CLIENT_USBNET_PROD;
	p->ipa_ep_cfg.hdr.hdr_len = ETH_HLEN;
	p->ipa_ep_cfg.mode.mode = IPA_BASIC;
	p->ipa_ep_cfg.nat.nat_en = IPA_SRC_NAT;
	p->desc_fifo_sz = (ctx->ipa_high_watermark + 1) * sizeof(struct sps_iovec);
	p->priv = ctx->dev;
	p->notify = usbnet_ipa_tx_notify;
	p->keep_ipa_awake = true;

	printk(KERN_INFO "@@ %s:%d desc_fifo_sz %d \n", __func__, __LINE__, p->desc_fifo_sz);

	res = ipa_setup_sys_pipe(p, &ctx->prod_hdl);
	if (res)
		goto fail_prod;

	p = &usbnet_cons_params;
	p->client = IPA_CLIENT_USBNET_CONS;
	p->ipa_ep_cfg.hdr.hdr_len = ETH_HLEN;
	p->ipa_ep_cfg.mode.mode = IPA_BASIC;
	p->ipa_ep_cfg.nat.nat_en = IPA_BYPASS_NAT;
	p->desc_fifo_sz = (ctx->ipa_high_watermark + 1) * sizeof(struct sps_iovec);
	p->priv = ctx->dev;
	p->notify = usbnet_ipa_rx_notify;
	p->keep_ipa_awake = true;

	res = ipa_setup_sys_pipe(p, &ctx->cons_hdl);
	if (res)
		goto fail_cons;

	return 0;
fail_cons:
	usbnet_ipa_info("%s:%d failed to setup cons sys pipe.\n", __func__, __LINE__);
	ipa_teardown_sys_pipe(ctx->prod_hdl);
	ctx->prod_hdl = 0;
fail_prod:
	usbnet_ipa_info("%s:%d\n failed to setup prod sys pipe.\n", __func__, __LINE__);
	return res;
}

static void
usbnet_ipa_lan_rm_notify(void *user_data, enum ipa_rm_event event, unsigned long data)
{
	usbnet_ipa_debug("%s:%d event %d\n", __func__, __LINE__, event);
	return;
}

static int
usbnet_ipa_lan_rm_cons_request(void)
{
	usbnet_ipa_debug("%s:%d\n", __func__, __LINE__);
	return 0;
}

static int
usbnet_ipa_lan_rm_cons_release(void)
{
	usbnet_ipa_debug("%s:%d\n", __func__, __LINE__);
	return 0;
}

static int
usbnet_ipa_lan_setup_rm(void)
{
	struct ipa_rm_create_params params = {0};
	int ret;

	memset(&params, 0, sizeof(params));
	params.name = IPA_RM_RESOURCE_USBNET_PROD;
	params.reg_params.user_data = usbnet_ipa;
	params.reg_params.notify_cb = usbnet_ipa_lan_rm_notify;
	params.floor_voltage = IPA_VOLTAGE_SVS;

	ret = ipa_rm_create_resource(&params);
	if (ret) {
		usbnet_ipa_info("%s:%d failed to create prod resource.\n", __func__, __LINE__);
		goto fail;
	}

	memset(&params, 0, sizeof(params));
	params.name = IPA_RM_RESOURCE_USBNET_CONS;
	params.request_resource = usbnet_ipa_lan_rm_cons_request;
	params.release_resource = usbnet_ipa_lan_rm_cons_release;
	params.floor_voltage = IPA_VOLTAGE_SVS;

	ret = ipa_rm_create_resource(&params);
	if (ret) {
		usbnet_ipa_info("%s:%d failed to create cons resource.\n", __func__, __LINE__);
		goto delete_prod;
	}

	ipa_rm_add_dependency_sync(IPA_RM_RESOURCE_USBNET_PROD, IPA_RM_RESOURCE_APPS_CONS);

	return ret;

delete_prod:
	ipa_rm_delete_resource(IPA_RM_RESOURCE_USBNET_PROD);
fail:
	return ret;
}

static int
usbnet_ipa_register_interface(void)
{
	struct ipa_tx_intf tx_intf = {0};
	struct ipa_ioc_tx_intf_prop tx_properties[2] = {{0},{0}};
	struct ipa_ioc_tx_intf_prop *ipv4_tx_property;
	struct ipa_ioc_tx_intf_prop *ipv6_tx_property;
	struct ipa_rx_intf rx_intf = {0};
	struct ipa_ioc_rx_intf_prop rx_properties[2] = {{0},{0}};
	struct ipa_ioc_rx_intf_prop *ipv4_rx_property;
	struct ipa_ioc_rx_intf_prop *ipv6_rx_property;
	int res;

	tx_intf.prop = tx_properties;
	ipv4_tx_property = &tx_intf.prop[0];
	ipv4_tx_property->ip = IPA_IP_v4;
	ipv4_tx_property->dst_pipe = IPA_CLIENT_USBNET_CONS;
	snprintf(ipv4_tx_property->hdr_name, IPA_RESOURCE_NAME_MAX, "%s%s", usbnet_ipa->devname, "_ipv4");
	ipv4_tx_property->hdr_l2_type = IPA_HDR_L2_ETHERNET_II;
	ipv6_tx_property = &tx_intf.prop[1];
	ipv6_tx_property->ip = IPA_IP_v6;
	ipv6_tx_property->dst_pipe = IPA_CLIENT_USBNET_CONS;
	snprintf(ipv6_tx_property->hdr_name, IPA_RESOURCE_NAME_MAX, "%s%s", usbnet_ipa->devname, "_ipv6");
	ipv6_tx_property->hdr_l2_type = IPA_HDR_L2_ETHERNET_II;
	tx_intf.num_props = 2;

	rx_intf.prop = rx_properties;
	ipv4_rx_property = &rx_intf.prop[0];
	ipv4_rx_property->ip = IPA_IP_v4;
	ipv4_rx_property->attrib.attrib_mask = 0;
	ipv4_rx_property->src_pipe = IPA_CLIENT_USBNET_PROD;
	ipv4_rx_property->hdr_l2_type = IPA_HDR_L2_ETHERNET_II;
	ipv6_rx_property = &rx_intf.prop[1];
	ipv6_rx_property->ip = IPA_IP_v6;
	ipv6_rx_property->attrib.attrib_mask = 0;
	ipv6_rx_property->src_pipe = IPA_CLIENT_USBNET_PROD;
	ipv6_rx_property->hdr_l2_type = IPA_HDR_L2_ETHERNET_II;
	rx_intf.num_props = 2;

	res = ipa_register_intf(usbnet_ipa->devname, &tx_intf, &rx_intf);
	if (res) {
		usbnet_ipa_info("%s:%d fail to register Tx/Rx interface.\n", __func__, __LINE__);
	}

	return res;
}

static int
usbnet_ipa_add_hdrs(struct usbnet_ipa_ctx *ctx)
{
	struct ipa_ioc_add_hdr *hdrs;
	struct ipa_hdr_add *ipv4_hdr;
	struct ipa_hdr_add *ipv6_hdr;
	struct ethhdr *eth_ipv4;
	struct ethhdr *eth_ipv6;
	int res;

	hdrs = kzalloc(sizeof(*hdrs) + sizeof(*ipv4_hdr) + sizeof(*ipv6_hdr),
		       GFP_KERNEL);
	if (!hdrs) {
		usbnet_ipa_info("%s:%d can not allocate memory.\n", __func__, __LINE__);
		res = -ENOMEM;
		goto out;
	}

	ipv4_hdr = &hdrs->hdr[0];
	eth_ipv4 = (struct ethhdr *)(ipv4_hdr->hdr);
	ipv6_hdr = &hdrs->hdr[1];
	eth_ipv6 = (struct ethhdr *)(ipv6_hdr->hdr);

	snprintf(ipv4_hdr->name, IPA_RESOURCE_NAME_MAX, "%s%s", ctx->devname, "_ipv4");
	memcpy(eth_ipv4->h_source, ctx->device_ethaddr, ETH_ALEN);

	usbnet_ipa_debug("device_ethaddr %02x:%02x:%02x:%02x:%02x:%02x\n",
	       eth_ipv4->h_source[0], eth_ipv4->h_source[1], eth_ipv4->h_source[2],
	       eth_ipv4->h_source[3], eth_ipv4->h_source[4], eth_ipv4->h_source[5]);

	eth_ipv4->h_proto = htons(ETH_P_IP);
	ipv4_hdr->hdr_len = ETH_HLEN;
	ipv4_hdr->is_partial = 1;
	ipv4_hdr->is_eth2_ofst_valid = 1;
	ipv4_hdr->eth2_ofst = 0;
	ipv4_hdr->type = IPA_HDR_L2_ETHERNET_II;
	snprintf(ipv6_hdr->name, IPA_RESOURCE_NAME_MAX, "%s%s", ctx->devname, "_ipv6");
	memcpy(eth_ipv6->h_source, ctx->device_ethaddr, ETH_ALEN);

	eth_ipv6->h_proto = htons(ETH_P_IPV6);
	ipv6_hdr->hdr_len = ETH_HLEN;
	ipv6_hdr->is_partial = 1;
	ipv6_hdr->is_eth2_ofst_valid = 1;
	ipv6_hdr->eth2_ofst = 0;
	ipv6_hdr->type = IPA_HDR_L2_ETHERNET_II;
	hdrs->commit = 1;
	hdrs->num_hdrs = 2;
	res = ipa_add_hdr(hdrs);
	if (res) {
		usbnet_ipa_info("%s:%d Fail on Header-Insertion(%d)\n", __func__, __LINE__, res);
		goto out_free_mem;
	}
	if (ipv4_hdr->status) {
		usbnet_ipa_info("%s:%d Fail on Header-Insertion ipv4 (%d)\n", __func__, __LINE__, res);
		res = ipv4_hdr->status;
		goto out_free_mem;
	}
	if (ipv6_hdr->status) {
		usbnet_ipa_info("%s:%d Fail on Header-Insertion ipv6 (%d)\n", __func__, __LINE__, res);
		res = ipv6_hdr->status;
		goto out_free_mem;
	}

	ctx->usbnet_ipv4_hdr_hdl = ipv4_hdr->hdr_hdl;
	ctx->usbnet_ipv6_hdr_hdl = ipv6_hdr->hdr_hdl;

	res = 0;

out_free_mem:
	kfree(hdrs);
out:
	return res;
}

static void
usbnet_ipa_deregister_interface(struct usbnet_ipa_ctx *ctx)
{
	int ret;

	ret = ipa_deregister_intf(ctx->devname);
	if (ret)
		usbnet_ipa_info("%s:%d failed to deregister Tx/Rx interface.\n", __func__, __LINE__);
}


static void
usbnet_ipa_cleanup_hdr(struct usbnet_ipa_ctx *ctx)
{
	struct ipa_ioc_del_hdr *del_hdr;
	struct ipa_hdr_del *ipv4;
	struct ipa_hdr_del *ipv6;
	int ret;

	del_hdr = kzalloc(sizeof(*del_hdr) + sizeof(*ipv4) + sizeof(*ipv6), GFP_KERNEL);
	if (del_hdr == NULL) {
		usbnet_ipa_info("%s:%d allocation failed.\n", __func__, __LINE__);
		return;
	}

	del_hdr->commit = 1;
	del_hdr->num_hdls = 2;
	ipv4 = &del_hdr->hdl[0];
	ipv4->hdl = ctx->usbnet_ipv4_hdr_hdl;
	ipv6 = &del_hdr->hdl[1];
	ipv6->hdl = ctx->usbnet_ipv6_hdr_hdl;
	ret = ipa_del_hdr(del_hdr);

	if(ret || ipv4->status || ipv6->status)
		usbnet_ipa_info("%s:%d failed to delete header.\n", __func__, __LINE__);

	kfree(del_hdr);
}

static void
usbnet_ipa_lan_cleanup_rm(void)
{
	int ret;

	ipa_rm_delete_dependency(IPA_RM_RESOURCE_USBNET_PROD, IPA_RM_RESOURCE_APPS_CONS);
	ret = ipa_rm_delete_resource(IPA_RM_RESOURCE_USBNET_PROD);
	if (ret)
		usbnet_ipa_info("%s:%d PROD resource delete failed %d.\n", __func__, __LINE__, ret);

	ret = ipa_rm_delete_resource(IPA_RM_RESOURCE_USBNET_CONS);
	if (ret)
	       usbnet_ipa_info("%s:%d CONS resource delete failed %d.\n", __func__, __LINE__, ret);
}

static void
usbnet_ipa_teardown_sys_pipe(struct usbnet_ipa_ctx *ctx)
{
	int ret;
	ret = ipa_teardown_sys_pipe(ctx->prod_hdl);
	if (ret)
		usbnet_ipa_info("%s:%d failed to teardown prod sys pipe.\n", __func__, __LINE__);

	ret = ipa_teardown_sys_pipe(ctx->cons_hdl);
	if (ret)
		usbnet_ipa_info("%s:%d failed to teardown cons sys pipe.\n", __func__, __LINE__);
}

static void
usbnet_ipa_send_routine(struct work_struct *work)
{
	struct usbnet_ipa_ctx *ctx = container_of(work, struct usbnet_ipa_ctx, ipa_send_task);
	struct usbnet_ipa_desc_node *node = NULL;
	struct ipa_tx_meta metadata;
	int ret;

	memset(&metadata, 0, sizeof(metadata));

	spin_lock_bh(&ctx->flow_ctrl_lock);

	if (unlikely(!ctx->pendq_cnt)) {
		usbnet_ipa_debug("%s no pending packts\n", __func__);
		spin_unlock_bh(&ctx->flow_ctrl_lock);
		return;
	}

	if (ctx->ipa_free_desc_cnt < ctx->ipa_low_watermark) {
		spin_unlock_bh(&ctx->flow_ctrl_lock);
		return;
	}

	while(ctx->ipa_free_desc_cnt && ctx->pendq_cnt) {
		node = list_first_entry(&ctx->pend_queue_head, struct usbnet_ipa_desc_node, link);
		list_del(&node->link);
		list_add_tail(&node->link, &ctx->free_queue_head);
		ctx->freeq_cnt++;
		ctx->pendq_cnt--;
		metadata.dma_address_valid = false;

		usbnet_ipa->stats.rx_ipa_send++;
		ret = ipa_tx_dp(IPA_CLIENT_USBNET_PROD, node->skb_ptr, &metadata);

		if (ret) {
			usbnet_ipa_debug("%s:%d failed to transmit packet to ipa. ", __func__, __LINE__);
			kfree(node->skb_ptr);
			/* @@ TODO : increase fail count */
			usbnet_ipa->stats.rx_ipa_send_fail++;
		} else {
			usbnet_ipa->ipa_free_desc_cnt--;
		}

	}
	spin_unlock_bh(&ctx->flow_ctrl_lock);
}

static int
usbnet_ipa_alloc_flow_ctrl_desc(struct usbnet_ipa_ctx *ctx)
{
	int i;
	struct usbnet_ipa_desc_node *node = NULL;

	for (i = 0; i < IPA_USBNET_DNE_PKTS_DESC; i++) {
		node = (struct usbnet_ipa_desc_node *) kzalloc(sizeof(struct usbnet_ipa_desc_node), GFP_KERNEL);

		if (node == NULL) {
			pr_err("%s : allocation failed.", __func__);
			return -ENOMEM;
		}

		spin_lock(&ctx->flow_ctrl_lock);
		ctx->freeq_cnt++;
		list_add_tail(&node->link, &ctx->free_queue_head);
		spin_unlock(&ctx->flow_ctrl_lock);
	}
	return 0;
}

static void
usbnet_ipa_free_flow_ctrl_desc(struct usbnet_ipa_ctx *ctx)
{
	struct usbnet_ipa_desc_node *node, *tmp;
	spin_lock_bh(&ctx->flow_ctrl_lock);
	list_for_each_entry_safe(node, tmp, &ctx->free_queue_head, link) {
		list_del(&node->link);
		kfree(node);
		ctx->freeq_cnt--;
	}
	spin_unlock_bh(&ctx->flow_ctrl_lock);
	if (ctx->freeq_cnt != 0) {
		pr_err("%s : Couldn't free all free_queue \n", __func__);
	}
}

static void
usbnet_ipa_flush_pend_queue(struct usbnet_ipa_ctx *ctx)
{
	struct usbnet_ipa_desc_node *node;
	spin_lock_bh(&ctx->flow_ctrl_lock);
	while (ctx->pendq_cnt) {
		node = list_first_entry(&ctx->pend_queue_head, struct usbnet_ipa_desc_node, link);
		list_del(&node->link);
		list_add_tail(&node->link, &ctx->free_queue_head);
		ctx->pendq_cnt--;
		ctx->freeq_cnt++;
	}

	if ((ctx->freeq_cnt != IPA_USBNET_DNE_PKTS_DESC) || (ctx->pendq_cnt != 0)) {
		pr_err("%s freeq_cnt %d pendq_cnt %d\n", __func__, ctx->freeq_cnt, ctx->pendq_cnt);
	}
	spin_unlock_bh(&ctx->flow_ctrl_lock);
}

static int usbnet_ipa_lan_set_perf_level(struct usbnet *dev)
{
#if 0 /* To Do */
	struct ipa_rm_perf_profile profile;
	int ret = 0;

	memset(&profile, 0, sizeof(profile));

	if (dev->udev->speed == USB_SPEED_SUPER)
		profile.max_supported_bandwidth_mbps = 800;
	else
		profile.max_supported_bandwidth_mbps = 400;

	ret = ipa_rm_set_perf_profile(IPA_RM_RESOURCE_USBNET_PROD,
					&profile);
	if (ret) {
		pr_err("Err to set BW: IPA_RM_RESOURCE_USBNET_PROD err:%d\n",
			ret);
		return ret;
	}

	ret = ipa_rm_set_perf_profile(IPA_RM_RESOURCE_USBNET_CONS,
					&profile);
	if (ret) {
		pr_err("Err to set BW: IPA_RM_RESOURCE_USBNET_CONS err:%d\n",
			ret);
		return ret;
	}

	return ret;
#endif /* To Do */
	return 0;
}

static int
usbnet_ipa_init(struct usbnet *dev)
{
	int				status;

	usbnet_ipa_info("%s:%d IPA initialization start.\n", __func__, __LINE__);
	usbnet_ipa = kzalloc(sizeof(*usbnet_ipa), GFP_KERNEL);
	if (!usbnet_ipa) {
		usbnet_ipa_info("%s:%d kzalloc error.\n", __func__, __LINE__);
	}

	dev->pusbnet_ipa = usbnet_ipa;
	usbnet_ipa->dev = dev;
	strlcpy(usbnet_ipa->devname, dev->net->name, sizeof(usbnet_ipa->devname));
	usbnet_ipa_debug("usbnet_ipa->devname %s\n", usbnet_ipa->devname);
	memcpy(usbnet_ipa->device_ethaddr, dev->net->dev_addr, ETH_ALEN);

	usbnet_ipa_debug("device_ethaddr %02x:%02x:%02x:%02x:%02x:%02x\n",
	       usbnet_ipa->device_ethaddr[0], usbnet_ipa->device_ethaddr[1], usbnet_ipa->device_ethaddr[2],
	       usbnet_ipa->device_ethaddr[3], usbnet_ipa->device_ethaddr[4], usbnet_ipa->device_ethaddr[5]);

	/* Initialize flow control valiables */
	usbnet_ipa->pendq_cnt = 0;
	usbnet_ipa->freeq_cnt = 0;
	usbnet_ipa->ipa_free_desc_cnt = IPA_USBNET_MAX_PKTS_DESC;
	usbnet_ipa->ipa_high_watermark = IPA_USBNET_MAX_PKTS_DESC;
	usbnet_ipa->ipa_low_watermark = IPA_USBNET_MIN_PKTS_DESC;
	spin_lock_init(&usbnet_ipa->flow_ctrl_lock);
	INIT_LIST_HEAD(&usbnet_ipa->pend_queue_head);
	INIT_LIST_HEAD(&usbnet_ipa->free_queue_head);
	usbnet_ipa_alloc_flow_ctrl_desc(usbnet_ipa);

	usbnet_ipa_info("%s:%d setup resouce manager.\n", __func__, __LINE__);
	usbnet_ipa_lan_setup_rm();

	usbnet_ipa_info("%s:%d setup sys pipe.\n", __func__, __LINE__);
	usbnet_ipa_sys_pipe_setup(usbnet_ipa);

	usbnet_ipa_info("%s:%d add headers.\n", __func__, __LINE__);
	usbnet_ipa_add_hdrs(usbnet_ipa);

	usbnet_ipa_info("%s:%d register interface.\n", __func__, __LINE__);
	usbnet_ipa_register_interface();

	INIT_WORK(&usbnet_ipa->ipa_send_task, usbnet_ipa_send_routine);

	status = usbnet_debugfs_init(dev);
	if (status)
		pr_err("USBNET: Debugfs Init Failed\n");

	status = usbnet_ipa_lan_set_perf_level(dev);
	if (status)
		pr_err("USBNET: perf_level set Failed\n");
	usbnet_ipa_info("%s:%d IPA initialization end.\n", __func__, __LINE__);
	return 0;
}

static int
usbnet_ipa_cleanup(struct usbnet_ipa_ctx *ctx)
{
	usbnet_ipa_info("%s:%d IPA Clean up start.\n", __func__, __LINE__);

	if (!usbnet_ipa) {
		usbnet_ipa_info("%s:%d Not initialized.\n", __func__, __LINE__);
		return -EFAULT;
	}

	cancel_work_sync(&ctx->ipa_send_task);

	usbnet_ipa_flush_pend_queue(ctx);

	usbnet_ipa_free_flow_ctrl_desc(ctx);

	usbnet_ipa_teardown_sys_pipe(ctx);
	usbnet_ipa_lan_cleanup_rm();
	usbnet_ipa_cleanup_hdr(ctx);
	usbnet_ipa_deregister_interface(ctx);
	usbnet_debugfs_exit(ctx->dev);
	ctx->dev->pusbnet_ipa = NULL;
	kfree(usbnet_ipa);

	usbnet_ipa_info("%s:%d IPA Clean up end.\n", __func__, __LINE__);
	return 0;
}
#endif /* CONFIG_USBNET_IPA */

/*-------------------------------------------------------------------------
 *
 * USB Device Driver support
 *
 *-------------------------------------------------------------------------*/
static ssize_t usbnet_ipa_debugfs_read_stats(struct file *file,
			char __user *user_buf, size_t count, loff_t *ppos)
{
	struct usbnet *dev = file->private_data;
	struct usbnet_ipa_ctx *usbnet_ipa = dev->pusbnet_ipa;
	char *buf;
	unsigned int len = 0, buf_len = 4096;
	ssize_t ret_cnt;

	if (unlikely(!usbnet_ipa)) {
		pr_err("%s NULL Pointer\n", __func__);
		return -EINVAL;
	}

	buf = kzalloc(buf_len, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;

	len += scnprintf(buf + len, buf_len - len, "%25s\n",
	"USBNET IPA stats");
	len += scnprintf(buf + len, buf_len - len, "%25s\n",
	"==================================================");
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA RX Pkt Send: ", usbnet_ipa->stats.rx_ipa_send);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA RX IPA Send Fail: ", usbnet_ipa->stats.rx_ipa_send_fail);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA RX Write done: ", usbnet_ipa->stats.rx_ipa_write_done);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA RX Exception: ", usbnet_ipa->stats.rx_ipa_excep);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA TX Send: ", usbnet_ipa->stats.tx_ipa_send);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA TX Send Err: ", usbnet_ipa->stats.tx_ipa_send_err);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA RX Packet Drops: ", usbnet_ipa->stats.flow_control_pkt_drop);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA flow ctrl pkt drop ", usbnet_ipa->stats.flow_control_pkt_drop);
	len += scnprintf(buf + len, buf_len - len, "%25s %10llu\n",
	"IPA low watermark cnt ", usbnet_ipa->stats.ipa_low_watermark_cnt);
#ifdef CONFIG_USBNET_IPA
	len += scnprintf(buf + len, buf_len - len, "%25s %10d\n",
	"IPA free desc cnt ", usbnet_ipa->ipa_free_desc_cnt);
	len += scnprintf(buf + len, buf_len - len, "%25s %10d\n",
	"IPA send qlen ", usbnet_ipa->pendq_cnt);
#else /* CONFIG_USBNET_IPA */
	len += scnprintf(buf + len, buf_len - len, "%25s %10d\n",
	"IPA free desc cnt ", dev->ipa_free_desc_cnt);
	len += scnprintf(buf + len, buf_len - len, "%25s %10d\n",
	"IPA send qlen ", dev->ipa_pendq.qlen);
#endif /* CONFIG_USBNET_IPA */

	if (len > buf_len)
		len = buf_len;

	ret_cnt = simple_read_from_buffer(user_buf, count, ppos, buf, len);
	kfree(buf);

	return ret_cnt;
}

static const struct file_operations fops_usbnet_ipa_stats = {
		.read = usbnet_ipa_debugfs_read_stats,
		.open = simple_open,
		.owner = THIS_MODULE,
		.llseek = default_llseek,
};

static int usbnet_debugfs_init(struct usbnet *dev)
{
	dev->pusbnet_ipa->debugfs_dir = debugfs_create_dir("usbnet", 0);
	if (!dev->pusbnet_ipa->debugfs_dir)
		return -ENOMEM;

	debugfs_create_file("stats", S_IRUSR, dev->pusbnet_ipa->debugfs_dir,
					dev, &fops_usbnet_ipa_stats);

	return 0;
}

void usbnet_debugfs_exit(struct usbnet *dev)
{
	debugfs_remove_recursive(dev->pusbnet_ipa->debugfs_dir);
}

static void usbnet_ipa_cleanup_rm(struct usbnet *dev)
{
	int ret;

	INIT_COMPLETION(dev->rm_prod_release_comp);

	ret =  ipa_rm_release_resource(IPA_RM_RESOURCE_ODU_ADAPT_PROD);
	if (ret) {
		if (ret != EINPROGRESS)
			dev_err(&dev->udev->dev,
				"Release ODU PROD resource failed:%d\n", ret);

		ret = wait_for_completion_timeout(&dev->rm_prod_release_comp,
						  msecs_to_jiffies(
						  IPA_ODU_RM_TIMEOUT_MSEC));
		if (ret == 0)
			dev_err(&dev->udev->dev,
				"Timeout releasing ODU prod resource\n");
	}

	ret = ipa_rm_delete_resource(IPA_RM_RESOURCE_ODU_ADAPT_PROD);
	if (ret)
		pr_warn("Resource:IPA_RM_RESOURCE_ODU_ADAPT_PROD del fail %d\n",
			ret);

	ret = ipa_rm_delete_resource(IPA_RM_RESOURCE_ODU_ADAPT_CONS);
	if (ret)
		pr_warn("Resource:IPA_RM_RESOURCE_ODU_ADAPT_CONS del fail %d\n",
			ret);
}

// precondition: never called in_interrupt

void usbnet_disconnect (struct usb_interface *intf)
{
	struct usbnet		*dev;
	struct usb_device	*xdev;
	struct net_device	*net;
	int retval = 0;

	dev = usb_get_intfdata(intf);
	usb_set_intfdata(intf, NULL);
	if (!dev)
		return;

	xdev = interface_to_usbdev (intf);

#ifdef CONFIG_USBNET_IPA
	usbnet_ipa_cleanup(usbnet_ipa);
#endif /* CONFIG_USBNET_IPA */

	netif_info(dev, probe, dev->net, "unregister '%s' usb-%s-%s, %s\n",
		   intf->dev.driver->name,
		   xdev->bus->bus_name, xdev->devpath,
		   dev->driver_info->description);

	net = dev->net;
	unregister_netdev (net);

	cancel_work_sync(&dev->kevent);

	if (enable_ipa_bridge) {
		usbnet_ipa_info("%s:%d IPA Clean up start.\n", __func__, __LINE__);
		skb_queue_purge(&dev->ipa_pendq);
		retval = odu_bridge_disconnect();
		if (retval)
			dev_dbg(&dev->udev->dev,
			"%s Disconnect from ODU bridge failed.\n",
			__func__);
		retval = odu_bridge_cleanup();
		if (retval)
			dev_dbg(&dev->udev->dev,
			"%s ODU bridge cleanup failed.\n",
			__func__);
		usbnet_ipa_cleanup_rm(dev);
		usbnet_debugfs_exit(dev);
		kfree(dev->pusbnet_ipa);
		usbnet_ipa_info("%s:%d IPA Clean up end.\n", __func__, __LINE__);
	}
	usb_scuttle_anchored_urbs(&dev->deferred);

	if (dev->driver_info->unbind)
		dev->driver_info->unbind (dev, intf);

	usb_kill_urb(dev->interrupt);
	usb_free_urb(dev->interrupt);

	free_netdev(net);
}
EXPORT_SYMBOL_GPL(usbnet_disconnect);

static const struct net_device_ops usbnet_netdev_ops = {
	.ndo_open		= usbnet_open,
	.ndo_stop		= usbnet_stop,
	.ndo_start_xmit		= usbnet_start_xmit,
	.ndo_tx_timeout		= usbnet_tx_timeout,
	.ndo_change_mtu		= usbnet_change_mtu,
#if defined(CONFIG_PLATFORM_NEC)
	.ndo_set_mac_address 	= usbnet_eth_mac_addr,
#else /* defined(CONFIG_PLATFORM_NEC) */
	.ndo_set_mac_address 	= eth_mac_addr,
#endif /* defined(CONFIG_PLATFORM_NEC) */
	.ndo_validate_addr	= eth_validate_addr,
};

/*-------------------------------------------------------------------------*/

// precondition: never called in_interrupt

static struct device_type wlan_type = {
	.name	= "wlan",
};

static struct device_type wwan_type = {
	.name	= "wwan",
};

static void usbnet_ipa_rm_notify(void *user_data, enum ipa_rm_event event,
							unsigned long data)
{
	struct usbnet *dev = (struct usbnet *)user_data;

	pr_debug(" %s IPA RM Evt: %d\n", __func__, event);

	switch (event) {
	case IPA_RM_RESOURCE_GRANTED:
		complete(&dev->rm_prod_granted_comp);
		break;
	case  IPA_RM_RESOURCE_RELEASED:
		complete(&dev->rm_prod_release_comp);
		break;
	default:
		dev_dbg(&dev->udev->dev,
			"Un-expected event %d\n", event);
		break;
	}
}

static int usbnet_ipa_rm_cons_request(void)
{
	/* Do Nothing*/
	return 0;
}

static int usbnet_ipa_rm_cons_release(void)
{
	/* Do Nothing*/
	return 0;
}

static int usbnet_ipa_setup_rm(struct usbnet *dev)
{
	struct ipa_rm_create_params create_params = {0};
	int ret;

	create_params.name = IPA_RM_RESOURCE_ODU_ADAPT_PROD;
	create_params.reg_params.user_data = dev;
	create_params.reg_params.notify_cb = usbnet_ipa_rm_notify;
	create_params.floor_voltage = IPA_VOLTAGE_SVS;

	ret = ipa_rm_create_resource(&create_params);
	if (ret) {
		dev_err(&dev->udev->dev,
			"Create ODU PROD RM resource failed: %d\n", ret);
		goto prod_fail;
	}

	memset(&create_params, 0, sizeof(create_params));
	create_params.name = IPA_RM_RESOURCE_ODU_ADAPT_CONS;
	create_params.request_resource = usbnet_ipa_rm_cons_request;
	create_params.release_resource = usbnet_ipa_rm_cons_release;
	create_params.floor_voltage = IPA_VOLTAGE_SVS;

	ret = ipa_rm_create_resource(&create_params);
	if (ret) {
		dev_err(&dev->udev->dev,
			"Create ODU CONC RM resource failed: %d\n", ret);
		goto delete_prod;
	}

	INIT_COMPLETION(dev->rm_prod_granted_comp);

	ret =  ipa_rm_request_resource(IPA_RM_RESOURCE_ODU_ADAPT_PROD);
	if (ret) {
		if (ret != EINPROGRESS) {
			dev_err(&dev->udev->dev,
				"Request ODU PROD resource failed: %d\n", ret);
			goto delete_cons;
		}
		ret = wait_for_completion_timeout(&dev->rm_prod_granted_comp,
						  msecs_to_jiffies(
						  IPA_ODU_RM_TIMEOUT_MSEC));
		if (ret == 0) {
			dev_err(&dev->udev->dev,
				"timeout requesting ODU prod resource\n");
			ret = -ETIMEDOUT;
			goto delete_cons;
		}
	}

	return ret;

delete_cons:
	ipa_rm_delete_resource(IPA_RM_RESOURCE_ODU_ADAPT_CONS);
delete_prod:
	ipa_rm_delete_resource(IPA_RM_RESOURCE_ODU_ADAPT_PROD);
prod_fail:
	return ret;

}

static void usbnet_ipa_tx_dp_cb(void *priv, enum ipa_dp_evt_type evt,
							unsigned long data)
{
	struct usbnet *dev = priv;
	struct usbnet_ipa_ctx *usbnet_ipa = dev->pusbnet_ipa;
	struct sk_buff *skb = (struct sk_buff *)data;
	int status;
	u32 qlen = 0;

	switch (evt) {
	case IPA_RECEIVE:
		/* Deliver SKB to network adapter */
		usbnet_ipa->stats.rx_ipa_excep++;
		skb->dev = dev->net;
		skb->protocol = eth_type_trans(skb, skb->dev);
		status = netif_rx_ni(skb);
		if (status != NET_RX_SUCCESS)
			pr_err("ERROR sending to nw stack %d\n", status);
		break;

	case IPA_WRITE_DONE:
		/* SKB send to IPA, safe to free */
		usbnet_ipa->stats.rx_ipa_write_done++;
		dev->net->stats.rx_packets++;
		dev->net->stats.rx_bytes += skb->len;
		dev_kfree_skb(skb);
		spin_lock(&dev->ipa_pendq.lock);
		qlen = skb_queue_len(&dev->ipa_pendq);
		dev->ipa_free_desc_cnt++;
		if (qlen && dev->ipa_free_desc_cnt < dev->ipa_low_watermark)
				usbnet_ipa->stats.ipa_low_watermark_cnt++;
		spin_unlock(&dev->ipa_pendq.lock);

		if (qlen)
			queue_work(usbnet_wq, &dev->ipa_send_task);
		break;

	default:
		pr_err("%s Invalid event from IPA\n", __func__);
		break;
	}
}

static void usbnet_ipa_tx_dl(void *priv, struct sk_buff *skb)
{
	struct usbnet *dev = priv;
	struct usbnet_ipa_ctx *usbnet_ipa = dev->pusbnet_ipa;
	netdev_tx_t ret = __NETDEV_TX_MIN;

	ret = usbnet_start_xmit(skb, dev->net);

	if (ret != NETDEV_TX_OK) {
		pr_err("%s usbnet_ipa_tx_dl() failed xmit returned %d\n",
					__func__, ret);
		usbnet_ipa->stats.tx_ipa_send_err++;
		dev_kfree_skb_any(skb);
	} else {
		usbnet_ipa->stats.tx_ipa_send++;
	}
}

static int usbnet_ipa_set_perf_level(struct usbnet *dev)
{
	struct ipa_rm_perf_profile profile;
	int ret = 0;

	memset(&profile, 0, sizeof(profile));

	if (dev->udev->speed == USB_SPEED_SUPER)
		profile.max_supported_bandwidth_mbps = 800;
	else
		profile.max_supported_bandwidth_mbps = 400;

	ret = ipa_rm_set_perf_profile(IPA_RM_RESOURCE_ODU_ADAPT_PROD,
					&profile);
	if (ret) {
		pr_err("Err to set BW: IPA_RM_RESOURCE_ODU_ADAPT_PROD err:%d\n",
			ret);
		return ret;
	}

	ret = ipa_rm_set_perf_profile(IPA_RM_RESOURCE_ODU_ADAPT_CONS,
					&profile);
	if (ret) {
		pr_err("Err to set BW: IPA_RM_RESOURCE_ODU_ADAPT_CONS err:%d\n",
			ret);
		return ret;
	}

	return ret;
}

#if !defined(CONFIG_USBNET_IPA)
/* usbnet_ipa_send_routine - Sends packets to IPA/ODU bridge Driver
 * Scheduled on RX of IPA_WRITE_DONE Event
 */
static void usbnet_ipa_send_routine(struct work_struct *work)
{
	struct usbnet *dev = container_of(work,
				struct usbnet, ipa_send_task);
	struct sk_buff *skb;
	struct ipa_tx_meta ipa_meta = {0x0};
	int ret = 0;

	/* Send all pending packets to IPA.
	 * Compute the number of desc left for HW and send packets accordingly
	 */
	spin_lock(&dev->ipa_pendq.lock);
	if (dev->ipa_free_desc_cnt < dev->ipa_low_watermark) {
		dev->pusbnet_ipa->stats.ipa_low_watermark_cnt++;
		spin_unlock(&dev->ipa_pendq.lock);
		return;
	}

	while (dev->ipa_free_desc_cnt &&
	       (skb = __skb_dequeue(&dev->ipa_pendq))) {
		ipa_meta.dma_address_valid = false;
		/* Send Packet to ODU bridge Driver */
		spin_unlock(&dev->ipa_pendq.lock);
		ret = odu_bridge_tx_dp(skb, &ipa_meta);
		spin_lock(&dev->ipa_pendq.lock);
		if (ret) {
			pr_err("%s: ret %d\n", __func__, ret);
			dev_kfree_skb(skb);
			dev->pusbnet_ipa->stats.rx_ipa_send_fail++;
		} else {
			dev->pusbnet_ipa->stats.rx_ipa_send++;
			dev->ipa_free_desc_cnt--;
		}
	}
	spin_unlock(&dev->ipa_pendq.lock);
}
#endif /* !defined(CONFIG_USBNET_IPA) */

int
usbnet_probe (struct usb_interface *udev, const struct usb_device_id *prod)
{
	struct usbnet			*dev;
	struct net_device		*net;
	struct usb_host_interface	*interface;
	struct driver_info		*info;
	struct usb_device		*xdev;
	int				status;
	const char			*name;
	struct usb_driver 	*driver = to_usb_driver(udev->dev.driver);
	struct usbnet_ipa_ctx *usbnet_ipa = NULL;
	struct odu_bridge_params *params_ptr, params;
	params_ptr = &params;

	/* usbnet already took usb runtime pm, so have to enable the feature
	 * for usb interface, otherwise usb_autopm_get_interface may return
	 * failure if RUNTIME_PM is enabled.
	 */
	if (!driver->supports_autosuspend) {
		driver->supports_autosuspend = 1;
		pm_runtime_enable(&udev->dev);
	}

	name = udev->dev.driver->name;
	info = (struct driver_info *) prod->driver_info;
	if (!info) {
		dev_dbg (&udev->dev, "blacklisted by %s\n", name);
		return -ENODEV;
	}
	xdev = interface_to_usbdev (udev);
	interface = udev->cur_altsetting;

	status = -ENOMEM;

	// set up our own records
	net = alloc_etherdev(sizeof(*dev));
	if (!net)
		goto out;

	/* netdev_printk() needs this so do it as early as possible */
	SET_NETDEV_DEV(net, &udev->dev);

	dev = netdev_priv(net);
	dev->udev = xdev;
	dev->intf = udev;
	dev->driver_info = info;
	dev->driver_name = name;
	dev->msg_enable = netif_msg_init (msg_level, NETIF_MSG_DRV
				| NETIF_MSG_PROBE | NETIF_MSG_LINK);
	init_waitqueue_head(&dev->wait);
	skb_queue_head_init (&dev->rxq);
	skb_queue_head_init (&dev->txq);
	skb_queue_head_init (&dev->done);
	skb_queue_head_init(&dev->rxq_pause);
	INIT_WORK(&dev->bh_w, usbnet_bh_w);
	INIT_WORK (&dev->kevent, kevent);
	init_usb_anchor(&dev->deferred);
	dev->delay.function = usbnet_bh;
	dev->delay.data = (unsigned long) dev;
	init_timer (&dev->delay);
	mutex_init (&dev->phy_mutex);
	mutex_init(&dev->interrupt_mutex);
	dev->interrupt_count = 0;
	dev->net = net;
	strcpy (net->name, "usb%d");
	memcpy (net->dev_addr, node_id, sizeof node_id);

	/* rx and tx sides can use different message sizes;
	 * bind() should set rx_urb_size in that case.
	 */
	dev->hard_mtu = net->mtu + net->hard_header_len;
#if 0
// dma_supported() is deeply broken on almost all architectures
	// possible with some EHCI controllers
	if (dma_supported (&udev->dev, DMA_BIT_MASK(64)))
		net->features |= NETIF_F_HIGHDMA;
#endif

	net->netdev_ops = &usbnet_netdev_ops;
	net->watchdog_timeo = TX_TIMEOUT_JIFFIES;
	net->ethtool_ops = &usbnet_ethtool_ops;

	// allow device-specific bind/init procedures
	// NOTE net->name still not usable ...
	if (info->bind) {
		status = info->bind (dev, udev);
		if (status < 0)
			goto out1;

		// heuristic:  "usb%d" for links we know are two-host,
		// else "eth%d" when there's reasonable doubt.  userspace
		// can rename the link if it knows better.
		if ((dev->driver_info->flags & FLAG_ETHER) != 0 &&
		    ((dev->driver_info->flags & FLAG_POINTTOPOINT) == 0 ||
		     (net->dev_addr [0] & 0x02) == 0))
			strcpy (net->name, "eth%d");
		/* WLAN devices should always be named "wlan%d" */
		if ((dev->driver_info->flags & FLAG_WLAN) != 0)
			strcpy(net->name, "wlan%d");
		/* WWAN devices should always be named "wwan%d" */
		if ((dev->driver_info->flags & FLAG_WWAN) != 0)
			strcpy(net->name, "wwan%d");

		/* devices that cannot do ARP */
		if ((dev->driver_info->flags & FLAG_NOARP) != 0)
			net->flags |= IFF_NOARP;

		/* maybe the remote can't receive an Ethernet MTU */
		if (net->mtu > (dev->hard_mtu - net->hard_header_len))
			net->mtu = dev->hard_mtu - net->hard_header_len;
	} else if (!info->in || !info->out)
		status = usbnet_get_endpoints (dev, udev);
	else {
		dev->in = usb_rcvbulkpipe (xdev, info->in);
		dev->out = usb_sndbulkpipe (xdev, info->out);
		if (!(info->flags & FLAG_NO_SETINT))
			status = usb_set_interface (xdev,
				interface->desc.bInterfaceNumber,
				interface->desc.bAlternateSetting);
		else
			status = 0;

	}
	if (status >= 0 && dev->status)
		status = init_status (dev, udev);
	if (status < 0)
		goto out3;

	if (!dev->rx_urb_size)
		dev->rx_urb_size = dev->hard_mtu;
	dev->maxpacket = usb_maxpacket (dev->udev, dev->out, 1);

	if ((dev->driver_info->flags & FLAG_WLAN) != 0)
		SET_NETDEV_DEVTYPE(net, &wlan_type);
	if ((dev->driver_info->flags & FLAG_WWAN) != 0)
		SET_NETDEV_DEVTYPE(net, &wwan_type);

	/* initialize max rx_qlen and tx_qlen */
	usbnet_update_max_qlen(dev);

	status = register_netdev (net);
	if (status)
		goto out4;
	netif_info(dev, probe, dev->net,
		   "register '%s' at usb-%s-%s, %s, %pM\n",
		   udev->dev.driver->name,
		   xdev->bus->bus_name, xdev->devpath,
		   dev->driver_info->description,
		   net->dev_addr);

	if (enable_ipa_bridge) {
		/* Init IPA Context */
		usbnet_ipa = kzalloc(sizeof(struct usbnet_ipa_ctx), GFP_KERNEL);
		if (!usbnet_ipa) {
			pr_err("kzalloc err.\n");
			status = -ENOMEM;
			goto out4;
		}

		dev->pusbnet_ipa = usbnet_ipa;
		dev->ipa_free_desc_cnt = USBNET_IPA_SYS_PIPE_MAX_PKTS_DESC;
		dev->ipa_high_watermark = USBNET_IPA_SYS_PIPE_MAX_PKTS_DESC;
		dev->ipa_low_watermark = USBNET_IPA_SYS_PIPE_MIN_PKTS_DESC;

#if 0
		/* Initialize flow control variables */
		skb_queue_head_init(&dev->ipa_pendq);
		INIT_WORK(&dev->ipa_send_task, usbnet_ipa_send_routine);
#endif

		status = usbnet_ipa_setup_rm(dev);
		if (status) {
			pr_err("USBNET: IPA Setup RM Failed\n");
			goto out4;
		}

		status = usbnet_debugfs_init(dev);
		if (status)
			pr_err("USBNET: Debugfs Init Failed\n");

		/* Initialize the ODU bridge driver now: odu_bridge_init()*/
		params_ptr->netdev_name = net->name;
		params_ptr->priv = dev;
		params.tx_dp_notify = usbnet_ipa_tx_dp_cb;
		params_ptr->send_dl_skb = (void *)&usbnet_ipa_tx_dl;
		memcpy(params_ptr->device_ethaddr, net->dev_addr, 6);
		params_ptr->ipa_desc_size = (dev->ipa_high_watermark + 1) *
					sizeof(struct sps_iovec);

		status = odu_bridge_init(params_ptr);
		if (status) {
			pr_err("Couldnt initialize ODU_Bridge Driver\n");
			goto out4;
		}
	}

	// ok, it's ready to go.
	usb_set_intfdata (udev, dev);

	netif_device_attach (net);

	if (dev->driver_info->flags & FLAG_LINK_INTR)
		usbnet_link_change(dev, 0, 0);

	if (enable_ipa_bridge) {
		status = odu_bridge_connect();
		if (status)
			pr_err("Could not connect to ODU bridge %d\n",
				status);
		else
			usbnet_ipa_set_perf_level(dev);
	}

#ifdef CONFIG_USBNET_IPA
	usbnet_ipa_init(dev);
#endif /* CONFIG_USBNET_IPA */

	return 0;

out4:
	usb_free_urb(dev->interrupt);
out3:
	if (info->unbind)
		info->unbind (dev, udev);
out1:
	free_netdev(net);
out:
	return status;
}
EXPORT_SYMBOL_GPL(usbnet_probe);

/*-------------------------------------------------------------------------*/

/*
 * suspend the whole driver as soon as the first interface is suspended
 * resume only when the last interface is resumed
 */

int usbnet_suspend (struct usb_interface *intf, pm_message_t message)
{
	struct usbnet		*dev = usb_get_intfdata(intf);

	if (!dev->suspend_count++) {
		spin_lock_irq(&dev->txq.lock);
		/* don't autosuspend while transmitting */
		if (dev->txq.qlen && PMSG_IS_AUTO(message)) {
			dev->suspend_count--;
			spin_unlock_irq(&dev->txq.lock);
			return -EBUSY;
		} else {
			set_bit(EVENT_DEV_ASLEEP, &dev->flags);
			spin_unlock_irq(&dev->txq.lock);
		}
		/*
		 * accelerate emptying of the rx and queues, to avoid
		 * having everything error out.
		 */
		netif_device_detach (dev->net);
		usbnet_terminate_urbs(dev);
		__usbnet_status_stop_force(dev);

		/*
		 * reattach so runtime management can use and
		 * wake the device
		 */
		netif_device_attach (dev->net);
	}
	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_suspend);

int usbnet_resume (struct usb_interface *intf)
{
	struct usbnet		*dev = usb_get_intfdata(intf);
	struct sk_buff          *skb;
	struct urb              *res;
	int                     retval;

	if (!--dev->suspend_count) {
		/* resume interrupt URB if it was previously submitted */
		__usbnet_status_start_force(dev, GFP_NOIO);

		spin_lock_irq(&dev->txq.lock);
		while ((res = usb_get_from_anchor(&dev->deferred))) {

			skb = (struct sk_buff *)res->context;
			retval = usb_submit_urb(res, GFP_ATOMIC);
			if (retval < 0) {
				dev_kfree_skb_any(skb);
				usb_free_urb(res);
				usb_autopm_put_interface_async(dev->intf);
			} else {
				dev->net->trans_start = jiffies;
				__skb_queue_tail(&dev->txq, skb);
			}
		}

		smp_mb();
		clear_bit(EVENT_DEV_ASLEEP, &dev->flags);
		spin_unlock_irq(&dev->txq.lock);

		if (test_bit(EVENT_DEV_OPEN, &dev->flags)) {
			/* handle remote wakeup ASAP
			 * we cannot race against stop
			 */
			if (netif_device_present(dev->net) &&
				!timer_pending(&dev->delay) &&
				!test_bit(EVENT_RX_HALT, &dev->flags))
					rx_alloc_submit(dev, GFP_NOIO);

			if (!(dev->txq.qlen >= TX_QLEN(dev)))
				netif_tx_wake_all_queues(dev->net);
			queue_work(usbnet_wq, &dev->bh_w);
		}
	}

	if (test_and_clear_bit(EVENT_DEVICE_REPORT_IDLE, &dev->flags))
		usb_autopm_get_interface_no_resume(intf);

	return 0;
}
EXPORT_SYMBOL_GPL(usbnet_resume);

/*
 * Either a subdriver implements manage_power, then it is assumed to always
 * be ready to be suspended or it reports the readiness to be suspended
 * explicitly
 */
void usbnet_device_suggests_idle(struct usbnet *dev)
{
	if (!test_and_set_bit(EVENT_DEVICE_REPORT_IDLE, &dev->flags)) {
		dev->intf->needs_remote_wakeup = 1;
		usb_autopm_put_interface_async(dev->intf);
	}
}
EXPORT_SYMBOL(usbnet_device_suggests_idle);

/*
 * For devices that can do without special commands
 */
int usbnet_manage_power(struct usbnet *dev, int on)
{
	dev->intf->needs_remote_wakeup = on;
	return 0;
}
EXPORT_SYMBOL(usbnet_manage_power);

void usbnet_link_change(struct usbnet *dev, bool link, bool need_reset)
{
	/* update link after link is reseted */
	if (link && !need_reset)
		netif_carrier_on(dev->net);
	else
		netif_carrier_off(dev->net);

	if (need_reset && link)
		usbnet_defer_kevent(dev, EVENT_LINK_RESET);
	else
		usbnet_defer_kevent(dev, EVENT_LINK_CHANGE);
}
EXPORT_SYMBOL(usbnet_link_change);

/*-------------------------------------------------------------------------*/
static int __usbnet_read_cmd(struct usbnet *dev, u8 cmd, u8 reqtype,
			     u16 value, u16 index, void *data, u16 size)
{
	void *buf = NULL;
	int err = -ENOMEM;

	netdev_dbg(dev->net, "usbnet_read_cmd cmd=0x%02x reqtype=%02x"
		   " value=0x%04x index=0x%04x size=%d\n",
		   cmd, reqtype, value, index, size);

	if (data) {
		buf = kmalloc(size, GFP_KERNEL);
		if (!buf)
			goto out;
	}

	err = usb_control_msg(dev->udev, usb_rcvctrlpipe(dev->udev, 0),
			      cmd, reqtype, value, index, buf, size,
			      USB_CTRL_GET_TIMEOUT);
	if (err > 0 && err <= size)
		memcpy(data, buf, err);
	kfree(buf);
out:
	return err;
}

static int __usbnet_write_cmd(struct usbnet *dev, u8 cmd, u8 reqtype,
			      u16 value, u16 index, const void *data,
			      u16 size)
{
	void *buf = NULL;
	int err = -ENOMEM;

	netdev_dbg(dev->net, "usbnet_write_cmd cmd=0x%02x reqtype=%02x"
		   " value=0x%04x index=0x%04x size=%d\n",
		   cmd, reqtype, value, index, size);

	if (data) {
		buf = kmemdup(data, size, GFP_KERNEL);
		if (!buf)
			goto out;
	}

	err = usb_control_msg(dev->udev, usb_sndctrlpipe(dev->udev, 0),
			      cmd, reqtype, value, index, buf, size,
			      USB_CTRL_SET_TIMEOUT);
	kfree(buf);

out:
	return err;
}

/*
 * The function can't be called inside suspend/resume callback,
 * otherwise deadlock will be caused.
 */
int usbnet_read_cmd(struct usbnet *dev, u8 cmd, u8 reqtype,
		    u16 value, u16 index, void *data, u16 size)
{
	int ret;

	if (usb_autopm_get_interface(dev->intf) < 0)
		return -ENODEV;
	ret = __usbnet_read_cmd(dev, cmd, reqtype, value, index,
				data, size);
	usb_autopm_put_interface(dev->intf);
	return ret;
}
EXPORT_SYMBOL_GPL(usbnet_read_cmd);

/*
 * The function can't be called inside suspend/resume callback,
 * otherwise deadlock will be caused.
 */
int usbnet_write_cmd(struct usbnet *dev, u8 cmd, u8 reqtype,
		     u16 value, u16 index, const void *data, u16 size)
{
	int ret;

	if (usb_autopm_get_interface(dev->intf) < 0)
		return -ENODEV;
	ret = __usbnet_write_cmd(dev, cmd, reqtype, value, index,
				 data, size);
	usb_autopm_put_interface(dev->intf);
	return ret;
}
EXPORT_SYMBOL_GPL(usbnet_write_cmd);

/*
 * The function can be called inside suspend/resume callback safely
 * and should only be called by suspend/resume callback generally.
 */
int usbnet_read_cmd_nopm(struct usbnet *dev, u8 cmd, u8 reqtype,
			  u16 value, u16 index, void *data, u16 size)
{
	return __usbnet_read_cmd(dev, cmd, reqtype, value, index,
				 data, size);
}
EXPORT_SYMBOL_GPL(usbnet_read_cmd_nopm);

/*
 * The function can be called inside suspend/resume callback safely
 * and should only be called by suspend/resume callback generally.
 */
int usbnet_write_cmd_nopm(struct usbnet *dev, u8 cmd, u8 reqtype,
			  u16 value, u16 index, const void *data,
			  u16 size)
{
	return __usbnet_write_cmd(dev, cmd, reqtype, value, index,
				  data, size);
}
EXPORT_SYMBOL_GPL(usbnet_write_cmd_nopm);

static void usbnet_async_cmd_cb(struct urb *urb)
{
	struct usb_ctrlrequest *req = (struct usb_ctrlrequest *)urb->context;
	int status = urb->status;

	if (status < 0)
		dev_dbg(&urb->dev->dev, "%s failed with %d",
			__func__, status);

	kfree(req);
	usb_free_urb(urb);
}

/*
 * The caller must make sure that device can't be put into suspend
 * state until the control URB completes.
 */
int usbnet_write_cmd_async(struct usbnet *dev, u8 cmd, u8 reqtype,
			   u16 value, u16 index, const void *data, u16 size)
{
	struct usb_ctrlrequest *req = NULL;
	struct urb *urb;
	int err = -ENOMEM;
	void *buf = NULL;

	netdev_dbg(dev->net, "usbnet_write_cmd cmd=0x%02x reqtype=%02x"
		   " value=0x%04x index=0x%04x size=%d\n",
		   cmd, reqtype, value, index, size);

	urb = usb_alloc_urb(0, GFP_ATOMIC);
	if (!urb) {
		netdev_err(dev->net, "Error allocating URB in"
			   " %s!\n", __func__);
		goto fail;
	}

	if (data) {
		buf = kmemdup(data, size, GFP_ATOMIC);
		if (!buf) {
			netdev_err(dev->net, "Error allocating buffer"
				   " in %s!\n", __func__);
			goto fail_free;
		}
	}

	req = kmalloc(sizeof(struct usb_ctrlrequest), GFP_ATOMIC);
	if (!req)
		goto fail_free_buf;

	req->bRequestType = reqtype;
	req->bRequest = cmd;
	req->wValue = cpu_to_le16(value);
	req->wIndex = cpu_to_le16(index);
	req->wLength = cpu_to_le16(size);

	usb_fill_control_urb(urb, dev->udev,
			     usb_sndctrlpipe(dev->udev, 0),
			     (void *)req, buf, size,
			     usbnet_async_cmd_cb, req);
	urb->transfer_flags |= URB_FREE_BUFFER;

	err = usb_submit_urb(urb, GFP_ATOMIC);
	if (err < 0) {
		netdev_err(dev->net, "Error submitting the control"
			   " message: status=%d\n", err);
		goto fail_free;
	}
	return 0;

fail_free_buf:
	kfree(buf);
fail_free:
	kfree(req);
	usb_free_urb(urb);
fail:
	return err;

}
EXPORT_SYMBOL_GPL(usbnet_write_cmd_async);
/*-------------------------------------------------------------------------*/

static int __init usbnet_init(void)
{
	/* Compiler should optimize this out. */
	BUILD_BUG_ON(
		FIELD_SIZEOF(struct sk_buff, cb) < sizeof(struct skb_data));

	eth_random_addr(node_id);

	usbnet_wq  = create_singlethread_workqueue("usbnet");
	if (!usbnet_wq) {
		pr_err("%s: Unable to create workqueue:usbnet\n", __func__);
		return -ENOMEM;
	}
	return 0;
}
module_init(usbnet_init);

static void __exit usbnet_exit(void)
{
	destroy_workqueue(usbnet_wq);
}
module_exit(usbnet_exit);

MODULE_AUTHOR("David Brownell");
MODULE_DESCRIPTION("USB network driver framework");
MODULE_LICENSE("GPL");
