/* Copyright (c) 2013-2014, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/err.h>
#include <linux/io.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/cpu.h>
#include <linux/sched.h>
#include <linux/of.h>
#include <linux/of_address.h>
#if defined(CONFIG_PLATFORM_NEC)
#include <mach/board.h>
#endif /* defined(CONFIG_PLATFORM_NEC) */

struct boot_stats {
	uint32_t bootloader_start;
	uint32_t bootloader_end;
	uint32_t bootloader_display;
	uint32_t bootloader_load_kernel;
};

static void __iomem *mpm_counter_base;
static uint32_t mpm_counter_freq;
static struct boot_stats __iomem *boot_stats;

static int mpm_parse_dt(void)
{
	struct device_node *np;
	u32 freq;

	np = of_find_compatible_node(NULL, NULL, "qcom,msm-imem-boot_stats");
	if (!np) {
		pr_err("can't find qcom,msm-imem node\n");
		return -ENODEV;
	}
	boot_stats = of_iomap(np, 0);
	if (!boot_stats) {
		pr_err("boot_stats: Can't map imem\n");
		return -ENODEV;
	}

	np = of_find_compatible_node(NULL, NULL, "qcom,mpm2-sleep-counter");
	if (!np) {
		pr_err("mpm_counter: can't find DT node\n");
		return -ENODEV;
	}

	if (!of_property_read_u32(np, "clock-frequency", &freq))
		mpm_counter_freq = freq;
	else
		return -ENODEV;

	if (of_get_address(np, 0, NULL, NULL)) {
		mpm_counter_base = of_iomap(np, 0);
		if (!mpm_counter_base) {
			pr_err("mpm_counter: cant map counter base\n");
			return -ENODEV;
		}
	}

	return 0;
}

static void print_boot_stats(void)
{
	pr_info("KPI: Bootloader start count = %u\n",
		readl_relaxed(&boot_stats->bootloader_start));
	pr_info("KPI: Bootloader end count = %u\n",
		readl_relaxed(&boot_stats->bootloader_end));
	pr_info("KPI: Bootloader display count = %u\n",
		readl_relaxed(&boot_stats->bootloader_display));
	pr_info("KPI: Bootloader load kernel count = %u\n",
		readl_relaxed(&boot_stats->bootloader_load_kernel));
	pr_info("KPI: Kernel MPM timestamp = %u\n",
		readl_relaxed(mpm_counter_base));
	pr_info("KPI: Kernel MPM Clock frequency = %u\n",
		mpm_counter_freq);
}

int boot_stats_init(void)
{
	int ret;

	ret = mpm_parse_dt();
	if (ret < 0)
		return -ENODEV;

	print_boot_stats();

	iounmap(boot_stats);
	iounmap(mpm_counter_base);

	return 0;
}

#if defined(CONFIG_PLATFORM_NEC)
#define MSM_BOOT_INFO_START	(0x08600000+0xAA0)

static struct msm_boot_info *boot_info;

static int boot_info_set(const char *str, struct kernel_param *kp)
{
	int offset = (int)kp->arg;
	u32 val;

	sscanf(str, "%u", &val);

	writel_relaxed(val, (void __iomem *)((u32)msm_boot_info_get() + offset));

	return 0;
}

static int boot_info_get(char *buffer, const struct kernel_param *kp)
{
	int offset = (int)kp->arg;

	return sprintf(buffer, "0x%08x",
	    readl_relaxed((void __iomem *)((u32)msm_boot_info_get() + offset)));
}

module_param_call(boot_info_boot_flags, boot_info_set, boot_info_get,
			(void *)0, 0644);
module_param_call(boot_info_rtctime, boot_info_set, boot_info_get,
			(void *)4, 0644);
module_param_call(boot_info_systime, boot_info_set, boot_info_get,
			(void *)8, 0644);
module_param_call(boot_info_reserved2, boot_info_set, boot_info_get,
			(void *)12, 0644);

struct msm_boot_info *
msm_boot_info_get(void)
{
	if (boot_info)
		return boot_info;
	boot_info = ioremap(MSM_BOOT_INFO_START, sizeof(*boot_info));
	if (boot_info == NULL)
		panic("%s\n", __func__);
	return boot_info;
}

#define MSM_MODE_INFO_START	(0x08600000+0xAB0)
extern bool msm_warm_reset_required;

static int mode_info_set(const char *str, struct kernel_param *kp)
{
	u32 val;

	sscanf(str, "%x", &val);

	writel_relaxed(val, (void __iomem *)(msm_mode_info_get()));

	if (val & 0x80000000)
		msm_warm_reset_required = true;

	return 0;
}

static int mode_info_get(char *buffer, const struct kernel_param *kp)
{
	return sprintf(buffer, "0x%08x", readl_relaxed((void __iomem *)(msm_mode_info_get())));
}

static uint32_t *mode_info;
module_param_call(mode_info, mode_info_set, mode_info_get,
			NULL, 0644);

uint32_t *
msm_mode_info_get(void)
{
	if (mode_info)
		return mode_info;
	mode_info = ioremap(MSM_MODE_INFO_START, sizeof(*mode_info));
	if (mode_info == NULL)
		panic("%s\n", __func__);
	return mode_info;
}
#endif /* defined(CONFIG_PLATFORM_NEC) */
