/* Copyright (c) 2014, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013 NEC AccessTechinica, Ltd., All rights reserved.
 * Copyright (c) 2014 - 2015, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>
#include <linux/qpnp/pin.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/leds.h>
#include <linux/regulator/consumer.h>
#include <linux/dma-mapping.h>
#include <linux/uaccess.h>

#include <linux/msm-sps.h>

#include "mdss.h"
#include "mdss_panel.h"
#include "mdss_qpic.h"
#include "mdss_qpic_panel.h"

#ifdef CONFIG_FB_MSM_NECPF_KITTING_IMG
#include "necpf_kitting_img.h"
#endif /* CONFIG_FB_MSM_NECPF_KITTING_IMG */

#define QPICP_DEBUG (0)
#if QPICP_DEBUG
#define QPICP_DPRINT(fmt, ...) pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define QPICP_DPRINT(fmt, ...)
#endif /* QPICP_DEBUG */

static u32 fb_used = 0;
static u32 panel_is_on = false;
static u32 display_is_on = false;
static u32 panel_refresh_rate;

#ifdef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
static struct mutex fb_used_mutex = __MUTEX_INITIALIZER(fb_used_mutex);

static int mdss_qpic_panel_wait_next_signal(void *te_base,
						u32 signal, u32 timeout);
#endif /* CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

static int (*qpic_panel_on)(struct qpic_panel_io_desc *qpic_panel_io);
static int (*qpic_display_on)(int lightup);
static void (*qpic_display_off)(void);
static void (*qpic_panel_off)(struct qpic_panel_io_desc *qpic_panel_io);

static int mdss_qpic_pinctrl_init(struct platform_device *pdev,
		struct qpic_panel_io_desc *qpic_panel_io);

#if 1
static void qpic_panel_bites_image_init(struct platform_device *, struct mdss_panel_data *);
#endif

u32 qpic_panel_get_framerate(void)
{
	return panel_refresh_rate;
}

/* write a frame of pixels to a MIPI screen */
u32 qpic_send_frame(u32 x_start,
				u32 y_start,
				u32 x_end,
				u32 y_end,
				u32 *data,
				u32 total_bytes)
{
	u8 param[4];
	u32 status;
	u32 start_0_7;
	u32 end_0_7;
	u32 start_8_15;
	u32 end_8_15;

	if (!panel_is_on) {
		pr_err("%s Illegal sequence occurred.\n", __func__);
		return 0;
	}

	/* convert to 16 bit representation */
	x_start = x_start & 0xffff;
	y_start = y_start & 0xffff;
	x_end = x_end & 0xffff;
	y_end = y_end & 0xffff;

	/* set column/page */
	start_0_7 = x_start & 0xff;
	end_0_7 = x_end & 0xff;
	start_8_15 = (x_start >> 8) & 0xff;
	end_8_15 = (x_end >> 8) & 0xff;
	param[0] = start_8_15;
	param[1] = start_0_7;
	param[2] = end_8_15;
	param[3] = end_0_7;
	status = qpic_send_pkt(OP_SET_COLUMN_ADDRESS, param, 4);
	if (status) {
		pr_err("Failed to set column address");
		return status;
	}

	start_0_7 = y_start & 0xff;
	end_0_7 = y_end & 0xff;
	start_8_15 = (y_start >> 8) & 0xff;
	end_8_15 = (y_end >> 8) & 0xff;
	param[0] = start_8_15;
	param[1] = start_0_7;
	param[2] = end_8_15;
	param[3] = end_0_7;
	status = qpic_send_pkt(OP_SET_PAGE_ADDRESS, param, 4);
	if (status) {
		pr_err("Failed to set page address");
		return status;
	}

	status = qpic_send_pkt(OP_WRITE_MEMORY_START, (u8 *)data, total_bytes);
	if (status) {
		pr_err("Failed to start memory write");
		return status;
	}
	return 0;
}

static int mdss_qpic_pinctrl_init(struct platform_device *pdev,
		struct qpic_panel_io_desc *qpic_panel_io)
{
	qpic_panel_io->pin_res.pinctrl = devm_pinctrl_get(&pdev->dev);
	if (IS_ERR_OR_NULL(qpic_panel_io->pin_res.pinctrl)) {
		pr_err("%s: failed to get pinctrl\n", __func__);
		return PTR_ERR(qpic_panel_io->pin_res.pinctrl);
	}

	qpic_panel_io->pin_res.gpio_state_active
		= pinctrl_lookup_state(qpic_panel_io->pin_res.pinctrl,
				MDSS_PINCTRL_STATE_DEFAULT);
	if (IS_ERR_OR_NULL(qpic_panel_io->pin_res.gpio_state_active))
		pr_warn("%s: cannot get default pinstate\n", __func__);

	qpic_panel_io->pin_res.gpio_state_suspend
		= pinctrl_lookup_state(qpic_panel_io->pin_res.pinctrl,
				MDSS_PINCTRL_STATE_SLEEP);
	if (IS_ERR_OR_NULL(qpic_panel_io->pin_res.gpio_state_suspend))
		pr_warn("%s: cannot get sleep pinstate\n", __func__);

	return 0;
}

int mdss_qpic_panel_on(struct mdss_panel_data *pdata,
	struct qpic_panel_io_desc *panel_io)
{
	int rc = 0;

	if (panel_is_on) {
		QPICP_DPRINT("ignore it.\n");
		return 0;
	}
	mdss_qpic_init();

	if (qpic_panel_on) {
		rc = qpic_panel_on(panel_io);
		if (rc) {
			pr_err("%s: failed to panel on\n" , __func__);
			return rc;
		}
	}

	panel_is_on = true;
	QPICP_DPRINT("done\n");
	return 0;
}

int mdss_qpic_display_on(void)
{
	return mdss_qpic_display_on_with_backlight(1);
}

int mdss_qpic_display_on_with_backlight(int lightup)
{
	if (!panel_is_on) {
		pr_err("%s Illegal sequence occurred.\n", __func__);
		return -1;
	}

	if (display_is_on) {
		QPICP_DPRINT("ignore it.\n");
		return 0;
	}

	if (qpic_display_on) {
		if (qpic_display_on(lightup) < 0) {
			pr_err("%s: failed to display on\n" , __func__);
			return -1;
		}
	}

	display_is_on = true;
	QPICP_DPRINT("done\n");
	return 0;
}

void mdss_qpic_display_off(void)
{
	if (!display_is_on) {
		QPICP_DPRINT("ignore it.\n");
		return;
	}

	if (qpic_display_off)
		qpic_display_off();

	display_is_on = false;
	QPICP_DPRINT("done\n");
}

int mdss_qpic_panel_off(struct mdss_panel_data *pdata,
	struct qpic_panel_io_desc *panel_io)
{
	if (!panel_is_on) {
		QPICP_DPRINT("ignore it.\n");
		return 0;
	}

	mdss_qpic_display_off();

	if (qpic_panel_off)
		qpic_panel_off(panel_io);

	panel_is_on = false;
	QPICP_DPRINT("done\n");
	return 0;
}

u32 mdss_qpic_panel_is_panel_on(void)
{
	return panel_is_on;
}

#ifdef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
static int mdss_qpic_panel_wait_next_signal(void *te_base,
						u32 signal, u32 timeout)
{
	u32 reg_val = 0;

	do {
		reg_val = readl_relaxed((te_base +
					QPIC_PANEL_TLMM_TE_GPIO_IN_OUT));
		if (signal == reg_val)
			break;

		if (timeout < (u32)ktime_to_ms(ktime_get())) {
			pr_err("%s te signal sync check timeout\n", __func__);
			return -1;
		}
		usleep(QPIC_PANEL_TLMM_TE_POLLING_TIME_US);
	} while(1);

	return 0;
}

int mdss_qpic_panel_wait_next_panel_vsync(void *te_base)
{
	u32 err;
	u32 timeout;
	u32 reg_val = 0;

	if (!display_is_on) {
		QPICP_DPRINT("ignore it.\n");
		return 0;
	}

	timeout = (u32)ktime_to_ms(ktime_get()) +
			QPIC_PANEL_TLMM_TE_POLLING_TIMEOUT_MS;

	reg_val = readl_relaxed((te_base + QPIC_PANEL_TLMM_TE_GPIO_IN_OUT));
	if (QPIC_PANEL_TLMM_TE_HIGH == reg_val) {
		err = mdss_qpic_panel_wait_next_signal(te_base,
					QPIC_PANEL_TLMM_TE_LOW, timeout);
		if (err)
			return err;
	}

	err = mdss_qpic_panel_wait_next_signal(te_base,
					QPIC_PANEL_TLMM_TE_HIGH, timeout);

	return err;
}
#endif /* CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

int mdss_qpic_panel_io_init(struct platform_device *pdev,
	struct qpic_panel_io_desc *qpic_panel_io)
{
	int rc = 0;
	struct device_node *np = pdev->dev.of_node;
#ifdef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
	int cs_gpio, te_gpio, rst_gpio;
#else
	int rst_gpio, cs_gpio, te_gpio, ad8_gpio, bl_gpio;
	struct regulator *vdd_vreg;
	struct regulator *avdd_vreg;
#endif /* CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

	rc = mdss_qpic_pinctrl_init(pdev, qpic_panel_io);
	if (rc)
		pr_warn("%s: failed to get pin resources\n", __func__);

	rst_gpio = of_get_named_gpio(np, "qcom,rst-gpio", 0);
	cs_gpio = of_get_named_gpio(np, "qcom,cs-gpio", 0);
	te_gpio = of_get_named_gpio(np, "qcom,te-gpio", 0);
#ifndef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
	ad8_gpio = of_get_named_gpio(np, "qcom,ad8-gpio", 0);
	bl_gpio = of_get_named_gpio(np, "qcom,bl-gpio", 0);
#endif /* !CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

	if (!gpio_is_valid(rst_gpio))
		pr_warn("%s: reset gpio not specified\n" , __func__);
	else
		qpic_panel_io->rst_gpio = rst_gpio;

	if (!gpio_is_valid(cs_gpio))
		pr_warn("%s: cs gpio not specified\n", __func__);
	else
		qpic_panel_io->cs_gpio = cs_gpio;

	if (!gpio_is_valid(te_gpio))
		pr_warn("%s: te gpio not specified\n", __func__);
	else
		qpic_panel_io->te_gpio = te_gpio;

#ifdef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
	qpic_panel_io->ad8_gpio  = 0;    /* no use */
	qpic_panel_io->bl_gpio   = 0;    /* no use */
	qpic_panel_io->vdd_vreg  = NULL; /* no use */
	qpic_panel_io->avdd_vreg = NULL; /* no use */

	QPICP_DPRINT("gpio no: cs=%d/te=%d/rst=%d\n",
			qpic_panel_io->cs_gpio,
			qpic_panel_io->te_gpio,
			qpic_panel_io->rst_gpio);
#else /* CONFIG_FB_MSM_QPIC_ILI_QVGA_PANEL */
	if (!gpio_is_valid(ad8_gpio))
		pr_warn("%s: ad8 gpio not specified\n", __func__);
	else
		qpic_panel_io->ad8_gpio = ad8_gpio;

	if (!gpio_is_valid(bl_gpio))
		pr_warn("%s: te gpio not specified\n", __func__);
	else
		qpic_panel_io->bl_gpio = bl_gpio;

	vdd_vreg = devm_regulator_get(&pdev->dev, "vdd");
	if (IS_ERR(vdd_vreg))
		pr_err("%s could not get vdd,", __func__);
	else
		qpic_panel_io->vdd_vreg = vdd_vreg;

	avdd_vreg = devm_regulator_get(&pdev->dev, "avdd");
	if (IS_ERR(avdd_vreg))
		pr_err("%s could not get avdd,", __func__);
	else
		qpic_panel_io->avdd_vreg = avdd_vreg;
#endif /* CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

	return 0;
}

static int mdss_panel_parse_dt(struct platform_device *pdev,
			       struct mdss_panel_data *panel_data)
{
	struct device_node *np = pdev->dev.of_node;
	u32 res[6], tmp;
	int rc;

	rc = of_property_read_u32_array(np, "qcom,mdss-pan-res", res, 2);
	if (rc) {
		pr_err("%s:%d, panel resolution not specified\n",
						__func__, __LINE__);
		return -EINVAL;
	}
	panel_data->panel_info.xres = (!rc ? res[0] : 240);
	panel_data->panel_info.yres = (!rc ? res[1] : 320);
	rc = of_property_read_u32(np, "qcom,mdss-pan-bpp", &tmp);
	if (rc) {
		pr_err("%s:%d, panel bpp not specified\n",
						__func__, __LINE__);
		return -EINVAL;
	}
	panel_data->panel_info.bpp = (!rc ? tmp : 24);
	of_property_read_u32(np, "qcom,refresh_rate", &panel_refresh_rate);

	panel_data->panel_info.type = EBI2_PANEL;
	panel_data->panel_info.pdest = DISPLAY_1;

	QPICP_DPRINT("xres=%d/yres=%d/bpp=%d/refreshrate=%d\n",
			panel_data->panel_info.xres,
			panel_data->panel_info.yres,
			panel_data->panel_info.bpp,
			panel_refresh_rate);

	return rc;
}

static int mdss_qpic_panel_probe(struct platform_device *pdev)
{
	int rc = 0;
	static struct mdss_panel_data vendor_pdata;
	static const char *panel_name;

	QPICP_DPRINT("device id=%d\n", pdev->id);
	if (!pdev->dev.of_node)
		return -ENODEV;

	panel_name = of_get_property(pdev->dev.of_node, "label", NULL);
	if (!panel_name)
		pr_info("%s:%d, panel name not specified\n",
						__func__, __LINE__);
	else
		pr_info("%s: Panel Name = %s\n", __func__, panel_name);

	rc = mdss_panel_parse_dt(pdev, &vendor_pdata);
	if (rc)
		return rc;

#ifdef CONFIG_FB_MSM_QPIC_PANEL_OTM3221B
	qpic_panel_on = otm3221b_panel_on;
	qpic_display_on = otm3221b_display_on;
	qpic_display_off = otm3221b_display_off;
	qpic_panel_off = otm3221b_panel_off;
#else /* CONFIG_FB_MSM_QPIC_ILI_QVGA_PANEL */
	/* select panel according to label */
	if (panel_name && !strcmp(panel_name, "ili qvga lcdc panel")) {
		qpic_panel_on = ili9341_on;
		qpic_panel_off = ili9341_off;
	} else {
		/* select default panel driver */
		pr_info("%s: select default panel driver\n", __func__);
		qpic_panel_on = ili9341_on;
		qpic_panel_off = ili9341_off;
	}
	qpic_display_on = NULL;
	qpic_display_off = NULL;
#endif /* CONFIG_FB_MSM_QPIC_PANEL_OTM3221B */

	rc = qpic_register_panel(&vendor_pdata);
	if (rc)
		return rc;

#if 1
	qpic_panel_bites_image_init(pdev, &vendor_pdata);
#endif
	QPICP_DPRINT("done\n");

	return 0;
}

#if 1
#include <linux/kthread.h>

#include <mach/board.h>
#include <soc/qcom/socinfo.h>

#include "bites_image.h"

enum writemode {
	WRITEMODE_DIRECT,
	WRITEMODE_BUFFER
};

static enum writemode writemode = WRITEMODE_DIRECT;
static u16 display_buffer[240*320];

static volatile u32 panel_disp_flag = 0;

#define PANEL_DISP_ON_BIT		(0)
#define PANEL_DISP_ON_MASK		(0x1)
  #define PANEL_STATE_DISP_ON		(0x1)

#define PANEL_DISP_BOOT_BIT		(1)
#define PANEL_DISP_BOOT_MASK		(0x1)
  #define PANEL_STATE_BOOTING		(0x1)

#define PANEL_DISP_VERTICAL_BIT		(2)
#define PANEL_DISP_VERTICAL_MASK	(0x1)
  #define PANEL_STATE_HORIZON		(0x0)
  #define PANEL_STATE_VERTICAL		(0x1)

#define PANEL_DISP_SLEEP_BIT		(4)
#define PANEL_DISP_SLEEP_MASK		(0x1)
  #define PANEL_STATE_RESUME		(0x0)
  #define PANEL_STATE_SUSPEND		(0x1)
#define PANEL_DISP_WAKEONBT_BIT		(5)
#define PANEL_DISP_WAKEONBT_MASK	(0x1)
  #define PANEL_STATE_WAKEONBT_OFF	(0x0)
  #define PANEL_STATE_WAKEONBT_ON	(0x1)
#define PANEL_DISP_PWOFF_BIT		(6)
#define PANEL_DISP_PWOFF_MASK		(0x1)
  #define PANEL_STATE_POWEROFF		(0x1)
#define PANEL_DISP_BATTERY_BIT		(8)
#define PANEL_DISP_BATTERY_MASK		(0x7f)
#define PANEL_DISP_CHARGE_BIT		(15)
#define PANEL_DISP_CHARGE_MASK		(0x1)
  #define PANEL_STATE_CHARGE_DISABLE	(0x0)
  #define PANEL_STATE_CHARGE_ENABLE	(0x1)

#define PANEL_DISP_CARRIER_SW_BIT	(16)
#define PANEL_DISP_CARRIER_SW_MASK	(0x1)
  #define PANEL_STATE_CARRIER_SW_EXEC	(0x1)


#define GET_PANEL_DISP_INFO(name)	\
	((panel_disp_flag >> PANEL_DISP_##name##_BIT) & PANEL_DISP_##name##_MASK)
#define SET_PANEL_DISP_INFO(name, val)					\
	panel_disp_flag |= ((val & PANEL_DISP_##name##_MASK) << PANEL_DISP_##name##_BIT)

static struct panel_softc {
	wait_queue_head_t wait_q;
	wait_queue_head_t sleep_wait_q;
	int wait_condition;
	struct mutex display_mutex;
	atomic_t sleeped;
} panel_sc;

static void bites_display_init(struct platform_device *pdev, struct mdss_panel_data *pdata);
static struct platform_device *bb_pdev;

static struct task_struct *bites_display_polling = NULL;
static void *bb_virt;
static u32 bb_phys;

extern struct qpic_data_type *qpic_res;
extern int msm_qpic_bus_set_vote(u32);

#include <linux/necpf_panel.h>
static volatile int msm_nand_init_comp = 0;

void necpf_qpic_panel_disp_allow(void)
{
	if (msm_nand_init_comp)
		return;
	mutex_lock(&panel_sc.display_mutex);
	msm_nand_init_comp = 1;
	mutex_unlock(&panel_sc.display_mutex);
	pr_info("@@@@@@@@@@@@@@ %s panel disp start\n", __func__);
}

static ssize_t qpic_panel_mode_info_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	uint32_t *mode_info;
	mode_info = msm_mode_info_get();
	return sprintf(buf, "0x%x\n", *mode_info);
}

static ssize_t qpic_panel_mode_info_store(struct device *dev,
	 struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	uint32_t *mode_info;

	mode_info = msm_mode_info_get();
	*mode_info = val;

	return count;
}

static DEVICE_ATTR(mode_info, S_IRUGO|S_IWUGO, qpic_panel_mode_info_show,
	qpic_panel_mode_info_store);


static ssize_t qpic_panel_sysfs_show_flag(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "0x%x\n", panel_disp_flag);
}

static ssize_t qpic_panel_sysfs_set_flag(struct device *dev,
	 struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	panel_disp_flag = val;
	sc->wait_condition = 1;
	mutex_unlock(&panel_sc.display_mutex);
	wake_up(&sc->wait_q);
	if ((val >> PANEL_DISP_ON_BIT) & PANEL_DISP_ON_MASK)
		return count;	/* _P[X */

	/* P[X */
	pr_info("@@@@@@@@@@@@@@ %s pre wait_event\n", __func__);
	wait_event(sc->sleep_wait_q, atomic_read(&sc->sleeped) == 1);
	pr_info("@@@@@@@@@@@@@@@ %s after wait_event\n", __func__);

	return count;
}

static DEVICE_ATTR(ctl_flag, S_IRUGO|S_IWUGO, qpic_panel_sysfs_show_flag,
	qpic_panel_sysfs_set_flag);

static struct attribute *fs_attrs[] = {
	&dev_attr_mode_info.attr,
	&dev_attr_ctl_flag.attr,
	NULL,
};

static struct attribute_group fs_attr_group = {
	.attrs = fs_attrs,
};

void mdss_qpic_panel_wait_anim_sleep(void)
{
	struct panel_softc *sc = &panel_sc;
	wait_event(sc->sleep_wait_q, atomic_read(&sc->sleeped) == 1);
	mdss_qpic_panel_set_fb_used(1);
	usleep(QPIC_PANEL_OFF_WAIT_US);
}

void mdss_qpic_panel_set_fb_used(u32 used)
{
	mutex_lock(&fb_used_mutex);
	fb_used = used;
	mutex_unlock(&fb_used_mutex);
}

static u32 mdss_qpic_panel_get_fb_used(void)
{
	u32 used;
	mutex_lock(&fb_used_mutex);
	used = fb_used;
	mutex_unlock(&fb_used_mutex);
	return used;
}

static void mdss_qpic_panel_wait_fb_off(void)
{
	u32 timeout;
	timeout = (u32)ktime_to_ms(ktime_get()) + QPIC_PANEL_OFF_TIMEOUT_MS;

	do {
		if (!mdss_qpic_panel_get_fb_used())
			break;

		if (timeout < (u32)ktime_to_ms(ktime_get())) {
			pr_err("%s: timed out\n", __func__);
			break;
		}
		usleep(QPIC_PANEL_OFF_WAIT_US);
	} while(1);

	usleep(QPIC_PANEL_OFF_WAIT_US);
	return;
}

static int qpic_panel_sysfs_add(struct device *dev)
{
	int retval;
	u32 boot_flag = get_msm_boot_flag();

	retval = sysfs_create_group(&dev->kobj, &fs_attr_group);

	mutex_lock(&panel_sc.display_mutex);
	if ((GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) ||
	    ((GET_BOOT_FLAG_INFO(boot_flag, TEST_MODE)
	      == BOOT_FLAG_TEST_MODE_ENABLE))
	    ) {
		panel_disp_flag = 0;
	} else {
		SET_PANEL_DISP_INFO(ON, PANEL_STATE_DISP_ON);
		SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	}
	mutex_unlock(&panel_sc.display_mutex);

	return retval;
}

static void
bites_convert_frame_data(const u16 *from, u32 *to,
    u32 pix_len)
{
	int i;

	for (i = 0; i < pix_len; i++, from++, to++) {
		*to = ((u32)(*from & 0xF800) << 7) |
		    ((u32)(*from & 0x0700) << 5) |
		    ((u32)(*from & 0x00C0) << 4) |
		    ((u32)(*from & 0x003F) << 2);
	}
}

static void set_display_writemode(enum writemode mode)
{
	writemode = mode;
}

static enum writemode get_display_writemode(void)
{
	return writemode;
}

static void flush_displaybuffer(void)
{
	bites_convert_frame_data(display_buffer, bb_virt, 320*240);

	/* Panel power on, if need. */
	mdss_qpic_panel_on(qpic_res->panel_data, &qpic_res->panel_io);

	msm_qpic_bus_set_vote(1);

	mdss_qpic_panel_wait_next_panel_vsync(qpic_res->te_base);

	qpic_send_frame(0, 0, 319, 239,	(u32 *)bb_phys, 320*240*4);

	msm_qpic_bus_set_vote(0);

	/* Panel display on, if need. Error is no care here. */
	mdss_qpic_display_on();
}

static void display_reset(struct display_info_t *info)
{
	int bit32_len;

	if (info == NULL)
		return;
	if ((info->image == NULL) || (info->coord == NULL))
		return;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->image->height; y++)
			for (x = 0; x < info->image->width; x++)
				display_buffer[320*(info->coord->y_start + y) + info->coord->x_start + x] = 0;
	}
	else {
		bit32_len = info->image->width * info->image->height * 4;
		memset(bb_virt, 0, bit32_len);

		/* Panel power on, if need. */
		mdss_qpic_panel_on(qpic_res->panel_data, &qpic_res->panel_io);

		msm_qpic_bus_set_vote(1);

		mdss_qpic_panel_wait_next_panel_vsync(qpic_res->te_base);

		qpic_send_frame(info->coord->x_start, info->coord->y_start,
				info->coord->x_end, info->coord->y_end,
				(u32 *)bb_phys, bit32_len);

		msm_qpic_bus_set_vote(0);

		/* Panel display on, if need. Error is no care here. */
		mdss_qpic_display_on();
	}
}

static void display_all_reset(void)
{
	struct display_info_t disp_info;
	struct image_info_t image;
	struct coordinate_info_t coord_info;

	image.width = 320;
	image.height = 240;
	coord_info.x_start = 0;
	coord_info.y_start = 0;
	coord_info.x_end = 319;
	coord_info.y_end = 239;

	disp_info.image = &image;
	disp_info.coord = &coord_info;

	display_reset(&disp_info);
}

static void display_picto(struct image_info_t *info,
			  struct coordinate_info_t *coord_info, const u16 *data)
{
	int pixel_len = 0;
	int bit32_len = 0;

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		int x, y;
		for (y = 0; y < info->height; y++) {
			for (x = 0; x < info->width; x++) {
				display_buffer[320*(coord_info->y_start + y) + coord_info->x_start + x]
					= data[info->width * y + x];
			}
		}
	}
	else {
		pixel_len = info->width * info->height;
		bit32_len = pixel_len * 4;

		bites_convert_frame_data(data, bb_virt, pixel_len);

		/* Panel power on, if need. */
		mdss_qpic_panel_on(qpic_res->panel_data, &qpic_res->panel_io);

		msm_qpic_bus_set_vote(1);

		mdss_qpic_panel_wait_next_panel_vsync(qpic_res->te_base);

		qpic_send_frame(coord_info->x_start, coord_info->y_start,
		    coord_info->x_end, coord_info->y_end,
		    (u32 *)bb_phys, bit32_len);

		msm_qpic_bus_set_vote(0);

		/* Panel display on, if need. Error is no care here. */
		mdss_qpic_display_on();
	}
}

static void display_animation_picto(int index, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	if (!vertical) {
		/* horizontal */
		coord_info = &coord_info_horizontal[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
	} else {
		/* vertical */
		coord_info = &coord_info_vertical[ANIME_PICTO];
		info = (struct image_info_t *)&anime_data[index];
	}

	data = (u16 *)anime_data[index].data;
	display_picto(info, coord_info, data);
}

static void display_sim_switch(int vertical)
{
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	u16 *data;
	u32 switch_type = GET_BOOT_FLAG_INFO(get_msm_boot_flag(),
					     SWAP_SIM_SELECT);

	if (!vertical) {
		if (switch_type) {
			info = (struct image_info_t *)&SIM_Fm2To1_h;
			data = (u16 *)SIM_Fm2To1_h.data;
		} else {
			info = (struct image_info_t *)&SIM_Fm1To2_h;
			data = (u16 *)SIM_Fm1To2_h.data;
		}
		coord_info = &coord_info_horizontal[SIM_SWITCH];
	} else {
		if (switch_type) {
			info = (struct image_info_t *)&SIM_Fm2To1_v;
			data = (u16 *)SIM_Fm2To1_v.data;
		} else {
			info = (struct image_info_t *)&SIM_Fm1To2_v;
			data = (u16 *)SIM_Fm1To2_v.data;
		}
		coord_info = &coord_info_vertical[SIM_SWITCH];
	}
	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();
	display_picto(info, coord_info, data);
	flush_displaybuffer();
	set_display_writemode(WRITEMODE_DIRECT);
}

#ifdef CONFIG_FB_MSM_NECPF_KITTING_IMG
static int display_kitting_boot_img(int vertical, u32 boot_flag)
{
	struct panel_softc *sc = &panel_sc;
	struct image_info_t info = { 320, 240, NULL };
	struct coordinate_info_t coord_info = { 0, 0, 319, 239 };
	u16 *data = NULL;

	mutex_lock(&panel_sc.display_mutex);
	while (!msm_nand_init_comp) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		wait_event_timeout(sc->wait_q, sc->wait_condition,
				   msecs_to_jiffies(500));
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);

	mutex_lock(&panel_sc.display_mutex);
	data = (u16 *)kitting_disp_boot_image(vertical);
	mutex_unlock(&panel_sc.display_mutex);
	if (data == NULL)
		return -1;

	set_display_writemode(WRITEMODE_DIRECT);
	if (GET_BOOT_FLAG_INFO(boot_flag, ATERM_INHERIT)
	    != BOOT_FLAG_ATERM_INHERIT) {
		display_all_reset();
	}
	display_picto(&info, &coord_info, data);

	mutex_lock(&panel_sc.display_mutex);
	SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	while (GET_PANEL_DISP_INFO(ON) &&
	      GET_PANEL_DISP_INFO(BOOT)) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		wait_event_timeout(sc->wait_q, sc->wait_condition,
				   msecs_to_jiffies(500));
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);

	display_all_reset();

	atomic_set(&sc->sleeped, 1);
	wake_up(&sc->sleep_wait_q);
	return 0;
}
#endif /* CONFIG_FB_MSM_NECPF_KITTING_IMG */

static void display_square_animation(int vertical, int msec)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;

	mutex_lock(&panel_sc.display_mutex);
	while (GET_PANEL_DISP_INFO(ON) &&
		GET_PANEL_DISP_INFO(BOOT)) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		display_animation_picto(i%2, vertical);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		wait_event_timeout(sc->wait_q, sc->wait_condition,
					msecs_to_jiffies(msec));
		i++;
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static void display_boot_animation(int vertical, int msec)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	u16 *data;
	u32 boot_flag = get_msm_boot_flag();

	if (GET_BOOT_FLAG_INFO(boot_flag,SWAP_SIM)
	    == BOOT_FLAG_SWAP_SIM_DISP) {

		display_sim_switch(vertical);

		mutex_lock(&panel_sc.display_mutex);
		/* 500ms * 10 */
		while ((GET_PANEL_DISP_INFO(ON) &&
			GET_PANEL_DISP_INFO(BOOT)) && i < 10) {
			mutex_unlock(&panel_sc.display_mutex);
			wait_event_timeout(sc->wait_q, sc->wait_condition,
					   msecs_to_jiffies(500));
			i++;
			mutex_lock(&panel_sc.display_mutex);
		}
		mutex_unlock(&panel_sc.display_mutex);
	}

	if (GET_BOOT_FLAG_INFO(boot_flag, ATERM_DISP)
	    == BOOT_FLAG_ATERM_DISP) {
#ifdef CONFIG_FB_MSM_NECPF_KITTING_IMG
		if (!display_kitting_boot_img(vertical, boot_flag))
			return;
#endif /* CONFIG_FB_MSM_NECPF_KITTING_IMG */
		/* Aterm S */
		if (!vertical) {
			info = (struct image_info_t *)&aterm_logo_h;
			coord_info = &coord_info_horizontal[ATERM_LOGO];
			data = (u16 *)aterm_logo_h.data;
		} else {
			info = (struct image_info_t *)&aterm_logo_v;
			coord_info = &coord_info_vertical[ATERM_LOGO];
			data = (u16 *)aterm_logo_v.data;
		}
		set_display_writemode(WRITEMODE_BUFFER);
		if (GET_BOOT_FLAG_INFO(boot_flag, ATERM_INHERIT)
		    != BOOT_FLAG_ATERM_INHERIT) {
			display_all_reset();
		}
		display_picto(info, coord_info, data);
	}

	display_square_animation(vertical, msec);

	display_all_reset();
}

static void display_carrier_switch(int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();

	if (!vertical) {
		info = (struct image_info_t *)&carrier_str_h;
		coord_info = &coord_info_horizontal[CARRIER_STR];
		data = (u16 *)carrier_str_h.data;
	} else {
		info = (struct image_info_t *)&carrier_str_v;
		coord_info = &coord_info_vertical[CARRIER_STR];
		data = (u16 *)carrier_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (!vertical) {
		info = (struct image_info_t *)&switch_str_h;
		coord_info = &coord_info_horizontal[SWITCH_STR];
		data = (u16 *)switch_str_h.data;
	} else {
		info = (struct image_info_t *)&switch_str_v;
		coord_info = &coord_info_vertical[SWITCH_STR];
		data = (u16 *)switch_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (!vertical) {
		info = (struct image_info_t *)&reboot_str_h;
		coord_info = &coord_info_horizontal[REBOOT_STR];
		data = (u16 *)reboot_str_h.data;
	} else {
		info = (struct image_info_t *)&reboot_str_v;
		coord_info = &coord_info_vertical[REBOOT_STR];
		data = (u16 *)reboot_str_v.data;
	}
	display_picto(info, coord_info, data);

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		flush_displaybuffer();
		set_display_writemode(WRITEMODE_DIRECT);
	}
}

static void display_resume(void)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	int vertical;

	mutex_lock(&panel_sc.display_mutex);
#ifdef CONFIG_ROTATE_DISPLAY
	vertical = GET_PANEL_DISP_INFO(VERTICAL);
#else
	vertical = 0;
#endif /* CONFIG_ROTATE_DISPLAY */
	mutex_unlock(&panel_sc.display_mutex);

#ifdef CONFIG_FB_MSM_NECPF_KITTING_IMG
	if (!display_kitting_boot_img(vertical, 0))
		return;
#endif /* CONFIG_FB_MSM_NECPF_KITTING_IMG */

	if (!vertical) {
		info = (struct image_info_t *)&aterm_logo_h;
		coord_info = &coord_info_horizontal[ATERM_LOGO];
		data = (u16 *)aterm_logo_h.data;
	} else {
		info = (struct image_info_t *)&aterm_logo_v;
		coord_info = &coord_info_vertical[ATERM_LOGO];
		data = (u16 *)aterm_logo_v.data;
	}

	set_display_writemode(WRITEMODE_BUFFER);
	display_all_reset();
	display_picto(info, coord_info, data);

	mutex_lock(&panel_sc.display_mutex);
	SET_PANEL_DISP_INFO(BOOT, PANEL_STATE_BOOTING);
	mutex_unlock(&panel_sc.display_mutex);

	display_square_animation(vertical, 500);
	display_all_reset();

	if (get_display_writemode() == WRITEMODE_BUFFER) {
		flush_displaybuffer();
		set_display_writemode(WRITEMODE_DIRECT);
	}
}

static void display_battery_picto(int remain, int charging, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	int batt_lv = BATT_CHG_LV1;

	if (remain <= 20)
		batt_lv = BATT_CHG_LV1;
	else if (remain <= 40)
		batt_lv = BATT_CHG_LV2;
	else if (remain <= 60)
		batt_lv = BATT_CHG_LV3;
	else if (remain <= 80)
		batt_lv = BATT_CHG_LV4;
	else
		batt_lv = BATT_CHG_LV5;

	if (!charging) {
		batt_lv += BATT_LV1;
		if (remain <= 5)
			batt_lv = BATT_LV0;
	}

	if (remain > 100 || remain < 0) {
		if (!charging)
			batt_lv = BATT_LV0;
		else
			batt_lv = BATT_CHG_LV1;
	}

	if (!vertical) {
		info = (struct image_info_t *)&batt_data[batt_lv];
		coord_info = &coord_info_horizontal[BATTERY_PICTO];
		data = (u16 *)batt_data[batt_lv].data;
	} else {
		info = (struct image_info_t *)&batt_v_data[batt_lv];
		coord_info = &coord_info_vertical[BATTERY_PICTO];
		data = (u16 *)batt_v_data[batt_lv].data;
	}

	display_picto(info, coord_info, data);
}

static void display_battery_remain(int remain, int vertical)
{
	u16 *data;
	struct image_info_t *image;
	struct coordinate_info_t *coord_info;
	struct display_info_t disp_info;
	int i;
	unsigned int num, tmp;
	int divider = 100;
	int num_disp = 0;

	/* % */
	if (!vertical) {
		image = (struct image_info_t *)&num_h_data[PERCENT];
		coord_info = &coord_info_horizontal[REMAIN_PER_PICTO];
		data = (u16 *)num_h_data[PERCENT].data;
	} else {
		image = (struct image_info_t *)&num_v_data[PERCENT];
		coord_info = &coord_info_vertical[REMAIN_PER_PICTO];
		data = (u16 *)num_v_data[PERCENT].data;
	}
	display_picto(image, coord_info, data);

	if (remain > 100 || remain < 0) {
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[MINUS];
			coord_info = &coord_info_horizontal[REMAIN_NUM2_PICTO];
			data = (u16 *)num_h_data[MINUS].data;
		} else {
			image = (struct image_info_t *)&num_v_data[MINUS];
			coord_info = &coord_info_vertical[REMAIN_NUM2_PICTO];
			data = (u16 *)num_v_data[MINUS].data;
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[REMAIN_NUM1_PICTO];
		} else {
			coord_info = &coord_info_vertical[REMAIN_NUM1_PICTO];
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[REMAIN_NUM0_PICTO];
		} else {
			coord_info = &coord_info_vertical[REMAIN_NUM0_PICTO];
		}
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);

		return;
	}

	num_disp = 0;
	tmp = remain;
	for (i = REMAIN_NUM0_PICTO; i <= REMAIN_NUM2_PICTO; i++) {
		if (divider <= 0)
			break;
		num = tmp / divider;
		tmp = tmp % divider;
		divider = divider / 10;
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[num];
			coord_info = &coord_info_horizontal[i];
			data = (u16 *)num_h_data[num].data;
		} else {
			image = (struct image_info_t *)&num_v_data[num];
			coord_info = &coord_info_vertical[i];
			data = (u16 *)num_v_data[num].data;
		}

		if ((num == 0) && (i != REMAIN_NUM2_PICTO) && !num_disp) {
			disp_info.image = image;
			disp_info.coord = coord_info;
			display_reset(&disp_info);
			continue;
		}

		display_picto(image, coord_info, data);
		num_disp = 1;
	}
}

static void display_battery_picto_pwoff(int remain, int charging, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;
	int batt_lv = BATT_CHG_LV1;

	if (remain <= 20)
		batt_lv = BATT_CHG_LV1;
	else if (remain <= 40)
		batt_lv = BATT_CHG_LV2;
	else if (remain <= 60)
		batt_lv = BATT_CHG_LV3;
	else if (remain <= 80)
		batt_lv = BATT_CHG_LV4;
	else
		batt_lv = BATT_CHG_LV5;

	if (!charging) {
		batt_lv += BATT_LV1;
		if (remain <= 5)
			batt_lv = BATT_LV0;
	}

	if (remain > 100 || remain < 0) {
		if (!charging)
			batt_lv = BATT_LV0;
		else
			batt_lv = BATT_CHG_LV1;
	}

	if (!vertical) {
		info = (struct image_info_t *)&batt_data[batt_lv];
		coord_info = &coord_info_horizontal[PWOFF_BATTERY_PICTO];
		data = (u16 *)batt_data[batt_lv].data;
	} else {
		info = (struct image_info_t *)&batt_v_data[batt_lv];
		coord_info = &coord_info_vertical[PWOFF_BATTERY_PICTO];
		data = (u16 *)batt_v_data[batt_lv].data;
	}

	display_picto(info, coord_info, data);
}

static void display_battery_remain_pwoff(int remain, int vertical)
{
	u16 *data;
	struct image_info_t *image;
	struct coordinate_info_t *coord_info;
	struct display_info_t disp_info;
	int i;
	unsigned int num, tmp;
	int divider = 100;
	int num_disp = 0;

	/* % */
	if (!vertical) {
		image = (struct image_info_t *)&num_h_data[PERCENT];
		coord_info = &coord_info_horizontal[PWOFF_REMAIN_PER_PICTO];
		data = (u16 *)num_h_data[PERCENT].data;
	} else {
		image = (struct image_info_t *)&num_v_data[PERCENT];
		coord_info = &coord_info_vertical[PWOFF_REMAIN_PER_PICTO];
		data = (u16 *)num_v_data[PERCENT].data;
	}
	display_picto(image, coord_info, data);

	if (remain > 100 || remain < 0) {
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[MINUS];
			coord_info = &coord_info_horizontal[PWOFF_REMAIN_NUM2_PICTO];
			data = (u16 *)num_h_data[MINUS].data;
		} else {
			image = (struct image_info_t *)&num_v_data[MINUS];
			coord_info = &coord_info_vertical[PWOFF_REMAIN_NUM2_PICTO];
			data = (u16 *)num_v_data[MINUS].data;
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[PWOFF_REMAIN_NUM1_PICTO];
		} else {
			coord_info = &coord_info_vertical[PWOFF_REMAIN_NUM1_PICTO];
		}
		display_picto(image, coord_info, data);

		if (!vertical) {
			coord_info = &coord_info_horizontal[PWOFF_REMAIN_NUM0_PICTO];
		} else {
			coord_info = &coord_info_vertical[PWOFF_REMAIN_NUM0_PICTO];
		}
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);

		return;
	}

	num_disp = 0;
	tmp = remain;
	for (i = PWOFF_REMAIN_NUM0_PICTO; i <= PWOFF_REMAIN_NUM2_PICTO; i++) {
		if (divider <= 0)
			break;
		num = tmp / divider;
		tmp = tmp % divider;
		divider = divider / 10;
		if (!vertical) {
			image = (struct image_info_t *)&num_h_data[num];
			coord_info = &coord_info_horizontal[i];
			data = (u16 *)num_h_data[num].data;
		} else {
			image = (struct image_info_t *)&num_v_data[num];
			coord_info = &coord_info_vertical[i];
			data = (u16 *)num_v_data[num].data;
		}

		if ((num == 0) && (i != PWOFF_REMAIN_NUM2_PICTO) && !num_disp) {
			disp_info.image = image;
			disp_info.coord = coord_info;
			display_reset(&disp_info);
			continue;
		}

		display_picto(image, coord_info, data);
		num_disp = 1;
	}
}

static void display_bt_wave(int enable, int vertical)
{
	u16 *data;
	struct image_info_t *image;
	struct coordinate_info_t *coord_info;

	if (!vertical) {
		image = (struct image_info_t *)&bt_wave_h_data;
		coord_info = &coord_info_horizontal[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_h_data.data;
	} else {
		image = (struct image_info_t *)&bt_wave_v_data;
		coord_info = &coord_info_vertical[BT_WAVE_PICTO];
		data = (u16 *)bt_wave_v_data.data;
	}

	if (!enable) {
		struct display_info_t disp_info;
		disp_info.image = image;
		disp_info.coord = coord_info;
		display_reset(&disp_info);
	} else {
		display_picto(image, coord_info, data);
	}
}

static void display_sleep_picto(int index, int vertical)
{
	u16 *data;
	struct image_info_t *info;
	struct coordinate_info_t *coord_info;

	if (index > 1 || index < 0)
		index = 0;

	if (!vertical) {
		/* horizontal */
		coord_info = &coord_info_horizontal[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_h_data[index];
		data = (u16 *)sleep_h_data[index].data;
	} else {
		/* vertical */
		coord_info = &coord_info_vertical[SLEEP_PICTO];
		info = (struct image_info_t *)&sleep_v_data[index];
		data = (u16 *)sleep_v_data[index].data;
	}

	display_picto(info, coord_info, data);
}

static void display_suspend(void)
{
	int i = 0;
	struct panel_softc *sc = &panel_sc;
	int batt_remain;
	int bt_enable;
	int charging;
	int vertical;
	u32 prev_panel_disp_flag = panel_disp_flag;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {

		batt_remain = GET_PANEL_DISP_INFO(BATTERY);
		bt_enable = GET_PANEL_DISP_INFO(WAKEONBT);
		charging = GET_PANEL_DISP_INFO(CHARGE);
#ifdef CONFIG_ROTATE_DISPLAY
		vertical = GET_PANEL_DISP_INFO(VERTICAL);
#else
		vertical = 0;
#endif /* CONFIG_ROTATE_DISPLAY */

		mutex_unlock(&panel_sc.display_mutex);
		set_display_writemode(WRITEMODE_BUFFER);
		display_all_reset();


		if (batt_remain > 100 || batt_remain < 0)
			batt_remain = -2;

		display_battery_picto(batt_remain, charging, vertical);

		display_battery_remain(batt_remain, vertical);

		display_bt_wave(bt_enable, vertical);

		mutex_lock(&panel_sc.display_mutex);
		while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(SLEEP)) {
			if (prev_panel_disp_flag != panel_disp_flag) {
				prev_panel_disp_flag = panel_disp_flag;
				break;
			}
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			display_sleep_picto(i%2, vertical);
			if (get_display_writemode() == WRITEMODE_BUFFER) {
				flush_displaybuffer();
				set_display_writemode(WRITEMODE_DIRECT);
			}
			wait_event_timeout(sc->wait_q, sc->wait_condition,
			    msecs_to_jiffies(1000));
			i++;
			mutex_lock(&panel_sc.display_mutex);
		}

		mutex_unlock(&panel_sc.display_mutex);
		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static void display_poweroff(void)
{
	int batt_remain;
	int bt_enable;
	int charging;
	int vertical;

	mutex_lock(&panel_sc.display_mutex);

	while (GET_PANEL_DISP_INFO(ON) && GET_PANEL_DISP_INFO(PWOFF)) {

		batt_remain = GET_PANEL_DISP_INFO(BATTERY);
		bt_enable = GET_PANEL_DISP_INFO(WAKEONBT);
		charging = GET_PANEL_DISP_INFO(CHARGE);
#ifdef CONFIG_ROTATE_DISPLAY
		vertical = GET_PANEL_DISP_INFO(VERTICAL);
#else
		vertical = 0;
#endif /* CONFIG_ROTATE_DISPLAY */

		mutex_unlock(&panel_sc.display_mutex);
		set_display_writemode(WRITEMODE_BUFFER);
		display_all_reset();


		if (batt_remain > 100 || batt_remain < 0)
			batt_remain = -2;

		display_battery_picto_pwoff(batt_remain, charging, vertical);

		display_battery_remain_pwoff(batt_remain, vertical);

		if (get_display_writemode() == WRITEMODE_BUFFER) {
			flush_displaybuffer();
			set_display_writemode(WRITEMODE_DIRECT);
		}
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);
}

static int bites_display_polling_thread(void *arg)
{
	int rc = 0;
	struct mdss_panel_data *pdata = (struct mdss_panel_data *)arg;
	int disp_len = 320 * 240 * 4;
	struct panel_softc *sc = &panel_sc;
	int vertical = 1;
	u32 boot_flag = get_msm_boot_flag();
	int goto_off = 1;

	bb_virt = dma_alloc_coherent(&bb_pdev->dev, disp_len,
				     &bb_phys, GFP_KERNEL | GFP_DMA);
	
	if (!bb_virt) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ Could not allocate external memory\n");
		return 0;
	}

	mutex_lock(&panel_sc.display_mutex);
	while (!msm_nand_init_comp) {
		sc->wait_condition = 0;
		mutex_unlock(&panel_sc.display_mutex);
		wait_event_timeout(sc->wait_q, sc->wait_condition,
				   msecs_to_jiffies(500));
		mutex_lock(&panel_sc.display_mutex);
	}
	mutex_unlock(&panel_sc.display_mutex);

	if (GET_BOOT_FLAG_INFO(boot_flag, LCD) == BOOT_FLAG_LCD_DISABLE) {
		dev_err(&bb_pdev->dev, "@@@@@@@@@@@ LCD disable\n");
	} else {
#ifdef CONFIG_ROTATE_DISPLAY
		if (GET_BOOT_FLAG_INFO(boot_flag, DISP_WAY)
		    == BOOT_FLAG_DISP_HORIZONTAL) {
			/* horizontal */
			vertical = 0;
		} else {
			/* vertical */
			vertical = 1;
		}
#else
		vertical = 0;
#endif /* CONFIG_ROTATE_DISPLAY */
		atomic_set(&sc->sleeped, 0);
		display_boot_animation(vertical, 500);
	}

	while (!kthread_should_stop()) {
		mutex_lock(&panel_sc.display_mutex);
		if (!GET_PANEL_DISP_INFO(ON)) {
			sc->wait_condition = 0;
			mutex_unlock(&panel_sc.display_mutex);
			dev_err(&bb_pdev->dev,
			    "*************** sleep!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);
			if (goto_off) {
				display_all_reset();
				mdss_qpic_display_off();
				rc = mdss_qpic_panel_off(pdata, &qpic_res->panel_io);
				goto_off = 0;
			}
			atomic_set(&sc->sleeped, 1);
			wake_up(&sc->sleep_wait_q);
			wait_event(sc->wait_q, sc->wait_condition);
			atomic_set(&sc->sleeped, 0);
			mdss_qpic_panel_wait_fb_off();
			dev_err(&bb_pdev->dev,
			    "*************** wake up!! (panel_disp_flag = 0x%x)\n",
				panel_disp_flag);

			/* NB: ʂ̃`ׁ̒̈Aʂɐ䂷 */
			display_all_reset();

			continue;
		}
		if (GET_PANEL_DISP_INFO(BOOT)) {
			mutex_unlock(&panel_sc.display_mutex);
			display_boot_animation(vertical, 500);
		} else if (GET_PANEL_DISP_INFO(CARRIER_SW)) {
			mutex_unlock(&panel_sc.display_mutex);
			display_carrier_switch(vertical);
		} else {
			if (GET_PANEL_DISP_INFO(PWOFF)) {
				mutex_unlock(&panel_sc.display_mutex);
				display_poweroff();
				goto_off = 1;
			} else if (GET_PANEL_DISP_INFO(SLEEP)) {
				mutex_unlock(&panel_sc.display_mutex);
				display_suspend();
				goto_off = 1;
			} else {
				mutex_unlock(&panel_sc.display_mutex);
				display_resume();
			}
		}
	}

	return 0;
}

static void bites_display_init(struct platform_device *pdev, struct mdss_panel_data *pdata)
{
	bites_display_polling = kthread_run(bites_display_polling_thread, pdata, "bites-display");
	if (IS_ERR(bites_display_polling)) {
		pr_err("failed to create polling thread: %ld\n", PTR_ERR(bites_display_polling));
		return;
	}
}

/* PDM0 (Clock) Control
 *     PDM_AHB_CBCR: 0x01844004 (GCC BASE: 0x01800000, PDM_AHB_CBCR: 0x44004)
 *     PDM_XO4_CBCR: 0x01844008 (GCC BASE: 0x01800000, PDM_XO8_CBCR: 0x44008)
 */
#define PDM_CLK_BASE        (0x01844000)
#define PDM_CLK_SIZE        (0xF)
#define PDM_AHB_CBCR_OFFSET (0x4)
#define PDM_XO4_CBCR_OFFSET (0x8)
static void __iomem *pdm_gcc_base = NULL;

static void msm_pdm_led_clk_enable(int enable)
{
	u32 ahb, xo4;

	ahb = readl_relaxed(pdm_gcc_base + PDM_AHB_CBCR_OFFSET);
	xo4 = readl_relaxed(pdm_gcc_base + PDM_XO4_CBCR_OFFSET);

	if (enable) {
		ahb |= BIT(0);
		xo4 |= BIT(0);
	} else {
		ahb &= ~(BIT(0));
		xo4 &= ~(BIT(0));
	}

	writel_relaxed(ahb, (pdm_gcc_base + PDM_AHB_CBCR_OFFSET));
	writel_relaxed(xo4, (pdm_gcc_base + PDM_XO4_CBCR_OFFSET));
}

void msm_pdm_led_clk_prepare_enable(void)
{
	msm_pdm_led_clk_enable(1);
	pr_info("%s: done\n", __func__);
}

void msm_pdm_led_clk_disable_unprepare(void)
{
	msm_pdm_led_clk_enable(0);
	pr_info("%s: done\n", __func__);
}

static int qpic_panel_pdm_init(void)
{
	pdm_gcc_base = ioremap(PDM_CLK_BASE, PDM_CLK_SIZE);
	if (!pdm_gcc_base) {
		pr_err("ioremap failed\n");
		return -ENOMEM;
	}
	pr_info("%s: pdm gcc p=0x%08x(v=0x%08x)\n",
		__func__, PDM_CLK_BASE, (u32)pdm_gcc_base);

	return 0;
}

static void qpic_panel_bites_image_init(struct platform_device *pdev,
    struct mdss_panel_data *pdata)
{
	static int initialized;
	struct panel_softc *sc = &panel_sc;
	int rc;

	if (initialized)
		return;

	mutex_init(&panel_sc.display_mutex);

	rc = qpic_panel_sysfs_add(&pdev->dev);
	if (rc)
		return;

	init_waitqueue_head(&sc->wait_q);
	init_waitqueue_head(&sc->sleep_wait_q);

	rc = qpic_panel_pdm_init();
	if (rc)
		return;

	bb_pdev = pdev;
	bites_display_init(pdev, pdata);

	initialized = 1;
}
#endif

static const struct of_device_id mdss_qpic_panel_match[] = {
	{.compatible = "qcom,mdss-qpic-panel"},
	{}
};

static struct platform_driver this_driver = {
	.probe  = mdss_qpic_panel_probe,
	.driver = {
		.name = "qpic_panel",
		.of_match_table = mdss_qpic_panel_match,
	},
};

static int __init mdss_qpic_panel_init(void)
{
	return platform_driver_register(&this_driver);
}
MODULE_DEVICE_TABLE(of, mdss_qpic_panel_match);
module_init(mdss_qpic_panel_init);
