#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/mtd/mtd.h>

#include "necpf_kitting_img.h"

#define NAND_FLASH_KITTING_PNAME "kitting"
#define NECPF_KITTING_CONF "kitting_conf"
#define NECPF_KITTING_IMG1 "kitting_img1"
#define NECPF_KITTING_IMG2 "kitting_img2"
#define NECPF_KITTING_IMG3 "kitting_img3"
#define NECPF_KITTING_IMG4 "kitting_img4"

#define KITTING_FNAME_LEN 64
struct kitting_head_t {
#define KITTING_HEADER_MAGIC (0x5454494b) //KITT
	uint32_t magic;
	uint32_t size;
	uint32_t num_pages;
	uint32_t cksum32;
	uint8_t  reserved[16];
	char     fname[KITTING_FNAME_LEN];
};
#define KITTING_IMG_HGIGHT (320)
#define KITTING_IMG_WIDTH  (240)
#define KITTING_IMG_SIZE (2 * KITTING_IMG_WIDTH * KITTING_IMG_HGIGHT)
static uint8_t boot_image3[KITTING_IMG_SIZE]; /* 16bit W:240 H:320 */
static uint8_t boot_image4[KITTING_IMG_SIZE]; /* 16bit W:240 H:320 */
static int load_image3 = -1;
static int load_image4 = -1;

static void
kitting_turnleft90_images(uint16_t *image)
{
	int x, y;
	uint16_t *tmp_image;

	tmp_image = (uint16_t *)kmalloc(KITTING_IMG_SIZE, GFP_KERNEL);
	if (tmp_image == NULL)
		return;

	for (y = 0; y < KITTING_IMG_HGIGHT; y++) {
		for (x = 0; x < KITTING_IMG_WIDTH; x++) {
			tmp_image[(KITTING_IMG_WIDTH - x - 1) * KITTING_IMG_HGIGHT + y]
				= image[KITTING_IMG_WIDTH * y + x];
		}
	}
	memcpy(image, tmp_image, KITTING_IMG_SIZE);
	kfree(tmp_image);

	return;
}

static int
kitting_mtd_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen,
		 uint8_t *buf)
{
	if (mtd_block_isbad(mtd, from))
		return -1; /* bad block */
	return mtd_read(mtd, from, len, retlen, buf);
}

static int
kitting_flread_check(struct mtd_info *mtd, int block,
		     const char *fname, int *load_flag, uint8_t *image)
{
	size_t retlen = 0;
	int rest;
	int ret = 0;
	uint8_t *buf;
	uint8_t *p;
	struct kitting_head_t *khead;
	uint32_t cp_sz;
	uint32_t num_blocks = (uint32_t)mtd->size / mtd->erasesize;

	buf = (uint8_t *)kmalloc(mtd->erasesize, GFP_KERNEL);

	rest = mtd->erasesize;
	ret = kitting_mtd_read(mtd, mtd->erasesize * block, mtd->erasesize, &retlen, buf);
	if (ret != 0) {
		printk(KERN_ERR "%s %u: mtd_read failed\n",
		       __func__, __LINE__);
		kfree(buf);
		return -1; /* bad block */
	}
	p = buf;
	khead = (struct kitting_head_t *)p;

	/* 1st page check */
	if (khead->magic != KITTING_HEADER_MAGIC) {
		printk(KERN_ERR "%s %u: %s not found(format error)\n",
		       __func__, __LINE__, fname);
		kfree(buf);
		*load_flag = 0; /* not found */
		return 0; /* format error */
	}
	rest -= mtd->writesize;
	p += mtd->writesize;

	/* rest page check */
	while (rest > 0) {
		khead = (struct kitting_head_t *)p;
		if (khead->magic == KITTING_HEADER_MAGIC) {
			if (!strcmp(khead->fname, fname)) {
				printk(KERN_INFO "%s %u: find %s\n",
				       __func__, __LINE__, fname);
				rest -= mtd->writesize;
				p += mtd->writesize;
				break;
			}
		}
		rest -= mtd->writesize;
		p += mtd->writesize;

		if (rest <= 0) {
			/* read next block */
			while (++block < num_blocks) {
				ret = kitting_mtd_read(mtd, mtd->erasesize * block,
					       mtd->erasesize, &retlen, buf);
				if (ret == 0) {
					/* next block read success */
					rest = mtd->erasesize;
					p = buf;
					break;
				}
				/* bad block */
				printk(KERN_ERR "%s %u: mtd_read failed\n",
				       __func__, __LINE__);
			}
		}
	}
	if (rest <= 0) {
		printk(KERN_ERR "%s %u: %s not found\n",
		       __func__, __LINE__, fname);
		kfree(buf);
		*load_flag = 0; /* not found */
		return 0; /* not found */
	}

	cp_sz = rest < KITTING_IMG_SIZE ? rest : KITTING_IMG_SIZE;
	memcpy(image, p, cp_sz);
	*load_flag = 1; /* find image */

	while ((cp_sz < KITTING_IMG_SIZE) &&
	       (++block < num_blocks)) {
		*load_flag = 0; /* not found */
		ret = kitting_mtd_read(mtd, mtd->erasesize * block,
			       KITTING_IMG_SIZE - cp_sz, &retlen, buf);
		if (ret == 0) {
			memcpy(image + cp_sz, buf, retlen);
			*load_flag = 1; /* find image */
			break;
		}
		printk(KERN_ERR "%s %u: mtd_read failed\n",
		       __func__, __LINE__);
	}

	kfree(buf);

	if (*load_flag) {
		kitting_turnleft90_images((uint16_t *)image);
	}

	return 0;
}

static int
kitting_get_img_data(const char *fname, int *load_flag, uint8_t *image)
{
	struct mtd_info *mtd = NULL;
	int ret = 0;
	int i;
	uint32_t num_blocks = 0;

	mtd = get_mtd_device_nm(NAND_FLASH_KITTING_PNAME);
	if (mtd == NULL) {
		printk(KERN_ERR "No device for name %s", NAND_FLASH_KITTING_PNAME);
		return -1;
	}

	num_blocks = (uint32_t)mtd->size / mtd->erasesize;
	for (i = 0; i < num_blocks; i++) {
		ret = kitting_flread_check(mtd, i, fname, load_flag, image);
		if (ret == 0) {
			break;
		}
		*load_flag = 0; /* not found */
	}

	return 0;
}

void * kitting_disp_boot_image(int cradle)
{
	if (!cradle) {
		if (load_image4 < 0)
			kitting_get_img_data(NECPF_KITTING_IMG4,
					     &load_image4, boot_image4);
		if (load_image4 > 0)
			return boot_image4;
	} else {
		if (load_image3 < 0)
			kitting_get_img_data(NECPF_KITTING_IMG3,
					     &load_image3, boot_image3);
		if (load_image3 > 0)
			return boot_image3;
	}
	return NULL;
}
