/*
 * Copyright (c) 2013 - 2015, The Linux Foundation. All rights reserved.
 * Copyright (c) 2013 NEC AccessTechinica, Ltd., All rights reserved.
 * Copyright (c) 2014 - 2015, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*****************************************************************************
 *  Includes
 *****************************************************************************/
#include <linux/memory.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/time.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/proc_fs.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/io.h>
#include <linux/leds.h>

#include <mach/hardware.h>

#include "mdss.h"
#include "mdss_qpic.h"
#include "mdss_qpic_panel.h"


/*****************************************************************************
 *  Definitions
 *****************************************************************************/
/* OQPQ_XXXX means Orise QVGA Panel with QPIC. */
/* for debug */
#define OQPQ_DEBUG (0)
#if OQPQ_DEBUG
#define OQPQ_DPRINT(fmt, ...) pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define OQPQ_DPRINT(fmt, ...)
#endif /* OQP_DEBUG */

/* LCD Backlight support */
#define SUPPORT_LCD_BACKLIGHT (1)

/* Panel operations */
#define OP_OTM3221B_TEON    (0x35)
#define OP_OTM3221B_MADCTR  (0x36)
#define OP_OTM3221B_COLMOD  (0x3a)

/* TEON(35h) design */
#define OQPQ_TEON_TELOM_ON_BIT   (1 << 0)
#define OQPQ_TEON_TELOM_OFF_BIT  (0 << 0)

#define OQPQ_TEON_PARAM01 (OQPQ_TEON_TELOM_OFF_BIT)

/* MADCTR(36h) design */
#define OQPQ_MADCTR_MY_BIT  (1 << 7)
#define OQPQ_MADCTR_MX_BIT  (1 << 6)
#define OQPQ_MADCTR_MV_BIT  (1 << 5)
#define OQPQ_MADCTR_ML_TB   (0 << 4)
#define OQPQ_MADCTR_ML_BT   (1 << 4)
#define OQPQ_MADCTR_RGB_RGB (0 << 3)
#define OQPQ_MADCTR_RGB_BGR (1 << 3)

#define OQPQ_MADCTR_PARAM01 \
	(OQPQ_MADCTR_ML_TB | OQPQ_MADCTR_RGB_RGB)

/* MADCTR(3Ah) design */
#define OQPQ_COLMOD_VIPF3_BIT (1 << 7)
#define OQPQ_COLMOD_VIPF2_BIT (1 << 6)
#define OQPQ_COLMOD_VIPF1_BIT (1 << 5)
#define OQPQ_COLMOD_VIPF0_BIT (1 << 4)
#define OQPQ_COLMOD_IFPF2_BIT (1 << 2)
#define OQPQ_COLMOD_IFPF1_BIT (1 << 1)
#define OQPQ_COLMOD_IFPF0_BIT (1 << 0)

#define OQPQ_COLMOD_VIPF_16BPP_2TIMES \
	(OQPQ_COLMOD_VIPF3_BIT | OQPQ_COLMOD_VIPF2_BIT | \
	 OQPQ_COLMOD_VIPF1_BIT | OQPQ_COLMOD_VIPF0_BIT)
#define OQPQ_COLMOD_VIPF_18BPP_3TIMES \
	(OQPQ_COLMOD_VIPF3_BIT | OQPQ_COLMOD_VIPF2_BIT | \
	 OQPQ_COLMOD_VIPF1_BIT)
#define OQPQ_COLMOD_VIPF_18BPP_1TIMES \
	(OQPQ_COLMOD_VIPF2_BIT | OQPQ_COLMOD_VIPF1_BIT)
#define OQPQ_COLMOD_VIPF_16BPP_1TIMES \
	(OQPQ_COLMOD_VIPF2_BIT | OQPQ_COLMOD_VIPF0_BIT)
#define OQPQ_COLMOD_IFPF_12BPP \
	(OQPQ_COLMOD_IFPF1_BIT | OQPQ_COLMOD_IFPF0_BIT)
#define OQPQ_COLMOD_IFPF_16BPP \
	(OQPQ_COLMOD_IFPF2_BIT | OQPQ_COLMOD_IFPF0_BIT)
#define OQPQ_COLMOD_IFPF_18BPP \
	(OQPQ_COLMOD_IFPF2_BIT | OQPQ_COLMOD_IFPF1_BIT)
#define OQPQ_COLMOD_PARAM01 \
	(OQPQ_COLMOD_VIPF_16BPP_2TIMES | OQPQ_COLMOD_IFPF_16BPP)

/* Max command argument size.
 *   (e.g. max argument = OQP_MAX_PARAM_SZ * 4) */
#define OQPQ_MAX_PARAM_SZ  (4)


/*****************************************************************************
 *  Private funciton declaration
 *****************************************************************************/
static int panel_io_init(struct qpic_panel_io_desc *panel_io);
static int panel_io_on(struct qpic_panel_io_desc *qpic_panel_io);
static void panel_io_off(struct qpic_panel_io_desc *qpic_panel_io);

static int set_pinctrl_state(struct qpic_panel_io_desc *qpic_panel_io,
				bool active);

static void set_panel_custom_param(void);
static int set_panel_display_on(int lightup);
static void set_panel_display_off(void);


/*****************************************************************************
 *  Private functions
 *****************************************************************************/
static int panel_io_init(struct qpic_panel_io_desc *qpic_panel_io)
{
	OQPQ_DPRINT("Check gpio(s) varidation\n");
	if (!(qpic_panel_io->rst_gpio) ||
		(!gpio_is_valid(qpic_panel_io->rst_gpio))) {
		OQPQ_DPRINT("request rst gpio is invalid\n");
		return -ENODEV;
	}

	OQPQ_DPRINT("Panel io initialized\n");
	return 0;
}

static int panel_io_on(struct qpic_panel_io_desc *qpic_panel_io)
{
	int value = 0;

	/* Set pin control state. And if success, controls gpio */
	if (set_pinctrl_state(qpic_panel_io, true)) {
		pr_err("%s: pinctrl not enabled\n", __func__);
		return -1;
	}
	OQPQ_DPRINT("set_pinctrl_state success\n");

	/* Power on:
	 *   VDD is auto controlled by HW. Not care of wait. */

	/* Check reset gpio state */
	value = gpio_get_value(qpic_panel_io->rst_gpio);
	if (1 == value) {
		pr_warn("%s reset pin is 1. set 0 before reset\n",
			__func__);
		gpio_set_value(qpic_panel_io->rst_gpio, 0);
		msleep(120);
	}

	/* SW reset:
	 *   Wait for min 10 us after reset,
	 *   and wait for min 120ms (OTP loading). */
	OQPQ_DPRINT("SOFT RESET(rst_gpio=1)\n");
	gpio_set_value(qpic_panel_io->rst_gpio, 1);
	msleep(130);

	OQPQ_DPRINT("Panel power on\n");
	return 0;
}

static void panel_io_off(struct qpic_panel_io_desc *qpic_panel_io)
{
	/* SW reset:
	 *   Wait formin 120ms. */
	OQPQ_DPRINT("SOFT RESET(rst_gpio=0)\n");
	gpio_set_value(qpic_panel_io->rst_gpio, 0);
	msleep(120);

	/* Power off:
	 *   VDD is auto controlled by HW. Not care of wait. */

	/* Set pin control state */
	if (set_pinctrl_state(qpic_panel_io, false))
		pr_warn("%s: pinctrl not disabled\n", __func__);

	OQPQ_DPRINT("Panel power off\n");
}

/*****************************************************************************/
static int set_pinctrl_state(struct qpic_panel_io_desc *qpic_panel_io,
				bool active)
{
	struct pinctrl_state *pin_state;
	int rc = -EFAULT;

	if (IS_ERR_OR_NULL(qpic_panel_io->pin_res.pinctrl)) {
		pr_err("%s: can not find or invalid pin resources\n",
			__func__);
		return rc;
	}

	pin_state = active ?
			qpic_panel_io->pin_res.gpio_state_active :
			qpic_panel_io->pin_res.gpio_state_suspend;
	if (!IS_ERR_OR_NULL(pin_state)) {
		rc = pinctrl_select_state(qpic_panel_io->pin_res.pinctrl,
			pin_state);
		if (rc)
			pr_err("%s: can not set %s pins\n", __func__,
				active ? MDSS_PINCTRL_STATE_DEFAULT
				: MDSS_PINCTRL_STATE_SLEEP);
	} else {
		pr_err("%s: invalid '%s' pinstate\n", __func__,
			active ? MDSS_PINCTRL_STATE_DEFAULT
			: MDSS_PINCTRL_STATE_SLEEP);
	}
	return rc;
}

static void set_panel_custom_param(void)
{
	u8 param[OQPQ_MAX_PARAM_SZ];

	/*
	 * Set panel initial parameters
	 */
	/* Set INVOFF(20h): Display Inversion Off */
	OQPQ_DPRINT("OP_EXIT_INVERT_MODE(20h)\n");
	qpic_send_pkt(OP_EXIT_INVERT_MODE, NULL, 0);
	msleep(10);

	/* Set TEON(35h): Tearing Effect Line ON */
	param[0] = OQPQ_TEON_PARAM01;
	OQPQ_DPRINT("TEON(35h) <-- 0x%02x\n" , param[0]);
	qpic_send_pkt(OP_OTM3221B_TEON, param, 1);
	msleep(10);

	/* Set MADCTR(36h): Memory Data Access Control */
	param[0] = OQPQ_MADCTR_PARAM01;
	OQPQ_DPRINT("MADCTR(36h) <-- 0x%02x\n" , param[0]);
	qpic_send_pkt(OP_OTM3221B_MADCTR, param, 1);
	msleep(10);

	/* Set COLMOD(3Ah): Interface Pixel Format */
	param[0] = OQPQ_COLMOD_PARAM01;
	OQPQ_DPRINT("COLMOD(3Ah) <-- 0x%02x\n" , param[0]);
	qpic_send_pkt(OP_OTM3221B_COLMOD, param, 1);
	msleep(10);

	/* Wait for 200 ms after completed to set custom parameters. */
	msleep(200);

	OQPQ_DPRINT("Panel custom parameters set\n");
}

static int set_panel_display_on(int lightup)
{
	/* Set SLPOUT(11h): Sleep Out
	 *   Wait for min 120 ms after sleep out. */
	OQPQ_DPRINT("OP_EXIT_SLEEP_MODE(11h)\n");
	qpic_send_pkt(OP_EXIT_SLEEP_MODE, NULL, 0);
	msleep(120);

	/* Set DISPON(29h): Display On
	 *   Wait for min 30 ms after display on.
	 *   30ms wait - led on (10ms wait). */
	OQPQ_DPRINT("OP_SET_DISPLAY_ON(29h)\n");
	qpic_send_pkt(OP_SET_DISPLAY_ON, NULL, 0);
	msleep(30);

#if SUPPORT_LCD_BACKLIGHT
	/* LED On, if need */
	if (lightup) {
		if (msm_pdm_led_on() < 0)
			return -1;
	}
#endif /* SUPPORT_LCD_BACKLIGHT */

	OQPQ_DPRINT("Panel display on\n");
	return 0;
}

static void set_panel_display_off(void)
{
#if SUPPORT_LCD_BACKLIGHT
	/* LED Off */
	msm_pdm_led_off();
	msleep(10);
#endif /* SUPPORT_LCD_BACKLIGHT */

	/* Set DISPOFF(28h): Display Off *
	 *   Wait for min 30 ms after display off. */
	OQPQ_DPRINT("OP_SET_DISPLAY_OFF(28h)\n");
	qpic_send_pkt(OP_SET_DISPLAY_OFF, NULL, 0);
	msleep(30);

	/* Set SLPOUT(10h): Sleep In
	 *   Wait for min 120 ms after sleep in. */
	OQPQ_DPRINT("OP_ENTER_SLEEP_MODE(10h)\n");
	qpic_send_pkt(OP_ENTER_SLEEP_MODE, NULL, 0);
	msleep(120);

	OQPQ_DPRINT("Panel display off\n");
}


/*****************************************************************************
 *  Public functions
 *****************************************************************************/
int otm3221b_panel_on(struct qpic_panel_io_desc *qpic_panel_io)
{
	int err = 0;

	/* Initialization */
	if (!qpic_panel_io->init) {
		err = panel_io_init(qpic_panel_io);
		if (err < 0)
			return err;

		qpic_panel_io->init = true;
	}

	/* Panel power on */
	err = panel_io_on(qpic_panel_io);
	if (err < 0)
		return err;

	/* Set custom parameters */
	set_panel_custom_param();

	return 0;
}

int otm3221b_display_on(int lightup)
{
	return set_panel_display_on(lightup);
}

void otm3221b_display_off(void)
{
	set_panel_display_off();
}

void otm3221b_panel_off(struct qpic_panel_io_desc *qpic_panel_io)
{
	panel_io_off(qpic_panel_io);
}
