#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <errno.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>

#include <glib.h>
#include <glib/gprintf.h>

#include <dbus/dbus.h>

static char*
get_adapter_path(DBusConnection *conn, const char *pattern)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;
	dbus_bool_t ret;
	const char *adapter_path;
	char *path;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.bluez.Manager",
					       "FindAdapter");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Manager#FindAdapter)\n");
 		return NULL;
	}

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &pattern);

	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Manager#FindAdapter'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return NULL;
	}

	ret = dbus_message_get_args(reply, &error,
				    DBUS_TYPE_OBJECT_PATH, &adapter_path,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		dbus_message_unref(reply);
		return NULL;
	}

	path = strdup(adapter_path);

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return path;
}

static char*
get_default_adapter_path(DBusConnection *conn)
{
	DBusMessage *message, *reply;
	DBusError error;
	dbus_bool_t ret;
	const char *adapter_path;
	char *path;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.bluez.Manager",
					       "DefaultAdapter");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Manager#DefaultAdapter)\n");
 		return NULL;
	}

	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Manager#DefaultAdapter'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return NULL;
	}

	ret = dbus_message_get_args(reply, &error,
				    DBUS_TYPE_OBJECT_PATH, &adapter_path,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		dbus_message_unref(reply);
		return NULL;
	}

	path = strdup(adapter_path);

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return path;
}

static char*
_get_device_path(DBusConnection *conn, const char *adapter, const char *address)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;
	char *object_path = NULL, *device_path = NULL;
	dbus_bool_t ret;

	message = dbus_message_new_method_call("org.bluez", adapter,
					       "org.bluez.Adapter",
					       "FindDevice");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Adapter#FindDevice)\n");
 		return NULL;
	}

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &address);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Adapter#FindDevice\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return NULL;
	}

	ret = dbus_message_get_args(reply, &error,
				    DBUS_TYPE_OBJECT_PATH, &object_path,
				    DBUS_TYPE_INVALID);
	if (!ret) {
		fprintf(stderr, "Can't get Device object path\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		dbus_message_unref(reply);
		return NULL;
	}

	dbus_message_unref(reply);

	device_path = strdup(object_path);

	return device_path;
}

static char*
get_device_path(DBusConnection *conn, const char *adapter, const char *address)
{
	char *device_path;

	device_path = _get_device_path(conn, adapter, address);
	if (device_path == NULL) {
		fprintf(stderr, "Can't find DevicePath of '%s'\n", address);
		exit(1);
	}

	return device_path;
}

static int
list_command_show_address_and_alias(DBusConnection *conn, const char *device)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter;
	DBusError error;
	DBusMessageIter reply_iter_entry;
	char *address, *alias;

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#GetProperties)\n");
 		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Can't get Device#GetProperties\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key, *value;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (strcmp(key, "Alias") != 0 && strcmp(key, "Address") != 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		/* Try to get the value */
		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);

		dbus_message_iter_get_basic(&iter_dict_val, &value);
		if (strcmp(key, "Alias") == 0) {
			alias = strdup(value);
		} else {
			address = strdup(value);
		}

		dbus_message_iter_next(&reply_iter_entry);
	}

	printf("%s %s\n", address, alias);

	free(alias);
	free(address);

	dbus_message_unref(reply);

	return 0;
}

static int
list_command(DBusConnection *conn, const char *adapter)
{
	DBusMessageIter reply_iter, iter_array;
	DBusError error;
	DBusMessage *message, *reply;

	message = dbus_message_new_method_call("org.bluez", adapter,
					       "org.bluez.Adapter",
					       "ListDevices");
	if (message == NULL) {
		fprintf(stderr, "Can't alloc method 'Adapter#ListDevices'\n");
		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Adapter#ListDevices\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		dbus_message_unref(reply);
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &iter_array);
	while (dbus_message_iter_get_arg_type(&iter_array) == DBUS_TYPE_OBJECT_PATH) {
		const char *device_path;
		int ret;

		dbus_message_iter_get_basic(&iter_array, &device_path);

		ret = list_command_show_address_and_alias(conn, device_path);
		if (ret != 0) {
			break;
		}
		dbus_message_iter_next(&iter_array);
	}

	dbus_message_unref(reply);

	return 0;
}

static void
show_services_property(DBusMessageIter *reply_iter)
{
	DBusMessageIter iter_array;

	if (dbus_message_iter_get_arg_type(reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] Iter is not Array\n", __func__);
		return;
	}

	dbus_message_iter_recurse(reply_iter, &iter_array);
	while (dbus_message_iter_get_arg_type(&iter_array) == DBUS_TYPE_OBJECT_PATH) {
		const char *path;
		DBusMessageIter service_entry;

		dbus_message_iter_recurse(&iter_array, &service_entry);
		dbus_message_iter_get_basic(&service_entry, &path);

		if (!path) {
			dbus_message_iter_next(&iter_array);
			continue;
		}

		printf("%s\n", path);
		dbus_message_iter_next(&iter_array);
	}
}

static int
services_command(DBusConnection *conn, const char *device)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry;
	DBusError error;

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#GetProperties)\n");
 		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Device#GetProperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iter is not Array\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (strcmp(key, "Services") != 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		show_services_property(&iter_dict_val);

		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);
	return 0;
}


static int
create_command(DBusConnection *conn, const char *adapter, const char *address)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;

	message = dbus_message_new_method_call("org.bluez", adapter,
					       "org.bluez.Adapter",
					       "CreateDevice");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Adapter#CreateDevice)\n");
 		return -1;
	}

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &address);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Adapter#CreateDevice'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_unref(reply);

	printf("Success: CreateDecice '%s'\n", address);
	return 0;
}

static int
remove_command(DBusConnection *conn, const char *adapter, const char *device)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter;
	DBusError error;

	message = dbus_message_new_method_call("org.bluez", adapter,
					       "org.bluez.Adapter",
					       "RemoveDevice");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Adapter#RemoveDevice)\n");
		return -1;
	}

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_OBJECT_PATH, &device);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	if (reply == NULL) {
		fprintf(stderr, "Can't call 'RemoveDevice'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_unref(message);

	if (dbus_error_is_set(&error)) {
		fprintf(stderr, "[%s] Error is set\n", __func__);
		return -1;
	}

	return 0;
}

static int
disconnect_command(DBusConnection *conn, const char *device)
{
	DBusMessage *message, *reply;
	DBusError error;

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "Disconnect");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#Disconnect)\n");
 		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Can't call 'Device#Disconnect'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_unref(message);

	if (dbus_error_is_set(&error)) {
		fprintf(stderr, "[%s] Error is set\n", __func__);
		return -1;
	}

	dbus_message_unref(reply);

	return 0;
}

static int
discover_command(DBusConnection *conn, const char *device, const char *pattern)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter, reply_iter, reply_iter_entry;
	DBusError error;
	const char *xml;

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "DiscoverServices");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#DiscoverServices)\n");
 		return -1;
	}

	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &pattern);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Adapter#FindDevice\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		dbus_message_unref(reply);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	/* Hopefully we only get one handle, or take a punt */
	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		uint32_t key;
		DBusMessageIter dict_entry;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (key == 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_get_basic(&dict_entry, &xml);
		if (xml != NULL) {
			printf("%s\n", xml);
		}
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return 0;
}

static int
show_device_property(DBusConnection *conn, const char *device,
		     const char *property, int type)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry;
	DBusError error;

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "GetProperties");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#GetProperties)\n");
 		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, NULL);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Device#GetProperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (strncmp(key, property, 32) != 0) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);

		switch (type) {
		case DBUS_TYPE_UINT32: {
			uint32_t value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);
			printf("0x%06x\n", value);
			break;
		}
		case DBUS_TYPE_STRING: {
			const char *value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);
			printf("%s\n", value);
			break;
		}
		case DBUS_TYPE_BOOLEAN: {
			gboolean value;
			dbus_message_iter_get_basic(&iter_dict_val, &value);

			printf("%d\n", value);
			break;
		}
		default:
			break;
		}

		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);
	return 0;
}

static int
class_command(DBusConnection *conn, const char *device)
{
	return show_device_property(conn, device, "Class", DBUS_TYPE_UINT32);
}

static int
name_command(DBusConnection *conn, const char *device)
{
	return show_device_property(conn, device, "Name", DBUS_TYPE_STRING);
}

static int
alias_command_get(DBusConnection *conn, const char *device)
{
	return show_device_property(conn, device, "Alias", DBUS_TYPE_STRING);
}

static int
alias_command_set(DBusConnection *conn, const char *device,
		  const char *new_alias)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter, sub;
	DBusError error;
	const char *alias_key = "Alias";
	char type_sig[2] = { DBUS_TYPE_STRING, '\0' };

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "SetProperty");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#SetProperty)\n");
 		return -1;
	}

	/* Set argument */
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &alias_key);

	dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT, type_sig, &sub);
	dbus_message_iter_append_basic(&sub, DBUS_TYPE_STRING, &new_alias);
	dbus_message_iter_close_container(&iter, &sub);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: 'Device#SetProperty'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	return 0;
}

static int
trusted_command_get(DBusConnection *conn, const char *device)
{
	return show_device_property(conn, device, "Trusted", DBUS_TYPE_BOOLEAN);
}

static int
set_device_bool_property(DBusConnection *conn, const char *device, const char *key, int bool)
{
	DBusMessage *message, *reply;
	DBusMessageIter iter, sub;
	DBusError error;
	char type_sig[2] = { DBUS_TYPE_BOOLEAN, '\0' };

	message = dbus_message_new_method_call("org.bluez", device,
					       "org.bluez.Device",
					       "SetProperty");
	if (message == NULL) {
		fprintf(stderr, "Can't allocate new method call(Device#SetProperty)\n");
 		return -1;
	}

	/* Set argument */
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);

	dbus_message_iter_open_container(&iter, DBUS_TYPE_VARIANT, type_sig, &sub);
	dbus_message_iter_append_basic(&sub, DBUS_TYPE_BOOLEAN, &bool);
	dbus_message_iter_close_container(&iter, &sub);

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Device#SetProperty\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	return 0;
}

static int
trusted_command_set(DBusConnection *conn, const char *device, int yes_or_no)
{
	return set_device_bool_property(conn, device, "Trusted", yes_or_no);
}

static int
blocked_command_get(DBusConnection *conn, const char *device)
{
	return show_device_property(conn, device, "Blocked", DBUS_TYPE_BOOLEAN);
}

static int
blocked_command_set(DBusConnection *conn, const char *device, int yes_or_no)
{
	return set_device_bool_property(conn, device, "Blocked", yes_or_no);
}

static void
usage(const char *prog)
{
	printf("Usage: %s <command>\n\n"
	       "<options>\n"
	       "  -i, --device=DEV_ID Device pattern\n"
	       "  -h, --help          Show usage\n\n"
	       "<command>\n"
	       "  list\n"
	       "  services <address>\n"
	       "  create <address>\n"
	       "  remove <address|path>\n"
	       "  disconnect <address>\n"
	       "  discover <address> [pattern]\n"
	       "  class <address>\n"
	       "  name <address>\n"
	       "  alias <address> [alias]\n"
	       "  trusted <address> [yes/no]\n"
	       "  blocked <address> [yes/no]\n",
	       prog);
}

enum device_command {
	DEV_CMD_LIST,
	DEV_CMD_SERVICES,
	DEV_CMD_CREATE,
	DEV_CMD_REMOVE,
	DEV_CMD_DISCONNECT,
	DEV_CMD_DISCOVER,
	DEV_CMD_CLASS,
	DEV_CMD_NAME,
	DEV_CMD_ALIAS,
	DEV_CMD_TRUSTED,
	DEV_CMD_BLOCKED,
	DEV_CMD_INVALID
};

static enum device_command
subcommand_to_device_command(const char *subcmd)
{
	if (strcmp(subcmd, "list") == 0) {
		return DEV_CMD_LIST;
	} else if (strcmp(subcmd, "services") == 0) {
		return DEV_CMD_SERVICES;
	} else if (strcmp(subcmd, "create") == 0) {
		return DEV_CMD_CREATE;
	} else if (strcmp(subcmd, "remove") == 0) {
		return DEV_CMD_REMOVE;
	} else if (strcmp(subcmd, "disconnect") == 0) {
		return DEV_CMD_DISCONNECT;
	} else if (strcmp(subcmd, "discover") == 0) {
		return DEV_CMD_DISCOVER;
	} else if (strcmp(subcmd, "class") == 0) {
		return DEV_CMD_CLASS;
	} else if (strcmp(subcmd, "name") == 0) {
		return DEV_CMD_NAME;
	} else if (strcmp(subcmd, "alias") == 0) {
		return DEV_CMD_ALIAS;
	} else if (strcmp(subcmd, "trusted") == 0) {
		return DEV_CMD_TRUSTED;
	} else if (strcmp(subcmd, "blocked") == 0) {
		return DEV_CMD_BLOCKED;
	}

	return DEV_CMD_INVALID;
}

static struct option main_options[] = {
	{ "device",	1, 0, 'i' },
	{ "help",	0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

int main(int argc, char *argv[])
{
	DBusConnection *conn;
	char *adapter_path = NULL;
	char *device_path = NULL;
	char *pattern = NULL;
	const char *subcommand;
	enum device_command devcmd;
	int ret, opt;

	while ((opt = getopt_long(argc, argv, "i:h", main_options, NULL)) != EOF) {
		switch(opt) {
		case 'i':
			pattern = optarg;
			break;
		case 'h':
			usage(argv[0]);
			exit(0);
		default:
			exit(1);
		}
	}

	argc -= optind;
	argv += optind;
	optind = 0;

	if (argc < 1) {
		usage(argv[0]);
		exit(1);
	}

	subcommand = argv[0];

	devcmd = subcommand_to_device_command(subcommand);
	if (devcmd == DEV_CMD_INVALID) {
		fprintf(stderr, "Invalid sub command '%s'\n", subcommand);
		return -1;
	}

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Error: Can't get on dbus system bus");
		exit(1);
	}

	if (pattern) {
		adapter_path = get_adapter_path(conn, pattern);
	} else {
		adapter_path = get_default_adapter_path(conn);
	}

	if (adapter_path == NULL) {
		fprintf(stderr, "Failed: Can't get default adapter\n");
		exit(1);
	}

	if (devcmd == DEV_CMD_LIST) {
		ret = list_command(conn, adapter_path);
	} else if (devcmd == DEV_CMD_SERVICES || devcmd == DEV_CMD_REMOVE
		   || devcmd == DEV_CMD_DISCONNECT || devcmd == DEV_CMD_CLASS
		   || devcmd == DEV_CMD_NAME) {
		const char *address;

		if (argc < 2) {
			fprintf(stderr, "Usage: necat-test-device '%s' <address>\n",
				subcommand);
			exit(1);
		}

		address = argv[1];
		device_path = get_device_path(conn, adapter_path, address);

		switch (devcmd) {
		case DEV_CMD_SERVICES:
			ret = services_command(conn, device_path);
			break;
		case DEV_CMD_REMOVE:
			ret = remove_command(conn, adapter_path, device_path);
			break;
		case DEV_CMD_DISCONNECT:
			ret = disconnect_command(conn, device_path);
			break;
		case DEV_CMD_CLASS:
			ret = class_command(conn, device_path);
			break;
		case DEV_CMD_NAME:
			ret = name_command(conn, device_path);
			break;
		default:
			break;
		}

		free(device_path);
	} else if (devcmd == DEV_CMD_CREATE) {
		const char *address;

		if (argc < 2) {
			fprintf(stderr, "Usage: necat-test-device 'create' <address>\n");
			exit(1);
		}

		address = argv[1];

		ret = create_command(conn, adapter_path, address);
	} else if (devcmd == DEV_CMD_DISCOVER) {
		const char *address, *pattern;

		if (argc < 2) {
			fprintf(stderr, "Usage: necat-test-device 'discover' <address>\n");
			exit(1);
		}

		address = argv[1];
		if (argc < 3) {
			pattern = "";
		} else {
			pattern = argv[2];
		}

		device_path = get_device_path(conn, adapter_path, address);

		ret = discover_command(conn, device_path, pattern);

		free(device_path);
	} else if (devcmd == DEV_CMD_ALIAS) {
		const char *address;

		if (argc < 2) {
			fprintf(stderr, "Usage: necat-test-device disconnect <address>\n");
			exit(1);
		}
		address = argv[1];

		device_path = get_device_path(conn, adapter_path, address);

		if (argc < 3) {
			ret = alias_command_get(conn, device_path);
		} else {
			const char *new_alias = argv[2];
			ret = alias_command_set(conn, device_path, new_alias);
		}

		free(device_path);
	} else if (devcmd == DEV_CMD_TRUSTED || devcmd == DEV_CMD_BLOCKED) {
		const char *address;

		if (argc < 2) {
			fprintf(stderr, "Usage: necat-test-device disconnect <address>\n");
			exit(1);
		}
		address = argv[1];

		device_path = get_device_path(conn, adapter_path, address);

		if (argc < 3) {
			if (devcmd == DEV_CMD_TRUSTED) {
				ret = trusted_command_get(conn, device_path);
			} else {
				ret = blocked_command_get(conn, device_path);
			}
		} else {
			const char *value = argv[2];
			int yes_or_no = 0;

			if (strcmp(value, "yes") == 0) {
				yes_or_no = 1;
			} else if (strcmp(value, "no") == 0) {
				yes_or_no = 0;
			} else {
				fprintf(stderr, "Please input yes/no\n");
				exit(1);
			}

			if (devcmd == DEV_CMD_TRUSTED) {
				ret = trusted_command_set(conn, device_path, yes_or_no);
			} else {
				ret = blocked_command_set(conn, device_path, yes_or_no);
			}
		}

		free(device_path);
	}

	free(adapter_path);

	if (ret != 0) {
		printf("Failed '%s'\n", subcommand);
	}

	return ret;
}
