#include <netinet/ether.h>

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <inttypes.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>

#include <dbus/dbus.h>
#include <gdbus.h>

#include <bluez5.h>

#include <sys/param.h>
#include <sys/uio.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/socket.h>

#ifdef USE_WLANAPI
#include <libwlanapi.h>
#include <lib_local_comm.h>
#endif /* USE_WLANAPI */

#include "bluetooth.h"
#include "hci.h"
#include "hci_lib.h"

static volatile sig_atomic_t __io_canceled = 0;
static volatile sig_atomic_t __io_terminated = 0;
static volatile sig_atomic_t exit_on_release = 1;

static int debug_mode;

#ifdef __unused
#undef __unused
#endif

#define __unused __attribute__((unused))

#define AGENT_PATH "/test/agent"
#define AGENT_INTERFACE "org.bluez.Agent1"

static int pairing_confirmation(DBusConnection *, const char *, int, void *, int, DBusMessage *);
static int get_device_name_btaddr(DBusConnection *, const char *, char **, char **);

static void sig_term(int sig __unused)
{
	__io_canceled = 1;
}

static DBusMessage *release_agent(DBusConnection *conn,
				  DBusMessage *msg, void *user_data)
{
	printf("Agent released\n");
	g_dbus_unregister_interface(conn, AGENT_PATH, AGENT_INTERFACE);
	return dbus_message_new_method_return(msg);
}

static DBusMessage *request_pincode(DBusConnection *conn,
				    DBusMessage *msg, void *user_data)
{
	const char *device;
	const char *pincode_ptr;
	int ret;

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_INVALID);

#ifdef USE_WLANAPI
	if (!debug_mode) {
		struct bt_pairing_RequestPincode data;
		int ret;

		ret = pairing_confirmation(conn, device,
					   BT_PAIRING_REQUEST_PINCODE,
					   (void *)&data, sizeof(data), msg);
		syslog(LOG_DEBUG, "%s: pairing_confirmation ret=%d pincode=%s",
		       __func__, ret, data.pincode);
		pincode_ptr = data.pincode;
	} else {
#endif /* USE_WLANAPI */
		char pincode[BT_PINCODE_MAXLEN];

		memset(pincode, 0, sizeof(pincode));

		printf("Pincode request for device %s\n", device);
		printf("Enter PinCode: ");

		ret = scanf("%15s", pincode);
		if (ret == EOF) {
			perror("scanf[pincode]");
			return NULL;
		}
		pincode_ptr = pincode;
#ifdef USE_WLANAPI
	}
#endif /* USE_WLANAPI */

	g_dbus_send_reply(conn, msg, DBUS_TYPE_STRING, &pincode_ptr,
			  DBUS_TYPE_INVALID);
	return NULL;
}

static DBusMessage *display_pincode(DBusConnection *conn,
				    DBusMessage *msg, void *user_data)
{
	const char *device;
	const char *pincode;

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_STRING, &pincode, DBUS_TYPE_INVALID);

	printf("PIN code: %s\n", pincode);
	return dbus_message_new_method_return(msg);
}

static DBusMessage *request_passkey(DBusConnection *conn,
				    DBusMessage *msg, void *user_data)
{
	const char *device;
	uint32_t passkey;
	int ret;

	printf("Request passkey\n");

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_INVALID);

	printf("Passkey request for device %s\n", device);
	printf("Enter passkey: ");
	ret = scanf("%" PRIu32, &passkey);
	if (ret == EOF) {
		perror("scanf[passkey]");
		return NULL;
	}

	g_dbus_send_reply(conn, msg, DBUS_TYPE_UINT32, &passkey,
			  DBUS_TYPE_INVALID);
	return NULL;
}

static DBusMessage *display_passkey(DBusConnection *conn,
				    DBusMessage *msg, void *user_data)
{
	const char *device;
	dbus_uint32_t passkey;
	dbus_uint16_t entered;
	char passkey_full[7];

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_UINT32, &passkey, DBUS_TYPE_UINT16,
			      &entered, DBUS_TYPE_INVALID);

	snprintf(passkey_full, sizeof(passkey_full), "%.6u", passkey);
	passkey_full[6] = '\0';

	if (entered > strlen(passkey_full))
		entered = strlen(passkey_full);

	printf("Passkey: %.*s %s\n",
	       entered, passkey_full, passkey_full + entered);

	return dbus_message_new_method_return(msg);
}

#ifdef USE_WLANAPI
static int
pairing_confirmation(DBusConnection *conn, const char *device,
		  int type, void *data, int data_len, DBusMessage *msg)
{
	int fd = -1, ret = -1;
	struct bt_pairing_hdr *hdr = (struct bt_pairing_hdr *)data;
	struct local_comm_hdr rcv_hdr;
	int rcv_size;
	void *buf = NULL;
	char *name = NULL, *bdaddr_str = NULL;
	struct ether_addr *addr;
	
	int dd, len, n;
	int hdev = 0;
	struct hci_filter nf, of;
	socklen_t olen;
	hci_event_hdr *ehdr;
	struct pollfd p;
	int try = 10;
	int to = 60000;
	int presult = BT_PAIRING_RESULT_FAILURE;
	unsigned char hbuf[HCI_MAX_EVENT_SIZE], *ptr;
	evt_simple_pairing_complete *spc;

	if (libbtutl_get_bt_pairable() == 0) {
		syslog(LOG_DEBUG, "%s: Reject pairing. NOT pairable now",
		       __func__);
		return -1;
	}

	{
		dd = hci_open_dev(hdev);
		if (dd < 0) {
			syslog(LOG_ERR , "Can't open device hci%d: %s (%d)\n",
							hdev, strerror(errno), errno);
			return -1;
		}
		
		olen = sizeof(of);
		if (getsockopt(dd, SOL_HCI, HCI_FILTER, &of, &olen) < 0)
			return -1;
		
		hci_filter_clear(&nf);
		hci_filter_set_ptype(HCI_EVENT_PKT,  &nf);
		hci_filter_set_event(EVT_SIMPLE_PAIRING_COMPLETE, &nf);
		hci_filter_set_event(EVT_DISCONN_COMPLETE, &nf);
		if (setsockopt(dd, SOL_HCI, HCI_FILTER, &nf, sizeof(nf)) < 0)
			goto end;
		
		syslog(LOG_DEBUG, "%s : %d\n", __func__, __LINE__);
	}
	
	get_device_name_btaddr(conn, device, &name, &bdaddr_str);

	addr = ether_aton(bdaddr_str);
	if (addr == NULL) {
		syslog(LOG_DEBUG, "%s: ether_aton failed", __func__);
		goto end;
	}

	strlcpy(hdr->name, name, sizeof(hdr->name));
	memcpy(hdr->bdaddr, addr, sizeof(hdr->bdaddr));

	syslog(LOG_DEBUG, "%s: [SEND] type=%d len=%d", __func__,
	       type, data_len);

	fd = lib_local_comm_client_open(BLUEZ_AGENT_SOCK_PATH);
	if (fd < 0) {
		syslog(LOG_DEBUG, "%s: lib_local_comm_client_open(%s) error",
		       __func__, BLUEZ_AGENT_SOCK_PATH);
		goto end;
	}

	ret = lib_local_comm_send(fd, type, 0, data, data_len);
	if (ret) {
		syslog(LOG_DEBUG, "%s:lib_local_comm_send error", __func__);
		goto end;
	}

	ret = lib_local_comm_recv(fd, &rcv_hdr, (void **)&buf, &rcv_size);
	if (ret) {
		syslog(LOG_DEBUG, "%s: lib_local_comm_recv error", __func__);
		goto end;
	}

	syslog(LOG_DEBUG, "%s: [RECV] type=%d status=%d", __func__,
	       rcv_hdr.type, rcv_hdr.status);

	if (rcv_hdr.type == type) {
		ret = rcv_hdr.status;
		if (buf != NULL && rcv_size > 0)
			memcpy(data, buf, data_len);
	} else {
		syslog(LOG_DEBUG, "%s: recv unknown message type=%d",
		    __func__, rcv_hdr.type);
		ret = -1;
	}
	
	if (ret) {
		g_dbus_send_error(conn, msg,
				  "org.bluez.Error.Rejected", NULL);
		syslog(LOG_DEBUG, "Pairing Failure\n");
		presult = BT_PAIRING_RESULT_FAILURE;
	} else {
		g_dbus_send_reply(conn, msg, DBUS_TYPE_INVALID);
		
		while (try--) {
			if (to) {
				p.fd = dd;
				p.events = POLLIN;
				while ((n = poll(&p, 1, to)) < 0) {
					if (errno == EAGAIN || errno == EINTR)
						continue;
					goto end;
				}
				
				if (!n) {
					errno = ETIMEDOUT;
					goto end;
				}
				
				to -= 10;
				if (to < 0)
					to = 0;
			}
			
			while ((len = read(dd, hbuf, sizeof(hbuf))) < 0) {
				if (errno == EAGAIN || errno == EINTR)
					continue;
				goto end;
			}
			
			ehdr = (void *) (hbuf + 1);
			ptr = hbuf + (1 + HCI_EVENT_HDR_SIZE);
			len -= (1 + HCI_EVENT_HDR_SIZE);
			
			syslog(LOG_INFO, "Event : %d\n", ehdr->evt);
			
			if(ehdr->evt == EVT_SIMPLE_PAIRING_COMPLETE) {
				spc = (void *) ptr;
				syslog(LOG_DEBUG, "EVT_SIMPLE_PAIRING_COMPLETE Status : 0x%d\n", spc->status);
				if(spc->status == 0) {
					syslog(LOG_DEBUG, "Pairing Success\n");
					presult = BT_PAIRING_RESULT_SUCCESS;
				} else {
					syslog(LOG_DEBUG, "Pairing Failure\n");
					presult = BT_PAIRING_RESULT_FAILURE;
				}
				break;
			}
			if(ehdr->evt == EVT_DISCONN_COMPLETE) {
				syslog(LOG_DEBUG, "EVT_DISCONN_COMPLETE\n");
				syslog(LOG_DEBUG, "Pairing Failure\n");
				presult = BT_PAIRING_RESULT_FAILURE;
				break;
			}
		}
	}
	
	ret = lib_local_comm_send(fd, BT_PAIRING_RESULT, 0, &presult, sizeof(presult));
	if (ret) {
		syslog(LOG_DEBUG, "%s:lib_local_comm_send error", __func__);
		goto end;
	}
	
 end:
	setsockopt(dd, SOL_HCI, HCI_FILTER, &of, sizeof(of));
	hci_close_dev(dd);
	
	if (name != NULL)
		free(name);
	if (bdaddr_str != NULL)
		free(bdaddr_str);
	if (buf != NULL)
		free(buf);
	if (fd >= 0) {
		lib_local_comm_client_close(fd);
	}

	return ret;
}
#endif /* USE_WLANAPI */

/*
 * @@@ free 'name' and 'bdaddr' after no longer used
 */
static int get_device_name_btaddr(DBusConnection *conn,
				  const char *object_path,
				  char **name, char **bdaddr)
{
	int ret;
	char value[128];

	ret = nb5_device1_get_string_property(conn, object_path,
					      "Alias", value,
					      sizeof(value));
	if (ret != 0)
		return -1;

	*name = strdup(value);

	ret = nb5_device1_get_string_property(conn, object_path,
					      "Address", value,
					      sizeof(value));
	if (ret != 0)
		return -1;

	*bdaddr = strdup(value);
	return 0;
}

static DBusMessage *request_confirmation(DBusConnection *conn,
					 DBusMessage *msg, void *user_data)
{
	const char *device;
	dbus_uint32_t passkey;
	int ret;
	
	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_UINT32, &passkey, DBUS_TYPE_INVALID);

#ifdef USE_WLANAPI
	if (!debug_mode) {
		struct bt_pairing_RequestConfirm data;
		
		memset(&data, 0, sizeof(data));
		data.passkey = passkey;
		ret = pairing_confirmation(conn, device,
					   BT_PAIRING_REQUEST_CONFIRMATION,
					   (void *)&data, sizeof(data), msg);

		syslog(LOG_DEBUG, "%s: pairing_confirmation ret=%d",
		       __func__, ret);

	} else {
#endif /* USE_WLANAPI */
		char yes_or_no[4];

		memset(yes_or_no, 0, sizeof(yes_or_no));

		printf("RequestConfirmation (%s, %06u)\n", device, passkey);
		printf("Confirm passkey (yes/no): ");
		ret = scanf("%3s", yes_or_no);
		if (ret == EOF) {
			perror("scanf[yes_or_no]");
			return NULL;
		}

		if (strncmp(yes_or_no, "yes", sizeof("yes")) != 0) {
			g_dbus_send_error(conn, msg,
					  "org.bluez.Error.Rejected", NULL);
			return NULL;
		}

		g_dbus_send_reply(conn, msg, DBUS_TYPE_INVALID);
#ifdef USE_WLANAPI
	}
#endif /* USE_WLANAPI */

	return NULL;
}

static DBusMessage *request_authorization(DBusConnection *conn,
					  DBusMessage *msg, void *user_data)
{
	const char *device;
	int ret;

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_INVALID);

#ifdef USE_WLANAPI
	if (!debug_mode) {
		struct bt_pairing_RequestAuthorize data;

		memset(&data, 0, sizeof(data));
		ret = pairing_confirmation(conn, device,
					   BT_PAIRING_REQUEST_AUTHORIZATION,
					   (void *)&data, sizeof(data), msg);

		syslog(LOG_DEBUG, "%s: pairing_confirmation ret=%d",
		       __func__, ret);

		if (ret)
			g_dbus_send_error(conn, msg,
					  "org.bluez.Error.Rejected", NULL);
		else
			g_dbus_send_reply(conn, msg, DBUS_TYPE_INVALID);
	} else {
#endif /* USE_WLANAPI */
		char yes_or_no[4];

		memset(yes_or_no, 0, sizeof(yes_or_no));

		printf("Request authorization(%s)\n", device);
		printf("Accept pairing ? (yes/no): ");
		ret = scanf("%3s", yes_or_no);
		if (ret == EOF) {
			perror("scanf[yes_or_no]");
			return NULL;
		}

		if (strncmp(yes_or_no, "yes", sizeof("yes")) != 0) {
			g_dbus_send_error(conn, msg,
					  "org.bluez.Error.Rejected", NULL);
			return NULL;
		}

		g_dbus_send_reply(conn, msg, DBUS_TYPE_INVALID);
#ifdef USE_WLANAPI
	}
#endif /* USE_WLANAPI */

	return NULL;
}

static DBusMessage *authorize_service(DBusConnection *conn,
				      DBusMessage *msg, void *user_data)
{
	const char *device, *uuid;

	printf("Authorize service\n");

	dbus_message_get_args(msg, NULL, DBUS_TYPE_OBJECT_PATH, &device,
			      DBUS_TYPE_STRING, &uuid, DBUS_TYPE_INVALID);

	/* Do not query "yes/no" */
	printf("Authorize service (%s, %s)\n", device, uuid);

	g_dbus_send_reply(conn, msg, DBUS_TYPE_INVALID);
	return NULL;
}

static DBusMessage *cancel_request(DBusConnection *conn,
				   DBusMessage *msg, void *user_data)
{
	printf("Request canceled\n");
	return dbus_message_new_method_return(msg);
}

static const GDBusMethodTable methods[] = {
	{ GDBUS_METHOD("Release",
		       NULL, NULL, release_agent) },
	{ GDBUS_ASYNC_METHOD("RequestPinCode",
			     GDBUS_ARGS({ "device", "o" }),
			     GDBUS_ARGS({ "pincode", "s" }), request_pincode) },
	{ GDBUS_METHOD("DisplayPinCode",
		       GDBUS_ARGS({ "device", "o" }, { "pincode", "s" }),
		       NULL, display_pincode) },
	{ GDBUS_ASYNC_METHOD("RequestPasskey",
			     GDBUS_ARGS({ "device", "o" }),
			     GDBUS_ARGS({ "passkey", "u" }), request_passkey) },
	{ GDBUS_METHOD("DisplayPasskey",
		       GDBUS_ARGS({ "device", "o" }, { "passkey", "u" },
				  { "entered", "q" }),
		       NULL, display_passkey) },
	{ GDBUS_ASYNC_METHOD("RequestConfirmation",
			     GDBUS_ARGS({ "device", "o" }, { "passkey", "u" }),
			     NULL, request_confirmation) },
	{ GDBUS_ASYNC_METHOD("RequestAuthorization",
			     GDBUS_ARGS({ "device", "o" }),
			     NULL, request_authorization) },
	{ GDBUS_ASYNC_METHOD("AuthorizeService",
			     GDBUS_ARGS({ "device", "o" }, { "uuid", "s" }),
			     NULL,  authorize_service) },
	{ GDBUS_METHOD("Cancel", NULL, NULL, cancel_request) },
	{ }
};

static int register_agent(DBusConnection *conn, const char *agent_path,
			  const char *capability)
{
	DBusMessage *msg, *reply;
	DBusError err;
	const char *object_path = "/org/bluez";

	msg = dbus_message_new_method_call("org.bluez", object_path,
					   "org.bluez.AgentManager1",
					   "RegisterAgent");
	if (msg == NULL) {
		fprintf(stderr, "Can't allocate new method");
		return -1;
	}

	printf("Register Agent\n");

	dbus_message_append_args(msg, DBUS_TYPE_OBJECT_PATH, &agent_path,
				 DBUS_TYPE_STRING, &capability,
				 DBUS_TYPE_INVALID);
	dbus_error_init(&err);

	reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);
	dbus_message_unref(msg);

	if (!reply) {
		fprintf(stderr, "Failed: AgentManager1#RegisterAgent\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		return -1;
	}

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return 0;
}

static int request_default_agent(DBusConnection *conn,
				 const char *agent_path)
{
	DBusMessage *msg, *reply;
	DBusError err;
	const char *object_path = "/org/bluez";

	msg = dbus_message_new_method_call("org.bluez", object_path,
					   "org.bluez.AgentManager1",
					   "RequestDefaultAgent");
	if (msg == NULL) {
		fprintf(stderr, "Can't allocate new method");
		return -1;
	}

	printf("Request Default Agent\n");

	dbus_message_append_args(msg, DBUS_TYPE_OBJECT_PATH, &agent_path,
				 DBUS_TYPE_INVALID);

	dbus_error_init(&err);

	reply = dbus_connection_send_with_reply_and_block(conn, msg, -1, &err);
	dbus_message_unref(msg);

	if (!reply) {
		fprintf(stderr, "Can't request default agent\n");
		if (dbus_error_is_set(&err)) {
			fprintf(stderr, "%s\n", err.message);
			dbus_error_free(&err);
		}
		return -1;
	}

	dbus_message_unref(reply);
	dbus_connection_flush(conn);

	return 0;
}

static struct option options[] = {
	{ "capability", 1, NULL, 'c' },
	{ "help", 0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

static void usage()
{
	printf("NECAT Bluetooth agent\n\n");
	printf("Usage: bluez5-agent [options]\n\n"
	       "Options:\n"
	       "\t-c, --capability : Set capability (Default: \"DisplayYesNo\")\n"
	       "\t-d, --debug_mode : Set debug_mode\n"
	       "\t-h, --help       : Show this message\n"
	       "\n");
}

int main(int argc, char *argv[])
{
	const char *capability = "DisplayYesNo";
	int opt;
	DBusConnection *conn;
	dbus_bool_t ret;
	struct sigaction sa;

	while ((opt = getopt_long(argc, argv, "c:dh", options, NULL)) != EOF) {
		switch (opt) {
		case 'c':
			capability = optarg;
			break;
		case 'd':
			debug_mode = 1;
			break;
		case 'h':
			usage();
			exit(0);
		default:
			usage();
			exit(1);
		}
	}

	argc -= optind;
	argv += optind;
	optind = 0;

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Can't get on dbus system bus");
		exit(1);
	}

	ret = g_dbus_register_interface(conn, AGENT_PATH,
					AGENT_INTERFACE, methods,
					NULL, NULL, NULL, NULL);
	if (ret == FALSE) {
		fprintf(stderr, "Can't register object path for agent\n");
		exit(1);
	}

	if (register_agent(conn, AGENT_PATH, capability) < 0) {
		dbus_connection_unref(conn);
		exit(1);
	}

	if (request_default_agent(conn, AGENT_PATH) < 0) {
		dbus_connection_unref(conn);
		exit(1);
	}

	memset(&sa, 0, sizeof(sa));
	sa.sa_flags = SA_NOCLDSTOP;
	sa.sa_handler = sig_term;
	sigaction(SIGTERM, &sa, NULL);
	sigaction(SIGINT, &sa, NULL);

	while (!__io_canceled && !__io_terminated) {
		if (dbus_connection_read_write_dispatch(conn, 500) != TRUE)
			break;
	}

	dbus_connection_unref(conn);

	return 0;
}
