#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>

#include <bluez5.h>

#define PROGRAM_NAME "bluez5-test-device"

static int
show_device_name_and_alias(DBusConnection *conn, const char *object_path)
{
	char buf[128];

	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Address", buf, sizeof(buf));
	if (ret != 0)
		return ret;

	printf("%s ", buf);

	ret = nb5_device1_get_string_property(conn, object_path,
					      "Name", buf, sizeof(buf));
	if (ret != 0) {
		printf("\n");
		return ret;
	}

	printf("%s\n", buf);
	return 0;
}

static int
interfaces_foreach(DBusConnection *conn, const char *object_path,
		   DBusMessageIter *interfaces_iter)
{
	while (dbus_message_iter_get_arg_type(interfaces_iter) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry;
		int ret;

		dbus_message_iter_recurse(interfaces_iter, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (strcmp(key, "org.bluez.Device1") != 0 ) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		ret = show_device_name_and_alias(conn, object_path);
		if (ret != 0)
			return ret;
		dbus_message_iter_next(interfaces_iter);
	}

	return 0;
}

static int
list_command(DBusConnection *conn)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter reply_iter, reply_iter_entry;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.freedesktop.DBus.ObjectManager",
					       "GetManagedObjects");
	if (message == NULL) {
		fprintf(stderr,
			"Can't allocate(ObjectManager#GetManagedObjects)\n");
		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Manager#Getproperties'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;
		int ret;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		ret = interfaces_foreach(conn, key, &iter_dict_val);
		if (ret != 0)
			return ret;
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return 0;
}

static int
remove_command(DBusConnection *conn, const char *address, const char *pattern)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter args;
	char adapter_path[128];
	char object_path[128];
	char *object_path_ptr = object_path;
	int ret;

	ret = nb5_find_adapter_path(conn, adapter_path, pattern);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_adapter(%s)\n", address);
		return -1;
	}

	ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	message = dbus_message_new_method_call("org.bluez",
					       adapter_path,
					       "org.bluez.Adapter1",
					       "RemoveDevice");

	if (message == NULL) {
		fprintf(stderr,
			"Can't allocate method(Adapter1#RemoveDevice)\n");
		return -1;
	}

	/* Set argument */
	dbus_message_iter_init_append(message, &args);
	dbus_message_iter_append_basic(&args, DBUS_TYPE_OBJECT_PATH,
				       &object_path_ptr);
	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message,
							  -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Adapter1#RemoveDevice(%s)\n",
			object_path);
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_unref(reply);
	return 0;
}

static int
connect_command(DBusConnection *conn, const char *address)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_connect(conn, object_path);
	if (ret != 0)
		return ret;

	return 0;
}

static int
connect_command_profile(DBusConnection *conn, const char *address,
			const char *profile_uuid)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_connect_profile(conn, object_path, profile_uuid);
	if (ret != 0)
		return ret;

	return 0;
}

static int
disconnect_command(DBusConnection *conn, const char *address)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_disconnect(conn, object_path);
	if (ret != 0)
		return ret;

	return 0;
}

static int
disconnect_command_profile(DBusConnection *conn, const char *address,
			   const char *profile)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_disconnect_profile(conn, object_path, profile);
	if (ret != 0)
		return ret;

	return 0;
}

static int
class_command(DBusConnection *conn, const char *address)
{
	char object_path[128];
	uint32_t class;

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_get_uint32_property(conn, object_path,
					      "Class", &class);
	if (ret != 0)
		return ret;

	printf("0x%06x\n", class);
	return 0;
}

static int
name_command(DBusConnection *conn, const char *address)
{
	char object_path[128];
	char name[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_get_string_property(conn, object_path,
					      "Name", name,
					      sizeof(name));
	if (ret != 0)
		return ret;

	printf("%s\n", name);
	return 0;
}

static int
alias_command_get(DBusConnection *conn, const char *address)
{
	char object_path[128];
	char alias[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_get_string_property(conn, object_path,
					      "Alias", alias,
					      sizeof(alias));
	if (ret != 0)
		return ret;

	printf("%s\n", alias);
	return 0;
}

static int
alias_command_set(DBusConnection *conn, const char *address,
		  const char *new_alias)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	return nb5_device1_set_string_property(conn, object_path,
					       "Alias", new_alias);
}

static int
trusted_command_get(DBusConnection *conn, const char *address)
{
	char object_path[128];
	gboolean trusted;

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_get_boolean_property(conn, object_path,
					       "Trusted", &trusted);
	if (ret != 0)
		return ret;

	printf("%d\n", trusted);
	return 0;
}

static int
trusted_command_set(DBusConnection *conn, const char *address,
		    int yes_or_no)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	return nb5_device1_set_boolean_property(conn, object_path,
						"Trusted", yes_or_no);
}

static int
blocked_command_get(DBusConnection *conn, const char *address)
{
	char object_path[128];
	gboolean blocked;

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	ret = nb5_device1_get_boolean_property(conn, object_path,
					       "Blocked", &blocked);
	if (ret != 0)
		return -1;

	printf("%d\n", blocked);
	return 0;
}

static int
blocked_command_set(DBusConnection *conn, const char *address,
		    int yes_or_no)
{
	char object_path[128];

	int ret = nb5_find_device(conn, address, object_path);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_device(%s)\n", address);
		return -1;
	}

	return nb5_device1_set_boolean_property(conn, object_path,
						"Blocked", yes_or_no);
}

static void
usage(const char *prog)
{
	printf("Usage: %s <command>\n\n"
	       "<options>\n"
	       "  -i, --device=DEV_ID Device pattern\n"
	       "  -h, --help          Show usage\n\n"
	       "<command>\n"
	       "  list\n"
	       "  create <address>\n"
	       "  remove <address|path>\n"
	       "  connect <address> [profile]\n"
	       "  disconnect <address> [profile]\n"
	       "  class <address>\n"
	       "  name <address>\n"
	       "  alias <address> [alias]\n"
	       "  trusted <address> [yes/no]\n"
	       "  blocked <address> [yes/no]\n",
	       prog);
}

enum device_command {
	DEV_CMD_LIST,
	DEV_CMD_REMOVE,
	DEV_CMD_CONNECT,
	DEV_CMD_DISCONNECT,
	DEV_CMD_CLASS,
	DEV_CMD_NAME,
	DEV_CMD_ALIAS,
	DEV_CMD_TRUSTED,
	DEV_CMD_BLOCKED,
	DEV_CMD_INVALID
};

static enum device_command
subcommand_to_device_command(const char *subcmd)
{
	if (strcmp(subcmd, "list") == 0) {
		return DEV_CMD_LIST;
	} else if (strcmp(subcmd, "remove") == 0) {
		return DEV_CMD_REMOVE;
	} else if (strcmp(subcmd, "connect") == 0) {
		return DEV_CMD_CONNECT;
	} else if (strcmp(subcmd, "disconnect") == 0) {
		return DEV_CMD_DISCONNECT;
	} else if (strcmp(subcmd, "class") == 0) {
		return DEV_CMD_CLASS;
	} else if (strcmp(subcmd, "name") == 0) {
		return DEV_CMD_NAME;
	} else if (strcmp(subcmd, "alias") == 0) {
		return DEV_CMD_ALIAS;
	} else if (strcmp(subcmd, "trusted") == 0) {
		return DEV_CMD_TRUSTED;
	} else if (strcmp(subcmd, "blocked") == 0) {
		return DEV_CMD_BLOCKED;
	}

	return DEV_CMD_INVALID;
}

static struct option main_options[] = {
	{ "device",	1, 0, 'i' },
	{ "help",	0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

static void usage_remove_subcommand()
{
	fprintf(stderr, "Usage: %s remove <address>\n",
		PROGRAM_NAME);
}

static void usage_connect_subcommand()
{
	fprintf(stderr, "Usage: %s connect <address> [profile]\n",
		PROGRAM_NAME);
}

static void usage_disconnect_subcommand()
{
	fprintf(stderr, "Usage: %s disconnect <address> [profile]\n",
		PROGRAM_NAME);
}

static void usage_name_subcommand()
{
	fprintf(stderr, "Usage: %s name <address>\n",
		PROGRAM_NAME);
}

static void usage_class_subcommand()
{
	fprintf(stderr, "Usage: %s class <address>\n",
		PROGRAM_NAME);
}

static void usage_alias_subcommand()
{
	fprintf(stderr, "Usage: %s alias <address> [new_alias]\n",
		PROGRAM_NAME);
}

static void usage_trusted_subcommand()
{
	fprintf(stderr, "Usage: %s trusted <address> [yes/no]\n",
		PROGRAM_NAME);
}

static void usage_blocked_subcommand()
{
	fprintf(stderr, "Usage: %s blocked <address> [yes/no]\n",
		PROGRAM_NAME);
}

static int yes_or_no_to_bool(const char *yes_or_no)
{
	int retval;

	if (strcmp(yes_or_no, "yes") == 0) {
		retval = 1;
	} else if (strcmp(yes_or_no, "no") == 0) {
		retval = 0;
	} else {
		fprintf(stderr, "Please input 'yes' or 'no'\n");
		exit(1);
	}

	return retval;
}

int main(int argc, char *argv[])
{
	DBusConnection *conn;
	char *pattern = NULL;
	const char *subcommand;
	enum device_command devcmd;
	int ret = 0, opt;
	const char *progname = argv[0];

	while ((opt = getopt_long(argc, argv, "i:h", main_options, NULL)) != EOF) {
		switch(opt) {
		case 'i':
			pattern = optarg;
			break;
		case 'h':
			usage(progname);
			exit(0);
		default:
			exit(1);
		}
	}

	argc -= optind;
	argv += optind;
	optind = 0;

	if (argc < 1) {
		usage(progname);
		exit(1);
	}

	subcommand = argv[0];

	devcmd = subcommand_to_device_command(subcommand);
	if (devcmd == DEV_CMD_INVALID) {
		fprintf(stderr, "Invalid sub command '%s'\n", subcommand);
		return -1;
	}

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Error: Can't get on dbus system bus");
		exit(1);
	}

	if (devcmd == DEV_CMD_LIST) {
		ret = list_command(conn);
	} else if (devcmd == DEV_CMD_REMOVE) {
		const char *address;
		if (argc < 2) {
			usage_remove_subcommand();
			exit(1);
		}

		address = argv[1];
		ret = remove_command(conn, address, pattern);
	} else if (devcmd == DEV_CMD_CONNECT) {
		const char *address;
		if (argc < 2) {
			usage_connect_subcommand();
			exit(1);
		}

		address = argv[1];

		if (argc < 3) {
			connect_command(conn, address);
		} else {
			const char *profile_uuid = argv[2];
			connect_command_profile(conn, address, profile_uuid);
		}
	} else if (devcmd == DEV_CMD_DISCONNECT) {
		const char *address;
		if (argc < 2) {
			usage_disconnect_subcommand();
			exit(1);
		}

		address = argv[1];

		if (argc < 3) {
			disconnect_command(conn, address);
		} else {
			const char *profile_uuid = argv[2];
			disconnect_command_profile(conn, address,
						   profile_uuid);
		}
	} else if (devcmd == DEV_CMD_NAME) {
		const char *address;
		if (argc < 2) {
			usage_name_subcommand();
			exit(1);
		}
		address = argv[1];
		name_command(conn, address);
	} else if (devcmd == DEV_CMD_CLASS) {
		const char *address;
		if (argc < 2) {
			usage_class_subcommand();
			exit(1);
		}
		address = argv[1];
		class_command(conn, address);
	} else if (devcmd == DEV_CMD_ALIAS) {
		const char *address;

		if (argc < 2) {
			usage_alias_subcommand();
			exit(1);
		}
		address = argv[1];

		if (argc < 3) {
			ret = alias_command_get(conn, address);
		} else {
			const char *new_alias = argv[2];
			ret = alias_command_set(conn, address, new_alias);
		}
	} else if (devcmd == DEV_CMD_TRUSTED) {
		const char *address;

		if (argc < 2) {
			usage_trusted_subcommand();
			exit(1);
		}
		address = argv[1];

		if (argc < 3) {
			ret = trusted_command_get(conn, address);
		} else {
			const char *value = argv[2];
			int yes_or_no = yes_or_no_to_bool(value);
			ret = trusted_command_set(conn, address, yes_or_no);
		}
	} else if (devcmd == DEV_CMD_BLOCKED) {
		const char *address;

		if (argc < 2) {
			usage_blocked_subcommand();
			exit(1);
		}
		address = argv[1];

		if (argc < 3) {
			ret = blocked_command_get(conn, address);
		} else {
			const char *value = argv[2];
			int yes_or_no = yes_or_no_to_bool(value);
			ret = blocked_command_set(conn, address, yes_or_no);
		}
	}

	if (ret != 0) {
		printf("Failed '%s'\n", subcommand);
	}

	dbus_connection_flush(conn);

	return ret;
}
