#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <bluez5.h>
#include <sdp.h>
#include <getopt.h>

static int service_flag = 0;

static void
print_string_range(const char *str, int start, int end)
{
	int i;

	for (i = start; i < end; i++) {
		putchar(str[i]);
	}
}

static int
ends_with(const char *str, const char *suffix)
{
	size_t len_str, len_suffix;

	if (str == NULL || suffix == NULL) {
		return 0;
	}

	len_str = strlen(str);
	len_suffix = strlen(suffix);

	if (len_suffix > len_str) {
		return 0;
	}

	return strncmp(str + len_str - len_suffix, suffix, len_suffix) == 0;
}

static int
uuid_to_integer(const char *uuid_str, int start)
{
	char tmp[5];
	long uuid;
	int i;

	for (i = start; i < start + 4; i++)
		tmp[i - start] = uuid_str[i];

	tmp[4] = '\0';

	uuid = strtol(tmp, NULL, 16);
	return (int)uuid;
}

static void
show_uuids_properties(DBusMessageIter *uuids_iter)
{
	DBusMessageIter iter_array;

	dbus_message_iter_recurse(uuids_iter, &iter_array);
	while (dbus_message_iter_get_arg_type(&iter_array) == DBUS_TYPE_STRING) {
		const char *uuid;

		dbus_message_iter_get_basic(&iter_array, &uuid);

		if (!uuid) {
			dbus_message_iter_next(&iter_array);
			continue;
		}

		if (ends_with(uuid, "-0000-1000-8000-00805f9b34fb")) {
			if (!service_flag)
				printf("0x");

			if (strncmp(uuid, "0000", 4) == 0) {
				if (service_flag) {
					int num = uuid_to_integer(uuid, 4);
					printf("'%s'", nb5_uuid_to_name(num));
				}
				else {
					print_string_range(uuid, 4, 8);
				}
			} else {
				print_string_range(uuid, 0, 8);
			}
		} else {
			printf("%s", uuid);
		}

		if (dbus_message_iter_next(&iter_array)) {
			putchar(' ');
		}
	}
}

static void
show_device1_uuids_property(DBusConnection *conn, const char *object_path)
{
	DBusMessage *message, *reply;
	DBusMessageIter reply_iter, reply_iter_entry, args;
	DBusError error;
	int reply_type;
	const char *interface_name = "org.bluez.Device1";
	const char *property_name = "UUIDs";

	message = dbus_message_new_method_call("org.bluez",
					       object_path,
					       "org.freedesktop.DBus.Properties",
					       "Get");
	if (message == NULL) {
		fprintf(stderr,
			"Failed org.freedesktop.DBus.Properties#Get(UUIDs)\n");
		return;
	}

	/* Set argument */
	dbus_message_iter_init_append(message, &args);
	dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING,
				       &interface_name);
	dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING,
				       &property_name);
	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);

	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: Getting property 'UUIDs'(%s)\n",
			object_path);
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);

	reply_type = dbus_message_iter_get_arg_type(&reply_iter);
	if (reply_type != DBUS_TYPE_VARIANT) {
		fprintf(stderr, "[%s] iterator is not VALIANT(%c)\n",
			__func__, reply_type);
		return;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	printf("\t%s = ", service_flag ? "Services" : "UUIDs");
	show_uuids_properties(&reply_iter_entry);
	printf("\n");
	dbus_message_unref(reply);
	return;
}

static void
show_device_address(DBusConnection *conn, const char *object_path)
{
	char address[128];
	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Address", address,
						  sizeof(address));
	if (ret != 0)
		return;

	printf("\tAddress = %s\n", address);
}

static void
show_device_name(DBusConnection *conn, const char *object_path)
{
	char name[128];
	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Name", name,
						  sizeof(name));
	if (ret != 0)
		return;

	printf("\tName = %s\n", name);
}

static void
show_device_icon(DBusConnection *conn, const char *object_path)
{
	char icon[128];
	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Icon", icon,
						  sizeof(icon));
	if (ret != 0)
		return;

	printf("\tIcon = %s\n", icon);
}

static void
show_device_class(DBusConnection *conn, const char *object_path)
{
	uint32_t class;
	int ret = nb5_device1_get_uint32_property(conn, object_path,
						  "Class", &class);
	if (ret != 0)
		return;

	printf("\tClass = %x\n", class);
}

static void
show_device_paired(DBusConnection *conn, const char *object_path)
{
	gboolean paired;
	int ret = nb5_device1_get_boolean_property(conn, object_path,
						   "Paired", &paired);
	if (ret != 0)
		return;

	printf("\tPaired = %s\n", paired ? "yes" : "no");
}

static void
show_device_connected(DBusConnection *conn, const char *object_path)
{
	gboolean connected;
	int ret = nb5_device1_get_boolean_property(conn, object_path,
						   "Connected", &connected);
	if (ret != 0)
		return;

	printf("\tConnected = %s\n", connected ? "yes" : "no");
}

static void
show_device_trusted(DBusConnection *conn, const char *object_path)
{
	gboolean trusted;
	int ret = nb5_device1_get_boolean_property(conn, object_path,
						   "Trusted", &trusted);
	if (ret != 0)
		return;

	printf("\tTrusted = %s\n", trusted ? "yes" : "no");
}

static void
show_device_blocked(DBusConnection *conn, const char *object_path)
{
	gboolean blocked;
	int ret = nb5_device1_get_boolean_property(conn, object_path,
						   "Blocked", &blocked);
	if (ret != 0)
		return;

	printf("\tBlocked = %s\n", blocked ? "yes" : "no");
}

static void
show_device_alias(DBusConnection *conn, const char *object_path)
{
	char alias[128];
	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Alias", alias,
						  sizeof(alias));
	if (ret != 0)
		return;

	printf("\tAlias = %s\n", alias);
}

static void
show_device_adapter(DBusConnection *conn, const char *object_path)
{
	char adapter[128];
	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Adapter", adapter,
						  sizeof(adapter));
	if (ret != 0)
		return;

	printf("\tAdapter = %s\n", adapter);
}

static void
show_device_legacy_pairing(DBusConnection *conn, const char *object_path)
{
	gboolean pairing;
	int ret = nb5_device1_get_boolean_property(conn, object_path,
						   "LegacyPairing",
						   &pairing);
	if (ret != 0)
		return;

	printf("\tLegacyPairing = %s\n", pairing ? "yes" : "no");
}

static void
show_device_properties(DBusConnection *conn, const char *object_path)
{
	printf("    [ %s ]\n", object_path);

	/*
	 * We cannot get some properties to some bluetooth devices.
	 *  - 'Appearance'
	 *  - 'Modalias'
	 *  - 'RSSI'
	 */
	show_device_address(conn, object_path);
	show_device_name(conn, object_path);
	show_device_icon(conn, object_path);
	show_device_class(conn, object_path);

	show_device1_uuids_property(conn, object_path);

	show_device_paired(conn, object_path);
	show_device_connected(conn, object_path);
	show_device_trusted(conn, object_path);
	show_device_blocked(conn, object_path);

	show_device_alias(conn, object_path);
	show_device_adapter(conn, object_path);
	show_device_legacy_pairing(conn, object_path);
	return;
}

static void
interfaces_foreach(DBusConnection *conn, const char *object_path,
		   DBusMessageIter *interfaces_iter)
{
	while (dbus_message_iter_get_arg_type(interfaces_iter) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry;

		dbus_message_iter_recurse(interfaces_iter, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (strcmp(key, "org.bluez.Device1") != 0 ) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		show_device_properties(conn, object_path);
		dbus_message_iter_next(interfaces_iter);
	}
}

static void
list_devices(DBusConnection *conn)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter reply_iter, reply_iter_entry;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.freedesktop.DBus.ObjectManager",
					       "GetManagedObjects");
	if (message == NULL) {
		fprintf(stderr,
			"Can't allocate(ObjectManager#GetManagedObjects)\n");
		return;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: ObjectManager#GetMangedObjects'\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		interfaces_foreach(conn, key, &iter_dict_val);
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);

	return;
}

static void
usage(const char *prog)
{
	printf("Usage: %s <options>\n\n"
	       "<options>\n"
	       "  -s, --service       Show services instead of UUIDs\n"
	       "  -h, --help          Show usage\n\n",
	       prog);
}

static struct option main_options[] = {
	{ "service", 0, 0, 's' },
	{ "help", 0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

int main(int argc, char *argv[])
{
	DBusConnection *conn;
	int opt;

	while ((opt = getopt_long(argc, argv, "hs", main_options, NULL)) != EOF) {
		switch(opt) {
		case 's':
			service_flag = 1;
			break;
		case 'h':
			usage(argv[0]);
			exit(0);
		default:
			exit(1);
		}
	}

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Can't get on dbus system bus");
		exit(1);
	}

	list_devices(conn);
	dbus_connection_flush(conn);

	return 0;
}
