#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

#include <bluez5.h>

static int
register_bridge(DBusConnection *conn, const char *bridge,
		const char *pattern)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter args;
	char adapter_path[128];
	char *service = "nap";
	int ret;

	printf("Register bridge: '%s'\n", bridge);

	ret = nb5_find_adapter_path(conn, adapter_path, pattern);
	if (ret != 0) {
		fprintf(stderr, "Failed: nb5_find_adapter\n");
		return -1;
	}

	message = dbus_message_new_method_call("org.bluez",
					       adapter_path,
					       "org.bluez.NetworkServer1",
					       "Register");

	if (message == NULL) {
		fprintf(stderr,
			"Can't allocate method(NetworkServer1#Register)\n");
		return -1;
	}

	/* Set argument */
	dbus_message_iter_init_append(message, &args);
	dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING,
				       &service);
	dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING,
				       &bridge);
	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message,
							  -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: NetworkServer1#Register\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_unref(reply);
	return 0;
}

static void
usage(const char *prog)
{
	printf("Usage: %s [options] bridge\n\n"
	       "<options>\n"
	       "  -i, --device=DEV_ID Device pattern\n"
	       "  -h, --help          Show usage\n\n",
	       prog);
}

static struct option main_options[] = {
	{ "device",	1, 0, 'i' },
	{ "help",	0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

int main(int argc, char *argv[])
{
	DBusConnection *conn;
	char *pattern = NULL;
	char *bridge;
	int ret = 0, opt;
	const char *progname = argv[0];

	while ((opt = getopt_long(argc, argv, "i:h", main_options, NULL)) != EOF) {
		switch(opt) {
		case 'i':
			pattern = optarg;
			break;
		case 'h':
			usage(progname);
			exit(0);
		default:
			exit(1);
		}
	}

	argc -= optind;
	argv += optind;
	optind = 0;

	if (argc < 1) {
		bridge = "br0";
	} else {
		bridge = argv[0];
	}

	conn = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		fprintf(stderr, "Error: Can't get on dbus system bus");
		exit(1);
	}

	register_bridge(conn, bridge, pattern);

	if (daemon(1, 0)) {
		fprintf(stderr, "daemon fail\n");
		dbus_connection_flush(conn);
		exit(EXIT_FAILURE);
	}

	while(1)
		sleep(1000);

	dbus_connection_flush(conn);
	return ret;
}
