#include <string.h>
#include <bluez5.h>

static gboolean
is_same_address(DBusConnection *conn, const char *object_path,
		const char *address)
{
	char address_prop[128];

	int ret = nb5_device1_get_string_property(conn, object_path,
						  "Address", address_prop,
						  sizeof(address_prop));
	if (ret != 0)
		return ret;

	return (strcmp(address_prop, address) == 0);
}

static int
find_device(DBusConnection *conn, const char *object_path,
	    DBusMessageIter *interfaces_iter,
	    const char *address)
{
	while (dbus_message_iter_get_arg_type(interfaces_iter) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry;

		dbus_message_iter_recurse(interfaces_iter, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (strcmp(key, "org.bluez.Device1") != 0 ) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (is_same_address(conn, object_path, address))
			return 1;

		dbus_message_iter_next(interfaces_iter);
	}

	return 0;
}

int
nb5_find_device(DBusConnection *conn, const char *address,
		char *object_path)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter reply_iter, reply_iter_entry;
	int ret = -1;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.freedesktop.DBus.ObjectManager",
					       "GetManagedObjects"
	);
	if (message == NULL) {
		fprintf(stderr,
			"Failed: set ObjectManager#GetManagedObjects\n");
		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: ObjectManager#GetManagedObjects\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		if (find_device(conn, key, &iter_dict_val, address)) {
			strcpy(object_path, key);
			ret = 0;
			break;
		}
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);
	return ret;
}

static gboolean
match_adapter_address(DBusConnection *conn, const char *object_path,
		      const char *pattern)
{
	char address_prop[128];

	int ret = nb5_adapter1_get_string_property(conn, object_path,
						   "Address", address_prop,
						   sizeof(address_prop));
	if (ret != 0)
		return ret;

	return (strcmp(address_prop, pattern) == 0);
}

static int
find_adapter(DBusConnection *conn, const char *object_path,
	     DBusMessageIter *interfaces_iter, const char *pattern)
{
	while (dbus_message_iter_get_arg_type(interfaces_iter) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry;

		dbus_message_iter_recurse(interfaces_iter, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		if (strcmp(key, "org.bluez.Adapter1") == 0) {
			if (pattern == NULL)
				return 1;

			if (match_adapter_address(conn, object_path, pattern))
				return 1;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(interfaces_iter);
			continue;
		}

		dbus_message_iter_next(interfaces_iter);
	}

	return 0;
}

int
nb5_find_adapter_path(DBusConnection *conn, char *adapter_path,
		      const char *pattern)
{
	DBusMessage *message, *reply;
	DBusError error;
	DBusMessageIter reply_iter, reply_iter_entry;
	int ret = -1;

	message = dbus_message_new_method_call("org.bluez", "/",
					       "org.freedesktop.DBus.ObjectManager",
					       "GetManagedObjects"
	);
	if (message == NULL) {
		fprintf(stderr,
			"Failed: set ObjectManager#GetManagedObjects\n");
		return -1;
	}

	dbus_error_init(&error);
	reply = dbus_connection_send_with_reply_and_block(conn, message, -1, &error);
	dbus_message_unref(message);

	if (reply == NULL) {
		fprintf(stderr, "Failed: ObjectManager#GetManagedObjects\n");
		if (dbus_error_is_set(&error)) {
			fprintf(stderr, "%s\n", error.message);
			dbus_error_free(&error);
		}
		return -1;
	}

	dbus_message_iter_init(reply, &reply_iter);
	if (dbus_message_iter_get_arg_type(&reply_iter) != DBUS_TYPE_ARRAY) {
		fprintf(stderr, "[%s] iterator is not ARRAY\n", __func__);
		return -1;
	}

	dbus_message_iter_recurse(&reply_iter, &reply_iter_entry);

	while (dbus_message_iter_get_arg_type(&reply_iter_entry) == DBUS_TYPE_DICT_ENTRY) {
		const char *key;
		DBusMessageIter dict_entry, iter_dict_val;

		dbus_message_iter_recurse(&reply_iter_entry, &dict_entry);

		dbus_message_iter_get_basic(&dict_entry, &key);
		if (!key) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		if (!dbus_message_iter_next(&dict_entry)) {
			dbus_message_iter_next(&reply_iter_entry);
			continue;
		}

		dbus_message_iter_recurse(&dict_entry, &iter_dict_val);
		if (find_adapter(conn, key, &iter_dict_val, pattern)) {
			strcpy(adapter_path, key);
			ret = 0;
			break;
		}
		dbus_message_iter_next(&reply_iter_entry);
	}

	dbus_message_unref(reply);
	return ret;
}
