/*
 * Copyright (C) 2007 Google Incorporated
 * Copyright (c) 2008-2016, The Linux Foundation. All rights reserved.
 * Copyright (c) 2016-2021, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/*****************************************************************************
 * Includes
 *****************************************************************************/
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/kthread.h>
#include <linux/sched.h>
#include <uapi/linux/sched/types.h>

#define NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0

#define NEC_PANEL_USE_ALLOCATED_DRM_FB
#include <drm/drm_fb_helper.h>

#include "nec_panel.h"
#include "nec_fb.h"



/*****************************************************************************
 * Types
 *****************************************************************************/
/* for debug */
#define NEC_FB_DEBUG		(0)
#define NEC_FB_VERBOSE		(0)

#if NEC_FB_DEBUG
#define NEC_FB_DPRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define NEC_FB_DPRINT(fmt, ...)
#define printk(...)
#endif /* NEC_FB_DEBUG */

#if NEC_FB_VERBOSE
#define NEC_FB_VPRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#else
#define NEC_FB_VPRINT(fmt, ...)
#define printk(...)
#endif /* NEC_FB_VERBOSE */

#define NEC_FB_PRINT(fmt, ...) \
		pr_info("%s: " fmt, __func__, ## __VA_ARGS__)
#define NEC_FB_PRINT_ERR(fmt, ...) \
		pr_err("%s: " fmt, __func__, ## __VA_ARGS__)


/*****************************************************************************
 * Statics
 *****************************************************************************/
struct nec_fb_panel_operation *nec_fb_panel_ops = NULL;
static struct drm_fb_helper *_helper = NULL;
static struct nec_fb_data *_nfd = NULL;

#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
// Backup vars for FB info
static struct fb_rel_info {
	struct fb_var_screeninfo var;
	u32 smem_len;				// Length of frame buffer mem
	size_t obj_size;			// size of the gem object (for mmap)
	unsigned long screen_size;	// size of (virtual) screen (for reada/write)
	unsigned int fb_height;		// height of FB
} _fbi_original, _fbi_tricked;
static int (*nec_fb_drm_fb_truncate)(struct drm_fb_helper *fb_helper, size_t size) = NULL;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static u32 nec_fb_pseudo_palette[NEC_FB_NUM_PALETTE_COLOR] = {
	0x00000000, 0xffffffff, 0xffffffff, 0xffffffff,
	0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
	0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff,
	0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff
};
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB


/*****************************************************************************
 * Private function declairs
 *****************************************************************************/
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int  nec_fb_open_common(struct nec_fb_data *nfd);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
static int  nec_fb_release_common(struct nec_fb_data *nfd,
					bool force_release);

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int  nec_fb_blank_unblank(struct nec_fb_data *nfd);
static int  nec_fb_blank_blank(struct nec_fb_data *nfd);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

static void nec_fb_print_fb_dump(struct nec_fb_data *nfd,
					int page, int line_s, int line_e);

static u16  nec_fb_get_color(int color_id);


/*****************************************************************************
 * NEC FB systemcall operation declairs
 *****************************************************************************/
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int  nec_fb_open(struct fb_info *info, int user);
static int  nec_fb_release(struct fb_info *info, int user);
static int  nec_fb_pan_display(struct fb_var_screeninfo *var,
						struct fb_info *info);
static int  nec_fb_check_var(struct fb_var_screeninfo *var,
						struct fb_info *info);
static int  nec_fb_set_par(struct fb_info *info);
static int  nec_fb_blank(int blank_mode, struct fb_info *info);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static struct fb_ops nec_fb_ops = {
	.owner		= THIS_MODULE,
	.fb_open	= nec_fb_open,
	.fb_release	= nec_fb_release,
	.fb_blank	= nec_fb_blank,
	.fb_pan_display	= nec_fb_pan_display,
	.fb_check_var	= nec_fb_check_var,
	.fb_set_par	= nec_fb_set_par,
};
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB


/*****************************************************************************
 * NEC FB SysFs operation declairs
 *****************************************************************************/
#define NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR( _name ) \
static ssize_t nec_fb_show_##_name (struct device *dev, \
	struct device_attribute *attr, char *buf); \
static ssize_t nec_fb_store_##_name (struct device *dev, \
	struct device_attribute *attr, const char *buf, size_t count); \
static DEVICE_ATTR( _name , (S_IRUGO|S_IWUSR) , \
	nec_fb_show_##_name , nec_fb_store_##_name )

#define NEC_FB_SYSFS_SHOW__FUNC_DECLAIR( _name ) \
static ssize_t nec_fb_show_##_name (struct device *dev, \
	struct device_attribute *attr, char *buf); \
static DEVICE_ATTR( _name , S_IRUGO , \
	nec_fb_show_##_name , NULL )

#define NEC_FB_SYSFS_STORE_FUNC_DECLAIR( _name ) \
static ssize_t nec_fb_store_##_name (struct device *dev, \
	struct device_attribute *attr, const char *buf, size_t count); \
static DEVICE_ATTR( _name , S_IWUSR, \
	NULL , nec_fb_store_##_name )

NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_enable);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_disp);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_cmd);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_cmd_data);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_backlight);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_brightness);
NEC_FB_SYSFS_SHOW_STORE_FUNC_DECLAIR(lcd_refresh_rate);
NEC_FB_SYSFS_STORE_FUNC_DECLAIR(lcd_send_frame);
NEC_FB_SYSFS_STORE_FUNC_DECLAIR(lcd_reset);
NEC_FB_SYSFS_STORE_FUNC_DECLAIR(lcd_user_enable);
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
NEC_FB_SYSFS_SHOW__FUNC_DECLAIR(panel_perf_print);
#endif // NEC_PANEL_ENABLE_PERF_CHECK
NEC_FB_SYSFS_STORE_FUNC_DECLAIR(fb_fill);
NEC_FB_SYSFS_STORE_FUNC_DECLAIR(fb_dump);

static struct attribute *nec_fb_attrs[] = {
	&dev_attr_lcd_enable.attr,
	&dev_attr_lcd_disp.attr,
	&dev_attr_lcd_cmd.attr,
	&dev_attr_lcd_cmd_data.attr,
	&dev_attr_lcd_backlight.attr,
	&dev_attr_lcd_brightness.attr,
	&dev_attr_lcd_refresh_rate.attr,
	&dev_attr_lcd_reset.attr,
	&dev_attr_lcd_send_frame.attr,
	&dev_attr_lcd_user_enable.attr,
#ifdef NEC_PANEL_ENABLE_PERF_CHECK
	&dev_attr_panel_perf_print.attr,
#endif // NEC_PANEL_ENABLE_PERF_CHECK
	&dev_attr_fb_fill.attr,
	&dev_attr_fb_dump.attr,
	NULL,
};

static struct attribute_group nec_fb_attr_group = {
	.attrs = nec_fb_attrs,
};


/*****************************************************************************
 * Power manegement operation declairs
 *****************************************************************************/
static int  nec_fb_suspend_common(struct nec_fb_data *nfd);
static int  nec_fb_resume_common(struct nec_fb_data *nfd);
#if defined(CONFIG_PM) && !defined(CONFIG_PM_SLEEP)
static int  nec_fb_suspend(struct platform_device *pdev, pm_message_t state);
static int  nec_fb_resume(struct platform_device *pdev);
#else
#define nec_fb_suspend	NULL
#define nec_fb_resume	NULL
#endif /* defined(CONFIG_PM) && !defined(CONFIG_PM_SLEEP) */
#ifdef CONFIG_PM_SLEEP
static int  nec_fb_pm_suspend(struct device *dev);
static int  nec_fb_pm_resume(struct device *dev);
#endif // CONFIG_PM_SLEEP


/*****************************************************************************
 * Driver register/unregister operation declairs
 *****************************************************************************/
static int  nec_fb_probe(struct platform_device *pdev);
static int  nec_fb_init_fbinfo(struct nec_fb_data *nfd);
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static void nec_fb_init_fb_resolution(struct nec_fb_data *nfd,
struct nec_panel_info *panel_info, struct fb_var_screeninfo *var);
static int  nec_fb_alloc_fb_mem(struct nec_fb_data *nfd);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
static int  nec_fb_create_sysfs(struct nec_fb_data *nfd);

static int  nec_fb_remove(struct platform_device *pdev);
static void nec_fb_remove_sysfs(struct nec_fb_data *nfd);

static void nec_fb_shutdown(struct platform_device *pdev);


/*****************************************************************************
 *  Private functions
 *****************************************************************************/
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int nec_fb_open_common(struct nec_fb_data *nfd)
{
	struct fb_info *fbi = nfd->fbi;
	int err = 0;

	/* Sync runtime */
	err = pm_runtime_get_sync(fbi->dev);
	if (err < 0) {
		NEC_FB_PRINT("Failed to wake up pm_runtime(err=%d)\n", err);
		return err;
	}

	/* On the nec panel if first open */
	if (nfd->ref_cnt == 0) {
		err = nec_fb_blank(FB_BLANK_UNBLANK, fbi);
		if (err) {
			NEC_FB_PRINT_ERR("Failed to turn on fb%d."
					"(err=%d)\n", nfd->id, err);
			return err;
		}
		NEC_FB_PRINT("turn on fb%d\n", nfd->id);
	}
	nfd->ref_cnt++;
	pm_runtime_put(fbi->dev);

	NEC_FB_DPRINT("fb%d: ref=%d\n", nfd->id, nfd->ref_cnt);
	return 0;
}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

static int nec_fb_release_common(struct nec_fb_data *nfd, bool force_release)
{
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->fbi;
	int err = 0;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	/* Force FB release */
	if (force_release) {
		NEC_FB_PRINT("Force release fb\n");
		nfd->ref_cnt = 1;
	}

	/* Off the nec panel if last close */
	if (nfd->ref_cnt == 1) {
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
		err = nec_fb_blank(FB_BLANK_NORMAL, fbi);
		if (err) {
			NEC_FB_PRINT_ERR("Failed to turn off fb%d."
					"(err=%d)\n", nfd->id, err);
			return err;
		}
		NEC_FB_PRINT("turn off fb%d\n", nfd->id);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	}
	nfd->ref_cnt--;
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	pm_runtime_put(fbi->dev);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	NEC_FB_DPRINT("fb%d: ref=%d\n", nfd->id, nfd->ref_cnt);
	return 0;
}

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int nec_fb_blank_unblank(struct nec_fb_data *nfd)
{
#ifndef CONFIG_64BIT
	int err = 0;
#else
	long err = 0;
#endif
	struct nec_fb_panel_operation *panel_ops = nfd->panel_ops;

	if (!panel_ops)
		return -ENODEV;

	/* Panel is on */
	if (panel_ops->panel_on) {
		err = panel_ops->panel_on(NEC_PANEL_USER_FB);
		if (err) {
			return err;
		}
	}

	NEC_FB_DPRINT("done\n");
	return 0;
}

static int nec_fb_blank_blank(struct nec_fb_data *nfd)
{
	struct nec_fb_panel_operation *panel_ops = nfd->panel_ops;

	if (!panel_ops)
		return -ENODEV;

	/* Force panel off, if shutdown started */
	if (nfd->shutdown_pending) {
		if (panel_ops->panel_force_off) {
			panel_ops->panel_force_off();

			NEC_FB_DPRINT("force panel off.\n");
			return 0;
		}
	}

	/* Panel is off */
	if (panel_ops->panel_off)
		panel_ops->panel_off(NEC_PANEL_USER_FB);

	NEC_FB_DPRINT("done\n");
	return 0;
}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

static void nec_fb_print_fb_dump(struct nec_fb_data *nfd,
				int page, int line_s, int line_e)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->drm_fbi;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->fbi;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_fix_screeninfo *fix = &(fbi->fix);
	struct fb_var_screeninfo *var = &(fbi->var);
	u16 *start = NULL;
	u32 fb_offset = 0;
	u16 pixel_data;
	u32 xres, yres, sz;
	u32 row, col, idx;

	xres = var->xres;
	yres = var->yres;
	sz = fix->line_length * yres;
	fb_offset = ((page == 1) ? sz : 0);
	start = (u16 *)(fbi->screen_base + fb_offset);
	//TODO: Make the address sure for DRM framebuffer
	{
		NEC_FB_PRINT("screen_base=0x%p\n", fbi->screen_base);
		return;
	}

#ifndef CONFIG_64BIT
	printk("FB DATA DUMP (page=%d/start=0x%08x/x=%d/y=%d/sz=0x%08x): "
		"line %d-%d\n", page, (int)start, xres, yres, sz,
		line_s, line_e);
#else
	printk("FB DATA DUMP (page=%d/start=0x%p/x=%d/y=%d/sz=0x%08x): "
		"line %d-%d\n", page, start, xres, yres, sz,
		line_s, line_e);
#endif

	for (row = 0; row < yres; row++) {
		if ((row < line_s) || (line_e < row))
			continue;

		printk("\nline: %4d", row);
		for (col = 0; col < xres; col++) {
			idx = (xres * row) + col;
			if (!(idx & 0xF))
				printk("\n");

			pixel_data = *(start + idx);
			printk("%04X ", pixel_data);
		}
		printk("\n");
	}
}

static u16 nec_fb_get_color(int color_id)
{
	switch (color_id) {
	/* each bit is on */
	case NEC_FB_COLOR_ID_B0: 	return NEC_FB_COLOR_B0;
	case NEC_FB_COLOR_ID_B1: 	return NEC_FB_COLOR_B1;
	case NEC_FB_COLOR_ID_B2: 	return NEC_FB_COLOR_B2;
	case NEC_FB_COLOR_ID_B3: 	return NEC_FB_COLOR_B3;
	case NEC_FB_COLOR_ID_B4: 	return NEC_FB_COLOR_B4;
	case NEC_FB_COLOR_ID_G0: 	return NEC_FB_COLOR_G0;
	case NEC_FB_COLOR_ID_G1: 	return NEC_FB_COLOR_G1;
	case NEC_FB_COLOR_ID_G2: 	return NEC_FB_COLOR_G2;
	case NEC_FB_COLOR_ID_G3: 	return NEC_FB_COLOR_G3;
	case NEC_FB_COLOR_ID_G4: 	return NEC_FB_COLOR_G4;
	case NEC_FB_COLOR_ID_G5: 	return NEC_FB_COLOR_G5;
	case NEC_FB_COLOR_ID_R0: 	return NEC_FB_COLOR_R0;
	case NEC_FB_COLOR_ID_R1: 	return NEC_FB_COLOR_R1;
	case NEC_FB_COLOR_ID_R2: 	return NEC_FB_COLOR_R2;
	case NEC_FB_COLOR_ID_R3: 	return NEC_FB_COLOR_R3;
	case NEC_FB_COLOR_ID_R4: 	return NEC_FB_COLOR_R4;
	/* common colors */
	case NEC_FB_COLOR_ID_GRAY: 	return NEC_FB_COLOR_GRAY;
	case NEC_FB_COLOR_ID_MAGENTA:	return NEC_FB_COLOR_MAGENTA;
	case NEC_FB_COLOR_ID_CYAN:	return NEC_FB_COLOR_CYAN;
	case NEC_FB_COLOR_ID_YELLOW:	return NEC_FB_COLOR_YELLOW;
	case NEC_FB_COLOR_ID_BLUE:	return NEC_FB_COLOR_BLUE;
	case NEC_FB_COLOR_ID_GREEN:	return NEC_FB_COLOR_GREEN;
	case NEC_FB_COLOR_ID_RED:	return NEC_FB_COLOR_RED;
	case NEC_FB_COLOR_ID_WHITE:	return NEC_FB_COLOR_WHITE;
	case NEC_FB_COLOR_ID_BLACK:
	default:			return NEC_FB_COLOR_BLACK;
	}
}

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
/*****************************************************************************
 * NEC FB systemcall operations
 *****************************************************************************/
static int nec_fb_open(struct fb_info *info, int user)
{
	struct nec_fb_data *nfd = (struct nec_fb_data *)info->par;
	int err = 0;

	if (!nfd)
		return -ENODEV;

	/* duaring shutdown */
	if (nfd->shutdown_pending) {
		pr_err_once("Shutdown pending. Skip it.\n");
		return -ESHUTDOWN;
	}

	err = nec_fb_open_common(nfd);
	return err;
}

static int nec_fb_release(struct fb_info *info, int user)
{
	struct nec_fb_data *nfd = (struct nec_fb_data *)info->par;
	struct task_struct *task = current->group_leader;
	int err = 0;

	if (!nfd)
		return -ENODEV;

	if (nfd->ref_cnt <= 0) {
		NEC_FB_PRINT("Unopened fb close! Skip it. "
				"(request from pid:%d name:%s)\n",
				current->tgid, task->comm);
		return -EINVAL;
	}

	err = nec_fb_release_common(nfd, false);
	return err;
}

static int nec_fb_blank(int blank_mode, struct fb_info *info)
{
	struct nec_fb_data *nfd = (struct nec_fb_data *)info->par;

	if (!nfd)
		return -ENODEV;

	NEC_FB_DPRINT("mode: %d\n", blank_mode);
	switch (blank_mode) {
	case FB_BLANK_UNBLANK:
		nec_fb_blank_unblank(nfd);
		break;
	case FB_BLANK_NORMAL:
	case FB_BLANK_POWERDOWN:
		nec_fb_blank_blank(nfd);
		break;
	case FB_BLANK_VSYNC_SUSPEND:
	case FB_BLANK_HSYNC_SUSPEND:
	default:
		NEC_FB_DPRINT("Unsupport operation(mode=%d)\n", blank_mode);
		return 0;
	}

	return 0;
}

static int nec_fb_pan_display(struct fb_var_screeninfo *var,
                               struct fb_info *info)
{
	return 0;
}

static int nec_fb_check_var(struct fb_var_screeninfo *var,
                             struct fb_info *info)
{
	struct nec_fb_data *nfd = (struct nec_fb_data *)info->par;
	const struct fb_videomode *mode;

	if (!nfd)
		return -ENODEV;

	if ((var->rotate != FB_ROTATE_UR) &&
		(var->rotate != FB_ROTATE_UD)) {
		return -EINVAL;
	}

	switch (var->bits_per_pixel) {
	case 16:
		if (
			(var->green.offset != 5) ||
			!((var->blue.offset == 11) ||
				(var->blue.offset == 0)) ||
			!((var->red.offset == 11) ||
				(var->red.offset == 0)) ||
			(var->blue.length != 5) ||
			(var->green.length != 6) ||
			(var->red.length != 5) ||
			(var->blue.msb_right != 0) ||
			(var->green.msb_right != 0) ||
			(var->red.msb_right != 0) ||
			(var->transp.offset != 0) ||
			(var->transp.length != 0)
			) {
			return -EINVAL;
		}
		break;
	case 24:
	case 32:
	default:
		return -EINVAL;
	}

	if ((var->xres_virtual <= 0) || (var->yres_virtual <= 0))
		return -EINVAL;

	if ((var->xres == 0) || (var->yres == 0)) {
		return -EINVAL;
	}

	if ((var->xres_virtual - var->xres) < var->xoffset)
		return -EINVAL;

	if ((var->yres_virtual - var->yres) < var->yoffset)
		return -EINVAL;

	if (info->mode) {
		mode = fb_match_mode(var, &info->modelist);
		if (!mode)
			return -EINVAL;
	}

	NEC_FB_DPRINT("done\n");
	return 0;
}

static int nec_fb_set_par(struct fb_info *info)
{
	struct nec_fb_data *nfd = (struct nec_fb_data *)info->par;
	struct fb_fix_screeninfo *fix;
	struct fb_var_screeninfo *var;
	const struct fb_videomode *mode;

	/* Get fb info */
	fix = &(info->fix);
	var = &(info->var);

	/* Check parameters */
	switch (var->bits_per_pixel) {
	case 16:
		if (var->red.offset == 0) {
			return -EINVAL;
		}
		break;
	case 24:
	case 32:
	default:
		return -EINVAL;
	}

	if (info->mode) {
		mode = fb_match_mode(var, &(info->modelist));
		if (!mode)
			return -EINVAL;

		if (fb_mode_is_equal(mode, info->mode)) {
			NEC_FB_DPRINT("mode is equal to current mode\n");
			return 0;
		}
	}

	/* Update bytes per line */
	fix->line_length = (var->xres * var->bits_per_pixel) /
					NEC_PANEL_BITS_PER_BYTE;

	/* if memory is not allocated yet, change memory size for fb */
	if (!fix->smem_start) {
		fix->smem_len =
			PAGE_ALIGN(fix->line_length * nfd->fbi->var.yres) *
			nfd->num_page;
	}

	NEC_FB_DPRINT("done\n");
	return 0;
}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB


/*****************************************************************************
 * NEC FB SysFs operations
 *****************************************************************************/
static ssize_t nec_fb_show_lcd_enable(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	bool panel_is_on;
	int on = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Get nec panel state */
	panel_is_on = nfd->panel_ops->panel_is_on();
	on = ((panel_is_on) ? 1 : 0);
	return snprintf(buf, PAGE_SIZE, "%d\n", on);
}

static ssize_t nec_fb_store_lcd_enable(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long on = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &on);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
			buf, err);
		return -EINVAL;
	}
	if ((on != 1) && (on != 0)) {
		return -EINVAL;
	}

	/* Operate nec panel */
	if (on) {
		err = nfd->panel_ops->panel_on(NEC_PANEL_USER_FB);
		if (err)
			len = -EBUSY;
	}
	else
		nfd->panel_ops->panel_off(NEC_PANEL_USER_FB);

	return len;
}

static ssize_t nec_fb_show_lcd_disp(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	int on = 0;

	if (!nfd || !(nfd->panel_res))
		return -ENODEV;

	/* Get nec panel display state */
	on = ((nfd->panel_res->display_is_on) ? 1 : 0);
	return snprintf(buf, PAGE_SIZE, "%d\n", on);
}

static ssize_t nec_fb_store_lcd_disp(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long on = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &on);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
			buf, err);
		return -EINVAL;
	}
	if ((on != 1) && (on != 0)) {
		return -EINVAL;
	}

	/* Operate nec panel display */
	if (on)
		nfd->panel_ops->panel_disp_on();
	else
		nfd->panel_ops->panel_disp_off();

	return len;
}

static ssize_t nec_fb_show_lcd_cmd(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	if (!nfd)
		return -ENODEV;

	/* Get LCD comand */
	return snprintf(buf, PAGE_SIZE, "0x%02x\n", nfd->lcd_cmd);
}

static ssize_t nec_fb_store_lcd_cmd(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	char *last = NULL;
	int cmd = 0;

	if (!nfd)
		return -ENODEV;

	/* Parse option */
	cmd = (u8)simple_strtoul(buf, &last, 0);
	if ((cmd < NEC_PANEL_MIN_LCD_CMD) ||
		(cmd > NEC_PANEL_MAX_LCD_CMD)) {
		return -EINVAL;
	}

	/* Set LCD command */
	nfd->lcd_cmd = cmd;
	NEC_FB_DPRINT("Update lcd_cmd=0x%02x\n", nfd->lcd_cmd);

	return count;
}

static ssize_t nec_fb_show_lcd_cmd_data(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long data_cnt = NEC_PANEL_MAX_LCD_CMD_DATA;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Read LCD data */
	nfd->panel_ops->read_panel_cmd(nfd->lcd_cmd, data_cnt);
	return 0;
}

static ssize_t nec_fb_store_lcd_cmd_data(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB
	int len = count;
	char *last = NULL;
	u8 data[NEC_PANEL_MAX_LCD_CMD_DATA] = { 0 };
	int data_cnt = 0;
	int i = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	/* 	Get data count */
	data_cnt = (u8)simple_strtoul(buf, &last, 0);
	last++;
	if ((data_cnt < NEC_PANEL_MIN_LCD_CMD_DATA) ||
		(data_cnt > NEC_PANEL_MAX_LCD_CMD_DATA)) {
		return -EINVAL;
	}
	/* 	Get LCD command data */
	for (i = 0; i < data_cnt; i++) {
		if (count <= (last - buf))
			return -EINVAL;

		data[i] = (u8)simple_strtoul(last, &last, 0);
		last++;
	}

	/* Send LCD command data */
	nfd->panel_ops->write_panel_cmd(nfd->lcd_cmd, data, data_cnt);

	return len;
}

static ssize_t nec_fb_show_lcd_backlight(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	int backlight = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Get LCD backlight status */
	backlight = nfd->panel_ops->get_backlight();
	return snprintf(buf, PAGE_SIZE, "%d\n", backlight);
}

static ssize_t nec_fb_store_lcd_backlight(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long on = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &on);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
				buf, err);
		return -EINVAL;
	}
	if ((on != 1) && (on != 0)) {
		return -EINVAL;
	}

	/* Set LCD backlight */
	err = nfd->panel_ops->set_backlight(on);
	if (err)
		len = -EBUSY;

	return len;
}

static ssize_t nec_fb_show_lcd_brightness(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	int brightness = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Get LCD backlight brightness */
	brightness = nfd->panel_ops->get_brightness();
	return snprintf(buf, PAGE_SIZE, "%d\n", brightness);
}

static ssize_t nec_fb_store_lcd_brightness(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long brightness = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &brightness);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
				buf, err);
		return -EINVAL;
	}
	if ((brightness < NEC_PANEL_MIN_LCD_BRIGHTNESS) ||
		(brightness > NEC_PANEL_MAX_LCD_BRIGHTNESS)) {
		return -EINVAL;
	}

	/* Set LCD backlight brightness */
	err = nfd->panel_ops->set_brightness(brightness);
	if (err)
		len = -EBUSY;

	return len;
}

static ssize_t nec_fb_show_lcd_refresh_rate(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	int refresh_rate = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Get LCD refresh rate */
	refresh_rate = nfd->panel_ops->get_refresh_rate();
	return snprintf(buf, PAGE_SIZE, "%d\n", refresh_rate);
}

static ssize_t nec_fb_store_lcd_refresh_rate(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
	struct fb_var_screeninfo *var = &(fbi->var);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_panel_info panel_info = {0};
	unsigned long refresh_rate = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &refresh_rate);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
				buf, err);
		return -EINVAL;
	}

	/* Set LCD backlight brightness */
	err = nfd->panel_ops->set_refresh_rate(refresh_rate);
	if (err)
		return -EBUSY;

	/* Update fbinfo */
	nec_panel_get_panel_info(&panel_info);
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	nec_fb_init_fb_resolution(nfd, &panel_info, var);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	return len;
}

static ssize_t nec_fb_store_lcd_reset(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	unsigned long hw_reset = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	err = kstrtoul(buf, 10, &hw_reset);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to convert value (buf=%s, err=%d)\n",
				buf, err);
		return -EINVAL;
	}
	if ((hw_reset != 1) && (hw_reset != 0)) {
		return -EINVAL;
	}

	/* Reset LCD device */
	if (hw_reset)
		nfd->panel_ops->panel_reset(true, false);
	else /* sw_reset */
		nfd->panel_ops->panel_reset(false, true);

	return len;
}

static ssize_t nec_fb_store_lcd_send_frame(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
	struct fb_info *fbi = nfd->drm_fbi;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_var_screeninfo var;
	int len = count;
	char *last = NULL;
	int fb_page = 0;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	fb_page = simple_strtoul(buf, &last, 0);
	fb_page = ((fb_page != 0) ? 1 : 0);

	/* Create var info */
	memcpy(&var, &(fbi->var), sizeof(struct fb_var_screeninfo));
	var.yoffset = ((fb_page == 0) ? 0 : var.yres);

#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	err = drm_fb_helper_pan_display(&var, fbi);
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	err = nec_fb_pan_display(&var, fbi);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	if (err)
		len = -EBUSY;

	return len;
}

static ssize_t nec_fb_store_lcd_user_enable(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	char *last = NULL;
	int on = 0;
	int user = 0;
	int len = count;
	int err = 0;

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	/* Parse option */
	/*      Get on/off */
	on = simple_strtoul(buf, &last, 0);
	last++;
	if (count <= (last - buf))
			return -EINVAL;
	on = ((on != 0) ? 1 : 0);
	/*      Get start row */
	user = (u8)simple_strtoul(last, &last, 0);
	user = ((user != NEC_PANEL_USER_FB) ?
		NEC_PANEL_USER_SPLASH : NEC_PANEL_USER_FB);

	/* Operate nec panel */
	if (on) {
		err = nfd->panel_ops->panel_on(user);
		if (err)
			len = -EBUSY;
	}
	else
		nfd->panel_ops->panel_off(user);

	return len;
}

#ifdef NEC_PANEL_ENABLE_PERF_CHECK
static ssize_t nec_fb_show_panel_perf_print(struct device *dev,
		struct device_attribute *attr, char *buf)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	if (!nfd || !(nfd->panel_ops))
		return -ENODEV;

	nfd->panel_ops->print_perf();
	return 0;
}
#endif // NEC_PANEL_ENABLE_PERF_CHECK

static ssize_t nec_fb_store_fb_fill(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
	struct fb_info *fbi = nfd->drm_fbi;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	int len = count;
	char *last = NULL;
	int page = 0;
	int offset = 0;
	u16 *start;
	int color_id, color;
	int w, h, idx;

	/* Parse option */
	/* 	Get fb page */
	page = simple_strtoul(buf, &last, 0);
	last++;
	if (count <= (last - buf))
			return -EINVAL;

	page = ((page != 0) ? 1 : 0);
	offset = ((page == 1) ? (fbi->fix.line_length * fbi->var.yres) : 0);
	/* 	Get color id */
	color_id = simple_strtoul(last, &last, 0);
	color = nec_fb_get_color(color_id);
	NEC_FB_DPRINT("fb_page=%d/color=%d\n", page, color);

	/* Set fb page address */
	start = (u16 *)(nfd->panel_res->fb_buf.virt + offset);
	//TODO: Make the address sure for DRM framebuffer
	{
		NEC_FB_DPRINT("offset=%d, line_len=%d, yres=%d\n", offset, fbi->fix.line_length, fbi->var.yres);
		NEC_FB_DPRINT("virtual addr=0x%p\n", nfd->panel_res->fb_buf.virt);
		return len;
	}
#ifndef CONFIG_64BIT
	NEC_FB_DPRINT("fill fb (page=%d/addr:0x%08x)\n",
			(int)page, (int)start);
#else
	NEC_FB_DPRINT("fill fb (page=%d/addr:0x%p)\n",
			(int)page, start);
#endif
	/* Fill FB */
	for (h = 0; h < fbi->var.yres; h++) {
		for (w = 0; w < fbi->var.xres; w++) {
			idx = (h * fbi->var.xres) + w;
			start[idx] = color;
		}
	}

	return len;
}

static ssize_t nec_fb_store_fb_dump(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t count)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = _nfd;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = dev_get_drvdata(dev);
	struct nec_fb_data *nfd = (struct nec_fb_data *)fbi->par;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	char *last = NULL;
	int page, line_s, line_e;

	if (!nfd)
		return -ENODEV;

	/* Parse option */
	/*      Get page */
	page = simple_strtoul(buf, &last, 0);
	last++;
	if (count <= (last - buf))
			return -EINVAL;
	page = ((page != 0) ? 1 : 0);
	/*      Get start row */
	line_s = simple_strtoul(last, &last, 0);
	last++;
	if (count <= (last - buf))
			return -EINVAL;
	/*      Get end row */
	line_e = simple_strtoul(last, &last, 0);

	/* Print FB Dump */
	nec_fb_print_fb_dump(nfd, page, line_s, line_e);
	return count;
}

/*****************************************************************************
 *  Public functions
 *****************************************************************************/
int nec_fb_set_panel_operation(struct nec_fb_panel_operation *panel_ops)
{
	if (nec_fb_panel_ops) {
		NEC_FB_PRINT_ERR("multiple regist the panel operation\n");
		return -EINVAL;
	}

	nec_fb_panel_ops = panel_ops;
	NEC_FB_DPRINT("Set nec panel operation\n");
	return 0;
}

struct drm_fb_helper *get_drm_fb_helper(void)
{
	if (_helper) {
		NEC_FB_PRINT_ERR("DRM fb helper = %p, info = %p\n", _helper, _helper->fbdev);
	}
	return _helper;
}

void nec_fb_set_original_fb_info(struct drm_fb_helper *helper, struct fb_var_screeninfo *ovar)
{
	struct fb_info *fbi;
	struct drm_framebuffer *fb;

	fbi = helper->fbdev;	// Use DRM's FB info
	if (!fbi) {
		NEC_FB_PRINT_ERR("exit, not yet registered DRM framebuffer \n");
		return;
	}
	fb = helper->fb;
	if (!fb) {
		NEC_FB_PRINT_ERR("exit, FB is invalid\n");
		return;
	}

	// Save current variable FB info
	if (ovar) {
		memcpy(ovar, &(fbi->var), sizeof(struct fb_var_screeninfo));
	}

	// Default working area is bottom area of virtual screen
	fbi->var.yoffset = fbi->var.yres_virtual - fbi->var.yres;
#if CONFIG_DRM_NEC_FB_RESERVED >= 100
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	// Recover original DRM FB info
	memcpy(&(fbi->var), &(_fbi_original.var), sizeof(struct fb_var_screeninfo));
	fbi->fix.smem_len = _fbi_original.smem_len;
	fbi->screen_size = _fbi_original.screen_size;
	fb->height = _fbi_original.fb_height;

	nec_fb_drm_fb_truncate(helper, _fbi_original.obj_size);

	// Set y position for NEC proprietary in the area of original DRM FB
	fbi->var.yoffset = _fbi_tricked.var.yres_virtual;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	// Save contents in DRM FB to NEC FB
	if (_nfd && _nfd->fbi && _nfd->fbi->screen_base) {
		int len =  (fbi->var.yres * CONFIG_DRM_NEC_FB_RESERVED / 100) * fbi->fix.line_length;
		// Above half is contents saving area, bottom half is NEC specific contents area
		memcpy(_nfd->fbi->screen_base, fbi->screen_base + fbi->var.yres_virtual * fbi->fix.line_length - len, len);
		memcpy(fbi->screen_base + fbi->var.yres_virtual * fbi->fix.line_length - len, _nfd->fbi->screen_base + len, len);
	}
	else {
		NEC_FB_PRINT_ERR("Contents not saved\n");
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100
}

void nec_fb_set_tricked_fb_info(struct drm_fb_helper *helper, struct fb_var_screeninfo *ovar)
{
	struct fb_info *fbi;
	struct drm_framebuffer *fb;

	fbi = helper->fbdev;	// Use DRM's FB info
	if (!fbi) {
		NEC_FB_PRINT_ERR("exit, not yet registered DRM framebuffer \n");
		return;
	}
	fb = helper->fb;
	if (!fb) {
		NEC_FB_PRINT_ERR("exit, FB is invalid\n");
		return;
	}

#if CONFIG_DRM_NEC_FB_RESERVED >= 100
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	nec_fb_drm_fb_truncate(helper, _fbi_tricked.obj_size);

	// Recover application FB related info
	fbi->fix.smem_len = _fbi_tricked.smem_len;
	fbi->screen_size = _fbi_tricked.screen_size;
	fb->height = _fbi_tricked.fb_height;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	// Recover contents in DRM FB from NEC FB
	if (_nfd && _nfd->fbi && _nfd->fbi->screen_base) {
		int len =  (fbi->var.yres * CONFIG_DRM_NEC_FB_RESERVED / 100) * fbi->fix.line_length;
		// Above half is contents saving area, bottom half is NEC specific contents area
		memcpy(_nfd->fbi->screen_base + len, fbi->screen_base + fbi->var.yres_virtual * fbi->fix.line_length - len, len);
		memcpy(fbi->screen_base + fbi->var.yres_virtual * fbi->fix.line_length - len, _nfd->fbi->screen_base, len);
	}
	else {
		NEC_FB_PRINT_ERR("Contents not recovered\n");
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100

	// Recover variable FB info
	if (ovar) {
		memcpy(&(fbi->var), ovar, sizeof(struct fb_var_screeninfo));
	}
}

void nec_drm_fb_attach(struct drm_fb_helper *helper, int (*trunc)(struct drm_fb_helper *fb_helper, size_t size))
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
#if CONFIG_DRM_NEC_FB_RESERVED >= 100
	struct fb_info *fbi;
	struct fb_var_screeninfo *var;
	struct fb_fix_screeninfo *fix;
	struct drm_framebuffer *fb;
	u32 nec_yres_virtual;
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB

	NEC_FB_DPRINT("start \n");
	if (helper) {
		_helper = helper;
		NEC_FB_DPRINT("DRM FB attached info:%p\n", helper->fbdev);

#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
#if CONFIG_DRM_NEC_FB_RESERVED >= 100
		nec_fb_drm_fb_truncate = trunc;
		fbi = helper->fbdev;
		if (!fbi) {
			NEC_FB_PRINT_ERR("exit, not yet registered DRM framebuffer \n");
			return;	// Not assigned
		}
		fb = helper->fb;
		if (!fb) {
			NEC_FB_PRINT_ERR("exit, FB is invalid\n");
			return;
		}

		/*
		 * Use NEC proprietary space in /dev/fb0
		 *
		 * /dev/fb0 (allocated whole area at one time)
		 * +----------------------------+  =+
		 * | application space          |   |
		 * | page 0                     |   |
		 * |                            |   |
		 * |                            |   |
		 * +- - - - - - - - - - - - - - +   |
		 * | page 1                     |   | overall area size is decided by CONFIG_DRM_FBDEV_OVERALLOC
		 * |                            |   |
		 * |                            |   |
		 * |                            |   |
		 * +----------------------------+   |  =+
		 * | NEC proprietary space      |   |   |
		 * | page 0                     |   |   | hidden area from application
		 * |                            |   |   | by changing framebuffer info
		 * |                            |   |   |
		 * +- - - - - - - - - - - - - - +   |   |
		 * | page 1                     |   |   | area size is decided by CONFIG_DRM_NEC_FB_RESERVED
		 * |                            |   |   |
		 * |                            |   |   |
		 * |                            |   |   |
		 * +----------------------------+  =+  =+
		 *
		 */

		var = &(fbi->var);
		fix = &(fbi->fix);

		// Save original DRM FB info
		memcpy(&(_fbi_original.var), var, sizeof(struct fb_var_screeninfo));
		_fbi_original.smem_len = fix->smem_len;
		_fbi_original.screen_size = fbi->screen_size;	// fbi->screen_size is used for read/write I/O
		_fbi_original.fb_height = fb->height;
		_fbi_original.obj_size = nec_fb_drm_fb_truncate(helper, 0);	// this is used for mmap I/O
		NEC_FB_DPRINT("DRM FB (%dx%d), pitches[]: %d %d %d %d, offsets[]: %d %d %d %d\n",
				fb->width, fb->height, fb->pitches[0], fb->pitches[1], fb->pitches[2], fb->pitches[3],
				fb->offsets[0], fb->offsets[1], fb->offsets[2], fb->offsets[3]);

		// Change the size for hiding NEC proprietary space
		nec_yres_virtual = var->yres * (CONFIG_DRM_NEC_FB_RESERVED / 100);
		memcpy(&_fbi_tricked, &_fbi_original, sizeof(struct fb_rel_info));
		var->yres_virtual -= nec_yres_virtual;
		fix->smem_len -= (nec_yres_virtual * fix->line_length);
		fbi->screen_size = fix->smem_len;
		_fbi_tricked.var.yres_virtual = var->yres_virtual;
		_fbi_tricked.smem_len = fix->smem_len;
		_fbi_tricked.screen_size = fbi->screen_size;
		nec_fb_drm_fb_truncate(helper, fix->smem_len);
		_fbi_tricked.obj_size =	nec_fb_drm_fb_truncate(helper, 0);

		NEC_FB_DPRINT("DRM FB original visual (%dx%d) virtual (%dx%d), offset (%d,%d), length: %d obj size: %ld screen_size: %ld\n",
				_fbi_original.var.xres, _fbi_original.var.yres,
				_fbi_original.var.xres_virtual, _fbi_original.var.yres_virtual,
				_fbi_original.var.xoffset, _fbi_original.var.yoffset,
				_fbi_original.smem_len, _fbi_original.obj_size, _fbi_original.screen_size);
		NEC_FB_DPRINT("DRM FB modified visual (%dx%d) virtual (%dx%d), offset (%d,%d), length: %d obj size: %ld screen_size: %ld\n",
				_fbi_tricked.var.xres, _fbi_tricked.var.yres,
				_fbi_tricked.var.xres_virtual, _fbi_tricked.var.yres_virtual,
				_fbi_tricked.var.xoffset, _fbi_tricked.var.yoffset,
				_fbi_tricked.smem_len, _fbi_tricked.obj_size, _fbi_tricked.screen_size);
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB
	}
	NEC_FB_DPRINT("exit \n");
}

void nec_drm_fb_dettach(struct drm_fb_helper *helper)
{
#if CONFIG_DRM_NEC_FB_RESERVED >= 100
	struct fb_info *fbi;
	struct drm_framebuffer *fb;
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100

	NEC_FB_DPRINT("start \n");
	if (_helper && _helper == helper) {
#if CONFIG_DRM_NEC_FB_RESERVED >= 100
		fbi = helper->fbdev;
		if (!fbi) {
			NEC_FB_PRINT_ERR("exit, FB info already invalid \n");
			return;	// Not assigned
		}
		fb = helper->fb;
		if (!fb) {
			NEC_FB_PRINT_ERR("exit, FB is invalid\n");
			return;
		}

		nec_fb_set_original_fb_info(helper, NULL);
#endif // CONFIG_DRM_NEC_FB_RESERVED >= 100

		_helper = NULL;
		NEC_FB_DPRINT("DRM fb dettached\n");
	}
	NEC_FB_DPRINT("exit \n");
}

/*****************************************************************************
 * Driver register/unregister operations
 *****************************************************************************/
static int nec_fb_do_suspend(struct nec_fb_data *nfd)
{
	int err = 0;

	if (!nfd)
		return -ENODEV;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* FB blank */
	nfd->suspend.panel_is_on = nfd->panel_res->panel_is_on;
	if (nfd->suspend.panel_is_on) {
		err = nec_fb_blank_blank(nfd);
		if (err) {
			NEC_FB_PRINT_ERR("Failed to blank fb.\n");
			return err;
		}
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	return err;
}

static int nec_fb_suspend_common(struct nec_fb_data *nfd)
{
	int err = 0;

	err = nec_fb_do_suspend(nfd);
	if (err)
		return err;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Suspend fb_info */
	fb_set_suspend(nfd->fbi, FBINFO_STATE_SUSPENDED);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	return 0;
}

static int nec_fb_do_resume(struct nec_fb_data *nfd)
{
	int err = 0;

	if (!nfd)
		return -ENODEV;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* FB unblank */
	if (nfd->suspend.panel_is_on) {
		err = nec_fb_blank_unblank(nfd);
		if (err) {
			NEC_FB_PRINT_ERR("Failed to unblank fb.\n");
			return err;
		}
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	return err;
}

static int nec_fb_resume_common(struct nec_fb_data *nfd)
{
	int err = 0;

	err = nec_fb_do_resume(nfd);
	if (err)
		return err;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Resume fb_info */
	fb_set_suspend(nfd->fbi, FBINFO_STATE_RUNNING);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	return 0;
}

#if defined(CONFIG_PM) && !defined(CONFIG_PM_SLEEP)
static int nec_fb_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct nec_fb_data *nfd = platform_get_drvdata(pdev);
	int err = 0;
	err = nec_fb_suspend_common(nfd);
	return err;
}

static int nec_fb_resume(struct platform_device *pdev)
{
	struct nec_fb_data *nfd = platform_get_drvdata(pdev);
	int err = 0;
	err = nec_fb_resume_common(nfd);
	return err;
}
#endif /* defined(CONFIG_PM) && !defined(CONFIG_PM_SLEEP) */

#ifdef CONFIG_PM_SLEEP
static int nec_fb_pm_suspend(struct device *dev)
{
	struct nec_fb_data *nfd = dev_get_drvdata(dev);
	int err = 0;
	err = nec_fb_suspend_common(nfd);
	return err;
}

static int nec_fb_pm_resume(struct device *dev)
{
	struct nec_fb_data *nfd = dev_get_drvdata(dev);
	int err = 0;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/*
	 * It is possible that the runtime status of the fb device may
	 * have been active when the system was suspended. Reset the runtime
	 * status to suspended state after a complete system resume.
	 */
	pm_runtime_disable(dev);
	pm_runtime_set_suspended(dev);
	pm_runtime_enable(dev);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	err = nec_fb_resume_common(nfd);
	return err;
}
#endif // CONFIG_PM_SLEEP


/*****************************************************************************/
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
extern struct drm_fb_helper *get_drm_fb_helper(void);
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB

static int nec_fb_probe(struct platform_device *pdev)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct drm_fb_helper *helper;
	struct fb_info *drm_fbi;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = NULL;
	struct fb_info     *fbi = NULL;
	int err = 0;

	printk(KERN_DEBUG "%s enter", __func__);
	/* Wait nec panel driver's probe */
	if (!nec_panel_is_initialized()) {
		NEC_FB_PRINT("Defer probe "
			"until nec_panel's probe is executed.\n");
		return -EPROBE_DEFER;
	}
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	helper = get_drm_fb_helper();
	if (!helper) {
		NEC_FB_PRINT("Defer after DRM FB helper is registered\n");
		return -EPROBE_DEFER;
	}
	drm_fbi = helper->fbdev;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB

	NEC_FB_DPRINT("start probe\n");

	/* Get memory for FB info */
	fbi = framebuffer_alloc(sizeof(struct nec_fb_data), NULL);
	if (!fbi) {
		NEC_FB_PRINT_ERR("can't allocate framebuffer info data!");
		return -ENOMEM;
	}
	_nfd = (struct nec_fb_data *)(fbi->par);
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	if (!drm_fbi)
		NEC_FB_DPRINT("DRM fbi is NULL\n");
	if (!(drm_fbi->par))
		NEC_FB_DPRINT("DRM par is NULL\n");

	// Use fb_info allocated here
	_nfd->drm_fbi = drm_fbi;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB

	/*
	 * Frame Buffer Initialization
	 */
	nfd = (struct nec_fb_data *)(fbi->par);
	nfd->id = 0;
	nfd->fbi = fbi;
	nfd->pdev = pdev;
	nfd->ref_cnt = 0;
	nfd->suspend.panel_is_on = false;
	nfd->shutdown_pending = false;

	/* Set driver data */
	/* 	NOTE: In SysFs Operations,
	 *	dev_get_drvdata(dev) function returns not nfd but fbi. */
	platform_set_drvdata(pdev, nfd);

	/* Get Panel resource */
	nec_panel_get_panel_resource(&(nfd->panel_res));
	nfd->panel_ops = nec_fb_panel_ops;

	/* Init FB info */
	nec_fb_init_fbinfo(nfd);

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Allocate Frame Buffer memory */
	err = nec_fb_alloc_fb_mem(nfd);
	if (err)
		return -ENOMEM;

	/* Allocate CMAP */
	err = fb_alloc_cmap(&(fbi->cmap), NEC_FB_NUM_PALETTES, 0);
	if (err)
		NEC_FB_PRINT_ERR("Failed to allocate fb cmap!\n");

	/* Regist Frame Buffer */
	err = register_framebuffer(fbi);
	if (err) {
		NEC_FB_PRINT_ERR("Failed to regist fb info\n");
		fb_dealloc_cmap(&(fbi->cmap));
		return err;
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	/* Create SysFs */
	err = nec_fb_create_sysfs(nfd);
	if (err) {
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
		fb_dealloc_cmap(&(fbi->cmap));
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
		return err;
	}

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Runtime init */
	err = pm_runtime_set_active(nfd->fbi->dev);
	if (err < 0)
		NEC_FB_PRINT_ERR("Failed to set pm_runtime active\n");
	pm_runtime_enable(nfd->fbi->dev);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	// Get initial panel status
	if (nfd->panel_ops->update_panel_status)
		nfd->panel_ops->update_panel_status();

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	pr_info("fb%d: w:%d/h:%d (a=%d/r=%d/g=%d/b=%d bpp=%d) "
		"registered successfully!\n",
		nfd->id, fbi->var.xres, fbi->var.yres,
		fbi->var.transp.length,
		fbi->var.red.length, fbi->var.green.length,
		fbi->var.blue.length, fbi->var.bits_per_pixel);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	NEC_FB_DPRINT("done\n");
	printk(KERN_DEBUG "%s exit", __func__);
	return 0;
}

static int nec_fb_init_fbinfo(struct nec_fb_data *nfd)
{
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->fbi;
	struct fb_fix_screeninfo *fix = &(fbi->fix);
	struct fb_var_screeninfo *var = &(fbi->var);
	u32 id = 0;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_panel_info panel_info = {0};

	nfd->pixel_format = NEC_FB_DEFAULT_PIXEL_FORMAT;

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	fbi->fbops = &nec_fb_ops;
	fbi->flags = FBINFO_FLAG_DEFAULT;

	fix->type_aux    = 0;
	fix->visual      = FB_VISUAL_TRUECOLOR;
	fix->ywrapstep   = 0;
	fix->mmio_start  = 0;
	fix->mmio_len    = 0;
	fix->accel       = FB_ACCEL_NONE;

	var->xoffset     = 0;
	var->yoffset     = 0;
	var->grayscale   = 0;
	var->nonstd      = 0;
	var->activate    = FB_ACTIVATE_VBL;
	var->height      = 0; /* mm */
	var->width       = 0; /* mm */
	var->accel_flags = 0;
	var->sync        = 0;
	var->rotate      = 0;

	switch (nfd->pixel_format) {
	case NEC_FB_RGB_565:
		fix->type = FB_TYPE_PACKED_PIXELS;
		fix->xpanstep        = 1;
		fix->ypanstep        = 1;
		var->vmode           = FB_VMODE_NONINTERLACED;
		var->blue.offset     = 0;
		var->green.offset    = 5;
		var->red.offset      = 11;
		var->blue.length     = 5;
		var->green.length    = 6;
		var->red.length      = 5;
		var->blue.msb_right  = 0;
		var->green.msb_right = 0;
		var->red.msb_right   = 0;
		var->transp.offset   = 0;
		var->transp.length   = 0;
		break;
	default:
		/* NOT REACH HERE. Support Only RGB565 */
		NEC_FB_PRINT_ERR("PixelFormat %d is unknown format!\n",
				nfd->pixel_format);
		return -EINVAL;
	}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	/* Set resolution */
	nec_panel_get_panel_info(&panel_info);
#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	nec_fb_init_fb_resolution(nfd, &panel_info, var);

	/* Set Framebuffer default size
	 *	NOTE: smem_len values will override at nec_fb_alloc_fb_mem() */
	fix->line_length = (var->xres * panel_info.Bpp); /* Bytes per pixel */
	fix->smem_len = PAGE_ALIGN(fix->line_length * var->yres_virtual);

	snprintf(fix->id, sizeof(fix->id), "nec-fb%x", id);
	fbi->pseudo_palette = nec_fb_pseudo_palette;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	NEC_FB_DPRINT("done\n");
	return 0;
}

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static void nec_fb_init_fb_resolution(struct nec_fb_data *nfd,
	struct nec_panel_info *panel_info, struct fb_var_screeninfo *var)
{
	u32 frame_rate = 0;
	unsigned long clk_rate = 0;
	unsigned long h_total = 0;
	unsigned long v_total = 0;

#if CONFIG_DRM_NEC_FB_RESERVED >= 100
	nfd->num_page = (CONFIG_DRM_NEC_FB_RESERVED / 100) * NEC_FB_NUM_FRAMES;
#else // !CONFIG_DRM_NEC_FB_RESERVED >= 100
	nfd->num_page = NEC_FB_NUM_FRAMES;
#endif // !CONFIG_DRM_NEC_FB_RESERVED >= 100

	var->xres           = panel_info->xres;
	var->yres           = panel_info->yres;
	var->xres_virtual   = var->xres;
	var->yres_virtual   = panel_info->yres * nfd->num_page;
	var->bits_per_pixel = panel_info->bpp;

	var->lower_margin = 0;
	var->upper_margin = 0;
	var->vsync_len    = 0;
	var->right_margin = 0;
	var->left_margin  = 0;
	var->hsync_len    = 0;

	/* Culc pixclock */
	frame_rate = panel_info->max_fps;
	h_total = var->xres + var->left_margin +
			var->right_margin + var->hsync_len;
	v_total = var->yres + var->lower_margin +
			var->upper_margin + var->vsync_len;
	clk_rate = h_total * v_total * frame_rate;
	var->pixclock = KHZ2PICOS(clk_rate / 1000);

	NEC_FB_DPRINT("done\n");
	return;
}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
static int nec_fb_alloc_fb_mem(struct nec_fb_data *nfd)
{
	int err = -ENOTSUPP;

	err = nfd->panel_ops->alloc_fb_mem(nfd);
	if (err)
		NEC_FB_PRINT_ERR("Framebuffer allocation err(%d)!\n", err);

	NEC_FB_DPRINT("done\n");
	return err;
}
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

static int nec_fb_create_sysfs(struct nec_fb_data *nfd)
{
	int err = 0;
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->drm_fbi;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->fbi;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct device *dev = fbi->dev;
#ifdef NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0
	struct kobject *fb_kobj = &dev->kobj;
	struct kobject *graphics_kobj = fb_kobj->parent;
	struct kobject *virtual_kobj;

	extern struct kobject *virtual_device_parent(struct device *dev);
#endif // NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0

	err = sysfs_create_group(&(dev->kobj), &(nec_fb_attr_group));
	if (err)
		NEC_FB_PRINT_ERR("Failed to create sysfs group!(%d)\n", err);

#ifdef NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0
	if(!graphics_kobj) {
		NEC_FB_PRINT_ERR("Parent graphics device not found on %s", kobject_name(fb_kobj));
		goto exit;
	}
	printk(KERN_DEBUG "%s: ../%s/%s\n", __func__, kobject_name(graphics_kobj), kobject_name(fb_kobj));
	if (strcmp(kobject_name(graphics_kobj), "graphics") != 0) {
		NEC_FB_PRINT_ERR("Parent device name is not graphics: %s", kobject_name(graphics_kobj));
		goto exit;
	}

	if (strcmp(kobject_name(fb_kobj), "fb0") != 0) {
		err = sysfs_create_link_nowarn(graphics_kobj, fb_kobj, "fb0");	// Create link /sys/devices/virtual/graphics/fb0 -> fb1
		if (err) {
			NEC_FB_PRINT_ERR("Unable to create link to %s err=%d\n", fb_kobj->name, err);
			goto exit;
		}
	}

	virtual_kobj = graphics_kobj->parent;
	if (!virtual_kobj) {
		NEC_FB_PRINT_ERR("Grandparent device not found on %s", kobject_name(graphics_kobj));
		goto exit;
	}
	if (strcmp(kobject_name(virtual_kobj), "virtual") != 0) {	// Real platform device?
		NEC_FB_PRINT_ERR("Grandparent name is not virtual: %s", kobject_name(virtual_kobj));
		// fbi->dev->kobj should be "fb0", not "fb1"
		// We would have /sys/device/virtual/graphics/fb0 for compatile API

		virtual_kobj = virtual_device_parent(NULL);
		err = sysfs_create_link_nowarn(virtual_kobj, graphics_kobj, "graphics");	// Create link /sys/device/virtual/graphics -> /sys/platform/<real-device>/graphics
		if (err) {
			NEC_FB_PRINT_ERR("Unable to create link to %s err=%d\n", kobject_name(graphics_kobj), err);
			goto exit;
		}
	}
exit:
#endif // NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0

	NEC_FB_DPRINT("done\n");
	return err;
}

/*****************************************************************************/
static int nec_fb_remove(struct platform_device *pdev)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct drm_fb_helper *helper;
	struct fb_info *drm_fbi;
#endif // NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct nec_fb_data *nfd = NULL;
	struct fb_info *fbi = NULL;
	int err = 0;

	nfd = (struct nec_fb_data *)platform_get_drvdata(pdev);
	if (!nfd)
		return -ENODEV;

	fbi = nfd->fbi;

	/* Remove SysFs */
	nec_fb_remove_sysfs(nfd);

#ifndef NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Runtime deinit */
	pm_runtime_disable(fbi->dev);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB

	/* Suspend fb */
	err = nec_fb_suspend_common(nfd);
	if (err)
		NEC_FB_PRINT_ERR("Failed to stop device(err=%d)\n", err);


#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	helper = get_drm_fb_helper();
	drm_fbi = nfd->drm_fbi;
	if (helper && drm_fbi) {
		helper->fbdev = nfd->drm_fbi;	// Recover previous fbi
	}
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Unregister framebuffer */
	fbi = nfd->fbi;
	unregister_framebuffer(fbi);
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	/* Release framebuffer */
	framebuffer_release(fbi);
	NEC_FB_DPRINT("done\n");
	return 0;
}

static void nec_fb_remove_sysfs(struct nec_fb_data *nfd)
{
#ifdef NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->drm_fbi;
#else // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct fb_info *fbi = nfd->fbi;
#endif // !NEC_PANEL_USE_ALLOCATED_DRM_FB
	struct device *dev = fbi->dev;
#ifdef NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0
	struct kobject *fb_kobj = &dev->kobj;
	struct kobject *graphics_kobj = fb_kobj->parent;
	struct kobject *virtual_kobj;

	extern struct kobject *virtual_device_parent(struct device *dev);

	if(!graphics_kobj) {
		NEC_FB_PRINT_ERR("Parent graphics device not found on %s", kobject_name(fb_kobj));
		goto exit;
	}
	printk(KERN_DEBUG "%s: ../%s/%s\n", __func__, kobject_name(graphics_kobj), kobject_name(fb_kobj));
	if (strcmp(kobject_name(graphics_kobj), "graphics") != 0) {
		NEC_FB_PRINT_ERR("Parent device name is not graphics: %s", kobject_name(graphics_kobj));
		goto exit;
	}

	if (strcmp(kobject_name(fb_kobj), "fb0") != 0) {
		sysfs_remove_link(graphics_kobj, "fb0");	// Remove link /sys/devices/virtual/graphics/fb0 -> fb1
	}

	virtual_kobj = graphics_kobj->parent;
	if (!virtual_kobj) {
		NEC_FB_PRINT_ERR("Grandparent device not found on %s", kobject_name(graphics_kobj));
		goto exit;
	}
	if (strcmp(kobject_name(virtual_kobj), "virtual") != 0) {	// Real platform device?
		NEC_FB_PRINT_ERR("Grandparent name is not virtual: %s", kobject_name(virtual_kobj));
		// fbi->dev->kobj should be "fb0", not "fb1"
		// We would have /sys/device/virtual/graphics/fb0 for compatile API

		virtual_kobj = virtual_device_parent(NULL);
		sysfs_remove_link(virtual_kobj, "graphics");	// Remove link /sys/device/virtual/graphics -> /sys/platform/<real-device>/graphics
	}
exit:
#endif // NEC_PANEL_CREATE_FAKE_VIRTUAL_GRAPHICS_FB0

	sysfs_remove_group(&dev->kobj, &nec_fb_attr_group);
	NEC_FB_DPRINT("done\n");
}


/*****************************************************************************/
static void nec_fb_shutdown(struct platform_device *pdev)
{
	struct nec_fb_data *nfd = platform_get_drvdata(pdev);

	nfd->shutdown_pending = true;

	/* Release FB */
	lock_fb_info(nfd->fbi);
	nec_fb_release_common(nfd, true);
	unlock_fb_info(nfd->fbi);

	NEC_FB_DPRINT("done\n");
}


/*****************************************************************************/
static const struct dev_pm_ops nec_fb_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(nec_fb_pm_suspend, nec_fb_pm_resume)
};

static const struct of_device_id nec_fb_of_match[] = {
	{ .compatible = NEC_FB_NAME,},
	{}
};
MODULE_DEVICE_TABLE(of, nec_fb_of_match);

static struct platform_driver nec_fb_driver = {
	.probe		= nec_fb_probe,
	.remove		= nec_fb_remove,
	.suspend	= nec_fb_suspend,
	.resume		= nec_fb_resume,
	.shutdown	= nec_fb_shutdown,
	.driver		= {
		.name		= NEC_FB_NAME,
		.of_match_table	= nec_fb_of_match,
		.pm		= &nec_fb_pm_ops,
	},
};
module_platform_driver(nec_fb_driver);
MODULE_LICENSE("GPL v2");
