/* SPDX-License-Identifier: GPL-2.0
 *
 * Copyright (c) 2020 - 2021 MediaTek Inc.
 * Copyright (c) 2017 - 2021, NEC Platforms, Ltd., All rights reserved.
 *
 */

#include <linux/gpio/consumer.h>
#include <linux/regulator/consumer.h>

#include <drm/drmP.h>
#include <drm/drm_panel.h>

#include "../mediatek/mtk_dbi.h"

#include <video/mipi_display.h>
#include <drm/drm_fb_helper.h>
#ifdef CONFIG_DRM_NEC
#include "../necpf/nec_panel.h"
#endif // CONFIG_DRM_NEC
// Define below for debug
//#define DEBUG_PANEL
#ifndef DEBUG_PANEL
#define printk(...)
#endif

// Define below if require to wait completion of last command send before suspending
#define WAIT_LAST_COMMAND_SENT
// Use to dynmiac detect LCD, if no exsited, return error
#define DYNAMIC_DETECT_LCD

// Screen active area size of InnoLux LCD module
#define PHYSICAL_DISPLAY_HEIGHT_MM 49
#define PHYSICAL_DISPLAY_WIDTH_MM 37

#define ST7789V_COLMOD_RGB_FMT_18BITS		(6 << 4)
#define ST7789V_COLMOD_CTRL_FMT_18BITS		(6 << 0)

#define ST7789V_RDDID_CMD	0X04
#define ST7789V_RDID1_CMD	0xDA
#define ST7789V_RDID2_CMD	0xDB
#define ST7789V_RDID3_CMD	0xDC

#define ST7789V_PORCTRL_CMD		0xb2
#define ST7789V_PORCTRL_IDLE_BP(n)		(((n) & 0xf) << 4)
#define ST7789V_PORCTRL_IDLE_FP(n)		((n) & 0xf)
#define ST7789V_PORCTRL_PARTIAL_BP(n)		(((n) & 0xf) << 4)
#define ST7789V_PORCTRL_PARTIAL_FP(n)		((n) & 0xf)

#define ST7789V_GCTRL_CMD		0xb7
#define ST7789V_GCTRL_VGHS(n)			(((n) & 7) << 4)
#define ST7789V_GCTRL_VGLS(n)			((n) & 7)

#define ST7789V_VCOMS_CMD		0xbb

#define ST7789V_VDVVRHEN_CMD		0xc2
#define ST7789V_VDVVRHEN_CMDEN			BIT(0)

#define ST7789V_VRHS_CMD		0xc3

#define ST7789V_VDVS_CMD		0xc4

#define ST7789V_FRCTRL2_CMD		0xc6

#define ST7789V_PWMFRSEL_CMD    0xcc

#define ST7789V_PWCTRL1_CMD		0xd0
#define ST7789V_PWCTRL1_MAGIC			0xa4
#define ST7789V_PWCTRL1_AVDD(n)			(((n) & 3) << 6)
#define ST7789V_PWCTRL1_AVCL(n)			(((n) & 3) << 4)
#define ST7789V_PWCTRL1_VDS(n)			((n) & 3)

#define ST7789V_PVGAMCTRL_CMD		0xe0
#define ST7789V_PVGAMCTRL_JP0(n)		(((n) & 3) << 4)
#define ST7789V_PVGAMCTRL_JP1(n)		(((n) & 3) << 4)
#define ST7789V_PVGAMCTRL_VP0(n)		((n) & 0xf)
#define ST7789V_PVGAMCTRL_VP1(n)		((n) & 0x3f)
#define ST7789V_PVGAMCTRL_VP2(n)		((n) & 0x3f)
#define ST7789V_PVGAMCTRL_VP4(n)		((n) & 0x1f)
#define ST7789V_PVGAMCTRL_VP6(n)		((n) & 0x1f)
#define ST7789V_PVGAMCTRL_VP13(n)		((n) & 0xf)
#define ST7789V_PVGAMCTRL_VP20(n)		((n) & 0x7f)
#define ST7789V_PVGAMCTRL_VP27(n)		((n) & 7)
#define ST7789V_PVGAMCTRL_VP36(n)		(((n) & 7) << 4)
#define ST7789V_PVGAMCTRL_VP43(n)		((n) & 0x7f)
#define ST7789V_PVGAMCTRL_VP50(n)		((n) & 0xf)
#define ST7789V_PVGAMCTRL_VP57(n)		((n) & 0x1f)
#define ST7789V_PVGAMCTRL_VP59(n)		((n) & 0x1f)
#define ST7789V_PVGAMCTRL_VP61(n)		((n) & 0x3f)
#define ST7789V_PVGAMCTRL_VP62(n)		((n) & 0x3f)
#define ST7789V_PVGAMCTRL_VP63(n)		(((n) & 0xf) << 4)

#define ST7789V_NVGAMCTRL_CMD		0xe1
#define ST7789V_NVGAMCTRL_JN0(n)		(((n) & 3) << 4)
#define ST7789V_NVGAMCTRL_JN1(n)		(((n) & 3) << 4)
#define ST7789V_NVGAMCTRL_VN0(n)		((n) & 0xf)
#define ST7789V_NVGAMCTRL_VN1(n)		((n) & 0x3f)
#define ST7789V_NVGAMCTRL_VN2(n)		((n) & 0x3f)
#define ST7789V_NVGAMCTRL_VN4(n)		((n) & 0x1f)
#define ST7789V_NVGAMCTRL_VN6(n)		((n) & 0x1f)
#define ST7789V_NVGAMCTRL_VN13(n)		((n) & 0xf)
#define ST7789V_NVGAMCTRL_VN20(n)		((n) & 0x7f)
#define ST7789V_NVGAMCTRL_VN27(n)		((n) & 7)
#define ST7789V_NVGAMCTRL_VN36(n)		(((n) & 7) << 4)
#define ST7789V_NVGAMCTRL_VN43(n)		((n) & 0x7f)
#define ST7789V_NVGAMCTRL_VN50(n)		((n) & 0xf)
#define ST7789V_NVGAMCTRL_VN57(n)		((n) & 0x1f)
#define ST7789V_NVGAMCTRL_VN59(n)		((n) & 0x1f)
#define ST7789V_NVGAMCTRL_VN61(n)		((n) & 0x3f)
#define ST7789V_NVGAMCTRL_VN62(n)		((n) & 0x3f)
#define ST7789V_NVGAMCTRL_VN63(n)		(((n) & 0xf) << 4)

#define ST7789V_BRIGHTNESS_MAX          0xff
#define ST7789V_BRIGHTNESS_MIN          0x00
#define ST7789V_DEFAULT_BRIGHTNESS		0x00	// Default value on reset

/* refresh rate */
#define ST7789V_DEFAULT_REFRESH_RATE	(60)	// Default value on reset
#define ST7789V_MAX_REFRESH_RATE	(82)
#define ST7789V_MIN_REFRESH_RATE	(40)
#define ST7789V_NUM_REFRESH_RATE_SETS	(11)

#define LCD_CLAMP( _val , _min , _max ) \
	( ( ( _val ) < ( _min ) ) ? ( _min ) : \
	( ( ( _max ) < ( _val ) ) ? ( _max ) : ( _val ) ) )

struct st7789v {
	bool prepared;
	struct drm_panel panel;
	struct device *dev;
#ifdef CONTROL_LCD_POWER_SUPPLY
	struct regulator *power;
#endif // CONTROL_LCD_POWER_SUPPLY
	int refresh_rate;
	bool is_enabled;
	int display_on;
	int lcd_on;
	int lcd_backlight;
	int brightness;
	int initialized;
};

enum st7789v_prefix {
	ST7789V_COMMAND = 0,
	ST7789V_DATA = 1,
};

typedef struct _lcd_refreshrate_data {
	int value;
	int cmd_data;
} lcd_refreshrate_data;

const struct mipi_dbi_host_ops *ops;
static int need_initialize = 0;	// 0: initialized by bootloader or explicitly initialize by application

static inline struct st7789v *panel_to_st7789v(struct drm_panel *panel)
{
	return container_of(panel, struct st7789v, panel);
}

static void st7789v_write_command(struct st7789v *ctx, u8 cmd)
{
	ops->send_cmd(&ctx->panel, cmd);
}

static void st7789v_write_data(struct st7789v *ctx, u8 data)
{
	ops->send_data(&ctx->panel, data);
}

static u8 st7789v_read_data(struct st7789v *ctx)
{
	return (u8)ops->read_data(&ctx->panel);
}

static void st7789v_occupy_data_pin(struct st7789v *ctx)
{
    ops->occupy_data_pin(&ctx->panel);
}

static void st7789v_release_data_pin(struct st7789v *ctx)
{
    ops->release_data_pin(&ctx->panel);
}

#ifdef CONFIG_MTK_DBI_RESET
static void st7789v_set_reset(struct st7789v *ctx, int signal)
{
	ops->set_reset(&ctx->panel, signal);
}
#endif // CONFIG_MTK_DBI_RESET

static int st7789v_is_busy(struct st7789v *ctx)
{
#ifdef CONFIG_MTK_DBI_CHECK_BUSY
	if (!ops) {	// Early beggining call doesn't effective
		printk(KERN_DEBUG "[%s] not yet ready \n", __func__);
		return -EBUSY;
	}
	return ops->is_busy(&ctx->panel);
#else // !CONFIG_MTK_DBI_CHECK_BUSY
	return 0;	// Not busy anytime
#endif // !CONFIG_MTK_DBI_CHECK_BUSY
}

static void st7789v_cmd_seq_begin(struct st7789v *ctx)
{
#ifdef CONFIG_MTK_DBI_MUTUAL_CMD_SEQ
	if (!ops) {	// Early beggining call doesn't effective
		printk(KERN_DEBUG "[%s] not yet ready \n", __func__);
		return;
	}
	ops->mutual_cmd_seq(&ctx->panel, 1);
#endif // CONFIG_MTK_DBI_MUTUAL_CMD_SEQ
}

static void st7789v_cmd_seq_end(struct st7789v *ctx)
{
#ifdef CONFIG_MTK_DBI_MUTUAL_CMD_SEQ
	if (!ops) {	// Early beggining call doesn't effective
		printk(KERN_DEBUG "[%s] not yet ready \n", __func__);
		return;
	}
	ops->mutual_cmd_seq(&ctx->panel, 0);
#endif // CONFIG_MTK_DBI_MUTUAL_CMD_SEQ
}

static void st7789v_set_panel_status(struct st7789v *ctx, bool panel_on)
{
#ifdef CONFIG_MTK_DBI_SET_PANEL_STATUS
	if (!ops) {	// Early beggining call doesn't effective
		printk(KERN_DEBUG "[%s] not yet ready \n", __func__);
		return;
	}
	ops->set_panel_status(&ctx->panel, panel_on);
#endif // !CONFIG_MTK_DBI_SET_PANEL_STATUS
}

static const struct drm_display_mode default_mode = {
	.name = "240x320-16",
	.clock = 6008,                // Pixel Clock (kHz): htotal * vtotal * vrefresh / 1000
	.hdisplay = 240,
	.hsync_start = 240 + 38,      // HFP
	.hsync_end = 240 + 38 + 10,   // HSA = HPW
	.htotal = 240 + 38 + 10 + 10, // HBP
	.vdisplay = 320,
	.vsync_start = 320 + 8,       // VFP
	.vsync_end = 320 + 8 + 4,     // VSA
	.vtotal = 320 + 8 + 4 + 4,    // VBP
	.vrefresh = 60,
	.type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED,
};

static int st7789v_get_modes(struct drm_panel *panel)
{
	struct drm_connector *connector = panel->connector;
	struct drm_display_mode *mode;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	mode = drm_mode_duplicate(panel->drm, &default_mode);
	if (!mode) {
		dev_err(panel->drm->dev, "failed to add mode %ux%ux@%u\n",
			default_mode.hdisplay, default_mode.vdisplay,
			default_mode.vrefresh);
		return -ENOMEM;
	}

	//drm_mode_set_name(mode);

	mode->type = DRM_MODE_TYPE_DRIVER | DRM_MODE_TYPE_PREFERRED;
	drm_mode_probed_add(connector, mode);

	panel->connector->display_info.width_mm = PHYSICAL_DISPLAY_WIDTH_MM;
	panel->connector->display_info.height_mm = PHYSICAL_DISPLAY_HEIGHT_MM;

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 1;
}

static void st7789v_send_command_data(struct drm_panel *panel, u8 cmd, u8 *data, int data_cnt)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int i = 0;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	st7789v_cmd_seq_begin(ctx);
	st7789v_write_command(ctx, cmd);

	while (i < data_cnt) {
		st7789v_write_data(ctx, data[i]);
		i++;
	}
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);
}

static int st7789v_recv_command_data(struct drm_panel *panel, u8 cmd, u8 *data, int max_data_cnt)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int i = 0;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	st7789v_cmd_seq_begin(ctx);
	st7789v_write_command(ctx, cmd);

	while (i < max_data_cnt) {
		if (data)
			data[i] = st7789v_read_data(ctx);
		else
			st7789v_read_data(ctx);
		i++;
	}
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit ret = %d\n", __func__, i);

	return i;
}

static int st7789v_init(struct drm_panel *panel)
{
	printk(KERN_DEBUG "[%s] enter \n", __func__);

	// Nothing to do

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static int st7789v_lcd_on_locked(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	// Sleep Out (11h)
	st7789v_write_command(ctx, MIPI_DCS_EXIT_SLEEP_MODE);
	/* We need to wait 120ms after a sleep out command */
	msleep(120);

	ctx->lcd_on = 1;
	st7789v_set_panel_status(ctx, true);

	return 0;
}

static int st7789v_lcd_on(struct drm_panel *panel)
{
    struct st7789v *ctx = panel_to_st7789v(panel);

    printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	st7789v_lcd_on_locked(panel);
	st7789v_cmd_seq_end(ctx);

    printk(KERN_DEBUG "[%s] exit \n", __func__);

    return 0;
}

static int st7789v_lcd_off(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	msleep(120);
	st7789v_set_panel_status(ctx, false);
	// Sleep in (10h)
	st7789v_write_command(ctx, MIPI_DCS_ENTER_SLEEP_MODE);

	ctx->lcd_on = 0;
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static int st7789v_reset(struct drm_panel *panel, int hwreset, int swreset)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter hwreset=%d, swreset=%d\n", __func__, hwreset, swreset);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	// Do not care LCD and DBI is still working (busy) before reset
	if (hwreset) {
		printk(KERN_DEBUG "[%s] set DBI I/F reset start \n", __func__);

#ifdef CONFIG_MTK_DBI_RESET
		// DBI LCM Reset Pin
		st7789v_set_reset(ctx, 1);
		msleep(10);
		st7789v_set_reset(ctx, 0);
		msleep(10);
		st7789v_set_reset(ctx, 1);
		msleep(10);
		// Need 120ms after H/W Reset
		msleep(120);

		printk(KERN_DEBUG "[%s] set DBI I/F reset done \n", __func__);
#else // !CONFIG_MTK_DBI_RESET
		printk(KERN_DEBUG "[%s] HW reset not supported. The request ignored\n", __func__);
#endif // !CONFIG_MTK_DBI_RESET
	}
	if (swreset) {
		printk(KERN_DEBUG "[%s] soft reset start \n", __func__);

		// Soft Reset (01h)
		st7789v_write_command(ctx, MIPI_DCS_SOFT_RESET);
		msleep(120);

		printk(KERN_DEBUG "[%s] soft reset done \n", __func__);
	}
	if (hwreset || swreset) {
		ctx->lcd_backlight = 0;
		ctx->brightness = ST7789V_DEFAULT_BRIGHTNESS;
		ctx->refresh_rate = ST7789V_DEFAULT_REFRESH_RATE;
		ctx->initialized = 0;
		ctx->display_on = 0;
		ctx->lcd_on = 0;
	}
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static lcd_refreshrate_data refrate_cmd_data[ST7789V_NUM_REFRESH_RATE_SETS] = {
	{ 82, 0x07}, { 72, 0x0A}, { 69, 0x0B}, { 67, 0x0C}, { 64, 0x0D},
	{ 60, 0x0F}, { 55, 0x12}, { 53, 0x13}, { 50, 0x15}, { 45, 0x19},
	{ 40, 0x1E},
};

static void st7789v_set_refresh_rate_locked(struct drm_panel *panel, int refresh_rate)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int i = 0;
	int value = 0;
	u8 data = 0x0F; /* default: 60Hz */

	printk(KERN_DEBUG "[%s] enter refresh_rate=%d\n", __func__, refresh_rate);

	value = LCD_CLAMP(refresh_rate, ST7789V_MIN_REFRESH_RATE, ST7789V_MAX_REFRESH_RATE);

	/* Updates refresh rate */
	for (i = 0; i < ST7789V_NUM_REFRESH_RATE_SETS; i++) {
		if (refrate_cmd_data[i].value <= value) {
			data = refrate_cmd_data[i].cmd_data;
			ctx->refresh_rate = refrate_cmd_data[i].value;
			break;
		}
	}

	/* Frame Rate Control in Normal Mode (C6h)  */
	// 	NLA[2-0]:0x0/RTNA[4-0]:0xF
	// 	NOTE: If change this parameters, please change
	// 		DEFAULT_REFRESH_RATE definition value.
	st7789v_write_command(ctx, ST7789V_FRCTRL2_CMD);
	st7789v_write_data(ctx, data);

	printk(KERN_DEBUG "[%s] exit refresh_rate=%d\n", __func__, ctx->refresh_rate);
}

static void st7789v_set_refresh_rate(struct drm_panel *panel, int refresh_rate)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter refresh_rate=%d\n", __func__, refresh_rate);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	st7789v_occupy_data_pin(ctx);
	st7789v_set_refresh_rate_locked(panel, refresh_rate);
	st7789v_release_data_pin(ctx);
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit refresh_rate=%d\n", __func__, ctx->refresh_rate);

	return;
}

static int st7789v_get_refresh_rate(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int refresh_rate = ctx->refresh_rate;

	printk(KERN_DEBUG "[%s] refresh_rate=%d\n", __func__, refresh_rate);

	return refresh_rate;
}

static int st7789v_init_params(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int t, ready = 0;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	for (t = 0; t < 120; t += 10) {
		if (!st7789v_is_busy(ctx)) {	// Check ready every 10ms
			ready = 1;
			break;
		}
		msleep(10);
	}
	// Timeout 120ms
	if (!ready) {
		printk(KERN_DEBUG "[%s] failed on busy \n", __func__);
		return -EBUSY;
	}

	st7789v_cmd_seq_begin(ctx);
	st7789v_occupy_data_pin(ctx);

	st7789v_lcd_on_locked(panel);

	// Memory Data Access Control (36h)
	//  MY:0/MX:0/MV:0/ML:0/RGB-order:1(BGR)/MH:0 
	st7789v_write_command(ctx, MIPI_DCS_SET_ADDRESS_MODE);
	st7789v_write_data(ctx, 0x08); // RGB order

	// Interface Pixel Format (COLMOD)  (3Ah)
	// 	D[6-4]:0(no care)/D[2-0]:0x5(16 bpp)
	st7789v_write_command(ctx, MIPI_DCS_SET_PIXEL_FORMAT);
	st7789v_write_data(ctx, 0x05);  //16bits

	// Porch Setting (B2h)
	// 	BPA[6-0]:0x0C/FPA[6-0]:0x0C/PSEN:0x00
	//  BPB[3-0]:0x3/FPB[3-0]:0x3/BPC[3-0]:0x3/FPC[3-0]:0x3
	st7789v_write_command(ctx, ST7789V_PORCTRL_CMD);
	st7789v_write_data(ctx, 0x0C);
	st7789v_write_data(ctx, 0x0C);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x33);
	st7789v_write_data(ctx, 0x33);

	// Gate Control (B7h)  */
	// 	VGHS[2-0]:0x3/VGLS[2-0]:0x5
	st7789v_write_command(ctx, ST7789V_GCTRL_CMD);
	st7789v_write_data(ctx, 0x35);

	// VCOMS Setting (BBh)
	// 	VCOMS[5-0]:0x32
	st7789v_write_command(ctx, ST7789V_VCOMS_CMD);
	st7789v_write_data(ctx, 0x32);

	// VDV and VRH Command Enable (C2h)
	// 	CMDEN:1
	st7789v_write_command(ctx, ST7789V_VDVVRHEN_CMD);
	st7789v_write_data(ctx, 0x01);

	// VRH Set (C3h)
	// 	VRHS[5-0]:0x18
	st7789v_write_command(ctx, ST7789V_VRHS_CMD);
	st7789v_write_data(ctx, 0x18);

	// VDV Set (C4h)
	// 	VDVS[5-0]:0x20
	st7789v_write_command(ctx, ST7789V_VDVS_CMD);
	st7789v_write_data(ctx, 0x20);

	st7789v_set_refresh_rate_locked(panel, ST7789V_DEFAULT_REFRESH_RATE);

	// PWM Frequency Selection (CCh)
	// 	CS[2-0]:0x0/CLK[2-0]:0x0 (39.2kHz)
	st7789v_write_command(ctx, ST7789V_PWMFRSEL_CMD);
	st7789v_write_data(ctx, 0x00);

	// Power Control (D0h)
	// 	1st data: 0xA4 (Fixed)
	// 	AVDD[1-0]:0x2/AVCL[1-0]:0x2/VDS[1-0]:0x1
	st7789v_write_command(ctx, ST7789V_PWCTRL1_CMD);
	st7789v_write_data(ctx, 0xA4);
	st7789v_write_data(ctx, 0xA1);

	// Positive Voltage Gamma Control (E0h)
	// 	1st - 14th data: Positive gamma control value
	st7789v_write_command(ctx, ST7789V_PVGAMCTRL_CMD);
	st7789v_write_data(ctx, 0xF0);
	st7789v_write_data(ctx, 0x03);
	st7789v_write_data(ctx, 0x0C);
	st7789v_write_data(ctx, 0x0C);
	st7789v_write_data(ctx, 0x11);
	st7789v_write_data(ctx, 0x0A);
	st7789v_write_data(ctx, 0x45);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x45);
	st7789v_write_data(ctx, 0x2A);
	st7789v_write_data(ctx, 0x13);
	st7789v_write_data(ctx, 0x11);
	st7789v_write_data(ctx, 0x25);
	st7789v_write_data(ctx, 0x25);

	// Negative Voltage Gamma Control (E1h)
	// 	1st - 14th data: Negative gamma control values
	st7789v_write_command(ctx, ST7789V_NVGAMCTRL_CMD);
	st7789v_write_data(ctx, 0xF0);
	st7789v_write_data(ctx, 0x03);
	st7789v_write_data(ctx, 0x0A);
	st7789v_write_data(ctx, 0x17);
	st7789v_write_data(ctx, 0x12);
	st7789v_write_data(ctx, 0x09);
	st7789v_write_data(ctx, 0x28);
	st7789v_write_data(ctx, 0x55);
	st7789v_write_data(ctx, 0x4F);
	st7789v_write_data(ctx, 0x09);
	st7789v_write_data(ctx, 0x15);
	st7789v_write_data(ctx, 0x17);
	st7789v_write_data(ctx, 0x17);
	st7789v_write_data(ctx, 0x1E);

	// Tearing Effect Line On (35h)
	// 	TEM:0x0(V-Blanking only)
	st7789v_write_command(ctx, MIPI_DCS_SET_TEAR_ON);
	st7789v_write_data(ctx, 0x00);  // vblank only
	msleep(20);

	// Memory Write (2Ch)
	//st7789v_write_command(ctx, MIPI_DCS_WRITE_MEMORY_START);

	// Column Address Set (2Ah)
	st7789v_write_command(ctx, MIPI_DCS_SET_COLUMN_ADDRESS);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x00);  // 0x00ef = 239
	st7789v_write_data(ctx, 0xEF);

	// Row Address Set (2Bh)
	st7789v_write_command(ctx, MIPI_DCS_SET_PAGE_ADDRESS);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x00);
	st7789v_write_data(ctx, 0x01);  // 0x013f = 319
	st7789v_write_data(ctx, 0x3F);

	ctx->initialized = 1;
	st7789v_release_data_pin(ctx);
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] done \n", __func__);
	return 0;
}

static int st7789v_reset_init(struct drm_panel *panel, int hwreset, int swreset)
{
	int ret;

	ret = st7789v_reset(panel, hwreset, swreset);
	if (!ret) {
		ret = st7789v_init_params(panel);
	}
	return ret;
}

static void st7789v_display_on(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	// Display On (29h)
	st7789v_write_command(ctx, MIPI_DCS_SET_DISPLAY_ON);
	msleep(20);

	ctx->display_on = 1;
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);
}

static void st7789v_display_off(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->is_enabled) {
		printk(KERN_DEBUG "[%s] not enabled, exit \n", __func__);
		return;				// Not accessible
	}

	st7789v_cmd_seq_begin(ctx);
	// Display On (29h)
	st7789v_write_command(ctx, MIPI_DCS_SET_DISPLAY_OFF);
	msleep(20);

	ctx->display_on = 0;
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);
}

void st7789v_get_status(struct drm_panel *panel, int *display_on, int *lcd_on)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (display_on)
		*display_on = ctx->display_on;
	if (lcd_on)
		*lcd_on = ctx->lcd_on;

	printk(KERN_DEBUG "[%s] exit display=%d, lcd=%d\n", __func__, ctx->display_on, ctx->lcd_on);
}

static int st7789v_set_brightness(struct drm_panel *panel, unsigned int level)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter level = %d\n", __func__, level);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	if (level > ST7789V_BRIGHTNESS_MAX)
		level = ST7789V_BRIGHTNESS_MAX;
	else if (level < ST7789V_BRIGHTNESS_MIN)
		level = ST7789V_BRIGHTNESS_MIN;
	else
		/* NO LEVEL ADJUSTMENT REQUIRED */;

	// Write Display Brightness (51h)
	st7789v_cmd_seq_begin(ctx);
	st7789v_occupy_data_pin(ctx);
	st7789v_write_command(ctx, MIPI_DCS_SET_DISPLAY_BRIGHTNESS);
	st7789v_write_data(ctx, level);
	st7789v_release_data_pin(ctx);
	ctx->brightness = level;
	st7789v_cmd_seq_end(ctx);

	printk(KERN_DEBUG "[%s] exit \n", __func__);
	return 0;
}

static int st7789v_get_brightness(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int level = ctx->brightness;

	printk(KERN_DEBUG "[%s] enter\n", __func__);

	printk(KERN_DEBUG "[%s] exit level = %d\n", __func__, level);
	return level;
}

static int st7789v_set_backlight(struct drm_panel *panel, int onoff)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter on/off = %d\n", __func__, onoff);

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;				// Not accessible
	}

	if (onoff != 0) {
		st7789v_lcd_on(panel);
		st7789v_display_on(panel);
	}

	st7789v_cmd_seq_begin(ctx);
	st7789v_occupy_data_pin(ctx);
	// Write control display (53h)
	st7789v_write_command(ctx, MIPI_DCS_WRITE_CONTROL_DISPLAY);
	if (onoff == 0)
		st7789v_write_data(ctx, 0x00);
	else
		st7789v_write_data(ctx, 0x24);    // BCTRL=On, DD=Off, BL=On
	st7789v_release_data_pin(ctx);
	msleep(20);

	ctx->lcd_backlight = onoff;
	st7789v_cmd_seq_end(ctx);

	if (onoff == 0) {
		st7789v_display_off(panel);
		st7789v_lcd_off(panel);
	}

	printk(KERN_DEBUG "[%s] exit \n", __func__);
	return 0;
}

static int st7789v_get_backlight(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int onoff;

	printk(KERN_DEBUG "[%s] enter\n", __func__);

	onoff = ctx->lcd_backlight;

	printk(KERN_DEBUG "[%s] exit on/off = %d\n", __func__, onoff);
	return onoff;
}

#ifdef CONFIG_DRM_NEC
static struct nec_panel_lcd_operation st7789v_lcd_ops = {
	.lcd_init		= st7789v_init,
	.lcd_reset		= st7789v_reset_init,
	.lcd_on			= st7789v_lcd_on,
	.lcd_off		= st7789v_lcd_off,
	.lcd_display_on		= st7789v_display_on,
	.lcd_display_off	= st7789v_display_off,
	.lcd_get_status		= st7789v_get_status,
	.lcd_set_backlight	= st7789v_set_backlight,
	.lcd_get_backlight	= st7789v_get_backlight,
	.lcd_set_brightness	= st7789v_set_brightness,
	.lcd_get_brightness	= st7789v_get_brightness,
	.lcd_set_refresh_rate	= st7789v_set_refresh_rate,
	.lcd_get_refresh_rate	= st7789v_get_refresh_rate,
	.lcd_send_cmd 	= st7789v_send_command_data,
	.lcd_recv_cmd	= st7789v_recv_command_data,
};

extern void nec_lcd_attach(struct drm_panel *panel, struct nec_panel_lcd_operation *ops);
#endif // CONFIG_DRM_NEC

/*
 * Copied comment from drm_panel_prepare() in drm_panel.h
 * Calling this function will enable power and deassert any reset signals to
 * the panel. After this has completed it is possible to communicate with any
 * integrated circuitry via a command bus.
 */
static int st7789v_prepare(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
#ifdef DYNAMIC_DETECT_LCD
    unsigned int da, db, dc;
    unsigned int d1, d2, d3, d4;
#endif // DYNAMIC_DETECT_LCD

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (ctx->prepared) {
		printk(KERN_DEBUG "[%s] already prepared, exit \n", __func__);
		return 0;
	}

#ifdef CONTROL_LCD_POWER_SUPPLY
	int ret;
	if (ctx->power) {
		ret = regulator_enable(ctx->power);
		if (ret)
			return ret;
	}
#endif // CONTROL_LCD_POWER_SUPPLY

#ifdef DYNAMIC_DETECT_LCD
	if (!ctx->initialized) {
		if (need_initialize) {
			ctx->prepared = true;	// Avoid checking prepared temporary
			st7789v_reset(panel, true, true);
			ctx->prepared = false;
		}
	}
	// Do not need sleep out as on usual initial ssequence
	// because ID commands are effective in any modes.

	// Get DDIC ID to check if Panel existed
    // Read DA
	st7789v_cmd_seq_begin(ctx);
    st7789v_occupy_data_pin(ctx);
    st7789v_write_command(ctx, ST7789V_RDID1_CMD);
    d1 = st7789v_read_data(ctx);
    da = st7789v_read_data(ctx);
    printk(KERN_DEBUG "[%s] read DAh (%02X, %02X) \n", __func__, d1, da);

    // Read DB
    st7789v_write_command(ctx, ST7789V_RDID2_CMD);
    d1 = st7789v_read_data(ctx);
    db = st7789v_read_data(ctx);
    printk(KERN_DEBUG "[%s] read DBh (%02X, %02X) \n", __func__, d1, db);

    // Read DC
    st7789v_write_command(ctx, ST7789V_RDID3_CMD);
    d1 = st7789v_read_data(ctx);
    dc = st7789v_read_data(ctx);
    printk(KERN_DEBUG "[%s] read DCh (%02X, %02X) \n", __func__, d1, dc);

    // Read DDIC ID
    st7789v_write_command(ctx, ST7789V_RDDID_CMD);
    d1 = st7789v_read_data(ctx);
    d2 = st7789v_read_data(ctx);
    d3 = st7789v_read_data(ctx);
    d4 = st7789v_read_data(ctx);
    st7789v_release_data_pin(ctx);
	st7789v_cmd_seq_end(ctx);
    printk(KERN_DEBUG "[%s] read DDIC ID (%02X, %02X, %02X, %02X) \n", __func__, d1, d2, d3, d4);

    if ((da == 0 && db == 0 && dc == 0) || da != d2 || db != d3 || dc != d4) {
        // Means there is no LCD detect, return error
        return -1;
    }
#endif // DYNAMIC_DETECT_LCD

	// Do not treat DBI here. Maybe prepared before.

	ctx->prepared = true;

	printk(KERN_DEBUG "[%s] exit prepared=%d\n", __func__, ctx->prepared);

	return 0;
}

/*
 * Copied comment from drm_panel_enable() in drm_panel.h
 * Calling this function will cause the panel display drivers to be turned on
 * and the backlight to be enabled. Content will be visible on screen after
 * this call completes.
*/
static int st7789v_enable(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int t, ready = 0;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (ctx->is_enabled) {
		printk(KERN_DEBUG "[%s] already enabled \n", __func__);
		return 0;
	}

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared yet \n", __func__);
		return -EAGAIN;
	}

	for (t = 0; t < 120; t += 10) {
		if (!st7789v_is_busy(ctx)) {	// Check ready every 10ms
			ready = 1;
			break;
		}
		msleep(10);
	}
	// Timeout 120ms
	if (!ready) {
		printk(KERN_DEBUG "[%s] failed on busy \n", __func__);
		return -EBUSY;
	}

	if (!ctx->initialized) {
		if (need_initialize) {
			st7789v_reset_init(panel, true, false);
		}
	}

	st7789v_lcd_on(panel);
	st7789v_set_backlight(panel, 1);   // Set backlight on
	st7789v_set_brightness(panel, ST7789V_BRIGHTNESS_MAX);

	msleep(120);
	st7789v_display_on(panel);

	ctx->is_enabled = true;

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static int st7789v_disable(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);
	int t, ready = 0;

	printk(KERN_DEBUG "[%s] enter \n", __func__);

	if (!ctx->is_enabled) {
		printk(KERN_DEBUG "[%s] already disabled \n", __func__);
		return 0;
	}

	if (!ctx->prepared) {
		printk(KERN_DEBUG "[%s] not prepared, exit \n", __func__);
		return -EAGAIN;
	}

	for (t = 0; t < 120; t += 10) {
		if (!st7789v_is_busy(ctx)) {
			ready = 1;
			break;
		}

		msleep(10);
	}
	// Timeout 120ms
	if (!ready) {
		printk(KERN_DEBUG "[%s] failed on busy \n", __func__);
		return -EBUSY;
	}
	st7789v_display_off(panel);

	st7789v_set_brightness(panel, ST7789V_BRIGHTNESS_MIN);
	st7789v_set_backlight(panel, 0);

	st7789v_lcd_off(panel);

#ifdef WAIT_LAST_COMMAND_SENT
	for (t = 0; t < 120; t += 10) {
		if (!st7789v_is_busy(ctx)) {
			ready = 1;
			break;
		}

		msleep(10);
	}
	// Timeout 120ms
	if (!ready) {
		printk(KERN_DEBUG "[%s] Last command not yet sent\n", __func__);
		return -EBUSY;
	}
#endif // WAIT_LAST_COMMAND_SENT

	ctx->is_enabled = false;

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static int st7789v_unprepare(struct drm_panel *panel)
{
	struct st7789v *ctx = panel_to_st7789v(panel);

	printk(KERN_DEBUG "[%s] enter \n", __func__);

#ifdef CONTROL_LCD_POWER_SUPPLY
	if (ctx->power) {
		regulator_disable(ctx->power);
	}
#endif // CONTROL_LCD_POWER_SUPPLY
	ctx->prepared = false;

	printk(KERN_DEBUG "[%s] exit \n", __func__);

	return 0;
}

static const struct drm_panel_funcs st7789v_drm_funcs = {
	.disable    = st7789v_disable,
	.enable     = st7789v_enable,
	.get_modes  = st7789v_get_modes,
	.prepare    = st7789v_prepare,
	.unprepare  = st7789v_unprepare,
};

static int st7789v_probe(struct device *dev)
{
	struct st7789v *ctx;
	struct mipi_dbi_device lcm_param;
	int ret;

	printk(KERN_DEBUG "st7789v_probe\n");

	ctx = devm_kzalloc(dev, sizeof(*ctx), GFP_KERNEL);
	if (!ctx)
		return -ENOMEM;

	lcm_param.ctrl_type = MTK_DBI_LCM_CTRL_TYPE_PARALLEL_DBI;

	lcm_param.clk_freq = MTK_DBI_LCM_CLOCK_FREQ_125M;
	lcm_param.data_width = MTK_DBI_LCM_DATA_WIDTH_8BITS;
	lcm_param.color_format = MTK_DBI_LCM_COLOR_FORMAT_RGB565;
	lcm_param.driving_current = MTK_DBI_LCM_DRIVING_CURRENT_12MA;
	lcm_param.color_order = MTK_DBI_LCM_COLOR_ORDER_BGR;
	lcm_param.te_control.software_te = 0;
	lcm_param.te_control.te_repeat = 0;
	lcm_param.te_control.sync_mode = 0;
	lcm_param.te_control.te_edge = 0;
	lcm_param.te_control.te_enable = 1;

	lcm_param.dbi_b_params.itf_size = MTK_DBI_LCM_INTERFACE_SIZE_8BITS;
#ifdef DYNAMIC_DETECT_LCD
    lcm_param.dbi_b_params.rw_timing.cs_to_read_hold_time = 6;
    lcm_param.dbi_b_params.rw_timing.cs_to_read_setup_time = 6;
    lcm_param.dbi_b_params.rw_timing.read_latency_time = 0xb;
#else // !DYNAMIC_DETECT_LCD
	lcm_param.dbi_b_params.rw_timing.cs_to_read_hold_time = 1;
	lcm_param.dbi_b_params.rw_timing.cs_to_read_setup_time = 1;
	lcm_param.dbi_b_params.rw_timing.read_latency_time = 4;
#endif // !DYNAMIC_DETECT_LCD
	lcm_param.dbi_b_params.rw_timing.cs_to_write_hold_time = 1;
	lcm_param.dbi_b_params.rw_timing.cs_to_write_setup_time = 1;
	lcm_param.dbi_b_params.rw_timing.write_wait_state_time = 3;
    lcm_param.dbi_b_params.rw_timing.cs_remain_high_time = 0;

	ctx->dev = dev;

	drm_panel_init(&ctx->panel);
	ctx->panel.dev = dev;
	ctx->panel.funcs = &st7789v_drm_funcs;

	if (!need_initialize) {		// Set status as in boot
		ctx->lcd_backlight = 1;
		ctx->brightness = ST7789V_BRIGHTNESS_MAX;
		ctx->refresh_rate = ST7789V_DEFAULT_REFRESH_RATE;
		ctx->initialized = 1;
		ctx->display_on = 1;
		ctx->lcd_on = 1;
		ctx->is_enabled = true;
	}
	else {						// Set status as reset
		ctx->lcd_backlight = 0;
		ctx->brightness = ST7789V_DEFAULT_BRIGHTNESS;
		ctx->refresh_rate = ST7789V_DEFAULT_REFRESH_RATE;
		ctx->initialized = 0;
		ctx->display_on = 0;
		ctx->lcd_on = 0;
		ctx->is_enabled = false;
	}

	printk(KERN_DEBUG "st7789v_probe2\n");
#ifdef CONTROL_LCD_POWER_SUPPLY
	ctx->power = devm_regulator_get(dev, "power");
	if (IS_ERR(ctx->power))
		return PTR_ERR(ctx->power);
#endif // CONTROL_LCD_POWER_SUPPLY

	dev_set_drvdata(dev, ctx);

	ret = drm_panel_add(&ctx->panel);
	if (ret < 0) {
		printk(KERN_DEBUG "st7789v_probe fail ret=%d\n", ret);
		return ret;
	}

	if (!need_initialize) {		// Set status as in boot
		mtk_dbi_host_attach_w_panel_status(&ctx->panel, &lcm_param, &ops, true, true);
	}
	else {
		mtk_dbi_host_attach_w_panel_status(&ctx->panel, &lcm_param, &ops, false, false);
	}		// Set status as in boot
#ifdef CONFIG_DRM_NEC
	nec_lcd_attach(&ctx->panel, &st7789v_lcd_ops);
#endif // CONFIG_DRM_NEC
	printk(KERN_DEBUG "st7789v_probe ops=%p\n", ops);
	printk(KERN_DEBUG "st7789v_probe ops->send_cmd=%p\n", ops->send_cmd);

	return 0;
}

static int st7789v_remove(struct device *dev)
{
	struct st7789v *ctx = dev_get_drvdata(dev);

	drm_panel_remove(&ctx->panel);

	return 0;
}

static const struct of_device_id st7789v_of_match[] = {
	{ .compatible = "sitronix,st7789v" },
	{ }
};
MODULE_DEVICE_TABLE(of, st7789v_of_match);

static int st7789v_driver_platform_probe(struct platform_device *pdev)
{
	const struct of_device_id *id;
	id = of_match_node(st7789v_of_match, pdev->dev.of_node);
	if (!id) {
		return -ENODEV;
	}

	return st7789v_probe(&pdev->dev);
}

static int st7789v_driver_platform_remove(struct platform_device *pdev)
{
	return st7789v_remove(&pdev->dev);
}

static struct platform_driver st7789v_driver_platform_driver = {
	.driver = {
		.name = "panel-innolux-st7789v",
		.of_match_table = st7789v_of_match,
	},
	.probe = st7789v_driver_platform_probe,
	.remove = st7789v_driver_platform_remove,
	//.shutdown = panel_simple_platform_shutdown,
};

static int __init st7789v_driver_init(void)
{
	int err;

	err = platform_driver_register(&st7789v_driver_platform_driver);
	if (err < 0) {
		return err;
	}

	return 0;
}
module_init(st7789v_driver_init);

static void __exit st7789v_driver_exit(void)
{
	platform_driver_unregister(&st7789v_driver_platform_driver);
}
module_exit(st7789v_driver_exit);

MODULE_DESCRIPTION("InnoLux Sitronix ST7789V LCD Driver for DBI-B");
MODULE_LICENSE("GPL v2");
