// SPDX-License-Identifier: GPL-2.0

/*
 * Copyright (c) 2019 MediaTek Inc.
 */


/**
 * @file	mtk_gpufreq_core
 * @brief   Driver for GPU-DVFS
 */

/**
 * ===============================================
 * SECTION : Include files
 * ===============================================
 */
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>
#include <linux/of.h>
#include <linux/of_address.h>
#include <linux/uaccess.h>
#include <linux/pm_qos.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/pm_runtime.h>
#include <linux/module.h>
#include <linux/thermal.h>
#include "mboot_params.h"
#include "mtk_gpufreq_common.h"
/* #define BRING_UP */

#include "mtk_gpufreq_core.h"
#include "mtk_gpufreq.h"

/* For freqency hopping */
#include "clk-mtk.h"
#include <dt-bindings/clock/mt6890-clk.h>

/* #include "mtk_thermal_typedefs.h" */
#ifdef FHCTL_READY
#include "mt_freqhopping.h"
#include "mt_fhreg.h"
#endif

#ifdef MT_GPUFREQ_PBM_SUPPORT
#include "mtk_pbm.h"
#endif /* ifdef MT_GPUFREQ_PBM_SUPPORT */

#ifdef CONFIG_MTK_QOS_SUPPORT
#include "mtk_gpu_bw.h"
#endif

#include <linux/random.h>

#ifdef CONFIG_MTK_STATIC_POWER
#include "mtk_static_power.h"
#endif /* ifdef CONFIG_MTK_STATIC_POWER */

#if IS_ENABLED(CONFIG_MTK_BATTERY_OC_POWER_THROTTLING) && \
		defined(MT_GPUFREQ_BATT_OC_PROTECT)
#include "mtk_battery_oc_throttling.h"
#endif
#if IS_ENABLED(CONFIG_MTK_LOW_BATTERY_POWER_THROTTLING) && \
		defined(MT_GPUFREQ_LOW_BATT_VOLT_PROTECT)
#include "mtk_low_battery_throttling.h"
#endif
#if IS_ENABLED(CONFIG_MTK_BATTERY_PERCENTAGE_POWER_THROTTLING) && \
		defined(MT_GPUFREQ_BATT_PERCENT_PROTECT)
#include "mtk_battery_percentage_throttling.h"
#endif

#include <linux/string.h>



/**
 * ===============================================
 * SECTION : Local functions declaration
 * ===============================================
 */

/*
To-do:
	__mt_gpufreq_vgpu_set_mode()
	__mt_gpufreq_get_opp_idx_by_volt()
	__mt_gpufreq_get_limited_freq_by_power()
	__mt_update_gpufreqs_power_table()
	__mt_gpufreq_update_max_limited_idx()

	Battery related
	__mt_gpufreq_batt_oc_protect()
	__mt_gpufreq_batt_percent_protect()
	__mt_gpufreq_low_batt_protect()
 */

static int __mt_gpufreq_pdrv_probe(struct platform_device *pdev);
static void __mt_gpufreq_set(unsigned int freq_old, unsigned int freq_new,
	unsigned int volt_old, unsigned int volt_new);
static void __mt_gpufreq_bucks_enable(void);
static void __mt_gpufreq_bucks_disable(void);
static unsigned int __mt_gpufreq_get_cur_volt(void);
static unsigned int __mt_gpufreq_get_cur_freq(void);
static enum g_post_divider_power_enum
__mt_gpufreq_get_post_divider_power(unsigned int freq, unsigned int efuse);
static void __mt_gpufreq_switch_to_clksrc(enum g_clock_source_enum clksrc);
static void __mt_gpufreq_kick_pbm(int enable);
static void __mt_gpufreq_clock_switch(unsigned int freq_new);
static void __mt_gpufreq_vcore_volt_switch(unsigned int volt_new);
static unsigned int __mt_gpufreq_calculate_dds(unsigned int freq_khz,
		enum g_post_divider_power_enum post_divider_power);
static void __mt_gpufreq_setup_opp_power_table(int num);
static void __mt_gpufreq_set_fixed_volt(int fixed_volt);
static void __mt_gpufreq_set_fixed_freq(int fixed_freq);

/**
 * ===============================================
 * SECTION : Local variables definition
 * ===============================================
 */
static void __iomem *g_apmixed_base;
static void __iomem *g_mfg_base;
static bool g_volt_enable_state;
static bool g_pre_pwr_off_opp_set;
static bool g_pre_pwr_off_state;
static bool g_fixed_freq_volt_state;
static bool g_keep_opp_freq_state;
static bool g_opp_stress_test_state;
static int g_cur_vcore_opp = VCORE_OPP_0;
static unsigned int g_max_limited_idx;
static unsigned int g_opp_idx_num;
static unsigned int g_cur_opp_volt;
static unsigned int g_cur_opp_freq;
static unsigned int g_cur_opp_cond_idx;
static unsigned int g_cur_opp_idx;
static unsigned int g_pre_pwr_off_opp_idx;
static unsigned int g_keep_opp_freq;
static unsigned int g_keep_opp_freq_idx;
static unsigned int g_fixed_freq;
static unsigned int g_fixed_volt;
static struct g_pmic_info *g_pmic;
static struct g_clk_info *g_clk;
static struct mt_gpufreq_power_table_info *g_power_table;
static struct g_opp_table_info *g_opp_table_default;
static struct g_opp_table_info *g_opp_table;
static enum g_post_divider_power_enum g_cur_post_divider_power;
static DEFINE_MUTEX(mt_gpufreq_lock);
static int g_clock_on;

static const struct of_device_id g_gpufreq_of_match[] = {
	{ .compatible = "mediatek,gpufreq" },
	{ /* sentinel */ }
};
static struct platform_driver g_gpufreq_pdrv = {
	.probe = __mt_gpufreq_pdrv_probe,
	.remove = NULL,
	.driver = {
		.name = "gpufreq",
		.owner = THIS_MODULE,
		.of_match_table = g_gpufreq_of_match,
	},
};
static struct g_opp_table_info g_opp_table_segment1[] = {
GPUOP(SEG1_GPU_DVFS_FREQ0, SEG1_GPU_DVFS_VOLT0, 0),
GPUOP(SEG1_GPU_DVFS_FREQ1, SEG1_GPU_DVFS_VOLT1, 1),
GPUOP(SEG1_GPU_DVFS_FREQ2, SEG1_GPU_DVFS_VOLT2, 2),
GPUOP(SEG1_GPU_DVFS_FREQ3, SEG1_GPU_DVFS_VOLT3, 3),
};
/**
 * ===============================================
 * SECTION : API definition
 * ===============================================
 */
/*
 * get OPP table index by freq (Hz)
 */
int mt_gpufreq_get_opp_idx_by_freq(unsigned int gpufreq_hz)
{
	int i = g_opp_idx_num - 1;
	unsigned int gpufreq_khz;

	gpufreq_khz = gpufreq_hz / 1000;
	while (i >= 0) {
		if (g_opp_table[i--].gpufreq_khz >= gpufreq_khz)
			goto EXIT;
	}

EXIT:
	return i+1;
}
EXPORT_SYMBOL(mt_gpufreq_get_opp_idx_by_freq);

unsigned int mt_gpufreq_get_pow_by_idx(unsigned int idx)
{
	if (idx < g_opp_idx_num) {
		gpufreq_pr_debug("@%s: idx = %d, volt = %d\n", __func__,
			idx, g_power_table[idx].gpufreq_power);
		return g_power_table[idx].gpufreq_power;
	}

	gpufreq_pr_debug("@%s: not found, idx = %d\n", __func__, idx);
	return 0;
}
EXPORT_SYMBOL(mt_gpufreq_get_pow_by_idx);

unsigned int mt_gpufreq_get_dyn_power(unsigned int freq_khz, unsigned int volt)
{
	unsigned int p_dynamic = 0;
	unsigned int ref_freq = 0;
	unsigned int ref_volt = 0;

	p_dynamic = GPU_ACT_REF_POWER;
	ref_freq = GPU_ACT_REF_FREQ;
	ref_volt = GPU_ACT_REF_VOLT;

	p_dynamic = p_dynamic *
			((freq_khz * 100) / ref_freq) *
			((volt * 100) / ref_volt) *
			((volt * 100) / ref_volt) / (100 * 100 * 100);
	return p_dynamic;
}
EXPORT_SYMBOL(mt_gpufreq_get_dyn_power);

/* API : get static leakage power */
unsigned int mt_gpufreq_get_leakage_mw(void)
{
	int temp = 0;
	struct thermal_zone_device *tzd;

	tzd = thermal_zone_get_zone_by_name("gpu0");
	if (IS_ERR(tzd)) {
		temp = 40;
	} else {
		thermal_zone_get_temp(tzd, &temp);
		temp /= 1000;

		if ((temp < -20) || (temp > 125)) {
			gpufreq_pr_debug("@%s: abnormal temp %d!\n", __func__, temp);
			temp = 65;
		}
	}

	/*
	unsigned int cur_vcore = __mt_gpufreq_get_cur_volt() / 100;
	int leak_power;
	leak_power = mt_spower_get_leakage(MTK_SPOWER_GPU, cur_vcore, temp);
	if (g_volt_enable_state && leak_power > 0)
		return leak_power;
	else
		return 0;
	*/
	return 130;
}

void mt_gpufreq_check_bus_idle(void)
{
	u32 val;

	gpufreq_pr_debug("@%s\n", __func__);

	/* MFG_QCHANNEL_CON (0x13fb_f0b4) bit [1:0] = 0x1 */
	writel(0x00000001, g_mfg_base + 0xb4);

	/* set register MFG_DEBUG_SEL (0x13fb_f170) bit [7:0] = 0x03 */
	writel(0x00000003, g_mfg_base + 0x170);

	/* polling register MFG_DEBUG_TOP (0x13fb_f178) bit 2 = 0x1 */
	/* => 1 for bus idle, 0 for bus non-idle */
	do {
		val = readl(g_mfg_base + 0x178);
	} while ((val & 0x4) != 0x4);
}
EXPORT_SYMBOL(mt_gpufreq_check_bus_idle);

void mt_gpufreq_external_cg_control(void)
{
	u32 val;

	gpufreq_pr_debug("@%s\n", __func__);

	/* [F] MFG_ASYNC_CON 0x13FB_F020 [22] MEM0_MST_CG_ENABLE = 0x1 */
	/* [J] MFG_ASYNC_CON 0x13FB_F020 [23] MEM0_SLV_CG_ENABLE = 0x1 */
	/* [H] MFG_ASYNC_CON 0x13FB_F020 [24] MEM1_MST_CG_ENABLE = 0x1 */
	/* [L] MFG_ASYNC_CON 0x13FB_F020 [25] MEM1_SLV_CG_ENABLE = 0x1 */
	val = readl(g_mfg_base + 0x20);
	val |= (1UL << 22);
	val |= (1UL << 23);
	val |= (1UL << 24);
	val |= (1UL << 25);
	writel(val, g_mfg_base + 0x20);

	/* [D] MFG_GLOBAL_CON 0x13FB_F0B0 [10] GPU_CLK_FREE_RUN = 0x0 */
	/* [D] MFG_GLOBAL_CON 0x13FB_F0B0 [9] MFG_SOC_OUT_AXI_FREE_RUN = 0x0 */
	val = readl(g_mfg_base + 0xB0);
	val &= ~(1UL << 10);
	val &= ~(1UL << 9);
	writel(val, g_mfg_base + 0xB0);

	/* [D] MFG_QCHANNEL_CON 0x13FB_F0B4 [4] QCHANNEL_ENABLE = 0x1 */
	val = readl(g_mfg_base + 0xB4);
	val |= (1UL << 4);
	writel(val, g_mfg_base + 0xB4);

	/* [E] MFG_GLOBAL_CON 0x13FB_F0B0 [19] PWR_CG_FREE_RUN = 0x0 */
	/* [P] MFG_GLOBAL_CON 0x13FB_F0B0 [8] MFG_SOC_IN_AXI_FREE_RUN = 0x0 */
	val = readl(g_mfg_base + 0xB0);
	val &= ~(1UL << 19);
	val &= ~(1UL << 8);
	writel(val, g_mfg_base + 0xB0);

	/*[O] MFG_ASYNC_CON_1 0x13FB_F024 [0] FAXI_CK_SOC_IN_EN_ENABLE = 0x1*/
	val = readl(g_mfg_base + 0x24);
	val |= (1UL << 0);
	writel(val, g_mfg_base + 0x24);

	return;
}
EXPORT_SYMBOL(mt_gpufreq_external_cg_control);

/*
 * API : handle frequency change request
 */
unsigned int mt_gpufreq_target(unsigned int idx)
{
	unsigned int target_freq;
	unsigned int target_volt;
	unsigned int target_idx;
	unsigned int target_cond_idx;

	mutex_lock(&mt_gpufreq_lock);

	if (g_opp_stress_test_state) {
		get_random_bytes(&idx, sizeof(idx));
		idx = idx % g_opp_idx_num;
		gpufreq_pr_debug("@%s: OPP stress test index: %d\n",
			__func__, idx);
	}

	if (idx > (g_opp_idx_num - 1)) {
		gpufreq_pr_debug("@%s: OPP index (%d) is out of range\n",
			__func__, idx);
		mutex_unlock(&mt_gpufreq_lock);
		return -1;
	}

	/* look up for the target OPP table */
	target_freq = g_opp_table[idx].gpufreq_khz;
	target_volt = g_opp_table[idx].gpufreq_volt;
	target_idx = g_opp_table[idx].gpufreq_idx;
	target_cond_idx = idx;

	gpufreq_pr_debug("@%s: receive freq: %d, index: %d\n",
		__func__, target_freq, target_idx);

	/* OPP freq is limited by Thermal/Power/PBM */
	if (g_max_limited_idx != 0) {
		if (target_freq > g_opp_table[g_max_limited_idx].gpufreq_khz) {
			target_idx = target_cond_idx = g_max_limited_idx;
			gpufreq_pr_debug(
				"@%s: OPP freq is limited by g_max, idx = %d\n",
				__func__, target_cond_idx);
		}
	}

	/* If /proc command keep OPP freq */
	if (g_keep_opp_freq_state) {
		target_idx = target_cond_idx = g_keep_opp_freq_idx;
		gpufreq_pr_debug("@%s: keep OPP freq, idx = %d\n",
			__func__, target_cond_idx);
	}

	/* If /proc command fix the freq and volt */
	if (g_fixed_freq_volt_state) {
		target_freq = g_fixed_freq;
		target_volt = g_fixed_volt;
		target_idx = target_cond_idx = 0;
		gpufreq_pr_debug(
			"@%s: fixed both freq and volt, freq = %d, volt = %d\n",
			__func__, target_freq, target_volt);
	} else {
		target_freq = g_opp_table[target_idx].gpufreq_khz;
		target_volt = g_opp_table[target_idx].gpufreq_volt;
	}

	/* target freq == current freq
	 * && target volt == current volt, skip it
	 */
	if (g_cur_opp_freq == target_freq && g_cur_opp_volt == target_volt) {
		gpufreq_pr_debug("@%s: Freq: %d ---> %d (skipped)\n",
			__func__, g_cur_opp_freq, target_freq);
		mutex_unlock(&mt_gpufreq_lock);
		return 0;
	}

	/* set to the target frequency and voltage */
	__mt_gpufreq_set(g_cur_opp_freq, target_freq, g_cur_opp_volt,
		target_volt);

	g_cur_opp_idx = target_idx;
	g_cur_opp_cond_idx = target_cond_idx;

	mutex_unlock(&mt_gpufreq_lock);

	return 0;
}
EXPORT_SYMBOL(mt_gpufreq_target);

/*
 * API : get current Thermal/Power/PBM limited OPP table index
 */
unsigned int mt_gpufreq_get_thermal_limit_index(void)
{
	gpufreq_pr_debug("@%s: current GPU Thermal/Power/PBM limit index is %d\n",
			__func__, g_max_limited_idx);
	return g_max_limited_idx;
}
EXPORT_SYMBOL(mt_gpufreq_get_thermal_limit_index);
/* API : get frequency via OPP table index */
unsigned int mt_gpufreq_get_freq_by_idx(unsigned int idx)
{
	if (idx < g_opp_idx_num) {
		gpufreq_pr_debug("@%s: idx = %d, freq = %d\n", __func__, idx,
			g_opp_table[idx].gpufreq_khz);
		return g_opp_table[idx].gpufreq_khz;
	}

	gpufreq_pr_debug("@%s: not found, idx = %d\n", __func__, idx);
	return 0;
}
EXPORT_SYMBOL(mt_gpufreq_get_freq_by_idx);
/*
 * API : get current OPP table conditional index
 */
unsigned int mt_gpufreq_get_cur_freq_index(void)
{
	gpufreq_pr_debug("@%s: current OPP table conditional index is %d\n",
		__func__, g_cur_opp_cond_idx);
	if (g_pre_pwr_off_state)
		return g_pre_pwr_off_opp_idx;

	return g_cur_opp_cond_idx;
}
EXPORT_SYMBOL(mt_gpufreq_get_cur_freq_index);
/* API : get OPP table index number */
unsigned int mt_gpufreq_get_dvfs_table_num(void)
{
	return g_opp_idx_num;
}
EXPORT_SYMBOL(mt_gpufreq_get_dvfs_table_num);
/*
 * API : get current OPP table frequency
 */
unsigned int mt_gpufreq_get_cur_freq(void)
{
	gpufreq_pr_debug(
		"@%s: current frequency is %d MHz\n",
		__func__, g_cur_opp_freq / 1000);
	return g_cur_opp_freq;
}
EXPORT_SYMBOL(mt_gpufreq_get_cur_freq);
/*
 * enable Clock Gating
 */
void mt_gpufreq_enable_CG(void)
{
	int ret;
	mutex_lock(&mt_gpufreq_lock);

#ifdef MT_GPUFREQ_SRAM_DEBUG
	aee_rr_rec_gpu_dvfs_status(0x10 | (aee_rr_curr_gpu_dvfs_status() & 0x0F));
#endif

	ret = clk_prepare_enable(g_clk->cg_bg3d);
	if (ret)
		gpufreq_pr_debug(
		"@%s: enable CLK_MFGCFG_BG3D failed, ret = %d\n",
		__func__, ret);

#ifdef MT_GPUFREQ_SRAM_DEBUG
	aee_rr_rec_gpu_dvfs_status(0x20 | (aee_rr_curr_gpu_dvfs_status() & 0x0F));
#endif
        gpufreq_pr_debug("@%s: enable CG done\n", __func__);
        g_clock_on = 1;
        mutex_unlock(&mt_gpufreq_lock);
}
EXPORT_SYMBOL(mt_gpufreq_enable_CG);
/*
 * disable Clock Gating
 */
void mt_gpufreq_disable_CG(void)
{
	mutex_lock(&mt_gpufreq_lock);

	g_clock_on = 0;

#ifdef MT_GPUFREQ_SRAM_DEBUG
	aee_rr_rec_gpu_dvfs_status(0x30 | (aee_rr_curr_gpu_dvfs_status() & 0x0F));
#endif

	clk_disable_unprepare(g_clk->cg_bg3d);

#ifdef MT_GPUFREQ_SRAM_DEBUG
	aee_rr_rec_gpu_dvfs_status(0x40 | (aee_rr_curr_gpu_dvfs_status() & 0x0F));
#endif
	gpufreq_pr_debug("@%s: disable CG done\n", __func__);
	mutex_unlock(&mt_gpufreq_lock);
}
EXPORT_SYMBOL(mt_gpufreq_disable_CG);

/*
 * enable MTCMOS
 */
void mt_gpufreq_enable_MTCMOS(bool bEnableHWAPM)
{
	int i32Ret;

	i32Ret = pm_runtime_get_sync(g_clk->mtcmos_mfg);
	if (i32Ret < 0)
		gpufreq_perr("@%s: failed when enable mtcmos_mfg\n",
		__func__);
	else
		gpufreq_pr_debug("@%s: enable MTCMOS done\n", __func__);

	mt_gpufreq_target(g_pre_pwr_off_opp_idx);
	g_pre_pwr_off_state = false;

	return;
}
EXPORT_SYMBOL(mt_gpufreq_enable_MTCMOS);
/*
 * disable MTCMOS
 */
void mt_gpufreq_disable_MTCMOS(bool bEnableHWAPM)
{
	int i32Ret;

	g_pre_pwr_off_opp_idx = g_cur_opp_idx;
	g_pre_pwr_off_opp_set = true;
	g_pre_pwr_off_state = true;

	i32Ret = pm_runtime_put_sync(g_clk->mtcmos_mfg);
	if (i32Ret < 0)
		gpufreq_perr("@%s: failed when disable mtcmos_mfg\n",
		__func__);
	else
		gpufreq_pr_debug("@%s: disable MTCMOS done\n", __func__);

	return;
}
EXPORT_SYMBOL(mt_gpufreq_disable_MTCMOS);
/*
 * disable bucks (VGPU)
 */
static void __mt_gpufreq_bucks_disable(void)
{
	__mt_gpufreq_vcore_volt_switch(0);
}
/*
 * enable bucks (VGPU)
 */
static void __mt_gpufreq_bucks_enable(void)
{
	__mt_gpufreq_vcore_volt_switch(g_cur_opp_volt);
}
/*
 * API : GPU voltage on/off setting
 * 0 : off
 * 1 : on
 */
unsigned int mt_gpufreq_voltage_enable_set(unsigned int enable)
{
	mutex_lock(&mt_gpufreq_lock);

	gpufreq_pr_debug("@%s: begin, enable = %d, g_volt_enable_state = %d\n",
		__func__, enable, g_volt_enable_state);

	if (enable == 1) {
		__mt_gpufreq_bucks_enable();
		__mt_gpufreq_kick_pbm(1);
		g_volt_enable_state = true;
		gpufreq_pr_debug("@%s: VGPU is on\n", __func__);
	} else if (enable == 0)  {
		__mt_gpufreq_bucks_disable();
		__mt_gpufreq_kick_pbm(0);
		g_volt_enable_state = false;
		gpufreq_pr_debug("@%s: VGPU is off\n", __func__);
	}

	gpufreq_pr_debug("@%s: end, enable = %d, g_volt_enable_state = %d\n",
			__func__, enable, g_volt_enable_state);

	mutex_unlock(&mt_gpufreq_lock);

	return 0;
}
EXPORT_SYMBOL(mt_gpufreq_voltage_enable_set);


#ifdef CONFIG_PROC_FS
/*
 * PROCFS : show important variables for debugging
 */
static int mt_gpufreq_var_dump_proc_show(struct seq_file *m, void *v)
{
	seq_printf(m, "g_cur_opp_idx = %d, g_cur_opp_cond_idx = %d\n",
			g_cur_opp_idx, g_cur_opp_cond_idx);
	seq_printf(m,
	"g_cur_opp_freq = %d, g_cur_opp_volt = %d, g_cur_opp_vsram_volt = %d\n",
		g_cur_opp_freq, g_cur_opp_volt, 75000);
	seq_printf(m, "real freq = %d, real volt = %d, real vsram_volt = %d\n",
			__mt_gpufreq_get_cur_freq(),
			__mt_gpufreq_get_cur_volt(),
			75000);
	seq_printf(m, "current vcore opp = %d\n", g_cur_vcore_opp);
	seq_printf(m, "g_volt_enable_state = %d\n", g_volt_enable_state);
	seq_printf(m, "g_opp_stress_test_state = %d\n",
		g_opp_stress_test_state);
	seq_printf(m, "g_max_limited_idx = %d\n", g_max_limited_idx);

	return 0;
}
/*
 * PROCFS : show OPP table
 */
static int mt_gpufreq_opp_dump_proc_show(struct seq_file *m, void *v)
{
	int i;
	int power = 0;

	for (i = 0; i < g_opp_idx_num; i++) {
		seq_printf(m, "[%d] ", i);
		seq_printf(m, "freq = %d, ", g_opp_table[i].gpufreq_khz);
		seq_printf(m, "volt = %d, ", g_opp_table[i].gpufreq_volt);
		seq_printf(m, "vsram = %d, ", 75000);

		power = mt_gpufreq_get_dyn_power(g_opp_table[i].gpufreq_khz,
										g_opp_table[i].gpufreq_volt);
		power += mt_gpufreq_get_leakage_mw();

		seq_printf(m, "gpufreq_power = %d\n", power);
	}

	return 0;
}


/*
 * PROCFS : show current stress test state
 */
static int mt_gpufreq_opp_stress_test_proc_show(struct seq_file *m, void *v)
{
	if (g_opp_stress_test_state) {
		seq_puts(m, "Stress test is enabled\n");
	} else
		seq_puts(m, "Stress test is disabled\n");

	return 0;
}

/*
 * PROCFS : For stress test
 * 0 : free run
 * 1 : stress test
 */
static ssize_t mt_gpufreq_opp_stress_test_proc_write(struct file *file,
		const char __user *buffer, size_t count, loff_t *data)
{
	char buf[64];
	unsigned int len = 0;
	unsigned int value = 0;
	int ret = -EFAULT;

	len = (count < (sizeof(buf) - 1)) ? count : (sizeof(buf) - 1);

	if (copy_from_user(buf, buffer, len))
		goto out;

	buf[len] = '\0';

	if (kstrtouint(buf, 10, &value) == 0) {
		if (value == 1) {
			ret = 0;
			g_opp_stress_test_state = true;
		} else {
			ret = 0;
			g_opp_stress_test_state = false;
		}
	}

out:
	return (ret < 0) ? ret : count;
}

/*
 * PROCFS : show current keeping OPP frequency state
 */
static int mt_gpufreq_opp_freq_proc_show(struct seq_file *m, void *v)
{
	if (g_keep_opp_freq_state) {
		seq_puts(m, "Keeping OPP frequency is enabled\n");
		seq_printf(m, "[%d] ", g_keep_opp_freq_idx);
		seq_printf(m, "freq = %d, ",
			g_opp_table[g_keep_opp_freq_idx].gpufreq_khz);
		seq_printf(m, "volt = %d, ",
			g_opp_table[g_keep_opp_freq_idx].gpufreq_volt);
		seq_printf(m, "vsram = %d\n", 75000);
	} else
		seq_puts(m, "Keeping OPP frequency is disabled\n");

	return 0;
}

/*
 * PROCFS : keeping OPP frequency setting
 * 0 : free run
 * 1 : keep OPP frequency
 */
static ssize_t mt_gpufreq_opp_freq_proc_write(struct file *file,
		const char __user *buffer, size_t count, loff_t *data)
{
	char buf[64];
	unsigned int len = 0;
	unsigned int value = 0;
	unsigned int i = 0;
	int ret = -EFAULT;

	len = (count < (sizeof(buf) - 1)) ? count : (sizeof(buf) - 1);

	if (copy_from_user(buf, buffer, len))
		goto out;

	buf[len] = '\0';

	if (kstrtouint(buf, 10, &value) == 0) {
		if (value == 0) {
			ret = 0;
			g_keep_opp_freq_state = false;
		} else {
			for (i = 0; i < g_opp_idx_num; i++) {
				if (value == g_opp_table[i].gpufreq_khz) {
					ret = 0;
					g_keep_opp_freq_idx = i;
					g_keep_opp_freq_state = true;
					g_keep_opp_freq = value;
					mt_gpufreq_voltage_enable_set(1);
					mt_gpufreq_target(i);
					break;
				}
			}
		}
	}

out:
	return (ret < 0) ? ret : count;
}

/*
 * PROCFS : show current fixed freq & volt state
 */
static int mt_gpufreq_fixed_freq_volt_proc_show(struct seq_file *m, void *v)
{
	if (g_fixed_freq_volt_state) {
		seq_puts(m, "Fixing freq & volt is enabled\n");
		seq_printf(m, "g_fixed_freq = %d\n", g_fixed_freq);
		seq_printf(m, "g_fixed_volt = %d\n", g_fixed_volt);
	} else {
		seq_puts(m, "Fixing freq & volt is disabled\n");
	}

	return 0;
}

/*
 * PROCFS : fixed freq & volt state setting
 */
static ssize_t mt_gpufreq_fixed_freq_volt_proc_write(struct file *file,
		const char __user *buffer, size_t count, loff_t *data)
{
	char buf[64];
	unsigned int len = 0;
	int ret = -EFAULT;
	unsigned int fixed_freq = 0;
	unsigned int fixed_volt = 0;

	len = (count < (sizeof(buf) - 1)) ? count : (sizeof(buf) - 1);

	if (copy_from_user(buf, buffer, len))
		goto out;

	buf[len] = '\0';

	if (sscanf(buf, "%d %d", &fixed_freq, &fixed_volt) == 2) {
		ret = 0;
		if ((fixed_freq == 0) && (fixed_volt == 0)) {
			g_fixed_freq_volt_state = false;
			g_fixed_freq = 0;
			g_fixed_volt = 0;
		} else {
			g_cur_opp_freq = __mt_gpufreq_get_cur_freq();

			mt_gpufreq_voltage_enable_set(1);
			if (fixed_freq > g_cur_opp_freq) {
				__mt_gpufreq_set_fixed_volt(fixed_volt);
				__mt_gpufreq_set_fixed_freq(fixed_freq);
			} else {
				__mt_gpufreq_set_fixed_freq(fixed_freq);
				__mt_gpufreq_set_fixed_volt(fixed_volt);
			}
			g_fixed_freq_volt_state = true;
			g_fixed_freq = fixed_freq;
			g_fixed_volt = fixed_volt;
		}
	}

out:
	return (ret < 0) ? ret : count;
}



/*
 * PROCFS : initialization
 */
PROC_FOPS_RO(gpufreq_opp_dump);
PROC_FOPS_RW(gpufreq_opp_freq);
PROC_FOPS_RO(gpufreq_var_dump);
PROC_FOPS_RW(gpufreq_fixed_freq_volt);
PROC_FOPS_RW(gpufreq_opp_stress_test);
static int __mt_gpufreq_create_procfs(void)
{
	struct proc_dir_entry *dir = NULL;
	int i;

	struct pentry {
		const char *name;
		const struct file_operations *fops;
	};

	const struct pentry entries[] = {
		PROC_ENTRY(gpufreq_opp_dump),
		PROC_ENTRY(gpufreq_opp_freq),
		PROC_ENTRY(gpufreq_var_dump),
		PROC_ENTRY(gpufreq_fixed_freq_volt),
		PROC_ENTRY(gpufreq_opp_stress_test),
	};

	dir = proc_mkdir("gpufreq", NULL);
	if (!dir) {
		gpufreq_perr("@%s: fail to create /proc/gpufreq\n", __func__);
		return -ENOMEM;
	}

	for (i = 0; i < ARRAY_SIZE(entries); i++) {
		if (!proc_create(entries[i].name, 0664, dir, entries[i].fops))
			gpufreq_perr("@%s: create /proc/gpufreq/%s failed\n",
				__func__, entries[i].name);
	}

	return 0;
}
#endif /* CONFIG_PROC_FS */

/*
 * set fixed frequency for PROCFS: fixed_freq_volt
 */
static void __mt_gpufreq_set_fixed_freq(int fixed_freq)
{
	gpufreq_pr_debug("@%s: before, g_fixed_freq = %d, g_fixed_volt = %d\n",
			__func__, g_fixed_freq, g_fixed_volt);

	g_fixed_freq = fixed_freq;
	g_fixed_volt = g_cur_opp_volt;

	__mt_gpufreq_clock_switch(g_fixed_freq);
	gpufreq_pr_debug("@%s: now, g_fixed_freq = %d, g_fixed_volt = %d\n",
			__func__, g_fixed_freq, g_fixed_volt);

	g_cur_opp_freq = g_fixed_freq;
}

/*
 * set fixed voltage for PROCFS: fixed_freq_volt
 */
static void __mt_gpufreq_set_fixed_volt(int fixed_volt)
{
	gpufreq_pr_debug("@%s: before, g_fixed_freq = %d, g_fixed_volt = %d\n",
			__func__, g_fixed_freq, g_fixed_volt);

	g_fixed_freq = g_cur_opp_freq;
	g_fixed_volt = fixed_volt;

	__mt_gpufreq_vcore_volt_switch(g_fixed_volt);
	gpufreq_pr_debug("@%s: now, g_fixed_freq = %d, g_fixed_volt = %d\n",
			__func__, g_fixed_freq, g_fixed_volt);

	g_cur_opp_volt = g_fixed_volt;
}

/*
 * kick Power Budget Manager(PBM) when OPP changed
 */
static void __mt_gpufreq_kick_pbm(int enable)
{
#ifdef MT_GPUFREQ_PBM_SUPPORT
	unsigned int power;
	unsigned int cur_freq;
	unsigned int cur_volt;
	unsigned int found = 0;
	int tmp_idx = -1;
	int i;

	cur_freq = __mt_gpufreq_get_cur_freq();
	cur_volt = __mt_gpufreq_get_cur_volt();

	if (enable) {
		for (i = 0; i < g_opp_idx_num; i++) {
			if (g_power_table[i].gpufreq_khz == cur_freq) {
				/* record idx since current voltage
				 * may not in DVFS table
				 */
				tmp_idx = i;

				if (g_power_table[i].gpufreq_volt == cur_volt) {
					power = g_power_table[i].gpufreq_power;
					found = 1;
					kicker_pbm_by_gpu(true,
						power, cur_volt / 100);
					gpufreq_pr_debug(
						"@%s: request GPU power = %d,",
						__func__, power);
					gpufreq_pr_debug(
						" cur_volt = %d uV, cur_freq = %d KHz\n",
					cur_volt * 10, cur_freq);
					return;
				}
			}
		}

		if (!found) {
			gpufreq_pr_debug("@%s: tmp_idx = %d\n",
				__func__, tmp_idx);
			if (tmp_idx != -1 && tmp_idx < g_opp_idx_num) {
				/* freq to find corresponding power budget */
				power = g_power_table[tmp_idx].gpufreq_power;
				kicker_pbm_by_gpu(true, power, cur_volt / 100);
				gpufreq_pr_debug("@%s: request GPU power = %d,",
				__func__, power);
				gpufreq_pr_debug(
				" cur_volt = %d uV, cur_freq = %d KHz\n",
				cur_volt * 10, cur_freq);
			} else {
				gpufreq_pr_debug(
				"@%s: Cannot found request power in power table",
				__func__);
				gpufreq_pr_debug(
				", cur_freq = %d KHz, cur_volt = %d uV\n",
				cur_freq, cur_volt * 10);
			}
		}
	} else {
		kicker_pbm_by_gpu(false, 0, cur_volt / 100);
	}
#endif /* ifdef MT_GPUFREQ_PBM_SUPPORT */
	return;
}

/*
 * switch to target clock source
 */
static void __mt_gpufreq_switch_to_clksrc(enum g_clock_source_enum clksrc)
{
	int ret;

	ret = clk_prepare_enable(g_clk->clk_mux);
	if (ret)
		gpufreq_pr_debug(
		"@%s: enable clk_mux(CLK_TOP_MFG_SEL) failed, ret = %d\n",
		__func__, ret);

	if (clksrc == CLOCK_MAIN) {
		clk_set_parent(g_clk->clk_mux, g_clk->clk_main_parent);
		gpufreq_pr_debug("@%s: switch to main clock source done\n",
			__func__);
	} else if (clksrc == CLOCK_SUB) {
		clk_set_parent(g_clk->clk_mux, g_clk->clk_sub_parent);
		gpufreq_pr_debug("@%s: switch to sub clock source done\n",
			__func__);
	} else {
		gpufreq_pr_debug(
			"@%s: clock source index is not valid, clksrc = %d\n",
			__func__, clksrc);
	}
	clk_disable_unprepare(g_clk->clk_mux);
}
/*
 * dds calculation for clock switching
 */
static unsigned int __mt_gpufreq_calculate_dds(unsigned int freq_khz,
		enum g_post_divider_power_enum post_divider_power)
{
	unsigned int dds = 0;

	gpufreq_pr_debug("@%s: request freq = %d, post_divider = %d\n",
		__func__, freq_khz, (1 << post_divider_power));

	/* [MT6890] dds is GPUPLL_CON2[21:0] */
	if ((freq_khz >= POST_DIV_8_MIN_FREQ) &&
		(freq_khz <= POST_DIV_4_MAX_FREQ)) {
		dds = (((freq_khz / TO_MHz_HEAD * (1 << post_divider_power))
		<< DDS_SHIFT) / GPUPLL_FIN + ROUNDING_VALUE) / TO_MHz_TAIL;
	} else {
		gpufreq_perr(
			"@%s: out of range, freq_khz = %d\n",
			__func__, freq_khz);
	}

	return dds;
}
/*
 * get post divider value
 * - VCO needs proper post divider value to get corresponding
 * dds value to adjust PLL value.
 * - e.g: In MT6758, VCO range is 2.0GHz - 4.0GHz, required frequency
 * is 900MHz, so post
 * divider could be 2(X), 4(3600/4), 8(X), 16(X).
 * - It may have special requiremt by DE in different efuse value
 * - e.g: In MT6779, efuse value(3'b001), VCO range is 1.5GHz - 3.8GHz,
 * required frequency
 * range is 375MHz - 900MHz, It can only use post divider 4, no post divider 2.
 */
static enum g_post_divider_power_enum
__mt_gpufreq_get_post_divider_power(unsigned int freq, unsigned int efuse)
{
	/* [MT6890]
	 *    VCO range: 1.5GHz - 3.8GHz by divider 1/2/4/8/16,
	 *    PLL range: 125MHz - 3.8GHz,
	 *    | VCO MAX | VCO MIN | POSTDIV | PLL OUT MAX | PLL OUT MIN |
	 *    |  3800   |  1500   |    1    |   3800MHz   |  1500MHz    | (X)
	 *    |  3800   |  1500   |    2    |   1900MHz   |   750MHz    | (X)
	 *    |  3800   |  1500   |    4    |   950MHz    |   375MHz    | (O)
	 *    |  3800   |  1500   |    8    |   475MHz    |   187.5MHz  | (O)
	 *    |  3800   |  2000   |   16    |   237.5MHz  |   125MHz    | (X)
	 */
	enum g_post_divider_power_enum post_divider_power = POST_DIV4;

	if (freq > 375000)
		post_divider_power = POST_DIV4;
	else
		post_divider_power = POST_DIV8;

	gpufreq_pr_debug(
		"@%s: freq = %d, post_divider_power = %d\n",
		__func__, freq, post_divider_power);

	return post_divider_power;
}
/*
 * switch clock(frequency) via PLL
 */
static void __mt_gpufreq_clock_switch(unsigned int freq_new)
{
	enum g_post_divider_power_enum post_divider_power;
	unsigned int cur_volt;
	unsigned int cur_freq;
	unsigned int dds;
	bool if_hopping_success;

	cur_volt = __mt_gpufreq_get_cur_volt();
	cur_freq = __mt_gpufreq_get_cur_freq();

	/* [MT6890] GPUPLL_CON2[24:26] is POST_DIVIDER
	 *    000 : /1
	 *    001 : /2
	 *    010 : /4
	 *    011 : /8
	 *    100 : /16
	 * [MT6890] GPUPLL_CON2[21:0] is SDM_PCW
	 */
	post_divider_power = __mt_gpufreq_get_post_divider_power(freq_new, 0);
	dds = __mt_gpufreq_calculate_dds(freq_new, post_divider_power);

	if (g_cur_post_divider_power == post_divider_power && mtk_fh_set_rate) {
		if_hopping_success = mtk_fh_set_rate(CLK_APMIXED_MFGPLL, dds, (int)(1 << post_divider_power));
		gpu_assert(if_hopping_success, GPU_FREQ_EXCEPTION,
			"hopping failing: %d\n", if_hopping_success);
	} else {
		gpufreq_pr_debug(
			"@%s: request GPU dds = 0x%x, cur_volt = %d, cur_freq = %d\n",
			__func__, dds, cur_volt, cur_freq);

		gpufreq_pr_debug("@%s: begin, freq = %d, GPUPLL_CON2 = 0x%x\n",
			__func__, freq_new, DRV_Reg32(GPUPLL_CON2));

		/* MFGPLL to MAINPLL_D5_D2 */
		__mt_gpufreq_switch_to_clksrc(CLOCK_SUB);
		/* dds = GPUPLL_CON2[21:0], POST_DIVIDER = GPUPLL_CON2[24:26] */
		DRV_WriteReg32(GPUPLL_CON2, (0x80000000) |
			(post_divider_power << POST_DIV_SHIFT) | dds);
		udelay(20);
		/* MAINPLL_D5_D2 to MFGPLL */
		__mt_gpufreq_switch_to_clksrc(CLOCK_MAIN);

		g_cur_post_divider_power = post_divider_power;

		gpufreq_pr_debug("@%s: end, freq = %d, GPUPLL_CON2 = 0x%x\n",
			__func__, freq_new, DRV_Reg32(GPUPLL_CON2));
	}

	return;
}

/*
 * switch VCore voltage via PMIC
 */
static void __mt_gpufreq_vcore_volt_switch(unsigned int volt_target)
{
	/*
	 * int regulator_set_voltage(struct regulator *regulator, int min_uV, int max_uV)
	 * @regulator: regulator source
	 * @min_uV: Minimum required voltage in uV
	 * @max_uV: Maximum acceptable voltage in uV  → Because this is vcore share regulator,  you must set as max voltage of this chip define (725000) or use INT_MAX
	 */

	if (volt_target >= 75000) {
		regulator_set_voltage(g_pmic->reg_vcore, VCORE_OPP_0, INT_MAX);
		g_cur_vcore_opp = VCORE_OPP_0;
	} else if (volt_target >= 65000) {
		regulator_set_voltage(g_pmic->reg_vcore, VCORE_OPP_1, INT_MAX);
		g_cur_vcore_opp = VCORE_OPP_1;
	} else if (volt_target >= 60000) {
		regulator_set_voltage(g_pmic->reg_vcore, VCORE_OPP_2, INT_MAX);
		g_cur_vcore_opp = VCORE_OPP_2;
	} else if (volt_target > 0) {
		regulator_set_voltage(g_pmic->reg_vcore, VCORE_OPP_3, INT_MAX);
		g_cur_vcore_opp = VCORE_OPP_3;
	} else {
		regulator_set_voltage(g_pmic->reg_vcore, VCORE_OPP_UNREQ, INT_MAX);
		g_cur_vcore_opp = VCORE_OPP_UNREQ;
	}
	return;
}

/*
 * frequency ramp up/down handler
 * - frequency ramp up need to wait voltage settle
 * - frequency ramp down do not need to wait voltage settle
 */
static void __mt_gpufreq_set(unsigned int freq_old, unsigned int freq_new,
	unsigned int volt_old, unsigned int volt_new)
{
	gpufreq_pr_debug(
		"@%s: freq: %d ---> %d, volt: %d ---> %d\n",
		__func__, freq_old, freq_new, volt_old, volt_new);

	if (freq_new > freq_old) {
		__mt_gpufreq_vcore_volt_switch(volt_new);
		__mt_gpufreq_clock_switch(freq_new);
	} else {
		__mt_gpufreq_clock_switch(freq_new);
		__mt_gpufreq_vcore_volt_switch(volt_new);
	}

	g_cur_opp_freq = __mt_gpufreq_get_cur_freq();

	gpu_assert(g_cur_opp_freq == freq_new, GPU_FREQ_EXCEPTION,
			"Clock switch failing: %d -> %d (target: %d)\n",
				freq_old, g_cur_opp_freq, freq_new);

	gpufreq_pr_debug(
		"@%s: real_freq = %d, real_volt = %d\n",
			__func__, __mt_gpufreq_get_cur_freq()
			, __mt_gpufreq_get_cur_volt());

	g_cur_opp_freq = freq_new;
	g_cur_opp_volt = volt_new;

	if (!g_pre_pwr_off_opp_set)
		__mt_gpufreq_kick_pbm(1);
	else
		g_pre_pwr_off_opp_set = false;
}

/*
 * get current frequency (KHZ)
 */
static unsigned int __mt_gpufreq_get_cur_freq(void)
{
	unsigned long mfgpll = 0;
	unsigned int post_divider_power = 0;
	unsigned int freq_khz = 0;
	unsigned long dds;

	mfgpll = DRV_Reg32(GPUPLL_CON2);
	dds = mfgpll & (0x3FFFFF);

	post_divider_power = (mfgpll & (0x7 << POST_DIV_SHIFT))
		>> POST_DIV_SHIFT;

	freq_khz = (((dds * TO_MHz_TAIL + ROUNDING_VALUE) * GPUPLL_FIN)
		>> DDS_SHIFT)
		/ (1 << post_divider_power) * TO_MHz_HEAD;

	gpufreq_pr_debug(
		"@%s: mfgpll = 0x%lx, freq = %d KHz, post_divider_power = %d\n",
		__func__, mfgpll, freq_khz, post_divider_power);

	return freq_khz;
}
/*
 * get current voltage (mV * 100)
 */
static unsigned int __mt_gpufreq_get_cur_volt(void)
{
	unsigned int volt = 0;

	if (g_volt_enable_state) {
		/* WARRNING: regulator_get_voltage prints uV */
		volt = regulator_get_voltage(g_pmic->reg_vcore) / 10;
	}

	gpufreq_pr_debug("@%s: volt = %d\n", __func__, volt);

	return volt;
}
/*
 *Set default OPP index at driver probe function
 */
static void __mt_gpufreq_set_initial(void)
{
	unsigned int cur_volt = 0;
	unsigned int cur_freq = 0;

	mutex_lock(&mt_gpufreq_lock);

	/* default OPP index */
	g_cur_opp_cond_idx = 3;

	/* set POST_DIVIDER initial value */
	g_cur_post_divider_power = POST_DIV_INIT;

	gpufreq_pr_debug("@%s: initial opp index = %d\n",
		__func__, g_cur_opp_cond_idx);

	cur_volt = __mt_gpufreq_get_cur_volt();
	cur_freq = __mt_gpufreq_get_cur_freq();

	__mt_gpufreq_set(cur_freq, g_opp_table[g_cur_opp_cond_idx].gpufreq_khz,
			cur_volt, g_opp_table[g_cur_opp_cond_idx].gpufreq_volt);

	g_cur_opp_freq = g_opp_table[g_cur_opp_cond_idx].gpufreq_khz;
	g_cur_opp_volt = g_opp_table[g_cur_opp_cond_idx].gpufreq_volt;
	g_cur_opp_idx = g_opp_table[g_cur_opp_cond_idx].gpufreq_idx;
	g_cur_opp_cond_idx = g_cur_opp_idx;
	g_pre_pwr_off_opp_idx = g_cur_opp_cond_idx;

	mutex_unlock(&mt_gpufreq_lock);
}
/*
 * (default) OPP table initialization
 */
static void __mt_gpufreq_setup_opp_table(struct g_opp_table_info *freqs,
	int num)
{
	int i = 0;

	g_opp_table = kzalloc((num) * sizeof(*freqs), GFP_KERNEL);
	g_opp_table_default = kzalloc((num) * sizeof(*freqs), GFP_KERNEL);

	if (g_opp_table == NULL || g_opp_table_default == NULL)
		return;

	for (i = 0; i < num; i++) {
		g_opp_table[i].gpufreq_khz = freqs[i].gpufreq_khz;
		g_opp_table[i].gpufreq_volt = freqs[i].gpufreq_volt;
		g_opp_table[i].gpufreq_idx = freqs[i].gpufreq_idx;

		g_opp_table_default[i].gpufreq_khz = freqs[i].gpufreq_khz;
		g_opp_table_default[i].gpufreq_volt = freqs[i].gpufreq_volt;
		g_opp_table_default[i].gpufreq_idx = freqs[i].gpufreq_idx;

		gpufreq_pr_debug(
			"@%s: [%d] idx = %u, freq_khz = %u, volt = %u",
			__func__, i, freqs[i].gpufreq_idx, freqs[i].gpufreq_khz,
			freqs[i].gpufreq_volt);
	}

	g_opp_idx_num = num;
	g_max_limited_idx = 0;

	return;
}
/* power calculation for power table */
static void __mt_gpufreq_calculate_power(unsigned int idx,
	unsigned int freq, unsigned int volt, unsigned int temp)
{
	unsigned int p_total = 0;
	unsigned int p_dynamic = 0;
	unsigned int ref_freq = 0;
	unsigned int ref_volt = 0;
	int p_leakage = 0;

	p_dynamic = GPU_ACT_REF_POWER;
	ref_freq = GPU_ACT_REF_FREQ;
	ref_volt = GPU_ACT_REF_VOLT;

	p_dynamic = p_dynamic *
			((freq * 100) / ref_freq) *
			((volt * 100) / ref_volt) *
			((volt * 100) / ref_volt) / (100 * 100 * 100);

#ifdef CONFIG_MTK_STATIC_POWER
	/*
	p_leakage = mt_spower_get_leakage(MTK_SPOWER_GPU, (volt / 100), temp);
	if (!g_volt_enable_state || p_leakage < 0)
		p_leakage = 0;
	*/
	p_leakage = 71;
#else
	p_leakage = 71;
#endif /* ifdef CONFIG_MTK_STATIC_POWER */

	p_total = p_dynamic + p_leakage;

	gpufreq_pr_debug(
	"@%s: idx = %d, p_dynamic = %d, p_leakage = %d, p_total = %d, temp = %d\n",
	__func__, idx, p_dynamic, p_leakage, p_total, temp);

	g_power_table[idx].gpufreq_power = p_total;
}
/*
 * OPP power table initialization
 */
static void __mt_gpufreq_setup_opp_power_table(int num)
{
	int i = 0;
	int temp = 0;
	struct thermal_zone_device *tzd;

	g_power_table = kzalloc((num) *
		sizeof(struct mt_gpufreq_power_table_info), GFP_KERNEL);

	if (g_power_table == NULL)
		return;

	/*
	 * Get temperature from GPU temperature sensor
	 * by using the linux kernel statndard API
	 *
	 * Location to look up termal_zone
	 * (6890.dtsi -> thermal_zone -> gpu0)
	 */
	tzd = thermal_zone_get_zone_by_name("gpu0");
	if (IS_ERR(tzd)) {
		temp = 40;
	} else {
		thermal_zone_get_temp(tzd, &temp);
		temp /= 1000;

		if ((temp < -20) || (temp > 125)) {
			gpufreq_pr_debug("@%s: abnormal temp %d!\n", __func__, temp);
			temp = 65;
		}
	}
	gpufreq_pr_debug("@%s: temp = %d\n", __func__, temp);

	for (i = 0; i < num; i++) {
		g_power_table[i].gpufreq_khz = g_opp_table[i].gpufreq_khz;
		g_power_table[i].gpufreq_volt = g_opp_table[i].gpufreq_volt;

		__mt_gpufreq_calculate_power(i, g_power_table[i].gpufreq_khz,
				g_power_table[i].gpufreq_volt, temp);

		gpufreq_pr_debug("@%s: [%d], freq_khz = %u, volt = %u, power = %u\n",
				__func__, i, g_power_table[i].gpufreq_khz,
				g_power_table[i].gpufreq_volt,
				g_power_table[i].gpufreq_power);
	}

	return;
}
/*
 * gpufreq driver probe
 */
static int __mt_gpufreq_pdrv_probe(struct platform_device *pdev)
{
	struct device_node *apmixed_node;
	struct device_node *mfg_node;
	struct device_node *node;
	int i;

	g_volt_enable_state = false;
	g_fixed_freq_volt_state = false;
	g_opp_stress_test_state = false;

	GPUFREQ_UNREFERENCED(i);

	node = of_find_matching_node(NULL, g_gpufreq_of_match);
	if (!node)
		gpufreq_perr("@%s: find GPU node failed\n", __func__);

	g_clk = kzalloc(sizeof(struct g_clk_info), GFP_KERNEL);
	if (g_clk == NULL)
		return -ENOMEM;

	g_clk->clk_mux = devm_clk_get(&pdev->dev, "clk_mux");
	if (IS_ERR(g_clk->clk_mux)) {
		gpufreq_perr("@%s: cannot get clk_mux\n", __func__);
		return PTR_ERR(g_clk->clk_mux);
	}

	g_clk->clk_main_parent = devm_clk_get(&pdev->dev, "clk_main_parent");
	if (IS_ERR(g_clk->clk_main_parent)) {
		gpufreq_perr("@%s: cannot get clk_main_parent\n", __func__);
		return PTR_ERR(g_clk->clk_main_parent);
	}

	g_clk->clk_sub_parent = devm_clk_get(&pdev->dev, "clk_sub_parent");
	if (IS_ERR(g_clk->clk_sub_parent)) {
		gpufreq_perr("@%s: cannot get clk_sub_parent\n", __func__);
		return PTR_ERR(g_clk->clk_sub_parent);
	}

	g_clk->cg_bg3d = devm_clk_get(&pdev->dev, "cg_bg3d");
	if (IS_ERR(g_clk->cg_bg3d)) {
		gpufreq_perr("@%s: cannot get cg_bg3d\n", __func__);
		return PTR_ERR(g_clk->cg_bg3d);
	}

	g_clk->mtcmos_mfg = &pdev->dev;
	if (IS_ERR(g_clk->mtcmos_mfg)) {
		gpufreq_perr("@%s: cannot get mtcmos_mfg\n", __func__);
		return PTR_ERR(g_clk->mtcmos_mfg);
	}
	pm_runtime_enable(g_clk->mtcmos_mfg);

	pr_info("[GPU/DVFS][INFO]@%s: clk_mux is at 0x%p, ",
		__func__, g_clk->clk_mux);
	pr_info("clk_sub_parent is at 0x%p, ", g_clk->clk_sub_parent);
	pr_info("clk_main_parent is at 0x%p, ", g_clk->clk_main_parent);
	pr_info("mtcmos_mfg is at 0x%p, ", g_clk->mtcmos_mfg);

	/* alloc PMIC regulator */
	g_pmic = kzalloc(sizeof(struct g_pmic_info), GFP_KERNEL);
	if (g_pmic == NULL)
		return -ENOMEM;

	g_pmic->reg_vcore = regulator_get(&pdev->dev, "dvfsrc-vcore");
	if (IS_ERR(g_pmic->reg_vcore)) {
		gpufreq_perr("@%s: cannot get VCORE\n", __func__);
		return PTR_ERR(g_pmic->reg_vcore);
	}

#ifdef CONFIG_MTK_STATIC_POWER
	/* Initial leackage power usage */
	/* mt_spower_init(); */
#endif /* CONFIG_MTK_STATIC_POWER */

	/* setup OPP table by device ID */
	__mt_gpufreq_setup_opp_table(g_opp_table_segment1,
		ARRAY_SIZE(g_opp_table_segment1));
	__mt_gpufreq_setup_opp_power_table(
		ARRAY_SIZE(g_opp_table_segment1));

	/* init APMIXED base address */
	apmixed_node =
		of_find_compatible_node(NULL, NULL, "mediatek,apmixed");
	g_apmixed_base = of_iomap(apmixed_node, 0);
	if (!g_apmixed_base) {
		gpufreq_perr("@%s: APMIXED iomap failed", __func__);
		return -ENOENT;
	}

	/* init g3d_config base address */
	mfg_node =
		of_find_compatible_node(NULL, NULL, "mediatek,g3d_config");
	g_mfg_base = of_iomap(mfg_node, 0);
	if (!g_mfg_base) {
		gpufreq_pr_info("@%s: ioremap failed at g3d_config",
			__func__);
		return -ENOENT;
	}

	mt_gpufreq_voltage_enable_set(1);

	/* setup initial opp */
	__mt_gpufreq_set_initial();

	mt_gpufreq_voltage_enable_set(0);
	return 0;
}

/*
 * register the gpufreq driver
 */
static int __init __mt_gpufreq_init(void)
{
	int ret = 0;

#ifdef BRING_UP
	/* Skip driver init in bring up stage */
	return 0;
#endif

	gpufreq_pr_debug("@%s: start to initialize gpufreq driver\n", __func__);

	/* register platform driver */
	ret = platform_driver_register(&g_gpufreq_pdrv);
	if (ret) {
		gpufreq_perr("@%s: fail to register gpufreq driver\n",
		__func__);
		goto out;
	}

#ifdef CONFIG_PROC_FS
	__mt_gpufreq_create_procfs();
#endif /* ifdef CONFIG_PROC_FS */

out:
	return ret;
}
/*
 * unregister the gpufreq driver
 */
static void __exit __mt_gpufreq_exit(void)
{
	platform_driver_unregister(&g_gpufreq_pdrv);
}

module_init(__mt_gpufreq_init);
module_exit(__mt_gpufreq_exit);

MODULE_DEVICE_TABLE(of, g_gpufreq_of_match);
MODULE_DESCRIPTION("MediaTek GPU-DVFS driver");
MODULE_LICENSE("GPL");