/* Copyright (c) 2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/of_gpio.h>
#include <linux/interrupt.h>
#include <linux/power_supply.h>

#include <linux/extcon-provider.h>
#include <linux/of_platform.h>
#include <linux/usb/role.h>

#define CONFIG_SUPPORT_NEC_CRADLE_DETECTION
#define CONFIG_SUPPORT_POWER_SUPPLY_IF
/* #define PIUSB_UPDATE_MAX_CURRENT_ENABLE */
#define PIUSB_UPDATE_ROLE_SYNC

#if 0  /* debug disable */
#undef dev_dbg
#define dev_dbg dev_info
#endif

#define PERICOM_I2C_NAME	"usb-type-c-pericom"
#define PERICOM_I2C_DELAY_MS	30

#define CCD_DEFAULT		0x1
#define CCD_MEDIUM		0x2
#define CCD_HIGH		0x3

#define MAX_CURRENT_MINIMUM	500
#define MAX_CURRENT_BC1P2	500
#define MAX_CURRENT_MEDIUM	1500
#define MAX_CURRENT_HIGH	3000

static bool disable_on_suspend;
module_param(disable_on_suspend , bool, S_IRUGO | S_IWUSR);
MODULE_PARM_DESC(disable_on_suspend,
	"Whether to disable chip on suspend if state is not attached");

#if defined(CONFIG_PLATFORM_NEC)
#define PIUSB_INIT_VALUE	0x4E

#define PIUSB_INT_INT_MASK	0x1

#define PIUSB_PS_MASK		( 0x80 | PIUSB_INT_INT_MASK)
#define PIUSB_PS_SHIFT		7
#define PIUSB_PS_ENABLE		0x1

#define PIUSB_PORTSETTING_MASK	(0x06 | PIUSB_INT_INT_MASK)
#define PIUSB_PORTSETTING_SHIFT	1
#define PIUSB_PORTSETTING_HOST	0x1
#define PIUSB_PORTSETTING_DR2	0x3
#endif /* defined(CONFIG_PLATFORM_NEC) */

u8 init_config[] = {0, PIUSB_INIT_VALUE};

struct piusb_regs {
	u8		dev_id;
	u8		control;
	u8		intr_status;
#define	INTS_ATTACH	0x1
#define	INTS_DETACH	0x2
#define INTS_ATTACH_MASK	0x3   /* current attach state interrupt */

	u8		port_status;
#define STS_PORT_MASK	(0x1c)	      /* attached port status  - device/host */
#define STS_CCD_MASK	(0x60)	      /* charging current status */
#define STS_VBUS_MASK	(0x80)	      /* vbus status */
#define STS_PORT_SHIFT	2
#define STS_PORT_STANDBY 0
#define STS_PORT_DEVICE	 1
#define STS_PORT_HOST	 2
#define STS_PORT_AUDIO	 3
#define STS_PORT_DBGACC	 4

} __packed;

struct pi_usb_type_c {
	struct device *dev;
	struct i2c_client	*client;
	struct piusb_regs	reg_data;
	struct power_supply	*charger_psy;
	struct notifier_block psy_nb;
	int					max_current;
	bool				attach_state;
	int					enb_gpio;
	int					enb_gpio_polarity;
	struct 	work_struct irq_work;

	struct extcon_dev *edev;
	struct device_connection dev_conn;
	struct usb_role_switch *role_sw;
	unsigned int		c_role;
	struct workqueue_struct *extcon_wq;
	bool binit;
#if defined(CONFIG_SUPPORT_POWER_SUPPLY_IF)
	struct power_supply *usb_type_c_psy;
	int otg_mode;
#define USB_TRANSFER_SPEED_HS	0	/* high-Speed */
#define USB_TRANSFER_SPEED_SS	1	/* SuperSpeed */
	int usb_transfer_speed;
#define AC_STOP_OFF	0	/* ac stop disable */
#define AC_STOP_ON	1	/* ac stop enable */
	int ac_stop;
#endif /* defined(CONFIG_SUPPORT_POWER_SUPPLY_IF) */
};

struct usb_role_info {
	struct pi_usb_type_c *pi_usb;
	struct delayed_work dwork;
	unsigned int d_role; /* desire data role */
};

enum {
	DUAL_PROP_MODE_UFP = 0,
	DUAL_PROP_MODE_DFP,
	DUAL_PROP_MODE_NONE,
};

enum {
	DUAL_PROP_PR_SRC = 0,
	DUAL_PROP_PR_SNK,
	DUAL_PROP_PR_NONE,
};

enum {
	DUAL_PROP_DR_HOST = 0,
	DUAL_PROP_DR_DEVICE,
	DUAL_PROP_DR_NONE,
};

static const unsigned int usb_extcon_cable[] = {
	EXTCON_USB,
	EXTCON_USB_HOST,
	EXTCON_NONE,
};

static int piusb_update_power_supply(struct pi_usb_type_c *pi_usb);

#if defined(CONFIG_PLATFORM_NEC)
static int piusb_set_powersaving(struct pi_usb_type_c *pi, bool enable);
static int piusb_set_portsetting(struct pi_usb_type_c *pi, bool ac_stop);

#define POWER_SUPPLY_RT9467_CHARGER_NAME	"rt9467-charger"
#define POWER_SUPPLY_BQ2589X_CHARGER_NAME	"bq2589x-charger"
//#include <linux/gpio.h>
extern int get_PKGID(void);

int get_charger_power_supply_handle(struct pi_usb_type_c *pi_usb)
{
	int pkgid;

	if (pi_usb->charger_psy)
		return 1;

	pkgid = get_PKGID();

	if (pkgid == 0) {
		pi_usb->charger_psy = power_supply_get_by_name(POWER_SUPPLY_RT9467_CHARGER_NAME);
	} else {
		pi_usb->charger_psy = power_supply_get_by_name(POWER_SUPPLY_BQ2589X_CHARGER_NAME);
	}

	if (pi_usb->charger_psy) {
		dev_info(pi_usb->dev, "%s: USB power_supply %s detected\n", __func__, pi_usb->charger_psy->desc->name);
	} else {
		return 0;
	}

	return 1;
}

#endif /* defined(CONFIG_PLATFORM_NEC) */

static void _piusb_extcon_update_role(struct pi_usb_type_c *pi_usb,unsigned int new_dr)
{
	unsigned int cur_dr;
	cur_dr = pi_usb->c_role;

	if (pi_usb->otg_mode == 2) {
		/* disable DUAL_PROP_DR_DEVICE */
		if (new_dr == DUAL_PROP_DR_DEVICE) {
			dev_dbg(pi_usb->dev, "%s disable DUAL_PROP_DR_DEVICE change new_dr\n",__func__);
			new_dr = DUAL_PROP_DR_NONE;
		}
	}

	dev_info(pi_usb->dev, "%s: cur_dr(%d) new_dr(%d)\n", __func__, cur_dr, new_dr);
	if (cur_dr == new_dr) {
		dev_info(pi_usb->dev, "%s same dr return\n",__func__);
		return;
	}

	/* none -> device */
	if (cur_dr == DUAL_PROP_DR_NONE &&
			new_dr == DUAL_PROP_DR_DEVICE) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB, true);
	/* none -> host */
	} else if (cur_dr == DUAL_PROP_DR_NONE &&
			new_dr == DUAL_PROP_DR_HOST) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB_HOST, true);
	/* device -> none */
	} else if (cur_dr == DUAL_PROP_DR_DEVICE &&
			new_dr == DUAL_PROP_DR_NONE) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB, false);
	/* host -> none */
	} else if (cur_dr == DUAL_PROP_DR_HOST &&
			new_dr == DUAL_PROP_DR_NONE) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB_HOST, false);
	/* device -> host */
	} else if (cur_dr == DUAL_PROP_DR_DEVICE &&
			new_dr == DUAL_PROP_DR_HOST) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB, false);
		extcon_set_state_sync(pi_usb->edev,	EXTCON_USB_HOST, true);
	/* host -> device */
	} else if (cur_dr == DUAL_PROP_DR_HOST &&
			new_dr == DUAL_PROP_DR_DEVICE) {
		extcon_set_state_sync(pi_usb->edev, EXTCON_USB_HOST, false);
		extcon_set_state_sync(pi_usb->edev,	EXTCON_USB, true);
	}

	/* usb role switch */
	if (pi_usb->role_sw) {
		dev_info(pi_usb->dev, "@@@ %s: role switch cur_dr(%d) new_dr(%d)\n", __func__, cur_dr, new_dr);
		if (new_dr == DUAL_PROP_DR_DEVICE) {
			if (pi_usb->usb_transfer_speed == USB_TRANSFER_SPEED_HS)
				piusb_set_powersaving(pi_usb, true);
			usb_role_switch_set_role(pi_usb->role_sw, USB_ROLE_DEVICE);
		}
		else if (new_dr == DUAL_PROP_DR_HOST)
			usb_role_switch_set_role(pi_usb->role_sw, USB_ROLE_HOST);
		else {
			usb_role_switch_set_role(pi_usb->role_sw, USB_ROLE_NONE);
			piusb_set_powersaving(pi_usb, false);
		}
	} else {
		dev_info(pi_usb->dev, "role_sw is none\n");
	}

	pi_usb->c_role = new_dr;

	piusb_update_power_supply(pi_usb);

	return;
}

#ifndef PIUSB_UPDATE_ROLE_SYNC
static void piusb_extcon_update_role(struct work_struct *work)
{
	struct usb_role_info *role = container_of(to_delayed_work(work),
					struct usb_role_info, dwork);
	struct pi_usb_type_c *pi_usb = role->pi_usb;

	_piusb_extcon_update_role(pi_usb,role->d_role);

	kfree(role);
}
#endif /* PIUSB_UPDATE_ROLE_SYNC */

static int piusb_extcon_set_role(struct pi_usb_type_c *pi_usb,
						unsigned int role)
{
#ifdef PIUSB_UPDATE_ROLE_SYNC
	_piusb_extcon_update_role(pi_usb,role);
#else /* PIUSB_UPDATE_ROLE_SYNC */
	struct usb_role_info *role_info;

	/* create and prepare worker */
	role_info = kzalloc(sizeof(*role_info), GFP_KERNEL);
	if (!role_info)
		return -ENOMEM;

	INIT_DELAYED_WORK(&role_info->dwork, piusb_extcon_update_role);

	role_info->pi_usb = pi_usb;
	role_info->d_role = role;
	/* issue connection work */
	queue_delayed_work(pi_usb->extcon_wq, &role_info->dwork, 0);
#endif /* PIUSB_UPDATE_ROLE_SYNC */

	return 0;
}

static int piusb_judge_and_set_role(struct pi_usb_type_c *pi, bool online, int usb_type)
{
	int attach_status;

	attach_status = (pi->reg_data.port_status & STS_PORT_MASK) >> STS_PORT_SHIFT;
	dev_info(pi->dev, "@@@ %s onine=%d attach_status=%d usb_type=0x%x\n",__FUNCTION__,online,attach_status,usb_type);
	if(online) {
		if (attach_status == STS_PORT_HOST) {
			if(usb_type == POWER_SUPPLY_USB_TYPE_SDP
			 || usb_type == POWER_SUPPLY_USB_TYPE_CDP) {
				piusb_extcon_set_role(pi, DUAL_PROP_DR_DEVICE);
			 }else{
			 	piusb_extcon_set_role(pi, DUAL_PROP_DR_NONE);
			 }
		} else if (attach_status == STS_PORT_DEVICE) {
			piusb_extcon_set_role(pi, DUAL_PROP_DR_HOST);
#ifdef CONFIG_SUPPORT_NEC_CRADLE_DETECTION
		/* NEC U2E Cradle. Data: Device-role, Power: Source-role */
		} else if (attach_status == STS_PORT_AUDIO) {
			piusb_extcon_set_role(pi, DUAL_PROP_DR_HOST);
#endif
		} else {	//STANDBY/AUDIO/DEBUGACC
			piusb_extcon_set_role(pi, DUAL_PROP_DR_NONE);
		}
	} else {
		piusb_extcon_set_role(pi, DUAL_PROP_DR_NONE);
	}

	return 0;
}

static int piusb_charger_is_online(struct pi_usb_type_c *pi, bool *online)
{
	union power_supply_propval pval;
	int ret;

#if defined(CONFIG_PLATFORM_NEC)
	ret = get_charger_power_supply_handle(pi);
	if (ret == 0)
		return -1;
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (pi->charger_psy == NULL) {
		pi->charger_psy = devm_power_supply_get_by_phandle(pi->dev, "charger");
		if (pi->charger_psy) {
			dev_info(pi->dev, "%s: USB power_supply %s detected\n", __func__, pi->charger_psy->desc->name);
		} else {
			return -1;
		}
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

	ret = power_supply_get_property(pi->charger_psy, POWER_SUPPLY_PROP_ONLINE, &pval);
	if (ret < 0) {
		return ret;
	}
	*online = pval.intval;

	return 0;
}

static int piusb_get_charger_usb_type(struct pi_usb_type_c *pi, int *type)
{
	union power_supply_propval pval;
	int ret;

#if defined(CONFIG_PLATFORM_NEC)
	ret = get_charger_power_supply_handle(pi);
	if (ret == 0)
		return -1;
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (pi->charger_psy == NULL) {
		pi->charger_psy = devm_power_supply_get_by_phandle(pi->dev, "charger");
		if (pi->charger_psy) {
			dev_info(pi->dev, "%s: USB power_supply %s detected\n", __func__, pi->charger_psy->desc->name);
		} else {
			return -1;
		}
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

	ret = power_supply_get_property(pi->charger_psy, POWER_SUPPLY_PROP_USB_TYPE, &pval);
	if (ret < 0) {
		return ret;
	}
	*type = pval.intval;

	return 0;
}

static int piusb_charger_psy_notifier(struct notifier_block *nb,
				unsigned long event, void *data)
{
	struct power_supply *psy = data;
	struct pi_usb_type_c *pi = container_of(nb, struct pi_usb_type_c, psy_nb);
	int ret;
	bool online;
	int usb_type;

	dev_dbg(pi->dev, "%s\n",__func__);
#if defined(CONFIG_PLATFORM_NEC)
	ret = get_charger_power_supply_handle(pi);
	if (ret == 0)
		return -1;
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (pi->charger_psy == NULL) {
		pi->charger_psy = devm_power_supply_get_by_phandle(pi->dev, "charger");
		if (pi->charger_psy) {
			dev_info(pi->dev, "%s: USB power_supply %s detected\n", __func__, pi->charger_psy->desc->name);
		} else {
			return -1;
		}
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

	if (event != PSY_EVENT_PROP_CHANGED || (psy != pi->charger_psy && psy != pi->usb_type_c_psy))
		return NOTIFY_DONE;

	ret = piusb_charger_is_online(pi, &online);
	if (ret < 0) {
		dev_info(pi->dev, "failed to get online\n");
		return NOTIFY_DONE;
	}
	ret = piusb_get_charger_usb_type(pi, &usb_type);
	if (ret < 0) {
		dev_info(pi->dev, "failed to get usb_type\n");
		return NOTIFY_DONE;
	}

	piusb_judge_and_set_role(pi, online, usb_type);

	return NOTIFY_DONE;
}

static int piusb_read_regdata(struct pi_usb_type_c *pi_usb)
{
	struct i2c_client *i2c = pi_usb->client;
	int rc;
	int data_length = sizeof(pi_usb->reg_data);
	uint16_t saddr = i2c->addr;
	struct i2c_msg msgs[] = {
		{
			.addr  = saddr,
			.flags = I2C_M_RD,
			.len   = data_length,
			.buf   = (u8 *)&pi_usb->reg_data,
		}
	};

	rc = i2c_transfer(i2c->adapter, msgs, 1);
	if (rc < 0) {
		/* i2c read may fail if device not enabled or not present */
		dev_dbg(&i2c->dev, "i2c read from 0x%x failed %d\n", saddr, rc);
		return -ENXIO;
	}

	dev_dbg(&i2c->dev, "i2c read from 0x%x-[%.2x %.2x %.2x %.2x]\n", saddr,
		    pi_usb->reg_data.dev_id, pi_usb->reg_data.control,
		    pi_usb->reg_data.intr_status, pi_usb->reg_data.port_status);

	return rc;
}

static int piusb_update_power_supply(struct pi_usb_type_c *pi_usb)
{
	union power_supply_propval val = {0,};
#if defined(PIUSB_UPDATE_MAX_CURRENT_ENABLE)
	val.intval = pi_usb->max_current * 1000;
#if defined(CONFIG_PLATFORM_NEC)
	if (get_charger_power_supply_handle(pi_usb) == 0)
		return -1;
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (pi_usb->charger_psy == NULL) {
		pi_usb->charger_psy = devm_power_supply_get_by_phandle(pi_usb->dev, "charger");
		if (pi_usb->charger_psy) {
			dev_info(pi_usb->dev, "%s: USB power_supply %s detected\n", __func__, pi_usb->charger_psy->desc->name);
		} else {
			return -1;
		}
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

	if (pi_usb->charger_psy)
		if (pi_usb->charger_psy->desc->set_property)
			return pi_usb->charger_psy->desc->set_property(pi_usb->charger_psy,
				POWER_SUPPLY_PROP_INPUT_CURRENT_LIMIT, &val);

	return -ENODEV;
#else /* defined(PIUSB_UPDATE_MAX_CURRENT_ENABLE) */
#if defined(CONFIG_PLATFORM_NEC)
	if (get_charger_power_supply_handle(pi_usb) == 0)
		return -1;
#else /* defined(CONFIG_PLATFORM_NEC) */
	if (pi_usb->charger_psy == NULL) {
		pi_usb->charger_psy = devm_power_supply_get_by_phandle(pi_usb->dev, "charger");
		if (pi_usb->charger_psy) {
			dev_info(pi_usb->dev, "%s: USB power_supply %s detected\n", __func__, pi_usb->charger_psy->desc->name);
		} else {
			return -1;
		}
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */
	val.intval = 1;
	if (pi_usb->charger_psy) {
		if (pi_usb->charger_psy->desc->set_property) {
			if(pi_usb->reg_data.port_status) {
				return pi_usb->charger_psy->desc->set_property(pi_usb->charger_psy,
					POWER_SUPPLY_PROP_NOTIFY_SRC, &val);
			}
		}
	}
	return 0;
#endif /* defined(PIUSB_UPDATE_MAX_CURRENT_ENABLE) */
}

static void piusb_update_max_current(struct pi_usb_type_c *pi_usb)
{
	u8 mask = STS_CCD_MASK;
	u8 shift = find_first_bit((void *)&mask, 8);
	u8 chg_mode = pi_usb->reg_data.port_status & mask;

	chg_mode >>= shift;

	/* if type-c detached */
	if (!pi_usb->attach_state) {
		pi_usb->max_current = MAX_CURRENT_MINIMUM;
		return;
	}

	switch (chg_mode) {
	case CCD_DEFAULT:
		pi_usb->max_current = MAX_CURRENT_BC1P2;
		break;
	case CCD_MEDIUM:
		pi_usb->max_current = MAX_CURRENT_MEDIUM;
		break;
	case CCD_HIGH:
		pi_usb->max_current = MAX_CURRENT_HIGH;
		break;
	default:
		dev_dbg(&pi_usb->client->dev, "wrong chg mode %x\n", chg_mode);
		pi_usb->max_current = MAX_CURRENT_BC1P2;
	}

	dev_dbg(&pi_usb->client->dev, "chg mode: %x, mA:%u\n", chg_mode,
							pi_usb->max_current);
}

static void piusb_irq_workfunc(struct work_struct *work)
{
	struct pi_usb_type_c *pi_usb = container_of(work, struct pi_usb_type_c, irq_work);
	int ret;
	bool online;
	int usb_type;
	u8 attach_state;

	/* i2c register update takes time, 30msec sleep required as per HPG */
	msleep(PERICOM_I2C_DELAY_MS);

	ret = piusb_read_regdata(pi_usb);
	if (ret < 0) {
		pi_usb->attach_state = false;
		return;
	}
	dev_info(pi_usb->dev, "%s: reg[1-4] = [%.2x %.2x %.2x %.2x]\n", __func__,
	    pi_usb->reg_data.dev_id, pi_usb->reg_data.control,
	    pi_usb->reg_data.intr_status, pi_usb->reg_data.port_status);

	if (pi_usb->binit) {
		if(pi_usb->reg_data.port_status) {
			piusb_charger_is_online(pi_usb, &online);
			piusb_get_charger_usb_type(pi_usb, &usb_type);
			piusb_judge_and_set_role(pi_usb, online, usb_type);
			pi_usb->binit = false;
		}
	}

	if (!pi_usb->reg_data.intr_status) {
		dev_err(pi_usb->dev, "intr_status is 0!, ignore interrupt\n");
		return;
	}

	attach_state = pi_usb->reg_data.intr_status & INTS_ATTACH_MASK;
	pi_usb->attach_state = (attach_state == INTS_ATTACH) ? true : false;
	piusb_update_max_current(pi_usb);

	ret = piusb_update_power_supply(pi_usb);
	if (ret < 0)
		dev_err(&pi_usb->client->dev, "failed to notify USB-%d\n", ret);
}

static irqreturn_t piusb_irq(int irq, void *data)
{
	struct pi_usb_type_c *pi_usb = data;

	schedule_work(&pi_usb->irq_work);
	return IRQ_HANDLED;
}

static int piusb_i2c_write(struct pi_usb_type_c *pi, u8 *data, int len)
{
	int ret;
	struct i2c_msg msgs[] = {
		{
			.addr  = pi->client->addr,
			.flags = 0,
			.len   = len,
			.buf   = data,
		}
	};

	ret = i2c_transfer(pi->client->adapter, msgs, 1);
	if (ret != 1) {
		dev_err(&pi->client->dev, "i2c write to [%x] failed %d\n",
				pi->client->addr, ret);
		return -EIO;
	}
	return 0;
}

static int piusb_i2c_enable(struct pi_usb_type_c *pi, bool enable)
{
	u8 rst_assert[] = {0, 0x1};		//Interrupt-Mask
	u8 pi_disable[] = {0, 0x80};	//Powersaving

	if (!enable) {
		if (piusb_i2c_write(pi, pi_disable, sizeof(pi_disable)))
			return -EIO;
		return 0;
	}

	if (piusb_i2c_write(pi, rst_assert, sizeof(rst_assert)))
		return -EIO;

	msleep(PERICOM_I2C_DELAY_MS);
	if (piusb_i2c_write(pi, init_config, sizeof(init_config)))
		return -EIO;

	return 0;
}

#if defined(CONFIG_PLATFORM_NEC)
static int piusb_set_powersaving(struct pi_usb_type_c *pi, bool enable)
{
	u8 pi_data[] = {0, PIUSB_INIT_VALUE};
	int i;

	dev_info(pi->dev, "@@@ %s enable=%d\n",__FUNCTION__,enable);
	pi_data[1] = pi->reg_data.control & ~PIUSB_PS_MASK;

	if (enable) {
		/* Powersaving Enable = Disable and low power state */
		pi_data[1] = pi_data[1] | (PIUSB_PS_ENABLE << PIUSB_PS_SHIFT);
	}

	disable_irq(pi->client->irq);
	if (piusb_i2c_write(pi, pi_data, sizeof(pi_data)))
		return -EIO;

	for (i=0; i< 10; i++) {
		msleep(PERICOM_I2C_DELAY_MS);
		piusb_read_regdata(pi);
		dev_dbg(pi->dev, "%s: reg[1-4] = [%.2x %.2x %.2x %.2x]\n", __func__,
			    pi->reg_data.dev_id, pi->reg_data.control,
			    pi->reg_data.intr_status, pi->reg_data.port_status);
		if (pi->reg_data.intr_status & INTS_ATTACH_MASK)
			break;
	}
	enable_irq(pi->client->irq);

	dev_info(pi->dev, "%s: reg[1-4] = [%.2x %.2x %.2x %.2x]\n", __func__,
		pi->reg_data.dev_id, pi->reg_data.control,
		pi->reg_data.intr_status, pi->reg_data.port_status);

	return 0;
}

static int piusb_set_portsetting(struct pi_usb_type_c *pi, bool ac_stop)
{
	u8 pi_data[] = {0, PIUSB_INIT_VALUE};

	dev_info(pi->dev, "@@@ %s ac_stop=%d\n",__FUNCTION__,ac_stop);
	pi_data[1] = pi->reg_data.control & ~PIUSB_PORTSETTING_MASK;
	if (ac_stop) {
		/* portsetting host bit[2:1]=01 */
		pi_data[1] = pi_data[1] | (PIUSB_PORTSETTING_HOST << PIUSB_PORTSETTING_SHIFT);
	} else {
		/* portsetting DualRole2 bit[2:1]=11 */
		pi_data[1] = pi_data[1] | (PIUSB_PORTSETTING_DR2 << PIUSB_PORTSETTING_SHIFT);
	}

	if (piusb_i2c_write(pi, pi_data, sizeof(pi_data)))
		return -EIO;

	pi->ac_stop = ac_stop;
	return 0;
}
#endif /* defined(CONFIG_PLATFORM_NEC) */

static int piusb_set_gpio_enable(struct pi_usb_type_c *pi, bool enable)
{
	if (!enable) {
		gpio_set_value(pi->enb_gpio, !pi->enb_gpio_polarity);
	} else {
		gpio_set_value(pi->enb_gpio, pi->enb_gpio_polarity);
		msleep(PERICOM_I2C_DELAY_MS);
	}

	return 0;
}

static int piusb_gpio_config(struct pi_usb_type_c *pi, bool enable)
{
	int ret = 0;

	if (!enable) {
		piusb_set_gpio_enable(pi, false);
		return 0;
	}

	ret = devm_gpio_request(&pi->client->dev, pi->enb_gpio,
					"pi_typec_enb_gpio");
	if (ret) {
		pr_err("%s:unable to request gpio [%d]\n", __func__, pi->enb_gpio);
		return ret;
	}

	ret = gpio_direction_output(pi->enb_gpio, pi->enb_gpio_polarity);
	if (ret) {
		dev_err(&pi->client->dev, "%s:set dir[%d] failed for gpio[%d]\n", __func__,
			pi->enb_gpio_polarity, pi->enb_gpio);
		return ret;
	}
	dev_dbg(&pi->client->dev, "%s:set dir[%d] for gpio[%d]\n", __func__,
			pi->enb_gpio_polarity, pi->enb_gpio);

	piusb_set_gpio_enable(pi, true);

	return ret;
}

static ssize_t piusb_show_registers(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;
	struct pi_usb_type_c *pi_usb = dev_get_drvdata(dev);
	u8 tmpbuf[300];
	int len;

	ret = piusb_read_regdata(pi_usb);
	if (ret < 0)
		return ret;

	len = snprintf(tmpbuf, PAGE_SIZE, "%s: reg[1-4] = [%.2x %.2x %.2x %.2x]\n", PERICOM_I2C_NAME,
	    pi_usb->reg_data.dev_id, pi_usb->reg_data.control,
	    pi_usb->reg_data.intr_status, pi_usb->reg_data.port_status);
	memcpy(&buf[0], tmpbuf, len);

	return len;
}

static ssize_t piusb_get_enable(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;
	struct pi_usb_type_c *pi = dev_get_drvdata(dev);

	ret = gpio_get_value(pi->enb_gpio);

	return sprintf(buf, "%d\n", pi->enb_gpio_polarity ? ret : !ret);
}

static ssize_t piusb_set_enable(struct device *dev,
				struct device_attribute *attr, const char *buf,
				size_t count)
{
	int ret;
	struct pi_usb_type_c *pi = dev_get_drvdata(dev);
	long val;

	if (kstrtol(buf, 10, &val) < 0)
		return -EINVAL;

	ret = piusb_set_gpio_enable(pi, val);
	return count;
}

static DEVICE_ATTR(registers, S_IRUGO, piusb_show_registers, NULL);
static DEVICE_ATTR(enable, S_IRUGO|S_IWUSR, piusb_get_enable, piusb_set_enable);

static struct attribute *piusb_attributes[] = {
	&dev_attr_registers.attr,
	&dev_attr_enable.attr,
	NULL,
};

static const struct attribute_group piusb_attr_group = {
	.attrs = piusb_attributes,
};

#if defined(CONFIG_SUPPORT_POWER_SUPPLY_IF)

static int piusb_set_prop_usb_otg(struct pi_usb_type_c *pi_usb,int d_otg)
{
	bool power_supply_change_req = false;

	switch (d_otg) {
	case 1:
		if (pi_usb->c_role == DUAL_PROP_DR_NONE) {
			dev_dbg(pi_usb->dev, "%s re detect req in\n",__func__);
			power_supply_change_req = true;
		}
		break;
	case 2:
		if (pi_usb->c_role == DUAL_PROP_DR_DEVICE) {
			dev_dbg(pi_usb->dev, "%s disconnect req in\n",__func__);
			power_supply_change_req = true;
		}
		break;
	default:
		dev_err(&pi_usb->client->dev, "%s d_otg=%d errr\n",__func__,d_otg);
		return pi_usb->otg_mode;
	}

	if (power_supply_change_req == true) {
		dev_dbg(pi_usb->dev, "%s power_supply_changed call\n",__func__);
		pi_usb->otg_mode = d_otg;
		power_supply_changed(pi_usb->usb_type_c_psy);
	}
	return d_otg;
}

static enum power_supply_property piusb_power_supply_props[] = {
	POWER_SUPPLY_PROP_CRADLE_STATUS,
	POWER_SUPPLY_PROP_USB_OTG,
	POWER_SUPPLY_PROP_USB_TRANSFER_SPEED,
	POWER_SUPPLY_PROP_AC_STOP,
	POWER_SUPPLY_PROP_NOTIFY_SRC,
};

static int piusb_power_supply_get_property(struct power_supply *psy,
						enum power_supply_property psp,
						union power_supply_propval *val)
{
	struct pi_usb_type_c *pi_usb = power_supply_get_drvdata(psy);
	int attach_status;

	attach_status = (pi_usb->reg_data.port_status & STS_PORT_MASK) >> STS_PORT_SHIFT;

	switch (psp) {
	case POWER_SUPPLY_PROP_CRADLE_STATUS:
		/* type-C reg0x04 bit[4-2]=001(STS_PORT_DEVICE) or 011(STS_PORT_AUDIO) */
		val->intval = 0;
		if ((attach_status == STS_PORT_DEVICE) || (attach_status == STS_PORT_AUDIO)) {
			if (pi_usb->c_role != DUAL_PROP_MODE_NONE) {
				val->intval = 1;
			}
		}
		break;
	case POWER_SUPPLY_PROP_USB_OTG:
		val->intval = pi_usb->otg_mode;
		break;
	case POWER_SUPPLY_PROP_USB_TRANSFER_SPEED:
		dev_info(pi_usb->dev, "@@@ %s POWER_SUPPLY_PROP_USB_TRANSFER_SPEED\n",__func__);
		val->intval = pi_usb->usb_transfer_speed;
		break;
	case POWER_SUPPLY_PROP_AC_STOP:
		dev_info(pi_usb->dev, "@@@ %s POWER_SUPPLY_PROP_AC_STOP\n",__func__);
		val->intval = pi_usb->ac_stop;
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static int piusb_power_supply_set_property(struct power_supply *psy,
					enum power_supply_property psp,
		const union power_supply_propval *val)
{
	struct pi_usb_type_c *pi_usb = power_supply_get_drvdata(psy);
	int ret = 0;

	switch (psp) {
	case POWER_SUPPLY_PROP_USB_OTG:
		pi_usb->otg_mode = piusb_set_prop_usb_otg(pi_usb,val->intval);
		break;
	case POWER_SUPPLY_PROP_USB_TRANSFER_SPEED:
		dev_info(pi_usb->dev, "@@@ %s POWER_SUPPLY_PROP_USB_TRANSFER_SPEED\n",__func__);
		if ((val->intval > USB_TRANSFER_SPEED_SS) || (val->intval < USB_TRANSFER_SPEED_HS)) {
			dev_err(pi_usb->dev, "@@@ POWER_SUPPLY_PROP_USB_TRANSFER_SPEED param %d err\n",val->intval);
			break;
		}
		pi_usb->usb_transfer_speed = val->intval;
		if (pi_usb->c_role == DUAL_PROP_DR_DEVICE) {
			extcon_set_state_sync(pi_usb->edev, EXTCON_USB, false);
			if (pi_usb->role_sw) {
				usb_role_switch_set_role(pi_usb->role_sw, USB_ROLE_NONE);
			}
			piusb_set_powersaving(pi_usb, false);
			pi_usb->c_role = DUAL_PROP_DR_NONE;
			dev_info(pi_usb->dev, "@@@ line=%d Discon\n",__LINE__);

			piusb_extcon_set_role(pi_usb, DUAL_PROP_DR_DEVICE);
			dev_info(pi_usb->dev, "@@@ line=%d Re connect\n",__LINE__);
		}
		break;
	case POWER_SUPPLY_PROP_AC_STOP:
		if ((val->intval != AC_STOP_OFF) && (val->intval != AC_STOP_ON)) {
			dev_err(pi_usb->dev, "@@@ POWER_SUPPLY_PROP_AC_STOP param %d err\n",val->intval);
			break;
		}
		if (val->intval == pi_usb->ac_stop) {
			dev_dbg(pi_usb->dev, "@@@ POWER_SUPPLY_PROP_AC_STOP same req %d\n",val->intval);
			break;
		}
		piusb_set_portsetting(pi_usb,val->intval);
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		{
			bool online;
			int usb_type;

			piusb_read_regdata(pi_usb);
			dev_info(pi_usb->dev, "%s: reg[1-4] = [%.2x %.2x %.2x %.2x]\n", __func__,
			    pi_usb->reg_data.dev_id, pi_usb->reg_data.control,
			    pi_usb->reg_data.intr_status, pi_usb->reg_data.port_status);
			piusb_charger_is_online(pi_usb, &online);
			piusb_get_charger_usb_type(pi_usb, &usb_type);
			piusb_judge_and_set_role(pi_usb, online, usb_type);
		}
		break;
	default:
		return -EINVAL;
	}
	return ret;
}

static int piusb_power_supply_is_writeable(struct power_supply *psy,
				       enum power_supply_property prop)
{
	int rc = 0;

	switch (prop) {
	case POWER_SUPPLY_PROP_USB_OTG:
	case POWER_SUPPLY_PROP_USB_TRANSFER_SPEED:
	case POWER_SUPPLY_PROP_AC_STOP:
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		rc = 1;
		break;
	default:
		rc = 0;
		break;
	}
	return rc;
}

static const struct power_supply_desc piusb_power_supply_desc = {
	.name = "type-c-usb",
	.type = POWER_SUPPLY_TYPE_USB,
	.properties = piusb_power_supply_props,
	.num_properties = ARRAY_SIZE(piusb_power_supply_props),
	.get_property = piusb_power_supply_get_property,
	.set_property = piusb_power_supply_set_property,
	.property_is_writeable = piusb_power_supply_is_writeable,
};

static int piusb_power_supply_init(struct pi_usb_type_c *pi_usb)
{
	struct power_supply_config psy_cfg = { .drv_data = pi_usb, };

	pi_usb->usb_type_c_psy = power_supply_register(pi_usb->dev, &piusb_power_supply_desc, &psy_cfg);

	return PTR_ERR_OR_ZERO(pi_usb->usb_type_c_psy);
}
#endif /* defined(CONFIG_SUPPORT_POWER_SUPPLY_IF) */

static int piusb_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret;
	struct pi_usb_type_c *pi_usb;
#if !defined(CONFIG_PLATFORM_NEC)
	struct power_supply *charger_psy;
#endif /* !defined(CONFIG_PLATFORM_NEC) */
	struct device_node *np = client->dev.of_node;
	enum of_gpio_flags flags;
	struct platform_device *conn_pdev;
	struct device_node *conn_np;

#if !defined(CONFIG_PLATFORM_NEC)
	charger_psy = devm_power_supply_get_by_phandle(&client->dev, "charger");
	if (charger_psy) {
		dev_info(&client->dev, "%s: USB power_supply %s detected\n", __func__, charger_psy->desc->name);
	} else {
		dev_info(&client->dev, "%s: USB power_supply not found, defer probe\n", __func__);
		return -EPROBE_DEFER;
	}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

	pi_usb = devm_kzalloc(&client->dev, sizeof(struct pi_usb_type_c),
				GFP_KERNEL);
	if (!pi_usb) {
		dev_err(&client->dev, "%s: out of memory\n", __func__);
		return -ENOMEM;
	}

	pi_usb->dev = &client->dev;

	pi_usb->edev = devm_extcon_dev_allocate(pi_usb->dev, usb_extcon_cable);
	if (IS_ERR(pi_usb->edev)) {
		dev_err(pi_usb->dev, "failed to allocate extcon device\n");
		return -ENOMEM;
	}

	ret = devm_extcon_dev_register(pi_usb->dev, pi_usb->edev);
	if (ret < 0) {
		dev_err(pi_usb->dev, "failed to register extcon device\n");
		return ret;
	}

	/* usb role switch */
	conn_np = of_parse_phandle(np, "dev-conn", 0);
	if (!conn_np) {
		dev_info(pi_usb->dev, "failed to get dev-conn node\n");
		return -EINVAL;
	}

	conn_pdev = of_find_device_by_node(conn_np);
	if (!conn_pdev) {
		dev_info(pi_usb->dev, "failed to get dev-conn pdev\n");
		return -EINVAL;
	}

	pi_usb->psy_nb.notifier_call = piusb_charger_psy_notifier;
	ret = power_supply_reg_notifier(&pi_usb->psy_nb);
	if (ret) {
		dev_err(pi_usb->dev, "fail to register notifer\n");
		return -EINVAL;
	}

	pi_usb->dev_conn.endpoint[0] = kasprintf(GFP_KERNEL,
				"%s-role-switch", dev_name(&conn_pdev->dev));
	pi_usb->dev_conn.endpoint[1] = dev_name(pi_usb->dev);
	pi_usb->dev_conn.id = "usb-role-switch";
	device_connection_add(&pi_usb->dev_conn);

	i2c_set_clientdata(client, pi_usb);
	pi_usb->client = client;
#if defined(CONFIG_PLATFORM_NEC)
	get_charger_power_supply_handle(pi_usb);
#else /* defined(CONFIG_PLATFORM_NEC) */
	pi_usb->charger_psy = charger_psy;
#endif /* defined(CONFIG_PLATFORM_NEC) */

	if (client->irq < 0) {
		dev_err(pi_usb->dev, "%s: irq not defined (%d)\n", __func__, client->irq);
		return -EINVAL;
	}

	/* override with module-param */
	if (!disable_on_suspend)
		disable_on_suspend = of_property_read_bool(np,
						"pericom,disable-on-suspend");
	pi_usb->enb_gpio = of_get_named_gpio_flags(np, "pericom,enb-gpio", 0, &flags);
	if (!gpio_is_valid(pi_usb->enb_gpio)) {
		dev_dbg(pi_usb->dev, "%s:enb gpio_get fail:%d\n", __func__, pi_usb->enb_gpio);
	} else {
		pi_usb->enb_gpio_polarity = !(flags & OF_GPIO_ACTIVE_LOW);
		ret = piusb_gpio_config(pi_usb, true);
		if (ret) {
			dev_err(pi_usb->dev, "%s:gpio init failed\n", __func__);
			goto out;
		}
	}

	ret = piusb_i2c_enable(pi_usb, true);
	if (ret) {
		dev_err(pi_usb->dev, "%s: i2c access failed\n", __func__);
		ret = -EPROBE_DEFER;
		goto gpio_disable;
	}

	ret = sysfs_create_group(&pi_usb->dev->kobj, &piusb_attr_group);
	if (ret) {
		dev_err(pi_usb->dev, "failed to register sysfs. err: %d\n", ret);
		goto i2c_disable;
	}

	pi_usb->c_role = DUAL_PROP_DR_NONE;
	pi_usb->extcon_wq = create_singlethread_workqueue("extcon_usb");
	pi_usb->role_sw = usb_role_switch_get(pi_usb->dev);
	if (IS_ERR(pi_usb->role_sw)) {
		dev_err(pi_usb->dev, "failed to get usb role\n");
		return PTR_ERR(pi_usb->role_sw);
	}

	pi_usb->binit = true;
	INIT_WORK(&pi_usb->irq_work, piusb_irq_workfunc);
	/* Update initial state to USB */
	schedule_work(&pi_usb->irq_work);

	ret = devm_request_threaded_irq(&client->dev, client->irq, NULL, piusb_irq,
					IRQF_ONESHOT | IRQF_TRIGGER_FALLING,
					PERICOM_I2C_NAME, pi_usb);
	if (ret) {
		dev_err(&client->dev, "%s: irq(%d) req failed-%d\n", __func__, client->irq, ret);
		goto irq_disable;
	}

#if defined(CONFIG_SUPPORT_POWER_SUPPLY_IF)
	ret = piusb_power_supply_init(pi_usb);
	if (ret < 0) {
		dev_err(pi_usb->dev, "Failed to register power supply\n");
	}
	pi_usb->otg_mode = 1;
	pi_usb->usb_transfer_speed = USB_TRANSFER_SPEED_SS;
	pi_usb->ac_stop = AC_STOP_OFF;
	pi_usb->reg_data.control = PIUSB_INIT_VALUE;
#endif /* defined(CONFIG_SUPPORT_POWER_SUPPLY_IF) */

	dev_dbg(&client->dev, "%s: finished, addr:%d\n", __func__, client->addr);

	return 0;

irq_disable:
	cancel_work_sync(&pi_usb->irq_work);
i2c_disable:
	piusb_i2c_enable(pi_usb, false);
gpio_disable:
	if (gpio_is_valid(pi_usb->enb_gpio))
		piusb_gpio_config(pi_usb, false);
out:
	return ret;
}

static int piusb_remove(struct i2c_client *client)
{
	struct pi_usb_type_c *pi_usb = i2c_get_clientdata(client);

	if (pi_usb->dev_conn.id)
		device_connection_remove(&pi_usb->dev_conn);

	cancel_work_sync(&pi_usb->irq_work);
	piusb_i2c_enable(pi_usb, false);
	if (gpio_is_valid(pi_usb->enb_gpio))
		piusb_gpio_config(pi_usb, false);
	devm_kfree(&client->dev, pi_usb);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int piusb_i2c_suspend(struct device *dev)
{
	struct i2c_client *i2c = to_i2c_client(dev);
	struct pi_usb_type_c *pi = i2c_get_clientdata(i2c);

	dev_dbg(dev, "pi_usb PM suspend.. attach(%d) disable(%d)\n",
			pi->attach_state, disable_on_suspend);
	disable_irq(pi->client->irq);
	/* Keep type-c chip enabled during session */
	if (pi->attach_state)
		return 0;

	if (disable_on_suspend)
		piusb_i2c_enable(pi, false);

	if (disable_on_suspend)
		gpio_set_value(pi->enb_gpio, !pi->enb_gpio_polarity);

	return 0;
}

static int piusb_i2c_resume(struct device *dev)
{
	int rc;
	struct i2c_client *i2c = to_i2c_client(dev);
	struct pi_usb_type_c *pi = i2c_get_clientdata(i2c);

	dev_dbg(dev, "pi_usb PM resume\n");
	/* suspend was no-op, just re-enable interrupt */
	if (pi->attach_state) {
		enable_irq(pi->client->irq);
		return 0;
	}

	if (disable_on_suspend) {
		gpio_set_value(pi->enb_gpio, pi->enb_gpio_polarity);
		msleep(PERICOM_I2C_DELAY_MS);
	}

	if (disable_on_suspend)
		rc = piusb_i2c_enable(pi, true);
	enable_irq(pi->client->irq);

	return rc;
}
#endif

static SIMPLE_DEV_PM_OPS(piusb_i2c_pm_ops, piusb_i2c_suspend,
			  piusb_i2c_resume);

static const struct i2c_device_id piusb_id[] = {
	{ PERICOM_I2C_NAME, 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, piusb_id);

#ifdef CONFIG_OF
static const struct of_device_id piusb_of_match[] = {
	{ .compatible = "pericom,usb-type-c", },
	{},
};
MODULE_DEVICE_TABLE(of, piusb_of_match);
#endif

static struct i2c_driver piusb_driver = {
	.driver = {
		.name = PERICOM_I2C_NAME,
		.of_match_table = of_match_ptr(piusb_of_match),
		.pm	= &piusb_i2c_pm_ops,
	},
	.probe		= piusb_probe,
	.remove		= piusb_remove,
	.id_table	= piusb_id,
};

module_i2c_driver(piusb_driver);

MODULE_DESCRIPTION("Pericom TypeC Detection driver");
MODULE_LICENSE("GPL v2");
