/* Copyright (C) 2008 Rodolfo Giometti <giometti@linux.it>
 * Copyright (C) 2008 Eurotech S.p.A. <info@eurotech.it>
 * Based on a previous work by Copyright (C) 2008 Texas Instruments, Inc.
 *
 * Copyright (c) 2011, The Linux Foundation. All rights reserved.
 * Copyright (c) 2011 - 2022, NEC Platforms, Ltd., All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/idr.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <asm/unaligned.h>
#include <linux/time.h>
#include <linux/gpio.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include "bq27421config.c"

#include "nec_charger_manager.h"

#define BQ27421_CONFREV 10	/* 注意！ガスゲージIC設定値のrevisionに応じて手動で変える必要がある */
#define DRIVER_VERSION			"1.0.0"
#define DEV_TYPE   0x0421
#define BQ27421_CONF_CHECK_SUM	0xFD	/* BQ27421_CONFREV=10 */

/* Bq27421 standard data commands */
#define BQ27421_REG_CNTL		0x00
#define BQ27421_REG_TEMP		0x02
#define BQ27421_REG_VOLT		0x04
#define BQ27421_REG_FLAGS		0x06
#define BQ27421_REG_FLAGS2		0x07
#define BQ27421_FLAG_DSG		BIT(0)
#define BQ27421_FLAG_SOCF		BIT(1)
#define BQ27421_FLAG_SOC1		BIT(2)
#define BQ27421_FLAG_CFG		BIT(4)
#define BQ27421_FLAG_CHG		BIT(8)
#define BQ27421_FLAG_FC			BIT(9)
#define BQ27421_FC_CHECK_MSK	(BQ27421_FLAG_CHG | BQ27421_FLAG_FC)
#define BQ27421_FLAG_INITCOMP	BIT(7)

#define BQ27421_REG_NAC			0x08
#define BQ27421_REG_FAC			0x0a
#define BQ27421_REG_RM			0x0c
#define BQ27421_REG_FCC			0x0e
#define BQ27421_REG_AI			0x10
#define BQ27421_REG_SI			0x12
#define BQ27421_REG_MLI			0x14
#define BQ27421_REG_AP			0x18
#define BQ27421_REG_SOC			0x1c
#define BQ27421_REG_INTTEMP		0x1e
#define BQ27421_REG_SOH			0x20
#define BQ27421_REG_DC			0x3c

/* Control subcommands */
#define BQ27421_SUBCMD_CTNL_STATUS	0x0000
#define BQ27421_SUBCMD_CTNL_SS		BIT(13)

#define BQ27421_SUBCMD_DEVCIE_TYPE	0x0001
#define BQ27421_SUBCMD_FW_VER		0x0002
#define BQ27421_SUBCMD_DF_CSUM		0x0004
#define BQ27421_SUBCMD_PREV_MACW	0x0007
#define BQ27421_SUBCMD_CHEM_ID		0x0008
#define BQ27421_SUBCMD_BAT_INS		0x000c
#define BQ27421_SUBCMD_BAT_REM		0x000d
#define BQ27421_SUBCMD_SET_HIB		0x0011
#define BQ27421_SUBCMD_CLR_HIB		0x0012
#define BQ27421_SUBCMD_SET_CFGUPD	0x0013
#define BQ27421_SUBCMD_ENABLE_SDOWN	0x0014
#define BQ27421_SUBCMD_SDOWN		0x0015
#define BQ27421_SUBCMD_TOGGLE_GPOUT	0x0016
#define BQ27421_SUBCMD_SEALED		0x0020
#define BQ27421_SUBCMD_RESET		0x0041
#define BQ27421_SUBCMD_SW_RESET		0x0042
#define BQ27421_SUBCMD_UNSEALED		0x8000

/* Bq27421 extended commands */
#define BQ27421_DATACLASS		0x3e
#define BQ27421_BLOCKDATACHECKSUM	0x60

#define ZERO_DEGREE_CELSIUS_IN_TENTH_KELVIN   (-2731)

#define BQ27421_RETRYLIMIT		3

/* If the system has several batteries we need a different name for each
 * of them...
 */
static DEFINE_IDR(battery_id);
static DEFINE_MUTEX(battery_mutex);
static DEFINE_MUTEX(battery_check_mutex);
static DEFINE_MUTEX(longlife_mutex);

#define BQ27421_PM_RESUME_REMAIN_TIMEOUT	3*1000	/* 3sec */

#define POWER_SUPPLY_CHARGER_NAME	"bq2589x-charger"

struct bq27421_device_info;
struct bq27421_access_methods {
	int (*read)(u8 reg, int *rt_value, int b_single,
		struct bq27421_device_info *di);
};

struct bq27421_device_info {
	struct device			*dev;
	int				id;
	struct bq27421_access_methods	*bus;
	struct i2c_client		*client;

#ifdef CONFIG_PM
	int				gp_out_wake_irq;
#endif

	struct mutex lock;

	struct charger_manager cm;

	wait_queue_head_t wait_q;
	atomic_t wait_condition;
	wait_queue_head_t pm_resume_remain_q;
	atomic_t pm_resume_remain_condition;

	struct task_struct *bq27421_polling;

	atomic_t power_on_charge_control_enable;

#define DEFAULT_SW_RESET_CYCLE	(3300)  /* cycle = 3300 * 10sec */
	int sw_reset_counter;
	int sw_reset_cycle;

	int charge_voltage_step;
	int previous_charge_voltage_step;

	int volt_read_cnt;
};

static enum power_supply_property bq27421_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,	/* VOLT */
	POWER_SUPPLY_PROP_CURRENT_NOW,	/* AI */
	POWER_SUPPLY_PROP_CAPACITY,	/* SOC */
	POWER_SUPPLY_PROP_CAPACITY_LEVEL,/* FLAGS */
	POWER_SUPPLY_PROP_TEMP,		/* TEMP */
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_CHARGE_NOW,	/* RM */
	POWER_SUPPLY_PROP_CHARGE_FULL,	/* FCC */
	POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN,/* DC */
	POWER_SUPPLY_PROP_ENERGY_AVG,	/* AP */
	POWER_SUPPLY_PROP_CURRENT_MAX,	/* MLI */
	POWER_SUPPLY_PROP_FORCE_DISCHARGE,
	POWER_SUPPLY_PROP_NOTIFY_SRC,
	POWER_SUPPLY_PROP_CHARGE_SAVE,
};

struct taper_voltage_val {
	unsigned char tapvolt_msb;
	unsigned char tapvolt_lsb;
} taper_voltage_table[] = {
	{ 0x10, 0x0C }, /* 4.108V */
	{ 0x0F, 0xDC }, /* 4.06V */
	{ 0x0F, 0xBC }, /* 4.028V */
	{ 0x0F, 0x8C }, /* 3.98V */
	{ 0x0F, 0x6C }, /* 3.948V */
	{ 0x0F, 0x3C }, /* 3.90V */
	{ 0x0F, 0x1C }, /* 3.868V */
};

#define ENABLE_GP_OUT
#define GP_OUT_GPIO		22
#define GPIO_BASE		12
#define GP_OUT_GPIO_NAME	"gp_out"

static int POLLING_ENABLE = 1;
static int ISSETTING = 0;


#define ENABLE_SOC_ZERO_CHK_VLT
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
	#define VOLT_CHECK_MAX  5
	#define VOLT_CHECK_VALUE 3400
#endif

static int bq27421_sw_reset(struct bq27421_device_info *di);
static int bq27421_power_on_check(struct bq27421_device_info *di);
static int bq27421_voltage_step(struct bq27421_device_info *di);
static int bq27421_control(struct bq27421_device_info *di);

static int bq27421_i2c_txsubcmd(u8 reg, unsigned short subcmd,
		struct bq27421_device_info *di);

int bq27421_getVBUSConnect(struct bq27421_device_info *di)
{
	union power_supply_propval pval;
	int ret;
	int vbus_connection = 0;

	if (!di->cm.charger_psy)
		di->cm.charger_psy = power_supply_get_by_name(POWER_SUPPLY_CHARGER_NAME);

	if (di->cm.charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(di->cm.charger_psy, POWER_SUPPLY_PROP_VBUS_CONNECTION, &pval);
	if (ret )
		return -2;
	
	vbus_connection = pval.intval;

	return vbus_connection;
}

int bq27421_getChargeStatus(struct bq27421_device_info *di)
{
	union power_supply_propval pval;
	int ret;
	int charge_status = POWER_SUPPLY_STATUS_UNKNOWN;

	if (!di->cm.charger_psy)
		di->cm.charger_psy = power_supply_get_by_name(POWER_SUPPLY_CHARGER_NAME);

	if (di->cm.charger_psy == NULL)
		return -1;

	ret = power_supply_get_property(di->cm.charger_psy, POWER_SUPPLY_PROP_STATUS, &pval);
	if (ret )
		return -2;

	charge_status = pval.intval;
	if (charge_status == POWER_SUPPLY_STATUS_CHARGING)
		return 1;

	return 0;
}

int bq27421_setChargeEnable(struct bq27421_device_info *di,int enable)
{
	union power_supply_propval pval;
	int ret;

	if (!di->cm.charger_psy)
		di->cm.charger_psy = power_supply_get_by_name(POWER_SUPPLY_CHARGER_NAME);

	if (di->cm.charger_psy == NULL)
		return -1;

	pval.intval = enable != 0 ? true : false;
	ret = power_supply_set_property(di->cm.charger_psy, POWER_SUPPLY_PROP_CHARGING_ENABLED, &pval);

	return ret;
}

int bq27421_setChargeVoltage(struct bq27421_device_info *di,int step)
{
	union power_supply_propval pval;
	int ret;

	if (!di->cm.charger_psy)
		di->cm.charger_psy = power_supply_get_by_name(POWER_SUPPLY_CHARGER_NAME);

	if (di->cm.charger_psy == NULL)
		return -1;

	dev_info(di->dev, "####bq27421_setChargeVoltage to power_supply_set_charge_voltage\n");

	pval.intval = step;
	ret = power_supply_set_property(di->cm.charger_psy, POWER_SUPPLY_PROP_CHARGE_VOLTAGE, &pval);

	return ret;
}

static int bq27421_read(u8 reg, int *rt_value, int b_single,
			struct bq27421_device_info *di)
{
	int ret;

	mutex_lock(&di->lock);
	ret = di->bus->read(reg, rt_value, b_single, di);
	mutex_unlock(&di->lock);

	return ret;
}

static int bq27421_battery_status(struct bq27421_device_info *di,
	union power_supply_propval *val)
{
	int status, flags, ret;

	ret = bq27421_read(BQ27421_REG_FLAGS, &flags, 0, di);
	if (ret) {
		dev_err(di->dev, "error battery status\n");
		return ret;
	}

	if ((flags & BQ27421_FC_CHECK_MSK) == BQ27421_FLAG_FC)
		status = POWER_SUPPLY_STATUS_FULL;
	else if (flags & BQ27421_FLAG_DSG)
		status = POWER_SUPPLY_STATUS_DISCHARGING;
	else
		status = POWER_SUPPLY_STATUS_CHARGING;

	val->intval = status;

	return 0;
}

static int bq27421_battery_capacity_level(struct bq27421_device_info *di,
	union power_supply_propval *val)
{
	int level, flags, ret;

	ret = bq27421_read(BQ27421_REG_FLAGS, &flags, 0, di);
	if (ret) {
		dev_err(di->dev, "error battery status\n");
		return ret;
	}

	if (flags & BQ27421_FLAG_FC)
		level = POWER_SUPPLY_CAPACITY_LEVEL_FULL;
	else if (flags & BQ27421_FLAG_SOCF)
		level = POWER_SUPPLY_CAPACITY_LEVEL_CRITICAL;
	else if (flags & BQ27421_FLAG_SOC1)
		level = POWER_SUPPLY_CAPACITY_LEVEL_LOW;
	else
		level = POWER_SUPPLY_CAPACITY_LEVEL_NORMAL;

	val->intval = level;

	return 0;
}

/*
 * Return the battery temperature in tenths of degree Celsius
 * Or < 0 if something fails.
 */
static int bq27421_battery_temperature(struct bq27421_device_info *di,
    union power_supply_propval *val)
{
	int ret;
	int temp = 0;

	ret = bq27421_read(BQ27421_REG_TEMP, &temp, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading temperature\n");
		return ret;
	}

	val->intval = temp + ZERO_DEGREE_CELSIUS_IN_TENTH_KELVIN;

	return 0;
}

static int bq27421_simple_value(struct bq27421_device_info *di, u8 reg,
    union power_supply_propval *val)
{
	int value = 0, ret;

	ret = bq27421_read(reg, &value, 0, di);
	if (value < 0)
		return value;

	val->intval = value;

	return 0;
}

static int bq27421_subcmd_read(struct bq27421_device_info *di, int *rt_value,
    u16 subcmd)
{
	int ret;

	mutex_lock(&di->lock);
	ret = bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, subcmd, di);
	if (ret < 0) {
		mutex_unlock(&di->lock);
		return ret;
	}

	udelay(66);

	ret = di->bus->read(BQ27421_REG_CNTL, rt_value, 0, di);

	mutex_unlock(&di->lock);

	return ret;
}

#define RETRY_COUNT 5
static int bq27421_retry_sleep_ms[RETRY_COUNT] = {
	10, 20, 30, 40, 50
};
/*
 * i2c specific code
 */
static int bq27421_i2c_txsubcmd(u8 reg, unsigned short subcmd,
		struct bq27421_device_info *di)
{
	s32 ret;
	int retry_count = 0;

retry:
	ret = i2c_smbus_write_word_data(di->client, reg, subcmd);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x subcmd 0x%04x\n", __func__, ret, reg, subcmd);
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c write fail: can't write %02x to %02x: %d\n",
			subcmd, reg, ret);
		return ret;
	}
	pr_debug("Writing 0x%02x=0x%04x\n", reg, subcmd);
	return 0;
}

static int bq27421_read_i2c(u8 reg, int *rt_value, int b_single,
			struct bq27421_device_info *di)
{
	s32 ret;
	int retry_count = 0;
	int success_count = 0;

retry:
	if (!b_single)
		ret = i2c_smbus_read_word_data(di->client, reg);
	else
		ret = i2c_smbus_read_byte_data(di->client, reg);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x\n", __func__, ret, reg);
	if ((ret >= 0) && (success_count == 0)) {
		success_count++;
		goto retry;
	}
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		success_count = 0;
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c read fail: can't read from %02x: %d\n", reg, ret);
		return ret;
	} else {
		*rt_value = (int)ret;
	}

	return 0;
}

static int bq27421_i2c_read(struct bq27421_device_info *di, u8 reg, unsigned char *buf, int len)
{
	int i;
	int tmp;
	
	mutex_lock(&di->lock);
	
	for (i = 0; i < len; i++, reg++, buf++) {
		bq27421_read_i2c(reg, &tmp, 1, di);
		*buf = (unsigned char)tmp;
	}

	mutex_unlock(&di->lock);

	return 0;
}

static int bq27421_i2c_write_byte(struct bq27421_device_info *di, u8 reg, const unsigned char *buf)
{
	s32 ret;
	int retry_count = 0;

retry:
	ret = i2c_smbus_write_byte_data(di->client, reg, *buf);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x buf 0x%02x\n", __func__, ret, reg, *buf);
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c write fail: can't write %02x to %02x: %d\n",
			*buf, reg, ret);
		return ret;
	}
	pr_debug("Writing 0x%02x=0x%02x\n", reg, *buf);
	return 0;
}

static int bq27421_i2c_write(struct bq27421_device_info *di, u8 reg, unsigned char *buf, int len)
{
	int i;

	mutex_lock(&di->lock);

	for (i = 0; i < len; i++, reg++, buf++)
		bq27421_i2c_write_byte(di, reg, buf);

	mutex_unlock(&di->lock);

	return 0;
}

static int isBatteryConnected(struct bq27421_device_info *di)
{
	int dev_type;

	/* Verify Device Type */
	bq27421_subcmd_read(di, &dev_type, BQ27421_SUBCMD_DEVCIE_TYPE);
	if (dev_type != DEV_TYPE) {
		dev_err(di->dev, "error: Nonexpected Device type 0x%04x\n", dev_type);
		return 0;
	}

	return 1;
}

static int isAlreadyConfigured(struct bq27421_device_info *di)
{	/* subclassID 107(0x6B)のcheck sumがBQ27421_CONF_CHECK_SUMかで確認する */
	unsigned char data[2];

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
	mutex_unlock(&di->lock);

	data[0] = 0x6B;
	data[1] = 0x00;
	bq27421_i2c_write(di, BQ27421_DATACLASS, data, 2);
	msleep(1); /* 少し間を開けないとエラーになる */
	bq27421_i2c_read(di, BQ27421_BLOCKDATACHECKSUM, data, 1);

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	if (data[0] == BQ27421_CONF_CHECK_SUM)
		return 1;
	else
		return 0;
}

static int bq27421_set_charge_voltage(struct bq27421_device_info *di, int charge_voltage_step)
{
	int cnt;
	unsigned char data[32];
	unsigned char OLD_TapVol_MSB;
	unsigned char OLD_TapVol_LSB;
	unsigned char OLD_Csum;
	unsigned char NEW_Csum;
	unsigned char temp;

set_charge_voltage_start:
	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	mutex_unlock(&di->lock);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SET_CFGUPD, di);
		mutex_unlock(&di->lock);
		msleep(1500);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, data, 1);
	} while(!(data[0] & BQ27421_FLAG_CFG) && ++cnt < BQ27421_RETRYLIMIT);

	if (cnt < BQ27421_RETRYLIMIT) {
		dev_info(di->dev,"####bq27421_set_charge_voltage BQ27421_FLAG_CFG 1-OK\n");
	}

	/* BlockDataControl */
	data[0] = 0;
	bq27421_i2c_write(di, 0x61, data, 1); /* BlockDataControl */

	msleep(5);
	/* DataBlockClass() command (0x3E) to access the State subclass
	 (82 decimal, 0x52 hex) containing the Taper Voltage parameter */
	data[0] = 0x52;
	data[1] = 0;
	bq27421_i2c_write(di, BQ27421_DATACLASS, data, 2); /* BlockDataControl */

	msleep(5);
	/* Read the 1-byte checksum using the BlockDataChecksum() command (0x60). */
	bq27421_i2c_read(di, BQ27421_BLOCKDATACHECKSUM, data, 1);
	OLD_Csum = data[0];

	dev_info(di->dev,"####bq27421_set_charge_voltage OLD_Csum:%02x\n", OLD_Csum);

	msleep(5);
	/* Read both Taper Voltage bytes starting at 0x5D (offset = 29). Block data
	starts at 0x40, so to read the data of a specific offset, use address 0x40 +
	mod(offset, 32). */
	bq27421_i2c_read(di, 0x5D, data, 2);
	
	OLD_TapVol_MSB = data[0];
	OLD_TapVol_LSB = data[1];

	dev_info(di->dev,"####bq27421_set_charge_voltage OLD_TapVol_MSB:%02x OLD_TapVol_LSB:%02x\n", OLD_TapVol_MSB, OLD_TapVol_LSB);
	msleep(5);
	/* Write both Taper Voltage bytes starting at 0x5D (offset = 29). */
	data[0] = taper_voltage_table[charge_voltage_step].tapvolt_msb;
	data[1] = taper_voltage_table[charge_voltage_step].tapvolt_lsb;
	dev_info(di->dev,"####bq27421_set_charge_voltage NEW_TapVol_MSB:%02x NEW_TapVol_LSB:%02x\n", data[0], data[1]);
	bq27421_i2c_write(di, 0x5D, data, 2);

	msleep(5);
	memset(data, 0, sizeof(data));
	/* Read back both Taper Voltage and show*/
	bq27421_i2c_read(di, 0x5D, data, 2);
	dev_info(di->dev,"####bq27421_set_charge_voltage read back NEW_TapVol_MSB:%02x NEW_TapVol_LSB:%02x\n", data[0], data[1]);

	/* Compute the new block checksum. The checksum is (255 - x) where x is the
	8-bit summation of the BlockData() (0x40 to 0x5F) on a byte-by-byte basis. A
	quick way to calculate the new checksum uses a data replacement method
	with the old and new data summation bytes. */

	temp = (255 - OLD_Csum - OLD_TapVol_MSB - OLD_TapVol_LSB)%256;
	NEW_Csum = 255 - ((temp + taper_voltage_table[charge_voltage_step].tapvolt_msb + 
			taper_voltage_table[charge_voltage_step].tapvolt_lsb)%256);
	dev_info(di->dev,"####bq27421_set_charge_voltage NEW_Csum:%02x\n", NEW_Csum);
/*
	temp = fmod(255 - OLD_Csum - OLD_TapVol_MSB - OLD_TapVol_LSB, 256);
	NEW_Csum = 255 - fmod(temp + taper_voltage_table[charge_voltage_step].tapvolt_msb + 
			taper_voltage_table[charge_voltage_step].tapvolt_lsb, 256);
*/
	msleep(5);
	/* Write new checksum. The data is actually transferred to the Data Memory
	when the correct checksum for the whole block (0x40 to 0x5F) is written to
	BlockDataChecksum() (0x60).*/

	data[0] = NEW_Csum;
	bq27421_i2c_write(di, BQ27421_BLOCKDATACHECKSUM, data, 1);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
		mutex_unlock(&di->lock);
		msleep(3000);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, data, 1);
	} while((data[0] & BQ27421_FLAG_CFG) && (++cnt < BQ27421_RETRYLIMIT));

	if (cnt == BQ27421_RETRYLIMIT) {
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_RESET, di);
		msleep(1000);
		dev_err(di->dev, "####bq27421_set_charge_voltage set charge voltage restart\n");
		goto set_charge_voltage_start;
	}
	dev_info(di->dev,"####bq27421_set_charge_voltage BQ27421_FLAG_CFG 2-OK\n");

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	return 0;
}

#define to_bq27421_device_info(x) container_of((x), \
				struct bq27421_device_info, bat);

static int bq27421_battery_get_property(struct power_supply *psy,
					enum power_supply_property psp,
					union power_supply_propval *val)
{
	int ret = 0;
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		ret = bq27421_battery_status(di, val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_VOLT, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = 1;
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_AI, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		if(atomic_read(&di->pm_resume_remain_condition) == 0 ) {
			dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));
			if (wait_event_timeout(di->pm_resume_remain_q, atomic_read(&di->pm_resume_remain_condition) == 1,
					       msecs_to_jiffies(BQ27421_PM_RESUME_REMAIN_TIMEOUT)) != 0) {
				pr_info("%s %u: wakeup @@@@\n", __func__, __LINE__);
			}
		}
		val->intval = di->cm.battery_remain;
		dev_dbg(di->dev,"@@@@ %s battery_remain=%d\n",__FUNCTION__,di->cm.battery_remain);
		ret = 0;
		break;
	case POWER_SUPPLY_PROP_CAPACITY_LEVEL:
		ret = bq27421_battery_capacity_level(di, val);
		break;
	case POWER_SUPPLY_PROP_TEMP:
		ret = bq27421_battery_temperature(di, val);
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = POWER_SUPPLY_TECHNOLOGY_LION;
		break;
	case POWER_SUPPLY_PROP_CHARGE_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_RM, val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL:
		ret = bq27421_simple_value(di, BQ27421_REG_FCC, val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN:
		ret = bq27421_simple_value(di, BQ27421_REG_DC, val);
		break;
	case POWER_SUPPLY_PROP_ENERGY_AVG:
		ret = bq27421_simple_value(di, BQ27421_REG_AP, val);
		break;
	case POWER_SUPPLY_PROP_CURRENT_MAX:
		ret = bq27421_simple_value(di, BQ27421_REG_MLI, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		ret = di->cm.discharge_status;
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		val->intval = charger_manager_get_force_discharge_info(&di->cm);
		break;
	case POWER_SUPPLY_PROP_CHARGE_SAVE:
		dev_info(di->dev, "@@@ %s POWER_SUPPLY_PROP_CHARGE_SAVE\n",__func__);
		val->intval = di->cm.charge_save;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int bq27421_battery_set_property(struct power_supply *psy,
					enum power_supply_property psp,
		const union power_supply_propval *val)
{
	int ret = 0;
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		charger_manager_reset_flag(&di->cm);
		atomic_set(&di->wait_condition, 1);
		wake_up(&di->wait_q);
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		charger_manager_set_force_discharge_req(&di->cm);
		break;
	case POWER_SUPPLY_PROP_CHARGE_SAVE:
		dev_info(di->dev, "@@@ %s POWER_SUPPLY_PROP_CHARGE_SAVE\n",__func__);
		charger_manager_reset_flag(&di->cm);
		di->cm.charge_save = val->intval;
		break;
	default:
		return -EINVAL;
	}
	return ret;
}

static int bq27421_battery_is_writeable(struct power_supply *psy,
				       enum power_supply_property prop)
{
	int rc = 0;

	switch (prop) {
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
	case POWER_SUPPLY_PROP_CHARGE_SAVE:
		rc = 1;
		break;
	default:
		rc = 0;
		break;
	}
	return rc;
}
static void bq27421_external_power_changed(struct power_supply *psy)
{
}

struct power_supply_config g_bq27421_psy_cfg;

static const struct power_supply_desc bq27421_power_supply_desc = {
	.name = "bq27421-battery",
	.type = POWER_SUPPLY_TYPE_BATTERY,
	.properties = bq27421_battery_props,
	.num_properties = ARRAY_SIZE(bq27421_battery_props),
	.get_property = bq27421_battery_get_property,
	.set_property = bq27421_battery_set_property,
	.external_power_changed = bq27421_external_power_changed,
	.property_is_writeable = bq27421_battery_is_writeable,
};

static int bq27421_powersupply_init(struct bq27421_device_info *di)
{
	int flags = 0, type = 0, fw_ver = 0;

	g_bq27421_psy_cfg.of_node = di->dev->of_node;
	g_bq27421_psy_cfg.drv_data = di;
	
	mutex_init(&di->lock);

	bq27421_subcmd_read(di, &flags, BQ27421_SUBCMD_CTNL_STATUS);

	bq27421_subcmd_read(di, &type, BQ27421_SUBCMD_DEVCIE_TYPE);

	bq27421_subcmd_read(di, &fw_ver, BQ27421_SUBCMD_FW_VER);

	dev_info(di->dev, "DEVICE_TYPE is 0x%02X, FIRMWARE_VERSION is 0x%02X\n",
			type, fw_ver);
	dev_info(di->dev, "Complete bq27421 configuration 0x%02X\n", flags);

	dev_info(di->dev, "support ver. %s enabled\n", DRIVER_VERSION);

	di->cm.battery_psy = power_supply_register(di->dev, &bq27421_power_supply_desc, &g_bq27421_psy_cfg);
	if (IS_ERR(di->cm.battery_psy)) {
		if (PTR_ERR(di->cm.battery_psy) == -EPROBE_DEFER)
			dev_dbg(di->dev, "failed to register battery, deferring probe\n");
		else
			dev_err(di->dev, "failed to register battery\n");

		di->cm.battery_psy = NULL;
		return -1;
	}

	return 0;
}

static void bq27421_powersupply_unregister(struct bq27421_device_info *di)
{
	power_supply_unregister(di->cm.battery_psy);
	mutex_destroy(&di->lock);
}

#if 1
static int stdreg;
static int subcmd;
static ssize_t bq27421_read_stdcmd(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	int ret;
	int temp = 0;
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	if (stdreg > 0x00) {
		ret = bq27421_read(stdreg, &temp, 0, di);
		if (ret)
			ret = snprintf(buf, PAGE_SIZE, "Read Error!\n");
		else
			ret = snprintf(buf, PAGE_SIZE, "0x%02x\n", temp);
	} else
		ret = snprintf(buf, PAGE_SIZE, "Register Error!\n");

	return ret;
}

static ssize_t bq27421_write_stdcmd(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	int cmd;

	sscanf(buf, "%x", &cmd);
	stdreg = cmd;
	return ret;
}

static ssize_t bq27421_read_subcmd(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	int ret;
	int temp = 0;
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	if (subcmd == BQ27421_SUBCMD_DEVCIE_TYPE ||
		 subcmd == BQ27421_SUBCMD_FW_VER ||
		 subcmd == BQ27421_SUBCMD_CHEM_ID) {

		ret = bq27421_subcmd_read(di, &temp, subcmd);

		if (ret)
			ret = snprintf(buf, PAGE_SIZE, "Read Error!\n");
		else
			ret = snprintf(buf, PAGE_SIZE, "0x%02x\n", temp);
	} else
		ret = snprintf(buf, PAGE_SIZE, "Register Error!\n");

	return ret;
}

static ssize_t bq27421_write_subcmd(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	int cmd;

	sscanf(buf, "%x", &cmd);
	subcmd = cmd;
	return ret;
}

static int REG = 0, LEN = 0;
static ssize_t bq27421_read_register(struct device *dev,
				     struct device_attribute *attr,
				     char *buf)
{
	int i, ret;
	unsigned char data[32];
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);
	
	bq27421_i2c_read(di, REG, data, LEN);

	ret = 0;
	for (i = 0; i < LEN; i++) {
		ret += snprintf(buf, PAGE_SIZE, "0x%02x: 0x%02x\n", REG + i, data[i]);
		buf += 11; /* len of "%#02x: %#02x\n" */
	}

	return ret;
}

/* format: {r/w} reg len [data...]  (all HEX) */
static ssize_t bq27421_write_register(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf,
				    size_t count)
{
	int i;
	unsigned char data[32];
	char read = buf[0];
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	if (read != 'r' && read != 'w') {
		dev_err(dev, "usage: {r/w} reg len [data...]  (all HEX. len<=0x20)\n");
		return -EINVAL;
	}

	buf++;

	if (sscanf(buf, "%x %x", &REG, &LEN) != 2)
		return -EINVAL;

	if (LEN > 0x20){
		dev_err(dev, "usage: {r/w} reg len [data...]  (all HEX. len<=0x20)\n");
		return -EINVAL;
	}

	if (read == 'r')
		return ret;

	/* REG, LEN は読み飛ばす */
	while (*buf == ' ') /* skip ' ' */
		buf++;
	while (*buf != ' ' && *buf != '\0') /* skip REG */
		buf++;
	while (*buf == ' ')	/* skip ' ' */
		buf++;
	while (*buf != ' ' && *buf != '\0') /* skip LEN */
		buf++;
	while (*buf == ' ') /* skip ' ' */
		buf++;

	for (i = 0; i < LEN; i++) {
		if (sscanf(buf, "%x", (unsigned int *)(&data[i])) != 1)
			return -EINVAL;

		while (*buf != ' ' && *buf != '\0')
			buf++;
		while (*buf == ' ') /* skip ' ' */
			buf++;
	}

	bq27421_i2c_write(di, REG, data, LEN);
	return ret;
}

static ssize_t bq27421_read_pollsetting(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", POLLING_ENABLE);	
}

static ssize_t bq27421_write_pollsetting(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	int enable;
	if (sscanf(buf, "%d", &enable) != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	if (enable != 0 && enable != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	POLLING_ENABLE = enable;

	return strnlen(buf, PAGE_SIZE);
}

static ssize_t bq27421_read_longlife(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d\n", di->cm.longlife);
}

static ssize_t bq27421_write_longlife(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	mutex_lock(&longlife_mutex);
	di->cm.longlife = val;
	if (di->cm.longlife == 1)
		charger_manager_reset_flag(&di->cm);
	mutex_unlock(&longlife_mutex);
	atomic_set(&di->wait_condition, 1);
	wake_up(&di->wait_q);

	return count;
}

static ssize_t bq27421_read_swreset_cycle(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d/%d\n", di->sw_reset_counter, di->sw_reset_cycle);
}

static ssize_t bq27421_write_swreset_cycle(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);
	unsigned long val = simple_strtoul(buf, NULL, 0);

	di->sw_reset_counter = di->sw_reset_cycle = val;

	return count;
}

static ssize_t bq27421_read_dischg_start_period(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d/%d\n", di->cm.dischg_counter, di->cm.dischg_start_period);
}

static ssize_t bq27421_write_dischg_start_period(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);
	unsigned long val = simple_strtoul(buf, NULL, 0);

	di->cm.dischg_counter = di->cm.dischg_start_period = val;

	return count;
}

static ssize_t bq27421_read_charge_control_enable(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d\n", atomic_read(&di->power_on_charge_control_enable));
}

static ssize_t bq27421_write_charge_control_enable(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);
	unsigned long val = simple_strtoul(buf, NULL, 0);

	if (val == 0)
		atomic_set(&di->power_on_charge_control_enable, 0);
	else
		atomic_set(&di->power_on_charge_control_enable, 1);

	atomic_set(&di->wait_condition, 1);
	wake_up(&di->wait_q);

	return count;
}

static ssize_t bq27421_read_charge_voltage_step(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d\n", di->charge_voltage_step);
}

static ssize_t bq27421_write_charge_voltage_step(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);
	unsigned long val = simple_strtoul(buf, NULL, 0);

	if (val >= (sizeof(taper_voltage_table)/sizeof(taper_voltage_table[0]))) {
		dev_err(dev, "invalid step=%ld >= %ld\n", val, (sizeof(taper_voltage_table)/sizeof(taper_voltage_table[0])));
		return count;
	}

	di->charge_voltage_step = val;
	dev_info(dev, "####bq27421_write_charge_voltage_step set%ld\n", val);

	return count;
}

#ifdef CHARGING_CONTROL_DEBUG
static ssize_t charging_control_debug_read(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	return snprintf(buf, PAGE_SIZE, "%d\n", di->cm.charging_control_debug);
}

static ssize_t charging_control_debug_write(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = power_supply_get_drvdata(psy);

	if (val == 1) {
		dev_info(dev, " debug start val=%ld\n", val);
		charger_manager_set_debug(&di->cm,1);
	} else {
		dev_info(dev, " debug stop val=%ld\n", val);
		charger_manager_set_debug(&di->cm,0);
	}

	return count;
}

#endif /* CHARGING_CONTROL_DEBUG */

static DEVICE_ATTR(std_cmd, 0664, bq27421_read_stdcmd,
	bq27421_write_stdcmd);
static DEVICE_ATTR(sub_cmd, 0664, bq27421_read_subcmd,
	bq27421_write_subcmd);
static DEVICE_ATTR(register, 0664, bq27421_read_register,
	bq27421_write_register);
static DEVICE_ATTR(polling, 0664, bq27421_read_pollsetting,
	bq27421_write_pollsetting);
static DEVICE_ATTR(longlife, 0664, bq27421_read_longlife,
	bq27421_write_longlife);
static DEVICE_ATTR(swreset_cycle, 0664, bq27421_read_swreset_cycle,
	bq27421_write_swreset_cycle);
static DEVICE_ATTR(dischg_start_period, 0664, bq27421_read_dischg_start_period,
	bq27421_write_dischg_start_period);
static DEVICE_ATTR(charge_control_enable, 0664, bq27421_read_charge_control_enable,
	bq27421_write_charge_control_enable);
static DEVICE_ATTR(charge_voltage_step, 0664, bq27421_read_charge_voltage_step,
	bq27421_write_charge_voltage_step);

#ifdef CHARGING_CONTROL_DEBUG
static DEVICE_ATTR(charging_control_debug, 0664, charging_control_debug_read,
	charging_control_debug_write);
#endif /* CHARGING_CONTROL_DEBUG */

static struct attribute *fs_attrs[] = {
	&dev_attr_std_cmd.attr,
	&dev_attr_sub_cmd.attr,
	&dev_attr_register.attr,
	&dev_attr_polling.attr,
	&dev_attr_longlife.attr,
	&dev_attr_swreset_cycle.attr,
	&dev_attr_dischg_start_period.attr,
	&dev_attr_charge_control_enable.attr,
	&dev_attr_charge_voltage_step.attr,
#ifdef CHARGING_CONTROL_DEBUG
	&dev_attr_charging_control_debug.attr,
#endif /* CHARGING_CONTROL_DEBUG */
	NULL,
};
static struct attribute_group fs_attr_group = {
	.attrs = fs_attrs,
};
#endif

static int set_necat_configure(struct bq27421_device_info *di)
{
	int cnt;
	int i, j, cmdtop;
	int reg, len, st;
	unsigned char data[32], val;

	cnt = 0;
	i = 0;
	cmdtop = 0;

	while (i < sizeof(BQ27421CONF)/sizeof(BQ27421CONF[0])) {
		if (i == cmdtop) {
			data[0] = 0;
			bq27421_i2c_write(di, 0x61, data, 1); /* BlockDataControl */
		}

		switch (BQ27421CONF[i]) {
		case 'W':
			reg = BQ27421CONF[i+1];
			len = BQ27421CONF[i+2];
			for (j = 0; j < len; j++)
				data[j] = (unsigned char)BQ27421CONF[i+3+j];

			bq27421_i2c_write(di, reg, data, len);

			i += 3 + len;
			msleep(1); /* 少し間を開けないとエラーになる */
			break;

		case 'X':
			st = BQ27421CONF[i+1];
			msleep(st);
			i += 2;
			break;

		case 'C':
			reg = BQ27421CONF[i+1];
			val = BQ27421CONF[i+2];
			bq27421_i2c_read(di, reg, data, 1);
			if (data[0] == val) {
				i += 3;
				cmdtop = i;
				cnt = 0;
			}
			else {
				i = cmdtop; /* 書き込みに失敗したらもう一度設定しなおす */
				if (++cnt >= BQ27421_RETRYLIMIT)
					return 0;
			}
			break;

		default:
			break;
		}
	}
	return 0;
}

static int pgcon_wakeup_detect;
module_param(pgcon_wakeup_detect, int, 0644);

#ifdef ENABLE_GP_OUT
static irqreturn_t
gp_out_wakeup_interrupt(int irq, void *arg)
{
	pr_err("%s\n", __func__);

	return IRQ_HANDLED;
}
#endif

static int bq27421_gpio_init(struct bq27421_device_info *di)
{
#define ERRMSG(X)							\
	dev_err(di->dev, "failed to request GPIO pin %d, error %d\n", X, err)
#define ERRMSG_DIR(X, D)						\
	dev_err(di->dev, "failed to config GPIO %d as " D " pin, err %d\n", X, err)

	int err = 0;
#ifdef ENABLE_GP_OUT
	int irq;

	if ((err = gpio_request(GP_OUT_GPIO + GPIO_BASE, GP_OUT_GPIO_NAME)) < 0) {
		ERRMSG(GP_OUT_GPIO);
		goto error;
	}

	irq = gpio_to_irq(GP_OUT_GPIO + GPIO_BASE);
	err = request_irq(irq, &gp_out_wakeup_interrupt,
	    IRQF_TRIGGER_FALLING, "gp_out", (void *)di);
	if (err < 0) {
		dev_err(di->dev, "setup fail wakeup interrupt\n");
		gpio_free(GP_OUT_GPIO + GPIO_BASE);
		goto error;
	}
	disable_irq(irq);
	di->gp_out_wake_irq = irq;
error:
#endif
	return err;
}

static int bq27421_init_config(struct bq27421_device_info *di)
{
	int cnt;
	unsigned char data;

	bq27421_setChargeEnable(di,0);

setting_start:
	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
	mutex_unlock(&di->lock);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SET_CFGUPD, di);
		mutex_unlock(&di->lock);
		msleep(1500);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(!(data & BQ27421_FLAG_CFG) && ++cnt < BQ27421_RETRYLIMIT);

	set_necat_configure(di);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
		mutex_unlock(&di->lock);
		msleep(3000);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(data & BQ27421_FLAG_CFG  && ++cnt < BQ27421_RETRYLIMIT);

	cnt = 0;
	do {
		msleep(1000);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(!(data & BQ27421_FLAG_INITCOMP) && ++cnt < BQ27421_RETRYLIMIT);
	if (cnt == BQ27421_RETRYLIMIT) {
		dev_err(di->dev, "bq27421 error: INITCOMP is not up\n");
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_RESET, di);
		msleep(1000);
		dev_err(di->dev, "bq27421: initialize restart\n");
		goto setting_start;
	}

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	return 0;
}

static int bq27421_sw_reset(struct bq27421_device_info *di)
{
	/* SW Reset */
	if ((bq27421_getVBUSConnect(di) > 0) && /* connect VBUS */
	    (bq27421_getChargeStatus(di) == 0)) { /* No Charge */
		if (di->sw_reset_counter-- <= 0) {
			mutex_lock(&di->lock);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
			mutex_unlock(&di->lock);
			dev_info(di->dev,
				 "bq27421 %s %u: SW Reset done @@@\n",
				 __func__, __LINE__);
			di->sw_reset_counter = di->sw_reset_cycle;
		}
	} else {
		di->sw_reset_counter = di->sw_reset_cycle;
	}
	return 0;
}

static int bq27421_power_on_check(struct bq27421_device_info *di)
{
	if (atomic_read(&di->power_on_charge_control_enable) == 1) {
		if (!di->cm.power_on_state) {
			dev_info(di->dev,"Exit power save Enter Power On\n");
			charger_manager_reset_flag(&di->cm);
			di->cm.power_on_state = true;
		}
	} else {
		if (di->cm.power_on_state) {
			dev_info(di->dev,"Enter power save PowerOff/Hibertante/(Waiting)\n");
			charger_manager_reset_flag(&di->cm);
			di->cm.power_on_state = false;
		}
	}
	return 0;
}

static int bq27421_voltage_step(struct bq27421_device_info *di)
{
	if (di->previous_charge_voltage_step != di->charge_voltage_step) {
		if (bq27421_getChargeStatus(di) == 0) {
			dev_info(di->dev, "####Do bq27421_set_charge_voltage\n");
			bq27421_set_charge_voltage(di, di->charge_voltage_step);
			dev_info(di->dev, "####Do bq27421_setChargeVoltage\n");
			bq27421_setChargeVoltage(di,di->charge_voltage_step);
			di->previous_charge_voltage_step = di->charge_voltage_step;
		} else {
			dev_info(di->dev, "####Skip bq27421_set_charge_voltage\n");
			di->charge_voltage_step = di->previous_charge_voltage_step;
		}
	}
	return 0;
}

static int bq27421_control(struct bq27421_device_info *di)
{
	if (di->cm.battery_remain < 0) {
		dev_err(di->dev, "bq27421 battery capacity get error.\n");
		bq27421_setChargeEnable(di,1);
		return -1;
	}

	if (!di->cm.charger_psy)
		di->cm.charger_psy = power_supply_get_by_name(POWER_SUPPLY_CHARGER_NAME);

	charger_manager_control(&di->cm);
	return 0;
}

static int monitoring(struct bq27421_device_info *di)
{
	int ctl_stat;
	unsigned char flags;
	union power_supply_propval val;
	int ret = 0;
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
	int val_volt = 0;
#endif	/* ENABLE_SOC_ZERO_CHK_VLT */

	mutex_lock(&battery_check_mutex);

	if (!isBatteryConnected(di)) {
		dev_err(di->dev, "bq27421 initialize error. (No battery connected?)\n");
		di->cm.battery_remain = -1;
		atomic_set(&di->pm_resume_remain_condition, 1);
		wake_up(&di->pm_resume_remain_q);
		dev_dbg(di->dev, "@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));
		mutex_unlock(&battery_check_mutex);
		return -1;
	}

	if (isAlreadyConfigured(di)) {
		bq27421_subcmd_read(di, &ctl_stat, BQ27421_SUBCMD_CTNL_STATUS);
		if (!(ctl_stat & BQ27421_SUBCMD_CTNL_SS)) {
			di->cm.battery_remain = -1;
			atomic_set(&di->pm_resume_remain_condition, 1);
			wake_up(&di->pm_resume_remain_q);
			dev_dbg(di->dev, "@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));
			mutex_unlock(&battery_check_mutex);
			ISSETTING = 1;
			mutex_lock(&battery_check_mutex);
			bq27421_setChargeEnable(di,0);

			bq27421_i2c_read(di, BQ27421_REG_FLAGS, &flags, 1);
			while(flags & BQ27421_FLAG_CFG) {
				mutex_lock(&di->lock);
				bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
				mutex_unlock(&di->lock);
				msleep(2000);
				bq27421_i2c_read(di, BQ27421_REG_FLAGS, &flags, 1);
			}

			mutex_lock(&di->lock);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
			mutex_unlock(&di->lock);

			msleep(2000);
		}
	}
	else {
		di->cm.battery_remain = -1;
		atomic_set(&di->pm_resume_remain_condition, 1);
		wake_up(&di->pm_resume_remain_q);
		dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));
		dev_info(di->dev,
			 "bq27421 %s %u: init_config @@@\n",
			 __func__, __LINE__);
		mutex_unlock(&battery_check_mutex);
		ISSETTING = 1;
		mutex_lock(&battery_check_mutex);
		bq27421_init_config(di);
	}

	ISSETTING = 0;
	ret = bq27421_simple_value(di, BQ27421_REG_SOC, &val);
	if (ret == 0)
	{
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
		if(val.intval == 0 )
		{
			ret = bq27421_read(BQ27421_REG_VOLT, &val_volt, 0, di);
			if (ret == 0)
			{
				if(val_volt <= VOLT_CHECK_VALUE )	/* 3400mv under */
				{
					di->volt_read_cnt++;
					if(di->volt_read_cnt < VOLT_CHECK_MAX ) {
						di->cm.battery_remain = 1;
					} else {
						di->cm.battery_remain = 0;
					}
					pr_info("battery_remain=%d  Voltage=0x%04x  under cnt=%d\n",di->cm.battery_remain,val_volt,di->volt_read_cnt);
				}
				else								/* 3400mv over */
				{
					di->cm.battery_remain = 1;
					pr_debug("battery_remain=%d  Voltage=0x%04x  under cnt=%d\n",di->cm.battery_remain,val_volt,di->volt_read_cnt);
					di->volt_read_cnt=0;
				}
			} else {
				di->cm.battery_remain = val.intval;
			}
		} else {
			di->volt_read_cnt=0;
			di->cm.battery_remain = val.intval;
		}
#else	/* ENABLE_SOC_ZERO_CHK_VLT */
		di->cm.battery_remain = val.intval;
#endif	/* ENABLE_SOC_ZERO_CHK_VLT */
		dev_dbg(di->dev,"@@@@ %s battery_remain=%d\n",__FUNCTION__,di->cm.battery_remain);
	}
	else
		dev_err(di->dev, "bq27421 battery capacity get error.\n");
	atomic_set(&di->pm_resume_remain_condition, 1);
	wake_up(&di->pm_resume_remain_q);
	dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));

	bq27421_control(di);

	/* SW Reset */
	bq27421_sw_reset(di);

	/* Power On Check */
	bq27421_power_on_check(di);

	/* set charge_voltage */
	bq27421_voltage_step(di);

	mutex_unlock(&battery_check_mutex);
	return 0;
}

static int bq27421_polling_thread(void *arg)
{
	struct i2c_client *client = arg;
	struct bq27421_device_info *di = i2c_get_clientdata(client);
	int err = 0;

	while (!kthread_should_stop()) {
		if (POLLING_ENABLE)
			err = monitoring(di);

		if (err == -1) {
			/* Polling Disable */
			POLLING_ENABLE = 0;
			err = 0;
			dev_err(di->dev, "bq27421 monitoring error. Stop monitoring\n");
		}

		atomic_set(&di->wait_condition, 0);
		if (wait_event_timeout(di->wait_q, atomic_read(&di->wait_condition) == 1,
				       msecs_to_jiffies(POLLING_INTERVAL)) != 0) {
			pr_info("%s %u: wakeup @@@@\n", __func__, __LINE__);
		}
	}
	return err;
}

static int bq27421_battery_probe(struct i2c_client *client,
				 const struct i2c_device_id *id)
{
	char *name;
	struct bq27421_device_info *di;
	struct bq27421_access_methods *bus;
	int num;
	int retval = 0;

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C))
		return -ENODEV;

	/* Get new ID for the new battery device */
	mutex_lock(&battery_mutex);
	num = idr_alloc(&battery_id, client, 0, 0, GFP_KERNEL);
	mutex_unlock(&battery_mutex);
	if (num < 0)
		return num;

	name = kasprintf(GFP_KERNEL, "%s-%d", id->name, num);
	if (!name) {
		dev_err(&client->dev, "failed to allocate device name\n");
		retval = -ENOMEM;
		goto batt_failed_1;
	}

	di = kzalloc(sizeof(*di), GFP_KERNEL);
	if (!di) {
		dev_err(&client->dev, "failed to allocate device info data\n");
		retval = -ENOMEM;
		goto batt_failed_2;
	}
	di->id = num;

	bus = kzalloc(sizeof(*bus), GFP_KERNEL);
	if (!bus) {
		dev_err(&client->dev, "failed to allocate access method "
					"data\n");
		retval = -ENOMEM;
		goto batt_failed_3;
	}

	init_waitqueue_head(&di->wait_q);
	init_waitqueue_head(&di->pm_resume_remain_q);
	atomic_set(&di->power_on_charge_control_enable, 1);
	atomic_set(&di->pm_resume_remain_condition, 1);

	di->volt_read_cnt = 0;

	di->sw_reset_counter = DEFAULT_SW_RESET_CYCLE;
	di->sw_reset_cycle = DEFAULT_SW_RESET_CYCLE;

	di->charge_voltage_step = 0;
	di->previous_charge_voltage_step = 0;

	i2c_set_clientdata(client, di);
	di->dev = &client->dev;
	bus->read = &bq27421_read_i2c;
	di->bus = bus;
	di->client = client;

	di->cm.dev = di->dev;

	charger_manager_init(&di->cm);

	retval = bq27421_powersupply_init(di);
	if (retval) {
		dev_err(&client->dev, "failed to setup bq27421\n");
		goto batt_failed_4;
	}

	retval = sysfs_create_group(&di->cm.battery_psy->dev.kobj, &fs_attr_group);
	if (retval)
		goto batt_failed_4;

	retval = bq27421_gpio_init(di);
	if (retval < 0)
		goto batt_failed_4;

	di->bq27421_polling = kthread_run(bq27421_polling_thread, client, "bq27421");
	if (IS_ERR(di->bq27421_polling)) {
		dev_err(di->dev, "failed to create polling thread\n");
		goto batt_failed_4;
	}
	return 0;

batt_failed_4:
	kfree(bus);
batt_failed_3:
	kfree(di);
batt_failed_2:
	kfree(name);
batt_failed_1:
	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, num);
	mutex_unlock(&battery_mutex);

	return retval;
}

static int bq27421_gpio_free(struct bq27421_device_info *di)
{
#ifdef ENABLE_GP_OUT
	gpio_free(GP_OUT_GPIO + GPIO_BASE);
#endif /* ENABLE_GP_OUT */
	return 0;
}

static int bq27421_battery_remove(struct i2c_client *client)
{
	struct bq27421_device_info *di = i2c_get_clientdata(client);

	bq27421_powersupply_unregister(di);

	kfree(di->bus);

	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, di->id);
	mutex_unlock(&battery_mutex);

	if (di->bq27421_polling) {
		kthread_stop(di->bq27421_polling);
		di->bq27421_polling = NULL;
	}
	bq27421_gpio_free(di);

	kfree(di);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int bq27421_pm_suspend(struct device *dev)
{
	struct bq27421_device_info *di = dev_get_drvdata(dev);

#ifdef ENABLE_GP_OUT
	enable_irq_wake(di->gp_out_wake_irq);
	enable_irq(di->gp_out_wake_irq);
#endif /* ENABLE_GP_OUT */
	dev_dbg(di->dev,"@@@@ %s wait_condition=%d\n",__FUNCTION__,atomic_read(&di->wait_condition));
	
	return 0;
}

static int bq27421_pm_resume(struct device *dev)
{
	struct bq27421_device_info *di = dev_get_drvdata(dev);

	atomic_set(&di->pm_resume_remain_condition, 0);
	dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&di->pm_resume_remain_condition));
	atomic_set(&di->wait_condition, 1);
	wake_up(&di->wait_q);
	dev_dbg(di->dev,"@@@@ %s wait_condition=%d\n",__FUNCTION__,atomic_read(&di->wait_condition));
#ifdef ENABLE_GP_OUT
	disable_irq_wake(di->gp_out_wake_irq);
	disable_irq_nosync(di->gp_out_wake_irq);
#endif /* ENABLE_GP_OUT */

	return 0;
}
#endif

#ifdef CONFIG_PM
static const struct dev_pm_ops bq27421_dev_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(bq27421_pm_suspend, bq27421_pm_resume)
};
#endif

static const struct i2c_device_id bq27421_id[] = {
	{ "bq27421", 1 },
	{},
};
MODULE_DEVICE_TABLE(i2c, BQ27421_id);

static struct i2c_driver bq27421_battery_driver = {
	.driver		= {
			.name = "bq27421-battery",
#ifdef CONFIG_PM
			.pm = &bq27421_dev_pm_ops,
#endif
	},
	.probe		= bq27421_battery_probe,
	.remove		= bq27421_battery_remove,
	.id_table	= bq27421_id,
};

static int __init bq27421_battery_init(void)
{
	int ret;

	ret = i2c_add_driver(&bq27421_battery_driver);
	if (ret)
		printk(KERN_ERR "Unable to register BQ27421 driver\n");

	return ret;
}
module_init(bq27421_battery_init);

static void __exit bq27421_battery_exit(void)
{
	i2c_del_driver(&bq27421_battery_driver);
}
module_exit(bq27421_battery_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC AccessTechinica, Ltd.");
MODULE_DESCRIPTION("BQ27421 battery monitor driver");
