// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2020 MediaTek Inc.
 * Copyright (c) 2020 - 2022, NEC Platforms, Ltd., All rights reserved.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/of_device.h>
#include <linux/of_irq.h>
#include <linux/of_gpio.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/mutex.h>
#include <linux/workqueue.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/pm_runtime.h>
#include <linux/power_supply.h>

#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/phy/phy.h>
#include <linux/reboot.h>

#ifdef CONFIG_REGMAP
#include <linux/regmap.h>
#endif

#if defined(CONFIG_PLATFORM_NEC)
/* boot type definitions */
enum boot_mode_t {
	NORMAL_BOOT = 0,
	META_BOOT = 1,
	RECOVERY_BOOT = 2,
	SW_REBOOT = 3,
	FACTORY_BOOT = 4,
	ADVMETA_BOOT = 5,
	ATE_FACTORY_BOOT = 6,
	ALARM_BOOT = 7,
	KERNEL_POWER_OFF_CHARGING_BOOT = 8,
	LOW_POWER_OFF_CHARGING_BOOT = 9,
	DONGLE_BOOT = 10,
	UNKNOWN_BOOT
};
#else /* defined(CONFIG_PLATFORM_NEC) */
#include <mt-plat/mtk_boot.h>
#endif /* defined(CONFIG_PLATFORM_NEC) */

#if defined(CONFIG_PLATFORM_NEC)
#include "rt9467_charger.h"
#else /* defined(CONFIG_PLATFORM_NEC) */
#include "rt9467.h"
#endif /* defined(CONFIG_PLATFORM_NEC) */
#define I2C_ACCESS_MAX_RETRY	5
#define RT9467_DRV_VERSION	"1.0.19_MTK"

/* ======================= */
/* RT9467 Parameter        */
/* ======================= */

#define PHY_MODE_BC11_SET 1
#define PHY_MODE_BC11_CLR 2

static const u32 rt9467_boost_oc_threshold[] = {
	500000, 700000, 1100000, 1300000, 1800000, 2100000, 2400000,
}; /* uA */

static const u32 rt9467_safety_timer[] = {
	4, 6, 8, 10, 12, 14, 16, 20,
}; /* hour */

enum rt9467_irq_idx {
	RT9467_IRQIDX_CHG_STATC = 0,
	RT9467_IRQIDX_CHG_FAULT,
	RT9467_IRQIDX_TS_STATC,
	RT9467_IRQIDX_CHG_IRQ1,
	RT9467_IRQIDX_CHG_IRQ2,
	RT9467_IRQIDX_CHG_IRQ3,
	RT9467_IRQIDX_DPDM_IRQ,
	RT9467_IRQIDX_MAX,
};

enum rt9467_irq_stat {
	RT9467_IRQSTAT_CHG_STATC = 0,
	RT9467_IRQSTAT_CHG_FAULT,
	RT9467_IRQSTAT_TS_STATC,
	RT9467_IRQSTAT_MAX,
};

enum rt9467_chg_type {
	RT9467_CHG_TYPE_NOVBUS = 0,
	RT9467_CHG_TYPE_UNDER_GOING,
	RT9467_CHG_TYPE_SDP,
	RT9467_CHG_TYPE_SDPNSTD,
	RT9467_CHG_TYPE_DCP,
	RT9467_CHG_TYPE_CDP,
	RT9467_CHG_TYPE_MAX,
};

enum rt9467_usbsw_state {
	RT9467_USBSW_CHG = 0,
	RT9467_USBSW_USB,
};

#if defined(CONFIG_PLATFORM_NEC)
static const u8 rt9467_irq_maskall[RT9467_IRQIDX_MAX] = {
	0xF0, /* 0x60 CHG_STATC_CTRL */
	0xF0, /* 0x61 CHG_FAULT_CTRL */
	0xFF, /* 0x62 TS_STATC_CTRL  */
	0xFF, /* 0x63 CHG_IRQ1_CTRL */
	0xFF, /* 0x64 CHG_IRQ2_CTRL */ /* Wd Enable = 0x08 */
	0xFF, /* 0x65 CHG_IRQ3_CTRL */
	0xFF, /* 0x66 DPDM_IRQ_CTRL */
};

#else /* defined(CONFIG_PLATFORM_NEC) */
static const u8 rt9467_irq_maskall[RT9467_IRQIDX_MAX] = {
	0xF0, 0xF0, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
};
#endif /* defined(CONFIG_PLATFORM_NEC) */

struct irq_mapping_tbl {
	const char *name;
	const int id;
};

#define RT9467_IRQ_MAPPING(_name, _id) {.name = #_name, .id = _id}
static const struct irq_mapping_tbl rt9467_irq_mapping_tbl[] = {
	RT9467_IRQ_MAPPING(chg_treg, 4),
	RT9467_IRQ_MAPPING(chg_aicr, 5),
	RT9467_IRQ_MAPPING(chg_mivr, 6),
	RT9467_IRQ_MAPPING(pwr_rdy, 7),
	RT9467_IRQ_MAPPING(chg_vsysuv, 12),
	RT9467_IRQ_MAPPING(chg_vsysov, 13),
	RT9467_IRQ_MAPPING(chg_vbatov, 14),
	RT9467_IRQ_MAPPING(chg_vbusov, 15),
	RT9467_IRQ_MAPPING(ts_batcold, 20),
	RT9467_IRQ_MAPPING(ts_batcool, 21),
	RT9467_IRQ_MAPPING(ts_batwarm, 22),
	RT9467_IRQ_MAPPING(ts_bathot, 23),
	RT9467_IRQ_MAPPING(ts_statci, 24),
	RT9467_IRQ_MAPPING(chg_faulti, 25),
	RT9467_IRQ_MAPPING(chg_statci, 26),
	RT9467_IRQ_MAPPING(chg_tmri, 27),
	RT9467_IRQ_MAPPING(chg_batabsi, 28),
	RT9467_IRQ_MAPPING(chg_adpbadi, 29),
	RT9467_IRQ_MAPPING(chg_rvpi, 30),
	RT9467_IRQ_MAPPING(otpi, 31),
	RT9467_IRQ_MAPPING(chg_aiclmeasi, 32),
	RT9467_IRQ_MAPPING(chg_ichgmeasi, 33),
	RT9467_IRQ_MAPPING(chgdet_donei, 34),
	RT9467_IRQ_MAPPING(wdtmri, 35),
	RT9467_IRQ_MAPPING(ssfinishi, 36),
	RT9467_IRQ_MAPPING(chg_rechgi, 37),
	RT9467_IRQ_MAPPING(chg_termi, 38),
	RT9467_IRQ_MAPPING(chg_ieoci, 39),
	RT9467_IRQ_MAPPING(adc_donei, 40),
	RT9467_IRQ_MAPPING(pumpx_donei, 41),
	RT9467_IRQ_MAPPING(bst_batuvi, 45),
	RT9467_IRQ_MAPPING(bst_midovi, 46),
	RT9467_IRQ_MAPPING(bst_olpi, 47),
	RT9467_IRQ_MAPPING(attachi, 48),
	RT9467_IRQ_MAPPING(detachi, 49),
	RT9467_IRQ_MAPPING(hvdcp_deti, 53),
	RT9467_IRQ_MAPPING(chgdeti, 54),
	RT9467_IRQ_MAPPING(dcdti, 55),
};

enum rt9467_charging_status {
	RT9467_CHG_STATUS_READY = 0,
	RT9467_CHG_STATUS_PROGRESS,
	RT9467_CHG_STATUS_DONE,
	RT9467_CHG_STATUS_FAULT,
	RT9467_CHG_STATUS_MAX,
};

static const u8 rt9467_val_en_hidden_mode[] = {
	0x49, 0x32, 0xB6, 0x27, 0x48, 0x18, 0x03, 0xE2,
};

enum rt9467_iin_limit_sel {
	RT9467_IINLMTSEL_3_2A = 0,
	RT9467_IINLMTSEL_CHG_TYP,
	RT9467_IINLMTSEL_AICR,
	RT9467_IINLMTSEL_LOWER_LEVEL, /* lower of above three */
};

enum rt9467_adc_sel {
	RT9467_ADC_VBUS_DIV5 = 1,
	RT9467_ADC_VBUS_DIV2,
	RT9467_ADC_VSYS,
	RT9467_ADC_VBAT,
	RT9467_ADC_TS_BAT = 6,
	RT9467_ADC_IBUS = 8,
	RT9467_ADC_IBAT,
	RT9467_ADC_REGN = 11,
	RT9467_ADC_TEMP_JC,
	RT9467_ADC_MAX,
};

/*
 * Unit for each ADC parameter
 * 0 stands for reserved
 * For TS_BAT, the real unit is 0.25.
 * Here we use 25, please remember to divide 100 while showing the value
 */
static const int rt9467_adc_unit[RT9467_ADC_MAX] = {
	0,
	RT9467_ADC_UNIT_VBUS_DIV5,
	RT9467_ADC_UNIT_VBUS_DIV2,
	RT9467_ADC_UNIT_VSYS,
	RT9467_ADC_UNIT_VBAT,
	0,
	RT9467_ADC_UNIT_TS_BAT,
	0,
	RT9467_ADC_UNIT_IBUS,
	RT9467_ADC_UNIT_IBAT,
	0,
	RT9467_ADC_UNIT_REGN,
	RT9467_ADC_UNIT_TEMP_JC,
};

static const int rt9467_adc_offset[RT9467_ADC_MAX] = {
	0,
	RT9467_ADC_OFFSET_VBUS_DIV5,
	RT9467_ADC_OFFSET_VBUS_DIV2,
	RT9467_ADC_OFFSET_VSYS,
	RT9467_ADC_OFFSET_VBAT,
	0,
	RT9467_ADC_OFFSET_TS_BAT,
	0,
	RT9467_ADC_OFFSET_IBUS,
	RT9467_ADC_OFFSET_IBAT,
	0,
	RT9467_ADC_OFFSET_REGN,
	RT9467_ADC_OFFSET_TEMP_JC,
};

struct rt9467_desc {
	u32 ichg;	/* uA */
	u32 ichg_dis_chg;/* uA */
	u32 aicr;	/* uA */
	u32 mivr;	/* uV */
	u32 cv;		/* uV */
	u32 ieoc;	/* uA */
	u32 safety_timer;	/* hour */
	u32 ircmp_resistor;	/* uohm */
	u32 ircmp_vclamp;	/* uV */
	u32 bc12_sel;
	bool en_te;
	bool en_wdt;
	bool en_irq_pulse;
	bool en_jeita;
	bool en_chgdet;
	int regmap_represent_slave_addr;
	const char *regmap_name;
	const char *chg_dev_name;
	bool ceb_invert;
};

/* These default values will be applied if there's no property in dts */
static struct rt9467_desc rt9467_default_desc = {
	.ichg = 2000000,	/* uA */
	.ichg_dis_chg = 2000000,/* uA */
	.aicr = 500000,		/* uA */
	.mivr = 4400000,	/* uV */
	.cv = 4350000,		/* uA */
	.ieoc = 250000,		/* uA */
	.safety_timer = 12,
#ifdef CONFIG_MTK_BIF_SUPPORT
	.ircmp_resistor = 0,		/* uohm */
	.ircmp_vclamp = 0,		/* uV */
#else
	.ircmp_resistor = 25000,	/* uohm */
	.ircmp_vclamp = 32000,		/* uV */
#endif /* CONFIG_MTK_BIF_SUPPORT */
	.bc12_sel = 0,
	.en_te = true,
	.en_wdt = true,
	.en_irq_pulse = false,
	.en_jeita = false,
	.en_chgdet = true,
	.regmap_represent_slave_addr = RT9467_SLAVE_ADDR,
	.regmap_name = "rt9467",
	.chg_dev_name = "primary_chg",
	.ceb_invert = false,
};

struct rt9467_info {
	struct i2c_client *client;
	struct mutex i2c_access_lock;
	struct mutex adc_access_lock;
	struct mutex irq_access_lock;
	struct mutex aicr_access_lock;
	struct mutex ichg_access_lock;
	struct mutex pe_access_lock;
	struct mutex hidden_mode_lock;
	struct mutex bc12_access_lock;
	struct mutex ieoc_lock;
	struct mutex tchg_lock;
	struct device *dev;
	struct rt9467_desc *desc;
	wait_queue_head_t wait_queue;
	int irq;
	int aicr_limit;
	u32 intr_gpio;
	u32 ceb_gpio;
	u8 chip_rev;
	u8 irq_flag[RT9467_IRQIDX_MAX];
	u8 irq_stat[RT9467_IRQSTAT_MAX];
	u8 irq_mask[RT9467_IRQIDX_MAX];
	u32 hidden_mode_cnt;
	struct regulator_dev *otg_rdev;
	bool bc12_en;
	bool pwr_rdy;
	u32 ieoc;
	u32 ichg;
	u32 ichg_dis_chg;
	bool ieoc_wkard;
	struct work_struct init_work;
	atomic_t bc12_sdp_cnt;
	atomic_t bc12_wkard;
	int tchg;

	u32 bootmode;
	u32 boottype;
	int psy_usb_type;
	/*power supply*/
	struct power_supply_desc psy_desc;
	/*for new framework get seft-power_supply*/
	struct power_supply *psy_self;

#if !defined(CONFIG_PLATFORM_NEC)
	struct work_struct chgdet_work;
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#ifdef CONFIG_REGMAP
	struct regmap *rm_dev;
#endif /* CONFIG_REGMAP */
#if defined(CONFIG_PLATFORM_NEC)
	int		ilim_det_result;
	struct 	power_supply *batt_psy;
	bool usb_detect;

	struct mutex 	float_v_lock;
	int				real_usb_psy_ma;
	int				prev_usb_psy_dis_ma;
	int				prev_usb_psy_sav_ma;
	bool			gadget_status;
	struct  power_supply *usb_psy;
	bool			chage_stop;
	bool			chg_enabled;
	bool			notify_src;
	int				cradle_status;
	int				detect_interval_sec;
	int				monitor_interval_sec;
	int				current_charging_voltage_step;
	char 			float_voltage;

	struct 	work_struct irq_work;
	struct 	delayed_work polling_work;
#define POLLING_INTERVAL_SEC	20	/* 20 Sec */
	int				polling_interval_sec;
#endif /* defined(CONFIG_PLATFORM_NEC) */
};

#if defined(CONFIG_PLATFORM_NEC)
struct init_reg_t {
	u8 reg;
	u8 val;
};
static struct init_reg_t hw_init_regs[] = {
	{0x00, 0x00},
	{0x01, 0x00},
	{0x02, 0x4B},
	{0x03, 0x22},
	{0x04, 0x3C},
	{0x05, 0x67},
	{0x06, 0x0B},
	{0x07, 0x44},
	{0x08, 0xA1},
	{0x09, 0x34},
	{0x0a, 0x58},
	{0x0b, 0x2C},
	{0x0c, 0x92},
	{0x0d, 0x72},
	{0x0e, 0x05},
	{0x0f, 0x00},
	{0x10, 0x16},
	{0x11, 0x00},
	{0x12, 0xD0},
	{0x13, 0x20},
	{0x14, 0x20},
	{0x18, 0x00},
	{0x19, 0x00},
	{0x1A, 0x40},
};

static char charge_voltage_table[] = {
	0x3C, /* 4.20V */
	0x34, /* 4.16V */
	0x2C, /* 4.128V */
	0x24, /* 4.08V */
	0x1C, /* 4.04V */
	0x14, /* 4.00V */
	0x0E, /* 3.97V */
};

#define CURRENT_100_MA		100
#define AICLR_100MA			0x00	/* 0x03 bit[7-2] : 000000 */
#define CURRENT_500_MA		500
#define AICLR_500MA			0x08	/* 0x03 bit[7-2] : 001000 */
#define CURRENT_1000_MA		1000
#define AICLR_1000MA		0x12	/* 0x03 bit[7-2] : 010010 */
#define CURRENT_1800_MA		1800
#define AICLR_1800MA		0x22	/* 0x03 bit[7-2] : 100010 */

static int rt9467_set_charging_voltage(struct rt9467_info *info, int step);
static int rt9467_battery_notify_usb_src_change(struct rt9467_info *info,int src_status);
static int rt9467_usb_notify_usb_src_change(struct rt9467_info *info,int src_status);
static int rt9467_get_prop_cradle_status(struct rt9467_info *info);
static int rt9467_get_prop_charge_status(struct rt9467_info *info);
static int rt9467_get_prop_usb_type_check(struct rt9467_info *info);
static int rt9467_get_prop_machine_status(struct rt9467_info *info);
static int rt9467_adc_read_temperature(struct rt9467_info *info);
static int rt9467_nec_hw_init_regs(struct rt9467_info *info, int recovery);
static int rt9467_enter_ship_mode(struct rt9467_info *info);
static int rt9467_enable_charger(struct rt9467_info *info, bool enable);
static int rt9467_set_input_current_limit(struct rt9467_info *info, int curr);
static int rt9467_get_input_current_limit(struct rt9467_info *info);

/* DEBUG */
#if 0
#undef pr_debug
#define pr_debug pr_info
#endif

/* #define WD_DEBUG */
#ifdef WD_DEBUG
static int wd_timeout_dbg = 0;
module_param(wd_timeout_dbg, int, 0644);
#endif /* WD_DEBUG */

static inline int rt9467_enable_wdt(struct rt9467_info *info, bool en);

#endif /* defined(CONFIG_PLATFORM_NEC) */

struct tag_bootmode {
	u32 size;
	u32 tag;
	u32 bootmode;
	u32 boottype;
};

/* ======================= */
/* Register Address        */
/* ======================= */

static const unsigned char rt9467_reg_addr[] = {
	RT9467_REG_CORE_CTRL0,
	RT9467_REG_CHG_CTRL1,
	RT9467_REG_CHG_CTRL2,
	RT9467_REG_CHG_CTRL3,
	RT9467_REG_CHG_CTRL4,
	RT9467_REG_CHG_CTRL5,
	RT9467_REG_CHG_CTRL6,
	RT9467_REG_CHG_CTRL7,
	RT9467_REG_CHG_CTRL8,
	RT9467_REG_CHG_CTRL9,
	RT9467_REG_CHG_CTRL10,
	RT9467_REG_CHG_CTRL11,
	RT9467_REG_CHG_CTRL12,
	RT9467_REG_CHG_CTRL13,
	RT9467_REG_CHG_CTRL14,
	RT9467_REG_CHG_CTRL15,
	RT9467_REG_CHG_CTRL16,
	RT9467_REG_CHG_ADC,
	RT9467_REG_CHG_DPDM1,
	RT9467_REG_CHG_DPDM2,
	RT9467_REG_CHG_DPDM3,
	RT9467_REG_CHG_CTRL19,
	RT9467_REG_CHG_CTRL17,
	RT9467_REG_CHG_CTRL18,
	RT9467_REG_DEVICE_ID,
	RT9467_REG_CHG_STAT,
	RT9467_REG_CHG_NTC,
	RT9467_REG_ADC_DATA_H,
	RT9467_REG_ADC_DATA_L,
	RT9467_REG_ADC_DATA_TUNE_H,
	RT9467_REG_ADC_DATA_TUNE_L,
	RT9467_REG_ADC_DATA_ORG_H,
	RT9467_REG_ADC_DATA_ORG_L,
	RT9467_REG_CHG_STATC,
	RT9467_REG_CHG_FAULT,
	RT9467_REG_TS_STATC,
	/* Skip IRQ evt to prevent reading clear while dumping registers */
	RT9467_REG_CHG_STATC_CTRL,
	RT9467_REG_CHG_FAULT_CTRL,
	RT9467_REG_TS_STATC_CTRL,
	RT9467_REG_CHG_IRQ1_CTRL,
	RT9467_REG_CHG_IRQ2_CTRL,
	RT9467_REG_CHG_IRQ3_CTRL,
	RT9467_REG_DPDM_IRQ_CTRL,
};

/* ========================= */
/* I2C operations            */
/* ========================= */

#ifdef CONFIG_REGMAP
static const struct regmap_config rt9467_regmap_config = {
	.reg_bits = 8,
	.val_bits = 8,
	.max_register = RT9467_REG_MAX,
	.cache_type = REGCACHE_NONE,
};

static int rt9467_register_regmap(struct rt9467_info *info)
{
	dev_info(info->dev, "%s\n", __func__);

	info->rm_dev = devm_regmap_init_i2c(info->client,
					    &rt9467_regmap_config);
	if (IS_ERR(info->rm_dev)) {
		dev_notice(info->dev, "%s fail(%ld)\n",
				      __func__, PTR_ERR(info->rm_dev));
		return -EIO;
	}

	return 0;
}
#endif /* CONFIG_REGMAP */

static int rt9467_device_read(void *client, u32 addr, int leng, void *dst)
{
	struct i2c_client *i2c = (struct i2c_client *)client;

	return i2c_smbus_read_i2c_block_data(i2c, addr, leng, dst);
}

static int rt9467_device_write(void *client, u32 addr, int leng,
	const void *src)
{
	struct i2c_client *i2c = (struct i2c_client *)client;

	return i2c_smbus_write_i2c_block_data(i2c, addr, leng, src);
}

static inline int __rt9467_i2c_write_byte(struct rt9467_info *info, u8 cmd,
	u8 data)
{
	int ret = 0, retry = 0;

	do {
#ifdef CONFIG_REGMAP
		ret = regmap_write(info->rm_dev, cmd, data);
#else
		ret = rt9467_device_write(info->client, cmd, 1, &data);
#endif /* CONFIG_REGMAP */
		retry++;
		if (ret < 0)
			udelay(10);
	} while (ret < 0 && retry < I2C_ACCESS_MAX_RETRY);

	if (ret < 0)
		dev_notice(info->dev, "%s: I2CW[0x%02X] = 0x%02X fail\n",
			__func__, cmd, data);
	else
		dev_dbg(info->dev, "%s: I2CW[0x%02X] = 0x%02X\n", __func__,
			cmd, data);

	return ret;
}

static int rt9467_i2c_write_byte(struct rt9467_info *info, u8 cmd, u8 data)
{
	int ret = 0;

	mutex_lock(&info->i2c_access_lock);
	ret = __rt9467_i2c_write_byte(info, cmd, data);
	mutex_unlock(&info->i2c_access_lock);

	return ret;
}

static inline int __rt9467_i2c_read_byte(struct rt9467_info *info, u8 cmd)
{
	int ret = 0, ret_val = 0, retry = 0;

	do {
#ifdef CONFIG_REGMAP
		ret = regmap_read(info->rm_dev, cmd, &ret_val);
#else
		ret = rt9467_device_read(info->client, cmd, 1, &ret_val);
#endif /* CONFIG_REGMAP */
		retry++;
		if (ret < 0)
			udelay(10);
	} while (ret < 0 && retry < I2C_ACCESS_MAX_RETRY);

	if (ret < 0) {
		dev_notice(info->dev, "%s: I2CR[0x%02X] fail\n", __func__, cmd);
		return ret;
	}

	ret_val = ret_val & 0xFF;

	dev_dbg(info->dev, "%s: I2CR[0x%02X] = 0x%02X\n", __func__, cmd,
		ret_val);

	return ret_val;
}

static int rt9467_i2c_read_byte(struct rt9467_info *info, u8 cmd)
{
	int ret = 0;

	mutex_lock(&info->i2c_access_lock);
	ret = __rt9467_i2c_read_byte(info, cmd);
	mutex_unlock(&info->i2c_access_lock);

	if (ret < 0)
		return ret;

	return (ret & 0xFF);
}

static inline int __rt9467_i2c_block_write(struct rt9467_info *info, u8 cmd,
	u32 leng, const u8 *data)
{
	int ret = 0;

#ifdef CONFIG_REGMAP
	ret = regmap_bulk_write(info->rm_dev, cmd, data, leng);
#else
	ret = rt9467_device_write(info->client, cmd, leng, data);
#endif /* CONFIG_REGMAP */

	return ret;
}


static int rt9467_i2c_block_write(struct rt9467_info *info, u8 cmd, u32 leng,
	const u8 *data)
{
	int ret = 0;

	mutex_lock(&info->i2c_access_lock);
	ret = __rt9467_i2c_block_write(info, cmd, leng, data);
	mutex_unlock(&info->i2c_access_lock);

	return ret;
}

static inline int __rt9467_i2c_block_read(struct rt9467_info *info, u8 cmd,
	u32 leng, u8 *data)
{
	int ret = 0;

#ifdef CONFIG_REGMAP
	ret = regmap_bulk_read(info->rm_dev, cmd, data, leng);
#else
	ret = rt9467_device_read(info->client, cmd, leng, data);
#endif /* CONFIG_REGMAP */

	return ret;
}


static int rt9467_i2c_block_read(struct rt9467_info *info, u8 cmd, u32 leng,
	u8 *data)
{
	int ret = 0;

	mutex_lock(&info->i2c_access_lock);
	ret = __rt9467_i2c_block_read(info, cmd, leng, data);
	mutex_unlock(&info->i2c_access_lock);

	return ret;
}


static int rt9467_i2c_test_bit(struct rt9467_info *info, u8 cmd, u8 shift,
	bool *is_one)
{
	int ret = 0;
	u8 data = 0;

	ret = rt9467_i2c_read_byte(info, cmd);
	if (ret < 0) {
		*is_one = false;
		return ret;
	}

	data = ret & (1 << shift);
	*is_one = (data == 0 ? false : true);

	return ret;
}

static int rt9467_i2c_update_bits(struct rt9467_info *info, u8 cmd, u8 data,
	u8 mask)
{
	int ret = 0;
	u8 reg_data = 0;

	mutex_lock(&info->i2c_access_lock);
	ret = __rt9467_i2c_read_byte(info, cmd);
	if (ret < 0) {
		mutex_unlock(&info->i2c_access_lock);
		return ret;
	}

	reg_data = ret & 0xFF;
	reg_data &= ~mask;
	reg_data |= (data & mask);

	ret = __rt9467_i2c_write_byte(info, cmd, reg_data);
	mutex_unlock(&info->i2c_access_lock);

	return ret;
}

static inline int rt9467_set_bit(struct rt9467_info *info, u8 reg, u8 mask)
{
	return rt9467_i2c_update_bits(info, reg, mask, mask);
}

static inline int rt9467_clr_bit(struct rt9467_info *info, u8 reg, u8 mask)
{
	return rt9467_i2c_update_bits(info, reg, 0x00, mask);
}

/* ================== */
/* Internal Functions */
/* ================== */
static int rt9467_get_charging_status(struct rt9467_info *info,
	enum rt9467_charging_status *chg_stat);

static inline int __rt9467_enable_hz(struct rt9467_info *info, bool en);
static int __rt9467_get_aicr(struct rt9467_info *info, u32 *aicr);
static int rt9467_enable_hidden_mode(struct rt9467_info *info, bool en);
#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_kick_wdt(struct rt9467_info *info);
static int __rt9467_set_ieoc(struct rt9467_info *info, u32 ieoc);
static int rt9467_get_ieoc(struct rt9467_info *info, u32 *ieoc);
static int __rt9467_set_ichg(struct rt9467_info *info, u32 ichg);
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline void rt9467_irq_set_flag(struct rt9467_info *info, u8 *irq,
	u8 mask)
{
	mutex_lock(&info->irq_access_lock);
	*irq |= mask;
	mutex_unlock(&info->irq_access_lock);
}

static inline void rt9467_irq_clr_flag(struct rt9467_info *info, u8 *irq,
	u8 mask)
{
	mutex_lock(&info->irq_access_lock);
	*irq &= ~mask;
	mutex_unlock(&info->irq_access_lock);
}

static inline const char *rt9467_get_irq_name(struct rt9467_info *info,
	int irqnum)
{
	int i = 0;

	for (i = 0; i < ARRAY_SIZE(rt9467_irq_mapping_tbl); i++) {
		if (rt9467_irq_mapping_tbl[i].id == irqnum)
			return rt9467_irq_mapping_tbl[i].name;
	}

	return "not found";
}

static inline void rt9467_irq_mask(struct rt9467_info *info, int irqnum)
{
	dev_dbg(info->dev, "%s: irq = %d, %s\n", __func__, irqnum,
		rt9467_get_irq_name(info, irqnum));
	info->irq_mask[irqnum / 8] |= (1 << (irqnum % 8));
}

static inline void rt9467_irq_unmask(struct rt9467_info *info, int irqnum)
{
	dev_dbg(info->dev, "%s: irq = %d, %s\n", __func__, irqnum,
		rt9467_get_irq_name(info, irqnum));
	info->irq_mask[irqnum / 8] &= ~(1 << (irqnum % 8));
}

static inline u8 rt9467_closest_reg(u32 min, u32 max, u32 step, u32 target)
{
	/* Smaller than minimum supported value, use minimum one */
	if (target < min)
		return 0;

	/* Greater than maximum supported value, use maximum one */
	if (target >= max)
		return (max - min) / step;

	return (target - min) / step;
}

static inline u8 rt9467_closest_reg_via_tbl(const u32 *tbl, u32 tbl_size,
	u32 target)
{
	u32 i = 0;

	/* Smaller than minimum supported value, use minimum one */
	if (target < tbl[0])
		return 0;

	for (i = 0; i < tbl_size - 1; i++) {
		if (target >= tbl[i] && target < tbl[i + 1])
			return i;
	}

	/* Greater than maximum supported value, use maximum one */
	return tbl_size - 1;
}

static inline u32 rt9467_closest_value(u32 min, u32 max, u32 step, u8 reg_val)
{
	u32 ret_val = 0;

	ret_val = min + reg_val * step;
	if (ret_val > max)
		ret_val = max;

	return ret_val;
}

static int __rt9467_get_ichg(struct rt9467_info *info, u32 *ichg);
static int rt9467_get_adc(struct rt9467_info *info,
	enum rt9467_adc_sel adc_sel, int *adc_val)
{
	int ret = 0, i = 0;
	const int max_wait_times = 6;
	u8 adc_data[6] = {0};
	u32 aicr = 0, ichg = 0;
	bool adc_start = false;

	mutex_lock(&info->adc_access_lock);

	rt9467_enable_hidden_mode(info, true);

	/* Select ADC to desired channel */
	/* RT9467_REG_CHG_ADC = 0x11 */
	ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_ADC,
		adc_sel << RT9467_SHIFT_ADC_IN_SEL, RT9467_MASK_ADC_IN_SEL);
	if (ret < 0) {
		dev_notice(info->dev, "%s: select ch to %d fail(%d)\n",
			__func__, adc_sel, ret);
		goto out;
	}

	/* Workaround for IBUS & IBAT */
	if (adc_sel == RT9467_ADC_IBUS) {
		mutex_lock(&info->aicr_access_lock);
		ret = __rt9467_get_aicr(info, &aicr);
		if (ret < 0) {
			dev_notice(info->dev, "%s: get aicr fail\n", __func__);
			goto out_unlock_all;
		}
	} else if (adc_sel == RT9467_ADC_IBAT) {
		mutex_lock(&info->ichg_access_lock);
		ret = __rt9467_get_ichg(info, &ichg);
		if (ret < 0) {
			dev_notice(info->dev, "%s: get ichg fail\n", __func__);
			goto out_unlock_all;
		}
	}

	/* Start ADC conversation */
	/* RT9467_REG_CHG_ADC = 0x11 */
	ret = rt9467_set_bit(info, RT9467_REG_CHG_ADC, RT9467_MASK_ADC_START);
	if (ret < 0) {
		dev_notice(info->dev, "%s: start con fail(%d), sel = %d\n",
			__func__, ret, adc_sel);
		goto out_unlock_all;
	}

	for (i = 0; i < max_wait_times; i++) {
		msleep(35);
		ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_ADC,
			RT9467_SHIFT_ADC_START, &adc_start);
		if (ret >= 0 && !adc_start)
			break;
	}

	if (i == max_wait_times) {
		dev_notice(info->dev, "%s: wait con fail(%d), sel = %d\n",
			__func__, ret, adc_sel);
		ret = -EINVAL;
		goto out_unlock_all;
	}

	mdelay(1);

	/* Read ADC data high/low byte */
	/* RT9467_REG_ADC_DATA_H =0x44 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_ADC_DATA_H, 6, adc_data);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read ADC data fail\n", __func__);
		goto out_unlock_all;
	}
	dev_dbg(info->dev,
		"%s: adc_tune = (0x%02X, 0x%02X), adc_org = (0x%02X, 0x%02X)\n",
		__func__, adc_data[2], adc_data[3], adc_data[4], adc_data[5]);

	/* Calculate ADC value */
	*adc_val = ((adc_data[0] << 8) + adc_data[1]) * rt9467_adc_unit[adc_sel]
		+ rt9467_adc_offset[adc_sel];

	dev_dbg(info->dev,
		"%s: adc_sel = %d, adc_h = 0x%02X, adc_l = 0x%02X, val = %d\n",
		__func__, adc_sel, adc_data[0], adc_data[1], *adc_val);

	ret = 0;

out_unlock_all:
	/* Coefficient of IBUS & IBAT */
	if (adc_sel == RT9467_ADC_IBUS) {
		if (aicr < 400000) /* 400mA */
			*adc_val = *adc_val * 67 / 100;
		mutex_unlock(&info->aicr_access_lock);
	} else if (adc_sel == RT9467_ADC_IBAT) {
		if (ichg >= 100000 && ichg <= 450000) /* 100~450mA */
			*adc_val = *adc_val * 57 / 100;
		else if (ichg >= 500000 && ichg <= 850000) /* 500~850mA */
			*adc_val = *adc_val * 63 / 100;
		mutex_unlock(&info->ichg_access_lock);
	}

out:
	rt9467_enable_hidden_mode(info, false);
	mutex_unlock(&info->adc_access_lock);
	return ret;
}

#if !defined(CONFIG_PLATFORM_NEC)
static bool is_usb_rdy(struct device *dev)
{
	struct device_node *node;
	bool ready = false;

	node = of_parse_phandle(dev->of_node, "usb", 0);
	if (node) {
		ready = of_property_read_bool(node, "gadget-ready");
		dev_info(dev, "gadget-ready=%d\n", ready);
	} else
		dev_info(dev, "usb node missing or invalid\n");

	return ready;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int DPDM_Switch_TO_CHG_upstream(struct rt9467_info *info,
						bool switch_to_chg)
{
	struct phy *phy;
	int mode = 0;
	int ret;

	mode = switch_to_chg ? PHY_MODE_BC11_SET : PHY_MODE_BC11_CLR;
	phy = phy_get(info->dev, "usb2-phy");
	if (IS_ERR_OR_NULL(phy)) {
		dev_info(info->dev, "phy_get fail\n");
		return -EINVAL;
	}

	ret = phy_set_mode_ext(phy, PHY_MODE_USB_DEVICE, mode);
	if (ret)
		dev_info(info->dev, "phy_set_mode_ext fail\n");

	phy_put(phy);

	return 0;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_set_usbsw_state(struct rt9467_info *info, int state)
{
	dev_info(info->dev, "%s: state = %d\n", __func__, state);

	if (state == RT9467_USBSW_CHG)
		DPDM_Switch_TO_CHG_upstream(info, true);
	else
		DPDM_Switch_TO_CHG_upstream(info, false);

	return 0;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline int __rt9467_enable_chgdet_flow(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	int ret = 0;
	enum rt9467_usbsw_state usbsw =
		en ? RT9467_USBSW_CHG : RT9467_USBSW_USB;

	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	rt9467_set_usbsw_state(info, usbsw);
	/* RT9467_REG_CHG_DPDM1 = 0x12 */
	ret = (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_DPDM1, RT9467_MASK_USBCHGEN);
	if (ret >= 0)
		info->bc12_en = en;

	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int rt9467_enable_chgdet_flow(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	int ret = 0, i = 0;
	bool pwr_rdy = false;
	const int max_wait_cnt = 200;

	dev_info(info->dev, "%s: en = %d\n", __func__, en);

	if (en) {
		/* Workaround for CDP port */
		for (i = 0; i < max_wait_cnt; i++) {
			if (is_usb_rdy(info->dev))
				break;
			dev_dbg(info->dev, "%s: CDP block\n", __func__);
			/* RT9467_REG_CHG_STATC = 0x50 */
			ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
				RT9467_SHIFT_PWR_RDY, &pwr_rdy);
			if (ret >= 0 && !pwr_rdy) {
				dev_info(info->dev, "%s: plug out\n",
					__func__);
				return 0;
			}
			msleep(100);
		}
		if (i == max_wait_cnt)
			dev_notice(info->dev, "%s: CDP timeout\n", __func__);
		else
			dev_info(info->dev, "%s: CDP free\n", __func__);
	}

	mutex_lock(&info->bc12_access_lock);
	ret = __rt9467_enable_chgdet_flow(info, en);
	mutex_unlock(&info->bc12_access_lock);
	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

#if 0
static int rt9467_inform_psy_changed(struct rt9467_info *info)
{
	int ret = 0;
	union power_supply_propval propval;

	dev_info(info->dev, "%s: pwr_rdy = %d, type = %d\n", __func__,
		info->pwr_rdy, info->chg_type);

	/* inform chg det power supply */
	propval.intval = info->pwr_rdy;
	ret = power_supply_set_property(info->psy_self, POWER_SUPPLY_PROP_ONLINE,
		&propval);
	if (ret < 0)
		dev_notice(info->dev, "%s: psy online fail(%d)\n", __func__,
			ret);

	propval.intval = info->chg_type;
	ret = power_supply_set_property(info->psy_self,
		POWER_SUPPLY_PROP_CHARGE_TYPE, &propval);
	if (ret < 0)
		dev_notice(info->dev, "%s: psy type fail(%d)\n", __func__, ret);

	return ret;
}
#endif

static inline int rt9467_enable_ilim(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL3 = 0x03 */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL3, RT9467_MASK_ILIM_EN);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static inline int rt9467_toggle_chgdet_flow(struct rt9467_info *info)
{
	int ret = 0;
	u8 data = 0;

	/* read data */
	/* RT9467_REG_CHG_DPDM1 = 0x12 */
	ret = i2c_smbus_read_i2c_block_data(info->client, RT9467_REG_CHG_DPDM1,
		1, &data);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read usbd fail\n", __func__);
		goto out;
	}

	/* usbd off */
	data &= ~RT9467_MASK_USBCHGEN;
	/* RT9467_REG_CHG_DPDM1 = 0x12 */
	ret = i2c_smbus_write_i2c_block_data(info->client, RT9467_REG_CHG_DPDM1,
		1, &data);
	if (ret < 0) {
		dev_notice(info->dev, "%s: usbd off fail\n", __func__);
		goto out;
	}

	dev_info(info->dev, "%s usbd off\n", __func__);
	udelay(40);

	/* usbd on */
	data |= RT9467_MASK_USBCHGEN;
	/* RT9467_REG_CHG_DPDM1 = 0x12 */
	ret = i2c_smbus_write_i2c_block_data(info->client, RT9467_REG_CHG_DPDM1,
		1, &data);
	if (ret < 0)
		dev_notice(info->dev, "%s: usbd on fail\n", __func__);
	dev_info(info->dev, "%s usbd on\n", __func__);
out:

	return ret;
}

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_bc12_sdp_workaround(struct rt9467_info *info)
{
	int ret = 0;

	dev_info(info->dev, "%s\n", __func__);

	mutex_lock(&info->i2c_access_lock);

	ret = rt9467_toggle_chgdet_flow(info);
	if (ret < 0)
		goto err;

	mdelay(10);

	ret = rt9467_toggle_chgdet_flow(info);
	if (ret < 0)
		goto err;

	goto out;
err:
	dev_notice(info->dev, "%s: fail\n", __func__);
out:
	mutex_unlock(&info->i2c_access_lock);
	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_chgdet_handler(struct rt9467_info *info)
{
	int ret = 0;
	bool pwr_rdy = false, inform_psy = true;
	u8 usb_status = 0;

	dev_info(info->dev, "%s\n", __func__);

	/* disabled by user, do nothing */
	if (!info->desc->en_chgdet) {
		dev_info(info->dev, "%s: bc12 is disabled by dts\n", __func__);
		return 0;
	}

	/* check power ready */
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
		RT9467_SHIFT_PWR_RDY, &pwr_rdy);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read pwr rdy state fail\n",
			__func__);
		return ret;
	}

	/* no change in pwr_rdy state */
	if (info->pwr_rdy == pwr_rdy &&
		atomic_read(&info->bc12_wkard) == 0) {
		dev_info(info->dev, "%s: pwr_rdy(%d) state is the same\n",
			__func__, pwr_rdy);
		inform_psy = false;
		goto out;
	}
	info->pwr_rdy = pwr_rdy;

	/* plug out */
	if (!pwr_rdy) {
		dev_info(info->dev, "%s: Charger Type: UNKONWN\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_UNKNOWN;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
		atomic_set(&info->bc12_sdp_cnt, 0);
		goto out;
	}

	/* plug in */
	/* RT9467_REG_CHG_DPDM2 = 0x13 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_DPDM2);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read type fail\n", __func__);
		return ret;
	}
	dev_info(info->dev,"%s: RT9467_REG_CHG_DPDM2 = 0x%x\n", __func__,ret);
	usb_status = (ret & RT9467_MASK_USB_STATUS) >> RT9467_SHIFT_USB_STATUS;

	switch (usb_status) {
	case RT9467_CHG_TYPE_UNDER_GOING:
		dev_info(info->dev, "%s: under going...\n", __func__);
		return ret;
	case RT9467_CHG_TYPE_SDP:
		dev_info(info->dev,
			  "%s: Charger Type: STANDARD_HOST\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_SDP;
		break;
	case RT9467_CHG_TYPE_SDPNSTD:
		dev_info(info->dev,
			  "%s: Charger Type: NONSTANDARD_CHARGER\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB_FLOAT;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
		break;
	case RT9467_CHG_TYPE_CDP:
		dev_info(info->dev,
			  "%s: Charger Type: CHARGING_HOST\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB_CDP;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_CDP;
		break;
	case RT9467_CHG_TYPE_DCP:
		dev_info(info->dev,
			  "%s: Charger Type: STANDARD_CHARGER\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB_DCP;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
		break;
	}

	/* BC12 workaround (NONSTD -> STP) */
	if (atomic_read(&info->bc12_sdp_cnt) < 2 &&
	    info->psy_usb_type == POWER_SUPPLY_USB_TYPE_SDP) {
		ret = rt9467_bc12_sdp_workaround(info);
		/* Workaround success, wait for next event */
		if (ret >= 0) {
			atomic_inc(&info->bc12_sdp_cnt);
			atomic_set(&info->bc12_wkard, 1);
			return ret;
		}
		goto out;
	}
out:
	atomic_set(&info->bc12_wkard, 0);

	/* turn off USB charger detection */
	ret = __rt9467_enable_chgdet_flow(info, false);
	if (ret < 0)
		dev_notice(info->dev, "%s: disable chrdet fail\n", __func__);

	if (inform_psy)
		power_supply_changed(info->psy_self);

	return 0;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_chgdet_handler(struct rt9467_info *info)
{
	int ret = 0;

	mutex_lock(&info->bc12_access_lock);
	ret = __rt9467_chgdet_handler(info);
	mutex_unlock(&info->bc12_access_lock);

	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_set_aicr(struct rt9467_info *info, u32 aicr)
{
	u8 reg_aicr = 0;

	reg_aicr = rt9467_closest_reg(RT9467_AICR_MIN, RT9467_AICR_MAX,
		RT9467_AICR_STEP, aicr);

	dev_info(info->dev, "%s: aicr = %d(0x%02X)\n", __func__, aicr,
		reg_aicr);

	/* RT9467_REG_CHG_CTRL3 = 0x03 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL3,
		reg_aicr << RT9467_SHIFT_AICR, RT9467_MASK_AICR);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static void rt9467_chgdet_work_handler(struct work_struct *work)
{
	int ret = 0;
	bool pwr_rdy = false;
	struct rt9467_info *info = (struct rt9467_info *)container_of(work,
		struct rt9467_info, chgdet_work);

	/* Check power ready */
	/* RT9467_REG_CHG_STATC = 0x50*/
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
		RT9467_SHIFT_PWR_RDY, &pwr_rdy);

	if (!pwr_rdy)
		return;

	/* Enable USB charger type detection */
	ret = rt9467_enable_chgdet_flow(info, true);
	if (ret < 0)
		dev_notice(info->dev, "%s: enable usb chrdet fail\n", __func__);

}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

/* Prevent back boost */
static int rt9467_toggle_cfo(struct rt9467_info *info)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	int ret = 0;
	u8 data = 0;

	dev_info(info->dev, "%s\n", __func__);
	mutex_lock(&info->i2c_access_lock);
	/* RT9467_REG_CHG_CTRL2 = 0x02 */
	ret = rt9467_device_read(info->client, RT9467_REG_CHG_CTRL2, 1, &data);
	if (ret < 0) {
		dev_notice(info->dev, "%s read cfo fail(%d)\n", __func__, ret);
		goto out;
	}

	/* CFO off */
	data &= ~RT9467_MASK_CFO_EN;
	ret = rt9467_device_write(info->client, RT9467_REG_CHG_CTRL2, 1, &data);
	if (ret < 0) {
		dev_notice(info->dev, "%s cfo off fail(%d)\n", __func__, ret);
		goto out;
	}

	/* CFO on */
	data |= RT9467_MASK_CFO_EN;
	ret = rt9467_device_write(info->client, RT9467_REG_CHG_CTRL2, 1, &data);
	if (ret < 0)
		dev_notice(info->dev, "%s cfo on fail(%d)\n", __func__, ret);

out:
	mutex_unlock(&info->i2c_access_lock);
	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

/* IRQ handlers */
static int rt9467_pwr_rdy_irq_handler(struct rt9467_info *info)
{
	int ret = 0;
	bool pwr_rdy = false;

	dev_notice(info->dev, "%s\n", __func__);
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
		RT9467_SHIFT_PWR_RDY, &pwr_rdy);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read pwr rdy fail\n", __func__);
		goto out;
	}

#if defined(CONFIG_PLATFORM_NEC)
	info->pwr_rdy = pwr_rdy;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	if (!pwr_rdy) {
		dev_info(info->dev, "%s: pwr rdy = 0\n", __func__);
		goto out;
	}

	ret = rt9467_enable_chgdet_flow(info, true);
	if (ret < 0)
		dev_notice(info->dev, "%s: en chgdet fail(%d)\n", __func__,
			ret);

out:
	return 0;
}

static int rt9467_chg_mivr_irq_handler(struct rt9467_info *info)
{
	int ret = 0;
	bool mivr_act = false;
	int adc_ibus = 0;

	dev_notice(info->dev, "%s\n", __func__);

	/* Check whether MIVR loop is active */
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
		RT9467_SHIFT_CHG_MIVR, &mivr_act);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read mivr stat fail\n", __func__);
		goto out;
	}

	if (!mivr_act) {
		dev_info(info->dev, "%s: mivr loop is not active\n", __func__);
		goto out;
	}

	if (strcmp(info->desc->chg_dev_name, "primary_chg") == 0) {
		/* Check IBUS ADC */
		ret = rt9467_get_adc(info, RT9467_ADC_IBUS, &adc_ibus);
		if (ret < 0) {
			dev_notice(info->dev, "%s: get ibus fail\n", __func__);
			return ret;
		}
		if (adc_ibus < 100000) { /* 100mA */
			ret = rt9467_toggle_cfo(info);
			return ret;
		}
	}
out:
	return 0;
}

static int rt9467_chg_aicr_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_treg_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_vsysuv_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_vsysov_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_vbatov_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_vbusov_irq_handler(struct rt9467_info *info)
{
	int ret = 0;
	bool vbusov = false;

	dev_notice(info->dev, "%s\n", __func__);
	/* RT9467_REG_CHG_FAULT = 0x51 */
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_FAULT,
		RT9467_SHIFT_VBUSOV, &vbusov);
	if (ret < 0)
		return ret;

	return 0;
}

static int rt9467_ts_bat_cold_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_ts_bat_cool_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_ts_bat_warm_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_ts_bat_hot_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_ts_statci_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_faulti_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_statci_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_tmri_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_batabsi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_adpbadi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_rvpi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_otpi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_aiclmeasi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	rt9467_irq_set_flag(info, &info->irq_flag[RT9467_IRQIDX_CHG_IRQ2],
		RT9467_MASK_CHG_AICLMEASI);
	wake_up_interruptible(&info->wait_queue);
	return 0;
}

static int rt9467_chg_ichgmeasi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chgdet_donei_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_wdtmri_irq_handler(struct rt9467_info *info)
{
	int ret = 0;
#if defined(CONFIG_PLATFORM_NEC)
#ifdef WD_DEBUG
	if (wd_timeout_dbg == 1) {
		printk(KERN_ERR "@@@ wd_timuout_dbg enable\n");
		wd_timeout_dbg = 0;
	}
#endif /* WD_DEBUG */
	dev_err(info->dev, "@@@ WD timeout!!\n");
	ret = rt9467_nec_hw_init_regs(info,1);
	if (ret < 0) {
		dev_err(info->dev, "Failed to nec_hw_init\n");
	}
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_notice(info->dev, "%s\n", __func__);
	ret = __rt9467_kick_wdt(info);
	if (ret < 0)
		dev_notice(info->dev, "%s: kick wdt fail\n", __func__);
#endif /* defined(CONFIG_PLATFORM_NEC) */

	return ret;
}

static int rt9467_ssfinishi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_rechgi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	power_supply_changed(info->psy_self);
	return 0;
}

static int rt9467_chg_termi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_chg_ieoci_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	power_supply_changed(info->psy_self);
	return 0;
}

static int rt9467_adc_donei_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_pumpx_donei_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_bst_batuvi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_bst_midovi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_bst_olpi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_attachi_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
#if defined(CONFIG_PLATFORM_NEC)
	schedule_work(&info->irq_work);
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */

	/* check bc12_en state */
	mutex_lock(&info->bc12_access_lock);
	if (!info->bc12_en) {
		dev_notice(info->dev, "%s: bc12 disabled, ignore irq\n",
			__func__);
		mutex_unlock(&info->bc12_access_lock);
		return 0;
	}
	mutex_unlock(&info->bc12_access_lock);
	return rt9467_chgdet_handler(info);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int rt9467_detachi_irq_handler(struct rt9467_info *info)
{
	int ret = 0;

#if defined(CONFIG_PLATFORM_NEC)
	schedule_work(&info->irq_work);
	return ret;
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_notice(info->dev, "%s\n", __func__);
	ret = rt9467_chgdet_handler(info);
	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int rt9467_chgdeti_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

static int rt9467_dcdti_irq_handler(struct rt9467_info *info)
{
	dev_notice(info->dev, "%s\n", __func__);
	return 0;
}

typedef int (*rt9467_irq_fptr)(struct rt9467_info *);
static rt9467_irq_fptr rt9467_irq_handler_tbl[56] = {
/* 0x60 CHG_STATC_CTRL */
	NULL,
	NULL,
	NULL,
	NULL,
	rt9467_chg_treg_irq_handler,
	rt9467_chg_aicr_irq_handler,
	rt9467_chg_mivr_irq_handler,
	rt9467_pwr_rdy_irq_handler,	/* Enable */
/* 0x61 CHG_FAULT_CTRL */
	NULL,
	NULL,
	NULL,
	NULL,
	rt9467_chg_vsysuv_irq_handler,
	rt9467_chg_vsysov_irq_handler,
	rt9467_chg_vbatov_irq_handler,
	rt9467_chg_vbusov_irq_handler,
/* 0x62 TS_STATC_CTRL  */
	NULL,
	NULL,
	NULL,
	NULL,
	rt9467_ts_bat_cold_irq_handler,
	rt9467_ts_bat_cool_irq_handler,
	rt9467_ts_bat_warm_irq_handler,
	rt9467_ts_bat_hot_irq_handler,
/* 0x63 CHG_IRQ1_CTRL */
	rt9467_ts_statci_irq_handler,
	rt9467_chg_faulti_irq_handler,
	rt9467_chg_statci_irq_handler,
	rt9467_chg_tmri_irq_handler,
	rt9467_chg_batabsi_irq_handler,
	rt9467_chg_adpbadi_irq_handler,
	rt9467_chg_rvpi_irq_handler,
	rt9467_chg_otpi_irq_handler,
/* 0x64 CHG_IRQ2_CTRL */
	rt9467_chg_aiclmeasi_irq_handler,
	rt9467_chg_ichgmeasi_irq_handler,
	rt9467_chgdet_donei_irq_handler,
	rt9467_wdtmri_irq_handler,	/* Enable */
	rt9467_ssfinishi_irq_handler,
	rt9467_chg_rechgi_irq_handler,
	rt9467_chg_termi_irq_handler,
	rt9467_chg_ieoci_irq_handler,
/* 0x65 CHG_IRQ3_CTRL */
	rt9467_adc_donei_irq_handler,
	rt9467_pumpx_donei_irq_handler,
	NULL,
	NULL,
	NULL,
	rt9467_bst_batuvi_irq_handler,
	rt9467_bst_midovi_irq_handler,
	rt9467_bst_olpi_irq_handler,
/* 0x66 DPDM_IRQ_CTRL */
	rt9467_attachi_irq_handler,	/* Enable */
	rt9467_detachi_irq_handler,	/* Enable */
	NULL,
	NULL,
	NULL,
	rt9467_chgdeti_irq_handler,
	rt9467_dcdti_irq_handler,
};

static inline int rt9467_enable_irqrez(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL13 = 0x0D */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL13, RT9467_MASK_IRQ_REZ);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int __rt9467_irq_handler(struct rt9467_info *info)
{
	int ret = 0, i = 0, j = 0;
	u8 evt[RT9467_IRQIDX_MAX] = {0};
	u8 mask[RT9467_IRQIDX_MAX] = {0};
	u8 stat[RT9467_IRQSTAT_MAX] = {0};
	u8 usb_status_old = 0, usb_status_new = 0;

	dev_info(info->dev, "%s\n", __func__);

	/* Read DPDM status before reading evts */
	/* RT9467_REG_CHG_DPDM2 = 0x13 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_DPDM2);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read type fail\n", __func__);
		goto err_read_irq;
	}
	usb_status_old = (ret & RT9467_MASK_USB_STATUS) >>
		RT9467_SHIFT_USB_STATUS;

	/* Read event and skip CHG_IRQ3 */
	/* RT9467_REG_CHG_IRQ1 = 0x53 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_CHG_IRQ1, 2, &evt[3]);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read evt1 fail(%d)\n", __func__,
			ret);
		goto err_read_irq;
	}

	/* RT9467_REG_DPDM_IRQ = 0x56 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_DPDM_IRQ, 1, &evt[6]);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read evt2 fail(%d)\n", __func__,
			ret);
		goto err_read_irq;
	}

	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_CHG_STATC, 3, evt);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read stat fail(%d)\n", __func__,
			ret);
		goto err_read_irq;
	}

	/* Read DPDM status after reading evts */
	/* RT9467_REG_CHG_DPDM2 = 0x13 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_DPDM2);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read type fail\n", __func__);
		goto err_read_irq;
	}
	usb_status_new = (ret & RT9467_MASK_USB_STATUS) >>
		RT9467_SHIFT_USB_STATUS;

	/* Read mask */
	/* RT9467_REG_CHG_STATC_CTRL = 0x60 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_CHG_STATC_CTRL,
		ARRAY_SIZE(mask), mask);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read mask fail(%d)\n", __func__,
			ret);
		goto err_read_irq;
	}

	/* Detach */
	if (usb_status_old != RT9467_CHG_TYPE_NOVBUS &&
		usb_status_new == RT9467_CHG_TYPE_NOVBUS)
		evt[RT9467_IRQIDX_DPDM_IRQ] |= 0x02;

	/* Attach */
	if (usb_status_new >= RT9467_CHG_TYPE_SDP &&
		usb_status_new <= RT9467_CHG_TYPE_CDP &&
		usb_status_old != usb_status_new)
		evt[RT9467_IRQIDX_DPDM_IRQ] |= 0x01;

	/* Store/Update stat */
	memcpy(stat, info->irq_stat, RT9467_IRQSTAT_MAX);

	for (i = 0; i < RT9467_IRQIDX_MAX; i++) {
		evt[i] &= ~mask[i];
		if (i < RT9467_IRQSTAT_MAX) {
			info->irq_stat[i] = evt[i];
			evt[i] ^= stat[i];
		}
		for (j = 0; j < 8; j++) {
			if (!(evt[i] & (1 << j)))
				continue;
			if (rt9467_irq_handler_tbl[i * 8 + j])
				rt9467_irq_handler_tbl[i * 8 + j](info);
		}
	}

err_read_irq:
	return ret;
}

static irqreturn_t rt9467_irq_handler(int irq, void *data)
{
	int ret = 0;
	struct rt9467_info *info = (struct rt9467_info *)data;

	dev_info(info->dev, "%s\n", __func__);

	ret = __rt9467_irq_handler(info);
	ret = rt9467_enable_irqrez(info, true);
	if (ret < 0)
		dev_notice(info->dev, "%s: en irqrez fail\n", __func__);

	return IRQ_HANDLED;
}

static int rt9467_irq_register(struct rt9467_info *info)
{
	int ret = 0, len = 0;
	char *name = NULL;

	if (strcmp(info->desc->chg_dev_name, "secondary_chg") == 0)
		return 0;

	dev_info(info->dev, "%s\n", __func__);

	/* request gpio */
	len = strlen(info->desc->chg_dev_name);
	name = devm_kzalloc(info->dev, len + 10, GFP_KERNEL);
	snprintf(name,  len + 10, "%s_irq_gpio", info->desc->chg_dev_name);
	ret = devm_gpio_request_one(info->dev, info->intr_gpio, GPIOF_IN, name);
	if (ret < 0) {
		dev_notice(info->dev, "%s: gpio request fail\n", __func__);
		return ret;
	}

	ret = gpio_to_irq(info->intr_gpio);
	if (ret < 0) {
		dev_notice(info->dev, "%s: irq mapping fail\n", __func__);
		return ret;
	}
	info->irq = ret;
	dev_info(info->dev, "%s: irq = %d\n", __func__, info->irq);

	/* Request threaded IRQ */
	name = devm_kzalloc(info->dev, len + 5, GFP_KERNEL);
	snprintf(name, len + 5, "%s_irq", info->desc->chg_dev_name);
	ret = devm_request_threaded_irq(info->dev, info->irq, NULL,
		rt9467_irq_handler, IRQF_TRIGGER_FALLING | IRQF_ONESHOT, name,
		info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: request thread irq fail\n",
			__func__);
		return ret;
	}
	device_init_wakeup(info->dev, true);

	return 0;
}

static inline int rt9467_maskall_irq(struct rt9467_info *info)
{
	dev_info(info->dev, "%s\n", __func__);
	/* RT9467_REG_CHG_STATC_CTRL = 0x60 */
	return rt9467_i2c_block_write(info, RT9467_REG_CHG_STATC_CTRL,
		ARRAY_SIZE(rt9467_irq_maskall), rt9467_irq_maskall);
}

static inline int rt9467_irq_init(struct rt9467_info *info)
{
	dev_info(info->dev, "%s\n", __func__);
	/* RT9467_REG_CHG_STATC_CTRL = 0x60 */
	return rt9467_i2c_block_write(info, RT9467_REG_CHG_STATC_CTRL,
		ARRAY_SIZE(info->irq_mask), info->irq_mask);
}

static bool rt9467_is_hw_exist(struct rt9467_info *info)
{
	int ret = 0;
	u8 vendor_id = 0, chip_rev = 0;

	/* RT9467_REG_DEVICE_ID = 0x40*/
	ret = i2c_smbus_read_byte_data(info->client, RT9467_REG_DEVICE_ID);
	if (ret < 0)
		return false;

	vendor_id = ret & 0xF0;
	chip_rev = ret & 0x0F;
	if (vendor_id != RT9467_VENDOR_ID) {
		dev_notice(info->dev, "%s: vendor id is incorrect (0x%02X)\n",
			__func__, vendor_id);
		return false;
	}

	dev_info(info->dev, "%s: 0x%02X\n", __func__, chip_rev);
	info->chip_rev = chip_rev;

	return true;
}

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_set_safety_timer(struct rt9467_info *info, u32 hr)
{
	u8 reg_st = 0;

	reg_st = rt9467_closest_reg_via_tbl(rt9467_safety_timer,
		ARRAY_SIZE(rt9467_safety_timer), hr);

	dev_info(info->dev, "%s: time = %d(0x%02X)\n", __func__, hr, reg_st);

	/* RT9467_REG_CHG_CTRL12 = 0x0C */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL12,
		reg_st << RT9467_SHIFT_WT_FC, RT9467_MASK_WT_FC);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline int rt9467_enable_wdt(struct rt9467_info *info, bool en)
{
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL13 = 0x0D */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL13, RT9467_MASK_WDT_EN);
}

#if !defined(CONFIG_PLATFORM_NEC)
static inline int rt9467_select_input_current_limit(struct rt9467_info *info,
	enum rt9467_iin_limit_sel sel)
{
	dev_info(info->dev, "%s: sel = %d\n", __func__, sel);
	/* RT9467_REG_CHG_CTRL2 = 0x02 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL2,
		sel << RT9467_SHIFT_IINLMTSEL, RT9467_MASK_IINLMTSEL);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static int rt9467_enable_hidden_mode(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	int ret = 0;

	mutex_lock(&info->hidden_mode_lock);

	if (en) {
		if (info->hidden_mode_cnt == 0) {
			ret = rt9467_i2c_block_write(info, 0x70,
				ARRAY_SIZE(rt9467_val_en_hidden_mode),
				rt9467_val_en_hidden_mode);
			if (ret < 0)
				goto err;
		}
		info->hidden_mode_cnt++;
	} else {
		if (info->hidden_mode_cnt == 1) /* last one */
			ret = rt9467_i2c_write_byte(info, 0x70, 0x00);
		info->hidden_mode_cnt--;
		if (ret < 0)
			goto err;
	}
	dev_dbg(info->dev, "%s: en = %d\n", __func__, en);
	goto out;

err:
	dev_notice(info->dev, "%s: en = %d fail(%d)\n", __func__, en, ret);
out:
	mutex_unlock(&info->hidden_mode_lock);
	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_set_iprec(struct rt9467_info *info, u32 iprec)
{
	u8 reg_iprec = 0;

	reg_iprec = rt9467_closest_reg(RT9467_IPREC_MIN, RT9467_IPREC_MAX,
		RT9467_IPREC_STEP, iprec);

	dev_info(info->dev, "%s: iprec = %d(0x%02X)\n", __func__, iprec,
		reg_iprec);

	/* RT9467_REG_CHG_CTRL8 = 0x08 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL8,
		reg_iprec << RT9467_SHIFT_IPREC, RT9467_MASK_IPREC);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_sw_workaround(struct rt9467_info *info)
{
	int ret = 0;

	dev_info(info->dev, "%s\n", __func__);

	rt9467_enable_hidden_mode(info, true);

	/* Modify UG driver */
	/* RT9467_REG_CHG_HIDDEN_CTRL4 = 0x23 */
	ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_HIDDEN_CTRL4, 0xC0,
		0xF0);
	if (ret < 0)
		dev_notice(info->dev, "%s: set UG driver fail\n", __func__);

	/* RT9467_REG_CHG_HIDDEN_CTRL4 = 0x23 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_HIDDEN_CTRL4);
	dev_info(info->dev, "%s: reg0x23 = 0x%02X\n", __func__, ret);

	/* Disable TS auto sensing */
	/* RT9467_REG_CHG_HIDDEN_CTRL15 = 0x2E */
	ret = rt9467_clr_bit(info, RT9467_REG_CHG_HIDDEN_CTRL15, 0x01);
	if (ret < 0)
		goto out;

	/* Set precharge current to 850mA, only do this in normal boot */
	if (info->chip_rev <= RT9467_CHIP_REV_E3) {
		/* Worst case delay: wait auto sensing */
		msleep(200);

		if (info->bootmode == NORMAL_BOOT) {
			ret = rt9467_set_iprec(info, 850000);
			if (ret < 0)
				goto out;

			/* Increase Isys drop threshold to 2.5A */
			/* RT9467_REG_CHG_HIDDEN_CTRL7 = 0x26 */
			ret = rt9467_i2c_write_byte(info,
				RT9467_REG_CHG_HIDDEN_CTRL7, 0x1c);
			if (ret < 0)
				goto out;
		}
	}

	/* Only revision <= E1 needs the following workaround */
	if (info->chip_rev > RT9467_CHIP_REV_E1)
		goto out;

	/* ICC: modify sensing node, make it more accurate */
	/* RT9467_REG_CHG_HIDDEN_CTRL8 = 0x27 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_HIDDEN_CTRL8, 0x00);
	if (ret < 0)
		goto out;

	/* DIMIN level */
	/* RT9467_REG_CHG_HIDDEN_CTRL8 = 0x27 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_HIDDEN_CTRL9, 0x86);

out:
	rt9467_enable_hidden_mode(info, false);
	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_enable_chg_type_det(struct rt9467_info *info, bool en)
{
	int ret = 0;
	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_kick_wdt(struct rt9467_info *info)
{
	enum rt9467_charging_status chg_status = RT9467_CHG_STATUS_READY;

	/* Any I2C communication can reset watchdog timer */
	return rt9467_get_charging_status(info, &chg_status);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline int __rt9467_enable_hz(struct rt9467_info *info, bool en)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL1 = 0x01 */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL1, RT9467_MASK_HZ_EN);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

/* Reset all registers' value to default */
static int rt9467_reset_chip(struct rt9467_info *info)
{
	int ret = 0;

	dev_info(info->dev, "%s\n", __func__);

	/* disable hz before reset chip */
	ret = __rt9467_enable_hz(info, false);
	if (ret < 0) {
		dev_notice(info->dev, "%s: disable hz fail\n", __func__);
		return ret;
	}

	/* RT9467_REG_CORE_CTRL0 = 0x00 */
	ret =  rt9467_set_bit(info, RT9467_REG_CORE_CTRL0, RT9467_MASK_RST);
	if (ret < 0)
		return ret;
#ifdef CONFIG_REGMAP
	regcache_mark_dirty(info->rm_dev);
#endif /* CONFIG_REGMAP */
	return ret;
}

#if !defined(CONFIG_PLATFORM_NEC)
static inline int rt9467_enable_jeita(struct rt9467_info *info, bool en)
{
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL16 = 0x10 */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL16, RT9467_MASK_JEITA_EN);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static inline int __rt9467_enable_te(struct rt9467_info *info, bool en)
{
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL2 = 0x02 */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL2, RT9467_MASK_TE_EN);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static inline int __rt9467_enable_safety_timer(struct rt9467_info *info,
	bool en)
{
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL12 = 0x0C */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL12, RT9467_MASK_TMR_EN);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_set_ieoc(struct rt9467_info *info, u32 ieoc)
{
	int ret = 0;
	u8 reg_ieoc = 0;

	/* IEOC workaround */
	if (info->ieoc_wkard)
		ieoc += 100000; /* 100mA */

	reg_ieoc = rt9467_closest_reg(RT9467_IEOC_MIN, RT9467_IEOC_MAX,
		RT9467_IEOC_STEP, ieoc);

	/* RT9467_REG_CHG_CTRL9 = 0x09 */
	ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL9,
		reg_ieoc << RT9467_SHIFT_IEOC, RT9467_MASK_IEOC);

	/* Store IEOC */
	return rt9467_get_ieoc(info, &info->ieoc);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_set_mivr(struct rt9467_info *info, u32 mivr)
{
	u8 reg_mivr = 0;

	reg_mivr = rt9467_closest_reg(RT9467_MIVR_MIN, RT9467_MIVR_MAX,
		RT9467_MIVR_STEP, mivr);

	dev_info(info->dev, "%s: mivr = %d(0x%02X)\n", __func__, mivr,
		reg_mivr);

	/* RT9467_REG_CHG_CTRL6 = 0x06 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL6,
		reg_mivr << RT9467_SHIFT_MIVR, RT9467_MASK_MIVR);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static int rt9467_get_charging_status(struct rt9467_info *info,
	enum rt9467_charging_status *chg_stat)
{
	int ret = 0;

	/* RT9467_REG_CHG_STAT = 0x42 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_STAT);
	if (ret < 0)
		return ret;

	*chg_stat = (ret & RT9467_MASK_CHG_STAT) >> RT9467_SHIFT_CHG_STAT;

	return ret;
}

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_get_ieoc(struct rt9467_info *info, u32 *ieoc)
{
	int ret = 0;
	u8 reg_ieoc = 0;

	/* RT9467_REG_CHG_CTRL9 = 0x09 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_CTRL9);
	if (ret < 0)
		return ret;

	reg_ieoc = (ret & RT9467_MASK_IEOC) >> RT9467_SHIFT_IEOC;
	*ieoc = rt9467_closest_value(RT9467_IEOC_MIN, RT9467_IEOC_MAX,
		RT9467_IEOC_STEP, reg_ieoc);

	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline int __rt9467_is_charging_enable(struct rt9467_info *info,
	bool *en)
{
	/* RT9467_REG_CHG_CTRL2 = 0x02 */
	return rt9467_i2c_test_bit(info, RT9467_REG_CHG_CTRL2,
		RT9467_SHIFT_CHG_EN, en);
}

static int __rt9467_get_ichg(struct rt9467_info *info, u32 *ichg)
{
	int ret = 0;
	u8 reg_ichg = 0;

	/* RT9467_REG_CHG_CTRL7 = 0x07 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_CTRL7);
	if (ret < 0)
		return ret;

	reg_ichg = (ret & RT9467_MASK_ICHG) >> RT9467_SHIFT_ICHG;
	*ichg = rt9467_closest_value(RT9467_ICHG_MIN, RT9467_ICHG_MAX,
		RT9467_ICHG_STEP, reg_ichg);

	return ret;
}

static int __rt9467_get_aicr(struct rt9467_info *info, u32 *aicr)
{
	int ret = 0;
	u8 reg_aicr = 0;

	/* RT9467_REG_CHG_CTRL3 = 0x03 */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_CTRL3);
	if (ret < 0)
		return ret;

	reg_aicr = (ret & RT9467_MASK_AICR) >> RT9467_SHIFT_AICR;
	*aicr = rt9467_closest_value(RT9467_AICR_MIN, RT9467_AICR_MAX,
		RT9467_AICR_STEP, reg_aicr);

	return ret;
}

#if !defined(CONFIG_PLATFORM_NEC)
static inline int rt9467_ichg_workaround(struct rt9467_info *info, u32 uA)
{
	int ret = 0;

	/* Vsys short protection */
	rt9467_enable_hidden_mode(info, true);

	/* RT9467_REG_CHG_HIDDEN_CTRL7 = 0x26 */
	if (info->ichg >= 900000 && uA < 900000)
		ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_HIDDEN_CTRL7,
			0x00, 0x60);
	else if (uA >= 900000 && info->ichg < 900000)
		ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_HIDDEN_CTRL7,
			0x40, 0x60);

	rt9467_enable_hidden_mode(info, false);
	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_set_ichg(struct rt9467_info *info, u32 ichg)
{
	int ret = 0;
	u8 reg_ichg = 0;

	if (strcmp(info->desc->chg_dev_name, "primary_chg") == 0) {
		ichg = (ichg < 500000) ? 500000 : ichg;
		rt9467_ichg_workaround(info, ichg);
	}

	reg_ichg = rt9467_closest_reg(RT9467_ICHG_MIN, RT9467_ICHG_MAX,
		RT9467_ICHG_STEP, ichg);

	dev_info(info->dev, "%s: ichg = %d(0x%02X)\n", __func__, ichg,
		reg_ichg);

	/* RT9467_REG_CHG_CTRL7 = 0x07 */
	ret = rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL7,
		reg_ichg << RT9467_SHIFT_ICHG, RT9467_MASK_ICHG);
	if (ret < 0)
		return ret;

	/* Store Ichg setting */
	__rt9467_get_ichg(info, &info->ichg);

	/* Workaround to make IEOC accurate */
	if (ichg < 900000 && !info->ieoc_wkard) { /* 900mA */
		ret = __rt9467_set_ieoc(info, info->ieoc + 100000);
		info->ieoc_wkard = true;
	} else if (ichg >= 900000 && info->ieoc_wkard) {
		info->ieoc_wkard = false;
		ret = __rt9467_set_ieoc(info, info->ieoc - 100000);
	}

	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int __rt9467_set_cv(struct rt9467_info *info, u32 cv)
{
	u8 reg_cv = 0;

	reg_cv = rt9467_closest_reg(RT9467_CV_MIN, RT9467_CV_MAX,
		RT9467_CV_STEP, cv);

	dev_info(info->dev, "%s: cv = %d(0x%02X)\n", __func__, cv, reg_cv);

	/* RT9467_REG_CHG_CTRL4 = 0x04 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL4,
		reg_cv << RT9467_SHIFT_CV, RT9467_MASK_CV);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_set_ircmp_resistor(struct rt9467_info *info, u32 uohm)
{
	u8 reg_resistor = 0;

	reg_resistor = rt9467_closest_reg(RT9467_IRCMP_RES_MIN,
		RT9467_IRCMP_RES_MAX, RT9467_IRCMP_RES_STEP, uohm);

	dev_info(info->dev, "%s: resistor = %d(0x%02X)\n", __func__, uohm,
		reg_resistor);

	/* RT9467_REG_CHG_CTRL18 = 0x1A */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL18,
		reg_resistor << RT9467_SHIFT_IRCMP_RES, RT9467_MASK_IRCMP_RES);
}
#endif /* defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_set_ircmp_vclamp(struct rt9467_info *info, u32 uV)
{
	u8 reg_vclamp = 0;

	reg_vclamp = rt9467_closest_reg(RT9467_IRCMP_VCLAMP_MIN,
		RT9467_IRCMP_VCLAMP_MAX, RT9467_IRCMP_VCLAMP_STEP, uV);

	dev_info(info->dev, "%s: vclamp = %d(0x%02X)\n", __func__, uV,
		reg_vclamp);

	/* RT9467_REG_CHG_CTRL18 = 0x1A */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL18,
		reg_vclamp << RT9467_SHIFT_IRCMP_VCLAMP,
		RT9467_MASK_IRCMP_VCLAMP);
}
#endif /* defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static inline int rt9467_enable_irq_pulse(struct rt9467_info *info, bool en)
{
	dev_info(info->dev, "%s: en = %d\n", __func__, en);
	/* RT9467_REG_CHG_CTRL1 = 0x01 */
	return (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL1, RT9467_MASK_IRQ_PULSE);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

static inline int rt9467_get_irq_number(struct rt9467_info *info,
	const char *name)
{
	int i = 0;

	if (!name) {
		dev_notice(info->dev, "%s: null name\n", __func__);
		return -EINVAL;
	}

	for (i = 0; i < ARRAY_SIZE(rt9467_irq_mapping_tbl); i++) {
		if (!strcmp(name, rt9467_irq_mapping_tbl[i].name))
			return rt9467_irq_mapping_tbl[i].id;
	}

	return -EINVAL;
}

static int rt9467_parse_dt(struct rt9467_info *info, struct device *dev)
{
	int ret = 0, irq_cnt = 0;
	struct rt9467_desc *desc = NULL;
	struct device_node *np = dev->of_node;
	const char *name = NULL;
	int irqnum = 0;

	dev_info(info->dev, "%s\n", __func__);

	if (!np) {
		dev_notice(info->dev, "%s: no device node\n", __func__);
		return -EINVAL;
	}

	info->desc = &rt9467_default_desc;

	desc = devm_kzalloc(dev, sizeof(struct rt9467_desc), GFP_KERNEL);
	if (!desc)
		return -ENOMEM;
	memcpy(desc, &rt9467_default_desc, sizeof(struct rt9467_desc));

	if (of_property_read_string(np, "charger_name",
		&desc->chg_dev_name) < 0)
		dev_notice(info->dev, "%s: no charger name\n", __func__);

#if (!defined(CONFIG_MTK_GPIO) || defined(CONFIG_MTK_GPIOLIB_STAND))
	ret = of_get_named_gpio(np, "rt,intr_gpio", 0);
	if (ret < 0)
		return ret;
	info->intr_gpio = ret;
	if (strcmp(desc->chg_dev_name, "secondary_chg") == 0) {
		ret = of_get_named_gpio(np, "rt,ceb_gpio", 0);
		if (ret < 0)
			return ret;
		info->ceb_gpio = ret;
	}
#else
	ret = of_property_read_u32(np, "rt,intr_gpio_num", &info->intr_gpio);
	if (ret < 0)
		return ret;
	if (strcmp(desc->chg_dev_name, "secondary_chg") == 0) {
		ret = of_property_read_u32(np, "rt,ceb_gpio_num",
			&info->ceb_gpio);
		if (ret < 0)
			return ret;
	}
#endif

	dev_info(info->dev, "%s: intr gpio = %d\n", __func__,
		info->intr_gpio);

	dev_info(info->dev, "%s: intr/ceb gpio = %d, %d\n", __func__,
		info->intr_gpio, info->ceb_gpio);

	/* request ceb gpio for secondary charger */
	if (strcmp(desc->chg_dev_name, "secondary_chg") == 0) {
		ret = devm_gpio_request_one(info->dev, info->ceb_gpio,
			GPIOF_DIR_OUT, "rt9467_sec_ceb_gpio");
		if (ret < 0) {
			dev_notice(info->dev, "%s: ceb gpio request fail\n",
				__func__);
			return ret;
		}
	}

	if (of_property_read_u32(np, "regmap_represent_slave_addr",
		&desc->regmap_represent_slave_addr) < 0)
		dev_notice(info->dev, "%s: no regmap slave addr\n", __func__);

	if (of_property_read_string(np, "regmap_name",
		&(desc->regmap_name)) < 0)
		dev_notice(info->dev, "%s: no regmap name\n", __func__);

	if (of_property_read_u32(np, "ichg", &desc->ichg) < 0)
		dev_notice(info->dev, "%s: no ichg\n", __func__);

	if (of_property_read_u32(np, "aicr", &desc->aicr) < 0)
		dev_notice(info->dev, "%s: no aicr\n", __func__);

	if (of_property_read_u32(np, "mivr", &desc->mivr) < 0)
		dev_notice(info->dev, "%s: no mivr\n", __func__);

	if (of_property_read_u32(np, "cv", &desc->cv) < 0)
		dev_notice(info->dev, "%s: no cv\n", __func__);

	if (of_property_read_u32(np, "ieoc", &desc->ieoc) < 0)
		dev_notice(info->dev, "%s: no ieoc\n", __func__);

	if (of_property_read_u32(np, "safety_timer", &desc->safety_timer) < 0)
		dev_notice(info->dev, "%s: no safety timer\n", __func__);

	if (of_property_read_u32(np, "ircmp_resistor",
		&desc->ircmp_resistor) < 0)
		dev_notice(info->dev, "%s: no ircmp resistor\n", __func__);

	if (of_property_read_u32(np, "ircmp_vclamp", &desc->ircmp_vclamp) < 0)
		dev_notice(info->dev, "%s: no ircmp vclamp\n", __func__);

	if (of_property_read_u32(np, "bc12_sel", &desc->bc12_sel) < 0)
		dev_notice(info->dev, "%s: no bc12_sel\n", __func__);

	desc->en_te = of_property_read_bool(np, "en_te");
	desc->en_wdt = of_property_read_bool(np, "en_wdt");
	desc->en_irq_pulse = of_property_read_bool(np, "en_irq_pulse");
	desc->en_jeita = of_property_read_bool(np, "en_jeita");
	desc->ceb_invert = of_property_read_bool(np, "ceb_invert");
	desc->en_chgdet = of_property_read_bool(np, "en_chgdet");

	while (true) {
		ret = of_property_read_string_index(np, "interrupt-names",
			irq_cnt, &name);
		if (ret < 0)
			break;
		dev_info(info->dev, "%s, name = %s\n", __func__, name);
		irq_cnt++;
		irqnum = rt9467_get_irq_number(info, name);
		if (irqnum >= 0)
			rt9467_irq_unmask(info, irqnum);
	}

	info->desc = desc;

	return 0;
}


/* =========================================================== */
/* Released interfaces                                         */
/* =========================================================== */

static int rt9467_set_boost_current_limit(struct rt9467_info *info,
	u32 current_limit)
{
	u8 reg_ilimit = 0;
	int ret = 0;

	reg_ilimit = rt9467_closest_reg_via_tbl(rt9467_boost_oc_threshold,
		ARRAY_SIZE(rt9467_boost_oc_threshold), current_limit);

	dev_info(info->dev, "%s: boost ilimit = %d(0x%02X)\n", __func__,
		current_limit, reg_ilimit);

	/*adjust inducer ocp level with different current limit*/
	/* RT9467_REG_CHG_CTRL13 = 0x0D */
	ret = ((current_limit >= 2100000) ? rt9467_set_bit :rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL13, RT9467_MASK_OCP);
	if (ret < 0) {
		dev_notice(info->dev, "%s: set ocp fail\n", __func__);
		return ret;
	}

	/* RT9467_REG_CHG_CTRL10 = 0x0A */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL10,
		reg_ilimit << RT9467_SHIFT_BOOST_OC, RT9467_MASK_BOOST_OC);
}

static int rt9467_enable_otg(struct rt9467_info *info, bool en)
{
	int ret = 0;
	bool en_otg = false;
	u8 hidden_val = en ? 0x00 : 0x0F;
	u8 lg_slew_rate = en ? 0xCC : 0xC3;

	dev_info(info->dev, "%s: en = %d\n", __func__, en);

	rt9467_enable_hidden_mode(info, true);

	/* Set OTG_OC to 1300mA */
	ret = rt9467_set_boost_current_limit(info, 1300000);
	if (ret < 0) {
		dev_notice(info->dev, "%s: set current limit fail\n", __func__);
		return ret;
	}

	/*
	 * Woraround : slow Low side mos Gate driver slew rate
	 * for decline VBUS noise
	 * reg[0x23] = 0xCC after entering OTG mode
	 * reg[0x23] = 0xC3 after leaving OTG mode
	 */
	/* RT9467_REG_CHG_HIDDEN_CTRL4 = 0x23 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_HIDDEN_CTRL4,
		lg_slew_rate);
	if (ret < 0) {
		dev_notice(info->dev,
			"%s: set Low side mos Gate drive speed fail(%d)\n",
			__func__, ret);
		goto out;
	}

	/* Enable WDT */
	if (en && info->desc->en_wdt) {
		ret = rt9467_enable_wdt(info, true);
		if (ret < 0) {
			dev_notice(info->dev, "%s: en wdt fail\n", __func__);
			goto err_en_otg;
		}
	}

	/* Switch OPA mode */
	/* RT9467_REG_CHG_CTRL1 = 0x01 */
	ret = (en ? rt9467_set_bit : rt9467_clr_bit)
		(info, RT9467_REG_CHG_CTRL1, RT9467_MASK_OPA_MODE);

	msleep(20);

	if (en) {
		/* RT9467_REG_CHG_CTRL1 = 0x01 */
		ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_CTRL1,
			RT9467_SHIFT_OPA_MODE, &en_otg);
		if (ret < 0 || !en_otg) {
			dev_notice(info->dev, "%s: otg fail(%d)\n", __func__,
				ret);
			goto err_en_otg;
		}
	}

	/*
	 * Woraround reg[0x25] = 0x00 after entering OTG mode
	 * reg[0x25] = 0x0F after leaving OTG mode
	 */
	/* RT9467_REG_CHG_HIDDEN_CTRL6 = 0x25 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_HIDDEN_CTRL6,
		hidden_val);
	if (ret < 0)
		dev_notice(info->dev, "%s: workaroud fail(%d)\n", __func__,
			ret);

	/* Disable WDT */
	if (!en) {
		ret = rt9467_enable_wdt(info, false);
		if (ret < 0)
			dev_notice(info->dev, "%s: disable wdt fail\n",
				__func__);
	}
	goto out;

err_en_otg:
	/* Disable WDT */
	ret = rt9467_enable_wdt(info, false);
	if (ret < 0)
		dev_notice(info->dev, "%s: disable wdt fail\n", __func__);

	/* Recover Low side mos Gate slew rate */
	/* RT9467_REG_CHG_HIDDEN_CTRL4 = 0x23 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_HIDDEN_CTRL4, 0x73);
	if (ret < 0)
		dev_notice(info->dev,
			"%s: recover Low side mos Gate drive speed fail(%d)\n",
			__func__, ret);
	ret = -EIO;
out:
	rt9467_enable_hidden_mode(info, false);
	return ret;
}

static int rt9467_enable_discharge(struct rt9467_info *info, bool en)
{
	int ret = 0, i = 0;
	const int check_dischg_max = 3;
	bool is_dischg = true;

	dev_info(info->dev, "%s: en = %d\n", __func__, en);

	ret = rt9467_enable_hidden_mode(info, true);
	if (ret < 0)
		return ret;

	/* Set bit2 of reg[0x21] to 1 to enable discharging */
	/* RT9467_REG_CHG_HIDDEN_CTRL2 = 0x21 */
	ret = (en ? rt9467_set_bit : rt9467_clr_bit)(info,
		RT9467_REG_CHG_HIDDEN_CTRL2, 0x04);
	if (ret < 0) {
		dev_notice(info->dev, "%s: en = %d, fail\n", __func__, en);
		return ret;
	}

	if (!en) {
		for (i = 0; i < check_dischg_max; i++) {
			/* RT9467_REG_CHG_HIDDEN_CTRL2 = 0x21 */
			ret = rt9467_i2c_test_bit(info,
				RT9467_REG_CHG_HIDDEN_CTRL2, 2, &is_dischg);
			if (ret >= 0 && !is_dischg)
				break;
			/* Disable discharging */
			/* RT9467_REG_CHG_HIDDEN_CTRL2 = 0x21 */
			ret = rt9467_clr_bit(info, RT9467_REG_CHG_HIDDEN_CTRL2,
				0x04);
		}
		if (i == check_dischg_max)
			dev_notice(info->dev, "%s: disable dischg fail(%d)\n",
				__func__, ret);
	}

	rt9467_enable_hidden_mode(info, false);
	return ret;
}

static int __rt9467_enable_auto_sensing(struct rt9467_info *info, bool en)
{
	int ret = 0;
	u8 auto_sense = 0;
	u8 *data = 0x00;

	/* enter hidden mode */
	/* RT9467_REG_MAX = 0x70 */
	ret = rt9467_device_write(info->client, 0x70,
		ARRAY_SIZE(rt9467_val_en_hidden_mode),
		rt9467_val_en_hidden_mode);
	if (ret < 0)
		return ret;

	/* RT9467_REG_CHG_HIDDEN_CTRL15 = 0x2E */
	ret = rt9467_device_read(info->client, RT9467_REG_CHG_HIDDEN_CTRL15, 1,
		&auto_sense);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read auto sense fail\n", __func__);
		goto out;
	}

	if (!en)
		auto_sense &= 0xFE; /* clear bit0 */
	else
		auto_sense |= 0x01; /* set bit0 */
	/* RT9467_REG_CHG_HIDDEN_CTRL15 = 0x2E */
	ret = rt9467_device_write(info->client, RT9467_REG_CHG_HIDDEN_CTRL15, 1,
		&auto_sense);
	if (ret < 0)
		dev_notice(info->dev, "%s: en = %d fail\n", __func__, en);

out:
	/* RT9467_REG_MAX = 0x70 */
	return rt9467_device_write(info->client, 0x70, 1, &data);
}

/*
 * This function is used in shutdown function
 * Use i2c smbus directly
 */
static int rt9467_sw_reset(struct rt9467_info *info)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	int ret = 0;
	u8 evt[RT9467_IRQIDX_MAX] = {0};

	/* Register 0x01 ~ 0x10 */
	u8 reg_data[] = {
		0x10, 0x03, 0x23, 0x3C, 0x67, 0x0B, 0x4C, 0xA1,
		0x3C, 0x58, 0x2C, 0x02, 0x52, 0x05, 0x00, 0x10
	};

	dev_info(info->dev, "%s\n", __func__);

	/* Disable auto sensing/Enable HZ,ship mode of secondary charger */
	if (strcmp(info->desc->chg_dev_name, "secondary_chg") == 0) {
		mutex_lock(&info->hidden_mode_lock);
		mutex_lock(&info->i2c_access_lock);
		__rt9467_enable_auto_sensing(info, false);
		mutex_unlock(&info->i2c_access_lock);
		mutex_unlock(&info->hidden_mode_lock);

		mdelay(50);
		reg_data[0] = 0x14; /* HZ */
		reg_data[1] = 0x83; /* Shipping mode */
	}

	/* Mask all irq */
	mutex_lock(&info->i2c_access_lock);
	/* RT9467_REG_CHG_STATC_CTRL = 0x60 */
	ret = rt9467_device_write(info->client, RT9467_REG_CHG_STATC_CTRL,
		ARRAY_SIZE(rt9467_irq_maskall), rt9467_irq_maskall);
	if (ret < 0)
		dev_notice(info->dev, "%s: mask all irq fail\n", __func__);

	/* Read all irq */
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_device_read(info->client, RT9467_REG_CHG_STATC, 5, evt);
	if (ret < 0)
		dev_notice(info->dev, "%s: read evt1 fail(%d)\n", __func__,
			ret);

	/* RT9467_REG_DPDM_IRQ = 0x56 */
	ret = rt9467_device_read(info->client, RT9467_REG_DPDM_IRQ, 1, &evt[6]);
	if (ret < 0)
		dev_notice(info->dev, "%s: read evt2 fail(%d)\n", __func__,
			ret);

	/* Reset necessary registers */
	/* RT9467_REG_CHG_CTRL1 = 0x01 */
	ret = rt9467_device_write(info->client, RT9467_REG_CHG_CTRL1,
		ARRAY_SIZE(reg_data), reg_data);
	if (ret < 0)
		dev_notice(info->dev, "%s: reset registers fail\n", __func__);
	mutex_unlock(&info->i2c_access_lock);

	return ret;
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

#if !defined(CONFIG_PLATFORM_NEC)
static void rt9467_get_boot_mode(struct rt9467_info *info)
{
	struct device_node *boot_node = NULL;
	struct tag_bootmode *tag = NULL;

	boot_node = of_parse_phandle(info->dev->of_node, "bootmode", 0);
	if (!boot_node)
		dev_info(info->dev, "%s: failed to get boot mode phandle\n",
			 __func__);
	else {
		tag = (struct tag_bootmode *)of_get_property(boot_node,
							"atag,boot", NULL);
		if (!tag)
			dev_info(info->dev, "%s: failed to get atag,boot\n",
				 __func__);
		else {
			dev_info(info->dev,
			"%s: size:0x%x tag:0x%x bootmode:0x%x boottype:0x%x\n",
				__func__, tag->size, tag->tag,
				tag->bootmode, tag->boottype);
			info->bootmode = tag->bootmode;
			info->boottype = tag->boottype;
		}
	}
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if !defined(CONFIG_PLATFORM_NEC)
static int rt9467_init_setting(struct rt9467_info *info)
{
	int ret = 0;
	struct rt9467_desc *desc = info->desc;
	u8 evt[RT9467_IRQIDX_MAX] = {0};

	dev_info(info->dev, "%s\n", __func__);

	/*get boot mode*/
	rt9467_get_boot_mode(info);

	/* RT9467_REG_CHG_CTRL2 = 0x02 */
	ret = rt9467_clr_bit(info, RT9467_REG_CHG_CTRL2, RT9467_MASK_CHG_EN);
	if (ret < 0) {	
		dev_notice(info->dev, "%s: disable chg_en fail\n",
			__func__);
		goto err;
	}

	/* disable USB charger type detection before reset IRQ */
	ret = rt9467_enable_chgdet_flow(info, false);
	if (ret < 0) {
		dev_notice(info->dev, "%s: disable usb chrdet fail\n",
			__func__);
		goto err;
	}

	/* RT9467_REG_CHG_DPDM1 = 0x12 */
	ret = rt9467_clr_bit(info, RT9467_REG_CHG_DPDM1, 0x40);
	if (ret < 0) {
		dev_notice(info->dev, "%s: disable attach delay fail\n",
			__func__);
		goto err;
	}

	/* mask all irq */
	ret = rt9467_maskall_irq(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: mask all irq fail\n", __func__);
		goto err;
	}

	/* clear event */
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_block_read(info, RT9467_REG_CHG_STATC, ARRAY_SIZE(evt),
		evt);
	if (ret < 0) {
		dev_notice(info->dev, "%s: clr evt fail(%d)\n", __func__, ret);
		goto err;
	}

	ret = __rt9467_set_ichg(info, desc->ichg);
	if (ret < 0)
		dev_notice(info->dev, "%s: set ichg fail\n", __func__);

	ret = __rt9467_set_aicr(info, desc->aicr);
	if (ret < 0)
		dev_notice(info->dev, "%s: set aicr fail\n", __func__);

	ret = __rt9467_set_mivr(info, desc->mivr);
	if (ret < 0)
		dev_notice(info->dev, "%s: set mivr fail\n", __func__);

	ret = __rt9467_set_cv(info, desc->cv);
	if (ret < 0)
		dev_notice(info->dev, "%s: set cv fail\n", __func__);

	ret = __rt9467_set_ieoc(info, desc->ieoc);
	if (ret < 0)
		dev_notice(info->dev, "%s: set ieoc fail\n", __func__);

	ret = __rt9467_enable_te(info, desc->en_te);
	if (ret < 0)
		dev_notice(info->dev, "%s: set te fail\n", __func__);

	ret = rt9467_set_safety_timer(info, desc->safety_timer);
	if (ret < 0)
		dev_notice(info->dev, "%s: set fast timer fail\n", __func__);

	ret = __rt9467_enable_safety_timer(info, true);
	if (ret < 0)
		dev_notice(info->dev, "%s: enable chg timer fail\n", __func__);

	ret = rt9467_enable_wdt(info, desc->en_wdt);
	if (ret < 0)
		dev_notice(info->dev, "%s: set wdt fail\n", __func__);

	ret = rt9467_enable_jeita(info, desc->en_jeita);
	if (ret < 0)
		dev_notice(info->dev, "%s: disable jeita fail\n", __func__);

	ret = rt9467_enable_irq_pulse(info, desc->en_irq_pulse);
	if (ret < 0)
		dev_notice(info->dev, "%s: set irq pulse fail\n", __func__);

	/* set ircomp according to BIF */
	ret = rt9467_set_ircmp_resistor(info, desc->ircmp_resistor);
	if (ret < 0)
		dev_notice(info->dev, "%s: set ircmp resistor fail\n",
			__func__);

	ret = rt9467_set_ircmp_vclamp(info, desc->ircmp_vclamp);
	if (ret < 0)
		dev_notice(info->dev, "%s: set ircmp clamp fail\n", __func__);

	ret = rt9467_sw_workaround(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: workaround fail\n", __func__);
		return ret;
	}

	/* Enable HZ mode of secondary charger */
	if (strcmp(info->desc->chg_dev_name, "secondary_chg") == 0) {
		ret = __rt9467_enable_hz(info, true);
		if (ret < 0)
			dev_notice(info->dev, "%s: hz sec chg fail\n",
				__func__);
	}
err:
	return ret;
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */


/*regulator otg ops*/
int rt9467_enable_regulator_otg(struct regulator_dev *rdev)
{
	int ret = 0;
	struct rt9467_info *info = rdev_get_drvdata(rdev);

	dev_info(info->dev, "%s\n", __func__);

	ret = rt9467_enable_discharge(info, false);
	if (ret < 0) {
		dev_notice(info->dev, "%s: disable discharge fail\n", __func__);
		return ret;
	}

	return rt9467_enable_otg(info, true);
}

int rt9467_disable_regulator_otg(struct regulator_dev *rdev)
{
	int ret = 0;
	struct rt9467_info *info = rdev_get_drvdata(rdev);

	dev_info(info->dev, "%s\n", __func__);

	ret = rt9467_enable_otg(info, false);
	if (ret < 0) {
		dev_notice(info->dev, "%s: disable otg fail\n", __func__);
		return ret;
	}

	return rt9467_enable_discharge(info, true);
}

static int rt9467_set_current_limit(struct regulator_dev *rdev,
						int min_uA, int max_uA)
{
	struct rt9467_info *info = rdev_get_drvdata(rdev);
	int num = ARRAY_SIZE(rt9467_boost_oc_threshold);

	if (min_uA < rt9467_boost_oc_threshold[0])
		min_uA = rt9467_boost_oc_threshold[0];
	else if (min_uA > rt9467_boost_oc_threshold[num - 1])
		min_uA = rt9467_boost_oc_threshold[num - 1];

	return rt9467_set_boost_current_limit(info, min_uA);
}

static int rt9467_get_current_limit(struct regulator_dev *rdev)
{
	int ret = 0, val = 0;
	struct rt9467_info *info = rdev_get_drvdata(rdev);

	/* RT9467_REG_CHG_CTRL10 = 0x0A */
	ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_CTRL10);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read otg_cc fail\n", __func__);
		return ret;
	}
	val = (ret & RT9467_MASK_BOOST_OC) >> RT9467_SHIFT_BOOST_OC;
	return rt9467_boost_oc_threshold[val];
}

static const struct regulator_ops rt9467_chg_otg_ops = {
	.enable = rt9467_enable_regulator_otg,
	.disable = rt9467_disable_regulator_otg,
	.is_enabled = regulator_is_enabled_regmap,
	.list_voltage = regulator_list_voltage_linear, 
	.set_voltage_sel = regulator_set_voltage_sel_regmap,
	.get_voltage_sel = regulator_get_voltage_sel_regmap,
	.set_current_limit = rt9467_set_current_limit,
	.get_current_limit = rt9467_get_current_limit,
};

static const struct regulator_desc rt9467_otg_rdesc = {
	.of_match = "usb-otg-vbus",
	.name = "usb-otg-vbus",
	.ops = &rt9467_chg_otg_ops,
	.owner = THIS_MODULE,
	.type = REGULATOR_VOLTAGE,
	.min_uV = 4425000,
	.uV_step = 25000, /* step  25mV */
	.n_voltages = 57, /* 4425mV to 5825mV */
	.vsel_reg = RT9467_REG_CHG_CTRL5,
	.vsel_mask = RT9467_MASK_BOOST_VOREG,
	.enable_reg = RT9467_REG_CHG_CTRL1,
	.enable_mask = RT9467_MASK_OPA_MODE,
};

/* ======================= */
/* rt9467 Power Supply Ops */
/* ======================= */
static int rt9467_charger_get_online(struct rt9467_info *info,
				     union power_supply_propval *val)
{
	val->intval = info->pwr_rdy;

	dev_info(info->dev, "%s: online = %d\n", __func__, val->intval);
	return 0;
}

static int rt9467_charger_set_online(struct rt9467_info *info,
				     const union power_supply_propval *val)
{
#if defined(CONFIG_PLATFORM_NEC)
	return 0;
#else /* defined(CONFIG_PLATFORM_NEC) */
	return __rt9467_enable_chg_type_det(info, val->intval);
#endif /* defined(CONFIG_PLATFORM_NEC) */
}

static int rt9467_charger_get_property(struct power_supply *psy,
				       enum power_supply_property psp,
				       union power_supply_propval *val)
{
	struct rt9467_info *info = power_supply_get_drvdata(psy);
	enum rt9467_charging_status chg_stat = RT9467_CHG_STATUS_MAX;
	int ret = 0;

	dev_dbg(info->dev, "%s: prop = %d\n", __func__, psp);
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
		ret = rt9467_charger_get_online(info, val);
		break;
	case POWER_SUPPLY_PROP_TYPE:
		val->intval = info->psy_desc.type;
		break;
	case POWER_SUPPLY_PROP_USB_TYPE:
		val->intval = info->psy_usb_type;
		break;
	case POWER_SUPPLY_PROP_STATUS:
#ifdef WD_DEBUG
		if (wd_timeout_dbg == 1) {
			val->intval = POWER_SUPPLY_STATUS_CHARGING;
			break;
		}
#endif /* WD_DEBUG */
		ret = rt9467_get_charging_status(info, &chg_stat);
		if (ret < 0)
			dev_info(info->dev,
				"%s: get mt6360 chg_status failed\n", __func__);

#if defined(CONFIG_PLATFORM_NEC)
		ret = 0;
#endif /* defined(CONFIG_PLATFORM_NEC) */
		switch (chg_stat) {
		case RT9467_CHG_STATUS_READY:
			val->intval = POWER_SUPPLY_STATUS_NOT_CHARGING;
			break;
		case RT9467_CHG_STATUS_PROGRESS:
			val->intval = POWER_SUPPLY_STATUS_CHARGING;
			break;
		case RT9467_CHG_STATUS_DONE:
			val->intval = POWER_SUPPLY_STATUS_FULL;
			break;
		case RT9467_CHG_STATUS_FAULT:
			val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
			break;
		default:
#if defined(CONFIG_PLATFORM_NEC)
			dev_err(info->dev, "%s:  failed chg_stat=%d \n", __func__,chg_stat);
			val->intval = POWER_SUPPLY_STATUS_UNKNOWN;
#else /* defined(CONFIG_PLATFORM_NEC) */
			ret = -ENODATA;
#endif /* defined(CONFIG_PLATFORM_NEC) */
			break;
		}
#if defined(CONFIG_PLATFORM_NEC)
		if (!info->pwr_rdy) {
			val->intval = POWER_SUPPLY_STATUS_DISCHARGING;
		}
#endif /* defined(CONFIG_PLATFORM_NEC) */
		break;
	case POWER_SUPPLY_PROP_CURRENT_MAX:
		if (info->psy_desc.type == POWER_SUPPLY_TYPE_USB)
			val->intval = 500000;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MAX:
		if (info->psy_desc.type == POWER_SUPPLY_TYPE_USB)
			val->intval = 5000000;
		break;
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_VBUS_CONNECTION:
		val->intval = info->pwr_rdy;
		break;
	case POWER_SUPPLY_PROP_CRADLE_STATUS:
		val->intval = rt9467_get_prop_cradle_status(info);
		break;
	case POWER_SUPPLY_PROP_MACHINE_STATUS:
#ifdef WD_DEBUG
		if (wd_timeout_dbg == 1) {
			val->intval = 0;
			break;
		}
#endif /* WD_DEBUG */
		val->intval = rt9467_get_prop_machine_status(info);
		break;
	case POWER_SUPPLY_PROP_CHARGE_STOP:
		val->intval = info->chage_stop;
		pr_debug("%s POWER_SUPPLY_PROP_CHARGE_STOP val=%d\n",__func__,val->intval);
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		val->intval = info->real_usb_psy_ma;
		break;
	case POWER_SUPPLY_PROP_CHARGE_VOLTAGE:
		val->intval = info->float_voltage;
		pr_debug("%s POWER_SUPPLY_PROP_CHARGE_VOLTAGE val=0x%x com=%s\n",__func__,val->intval,current->comm);
		break;
	case POWER_SUPPLY_PROP_DCP_TYPE:
		val->intval = rt9467_get_prop_usb_type_check(info);
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
#ifdef WD_DEBUG
		if (wd_timeout_dbg == 1) {
			val->intval = 0;
			break;
		}
#endif /* WD_DEBUG */
		val->intval = info->notify_src;
		break;
	case POWER_SUPPLY_PROP_TEMP:
#ifdef WD_DEBUG
		if (wd_timeout_dbg == 1) {
			val->intval = 3200;
			break;
		}
#endif /* WD_DEBUG */
		val->intval = rt9467_adc_read_temperature(info);
		dev_dbg(info->dev,"%s: POWER_SUPPLY_PROP_TEMP val=%d\n", __func__,val->intval);
		break;
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
#ifdef WD_DEBUG
		if (wd_timeout_dbg == 1) {
			val->intval = 1;
			break;
		}
#endif /* WD_DEBUG */
		val->intval = rt9467_get_prop_charge_status(info);
		break;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	default:
		ret = -ENODATA;
	}
	return ret;
}

static int rt9467_charger_set_property(struct power_supply *psy,
				       enum power_supply_property psp,
				       const union power_supply_propval *val)
{
	struct rt9467_info *info = power_supply_get_drvdata(psy);
	int ret;

	dev_dbg(info->dev, "%s: prop = %d\n", __func__, psp);
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
		ret = rt9467_charger_set_online(info, val);
		break;
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
		dev_dbg(info->dev, "%s POWER_SUPPLY_PROP_CHARGING_ENABLED val=%d\n",__func__,val->intval);
		info->chg_enabled = val->intval;
		rt9467_enable_charger(info,info->chg_enabled);
		break;
	case POWER_SUPPLY_PROP_CHARGE_STOP:
		dev_dbg(info->dev, "%s POWER_SUPPLY_PROP_CHARGE_STOP val=%d\n",__func__,val->intval);
		info->chage_stop = val->intval;
		rt9467_enable_charger(info,info->chg_enabled);
		break;
	case POWER_SUPPLY_PROP_FORCE_DISCHARGE:
		dev_dbg(info->dev, "%s POWER_SUPPLY_PROP_FORCE_DISCHARGE val=%d\n",__func__,val->intval);
		if (val->intval) {
			if (val->intval != info->real_usb_psy_ma) {
				printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Force input current %dmA\n", val->intval);
				info->prev_usb_psy_dis_ma = info->real_usb_psy_ma;
				rt9467_set_input_current_limit(info,val->intval);
				info->real_usb_psy_ma = val->intval;
			}
		} else {
			if ((info->prev_usb_psy_dis_ma != 0) && (info->prev_usb_psy_dis_ma != info->real_usb_psy_ma)) {
				if (info->pwr_rdy) {
					printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Restore input current %dmA to %dmA\n", info->real_usb_psy_ma, info->prev_usb_psy_dis_ma);
					rt9467_set_input_current_limit(info,info->prev_usb_psy_dis_ma);
					info->real_usb_psy_ma = info->prev_usb_psy_dis_ma;
				}
				/* printk("POWER_SUPPLY_PROP_FORCE_DISCHARGE Restore stay\n"); */
				info->prev_usb_psy_dis_ma = 0;
			}
		}
		break;
	case POWER_SUPPLY_PROP_CHARGE_VOLTAGE:
		dev_info(info->dev, "####rt9467_charger_set_property POWER_SUPPLY_PROP_CHARGE_VOLTAGE\n");
		rt9467_set_charging_voltage(info, val->intval);
		info->current_charging_voltage_step = val->intval;
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		dev_info(info->dev, "%s POWER_SUPPLY_PROP_NOTIFY_SRC val=%d\n",__func__,val->intval);
		cancel_delayed_work_sync(&info->polling_work);
		info->polling_interval_sec = info->detect_interval_sec;
		info->notify_src = 1;
		dev_info(info->dev,"@@@ %s notify_src in polling interval_sec=%d\n",__FUNCTION__,info->polling_interval_sec);
		schedule_delayed_work(&info->polling_work, info->polling_interval_sec * HZ);
		break;
#endif /* defined(CONFIG_PLATFORM_NEC) */
	default:
		ret = -EINVAL;
	}
	return ret;
}

static int rt9467_charger_property_is_writeable(struct power_supply *psy,
						enum power_supply_property psp)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_ONLINE:
#if defined(CONFIG_PLATFORM_NEC)
	case POWER_SUPPLY_PROP_CHARGE_STOP:
	case POWER_SUPPLY_PROP_CHARGING_ENABLED:
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
#endif /* defined(CONFIG_PLATFORM_NEC) */
		return 1;
	default:
		return 0;
	}
}

static enum power_supply_property rt9467_charger_properties[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_ONLINE,
	POWER_SUPPLY_PROP_TYPE,
	POWER_SUPPLY_PROP_USB_TYPE,
	POWER_SUPPLY_PROP_CURRENT_MAX,
	POWER_SUPPLY_PROP_VOLTAGE_MAX,
#if defined(CONFIG_PLATFORM_NEC)
	POWER_SUPPLY_PROP_VBUS_CONNECTION,
	POWER_SUPPLY_PROP_CRADLE_STATUS,
	POWER_SUPPLY_PROP_MACHINE_STATUS,
	POWER_SUPPLY_PROP_CHARGE_STOP,
	POWER_SUPPLY_PROP_FORCE_DISCHARGE,
	POWER_SUPPLY_PROP_CHARGE_VOLTAGE,
	POWER_SUPPLY_PROP_DCP_TYPE,
	POWER_SUPPLY_PROP_NOTIFY_SRC
	,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_CHARGING_ENABLED,
#endif /* defined(CONFIG_PLATFORM_NEC) */
};

static enum power_supply_usb_type rt9467_charger_usb_types[] = {
	POWER_SUPPLY_USB_TYPE_UNKNOWN,
	POWER_SUPPLY_USB_TYPE_SDP,
	POWER_SUPPLY_USB_TYPE_DCP,
	POWER_SUPPLY_USB_TYPE_CDP,
	POWER_SUPPLY_USB_TYPE_C,
	POWER_SUPPLY_USB_TYPE_PD,
	POWER_SUPPLY_USB_TYPE_PD_DRP,
	POWER_SUPPLY_USB_TYPE_APPLE_BRICK_ID
};

static const struct power_supply_desc rt9467_charger_desc = {
#if defined(CONFIG_PLATFORM_NEC)
	.name = "rt9467-charger",
#endif /* defined(CONFIG_PLATFORM_NEC) */
	.type			= POWER_SUPPLY_TYPE_USB,
	.properties		= rt9467_charger_properties,
	.num_properties		= ARRAY_SIZE(rt9467_charger_properties),
	.get_property		= rt9467_charger_get_property,
	.set_property		= rt9467_charger_set_property,
	.property_is_writeable	= rt9467_charger_property_is_writeable,
	.usb_types		= rt9467_charger_usb_types,
	.num_usb_types		= ARRAY_SIZE(rt9467_charger_usb_types),
};

static char *rt9467_charger_supplied_to[] = {
	"battery",
	"mtk-master-charger"
};

#if !defined(CONFIG_PLATFORM_NEC)
static void rt9467_init_setting_work_handler(struct work_struct *work)
{
	int ret = 0, retry_cnt = 0;
	struct rt9467_info *info = (struct rt9467_info *)container_of(work,
		struct rt9467_info, init_work);

	do {
		/* Select IINLMTSEL to use AICR */
		ret = rt9467_select_input_current_limit(info,
			RT9467_IINLMTSEL_AICR);
		if (ret < 0) {
			dev_notice(info->dev, "%s: sel ilmtsel fail\n",
				__func__);
			retry_cnt++;
		}
	} while (retry_cnt < 5 && ret < 0);

	msleep(150);

	retry_cnt = 0;
	do {
		/* Disable hardware ILIM */
		ret = rt9467_enable_ilim(info, false);
		if (ret < 0) {
			dev_notice(info->dev, "%s: disable ilim fail\n",
				__func__);
			retry_cnt++;
		}
	} while (retry_cnt < 5 && ret < 0);
}
#endif /* !defined(CONFIG_PLATFORM_NEC) */

/* ========================= */
/* I2C driver function       */
/* ========================= */

static ssize_t shipping_mode_store(struct device *dev,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct rt9467_info *info = dev_get_drvdata(dev);
	int ret = 0;
	bool is_main_chg = strcmp(info->desc->chg_dev_name,
		"primary_chg") == 0 ? true : false;

	mutex_lock(&info->adc_access_lock);
	if (is_main_chg) {
		ret = __rt9467_enable_auto_sensing(info, false);
		if (ret < 0) {
			dev_notice(dev, "%s: disable auto sensing fail\n",
				__func__);
			goto out;
		}
		mdelay(50);
	}

	ret = rt9467_sw_reset(info);
	if (ret < 0) {
		dev_notice(dev, "%s: sw reset fail\n", __func__);
		goto out;
	}

	if (is_main_chg) {
		/* RT9467_REG_CHG_CTRL2 = 0x02 */
		ret = rt9467_set_bit(info, RT9467_REG_CHG_CTRL2,
			RT9467_MASK_SHIP_MODE);
		if (ret < 0)
			dev_notice(dev, "%s: set shipping mode fail\n",
				__func__);
	}
out:
	mutex_unlock(&info->adc_access_lock);
	return ret;
}

static const DEVICE_ATTR_WO(shipping_mode);

#if defined(CONFIG_PLATFORM_NEC)
static ssize_t rt9467_fs_show_registers(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct rt9467_info *info = dev_get_drvdata(dev);
	u8 addr;
	u8 val = 0;
	u8 tmpbuf[300];
	int len;
	int idx = 0;
	int ret;
	u8 mask[RT9467_IRQIDX_MAX] = {0};

	idx = snprintf(buf, PAGE_SIZE, "%s:\n", "Charger");
	for (addr = 0x0; addr <= 0x1A; addr++) {
		ret = rt9467_i2c_read_byte(info,addr);
		if (ret >= 0) {
			val = ret;
		} else {
			continue;
		}
		
		len = snprintf(tmpbuf, PAGE_SIZE - idx,"Reg[0x%.2x] = 0x%.2x\n", addr, val);
		memcpy(&buf[idx], tmpbuf, len);
		idx += len;
	}
	for (addr = 0x40; addr <= 0x45; addr++) {
		ret = rt9467_i2c_read_byte(info,addr);
		if (ret >= 0) {
			val = ret;
		} else {
			continue;
		}
		
		len = snprintf(tmpbuf, PAGE_SIZE - idx,"Reg[0x%.2x] = 0x%.2x\n", addr, val);
		memcpy(&buf[idx], tmpbuf, len);
		idx += len;
	}
	for (addr = 0x50; addr <= 0x52; addr++) {
		ret = rt9467_i2c_read_byte(info,addr);
		if (ret >= 0) {
			val = ret;
		} else {
			continue;
		}
		
		len = snprintf(tmpbuf, PAGE_SIZE - idx,"Reg[0x%.2x] = 0x%.2x\n", addr, val);
		memcpy(&buf[idx], tmpbuf, len);
		idx += len;
	}

	/* Read mask */
	/* RT9467_REG_CHG_STATC_CTRL = 0x60 */
	rt9467_i2c_block_read(info, RT9467_REG_CHG_STATC_CTRL,
		ARRAY_SIZE(mask), mask);
	for (addr = 0x60; addr <= 0x66; addr++) {
		ret = mask[addr-0x60];
		if (ret >= 0) {
			val = ret;
		} else {
			continue;
		}
		
		len = snprintf(tmpbuf, PAGE_SIZE - idx,"Reg[0x%.2x] = 0x%.2x\n", addr, val);
		memcpy(&buf[idx], tmpbuf, len);
		idx += len;
	}

	return idx;
}

static ssize_t rt9467_fs_show_temp_read(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	struct rt9467_info *info = dev_get_drvdata(dev);
	int idx = 0;
	int ret;

	ret = rt9467_adc_read_temperature(info);

	idx = snprintf(buf, PAGE_SIZE, "Charger temp = %d\n", ret);

	return idx;
}

static DEVICE_ATTR(registers, S_IRUGO, rt9467_fs_show_registers, NULL);
static DEVICE_ATTR(temp_read, S_IRUGO, rt9467_fs_show_temp_read, NULL);

static struct attribute *rt9467_attributes[] = {
	&dev_attr_registers.attr,
	&dev_attr_temp_read.attr,
	NULL,
};

static const struct attribute_group rt9467_attr_group = {
	.attrs = rt9467_attributes,
};

static int rt9467_set_charging_voltage(struct rt9467_info *info, int step)
{
	int ret;
	
	/* RT9467_REG_CHG_CTRL4 = 0x04 */
	ret = rt9467_i2c_write_byte(info, RT9467_REG_CHG_CTRL4, charge_voltage_table[step]);
	if (ret < 0) {
		dev_err(info->dev, "Unable to set RT9467_REG_CHG_CTRL4 ret = %d\n", ret);
		return ret;
	}
	printk("####rt9467_set_charging_voltage OK step%d set%02x\n", step, charge_voltage_table[step]);
	info->float_voltage = charge_voltage_table[step];
	return 0;
}

#define POWER_SUPPLY_TYPEC_NAME		"type-c-usb"
#define POWER_SUPPLY_BATTERY_NAME	"rt9426a-battery"

static int rt9467_battery_notify_usb_src_change(struct rt9467_info *info,int src_status)
{
	union power_supply_propval pval;
	int ret = -ENODEV;

	pval.intval = src_status;

	if (!info->batt_psy)
		info->batt_psy = power_supply_get_by_name(POWER_SUPPLY_BATTERY_NAME);

	if (info->batt_psy) {
		ret = power_supply_set_property(info->batt_psy, POWER_SUPPLY_PROP_NOTIFY_SRC, &pval);
	}

	return ret;
}

static int rt9467_usb_notify_usb_src_change(struct rt9467_info *info,int src_status)
{
	union power_supply_propval pval;
	int ret = -ENODEV;

	pval.intval = src_status;

	if (!info->usb_psy)
		info->usb_psy = power_supply_get_by_name(POWER_SUPPLY_TYPEC_NAME);

	if (info->usb_psy) {
		ret = power_supply_set_property(info->usb_psy, POWER_SUPPLY_PROP_NOTIFY_SRC, &pval);
	}

	return ret;
}

/* 1:connected 0:unconnected -1:-ENODEV */
static int rt9467_get_prop_cradle_status(struct rt9467_info *info)
{
	union power_supply_propval pval;
	int ret = -ENODEV;

	if (!info->usb_psy)
		info->usb_psy = power_supply_get_by_name(POWER_SUPPLY_TYPEC_NAME);

	if (info->usb_psy) {
		ret = power_supply_get_property(info->usb_psy, POWER_SUPPLY_PROP_CRADLE_STATUS, &pval);
		if (ret == 0) {
			ret = pval.intval;
		}
	}
	return ret;
}

static int rt9467_get_prop_charge_status(struct rt9467_info *info)
{
	bool enable;

	__rt9467_is_charging_enable(info,&enable);
	return enable;
}

static int rt9467_get_prop_usb_type_check(struct rt9467_info *info)
{
	return 0;
}

#define MACHINE_STATUS_GOOD				0
#define MACHINE_STATUS_HOT_HARD_LIMIT	1

static int rt9467_get_prop_machine_status(struct rt9467_info *info)
{
#define MACHINE_STATUS_CHECK	3
	int i, ret;
	int bat_ntc_fault;

	for (i = 0; i < MACHINE_STATUS_CHECK; i++) {
		/* RT9467_REG_CHG_NTC = 0x43 */
		ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_NTC);
		if (ret < 0) {
			printk(KERN_ERR "%s RT9467_REG_CHG_NTC read err=%d\n",__func__,ret);
			return 0;
		}

		pr_debug("%s RT9467_REG_CHG_NTC val=0x%x\n",__func__,ret);

		bat_ntc_fault = (ret & RT9467_MASK_BAT_NTC_FAULT) >> RT9467_SHIFT_BAT_NTC_FAULT;
		/* 0=000 : Normal */
		/* 2=010 : Warm */
		/* 3=011 : Cool */
		/* 5=101 : Cold */
		/* 6=110 : Hot */
		if (bat_ntc_fault == 0x6) {
			ret = MACHINE_STATUS_HOT_HARD_LIMIT;
		} else {
			ret = MACHINE_STATUS_GOOD;
			break;
		}
	}
	return ret;
}

static int rt9467_adc_read_temperature(struct rt9467_info *info)
{
	int ret;
	int adc_val = 0;
	int ts_value;
	int temp;
	
	ret = rt9467_get_adc(info,RT9467_ADC_TS_BAT,&adc_val);
	if (ret < 0) {
		dev_err(info->dev, "%s rt9467_get_adc RT9467_ADC_TS_BAT err=%d\n",__func__,ret);
		return ret;
	} else {
		ts_value = adc_val * (1000/100);
		dev_dbg(info->dev, "%s adc_val =%d\n",__func__,adc_val);
		dev_dbg(info->dev, "%s ts_value =%d\n",__func__,ts_value);
		
		/* 112.97                - (1.5209                  *  (ts_value / 1000)       ) */
		/* (112.97 * 1000*10000) - ((1.5209 *10000)         *  (ts_value))             ) */
		temp = (1129700000 - (15209 * ts_value)) /100000;
	}
	return temp;
}

static int rt9467_nec_hw_init_regs(struct rt9467_info *info, int recovery)
{
	struct init_reg_t *init_regs = hw_init_regs;
	int ret;
	int i;

	for (i = 0; i < ARRAY_SIZE(hw_init_regs); i++, init_regs++) {
		ret = rt9467_i2c_write_byte(info, init_regs->reg, init_regs->val);
		if (ret) {
			dev_err(info->dev, "rt9467_i2c_write_byte error %02x %02x ret=%d\n",
			    init_regs->reg, init_regs->val, ret);
			break;
		}
	}

	if (recovery == 1) {
		ret = rt9467_set_input_current_limit(info, info->real_usb_psy_ma);
		if (ret < 0) {
			dev_err(info->dev, "%s:Failed to set input current limit:%d\n", __func__, ret);
		}

		ret = rt9467_enable_charger(info,info->chg_enabled);
		dev_info(info->dev, "%s: start charging ret=%d\n", __func__,ret);

		ret = rt9467_set_charging_voltage(info,info->current_charging_voltage_step);
		if (ret < 0) {
			dev_err(info->dev, "%s:Failed to charging_voltage:%d\n", __func__, ret);
		}
	}
	return ret;
}

static int rt9467_enter_ship_mode(struct rt9467_info *info)
{
	int ret;

	ret = rt9467_set_bit(info, RT9467_REG_CHG_CTRL2,RT9467_MASK_SHIP_MODE);
	if (ret < 0)
		dev_notice(info->dev, "%s: set shipping mode fail\n",	__func__);

	return ret;
}

static int rt9467_enable_charger(struct rt9467_info *info, bool enable)
{
	pr_debug("%s enable=%d chage_stop=%d\n",__func__,enable,info->chage_stop);
	if( info->chage_stop == 1 ) {
		/* force chg disable */
		/* RT9467_REG_CHG_CTRL2 = 0x02 */
		return rt9467_clr_bit(info, RT9467_REG_CHG_CTRL2, RT9467_MASK_CHG_EN);
	} else {
		/* RT9467_REG_CHG_CTRL2 = 0x02 */
		return (enable ? rt9467_set_bit : rt9467_clr_bit)
			(info, RT9467_REG_CHG_CTRL2, RT9467_MASK_CHG_EN);
	}
}

static int rt9467_set_input_current_limit(struct rt9467_info *info, int curr)
{
	int aicr_value;

	switch(curr) {
		case CURRENT_1000_MA:
			aicr_value = AICLR_1000MA;
			break;
		case CURRENT_1800_MA:
			aicr_value = AICLR_1800MA;
			break;
		case CURRENT_100_MA:
			aicr_value = AICLR_100MA;
			break;
		case CURRENT_500_MA:
		default:
			aicr_value = AICLR_500MA;
			break;
	}

	/* RT9467_REG_CHG_CTRL3 = 0x03 */
	return rt9467_i2c_update_bits(info, RT9467_REG_CHG_CTRL3,
		aicr_value << RT9467_SHIFT_AICR, RT9467_MASK_AICR);
}

static int rt9467_get_input_current_limit(struct rt9467_info *info)
{
	u32 aicr_value;
	int ret;

	ret = __rt9467_get_aicr(info, &aicr_value);
	if (ret < 0)
		return 0;

	return aicr_value;
}

static int rt9467_read_batt_rsoc(struct rt9467_info *info)
{
	union power_supply_propval ret = {0,};

	if (!info->batt_psy)
		info->batt_psy = power_supply_get_by_name(POWER_SUPPLY_BATTERY_NAME);

	if (info->batt_psy) {
		if (info->batt_psy->desc) {
			info->batt_psy->desc->get_property(info->batt_psy,POWER_SUPPLY_PROP_CAPACITY,&ret);
			return ret.intval;
		}
	}

	return -1;
}

static int rt9467_wd_check(struct rt9467_info *info)
{
#define WD_TIMEOUT_CHECK	3
	int i, ret;
	int result = 0;
	u8 vendor_id = 0, chip_rev = 0;

	for (i = 0; i < WD_TIMEOUT_CHECK; i++) {
		/* RT9467_REG_DEVICE_ID = 0x40*/
		ret = i2c_smbus_read_byte_data(info->client, RT9467_REG_DEVICE_ID);
		if (ret < 0) {
			break;
		}

		vendor_id = ret & 0xF0;
		chip_rev = ret & 0x0F;
		if (vendor_id == RT9467_VENDOR_ID) {
			result = 1;
			break;
		}
	}
#ifdef WD_DEBUG
	if (wd_timeout_dbg == 2) {
		result = 0;
		wd_timeout_dbg = 0;
	}
#endif /* WD_DEBUG */
	return result;
}

static void rt9467_polling_workfunc(struct work_struct *work)
{
	int ret;
	struct rt9467_info *info = container_of(work, struct rt9467_info, polling_work.work);
	enum rt9467_charging_status chg_stat = RT9467_CHG_STATUS_MAX;

	if (!info->pwr_rdy) {
		goto end_func;
	}

#ifdef WD_DEBUG
	if (wd_timeout_dbg == 1) {
		printk(KERN_ERR "@@@ wd_timuout_dbg enable skip reset wd\n");
		goto end_func;
	}
#endif

	if (rt9467_wd_check(info) == 0) {
		dev_err(info->dev, "@@@ WD Check Hit!!\n");
		ret = rt9467_nec_hw_init_regs(info,1);
		if (ret < 0) {
			dev_err(info->dev, "Failed to nec_hw_init\n");
		}
	}

	rt9467_get_charging_status(info, &chg_stat);
	dev_info(info->dev, "%s:soc:%d, chg_stat:%d ce:%d Ilim=%d\n", __func__,
				rt9467_read_batt_rsoc(info),
				chg_stat,
				rt9467_get_prop_charge_status(info),
				rt9467_get_input_current_limit(info)
	);

end_func:
	if (info->notify_src) {
		info->notify_src = 0;
		info->polling_interval_sec = info->monitor_interval_sec;
		dev_info(info->dev,"@@@ %s notify_src in polling interval_sec=%d\n",__FUNCTION__,info->polling_interval_sec);
		schedule_work(&info->irq_work);
	}
	schedule_delayed_work(&info->polling_work, info->polling_interval_sec * HZ);
}

static void rt9467_charger_irq_workfunc(struct work_struct *work)
{
	int ret;
	struct rt9467_info *info = container_of(work, struct rt9467_info, irq_work);
	bool pwr_rdy = false;
	int check_cnt;
	u8 usb_status = 0;

	dev_info(info->dev, "%s\n", __func__);

	/* check power ready */
	/* RT9467_REG_CHG_STATC = 0x50 */
	ret = rt9467_i2c_test_bit(info, RT9467_REG_CHG_STATC,
		RT9467_SHIFT_PWR_RDY, &pwr_rdy);
	if (ret < 0) {
		dev_notice(info->dev, "%s: read pwr rdy state fail\n",
			__func__);
		return;
	}

	dev_info(info->dev, "pwr_rdy now(%d) old(%d) usb_detect(%d)\n", pwr_rdy,info->pwr_rdy,info->usb_detect);

#if 0
	if (info->pwr_rdy == pwr_rdy &&
		info->pwr_rdy == info->usb_detect) {
		dev_info(info->dev, "%s: pwr_rdy(%d) state is the same\n",__func__, info->pwr_rdy);
		return;
	}
#endif

	info->pwr_rdy = pwr_rdy;

	/* plug out */
	if (!info->pwr_rdy) {
		dev_info(info->dev, "%s: Charger Type: UNKONWN\n", __func__);
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
		info->real_usb_psy_ma = CURRENT_500_MA;
		goto ilimset;
	}

	/* plug in */
#define RT9467_MASK_USB_SRC			0x0F
#define RT9467_CHG_DPDM1_SDP		(1 << 0)
#define RT9467_CHG_DPDM1_CDP		(1 << 1)
#define RT9467_CHG_DPDM1_DCP		(1 << 2)
#define RT9467_CHG_DPDM1_HVDCP		(1 << 3)

	if (info->usb_detect == 0) {
		msleep(600);
		rt9467_toggle_chgdet_flow(info);
	}

	/* check status */
#define STATUS_CHECK_TIMEOUT	2000
#define STATUS_CHECK_TIME		100
	for (check_cnt = 0;check_cnt < (STATUS_CHECK_TIMEOUT / STATUS_CHECK_TIME);check_cnt++) {
		/* RT9467_REG_CHG_DPDM1 = 0x12 */
		ret = rt9467_i2c_read_byte(info, RT9467_REG_CHG_DPDM1);
		dev_dbg(info->dev, "0x12=0x%x\n",ret);
		if (ret < 0) {
			dev_notice(info->dev, "%s: read type fail\n", __func__);
			return;
		} else if (ret & RT9467_MASK_USB_SRC) {
			break;
		}
		msleep(STATUS_CHECK_TIME);
	}

	dev_info(info->dev,"%s: RT9467_REG_CHG_DPDM1 = 0x%x\n", __func__,ret);
	usb_status = (ret & RT9467_MASK_USB_SRC);

	switch (usb_status) {
	case RT9467_CHG_DPDM1_SDP:
		dev_info(info->dev,
			  "%s: Charger Type: RT9467_CHG_DPDM1_SDP\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_SDP;
		info->real_usb_psy_ma = CURRENT_500_MA;
		break;
	case RT9467_CHG_DPDM1_HVDCP:
		dev_info(info->dev,
			  "%s: Charger Type: RT9467_CHG_DPDM1_HVDCP\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
		info->real_usb_psy_ma = CURRENT_1000_MA;
		break;
	case RT9467_CHG_DPDM1_CDP:
		dev_info(info->dev,
			  "%s: Charger Type: RT9467_CHG_DPDM1_CDP\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB_CDP;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_CDP;
		info->real_usb_psy_ma = CURRENT_500_MA;
		break;
	case RT9467_CHG_TYPE_DCP:
		dev_info(info->dev,
			  "%s: Charger Type: STANDARD_CHARGER\n", __func__);
		info->psy_desc.type = POWER_SUPPLY_TYPE_USB_DCP;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
		info->real_usb_psy_ma = CURRENT_1800_MA;
		break;
	default:
		dev_info(info->dev,
			  "%s: Charger Type: default=0x%x\n", __func__,usb_status);
		info->psy_desc.type = POWER_SUPPLY_TYPE_UNKNOWN;
		info->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
		info->real_usb_psy_ma = CURRENT_500_MA;
		break;
	}

ilimset:
	rt9467_usb_notify_usb_src_change(info,(int)info->usb_detect);
	info->cradle_status = rt9467_get_prop_cradle_status(info);
	dev_info(info->dev, "%s:cradle_status:%d\n", __func__, info->cradle_status);
	if (info->cradle_status == 1) {
		if (info->pwr_rdy == 1) {
			info->psy_usb_type = POWER_SUPPLY_USB_TYPE_DCP;
			info->real_usb_psy_ma = CURRENT_1800_MA;
		} else {
			info->psy_usb_type = POWER_SUPPLY_USB_TYPE_UNKNOWN;
			info->real_usb_psy_ma = CURRENT_500_MA;
		}
	}

	rt9467_set_input_current_limit(info, info->real_usb_psy_ma);
	rt9467_battery_notify_usb_src_change(info,(int)info->usb_detect);
	info->ilim_det_result = rt9467_get_input_current_limit(info) * 1000;
	dev_info(info->dev, "%s:Ilim Det-reslt:%d uA\n", __func__, info->ilim_det_result);

	if (info->psy_usb_type == POWER_SUPPLY_USB_TYPE_UNKNOWN) {
		rt9467_enable_charger(info,false);
		ret = rt9467_enable_wdt(info,false);
		if (ret < 0) {
			dev_err(info->dev, "%s:Failed to disable watchdog timer:%d\n", __func__, ret);
		}
	} else {
		rt9467_enable_charger(info,true);
		ret = rt9467_enable_wdt(info,true);
		if (ret < 0) {
			dev_err(info->dev, "%s:Failed to enable watchdog timer:%d\n", __func__, ret);
		}
	}
	info->usb_detect = info->pwr_rdy;
	return;
}
#endif /* defined(CONFIG_PLATFORM_NEC) */

static int rt9467_probe(struct i2c_client *client,
	const struct i2c_device_id *dev_id)
{
	int ret = 0;
	struct rt9467_info *info = NULL;
	struct power_supply_config charger_cfg = {};
	struct regulator_config config = { };

	pr_info("%s(%s)\n", __func__, RT9467_DRV_VERSION);

	info = devm_kzalloc(&client->dev, sizeof(struct rt9467_info),
		GFP_KERNEL);
	if (!info)
		return -ENOMEM;

	mutex_init(&info->i2c_access_lock);
	mutex_init(&info->adc_access_lock);
	mutex_init(&info->irq_access_lock);
	mutex_init(&info->aicr_access_lock);
	mutex_init(&info->ichg_access_lock);
	mutex_init(&info->hidden_mode_lock);
	mutex_init(&info->pe_access_lock);
	mutex_init(&info->bc12_access_lock);
	mutex_init(&info->ieoc_lock);
	mutex_init(&info->tchg_lock);
	atomic_set(&info->bc12_sdp_cnt, 0);
	atomic_set(&info->bc12_wkard, 0);

	info->client = client;
	info->dev = &client->dev;
	info->aicr_limit = -1;
	info->hidden_mode_cnt = 0;
	info->bc12_en = true;
	info->ieoc_wkard = false;
	info->ieoc = 250000; /* register default value 250mA */
	info->ichg = 2000000; /* register default value 2000mA */
	info->ichg_dis_chg = 2000000;
	info->tchg = 25;
	memcpy(info->irq_mask, rt9467_irq_maskall, RT9467_IRQIDX_MAX);

	/* Init wait queue head */
	init_waitqueue_head(&info->wait_queue);

#if !defined(CONFIG_PLATFORM_NEC)
	INIT_WORK(&info->init_work, rt9467_init_setting_work_handler);
	INIT_WORK(&info->chgdet_work, rt9467_chgdet_work_handler);
#endif /* !defined(CONFIG_PLATFORM_NEC) */

#if defined(CONFIG_PLATFORM_NEC)
	info->batt_psy = NULL;
	info->usb_detect = 0;
	mutex_init(&info->float_v_lock);
	info->prev_usb_psy_dis_ma = 0;
	info->prev_usb_psy_sav_ma = 0;
	info->real_usb_psy_ma = CURRENT_500_MA;
	info->chage_stop = 0;
	info->cradle_status = 0;
	info->current_charging_voltage_step = 0;
	info->float_voltage = charge_voltage_table[0];
	info->polling_interval_sec = POLLING_INTERVAL_SEC;
	INIT_WORK(&info->irq_work, rt9467_charger_irq_workfunc);
	INIT_DELAYED_WORK(&info->polling_work, rt9467_polling_workfunc);
#endif /* defined(CONFIG_PLATFORM_NEC) */

	/* Is HW exist */
	if (!rt9467_is_hw_exist(info)) {
		dev_notice(info->dev, "%s: no rt9467 exists\n", __func__);
		ret = -ENODEV;
		goto err_no_dev;
	}
	i2c_set_clientdata(client, info);

	ret = rt9467_parse_dt(info, &client->dev);
	if (ret < 0) {
		dev_notice(info->dev, "%s: parse dt fail\n", __func__);
		goto err_parse_dt;
	}

#ifdef CONFIG_REGMAP
	ret = rt9467_register_regmap(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s register regmap fail(%d)\n",
				      __func__, ret);
		goto err_register_rm;
	}
#endif /* CONFIG_REGMAP */

	ret = rt9467_reset_chip(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: reset chip fail\n", __func__);
		goto err_reset_chip;
	}

#if defined(CONFIG_PLATFORM_NEC)
	ret = rt9467_nec_hw_init_regs(info,0);
	if (ret < 0) {
		dev_err(info->dev, "Failed to nec_hw_init\n");
		goto err_init_setting;
	}
	/* mask all irq */
	ret = rt9467_maskall_irq(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: mask all irq fail\n", __func__);
		goto err_init_setting;
	}
#else /* defined(CONFIG_PLATFORM_NEC) */
	ret = rt9467_init_setting(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: init setting fail\n", __func__);
		goto err_init_setting;
	}
#endif /* defined(CONFIG_PLATFORM_NEC) */

	ret = rt9467_irq_register(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: irq register fail\n", __func__);
		goto err_irq_register;
	}

	ret = rt9467_irq_init(info);
	if (ret < 0) {
		dev_notice(info->dev, "%s: irq init fail\n", __func__);
		goto err_irq_init;
	}

	/* otg regulator */
	config.dev = info->dev;
	config.driver_data = info;
#ifdef CONFIG_REGMAP
	config.regmap = info->rm_dev;
#endif /* CONFIG_REGMAP */

	info->otg_rdev = devm_regulator_register(info->dev, &rt9467_otg_rdesc,
						&config);
	if (IS_ERR(info->otg_rdev)) {
		dev_notice(info->dev, "%s : regulator register fail\n", __func__);
		goto err_regulator_dev;
	}

	/* power supply register */
	memcpy(&info->psy_desc,
		&rt9467_charger_desc, sizeof(info->psy_desc));
#if !defined(CONFIG_PLATFORM_NEC)
	info->psy_desc.name = dev_name(info->dev);
#endif /* !defined(CONFIG_PLATFORM_NEC) */
	charger_cfg.drv_data = info;
	charger_cfg.of_node = info->dev->of_node;
	charger_cfg.supplied_to = rt9467_charger_supplied_to;
	charger_cfg.num_supplicants = ARRAY_SIZE(rt9467_charger_supplied_to);
	info->psy_self = devm_power_supply_register(info->dev,
					&info->psy_desc, &charger_cfg);
	if (IS_ERR_OR_NULL(info->psy_self)) {
		dev_notice(info->dev,
			   "Fail to register power supply dev, is NULL = %d\n",
			   (info->psy_self == NULL));
		ret = PTR_ERR(info->psy_self);
		goto err_register_psy;
	}

#if !defined(CONFIG_PLATFORM_NEC)
	schedule_work(&info->init_work);
#endif /* !defined(CONFIG_PLATFORM_NEC) */

	ret = device_create_file(info->dev, &dev_attr_shipping_mode);
	if (ret < 0) {
		dev_notice(info->dev, "%s: create shipping attr fail\n",
			__func__);
		goto err_register_ls_dev;
	}

#if defined(CONFIG_PLATFORM_NEC)
	ret = sysfs_create_group(&info->dev->kobj, &rt9467_attr_group);
	if (ret) {
		dev_err(info->dev, "failed to register sysfs. err: %d\n", ret);
	}
	info->monitor_interval_sec = info->polling_interval_sec;
	info->detect_interval_sec = 1;
	info->polling_interval_sec = info->detect_interval_sec;
	info->notify_src = 1;
	schedule_delayed_work(&info->polling_work, info->polling_interval_sec * HZ);
#endif /* defined(CONFIG_PLATFORM_NEC) */

	dev_info(info->dev, "%s: successfully\n", __func__);
	return ret;

err_register_ls_dev:
err_register_psy:
err_regulator_dev:
err_irq_init:
err_irq_register:
err_init_setting:
err_reset_chip:
#ifdef CONFIG_REGMAP
err_register_rm:
#endif /* CONFIG_REGMAP */
err_parse_dt:
err_no_dev:
	mutex_destroy(&info->i2c_access_lock);
	mutex_destroy(&info->adc_access_lock);
	mutex_destroy(&info->irq_access_lock);
	mutex_destroy(&info->aicr_access_lock);
	mutex_destroy(&info->ichg_access_lock);
	mutex_destroy(&info->hidden_mode_lock);
	mutex_destroy(&info->pe_access_lock);
	mutex_destroy(&info->bc12_access_lock);
	mutex_destroy(&info->ieoc_lock);
	mutex_destroy(&info->tchg_lock);
//	mutex_destroy(&info->attach_lock);
	return ret;
}


static int rt9467_remove(struct i2c_client *client)
{
	int ret = 0;
	struct rt9467_info *info = i2c_get_clientdata(client);

	pr_info("%s\n", __func__);

	if (info) {
		device_remove_file(info->dev, &dev_attr_shipping_mode);
		mutex_destroy(&info->i2c_access_lock);
		mutex_destroy(&info->adc_access_lock);
		mutex_destroy(&info->irq_access_lock);
		mutex_destroy(&info->aicr_access_lock);
		mutex_destroy(&info->ichg_access_lock);
		mutex_destroy(&info->hidden_mode_lock);
		mutex_destroy(&info->pe_access_lock);
		mutex_destroy(&info->bc12_access_lock);
		mutex_destroy(&info->ieoc_lock);
		mutex_destroy(&info->tchg_lock);
	}

	return ret;
}

static void rt9467_shutdown(struct i2c_client *client)
{
	int ret = 0;
	struct rt9467_info *info = i2c_get_clientdata(client);

	pr_info("%s\n", __func__);
	if (info) {
#if defined(CONFIG_PLATFORM_NEC)
		dev_info(info->dev, "%s: shutdown\n", __func__);
		rt9467_enter_ship_mode(info);
#endif /* defined(CONFIG_PLATFORM_NEC) */
		ret = rt9467_sw_reset(info);
		if (ret < 0)
			pr_notice("%s: sw reset fail\n", __func__);
	}
}

static int rt9467_suspend(struct device *dev)
{
	struct rt9467_info *info = dev_get_drvdata(dev);

	dev_info(dev, "%s\n", __func__);
	if (device_may_wakeup(dev))
		enable_irq_wake(info->irq);

	return 0;
}

static int rt9467_resume(struct device *dev)
{
	struct rt9467_info *info = dev_get_drvdata(dev);

	dev_info(dev, "%s\n", __func__);
	if (device_may_wakeup(dev))
		disable_irq_wake(info->irq);

	return 0;
}

static SIMPLE_DEV_PM_OPS(rt9467_pm_ops, rt9467_suspend, rt9467_resume);

static const struct i2c_device_id rt9467_i2c_id[] = {
	{"rt9467", 0},
	{}
};
MODULE_DEVICE_TABLE(i2c, rt9467_i2c_id);

static const struct of_device_id rt9467_of_match[] = {
	{ .compatible = "richtek,rt9467", },
	{},
};
MODULE_DEVICE_TABLE(of, rt9467_of_match);

#ifndef CONFIG_OF
#define RT9467_BUSNUM 1

static struct i2c_board_info rt9467_i2c_board_info __initdata = {
	I2C_BOARD_INFO("rt9467", RT9467_SALVE_ADDR)
};
#endif /* CONFIG_OF */


static struct i2c_driver rt9467_i2c_driver = {
	.driver = {
#if defined(CONFIG_PLATFORM_NEC)
		.name = "rt9467-charger",
#else /* defined(CONFIG_PLATFORM_NEC) */
		.name = "rt9467",
#endif /* defined(CONFIG_PLATFORM_NEC) */
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(rt9467_of_match),
		.pm = &rt9467_pm_ops,
	},
	.probe = rt9467_probe,
	.remove = rt9467_remove,
	.shutdown = rt9467_shutdown,
	.id_table = rt9467_i2c_id,
};

static int __init rt9467_init(void)
{
	int ret = 0;

#ifdef CONFIG_OF
	pr_info("%s: with dts\n", __func__);
#else
	pr_info("%s: without dts\n", __func__);
	i2c_register_board_info(RT9467_BUSNUM, &rt9467_i2c_board_info, 1);
#endif

	ret = i2c_add_driver(&rt9467_i2c_driver);
	if (ret < 0)
		pr_notice("%s: register i2c driver fail\n", __func__);

	return ret;
}
module_init(rt9467_init);


static void __exit rt9467_exit(void)
{
	i2c_del_driver(&rt9467_i2c_driver);
}
module_exit(rt9467_exit);


MODULE_LICENSE("GPL");
MODULE_AUTHOR("ShuFanLee <shufan_lee@richtek.com>");
MODULE_DESCRIPTION("RT9467 Charger Driver");
MODULE_VERSION(RT9467_DRV_VERSION);

/*
 * Release Note
 * 1.0.19
 * (1) Fast toggle chgdet flow
 * (2) When chg_type is DCP, enable chgdet for D+ 0.6V
 * (3) mutex_unlock() once in rt9467_attachi_irq_handler()
 * (4) Lower UG driver slew rate
 * (5) Show VBUS, CV in rt9467_dump_register()
 * (6) Revise enable_auto_sensing function, set auto sensing bit when enable
	it, otherwise clear it.
 * (7) Add shipping_mode_store node, and add adc_access_lock to avoid
	conflicting access with adc measurement.
 * (8) Add safety_check ops, trigger ieoc event, when ibat < ieoc for 3 times.
 * (9) Add workaround for vsys overshoot
 *
 * 1.0.18
 * (1) Check tchg 3 times if it >= 120 degree
 *
 * 1.0.17
 * (1) Add ichg workaround
 *
 * 1.0.16
 * (1) Fix type error of enable_auto_sensing in sw_reset
 * (2) Move irq_mask to info structure
 * (3) Remove config of Charger_Detect_Init/Release
 *
 * 1.0.15
 * (1) Do ilim select in WQ and register charger class in probe
 *
 * 1.0.14
 * (1) Disable attach delay
 * (2) Enable IRQ_RZE at the end of irq handler
 * (3) Remove IRQ related registers from reg_addr
 * (4) Recheck status in irq handler
 * (5) Use bc12_access_lock instead of chgdet_lock
 *
 * 1.0.13
 * (1) Add do event interface for polling mode
 * (2) Check INT pin after reading evt
 *
 * 1.0.12
 * (1) Add MTK_SSUSB config for Charger_Detect_Init/Release
 *
 * 1.0.11
 * (1) Disable psk mode in pep20_reest
 * (2) For secondary chg, enter shipping mode before shdn
 * (3) Add BC12 sdp workaround
 * (4) Remove enabling/disabling ILIM in chgdet_handler
 *
 * 1.0.10
 * (1) Add IEOC workaround
 * (2) Release set_ieoc/enable_te interface
 * (3) Fix type errors
 *
 * 1.0.9
 * (1) Add USB workaround for CDP port
 * (2) Plug in -> usbsw to charger, after chgdet usbsw to AP
 *     Plug out -> usbsw to AP
 * (3) Filter out not changed irq state
 * (4) Not to use CHG_IRQ3
 *
 * 1.0.8
 * (1) Set irq to wake up system
 * (2) Refine I2C driver related table
 *
 * 1.0.7
 * (1) Enable/Disable ILIM in chgdet_handler
 *
 * 1.0.6
 * (1) Prevent backboot
 * (2) Add CEB pin control for secondary charger
 * (3) After PE pattern -> Enable skip mode
 *     Disable skip mode -> Start PE pattern
 * (4) Disable BC12 detection before reset IRQ in init_setting
 *
 * 1.0.5
 * (1) Remove wait CDP flow
 * (2) Add rt9467_chgdet_handler for attachi/detachi
 * (3) Set secondary chg to HZ if it is not in charging mode
 * (4) Add is_charging_enabled, get_min_ichg OPS
 *
 * 1.0.4
 * (1) Set ichg&aicr, enable chg before sending PE+ series pattern
 * (2) Add enable_cable_drop_com OPS
 *
 * 1.0.3
 * (1) IRQs are default unmasked before E4, need to mask them manually
 *
 * 1.0.2
 * (1) Fix AICL workqueue lock issue
 *
 * 1.0.1
 * (1) Fix IRQ init sequence
 *
 * 1.0.0
 * (1) Initial released
 */
