// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2020 MediaTek Inc.
 */

#include <linux/err.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/of_device.h>
#include <linux/platform_device.h>
#include <md_cooling.h>
#include <mtk_thermal_trace.h>

#define THROTTLE_ACTIVE_PERIOD_LV1	(400)
#define THROTTLE_ACTIVE_PERIOD_LV2	(300)
#define THROTTLE_ACTIVE_PERIOD_LV3	(200)
#define THROTTLE_ACTIVE_PERIOD_LV4	(100)
#define THROTTLE_ACTIVE_PERIOD_LV5	(100)
#define THROTTLE_ACTIVE_PERIOD_LV6	(100)
#define THROTTLE_ACTIVE_PERIOD_LV7	(100)

#define THROTTLE_SUSPEND_PERIOD_LV1	(100)
#define THROTTLE_SUSPEND_PERIOD_LV2	(100)
#define THROTTLE_SUSPEND_PERIOD_LV3	(100)
#define THROTTLE_SUSPEND_PERIOD_LV4	(200)
#define THROTTLE_SUSPEND_PERIOD_LV5	(300)
#define THROTTLE_SUSPEND_PERIOD_LV6	(400)
#define THROTTLE_SUSPEND_PERIOD_LV7	(600)
#define MAX_IMS_LV (7)

static int md_cooling_ul_control_get_max_state(
	struct thermal_cooling_device *cdev, unsigned long *state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;

	*state = md_cdev->max_level;

	return 0;
}

static int md_cooling_ul_control_get_cur_state(
	struct thermal_cooling_device *cdev, unsigned long *state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;

	*state = md_cdev->target_level;

	return 0;
}

static int md_cooling_ul_control_set_cur_state(
		struct thermal_cooling_device *cdev, unsigned long state)
{
	struct md_cooling_device *md_cdev = cdev->devdata;
	struct device *dev = (struct device *)md_cdev->dev_data;
	enum md_status status;
	unsigned int ul_control[MAX_IMS_LV][MAX_IMS_LV] = {
		{THROTTLE_ACTIVE_PERIOD_LV1, THROTTLE_SUSPEND_PERIOD_LV1},
		{THROTTLE_ACTIVE_PERIOD_LV2, THROTTLE_SUSPEND_PERIOD_LV2},
		{THROTTLE_ACTIVE_PERIOD_LV3, THROTTLE_SUSPEND_PERIOD_LV3},
		{THROTTLE_ACTIVE_PERIOD_LV4, THROTTLE_SUSPEND_PERIOD_LV4},
		{THROTTLE_ACTIVE_PERIOD_LV5, THROTTLE_SUSPEND_PERIOD_LV5},
		{THROTTLE_ACTIVE_PERIOD_LV6, THROTTLE_SUSPEND_PERIOD_LV6},
		{THROTTLE_ACTIVE_PERIOD_LV7, THROTTLE_SUSPEND_PERIOD_LV7}
	};
	unsigned int msg;
	int ret = 0;

	/* Request state should be less than max_level */
	if (WARN_ON(state > md_cdev->max_level))
		return -EINVAL;

	if (md_cdev->target_level == state)
		return 0;

	status = get_md_status();
	if (is_md_off(status)) {
		dev_info(dev, "skip ul_control due to MD is off\n");
		md_cdev->target_level = MD_COOLING_UNLIMITED_LV;
		return 0;
	}

	if (state == 0)
		msg = TMC_THROTTLE_DISABLE_MSG;
	else if (state == 8)
		/* set active/suspend=1/255 for no IMS case */
		msg = duty_ctrl_to_tmc_msg(1, 255, 0);
	else
		msg = duty_ctrl_to_tmc_msg(ul_control[state - 1][0]/100,
				ul_control[state - 1][1]/100, 1);

	ret = send_throttle_msg(msg);
	if (!ret)
		md_cdev->target_level = state;

	dev_info(dev, "%s: set lv = %ld done\n", md_cdev->name, state);
	trace_md_ul_control(md_cdev, status);

	return ret;
}

static const struct of_device_id md_cooling_ul_control_of_match[] = {
	{ .compatible = "mediatek,md-cooler-ul-control", },
	{},
};
MODULE_DEVICE_TABLE(of, md_cooling_ul_control_of_match);

static struct thermal_cooling_device_ops md_cooling_ul_control_ops = {
	.get_max_state		= md_cooling_ul_control_get_max_state,
	.get_cur_state		= md_cooling_ul_control_get_cur_state,
	.set_cur_state		= md_cooling_ul_control_set_cur_state,
};

static int md_cooling_ul_control_probe(struct platform_device *pdev)
{
	struct device_node *np = pdev->dev.of_node;
	struct device *dev = &pdev->dev;
	int ret = -1;

	if (!np) {
		dev_err(dev, "MD cooler DT node not found\n");
		return -ENODEV;
	}

	ret = md_cooling_register(np,
				MD_COOLING_TYPE_UL_CONTROL,
				MAX_NUM_UL_CONTROL_LV,
				NULL,
				&md_cooling_ul_control_ops,
				dev);
	if (ret) {
		dev_err(dev, "register ul-control cdev failed!\n");
		return ret;
	}

	return ret;
}

static int md_cooling_ul_control_remove(struct platform_device *pdev)
{
	md_cooling_unregister(MD_COOLING_TYPE_UL_CONTROL);

	return 0;
}

static struct platform_driver md_cooling_ul_control_driver = {
	.probe = md_cooling_ul_control_probe,
	.remove = md_cooling_ul_control_remove,
	.driver = {
		.name = "mtk-md-cooling-ul-control",
		.of_match_table = md_cooling_ul_control_of_match,
	},
};
module_platform_driver(md_cooling_ul_control_driver);

MODULE_AUTHOR("Meng-Hsun Chou <meng-hsun.chou@mediatek.com>");
MODULE_DESCRIPTION("Mediatek modem cooling UL_control driver");
MODULE_LICENSE("GPL v2");
