#include <typedefs.h>
#include <pmic.h>
#include <regulator/mtk_regulator_core.h>
#include <mt6330.h>


#ifdef LDO_SUPPORT
static const unsigned int vio18_2_volts[] = {
	1600000,
	1700000,
	1800000,
};

static const unsigned int vmdd2_volts[] = {
	1000000,
	1050000,
	1100000,
	1150000,
};

static const unsigned int vefuse_volts[] = {
	1700000,
	1800000,
	2000000,
};

static const unsigned int vio18_1_volts[] = {
	1600000,
	1700000,
	1800000,
};

static const unsigned int vmddr_volts[] = {
	700000,
	750000,
	800000,
};

static const unsigned int vmddq_volts[] = {
	550000,
	600000,
	650000,
};

static const unsigned int vio18_2_idxs[] = {
	10, 11, 12,
};

static const unsigned int vmdd2_idxs[] = {
	6, 7, 8, 9,
};

static const unsigned int vefuse_idxs[] = {
	3, 4, 5,
};

static const unsigned int vio18_1_idxs[] = {
	10, 11, 12,
};

static const unsigned int vmddr_idxs[] = {
	3, 4, 5,
};

static const unsigned int vmddq_idxs[] = {
	0, 1, 2,
};

static struct mt6330_ldo_info ldo_ext_info[] = {
	mt6330_ldo_decl(vio18_2_volts, vio18_2_idxs),
	mt6330_ldo_decl(vmdd2_volts, vmdd2_idxs),
	mt6330_ldo_decl(vefuse_volts, vefuse_idxs),
	mt6330_ldo_decl(vio18_1_volts, vio18_1_idxs),
	mt6330_ldo_decl(vmddr_volts, vmddr_idxs),
	mt6330_ldo_decl(vmddq_volts, vmddq_idxs),
};
#endif /*--LDO_SUPPORT--*/

#ifdef LDO_VOTRIM_SUPPORT
static const int votrim_1_type[] = {
	0, (-1)*20000, (-1)*40000, (-1)*60000, (-1)*80000, (-1)*100000, (-1)*120000, (-1)*140000,
	160000, 140000, 1200000, 100000, 80000, 60000, 40000, 20000,
};

static const int votrim_2_type[] = {
	0, (-1)*10000, (-1)*20000, (-1)*30000, (-1)*40000, (-1)*50000, (-1)*60000, (-1)*70000,
	80000, 70000, 60000, 50000, 40000, 30000, 20000, 10000,
};

static const int votrim_3_type[] = {
	0, (-1)*5000, (-1)*10000, (-1)*15000, (-1)*20000, (-1)*25000, (-1)*30000, (-1)*35000,
	40000, 35000, 30000, 25000, 20000, 15000, 10000, 5000,
};

static struct mt6330_ldo_trim_info ldo_trim_info[] = {
	mt6330_ldo_trim_decl(vio18_2, votrim_2_type),
	mt6330_ldo_trim_decl(vmdd2, votrim_3_type),
	mt6330_ldo_trim_decl(vefuse, votrim_1_type),
	mt6330_ldo_trim_decl(vio18_1, votrim_2_type),
	mt6330_ldo_trim_decl(vmddr, votrim_3_type),
	mt6330_ldo_trim_decl(vmddq, votrim_3_type),
};
#endif /*--LDO_VOTRIM_SUPPORT--*/

static struct mt6330_regulator_info mt6330_regu_info[] = {
	mt6330_decl(vmd12),
	mt6330_decl(vrfdig),
	mt6330_decl(vcore),
	mt6330_decl(vmd11),
	mt6330_decl(vsram_md),
	mt6330_decl(vsram_core),
	mt6330_decl(vsram_proc),
	mt6330_decl(vsram_rfdig),
#ifdef LDO_SUPPORT
	mt6330_decl(vio18_2),
	mt6330_decl(vmdd2),
	mt6330_decl(vefuse),
	mt6330_decl(vio18_1),
	mt6330_decl(vmddr),
	mt6330_decl(vmddq),
#endif /*--LDO_SUPPORT--*/
};

static unsigned char regu_size = ARRAY_SIZE(mt6330_regu_info);

static int mt6330_set_voltage(unsigned char id, unsigned int volt, unsigned int max_volt) {
	unsigned short selector = 0;
	int ret = 0;

	if (volt < mt6330_regu_info[id].min_uV ||
	    volt > mt6330_regu_info[id].max_uV + 100000 ||
	    (mt6330_regu_info[id].rtype == REGULAR_VOLTAGE && volt > mt6330_regu_info[id].max_uV)) {
		mreg_dbg_print("vp\n");
		return -1;
	}

	if (mt6330_regu_info[id].rtype == REGULAR_VOLTAGE)
		selector = DIV_ROUND_UP((volt - mt6330_regu_info[id].min_uV),
			mt6330_regu_info[id].step_uV);
#ifdef LDO_SUPPORT
	else if (mt6330_regu_info[id].rtype == NON_REGULAR_VOLTAGE) {
		selector = mt6330_ldo_convert_data(id, volt, VOLTOSEL);
		if (selector == 0xFFFF) {
			mreg_dbg_print("vnf\n"); /* voltage not found */
			return -1;
		}
	} else if (mt6330_regu_info[id].rtype == FIXED_REGULAR_VOLTAGE) {
		if (mt6330_ldo_convert_data(id, volt, VOLTOSEL) == 0)
			return 0;
		else {
			mreg_dbg_print("vswf\n");
			return -1;
		}
	}
#else
	else {
		mreg_dbg_print("ldo not support\n");
		return -1;
	}
#endif /*--LDO_SUPPORT--*/

	mreg_dbg_print("1 %d,%d\n", id, selector);

	ret = pmic_config_interface(mt6330_regu_info[id].vol_reg, selector,
				    mt6330_regu_info[id].vol_mask,
				    mt6330_regu_info[id].vol_shift);
	return ret;
}

static int mt6330_get_voltage(unsigned char id)
{
	unsigned int selector = 0, offset = 0;
	unsigned int volt = 0;
	int ret = 0;

	if (mt6330_regu_info[id].da_vol_reg != 0) {
		ret = pmic_read_interface(
			mt6330_regu_info[id].da_vol_reg,
			&selector,
			mt6330_regu_info[id].da_vol_mask,
			mt6330_regu_info[id].da_vol_shift);
	} else {
		ret = pmic_read_interface(
			mt6330_regu_info[id].vol_reg,
			&selector,
			mt6330_regu_info[id].vol_mask,
			mt6330_regu_info[id].vol_shift);
	}

	if (ret)
		return -1;

	if (mt6330_regu_info[id].rtype == REGULAR_VOLTAGE)
		volt = ((selector * mt6330_regu_info[id].step_uV) + mt6330_regu_info[id].min_uV);
#ifdef LDO_SUPPORT
	else if (mt6330_regu_info[id].rtype == NON_REGULAR_VOLTAGE)
		volt = mt6330_ldo_convert_data(id, selector, SELTOVOL);
	else if (mt6330_regu_info[id].rtype == FIXED_REGULAR_VOLTAGE)
		volt = *((int *)(mt6330_regu_info[id].extinfo->pvoltages));
	if (mt6330_regu_info[id].rtype != REGULAR_VOLTAGE) {
		ret = pmic_read_interface(mt6330_regu_info[id].vocal_reg,
					  &offset,
					  mt6330_regu_info[id].vocal_mask,
					  mt6330_regu_info[id].vocal_shift);
		if (id == 9 || id == 12 || id == 13)
			volt += (offset / 2) * 10000;
		else
			volt += offset * 10000;
	}
#else
	else
		return -1;
#endif /*--LDO_SUPPORT--*/

	mreg_dbg_print("get volt %d, %d, %d, %d\n", id, selector, offset, volt);
	if (volt < mt6330_regu_info[id].min_uV ||
	    volt > mt6330_regu_info[id].max_uV + 100000 ||
	    (mt6330_regu_info[id].rtype == REGULAR_VOLTAGE && volt > mt6330_regu_info[id].max_uV)) {
		mreg_dbg_print("vgw\n");
		return -1;
	}

	return volt;
}

static int mt6330_enable(unsigned char id, unsigned char en)
{
	int ret = 0;

	if (mt6330_regu_info[id].enable_reg == 0)
		return -1;
	mreg_dbg_print("2 %d,%d\n", id, en);
	ret = pmic_config_interface(mt6330_regu_info[id].enable_reg, en, 0x1,
		mt6330_regu_info[id].enable_shift);

	return ret;
}

static int mt6330_is_enabled(unsigned char id)
{
	unsigned int en = 0;
	unsigned int ret = 0;

	if (mt6330_regu_info[id].enable_reg == 0)
		return -1;
	ret = pmic_read_interface(mt6330_regu_info[id].enable_reg, &en, 0x1,
		mt6330_regu_info[id].enable_shift);
	mreg_dbg_print("3 %d,%d\n", id, en);

	return (ret ? ret : en) ;
}

static int mt6330_set_mode(unsigned char id, unsigned char mode)
{
	int ret = 0;

	if (mt6330_regu_info[id].mode_reg == 0)
		return -1;
	mreg_dbg_print("4 %d,%d\n", id, mode);
	ret = pmic_config_interface(mt6330_regu_info[id].mode_reg, mode, 0x1,
		mt6330_regu_info[id].mode_shift);
	/* Special setting for VMD11/12 2phase buck */
	if (id == MTK_REGULATOR_VMD11)
		ret = pmic_config_interface(PMIC_RG_VMD12_FCCM_ADDR, mode,
					    PMIC_RG_VMD12_FCCM_MASK,
					    PMIC_RG_VMD12_FCCM_SHIFT);

	return ret;
}

static int mt6330_get_mode(unsigned char id)
{
	int mode = 0;
	int ret = 0;

	if (mt6330_regu_info[id].mode_reg == 0)
		return -1;
	ret = pmic_read_interface(mt6330_regu_info[id].mode_reg, &mode, 0x1,
		mt6330_regu_info[id].mode_shift);
	mreg_dbg_print("5 %d,%d\n", id, mode);

	return (ret ? ret: mode);
}

#ifdef LDO_VOTRIM_SUPPORT
static int mt6330_ldo_votrim_convert_data(unsigned char id, int cnvdata, trimseltran transtype)
{
	int i = 0, trim_size = 0, choice = -1;
	const int *trim_Voltage;

	if ((mt6330_regu_info[id].triminfo->trim_voltages != NULL)) {
		trim_Voltage = mt6330_regu_info[id].triminfo->trim_voltages;
		trim_size = mt6330_regu_info[id].triminfo->trim_size;
		mreg_dbg_print("votrim_size %d, cnvdata %d\n", trim_size, cnvdata);
		switch (transtype) {
		case TRIMTOSEL:
			if (cnvdata > 0) {
				for (i = trim_size/2; i < trim_size; i++) {
					choice = i;
					if (trim_Voltage[i] <= cnvdata) {
						dbg_print("trim_Voltage:%d, cnvdata:%d\n", trim_Voltage[i], cnvdata);
						break;
					}
				}
			} else if (cnvdata < 0) {
				for (i = trim_size/2 - 1; i >= 0; i--) {
					choice = i;
					if (trim_Voltage[i] >= cnvdata) {
						dbg_print("trim_Voltage:%d, cnvdata:%d\n", trim_Voltage[i], cnvdata);
						break;
					}
				}
			} else
				choice = 0;
			break;
		case SELTOTRIM:
			choice = trim_Voltage[cnvdata];
			break;
		default:
			break;
		}
	}

	return choice;
}

static int mt6330_set_votrim(unsigned char id, int trim_volt) {
	int selector = 0;
	int ret = 0;

	selector = mt6330_ldo_votrim_convert_data(id, trim_volt, TRIMTOSEL);
	if (selector == -1)
		return -1;

	if ((mt6330_regu_info[id].triminfo->trim_voltages != NULL)) {
		mreg_dbg_print("6 %d,%d\n", id, selector);
			pmic_config_interface(PMIC_TMA_KEY_L_ADDR, 0xcf,
			      PMIC_TMA_KEY_L_MASK,
			      PMIC_TMA_KEY_L_SHIFT);
			pmic_config_interface(PMIC_TMA_KEY_H_ADDR, 0x9c,
			      PMIC_TMA_KEY_H_MASK,
			      PMIC_TMA_KEY_H_SHIFT);
		ret = pmic_config_interface(
			mt6330_regu_info[id].triminfo->trim_reg,
			(unsigned int)selector,
			mt6330_regu_info[id].triminfo->trim_mask,
			mt6330_regu_info[id].triminfo->trim_shift);
			pmic_config_interface(PMIC_TMA_KEY_L_ADDR, 0,
			      PMIC_TMA_KEY_L_MASK,
			      PMIC_TMA_KEY_L_SHIFT);
			pmic_config_interface(PMIC_TMA_KEY_H_ADDR, 0,
			      PMIC_TMA_KEY_H_MASK,
			      PMIC_TMA_KEY_H_SHIFT);
	}

	return ret;
}

static int mt6330_get_votrim(unsigned char id)
{
	unsigned int selector = 0;
	int ret = 0;

	ret = pmic_read_interface(
		mt6330_regu_info[id].triminfo->trim_reg,
		&selector,
		mt6330_regu_info[id].triminfo->trim_mask,
		mt6330_regu_info[id].triminfo->trim_shift);
	if (ret)
		return -1;

	ret = mt6330_ldo_votrim_convert_data(id, selector, SELTOTRIM);

	mreg_dbg_print("7 %d,%d,%d\n", id, selector,ret);

	return ret;
}
#endif /*--LDO_VOTRIM_SUPPORT--*/

#ifdef LDO_SUPPORT
static unsigned int mt6330_ldo_convert_data(unsigned char id, unsigned int cnvdata, volseltran transtype)
{
	int i = 0, n_size = 0;
	unsigned int choice = 0xFFFF;
	unsigned int offset = 0;
	const unsigned int *pVoltage;
	const unsigned int *iDx;

	if ((mt6330_regu_info[id].extinfo->pvoltages == NULL) || (mt6330_regu_info[id].extinfo->idxs == NULL))
		return choice;
	pVoltage = mt6330_regu_info[id].extinfo->pvoltages;
	iDx = mt6330_regu_info[id].extinfo->idxs;
	n_size = mt6330_regu_info[id].extinfo->n_size;
	mreg_dbg_print("%s, cnvdata %d, n_size %d\n", __func__, cnvdata, n_size);
	for (i = 0; i <= n_size; i++) {
		if (id == 9 || id == 12 || id == 13) {
			if (transtype == VOLTOSEL) {
				if (cnvdata >= pVoltage[i] && cnvdata < pVoltage[i] + 50000) {
					choice = iDx[i];
					offset = ((cnvdata - pVoltage[i]) / 10000);
					pmic_config_interface(mt6330_regu_info[id].vocal_reg,
							      offset * 2,
							      mt6330_regu_info[id].vocal_mask,
							      mt6330_regu_info[id].vocal_shift);
					break;
				}
			} else if (transtype == SELTOVOL && iDx[i] == cnvdata) {
				choice = pVoltage[i];
				break;
			}
		} else {
			if (transtype == VOLTOSEL) {
				if (cnvdata >= pVoltage[i] && cnvdata < pVoltage[i] + 100000) {
					choice = iDx[i];
					offset = (cnvdata - pVoltage[i]) / 10000;
					pmic_config_interface(mt6330_regu_info[id].vocal_reg,
							      offset,
							      mt6330_regu_info[id].vocal_mask,
							      mt6330_regu_info[id].vocal_shift);
					break;
				}
			} else if (transtype == SELTOVOL && iDx[i] == cnvdata) {
				choice = pVoltage[i];
				break;
			}
		}
	}

	return choice;
}
#endif /*--LDO_SUPPORT--*/

static struct regulator_ctrl mt6330_regulator_ctrl = {
	.set_voltage = mt6330_set_voltage,
	.get_voltage = mt6330_get_voltage,
	.enable = mt6330_enable,
	.is_enabled = mt6330_is_enabled,
	.set_mode = mt6330_set_mode,
	.get_mode = mt6330_get_mode,
#ifdef LDO_VOTRIM_SUPPORT
	.set_votrim = mt6330_set_votrim,
	.get_votrim = mt6330_get_votrim,
#endif /*--LDO_VOTRIM_SUPPORT--*/
};

static struct mtk_regulator mt6330_regulator[] = {
	{
		.name = "vmd12",
		.id = MTK_REGULATOR_VMD12,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vrfdig",
		.id = MTK_REGULATOR_VRFDIG,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vcore",
		.id = MTK_REGULATOR_VCORE,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vmd11",
		.id = MTK_REGULATOR_VMD11,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vsram_md",
		.id = MTK_REGULATOR_VSRAM_MD,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vsram_core",
		.id = MTK_REGULATOR_VSRAM_CORE,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vsram_proc",
		.id = MTK_REGULATOR_VSRAM_PROC,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vsram_rfdig",
		.id = MTK_REGULATOR_VSRAM_RFDIG,
		.reg_ops = &mt6330_regulator_ctrl,
	},
#ifdef LDO_SUPPORT
	{
		.name = "vio18_2",
		.id = MTK_REGULATOR_VIO18_2,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
#ifdef EXT_BUCK_MT6315
		.name = "vmdd2_no_used",
#else
		.name = "vmdd2",
#endif
		.id = MTK_REGULATOR_VMDD2,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vefuse",
		.id = MTK_REGULATOR_VEFUSE,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vio18_1",
		.id = MTK_REGULATOR_VIO18_1,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vmddr",
		.id = MTK_REGULATOR_VMDDR,
		.reg_ops = &mt6330_regulator_ctrl,
	},
	{
		.name = "vmddq",
		.id = MTK_REGULATOR_VMDDQ,
		.reg_ops = &mt6330_regulator_ctrl,
	},
#endif /*--LDO_SUPPORT--*/
};

void mt6330_wdtdbg_vosel(void)
{
	unsigned int val = 0;

	pmic_read_interface(PMIC_BUCK_VCORE_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_BUCK_VCORE_WDTDBG_VOSEL_MASK,
			    PMIC_BUCK_VCORE_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC] latch VCORE %d uV(0x%x)\n", vcore_min_uV + vcore_step_uV * gray_to_binary(val), val);

	pmic_read_interface(PMIC_BUCK_VMD11_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_BUCK_VMD11_WDTDBG_VOSEL_MASK,
			    PMIC_BUCK_VMD11_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC] latch VMD11 %d uV(0x%x)\n", vmd11_min_uV + vmd11_step_uV * gray_to_binary(val), val);

	pmic_read_interface(PMIC_BUCK_VRFDIG_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_BUCK_VRFDIG_WDTDBG_VOSEL_MASK,
			    PMIC_BUCK_VRFDIG_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC] latch VRFDIG %d uV(0x%x)\n", vrfdig_min_uV + vrfdig_step_uV * gray_to_binary(val), val);

	pmic_read_interface(PMIC_BUCK_VSRAM_MD_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_BUCK_VSRAM_MD_WDTDBG_VOSEL_MASK,
			    PMIC_BUCK_VSRAM_MD_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC] latch VSRAM_MD %d uV(0x%x)\n", vsram_md_min_uV + vsram_md_step_uV * gray_to_binary(val), val);

	pmic_read_interface(PMIC_LDO_VSRAM_PROC_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_LDO_VSRAM_PROC_WDTDBG_VOSEL_MASK,
			    PMIC_LDO_VSRAM_PROC_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC] latch VSRAM_PROC %d uV(0x%x)\n", vsram_proc_min_uV + vsram_proc_step_uV * gray_to_binary(val), val);

	pmic_read_interface(PMIC_LDO_VSRAM_CORE_WDTDBG_VOSEL_ADDR, &val,
			    PMIC_LDO_VSRAM_CORE_WDTDBG_VOSEL_MASK,
			    PMIC_LDO_VSRAM_CORE_WDTDBG_VOSEL_SHIFT);
	mreg_dbg_print("[PMIC]latch VSRAM_CORE %d uV(0x%x)\n", vsram_core_min_uV + vsram_core_step_uV * gray_to_binary(val), val);

}
/* ====================
 * Driver operations
 * ====================
 */
int mt6330_probe(void)
{
	int ret = 0;
	unsigned int i = 0;

	for (i = 0; i < regu_size; i++) {
#ifdef LDO_SUPPORT
		if (mt6330_regu_info[i].rtype != REGULAR_VOLTAGE && i >= MTK_REGULATOR_LDO_SUPPORT)
			mt6330_regu_info[i].extinfo = &ldo_ext_info[(i-MTK_REGULATOR_LDO_SUPPORT)];
#endif /*--LDO_SUPPORT--*/
#ifdef LDO_VOTRIM_SUPPORT
		if (mt6330_regu_info[i].rtype != REGULAR_VOLTAGE  && i >= MTK_REGULATOR_LDO_SUPPORT)
			mt6330_regu_info[i].triminfo = &ldo_trim_info[(i-MTK_REGULATOR_LDO_SUPPORT)];
#endif /*--LDO_SUPPORT--*/
		ret = mtk_simple_regulator_register(&mt6330_regulator[i]);
		if (ret < 0) {
			/* register mtk regulator error */
			mreg_dbg_print("[PMIC] regulator %s\n", mt6330_regulator[i].name);
			return ret;
		}
	}

	return 0;
}
