#include "boot/dev_interface/gpio_interface.h"
#include "boot/protocol_callbacks.h"
#include "error_code.h"
#include "nandx_util.h"
#include "nandx_core.h"
#include "bbt.h"
#include "lite_nftl.h"

enum nand_type {
	NAND_SPI,
	NAND_SLC,
	NAND_MLC,
	NAND_TLC
};

static s8 get_boot_device()
{
	enum BOOT_TRAP_E val;

	val = gpio_get_boot_trap();
	if (val == BOOT_TRAP_SNAND)
		return NAND_SPI;
	else if (val == BOOT_TRAP_PNAND)
		return NAND_SLC;
	else
		return -1;
}

struct nandx_info g_nandi;
static u8 *head_buf, *tail_buf;

typedef int (*func_nftl_operation)(struct nftl_part *, u8 *, u64, size_t);

int erase_operation(u64 offset, size_t len,
		    const struct progress_cb *cb, bool is_force)
{
	u32	erase_done, erase_total;
	u32 block_size = g_nandi.block_size;
	u32     progress = 0;
	u64 start_offset = div_down(offset, block_size) * block_size;
	int ret = 0;

	pr_debug("addr 0x%llx, len 0x%lx\n", offset, len);

	erase_done = 0;
	erase_total = div_up(len, block_size);

	pr_debug("erase %d blocks from 0x%llx\n", erase_total, start_offset);

	while (erase_done < erase_total) {
		pr_debug("erase at blk %llx\n", start_offset);
		if (!is_force) {
			if (bbt_is_bad(&g_nandi, start_offset)) {
				pr_info("Skip bad block 0x%llx, not erase!\n", start_offset);
			} else {
				ret = nandx_erase(start_offset, block_size);
				if (ret < 0)
					pr_err("erase fail at blk 0x%llx, ret: %d\n", start_offset, ret);

				if (ret == -ENANDERASE)
					ret = bbt_mark_bad(&g_nandi, start_offset);

				if (ret < 0)
					return ret;
			}
		} else {
			ret = nandx_erase(start_offset, block_size);
			if (ret < 0) {
				pr_err("erase fail at blk 0x%llx, ret: %d\n", start_offset, ret);
				/* Ignore bad block for force erase. */
				if (ret == -ENANDERASE)
					ret = 0;
				else
					return ret;
			}
		}

		start_offset += block_size;

		erase_done++;
		if (cb) {
			u32	progress = 0;

			progress = (erase_done * 100) / erase_total;
			if (progress % 10 == 0)
				ret = cb->cb(cb->user_arg, progress, "Erase");
		}
	}
	return ret;
}

int nand_bbt_scan(const struct progress_cb *cb)
{
	/** DA call this fun too late after PMT read
	 *  so we make scan_bbt in interface_nandx_init
	 */
	return 0;
}

int rw_operation(struct nftl_part *part, u8 *buf, u64 offset, size_t len,
		 bool read)
{
	struct nandx_split64 split = {0};
	func_nftl_operation operation;
	u64 val, i;
	int ret = 0, blocks;
	u8 *lbuf = buf;
	u32 split_size = g_nandi.page_size;

	pr_debug("addr %x, len %x\n", (u32)offset, (u32)len);

	operation = read ? nftl_read : nftl_write;

	nandx_split(&split, offset, len, val, split_size);
	if (split.head_len) {
		if (!read) {
			memset(head_buf, 0xff, split_size);
			memcpy(head_buf + reminder(split.head, split_size),
			       lbuf, split.head_len);
		}

		ret = operation(part, head_buf,
				div_round_down(split.head, split_size),
				split_size);
		if (ret < 0) {
			pr_err("%s, %d, fail read: %d, offset: %x, ret: %d\n",
			       __func__, __LINE__, read, (u32)split.head, ret);
			return STATUS_NAND_ERR;
		}
		lbuf += split.head_len;
	}

	if (split.body_len) {
		ret = operation(part, lbuf, split.body, split.body_len);
		if (ret < 0) {
			pr_err("%s, %d, fail read: %d, offset: %x, ret: %d\n",
			       __func__, __LINE__, read, (u32)split.body, ret);
			return STATUS_NAND_ERR;
		}

		lbuf += split.body_len;
	}

	if (split.tail_len) {
		if (!read) {
			memset(tail_buf, 0xff, split_size);
			memcpy(tail_buf, lbuf, split.tail_len);
		}

		ret = operation(part, tail_buf,
				div_round_down(split.tail, split_size),
				split_size);
		if (ret < 0) {
			pr_err("%s, %d, fail read: %d, offset: %x, ret: %d\n",
			       __func__, __LINE__, read, (u32)split.tail, ret);
			return STATUS_NAND_ERR;
		}
	}

	if (read && split.head_len)
		memcpy(buf, head_buf + reminder(split.head, split_size),
		       split.head_len);
	if (read && split.tail_len)
		memcpy(buf + split.head_len + split.body_len, tail_buf,
		       split.tail_len);

	return 0;
}


static void nand_gpio_init(u8 nand_type)
{

#define GPIO_BASE                   (0x10005000)    // GPIO

/* For NFI GPIO Pinmux setting */
#define NFI_GPIO_MODE26         (GPIO_BASE + 0x4A0)
#define NFI_GPIO_MODE27         (GPIO_BASE + 0x4B0)
#define NFI_GPIO_MODE28         (GPIO_BASE + 0x4C0)
#define SNFI_GPIO_MODE27        (GPIO_BASE + 0x4B0)
#define SNFI_GPIO_MODE28        (GPIO_BASE + 0x4C0)

#define GPIO_IOCFG_TL_BASE      (0x11F00000)
#define GPIO_DRV_CFG0           (GPIO_IOCFG_TL_BASE)
#define GPIO_DRV_CFG1           (GPIO_IOCFG_TL_BASE + 0x10)
#define GPIO_PD_CFG0            (GPIO_IOCFG_TL_BASE + 0x40)
#define GPIO_PU_CFG0            (GPIO_IOCFG_TL_BASE + 0x50)
#define GPIO_RDSEL_CFG0         (GPIO_IOCFG_TL_BASE + 0x60)
#define GPIO_TDSEL_CFG0         (GPIO_IOCFG_TL_BASE + 0x80)

	if (nand_type == NAND_SLC)
	{
		nandx_set_bits32(NFI_GPIO_MODE26, 0x70000000, (1 << 28));
		/* NCEB1, NLD7, NLD6, NLD5, NLD4, NLD3, NLD2, NLD1 */
		nandx_set_bits32(NFI_GPIO_MODE27, 0x77777777, (1 << 28) | (1 << 24) | (1 << 20) | (1 << 16) | (1 << 12) | (1 << 8) | (1 << 4) | (1 << 0));
		/* NCLE, NALE, NWEB, NREB, NRNB, NCEB0 */
		nandx_set_bits32(NFI_GPIO_MODE28, 0x777777,  (1 << 20) | (1 << 16) | (1 << 12) | (1 << 8) | (1 << 4) | (1 << 0));
		/* Pull up/down setting */
		/*          25     24     23     22     21     20     19     18     17     16
		         NWEB   NRNB   NREB   NLD7   NLD6   NLD5   NLD4   NLD3   NLD2   NLD1
		         15     14     13     12     11
		         NLD0   NCLE   NCEB1  NCEB0  NALE
		*/
		nandx_set_bits32(GPIO_PU_CFG0, 0x03FFF800, (0x3 << 24) | (0x8 << 20) | (0x0 << 16) | (0x3 << 12) | (0x0 << 8));
		nandx_set_bits32(GPIO_PD_CFG0, 0x03FFF800, (0x0 << 24) | (0x7 << 20) | (0xF << 16) | (0xC << 12) | (0x8 << 8));
		/* Driving setting */
		nandx_set_bits32(GPIO_DRV_CFG0, 0x3FFF8000, (0x1 << 27) | (0x1 << 24) | (0x1 << 21) | (0x1 << 18) | (0x1 << 15));
		nandx_set_bits32(GPIO_DRV_CFG1, 0xFFFF, (0x1 << 9) | (0x1 << 6) | (0x1 << 3) | (0x1));
		/* TDSEL/RDSEL -> default */
		nandx_set_bits32(GPIO_RDSEL_CFG0, 0xFF, (0x0 << 6) | (0x0 << 4) | (0x0 << 2) | (0x0));
		nandx_set_bits32(GPIO_TDSEL_CFG0, 0xFFFF, (0x0 << 12) | (0x0 << 8) | (0x0 << 4) | (0x0));
	} else if (nand_type == NAND_SPI)
	{
		nandx_set_bits32(SNFI_GPIO_MODE27, 0x70000000, (2 << 28));
		/* SNFI_MISO, SNFI_CLK, SNFI_HOLD, SNFI_WP, SNFI_CS */
		nandx_set_bits32(SNFI_GPIO_MODE28, 0x77777, (2 << 16) | (2 << 12) | (2 << 8) | (2 << 4) | (2 << 0));
		/* Pull up/down setting */
		/*             25          24          23
		NWEB(CLK)    NRNB(WP)   NREB(HOLD)
		13          12          11
		NCEB1(MOSI)  NCEB0(CS)  NALE(MISO)
		*/
		nandx_set_bits32(GPIO_PU_CFG0, 0x03803800, (0x1 << 24) | (0x8 << 20) | (0x1 << 12) | (0x0 << 8));
		nandx_set_bits32(GPIO_PD_CFG0, 0x03803800, (0x2 << 24) | (0x0 << 20) | (0x2 << 12) | (0x8 << 8));
		/* Driving setting */
		nandx_set_bits32(GPIO_DRV_CFG0, 0x38FF8000, (0x1 << 27) | (0x1 << 21) | (0x1 << 18) | (0x1 << 15));
		nandx_set_bits32(GPIO_DRV_CFG1, 0x3F, (0x1 << 3) | (0x1));
		/* TDSEL/RDSEL -> default */
		nandx_set_bits32(GPIO_RDSEL_CFG0, 0xF0, (0x0 << 6) | (0x0 << 4));
		nandx_set_bits32(GPIO_TDSEL_CFG0, 0xFF00, (0x0 << 12) | (0x0 << 8));
	} else {
	}
}

static u32 nand_clock_init(void)
{
	return 156 * 1000 * 1000;
}

static void nand_hard_reset(void)
{
	u32 val;

	val = readl(INFRACFG_AO_BASE + 0x730);
	val |= BIT(10);
	writel(val, INFRACFG_AO_BASE + 0x730);

	//nandx_udelay(5);

	val = readl(INFRACFG_AO_BASE + 0x734);
	val |= BIT(10);
	writel(val, INFRACFG_AO_BASE + 0x734);

}

int interface_nandx_init()
{
	struct nfi_resource res = {
			NANDX_MT6880, NULL,
			(void *)NFIECC_BASE, 197,
			(void *)NFI_BASE, 196,
			26000000, NULL, 0, 32
	};
	int ret, arg = 1;


	pr_info("%s @ %d ...\n", __func__, __LINE__);
	ret = get_boot_device();
	if (ret == -1)
		return ret;
	res.nand_type = ret;


#if (DA_MODE_FPGA == 0)
	nand_hard_reset();
#endif
	nand_gpio_init(res.nand_type);

	res.clock_1x = nand_clock_init();

	ret = nandx_init(&res);
	if (ret) {
		pr_err("nandx init error (%d)!\n", ret);
		return ret;
	}

	nandx_ioctl(NFI_CTRL_DMA, &arg);
	nandx_ioctl(NFI_CTRL_ECC, &arg);
	nandx_ioctl(NFI_CTRL_BAD_MARK_SWAP, &arg);
	nandx_ioctl(NFI_CTRL_NFI_IRQ, &g_nandi);
	nandx_ioctl(NFI_CTRL_ECC_IRQ, &g_nandi);
	nandx_ioctl(CORE_CTRL_NAND_INFO, &g_nandi);

	head_buf = mem_alloc(1, g_nandi.block_size);
	if (!head_buf) {
		ret = -ENOMEM;
		pr_err("%s, %d, mem alloc fail!!! len:%d\n",
				__func__, __LINE__, g_nandi.block_size);
		return ret;
	}
	tail_buf = mem_alloc(1, g_nandi.block_size);
	if (!tail_buf) {
		ret = -ENOMEM;
		pr_err("%s, %d, mem alloc fail!!! len:%d\n",
				__func__, __LINE__, g_nandi.block_size);
		return ret;
	}

	ret = nftl_init();
	if (ret) {
		pr_err("nftl init error (%d)!\n", ret);
		return ret;
	}

	ret = scan_bbt(&g_nandi);
	if (ret) {
		pr_err("bbt init error (%d)!\n", ret);
		return ret;
	}

	return 0;
}

int interface_get_nandx_info(uint32 *page_size, uint32 *block_size, uint32 *spare_size, u64 *total_size, u64 *available_size,
                             uint8 *nand_bmt_exist, uint8 *plane_num, uint8 *nand_id, uint32 *page_parity_size)
{
	int i;
	*page_size = g_nandi.page_size;
	*block_size = g_nandi.block_size;
	*spare_size = g_nandi.oob_size;
	*total_size = g_nandi.total_size;
	*page_parity_size = g_nandi.page_parity_size;
	*plane_num = g_nandi.plane_num;
	/* reserve 4 blocks for bbt */
	*available_size =  g_nandi.total_size
		- NAND_BBT_SCAN_MAXBLOCKS * g_nandi.block_size;

	/* no bmt */
	*nand_bmt_exist = 0;

	for (i = 0; i < sizeof(u64); i++)
		nand_id[i] = g_nandi.id >> (i << 3) & 0xff;

	return 0;
}

u32 nand_get_device_id(u8 *id, u32 len)
{
	int i;

	if (!id)
		return -EINVAL;

	if (len > sizeof(u64))
		len = sizeof(u64);

	for (i = 0; i < len; i++)
		id[i] = g_nandi.id >> (i << 3) & 0xff;

	return 0;
}

int interface_nandx_device_ctrl(uint32 ctrl_code, void* in, uint32 in_len,
	void* out, uint32 out_len, uint32* ret_len)
{
	/* NOP */
	return 0;
}

int interface_nandx_flush(void)
{
	/* NOP */
	return 0;
}

int interface_nandx_ecc_header_block_encode(u8 *data, int len)
{
	return nandx_ecc_encode(data, len);
}
