#include <linux/version.h>
#include "moal_main.h"
#include "moal_uap.h"
#include "moal_cfg80211.h"

/*************************************************************
 * netlink function for wlevtd
 ************************************************************/
int
woal_add_nec_netlink(moal_handle *handle, int netlink_num)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 6, 0)
	struct netlink_kernel_cfg cfg = {
		.groups = NL_MULTICAST_GROUP,
	};
#endif

	if (netlink_num <= 0)
		return -1;

	do {
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
		handle->wlevtd_nl_sk = netlink_kernel_create(netlink_num, NULL);
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 22)
		handle->wlevtd_nl_sk =
			netlink_kernel_create(netlink_num, NL_MULTICAST_GROUP,
					      NULL, THIS_MODULE);
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 24)
		handle->wlevtd_nl_sk =
			netlink_kernel_create(netlink_num, NL_MULTICAST_GROUP,
					      NULL, NULL, THIS_MODULE);
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 6, 0)
		handle->wlevtd_nl_sk =
			netlink_kernel_create(&init_net, netlink_num,
					      NL_MULTICAST_GROUP, NULL, NULL,
					      THIS_MODULE);
#else
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 7, 0)
		handle->wlevtd_nl_sk =
			netlink_kernel_create(&init_net, netlink_num,
					      THIS_MODULE, &cfg);
#else
		handle->wlevtd_nl_sk =
			netlink_kernel_create(&init_net, netlink_num, &cfg);
#endif
#endif
#endif
#endif
#endif
		if (handle->wlevtd_nl_sk) {
			PRINTM(MINFO, "Netlink number (wlevtd) = %d\n", netlink_num);
			handle->wlevtd_netlink_num = netlink_num;
			break;
		}
		netlink_num--;
	} while (netlink_num > 0);

	if (handle->wlevtd_nl_sk == NULL) {
		PRINTM(MERROR,
		       "Could not initialize netlink event passing mechanism! (wlevtd)\n");
		return -1;
	}
	return 0;
}

int
woal_free_nec_netlink(moal_handle *handle)
{
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 25)
	if ((handle->wlevtd_nl_sk) && ((handle->wlevtd_nl_sk)->sk_socket)) {
		sock_release((handle->wlevtd_nl_sk)->sk_socket);
		handle->wlevtd_nl_sk = NULL;
	}
#else
	netlink_kernel_release(handle->wlevtd_nl_sk);
#endif
	return 0;
}

/**
 *  @brief This function handles events generated by firmware
 *
 *  @param priv     A pointer to moal_private structure
 *  @param payload  A pointer to payload buffer
 *  @param len      Length of the payload
 *  @return         MLAN_STATUS_SUCCESS or MLAN_STATUS_FAILURE
 */
mlan_status
woal_broadcast_event_to_wlevtd(moal_private * priv, enum wlevtd_eventid eventid, t_u8 * payload, t_u32 len)
{
	mlan_status ret = MLAN_STATUS_SUCCESS;
	struct sk_buff *skb = NULL;
	struct nlmsghdr *nlh = NULL;
	moal_handle *handle = priv->phandle;
	struct net_device *netdev = priv->netdev;
	struct sock *sk = handle->wlevtd_nl_sk;

	ENTER();

	/* interface name to be prepended to event */
	if ((len + IFNAMSIZ + sizeof(eventid)) > NL_MAX_PAYLOAD) {
		PRINTM(MERROR, "event size is too big, len=%d\n", (int)len);
		ret = MLAN_STATUS_FAILURE;
		goto done;
	}
	if (sk) {
		/* Allocate skb */
		skb = alloc_skb(NLMSG_SPACE(NL_MAX_PAYLOAD),
				GFP_ATOMIC);
		if (!skb) {
			PRINTM(MERROR,
			       "Could not allocate skb for netlink\n");
			ret = MLAN_STATUS_FAILURE;
			goto done;
		}

		nlh = (struct nlmsghdr *)skb->data;
		nlh->nlmsg_len = NLMSG_SPACE(len + IFNAMSIZ + sizeof(eventid));

		/* From kernel */
		nlh->nlmsg_pid = 0;
		nlh->nlmsg_flags = 0;

		/* Data */
		skb_put(skb, nlh->nlmsg_len);
		memcpy(NLMSG_DATA(nlh), netdev->name, IFNAMSIZ);
		memcpy(((t_u8 *) (NLMSG_DATA(nlh))) + IFNAMSIZ, &eventid, sizeof(eventid));
		memcpy(((t_u8 *) (NLMSG_DATA(nlh))) + IFNAMSIZ + sizeof(eventid), payload, len);

		/* From Kernel */
#if LINUX_VERSION_CODE < KERNEL_VERSION(3, 7, 0)
		NETLINK_CB(skb).pid = 0;
#else
		NETLINK_CB(skb).portid = 0;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 20)
		/* Multicast message */
		NETLINK_CB(skb).dst_pid = 0;
#endif

		/* Multicast group number */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2, 6, 14)
		NETLINK_CB(skb).dst_groups = NL_MULTICAST_GROUP;
#else
		NETLINK_CB(skb).dst_group = NL_MULTICAST_GROUP;
#endif

		/* Send message */
		ret = netlink_broadcast(sk, skb, 0, NL_MULTICAST_GROUP,
					GFP_ATOMIC);
		if (ret) {
			PRINTM(MWARN, "netlink_broadcast failed: ret=%d\n",
			       ret);
			goto done;
		}

		ret = MLAN_STATUS_SUCCESS;
	} else {
		PRINTM(MERROR,
		       "Could not send event through NETLINK. Link down.\n");
		ret = MLAN_STATUS_FAILURE;
	}
done:
	LEAVE();
	return ret;
}
