/*
 * iwcap.c - The pktcnt utility display each network interface packets per second.
 *
 */

#include <stdio.h>
//#include <stdint.h>
#include <stdlib.h>
//#include <stdarg.h>
#include <unistd.h>
#include <string.h>
//#include <signal.h>
#//include <syslog.h>
#include <errno.h>
//#include <byteswap.h>
//#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
//#include <sys/socket.h>
//#include <net/ethernet.h>
#include <net/if.h>
//#include <netinet/in.h>
//#include <linux/if_packet.h>

#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <ctype.h>
#include <ifaddrs.h>




//#define DPRINTF(fmt, args...) printf(fmt, ## args)
//#define DPRINTF(fmt, args...) printf("{%s.%d} " fmt, __func__, __LINE__, ## args)
#define DPRINTF(fmt, args...)

#define APP_NAME		"pktcnt"
#define PATH_PROC_STAT		"/proc/stat"
#define PATH_PROC_NET_DEV	"/proc/net/dev"
#define PATH_PROC_NET_DEV_V6	"/proc/net/if_inet6"

static void
usage(void)
{
	fprintf(stderr, "Packet Counter");
	fprintf(stderr, "pktcnt -cCOUNT [-sDELAY] [MODES]\n");
	fprintf(stderr, "\t-c: check count\n");
	fprintf(stderr, "\t-s: check delay\n");
	fprintf(stderr, "\t-b: bytes\n");
	fprintf(stderr, "\t-p: pktcnt <default>\n");
	fprintf(stderr, "\t-e: errcnt\n");
	fprintf(stderr, "\t-d: dropcnt\n");
	fprintf(stderr, "\t-f: fifo_errcnt\n");
	fprintf(stderr, "\t-o: option <b'2=CPU, b'1=DATE, b'0=HEADER>\n");
	fprintf(stderr, "\t-a: display all interface\n");
}

struct _cpu_stats {
	unsigned int user;
	unsigned int nice;
	unsigned int system;
	unsigned int idle;
	unsigned int iowait; /* kernel 2.5.41- */
	unsigned int irq;    /* kernel 2.6. 0- */
	unsigned int softirq;/* kernel 2.6. 0- */
	unsigned int steal;  /* kernel 2.6.11- */
	unsigned int guest;  /* kernel 2.6.24- */
	unsigned int guest_nice;    /* 2.6.33- */
	unsigned int run;
	unsigned int total;
};

struct cpu_stats {
	unsigned int user;
	unsigned int nice;
	unsigned int system;
	unsigned int idle;
	unsigned int iowait; /* kernel 2.5.41- */
	unsigned int irq;    /* kernel 2.6. 0- */
	unsigned int softirq;/* kernel 2.6. 0- */
	unsigned int steal;  /* kernel 2.6.11- */
	unsigned int guest;  /* kernel 2.6.24- */
	unsigned int guest_nice;    /* 2.6.33- */
	unsigned int run;
	unsigned int total;
#define MAX_CPU_NUM	4
	struct _cpu_stats cpus[MAX_CPU_NUM];
};

struct dev_stats {
	unsigned long long rx_bytes;
	unsigned long long rx_pktcnt;
	unsigned long rx_errcnt;
	unsigned long rx_dropcnt;
	unsigned long rx_fifo_errcnt;
	unsigned long rx_frame_errcnt;
	unsigned long rx_compress;
	unsigned long rx_multicast;
	unsigned long long tx_bytes;
	unsigned long long tx_pktcnt;
	unsigned long tx_errcnt;
	unsigned long tx_dropcnt;
	unsigned long tx_fifo_errcnt;
	unsigned long collisions;
	unsigned long tx_carrier_errcnt;
	unsigned long tx_compress;
};

struct net_dev_info {
	int disp;
	char dev_name[IFNAMSIZ];
	struct dev_stats st_new;
	struct dev_stats st_old;
} dev_info[] = {
	{ 0, "lte0pdn0",{0}, {0} },
//	{ 0, "wimax",	{0}, {0} },
//	{ 0, "mlan0",	{0}, {0} },
//	{ 0, "mmlan0",	{0}, {0} },
	{ 0, "br0",	{0}, {0} },
	{ 0, "eth0",	{0}, {0} },
	{ 0, "usb0",	{0}, {0} },
	{ 0, "uap0",	{0}, {0} },
	{ 0, "uap1",	{0}, {0} },
	{ 0, "muap0",	{0}, {0} },
	{ 0, "muap1",	{0}, {0} },
	{ 0, "\0",	{0}, {0} }, /* END */
};

#define MAX_DEV_NUM 10
static char dev_names[MAX_DEV_NUM][IFNAMSIZ];

static int
get_ifup_dev_names(void)
{
	struct ifaddrs *ifa_list;
	struct ifaddrs *ifa;
	int n, i = 0;
	int r, ignore;
	char ignore_dev[][IFNAMSIZ] = {
		"lo",
	};

	n = getifaddrs(&ifa_list);
	if (n != 0)
		return -1;

	for(ifa = ifa_list; ifa != NULL; ifa=ifa->ifa_next) {
		if (i > MAX_DEV_NUM)
			break;

		if ((ifa->ifa_flags & IFF_UP) != IFF_UP)
			continue;

		if (ifa->ifa_addr->sa_family == AF_INET)
			continue;
		else if (ifa->ifa_addr->sa_family == AF_INET6)
			continue;

		for (r = 0, ignore = 0; r < (sizeof(ignore_dev) / sizeof(ignore_dev[0])); r++) {
			if (strcmp(ignore_dev[r], ifa->ifa_name) == 0) {
				ignore = 1;
				break;
			}
		}
		if (ignore)
			continue;

		snprintf(dev_names[i], IFNAMSIZ, "%s", ifa->ifa_name);
		i++;
	}

	freeifaddrs(ifa_list);

	return i;
}


static int
get_cpu_info(struct cpu_stats *cpu)
{
	char buf[IFNAMSIZ];
	FILE *fp;
	int i;

	if ((fp = fopen(PATH_PROC_STAT, "r")) == NULL) {
		fprintf(stderr, "ERROR: failed to open %s, errno=%d\n",
			PATH_PROC_STAT, errno);
		return (-1);
	}

	/* cpu */
	fscanf(fp, "%s%d%d%d%d%d%d%d%d%d%d", (char *)&buf,
		&cpu->user, &cpu->nice, &cpu->system, &cpu->idle, &cpu->iowait, &cpu->irq,
		&cpu->softirq, &cpu->steal, &cpu->guest, &cpu->guest_nice);

	cpu->run = cpu->user + cpu->nice + cpu->system + cpu->iowait
		+ cpu->irq + cpu->steal + cpu->softirq + cpu->guest + cpu->guest_nice;
	cpu->total = cpu->run + cpu->idle;

	/* cpu0 - cpu3 */
	for (i=0; i < MAX_CPU_NUM; i++) {
		fscanf(fp, "%s%d%d%d%d%d%d%d%d%d%d", (char *)&buf,
			&cpu->cpus[i].user, &cpu->cpus[i].nice, &cpu->cpus[i].system, &cpu->cpus[i].idle, &cpu->cpus[i].iowait, &cpu->cpus[i].irq,
			&cpu->cpus[i].softirq, &cpu->cpus[i].steal, &cpu->cpus[i].guest, &cpu->cpus[i].guest_nice);

		cpu->cpus[i].run = cpu->cpus[i].user + cpu->cpus[i].nice + cpu->cpus[i].system + cpu->cpus[i].iowait
			+ cpu->cpus[i].irq + cpu->cpus[i].steal + cpu->cpus[i].softirq + cpu->cpus[i].guest + cpu->cpus[i].guest_nice;
		cpu->cpus[i].total = cpu->cpus[i].run + cpu->cpus[i].idle;
	}

	fclose(fp);
	return 0;
}

static int
get_dev_info(struct net_dev_info *dev)
{
	char buf[256];
	struct net_dev_info tmp;
	int i;
	FILE *fp;

	if ((fp = fopen(PATH_PROC_NET_DEV, "r")) == NULL) {
		fprintf(stderr, "ERROR: failed to open %s, errno=%d\n",
			PATH_PROC_NET_DEV, errno);
		return 0;
	}

	/* skip 2 line */
	fgets(buf, sizeof(buf), fp);
	fgets(buf, sizeof(buf), fp);

	while (fgets(buf, sizeof(buf), fp)) {
		sscanf(buf, " %[^:]:%llu%llu%lu%lu%lu%lu%lu%lu%llu%llu%lu%lu%lu%lu%lu%lu",
			(char *)&tmp.dev_name,
			&tmp.st_new.rx_bytes,
			&tmp.st_new.rx_pktcnt,
			&tmp.st_new.rx_errcnt,
			&tmp.st_new.rx_dropcnt,
			&tmp.st_new.rx_fifo_errcnt,
			&tmp.st_new.rx_frame_errcnt,
			&tmp.st_new.rx_compress,
			&tmp.st_new.rx_multicast,
			&tmp.st_new.tx_bytes,
			&tmp.st_new.tx_pktcnt,
			&tmp.st_new.tx_errcnt,
			&tmp.st_new.tx_dropcnt,
			&tmp.st_new.tx_fifo_errcnt,
			&tmp.st_new.collisions,
			&tmp.st_new.tx_carrier_errcnt,
			&tmp.st_new.tx_compress
		);
		DPRINTF("tmp.dev_name=%s\n", (char *)&tmp.dev_name);

		for (i=0,tmp.disp=0;;i++) {
			DPRINTF("dev[%d].dev_name=%s .disp=%d\n", i, (char *)dev[i].dev_name, dev[i].disp);
			if (dev[i].dev_name[0] == '\0') {
				DPRINTF("break: name==0\n");
				break;
			}
			if (tmp.disp == 1) {/* skip duplicate name */
				DPRINTF("break: tmp.disp==1\n");
				break;
			}
			if (strcmp(dev[i].dev_name, tmp.dev_name) == 0) {
				tmp.disp = 1; /* trick */
				dev[i].disp = 1;
				memcpy(&dev[i].st_new, &tmp.st_new, sizeof(struct dev_stats));
				DPRINTF("break: much\n");
				break;
			}
		}
	}
	fclose(fp);
	return 1;
}

int
main(int argc, char *argv[])
{
	extern char *optarg;
	extern int optind;
	int c, cnt = 0x7fffffff;
	int delay = 1;
	int i;
	struct cpu_stats cpu_new = {0}, cpu_old = {0};
	char str[512], *pstr;
	struct net_dev_info *dev;
	int use_dev_info = 0;
	enum t_drv_mode {
		MODE_BYTES=1,     /*bytes*/
		MODE_PKTCNT,      /*pktcnt*/
		MODE_ERRCNT,      /*errcnt*/
		MODE_DROPCNT,     /*dropcnt*/
		MODE_FIFO_ERRCNT, /*fifo_errcnt*/
	} mode = MODE_PKTCNT;
	enum t_option {
		HEADER    = (1 << 0),
		DATE_TIME = (1 << 1),
		CPU_USAGE = (1 << 2)
	} option = CPU_USAGE | DATE_TIME | HEADER;

	while ((c = getopt(argc, argv, "abpedfhc:s:o:")) > 0) {
		switch(c) {
		case 'h':	/* help */
			usage();
			return 0;
		case 'a':
			use_dev_info = 1;
			dev = (struct net_dev_info *)dev_info;
			break;
		case 'o':
			option = atoi(optarg);
			break;
		case 'c':
			cnt = atoi(optarg);
			break;
		case 's':
			delay = atoi(optarg);
			break;
		case 'b':
			mode = MODE_BYTES;
			break;
		case 'e':
			mode = MODE_ERRCNT;
			break;
		case 'd':
			mode = MODE_DROPCNT;
			break;
		case 'f':
			mode = MODE_FIFO_ERRCNT;
			break;
		case 'p':
		default:
			mode = MODE_PKTCNT;
			break;
		}
	}

	if (use_dev_info == 0) {
		int num, n;
		num = get_ifup_dev_names();
		if (num <= 0) {
			printf("Active interface is not found.\n");
			return 1;
		} else {
			dev = (struct net_dev_info *)calloc(num + 1, sizeof(struct net_dev_info));
			for (n = 0; n < num; n++) {
				snprintf(dev[n].dev_name, IFNAMSIZ, "%s", dev_names[n]);
			}
		}
	}

	printf("option: %s%s%s\n",
		(option & HEADER) ? "HEADER, " : ""
		,(option & DATE_TIME) ? "DATE, " : ""
		,(option & CPU_USAGE) ? "CPU" : ""
	);
	switch (mode) {
	case MODE_BYTES: /*bytes*/
		printf("[ Bytes Handling/sec ] order: Interface(Tx Rx) ...\n");
		break;
	case MODE_ERRCNT: /*errcnt*/
		printf("[ Packet Errors/sec ] order: Interface(Tx Rx) ...\n");
		break;
	case MODE_DROPCNT: /*dropcnt*/
		printf("[ Packet Drop/sec ] order: Interface(Tx Rx) ...\n");
		break;
	case MODE_FIFO_ERRCNT: /*fifo_errcnt*/
		printf("[ FIFO Errors/sec ] order: Interface(Tx Rx) ...\n");
		break;
	case MODE_PKTCNT: /*pktcnt*/
		printf("[ Packet Handling/sec ] order: Interface(Tx Rx) ...\n");
	default:
		break;
	}

	pstr = str;

	if (option & DATE_TIME) {
		time_t t = time(NULL);
		struct tm *tm = localtime(&t);
		pstr += sprintf(pstr, "%4d/%02d/%02d-%02d:%02d:%02d, ",
				1900+tm->tm_year, tm->tm_mon+1, tm->tm_mday,
				tm->tm_hour, tm->tm_min, tm->tm_sec);
	}

	pstr += sprintf(pstr, "cpu(avg)%%,cpu0%%,cpu1%%,cpu2%%,cpu3%%");
	get_dev_info(dev);
	for(i=0;;i++) {
		if(dev[i].dev_name[0] == '\0')
			break;
		if(dev[i].disp == 1)
			pstr += sprintf(pstr,", %s,Rx", dev[i].dev_name);
	}
	printf("%s\n",str);
	printf("--------------------------------------------------------------------\n");
	for (c = 0; c < cnt; c++) {
		pstr = str;
		sleep(delay);
		if (option & HEADER) {
			pstr += sprintf(pstr, "%s, ", APP_NAME);
		}
		if (option & DATE_TIME) {
			time_t t = time(NULL);
			struct tm *tm = localtime(&t);
			pstr += sprintf(pstr, "%02d:%02d:%02d, ",
				tm->tm_hour, tm->tm_min, tm->tm_sec);
		}
		memcpy(&cpu_old, &cpu_new, sizeof(struct cpu_stats));
		get_cpu_info(&cpu_new);
		if (cpu_new.total - cpu_old.total) {
			pstr += sprintf(pstr, "%2d,%2d,%2d,%2d,%2d",
				(cpu_new.run - cpu_old.run) * 100/ (cpu_new.total - cpu_old.total),
				(cpu_new.cpus[0].run - cpu_old.cpus[0].run) * 100/ (cpu_new.cpus[0].total - cpu_old.cpus[0].total),
				(cpu_new.cpus[1].run - cpu_old.cpus[1].run) * 100/ (cpu_new.cpus[1].total - cpu_old.cpus[1].total),
				(cpu_new.cpus[2].run - cpu_old.cpus[2].run) * 100/ (cpu_new.cpus[2].total - cpu_old.cpus[2].total),
				(cpu_new.cpus[3].run - cpu_old.cpus[3].run) * 100/ (cpu_new.cpus[3].total - cpu_old.cpus[3].total)
			);
		} else {
			pstr += sprintf(pstr, "%2d", 0);
		}
		DPRINTF("us=%d sy=%d ni=%d id=%d wa=%d hi=%d si=%d st=%d gu=%d gn=%d\n", 
			cpu.user, cpu.system, cpu.nice, cpu.idle,
			cpu.iowait, cpu.irq, cpu.softirq, cpu.steal,
			cpu.guest, cpu.guest_nice);

		get_dev_info(dev);
		for(i=0;;i++) {
			if(dev[i].dev_name[0] == '\0') {
				break;
			}
			if(dev[i].disp == 0) {
				continue;
			}
			switch (mode) {
			case MODE_BYTES: /*bytes*/
				pstr += sprintf(pstr, ", %lld,%lld",
					dev[i].st_new.tx_bytes - dev[i].st_old.tx_bytes,
					dev[i].st_new.rx_bytes - dev[i].st_old.rx_bytes);
				break;
			case MODE_ERRCNT: /*errcnt*/
				pstr += sprintf(pstr, ", %ld,%ld",
					dev[i].st_new.tx_errcnt - dev[i].st_old.tx_errcnt,
					dev[i].st_new.rx_errcnt - dev[i].st_old.rx_errcnt);
				break;
			case MODE_DROPCNT: /*dropcnt*/
				pstr += sprintf(pstr, ", %ld,%ld",
					dev[i].st_new.tx_dropcnt - dev[i].st_old.tx_dropcnt,
					dev[i].st_new.rx_dropcnt - dev[i].st_old.rx_dropcnt);
				break;
			case MODE_FIFO_ERRCNT: /*fifo_errcnt*/
				pstr += sprintf(pstr, ", %ld,%ld",
					dev[i].st_new.tx_fifo_errcnt - dev[i].st_old.tx_fifo_errcnt,
					dev[i].st_new.rx_fifo_errcnt - dev[i].st_old.rx_fifo_errcnt);
				break;
			case MODE_PKTCNT: /*pktcnt*/
			default:
				pstr += sprintf(pstr, ", %lld,%lld",
					dev[i].st_new.tx_pktcnt - dev[i].st_old.tx_pktcnt,
					dev[i].st_new.rx_pktcnt - dev[i].st_old.rx_pktcnt);
				break;
			}
			memcpy(&dev[i].st_old, &dev[i].st_new, sizeof(struct dev_stats));
		}
		if (c)
			printf("%s\n", str);
	}

	if (use_dev_info == 0)
		free(dev);

	return 0;
}

