#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdarg.h>
#include <time.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <errno.h>
#include <openssl/sha.h>
#include <openssl/md5.h>

#include "lib/bluetooth.h"
#include "lib/hci.h"
#include "lib/hci_lib.h"
#include "lib/l2cap.h"
#include "lib/uuid.h"

#include "src/shared/mainloop.h"
#include "src/shared/util.h"
#include "src/shared/att.h"
#include "src/shared/queue.h"
#include "src/shared/timeout.h"
#include "src/shared/gatt-db.h"
#include "src/shared/gatt-server.h"

#include "tools/btgatt-nec-io.h"

#include <libwlanapi/libwlanapi.h>
#include <libsysmgr/lib_sysmgr.h>
#include <libcli/clilib.h>
#include <pfmg/pfmgvar.h>
#include <libtrafficmon/trafficmonlib.h>

struct wireless_config_info {
	char ssid[LIBSYSMGR_MAX_STRLEN];
	char key[LIBSYSMGR_MAX_STRLEN];
	int mode;
	int interface;
	int ssid_len;
	int key_len;
};

struct set_data_list {
	int mode;
	uint8_t set_value;
};

#define DO_CLI_CMD(fd, ...) \
do { \
	if (cli_write_cmd(fd, '#', __VA_ARGS__) < 0) {	\
		cli_close(fd);				\
		return -1;				\
	}						\
} while (0)

#define SHA256_DIGEST_LENGTH 512
#define SBUF_POOL_MAX 3
#define CMD_PARAMETER_MAX 128
static char Sbuf[SBUF_POOL_MAX][CMD_PARAMETER_MAX];
static int Sidx = 0;
static int traffic_year = 0;
static int traffic_month = 0;
static int traffic_day = 0;

static char *
Ssnprintf(const char *format, ...)
{
	va_list arg_list;
	int idx;

	va_start(arg_list, format);
	vsnprintf(Sbuf[Sidx], sizeof(Sbuf[Sidx]), format, arg_list);
	va_end(arg_list);

	idx = Sidx;
	if ((Sidx + 1) < SBUF_POOL_MAX)
		Sidx += 1;
	else
		Sidx = 0;

	return Sbuf[idx];
}

static char *
btgatt_get_random_data(void)
{
	struct timespec ntv;
	unsigned long randval;

	if (0 != clock_gettime(CLOCK_REALTIME, &ntv))
		return "";

	srandom(ntv.tv_nsec);
	randval = random();

	/* At@xbg啶ɕϊĖ߂lԂ */
	return Ssnprintf("%08X", randval);
}

static int
do_cli_open(int *fdp, int *read_optp)
{
	int fd, retry = 0;

retry:
	fd = cli_open_config(LOCAL_SOCKET, LOCK_FILE, 1, CLI_MODE_RDONLY);
	if (fd < 0) {
		return -1;
	}
	*read_optp = cli_read_opt(CLI_READ_OPT_TIME_MID);
	*fdp = fd;

	return 0;
}

static void
do_cli_close(int fd, int read_opt)
{
	cli_read_opt(read_opt);
	cli_close(fd);
}

static int
wireless_config_interface_match(int nmatch, char *match[], void *userp)
{
	struct wireless_config_info *info = (struct wireless_config_info *)userp;

	if (strstr(match[0], "WirelessEthernet0.0")) {
		info->interface = 1;
	} else if (strstr(match[0], "WirelessEthernet0.1")) {
		info->interface = 2;
	} else {
		info->interface = 0;
	}

	return 0;
}

static int
wireless_config_ssid_match(int nmatch, char *match[], void *userp)
{
	struct wireless_config_info *info = (struct wireless_config_info *)userp;

	if (info->interface == 1) {
		strlcpy(info->ssid, match[0], LIBSYSMGR_MAX_STRLEN);
	}

	return 0;
}

static int
wireless_config_eap_key_match(int nmatch, char *match[], void *userp)
{
	struct wireless_config_info *info = (struct wireless_config_info *)userp;

	if (info->interface == 1) {
		if (63 < strlen(match[0])) {
			strlcpy(info->key, &match[0][2], LIBSYSMGR_MAX_STRLEN);
		} else {
			strlcpy(info->key, match[0], LIBSYSMGR_MAX_STRLEN);
		}
	}

	return 0;
}

static int
wireless_config_eap_mode_match(int nmatch, char *match[], void *userp)
{
	struct wireless_config_info *info = (struct wireless_config_info *)userp;

	if (info->interface == 1) {
		if(match[0] == NULL){
			info->mode = 0;
		} else if(0 == strcmp(match[0], "wpa-wpa2-mixed-psk")){
			info->mode = 1;
		} else if(0 == strcmp(match[0], "wpa-wpa2-aes")){
			info->mode = 2;
		} else if(0 == strcmp(match[0], "aes2")){
			info->mode = 3;
		} else {
			info->mode = 0;
		}
	}

	return 0;
}

static int
wireless_config_get(struct wireless_config_info *info)
{
	int fd, read_opt;
	const cli_match_tbl_t tbl[] = {
		{"^interface (WirelessEthernet0.0|WirelessEthernet0.1)", wireless_config_interface_match},
		{"^ +ssid (.+)$", wireless_config_ssid_match},
		{"^ +encryption eap-key EAP (.+) interval [0-9]+", wireless_config_eap_key_match},
		{"^ +encryption mode (.+) (.*)", wireless_config_eap_mode_match},
		{NULL, NULL}
	};

	if (do_cli_open(&fd, &read_opt)){
		return -1;
	}

	memset(info, 0, sizeof(struct wireless_config_info));

	DO_CLI_CMD(fd, "show wireless config\n");
	if (cli_read_regex(fd, tbl, info, '#') >= 0) {
		if ((info->key[0] == '\0') || (info->mode == 0)) {
			info->key[0] = '\0';
			info->mode = 0;
		}
		info->ssid_len = strlen(info->ssid);
		info->key_len  = strlen(info->key);
	}

	do_cli_close(fd, read_opt);

	return 0;
}

static uint8_t
set_data_to_viw(const struct set_data_list *tmp_list, int mode)
{
	while (tmp_list->mode != -1){
		if (tmp_list->mode == mode){
			return tmp_list->set_value;
		}
		tmp_list++;
	}
	return 0xff;
}

void gap_device_name_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	char dev_name[BT_DEVNAME_MAXLEN + 1];
	int ret;

	PRLOG("GAP : Device Name Read called\n");
#if 0
	len = server->name_len;

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &server->device_name[offset] : NULL;
#else
	memset(dev_name, 0, sizeof(dev_name));
	ret = libbtutl_get_bt_name(dev_name);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	len = strlen(dev_name) + 1;

	len -= offset;
	value = len ? &dev_name[offset] : NULL;
#endif
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void gap_device_name_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;

	PRLOG("GAP : Device Name Write called\n");

	/* If the value is being completely truncated, clean up and return */
	if (!(offset + len)) {
		free(server->device_name);
		server->device_name = NULL;
		server->name_len = 0;
		goto done;
	}

	/* Implement this as a variable length attribute value. */
	if (offset > server->name_len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	if (offset + len != server->name_len) {
		uint8_t *name;

		name = realloc(server->device_name, offset + len);
		if (!name) {
			error = BT_ATT_ERROR_INSUFFICIENT_RESOURCES;
			goto done;
		}

		server->device_name = name;
		server->name_len = offset + len;
	}

	if (value)
		memcpy(server->device_name + offset, value, len);

done:
	gatt_db_attribute_write_result(attrib, id, error);
}

void gap_device_name_ext_prop_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	uint8_t value[2];

	PRLOG("GAP : Device Name Extended Properties Read called\n");

	value[0] = BT_GATT_CHRC_EXT_PROP_RELIABLE_WRITE;
	value[1] = 0;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void gap_appearance_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 2;
	const uint8_t *value = NULL;
	uint8_t appearance[2];

	PRLOG("GAP : Appearance read request\n");

	if (offset > 2) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	appearance[0] = server->appearance & 0x00ff;
	appearance[1] = (server->appearance >> 8) & 0x001f;

	len -= offset;
	value = len ? &appearance[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void gap_peripheral_pref_conn_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 2;
	const uint8_t *value = NULL;
	uint8_t peripheral_pref_conn[2];

	PRLOG("GAP : peripheral_pref_conn read request\n");

	if (offset > 2) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	peripheral_pref_conn[0] = server->peripheral_pref_conn & 0x00ff;
	peripheral_pref_conn[1] = server->peripheral_pref_conn >> 8;

	len -= offset;
	value = len ? &peripheral_pref_conn[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void gap_central_addr_resol_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 2;
	const uint8_t *value = NULL;
	uint8_t central_addr_resol[2];

	PRLOG("GAP : central_addr_resol read request\n");

	if (offset > 2) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	central_addr_resol[0] = server->central_addr_resol & 0x00ff;
	central_addr_resol[1] = server->central_addr_resol >> 8;

	len -= offset;
	value = len ? &central_addr_resol[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void gatt_service_changed_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	PRLOG("GATT : Service Changed Read called\n");

	gatt_db_attribute_read_result(attrib, id, 0, NULL, 0);
}

void gatt_svc_chngd_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("Service Changed CCC Read called\n");

	value[0] = server->svc_chngd_enabled ? 0x02 : 0x00;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void gatt_svc_chngd_ccc_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;

	PRLOG("Service Changed CCC Write called\n");

	if (!value || len != 2) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	if (value[0] == 0x00)
		server->svc_chngd_enabled = false;
	else if (value[0] == 0x02)
		server->svc_chngd_enabled = true;
	else
		ecode = 0x80;

	PRLOG("Service Changed Enabled: %s\n",
				server->svc_chngd_enabled ? "true" : "false");

done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

void devinfo_manuf_name_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("Device Info : Manufacture Name String called\n");

	len = server->manuf_len;

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &server->manuf_name[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void devinfo_model_number_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	char serial_num[KFS_PFMG_SIZE];

	PRLOG("Device Info : Model Number String called\n");
	
	pfmg_read_serial_number(serial_num, sizeof(serial_num));
	len = strlen(serial_num);

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &serial_num[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void devinfo_fw_revision_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	char oem_version[VERSION_LENGTH];

	PRLOG("Device Info : Firmware Revision String called\n");

	memset(oem_version, 0, VERSION_LENGTH);

	len = read_oem_version(oem_version, VERSION_LENGTH);

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &oem_version[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void devinfo_hw_revision_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("Device Info : Hardware Revision String called\n");

	len = server->hw_rev_len;

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &server->hw_revision[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void devinfo_pnp_id_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("Device Info PnP ID String called\n");

	len = server->pnp_id_len;

	if (offset > len) {
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &server->pnp_id[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void auth_key_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = SHA256_DIGEST_LENGTH;
	const uint8_t *value = NULL;
	int i;
	char auth_key[SHA256_DIGEST_LENGTH];
	static struct sysmgr_ble_msg_data ble_msg_data;
	
	PRLOG("Authorization Service : Authorization Key read request\n");

	memset(auth_key, 0, SHA256_DIGEST_LENGTH);

	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	if(offset == 0){
		sysmgrble_get_auth_key_data(&ble_msg_data);
		memcpy(auth_key, ble_msg_data.msg, SHA256_DIGEST_LENGTH);
	} else {
		memcpy(auth_key, server->auth_key, SHA256_DIGEST_LENGTH);
	}

	len -= offset;
	value = len ? &auth_key[offset] : NULL;

//	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void auth_stat_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t auth_stat[4];

	PRLOG("Authorization Service : Authorization Status read request\n");

	if (offset > 4) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	auth_stat[0] = server->auth_stat & 0x000000ff;
	auth_stat[1] = (server->auth_stat >> 8) & 0x000000ff;
	auth_stat[2] = (server->auth_stat >> 16) & 0x000000ff;
	auth_stat[3] = (server->auth_stat >> 24) & 0x000000ff;

	len -= offset;
	value = len ? &auth_stat[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void auth_val_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	static uint8_t ecode;
	static uint8_t tmp_val[32];
	static int ng_flg;

	PRLOG("Authorization Service : Authorization Value write request\n");
	PRLOG("offset : %d,   len : %d\n", (int)offset, (int)len);
	PRLOG_DUMP(value, len);
	
	if (offset == 0) {
		ecode = 0;
		memset(tmp_val, 0, 32);
		ng_flg = 0;
	}
	
	if (ecode != 0) {
		goto done;
	}
	
	if (value == NULL) {
#ifdef USE_BTGATT_AUTH
		if (ng_flg == 0) {
			PRLOG("Authorization Lock\n");
			server->auth_stat = 1;
			sysmgrble_set_auth_ng();
			ng_flg = 1;
		}
#endif
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}
	
	if (offset + len > 32) {
#ifdef USE_BTGATT_AUTH
		if (ng_flg == 0) {
			PRLOG("Authorization Lock\n");
			server->auth_stat = 1;
			sysmgrble_set_auth_ng();
			ng_flg = 1;
		}
#endif
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	} else {
		memcpy(tmp_val + offset, value, len);
	}
	
#ifdef USE_BTGATT_AUTH
	if (offset + len == 32) {
		uint8_t empty[32];
		memset(empty, 0, 32);
		if ((memcmp(server->auth_key, empty, 32) == 0) && (memcmp(server->auth_val, empty, 32) == 0)) {
			if (ng_flg == 0) {
				PRLOG("Authorization Lock\n");
				server->auth_stat = 1;
				sysmgrble_set_auth_ng();
				ng_flg = 1;
			}
			ecode = BT_ATT_ERROR_AUTH_FAILED;
			goto done;
		}
		
		if (memcmp(server->auth_val, tmp_val, 32) == 0) {
			PRLOG("Authorization Unlock\n");
			server->auth_stat = 0;
		} else {
			if (ng_flg == 0) {
				PRLOG("Authorization Lock\n");
				server->auth_stat = 1;
				sysmgrble_set_auth_ng();
				ng_flg = 1;
			}
			ecode = BT_ATT_ERROR_AUTH_FAILED;
		}
	} else {
		if (memcmp(server->auth_val + offset, value, len) != 0) {
			if (ng_flg == 0) {
				PRLOG("Authorization Lock\n");
				server->auth_stat = 1;
				sysmgrble_set_auth_ng();
				ng_flg = 1;
			}
		}
	}
#endif
	
done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

void necpf_rt_notify_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("NECPF Service : Router Control Notify Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_rt_notify_cb(struct server *server, uint32_t notify)
{
	uint16_t len = 4;
	uint8_t pdu[4];
	
	PRLOG("NECPF Service : Router Notification called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		return;
	}
	
	pdu[0] = notify & 0x000000ff;
	pdu[1] = (notify >> 8) & 0x000000ff;
	pdu[2] = (notify >> 16) & 0x000000ff;
	pdu[3] = (notify >> 24) & 0x000000ff;
	
	bt_gatt_server_send_notification(server->gatt, server->necpf_rt_notify_handle, pdu, len);
}

void necpf_rt_notify_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("NECPF Service : Router Notify CCC Read called\n");

	value[0] = 0x01;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void necpf_rt_status_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int ret;
	int rt_state = 0;
	int wifi_cac = 0;
	int wifi_band = 0;
	int lte_state = 0;
	int lte_lv = 0;
	uint32_t lte_flags = 0;
	int stop_reason = 0;

	PRLOG("NECPF Service : Router Status Read called\n");

	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	ret = sysmgrble_get_router_state(&rt_state);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_wifi_band(&wifi_cac, &wifi_band);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_lte_state(&lte_state, &lte_lv, &lte_flags);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	PRLOG("rt_state : 0x%02x\n", rt_state);
	PRLOG("wifi_cac : 0x%02x, wifi_band : 0x%02x, stop_reason : 0x%08x\n", wifi_cac, wifi_band, stop_reason);
	PRLOG("lte_state : 0x%02x, lte_lv : 0x%02x, lte_flags : 0x%08x\n", lte_state, lte_lv, (int)lte_flags);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (rt_state == 0) {
		set_data[0] = 0x00;
		set_data[1] = 0x00;
	} else if ((rt_state == 1) || (rt_state == 2)) {
		if (stop_reason != 0) {
			set_data[0] = 0x02;
			if (stop_reason & 0x08) {         // SYS2BLE_REASON_HEAT
				set_data[1] = 0x04;
			} else if (stop_reason & 0x10) {  // SYS2BLE_REASON_PROCESS
				set_data[1] = 0x05;
			} else if (stop_reason & 0x80) {  // SYS2BLE_REASON_VERUP
				set_data[1] = 0x08;
			} else if (stop_reason & 0x20) {  // SYS2BLE_REASON_CAC
				set_data[1] = 0x06;
			} else if (stop_reason & 0x01) {  // SYS2BLE_REASON_OUTER
				set_data[1] = 0x01;
			} else if (stop_reason & 0x02) {  // SYS2BLE_REASON_OVER
				set_data[1] = 0x02;
			} else if (stop_reason & 0x04) {  // SYS2BLE_REASON_EXPIRED
				set_data[1] = 0x03;
			} else if (stop_reason & 0x40) {  // SYS2BLE_REASON_LTE_STOP
				set_data[1] = 0x10;
			} else {
				set_data[1] = 0x20;
			}
		} else {
			if ((lte_flags == 0) || (lte_flags == 0x08)) {
				if (lte_lv == 0) {
					set_data[0] = 0x02;
					set_data[1] = 0x01;
				} else {
					set_data[0] = 0x01;
					set_data[1] = 0x00;
				}
			} else {
				set_data[0] = 0x02;
				if (lte_flags & 0x04) {   // SYS2BLE_LTE_TMP_STOP
					set_data[1] = 0x04;
				} else if (lte_flags & 0x02) {   // SYS2BLE_LTE_REJECT
					if (stop_reason & 0x02) {        // SYS2BLE_REASON_OVER
						set_data[1] = 0x02;
					} else if (stop_reason & 0x04) { // SYS2BLE_REASON_EXPIRED
						set_data[1] = 0x03;
					} else {
						set_data[1] = 0x10;
					}
				} else if (lte_flags & 0x01) {          // SYS2BLE_LTE_OUTER
					set_data[1] = 0x01;
				} else if (lte_flags & 0x10) {   // SYS2BLE_LTE_FWUPDATE
					set_data[1] = 0x10;
				} else {
					set_data[1] = 0x10;
				}
			}
		}
	} else {
		set_data[0] = 0x02;
		set_data[1] = 0xff;
	}
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_rt_cmd_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;
	int tmp_data = 0;

	PRLOG("NECPF Service : Router Control Command write called\n");
	PRLOG_DUMP(value, len);
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		ecode = BT_ATT_ERROR_WRITE_NOT_PERMITTED;
		goto done;
	}
	
	if (!value || len != 4) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	/* ύXR}hグ */
	if(value[0] == 0x00){
		tmp_data = 0;
	} else if(value[0] == 0x01){
		tmp_data = 1;
	} else if(value[0] == 0x02) {
		tmp_data = 2;
	} else {
		ecode = BT_ERROR_OUT_OF_RANGE;
		goto done;
	}
	if(-1 == sysmgrble_set_router_state(tmp_data)){
		/* ݒ莸s */
		ecode = BT_ATT_ERROR_SET_FAILED;
		goto done;
	}

done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

void necpf_wifi_notify_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("NECPF Service : Wi-Fi Control Notify Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_notify_cb(struct server *server, uint32_t notify)
{
	uint16_t len = 4;
	uint8_t pdu[4];
	
	PRLOG("NECPF Service : Wi-Fi Notification called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		return;
	}
	
	pdu[0] = notify & 0x000000ff;
	pdu[1] = (notify >> 8) & 0x000000ff;
	pdu[2] = (notify >> 16) & 0x000000ff;
	pdu[3] = (notify >> 24) & 0x000000ff;
	
	bt_gatt_server_send_notification(server->gatt, server->necpf_wifi_notify_handle, pdu, len);
}

void necpf_wifi_notify_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("NECPF Service : Wi-Fi Notify CCC Read called\n");

	value[0] = 0x01;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void necpf_wifi_status_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int ret;
	int rt_state = 0;
	int wifi_cac = 0;
	int wifi_band = 0;
	int stop_reason = 0;

	PRLOG("NECPF Service : Wi-Fi Status Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	ret = sysmgrble_get_router_state(&rt_state);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_wifi_band(&wifi_cac, &wifi_band);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	PRLOG("rt_state : 0x%02x\n", rt_state);
	PRLOG("wifi_cac : 0x%02x, wifi_band : 0x%02x, stop_reason : 0x%08x\n", wifi_cac, wifi_band, stop_reason);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (rt_state == 0) {
		set_data[0] = 0x00;
		set_data[1] = 0x07;
	} else if ((rt_state == 1) || (rt_state == 2)) {
		if (stop_reason != 0) {
			set_data[0] = 0x00;
			if (stop_reason & 0x08) {          // SYS2BLE_REASON_HEAT
				set_data[1] = 0x04;
			} else if (stop_reason & 0x10) {   // SYS2BLE_REASON_PROCESS
				set_data[1] = 0x05;
			} else if (stop_reason & 0x80) {   // SYS2BLE_REASON_VERUP
				set_data[1] = 0x08;
			} else if (stop_reason & 0x20) {   // SYS2BLE_REASON_CAC
				set_data[0] = 0x03;
			} else if (stop_reason & 0x01) {   // SYS2BLE_REASON_OUTER
				set_data[1] = 0x01;
			} else if (stop_reason & 0x02) {   // SYS2BLE_REASON_OVER
				set_data[1] = 0x02;
			} else if (stop_reason & 0x04) {   // SYS2BLE_REASON_EXPIRED
				set_data[1] = 0x03;
			} else if (stop_reason & 0x40) {   // SYS2BLE_REASON_LTE_STOP
				set_data[1] = 0x10;
			} else {
				set_data[1] = 0xff;
			}
		} else {
			if (wifi_band == 1) {                  // SYS2UI_LANWIFI_ENABLE_2G
				set_data[0] = 0x01;
			} else if (wifi_band == 3) {           // SYS2UI_LANWIFI_ENABLE_5G_OUT
				if (wifi_cac == 0) {
					set_data[0] = 0x02;
				} else {
					set_data[0] = 0x03;
				}
			} else {
				set_data[0] = 0x00;
				set_data[1] = 0xff;
			}
		}
	} else {
		set_data[0] = 0x00;
		set_data[1] = 0xff;
	}
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_band_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int wifi_cac = 0;
	int wifi_band = 0;

	PRLOG("NECPF Service : Wi-Fi Band Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_wifi_band(&wifi_cac, &wifi_band);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (wifi_band == 0) {
		set_data[0] = 0x00;
	} else if (wifi_band == 1) {
		set_data[0] = 0x01;
	} else if (wifi_band == 3) {
		if (wifi_cac == 0) {
			set_data[0] = 0x02;
		} else {
			set_data[0] = 0x03;
		}
	}
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_ssid_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	static struct wireless_config_info info;
	int ret;
	int stop_reason = 0;

	PRLOG("NECPF Service : Wi-Fi SSID Read called\n");

	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	if (stop_reason & 0x10) {   // SYS2BLE_REASON_PROCESS
		PRLOG("F/W Updating\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset == 0){
		if(wireless_config_get(&info)){
			len = 0;
			value = NULL;
			error = BT_ATT_ERROR_GET_FAILED;
			goto done;
		}
	}

	len = info.ssid_len;

	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &info.ssid[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_pass_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	static struct wireless_config_info info;
	int ret;
	int i;
	int stop_reason = 0;

	PRLOG("NECPF Service : Wi-Fi Password Read called\n");
	
	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	if (stop_reason & 0x10) {   // SYS2BLE_REASON_PROCESS
		PRLOG("F/W Updating\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset == 0) {
		if(wireless_config_get(&info)){
			len = 0;
			value = NULL;
			error = BT_ATT_ERROR_GET_FAILED;
			goto done;
		}
	}

	len = info.key_len;

	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	len -= offset;
	value = len ? &info.key[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_count_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int wifi_sta = 0;

	PRLOG("NECPF Service : Wi-Fi Connected STA Count Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_wifi_sta(&wifi_sta);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	set_data[0] = (uint8_t)wifi_sta;
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_encrypt_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	struct wireless_config_info info;
	const struct set_data_list tmp_list[] = {
		{0,0x00},
		{1,0x01},
		{2,0x02},
		{3,0x03},
		{-1, 0xff} /* ԕ */
	};
	int ret;
	int stop_reason = 0;

	memset(set_data, 0, len);

	PRLOG("NECPF Service : Wi-Fi Encrypttion Mode Read called\n");

	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	if (stop_reason & 0x10) {   // SYS2BLE_REASON_PROCESS
		PRLOG("F/W Updating\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if(wireless_config_get(&info)){
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}

	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	set_data[0] = set_data_to_viw(tmp_list, info.mode);

	len -= offset;
	value = len ? &set_data[offset] : NULL;

	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_wifi_cmd_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;
	int tmp_data = 0;

	PRLOG("NECPF Service : Wi-Fi Control Command write called\n");
	PRLOG_DUMP(value, len);
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		ecode = BT_ATT_ERROR_WRITE_NOT_PERMITTED;
		goto done;
	}
	
	if (!value || len != 4) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	/* ύXR}hグ */
	if (value[0] == 0x01) {
		tmp_data = 1;
	} else if (value[0] == 0x03) {
		tmp_data = 3;
	} else {
		ecode = BT_ERROR_OUT_OF_RANGE;
		goto done;
	}
	if(-1 == sysmgrble_set_wifi_band(tmp_data)){
		ecode = BT_ATT_ERROR_SET_FAILED;
		goto done;
	}
	
done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

void necpf_lte_notify_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("NECPF Service : LTE Control Notify Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_lte_notify_cb(struct server *server, uint32_t notify)
{
	uint16_t len = 4;
	uint8_t pdu[4];
	
	PRLOG("NECPF Service : LTE Notification called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		return;
	}
	
	pdu[0] = notify & 0x000000ff;
	pdu[1] = (notify >> 8) & 0x000000ff;
	pdu[2] = (notify >> 16) & 0x000000ff;
	pdu[3] = (notify >> 24) & 0x000000ff;
	
	bt_gatt_server_send_notification(server->gatt, server->necpf_lte_notify_handle, pdu, len);
}

void necpf_lte_notify_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("NECPF Service : LTE Notify CCC Read called\n");

	value[0] = 0x01;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void necpf_lte_status_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int ret;
	int lte_state = 0;
	int lte_lv = 0;
	uint32_t lte_flags = 0;
	int stop_reason = 0;

	PRLOG("NECPF Service : LTE Status Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	ret = sysmgrble_get_lte_state(&lte_state, &lte_lv, &lte_flags);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	ret = sysmgrble_get_stop_reason(&stop_reason);
	if (ret != 0) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	PRLOG("lte_state : 0x%02x, lte_lv : 0x%02x, lte_flags : 0x%08x\n", lte_state, lte_lv, (int)lte_flags);
	PRLOG("stop_reason : 0x%08x\n", stop_reason);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (lte_state == 1) {
		if ((lte_flags == 0) || (lte_flags == 0x08)) {
			if (lte_lv == 0) {
				set_data[0] = 0x00;
			} else {
				set_data[0] = 0x01;
			}
		} else if (lte_flags & 0x04) { // SYS2BLE_LTE_TMP_STOP
			set_data[0] = 0x02;
		} else if (lte_flags & 0x02) { // SYS2BLE_LTE_REJECT
			if (stop_reason & 0x02) {  // SYS2BLE_REASON_OVER
				set_data[0] = 0x03;
			} else if (stop_reason & 0x04) { // SYS2BLE_REASON_EXPIRED
				set_data[0] = 0x04;
			} else {
				set_data[0] = 0x05;
			}
		} else if (lte_flags & 0x01) { // SYS2BLE_LTE_OUTER
			set_data[0] = 0x00;
		} else if (lte_flags & 0x10) { // SYS2BLE_LTE_FWUPDATE
			set_data[0] = 0x06;
		} else {
			set_data[0] = 0xff;
		}
	} else if (lte_state == 0) {
		set_data[0] = 0x07;
	} else {
		set_data[0] = 0xff;
	}
	set_data[1] = (uint8_t)lte_lv;
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_lte_traffic_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	struct libtmon_db_t db;
	struct libtmon_daily_db_t daily_db;
	int mgrtype;
	int i, ret, diff_day;
	uint64_t traffic = 0;
	uint32_t traffic_mb = 0;

	PRLOG("NECPF Service : LTE Traffic Data Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	
	ret = libtmon_read_traffic_daily(&daily_db, &mgrtype, LIBTMON_GET_TRAFFIC_DAILY, LIBTMON_SIM1);
	if (ret != LIBTMON_OK) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_GET_FAILED;
		goto done;
	}
	
	PRLOG("Today    : %04d/%02d/%02d\n", daily_db.traffic_list[0].year, daily_db.traffic_list[0].month, daily_db.traffic_list[0].day);
	PRLOG("Set Date : %04d/%02d/%02d\n", traffic_year, traffic_month, traffic_day);
	for (i = 0; i < LIBTMON_DAILY_DATA_MAX; i++) {
		PRLOG("%04d/%02d/%02d : %d\n", daily_db.traffic_list[i].year, daily_db.traffic_list[i].month, daily_db.traffic_list[i].day, (int)(daily_db.traffic_list[i].traffic));
	}
	
	diff_day = ((daily_db.traffic_list[0].year * 10000) + (daily_db.traffic_list[0].month * 100) + daily_db.traffic_list[0].day) - 
	((traffic_year * 10000) + (traffic_month * 100) + traffic_day);
	
	if (diff_day < 0) {
		/* ̓t̏ꍇ */
		traffic = 0;
	} else if ((diff_day >= 0) && (diff_day < 14)) {
		/* NŁA14ȓ̏ꍇ */
		for (i = 0; i <= diff_day; i++) {
			traffic += daily_db.traffic_list[i].traffic;
		}
	} else if ((diff_day >=14) && (diff_day < 31)) {
		/* NŁA15ȏ̏ꍇ */
		for (i = 0; i < LIBTMON_DAILY_DATA_MAX; i++) {
			traffic += daily_db.traffic_list[i].traffic;
		}
	} else {
		/* N܂ׂ͌ꍇ */
		ret = libtmon_read_traffic(&db, &mgrtype, LIBTMON_GET_TRAFFIC_CURRENT, LIBTMON_SIM1);
		if (ret != LIBTMON_OK) {
			len = 0;
			value = NULL;
			error = BT_ATT_ERROR_GET_FAILED;
			goto done;
		}
		traffic = db.traffic;
	}
	
	traffic_mb = (traffic + ((uint64_t)512 * LIBTMON_KILO)) / LIBTMON_MEGA;    // ľܓMBPʂvZ
	PRLOG("Traffic Data : %llu Byte\n", traffic);
	PRLOG("Traffic Data : %lu MByte\n", traffic_mb);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	set_data[0] = traffic_mb & 0x000000ff;
	set_data[1] = (traffic_mb >> 8) & 0x000000ff;
	set_data[2] = (traffic_mb >> 16) & 0x000000ff;
	set_data[3] = (traffic_mb >> 24) & 0x000000ff;
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_lte_cmd_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;
	int year, month, day;

	PRLOG("NECPF Service : LTE Control Command write called\n");
	PRLOG_DUMP(value, len);
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		ecode = BT_ATT_ERROR_WRITE_NOT_PERMITTED;
		goto done;
	}
	
	if (!value || len != 4) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (value[0] <= 31) {
		day = (int)value[0];
	} else {
		ecode = BT_ERROR_OUT_OF_RANGE;
		goto done;
	}
	if (value[1] <= 12) {
		month = (int)value[1];
	} else {
		ecode = BT_ERROR_OUT_OF_RANGE;
		goto done;
	}
	year = (int)value[2];
	year += ((int)value[3] * 0x100);
	
	traffic_year = year;
	traffic_month = month;
	traffic_day = day;
	PRLOG("Set traffic day : %04d/%02d/%02d\n", traffic_year, traffic_month, traffic_day);
	
done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

void necpf_fw_notify_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("NECPF Service : F/W Control Notify Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_fw_notify_cb(struct server *server, uint32_t notify)
{
	uint16_t len = 4;
	uint8_t pdu[4];
	
	PRLOG("NECPF Service : F/W Notification called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		return;
	}
	
	pdu[0] = notify & 0x000000ff;
	pdu[1] = (notify >> 8) & 0x000000ff;
	pdu[2] = (notify >> 16) & 0x000000ff;
	pdu[3] = (notify >> 24) & 0x000000ff;
	
	bt_gatt_server_send_notification(server->gatt, server->necpf_fw_notify_handle, pdu, len);
}

void necpf_fw_notify_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("NECPF Service : F/W Notify CCC Read called\n");

	value[0] = 0x01;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void necpf_fw_status_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int fw_stat = 0;
	int fw_upd = 0;

	PRLOG("NECPF Service : F/W Status Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_verup_state(&fw_stat);
	sysmgrble_get_newfw_state(&fw_upd);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (fw_stat == 0 ) {
		if (fw_upd == 0) {
			set_data[0] = 0x00;
		} else {
			set_data[0] = 0x01;
		}
	} else {
		if (fw_stat == 0x01) {        // SYS2BLE_VERUP_DOWNLOAD
			set_data[0] = 0x02;
		} else if (fw_stat == 0x02) { // SYS2BLE_VERUP_UPDATING
			set_data[0] = 0x03;
		} else if (fw_stat == 0x03) { // SYS2BLE_VERUP_SUCCESS
			set_data[0] = 0x04;
		} else if (fw_stat == 0x04) { // SYS2BLE_VERUP_FAIL
			set_data[0] = 0x05;
		} else if (fw_stat == 0x05) { // SYS2BLE_VERUP_FAIL_DOWNLOAD
			set_data[0] = 0x06;
		}
	}
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_fw_update_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int fw_upd = 0;

	PRLOG("NECPF Service : F/W Update Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_newfw_state(&fw_upd);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	if (fw_upd == 0 ) {
		set_data[0] = 0x00;
	} else {
		set_data[0] = 0x01;
	}
	
	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_fw_cmd_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;
	int tmp_data = 0;

	PRLOG("NECPF Service : F/W Control Command write called\n");
	PRLOG_DUMP(value, len);
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		ecode = BT_ATT_ERROR_WRITE_NOT_PERMITTED;
		goto done;
	}
	
	if (!value || len != 4) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	/* ύXR}hグ */
	if (value[0] == 0x01) {
		/* if (-1 == sysmgrble_set_verup_start()) { */
		if (0 > sysmgrble_set_verup_start()) {
			ecode = BT_ATT_ERROR_SET_FAILED;
			goto done;
		}
	} else {
		ecode = BT_ERROR_OUT_OF_RANGE;
		goto done;
	}
	
done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

/* 20190531 AddIF */
void necpf_server_control_notify_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;

	PRLOG("NECPF Service : Server Control Notify Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_server_control_notify_cb(struct server *server, uint32_t notify)
{
	uint16_t len = 4;
	uint8_t pdu[4];
	
	PRLOG("NECPF Service : Server Notification called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		return;
	}
	
	pdu[0] = notify & 0x000000ff;
	pdu[1] = (notify >> 8) & 0x000000ff;
	pdu[2] = (notify >> 16) & 0x000000ff;
	pdu[3] = (notify >> 24) & 0x000000ff;
	
	bt_gatt_server_send_notification(server->gatt, server->necpf_server_control_notify_handle, pdu, len);
}

void necpf_server_control_notify_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	PRLOG("NECPF Service : Server Notify CCC Read called\n");

	value[0] = 0x01;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, sizeof(value));
}

void necpf_server_status_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	int server_stat = 0;

	PRLOG("NECPF Service : Server Status Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_availability(&server_stat);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	set_data[0] = server_stat;

	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_server_traffic_data_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	uint8_t set_data[len];
	uint64_t traffic = 0;
	uint32_t traffic_mb = 0;

	PRLOG("NECPF Service : Server Traffic Data Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	memset(set_data, 0, len);
	sysmgrble_get_contract_capa(&traffic);
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	traffic_mb = (traffic + ((uint64_t)512 * LIBTMON_KILO)) / LIBTMON_MEGA;    // ľܓMBPʂvZ
	PRLOG("Traffic Data : %llu Byte\n", traffic);
	PRLOG("Traffic Data : %lu MByte\n", traffic_mb);

	set_data[0] = traffic_mb & 0x000000ff;
	set_data[1] = (traffic_mb >> 8) & 0x000000ff;
	set_data[2] = (traffic_mb >> 16) & 0x000000ff;
	set_data[3] = (traffic_mb >> 24) & 0x000000ff;

	len -= offset;
	value = len ? &set_data[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_server_free_date_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	static struct sysmgr_ble_msg_data ble_msg_data;
	char *p;

	PRLOG("NECPF Service : Server Free Date Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	/* Mf[^ʂđM邽߁A񂾂l擾 */
	if (offset == 0){
		sysmgrble_get_free_period_to(&ble_msg_data);
		if(ble_msg_data.msg != NULL){
			while ((p = strstr(ble_msg_data.msg , "-")) != NULL) {
				*p = '/';
			}
		}
	}
	len = ble_msg_data.len;
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}
	
	len -= offset;
	value = len ? &ble_msg_data.msg[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_server_availability_dams_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 0;
	const uint8_t *value = NULL;
	static struct sysmgr_ble_msg_data ble_msg_data;

	PRLOG("NECPF Service : Server Availability DaMs Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	/* Mf[^ʂđM邽߁A񂾂l擾 */
	if (offset == 0){
		sysmgrble_get_avail_da_msg(&ble_msg_data);
	}
	len = ble_msg_data.len;
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &ble_msg_data.msg[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

void necpf_server_free_alert_dams_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t error = 0;
	size_t len = 4;
	const uint8_t *value = NULL;
	static struct sysmgr_ble_msg_data ble_msg_data;

	PRLOG("NECPF Service : Server Free Alert DaMs Read called\n");
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_READ_NOT_PERMITTED;
		goto done;
	}
	
	/* Mf[^ʂđM邽߁A񂾂l擾 */
	if (offset == 0){
		sysmgrble_get_free_da_msg(&ble_msg_data);
	}
	len = ble_msg_data.len;
	
	if (offset > len) {
		len = 0;
		value = NULL;
		error = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	len -= offset;
	value = len ? &ble_msg_data.msg[offset] : NULL;
	
	PRLOG_DUMP(value, len);
done:
	gatt_db_attribute_read_result(attrib, id, error, value, len);
}

#define SERVER_CMD_LENGTH 15

void necpf_server_control_command_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;
	char server_userId[SERVER_CMD_LENGTH+1];
	int i = 0, j = 0;

	PRLOG("NECPF Service : Server Control Command write called\n");
	PRLOG_DUMP(value, len);
	
	if (server->auth_stat != 0) {
		PRLOG("Authorization Locked\n");
		ecode = BT_ATT_ERROR_WRITE_NOT_PERMITTED;
		goto done;
	}
	
	if (!value || len > SERVER_CMD_LENGTH) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	for(i = 0; i < len; i++){
		server_userId[i] = value[i];
	}
	server_userId[len] = '\0';

	/* ύXR}hグ */
	if (-1 == sysmgrble_set_user_id(server_userId, len)) {
		ecode = BT_ATT_ERROR_SET_FAILED;
		goto done;
	}

done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}
