/*
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2014  Google Inc.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdbool.h>
#include <stdint.h>
#include <time.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>
#include <errno.h>
#include <sys/signalfd.h>
#include <pthread.h>

#include "lib/bluetooth.h"
#include "lib/hci.h"
#include "lib/hci_lib.h"
#include "lib/l2cap.h"
#include "lib/uuid.h"
#include "lib/mgmt.h"

#include "src/shared/mainloop.h"
#include "src/shared/util.h"
#include "src/shared/att.h"
#include "src/shared/queue.h"
#include "src/shared/timeout.h"
#include "src/shared/gatt-db.h"
#include "src/shared/gatt-server.h"
#include "src/shared/mgmt.h"

#include "tools/btgatt-nec-io.h"

#include <pfmg/pfmgvar.h>
#include <libbtgatt/libbtgatt.h>
#include <liblocal_comm/lib_local_comm.h>
#include <libselect.h>

static struct mgmt *mgmt = NULL;

static const char test_device_name[] = "NECPF Wi-Fi Router";
static const char test_manuf_name[] = "NEC Platforms, Ltd";
static const char test_model_num[] = "WWWWxxxxxxxxZ0";
static const char test_fw_revision[] = "1.0.0";
static const char test_hw_revision[] = "1.0.0";
static const char test_pnp_id[] = "0";
static bool verbose = false;
static int flg_finish = 0;
static int accept_flg = 0;
static pthread_t select_tid = 0;
static void *h = NULL;

struct server *btgatt_server = NULL;

const uint128_t uuid_auth_btorder = {
	.data = { 0xd8, 0x0d, 0xd8, 0x6d, 0x02, 0x91, 0x44, 0x0a,
		0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_auth_key_btorder = {
	.data = { 0xd8, 0x0d, 0x10, 0x01, 0x02, 0x91, 0x44, 0x0a,
		0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_auth_stat_btorder = {
	.data = { 0xd8, 0x0d, 0x10, 0x02, 0x02, 0x91, 0x44, 0x0a,
		0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_auth_val_btorder = {
	.data = { 0xd8, 0x0d, 0x10, 0x10, 0x02, 0x91, 0x44, 0x0a,
		0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_necpf_btorder = {
	.data = { 0xd8, 0x0c, 0xd8, 0x6d, 0x02, 0x91, 0x44, 0x0a,
		0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_rt_notify_btorder = {	
	.data = { 0xd8, 0x0c, 0x10, 0x01, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_rt_status_btorder = {	
	.data = { 0xd8, 0x0c, 0x10, 0x02, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_rt_cmd_btorder = {	
	.data = { 0xd8, 0x0c, 0x10, 0x10, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_notify_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x01, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_status_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x02, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_band_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x03, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_ssid_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x04, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_pass_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x05, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_count_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x06, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_encrypt_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x07, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_wifi_cmd_btorder = {	
	.data = { 0xd8, 0x0c, 0x20, 0x10, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_lte_notify_btorder = {	
	.data = { 0xd8, 0x0c, 0x30, 0x01, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_lte_status_btorder = {	
	.data = { 0xd8, 0x0c, 0x30, 0x02, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_lte_traffic_btorder = {	
	.data = { 0xd8, 0x0c, 0x30, 0x03, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_lte_cmd_btorder = {	
	.data = { 0xd8, 0x0c, 0x30, 0x10, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_fw_notify_btorder = {	
	.data = { 0xd8, 0x0c, 0x40, 0x01, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_fw_status_btorder = {	
	.data = { 0xd8, 0x0c, 0x40, 0x02, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_fw_update_btorder = {	
	.data = { 0xd8, 0x0c, 0x40, 0x03, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_fw_cmd_btorder = {	
	.data = { 0xd8, 0x0c, 0x40, 0x10, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

	/* SpecChange(2019/05/31) Add I/F */
const uint128_t uuid_server_control_notify_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x01, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_status_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x02, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_traffic_data_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x03, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_free_date_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x04, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_availability_dams_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x05, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_free_alert_dams_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x06, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };

const uint128_t uuid_server_control_command_btorder = {
	.data = { 0xd8, 0x0c, 0x50, 0x10, 0x02, 0x91, 0x44, 0x0a,
	0x8e, 0x49, 0x60, 0xc5, 0x6e, 0x33, 0x2d, 0xdc } };


static void att_disconnect_cb(int err, void *user_data)
{
	PRLOG("Device disconnected: %s\n", strerror(err));

	mainloop_quit();
}

static void att_debug_cb(const char *str, void *user_data)
{
	const char *prefix = user_data;

	PRLOG(COLOR_BOLDGRAY "%s" COLOR_BOLDWHITE "%s\n" COLOR_OFF, prefix,
									str);
}

static void gatt_debug_cb(const char *str, void *user_data)
{
	const char *prefix = user_data;

	PRLOG(COLOR_GREEN "%s%s\n" COLOR_OFF, prefix, str);
}

static void hr_msrmt_ccc_read_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t value[2];

	value[0] = server->hr_msrmt_enabled ? 0x01 : 0x00;
	value[1] = 0x00;

	gatt_db_attribute_read_result(attrib, id, 0, value, 2);
}

static bool hr_msrmt_cb(void *user_data)
{
	struct server *server = user_data;
	bool expended_present = !(server->hr_ee_count % 10);
	uint16_t len = 2;
	uint8_t pdu[4];
	uint32_t cur_ee;

	pdu[0] = 0x06;
	pdu[1] = 90 + (rand() % 40);

	if (expended_present) {
		pdu[0] |= 0x08;
		put_le16(server->hr_energy_expended, pdu + 2);
		len += 2;
	}

	bt_gatt_server_send_notification(server->gatt,
						server->hr_msrmt_handle,
						pdu, len);


	cur_ee = server->hr_energy_expended;
	server->hr_energy_expended = MIN(UINT16_MAX, cur_ee + 10);
	server->hr_ee_count++;

	return true;
}

static void update_hr_msrmt_simulation(struct server *server)
{
	if (!server->hr_msrmt_enabled || !server->hr_visible) {
		timeout_remove(server->hr_timeout_id);
		return;
	}

	server->hr_timeout_id = timeout_add(1000, hr_msrmt_cb, server, NULL);
}

static void hr_msrmt_ccc_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;

	if (!value || len != 2) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	if (value[0] == 0x00)
		server->hr_msrmt_enabled = false;
	else if (value[0] == 0x01) {
		if (server->hr_msrmt_enabled) {
			PRLOG("HR Measurement Already Enabled\n");
			goto done;
		}

		server->hr_msrmt_enabled = true;
	} else
		ecode = 0x80;

	PRLOG("HR: Measurement Enabled: %s\n",
				server->hr_msrmt_enabled ? "true" : "false");

	update_hr_msrmt_simulation(server);

done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

static void hr_control_point_write_cb(struct gatt_db_attribute *attrib,
					unsigned int id, uint16_t offset,
					const uint8_t *value, size_t len,
					uint8_t opcode, struct bt_att *att,
					void *user_data)
{
	struct server *server = user_data;
	uint8_t ecode = 0;

	if (!value || len != 1) {
		ecode = BT_ATT_ERROR_INVALID_ATTRIBUTE_VALUE_LEN;
		goto done;
	}

	if (offset) {
		ecode = BT_ATT_ERROR_INVALID_OFFSET;
		goto done;
	}

	if (value[0] == 1) {
		PRLOG("HR: Energy Expended value reset\n");
		server->hr_energy_expended = 0;
	}

done:
	gatt_db_attribute_write_result(attrib, id, ecode);
}

static void confirm_write(struct gatt_db_attribute *attr, int err,
							void *user_data)
{
	if (!err)
		return;

	PRERR("Error caching attribute %p - err: %d\n", attr, err);
	exit(1);
}

static void populate_gap_service(struct server *server)
{
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *tmp;
	uint16_t appearance;

	/* Add the GAP service */
	bt_uuid16_create(&uuid, UUID_GAP);
	service = gatt_db_add_service(server->db, &uuid, true, 9);

	/*
	 * Device Name characteristic. Make the value dynamically read and
	 * written via callbacks.
	 */
	bt_uuid16_create(&uuid, GATT_CHARAC_DEVICE_NAME);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					gap_device_name_read_cb,
					NULL,
					server);

	/*
	 * Appearance characteristic. Reads and writes should obtain the value
	 * from the database.
	 */
	bt_uuid16_create(&uuid, GATT_CHARAC_APPEARANCE);
	tmp = gatt_db_service_add_characteristic(service, &uuid,
							BT_ATT_PERM_READ,
							BT_GATT_CHRC_PROP_READ,
							gap_appearance_read_cb,
							NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_PERIPHERAL_PREF_CONN);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					gap_peripheral_pref_conn_read_cb,
					NULL,
					server);
	
	
	bt_uuid16_create(&uuid, UUID_CENTRAL_ADDR_RESOL);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					gap_central_addr_resol_read_cb,
					NULL,
					server);
	
	gatt_db_service_set_active(service, true);
}

static void populate_gatt_service(struct server *server)
{
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *svc_chngd;

	/* Add the GATT service */
	bt_uuid16_create(&uuid, UUID_GATT);
	service = gatt_db_add_service(server->db, &uuid, true, 4);

	bt_uuid16_create(&uuid, GATT_CHARAC_SERVICE_CHANGED);
	svc_chngd = gatt_db_service_add_characteristic(service, &uuid,
			BT_ATT_PERM_NONE,
			BT_GATT_CHRC_PROP_INDICATE,
			NULL, NULL, server);
	server->gatt_svc_chngd_handle = gatt_db_attribute_get_handle(svc_chngd);

	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_READ | BT_ATT_PERM_WRITE,
				gatt_svc_chngd_ccc_read_cb,
				gatt_svc_chngd_ccc_write_cb, server);

	gatt_db_service_set_active(service, true);
}

static void populate_devinfo_service(struct server *server)
{
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *svc_chngd;
	
	bt_uuid16_create(&uuid, UUID_DEVINFO);
	service = gatt_db_add_service(server->db, &uuid, true, 11);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_MANUFACTURER_NAME_STRING);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					devinfo_manuf_name_read_cb,
					NULL,
					server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_MODEL_NUMBER_STRING);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					devinfo_model_number_read_cb,
					NULL,
					server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_FIRMWARE_REVISION_STRING);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					devinfo_fw_revision_read_cb,
					NULL,
					server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_HARDWARE_REVISION_STRING);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					devinfo_hw_revision_read_cb,
					NULL,
					server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_PNP_ID);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					devinfo_pnp_id_read_cb,
					NULL,
					server);
	
	gatt_db_service_set_active(service, true);
}

static void populate_auth_service(struct server *server)
{
	uint128_t uuid_auth, uuid_auth_key, uuid_auth_stat, uuid_auth_val;
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *svc_chngd;
	
	btoh128(&uuid_auth_btorder, &uuid_auth);
	bt_uuid128_create(&uuid, uuid_auth);
	service = gatt_db_add_service(server->db, &uuid, true, 10);
	
	btoh128(&uuid_auth_key_btorder, &uuid_auth_key);
	bt_uuid128_create(&uuid, uuid_auth_key);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					auth_key_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_auth_stat_btorder, &uuid_auth_stat);
	bt_uuid128_create(&uuid, uuid_auth_stat);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					auth_stat_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_auth_val_btorder, &uuid_auth_val);
	bt_uuid128_create(&uuid, uuid_auth_val);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					auth_val_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	gatt_db_service_set_active(service, true);
}

static void populate_necpf_service(struct server *server)
{
	uint128_t uuid_necpf, uuid_rt_notify, uuid_rt_status, uuid_rt_cmd, uuid_wifi_notify, 
		uuid_wifi_status, uuid_wifi_band, uuid_wifi_ssid, uuid_wifi_pass, uuid_wifi_count, 
		uuid_wifi_encrypt, uuid_wifi_cmd, uuid_lte_notify, uuid_lte_status, uuid_lte_traffic, 
		uuid_lte_cmd, uuid_fw_notify, uuid_fw_status, uuid_fw_update, uuid_fw_cmd,
		uuid_server_control_notify, uuid_server_status, uuid_server_traffic_data, 
		uuid_server_free_date, uuid_server_availability_dams, uuid_server_free_alert_dams, uuid_server_control_command;
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *necpf_rt_notify, *necpf_wifi_notify, *necpf_lte_notify, *necpf_fw_notify, *necpf_server_control_notify;
	
	btoh128(&uuid_necpf_btorder, &uuid_necpf);
	bt_uuid128_create(&uuid, uuid_necpf);
	service = gatt_db_add_service(server->db, &uuid, true, 84);
	
	
	btoh128(&uuid_rt_notify_btorder, &uuid_rt_notify);
	bt_uuid128_create(&uuid, uuid_rt_notify);
	necpf_rt_notify = gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ | BT_GATT_CHRC_PROP_NOTIFY,
					necpf_rt_notify_read_cb,
					NULL, server);
	server->necpf_rt_notify_handle = gatt_db_attribute_get_handle(necpf_rt_notify);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid, BT_ATT_PERM_READ,
				necpf_rt_notify_ccc_read_cb, NULL, server);
	
	
	btoh128(&uuid_rt_status_btorder, &uuid_rt_status);
	bt_uuid128_create(&uuid, uuid_rt_status);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_rt_status_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_rt_cmd_btorder, &uuid_rt_cmd);
	bt_uuid128_create(&uuid, uuid_rt_cmd);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					necpf_rt_cmd_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_notify_btorder, &uuid_wifi_notify);
	bt_uuid128_create(&uuid, uuid_wifi_notify);
	necpf_wifi_notify = gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ | BT_GATT_CHRC_PROP_NOTIFY,
					necpf_wifi_notify_read_cb,
					NULL, server);
	server->necpf_wifi_notify_handle = gatt_db_attribute_get_handle(necpf_wifi_notify);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid, BT_ATT_PERM_READ,
				necpf_wifi_notify_ccc_read_cb, NULL, server);
	
	
	btoh128(&uuid_wifi_status_btorder, &uuid_wifi_status);
	bt_uuid128_create(&uuid, uuid_wifi_status);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_status_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_band_btorder, &uuid_wifi_band);
	bt_uuid128_create(&uuid, uuid_wifi_band);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_band_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_ssid_btorder, &uuid_wifi_ssid);
	bt_uuid128_create(&uuid, uuid_wifi_ssid);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_ssid_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_pass_btorder, &uuid_wifi_pass);
	bt_uuid128_create(&uuid, uuid_wifi_pass);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_pass_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_count_btorder, &uuid_wifi_count);
	bt_uuid128_create(&uuid, uuid_wifi_count);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_count_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_encrypt_btorder, &uuid_wifi_encrypt);
	bt_uuid128_create(&uuid, uuid_wifi_encrypt);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_wifi_encrypt_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_wifi_cmd_btorder, &uuid_wifi_cmd);
	bt_uuid128_create(&uuid, uuid_wifi_cmd);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					necpf_wifi_cmd_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_lte_notify_btorder, &uuid_lte_notify);
	bt_uuid128_create(&uuid, uuid_lte_notify);
	necpf_lte_notify = gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ | BT_GATT_CHRC_PROP_NOTIFY,
					necpf_lte_notify_read_cb,
					NULL, server);
	server->necpf_lte_notify_handle = gatt_db_attribute_get_handle(necpf_lte_notify);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid, BT_ATT_PERM_READ,
				necpf_lte_notify_ccc_read_cb, NULL, server);
	
	
	btoh128(&uuid_lte_status_btorder, &uuid_lte_status);
	bt_uuid128_create(&uuid, uuid_lte_status);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_lte_status_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_lte_traffic_btorder, &uuid_lte_traffic);
	bt_uuid128_create(&uuid, uuid_lte_traffic);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_lte_traffic_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_lte_cmd_btorder, &uuid_lte_cmd);
	bt_uuid128_create(&uuid, uuid_lte_cmd);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					necpf_lte_cmd_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_fw_notify_btorder, &uuid_fw_notify);
	bt_uuid128_create(&uuid, uuid_fw_notify);
	necpf_fw_notify = gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ | BT_GATT_CHRC_PROP_NOTIFY,
					necpf_fw_notify_read_cb,
					NULL, server);
	server->necpf_fw_notify_handle = gatt_db_attribute_get_handle(necpf_fw_notify);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid, BT_ATT_PERM_READ,
				necpf_fw_notify_ccc_read_cb, NULL, server);
	
	
	btoh128(&uuid_fw_status_btorder, &uuid_fw_status);
	bt_uuid128_create(&uuid, uuid_fw_status);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_fw_status_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_fw_update_btorder, &uuid_fw_update);
	bt_uuid128_create(&uuid, uuid_fw_update);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_fw_update_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	btoh128(&uuid_fw_cmd_btorder, &uuid_fw_cmd);
	bt_uuid128_create(&uuid, uuid_fw_cmd);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					necpf_fw_cmd_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);

	/* SpecChange(2019/05/31) Add I/F */
	/* server_control_notify */
	btoh128(&uuid_server_control_notify_btorder, &uuid_server_control_notify);
	bt_uuid128_create(&uuid, uuid_server_control_notify);
	necpf_server_control_notify = gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ | BT_GATT_CHRC_PROP_NOTIFY,
					necpf_server_control_notify_read_cb,
					NULL, server);
	server->necpf_server_control_notify_handle = gatt_db_attribute_get_handle(necpf_server_control_notify);

	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);

	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid, BT_ATT_PERM_READ,
				necpf_server_control_notify_ccc_read_cb, NULL, server);

	
	/* server_status */
	btoh128(&uuid_server_status_btorder, &uuid_server_status);
	bt_uuid128_create(&uuid, uuid_server_status);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_server_status_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	/* server_traffic_data */
	btoh128(&uuid_server_traffic_data_btorder, &uuid_server_traffic_data);
	bt_uuid128_create(&uuid, uuid_server_traffic_data);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_server_traffic_data_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);

	
	/* server_free_date */
	btoh128(&uuid_server_free_date_btorder, &uuid_server_free_date);
	bt_uuid128_create(&uuid, uuid_server_free_date);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_server_free_date_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	/* server_availability_dams */
	btoh128(&uuid_server_availability_dams_btorder, &uuid_server_availability_dams);
	bt_uuid128_create(&uuid, uuid_server_availability_dams);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_server_availability_dams_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	/* server_free_alert_dams */
	btoh128(&uuid_server_free_alert_dams_btorder, &uuid_server_free_alert_dams);
	bt_uuid128_create(&uuid, uuid_server_free_alert_dams);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_READ,
					BT_GATT_CHRC_PROP_READ,
					necpf_server_free_alert_dams_read_cb,
					NULL, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);
	
	
	/* server_control_command */
	btoh128(&uuid_server_control_command_btorder, &uuid_server_control_command);
	bt_uuid128_create(&uuid, uuid_server_control_command);
	gatt_db_service_add_characteristic(service, &uuid,
					BT_ATT_PERM_WRITE,
					BT_GATT_CHRC_PROP_WRITE,
					NULL,
					necpf_server_control_command_write_cb, server);
	
	bt_uuid16_create(&uuid, GATT_CHARAC_USER_DESC_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
				BT_ATT_PERM_NONE,
				NULL,
				NULL, server);

	
	gatt_db_service_set_active(service, true);
}

static void populate_hr_service(struct server *server)
{
	bt_uuid_t uuid;
	struct gatt_db_attribute *service, *hr_msrmt, *body;
	uint8_t body_loc = 1;  /* "Chest" */

	/* Add Heart Rate Service */
	bt_uuid16_create(&uuid, UUID_HEART_RATE);
	service = gatt_db_add_service(server->db, &uuid, true, 8);
	server->hr_handle = gatt_db_attribute_get_handle(service);

	/* HR Measurement Characteristic */
	bt_uuid16_create(&uuid, UUID_HEART_RATE_MSRMT);
	hr_msrmt = gatt_db_service_add_characteristic(service, &uuid,
						BT_ATT_PERM_NONE,
						BT_GATT_CHRC_PROP_NOTIFY,
						NULL, NULL, NULL);
	server->hr_msrmt_handle = gatt_db_attribute_get_handle(hr_msrmt);

	bt_uuid16_create(&uuid, GATT_CLIENT_CHARAC_CFG_UUID);
	gatt_db_service_add_descriptor(service, &uuid,
					BT_ATT_PERM_READ | BT_ATT_PERM_WRITE,
					hr_msrmt_ccc_read_cb,
					hr_msrmt_ccc_write_cb, server);

	/*
	 * Body Sensor Location Characteristic. Make reads obtain the value from
	 * the database.
	 */
	bt_uuid16_create(&uuid, UUID_HEART_RATE_BODY);
	body = gatt_db_service_add_characteristic(service, &uuid,
						BT_ATT_PERM_READ,
						BT_GATT_CHRC_PROP_READ,
						NULL, NULL, server);
	gatt_db_attribute_write(body, 0, (void *) &body_loc, sizeof(body_loc),
							BT_ATT_OP_WRITE_REQ,
							NULL, confirm_write,
							NULL);

	/* HR Control Point Characteristic */
	bt_uuid16_create(&uuid, UUID_HEART_RATE_CTRL);
	gatt_db_service_add_characteristic(service, &uuid,
						BT_ATT_PERM_WRITE,
						BT_GATT_CHRC_PROP_WRITE,
						NULL, hr_control_point_write_cb,
						server);

	if (server->hr_visible)
		gatt_db_service_set_active(service, true);
}

static void populate_db(struct server *server)
{
	populate_gap_service(server);
	populate_gatt_service(server);
	populate_devinfo_service(server);
	populate_auth_service(server);
	populate_necpf_service(server);
//	populate_hr_service(server);
}

static struct server *server_create(int fd, uint16_t mtu, bool hr_visible)
{
	struct server *server;
	size_t name_len = strlen(test_device_name);
	size_t manuf_len = strlen(test_manuf_name);
	size_t model_len = strlen(test_model_num);
	size_t fw_rev_len = strlen(test_fw_revision);
	size_t hw_rev_len = strlen(test_hw_revision);
	size_t pnp_id_len = strlen(test_pnp_id);

	server = new0(struct server, 1);
	if (!server) {
		PRERR("Failed to allocate memory for server\n");
		return NULL;
	}

	server->att = bt_att_new(fd, false);
	if (!server->att) {
		PRERR("Failed to initialze ATT transport layer\n");
		goto fail;
	}

	if (!bt_att_set_close_on_unref(server->att, true)) {
		PRERR("Failed to set up ATT transport layer\n");
		goto fail;
	}

	if (!bt_att_register_disconnect(server->att, att_disconnect_cb, NULL,
									NULL)) {
		PRERR("Failed to set ATT disconnect handler\n");
		goto fail;
	}

	server->name_len = name_len + 1;
	server->device_name = malloc(name_len + 1);
	if (!server->device_name) {
		PRERR("Failed to allocate memory for device name\n");
		goto fail;
	}

	memcpy(server->device_name, test_device_name, name_len);
	server->device_name[name_len] = '\0';

	server->appearance = 0x0000;
	server->peripheral_pref_conn = 0x0000;
	server->central_addr_resol = 0x0001;
	
	server->manuf_len = manuf_len + 1;
	server->manuf_name = malloc(manuf_len + 1);
	if (!server->manuf_name) {
		PRERR("Failed to allocate memory for manuf name\n");
		goto fail;
	}

	memcpy(server->manuf_name, test_manuf_name, manuf_len);
	server->manuf_name[manuf_len] = '\0';
	
	
	server->model_len = model_len + 1;
	server->model_num = malloc(model_len + 1);
	if (!server->model_num) {
		PRERR("Failed to allocate memory for model number\n");
		goto fail;
	}

	memcpy(server->model_num, test_model_num, model_len);
	server->model_num[model_len] = '\0';
	
	
	server->fw_rev_len = fw_rev_len + 1;
	server->fw_revision = malloc(fw_rev_len + 1);
	if (!server->fw_revision) {
		PRERR("Failed to allocate memory for firmware revision\n");
		goto fail;
	}

	memcpy(server->fw_revision, test_fw_revision, fw_rev_len);
	server->fw_revision[fw_rev_len] = '\0';
	
	
	server->hw_rev_len = hw_rev_len + 1;
	server->hw_revision = malloc(hw_rev_len + 1);
	if (!server->hw_revision) {
		PRERR("Failed to allocate memory for hardware revision\n");
		goto fail;
	}

	memcpy(server->hw_revision, test_hw_revision, hw_rev_len);
	server->hw_revision[hw_rev_len] = '\0';
	
	
	server->pnp_id_len = pnp_id_len + 1;
	server->pnp_id = malloc(pnp_id_len + 1);
	if (!server->pnp_id) {
		PRERR("Failed to allocate memory for PnP ID\n");
		goto fail;
	}

	memcpy(server->pnp_id, test_pnp_id, pnp_id_len);
	server->pnp_id[pnp_id_len] = '\0';
	
	
	memset(server->auth_key, 0, 32);
#ifdef USE_BTGATT_AUTH
	server->auth_stat = 1;
#else
	server->auth_stat = 0;
#endif
	memset(server->auth_val, 0, 32);
	memset(server->auth_val_512, 0, 64);
	memset(server->auth_val_512_s, 0, 64);
	
	server->fd = fd;
	server->db = gatt_db_new();
	if (!server->db) {
		PRERR("Failed to create GATT database\n");
		goto fail;
	}

	server->gatt = bt_gatt_server_new(server->db, server->att, mtu);
	if (!server->gatt) {
		PRERR("Failed to create GATT server\n");
		goto fail;
	}

	server->hr_visible = hr_visible;

	if (verbose) {
		bt_att_set_debug(server->att, att_debug_cb, "att: ", NULL);
		bt_gatt_server_set_debug(server->gatt, gatt_debug_cb,
							"server: ", NULL);
	}

	/* Random seed for generating fake Heart Rate measurements */
	srand(time(NULL));

	/* bt_gatt_server already holds a reference */
	populate_db(server);

	return server;

fail:
	gatt_db_unref(server->db);
	free(server->device_name);
	bt_att_unref(server->att);
	free(server);

	return NULL;
}

static void server_destroy(struct server *server)
{
	timeout_remove(server->hr_timeout_id);
	bt_gatt_server_unref(server->gatt);
	gatt_db_unref(server->db);
}

static void usage(void)
{
	printf("btgatt-server\n");
	printf("Usage:\n\tbtgatt-server [options]\n");

	printf("Options:\n"
		"\t-i, --index <id>\t\tSpecify adapter index, e.g. hci0\n"
		"\t-m, --mtu <mtu>\t\t\tThe ATT MTU to use\n"
		"\t-s, --security-level <sec>\tSet security level (low|"
								"medium|high)\n"
		"\t-t, --type [random|public] \t The source address type\n"
		"\t-v, --verbose\t\t\tEnable extra logging\n"
		"\t-h, --help\t\t\tDisplay help\n");
}

static struct option main_options[] = {
	{ "index",		1, 0, 'i' },
	{ "mtu",		1, 0, 'm' },
	{ "security-level",	1, 0, 's' },
	{ "type",		1, 0, 't' },
	{ "verbose",		0, 0, 'v' },
	{ "heart-rate",		0, 0, 'r' },
	{ "help",		0, 0, 'h' },
	{ }
};

static int l2cap_le_att_listen_and_accept(bdaddr_t *src, int sec,
							uint8_t src_type)
{
	int sk, nsk;
	struct sockaddr_l2 srcaddr, addr;
	socklen_t optlen;
	struct bt_security btsec;
	char ba[18];

	sk = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
	if (sk < 0) {
		perror("Failed to create L2CAP socket");
		return -1;
	}

	/* Set up source address */
	memset(&srcaddr, 0, sizeof(srcaddr));
	srcaddr.l2_family = AF_BLUETOOTH;
	srcaddr.l2_cid = htobs(ATT_CID);
	srcaddr.l2_bdaddr_type = src_type;
	bacpy(&srcaddr.l2_bdaddr, src);

	if (bind(sk, (struct sockaddr *) &srcaddr, sizeof(srcaddr)) < 0) {
		perror("Failed to bind L2CAP socket");
		goto fail;
	}

	/* Set the security level */
	memset(&btsec, 0, sizeof(btsec));
	btsec.level = sec;
	if (setsockopt(sk, SOL_BLUETOOTH, BT_SECURITY, &btsec,
							sizeof(btsec)) != 0) {
		PRERR("Failed to set L2CAP security level\n");
		goto fail;
	}

	if (listen(sk, 10) < 0) {
		perror("Listening on socket failed");
		goto fail;
	}

	PRLOG("Started listening on ATT channel. Waiting for connections\n");

	memset(&addr, 0, sizeof(addr));
	optlen = sizeof(addr);
	nsk = accept(sk, (struct sockaddr *) &addr, &optlen);
	if (nsk < 0) {
		perror("Accept failed");
		goto fail;
	}

	ba2str(&addr.l2_bdaddr, ba);
	PRLOG("Connect from %s\n", ba);
	close(sk);

	return nsk;

fail:
	close(sk);
	return -1;
}

static void notify_usage(void)
{
	printf("Usage: notify [options] <value_handle> <value>\n"
					"Options:\n"
					"\t -i, --indicate\tSend indication\n"
					"e.g.:\n"
					"\tnotify 0x0001 00 01 00\n");
}

static struct option notify_options[] = {
	{ "indicate",	0, 0, 'i' },
	{ }
};

static bool parse_args(char *str, int expected_argc,  char **argv, int *argc)
{
	char **ap;

	for (ap = argv; (*ap = strsep(&str, " \t")) != NULL;) {
		if (**ap == '\0')
			continue;

		(*argc)++;
		ap++;

		if (*argc > expected_argc)
			return false;
	}

	return true;
}

static void conf_cb(void *user_data)
{
	PRLOG("Received confirmation\n");
}

static void cmd_notify(struct server *server, char *cmd_str)
{
	int opt, i;
	char *argvbuf[516];
	char **argv = argvbuf;
	int argc = 1;
	uint16_t handle;
	char *endptr = NULL;
	int length;
	uint8_t *value = NULL;
	bool indicate = false;

	if (!parse_args(cmd_str, 514, argv + 1, &argc)) {
		PRLOG("Too many arguments\n");
		notify_usage();
		return;
	}

	optind = 0;
	argv[0] = "notify";
	while ((opt = getopt_long(argc, argv, "+i", notify_options,
								NULL)) != -1) {
		switch (opt) {
		case 'i':
			indicate = true;
			break;
		default:
			notify_usage();
			return;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 1) {
		notify_usage();
		return;
	}

	handle = strtol(argv[0], &endptr, 16);
	if (!endptr || *endptr != '\0' || !handle) {
		PRLOG("Invalid handle: %s\n", argv[0]);
		return;
	}

	length = argc - 1;

	if (length > 0) {
		if (length > UINT16_MAX) {
			PRLOG("Value too long\n");
			return;
		}

		value = malloc(length);
		if (!value) {
			PRLOG("Failed to construct value\n");
			return;
		}

		for (i = 1; i < argc; i++) {
			if (strlen(argv[i]) != 2) {
				PRLOG("Invalid value byte: %s\n",
								argv[i]);
				goto done;
			}

			value[i-1] = strtol(argv[i], &endptr, 16);
			if (endptr == argv[i] || *endptr != '\0'
							|| errno == ERANGE) {
				PRLOG("Invalid value byte: %s\n",
								argv[i]);
				goto done;
			}
		}
	}

	if (indicate) {
		if (!bt_gatt_server_send_indication(server->gatt, handle,
							value, length,
							conf_cb, NULL, NULL))
			PRLOG("Failed to initiate indication\n");
	} else if (!bt_gatt_server_send_notification(server->gatt, handle,
								value, length))
		PRLOG("Failed to initiate notification\n");

done:
	free(value);
}

static void heart_rate_usage(void)
{
	PRLOG("Usage: heart-rate on|off\n");
}

static void cmd_heart_rate(struct server *server, char *cmd_str)
{
	bool enable;
	uint8_t pdu[4];
	struct gatt_db_attribute *attr;

	if (!cmd_str) {
		heart_rate_usage();
		return;
	}

	if (strcmp(cmd_str, "on") == 0)
		enable = true;
	else if (strcmp(cmd_str, "off") == 0)
		enable = false;
	else {
		heart_rate_usage();
		return;
	}

	if (enable == server->hr_visible) {
		PRLOG("Heart Rate Service already %s\n",
						enable ? "visible" : "hidden");
		return;
	}

	server->hr_visible = enable;
	attr = gatt_db_get_attribute(server->db, server->hr_handle);
	gatt_db_service_set_active(attr, server->hr_visible);
	update_hr_msrmt_simulation(server);

	if (!server->svc_chngd_enabled)
		return;

	put_le16(server->hr_handle, pdu);
	put_le16(server->hr_handle + 7, pdu + 2);

	server->hr_msrmt_enabled = false;
	update_hr_msrmt_simulation(server);

	bt_gatt_server_send_indication(server->gatt,
						server->gatt_svc_chngd_handle,
						pdu, 4, conf_cb, NULL, NULL);
}

static void print_uuid(const bt_uuid_t *uuid)
{
	char uuid_str[MAX_LEN_UUID_STR];
	bt_uuid_t uuid128;

	bt_uuid_to_uuid128(uuid, &uuid128);
	bt_uuid_to_string(&uuid128, uuid_str, sizeof(uuid_str));

	PRLOG("%s\n", uuid_str);
}

static void print_incl(struct gatt_db_attribute *attr, void *user_data)
{
	struct server *server = user_data;
	uint16_t handle, start, end;
	struct gatt_db_attribute *service;
	bt_uuid_t uuid;

	if (!gatt_db_attribute_get_incl_data(attr, &handle, &start, &end))
		return;

	service = gatt_db_get_attribute(server->db, start);
	if (!service)
		return;

	gatt_db_attribute_get_service_uuid(service, &uuid);

	PRLOG("\t  " COLOR_GREEN "include" COLOR_OFF " - handle: "
					"0x%04x, - start: 0x%04x, end: 0x%04x,"
					"uuid: ", handle, start, end);
	print_uuid(&uuid);
}

static void print_desc(struct gatt_db_attribute *attr, void *user_data)
{
	PRLOG("\t\t  " COLOR_MAGENTA "descr" COLOR_OFF
					" - handle: 0x%04x, uuid: ",
					gatt_db_attribute_get_handle(attr));
	print_uuid(gatt_db_attribute_get_type(attr));
}

static void print_chrc(struct gatt_db_attribute *attr, void *user_data)
{
	uint16_t handle, value_handle;
	uint8_t properties;
	bt_uuid_t uuid;

	if (!gatt_db_attribute_get_char_data(attr, &handle,
								&value_handle,
								&properties,
								&uuid))
		return;

	PRLOG("\t  " COLOR_YELLOW "charac" COLOR_OFF
					" - start: 0x%04x, value: 0x%04x, "
					"props: 0x%02x, uuid: ",
					handle, value_handle, properties);
	print_uuid(&uuid);

	gatt_db_service_foreach_desc(attr, print_desc, NULL);
}

static void print_service(struct gatt_db_attribute *attr, void *user_data)
{
	struct server *server = user_data;
	uint16_t start, end;
	bool primary;
	bt_uuid_t uuid;

	if (!gatt_db_attribute_get_service_data(attr, &start, &end, &primary,
									&uuid))
		return;

	PRLOG(COLOR_RED "service" COLOR_OFF " - start: 0x%04x, "
				"end: 0x%04x, type: %s, uuid: ",
				start, end, primary ? "primary" : "secondary");
	print_uuid(&uuid);

	gatt_db_service_foreach_incl(attr, print_incl, server);
	gatt_db_service_foreach_char(attr, print_chrc, NULL);

	PRLOG("\n");
}

static void cmd_services(struct server *server, char *cmd_str)
{
	gatt_db_foreach_service(server->db, NULL, print_service, server);
}

static bool convert_sign_key(char *optarg, uint8_t key[16])
{
	int i;

	if (strlen(optarg) != 32) {
		PRLOG("sign-key length is invalid\n");
		return false;
	}

	for (i = 0; i < 16; i++) {
		if (sscanf(optarg + (i * 2), "%2hhx", &key[i]) != 1)
			return false;
	}

	return true;
}

static void set_sign_key_usage(void)
{
	printf("Usage: set-sign-key [options]\nOptions:\n"
		"\t -c, --sign-key <remote csrk>\tRemote CSRK\n"
		"e.g.:\n"
		"\tset-sign-key -c D8515948451FEA320DC05A2E88308188\n");
}

static bool remote_counter(uint32_t *sign_cnt, void *user_data)
{
	static uint32_t cnt = 0;

	if (*sign_cnt < cnt)
		return false;

	cnt = *sign_cnt;

	return true;
}

static void cmd_set_sign_key(struct server *server, char *cmd_str)
{
	char *argv[3];
	int argc = 0;
	uint8_t key[16];

	memset(key, 0, 16);

	if (!parse_args(cmd_str, 2, argv, &argc)) {
		set_sign_key_usage();
		return;
	}

	if (argc != 2) {
		set_sign_key_usage();
		return;
	}

	if (!strcmp(argv[0], "-c") || !strcmp(argv[0], "--sign-key")) {
		if (convert_sign_key(argv[1], key))
			bt_att_set_remote_key(server->att, key, remote_counter,
									server);
	} else
		set_sign_key_usage();
}

static void btgatt_start_bluetooth(void)
{
	pfmg_macaddr_id_t mac_id;
	unsigned char mac[PFMG_ETHER_ADDR_LEN];
	char cmd_buf[256];
	
	mac_id = PFMG_MACADDR_BLUETOOTH;
	pfmg_read_macaddr(mac_id, mac);
	unlink("/etc/bluetooth/.bt_nv.bin");
	snprintf(cmd_buf, sizeof(cmd_buf),
	   "btnvtool -b %02x.%02x.%02x.%02x.%02x.%02x",
	   mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	system(cmd_buf);
	system("/etc/init.d/bluetooth-5 start");
	sleep(2);
}

static void btgatt_stop_bluetooth(void)
{
	system("/etc/init.d/bluetooth-5 stop");
}

static void btgatt_set_advertising(void)
{
	int i, n;
	char cmd_buf[256];
	char serial_num[KFS_PFMG_SIZE];
	
	pfmg_read_serial_number(serial_num, sizeof(serial_num));
	
	system("btmgmt power off");
	system("btmgmt le on");
	system("btmgmt bredr off");
	system("btmgmt sc on");
	system("btmgmt debug-keys off");
	system("btmgmt bondable on");
	system("btmgmt name NEC_Test");
	system("btmgmt connectable on");
	system("btmgmt power on");
	system("hcitool -i hci0 cmd 0x08 0x0008 1f 02 01 06 09 09 4e 45 43 20 54 65 73 74 11 06 dc 2d 33 6e c5 60 49 8e 0a 44 91 02 6d d8 0c d8");
	n = snprintf(cmd_buf, sizeof(cmd_buf), "hcitool -i hci0 cmd 0x08 0x0009 12 11 ff 22 00");
	for (i = 0; i < 14; i++) {
		n += snprintf(cmd_buf + n, sizeof(cmd_buf) - n, " %02x", serial_num[i]);
	}
	for (i = 0; i < 13; i++) {
		n += snprintf(cmd_buf + n, sizeof(cmd_buf) - n, " 00");
	}
	system(cmd_buf);
//	system("btmgmt advertising on");
}

static void btgatt_enable_advertising(int enable)
{
	if (enable) {
		system("btmgmt connectable on");
		//system("btmgmt advertising on");
		system("hcitool -i hci0 cmd 0x08 0x000a 01");
	} else {
		system("btmgmt advertising off");
		//system("btmgmt connectable off");
		system("hcitool -i hci0 cmd 0x08 0x000a 00");
	}
}

static void cmd_help(struct server *server, char *cmd_str);

typedef void (*command_func_t)(struct server *server, char *cmd_str);

static struct {
	char *cmd;
	command_func_t func;
	char *doc;
} command[] = {
	{ "help", cmd_help, "\tDisplay help message" },
	{ "notify", cmd_notify, "\tSend handle-value notification" },
	{ "heart-rate", cmd_heart_rate, "\tHide/Unhide Heart Rate Service" },
	{ "services", cmd_services, "\tEnumerate all services" },
	{ "set-sign-key", cmd_set_sign_key,
			"\tSet remote signing key for signed write command"},
	{ }
};

static void cmd_help(struct server *server, char *cmd_str)
{
	int i;

	printf("Commands:\n");
	for (i = 0; command[i].cmd; i++)
		printf("\t%-15s\t%s\n", command[i].cmd, command[i].doc);
}

static void prompt_read_cb(int fd, uint32_t events, void *user_data)
{
	ssize_t read;
	size_t len = 0;
	char *line = NULL;
	char *cmd = NULL, *args;
	struct server *server = user_data;
	int i;

	if (events & (EPOLLRDHUP | EPOLLHUP | EPOLLERR)) {
		mainloop_quit();
		flg_finish = 1;
		return;
	}

	read = getline(&line, &len, stdin);
	if (read < 0)
		return;

	if (read <= 1) {
//		cmd_help(server, NULL);
		print_prompt();
		return;
	}

	line[read-1] = '\0';
	args = line;

	while ((cmd = strsep(&args, " \t")))
		if (*cmd != '\0')
			break;

	if (!cmd)
		goto failed;

	for (i = 0; command[i].cmd; i++) {
		if (strcmp(command[i].cmd, cmd) == 0)
			break;
	}

	if (command[i].cmd)
		command[i].func(server, args);
	else
		PRERR("Unknown command: %s\n", line);

failed:
	print_prompt();

	free(line);
}

static void signal_cb(int signum, void *user_data)
{
	struct server *server = (struct server *)user_data;
	
	switch (signum) {
	case SIGINT:
	case SIGTERM:
		mainloop_quit();
		flg_finish = 1;
		
		break;
	default:
		break;
	}
}

static void sig_btgatt_handler(int signum)
{
	switch (signum) {
	case SIGINT:
	case SIGTERM:
		if (accept_flg == 0) {
			mainloop_quit();
			flg_finish = 1;
		} else {
//			btgatt_stop_bluetooth();
			btgatt_enable_advertising(0);
			pthread_cancel(select_tid);
			exit(0);
		}
		
		break;
	default:
		break;
	}
}

static void
btgatt_stat_recv(void *arg, int s, struct local_comm_hdr *hdr,
    void *dbuf, uint32_t dlen)
{
	uint32_t notify;
	uint32_t rt_notify = 0;
	uint32_t wifi_notify = 0;
	uint32_t lte_notify = 0;
	uint32_t fw_notify = 0;
	uint32_t rt_auth = 0;
	uint32_t server_notify = 0;
	
	switch(hdr->type) {
	case LIBBTGATT_TYPE_NOTIFY :
		notify = (uint32_t)(hdr->status);
		PRLOG("Notify From Host : %08x\n", notify);
		
		if (btgatt_server == NULL) {
			PRLOG("BTGATT Server Down\n");
			return;
		}
		
		if (btgatt_server->auth_stat != 0) {
			PRLOG("Authorization Locked\n");
			return;
		}
		
		/* [^NOTIFY */
		if (notify & LIBBTGATT_RT_STAT_NOTIFY) {
			rt_notify |= BTGATT_NOTIFY_RT_STATUS;
		}
		if (rt_notify != 0) {
			necpf_rt_notify_cb(btgatt_server, rt_notify);
		}
		
		/* Wi-FiNOTIFY */
		if (notify & LIBBTGATT_WIFI_STAT_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_STATUS;
		}
		if (notify & LIBBTGATT_WIFI_SSID_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_SSID;
		}
		if (notify & LIBBTGATT_WIFI_PASS_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_PASS;
		}
		if (notify & LIBBTGATT_WIFI_STA_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_STA;
		}
		if (notify & LIBBTGATT_WIFI_ENC_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_ENC;
		}
		if (notify & LIBBTGATT_WIFI_CACDETECT_NOTIFY) {
			wifi_notify |= BTGATT_NOTIFY_WIFI_CACDETECT;
		}
		if (wifi_notify != 0) {
			necpf_wifi_notify_cb(btgatt_server, wifi_notify);
		}
		
		/* LTENOTIFY */
		if (notify & LIBBTGATT_LTE_STAT_NOTIFY) {
			lte_notify |= BTGATT_NOTIFY_LTE_STATUS;
		}
		if (notify & LIBBTGATT_LTE_TRAF_NOTIFY) {
			lte_notify |= BTGATT_NOTIFY_LTE_TRAF;
		}
		if (lte_notify != 0) {
			necpf_lte_notify_cb(btgatt_server, lte_notify);
		}
		
		/* F/WNOTIFY */
		if (notify & LIBBTGATT_FW_STAT_NOTIFY) {
			fw_notify |= BTGATT_NOTIFY_FW_STATUS;
		}
		if (notify & LIBBTGATT_FW_UPD_NOTIFY) {
			fw_notify |= BTGATT_NOTIFY_FW_UPDATE;
		}
		if (fw_notify != 0) {
			necpf_fw_notify_cb(btgatt_server, fw_notify);
		}
		
		/* Sever ControlNOTIFY */
		if (notify & LIBBTGATT_SERVER_CONTROL_NOTIFY) {
			server_notify |= BTGATT_NOTIFY_SERVER_CONTROL;
		}
		if (server_notify != 0) {
			necpf_server_control_notify_cb(btgatt_server, server_notify);
		}
		
		break;
	case LIBBTGATT_TYPE_GETAUTH :
		if (btgatt_server == NULL) {
			PRLOG("BTGATT Server Down\n");
			rt_auth = 2;
		} else {
			rt_auth = btgatt_server->auth_stat;
		}
		lib_local_comm_send(s, LIBBTGATT_TYPE_GETAUTH, rt_auth, NULL, 0);
		break;
	case LIBBTGATT_TYPE_SETAUTH:
		if (btgatt_server == NULL) {
			PRLOG("BTGATT Server Down\n");
			return;
		}
		btgatt_server->auth_stat = (int)(hdr->status);
		if (btgatt_server->auth_stat == 1) {
			sysmgrble_set_auth_ng();
		}
		break;
	default :
		break;
	}
}

static void sig_select_handler(int signum)
{
	if (h != NULL) {
		lib_local_comm_server_close(h);
		h = NULL;
		unlink(BTGATT_EVENT_SOCKFILE);
	}
	exit(0);
}

static void *btgatt_thread_select_event_loop(void *arg)
{
	unlink(BTGATT_EVENT_SOCKFILE);
	h = lib_local_comm_server_open(BTGATT_EVENT_SOCKFILE, btgatt_stat_recv, NULL);
	if (h == NULL) {
		PRERR("BTGATT : lib_local_comm_server_open failure\n");
		exit(EXIT_FAILURE);
	}
	select_event_loop();
	
	return NULL;
}

int main(int argc, char *argv[])
{
	int opt, ret;
	bdaddr_t src_addr;
	int dev_id = -1;
	int fd;
	int sec = BT_SECURITY_LOW;
	uint8_t src_type = BDADDR_LE_PUBLIC;
	uint16_t mtu = 0;
	sigset_t mask;
	bool hr_visible = false;
	struct sigaction act;

	while ((opt = getopt_long(argc, argv, "+hvrs:t:m:i:",
						main_options, NULL)) != -1) {
		switch (opt) {
		case 'h':
			usage();
			return EXIT_SUCCESS;
		case 'v':
			verbose = true;
			break;
		case 's':
			if (strcmp(optarg, "low") == 0)
				sec = BT_SECURITY_LOW;
			else if (strcmp(optarg, "medium") == 0)
				sec = BT_SECURITY_MEDIUM;
			else if (strcmp(optarg, "high") == 0)
				sec = BT_SECURITY_HIGH;
			else {
				PRERR("Invalid security level\n");
				return EXIT_FAILURE;
			}
			break;
		case 't':
			if (strcmp(optarg, "random") == 0)
				src_type = BDADDR_LE_RANDOM;
			else if (strcmp(optarg, "public") == 0)
				src_type = BDADDR_LE_PUBLIC;
			else {
				PRERR("Allowed types: random, public\n");
				return EXIT_FAILURE;
			}
			break;
		case 'm': {
			int arg;

			arg = atoi(optarg);
			if (arg <= 0) {
				PRERR("Invalid MTU: %d\n", arg);
				return EXIT_FAILURE;
			}

			if (arg > UINT16_MAX) {
				PRERR("MTU too large: %d\n", arg);
				return EXIT_FAILURE;
			}

			mtu = (uint16_t) arg;
			break;
		}
		case 'i':
			dev_id = hci_devid(optarg);
			if (dev_id < 0) {
				perror("Invalid adapter");
				return EXIT_FAILURE;
			}

			break;
		default:
			PRERR("Invalid option: %c\n", opt);
			return EXIT_FAILURE;
		}
	}

	argc -= optind;
	argv -= optind;
	optind = 0;

	if (argc) {
		usage();
		return EXIT_SUCCESS;
	}
	
	pthread_create(&select_tid, NULL, btgatt_thread_select_event_loop, NULL);
	
//	btgatt_start_bluetooth();
	
	if (dev_id == -1)
		bacpy(&src_addr, BDADDR_ANY);
	else if (hci_devba(dev_id, &src_addr) < 0) {
		perror("Adapter not available");
		return EXIT_FAILURE;
	}
	
//	btgatt_set_advertising();

	while(1) {
		btgatt_enable_advertising(1);
		accept_flg = 1;
		fd = l2cap_le_att_listen_and_accept(&src_addr, sec, src_type);
		if (fd < 0) {
			PRERR("Failed to accept L2CAP ATT connection\n");
			return EXIT_FAILURE;
		}
		accept_flg = 0;
		btgatt_enable_advertising(0);
		sysmgrble_set_ble_state(1);
		
		mainloop_init();
		
		btgatt_server = server_create(fd, mtu, hr_visible);
		if (!btgatt_server) {
			close(fd);
			return EXIT_FAILURE;
		}
		
/*		if (mainloop_add_fd(fileno(stdin),
					EPOLLIN | EPOLLRDHUP | EPOLLHUP | EPOLLERR,
					prompt_read_cb, btgatt_server, NULL) < 0) {
			PRERR("Failed to initialize console\n");
			server_destroy(btgatt_server);
			btgatt_server = NULL;
		
			return EXIT_FAILURE;
		} */
		
		PRLOG("Running GATT server\n");
		
/*		sigemptyset(&mask);
		sigaddset(&mask, SIGINT);
		sigaddset(&mask, SIGTERM);
		
		mainloop_set_signal(&mask, signal_cb, btgatt_server, NULL); */
		signal(SIGINT, sig_btgatt_handler);
		signal(SIGTERM, sig_btgatt_handler);
		
//		print_prompt();
		
		mainloop_run();
		
		sysmgrble_set_ble_state(0);
		PRLOG("\n\nShutting down...\n");
		
		server_destroy(btgatt_server);
		btgatt_server = NULL;
		if (flg_finish) {
			break;
		}
	}
//	btgatt_stop_bluetooth();
	pthread_cancel(select_tid);

	return EXIT_SUCCESS;
}
