/* Copyright (C) 2008 Rodolfo Giometti <giometti@linux.it>
 * Copyright (C) 2008 Eurotech S.p.A. <info@eurotech.it>
 * Based on a previous work by Copyright (C) 2008 Texas Instruments, Inc.
 *
 * Copyright (c) 2011, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/idr.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <asm/unaligned.h>
#include <linux/time.h>
#include <linux/gpio.h>
#include <linux/err.h>
#include <linux/delay.h>
#include <linux/kthread.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <soc/qcom/socinfo.h>
#include <linux/smb135x-charger.h>
#include "bq27421config.c"

#define BQ27421_CONFREV 10	/* 注意！ガスゲージIC設定値のrevisionに応じて手動で変える必要がある */
#define DRIVER_VERSION			"1.0.0"
#define DEV_TYPE   0x0421
#if BQ27421_CONFREV == 03 || BQ27421_CONFREV == 06 || BQ27421_CONFREV == 07
#define isAlreadyConfigured(x)	isAlreadyConfigured03(x)
#elif (BQ27421_CONFREV == 10)
#define isAlreadyConfigured(x)	isAlreadyConfigured12(x)
#else
#error "This revision(BQ27421_CONFREV) is not supported."
#endif

/* Bq27421 standard data commands */
#define BQ27421_REG_CNTL		0x00
#define BQ27421_REG_TEMP		0x02
#define BQ27421_REG_VOLT		0x04
#define BQ27421_REG_FLAGS		0x06
#define BQ27421_REG_FLAGS2		0x07
#define BQ27421_REG_NAC			0x08
#define BQ27421_REG_FAC			0x0a
#define BQ27421_REG_RM			0x0c
#define BQ27421_REG_FCC			0x0e
#define BQ27421_REG_AI			0x10
#define BQ27421_REG_SI			0x12
#define BQ27421_REG_MLI			0x14
#define BQ27421_REG_AP			0x18
#define BQ27421_REG_SOC			0x1c
#define BQ27421_REG_INTTEMP		0x1e
#define BQ27421_REG_SOH			0x20

#define BQ27421_REG_DC			0x3c

#define BQ27421_FLAG_DSG		BIT(0)
#define BQ27421_FLAG_SOCF		BIT(1)
#define BQ27421_FLAG_SOC1		BIT(2)
#define BQ27421_FLAG_CFG		BIT(4)
#define BQ27421_FLAG_FC			BIT(9)
#define BQ27421_FLAG_INITCOMP		BIT(7)
#define BQ27421_FLAG_SS			BIT(13)

/* Control subcommands */
#define BQ27421_SUBCMD_CTNL_STATUS	0x0000
#define BQ27421_SUBCMD_DEVCIE_TYPE	0x0001
#define BQ27421_SUBCMD_FW_VER		0x0002
#define BQ27421_SUBCMD_DF_CSUM		0x0004
#define BQ27421_SUBCMD_PREV_MACW	0x0007
#define BQ27421_SUBCMD_CHEM_ID		0x0008
#define BQ27421_SUBCMD_BAT_INS		0x000c
#define BQ27421_SUBCMD_BAT_REM		0x000d
#define BQ27421_SUBCMD_SET_HIB		0x0011
#define BQ27421_SUBCMD_CLR_HIB		0x0012
#define BQ27421_SUBCMD_SET_CFGUPD	0x0013
#define BQ27421_SUBCMD_ENABLE_SDOWN	0x0014
#define BQ27421_SUBCMD_SDOWN		0x0015
#define BQ27421_SUBCMD_TOGGLE_GPOUT	0x0016
#define BQ27421_SUBCMD_SEALED		0x0020
#define BQ27421_SUBCMD_RESET		0x0041
#define BQ27421_SUBCMD_SW_RESET		0x0042
#define BQ27421_SUBCMD_UNSEALED		0x8000

/* Bq27421 extended commands */
#define BQ27421_DATACLASS		0x3e
#define BQ27421_BLOCKDATACHECKSUM	0x60

#define ZERO_DEGREE_CELSIUS_IN_TENTH_KELVIN   (-2731)

#define BQ27421_POLLING_INTERVAL	10000	/* ms */
#define BQ27421_RETRYLIMIT		3

#define BOOTMODE_BT	0x80000000
#define BOOTMODE_PT1	0x90000000
#define BOOTMODE_PT2	0xA0000000
#define BOOTMODE_PT3	0xB0000000
#define BOOTMODE_PT4	0xC0000000
#define BOOTMODE_PT5	0xD0000000
#define BOOTMODE_MASK	0xF0000000
#define IS_BOOTMODE_NO_BATTERY(__bflag)		\
	((((__bflag) & BOOTMODE_MASK) == BOOTMODE_PT1) || \
	 (((__bflag) & BOOTMODE_MASK) == BOOTMODE_PT3) || \
	 (((__bflag) & BOOTMODE_MASK) == BOOTMODE_PT4) || \
	 (((__bflag) & BOOTMODE_MASK) == BOOTMODE_PT5))

#define CURRENT_100_MA		100
#define CURRENT_100_MA		100
#define CURRENT_150_MA		150
#define CURRENT_500_MA		500
#if defined(CONFIG_PLATFORM_NEC)
#define CURRENT_700_MA		700
#endif /* defined(CONFIG_PLATFORM_NEC) */
#define CURRENT_900_MA		900
#if defined(CONFIG_PLATFORM_NEC)
#define CURRENT_950_MA		950
#define CURRENT_1000_MA		1000
#define CURRENT_1400_MA		1400
#endif /* defined(CONFIG_PLATFORM_NEC) */

static volatile int battery_remain = -1;

#define DEFAULT_SW_RESET_CYCLE	(3300)  /* cycle = 3300 * 10sec */
static volatile int sw_reset_counter = DEFAULT_SW_RESET_CYCLE;
static volatile int sw_reset_cycle = DEFAULT_SW_RESET_CYCLE;

#define DEFAULT_COUNT_TO_FORCED_DISCHG	(4400)  /* cycle = 4400 * 10sec 12hours */
static volatile int dischg_counter = DEFAULT_COUNT_TO_FORCED_DISCHG;
static volatile int dischg_start_period = DEFAULT_COUNT_TO_FORCED_DISCHG;

/* If the system has several batteries we need a different name for each
 * of them...
 */
static DEFINE_IDR(battery_id);
static DEFINE_MUTEX(battery_mutex);
static DEFINE_MUTEX(battery_check_mutex);
static DEFINE_MUTEX(longlife_mutex);

static wait_queue_head_t wait_q;
static atomic_t wait_condition;

#define BQ27421_PM_RESUME_REMAIN_TIMEOUT	3*1000	/* 3sec */
static wait_queue_head_t pm_resume_remain_q;
static atomic_t pm_resume_remain_condition;

struct bq27421_device_info;
struct bq27421_access_methods {
	int (*read)(u8 reg, int *rt_value, int b_single,
		struct bq27421_device_info *di);
};

struct bq27421_device_info {
	struct device			*dev;
	int				id;
	struct bq27421_access_methods	*bus;
	struct i2c_client		*client;

	struct power_supply		bat;

#ifdef CONFIG_PM
	int				gp_out_wake_irq;
#endif

	struct mutex lock;
};

static enum power_supply_property bq27421_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,	/* VOLT */
	POWER_SUPPLY_PROP_CURRENT_NOW,	/* AI */
	POWER_SUPPLY_PROP_CAPACITY,	/* SOC */
	POWER_SUPPLY_PROP_CAPACITY_LEVEL,/* FLAGS */
	POWER_SUPPLY_PROP_TEMP,		/* TEMP */
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_CHARGE_NOW,	/* RM */
	POWER_SUPPLY_PROP_CHARGE_FULL,	/* FCC */
	POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN,/* DC */
	POWER_SUPPLY_PROP_ENERGY_AVG,	/* AP */
	POWER_SUPPLY_PROP_CURRENT_MAX,	/* MLI */
	POWER_SUPPLY_PROP_NOTIFY_SRC,
};

//#define ENABLE_CHG_PG
#ifdef ENABLE_CHG_PG
static int /*GPIO_BIN_CONT, GPIO_CHG_CE, */GPIO_CHG_PG;
#endif /* ENABLE_CHG_PG */
static struct task_struct *bq27421_polling = NULL;
static int POLLING_ENABLE = 1;
static int ISSETTING = 0;

static atomic_t power_on_charge_control_enable;
static bool execute_auto_charge_control = false;
static bool discharge_timer_set = false;
static bool discharge_status = false;
static bool power_on_state = true;
static bool discharge_timer_expired = false;
static bool fullcharge_on_powoff = false;

#define GPIO_GP_OUT	20

#define ENABLE_SOC_ZERO_CHK_VLT
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
	#define VOLT_CHECK_MAX  5
	int volt_read_cnt;
	#define VOLT_CHECK_VALUE 3200
#endif

static int bq27421_i2c_txsubcmd(u8 reg, unsigned short subcmd,
		struct bq27421_device_info *di);

int battery_setForceDischarge(int enable)
{
	struct power_supply *battery_psy;

	battery_psy = power_supply_get_by_name("battery");

	if (battery_psy == NULL)
		return -1;
		
	discharge_status = (enable != 0 ? true : false);

	return power_supply_set_force_discharge(battery_psy,
						 enable != 0 ? CURRENT_100_MA : 0);
	return 0;
}

int battery_setSaveCurrent(int enable)
{
	struct power_supply *battery_psy;

	battery_psy = power_supply_get_by_name("battery");

	if (battery_psy == NULL)
		return -1;
		
	return power_supply_set_save_current(battery_psy,
						 enable != 0 ? CURRENT_700_MA : 0);
	return 0;
}

static void
restore_discharge_ctrl_flag(void)
{
	if ((execute_auto_charge_control != false) || (discharge_timer_set != false) || (discharge_status != false) || (discharge_timer_expired != false))
		printk("Clear discharge control\n");
	execute_auto_charge_control = false;
	discharge_timer_set = false;
	discharge_timer_expired = false;
	fullcharge_on_powoff = false;
	battery_setForceDischarge(0);
	battery_setSaveCurrent(0);


}

int battery_getVBUSConnect(void)
{
	int ret;
	struct power_supply *battery_psy;
	int vbus_connection = 0;

	battery_psy = power_supply_get_by_name("battery");

	if (battery_psy == NULL)
		return -1;

	ret = power_supply_get_vbus_connection(battery_psy, &vbus_connection);
	if (ret)
		return -2;
		
	if (!vbus_connection) {
		restore_discharge_ctrl_flag();
	}

	return vbus_connection;
}

int battery_getChargeEnable(void)
{
	int ret;
	int enable;
	struct power_supply *battery_psy;

	battery_psy = power_supply_get_by_name("battery");

	if (battery_psy == NULL)
		return -1;

	ret = power_supply_get_charging_enabled(battery_psy, &enable);
	if (ret)
		return -2;
	return enable;

	return 0;
}

static int bq27421_read(u8 reg, int *rt_value, int b_single,
			struct bq27421_device_info *di)
{
	int ret;

	mutex_lock(&di->lock);
	ret = di->bus->read(reg, rt_value, b_single, di);
	mutex_unlock(&di->lock);

	return ret;
}

static int bq27421_battery_status(struct bq27421_device_info *di,
	union power_supply_propval *val)
{
	int status, flags, ret;

	ret = bq27421_read(BQ27421_REG_FLAGS, &flags, 0, di);
	if (ret) {
		dev_err(di->dev, "error battery status\n");
		return ret;
	}

	if (flags & BQ27421_FLAG_FC)
		status = POWER_SUPPLY_STATUS_FULL;
	else if (flags & BQ27421_FLAG_DSG)
		status = POWER_SUPPLY_STATUS_DISCHARGING;
	else
		status = POWER_SUPPLY_STATUS_CHARGING;

	val->intval = status;

	return 0;
}

static int bq27421_battery_capacity_level(struct bq27421_device_info *di,
	union power_supply_propval *val)
{
	int level, flags, ret;

	ret = bq27421_read(BQ27421_REG_FLAGS, &flags, 0, di);
	if (ret) {
		dev_err(di->dev, "error battery status\n");
		return ret;
	}

	if (flags & BQ27421_FLAG_FC)
		level = POWER_SUPPLY_CAPACITY_LEVEL_FULL;
	else if (flags & BQ27421_FLAG_SOCF)
		level = POWER_SUPPLY_CAPACITY_LEVEL_CRITICAL;
	else if (flags & BQ27421_FLAG_SOC1)
		level = POWER_SUPPLY_CAPACITY_LEVEL_LOW;
	else
		level = POWER_SUPPLY_CAPACITY_LEVEL_NORMAL;

	val->intval = level;

	return 0;
}

/*
 * Return the battery temperature in tenths of degree Celsius
 * Or < 0 if something fails.
 */
static int bq27421_battery_temperature(struct bq27421_device_info *di,
    union power_supply_propval *val)
{
	int ret;
	int temp = 0;

	ret = bq27421_read(BQ27421_REG_TEMP, &temp, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading temperature\n");
		return ret;
	}

	val->intval = temp + ZERO_DEGREE_CELSIUS_IN_TENTH_KELVIN;

	return 0;
}

static int bq27421_simple_value(struct bq27421_device_info *di, u8 reg,
    union power_supply_propval *val)
{
	int value = 0, ret;

	ret = bq27421_read(reg, &value, 0, di);
	if (value < 0)
		return value;

	val->intval = value;

	return 0;
}

static int bq27421_subcmd_read(struct bq27421_device_info *di, int *rt_value,
    u16 subcmd)
{
	int ret;

	mutex_lock(&di->lock);
	ret = bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, subcmd, di);
	if (ret < 0) {
		mutex_unlock(&di->lock);
		return ret;
	}

	udelay(66);

	ret = di->bus->read(BQ27421_REG_CNTL, rt_value, 0, di);

	mutex_unlock(&di->lock);

	return ret;
}

#define RETRY_COUNT 5
static int bq27421_retry_sleep_ms[RETRY_COUNT] = {
	10, 20, 30, 40, 50
};
/*
 * i2c specific code
 */
static int bq27421_i2c_txsubcmd(u8 reg, unsigned short subcmd,
		struct bq27421_device_info *di)
{
	s32 ret;
	int retry_count = 0;

retry:
	ret = i2c_smbus_write_word_data(di->client, reg, subcmd);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x subcmd 0x%04x\n", __func__, ret, reg, subcmd);
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c write fail: can't write %02x to %02x: %d\n",
			subcmd, reg, ret);
		return ret;
	}
	pr_debug("Writing 0x%02x=0x%04x\n", reg, subcmd);
	return 0;
}

static int bq27421_read_i2c(u8 reg, int *rt_value, int b_single,
			struct bq27421_device_info *di)
{
	s32 ret;
	int retry_count = 0;
	int success_count = 0;

retry:
	if (!b_single)
		ret = i2c_smbus_read_word_data(di->client, reg);
	else
		ret = i2c_smbus_read_byte_data(di->client, reg);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x\n", __func__, ret, reg);
	if ((ret >= 0) && (success_count == 0)) {
		success_count++;
		goto retry;
	}
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		success_count = 0;
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c read fail: can't read from %02x: %d\n", reg, ret);
		return ret;
	} else {
		*rt_value = (int)ret;
	}

	return 0;
}

static int bq27421_i2c_read(struct bq27421_device_info *di, u8 reg, unsigned char *buf, int len)
{
	int i;
	int tmp;
	
	mutex_lock(&di->lock);
	
	for (i = 0; i < len; i++, reg++, buf++) {
		bq27421_read_i2c(reg, &tmp, 1, di);
		*buf = (unsigned char)tmp;
	}

	mutex_unlock(&di->lock);

	return 0;
}

static int bq27421_i2c_write_byte(struct bq27421_device_info *di, u8 reg, const unsigned char *buf)
{
	s32 ret;
	int retry_count = 0;

retry:
	ret = i2c_smbus_write_byte_data(di->client, reg, *buf);
	if (ret < 0)
		pr_info("%s ret %d reg 0x%02x buf 0x%02x\n", __func__, ret, reg, *buf);
	if (ret < 0 && retry_count < RETRY_COUNT) {
		/* sleep for few ms before retrying */
		msleep(bq27421_retry_sleep_ms[retry_count++]);
		goto retry;
	}
	if (ret < 0) {
		dev_err(di->dev,
			"i2c write fail: can't write %02x to %02x: %d\n",
			*buf, reg, ret);
		return ret;
	}
	pr_debug("Writing 0x%02x=0x%02x\n", reg, *buf);
	return 0;
}

static int bq27421_i2c_write(struct bq27421_device_info *di, u8 reg, unsigned char *buf, int len)
{
	int i;

	mutex_lock(&di->lock);

	for (i = 0; i < len; i++, reg++, buf++)
		bq27421_i2c_write_byte(di, reg, buf);

	mutex_unlock(&di->lock);

	return 0;
}

static int isBatteryConnected(struct bq27421_device_info *di)
{
	int dev_type;

	/* Verify Device Type */
	bq27421_subcmd_read(di, &dev_type, BQ27421_SUBCMD_DEVCIE_TYPE);
	if (dev_type != DEV_TYPE) {
		dev_err(di->dev, "error: Nonexpected Device type 0x%04x\n", dev_type);
		return 0;
	}

	return 1;
}

#if BQ27421_CONFREV == 03 || BQ27421_CONFREV == 06 || BQ27421_CONFREV == 07
static int isAlreadyConfigured03(struct bq27421_device_info *di)
{	/* rev0.3 ではsubclassID 107(0x6B)のcheck sumが0x3Aかで確認する */
	unsigned char data[2];

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
	mutex_unlock(&di->lock);

	data[0] = 0x6B;
	data[1] = 0x00;
	bq27421_i2c_write(di, BQ27421_DATACLASS, data, 2);
	msleep(1); /* 少し間を開けないとエラーになる */
	bq27421_i2c_read(di, BQ27421_BLOCKDATACHECKSUM, data, 1);

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	if (data[0] == 0x3A)
		return 1;
	else
		return 0;
}
#endif /* BQ27421_CONFREV 0.3||0.6||0.7 */

#if BQ27421_CONFREV == 10
static int isAlreadyConfigured12(struct bq27421_device_info *di)
{	/* rev0.12 ではsubclassID 107(0x6B)のcheck sumが0xFDかで確認する */
	unsigned char data[2];

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
	mutex_unlock(&di->lock);

	data[0] = 0x6B;
	data[1] = 0x00;
	bq27421_i2c_write(di, BQ27421_DATACLASS, data, 2);
	msleep(1); /* 少し間を開けないとエラーになる */
	bq27421_i2c_read(di, BQ27421_BLOCKDATACHECKSUM, data, 1);

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	if (data[0] == 0xFD)
		return 1;
	else
		return 0;
}
#endif /* BQ27421_CONFREV == 10 */

#define to_bq27421_device_info(x) container_of((x), \
				struct bq27421_device_info, bat);

static int bq27421_battery_get_property(struct power_supply *psy,
					enum power_supply_property psp,
					union power_supply_propval *val)
{
	int ret = 0;
	struct bq27421_device_info *di = to_bq27421_device_info(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		ret = bq27421_battery_status(di, val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_VOLT, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = 1;
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_AI, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		if(atomic_read(&pm_resume_remain_condition) == 0 ) {
			dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));
			if (wait_event_timeout(pm_resume_remain_q, atomic_read(&pm_resume_remain_condition) == 1,
					       msecs_to_jiffies(BQ27421_PM_RESUME_REMAIN_TIMEOUT)) != 0) {
				pr_info("%s %u: wakeup @@@@\n", __func__, __LINE__);
			}
		}
		val->intval = battery_remain;
		dev_dbg(di->dev,"@@@@ %s battery_remain=%d\n",__FUNCTION__,battery_remain);
		ret = 0;
		break;
	case POWER_SUPPLY_PROP_CAPACITY_LEVEL:
		ret = bq27421_battery_capacity_level(di, val);
		break;
	case POWER_SUPPLY_PROP_TEMP:
		ret = bq27421_battery_temperature(di, val);
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = POWER_SUPPLY_TECHNOLOGY_LION;
		break;
	case POWER_SUPPLY_PROP_CHARGE_NOW:
		ret = bq27421_simple_value(di, BQ27421_REG_RM, val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL:
		ret = bq27421_simple_value(di, BQ27421_REG_FCC, val);
		break;
	case POWER_SUPPLY_PROP_CHARGE_FULL_DESIGN:
		ret = bq27421_simple_value(di, BQ27421_REG_DC, val);
		break;
	case POWER_SUPPLY_PROP_ENERGY_AVG:
		ret = bq27421_simple_value(di, BQ27421_REG_AP, val);
		break;
	case POWER_SUPPLY_PROP_CURRENT_MAX:
		ret = bq27421_simple_value(di, BQ27421_REG_MLI, val);
		if (ret == 0)
			val->intval *= 1000;
		break;
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		ret = discharge_status;
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int bq27421_battery_set_property(struct power_supply *psy,
					enum power_supply_property psp,
					union power_supply_propval *val)
{
	int ret = 0;
	struct bq27421_device_info *di = to_bq27421_device_info(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		atomic_set(&wait_condition, 1);
		wake_up(&wait_q);
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int bq27421_battery_is_writeable(struct power_supply *psy,
				       enum power_supply_property prop)
{
	int rc;

	switch (prop) {
	case POWER_SUPPLY_PROP_NOTIFY_SRC:
		rc = 1;
		break;
	default:
		rc = 0;
		break;
	}
	return rc;
}
static void bq27421_external_power_changed(struct power_supply *psy)
{
}

static int bq27421_powersupply_init(struct bq27421_device_info *di)
{
	int ret, flags = 0, type = 0, fw_ver = 0;

	mutex_init(&di->lock);

	di->bat.type = POWER_SUPPLY_TYPE_BATTERY;
	di->bat.properties = bq27421_battery_props;
	di->bat.num_properties = ARRAY_SIZE(bq27421_battery_props);
	di->bat.get_property = bq27421_battery_get_property;
	di->bat.set_property = bq27421_battery_set_property;
	di->bat.external_power_changed = bq27421_external_power_changed;
	di->bat.property_is_writeable = bq27421_battery_is_writeable;

	bq27421_subcmd_read(di, &flags, BQ27421_SUBCMD_CTNL_STATUS);

	bq27421_subcmd_read(di, &type, BQ27421_SUBCMD_DEVCIE_TYPE);

	bq27421_subcmd_read(di, &fw_ver, BQ27421_SUBCMD_FW_VER);

	dev_info(di->dev, "DEVICE_TYPE is 0x%02X, FIRMWARE_VERSION is 0x%02X\n",
			type, fw_ver);
	dev_info(di->dev, "Complete bq27421 configuration 0x%02X\n", flags);

	dev_info(di->dev, "support ver. %s enabled\n", DRIVER_VERSION);

	ret = power_supply_register(di->dev, &di->bat);
	if (ret) {
		dev_err(di->dev, "failed to register battery: %d\n", ret);
		return ret;
	}

	return 0;
}

static void bq27421_powersupply_unregister(struct bq27421_device_info *di)
{
	power_supply_unregister(&di->bat);

	mutex_destroy(&di->lock);
}

#if 1
static int stdreg;
static int subcmd;
static ssize_t bq27421_read_stdcmd(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	int ret;
	int temp = 0;
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = to_bq27421_device_info(psy);

	if (stdreg > 0x00) {
		ret = bq27421_read(stdreg, &temp, 0, di);
		if (ret)
			ret = snprintf(buf, PAGE_SIZE, "Read Error!\n");
		else
			ret = snprintf(buf, PAGE_SIZE, "0x%02x\n", temp);
	} else
		ret = snprintf(buf, PAGE_SIZE, "Register Error!\n");

	return ret;
}

static ssize_t bq27421_write_stdcmd(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	int cmd;

	sscanf(buf, "%x", &cmd);
	stdreg = cmd;
	return ret;
}

static ssize_t bq27421_read_subcmd(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	int ret;
	int temp = 0;
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = to_bq27421_device_info(psy);

	if (subcmd == BQ27421_SUBCMD_DEVCIE_TYPE ||
		 subcmd == BQ27421_SUBCMD_FW_VER ||
		 subcmd == BQ27421_SUBCMD_CHEM_ID) {

		ret = bq27421_subcmd_read(di, &temp, subcmd);

		if (ret)
			ret = snprintf(buf, PAGE_SIZE, "Read Error!\n");
		else
			ret = snprintf(buf, PAGE_SIZE, "0x%02x\n", temp);
	} else
		ret = snprintf(buf, PAGE_SIZE, "Register Error!\n");

	return ret;
}

static ssize_t bq27421_write_subcmd(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	int cmd;

	sscanf(buf, "%x", &cmd);
	subcmd = cmd;
	return ret;
}

static int REG = 0, LEN = 0;
static ssize_t bq27421_read_register(struct device *dev,
				     struct device_attribute *attr,
				     char *buf)
{
	int i, ret;
	unsigned char data[32];
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = to_bq27421_device_info(psy);
	
	bq27421_i2c_read(di, REG, data, LEN);

	ret = 0;
	for (i = 0; i < LEN; i++) {
		ret += snprintf(buf, PAGE_SIZE, "0x%02x: 0x%02x\n", REG + i, data[i]);
		buf += 11; /* len of "%#02x: %#02x\n" */
	}

	return ret;
}

/* format: {r/w} reg len [data...]  (all HEX) */
static ssize_t bq27421_write_register(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf,
				    size_t count)
{
	int i;
	unsigned char data[32];
	char read = buf[0];
	ssize_t ret = strnlen(buf, PAGE_SIZE);
	struct power_supply *psy = dev_get_drvdata(dev);
	struct bq27421_device_info *di = to_bq27421_device_info(psy);

	if (read != 'r' && read != 'w') {
		dev_err(dev, "usage: {r/w} reg len [data...]  (all HEX. len<=0x20)\n");
		return -EINVAL;
	}

	buf++;

	if (sscanf(buf, "%x %x", &REG, &LEN) != 2)
		return -EINVAL;

	if (LEN > 0x20){
		dev_err(dev, "usage: {r/w} reg len [data...]  (all HEX. len<=0x20)\n");
		return -EINVAL;
	}

	if (read == 'r')
		return ret;

	/* REG, LEN は読み飛ばす */
	while (*buf == ' ') /* skip ' ' */
		buf++;
	while (*buf != ' ' && *buf != '\0') /* skip REG */
		buf++;
	while (*buf == ' ')	/* skip ' ' */
		buf++;
	while (*buf != ' ' && *buf != '\0') /* skip LEN */
		buf++;
	while (*buf == ' ') /* skip ' ' */
		buf++;

	for (i = 0; i < LEN; i++) {
		if (sscanf(buf, "%x", (unsigned int *)(&data[i])) != 1)
			return -EINVAL;

		while (*buf != ' ' && *buf != '\0')
			buf++;
		while (*buf == ' ') /* skip ' ' */
			buf++;
	}

	bq27421_i2c_write(di, REG, data, LEN);
	return ret;
}

static ssize_t bq27421_read_pollsetting(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", POLLING_ENABLE);	
}

static ssize_t bq27421_write_pollsetting(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	int enable;
	if (sscanf(buf, "%d", &enable) != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	if (enable != 0 && enable != 1) {
		dev_err(dev, "Usage: {0/1}\n");
		return -EINVAL;	
	}

	POLLING_ENABLE = enable;

	return strnlen(buf, PAGE_SIZE);
}

static int longlife = 0;
static ssize_t bq27421_read_longlife(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", longlife);
}

static ssize_t bq27421_write_longlife(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);

	mutex_lock(&longlife_mutex);
	longlife = val;
	if (longlife == 1)
		restore_discharge_ctrl_flag();
	mutex_unlock(&longlife_mutex);
	atomic_set(&wait_condition, 1);
	wake_up(&wait_q);

	return count;
}

static ssize_t bq27421_read_swreset_cycle(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d/%d\n", sw_reset_counter, sw_reset_cycle);
}

static ssize_t bq27421_write_swreset_cycle(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);

	sw_reset_counter = sw_reset_cycle = val;

	return count;
}

static ssize_t bq27421_read_dischg_start_period(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d/%d\n", dischg_counter, dischg_start_period);
}

static ssize_t bq27421_write_dischg_start_period(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);

	dischg_counter = dischg_start_period = val;

	return count;
}

static ssize_t bq27421_read_charge_control_enable(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", power_on_charge_control_enable);
}

static ssize_t bq27421_write_charge_control_enable(struct device *dev,
					 struct device_attribute *attr,
					 const char *buf,
					 size_t count)
{
	unsigned long val = simple_strtoul(buf, NULL, 0);

	if (val == 0)
		atomic_set(&power_on_charge_control_enable, 0);
	else
		atomic_set(&power_on_charge_control_enable, 1);

	atomic_set(&wait_condition, 1);
	wake_up(&wait_q);

	return count;
}


static DEVICE_ATTR(std_cmd, 0664, bq27421_read_stdcmd,
	bq27421_write_stdcmd);
static DEVICE_ATTR(sub_cmd, 0664, bq27421_read_subcmd,
	bq27421_write_subcmd);
static DEVICE_ATTR(register, 0664, bq27421_read_register,
	bq27421_write_register);
static DEVICE_ATTR(polling, 0664, bq27421_read_pollsetting,
	bq27421_write_pollsetting);
static DEVICE_ATTR(longlife, 0664, bq27421_read_longlife,
	bq27421_write_longlife);
static DEVICE_ATTR(swreset_cycle, 0664, bq27421_read_swreset_cycle,
	bq27421_write_swreset_cycle);
static DEVICE_ATTR(dischg_start_period, 0664, bq27421_read_dischg_start_period,
	bq27421_write_dischg_start_period);
static DEVICE_ATTR(charge_control_enable, 0664, bq27421_read_charge_control_enable,
	bq27421_write_charge_control_enable);

static struct attribute *fs_attrs[] = {
	&dev_attr_std_cmd.attr,
	&dev_attr_sub_cmd.attr,
	&dev_attr_register.attr,
	&dev_attr_polling.attr,
	&dev_attr_longlife.attr,
	&dev_attr_swreset_cycle.attr,
	&dev_attr_dischg_start_period.attr,
	&dev_attr_charge_control_enable.attr,
	NULL,
};
static struct attribute_group fs_attr_group = {
	.attrs = fs_attrs,
};
#endif

static int set_necat_configure(struct bq27421_device_info *di)
{
	int cnt;
	int i, j, cmdtop;
	int reg, len, st;
	unsigned char data[32], val;

	cnt = 0;
	i = 0;
	cmdtop = 0;

	while (i < sizeof(BQ27421CONF)/sizeof(BQ27421CONF[0])) {
		if (i == cmdtop) {
			data[0] = 0;
			bq27421_i2c_write(di, 0x61, data, 1); /* BlockDataControl */
		}

		switch (BQ27421CONF[i]) {
		case 'W':
			reg = BQ27421CONF[i+1];
			len = BQ27421CONF[i+2];
			for (j = 0; j < len; j++)
				data[j] = (unsigned char)BQ27421CONF[i+3+j];

			bq27421_i2c_write(di, reg, data, len);

			i += 3 + len;
			msleep(1); /* 少し間を開けないとエラーになる */
			break;

		case 'X':
			st = BQ27421CONF[i+1];
			msleep(st);
			i += 2;
			break;

		case 'C':
			reg = BQ27421CONF[i+1];
			val = BQ27421CONF[i+2];
			bq27421_i2c_read(di, reg, data, 1);
			if (data[0] == val) {
				i += 3;
				cmdtop = i;
				cnt = 0;
			}
			else {
				i = cmdtop; /* 書き込みに失敗したらもう一度設定しなおす */
				if (++cnt >= BQ27421_RETRYLIMIT)
					return 0;
			}
			break;

		default:
			break;
		}
	}
	return 0;
}

static int pgcon_wakeup_detect;
module_param(pgcon_wakeup_detect, int, 0644);

static irqreturn_t
pgcon_wakeup_interrupt(int irq, void *ignored)
{
	pgcon_wakeup_detect = 1;
	return IRQ_HANDLED;
}

#define PMIC_RESET_GPIO		89

#ifdef CONFIG_PM_SLEEP
static irqreturn_t
gp_out_wakeup_interrupt(int irq, void *arg)
{
	pr_info("%s\n", __func__);

	gpio_set_value(PMIC_RESET_GPIO, 1);

	return IRQ_HANDLED;
}
#endif

static int bq27421_gpio_init(struct bq27421_device_info *di)
{
#define ERRMSG(X)							\
	dev_err(di->dev, "failed to request GPIO pin %d, error %d\n", X, err)
#define ERRMSG_DIR(X, D)						\
	dev_err(di->dev, "failed to config GPIO %d as " D " pin, err %d\n", X, err)

	int err;
	struct device_node *node = di->dev->of_node;
	int irq;

//	of_property_read_u32(node, "ti,bin_cont", &GPIO_BIN_CONT);
//	of_property_read_u32(node, "ti,chg_ce", &GPIO_CHG_CE);
#ifdef ENABLE_CHG_PG
	of_property_read_u32(node, "ti,chg_pg", &GPIO_CHG_PG);
#endif /* ENABLE_CHG_PG */

//	if ((err = gpio_request(GPIO_CHG_CE, "Charge Enable")) < 0) {
//		ERRMSG(GPIO_CHG_CE);
//		goto CE_fail;
//	}
#ifdef ENABLE_CHG_PG
	if ((err = gpio_request(GPIO_CHG_PG, "PG_connection")) < 0) {
		ERRMSG(GPIO_CHG_PG);
		goto PG_fail;
	}
#endif /* ENABLE_CHG_PG */
//	if ((err = gpio_request(GPIO_BIN_CONT, "BIN cont")) < 0) {
//		ERRMSG(GPIO_BIN_CONT);
//		goto BINCONT_fail;
//	}

#ifdef ENABLE_CHG_PG
	irq = gpio_to_irq(GPIO_CHG_PG);
	err = request_irq(irq, &pgcon_wakeup_interrupt,
	    IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "pgcon_wakeup", NULL);
	if (err < 0) {
		dev_err(di->dev, "setup fail wakeup interrupt\n");
		goto error;
	}
	enable_irq_wake(irq);
#endif /* ENABLE_CHG_PG */

#ifdef CONFIG_PM
	if ((err = gpio_request(GPIO_GP_OUT, "GP_OUT")) < 0) {
		ERRMSG(GPIO_GP_OUT);
		goto BINCONT_fail;
	}
	if ((err = gpio_direction_input(GPIO_GP_OUT)) < 0) {
		ERRMSG_DIR(GPIO_GP_OUT, "input");
		goto error;
	}
	gpio_export(GPIO_GP_OUT, false);

	irq = gpio_to_irq(GPIO_GP_OUT);
	err = request_irq(irq, &gp_out_wakeup_interrupt,
	    IRQF_TRIGGER_FALLING, "gp_out", (void *)di);
	if (err < 0) {
		dev_err(di->dev, "setup fail wakeup interrupt\n");
		goto error;
	}
	disable_irq(irq);
	di->gp_out_wake_irq = irq;
#endif

//	if ((err = gpio_direction_output(GPIO_CHG_CE, 0)) < 0) {
//		ERRMSG_DIR(GPIO_CHG_CE, "output");
//		goto error;		
//	}
#ifdef ENABLE_CHG_PG
	if ((err = gpio_direction_input(GPIO_CHG_PG)) < 0) {
		ERRMSG_DIR(GPIO_CHG_PG, "input");
		goto error;		
	}
	gpio_export(GPIO_CHG_PG, false);
#endif /* ENABLE_CHG_PG */
//	if ((err = gpio_direction_output(GPIO_BIN_CONT, 0)) < 0) {
//		ERRMSG_DIR(GPIO_BIN_CONT, "output");
//		goto error;		
//	}
	return 0;

error:
//	gpio_free(GPIO_BIN_CONT);
BINCONT_fail:
#ifdef ENABLE_CHG_PG
	gpio_free(GPIO_CHG_PG);
#endif /* ENABLE_CHG_PG */
PG_fail:
//	gpio_free(GPIO_CHG_CE);
#ifdef CONFIG_PM
	gpio_free(GPIO_GP_OUT);
#endif
//CE_fail:
	return err;
}

static int bq27421_check_longlife(struct bq27421_device_info *di)
{
	static int prelonglife = 0;
	if ((prelonglife != longlife) && (prelonglife == 1)) {
		dev_info(di->dev,
			 "bq27421 charging enable\n");
		smb135x_setChargeEnable(1);
	}
	prelonglife = longlife;

	return longlife;
}

static int bq27421_control_charging(struct bq27421_device_info *di)
{
	int ret;
	unsigned char flags;

	ret = battery_getVBUSConnect();
	/* ret = smb135x_getVBUSConnect(); */
	if (ret <= 0) {
		if (ret) {
			dev_err(di->dev,
				"%s %u: bq27421 vbus connect get error.\n",
				__func__, __LINE__);
			return -1;
		}
		smb135x_setChargeEnable(1);
		return 0;
	}

	ret = smb135x_getChargeStatus();
	if (ret < 0) {
		dev_err(di->dev, "%s %u: bq27421 charge type get error.\n",
			__func__, __LINE__);
		return -2;
	}

	bq27421_i2c_read(di, BQ27421_REG_FLAGS2, &flags, 1);
	if (ret) {
		if(battery_remain >= 95)
			battery_setSaveCurrent(1);
		else
			battery_setSaveCurrent(0);
		if ((flags & 0x3) == 0x2) {
			printk("%s %u: battery_setChargeEnable(0);\n",__func__, __LINE__);
			smb135x_setChargeEnable(0);
			if (!discharge_timer_set && power_on_state) {
				printk("Start timer to discharge\n");
				discharge_timer_set = true;
			} else if (!power_on_state) {
				if (!fullcharge_on_powoff)
					printk("Full charged on power off state\n");
				fullcharge_on_powoff = true;
			}
		}
	} else {
		if(!discharge_status)
			battery_setSaveCurrent(0);
		if (battery_getChargeEnable()){
			if((flags & 0x2) == 0x0) {
				if (power_on_state && (discharge_timer_expired || fullcharge_on_powoff)) {
					if (!execute_auto_charge_control)
						printk("Stop force discharge behave as longlife charging\n");
					execute_auto_charge_control = true;
					fullcharge_on_powoff = false;
					battery_setForceDischarge(0);
					discharge_timer_set = false;
				} else if (!bq27421_check_longlife(di))
					smb135x_setChargeEnable(1);
			} else {
				if (power_on_state && !discharge_timer_set && !discharge_timer_expired) {
					printk("Full charged on power off start discharge timer\n");
					discharge_timer_set = true;
				}
			}
		}

	}

	return 0;
}

static int bq27421_control_longlife(struct bq27421_device_info *di)
{
	int ret;

	ret = battery_getVBUSConnect();
	/* ret = smb135x_getVBUSConnect(); */
	if (ret < 0) {
		dev_err(di->dev, "bq27421 vbus connect get error.\n");
		return -1;
	}
	if ((!bq27421_check_longlife(di) && !execute_auto_charge_control) || !ret ) {
		bq27421_control_charging(di);
		return 0;
	}

	if (battery_remain < 0) {
		dev_err(di->dev, "bq27421 battery capacity get error.\n");
		smb135x_setChargeEnable(1);
		return -1;
	}

	ret = smb135x_getChargeStatus();
	if (ret < 0) {
		dev_err(di->dev, "bq27421 charge type get error.\n");
		smb135x_setChargeEnable(1);
		return -2;
	}

	if (ret) {
		if (battery_remain >= 70) {
			dev_info(di->dev,
				 "bq27421 stop charging(batt=%d%%)\n",
				 battery_remain);
			printk("LL stop charge\n");
			smb135x_setChargeEnable(0);
		}
	} else {
		if (battery_remain <= 60) {
			dev_info(di->dev,
				 "bq27421 restart charging(batt=%d%%)\n",
				 battery_remain);
			printk("LL (re)start charge\n");
			smb135x_setChargeEnable(1);
		}
	}

	return 0;
}

static int bq27421_init_config(struct bq27421_device_info *di)
{
	int cnt;
	unsigned char data;

	smb135x_setChargeEnable(0);

setting_start:
	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
	mutex_unlock(&di->lock);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SET_CFGUPD, di);
		mutex_unlock(&di->lock);
		msleep(1500);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(!(data & BQ27421_FLAG_CFG) && ++cnt < BQ27421_RETRYLIMIT);

	set_necat_configure(di);

	cnt = 0;
	do {
		mutex_lock(&di->lock);
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
		mutex_unlock(&di->lock);
		msleep(3000);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(data & BQ27421_FLAG_CFG  && ++cnt < BQ27421_RETRYLIMIT);

	cnt = 0;
	do {
		msleep(1000);
		bq27421_i2c_read(di, BQ27421_REG_FLAGS, &data, 1);
	} while(!(data & BQ27421_FLAG_INITCOMP) && ++cnt < BQ27421_RETRYLIMIT);
	if (cnt == BQ27421_RETRYLIMIT) {
		dev_err(di->dev, "bq27421 error: INITCOMP is not up\n");
		bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_RESET, di);
		msleep(1000);
		dev_err(di->dev, "bq27421: initialize restart\n");
		goto setting_start;
	}

	mutex_lock(&di->lock);
	bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
	mutex_unlock(&di->lock);

	bq27421_control_longlife(di);

	return 0;
}

static int monitoring(struct bq27421_device_info *di)
{
	int ctl_stat;
	unsigned char flags;
	union power_supply_propval val;
	int ret = 0;
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
	int val_volt = 0;
#endif	/* ENABLE_SOC_ZERO_CHK_VLT */

	mutex_lock(&battery_check_mutex);

	if (!isBatteryConnected(di)) {
		dev_err(di->dev, "bq27421 initialize error. (No battery connected?)\n");
		battery_remain = -1;
		atomic_set(&pm_resume_remain_condition, 1);
		wake_up(&pm_resume_remain_q);
		dev_dbg(di->dev, "@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));
		mutex_unlock(&battery_check_mutex);
		return -1;
	}

	if (isAlreadyConfigured(di)) {
		bq27421_subcmd_read(di, &ctl_stat, BQ27421_SUBCMD_CTNL_STATUS);
		if (!(ctl_stat & BQ27421_FLAG_SS)) {
			battery_remain = -1;
			atomic_set(&pm_resume_remain_condition, 1);
			wake_up(&pm_resume_remain_q);
			dev_dbg(di->dev, "@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));
			mutex_unlock(&battery_check_mutex);
			ISSETTING = 1;
			mutex_lock(&battery_check_mutex);
			smb135x_setChargeEnable(0);

			bq27421_i2c_read(di, BQ27421_REG_FLAGS, &flags, 1);
			while(flags & BQ27421_FLAG_CFG) {
				mutex_lock(&di->lock);
				bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
				mutex_unlock(&di->lock);
				msleep(2000);
				bq27421_i2c_read(di, BQ27421_REG_FLAGS, &flags, 1);
			}

			mutex_lock(&di->lock);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
			mutex_unlock(&di->lock);

			msleep(2000);
//			gpio_set_value(GPIO_CHG_CE, 0);
			bq27421_control_longlife(di);
		} else {
			/* longlife monitoring */
			bq27421_control_longlife(di);
		}
	}
	else {
		battery_remain = -1;
		atomic_set(&pm_resume_remain_condition, 1);
		wake_up(&pm_resume_remain_q);
		dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));
		dev_info(di->dev,
			 "bq27421 %s %u: init_config @@@\n",
			 __func__, __LINE__);
		mutex_unlock(&battery_check_mutex);
		ISSETTING = 1;
		mutex_lock(&battery_check_mutex);
		bq27421_init_config(di);
	}

	ISSETTING = 0;
	ret = bq27421_simple_value(di, BQ27421_REG_SOC, &val);
	if (ret == 0)
	{
#ifdef	ENABLE_SOC_ZERO_CHK_VLT
		if(val.intval == 0 )
		{
			ret = bq27421_read(BQ27421_REG_VOLT, &val_volt, 0, di);
			if (ret == 0)
			{
				if(val_volt <= VOLT_CHECK_VALUE )	/* 3400mv under */
				{
					volt_read_cnt++;
					if(volt_read_cnt < VOLT_CHECK_MAX ) {
						battery_remain = 1;
					} else {
						battery_remain = 0;
					}
					pr_info("battery_remain=%d  Voltage=0x%04x  under cnt=%d\n",battery_remain,val_volt,volt_read_cnt);
				}
				else								/* 3400mv over */
				{
					battery_remain = 1;
					pr_debug("battery_remain=%d  Voltage=0x%04x  under cnt=%d\n",battery_remain,val_volt,volt_read_cnt);
					volt_read_cnt=0;
				}
			} else {
				battery_remain = val.intval;
			}
		} else {
			volt_read_cnt=0;
			battery_remain = val.intval;
		}
#else	/* ENABLE_SOC_ZERO_CHK_VLT */
		battery_remain = val.intval;
#endif	/* ENABLE_SOC_ZERO_CHK_VLT */
		dev_dbg(di->dev,"@@@@ %s battery_remain=%d\n",__FUNCTION__,battery_remain);
	}
	else
		dev_err(di->dev, "bq27421 battery capacity get error.\n");
	atomic_set(&pm_resume_remain_condition, 1);
	wake_up(&pm_resume_remain_q);
	dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));

	/* SW Reset */
	if ((battery_getVBUSConnect() > 0) && /* connect VBUS */
	/*if ((smb135x_getVBUSConnect() > 0) && */ /* connect VBUS */
	    (smb135x_getChargeStatus() == 0)) { /* No Charge */
		if (sw_reset_counter-- <= 0) {
			mutex_lock(&di->lock);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_UNSEALED, di);	/* UNSEAL コマンドは2回発行する必要がある */
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SW_RESET, di);
			bq27421_i2c_txsubcmd(BQ27421_REG_CNTL, BQ27421_SUBCMD_SEALED, di);
			mutex_unlock(&di->lock);
			dev_info(di->dev,
				 "bq27421 %s %u: SW Reset done @@@\n",
				 __func__, __LINE__);
			sw_reset_counter = sw_reset_cycle;
		}
		if (discharge_timer_set) {
			if (dischg_counter-- <= 0) {
				battery_setForceDischarge(1);
				printk("Force Discharge start\n");
				dischg_counter = dischg_start_period;
				discharge_timer_set = false;
				discharge_timer_expired = true;
			}
		} else {
			dischg_counter = dischg_start_period;
		}
	} else {
		sw_reset_counter = sw_reset_cycle;
		dischg_counter = dischg_start_period;
	}
	if (atomic_read(&power_on_charge_control_enable) == 1) {
		if (!power_on_state) {
			printk("Exit power save Enter Power On\n");
			power_on_state = true;
		}
	} else {
		if (power_on_state) {
			printk("Enter power save PowerOff/Hibertante/(Waiting)\n");
			restore_discharge_ctrl_flag();
			power_on_state = false;
		}
	}

	mutex_unlock(&battery_check_mutex);
	return 0;
}

static int bq27421_polling_thread(void *arg)
{
	struct i2c_client *client = arg;
	struct bq27421_device_info *di = i2c_get_clientdata(client);
	int err = 0;
	
	while (!kthread_should_stop()) {
		if (POLLING_ENABLE)
			err = monitoring(di);
#if 0 /* bug#36500 */
		msleep(BQ27421_POLLING_INTERVAL);
#else
		atomic_set(&wait_condition, 0);
		if (wait_event_timeout(wait_q, atomic_read(&wait_condition) == 1,
				       msecs_to_jiffies(BQ27421_POLLING_INTERVAL)) != 0) {
			pr_info("%s %u: wakeup @@@@\n", __func__, __LINE__);
		}
#endif /* bug#36500 */
	}
	return err;
}

static int bq27421_battery_probe(struct i2c_client *client,
				 const struct i2c_device_id *id)
{
	char *name;
	struct bq27421_device_info *di;
	struct bq27421_access_methods *bus;
	uint32_t bootflag = get_msm_boot_flag();
	int num;
	int retval = 0;

	if (GET_BOOT_FLAG_INFO(bootflag, BATT_DETECT)
	    == BOOT_FLAG_BATT_UNDETECT) {
		pr_info("%s bq27421 disable @@@@@@ \n", __func__);
		return -ENODEV;
	}
	init_waitqueue_head(&wait_q);
	init_waitqueue_head(&pm_resume_remain_q);
	atomic_set(&power_on_charge_control_enable, 1);
	atomic_set(&pm_resume_remain_condition, 1);

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C))
		return -ENODEV;

	/* Get new ID for the new battery device */
	mutex_lock(&battery_mutex);
	num = idr_alloc(&battery_id, client, 0, 0, GFP_KERNEL);
	mutex_unlock(&battery_mutex);
	if (num < 0)
		return num;

	name = kasprintf(GFP_KERNEL, "%s-%d", id->name, num);
	if (!name) {
		dev_err(&client->dev, "failed to allocate device name\n");
		retval = -ENOMEM;
		goto batt_failed_1;
	}

	di = kzalloc(sizeof(*di), GFP_KERNEL);
	if (!di) {
		dev_err(&client->dev, "failed to allocate device info data\n");
		retval = -ENOMEM;
		goto batt_failed_2;
	}
	di->id = num;

	bus = kzalloc(sizeof(*bus), GFP_KERNEL);
	if (!bus) {
		dev_err(&client->dev, "failed to allocate access method "
					"data\n");
		retval = -ENOMEM;
		goto batt_failed_3;
	}

	i2c_set_clientdata(client, di);
	di->dev = &client->dev;
	di->bat.name = "bq27421-battery";
	bus->read = &bq27421_read_i2c;
	di->bus = bus;
	di->client = client;

	retval = bq27421_powersupply_init(di);
	if (retval) {
		dev_err(&client->dev, "failed to setup bq27421\n");
		goto batt_failed_4;
	}

#if 1
	retval = sysfs_create_group(&di->bat.dev->kobj, &fs_attr_group);
	if (retval)
		goto batt_failed_4;
#endif

	retval = bq27421_gpio_init(di);
	if (retval < 0)
		goto batt_failed_4;

	if (!IS_BOOTMODE_NO_BATTERY(bootflag)) {
		bq27421_polling = kthread_run(bq27421_polling_thread, client, "bq27421");
		if (IS_ERR(bq27421_polling)) {
			dev_err(di->dev, "failed to create polling thread\n");
			goto batt_failed_4;
		}
	}
	return 0;

batt_failed_4:
	kfree(bus);
batt_failed_3:
	kfree(di);
batt_failed_2:
	kfree(name);
batt_failed_1:
	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, num);
	mutex_unlock(&battery_mutex);

	return retval;
}

static int bq27421_gpio_free(struct bq27421_device_info *di)
{
//	gpio_free(GPIO_BIN_CONT);
//	gpio_free(GPIO_CHG_CE);
#ifdef ENABLE_CHG_PG
	gpio_free(GPIO_CHG_PG);
#endif /* ENABLE_CHG_PG */
	return 0;
}

static int bq27421_battery_remove(struct i2c_client *client)
{
	struct bq27421_device_info *di = i2c_get_clientdata(client);

	bq27421_powersupply_unregister(di);

	kfree(di->bus);

	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, di->id);
	mutex_unlock(&battery_mutex);

	if (bq27421_polling) {
		kthread_stop(bq27421_polling);
		bq27421_polling = NULL;
	}
	bq27421_gpio_free(di);

	kfree(di);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int bq27421_pm_suspend(struct device *dev)
{
	struct bq27421_device_info *di = dev_get_drvdata(dev);

	enable_irq_wake(di->gp_out_wake_irq);
	enable_irq(di->gp_out_wake_irq);
	dev_dbg(di->dev,"@@@@ %s wait_condition=%d\n",__FUNCTION__,atomic_read(&wait_condition));
	
	return 0;
}

static int bq27421_pm_resume(struct device *dev)
{
	struct bq27421_device_info *di = dev_get_drvdata(dev);

	atomic_set(&pm_resume_remain_condition, 0);
	dev_dbg(di->dev,"@@@@ %s line=%d pm_resume_remain_condition=%d\n",__FUNCTION__,__LINE__,atomic_read(&pm_resume_remain_condition));
	atomic_set(&wait_condition, 1);
	wake_up(&wait_q);
	dev_dbg(di->dev,"@@@@ %s wait_condition=%d\n",__FUNCTION__,atomic_read(&wait_condition));
	disable_irq_wake(di->gp_out_wake_irq);
	disable_irq_nosync(di->gp_out_wake_irq);

	return 0;
}
#endif

#ifdef CONFIG_PM
static const struct dev_pm_ops bq27421_dev_pm_ops = {
	SET_SYSTEM_SLEEP_PM_OPS(bq27421_pm_suspend, bq27421_pm_resume)
};
#endif

static const struct i2c_device_id bq27421_id[] = {
	{ "bq27421", 1 },
	{},
};
MODULE_DEVICE_TABLE(i2c, BQ27421_id);

static struct i2c_driver bq27421_battery_driver = {
	.driver		= {
			.name = "bq27421-battery",
#ifdef CONFIG_PM
			.pm = &bq27421_dev_pm_ops,
#endif
	},
	.probe		= bq27421_battery_probe,
	.remove		= bq27421_battery_remove,
	.id_table	= bq27421_id,
};

static int __init bq27421_battery_init(void)
{
	int ret;

	ret = i2c_add_driver(&bq27421_battery_driver);
	if (ret)
		printk(KERN_ERR "Unable to register BQ27421 driver\n");

	return ret;
}
module_init(bq27421_battery_init);

static void __exit bq27421_battery_exit(void)
{
	i2c_del_driver(&bq27421_battery_driver);
}
module_exit(bq27421_battery_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("NEC AccessTechinica, Ltd.");
MODULE_DESCRIPTION("BQ27421 battery monitor driver");
