#!/bin/bash
#global variables:
bl2_start_dec=0
bl2_start_hex=0x0
rf_start_dec=0
rf_start_hex=0x0
fip_start_dec=0
fip_start_hex=0x0
kernel_start_dec=0
kernel_start_hex=0x0
rootfs_start_dec=0
rootfs_start_hex=0x0
rootfs_image=""

usage() {
	printf 'Usage:\n'\
'  ./mk_image -p <platform>\n'\
'             -d <flash device type>\n'\
'             -b <BL2 image>\n'\
'             -r <RF image>\n'\
'             -f <FIP image>\n'\
'             -k <kernel image>\n'\
'             -g <GPT table>\n'\
'             -h <usage menu>\n'\
'             -o <single image name>\n'\
'  example:\n'\
'    ./mk_image.sh -p mt7986a -d emmc \\\n'\
'                  -g GPT_EMMC-iap-20220125 \\\n'\
'                  -f fip-iap-emmc-20220125.bin \\\n'\
'                  -k OF_openwrt-mediatek-mt7986-mt7986a-ax6000-emmc-rfb-squashfs-sysupgrade-20220125030158.bin\n'\
'    ./mk_image.sh -p mt7986a -d spim-nand \\\n'\
'                  -b bl2-iap-snand-20220114.img \\\n'\
'                  -f fip-snand-20220114.bin \\\n'\
'                  -k OF_openwrt-mediatek-mt7986-mt7986a-ax6000-spim-nand-rfb-squashfs-factory-20220125030158.bin \\\n'
	exit
}

load_partition() {
# NAND flash layout:
#       0x0~0x100000: BL2,       1024K
#  0x100000~0x180000: Uboot env,  512K
#  0x180000~0x380000: RF,        2048K
#  0x380000~0x580000: FIP,       2048K
#  0x580000~        : firmware
	if [ $flash_type == "spim-nand" ]
	then
		bl2_start_hex=0x0
		rf_start_hex=0x180000
		fip_start_hex=0x380000
		kernel_start_hex=0x580000
# NOR flash layout:
#      0x0~0x40000 : BL2,        256K
#  0x40000~0x50000 : Uboot env,   64K
#  0x50000~0x100000: RF,         704K
# 0x100000~0x180000: FIP,        512K
# 0x180000~        : firmware
	elif [ $flash_type == "spim-nor" ]
	then
		bl2_start_hex=0x0
		rf_start_hex=0x50000
		fip_start_hex=0x100000
		kernel_start_hex=0x180000
# eMMC layout:
#       0x0~0x4400    : GPT
#  0x400000~0x480000  : u-boot-env
#  0x480000~0x680000  : rf
#  0x680000~0x880000  : fip
#  0x880000~0x2880000 : kernel
# 0x2880000~0x12880000: rootfs
	elif [ $flash_type == "emmc" ]
	then
		rf_start_hex=0x480000
		fip_start_hex=0x680000
		kernel_start_hex=0x880000
		rootfs_start_hex=0x2880000
# SD-card layout:
#       0x0~0x4400    : GPT
#    0x4400~0x400000  : bl2
#  0x400000~0x480000  : u-boot-env
#  0x480000~0x680000  : rf
#  0x680000~0x880000  : fip
#  0x880000~0x2880000 : kernel
# 0x2880000~0x12880000: rootfs
	elif [ $flash_type == "sd" ]
	then
		bl2_start_hex=0x4400
		rf_start_hex=0x480000
		fip_start_hex=0x680000
		kernel_start_hex=0x880000
		rootfs_start_hex=0x2880000
	fi
	bl2_start_dec=`printf '%d' $bl2_start_hex`
	rf_start_dec=`printf '%d' $rf_start_hex`
	fip_start_dec=`printf '%d' $fip_start_hex`
	kernel_start_dec=`printf '%d' $kernel_start_hex`
	rootfs_start_dec=`printf '%d' $rootfs_start_hex`
}

prepare_image() {
	# Pad empty bytes to single image first
	if [[ -z $kernel_image ]]
	then
		if [[ $flash_type == "emmc" ]] || [[ $flash_type == "sd" ]]
		then
			dd if=/dev/zero ibs=$fip_start_dec count=1 > $single_image
		else
			dd if=/dev/zero ibs=$fip_start_dec count=1 | tr "\000" "\377" > $single_image
		fi
	else
		if [[ $flash_type == "emmc" ]] || [[ $flash_type == "sd" ]]
		then
			dd if=/dev/zero ibs=$kernel_start_dec count=1 > $single_image
		else
			dd if=/dev/zero ibs=$kernel_start_dec count=1 | tr "\000" "\377" > $single_image
		fi
	fi
}

extract_sdmmc_kernel() {
	output=`tar -xvf $kernel_image | awk {'print $1'}`
	IFS=$'\n' read -d "\034" -r -a output_arr <<< "$output"

	#For debugging
	#echo "There are ${#output_arr[*]}" next lines in the output.

	for filename in "${output_arr[@]}";
	do
		if [[ "$filename" == *"kernel" ]]
		then
			kernel_image=$filename
		elif [[ "$filename" == *"root" ]]
		then
			rootfs_image=$filename
		fi
	done
}

start_wrapping() {
	printf "[Start wrapping %s single image......]\n" $flash_type

	if [[ $flash_type != "emmc" ]]
	then
		printf "[wrapping BL2 image......]\n"
		dd if=$bl2_image of=$single_image bs=512 seek=$(( ($bl2_start_dec/512) )) conv=notrunc
	fi

	if [[ $flash_type == "emmc" ]]
	then
		printf "[wrapping GPT......]\n"
		dd if=$gpt of=$single_image bs=512 seek=0 conv=notrunc
	fi

	if [[ -n $rf_image ]]
	then
		printf "[wrapping RF image......]\n"
		dd if=$rf_image of=$single_image bs=512 seek=$(( ($rf_start_dec/512) )) conv=notrunc
	fi

	printf "[wrapping FIP image......]\n"
	dd if=$fip_image of=$single_image bs=512 seek=$(( ($fip_start_dec/512) )) conv=notrunc

	if [[ -n $kernel_image ]]
	then
		printf "[wrapping kernel image......]\n"
		if [[ $flash_type == "emmc" ]] || [[ $flash_type == "sd" ]]
		then
			extract_sdmmc_kernel
			dd if=$kernel_image of=$single_image bs=512 seek=$(( ($kernel_start_dec/512) )) conv=notrunc
			printf "[wrapping rootfs image......]\n"
			dd if=$rootfs_image of=$single_image bs=512 seek=$(( ($rootfs_start_dec/512) )) conv=notrunc
		else
			dd if=$kernel_image of=$single_image bs=512 seek=$(( ($kernel_start_dec/512) )) conv=notrunc
		fi
	fi
}

if [ $# -lt 1 ]
then
	usage
	exit 1
fi

while [ "$1" != "" ]; do
	case $1 in
	-h )
		usage
		;;
	-p )
		shift
		platform=$1
		;;
	-d )
		shift
		flash_type=$1
		;;
	-b )
		shift
		bl2_image=$1
		;;
	-f )
		shift
		fip_image=$1
		;;
	-k )
		shift
		kernel_image=$1
		;;
	-g )
		shift
		gpt=$1
		;;
	-o )
		shift
		single_image=$1
		;;
	-r )
		shift
		rf_image=$1
		;;
	esac
	shift
done


######## Check if variables are valid ########
check_ok=1
if ! [[ $platform =~ ^(mt7981abd|mt7981c|mt7986a|mt7986b)$ ]]; then
	printf "Platform must be in mt7981abd|mt7981c|mt7986a|mt7986b\n"
	usage
	check_ok=0
fi
if ! [[ $flash_type =~ ^(spim-nand|spim-nor|emmc|sd)$ ]]; then
	printf "Flash type must be in spim-nand|spim-nor|emmc|sd\n"
	usage
	check_ok=0
fi
if [[ $flash_type =~ ^(emmc|sd)$ ]] && [[ -z $gpt ]]; then
	printf "GPT table must be provided if flash type is emmc or sd\n"
	usage
	check_ok=0
fi

if [[ -n $gpt ]] && ! [[ -f $gpt ]]; then
	printf "GPT table provided doesn't exist.\n"
	check_ok=0
fi
if [[ -n $rf_image ]] && ! [[ -f $rf_image ]]; then
	printf "RF image provided doesn't exist.\n"
	check_ok=0
fi
if [[ -n $kernel_image ]] && ! [[ -f $kernel_image ]]; then
	printf "Kernel image provided doesn't exist.\n"
	check_ok=0
fi

##############################################

if [[ -n $bl2_image ]] && ! [[ -f $bl2_image ]]
then
	printf "BL2 image provided doesn't exist.\n"
elif [[ -z $bl2_image ]]
then
	bl2_image="bl2.img"
	printf "Use default BL2 image name\n"
fi

if [[ -n $fip_image ]] && ! [[ -f $fip_image ]]
then
	printf "FIP image provided doesn't exist.\n"
elif [[ -z $fip_image ]]
then
	fip_image="fip.bin"
	printf "Use default FIP image name\n"
fi

if [[ -z $single_image ]]
then
	single_image="$platform-$flash_type-$(date +%Y%m%d)-single-image"
	printf "Use default single image name\n"
fi

if [[ $check_ok == 1 ]]; then
	printf "./mk_image -p %s -d %s\n" $platform $flash_type
	load_partition
	prepare_image
	start_wrapping
fi
