/* vi: set sw=4 ts=4: */
/* dirutil.c ... directory utilities.
 *               C. Scott Ananian <cananian@alumni.princeton.edu>
 *
 * $Id: dirutil.c,v 1.1.1.1 2005/05/19 10:53:07 r01122 Exp $
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#ifdef MEMWATCH
#include "memwatch.h"
#else
#include <stdlib.h>
#endif
#include "dirutil.h"

/* Returned malloc'ed string representing basename */
char *basenamex(char *pathname)
{
	char *dup, *ptr;

	if (pathname == NULL)
		return NULL;

	dup = strdup(pathname);
	if (dup == NULL)
		return NULL;

	ptr = strrchr(stripslash(dup), '/');
	if (ptr==NULL) return dup;
	ptr = strdup(ptr+1);
	free(dup);
	return ptr;
}

/* Return malloc'ed string representing directory name (no trailing slash) */
char *dirname(const char *pathname)
{
	char *dup, *ptr;

	if (pathname == NULL)
		return NULL;

	dup = strdup(pathname);
	if (dup == NULL)
		return NULL;

	ptr = strrchr(stripslash(dup), '/');
	if (ptr==NULL) { free(dup); return strdup("."); }
	if (ptr==dup && dup[0]=='/') ptr++;
	*ptr = '\0';
	return dup;
}
/* In-place modify a string to remove trailing slashes.  Returns arg.
 * stripslash("/") returns "/";
 */
char *stripslash(char *pathname)
{
	int len;

	if (pathname == NULL)
		return NULL;

	len = strlen(pathname);
	while (len > 1 && pathname[len-1]=='/')
		pathname[--len]='\0';
	return pathname;
}
/* ensure dirname exists, creating it if necessary. */
int make_valid_path(char *dir, mode_t mode)
{
	struct stat st;
	char *tmp=NULL, *path;
	int retval;

	if (dir == NULL)
		return 0;

	path = stripslash(strdup(dir));
	if (path == NULL)
		return 0;

	if (stat(path, &st) == 0)
	{	/* file exists */
		if (S_ISDIR(st.st_mode)) { retval=1; goto end; }
		else { retval=0; goto end; } /* not a directory.  Oops. */
	}
	/* Directory doesn't exist.  Let's make it. */
	/*   Make parent first. */
	if (!make_valid_path(tmp = dirname(path), mode)) { retval=0; goto end; }
	/*   Now make this 'un. */
	if (mkdir(path, mode) < 0) { retval=0; goto end; }
	/* Success. */
	retval=1;

end:
	if (tmp!=NULL) free(tmp);
	if (path!=NULL) free(path);
	return retval;
}
