/* vi: set sw=4 ts=4: */
/*
 *	strobj.c
 *
 *	String object. A helper to manipulate string text.
 *	Created by David Hsieh <david_hsieh@alphanetworks.com>
 *	Copyright (C) 2007-2009 by Alpha Networks, Inc.
 *
 *	This file is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Lesser General Public
 *	License as published by the Free Software Foundation; either'
 *	version 2.1 of the License, or (at your option) any later version.
 *
 *	The GNU C Library is distributed in the hope that it will be useful,'
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Lesser General Public License for more details.
 *
 *	You should have received a copy of the GNU Lesser General Public
 *	License along with the GNU C Library; if not, write to the Free
 *	Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 *	02111-1307 USA.
 */

#include <stdio.h>
#ifdef MEMWATCH
#include "memwatch.h"
#else
#include <stdlib.h>
#endif
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <ctype.h>

#include <dtrace.h>
#include <dlist.h>
#include <mem_helper.h>
#include <strobj.h>
#include <limits.h>
#include <errno.h>

#define STROBJ_STEP_SIZE	32

#define __round_mask(x, y) ((__typeof__(x))((y)-1))
#define round_up(x, y) ((((x)-1) | __round_mask(x, y))+1)

#if 0
struct strobj
{
	struct dlist_head list;
	unsigned int flags;
	size_t total;	/* allocated size, not including the terminated NULL. */
	size_t size;	/* used size, not including the terminated NULL. */
	char * buff;	/* pointer to the buffer */
};

struct strobj_list
{
	struct dlist_head head;
	struct strobj * curr;
};
#endif

/****************************************************************************/

solist_t solist_new(void)
{
	struct strobj_list * sol = xmalloc(sizeof(struct strobj_list));
	if (sol)
	{
		INIT_DLIST_HEAD(&sol->head);
		sol->curr = NULL;
	}
	return sol;
}

void solist_free(solist_t sol)
{
	struct dlist_head * entry;
	struct strobj * so = NULL;

	if (!sol)
		return;

	while (!dlist_empty(&sol->head))
	{
		entry = sol->head.next;
		dlist_del(entry);
		so = dlist_entry(entry, struct strobj, list);
		sobj_del(so);
	}
	sol->curr = NULL;
}

void solist_del(solist_t list)
{
	if (!list)
		return;

	solist_free(list);
	xfree(list);
}

strobj_t solist_get_next(solist_t sol)
{
	struct dlist_head * entry;
	struct strobj * so = NULL;

	if (!sol)
		return NULL;

	if (sol->curr)
		entry = sol->curr->list.next;
	else
		entry = sol->head.next;

	if (entry != &sol->head)
		so = dlist_entry(entry, struct strobj, list);
	sol->curr = so;
	return so;
}

strobj_t solist_get_prev(solist_t sol)
{
	struct dlist_head * entry;
	struct strobj * so = NULL;

	if (!sol)
		return NULL;

	if (sol->curr)
		entry = sol->curr->list.prev;
	else
		entry = sol->head.prev;

	if (entry != &sol->head)
		so = dlist_entry(entry, struct strobj, list);
	sol->curr = so;
	return so;
}

void solist_get_reset(solist_t sol)
{
	if (sol)
		sol->curr = NULL;
}

void solist_remove(solist_t list, strobj_t obj)
{
	if (!list || !obj)
		return;

	if (list->curr == obj)
		list->curr = NULL;
	dlist_del(&obj->list);
}

void solist_add(solist_t list, strobj_t obj)
{
	if (!list || !obj)
		return;

	dlist_add_tail(&obj->list, &list->head);
}

int solist_get_count(solist_t sol)
{
	struct dlist_head * entry;
	int count = 0;

	if (!sol)
		return count;

	entry = sol->head.next;
	while (entry != &sol->head)
	{
		count++;
		entry = entry->next;
	}
	return count;
}

/****************************************************************************/

/* Create a string object. */
strobj_t sobj_new(void)
{
	struct strobj * sobj = xmalloc(sizeof(struct strobj));

	if (!sobj)
		return NULL;

	memset(sobj, 0, sizeof(struct strobj));
	INIT_DLIST_HEAD(&sobj->list);

	return sobj;
}

/* realloc the space of the object at least size */
static int sobj_realloc_space(struct strobj *sobj, size_t size)
{
	size_t msize;
	void *tmp;

	if (!sobj)
		return -1;

	/* size enough no need to change */
	if (sobj->total >= size)
		return 0;

	msize = round_up(size + 1, STROBJ_STEP_SIZE);
	tmp = xrealloc(sobj->buff, msize);
	if (!tmp)
		return -1;
	sobj->buff = tmp;

	/* init the buffer at first time */
	if (sobj->total == 0)
		sobj->buff[0] = '\0';

	/* reserved for null end */
	sobj->total = msize - 1;

	return 0;
}

#if defined(__ECOS__)
//+++ add by siyou for performance tunning
strobj_t sobj_new_wsize(int size)
{
	struct strobj *so;

	so = (struct strobj *) sobj_new();
	if (so && size > 0)
	{
		if (sobj_realloc_space(so, size) == -1) {
			sobj_del(so);
			return NULL;
		}
	}

	return so;
}
#endif

/* Destroy the string object. */
void sobj_del(strobj_t sobj)
{
	if (!sobj)
		return;

	if (sobj->buff)
		xfree(sobj->buff);
	xfree(sobj);
}

/* Increase the space of the object. */
int sobj_inc_space(strobj_t sobj)
{
	size_t size;

	dassert(sobj);
	if (!sobj)
		return -1;

	size = round_up(sobj->total, STROBJ_STEP_SIZE) + STROBJ_STEP_SIZE;
	return sobj_realloc_space(sobj, size - 1);
}

/* add a character at the tail of the object */
int sobj_add_char(strobj_t sobj, int c)
{
	dassert(sobj);
	if (!sobj)
		return -1;
	if ((sobj->total - sobj->size) < 1)
		if (sobj_inc_space(sobj) < 0)
			return -1;
#if defined(__ECOS__)
	if(malloc_check_address(&(sobj->buff[sobj->size]))<0)
	{
		if(malloc_check_address(sobj->buff)==0)free(sobj->buff);
		return -1;
	} // kingkong add this to avoid some exception problem.
#endif
	sobj->buff[sobj->size++] = (char)c;
	sobj->buff[sobj->size] = '\0';
	return 0;
}

/* append a string at the tail of the object */
int sobj_add_string(strobj_t sobj, const char * string)
{
	size_t i;

	dassert(sobj);
	if (!sobj)
		return -1;
	if (!string)
		return 0;
	i = strlen(string);
	if (i == 0)
		return 0;

	/*This boundary check is patched from DIR-600NB to avoid some exception problem.
	  However, it will cause xmldbc be in disorder. */
#if defined(__ECOS__)
	if (sobj->size < 0) {
		//diag_printf("-----sobj_add_string----------sobj->size=%d---------------\n",sobj->size);
		return -1;}
#endif

	if ((sobj->total - sobj->size) < i)
		if (sobj_realloc_space(sobj, sobj->size + i) < 0)
			return -1;

#if defined(__ECOS__)
	if(malloc_check_address(&(sobj->buff[sobj->size]))<0)
	{
		if(malloc_check_address(sobj->buff)==0)free(sobj->buff);
		return -1;
	} // kingkong add this to avoid some exception problem.
#endif

	strcpy(&(sobj->buff[sobj->size]), string);
	sobj->size += i;
	return 0;
}

/* format string */
int sobj_format(strobj_t obj, const char * format, ...)
{
	va_list marker;
	char buff[1024];
	int len;

	if (!obj)
		return -1;
	sobj_free(obj);

	if (!format)
		return 0;
	
	va_start(marker, format);
 	len = vsnprintf(buff, sizeof(buff), format, marker);
	va_end(marker);

	/* error or overflow */
	if (len < 0 || (size_t)len >= sizeof(buff))
		return -1;

	return sobj_add_string(obj, buff);
}

/* append format string */
int sobj_add_format(strobj_t obj, const char * format, ...)
{
	va_list marker;
	char buff[1024];
	int len;

	if (!obj)
		return -1;
	if (!format)
		return 0;

	va_start(marker, format);
	len = vsnprintf(buff, sizeof(buff), format, marker);
	va_end(marker);

	/* error or overflow */
	if (len < 0 || (size_t)len >= sizeof(buff))
		return -1;

	return sobj_add_string(obj, buff);
}

/* duplicate a string from the object. */
char * sobj_strdup(strobj_t sobj)
{
	if (!sobj)
		return NULL;

	if (sobj->buff)
		return xstrdup(sobj->buff);
	return xstrdup("");
}

/* get the string of the object */
const char * sobj_get_string(strobj_t sobj)
{
	if (!sobj)
		return NULL;
	return sobj->buff ? sobj->buff : "";
}

/* get the flags of string object */
unsigned int sobj_get_flags(strobj_t sobj)
{
	if (!sobj)
		return 0;

	return sobj->flags;
}

/* remove the white character of the string. */
const char * sobj_eat_all_white(strobj_t sobj)
{
	const char * ptr;
	char *tmp;

	if (!sobj || !sobj->buff)
		return NULL;

	tmp = sobj->buff;
	ptr = sobj_eatwhite(sobj_reatwhite(sobj->buff));

	while (ptr && *ptr) {
		if (sobj_iswhite(*ptr)) {
			/* save only one white space */
			if (tmp != ptr)
				*tmp = ' ';
			ptr = sobj_eatwhite(ptr);
		} else {
			if (tmp != ptr)
				*tmp = *ptr;
			ptr++;
		}
		tmp++;
	}
	*tmp = '\0';
	sobj->size = strlen(sobj->buff);

	return sobj->buff;
}

/* remove the indent of the string. */
const char * sobj_eat_indent(strobj_t sobj)
{
	const char * ptr;

	if (!sobj || !sobj->buff)
		return NULL;

	ptr = sobj_eatindent(sobj_reatindent(sobj->buff));

	if (sobj->buff != ptr) {
		sobj->size = strlen(ptr);
		/* copy with null end */
		memmove(sobj->buff, ptr, sobj->size + 1);
	} else
		sobj->size = strlen(sobj->buff);

	return sobj->buff;
}

int sobj_free(strobj_t sobj)
{
	if (!sobj)
		return -1;

	if (sobj->buff)
		sobj->buff[0] = '\0';
	sobj->size = 0;
	sobj->flags = 0;

	return 0;
}

int sobj_strcpy(strobj_t sobj, const char * string)
{
	if (!sobj)
		return -1;

	sobj->size = 0;
	sobj->flags = 0;
	if (sobj->buff)
		sobj->buff[0] = '\0';
	return sobj_add_string(sobj, string);
}

int sobj_move(strobj_t tobj, strobj_t fobj)
{
	if (!tobj || !fobj)
		return -1;

	if (tobj->buff)
		xfree(tobj->buff);
	tobj->buff = fobj->buff;
	tobj->total = fobj->total;
	tobj->size = fobj->size;
	tobj->flags = fobj->flags;

	fobj->buff = NULL;
	fobj->total = 0;
	fobj->size = 0;
	fobj->flags = 0;
	return 0;
}

char sobj_get_char(strobj_t sobj, size_t i)
{
	if (!sobj || !sobj->buff)
		return '\0';
	if (i > sobj->size)
		return '\0';
	return sobj->buff[i];
}

size_t sobj_get_length(strobj_t sobj)
{
	if (!sobj)
		return 0;
	return sobj->size;
}

int sobj_empty(strobj_t obj)
{
	return (sobj_get_length(obj)==0) ? 1 : 0;
}

strobj_t sobj_split(strobj_t sobj, size_t at)
{
	struct strobj * newobj;

	if (!sobj)
		return NULL;
	if(!(newobj = sobj_new()))
		return NULL;
	if (at >= sobj->size)
		return newobj;
	if (sobj_add_string(newobj, &(sobj->buff[at])) < 0)
	{
		sobj_del(newobj);
		return NULL;
	}
	sobj->buff[at] = '\0';
	sobj->size = strlen(sobj->buff);
	return newobj;
}

int sobj_strchr(strobj_t sobj, int c)
{
	char * ptr;

	if (!sobj || !sobj->buff)
		return -1;

	ptr = strchr(sobj->buff, c);
	if (ptr)
		return (int)(ptr - sobj->buff);
	return -1;
}

int sobj_strrchr(strobj_t sobj, int c)
{
	char * ptr;

	if (!sobj || !sobj->buff)
		return -1;

	ptr = strrchr(sobj->buff, c);
	if (ptr)
		return (int)(ptr - sobj->buff);
	return -1;
}

int sobj_strstr(strobj_t sobj, const char * str)
{
	char * ptr;

	if (!sobj || !sobj->buff)
		return -1;

	ptr = strstr(sobj->buff, str);
	if (ptr)
		return (int)(ptr - sobj->buff);
	return -1;
}

int sobj_strcmp(strobj_t sobj, const char *s)
{
	char * ptr;

	if (!sobj || !s)
		return -1;
	ptr = sobj->buff ? sobj->buff : "";
	return strcmp(ptr, s);
}

int sobj_strncmp(strobj_t sobj, const char *s, size_t n)
{
	char * ptr;

	if (!sobj || !s)
		return -1;
	ptr = sobj->buff ? sobj->buff : "";
	return strncmp(ptr, s, n);
}

int sobj_strcasecmp(strobj_t sobj, const char * s)
{
	char * ptr;

	if (!sobj || !s)
		return -1;
	ptr = sobj->buff ? sobj->buff : "";
	return strcasecmp(ptr, s);
}

int sobj_strncasecmp(strobj_t sobj, const char * s, size_t n)
{
	char * ptr;

	if (!sobj || !s)
		return -1;
	ptr = sobj->buff ? sobj->buff : "";
	return strncasecmp(ptr, s, n);
}

int sobj_remove_char(strobj_t sobj, size_t at)
{

	if (!sobj || !sobj->buff)
		return -1;
	if (at >= sobj->size)
		return -1;
	while (at < sobj->size)
	{
		sobj->buff[at] = sobj->buff[at+1];
		at++;
	}
	sobj->size--;
	return 0;
}

int sobj_remove_tail(strobj_t sobj)
{

	if (!sobj || !sobj->buff)
		return -1;

	if (sobj->size > 0) {
		sobj->size--;
		sobj->buff[sobj->size] = '\0';
	}
	return 0;
}

#if 1
/* str list must be orderd for binary search */
static int find_pattern(char c, const char * str)
{
	int low, high, mid, n;

	if (!str)
		return 0;
	if ((n = strlen(str)) <= 0)
		return 0;

        low = 0;
        high = n - 1;
        while (low <= high) {
                mid = (low + high) / 2;
                if (c < str[mid])
                        high = mid - 1;
                else if (c > str[mid])
                        low = mid + 1;
                else    /* found match */
                        return str[mid];
        }

	return 0;
}
#else
static int find_pattern(char c, const char * str)
{
	while (*str && (*str!=c))
		str++;
	return (int)(*str);
}
#endif

int sobj_xstream_eatwhite(xstream_t fd)
{
	int c;
	while ((c=xs_getc(fd)) != EOF)
	{
		if (!sobj_iswhite(c))
		{
			xs_ungetc(c, fd);
			break;
		}
	}
	return c;
}

/* This function will read a token from xstream.
 * The token is ended by white space, 'end' character or quotation mark.
 * The return vaule is 0 when error occur.
 * Otherwise the return value is the end of this token.
 * end list must be orderd for binary search */
int sobj_xstream_read(xstream_t fd, strobj_t sobj, const char * end, int * quot)
{
	return sobj_xstream_read_esc(fd, sobj, end, quot, '\\');
}

/* end list must be orderd for binary search */
int sobj_xstream_read_esc(xstream_t fd, strobj_t sobj, const char * end, int * quot, char esc)
{
	int err=0, escape=0, q=-1, c;

	while ((c=xs_getc(fd)) != EOF)
	{
		/* We don't want the NULL character. */
		if (c == 0) { err++; break; }

		/* Check the first character to determine
		 * if the string is in quotation marks. */
		if (q < 0)
		{
			if (c=='\"' || c=='\'')
			{
				/* Now we got the quotation mark,
				 * save it and read the next character. */
				q = c;
				continue;
			}
			/* This string is not beginning with quot,
			 * it is not inside the quot. marks.  */
			q = 0;
		}

		if (q)
		{
			/* Read string inside the quotation mark. */
			/* Handle escape character. */
			if (escape)
			{
				switch (c)
				{
				case 'n': c = '\n'; break;
				case 'r': c = '\r'; break;
				case 't': c = '\t'; break;
				}
				if (sobj_add_char(sobj, c) < 0) { err++; break; }
				escape = 0;
			}
			/* got escape character ? */
			else if (esc && c==esc) { escape = 1; }
			/* Reach the end ? */
			else if (c==q) { sobj->flags |= SO_FLAG_QUOT_STRING; break; }
			/* normal character, save it. */
			else if (sobj_add_char(sobj, c) < 0) { err++; break; }
		}
		else
		{
			/* Read string without quotation marks */
			if (c=='\"' || c=='\'')
			{
				/* Reach the end, restore the quotation mark. */
				xs_ungetc(c, fd);
				sobj->flags &= ~SO_FLAG_QUOT_STRING;
				break;
			}
			else if (sobj_iswhite(c) || (end && find_pattern(c, end)))
			{
				/* Reach the end */
				sobj->flags &= ~SO_FLAG_QUOT_STRING;
				break;
			}
			/* Normal character, save it. */
			else if (sobj_add_char(sobj, c) < 0) { err++; break; }
		}
	}
	if (quot) *quot = q;
	if (err) return 0;
	return (int)c;
}

/* delimiter list must be orderd for binary search and include end */
int sobj_xstream_read_tokens(xstream_t fd, solist_t list, char end, const char * delimiter)
{
	int quot, c = 0, err = 0;
	struct strobj * token = NULL;

	//d_dbg("%s: end [%c], delimiter [%s]\n",__func__, end, delimiter);

	do
	{
		do
		{
			/* allocate string object for token */
			token = sobj_new();
			if (!token) { err++; break; }
			/* read token */
			sobj_xstream_eatwhite(fd);
			c = sobj_xstream_read(fd, token, delimiter, &quot);
			if (c == 0) { err++; break; }

			/* add the token to the list. */
			if (sobj_get_length(token) == 0) sobj_del(token);
			else dlist_add_tail(&token->list, &list->head);
			token = NULL;

			if (c == (int)end) break;
			else if (delimiter && find_pattern(c, delimiter))
			{
				token = sobj_new();
				if (!token) { err++; break; }
				if (sobj_add_char(token, c) < 0) { err++; break; }
				dlist_add_tail(&token->list, &list->head);
				token = NULL;
			}
		} while (!err && c!=EOF);
	} while (0);

	if (token) sobj_del(token);
	if (err)
	{
		solist_free(list);
		return 0;
	}
	return (int)c;
}

/*************************************************************************/

/* sc list must be orderd for binary search */
static int escape_character(char escape, const char * sc, const char * from, strobj_t to)
{
	while (*from)
	{
		if (find_pattern(*from, sc))
			if (sobj_add_char(to, escape) == -1)
				return -1;
		if (sobj_add_char(to, *from++) == -1)
			return -1;
	}
	return 0;
}

#if 0
static void unescape_character(char escape, const char * from, strobj_t to)
{
	int escape_mode = 0;
	while (*from)
	{
		if (escape_mode) { sobj_add_char(to, *from); escape_mode = 0; }
		else if (*from == escape) { escape_mode++; }
		else { sobj_add_char(to, *from); }
		from++;
	}
}
#endif

int sobj_escape_javascript(const char * from, strobj_t to)
{
	const char patt[] = {'"', '\'', '\\', '\0'};

	if (!from || !to)
		return -1;

	return escape_character('\\', patt, from, to);
}

/* The converted string should be in "". */
int sobj_escape_shellscript(const char * from, strobj_t to)
{
	const char patt[] = {'"', '$', '\\', '`', '\0'};

	if (!from || !to)
		return -1;

	return escape_character('\\', patt, from, to);
}

/* HTML special characters
 * '&' -> '&amp;'	- ampersand,
 * '<' -> '&lt;'	- less then,
 * '>' -> '&gt;'	- greater then,
 * ' ' -> '&nbsp;'	- non-breaking space,
 * ''' -> '&apos;'	- apostrophe,
 * '"' -> '&quot;'	- quotation mark. */
int sobj_escape_html_sc(const char * from, strobj_t to)
{
	char *espstr;

	if (!from || !to)
		return -1;

	while (*from)
	{
		espstr = NULL;
		switch (*from)
		{
		case '&':	espstr = "&amp;";	break;
		case '<':	espstr = "&lt;";	break;
		case '>':	espstr = "&gt;";	break;
		/* Don't convert the space, white space wrapping will failed
		 * if we convert all the white space to &nbsp;
		 * David Hsieh*/
		//case ' ':	espstr = "&nbsp;";	break;
		case '"':	espstr = "&quot;";	break;
		//case '\'':	espstr = "&apos;";	break;
		}

		if (espstr) {
			if (sobj_add_string(to, espstr) == -1)
				return -1;
		} else {
			if (sobj_add_char(to, *from) == -1)
				return -1;
		}
		from++;
	}
	return 0;
}

int sobj_unescape_html_sc(const char * from, strobj_t to)
{
	char tmp;

	if (!from || !to)
		return -1;

	while ((tmp = *from)) {
		if (*from != '&') {
			from++;
		} else if (strncmp(from, "&amp;", 5)==0) {
			tmp = '&';
			from+=5;
		} else if (strncmp(from, "&lt;", 4)==0)	{
			tmp = '<';
			from+=4;
		} else if (strncmp(from, "&gt;", 4)==0) {
			tmp = '>';
			from+=4;
		} else if (strncmp(from, "&nbsp;", 6)==0) {
			tmp = ' ';
			from+=6;
		} else if (strncmp(from, "&quot;", 6)==0) {
			tmp = '"';
			from+=6;
		} else if (strncmp(from, "&apos;", 6)==0) {
			tmp = '\'';
			from+= 6;
		} else
			from++;

		if (sobj_add_char(to, tmp) == -1)
			return -1;
	}
	return 0;
}

/* XML special characters
 * '&' -> '&amp;'	- ampersand,
 * '<' -> '&lt;'	- less then,
 * '>' -> '&gt;'	- greater then,
 * ''' -> '&apos;'	- apostrophe,
 * '"' -> '&quot;'	- quotation mark. */
int sobj_escape_xml_sc(const char * from, strobj_t to)
{
	char *espstr;

	if (!from || !to)
		return -1;

	while (*from) {
		espstr = NULL;
		switch (*from) {
		case '&':
			espstr = "&amp;";
			break;
		case '<':
			espstr = "&lt;";
			break;
		case '>':
			espstr = "&gt;";
			break;
		case '"':
			espstr = "&quot;";
			break;
#if !defined(__ECOS__)
		/*
		//IE does not work by jef
		case '\'':
			espstr = "&apos;";
			break;
		*/
		case '\'':
			espstr = "&#39;";
			break;
#endif
		}

		if (espstr) {
			if (sobj_add_string(to, espstr) == -1)
				return -1;
		} else {
			if (sobj_add_char(to, *from) == -1)
				return -1;
		}
		from++;
	}
	return 0;
}

int sobj_unescape_xml_sc(const char * from, strobj_t to)
{
	char tmp;

	if (!from || !to)
		return -1;

	while ((tmp = *from))
	{
		if (*from != '&') from++;
		else if	(strncmp(from, "&amp;", 5)==0)	{ tmp = '&'; from+=5; }
		else if	(strncmp(from, "&lt;", 4)==0)	{ tmp = '<'; from+=4; }
		else if (strncmp(from, "&gt;", 4)==0)	{ tmp = '>'; from+=4; }
		else if	(strncmp(from, "&quot;", 6)==0)	{ tmp = '"'; from+=6; }
#if !defined(__ECOS__)
		else if	(strncmp(from, "&apos;", 6)==0)	{ tmp = '\''; from+= 6; }
		else if (strncmp(from, "&#39;", 5)==0)	{ tmp = '\''; from+= 5; }
#endif
		else from++;

		if (sobj_add_char(to, tmp) == -1)
			return -1;
	}
	return 0;
}

static int ctoi(char c)
{
	if (c>='0' && c<='9') return (int)(c-'0');
	if (c>='a' && c<='f') return (int)(c-'a'+10);
	if (c>='A' && c<='F') return (int)(c-'A'+10);
	return 0;
}

strobj_t sobj_unescape_uri(strobj_t sobj)
{
	char *ptr, *tmp, *end;

	dassert(sobj);
	if (!sobj)
		return NULL;

	ptr = tmp = sobj->buff;
	end = ptr + sobj->size;

	while (tmp < end) {
		if (*tmp == '%' && (tmp + 2 < end) &&
				isxdigit(*(tmp+1)) && isxdigit(*(tmp+2))) {
			*ptr = ctoi(*(tmp+1)) * 16 + ctoi(*(tmp+2));
			tmp += 3;
		} else {
			if (ptr != tmp)
				*ptr = *tmp;
			tmp++;
		}
		ptr++;
	}
	*ptr = '\0';
	sobj->size = strlen(sobj->buff);

	return sobj;
}

int sobj_urlencode_sc(const char * from, strobj_t to)
{
	const char hex[16] = "0123456789ABCDEF";
	char endstr[8];

	if (!from || !to)
		return -1;

	while (*from) {
		if (isalnum(*from) || *from == '-' || *from == '_' || *from == '.' ||
				*from == '!' || *from == '~' || *from == '*' ||
				*from == '\'' || *from == '(' || *from == ')') {
			if (sobj_add_char(to, *from) == -1)
				return -1;
		} else {
			endstr[0] = '%';
			endstr[1] = hex[(*from >> 4) & 0x0f];
			endstr[2] = hex[(*from & 0x0f)];
			endstr[3] = '\0';
			if (sobj_add_string(to, endstr) == -1)
				return -1;
		}
		from++;
	}
	return 0;
}

int sobj_urlencode_sc_for_mydlink(const char * from, strobj_t to)
{
	const char hex[16] = "0123456789ABCDEF";
	char endstr[8];

	if (!from || !to)
		return -1;

	while (*from) {
		if(isalnum(*from)) {
			if (sobj_add_char(to, *from) == -1)
				return -1;
		} else {
			endstr[0] = '%';
			endstr[1] = hex[(*from >> 4) & 0x0f];
			endstr[2] = hex[(*from & 0x0f)];
			endstr[3] = '\0';
			if (sobj_add_string(to, endstr) == -1)
				return -1;
		}
		from++;
	}
	return 0;
}

int sobj_urldecode(const char * from, strobj_t to)
{
	char c;
	const char *end, *tmp;

	if (!from || !to)
		return -1;

	end = from + strlen(from);
	tmp = from;
	while (tmp < end) {
		if (*tmp == '%' && (tmp + 2 < end) &&
			isxdigit(*(tmp+1)) && isxdigit(*(tmp+2))) {
			c = ctoi(*(tmp+1)) * 16 + ctoi(*(tmp+2));
			tmp += 3;
		} else if(*tmp == '+') {
			c = ' ';
			tmp++;
		} else {
			c = *tmp++;
		}

		/* should i clean the buffer? */
		if (sobj_add_char(to, c) == -1)
			return -1;
	}

	return 0;
}

/*
 * string to unsigned long
 * return 0 success. others failed.
 */
int sobj_strtoul(strobj_t sobj, unsigned long int *ul, int base)
{
	char *end;

	if (!sobj || !sobj->buff)
		return -1;

	errno = 0;
	*ul = strtoul(sobj->buff, &end, base);
	/* do the following check
	 * not a decimal number
	 * extra characters at end of input
	 * out of range
	 */
	if ((end == sobj->buff) ||
			('\0' != *end) ||
			errno) {
		return -1;
	}

	return 0;
}

/*
 * string to long
 * return 0 success. others failed.
 */
int sobj_strtol(strobj_t sobj, long int *sl, int base)
{
	char *end;

	if (!sobj || !sobj->buff)
		return -1;

	errno = 0;
	*sl = strtol(sobj->buff, &end, base);
	/* do the following check
	 * not a decimal number
	 * extra characters at end of input
	 * out of range of type long
	 */
	if ((end == sobj->buff) ||
			('\0' != *end) ||
			errno) {
		return -1;
	}

	return 0;
}

/*
 * string to integer
 * return 0 success. others failed.
 */
int sobj_strtoi(strobj_t sobj, int *si, int base)
{
	char *end;
	long sl;

	if (!sobj || !sobj->buff)
		return -1;

	errno = 0;
	sl = strtol(sobj->buff, &end, base);
	/* do the following check
	 * not a decimal number
	 * extra characters at end of input
	 * out of range of type long
	 * greater than INT_MAX
	 * less than INT_MIN
	 */
	if ((end == sobj->buff) ||
			('\0' != *end) ||
			errno ||
			(sl > INT_MAX) || (sl < INT_MIN)) {
		return -1;
	} else {
		*si = (int)sl;
	}

	return 0;
}
