// SPDX-License-Identifier: GPL-2.0-or-later
/*
 *	Handle incoming frames
 *	Linux ethernet bridge
 *
 *	Authors:
 *	Lennert Buytenhek		<buytenh@gnu.org>
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/netfilter_bridge.h>
#ifdef CONFIG_NETFILTER_FAMILY_BRIDGE
#include <net/netfilter/nf_queue.h>
#endif
#include <linux/neighbour.h>
#include <net/arp.h>
#include <linux/export.h>
#include <linux/rculist.h>
#include "br_private.h"
#include "br_private_tunnel.h"

#ifdef CONFIG_BRIDGE_IGMP_MANAGER
#include <linux/igmp.h>
#endif

#ifdef CONFIG_BRIDGE_MLD_MANAGER
#include <linux/in6.h>
#include <net/mld.h>
#include <linux/icmpv6.h>

#define IPV6_ROUTER_ALTER_OPTION 0x05020000
#define HOP_BY_HOP_OPTIONS_HEADER 0
#define ROUTING_HEADER 43
#define FRAGMENT_HEADER 44
#define DESTINATION_OPTION_HEADER 60

#define PIM_PROTOCOL 103
#define MOSPF_PROTOCOL 89
#define TCP_PROTOCOL 6
#define UDP_PROTOCOL 17
#define NO_NEXT_HEADER 59
#define ICMP_PROTOCOL 58

#define MLD_QUERY 130
#define MLDV1_REPORT 131
#define MLDV1_DONE 132
#define MLDV2_REPORT 143

#define IS_IPV6_PIM_ADDR(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x0000000D))
#define IS_IPV6_MOSPF_ADDR1(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x00000005))
#define IS_IPV6_MOSPF_ADDR2(ipv6addr) ((ipv6addr[0] == 0xFF020000)&&(ipv6addr[1] == 0x00000000)&&(ipv6addr[2] == 0x00000000)&&(ipv6addr[3] ==0x00000006))
#endif

#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
extern void write_alpha_multicast (unsigned char op, struct net_bridge *br, unsigned char *dest, struct sk_buff *skb);
#endif

#ifdef CONFIG_BRIDGE_GUEST_ZONE
typedef int (*gz_skb_mark_cb)(const struct net_bridge *, struct sk_buff *);
gz_skb_mark_cb gz_skb_mark=NULL;

void register_gz_skb_mark_init_callback(gz_skb_mark_cb funa)
{
	gz_skb_mark = funa;
}

void unregister_gz_skb_mark_init_callback(void)
{
	gz_skb_mark = NULL;
}

EXPORT_SYMBOL(register_gz_skb_mark_init_callback);
EXPORT_SYMBOL(unregister_gz_skb_mark_init_callback);
#endif

static int
br_netif_receive_skb(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	br_drop_fake_rtable(skb);
	return netif_receive_skb(skb);
}

static int br_pass_frame_up(struct sk_buff *skb)
{
	struct net_device *indev, *brdev = BR_INPUT_SKB_CB(skb)->brdev;
	struct net_bridge *br = netdev_priv(brdev);
	struct net_bridge_vlan_group *vg;
	struct pcpu_sw_netstats *brstats = this_cpu_ptr(br->stats);

	u64_stats_update_begin(&brstats->syncp);
	brstats->rx_packets++;
	brstats->rx_bytes += skb->len;
	u64_stats_update_end(&brstats->syncp);

	vg = br_vlan_group_rcu(br);

	/* Reset the offload_fwd_mark because there could be a stacked
	 * bridge above, and it should not think this bridge it doing
	 * that bridge's work forwarding out its ports.
	 */
	br_switchdev_frame_unmark(skb);

	/* Bridge is just like any other port.  Make sure the
	 * packet is allowed except in promisc modue when someone
	 * may be running packet capture.
	 */
	if (!(brdev->flags & IFF_PROMISC) &&
	    !br_allowed_egress(vg, skb)) {
		kfree_skb(skb);
		return NET_RX_DROP;
	}

#ifdef CONFIG_BRIDGE_GUEST_ZONE
	if (gz_skb_mark)
	{
		if (!gz_skb_mark(br, skb))
		{
			kfree_skb(skb);
			return NET_RX_DROP;
		}
	}
#endif

	indev = skb->dev;
	skb->dev = brdev;
	skb = br_handle_vlan(br, NULL, vg, skb);
	if (!skb)
		return NET_RX_DROP;
	/* update the multicast stats if the packet is IGMP/MLD */
	br_multicast_count(br, NULL, skb, br_multicast_igmp_type(skb),
			   BR_MCAST_DIR_TX);

	return NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN,
		       dev_net(indev), NULL, skb, indev, NULL,
		       br_netif_receive_skb);
}

#ifdef CONFIG_BRIDGE_IGMP_MANAGER
//#define DBG_IGMP	//enable it to debug igmp check
#ifdef DBG_IGMP
#define NIPQUAD(addr) \
        ((unsigned char *)&addr)[0], \
        ((unsigned char *)&addr)[1], \
        ((unsigned char *)&addr)[2], \
        ((unsigned char *)&addr)[3]
#endif

//---	if group is an reserved ip address(upnp, nec adp deamon), then return 0;
//---	convert mac to IPv4 multicast. [0x01005e(24bits) + multicast ip(smallest 24bits)]
int ConvertMulticatIPtoMacAddr(__u32 group, unsigned char *gmac)
{
	__u32 u32tmp, tmp;
	int i;
	int ret=1;

	if((group&0xFFFFFF00)==0xEFFFFF00 || group == 0xEFFFFC84 || group == 0xE0001700 || (group&0xFFFFFF00)==0xE0000000)
	{
		//UPnP etc.:239.255.255.0/24, Aterm Discovery Protocol:239.255.252.132, ECHONET Lite 224.0.23.0, 224.0.0.0/24
		ret = 0;
	}
	
	u32tmp = group & 0x007FFFFF;
	gmac[0]=0x01; gmac[1]=0x00; gmac[2]=0x5e;
	for (i=5; i>=3; i--) {
		tmp=u32tmp&0xFF;
		gmac[i]=tmp;
		u32tmp >>= 8;
	}
	return ret;
}

//---	check IGMP message.
//---	return	-1	->	packet may have some problems( header shortage, illegal multicast address), drop this packet.
//---	return	1	->	membership exclude report or unknown reports, add case
//---	return	2	->	membership include report or leave, delete case
static char igmp_type_check(struct sk_buff *skb, unsigned char *gmac,unsigned int *gIndex,unsigned int *moreFlag)
{
	struct iphdr *iph;
	__u8 hdrlen;
	struct igmphdr *igmph;
	int i;
	unsigned int groupAddr=0;// add for fit igmp v3
	*moreFlag=0;

	iph=(struct iphdr *)skb_network_header(skb);
	hdrlen = iph->ihl << 2;
	if ((iph->version != 4) &&  (hdrlen < 20))
		return -1;
	if (ip_fast_csum((u8 *)iph, iph->ihl) != 0)
		return -1;
	{ /* check the length */
		__u32 len = ntohs(iph->tot_len);
		if (skb->len < len || len < hdrlen)
			return -1; 
	}
	/* parsing the igmp packet */
	igmph = (struct igmphdr *)((u8*)iph+hdrlen);
	
	if ((igmph->type==IGMP_HOST_MEMBERSHIP_REPORT) ||
		(igmph->type==IGMPV2_HOST_MEMBERSHIP_REPORT)) 
	{
		groupAddr = ntohl(igmph->group);
		if(!IN_MULTICAST(groupAddr))
		{			
			return -1;
		}
		
		if(ConvertMulticatIPtoMacAddr(groupAddr, gmac))
			return 1; /* report and add it */
		else
			return -1;
	}
	else if (igmph->type==IGMPV3_HOST_MEMBERSHIP_REPORT)
	{ 
		//for support igmp v3   	
		struct igmpv3_report *igmpv3report=(struct igmpv3_report * )igmph;
		struct igmpv3_grec	*igmpv3grec=NULL; 
#ifdef DBG_IGMP
		printk("%s:%d,*gIndex is %d,igmpv3report->ngrec is %d\n",__FUNCTION__,__LINE__,*gIndex,igmpv3report->ngrec);
#endif
		if(*gIndex>=ntohs(igmpv3report->ngrec))
		{
			*moreFlag=0;
			return -1;
		}
		for(i=0;i<ntohs(igmpv3report->ngrec);i++)
		{
			if(i==0)
			{
				igmpv3grec = (struct igmpv3_grec *)(&(igmpv3report->grec)); /*first igmp group record*/
			}
			else
			{
				igmpv3grec=(struct igmpv3_grec *)((unsigned char*)igmpv3grec+8+ntohs(igmpv3grec->grec_nsrcs)*4+(igmpv3grec->grec_auxwords)*4);
				
			}
			
			if(i!=*gIndex)
			{	
				continue;
			}
			if(i==(ntohs(igmpv3report->ngrec)-1))
			{
				/*last group record*/
				*moreFlag=0;
			}
			else
			{
				*moreFlag=1;
			}
			
			/*gIndex move to next group*/
			*gIndex=*gIndex+1;	
			groupAddr= ntohl(igmpv3grec->grec_mca);
#ifdef DBG_IGMP
			printk("%s:%d,groupAddr is %d.%d.%d.%d\n",__FUNCTION__,__LINE__,NIPQUAD(groupAddr));
#endif
			if(!IN_MULTICAST(groupAddr))
			{			
				return -1;
			}
			
			if(!ConvertMulticatIPtoMacAddr(groupAddr, gmac))
				return -1;
			if(((igmpv3grec->grec_type == IGMPV3_CHANGE_TO_INCLUDE) || (igmpv3grec->grec_type == IGMPV3_MODE_IS_INCLUDE))&& (ntohs(igmpv3grec->grec_nsrcs)==0))
			{	
				return 2; /* leave and delete it */	
			}
			else if((igmpv3grec->grec_type == IGMPV3_CHANGE_TO_EXCLUDE) ||
				(igmpv3grec->grec_type == IGMPV3_MODE_IS_EXCLUDE) ||
				(igmpv3grec->grec_type == IGMPV3_ALLOW_NEW_SOURCES))
			{
				return 1;
			}
			else
			{
				return 1; //unknown igmp report is regarded as join
			}
			
			return -1;
		}
		
		/*avoid dead loop in case of initial gIndex is too big*/
		if(i>=(ntohs(igmpv3report->ngrec)-1))
		{
			/*last group record*/
			*moreFlag=0;
			return -1;
		}

	}
	else if (igmph->type==IGMP_HOST_LEAVE_MESSAGE){
		groupAddr = ntohl(igmph->group);
		if(!IN_MULTICAST(groupAddr))
		{			
				return -1;
		}
		
		if(ConvertMulticatIPtoMacAddr(groupAddr, gmac))
			return 2; /* leave and delete it */
		else
			return -1;
	}	
	
	return -1;
}
#endif

#ifdef CONFIG_BRIDGE_MLD_MANAGER
//#define DBG_ICMPv6	//enable it to debug icmpv6 check
//---	parse IPv6 header, check packet's Transport Protocol.
int getIpv6TransportProtocol(struct ipv6hdr* ipv6h)
{
	unsigned char *ptr=NULL;
	unsigned char *startPtr=NULL;
	unsigned char *lastPtr=NULL;
	unsigned char nextHeader=0;
	unsigned short extensionHdrLen=0;

	unsigned char  optionDataLen=0;
	unsigned char  optionType=0;
	unsigned int ipv6RAO=0;
	unsigned int ipv6addr[4] = {0};

	if(ipv6h==NULL)
	{
		return -1;
	}

	if(ipv6h->version!=6)
	{
		return -1;
	}

	startPtr= (unsigned char *)ipv6h;
	lastPtr=startPtr+sizeof(struct ipv6hdr)+ntohs(ipv6h->payload_len);
	nextHeader= ipv6h ->nexthdr;
	ptr=startPtr+sizeof(struct ipv6hdr);

	ipv6addr[0] = ntohl(ipv6h->daddr.s6_addr32[0]);
	ipv6addr[1] = ntohl(ipv6h->daddr.s6_addr32[1]);
	ipv6addr[2] = ntohl(ipv6h->daddr.s6_addr32[2]);
	ipv6addr[3] = ntohl(ipv6h->daddr.s6_addr32[3]);

	while(ptr<lastPtr)
	{
		switch(nextHeader)
		{
			case HOP_BY_HOP_OPTIONS_HEADER:
				/*parse hop-by-hop option*/
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+2;

				while(ptr<(startPtr+extensionHdrLen+sizeof(struct ipv6hdr)))
				{
					optionType=ptr[0];
					/*pad1 option*/
					if(optionType==0)
					{
						ptr=ptr+1;
						continue;
					}

					/*padN option*/
					if(optionType==1)
					{
						optionDataLen=ptr[1];
						ptr=ptr+optionDataLen+2;
						continue;
					}

					/*router altert option*/
					if(ntohl(*(uint32 *)(ptr))==IPV6_ROUTER_ALTER_OPTION)
					{
						ipv6RAO=IPV6_ROUTER_ALTER_OPTION;
						ptr=ptr+4;
						continue;
					}

					/*other TLV option*/
					if((optionType!=0) && (optionType!=1))
					{
						optionDataLen=ptr[1];
						ptr=ptr+optionDataLen+2;
						continue;
					}
				}

				break;

			case ROUTING_HEADER:
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+extensionHdrLen;
				break;

			case FRAGMENT_HEADER:
				nextHeader=ptr[0];
				ptr=ptr+8;
				break;

			case DESTINATION_OPTION_HEADER:
				nextHeader=ptr[0];
				extensionHdrLen=((uint16)(ptr[1])+1)*8;
				ptr=ptr+extensionHdrLen;
				break;

			case ICMP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if((ptr[0]==MLD_QUERY) ||(ptr[0]==MLDV1_REPORT) ||(ptr[0]==MLDV1_DONE) ||(ptr[0]==MLDV2_REPORT))
				{
					return ICMP_PROTOCOL;

				}
				break;

			case PIM_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if(IS_IPV6_PIM_ADDR(ipv6addr))
				{
					return PIM_PROTOCOL;
				}

				break;

			case MOSPF_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				if(IS_IPV6_MOSPF_ADDR1(ipv6addr) || IS_IPV6_MOSPF_ADDR2(ipv6addr))
				{
					return MOSPF_PROTOCOL;
				}
				break;

			case TCP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				return TCP_PROTOCOL;

				break;

			case UDP_PROTOCOL:
				nextHeader=NO_NEXT_HEADER;
				return UDP_PROTOCOL;

				break;

			default:
				/*not ipv6 multicast protocol*/
				return -1;
				break;
		}
	}
	return -1;
}

/*Convert  MultiCatst IPV6_Addr to MAC_Addr*/
//---	if group is an reserved ip address, then return 0;
//---	convert mac to IPv6 multicast. [0x3333(16 bits) + multicast ip(smallest 32bits)]
void CIPV6toMac(unsigned char* icmpv6_McastAddr, unsigned char *gmac )
{
	/*ICMPv6 valid addr 2^32 -1*/
	gmac[0] = 0x33;
	gmac[1] = 0x33;
	gmac[2] = icmpv6_McastAddr[12];
	gmac[3] = icmpv6_McastAddr[13];
	gmac[4] = icmpv6_McastAddr[14];
	gmac[5] = icmpv6_McastAddr[15];			
}

//add mld v2 support
//---	check IPMPv6 MLD message.
//---	return	-1	->	packet may have some problems or not MLD/MLD2 protocol, drop this packet.
//---	return	1	->	membership exclude report or unknown reports, add case
//---	return	2	->	membership include report or listener done, delete case
char ICMPv6_check(struct sk_buff *skb , unsigned char *gmac, unsigned int *gIndex,unsigned int *moreFlag)
{
	struct ipv6hdr *ipv6h;
	char* protoType;
	int i = 0;
	*moreFlag=0;

	/* check IPv6 header information */
	//ipv6h = skb->nh.ipv6h;
	ipv6h = (struct ipv6hdr *)skb_network_header(skb);
	if(ipv6h->version != 6){
#ifdef DBG_ICMPv6
		printk("ipv6h->version != 6\n");
#endif
		return -1;
	}
 
	/*Next header: IPv6 hop-by-hop option (0x00)*/
	if(ipv6h->nexthdr == 0) 
	{
		protoType = (unsigned char*)( (unsigned char*)ipv6h + sizeof(struct ipv6hdr) );
	}
	else
	{
#ifdef DBG_ICMPv6
		printk("ipv6h->nexthdr != 0\n");
#endif
		return -1;
	}

	if(protoType[0] == 0x3a)
	{
#ifdef DBG_ICMPv6
		printk("recv icmpv6 packet\n");
#endif
		struct icmp6hdr* icmpv6h = (struct icmp6hdr*)(protoType + 8);
		unsigned char *icmpv6_McastAddr;
 
		if(icmpv6h->icmp6_type == ICMPV6_MGM_REPORT)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8);
			
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener report) \n",icmpv6h->icmp6_type);
#endif
			if(icmpv6_McastAddr[0] != 0xFF)
				return -1;
			if(icmpv6_McastAddr[0] == 0xFF && icmpv6_McastAddr[1]==0x02) //flooding ff02::/16
			{
#ifdef DBG_ICMPv6
				printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
					icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
					icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
					icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
					icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
#endif
				return -1;
			}
			if(icmpv6_McastAddr[0] == 0xFF && icmpv6_McastAddr[1]==0x05 && icmpv6_McastAddr[2]==0x00
			&& icmpv6_McastAddr[3] == 0x00 && icmpv6_McastAddr[4]==0x00 && icmpv6_McastAddr[5]==0x00
			&& icmpv6_McastAddr[6] == 0x00 && icmpv6_McastAddr[7]==0x00 && icmpv6_McastAddr[8]==0x00
			&& icmpv6_McastAddr[9] == 0x00 && icmpv6_McastAddr[10]==0x00 && icmpv6_McastAddr[11]==0x00
			&& icmpv6_McastAddr[12] == 0x00 && icmpv6_McastAddr[13]==0x00 && icmpv6_McastAddr[14]==0x00
			&& icmpv6_McastAddr[15]==0x0C) //flooding SSDP ff05::C
			{
#ifdef DBG_ICMPv6
				printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
					icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
					icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
					icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
					icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
#endif
				return -1;
			}
			CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
			printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
				icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
				icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
				icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
				icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
			printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
				gmac[0],gmac[1],gmac[2],
				gmac[3],gmac[4],gmac[5]);
#endif
			return 1;//add

		}
		else if(icmpv6h->icmp6_type == ICMPV6_MLD2_REPORT)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 + 4);
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener report v2) \n",icmpv6h->icmp6_type);
#endif
        	struct mld2_report *mldv2report = (struct mld2_report * )icmpv6h;
           	struct mld2_grec *mldv2grec = NULL;
#ifdef DBG_ICMPv6
			printk("%s:%d,*gIndex is %d,mldv2report->ngrec is %d\n",__FUNCTION__,__LINE__,*gIndex, ntohs(mldv2report->mld2r_ngrec));
#endif
			if(*gIndex >= ntohs(mldv2report->mld2r_ngrec))
			{
				*moreFlag=0;
				return -1;
			}

			for(i=0;i< ntohs(mldv2report->mld2r_ngrec);i++)
			{
				if(i==0)
				{
					mldv2grec = (struct mld2_grec *)(&(mldv2report->mld2r_grec)); /*first igmp group record*/
				}
				else
				{
					mldv2grec = (struct mld2_grec *)((unsigned char*)mldv2grec+4+16+ntohs(mldv2grec->grec_nsrcs)*16+(mldv2grec->grec_auxwords)*4);
				}

				if(i!=*gIndex)
				{
					continue;
				}

				if(i==(ntohs(mldv2report->mld2r_ngrec)-1))
				{
					/*last group record*/
					*moreFlag=0;
				}
				else
				{
					*moreFlag=1;
				}

				/*gIndex move to next group*/
				*gIndex=*gIndex+1;

				icmpv6_McastAddr = (unsigned char *)&mldv2grec->grec_mca;
#ifdef DBG_ICMPv6
				printk("%s:%d,groupAddr is %d.%d.%d.%d\n",__FUNCTION__,__LINE__,NIPQUAD(groupAddr));
#endif
				if(icmpv6_McastAddr[0] != 0xFF)
					return -1;
				if(icmpv6_McastAddr[0] == 0xFF && icmpv6_McastAddr[1]==0x02) //flooding ff02::/16
				{
#ifdef DBG_ICMPv6
					printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
						icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
						icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
						icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
						icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
#endif
					return -1;
				}
				if(icmpv6_McastAddr[0] == 0xFF && icmpv6_McastAddr[1]==0x05 && icmpv6_McastAddr[2]==0x00
				&& icmpv6_McastAddr[3] == 0x00 && icmpv6_McastAddr[4]==0x00 && icmpv6_McastAddr[5]==0x00
				&& icmpv6_McastAddr[6] == 0x00 && icmpv6_McastAddr[7]==0x00 && icmpv6_McastAddr[8]==0x00
				&& icmpv6_McastAddr[9] == 0x00 && icmpv6_McastAddr[10]==0x00 && icmpv6_McastAddr[11]==0x00
				&& icmpv6_McastAddr[12] == 0x00 && icmpv6_McastAddr[13]==0x00 && icmpv6_McastAddr[14]==0x00
				&& icmpv6_McastAddr[15]==0x0C) //flooding SSDP ff05::C
				{
#ifdef DBG_ICMPv6
					printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
						icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
						icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
						icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
						icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
#endif
					return -1;
				}
				CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
				printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
					icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
					icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
					icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
					icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
				printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
					gmac[0],gmac[1],gmac[2],
					gmac[3],gmac[4],gmac[5]);
 
				printk("grec_type = %d, src_num = %d\n", mldv2grec->grec_type, mldv2grec->grec_nsrcs);
#endif
 
				if(mldv2grec->grec_type== MLD2_BLOCK_OLD_SOURCES && (ntohs(mldv2grec->grec_nsrcs)!=0))
				{
					return 2; /* MLD2_BLOCK_OLD_SOURCES -> delete it */
				}
				else if(((mldv2grec->grec_type== MLD2_CHANGE_TO_INCLUDE) || (mldv2grec->grec_type == MLD2_MODE_IS_INCLUDE))&& (ntohs(mldv2grec->grec_nsrcs)==0))
				{
					return 2; /* leave and delete it */
				}
				else if((mldv2grec->grec_type == MLD2_CHANGE_TO_EXCLUDE) ||
						(mldv2grec->grec_type == MLD2_MODE_IS_EXCLUDE) ||
						(mldv2grec->grec_type == MLD2_ALLOW_NEW_SOURCES))
				{
					return 1;
				}
				else
				{
					return 1; //unknown mld report is regarded as join
				}
 
				return -1;
			}
 
			/*avoid dead loop in case of initial gIndex is too big*/
			if(i>=(ntohs(mldv2report->mld2r_ngrec)-1))
			{
				/*last group record*/
				*moreFlag=0;
				return -1;
			}
		}
		//---	Multicast Linster Done
		else if(icmpv6h->icmp6_type == ICMPV6_MGM_REDUCTION)
		{
			icmpv6_McastAddr = (unsigned char*)((unsigned char*)icmpv6h + 8 );
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (Multicast listener done ) \n",icmpv6h->icmp6_type);
#endif
			if(icmpv6_McastAddr[0] != 0xFF)
				return -1;
			CIPV6toMac(icmpv6_McastAddr, gmac);
#ifdef DBG_ICMPv6
			printk("MCAST_IPV6Addr:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x \n",
				icmpv6_McastAddr[0],icmpv6_McastAddr[1],icmpv6_McastAddr[2],icmpv6_McastAddr[3],
				icmpv6_McastAddr[4],icmpv6_McastAddr[5],icmpv6_McastAddr[6],icmpv6_McastAddr[7],
				icmpv6_McastAddr[8],icmpv6_McastAddr[9],icmpv6_McastAddr[10],icmpv6_McastAddr[11],
				icmpv6_McastAddr[12],icmpv6_McastAddr[13],icmpv6_McastAddr[14],icmpv6_McastAddr[15]);
 
			printk("group_mac [%02x:%02x:%02x:%02x:%02x:%02x] \n",
				gmac[0],gmac[1],gmac[2],
				gmac[3],gmac[4],gmac[5]);
#endif
			return 2;//del
		}
		else
		{
#ifdef DBG_ICMPv6
			printk("Type: 0x%x (unknow type)\n",icmpv6h->icmp6_type);
#endif
			return -1;
		}
	}
	else
	{
#ifdef DBG_ICMPv6
		printk("protoType[0] != 0x3a\n");
#endif
		return -1;//not icmpv6 type
	}
 
	return -1;
}
#endif

/* note: already called with rcu_read_lock */
int br_handle_frame_finish(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);
	enum br_pkt_type pkt_type = BR_PKT_UNICAST;
	struct net_bridge_fdb_entry *dst = NULL;
	struct net_bridge_mdb_entry *mdst;
	bool local_rcv, mcast_hit = false;
	struct net_bridge *br;
	u16 vid = 0;

	if (!p || p->state == BR_STATE_DISABLED)
		goto drop;

	if (!br_allowed_ingress(p->br, nbp_vlan_group_rcu(p), skb, &vid))
		goto out;

	nbp_switchdev_frame_mark(p, skb);

	/* insert into forwarding database after filtering to avoid spoofing */
	br = p->br;
	if (p->flags & BR_LEARNING)
		br_fdb_update(br, p, eth_hdr(skb)->h_source, vid, false);

	local_rcv = !!(br->dev->flags & IFF_PROMISC);

#ifdef CONFIG_NET_PPPOE_IPV6_PTHROUGH
	{
		// default let pppoe packet to send up to support pppoe passthrough.
		unsigned short proto;
		struct ethhdr *eth = eth_hdr(skb);

		proto = eth->h_proto;
		if ((proto == htons(ETH_P_PPP_SES)) || (proto == htons(ETH_P_PPP_DISC))) {
			local_rcv = true;
		}
	}
#endif

	if (is_multicast_ether_addr(eth_hdr(skb)->h_dest)) {
		/* by definition the broadcast is also a multicast address */
		if (is_broadcast_ether_addr(eth_hdr(skb)->h_dest)) {
			pkt_type = BR_PKT_BROADCAST;
			local_rcv = true;
		} else {
			pkt_type = BR_PKT_MULTICAST;
			if (br_multicast_rcv(br, p, skb, vid))
				goto drop;
		}
	}

	BR_INPUT_SKB_CB(skb)->brdev = br->dev;

	if (skb->protocol == htons(ETH_P_PAE) && !br->disable_eap_hack)
		return br_pass_frame_up(skb);

	if (p->state == BR_STATE_LEARNING)
		goto drop;

	BR_INPUT_SKB_CB(skb)->src_port_isolated = !!(p->flags & BR_ISOLATED);

	if (IS_ENABLED(CONFIG_INET) &&
	    (skb->protocol == htons(ETH_P_ARP) ||
	     skb->protocol == htons(ETH_P_RARP))) {
		br_do_proxy_suppress_arp(skb, br, vid, p);
	} else if (IS_ENABLED(CONFIG_IPV6) &&
		   skb->protocol == htons(ETH_P_IPV6) &&
		   br_opt_get(br, BROPT_NEIGH_SUPPRESS_ENABLED) &&
		   pskb_may_pull(skb, sizeof(struct ipv6hdr) +
				 sizeof(struct nd_msg)) &&
		   ipv6_hdr(skb)->nexthdr == IPPROTO_ICMPV6) {
			struct nd_msg *msg, _msg;

			msg = br_is_nd_neigh_msg(skb, &_msg);
			if (msg)
				br_do_suppress_nd(skb, br, vid, p, msg);
	}

	switch (pkt_type) {
	case BR_PKT_MULTICAST:
		mdst = br_mdb_get(br, skb, vid);
		if ((mdst || BR_INPUT_SKB_CB_MROUTERS_ONLY(skb)) &&
		    br_multicast_querier_exists(br, eth_hdr(skb))) {
			if ((mdst && mdst->host_joined) ||
			    br_multicast_is_router(br)) {
				local_rcv = true;
				DEV_STATS_INC(br->dev, multicast);
			}
			mcast_hit = true;
		} else {
			local_rcv = true;
			DEV_STATS_INC(br->dev, multicast);
		}
		break;
	case BR_PKT_UNICAST:
		dst = br_fdb_find_rcu(br, eth_hdr(skb)->h_dest, vid);
	default:
		break;
	}

#if defined(CONFIG_BRIDGE_IGMP_MANAGER) || defined(CONFIG_BRIDGE_MLD_MANAGER)
	if (skb) {
		if (is_multicast_ether_addr(eth_hdr(skb)->h_dest))
		{
#ifdef CONFIG_BRIDGE_IGMP_MANAGER
			struct iphdr *iph=NULL;
#endif
#ifdef CONFIG_BRIDGE_MLD_MANAGER
			struct ipv6hdr *ipv6h=NULL;
#endif
			unsigned char proto=0;
			unsigned char macAddr[6];
			unsigned char operation;
			char tmpOp;
			unsigned int gIndex=0;
			unsigned int moreFlag=1;
#ifdef CONFIG_BRIDGE_IGMP_MANAGER
			//---	if it's a multicast mac and IPv4 protocol. check igmp.
			if (MULTICAST_MAC(eth_hdr(skb)->h_dest) 
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IP)))
			{
				//---	get the network layer header start position.
				iph=(struct iphdr *)skb_network_header(skb);
				proto =  iph->protocol;
				
				//---	if network layer's protocol is IGMP
				if (proto == IPPROTO_IGMP) 
				{
					//printk("[%s:%d]process IGMP\n",__FUNCTION__,__LINE__);
					while(moreFlag)
					{
						//---	check igmp columns if it's a add/delete/error packet.
						tmpOp=igmp_type_check(skb, macAddr, &gIndex, &moreFlag);
						if(tmpOp>0)
						{
							operation=(unsigned char)tmpOp;
#ifdef DBG_IGMP
							if( operation == 1)
								printk("igmp add from frame finish\n");
							else if(operation == 2)
								printk("igmp del from frame finish\n");
#endif
							write_alpha_multicast(operation, br, macAddr, skb);
						}
					}
				}
			}
#endif
			//---	if it's a multicast mac and IPv6 protocol. check icmpv6_mld.
#if defined(CONFIG_BRIDGE_IGMP_MANAGER) && defined(CONFIG_BRIDGE_MLD_MANAGER)
			else if(IPV6_MULTICAST_MAC(eth_hdr(skb)->h_dest)
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)))
#elif defined(CONFIG_BRIDGE_MLD_MANAGER)
			if(IPV6_MULTICAST_MAC(eth_hdr(skb)->h_dest)
				&& (eth_hdr(skb)->h_proto == htons(ETH_P_IPV6)))
#endif
#ifdef CONFIG_BRIDGE_MLD_MANAGER
			{
				//---	get the network layer header start position.
				ipv6h=(struct ipv6hdr *)skb_network_header(skb);
				proto =  getIpv6TransportProtocol(ipv6h);
				/*icmp protocol*/
				if (proto == IPPROTO_ICMPV6) 
				{
#ifdef DBG_ICMPv6
					if(net_ratelimit())
						printk("[%s:%d]ipv6 ICMPV6,\n",__FUNCTION__,__LINE__);		
#endif
					while(moreFlag)
					{
						//---	check ICMPv6 columns if it's a add/delete/error packet.
						tmpOp=ICMPv6_check(skb , macAddr, &gIndex, &moreFlag);
						if(tmpOp > 0)
						{
							operation=(unsigned char)tmpOp;
#ifdef DBG_ICMPv6
							if( operation == 1)
								printk("icmpv6 add from frame finish\n");
							else if(operation == 2)
								printk("icmpv6 del from frame finish\n");
#endif
							write_alpha_multicast(operation, br, macAddr, skb);
						}
					}
				}
			}
#endif
		}
	}
#endif

	if (dst) {
		unsigned long now = jiffies;

		if (dst->is_local)
			return br_pass_frame_up(skb);

		if (now != dst->used)
			dst->used = now;
		br_forward(dst->dst, skb, local_rcv, false);
	} else {
		if (!mcast_hit)
			br_flood(br, skb, pkt_type, local_rcv, false);
		else
			br_multicast_flood(mdst, skb, local_rcv, false);
	}

	if (local_rcv)
		return br_pass_frame_up(skb);

out:
	return 0;
drop:
	kfree_skb(skb);
	goto out;
}
EXPORT_SYMBOL_GPL(br_handle_frame_finish);

static void __br_handle_local_finish(struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);
	u16 vid = 0;

	/* check if vlan is allowed, to avoid spoofing */
	if ((p->flags & BR_LEARNING) &&
	    !br_opt_get(p->br, BROPT_NO_LL_LEARN) &&
	    br_should_learn(p, skb, &vid))
		br_fdb_update(p->br, p, eth_hdr(skb)->h_source, vid, false);
}

/* note: already called with rcu_read_lock */
static int br_handle_local_finish(struct net *net, struct sock *sk, struct sk_buff *skb)
{
	struct net_bridge_port *p = br_port_get_rcu(skb->dev);

	if (p->state != BR_STATE_DISABLED)
	__br_handle_local_finish(skb);

	/* return 1 to signal the okfn() was called so it's ok to use the skb */
	return 1;
}

static int nf_hook_bridge_pre(struct sk_buff *skb, struct sk_buff **pskb)
{
#ifdef CONFIG_NETFILTER_FAMILY_BRIDGE
	struct nf_hook_entries *e = NULL;
	struct nf_hook_state state;
	unsigned int verdict, i;
	struct net *net;
	int ret;

	net = dev_net(skb->dev);
#ifdef HAVE_JUMP_LABEL
	if (!static_key_false(&nf_hooks_needed[NFPROTO_BRIDGE][NF_BR_PRE_ROUTING]))
		goto frame_finish;
#endif

	e = rcu_dereference(net->nf.hooks_bridge[NF_BR_PRE_ROUTING]);
	if (!e)
		goto frame_finish;

	nf_hook_state_init(&state, NF_BR_PRE_ROUTING,
			   NFPROTO_BRIDGE, skb->dev, NULL, NULL,
			   net, br_handle_frame_finish);

	for (i = 0; i < e->num_hook_entries; i++) {
		verdict = nf_hook_entry_hookfn(&e->hooks[i], skb, &state);
		switch (verdict & NF_VERDICT_MASK) {
		case NF_ACCEPT:
			if (BR_INPUT_SKB_CB(skb)->br_netfilter_broute) {
				*pskb = skb;
				return RX_HANDLER_PASS;
			}
			break;
		case NF_DROP:
			kfree_skb(skb);
			return RX_HANDLER_CONSUMED;
		case NF_QUEUE:
			ret = nf_queue(skb, &state, i, verdict);
			if (ret == 1)
				continue;
			return RX_HANDLER_CONSUMED;
		default: /* STOLEN */
			return RX_HANDLER_CONSUMED;
		}
	}
frame_finish:
	net = dev_net(skb->dev);
	br_handle_frame_finish(net, NULL, skb);
#else
	br_handle_frame_finish(dev_net(skb->dev), NULL, skb);
#endif
	return RX_HANDLER_CONSUMED;
}

/*
 * Return NULL if skb is handled
 * note: already called with rcu_read_lock
 */
rx_handler_result_t br_handle_frame(struct sk_buff **pskb)
{
	struct net_bridge_port *p;
	struct sk_buff *skb = *pskb;
	const unsigned char *dest = eth_hdr(skb)->h_dest;

	if (unlikely(skb->pkt_type == PACKET_LOOPBACK))
		return RX_HANDLER_PASS;

	if (!is_valid_ether_addr(eth_hdr(skb)->h_source))
		goto drop;

	skb = skb_share_check(skb, GFP_ATOMIC);
	if (!skb)
		return RX_HANDLER_CONSUMED;

	memset(skb->cb, 0, sizeof(struct br_input_skb_cb));

	p = br_port_get_rcu(skb->dev);
	if (p->flags & BR_VLAN_TUNNEL) {
		if (br_handle_ingress_vlan_tunnel(skb, p,
						  nbp_vlan_group_rcu(p)))
			goto drop;
	}

	if (unlikely(is_link_local_ether_addr(dest))) {
		u16 fwd_mask = p->br->group_fwd_mask_required;

		/*
		 * See IEEE 802.1D Table 7-10 Reserved addresses
		 *
		 * Assignment		 		Value
		 * Bridge Group Address		01-80-C2-00-00-00
		 * (MAC Control) 802.3		01-80-C2-00-00-01
		 * (Link Aggregation) 802.3	01-80-C2-00-00-02
		 * 802.1X PAE address		01-80-C2-00-00-03
		 *
		 * 802.1AB LLDP 		01-80-C2-00-00-0E
		 *
		 * Others reserved for future standardization
		 */
		fwd_mask |= p->group_fwd_mask;
		switch (dest[5]) {
		case 0x00:	/* Bridge Group Address */
			/* If STP is turned off,
			   then must forward to keep loop detection */
			if (p->br->stp_enabled == BR_NO_STP ||
			    fwd_mask & (1u << dest[5]))
				goto forward;
			*pskb = skb;
			__br_handle_local_finish(skb);
			return RX_HANDLER_PASS;

		case 0x01:	/* IEEE MAC (Pause) */
			goto drop;

		case 0x0E:	/* 802.1AB LLDP */
			fwd_mask |= p->br->group_fwd_mask;
			if (fwd_mask & (1u << dest[5]))
				goto forward;
			*pskb = skb;
			__br_handle_local_finish(skb);
			return RX_HANDLER_PASS;

		default:
			/* Allow selective forwarding for most other protocols */
			fwd_mask |= p->br->group_fwd_mask;
			if (fwd_mask & (1u << dest[5]))
				goto forward;
		}

		/* The else clause should be hit when nf_hook():
		 *   - returns < 0 (drop/error)
		 *   - returns = 0 (stolen/nf_queue)
		 * Thus return 1 from the okfn() to signal the skb is ok to pass
		 */
		if (NF_HOOK(NFPROTO_BRIDGE, NF_BR_LOCAL_IN,
			    dev_net(skb->dev), NULL, skb, skb->dev, NULL,
			    br_handle_local_finish) == 1) {
			return RX_HANDLER_PASS;
		} else {
			return RX_HANDLER_CONSUMED;
		}
	}

forward:
	switch (p->state) {
	case BR_STATE_DISABLED:
		if (ether_addr_equal(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		if (NF_HOOK(NFPROTO_BRIDGE, NF_BR_PRE_ROUTING,
			dev_net(skb->dev), NULL, skb, skb->dev, NULL,
			br_handle_local_finish) == 1) {
			return RX_HANDLER_PASS;
		}
		break;

	case BR_STATE_FORWARDING:
	case BR_STATE_LEARNING:
		if (ether_addr_equal(p->br->dev->dev_addr, dest))
			skb->pkt_type = PACKET_HOST;

		return nf_hook_bridge_pre(skb, pskb);
	default:
drop:
		kfree_skb(skb);
	}
	return RX_HANDLER_CONSUMED;
}
