/*
 * Copyright (c) 2023, MediaTek Inc. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <assert.h>
#include <arch_helpers.h>
#include <common/debug.h>
#include <lib/psci/psci.h>
#include <lib/mmio.h>
#include <mtk_gic_v3.h>
#include <mtspmc.h>
#include <mcucfg.h>
#include <plat_pm.h>

uintptr_t secure_entrypoint;

static void plat_cpu_pwron_common(void)
{
	/* Enable the gic cpu interface */
	mt_gic_cpuif_enable();
	mt_gic_rdistif_init();
	mt_gic_rdistif_restore();
}

static void plat_cpu_pwrdwn_common(void)
{
	/* Prevent interrupts from spuriously waking up this cpu */
	mt_gic_rdistif_save();
	mt_gic_cpuif_disable();
}

int mtk_plat_power_domain_on(unsigned long mpidr)
{
	int cpu_id = MPIDR_AFFLVL0_VAL(mpidr);
	int arm64 = 1;

	INFO("%s: cpu: %d (total cpu: %d)\n", __func__, cpu_id,
	     PLATFORM_CORE_COUNT);

	if (arm64)
		mmio_setbits_32(mcucfg_reg(MCUCFG_INITARCH),
				1 << (MCUCFG_INITARCH_SHIFT + cpu_id));
	else
		mmio_clrbits_32(mcucfg_reg(MCUCFG_INITARCH),
				1 << (MCUCFG_INITARCH_SHIFT + cpu_id));

	mmio_write_32(mcucfg_reg(MCUCFG_BOOTADDR[cpu_id]), secure_entrypoint);

	INFO("cpu_on[%d], entrypoint 0x%lx\n", cpu_id, secure_entrypoint);
	spmc_cpu_corex_onoff(cpu_id, STA_POWER_ON, MODE_SPMC_HW);

	return PSCI_E_SUCCESS;
}

void mtk_plat_power_domain_on_finish(const psci_power_state_t *state)
{
	plat_cpu_pwron_common();
}

void mtk_plat_power_domain_off(const psci_power_state_t *state)
{
	uint64_t mpidr = read_mpidr();
	int cpu_id = MPIDR_AFFLVL0_VAL(mpidr);

	plat_cpu_pwrdwn_common();

	INFO("cpu_off[%d]\n", cpu_id);
	spmc_cpu_corex_onoff(cpu_id, STA_POWER_DOWN, MODE_AUTO_SHUT_OFF);
}
