#include "halmac_2_platform.h"
#include "halmac_type.h"
#if HALMAC_PLATFORM_WINDOWS == 1 
#include "MP_Precomp.h" 
#endif
#include "halmac_88xx/halmac_api_88xx.h"
#include "halmac_88xx/halmac_88xx_cfg.h"
#include "halmac_88xx/halmac_8822b/halmac_8822b_cfg.h"
#include "halmac_88xx/halmac_8821c/halmac_8821c_cfg.h"
#include "halmac_88xx/halmac_8197f/halmac_8197f_cfg.h"


HALMAC_RET_STATUS	
halmac_check_platform_api(
	IN VOID *pDriver_adapter,
	IN HALMAC_INTERFACE halmac_interface,	
	IN PHALMAC_PLATFORM_API pHalmac_platform_api
);

VOID
halmac_init_adapter_parameter(
	IN	PHALMAC_ADAPTER pHalmac_adapter
);

HALMAC_RET_STATUS
halmac_mount_halmac_api(
	IN	PHALMAC_ADAPTER pHalmac_adapter
);

VOID
halmac_init_state_machine(
	IN	PHALMAC_ADAPTER pHalmac_adapter
);

/**
* halmac_init_adapter() - init halmac_adapter
* @pDriver_adapter
* @pHalmac_platform_api : platform api for halmac used
* @halmac_interface : PCIE, USB, or SDIO
* @ppHalmac_adapter
* @ppHalmac_api
* Author : KaiYuan Chang
* Return : HALMAC_RET_STATUS
*/
HALMAC_RET_STATUS  
halmac_init_adapter(
	IN void *pDriver_adapter,
	IN PHALMAC_PLATFORM_API pHalmac_platform_api,
	IN HALMAC_INTERFACE halmac_interface,
	OUT PHALMAC_ADAPTER *ppHalmac_adapter,
	OUT PHALMAC_API	*ppHalmac_api
)
{
	PHALMAC_ADAPTER pHalmac_adapter = (PHALMAC_ADAPTER)NULL;
	HALMAC_RET_STATUS status = HALMAC_RET_SUCCESS;
	
	union {
        u32 i;
        u8 x[4];
    } ENDIAN_CHECK = {0x01000000};

	status = halmac_check_platform_api(pDriver_adapter, halmac_interface, pHalmac_platform_api);
	if(HALMAC_RET_SUCCESS != status)
		return status;
	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, HALMAC_SVN_VER"\n");
	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "HALMAC_MAJOR_VER = %x\n",HALMAC_MAJOR_VER);
	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "HALMAC_PROTOTYPE_VER = %x\n",HALMAC_PROTOTYPE_VER);
	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "HALMAC_MINOR_VER = %x\n",HALMAC_MINOR_VER);	
	
	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_TRACE, "halmac_init_adapter_88xx ==========>\n");

	/* Check endian setting - Little endian : 1, Big endian : 0*/
	if (HALMAC_SYSTEM_ENDIAN == ENDIAN_CHECK.x[0]) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "Endian setting Err!!\n");
		pHalmac_adapter = (PHALMAC_ADAPTER)NULL;
		return HALMAC_RET_ENDIAN_ERR;
	}
		
	pHalmac_adapter = (PHALMAC_ADAPTER)pHalmac_platform_api->RTL_MALLOC(pDriver_adapter, sizeof(HALMAC_ADAPTER));
	if (NULL == pHalmac_adapter) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "Malloc HAL Adapter Err!!\n");	
		return HALMAC_RET_MALLOC_FAIL;
	}

	/* return halmac adapter address to caller */
	*ppHalmac_adapter = pHalmac_adapter;

	/* Record caller info */
	pHalmac_adapter->pHalmac_platform_api = pHalmac_platform_api;	
	pHalmac_adapter->pDriver_adapter = pDriver_adapter;
	pHalmac_adapter->halmac_interface = halmac_interface;

	PLATFORM_MUTEX_INIT(pDriver_adapter, &(pHalmac_adapter->EfuseMutex));
	PLATFORM_MUTEX_INIT(pDriver_adapter, &(pHalmac_adapter->h2c_seq_mutex));
	
	halmac_init_adapter_parameter(pHalmac_adapter);

	/* Assign function pointer to halmac API */
	status = halmac_mount_halmac_api(pHalmac_adapter);

	/* Return halmac API function pointer */
	*ppHalmac_api = (PHALMAC_API)pHalmac_adapter->pHalmac_api;

	PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_TRACE, "halmac_init_adapter_88xx <==========\n");		

	return status;

}

VOID
halmac_init_adapter_parameter(
	IN	PHALMAC_ADAPTER pHalmac_adapter
)
{	
	pHalmac_adapter->api_record.array_wptr = 0;
	pHalmac_adapter->pHalAdapter_backup = pHalmac_adapter;
	pHalmac_adapter->h2c_packet_seq = 0;
	pHalmac_adapter->pHalEfuse_map = (u8*)NULL;
	pHalmac_adapter->hal_efuse_map_valid = _FALSE;
	pHalmac_adapter->efuse_end = 0;
	pHalmac_adapter->pHal_mac_addr[0].Address_L_H.Address_Low = 0;
	pHalmac_adapter->pHal_mac_addr[0].Address_L_H.Address_High = 0;	
	pHalmac_adapter->pHal_mac_addr[1].Address_L_H.Address_Low = 0;
	pHalmac_adapter->pHal_mac_addr[1].Address_L_H.Address_High = 0;	
	pHalmac_adapter->pHal_bss_addr[0].Address_L_H.Address_Low = 0;
	pHalmac_adapter->pHal_bss_addr[0].Address_L_H.Address_High = 0;	
	pHalmac_adapter->pHal_bss_addr[1].Address_L_H.Address_Low = 0;
	pHalmac_adapter->pHal_bss_addr[1].Address_L_H.Address_High = 0;


    pHalmac_adapter->low_clk = _FALSE;
	pHalmac_adapter->h2c_buf_free_space = 0;
	pHalmac_adapter->max_download_size = HALMAC_FW_MAX_DL_SIZE;
		
	/* Init LPS Option */
	pHalmac_adapter->fwlps_option.mode = 0x01; /*0:Active 1:LPS 2:WMMPS*/
	pHalmac_adapter->fwlps_option.awake_interval = 1;
	pHalmac_adapter->fwlps_option.enter_32K= 1; 
	pHalmac_adapter->fwlps_option.clk_request = 0;
	pHalmac_adapter->fwlps_option.rlbm = 0;
	pHalmac_adapter->fwlps_option.smart_ps = 0;
	pHalmac_adapter->fwlps_option.awake_interval = 1;
	pHalmac_adapter->fwlps_option.all_queue_uapsd = 0;
	pHalmac_adapter->fwlps_option.pwr_state = 0;
	pHalmac_adapter->fwlps_option.low_pwr_rx_beacon = 0;
	pHalmac_adapter->fwlps_option.ant_auto_switch = 0;
	pHalmac_adapter->fwlps_option.ps_allow_bt_high_Priority = 0;
	pHalmac_adapter->fwlps_option.protect_bcn = 0;
	pHalmac_adapter->fwlps_option.silence_period = 0;
	pHalmac_adapter->fwlps_option.fast_bt_connect = 0;
	pHalmac_adapter->fwlps_option.two_antenna_en = 0;
	pHalmac_adapter->fwlps_option.adopt_user_Setting = 1;
	pHalmac_adapter->fwlps_option.drv_bcn_early_shift = 0;
		
	pHalmac_adapter->txff_bndy.tx_fifo_pg_num = 0;
	pHalmac_adapter->txff_bndy.ac_q_pg_num = 0;
	pHalmac_adapter->txff_bndy.rsvd_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.rsvd_drv_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.rsvd_h2c_extra_info_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.rsvd_h2c_queue_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.rsvd_cpu_instr_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.rsvd_fw_txbuff_pg_bndy = 0;
	pHalmac_adapter->txff_bndy.pub_queue_pg_num = 0;
	pHalmac_adapter->txff_bndy.la_mode = HALMAC_LA_MODE_DISABLE;
		
	pHalmac_adapter->config_para_info.pCfg_para_buf = NULL;
	pHalmac_adapter->config_para_info.pPara_buf_w = NULL;
	pHalmac_adapter->config_para_info.para_num = 0;
	pHalmac_adapter->config_para_info.full_fifo_mode = _FALSE;
	pHalmac_adapter->config_para_info.para_buf_size = 0;
	pHalmac_adapter->config_para_info.avai_para_buf_size = 0;
	pHalmac_adapter->config_para_info.offset_accumulation =0;
	pHalmac_adapter->config_para_info.value_accumulation =0;
	pHalmac_adapter->config_para_info.datapack_segment =0;
	
	pHalmac_adapter->ch_sw_info.ch_info_buf = NULL;
	pHalmac_adapter->ch_sw_info.ch_info_buf_w = NULL;
	pHalmac_adapter->ch_sw_info.extra_info_en = 0;
	pHalmac_adapter->ch_sw_info.buf_size = 0;
	pHalmac_adapter->ch_sw_info.avai_buf_size = 0;
	pHalmac_adapter->ch_sw_info.total_size = 0;
	pHalmac_adapter->ch_sw_info.ch_num = 0;

	pHalmac_adapter->gen_info_valid = _FALSE;

	PLATFORM_RTL_MEMSET(pHalmac_adapter->pDriver_adapter, pHalmac_adapter->api_record.api_array, HALMAC_API_STUFF, sizeof(pHalmac_adapter->api_record.api_array));
	
    halmac_init_state_machine(pHalmac_adapter);
		
#if HALMAC_PLATFORM_TESTPROGRAM
	u32 i = 0;
	VOID *pDriver_adapter = pHalmac_adapter->pDriver_adapter;
	
	PLATFORM_MUTEX_INIT(pDriver_adapter, &(pHalmac_adapter->agg_buff_mutex));	
	for (i=0;i<4;i++) {
		pHalmac_adapter->halmac_tx_buf_info[i].pTx_agg_buf = (u8*)PLATFORM_RTL_MALLOC(pDriver_adapter,HALMAC_TX_AGG_BUFF_SIZE);
		PLATFORM_RTL_MEMSET(pDriver_adapter, pHalmac_adapter->halmac_tx_buf_info[i].pTx_agg_buf, 0x00, HALMAC_TX_AGG_BUFF_SIZE);
		pHalmac_adapter->halmac_tx_buf_info[i].pCurr_pkt_buf = pHalmac_adapter->halmac_tx_buf_info[i].pTx_agg_buf;
		pHalmac_adapter->halmac_tx_buf_info[i].avai_buf_size = HALMAC_TX_AGG_BUFF_SIZE;
		pHalmac_adapter->halmac_tx_buf_info[i].total_pkt_size = 0;
		pHalmac_adapter->halmac_tx_buf_info[i].agg_num =0 ;
	}
	pHalmac_adapter->max_agg_num = 3;
#endif

}

VOID
halmac_init_state_machine(
	IN	PHALMAC_ADAPTER pHalmac_adapter
)
{
	PHALMAC_STATE pState = &(pHalmac_adapter->halmac_state);

    pState->efuse_state_set.efuse_cmd_construct_state = HALMAC_EFUSE_CMD_CONSTRUCT_IDLE;
    pState->efuse_state_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->efuse_state_set.seq_num = pHalmac_adapter->h2c_packet_seq;

    pState->cfg_para_state_set.cfg_para_cmd_construct_state = HALMAC_CFG_PARA_CMD_CONSTRUCT_IDLE;
    pState->cfg_para_state_set.process_status = HALMAC_CMD_PROCESS_IDLE;
    pState->cfg_para_state_set.seq_num = pHalmac_adapter->h2c_packet_seq;
    
    pState->scan_state_set.scan_cmd_construct_state = HALMAC_SCAN_CMD_CONSTRUCT_IDLE;
    pState->scan_state_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->scan_state_set.seq_num = pHalmac_adapter->h2c_packet_seq;
	
    pState->update_packet_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->update_packet_set.seq_num = pHalmac_adapter->h2c_packet_seq;

	pState->iqk_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->iqk_set.seq_num = pHalmac_adapter->h2c_packet_seq;
	
	pState->power_tracking_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->power_tracking_set.seq_num = pHalmac_adapter->h2c_packet_seq;
	
	pState->psd_set.process_status = HALMAC_CMD_PROCESS_IDLE;
	pState->psd_set.seq_num = pHalmac_adapter->h2c_packet_seq;
	pState->psd_set.data_size = 0;
	pState->psd_set.segment_size = 0;
	pState->psd_set.pData = NULL;

	pState->api_state = HALMAC_API_STATE_INIT;

	pState->dlfw_state = HALMAC_DLFW_NONE;
	pState->mac_power = HALMAC_MAC_POWER_OFF;
	pState->ps_state = HALMAC_PS_STATE_UNDEFINE;
}

HALMAC_RET_STATUS
halmac_mount_halmac_api(
	IN	PHALMAC_ADAPTER pHalmac_adapter
)
{		
	VOID *pDriver_adapter = pHalmac_adapter->pDriver_adapter;
	PHALMAC_API	pHalmac_api = (PHALMAC_API)NULL;
	u8 chip_id,chip_version,polling_count;

	
	pHalmac_adapter->pHalmac_api = (PHALMAC_API)PLATFORM_RTL_MALLOC(pDriver_adapter, sizeof(HALMAC_API));
	if (NULL == pHalmac_adapter->pHalmac_api)
		return HALMAC_RET_MALLOC_FAIL;
	pHalmac_api = (PHALMAC_API)pHalmac_adapter->pHalmac_api;

	/* Mount function pointer */
	pHalmac_api->halmac_download_firmware = halmac_download_firmware_88xx;
	pHalmac_api->halmac_get_fw_version = halmac_get_fw_version_88xx;
	pHalmac_api->halmac_cfg_mac_addr = halmac_cfg_mac_addr_88xx;
	pHalmac_api->halmac_cfg_bssid = halmac_cfg_bssid_88xx;
	pHalmac_api->halmac_cfg_multicast_addr = halmac_cfg_multicast_addr_88xx;
	pHalmac_api->halmac_pre_init_system_cfg = halmac_pre_init_system_cfg_88xx;
	pHalmac_api->halmac_init_system_cfg = halmac_init_system_cfg_88xx;
	pHalmac_api->halmac_init_protocol_cfg = halmac_init_protocol_cfg_88xx;
	pHalmac_api->halmac_init_edca_cfg = halmac_init_edca_cfg_88xx;
	pHalmac_api->halmac_cfg_operation_mode = halmac_cfg_operation_mode_88xx;
	pHalmac_api->halmac_cfg_ch_bw = halmac_cfg_ch_bw_88xx;	
	pHalmac_api->halmac_cfg_bw = halmac_cfg_bw_88xx;		
	pHalmac_api->halmac_init_wmac_cfg = halmac_init_wmac_cfg_88xx;
	pHalmac_api->halmac_init_mac_cfg = halmac_init_mac_cfg_88xx;
	pHalmac_api->halmac_init_sdio_cfg = halmac_init_sdio_cfg_88xx;
	pHalmac_api->halmac_init_usb_cfg = halmac_init_usb_cfg_88xx;
	pHalmac_api->halmac_init_pcie_cfg = halmac_init_pcie_cfg_88xx;
	pHalmac_api->halmac_deinit_sdio_cfg = halmac_deinit_sdio_cfg_88xx;
	pHalmac_api->halmac_deinit_usb_cfg = halmac_deinit_usb_cfg_88xx;
	pHalmac_api->halmac_deinit_pcie_cfg = halmac_deinit_pcie_cfg_88xx;
	pHalmac_api->halmac_dump_efuse_map = halmac_dump_efuse_map_88xx;
	pHalmac_api->halmac_dump_logical_efuse_map = halmac_dump_logical_efuse_map_88xx;
	pHalmac_api->halmac_write_efuse = halmac_write_efuse_88xx;
	pHalmac_api->halmac_pg_efuse_by_map = halmac_pg_efuse_by_map_88xx;
	pHalmac_api->halmac_get_efuse_size = halmac_get_efuse_size_88xx;	
	pHalmac_api->halmac_get_c2h_info = halmac_get_c2h_info_88xx;
	pHalmac_api->halmac_read_efuse = halmac_read_efuse_88xx;
	
	pHalmac_api->halmac_get_logical_efuse_size = halmac_get_logical_efuse_size_88xx;	
	
	pHalmac_api->halmac_write_logical_efuse = halmac_write_logical_efuse_88xx;
	pHalmac_api->halmac_read_logical_efuse = halmac_read_logical_efuse_88xx;	
		
	pHalmac_api->halmac_cfg_fwlps_option = halmac_cfg_fwlps_option_88xx;
	pHalmac_api->halmac_cfg_fwips_option = halmac_cfg_fwips_option_88xx;
	pHalmac_api->halmac_enter_wowlan = halmac_enter_wowlan_88xx;
	pHalmac_api->halmac_leave_wowlan = halmac_leave_wowlan_88xx;
	pHalmac_api->halmac_enter_ps = halmac_enter_ps_88xx;
	pHalmac_api->halmac_leave_ps = halmac_leave_ps_88xx;
	pHalmac_api->halmac_h2c_lb = halmac_h2c_lb_88xx;
	pHalmac_api->halmac_debug = halmac_debug_88xx;
	pHalmac_api->halmac_cfg_parameter = halmac_cfg_parameter_88xx;
	pHalmac_api->halmac_update_datapack = halmac_update_datapack_88xx;	
	pHalmac_api->halmac_run_datapack = halmac_run_datapack_88xx;	
	pHalmac_api->halmac_cfg_drv_info = halmac_cfg_drv_info_88xx; 
	pHalmac_api->halmac_send_bt_coex = halmac_send_bt_coex_88xx; 
	pHalmac_api->halmac_verify_platform_api = halmac_verify_platform_api_88xx;		
	pHalmac_api->halmac_update_packet = halmac_update_packet_88xx;
	pHalmac_api->halmac_bcn_ie_filter = halmac_bcn_ie_filter_88xx;
	pHalmac_api->halmac_cfg_txbf = halmac_cfg_txbf_88xx;
	pHalmac_api->halmac_cfg_mumimo = halmac_cfg_mumimo_88xx;
	pHalmac_api->halmac_cfg_sounding = halmac_cfg_sounding_88xx;
	pHalmac_api->halmac_del_sounding = halmac_del_sounding_88xx;
	pHalmac_api->halmac_su_bfer_entry_init = halmac_su_bfer_entry_init_88xx;
	pHalmac_api->halmac_su_bfee_entry_init = halmac_su_bfee_entry_init_88xx;
	pHalmac_api->halmac_mu_bfer_entry_init = halmac_mu_bfer_entry_init_88xx;
	pHalmac_api->halmac_mu_bfee_entry_init = halmac_mu_bfee_entry_init_88xx;
	pHalmac_api->halmac_su_bfer_entry_del = halmac_su_bfer_entry_del_88xx;
	pHalmac_api->halmac_su_bfee_entry_del = halmac_su_bfee_entry_del_88xx;
	pHalmac_api->halmac_mu_bfer_entry_del = halmac_mu_bfer_entry_del_88xx;
	pHalmac_api->halmac_mu_bfee_entry_del = halmac_mu_bfee_entry_del_88xx;
	
	pHalmac_api->halmac_add_ch_info = halmac_add_ch_info_88xx;
	pHalmac_api->halmac_add_extra_ch_info = halmac_add_extra_ch_info_88xx;
	pHalmac_api->halmac_ctrl_ch_switch = halmac_ctrl_ch_switch_88xx;	
	pHalmac_api->halmac_clear_ch_info = halmac_clear_ch_info_88xx;	
	pHalmac_api->halmac_send_general_info = halmac_send_general_info_88xx;	

	pHalmac_api->halmac_start_iqk = halmac_start_iqk_88xx;
	pHalmac_api->halmac_ctrl_pwr_tracking = halmac_ctrl_pwr_tracking_88xx;
	pHalmac_api->halmac_psd = halmac_psd_88xx;
	pHalmac_api->halmac_cfg_la_mode = halmac_cfg_la_mode_88xx;

	pHalmac_api->halmac_query_status = halmac_query_status_88xx;
	pHalmac_api->halmac_reset_feature = halmac_reset_feature_88xx;
	pHalmac_api->halmac_check_fw_status = halmac_check_fw_status_88xx;
	pHalmac_api->halmac_dump_fw_dmem = halmac_dump_fw_dmem_88xx;
	pHalmac_api->halmac_cfg_max_dl_size= halmac_cfg_max_dl_size_88xx;	
	
	pHalmac_api->halmac_dump_fifo = halmac_dump_fifo_88xx;
	pHalmac_api->halmac_get_fifo_size = halmac_get_fifo_size_88xx;
		
	if (HALMAC_INTERFACE_SDIO == pHalmac_adapter->halmac_interface) {
		pHalmac_api->halmac_cfg_rx_aggregation=halmac_cfg_rx_aggregation_88xx_sdio;
		pHalmac_api->halmac_init_interface_cfg=halmac_init_sdio_cfg_88xx;
		pHalmac_api->halmac_deinit_interface_cfg=halmac_deinit_sdio_cfg_88xx;	
		pHalmac_api->halmac_reg_read_8=halmac_reg_read_8_sdio;
		pHalmac_api->halmac_reg_write_8=halmac_reg_write_8_sdio;
		pHalmac_api->halmac_reg_read_16=halmac_reg_read_16_sdio;
		pHalmac_api->halmac_reg_write_16=halmac_reg_write_16_sdio;
		pHalmac_api->halmac_reg_read_32=halmac_reg_read_32_sdio;
		pHalmac_api->halmac_reg_write_32=halmac_reg_write_32_sdio;
	} else if(HALMAC_INTERFACE_USB ==pHalmac_adapter->halmac_interface) {
		pHalmac_api->halmac_cfg_rx_aggregation=halmac_cfg_rx_aggregation_88xx_usb;
		pHalmac_api->halmac_init_interface_cfg=halmac_init_usb_cfg_88xx;
		pHalmac_api->halmac_deinit_interface_cfg=halmac_deinit_usb_cfg_88xx;			
		pHalmac_api->halmac_reg_read_8=halmac_reg_read_8_usb;
		pHalmac_api->halmac_reg_write_8=halmac_reg_write_8_usb;
		pHalmac_api->halmac_reg_read_16=halmac_reg_read_16_usb;
		pHalmac_api->halmac_reg_write_16=halmac_reg_write_16_usb;
		pHalmac_api->halmac_reg_read_32=halmac_reg_read_32_usb;
		pHalmac_api->halmac_reg_write_32=halmac_reg_write_32_usb;
	} else if(HALMAC_INTERFACE_PCIE ==pHalmac_adapter->halmac_interface) {
		pHalmac_api->halmac_cfg_rx_aggregation=halmac_cfg_rx_aggregation_88xx_pcie;
		pHalmac_api->halmac_init_interface_cfg=halmac_init_pcie_cfg_88xx;
		pHalmac_api->halmac_deinit_interface_cfg=halmac_deinit_pcie_cfg_88xx;			
		pHalmac_api->halmac_reg_read_8=halmac_reg_read_8_pcie;
		pHalmac_api->halmac_reg_write_8=halmac_reg_write_8_pcie;
		pHalmac_api->halmac_reg_read_16=halmac_reg_read_16_pcie;
		pHalmac_api->halmac_reg_write_16=halmac_reg_write_16_pcie;
		pHalmac_api->halmac_reg_read_32=halmac_reg_read_32_pcie;
		pHalmac_api->halmac_reg_write_32=halmac_reg_write_32_pcie;
	} else {
		PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "Set halmac io function Error!!\n");			
	}

	pHalmac_api->halmac_set_bulkout_num = halmac_set_bulkout_num_88xx;
	pHalmac_api->halmac_get_sdio_tx_addr = halmac_get_sdio_tx_addr_88xx;
	pHalmac_api->halmac_get_usb_bulkout_id =halmac_get_usb_bulkout_id_88xx;
	pHalmac_api->halmac_timer_2s = halmac_timer_2s_88xx;
	pHalmac_api->halmac_fill_txdesc_checksum = halmac_fill_txdesc_check_sum_88xx;
	
	/* Get Chip_id and Chip_version */

	chip_id = HALMAC_REG_READ_8(pHalmac_adapter, REG_SYS_CFG2);
	if (HALMAC_INTERFACE_SDIO == pHalmac_adapter->halmac_interface) {
		if (chip_id == 0xEA)
			HALMAC_REG_WRITE_8(pHalmac_adapter, REG_SDIO_HSUS_CTRL, HALMAC_REG_READ_8(pHalmac_adapter, REG_SDIO_HSUS_CTRL) & ~(BIT(0)));

		polling_count = HALMAC_POLLING_READY_TIMEOUT_COUNT;
		while (!(HALMAC_REG_READ_8(pHalmac_adapter, REG_SDIO_HSUS_CTRL) & 0x02)) {
			polling_count--;
			if (polling_count == 0)
				return	HALMAC_RET_SDIO_LEAVE_SUSPEND_FAIL;
		}
		
		chip_id = HALMAC_REG_READ_8(pHalmac_adapter, REG_SYS_CFG2);
	}
	chip_version = (u8)HALMAC_REG_READ_8(pHalmac_adapter, REG_SYS_CFG1 + 1) >> 4 ; 
	
	pHalmac_adapter->chip_version = (HALMAC_CHIP_VER)chip_version;
	
	if (HALMAC_CHIP_ID_HW_DEF_8822B == chip_id) {
		pHalmac_adapter->chip_id = HALMAC_CHIP_ID_8822B;
		pHalmac_adapter->hw_config_info.efuse_size = HALMAC_EFUSE_SIZE_8822B;
		pHalmac_adapter->hw_config_info.txdesc_size = HALMAC_TX_DESC_SIZE_8822B;
		pHalmac_adapter->hw_config_info.rxdesc_size = HALMAC_RX_DESC_SIZE_8822B;
		pHalmac_adapter->hw_config_info.tx_fifo_size = HALMAC_TX_FIFO_SIZE_8822B;
		pHalmac_adapter->hw_config_info.rx_fifo_size = HALMAC_RX_FIFO_SIZE_8822B;
#if HALMAC_8822B_SUPPORT 
		pHalmac_api->halmac_init_trx_cfg = halmac_init_trx_cfg_8822b;
		pHalmac_api->halmac_init_h2c = halmac_init_h2c_8822b;
	
		if (HALMAC_INTERFACE_SDIO == pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_tx_allowed_sdio = halmac_tx_allowed_sdio_8822b;		
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8822b_sdio;	
		} else if(HALMAC_INTERFACE_USB== pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8822b_usb;	
		} else if(HALMAC_INTERFACE_PCIE== pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8822b_pcie;	
		}	
#endif


	} else if(HALMAC_CHIP_ID_HW_DEF_8821C == chip_id) {
		pHalmac_adapter->chip_id = HALMAC_CHIP_ID_8821C;	
		pHalmac_adapter->hw_config_info.efuse_size = HALMAC_EFUSE_SIZE_8821C;
		pHalmac_adapter->hw_config_info.txdesc_size = HALMAC_TX_DESC_SIZE_8821C;
		pHalmac_adapter->hw_config_info.rxdesc_size = HALMAC_RX_DESC_SIZE_8821C;
		pHalmac_adapter->hw_config_info.tx_fifo_size = HALMAC_TX_FIFO_SIZE_8821C;
		pHalmac_adapter->hw_config_info.rx_fifo_size = HALMAC_RX_FIFO_SIZE_8821C;			
#if HALMAC_8821C_SUPPORT 
		pHalmac_api->halmac_init_trx_cfg = halmac_init_trx_cfg_8821C;
		pHalmac_api->halmac_init_h2c = halmac_init_h2c_8821c;
	
		if (HALMAC_INTERFACE_SDIO == pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_tx_allowed_sdio = halmac_tx_allowed_sdio_8821c;
			pHalmac_api->halmac_cfg_tx_agg_align=halmac_cfg_tx_agg_align_8821C_sdio;
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8821c_sdio;	
		} else if (HALMAC_INTERFACE_USB== pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8821c_usb;	
		} else if (HALMAC_INTERFACE_PCIE== pHalmac_adapter->halmac_interface) {
			pHalmac_api->halmac_mac_power_switch=halmac_mac_power_switch_8821c_pcie;	
		}
#endif


	} else if(HALMAC_CHIP_ID_HW_DEF_8197F == chip_id) {
		pHalmac_adapter->chip_id = HALMAC_CHIP_ID_8197F;
	
		pHalmac_adapter->hw_config_info.efuse_size = HALMAC_EFUSE_SIZE_8197F;
		pHalmac_adapter->hw_config_info.txdesc_size = HALMAC_TX_DESC_SIZE_8197F;
		pHalmac_adapter->hw_config_info.rxdesc_size = HALMAC_RX_DESC_SIZE_8197F;
		pHalmac_adapter->hw_config_info.tx_fifo_size = HALMAC_TX_FIFO_SIZE_8197F;
		pHalmac_adapter->hw_config_info.rx_fifo_size = HALMAC_RX_FIFO_SIZE_8197F;
	} else {
		PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "Chip ID undefine!!\n");
		//return HALMAC_RET_CHIP_NOT_SUPPORT;
	}
	
#if HALMAC_PLATFORM_TESTPROGRAM
	pHalmac_api->halmac_gen_txdesc = halmac_gen_tx_desc_88xx;
	pHalmac_api->halmac_txdesc_parser = halmac_tx_desc_parser_88xx;
	pHalmac_api->halmac_rxdesc_parser = halmac_rx_desc_parser_88xx;
	pHalmac_api->halmac_get_txdesc_size = halmac_get_txdesc_size_88xx;
	pHalmac_api->halmac_send_packet = halmac_send_packet_88xx;
	pHalmac_api->halmac_parse_packet = halmac_parse_packet_88xx;
	
	pHalmac_api->halmac_get_pcie_packet = halmac_get_pcie_packet_88xx;
	pHalmac_api->halmac_gen_txagg_desc = halmac_gen_txagg_desc_88xx;
	
	pHalmac_api->halmac_bb_reg_read = halmac_bb_reg_read_88xx;
	pHalmac_api->halmac_bb_reg_write = halmac_bb_reg_write_88xx;
	
	pHalmac_api->halmac_rf_reg_read = halmac_rf_ac_reg_read_88xx;
	pHalmac_api->halmac_rf_reg_write = halmac_rf_ac_reg_write_88xx;
	pHalmac_api->halmac_init_antenna_selection = halmac_init_antenna_selection_88xx;
	pHalmac_api->halmac_bb_preconfig = halmac_bb_preconfig_88xx;
	pHalmac_api->halmac_init_crystal_capacity = halmac_init_crystal_capacity_88xx;
	pHalmac_api->halmac_trx_antenna_setting = halmac_trx_antenna_setting_88xx;
	
	pHalmac_api->halmac_himr_setting_sdio = halmac_himr_setting_88xx_sdio;
	
	pHalmac_api->halmac_get_cam_num = halmac_get_cam_num_8822b;
	pHalmac_api->halmac_dump_cam_table = halmac_dump_cam_table_8822b;
	pHalmac_api->halmac_send_beacon = halmac_send_beacon_88xx;
	pHalmac_api->halmac_stop_beacon = halmac_stop_beacon_88xx;
	pHalmac_api->halmac_check_trx_status = halmac_check_trx_status_88xx;	
	pHalmac_api->halmac_set_agg_num = halmac_set_agg_num_88xx;
	pHalmac_api->halmac_send_management= halmac_send_management_88xx;
	pHalmac_api->halmac_send_control= halmac_send_control_88xx;
	pHalmac_api->halmac_send_hiqueue = halmac_send_hiqueue_88xx;
	//pHalmac_api->halmac_run_pwrseq = halmac_run_pwrseq_88xx;
	pHalmac_api->halmac_media_status_rpt = halmac_media_status_rpt_88xx;
	
	pHalmac_api->halmac_timer_10ms = halmac_timer_10ms_88xx;
	
	pHalmac_api->halmac_download_firmware_fpag = halmac_download_firmware_fpga_88xx;
	pHalmac_api->halmac_download_rom_fpga = halmac_download_rom_fpga_88xx;
	pHalmac_api->halmac_send_nlo = halmac_send_nlo_88xx;

	pHalmac_api->halmac_get_chip_type = halmac_get_chip_type_88xx;
	
	if (pHalmac_adapter->chip_id == HALMAC_CHIP_ID_8822B)
		pHalmac_api->halmac_run_pwrseq = halmac_run_pwrseq_8822b;
	else if (pHalmac_adapter->chip_id == HALMAC_CHIP_ID_8821C)
		pHalmac_api->halmac_run_pwrseq = halmac_run_pwrseq_8821c;

	if (HALMAC_INTERFACE_SDIO == pHalmac_adapter->halmac_interface) {
		pHalmac_api->halmac_reg_read_8 = halmac_reg_read_8_sdio_testprogram;
		pHalmac_api->halmac_reg_write_8 = halmac_reg_write_8_sdio_testprogram;
		pHalmac_api->halmac_reg_read_16 = halmac_reg_read_16_sdio_testprogram;
		pHalmac_api->halmac_reg_write_16 = halmac_reg_write_16_sdio_testprogram;
		pHalmac_api->halmac_reg_read_32 = halmac_reg_read_32_sdio_testprogram;
		pHalmac_api->halmac_reg_write_32 = halmac_reg_write_32_sdio_testprogram;
	}
#endif
	return HALMAC_RET_SUCCESS;
}

/**
* halmac_halt_api() - halt all halmac api
* @pHalmac_adapter
* Author : Ivan Lin
* Return : HALMAC_RET_STATUS
*/
HALMAC_RET_STATUS  
halmac_halt_api(
	IN PHALMAC_ADAPTER pHalmac_adapter
)
{
	VOID *pDriver_adapter = NULL;
	PHALMAC_PLATFORM_API pHalmac_platform_api = (PHALMAC_PLATFORM_API)NULL;

	if (HALMAC_RET_SUCCESS != halmac_adapter_validate(pHalmac_adapter))
		return HALMAC_RET_ADAPTER_INVALID;
	
	pDriver_adapter = pHalmac_adapter->pDriver_adapter;
	pHalmac_platform_api = pHalmac_adapter->pHalmac_platform_api;
	
	PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "halmac_halt_api ==========>\n");
	pHalmac_adapter->halmac_state.api_state = HALMAC_API_STATE_HALT;
	PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "halmac_halt_api ==========>\n");
	return HALMAC_RET_SUCCESS;
}

/**
* halmac_deinit_adapter() - deinit halmac adapter
* @pHalmac_adapter
* Author : KaiYuan Chang
* Return : HALMAC_RET_STATUS
*/
HALMAC_RET_STATUS  
halmac_deinit_adapter(
	IN PHALMAC_ADAPTER pHalmac_adapter
)
{	
	VOID *pDriver_adapter = NULL;
	PHALMAC_PLATFORM_API pHalmac_platform_api = (PHALMAC_PLATFORM_API)NULL;

	if (HALMAC_RET_SUCCESS != halmac_adapter_validate(pHalmac_adapter))
		return HALMAC_RET_ADAPTER_INVALID;
	
	pDriver_adapter = pHalmac_adapter->pDriver_adapter;
	pHalmac_platform_api = pHalmac_adapter->pHalmac_platform_api;
	
	PLATFORM_MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_TRACE, "halmac_deinit_adapter_88xx ==========>\n");
	
	PLATFORM_MUTEX_DEINIT(pDriver_adapter, &(pHalmac_adapter->EfuseMutex));
	PLATFORM_MUTEX_DEINIT(pDriver_adapter, &(pHalmac_adapter->h2c_seq_mutex));

	if (NULL != pHalmac_adapter->pHalEfuse_map) {
		PLATFORM_RTL_FREE(pDriver_adapter, pHalmac_adapter->pHalEfuse_map, pHalmac_adapter->hw_config_info.efuse_size);
		pHalmac_adapter->pHalEfuse_map = (u8*)NULL;
	}
	
	if (NULL != pHalmac_adapter->halmac_state.psd_set.pData) {
		PLATFORM_RTL_FREE(pDriver_adapter, pHalmac_adapter->halmac_state.psd_set.pData, pHalmac_adapter->halmac_state.psd_set.data_size);
		pHalmac_adapter->halmac_state.psd_set.pData = (u8*)NULL;
	}
	
	if (NULL != pHalmac_adapter->pHalmac_api) {
		PLATFORM_RTL_FREE(pDriver_adapter, pHalmac_adapter->pHalmac_api, sizeof(HALMAC_API));
		pHalmac_adapter->pHalmac_api = NULL;
	}

	if (NULL != pHalmac_adapter) {
		pHalmac_adapter->pHalAdapter_backup = NULL;
		PLATFORM_RTL_FREE(pDriver_adapter, pHalmac_adapter, sizeof(HALMAC_ADAPTER));
		pHalmac_adapter = (PHALMAC_ADAPTER)NULL;
	}

	return HALMAC_RET_SUCCESS;
}

/**
* halmac_check_platform_api() - check platform api pointers
* @pDriver_adapter
* @halmac_interface : PCIE, USB or SDIO
* @pHalmac_platform_api
* Author : KaiYuan Chang
* Return : HALMAC_RET_STATUS
*/
HALMAC_RET_STATUS	
halmac_check_platform_api(
	IN VOID *pDriver_adapter,
	IN HALMAC_INTERFACE halmac_interface,	
	IN PHALMAC_PLATFORM_API pHalmac_platform_api
)
{
	VOID *pAdapter_Local=NULL;
	pAdapter_Local = pDriver_adapter;

	if (NULL == pHalmac_platform_api)
		return HALMAC_RET_PLATFORM_API_NULL;

	if (NULL == pHalmac_platform_api->MSG_PRINT)
		return HALMAC_RET_PLATFORM_API_NULL;
	
	if (HALMAC_INTERFACE_SDIO == halmac_interface) {
		if (NULL == pHalmac_platform_api->SDIO_CMD52_READ) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD52_READ)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_READ_8) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_READ_8)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_READ_16) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_READ_16)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_READ_32) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_READ_32)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD52_WRITE) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD52_WRITE)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_WRITE_8) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_WRITE_8)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_WRITE_16) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_WRITE_16)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->SDIO_CMD53_WRITE_32) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->SDIO_CMD53_WRITE_32)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
	}
	
	if ((HALMAC_INTERFACE_USB == halmac_interface) || (HALMAC_INTERFACE_PCIE == halmac_interface)) {		
		if (NULL == pHalmac_platform_api->REG_READ_8) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_READ_8)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->REG_READ_16) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_READ_16)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->REG_READ_32) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_READ_32)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->REG_WRITE_8) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_WRITE_8)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}	
		if (NULL == pHalmac_platform_api->REG_WRITE_16) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_WRITE_16)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}
		if (NULL == pHalmac_platform_api->REG_WRITE_32) {
			pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->REG_WRITE_32)\n");
			return HALMAC_RET_PLATFORM_API_NULL;
		}		
	}

	if (NULL == pHalmac_platform_api->RTL_FREE) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->RTL_FREE)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}
	
	if (NULL == pHalmac_platform_api->RTL_MALLOC) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->RTL_MALLOC)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}
	if (NULL == pHalmac_platform_api->RTL_MEMCPY) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->RTL_MEMCPY)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}
	if (NULL == pHalmac_platform_api->RTL_MEMSET) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->RTL_MEMSET)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}
	if (NULL == pHalmac_platform_api->RTL_DELAY_US) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->RTL_DELAY_US)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}

	if (NULL == pHalmac_platform_api->MUTEX_INIT) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->MUTEX_INIT)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}	
	if (NULL == pHalmac_platform_api->MUTEX_DEINIT) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->MUTEX_DEINIT)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}	
	if (NULL == pHalmac_platform_api->MUTEX_LOCK) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->MUTEX_LOCK)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}	
	if (NULL == pHalmac_platform_api->MUTEX_UNLOCK) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->MUTEX_UNLOCK)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}
	if (NULL == pHalmac_platform_api->EVENT_INDICATION) {
		pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_ERR, "(NULL==pHalmac_platform_api->EVENT_INDICATION)\n");
		return HALMAC_RET_PLATFORM_API_NULL;
	}		

	pHalmac_platform_api->MSG_PRINT(pDriver_adapter, HALMAC_MSG_INIT, HALMAC_DBG_TRACE, "halmac_check_platform_api ==========>\n");

	return HALMAC_RET_SUCCESS;
}

HALMAC_RET_STATUS  
halmac_get_version(
	OUT HALMAC_VER *version
)
{
	version->major_ver = (u8)HALMAC_MAJOR_VER;
	version->prototype_ver = (u8)HALMAC_PROTOTYPE_VER;
	version->minor_ver = (u8)HALMAC_MINOR_VER;

	return HALMAC_RET_SUCCESS;
}
